#
# $Id: lonwhatsnew.pm,v 1.67 2006/12/20 23:02:33 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#


package Apache::lonwhatsnew;

use strict;
use lib qw(/home/httpd/lib/perl);
use Apache::lonnet;
use Apache::loncommon();
use Apache::lonhtmlcommon();
use Apache::lonlocal;
use Apache::loncoursedata();
use Apache::lonnavmaps();
use Apache::lonuserstate;
use Apache::Constants qw(:common :http);
use Time::Local;
use GDBM_File;
use lib '/home/httpd/lib/perl/';
use LONCAPA;

#----------------------------
# handler
#
#----------------------------

sub handler {
    my $r = shift;
    if ($r->header_only) {
        &Apache::loncommon::content_type($r,'text/html');
        $r->send_http_header;
        return OK;
    }
    &Apache::loncommon::get_unprocessed_cgi(
                                   $ENV{'QUERY_STRING'},['command','refpage']);

    my $command = $env{'form.command'};
    my $refpage = $env{'form.refpage'};

    my %checkallowed = ( coursenormalmail => 1,
			 coursecritmail => 1, );
    foreach my $perm_check (['whn','whatsnew',1],
			    ['pch','coursediscussion',1],
			    ['mgr','handgrading',1],
			    ['vgr','abovethreshold',1],
			    ['opa','haserrors',1],
			    ['mdc','versionchanges',0],
			    ) {
	my ($perm,$key,$check_section) = @{ $perm_check };
	my $scope = $env{'request.course.id'};
	if (!($checkallowed{$key} = &Apache::lonnet::allowed($perm,$scope))) {
	    $scope .= '/'.$env{'request.course.sec'};
	    if ( $check_section ) {
		$checkallowed{$key} = &Apache::lonnet::allowed($perm,$scope);
	    }
	    if ($checkallowed{$key}) {
		$checkallowed{$key.'_section'} = $env{'request.course.sec'};
	    }
	}
    }

    if ( ! $env{'request.course.fn'} || ! $checkallowed{'whatsnew'}) {
        # Not in a course, or no whn priv in course
        $env{'user.error.msg'}="/adm/whatsnew::whn:0:0:Cannot display what's new page";
        return HTTP_NOT_ACCEPTABLE;
    }

    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;

    $r->print(&display_header($command,\%checkallowed));

    &Apache::lonhtmlcommon::clear_breadcrumbs();
    &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/whatsnew',
              text=>"Display Action Items"});
    if (($command eq 'chgthreshold') && $checkallowed{'abovethreshold'}) {
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/whatsnew?command=chgthreshold&refpage='.$refpage,
              text=>"Change thresholds"});
        $r->print(&Apache::lonhtmlcommon::breadcrumbs
            ("What's New?",#'Course_Action_Items_Thresholds'
	     ));
    } elsif (($command eq 'chginterval') && $checkallowed{'versionchanges'} ) {
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/whatsnew?command=chginterval&refpage='.$refpage,
              text=>"Change interval"});
        $r->print(&Apache::lonhtmlcommon::breadcrumbs
            ("What's New?",#'Course_Action_Items_Intervals'
	     ));
    } elsif (($command eq 'chgdisc') && $checkallowed{'coursediscussion'}) {
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/whatsnew?command=chgdisc&refpage='.$refpage,
              text=>"Change discussion display"});
        $r->print(&Apache::lonhtmlcommon::breadcrumbs
            ("What's New?",#'Course_Action_Items_Intervals'
	     ));
    } elsif ($command eq 'courseinit') {
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/whatsnew?command=courseinit&refpage='.$refpage,
              text=>"Course initialization preference"});
        $r->print(&Apache::lonhtmlcommon::breadcrumbs
            ("What's New?",#'Course_Action_Items_Initialization'
	     ));
    } else {
        $r->print(&Apache::lonhtmlcommon::breadcrumbs
            ("What's New?",#'Course_Action_Items_Display'
	     ));
    }
    &display_main_box($r,$command,$refpage,\%checkallowed);
    return OK;
}

#------------------------------
# display_main_box
#
# Display all the elements within the main box
#------------------------------
                                                                                
sub display_main_box {
    my ($r,$command,$refpage,$checkallowed) = @_;
    my $domain=&Apache::loncommon::determinedomain();
    my $function = &Apache::loncommon::get_users_function();
    my $tabbg=&Apache::loncommon::designparm($function.'.tabbg',$domain);
    my $lctype = lc(&Apache::loncommon::course_type());
    $r->print('<table width="100%" border="0" cellpadding="5" cellspacing="0"><tr><td width="100%">');

    my %threshold_titles = &Apache::lonlocal::texthash (
                         av_attempts => 'Average number of attempts',
                         degdiff => 'Degree of difficulty',
                         numstudents => 'Total number of students with submissions',
    );

    my %interval_titles = &Apache::lonlocal::texthash (
                            -1 => "since start of $lctype",
                       2592000 => 'since last month',
                        604800 => 'since last week',
                         86400 => 'since yesterday',
    );

    my %initpage = &Apache::lonlocal::texthash (
                     firstres => "first resource in the $lctype",
                     whatsnew => "what's new? page",
                     userpref => 'your general user preferences',
                     coursespecific => "specific setting for this $lctype",
                   );
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};

    if (($command eq 'chgthreshold') 
	&& $checkallowed->{'abovethreshold'}) {
        &display_threshold_config($r,$refpage,$tabbg,\%threshold_titles,
                                                                   $cdom,$crs);
    } elsif (($command eq 'chginterval') 
	     && $checkallowed->{'versionchanges'}) {
        &display_interval_config($r,$refpage,\%interval_titles);
    } elsif (($command eq 'chgdisc') 
	     && $checkallowed->{'coursediscussion'}) {
        &display_discussion_config($r,$refpage);
    } elsif ($command eq 'courseinit') {
        &courseinit_config($r,$refpage,\%initpage);
    } else {
        &display_actions_box($r,$tabbg,$command,$refpage,\%threshold_titles,
                        \%interval_titles,\%initpage,$cdom,$crs,$checkallowed);
    }
    my $end_page = &Apache::loncommon::end_page();
    $r->print(<<END_OF_BLOCK);
  </td>
 </tr>
</table><br />
$end_page
END_OF_BLOCK
}

#-------------------------------
# display_header
#
# Display the header information and set
# up the HTML
#-------------------------------

sub display_header {
    my ($command,$checkallowed) = @_;
    
    my $scripttag;
    unless ($command eq 'chgthreshold' || $command eq 'chginterval') {
       $scripttag = <<"END";
<script type="text/javascript">
function change_display(caller,change) {
    caller.value = change;
    document.visible.submit(); 
}

function changeAll(change) {
END
        foreach my $item (keys(%{$checkallowed})) {
	    if ($item =~ /_section$/) { next; }
            if ($$checkallowed{$item}) {
                $scripttag.='document.visible.display_'.$item.'.value=change'.
                            "\n";
            }
        }
        $scripttag.='document.visible.submit();
}
</script>
';
    }
    my $course_type=&Apache::loncommon::course_type();
    return &Apache::loncommon::start_page($course_type.' Action Items',
					  $scripttag);
}

#-------------------------------
# display_actions_box
#
# Display the action items
#
#-------------------------------
                                                                                
sub display_actions_box {
    my ($r,$tabbg,$command,$refpage,$threshold_titles,$interval_titles,
                                      $initpage,$cdom,$crs,$checkallowed) = @_;
    my $rowColor1 = "#ffffff";
    my $rowColor2 = "#eeeeee";

    my $udom = $env{'user.domain'};
    my $uname = $env{'user.name'};
    my $cid = $env{'request.course.id'};
    my $crstype = &Apache::loncommon::course_type();
    my $lctype = lc($crstype);
    my %stulabel = (
                    'Course' => 'students',
                    'Group' => 'members',
                   );
    my %lt = &Apache::lonlocal::texthash(
                 'yacc' => 'You are accessing an invalid course',
                 'gtfr' => 'Go to first resource',
                 'hial' => 'Hide all',
                 'shal' => 'Show all',
    );

    my %unread = ();
    my %ungraded = ();
    my %bombed = ();
    my %triggered = ();
    my %changed = ();
    my @newmsgs = ();
    my @critmsgs = ();
    my @newdiscussions = ();
    my @tograde = ();
    my @bombs = ();
    my @warnings = ();
    my $msgcount = 0;
    my $critmsgcount = 0;

    my %res_title = ();
    my %show = ();
    my $needitems = 0;
    my $boxcount = 0;

    my $result;
    if ($command eq 'newcourseinit') {
        $result = &store_courseinit_setting($uname,$udom,$cid,$initpage);
    }

    my %threshold = ();
    my %pagedesc = &Apache::lonlocal::texthash (
                     firstres => 'First resource',
                     whatsnew => "What's New? page",
                     userpref => 'user preference',
                     coursespecific => $lctype.' only',
                     default => 'default',
                   );

    my ($initcontrol,$initdisp) = &curr_courseinit();
    my $currinit = $pagedesc{$initdisp}.' ('.$pagedesc{$initcontrol}.')';

    unless ($cid) {
        $r->print('<br /><b><center>'.$lt{'yacc'}.'</center></b><br /><br />');
        return;
    }

    if ($refpage eq 'start') {
        if (tie(my %bighash,'GDBM_File',$env{'request.course.fn'}.'.db',
            &GDBM_READER(),0640)) {
            my $furl=$bighash{'first_url'};
            untie(%bighash);
            $r->print('<font size="+1"><a href="'.$furl.'">'.$lt{'gtfr'}.
                  '</a></font><br />');
        }
    }
    $r->print(&mt('Page set to be displayed after you have selected a role in this '.$lctype).
              '. <nobr>'.&mt('Currently: <i>[_1]</i>',$currinit).'.&nbsp;&nbsp;'.
              &mt('<b>Change</b> for just <a href="/adm/whatsnew?command=courseinit&refpage=[_1]">this '.$lctype.'</a>',$refpage).' '.
              &mt('or for all <a href="/adm/preferences?action=changecourseinit&refpage=[_1]">your courses</a>',$refpage).'</nobr><br /><hr />');

    if ($command eq 'reset') {
        $result = &process_reset($cdom,$crs);
    } elsif ($command eq 'update') {
        $result = &process_update($uname,$udom,$threshold_titles);
    } elsif ($command eq 'newinterval') {
        $result = &store_interval_setting($uname,$udom,$cid,$interval_titles);
    } elsif ($command eq 'newdiscconf') {
        $result = &store_discussion_setting($uname,$udom,$cid);
    }

    my $store_result=&store_display_settings($uname,$udom,$cid,$checkallowed);

    unless ($store_result eq 'ok') { 
        &Apache::lonnet::logthis('Error storing whatsnew settings: '.
            $store_result.' for '.'user '.$uname.':'.$udom.' in '.$lctype.' '.$cid);
        $result .= &mt('Unable to store visibility settings due to [_1]',
                       $store_result); 
    }

    if ($result) {
        $r->print($result.'<hr width="100%" />');
    }
    $r->rflush();


    my %display_settings = &get_display_settings($uname,$udom,$cid);
    my $timediff = $display_settings{$cid.':interval'};
    unless (defined($timediff)) { $timediff = 604800; } 
    my $now = time;
    my $interval = $$interval_titles{$timediff};
    if ($timediff == -1) {
        $timediff = time;
    } 
    my $starttime = $now - $timediff;
    my $countunread = $display_settings{$cid.':countunread'};
    unless (defined($countunread)) {
        $countunread = 'on';
    }

    my %headings = &Apache::lonlocal::texthash(
                coursediscussion =>  'Unread '.$lctype.' discussion posts',
                handgrading =>  'Problems requiring handgrading',
                haserrors => 'Problems with errors',
                versionchanges => 'Resources in '.$lctype.' with version changes '.$interval,
                coursenormalmail => 'New '.$lctype.' messages',
                coursecritmail => 'New critical messages in '.$lctype,
    );

    if ($$checkallowed{'abovethreshold'}) {
        &get_curr_thresholds(\%threshold,$uname,$udom,$cid,$cdom,$crs);
    }

    $headings{'abovethreshold'} = 
	&mt('Problems with av. attempts &ge; [_1] or deg. difficulty &ge; [_2] <br /> and total number of '.$stulabel{$crstype}.' with submissions &ge; [_3]',
	    $threshold{'av_attempts'},$threshold{'degdiff'},
	    $threshold{'numstudents'});

    my @actionorder = ('handgrading','haserrors','abovethreshold','versionchanges','coursediscussion','coursenormalmail','coursecritmail');

    foreach my $key (keys(%{$checkallowed})) {
	if ($key =~ /_section$/) { next; }
        $show{$key} = 0;
        if ($$checkallowed{$key}) {
            unless ($display_settings{$cid.':'.$key} eq 'hide') {
                $show{$key} = 1;
            }
        }
    }

    foreach my $item (@actionorder) {
        unless ($item eq 'coursenormalmail' || $item eq 'coursecritmail') {
            if ($show{$item}) {
                $needitems = 1;
                last;
            }
        }
    }

    if ($needitems) {
        &getitems(\%unread,\%ungraded,\%bombed,\%triggered,\%changed,\@newdiscussions,\@tograde,\@bombs,\@warnings,$rowColor1,$rowColor2,\%threshold,$cdom,$crs,\%res_title,\%show,$starttime,$countunread);
    }
    if ($show{'coursenormalmail'}) {
        $msgcount = &getnormalmail(\@newmsgs);
    }
    if ($show{'coursecritmail'}) {
        $critmsgcount = &getcritmail(\@critmsgs);
    }

    $r->print(qq|<a href="javascript:changeAll('hide');">$lt{'hial'}</a>
     &nbsp;&nbsp;<a href="javascript:changeAll('show');">$lt{'shal'}</a>
     <form method="post" name="visible" action="/adm/whatsnew">\n|);
    foreach my $item (keys(%{$checkallowed})) {
	if ($item =~ /_section$/) { next; }
        if ($$checkallowed{$item}) {
            $r->print('<input type="hidden" name="display_'.$item.'" />'."\n");
        }
    }

    $r->print('<input type="hidden" name="refpage" value="'.$refpage.'"></form><br /><table border="0" width="100%" cellpadding="2" cellspacing="4"><tr><td align="left" valign="top" width="45%">');

    my $displayed = 0;
    my $totalboxes = 0;
    foreach my $key (keys(%{$checkallowed})) {
	if ($key =~ /_section$/) { next; }
	if ($key eq 'whatsnew' ) { next; } # whatsnew check creates no box
        if ($$checkallowed{$key}) {
            $totalboxes ++;
        }
    }
    my $halfway = int($totalboxes/2) + $totalboxes%2;
    foreach my $actionitem (@actionorder) {
        if ($$checkallowed{$actionitem}) {
            if ($displayed == $halfway) {
                $r->print('</td><td width="6%">&nbsp;</td><td align="left" valign="top" width="47%">');
            }
            &display_launcher($r,$actionitem,$refpage,$checkallowed,$tabbg,$rowColor1,$rowColor2,\%show,\%headings,\%res_title,\@tograde,\%ungraded,\@bombs,\%bombed,\%changed,\@warnings,\%triggered,\@newdiscussions,\%unread,$msgcount,\@newmsgs,$critmsgcount,\@critmsgs,$interval,$countunread);
            $displayed ++; 
        }
    }
    $r->print('
           </table>
          </td>
         </tr>
        </table>
      </td>
    </tr>
   </table>');
}

#-------------------------------
# display_threshold_config
#
# Display the threshold setting screen 
#
#-------------------------------
                                                                                
sub display_threshold_config {
    my ($r,$refpage,$tabbg,$threshold_titles,$cdom,$crs) = @_;
    my $uname = $env{'user.name'};
    my $udom = $env{'user.dom'};
    my $cid = $env{'request.course.id'};
    my %threshold = ();
    my $rowColor1 = "#ffffff";
    my $rowColor2 = "#eeeeee";
    my $rowColor;

    my @thresholditems = ("av_attempts","degdiff","numstudents");
    my %threshold_titles = &Apache::lonlocal::texthash(
                         av_attempts => 'Average number of attempts',
                         degdiff => 'Degree of difficulty',
                         numstudents => 'Total number of students with submissions',
                         );
    &get_curr_thresholds(\%threshold,$uname,$udom,$cid,$cdom,$crs);

    $r->print('<br /><form name="thresholdform" method="post" action="/adm/whatsnew">
        <table border="0" cellpadding="2" cellspacing="4">
         <tr>
          <td align="left" valign="top" width="45%">
           <table border="0" cellpadding="0" cellspacing="0" bgcolor="#000000">
            <tr>
             <td>
               <table border="0" cellpadding="1" cellspacing="1" bgcolor="#000000">
                <tr>
                <td bgcolor="#ffffff">
                 <table cellspacing="0" cellpadding="4" border="0">
     <tr bgcolor="'.$tabbg.'">
      <th>Threshold Name</th>
      <th>Current value</th>
      <th>Change?</th>
     </tr>');
    my $rowNum =0;
    foreach my $type (@thresholditems) {
        my $parameter = $env{'request.course.id'}.':threshold_'.$type;
# onchange is javascript to automatically check the 'Set' button.
        my $onchange = 'onFocus="javascript:window.document.forms'.
              "['thresholdform'].elements['".$parameter."_setparmval']".
              '.checked=true;"';
        if ($rowNum %2 == 1) {
            $rowColor = $rowColor1;
        } else {
            $rowColor = $rowColor2;
        }
        $r->print('
     <tr bgcolor="'.$rowColor.'">
      <td>'.$threshold_titles{$type}.'</td>
      <td>'.&Apache::lonhtmlcommon::textbox($parameter.'_value',
                                            $threshold{$type},
                                            10,$onchange).'</td>
      <td>'
           .&Apache::lonhtmlcommon::checkbox($parameter.'_setparmval').
      '</td>
     </tr>');
        $rowNum ++;
    }
    $r->print('</table></td></tr></table></td></tr></table>
           <br /><input type="submit" name="threshold" value="Make changes" />
                 <input type="hidden" name="command" value="update" />
                 <input type="hidden" name="refpage" value="'.$refpage.'" />
               </form>');
}

#-------------------------------
# display_interval_config
#
# Display the interval setting screen
#
#-------------------------------
                                                                                   
sub display_interval_config {
    my ($r,$refpage,$interval_titles) = @_;
    my $lctype = lc(&Apache::loncommon::course_type());
    my $current = &get_current($env{'user.name'},$env{'user.domain'},
                                $env{'request.course.id'},'interval');
    $r->print('<br />'.&mt('Choose the time window to use for display of version changes for resources in the '.$lctype.'.'));
    unless ($current eq '') {
        $r->print(' '.&mt('Current value is [_1]','<b>'.
                  $$interval_titles{$current}.'</b>.'));
    }
    $r->print('<br /><br />
<form method="post" name="intervalswitch" action="/adm/whatsnew">
<input type="hidden" name="command" value="newinterval" />
<input type="hidden" name="refpage" value="'.$refpage.'" />
<select name="interval">
');
    foreach my $key (reverse sort ({$a cmp $b} (keys(%{$interval_titles})))) {
        $r->print('<option value="'.$key.'">Version changes '.$$interval_titles{$key}.
                  '</option>'."\n");
    }
    $r->print('</select>&nbsp;&nbsp;
               <input type="submit" name="display" value="'.
               &mt('Change interval').'" /></form>');
    return;
}

#----------------------------------------------
# display_discussion_config
#
# Display the discussion display setting screen
#
#----------------------------------------------
                                                                                  
sub display_discussion_config {
    my ($r,$refpage) = @_;
    my $current = &get_current($env{'user.name'},$env{'user.domain'},
                                $env{'request.course.id'},'countunread');
    if ($current eq '') {
        $current = 'on';
    }
    my %opposite = ( 
                      'on' => 'off',
                      'off' => 'on',
                    );
    $r->print('<script type="text/javascript">
function toggle_countunread(choice) {
    if (choice == "unchanged") {
        document.discussionswitch.command.value = "";
    }
    document.discussionswitch.submit();
}
</script>');
    $r->print('<br />'.&mt('Choose whether or not to display a count of the number of new posts for each resource or bulletin board which has unread posts.').'<br />'.&mt('This can increase the time taken to gather data for the [_1] page by a few seconds.',"<i>What's New?</i>").'&nbsp;&nbsp;'.&mt('Currently set to [_1].','<b>'.$current.'</b>'));
    $r->print('<br /><br />
<form method="post" name="discussionswitch" action="/adm/whatsnew">
<input type="hidden" name="command" value="newdiscconf" />
<input type="hidden" name="refpage" value="'.$refpage.'" />
<input type="hidden" name="countunread" value="'.$opposite{$current}.'" />
');
    $r->print('<br/>
               <input type="button" name="display" value="'.
               &mt('Change to [_1]',$opposite{$current}).'" 
               onclick="javascript:toggle_countunread('."'change'".')" />
               &nbsp;&nbsp;&nbsp;&nbsp;&nbsp&nbsp;&nbsp;
               <input type="button" name="nochange" value="'.
               &mt("No change").'" 
               onclick="javascript:toggle_countunread('."'unchanged'".')" />
               </form>');
    return;
}

#---------------------------------------------------
# courseinit_config
#
# Set page displayed when course loads after 
# selecting a role in the course from the roles page. 
#
#---------------------------------------------------

sub courseinit_config {
    my ($r,$refpage,$initpage) = @_;
    my ($control,$current) = &curr_courseinit();
    my @chgstate = ('userpref','coursespecific');
    my @chgentry = ('firstres','whatsnew');
    my $lctype = lc(&Apache::loncommon::course_type());
    my %lt = &Apache::lonlocal::texthash(
                             'chwp' => "Choose which page will be displayed when you enter this $lctype after selecting a role.",
                             'cuva' => 'Current value is determined by',
                             'anis' => 'and is set to display',
                             'padc' => 'Page display controlled by',
                             'chce' => 'Choose '.$lctype.' entry',
                             'moce' => 'Modify '.$lctype.' entry',
    );
    $r->print(<<"END"); 
<br />$lt{'chwp'}
<br />$lt{'cuva'}: <b>
$$initpage{$control}</b> $lt{'anis'} <b>
$$initpage{$current}</b>.<br /><br />
<form method="post" name="courseinitswitch" action="/adm/whatsnew">
<input type="hidden" name="command" value="newcourseinit" />
<input type="hidden" name="refpage" value="$refpage" />
$lt{'padc'}:&nbsp;&nbsp;
END
    foreach my $choice (@chgstate) {
        my $chkstring;
        if ($choice eq $control) {
            $chkstring = ' checked="checked" ';
        }  
        $r->print('<nobr><label><input type="radio" name="courseinit_control" value="'.
                   $choice.'"'.$chkstring.'/>'.$$initpage{$choice}.
                   '&nbsp;&nbsp;</label></nobr>');
    }
    $r->print('<br /><br />'.&mt('If').' '.$$initpage{'coursespecific'}.
              ' - <br />'.$lt{'chce'}.": \n");
    foreach my $choice (@chgentry) {
        my $chkstring;
        if (($choice eq $current) && ($control eq 'coursespecific')) {
            $chkstring = ' checked="checked" ';
        }
        $r->print('<nobr><label><input type="radio" name="courseinit_page" value="'.
                  $choice.'"'.$chkstring.'/>'.$$initpage{$choice}.
                  '&nbsp;&nbsp;</label></nobr>');
    }
    $r->print('<br /><br /><input type="submit" name="display" value="'.
               $lt{'moce'}.'" /></form>');
    return;
}

sub curr_courseinit {
    my $current = &get_current($env{'user.name'},$env{'user.domain'},
                                $env{'request.course.id'},'courseinit');
    my $control;
    if ($current) {
        $control = 'coursespecific';
    } else {
        $control = 'userpref';
        my %userenv = &Apache::lonnet::get('environment',
                                                      ['course_init_display']);
        if (exists($userenv{'course_init_display'})) {
            $current = $userenv{'course_init_display'};
        }
        unless ($current) {
            $current = 'whatsnew';
        }
    }
    return ($control,$current);
}

sub display_launcher {
    my ($r,$action,$refpage,$checkallowed,$tabbg,$rowColor1,$rowColor2,$show,
        $headings,$res_title,$tograde,$ungraded,$bombs,$bombed,$changed,
        $warnings,$triggered,$newdiscussions,$unread,$msgcount,$newmsgs,
                          $critmsgcount,$critmsgs,$interval,$countunread) = @_;

    if ($$checkallowed{$action}) {
        &start_box($r,$tabbg,$show,$headings,$action,$refpage,$action);
        if ($$show{$action}) {
            if ($action eq 'handgrading') {    # UNGRADED ITEMS
                &display_handgrade($r,$tograde,$rowColor1,$rowColor2,
                                                                    $ungraded);
            } elsif ($action eq 'haserrors') { # BOMBS
                &display_haserrors($r,$bombs,$rowColor1,$rowColor2,$bombed,
                                                                   $res_title);
            } elsif ($action eq 'versionchanges') { # VERSION CHANGES
                &display_versionchanges($r,$changed,$res_title,$rowColor1,
                                                         $rowColor2,$interval);

            } elsif ($action eq 'abovethreshold') { # DEGDIFF/AV. TRIES TRIGGERS
                &display_abovethreshold($r,$refpage,$warnings,$triggered,
					$res_title);
            } elsif ($action eq 'coursediscussion') { # UNREAD COURSE DISCUSSION
                &display_coursediscussion($r,$newdiscussions,$unread,
                                $countunread,$res_title,$rowColor1,$rowColor2);
            } elsif ($action eq 'coursenormalmail') { # NORMAL MESSAGES
                &display_coursenormalmail($r,$msgcount,$newmsgs,$rowColor1,
                                                                   $rowColor2);
            } elsif ($action eq 'coursecritmail') { # CRITICAL MESSAGES
                &display_coursecritmail($r,$critmsgcount,$critmsgs,$rowColor1,
                                                                   $rowColor2);
            }
        }
        &end_box($r);
    }
    return;
}

sub getitems {
    my ($unread,$ungraded,$bombed,$triggered,$changed,$newdiscussions,
        $tograde,$bombs,$warnings,$rowColor1,$rowColor2,$threshold,$cdom,$crs,
                                 $res_title,$show,$starttime,$countunread) = @_;
    my $navmap = Apache::lonnavmaps::navmap->new();
    # force retrieve Resource to seed the part id cache we'll need it later
    my @allres=$navmap->retrieveResources(undef,
                     sub {if ($_[0]->is_problem) { $_[0]->parts();} return 1;});
    my %resourcetracker;
    my $discussiontime;

# Resource version changes
    if ($$show{'versionchanges'}) {
        &checkversions($cdom,$crs,$navmap,$changed,$starttime);
    }

    if ($$show{'abovethreshold'}) {
        %resourcetracker =  &Apache::lonnet::dump('nohist_resourcetracker',
                                                                   $cdom,$crs);
    }

    foreach my $resource (@allres) {
        my $result = '';
        my $applies = 0;
        my $symb = $resource->symb();
        %{$$bombed{$symb}} = ();
        %{$$ungraded{$symb}} = ();
        %{$$triggered{$symb}} = ();
        $$triggered{$symb}{numparts} = 0;
        if ($resource->encrypted()) {
            $$triggered{$symb}{'enclink'} = $resource->link();
            $$triggered{$symb}{'encsymb'} = $resource->shown_symb();
        }
        my $title = $resource->compTitle();
        $$res_title{$symb} = $title;
        my $ressymb = $resource->wrap_symb();

# Check if there are unread discussion postings
        if ($$show{'coursediscussion'}) {
            &check_discussions($resource,$symb,$ressymb,$title,
			       $newdiscussions,$unread,$countunread);
        }

# Check for ungraded problems
        if ($resource->is_problem()) {
            if ($$show{'handgrading'}) {
                &check_handgraded($resource,$symb,$title,$cdom,$crs,$ungraded,
                                                                     $tograde);
            }
        }

# Check for bombs
        if ($$show{'haserrors'}) {
            &check_bombed($resource,$symb,$title,$bombs,$bombed);
        }

# Maxtries and degree of difficulty for problem parts, unless handgradeable
        if ($$show{'abovethreshold'}) {  
            &check_thresholds($resource,$symb,\%resourcetracker,
			      $triggered,$threshold,$warnings);
        }

    }
}

sub check_discussions {
    my ($resource,$symb,$ressymb,$title,$newdiscussions,$unread,
	$countunread) = @_;

    if (!$resource->hasDiscussion()) { return; }

    %{$$unread{$ressymb}} = ();
    $$unread{$ressymb}{'title'} = $title;
    $$unread{$ressymb}{'symb'} = $symb;
    if ($resource->encrypted()) {
        $$unread{$ressymb}{'enclink'} = $resource->link();
        $$unread{$ressymb}{'encsymb'} = $resource->shown_symb();
    }
    push(@{$newdiscussions}, $ressymb);
    
    $$unread{$ressymb}{'lastpost'} = $resource->last_post_time();
    
    if ($countunread eq 'on') {
	$$unread{$ressymb}{'unreadcount'} = 
                            $resource->discussion_info('unread');
    }
}

sub check_handgraded {
    my ($resource,$symb,$title,$cdom,$cnum,$ungraded,$tograde) = @_;
    if ($resource->is_problem()) {
        my ($map,$ind,$url)=&Apache::lonnet::decode_symb($symb);
        my $partlist=$resource->parts();
        my $handgradeable;
        foreach my $part (@$partlist) {
            if ($resource->handgrade($part) eq 'yes') {
                $handgradeable=1; last;
            }
        }
        if ($handgradeable) {
            my @ungraded = &Apache::bridgetask::get_queue_symb_status(
                                             'gradingqueue',$symb,$cdom,$cnum);
            if (@ungraded > 0) {
                $$ungraded{$symb}{count} = scalar(@ungraded);
                $$ungraded{$symb}{title} = $title;
                if ($resource->encrypted()) {
                    $$ungraded{$symb}{'enclink'} = $resource->link();
                    $$ungraded{$symb}{'encsymb'} = $resource->shown_symb();
                }
                push(@{$tograde},$symb);
            }
        }
    }
}

sub check_bombed {
    my ($resource,$symb,$title,$bombs,$bombed) = @_;
    if ($resource->getErrors()) {
        my $errors = $resource->getErrors();
        $errors =~ s/^,//;
        my @bombs = split(/,/, $errors);
        my $errorcount = scalar(@bombs);
        my $errorlink = '<a href="/adm/email?display='.
                        &escape($bombs[0]).'">'.
                        $title.'</a>';
        $$bombed{$symb}{errorcount} = $errorcount;
        $$bombed{$symb}{errorlink} = $errorlink;
        push(@{$bombs}, $symb);
    }
}

sub check_thresholds {
    my ($resource,$symb,$resourcetracker,$triggered,$threshold,$warnings) = @_;
# Compile maxtries and degree of difficulty for problem parts, unless handgradeable
    my @parts = @{$resource->parts()};
    my %stats;
    my %lastreset = ();
    my $warning = 0;
    my $rowColor;
    foreach my $part (@parts) {
        if ($resource->handgrade($part) eq 'yes') {
            next;
        }
        if ($resource->is_survey($part)) {
            next;
        }
        %{$stats{$part}} = ();
        my ($attempts,$users,$corrects,$degdiff,$av_attempts);
        if (exists($$resourcetracker{$symb."\0".$part."\0attempts"})) {
            $attempts = $$resourcetracker{$symb."\0".$part."\0attempts"};
        }
        if (exists($$resourcetracker{$symb."\0".$part."\0users"})) {
            $users = $$resourcetracker{$symb."\0".$part."\0users"};
        }
        if (exists($$resourcetracker{$symb."\0".$part."\0correct"})) {
            $corrects = $$resourcetracker{$symb."\0".$part."\0correct"};
        }
        if ($attempts > 0) {
            $degdiff =  1 - ($corrects/$attempts);
            $degdiff = sprintf("%.2f",$degdiff);
        }
        if ($users > 0) {
            $av_attempts = $attempts/$users;
            $av_attempts = sprintf("%.2f",$av_attempts);
        }
        if ((($degdiff ne '' && $degdiff >= $$threshold{'degdiff'}) || ($av_attempts ne '' && $av_attempts >= $$threshold{'av_attempts'})) && ($users >= $$threshold{'numstudents'})) {
            $stats{$part}{degdiff} = $degdiff;
            $stats{$part}{attempts} = $av_attempts;
            $stats{$part}{users} = $users;
            $lastreset{$part} = $$resourcetracker{$symb."\0".$part."\0resettime"};
            if ($lastreset{$part}) {
                $lastreset{$part} = &Apache::lonnavmaps::timeToHumanString($lastreset{$part});
            }
            $warning = 1;
        }
    }
    if ($warning) {
        $$triggered{$symb}{title} = $resource->title;
        my $partcount = 0;
        @{$$triggered{$symb}{text}} = ();
        foreach my $part (@parts) {
            if (exists($stats{$part}{users})) {
                my $resetname = 'reset_'.&escape($symb."\0".$part);
                my $resettitle = 'title_'.&escape($symb."\0".$part);
                if (@parts > 1) {
                    $$triggered{$symb}{text}[$partcount] = '
                     <td>part - '.$part.'</td>';
                } else {
                    $$triggered{$symb}{text}[$partcount] = '
                     <td>single part</td>';
                }
                $$triggered{$symb}{text}[$partcount] .= '
                     <td>'.$stats{$part}{users}.'</td>
                     <td>'.$stats{$part}{attempts}.'</td>
                     <td>'.$stats{$part}{degdiff}.'</td>
                     <td>'.$lastreset{$part}.'</td>
                     <td><input type="checkbox" name="'.$resetname.'" /><input type="hidden" name="'.$resettitle.'" value="'.&escape($$triggered{$symb}{title}).'" /></td>';
                $partcount ++;
            }
            $$triggered{$symb}{numparts} = $partcount;
        }
        push(@{$warnings},$symb);
    }
}


sub get_curr_thresholds {
    my ($threshold,$uname,$udom,$cid,$cdom,$crs) = @_;
    # set default values
    %$threshold = (av_attempts => 2,
                  degdiff => 0.5,
                  numstudents => 2
                  );
    my %thresholdsettings = &Apache::lonnet::dump('nohist_whatsnew',$udom,
                                                     $uname,$cid.':threshold');
    my $thresholdcount = 0;
    my ($tmp) = %thresholdsettings;
    unless ($tmp =~ /^(con_lost|error|no_such_host)/i) {
        foreach my $item (keys %{$threshold}) { 
            if (exists($thresholdsettings{$cid.':threshold_'.$item})) {
                $$threshold{$item} = 
                             $thresholdsettings{$cid.':threshold_'.$item};
                $thresholdcount ++;
            }
        }
    }
    if ($thresholdcount == 3) {
        return;
    }
    my %coursesettings = &Apache::lonnet::dump('environment',
                                              $cdom,$crs,'internal.threshold');
    my ($temp) = %coursesettings;
    unless ($temp =~ /^(con_lost|error|no_such_host)/i) {  
        foreach my $item (keys %{$threshold}) {
            unless (exists($thresholdsettings{$cid.':threshold_'.$item})) {
                if (exists($coursesettings{'internal.threshold_'.$item})) {
                    $$threshold{$item} = 
                             $coursesettings{'internal.threshold_'.$item};
                }
            }
        }
    }
    return;
}

sub get_current {
    my ($uname,$udom,$cid,$caller) = @_;
    my $currvalue;
    my %settings = &Apache::lonnet::dump('nohist_whatsnew',$udom,$uname,$cid.
                                                                ':'.$caller);
    my ($tmp) = %settings;
    unless ($tmp =~ /^(con_lost|error|no_such_host)/i) {
        $currvalue = $settings{$cid.':'.$caller};
    }
    return $currvalue;
}

sub process_reset {
    my ($dom,$crs) = @_;
    my $result = '<b>'.&mt('Counters reset for following problems (and parts):').
                           '</b><br />';
    my @agg_types = ('attempts','users','correct');
    my %agg_titles = &Apache::lonlocal::texthash (
                     attempts => 'Number of submissions',
                     users => 'Students with submissions',
                     correct => 'Number of correct submissions',
                     );
    my @resets = ();
    my %titles = ();
    foreach my $key (keys(%env)) {
        next if ($key !~ /^form\.reset_(.+)$/);
        my $title = &unescape($env{'form.title_'.$1});
        my $reset_item = &unescape($1);
        my %curr_aggregates = &Apache::lonnet::dump('nohist_resourcetracker',$dom,$crs,$reset_item);
        my %aggregates = ();
        my ($symb,$part) = split(/\0/,$reset_item);
        foreach my $type (@agg_types) {
            $aggregates{$reset_item."\0".$type} = 0;
        }  
	$aggregates{$reset_item."\0".'resettime'} = time;
        my $putresult = &Apache::lonnet::put('nohist_resourcetracker',\%aggregates,
                          $dom,$crs);
        if ($putresult eq 'ok') {
            $result .= $title.' -part '.$part.': ';
            my %new_aggregates = &Apache::lonnet::dump('nohist_resourcetracker',$dom,$crs,$reset_item);
            foreach my $type (@agg_types) {
                $result .= $agg_titles{$type}.' = '.$new_aggregates{$reset_item."\0".$type}.'; ';
            }
            $result =~ s/; $//;
            $result .= '<br />';
        } else {
            $result = $title.' -part '.$part.': '.&mt('Unable to reset counters to zero due to [_1]',$putresult).'.<br />'."\n";
        }
    }
    return $result;
}

sub process_update {
    my ($uname,$udom,$threshold_titles) = @_;
    my $setoutput = '<b>Changes to threshold(s) for problem tracking:</b><br />';
    foreach (keys %env) {
        next if ($_!~/^form\.(.+)\_setparmval$/);
        my $name  = $1;
        my $value = $env{'form.'.$name.'_value'};
        if ($name && defined($value) && ($value ne '')) {
            my $put_result = &Apache::lonnet::put('nohist_whatsnew',
                                                  {$name=>$value},$udom,$uname);
           
            my ($shortname) = ($name =~ /^\Q$env{'request.course.id'}\E:threshold_(.+)$/);
            if ($put_result eq 'ok') {
                $setoutput.=&mt('Set threshold for [_1] to [_2]',
				'<b>'.$$threshold_titles{$shortname}.'</b>',
				'<b>'.$value.'</b>').'<br />';
	    } else {
                $setoutput.=&mt('Unable to set threshold for [_1] to [_2] due to [_3].',
				'<b>'.$name.'</b>','<b>'.$value.'</b>',
				'<tt>'.$put_result.'</tt>').'<br />';
            }
        }
    }
    return $setoutput;
}

sub getnormalmail {
    my ($newmsgs) = @_;
# Check for unread mail in course
    my $msgcount = 0;

    my @messages = sort(&Apache::lonnet::getkeys('nohist_email'));
    foreach my $message (@messages) {
	my $msgid=&escape($message);
        my ($sendtime,$shortsubj,$fromname,$fromdom,$status,$fromcid)=
            &Apache::lonmsg::unpackmsgid($msgid);
        if (($fromcid) && ($fromcid eq $env{'request.course.id'})) {
            if (defined($sendtime) && $sendtime!~/error/) {
                my $numsendtime = $sendtime;
                $sendtime = &Apache::lonlocal::locallocaltime($sendtime);
                if ($status eq 'new') {
                    $msgcount ++;
                    if ($shortsubj eq '') {
                        $shortsubj = &mt('No subject');
                    }
                    push(@{$newmsgs}, {
                        msgid    => $msgid,
                        sendtime => $sendtime,
                        shortsub => $shortsubj,
                        from     => $fromname,
                        fromdom  => $fromdom
                        });
                }
            }
        }
    }
    return $msgcount;
}

sub getcritmail {
    my ($critmsgs) = @_; 
# Check for critical messages in course
    my %what=&Apache::lonnet::dump('critical');
    my $result = '';
    my $critmsgcount = 0;
    foreach my $msgid (sort(keys(%what))) {
        my ($sendtime,$shortsubj,$fromname,$fromdom,$status,$fromcid)=
            &Apache::lonmsg::unpackmsgid($msgid);
        if (($fromcid) && ($fromcid eq  $env{'request.course.id'})) {
            if (defined($sendtime) && $sendtime!~/error/) {
                my $numsendtime = $sendtime;
                $sendtime = &Apache::lonlocal::locallocaltime($sendtime);
                $critmsgcount ++;
                if ($shortsubj eq '') {
                    $shortsubj = &mt('No subject');
                }
                push(@{$critmsgs}, {
                        msgid    => $msgid,
                        sendtime => $sendtime,
                        shortsub => $shortsubj,
                        from     => $fromname,
                        fromdom  => $fromdom
                        });
            }
        }
    }
    return $critmsgcount;
}


sub checkversions {
    my ($cdom,$crs,$navmap,$changed,$starttime) = @_;
    my %changes=&Apache::lonnet::dump('versionupdate',$cdom,$crs);
    my ($tmp) = keys(%changes);
    unless ($tmp =~ /^(con_lost|error|no_such_host)/i) {
        if (keys(%changes) > 0) {
            foreach my $key (sort(keys(%changes))) {
                if ($changes{$key} > $starttime) {
                    my $version;
                    my ($root,$extension)=($key=~/^(.*)\.(\w+)$/);
                    my $currentversion=&Apache::lonnet::getversion($key);
                    my $revdate = 
                          &Apache::lonnet::metadata($root.'.'.$extension,
                                                     'lastrevisiondate');
                    $revdate =  &Apache::lonlocal::locallocaltime($revdate);
                    my $linkurl=&Apache::lonnet::clutter($key);
                    my $usedversion=$navmap->usedVersion('version_'.$linkurl);
                    my @resources = $navmap->getResourceByUrl($linkurl,1);
                    if (($usedversion) && ($usedversion ne 'mostrecent')) {
                        $version = $usedversion;     
                    } else {
                        $version = $currentversion;
                    }
                    foreach my $res (@resources) {
                         if (ref($res) eq 'Apache::lonnavmaps::resource') { 
                            my $symb = $res->symb();
                            %{$$changed{$symb}} = (
                                                current => $currentversion,
                                                version => $version,
                                                revdate => $revdate,
                            );
                        }
                    }
                }
            }
        }
    }
    return;
}

sub display_handgrade {
    my ($r,$tograde,$rowColor1,$rowColor2,$ungraded) = @_;
    my $rowColor;
    my %lt = &Apache::lonlocal::texthash(
                        'prna' => 'Problem Name',
                        'nmun' => 'Number ungraded',
                        'nopr' => 'No problems require handgrading',
    );
    if (@{$tograde} > 0) {
        $r->print('<tr bgcolor="#cccccc"><td><b><small>'.$lt{'prna'}.'</small></b></td><td align="right"><b><small>'.$lt{'nmun'}.'</small></b></td></tr>');
        my $rowNum = 0;
        foreach my $res (@{$tograde}) {
            if ($rowNum %2 == 1) {
                $rowColor = $rowColor1;
            } else {
                $rowColor = $rowColor2;
            }
            my ($map,$id,$url)=&Apache::lonnet::decode_symb($res);
            my $linkurl=&Apache::lonnet::clutter($url);
            $linkurl .= '?symb='.&escape($res);
            if ($$ungraded{$res}{'enclink'}) {
                $linkurl = 
                    $$ungraded{$res}{'enclink'}.'?symb='.$$ungraded{$res}{'encsymb'};
            } 
            $r->print('<tr bgcolor="'.$rowColor.'"><td><a href="'.$linkurl.'"><small>'.$$ungraded{$res}{title}.'</small></a></td><td align="right"><small>'.$$ungraded{$res}{count}.'</small></td></tr>');
            $rowNum ++;
        }
    } else {
        $r->print('<tr><td bgcolor="#ffffff"><br><center><i><b><small>&nbsp;&nbsp;'.$lt{'nopr'}.'&nbsp;&nbsp;</small><br><br></b></i></td></tr>');
    }
}

sub display_haserrors {
    my ($r,$bombs,$rowColor1,$rowColor2,$bombed,$res_title) = @_;
    my $bombnum = 0;
    my $rowColor;
    my %lt = &Apache::lonlocal::texthash(
                                   reso => 'Resource',
                                   nmer => 'Number of errors',
                                   noer => 'No problems with errors',
    );
    if (@{$bombs} > 0) {
        $r->print('<tr bgcolor="#cccccc"><td><b><small>'.$lt{'reso'}.'</small></b></td><td align="right"><b><small>'.$lt{'nmer'}.'</small></b></td></tr>');
        @{$bombs} = sort { &cmp_title($a,$b,$res_title) } @{$bombs};
        foreach my $bomb (@{$bombs}) {
            if ($bombnum %2 == 1) {
                $rowColor = $rowColor1;
            } else {
                $rowColor = $rowColor2;
            }
            $r->print('<tr bgcolor="'.$rowColor.'"><td><small>'.$$bombed{$bomb}{errorlink}.'</small></td><td align="right"><small>'.$$bombed{$bomb}{errorcount}.'</small></td></tr>');
            $bombnum ++;
        }
    } else {
        $r->print('<tr><td bgcolor="#ffffff"><br /><center><b><i><small>'.$lt{'noer'}.'</small></i></b></center><br /></td></tr>');
    }
    return;
}

sub display_abovethreshold {
    my ($r,$refpage,$warnings,$triggered,$res_title) = @_;
    my %lt = &Apache::lonlocal::texthash(
                 reso => 'Resource',
                 part => 'Part',
                 nust => 'Num. students',
                 avat => 'Av. Attempts',
                 dedi => 'Deg. Diff',
                 lare => 'Last Reset',
                 reco => 'Reset Count?',
                 rese => 'Reset counters to 0',
                 nopr => 'No problems satisfy threshold criteria',
    );
    if (@{$warnings} > 0) {
        @{$warnings} = sort { &cmp_title($a,$b,$res_title) } @{$warnings};
        $r->print('<form name="reset_tracking" method="post" action="/adm/whatsnew">'.
                ' <input type="hidden" name="command" value="reset" />'."\n".
                ' <input type="hidden" name="refpage" value="'.$refpage.'" />'.
                "\n");
        $r->print('<tr class="LC_info_row">'.
		  '<td class="LC_first_item">'.$lt{'reso'}.'</td>'.
		  '<td>'.$lt{'part'}.'</td><td>'.$lt{'nust'}.'</td>'.
		  '<td>'.$lt{'avat'}.'</td><td>'.$lt{'dedi'}.'</td>'.
		  '<td>'.$lt{'lare'}.'</td><td>'.$lt{'reco'}.'</td></tr>');
	my $row;
        foreach my $res (@{$warnings}) {
	    $row++;
            my ($map,$id,$url)=&Apache::lonnet::decode_symb($res);
            my $linkurl=&Apache::lonnet::clutter($url);
            my $rowspan;
            if ($$triggered{$res}{numparts} > 1) {
                $rowspan = 'rowspan="'.$$triggered{$res}{numparts}.'"';
            }
            $linkurl .= '?symb='.&escape($res);
            if ($$triggered{$res}{'enclink'}) {
                $linkurl = 
                  $$triggered{$res}{'enclink'}.'?symb='.$$triggered{$res}{'encsymb'};
            }
            my $css_class = $row%2?'LC_odd_row':'';
            $r->print('<tr class="'.$css_class.'">'.
		      '<td class="LC_first_item" '.$rowspan.'><a href="'.$linkurl.'">'.
		      $$triggered{$res}{title}.'</a></td>');
                      if (ref($$triggered{$res}{text}) eq 'ARRAY') {
		          $r->print($$triggered{$res}{text}[0]);
                      }
                      $r->print('</tr>');
            if (ref($$triggered{$res}{text}) eq 'ARRAY') {
                if (@{$$triggered{$res}{text}} > 1) {
                    for (my $i=1; $i<@{$$triggered{$res}{text}}; $i++) {
                        $r->print('<tr class="'.$css_class.'">'.
                                  $$triggered{$res}{text}[$i].'</tr>');
                    }
                }
            }
        }
        $r->print('<tr class="LC_info_row"><td colspan="7"><br /><input type="submit" name="counters" value="'.$lt{'rese'}.'" /></td></tr></form>');
    } else {
        $r->print('<tr class="LC_empty_row"><td>'.$lt{'nopr'}.'</td></tr>');
    }
}

sub display_versionchanges {
    my ($r,$changed,$res_title,$rowColor1,$rowColor2,$interval) = @_;
    my %lt = &Apache::lonlocal::texthash(
        'reso' => 'Resource',
        'revd' => 'Last revised',
        'newv' => 'New version',
        'veru' => 'Version used',
        'noup' => 'No updated versions', 
    );
    my $rowColor;
    if (keys(%{$changed}) > 0) {
        $r->print('<tr bgcolor="#cccccc"><td><b><small>'.$lt{'reso'}.'</small></b></td><td><b><small>'.$lt{'revd'}.'</small></b></td><td><b><small>'.$lt{'newv'}.'</small></b></td><td><b><small>'.$lt{'veru'}.'</small></b></td></tr>');
        
        
        my @changes = sort { &cmp_title($a,$b,$res_title) } keys(%{$changed});
        my $changenum = 0;
        foreach my $item (@changes) {
            if ($changenum %2 == 1) {
                $rowColor = $rowColor1;
            } else {
                $rowColor = $rowColor2;
            }
            my ($map,$id,$url)=&Apache::lonnet::decode_symb($item);
            my $linkurl=&Apache::lonnet::clutter($url);
            $linkurl .= '?symb='.&escape($item);

            $r->print('<tr bgcolor="'.$rowColor.'"><td><small><a href="'.$linkurl.'">'.$$res_title{$item}.'</a></small></td><td><small>'.$$changed{$item}{'revdate'}.'</small></td><td><small>'.$$changed{$item}{'current'}.'</small></td><td><small>'.$$changed{$item}{'version'}.'</small></td></tr>');
            $changenum ++;
        }
    } else {
        $r->print('<tr><td bgcolor="#ffffff"><br /><center><b><i><small>'.$lt{'noup'}.' '.$interval.'</small></i></b></center><br /></td></tr>');
    }
    return;
}
 
sub display_coursediscussion {
    my ($r,$newdiscussions,$unread,$countunread,$res_title,$rowColor1,
                                                              $rowColor2) = @_;
    my $lctype = lc(&Apache::loncommon::course_type());
    my %lt = &Apache::lonlocal::texthash(
                'loca' => 'Location',
                'type' => 'Type',
                'numn' => 'Number of new posts',
                'noun' => 'No unread posts in '.$lctype.' discussions',
                'tmlp' => 'Time of last post', 
    );
    my $rowColor;
    if (@{$newdiscussions} > 0) {
        $r->print('<tr bgcolor="#cccccc"><td><b><small>'.$lt{'loca'}.
                  '</small></b></td><td><b><small>'.$lt{'type'}.
                  '</small></b>');
        if ($countunread eq 'on') {
            $r->print('<td><b><small>'.$lt{'tmlp'}.'</small></b></td>'.
                      '<td align="right"><b><small>'.$lt{'numn'}.
                      '</small></b></td>');
        } else {
            $r->print('<td align="right"><b><small>'.$lt{'tmlp'}.
                         '</small></b></td>');
        }
        $r->print("</tr>\n");
        @{$newdiscussions} = sort { &cmp_title($a,$b,$res_title) }
                                                            @{$newdiscussions};
        my $rowNum = 0;
        foreach my $ressymb (@{$newdiscussions}) {
            my $forum_title = $$unread{$ressymb}{'title'};
            my $type = 'Resource';
            my $feedurl=&Apache::lonfeedback::get_feedurl($ressymb);
            my $disclink = $feedurl.'?symb='.$$unread{$ressymb}{symb};
            if ($feedurl =~ /bulletinboard/) {
                $type = 'Bulletin Board';
            }
            if ($$unread{$ressymb}{'enclink'}) {
                $disclink = $$unread{$ressymb}{'enclink'}.'?symb='.$$unread{$ressymb}{'encsymb'};
            }
            if ($rowNum %2 == 1) {
                $rowColor = $rowColor1;
            } else {
                $rowColor = $rowColor2;
            }
            my $lastpost = &Apache::lonnavmaps::timeToHumanString(
                                               $$unread{$ressymb}{'lastpost'});
            $r->print('<tr bgcolor="'.$rowColor.'"><td><small><a href="'.$disclink.'">'.$forum_title.'</a>&nbsp;</td><td><small>'.$type.'&nbsp;</small></td>');
            if ($countunread eq 'on') {
                my $unreadnum = $$unread{$ressymb}{'unreadcount'};
                $r->print('<td><small>'.$lastpost.'<small></td><td align="right">'.
                          '<small>',$unreadnum.'&nbsp;</small></td>');
            } else {
                $r->print('<td align="right"><small>'.$lastpost.'</small></td>');
            }
            $r->print("</tr>\n");
            $rowNum ++;
        }
    } else {
        $r->print('<tr><td bgcolor="#ffffff"><br><center>&nbsp;<i><b><small>'.
                  $lt{'noun'}.'</small></b></i><br><br></td></tr>');
    }
}

sub display_coursenormalmail {
    my ($r,$msgcount,$newmsgs,$rowColor1,$rowColor2) = @_;
    my $rowColor;
    my $lctype = lc(&Apache::loncommon::course_type());
    if ($msgcount > 0) {
        $r->print('<tr bgcolor="#cccccc"><td><b><small>'.&mt('Number').'</small></b></td><td><b><small>'.&mt('Subject').'</small></b></td><td><b><small>'.&mt('Sender').'</small></b></td><td><b><small>'.&mt('Date/Time').'</small></b></td></tr>');
        my $rowNum = 0;
        my $mailcount = 1;
        foreach my $msg (@{$newmsgs}) {
            if ($rowNum %2 == 1) {
                $rowColor = $rowColor1;
            } else {
                $rowColor = $rowColor2;
            }
            $r->print('<tr bgcolor="'.$rowColor.'"><td valign="top"><small>'.$mailcount.'. &nbsp;</small></td><td valign="top"><small><a href="/adm/communicate">'.$msg->{'shortsub'}.'</a>&nbsp; &nbsp;</small></td><td valign="top"><small>&nbsp;'.$msg->{'from'}.'@'.$msg->{'fromdom'}.'&nbsp;</small></td><td valign="top"><small>'.$msg->{'sendtime'}.'</small></td></tr>');
            $rowNum ++;
            $mailcount ++;
        }
    } else {
        $r->print('<tr><td bgcolor="#ffffff" width="100%"><center><br /><b><i><small>'.&mt('No new '.$lctype.' messages').'</small></i></b><br /><br /></center></td></tr>');
    }
}

sub display_coursecritmail {
    my ($r,$critmsgcount,$critmsgs,$rowColor1,$rowColor2) = @_;
    my $rowColor;
    my $lctype = lc(&Apache::loncommon::course_type());
    if ($critmsgcount > 0) {
        $r->print('<tr bgcolor="#cccccc"><td><b><small>'.&mt('Number').'</small></b></td><td><b><small>'.&mt('Subject').'</small></b></td><td><b><small>'.&mt('Sender').'</small></b></td><td><b><small>'.&mt('Date/Time').'</small></b></td></tr>');
        my $rowNum = 0;
        my $mailcount = 1;
        foreach my $msg (@{$critmsgs}) {
            if ($rowNum %2 == 1) {
                $rowColor = $rowColor1;
            } else {
                $rowColor = $rowColor2;
            }
            $r->print('<tr bgcolor="'.$rowColor.'"><td valign="top"><small>'.$mailcount.'. &nbsp;<small></td><td valign="top"><small><a href="/adm/email?folder=critical">'.$msg->{'shortsub'}.'</a>&nbsp; &nbsp;</small></td><td valign="top"><small>&nbsp;'.$msg->{'from'}.'@'.$msg->{'fromdom'}.'&nbsp;</small></td><td valign="top"><small>'.$msg->{'sendtime'}.'</small></td></tr>');
            $rowNum ++;
            $mailcount ++;
        }
    } else {
        $r->print('<tr><td bgcolor="#ffffff" width="100%"><center><br /><b><i><small>'.&mt('No unread critical messages in '.$lctype).'</small></i></b><br /><br /></center></td></tr>');
    }
}

sub cmp_title {
    my ($a,$b,$res_title) = @_;
    my ($atitle,$btitle) = (lc($$res_title{$a}),lc($$res_title{$b}));
    $atitle=~s/^\s*//;
    $btitle=~s/^\s*//;
    return $atitle cmp $btitle;
}

sub get_display_settings {
    my ($uname,$udom,$cid) = @_;
    my %settings = &Apache::lonnet::dump('nohist_whatsnew',$udom,$uname,$cid); 
    my ($tmp) = keys(%settings);
    if ($tmp=~ /^(con_lost|error|no_such_host)/i) {
        %settings = ();
        unless ($tmp =~ /^error: 2 /) {
	    my $lctype = lc(&Apache::loncommon::course_type());
            &Apache::lonnet::logthis('Error retrieving whatsnew settings: '.
            $tmp.' for '.$uname.':'.$udom.' for '.$lctype.': '.$cid);
        }
    }
    return %settings;
}

sub store_display_settings {
    my ($uname,$udom,$cid,$checkallowed) = @_;
    my %whatsnew_settings;
    my $result;
    foreach my $key (keys(%{$checkallowed})) {
	if ($key =~ /_section$/) { next; }
        if (exists($env{'form.display_'.$key})) {
            unless ($env{'form.display_'.$key} eq '') {
                $whatsnew_settings{$cid.':'.$key} = $env{'form.display_'.$key};
            }
        }
    }
    if (keys(%whatsnew_settings)) {
        $result = &Apache::lonnet::put('nohist_whatsnew',\%whatsnew_settings,
                                                                 $udom,$uname);
    } else {
        $result = 'ok';
    }
    return $result;
}

sub store_interval_setting {
    my ($uname,$udom,$cid,$interval_titles) = @_;
    my %interval_settings = ();
    my $result;
    if (defined($env{'form.interval'})) {
        $interval_settings{$cid.':interval'} = $env{'form.interval'};
        my $outcome = &Apache::lonnet::put('nohist_whatsnew',
                                             \%interval_settings,$udom,$uname);
        if ($outcome eq 'ok') {
            $result = &mt('Interval set to version changes [_1]',
                  '<b>'.$$interval_titles{$env{'form.interval'}}.'</b><br />');

        } else {
	    my $lctype = lc(&Apache::loncommon::course_type());
            &Apache::lonnet::logthis('Error storing whatsnew interval setting'.
                ' '.$outcome.' for '.$uname.':'.$udom.' in '.$lctype.' '.$cid);
            $result = &mt('Unable to set interval to [_1] due to [_2].',
                         '<b>'.$$interval_titles{$env{'form.interval'}}.'</b>',
                         '<tt>'.$outcome.'</tt>.<br />');
        }
    }
    return $result;
}

sub store_discussion_setting {
    my ($uname,$udom,$cid) = @_;
    my %discussion_settings;
    my $result;
    if (defined($env{'form.countunread'})) {
        $discussion_settings{$cid.':countunread'} = $env{'form.countunread'};
        my $outcome = &Apache::lonnet::put('nohist_whatsnew',
                                             \%discussion_settings,$udom,$uname);
        if ($outcome eq 'ok') {
            $result = &mt('Count unread posts in discussions display set to [_1]',
                  '<b>'.$env{'form.countunread'}.'</b><br />');
                                                                                  
        } else {
	    my $lctype = lc(&Apache::loncommon::course_type());
            &Apache::lonnet::logthis('Error storing whatsnew countunread setting'.
                ' '.$outcome.' for '.$uname.':'.$udom.' in '.$lctype.' '.$cid);
            $result = &mt('Unable to set "number unread posts display" to [_1]'.
                          ' due to [_2].',
                         '<b>'.$env{'form.countunread'}.'</b>',
                         '<tt>'.$outcome.'</tt>.<br />');
        }
    }
    return $result;
}

sub store_courseinit_setting {
    my ($uname,$udom,$cid,$initpage) = @_;
    my %courseinit_settings;
    my $page_control;
    my $result;
    if (defined($env{'form.courseinit_control'})) {
        if ($env{'form.courseinit_control'} eq 'userpref') {
            $courseinit_settings{$cid.':courseinit'} = '';
            $page_control = 'global preferences';
        } else {
            if (defined($env{'form.courseinit_page'})) {
                $courseinit_settings{$cid.':courseinit'} = 
                                                  $env{'form.courseinit_page'};
                $page_control = 'course specific setting';
            }
        }
        if ($page_control) {
	    my $lctype = lc(&Apache::loncommon::course_type());
            my $outcome = &Apache::lonnet::put('nohist_whatsnew',
                                           \%courseinit_settings,$udom,$uname);
            if ($outcome eq 'ok') {
                if ($page_control eq 'global preferences') {
                    $result = &mt("Page displayed after role selection in $lctype now set by <b>user's global preferences</b>.");
                } else {
                    $result = &mt('Page displayed after role selection in this '.$lctype.' set to <b>[_2]</b>',$lctype,$$initpage{$env{'form.courseinit_page'}});
                }
            } else {
                &Apache::lonnet::logthis('Error storing whatsnew courseinit '.
                                         'setting: '.$outcome.' for '.$uname.
                                         ':'.$udom.' in '.$lctype.' '.$cid);
                if ($page_control eq 'global preferences') {
                    $result = &mt('Unable to set control of page display to [_1]'.
                          ' due to [_2].',
                         '<b>'.$page_control.'</b>',
                         '<tt>'.$outcome.'</tt>.<br />');
                } else {
                    $result = &mt('Unable to set page display, after role selection, for this '.$lctype.' to <b>[_2]</b> due to <tt>[_3]</tt>.<br />',
                         $$initpage{$env{'form.courseinit_page'}},$outcome);
                }
            }
        }
    }
    return $result;
}

sub start_box {
    my ($r,$tabbg,$show,$heading,$caller,$refpage) = @_;
    my %lt = &Apache::lonlocal::texthash( 
                       chth => 'Change thresholds?',
                       chin => 'Change interval?',
                       chop => 'Change options?',
    );
    my ($showhide,$class);
    if ($$show{$caller}) {
        $showhide = '<b><a href="javascript:change_display(document.visible.'.
                               'display_'.$caller.",'hide'".');">Hide</a></b>';
   
    } else {
        $showhide = '<b><a href="javascript:change_display(document.visible.'.
                               'display_'.$caller.",'show'".');">Show</a></b>';
    }
    
    $r->print('
         <table border="0" cellpadding="0" cellspacing="0" bgcolor="#000000" width="100%">
          <tr>
           <td>
            <table border="0" cellpadding="1" cellspacing="1" bgcolor="#000000" width="100%">
              <tr>
               <td bgcolor="'.$tabbg.'">
                <table width="100%" border="0" cellspacing="0" cellpadding="0">
                 <tr>
                  <td><b>'.$$heading{$caller}.'</b></td>
                  <td valign="top" align="right">'.$showhide.'</td>
                 </tr>
                </table>
               </td>
              </tr>');
     if (($caller eq 'abovethreshold') && ($$show{$caller})) {
         if ($$show{$caller}) {
             $r->print('
              <tr>
                <td bgcolor="'.$tabbg.'" align="right"><a href="/adm/whatsnew?command=chgthreshold&refpage='.$refpage.'"><b><small>'.$lt{'chth'}.'</small></b></a></td>
              </tr>');
         }
	 $class='class="LC_whatsnew"';
     } elsif (($caller eq 'versionchanges') && ($$show{$caller})) {
         if ($$show{$caller}) {
             $r->print('
              <tr>
                <td bgcolor="'.$tabbg.'" align="right"><a href="/adm/whatsnew?command=chginterval&refpage='.$refpage.'"><b><small>'.$lt{'chin'}.'</small></b></a></td>
              </tr>');
         }
     } elsif ($caller eq 'coursediscussion') {
         if ($$show{$caller}) {
             $r->print('
              <tr>
                <td bgcolor="'.$tabbg.'" align="right"><a href="/adm/whatsnew?command=chgdisc&refpage='.$refpage.'"><b><small>'.$lt{'chop'}.'</small></b></a></td>
              </tr>');
         }
     }
    $r->print('
              <tr>
               <td bgcolor="#ffffff">
                <table cellpadding="2" cellspacing="0" border="0" width="100%" '.$class.'>
');
    return;
}

sub end_box {
    my ($r) = shift;
    $r->print('
      </table>
     </td>
    </tr>
   </table>
  </td>
 </tr>
</table><br />');
    return;
}

1;
