# The LearningOnline Network with CAPA
# Configuration file reader
#
# $Id: Configuration.pm,v 1.13 2006/09/15 20:49:27 raeburn Exp $
#
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
# YEAR=2002
#
###

# POD documentation is at the end of this short module.

package LONCAPA::Configuration;

$VERSION = sprintf("%d.%02d", q$Revision: 1.13 $ =~ /(\d+)\.(\d+)/);

use strict;

my @confdirs=('/etc/httpd/conf/','/etc/apache2/');

# ------------------- Subroutine read_conf: read LON-CAPA server configuration.
# This subroutine reads PerlSetVar values out of specified web server
# configuration files.
sub read_conf {
    my (@conf_files)=@_;
    my (%perlvar,%configdirs);
    foreach my $filename (@conf_files,'loncapa_apache.conf') {
        my $configdir = '';
        $configdirs{$filename} = [@confdirs];
        while ($configdir eq '' && @{$configdirs{$filename}} > 0) {
            my $testdir = shift(@{$configdirs{$filename}});
            if (-e $testdir.$filename) {
                $configdir = $testdir;
            }
        }
        if ($configdir eq '') {
            die("Couldn't find a directory containing $filename");
        }
	open(CONFIG,'<'.$configdir.$filename) or
	    die("Can't read $configdir$filename");
	while (my $configline=<CONFIG>) {
	    if ($configline =~ /^[^\#]*PerlSetVar/) {
		my ($unused,$varname,$varvalue)=split(/\s+/,$configline);
		chomp($varvalue);
		$perlvar{$varname}=$varvalue;
	    }
	}
	close(CONFIG);
    }
    my $perlvarref=\%perlvar;
    return ($perlvarref);
}

#---------------------- Subroutine read_hosts: Read a LON-CAPA hosts.tab
# formatted configuration file.
#
my $RequiredCount = 5;		# Required item count in hosts.tab.
my $DefaultMaxCon = 5;		# Default value for maximum connections.
my $DefaultIdle   = 1000;       # Default connection idle time in seconds.
my $DefaultMinCon = 0;          # Default value for minimum connections.
sub read_hosts {
    my $Filename = shift;
    my %HostsTab;
    
    open(CONFIG,'<'.$Filename) or die("Can't read $Filename");
    while (my $line = <CONFIG>) {
	if (!($line =~ /^\s*\#/)) {
	    my @items = split(/:/, $line);
	    if(scalar @items >= $RequiredCount) {
		if (scalar @items == $RequiredCount) { # Only required items:
		    $items[$RequiredCount] = $DefaultMaxCon;
		}
		if(scalar @items == $RequiredCount + 1) { # up through maxcon.
		    $items[$RequiredCount+1] = $DefaultIdle;
		}
		if(scalar @items == $RequiredCount + 2) { # up through idle.
		    $items[$RequiredCount+2] = $DefaultMinCon;
		}
		{
		    my @list = @items; # probably not needed but I'm unsure of 
		    # about the scope of item so...
		    $HostsTab{$list[0]} = \@list; 
		}
	    }
	}
    }
    close(CONFIG);
    my $hostref = \%HostsTab;
    return ($hostref);
}

1;
__END__

=pod

=head1 NAME

B<LONCAPA::Configuration> - configuration file reader

=head1 SYNOPSIS

 use lib '/home/httpd/lib/perl/';
 use LONCAPA::Configuration;

 LONCAPA::Configuration::read_conf('loncapa.conf');
 LONCAPA::Configuration::read_hosts(filename);

=head1 DESCRIPTION

Many different parts of the LON-CAPA software need to reads in the
machine-specific configuration information.  These included scripts
controlling the TCP/IP layer (e.g. F<lonc> and F<lond>), testing scripts
(e.g. test_weblayer.pl and sqltest.pl), and utility scripts
(e.g. clusterstatus.pl and metadata_keywords.pl).

The following methods are available:

=over 4

=item $perlvarref = LONCAPA::Configuration::read_conf( @filename_list );

On a typical LON-CAPA server, the filename list argument will consist of
just one element, "loncapa.conf".

If there are multiple elements within the filename list, then these
filenames are processed consecutively.

A hash reference is returned and consists of those values which
have been initialized from the configuration filenames indicated
in the arguments.

If multiple file names define the same hash key, then priority is
given toward the B<last> file name processed.

=back

=over 4
=item $hostinfo = LONCAPA::Configuration::read_hosts(filename);

  The parameter is the name of a file in hosts.tab form.  The file is read and
parsed.  The return value is a reference to a hash.   The hash is indexed by
host and each element of the has is in turn a reference to an anonymous list
containing:

=over 4
=item host
   The loncapa hostname of the system. (This may be different than the 
   network hostname, see below).
=item domain
   The loncapa domain in which the host lives.
=item role
    The role of the system, currently allowed values are access for an
    access server and library for a library server.
=item dns
    The DNS hostname of the system. 
=item ip
    The IP address corresponding to the dns hostname of the system.
=item maxconn 
    The maximum number of connections this system should hold to the
    target system's lond.  If the file has no value, a default is supplied
    here by the function.
=item idle
    The number of seconds the oldest idle connection can be idle before it
    should be adaptively dropped.  If the file has no value, a default
    is supplied by the function.
=item mincon
    The minimum number of connections this system should hold to the
    target system's lond.  If the file has no value, a default is supplied by
    the funciton.

=back

=back



=head1 AUTHORS

This library is free software; you can redistribute it and/or
modify it under the same terms as LON-CAPA itself.

=cut
