# The LearningOnline Network with CAPA
# Handler to retrieve an old version of a file
#
# $Id: lonretrieve.pm,v 1.34 2006/12/06 22:22:39 albertel Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#
###

package Apache::lonretrieve;

use strict;
use Apache::File;
use File::Copy;
use Apache::Constants qw(:common :http :methods);
use Apache::loncacc;
use Apache::loncommon();
use Apache::lonlocal;
use Apache::lonnet;
use LONCAPA();

# ------------------------------------ Interface for selecting previous version
sub phaseone {
    my ($r,$fn,$uname,$udom)=@_;
    my $docroot=$r->dir_config('lonDocRoot');

    my $urldir='/res/'.$udom.'/'.$uname.$fn;
    $urldir=~s/\/[^\/]+$/\//;

    my $resfn=$docroot.'/res/'.$udom.'/'.$uname.$fn;
    my $resdir=$resfn;
    $resdir=~s/\/[^\/]+$/\//;

    my ($main,$suffix,$is_meta) = &get_file_info($fn);
    
    if (-e $resfn) {  
	$r->print('<form action="/adm/retrieve" method="POST">'.
		  '<input type="hidden" name="filename" value="/~'.$uname.$fn.'" />'.
		  '<input type="hidden" name="phase" value="two" />'.
		  &Apache::loncommon::start_data_table().
		  &Apache::loncommon::start_data_table_header_row().
		  '<th>'.&mt('Select').'</th>'.
		  '<th>'.&mt('Version').'</th>'.
		  '<th>'.&mt('Published on ...').'</th>');
	if (!$is_meta) {
	    $r->print('<th>'.&mt('Metadata').'</th>');
	}
	if ($is_meta
	    || &Apache::loncommon::fileembstyle($suffix) eq 'ssi') {
	    $r->print('<th>'.&mt('Diffs').'</th>');
	}
	$r->print(&Apache::loncommon::end_data_table_header_row());
	
	opendir(DIR,$resdir);
	my @files = grep(/^\Q$main\E\.(\d+)\.\Q$suffix\E$/,readdir(DIR));
	@files = sort {
	    my ($aver) = ($a=~/^\Q$main\E\.(\d+)\.\Q$suffix\E$/);
	    my ($bver) = ($b=~/^\Q$main\E\.(\d+)\.\Q$suffix\E$/);
	    return $aver <=> $bver;
	} (@files);
	closedir(DIR);
	
	foreach my $filename (@files) {
	    if ($filename=~/^\Q$main\E\.(\d+)\.\Q$suffix\E$/) {
		my $version=$1;
		my $rmtime=&Apache::lonnet::metadata($resdir.'/'.$filename,'lastrevisiondate');
		$r->print(&Apache::loncommon::start_data_table_row().
			  '<td><input type="radio" name="version" value="'.
			  $version.'" /></td><td>'.&mt('Previously published version').' '.$version.'</td><td>'.
			  localtime($rmtime).'</td>');
		
		if (!$is_meta) {
		    $r->print('<td><a href="'.$urldir.$filename.'.meta" target=cat>'.
			      &mt('Metadata Version').' '.$version.'</a></td>');
		}
		if ($is_meta
		    || &Apache::loncommon::fileembstyle($suffix) eq 'ssi') {
		    $r->print(
			      '<td><a target="cat" href="/adm/diff?filename=/~'.
			      $uname.$fn.
			      '&amp;versiontwo=priv&amp;versionone='.$version.
			      '">'.&mt('Diffs with Version').' '.$version.
			      '</a></td>');
		}
		$r->print(&Apache::loncommon::end_data_table_row());
	    }
	}
	closedir(DIR);
	my $rmtime=&Apache::lonnet::metadata($resfn,'lastrevisiondate');
	$r->print(&Apache::loncommon::start_data_table_row().
		  '<td><input type="radio" name="version" value="new" /></td>'.
		  '<td><b>'.&mt('Currently published version').'</b></td><td>'.localtime($rmtime).
		  '</td>');
	if (!$is_meta) {
	    $r->print('<td><a href="'.$urldir.$main.'.'.$suffix.'.meta" target=cat>'.
		      &mt('Metadata current version').'</a></td>');           
	}
	if ($is_meta 
	    || &Apache::loncommon::fileembstyle($suffix) eq 'ssi') {
	    $r->print(
		      '<td><a target="cat" href="/adm/diff?filename=/~'.
		      $uname.$fn.
		      '&amp;versiontwo=priv'.
		      '">'.&mt('Diffs with current Version').'</a></td>');
	}
	$r->print(&Apache::loncommon::end_data_table_row().
		  &Apache::loncommon::end_data_table().
		  '<p>'.'<span class="LC_warning">'.
		  &mt('Retrieval of an old version will overwrite the file currently in construction space').'</span></p>');
	if (!$is_meta) {
	    $r->print('<p>'.'<span class="LC_warning">'.
		      &mt('This will only retrieve the resource, if you want to retrieve the metadata you will need to do that separately.').
		      '</span></p>');
	}
	$r->print('<input type="submit" value="'.&mt('Retrieve version').'" /></form>');
    } else {
	$r->print('<h3>'.&mt('No previous versions published.').'</h3>');
    }
    $r->print('<p><a href="/priv/'.$uname.$fn.'">'.&mt('Back to').' '.$fn.
	      '</a></p>'); 
}

# ---------------------------------- Interface for presenting specified version
sub phasetwo {
    my ($r,$fn,$uname,$udom)=@_;
    if ($env{'form.version'}) {
        my $version=$env{'form.version'};
	if ($version eq 'new') {
	    $r->print('<h3>'.&mt('Retrieving current (most recent) version').'</h3>');
        } else {
            $r->print('<h3>'.&mt('Retrieving old version').' '.$version.'</h3>');
        }
	my ($main,$suffix,$is_meta) = &get_file_info($fn);

        my $logfile;
        my $ctarget='/home/'.$uname.'/public_html'.$fn;
        my $vfn=$fn;
        if ($version ne 'new') {
	    $vfn=~s/\.(\Q$suffix\E)$/\.$version\.$1/;
        }

        my $csource=$r->dir_config('lonDocRoot').'/res/'.$udom.'/'.$uname.$vfn;

	my $logname = $ctarget;
	if ($is_meta) { $logname =~ s/\.meta$//; }
	$logname = $ctarget.'.log';
        unless ($logfile=Apache::File->new('>>'.$logname)) {
	  $r->print(
         '<font color=red>'.&mt('No write permission to user directory, FAIL').'</font>');
        }
        print $logfile 
"\n\n================= Retrieve ".localtime()." ================\n".
"Version: $version\nSource: $csource\nTarget: $ctarget\n";
        $r->print('<p>'.&mt('Copying file').': ');
	if (copy($csource,$ctarget)) {
	    $r->print('ok<p>');
            print $logfile "Copied sucessfully.\n\n";
        } else {
            my $error=$!;
	    $r->print('fail, '.$error.'<p>');
            print $logfile "Copy failed: $error\n\n";
        }
        $r->print('<font size=+2><a href="/priv/'.$uname.$fn.
                  '">'.&mt('Back to').' '.$fn.'</a></font>'); 
    } else {
       $r->print('<span class="LC_warning">'.&mt('Please pick a version to retrieve').'</span><p>');
       &phaseone($r,$fn,$uname,$udom);
    }
}

sub get_file_info {
    my ($fn) = @_;
    my ($main,$suffix) = ($fn=~/\/([^\/]+)\.(\w+)$/);
    my $is_meta=0;
    if ($suffix eq 'meta') {
	$is_meta = 1;
	($main,$suffix) = ($main=~/(.+)\.(\w+)$/);	    
	$suffix .= '.meta';
    }
    return ($main,$suffix,$is_meta);
}

# ---------------------------------------------------------------- Main Handler
sub handler {

  my $r=shift;

  my $fn;


# Get query string for limited number of parameters

  &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
					  ['filename']);

  if ($env{'form.filename'}) {
      $fn=$env{'form.filename'};
      $fn=~s/^http\:\/\/[^\/]+//;
  } else {
     $r->log_reason($env{'user.name'}.' at '.$env{'user.domain'}.
         ' unspecified filename for retrieval', $r->filename); 
     return HTTP_NOT_FOUND;
  }

  unless ($fn) { 
     $r->log_reason($env{'user.name'}.' at '.$env{'user.domain'}.
         ' trying to retrieve non-existing file', $r->filename); 
     return HTTP_NOT_FOUND;
  } 

# ----------------------------------------------------------- Start page output
  my $uname;
  my $udom;

  ($uname,$udom)=
    &Apache::loncacc::constructaccess($fn,$r->dir_config('lonDefDomain'));
  unless (($uname) && ($udom)) {
     $r->log_reason($uname.' at '.$udom.
         ' trying to publish file '.$env{'form.filename'}.
         ' ('.$fn.') - not authorized', 
         $r->filename); 
     return HTTP_NOT_ACCEPTABLE;
  }

  $fn=~s{/~($LONCAPA::username_re)}{};

  &Apache::loncommon::content_type($r,'text/html');
  $r->send_http_header;

  $r->print(&Apache::loncommon::start_page('Retrieve Published Resources'));

  
  $r->print('<h1>'.&mt('Retrieve previous versions of').' <tt>'.$fn.'</tt></h1>');
  
  if (($uname ne $env{'user.name'}) || ($udom ne $env{'user.domain'})) {
          $r->print('<h3><span class="LC_diff_coauthor">'.&mt('Co-Author').': '.$uname.
		    &mt(' at ').$udom.
               '</span></h3>');
  }


  if ($env{'form.phase'} eq 'two') {
      &phasetwo($r,$fn,$uname,$udom);
  } else {
      &phaseone($r,$fn,$uname,$udom);
  }

  $r->print(&Apache::loncommon::end_page());
  return OK;  
}

1;
__END__

=head1 NAME

Apache::lonretrieve - retrieves an old version of a file

=head1 SYNOPSIS

Invoked by /etc/httpd/conf/srm.conf:

 <Location /adm/retrieve>
 PerlAccessHandler       Apache::lonacc
 SetHandler perl-script
 PerlHandler Apache::lonretrieve
 ErrorDocument     403 /adm/login
 ErrorDocument     404 /adm/notfound.html
 ErrorDocument     406 /adm/unauthorized.html
 ErrorDocument	  500 /adm/errorhandler
 </Location>

=head1 INTRODUCTION

This module retrieves an old published version of a file.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 HANDLER SUBROUTINE

This routine is called by Apache and mod_perl.

=over 4

=item *

Get query string for limited number of parameters

=item *

Start page output

=item *

print phase relevant output

=item *

(phase one is to select version; phase two retrieves version)

=back

=head1 OTHER SUBROUTINES

=over 4

=item *

phaseone() : Interface for selecting previous version.

=item *

phasetwo() : Interface for presenting specified version.

=back

=cut
