# The LearningOnline Network with CAPA
# The LON-CAPA Grading handler
#
# $Id: grades.pm,v 1.399.2.1 2007/06/13 17:20:12 albertel Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

package Apache::grades;
use strict;
use Apache::style;
use Apache::lonxml;
use Apache::lonnet;
use Apache::loncommon;
use Apache::lonhtmlcommon;
use Apache::lonnavmaps;
use Apache::lonhomework;
use Apache::loncoursedata;
use Apache::lonmsg();
use Apache::Constants qw(:common);
use Apache::lonlocal;
use Apache::lonenc;
use String::Similarity;
use lib '/home/httpd/lib/perl';
use LONCAPA;

use POSIX qw(floor);

my %oldessays=();
my %perm=();

# ----- These first few routines are general use routines.----
#
# --- Retrieve the parts from the metadata file.---
sub getpartlist {
    my ($symb) = @_;
    my (undef,undef,$url) = &Apache::lonnet::decode_symb($symb);
    my $partorder = &Apache::lonnet::metadata($url, 'partorder');
    my @parts;
    if ($partorder) {
	for my $part (split (/,/,$partorder)) {
	    if (!&Apache::loncommon::check_if_partid_hidden($part,$symb)) {
		push(@parts, $part);
	    }
	}	    
    } else {
	my $metadata = &Apache::lonnet::metadata($url, 'packages');
	foreach (split(/\,/,$metadata)) {
	    if ($_ =~ /^part_(.*)$/) {
		if (!&Apache::loncommon::check_if_partid_hidden($1,$symb)) {
		    push(@parts, $1);
		}
	    }
	}
    }
    my @stores;
    foreach my $part (@parts) {
	my (@metakeys) = split(/,/,&Apache::lonnet::metadata($url,'keys'));
	foreach my $key (@metakeys) {
	    if ($key =~ m/^stores_\Q$part\E_/) { push(@stores,$key); }
	}
    }
    return @stores;
}

# --- Get the symbolic name of a problem and the url
sub get_symb {
    my ($request,$silent) = @_;
    (my $url=$env{'form.url'}) =~ s-^http://($ENV{'SERVER_NAME'}|$ENV{'HTTP_HOST'})--;
    my $symb=($env{'form.symb'} ne '' ? $env{'form.symb'} : (&Apache::lonnet::symbread($url)));
    if ($symb eq '') { 
	if (!$silent) {
	    $request->print("Unable to handle ambiguous references:$url:.");
	    return ();
	}
    }
    return ($symb);
}

#--- Format fullname, username:domain if different for display
#--- Use anywhere where the student names are listed
sub nameUserString {
    my ($type,$fullname,$uname,$udom) = @_;
    if ($type eq 'header') {
	return '<b>&nbsp;Fullname&nbsp;</b><span class="LC_internal_info">(Username)</span>';
    } else {
	return '&nbsp;'.$fullname.'<span class="LC_internal_info">&nbsp;('.$uname.
	    ($env{'user.domain'} eq $udom ? '' : ' ('.$udom.')').')</span>';
    }
}

#--- Get the partlist and the response type for a given problem. ---
#--- Indicate if a response type is coded handgraded or not. ---
sub response_type {
    my ($symb) = shift;

    my $navmap = Apache::lonnavmaps::navmap->new();
    my $res = $navmap->getBySymb($symb);
    my $partlist = $res->parts();
    my %vPart = 
	map { $_ => 1 } (&Apache::loncommon::get_env_multiple('form.vPart'));
    my (%response_types,%handgrade);
    foreach my $part (@{ $partlist }) {
	next if (%vPart && !exists($vPart{$part}));

	my @types = $res->responseType($part);
	my @ids = $res->responseIds($part);
	for (my $i=0; $i < scalar(@ids); $i++) {
	    $response_types{$part}{$ids[$i]} = $types[$i];
	    $handgrade{$part.'_'.$ids[$i]} = 
		&Apache::lonnet::EXT('resource.'.$part.'_'.$ids[$i].
				     '.handgrade',$symb);
	}
    }
    return ($partlist,\%handgrade,\%response_types);
}

sub flatten_responseType {
    my ($responseType) = @_;
    my @part_response_id =
	map { 
	    my $part = $_;
	    map {
		[$part,$_]
		} sort(keys(%{ $responseType->{$part} }));
	} sort(keys(%$responseType));
    return @part_response_id;
}

sub get_display_part {
    my ($partID,$symb)=@_;
    my $display=&Apache::lonnet::EXT('resource.'.$partID.'.display',$symb);
    if (defined($display) and $display ne '') {
	$display.= " (<span class=\"LC_internal_info\">id $partID</span>)";
    } else {
	$display=$partID;
    }
    return $display;
}

#--- Show resource title
#--- and parts and response type
sub showResourceInfo {
    my ($symb,$probTitle,$checkboxes) = @_;
    my $col=3;
    if ($checkboxes) { $col=4; }
    my $result = '<h3>'.&mt('Current Resource').': '.$probTitle.'</h3>'."\n";
    $result .='<table border="0">';
    my ($partlist,$handgrade,$responseType) = &response_type($symb);
    my %resptype = ();
    my $hdgrade='no';
    my %partsseen;
    foreach my $partID (sort keys(%$responseType)) {
	foreach my $resID (sort keys(%{ $responseType->{$partID} })) {
	    my $handgrade=$$handgrade{$partID.'_'.$resID};
	    my $responsetype = $responseType->{$partID}->{$resID};
	    $hdgrade = $handgrade if ($handgrade eq 'yes');
	    $result.='<tr>';
	    if ($checkboxes) {
		if (exists($partsseen{$partID})) {
		    $result.="<td>&nbsp;</td>";
		} else {
		    $result.="<td><input type='checkbox' name='vPart' value='$partID' checked='on' /></td>";
		}
		$partsseen{$partID}=1;
	    }
	    my $display_part=&get_display_part($partID,$symb);
	    $result.='<td><b>Part: </b>'.$display_part.' <span class="LC_internal_info">'.
		$resID.'</span></td>'.
		'<td><b>Type: </b>'.$responsetype.'</td></tr>';
#	    '<td><b>Handgrade: </b>'.$handgrade.'</td></tr>';
	}
    }
    $result.='</table>'."\n";
    return $result,$responseType,$hdgrade,$partlist,$handgrade;
}


sub get_order {
    my ($partid,$respid,$symb,$uname,$udom)=@_;
    my (undef,undef,$url)=&Apache::lonnet::decode_symb($symb);
    $url=&Apache::lonnet::clutter($url);
    my $subresult=&Apache::lonnet::ssi($url,
				       ('grade_target' => 'analyze'),
				       ('grade_domain' => $udom),
				       ('grade_symb' => $symb),
				       ('grade_courseid' => 
					        $env{'request.course.id'}),
				       ('grade_username' => $uname));
    (undef,$subresult)=split(/_HASH_REF__/,$subresult,2);
    my %analyze=&Apache::lonnet::str2hash($subresult);
    return ($analyze{"$partid.$respid.shown"});
}
#--- Clean response type for display
#--- Currently filters option/rank/radiobutton/match/essay/Task
#        response types only.
sub cleanRecord {
    my ($answer,$response,$symb,$partid,$respid,$record,$order,$version,
	$uname,$udom) = @_;
    my $grayFont = '<span class="LC_internal_info">';
    if ($response =~ /^(option|rank)$/) {
	my %answer=&Apache::lonnet::str2hash($answer);
	my %grading=&Apache::lonnet::str2hash($record->{$version."resource.$partid.$respid.submissiongrading"});
	my ($toprow,$bottomrow);
	foreach my $foil (@$order) {
	    if ($grading{$foil} == 1) {
		$toprow.='<td><b>'.$answer{$foil}.'&nbsp;</b></td>';
	    } else {
		$toprow.='<td><i>'.$answer{$foil}.'&nbsp;</i></td>';
	    }
	    $bottomrow.='<td>'.$grayFont.$foil.'</span>&nbsp;</td>';
	}
	return '<blockquote><table border="1">'.
	    '<tr valign="top"><td>Answer</td>'.$toprow.'</tr>'.
	    '<tr valign="top"><td>'.$grayFont.'Option ID</span></td>'.
	    $grayFont.$bottomrow.'</tr>'.'</table></blockquote>';
    } elsif ($response eq 'match') {
	my %answer=&Apache::lonnet::str2hash($answer);
	my %grading=&Apache::lonnet::str2hash($record->{$version."resource.$partid.$respid.submissiongrading"});
	my @items=&Apache::lonnet::str2array($record->{$version."resource.$partid.$respid.submissionitems"});
	my ($toprow,$middlerow,$bottomrow);
	foreach my $foil (@$order) {
	    my $item=shift(@items);
	    if ($grading{$foil} == 1) {
		$toprow.='<td><b>'.$item.'&nbsp;</b></td>';
		$middlerow.='<td><b>'.$grayFont.$answer{$foil}.'&nbsp;</span></b></td>';
	    } else {
		$toprow.='<td><i>'.$item.'&nbsp;</i></td>';
		$middlerow.='<td><i>'.$grayFont.$answer{$foil}.'&nbsp;</span></i></td>';
	    }
	    $bottomrow.='<td>'.$grayFont.$foil.'</span>&nbsp;</td>';
	}
	return '<blockquote><table border="1">'.
	    '<tr valign="top"><td>Answer</td>'.$toprow.'</tr>'.
	    '<tr valign="top"><td>'.$grayFont.'Item ID</span></td>'.
	    $middlerow.'</tr>'.
	    '<tr valign="top"><td>'.$grayFont.'Option ID</span></td>'.
	    $bottomrow.'</tr>'.'</table></blockquote>';
    } elsif ($response eq 'radiobutton') {
	my %answer=&Apache::lonnet::str2hash($answer);
	my ($toprow,$bottomrow);
	my $correct=($order->[0])+1;
	for (my $i=1;$i<=$#$order;$i++) {
	    my $foil=$order->[$i];
	    if (exists($answer{$foil})) {
		if ($i == $correct) {
		    $toprow.='<td><b>true</b></td>';
		} else {
		    $toprow.='<td><i>true</i></td>';
		}
	    } else {
		$toprow.='<td>false</td>';
	    }
	    $bottomrow.='<td>'.$grayFont.$foil.'</span>&nbsp;</td>';
	}
	return '<blockquote><table border="1">'.
	    '<tr valign="top"><td>Answer</td>'.$toprow.'</tr>'.
	    '<tr valign="top"><td>'.$grayFont.'Option ID</span></td>'.
	    $grayFont.$bottomrow.'</tr>'.'</table></blockquote>';
    } elsif ($response eq 'essay') {
	if (! exists ($env{'form.'.$symb})) {
	    my (%keyhash) = &Apache::lonnet::dump('nohist_handgrade',
						  $env{'course.'.$env{'request.course.id'}.'.domain'},
						  $env{'course.'.$env{'request.course.id'}.'.num'});

	    my $loginuser = $env{'user.name'}.':'.$env{'user.domain'};
	    $env{'form.keywords'} = $keyhash{$symb.'_keywords'} ne '' ? $keyhash{$symb.'_keywords'} : '';
	    $env{'form.kwclr'}    = $keyhash{$loginuser.'_kwclr'} ne '' ? $keyhash{$loginuser.'_kwclr'} : 'red';
	    $env{'form.kwsize'}   = $keyhash{$loginuser.'_kwsize'} ne '' ? $keyhash{$loginuser.'_kwsize'} : '0';
	    $env{'form.kwstyle'}  = $keyhash{$loginuser.'_kwstyle'} ne '' ? $keyhash{$loginuser.'_kwstyle'} : '';
	    $env{'form.'.$symb} = 1; # so that we don't have to read it from disk for multiple sub of the same prob.
	}
	$answer =~ s-\n-<br />-g;
	return '<br /><br /><blockquote><tt>'.&keywords_highlight($answer).'</tt></blockquote>';
    } elsif ( $response eq 'organic') {
	my $result='Smile representation: "<tt>'.$answer.'</tt>"';
	my $jme=$record->{$version."resource.$partid.$respid.molecule"};
	$result.=&Apache::chemresponse::jme_img($jme,$answer,400);
	return $result;
    } elsif ( $response eq 'Task') {
	if ( $answer eq 'SUBMITTED') {
	    my $files = $record->{$version."resource.$respid.$partid.bridgetask.portfiles"};
	    my $result = &Apache::bridgetask::file_list($files,$uname,$udom);
	    return $result;
	} elsif ( grep(/^\Q$version\E.*?\.instance$/, keys(%{$record})) ) {
	    my @matches = grep(/^\Q$version\E.*?\.instance$/,
			       keys(%{$record}));
	    return join('<br />',($version,@matches));
			       
			       
	} else {
	    my $result =
		'<p>'
		.&mt('Overall result: [_1]',
		     $record->{$version."resource.$respid.$partid.status"})
		.'</p>';
	    
	    $result .= '<ul>';
	    my @grade = grep(/^\Q${version}resource.$respid.$partid.\E[^.]*[.]status$/,
			     keys(%{$record}));
	    foreach my $grade (sort(@grade)) {
		my ($dim) = ($grade =~/[.]([^.]+)[.]status$/);
		$result.= '<li>'.&mt("Dimension: [_1], status [_2] ",
				     $dim, $record->{$grade}).
			  '</li>';
	    }
	    $result.='</ul>';
	    return $result;
	}
       
    }
    return $answer;
}

#-- A couple of common js functions
sub commonJSfunctions {
    my $request = shift;
    $request->print(<<COMMONJSFUNCTIONS);
<script type="text/javascript" language="javascript">
    function radioSelection(radioButton) {
	var selection=null;
	if (radioButton.length > 1) {
	    for (var i=0; i<radioButton.length; i++) {
		if (radioButton[i].checked) {
		    return radioButton[i].value;
		}
	    }
	} else {
	    if (radioButton.checked) return radioButton.value;
	}
	return selection;
    }

    function pullDownSelection(selectOne) {
	var selection="";
	if (selectOne.length > 1) {
	    for (var i=0; i<selectOne.length; i++) {
		if (selectOne[i].selected) {
		    return selectOne[i].value;
		}
	    }
	} else {
            // only one value it must be the selected one
	    return selectOne.value;
	}
    }
</script>
COMMONJSFUNCTIONS
}

#--- Dumps the class list with usernames,list of sections,
#--- section, ids and fullnames for each user.
sub getclasslist {
    my ($getsec,$filterlist) = @_;
    my @getsec;
    if (!ref($getsec)) {
	if ($getsec ne '' && $getsec ne 'all') {
	    @getsec=($getsec);
	}
    } else {
	@getsec=@{$getsec};
    }
    if (grep(/^all$/,@getsec)) { undef(@getsec); }

    my $classlist=&Apache::loncoursedata::get_classlist();
    # Bail out if we were unable to get the classlist
    return if (! defined($classlist));
    #
    my %sections;
    my %fullnames;
    foreach my $student (keys(%$classlist)) {
        my $end      = 
            $classlist->{$student}->[&Apache::loncoursedata::CL_END()];
        my $start    = 
            $classlist->{$student}->[&Apache::loncoursedata::CL_START()];
        my $id       = 
            $classlist->{$student}->[&Apache::loncoursedata::CL_ID()];
        my $section  = 
            $classlist->{$student}->[&Apache::loncoursedata::CL_SECTION()];
        my $fullname = 
            $classlist->{$student}->[&Apache::loncoursedata::CL_FULLNAME()];
        my $status   = 
            $classlist->{$student}->[&Apache::loncoursedata::CL_STATUS()];
	# filter students according to status selected
	if ($filterlist && $env{'form.Status'} ne 'Any') {
	    if ($env{'form.Status'} ne $status) {
		delete ($classlist->{$student});
		next;
	    }
	}
	$section = ($section ne '' ? $section : 'none');
	if (&canview($section)) {
	    if (!@getsec || grep(/^\Q$section\E$/,@getsec)) {
		$sections{$section}++;
		$fullnames{$student}=$fullname;
	    } else {
		delete($classlist->{$student});
	    }
	} else {
	    delete($classlist->{$student});
	}
    }
    my %seen = ();
    my @sections = sort(keys(%sections));
    return ($classlist,\@sections,\%fullnames);
}

sub canmodify {
    my ($sec)=@_;
    if ($perm{'mgr'}) {
	if (!defined($perm{'mgr_section'})) {
	    # can modify whole class
	    return 1;
	} else {
	    if ($sec eq $perm{'mgr_section'}) {
		#can modify the requested section
		return 1;
	    } else {
		# can't modify the request section
		return 0;
	    }
	}
    }
    #can't modify
    return 0;
}

sub canview {
    my ($sec)=@_;
    if ($perm{'vgr'}) {
	if (!defined($perm{'vgr_section'})) {
	    # can modify whole class
	    return 1;
	} else {
	    if ($sec eq $perm{'vgr_section'}) {
		#can modify the requested section
		return 1;
	    } else {
		# can't modify the request section
		return 0;
	    }
	}
    }
    #can't modify
    return 0;
}

#--- Retrieve the grade status of a student for all the parts
sub student_gradeStatus {
    my ($symb,$udom,$uname,$partlist) = @_;
    my %record     = &Apache::lonnet::restore($symb,$env{'request.course.id'},$udom,$uname);
    my %partstatus = ();
    foreach (@$partlist) {
	my ($status,undef)   = split(/_/,$record{"resource.$_.solved"},2);
	$status              = 'nothing' if ($status eq '');
	$partstatus{$_}      = $status;
	my $subkey           = "resource.$_.submitted_by";
	$partstatus{$subkey} = $record{$subkey} if ($record{$subkey} ne '');
    }
    return %partstatus;
}

# hidden form and javascript that calls the form
# Use by verifyscript and viewgrades
# Shows a student's view of problem and submission
sub jscriptNform {
    my ($symb) = @_;
    my $jscript='<script type="text/javascript" language="javascript">'."\n".
	'    function viewOneStudent(user,domain) {'."\n".
	'	document.onestudent.student.value = user;'."\n".
	'	document.onestudent.userdom.value = domain;'."\n".
	'	document.onestudent.submit();'."\n".
	'    }'."\n".
	'</script>'."\n";
    $jscript.= '<form action="/adm/grades" method="post" name="onestudent">'."\n".
	'<input type="hidden" name="symb"    value="'.$symb.'" />'."\n".
	'<input type="hidden" name="saveState" value="'.$env{'form.saveState'}.'" />'."\n".
	'<input type="hidden" name="probTitle" value="'.$env{'form.probTitle'}.'" />'."\n".
	'<input type="hidden" name="Status"  value="'.$env{'form.Status'}.'" />'."\n".
	'<input type="hidden" name="command" value="submission" />'."\n".
	'<input type="hidden" name="student" value="" />'."\n".
	'<input type="hidden" name="userdom" value="" />'."\n".
	'</form>'."\n";
    return $jscript;
}

# Given the score (as a number [0-1] and the weight) what is the final
# point value? This function will round to the nearest tenth, third,
# or quarter if one of those is within the tolerance of .00001.
sub compute_points {
    my ($score, $weight) = @_;
    
    my $tolerance = .00001;
    my $points = $score * $weight;

    # Check for nearness to 1/x.
    my $check_for_nearness = sub {
        my ($factor) = @_;
        my $num = ($points * $factor) + $tolerance;
        my $floored_num = floor($num);
        if ($num - $floored_num < 2 * $tolerance * $factor) {
            return $floored_num / $factor;
        }
        return $points;
    };

    $points = $check_for_nearness->(10);
    $points = $check_for_nearness->(3);
    $points = $check_for_nearness->(4);
    
    return $points;
}

#------------------ End of general use routines --------------------

#
# Find most similar essay
#

sub most_similar {
    my ($uname,$udom,$uessay)=@_;

# ignore spaces and punctuation

    $uessay=~s/\W+/ /gs;

# ignore empty submissions (occuring when only files are sent)

    unless ($uessay=~/\w+/) { return ''; }

# these will be returned. Do not care if not at least 50 percent similar
    my $limit=0.6;
    my $sname='';
    my $sdom='';
    my $scrsid='';
    my $sessay='';
# go through all essays ...
    foreach my $tkey (keys %oldessays) {
	my ($tname,$tdom,$tcrsid)=split(/\./,$tkey);
# ... except the same student
        if (($tname ne $uname) || ($tdom ne $udom)) {
	    my $tessay=$oldessays{$tkey};
            $tessay=~s/\W+/ /gs;
# String similarity gives up if not even limit
            my $tsimilar=&String::Similarity::similarity($uessay,$tessay,$limit);
# Found one
            if ($tsimilar>$limit) {
		$limit=$tsimilar;
                $sname=$tname;
                $sdom=$tdom;
                $scrsid=$tcrsid;
                $sessay=$oldessays{$tkey};
            }
        } 
    }
    if ($limit>0.6) {
       return ($sname,$sdom,$scrsid,$sessay,$limit);
    } else {
       return ('','','','',0);
    }
}

#-------------------------------------------------------------------

#------------------------------------ Receipt Verification Routines
#
#--- Check whether a receipt number is valid.---
sub verifyreceipt {
    my $request  = shift;

    my $courseid = $env{'request.course.id'};
    my $receipt  = &Apache::lonnet::recprefix($courseid).'-'.
	$env{'form.receipt'};
    $receipt     =~ s/[^\-\d]//g;
    my ($symb)   = &get_symb($request);

    my $title.='<h3><span class="LC_info">Verifying Submission Receipt '.
	$receipt.'</h3></span>'."\n".
	'<h4><b>Resource: </b>'.$env{'form.probTitle'}.'</h4><br /><br />'."\n";

    my ($string,$contents,$matches) = ('','',0);
    my (undef,undef,$fullname) = &getclasslist('all','0');
    
    my $receiptparts=0;
    if ($env{"course.$courseid.receiptalg"} eq 'receipt2' ||
	$env{"course.$courseid.receiptalg"} eq 'receipt3') { $receiptparts=1; }
    my $parts=['0'];
    if ($receiptparts) { ($parts)=&response_type($symb); }
    foreach (sort 
	     {
		 if (lc($$fullname{$a}) ne lc($$fullname{$b})) {
		     return (lc($$fullname{$a}) cmp lc($$fullname{$b}));
		 }
		 return $a cmp $b;
	     } (keys(%$fullname))) {
	my ($uname,$udom)=split(/\:/);
	foreach my $part (@$parts) {
	    if ($receipt eq &Apache::lonnet::ireceipt($uname,$udom,$courseid,$symb,$part)) {
		$contents.='<tr bgcolor="#ffffe6"><td>&nbsp;'."\n".
		    '<a href="javascript:viewOneStudent(\''.$uname.'\',\''.$udom.
		    '\')"; TARGET=_self>'.$$fullname{$_}.'</a>&nbsp;</td>'."\n".
		    '<td>&nbsp;'.$uname.'&nbsp;</td>'.
		    '<td>&nbsp;'.$udom.'&nbsp;</td>';
		if ($receiptparts) {
		    $contents.='<td>&nbsp;'.$part.'&nbsp;</td>';
		}
		$contents.='</tr>'."\n";
		
		$matches++;
	    }
	}
    }
    if ($matches == 0) {
	$string = $title.'No match found for the above receipt.';
    } else {
	$string = &jscriptNform($symb).$title.
	    'The above receipt matches the following student'.
	    ($matches <= 1 ? '.' : 's.')."\n".
	    '<table border="0"><tr><td bgcolor="#777777">'."\n".
	    '<table border="0"><tr bgcolor="#e6ffff">'."\n".
	    '<td><b>&nbsp;Fullname&nbsp;</b></td>'."\n".
	    '<td><b>&nbsp;Username&nbsp;</b></td>'."\n".
	    '<td><b>&nbsp;Domain&nbsp;</b></td>';
	if ($receiptparts) {
	    $string.='<td>&nbsp;Problem Part&nbsp;</td>';
	}
	$string.='</tr>'."\n".$contents.
	    '</table></td></tr></table>'."\n";
    }
    return $string.&show_grading_menu_form($symb);
}

#--- This is called by a number of programs.
#--- Called from the Grading Menu - View/Grade an individual student
#--- Also called directly when one clicks on the subm button 
#    on the problem page.
sub listStudents {
    my ($request) = shift;

    my ($symb) = &get_symb($request);
    my $cdom      = $env{"course.$env{'request.course.id'}.domain"};
    my $cnum      = $env{"course.$env{'request.course.id'}.num"};
    my $getsec    = $env{'form.section'} eq '' ? 'all' : $env{'form.section'};
    my $submitonly= $env{'form.submitonly'} eq '' ? 'all' : $env{'form.submitonly'};

    my $viewgrade = $env{'form.showgrading'} eq 'yes' ? 'View/Grade/Regrade' : 'View';
    $env{'form.probTitle'} = $env{'form.probTitle'} eq '' ? 
	&Apache::lonnet::gettitle($symb) : $env{'form.probTitle'};

    my $result='<h3><span class="LC_info">&nbsp;'.$viewgrade.
	' Submissions for a Student or a Group of Students</span></h3>';

    my ($table,undef,$hdgrade,$partlist,$handgrade) = &showResourceInfo($symb,$env{'form.probTitle'},($env{'form.showgrading'} eq 'yes'));

    $request->print(<<LISTJAVASCRIPT);
<script type="text/javascript" language="javascript">
    function checkSelect(checkBox) {
	var ctr=0;
	var sense="";
	if (checkBox.length > 1) {
	    for (var i=0; i<checkBox.length; i++) {
		if (checkBox[i].checked) {
		    ctr++;
		}
	    }
	    sense = "a student or group of students";
	} else {
	    if (checkBox.checked) {
		ctr = 1;
	    }
	    sense = "the student";
	}
	if (ctr == 0) {
	    alert("Please select "+sense+" before clicking on the Next button.");
	    return false;
	}
	document.gradesub.submit();
    }

    function reLoadList(formname) {
	if (formname.saveStatusOld.value == pullDownSelection(formname.Status)) {return;}
	formname.command.value = 'submission';
	formname.submit();
    }
</script>
LISTJAVASCRIPT

    &commonJSfunctions($request);
    $request->print($result);

    my $checkhdgrade = ($env{'form.handgrade'} eq 'yes' && scalar(@$partlist) > 1 ) ? 'checked' : '';
    my $checklastsub = $checkhdgrade eq '' ? 'checked' : '';
    my $gradeTable='<form action="/adm/grades" method="post" name="gradesub">'.
	"\n".$table.
	'&nbsp;<b>View Problem Text: </b><label><input type="radio" name="vProb" value="no" checked="on" /> no </label>'."\n".
	'<label><input type="radio" name="vProb" value="yes" /> one student </label>'."\n".
	'<label><input type="radio" name="vProb" value="all" /> all students </label><br />'."\n".
	'&nbsp;<b>View Answer: </b><label><input type="radio" name="vAns" value="no"  /> no </label>'."\n".
	'<label><input type="radio" name="vAns" value="yes" /> one student </label>'."\n".
	'<label><input type="radio" name="vAns" value="all" checked="on" /> all students </label><br />'."\n".
	'&nbsp;<b>Submissions: </b>'."\n";
    if ($env{'form.handgrade'} eq 'yes' && scalar(@$partlist) > 1) {
	$gradeTable.='<label><input type="radio" name="lastSub" value="hdgrade" '.$checkhdgrade.' /> essay part only </label>'."\n";
    }

    my $saveStatus = $env{'form.Status'} eq '' ? 'Active' : $env{'form.Status'};
    $env{'form.Status'} = $saveStatus;

    $gradeTable.='<label><input type="radio" name="lastSub" value="lastonly" '.$checklastsub.' /> last submission only </label>'."\n".
	'<label><input type="radio" name="lastSub" value="last" /> last submission & parts info </label>'."\n".
	'<label><input type="radio" name="lastSub" value="datesub" /> by dates and submissions </label>'."\n".
	'<label><input type="radio" name="lastSub" value="all" /> all details</label><br />'."\n".
        '&nbsp;<b>Grading Increments:</b> <select name="increment">'.
        '<option value="1">Whole Points</option>'.
        '<option value=".5">Half Points</option>'.
        '<option value=".25">Quarter Points</option>'.
        '<option value=".1">Tenths of a Point</option>'.
        '</select>'.

	'<input type="hidden" name="section"     value="'.$getsec.'" />'."\n".
	'<input type="hidden" name="submitonly"  value="'.$submitonly.'" />'."\n".
	'<input type="hidden" name="handgrade"   value="'.$env{'form.handgrade'}.'" /><br />'."\n".
	'<input type="hidden" name="showgrading" value="'.$env{'form.showgrading'}.'" /><br />'."\n".
	'<input type="hidden" name="saveState"   value="'.$env{'form.saveState'}.'" />'."\n".
	'<input type="hidden" name="probTitle"   value="'.$env{'form.probTitle'}.'" />'."\n".
	'<input type="hidden" name="symb" value="'.$symb.'" />'."\n".
	'<input type="hidden" name="saveStatusOld" value="'.$saveStatus.'" />'."\n";

    if (exists($env{'form.gradingMenu'}) && exists($env{'form.Status'})) {
	$gradeTable.='<input type="hidden" name="Status"   value="'.$env{'form.Status'}.'" />'."\n";
    } else {
	$gradeTable.='<b>Student Status:</b> '.
	    &Apache::lonhtmlcommon::StatusOptions($saveStatus,undef,1,'javascript:reLoadList(this.form);').'<br />';
    }

    $gradeTable.='To '.lc($viewgrade).' a submission or a group of submissions, click on the check box(es) '.
	'next to the student\'s name(s). Then click on the Next button.<br />'."\n".
	'<input type="hidden" name="command" value="processGroup" />'."\n";

# checkall buttons
    $gradeTable.=&check_script('gradesub', 'stuinfo');
    $gradeTable.='<input type="button" '."\n".
	'onClick="javascript:checkSelect(this.form.stuinfo);" '."\n".
	'value="Next->" /> <br />'."\n";
    $gradeTable.=&check_buttons();
    $gradeTable.='<label><input type="checkbox" name="checkPlag" checked="on" />Check For Plagiarism</label>';
    my ($classlist, undef, $fullname) = &getclasslist($getsec,'1');
    $gradeTable.='<table border="0"><tr><td bgcolor="#777777">'.
	'<table border="0"><tr bgcolor="#e6ffff">';
    my $loop = 0;
    while ($loop < 2) {
	$gradeTable.='<td><b>&nbsp;No.</b>&nbsp;</td><td><b>&nbsp;Select&nbsp;</b></td>'.
	    '<td>'.&nameUserString('header').'&nbsp;Section/Group</td>';
	if ($env{'form.showgrading'} eq 'yes' 
	    && $submitonly ne 'queued'
	    && $submitonly ne 'all') {
	    foreach (sort(@$partlist)) {
		my $display_part=&get_display_part((split(/_/))[0],$symb);
		$gradeTable.='<td><b>&nbsp;Part: '.$display_part.
		    ' Status&nbsp;</b></td>';
	    }
	} elsif ($submitonly eq 'queued') {
	    $gradeTable.='<td><b>&nbsp;'.&mt('Queue Status').'&nbsp;</b></td>';
	}
	$loop++;
#	$gradeTable.='<td></td>' if ($loop%2 ==1);
    }
    $gradeTable.='</tr>'."\n";

    my $ctr = 0;
    foreach my $student (sort 
			 {
			     if (lc($$fullname{$a}) ne lc($$fullname{$b})) {
				 return (lc($$fullname{$a}) cmp lc($$fullname{$b}));
			     }
			     return $a cmp $b;
			 }
			 (keys(%$fullname))) {
	my ($uname,$udom) = split(/:/,$student);

	my %status = ();

	if ($submitonly eq 'queued') {
	    my %queue_status = 
		&Apache::bridgetask::get_student_status($symb,$cdom,$cnum,
							$udom,$uname);
	    next if (!defined($queue_status{'gradingqueue'}));
	    $status{'gradingqueue'} = $queue_status{'gradingqueue'};
	}

	if ($env{'form.showgrading'} eq 'yes' 
	    && $submitonly ne 'queued'
	    && $submitonly ne 'all') {
	    (%status) =&student_gradeStatus($symb,$udom,$uname,$partlist);
	    my $submitted = 0;
	    my $graded = 0;
	    my $incorrect = 0;
	    foreach (keys(%status)) {
		$submitted = 1 if ($status{$_} ne 'nothing');
		$graded = 1 if ($status{$_} =~ /^ungraded/);
		$incorrect = 1 if ($status{$_} =~ /^incorrect/);
		
		my ($foo,$partid,$foo1) = split(/\./,$_);
		if ($status{'resource.'.$partid.'.submitted_by'} ne '') {
		    $submitted = 0;
		    my ($part)=split(/\./,$partid);
		    $gradeTable.='<input type="hidden" name="'.
			$student.':'.$part.':submitted_by" value="'.
			$status{'resource.'.$partid.'.submitted_by'}.'" />';
		}
	    }
	    
	    next if (!$submitted && ($submitonly eq 'yes' ||
				     $submitonly eq 'incorrect' ||
				     $submitonly eq 'graded'));
	    next if (!$graded && ($submitonly eq 'graded'));
	    next if (!$incorrect && $submitonly eq 'incorrect');
	}

	$ctr++;
	my $section = $classlist->{$student}->[&Apache::loncoursedata::CL_SECTION()];

	if ( $perm{'vgr'} eq 'F' ) {
	    $gradeTable.='<tr bgcolor="#ffffe6">' if ($ctr%2 ==1);
	    $gradeTable.='<td align="right">'.$ctr.'&nbsp;</td>'.
               '<td align="center"><label><input type=checkbox name="stuinfo" value="'.
               $student.':'.$$fullname{$student}.':::SECTION'.$section.
	       ')&nbsp;" />&nbsp;&nbsp;</label></td>'."\n".'<td>'.
	       &nameUserString(undef,$$fullname{$student},$uname,$udom).
	       '&nbsp;'.$section.'</td>'."\n";

	    if ($env{'form.showgrading'} eq 'yes' && $submitonly ne 'all') {
		foreach (sort keys(%status)) {
		    next if (/^resource.*?submitted_by$/);
		    $gradeTable.='<td align="center">&nbsp;'.$status{$_}.'&nbsp;</td>'."\n";
		}
	    }
#	    $gradeTable.='<td></td>' if ($ctr%2 ==1);
	    $gradeTable.='</tr>'."\n" if ($ctr%2 ==0);
	}
    }
    if ($ctr%2 ==1) {
	$gradeTable.='<td>&nbsp;</td><td>&nbsp;</td><td>&nbsp;</td>';
	    if ($env{'form.showgrading'} eq 'yes' 
		&& $submitonly ne 'queued'
		&& $submitonly ne 'all') {
		foreach (@$partlist) {
		    $gradeTable.='<td>&nbsp;</td>';
		}
	    } elsif ($submitonly eq 'queued') {
		$gradeTable.='<td>&nbsp;</td>';
	    }
	$gradeTable.='</tr>';
    }

    $gradeTable.='</table></td></tr></table>'."\n".
	'<input type="button" '.
	'onClick="javascript:checkSelect(this.form.stuinfo);" '.
	'value="Next->" /></form>'."\n";
    if ($ctr == 0) {
	my $num_students=(scalar(keys(%$fullname)));
	if ($num_students eq 0) {
	    $gradeTable='<br />&nbsp;<span class="LC_warning">There are no students currently enrolled.</span>';
	} else {
	    my $submissions='submissions';
	    if ($submitonly eq 'incorrect') { $submissions = 'incorrect submissions'; }
	    if ($submitonly eq 'graded'   ) { $submissions = 'ungraded submissions'; }
	    if ($submitonly eq 'queued'   ) { $submissions = 'queued submissions'; }
	    $gradeTable='<br />&nbsp;<span class="LC_warning">'.
		'No '.$submissions.' found for this resource for any students. ('.$num_students.
		' students checked for '.$submissions.')</span><br />';
	}
    } elsif ($ctr == 1) {
	$gradeTable =~ s/type=checkbox/type=checkbox checked/;
    }
    $gradeTable.=&show_grading_menu_form($symb);
    $request->print($gradeTable);
    return '';
}

#---- Called from the listStudents routine

sub check_script {
    my ($form, $type)=@_;
    my $chkallscript='<script type="text/javascript">
    function checkall() {
        for (i=0; i<document.forms.'.$form.'.elements.length; i++) {
            ele = document.forms.'.$form.'.elements[i];
            if (ele.name == "'.$type.'") {
            document.forms.'.$form.'.elements[i].checked=true;
                                       }
        }
    }

    function checksec() {
        for (i=0; i<document.forms.'.$form.'.elements.length; i++) {
            ele = document.forms.'.$form.'.elements[i];
           string = document.forms.'.$form.'.chksec.value;
           if
          (ele.value.indexOf(":::SECTION"+string)>0) {
              document.forms.'.$form.'.elements[i].checked=true;
            }
        }
    }


    function uncheckall() {
        for (i=0; i<document.forms.'.$form.'.elements.length; i++) {
            ele = document.forms.'.$form.'.elements[i];
            if (ele.name == "'.$type.'") {
            document.forms.'.$form.'.elements[i].checked=false;
                                       }
        }
    }

</script>'."\n";
    return $chkallscript;
}

sub check_buttons {
    my $buttons.='<input type="button" onclick="checkall()" value="Check All" />';
    $buttons.='<input type="button" onclick="uncheckall()" value="Uncheck All" />&nbsp;';
    $buttons.='<input type="button" onclick="checksec()" value="Check Section/Group" />';
    $buttons.='<input type="text" size="5" name="chksec" />&nbsp;';
    return $buttons;
}

#     Displays the submissions for one student or a group of students
sub processGroup {
    my ($request)  = shift;
    my $ctr        = 0;
    my @stuchecked = &Apache::loncommon::get_env_multiple('form.stuinfo');
    my $total      = scalar(@stuchecked)-1;

    foreach my $student (@stuchecked) {
	my ($uname,$udom,$fullname) = split(/:/,$student);
	$env{'form.student'}        = $uname;
	$env{'form.userdom'}        = $udom;
	$env{'form.fullname'}       = $fullname;
	&submission($request,$ctr,$total);
	$ctr++;
    }
    return '';
}

#------------------------------------------------------------------------------------
#
#-------------------------- Next few routines handles grading by student, essentially
#                           handles essay response type problem/part
#
#--- Javascript to handle the submission page functionality ---
sub sub_page_js {
    my $request = shift;
    $request->print(<<SUBJAVASCRIPT);
<script type="text/javascript" language="javascript">
    function updateRadio(formname,id,weight) {
	var gradeBox = formname["GD_BOX"+id];
	var radioButton = formname["RADVAL"+id];
	var oldpts = formname["oldpts"+id].value;
	var pts = checkSolved(formname,id) == 'update' ? gradeBox.value : oldpts;
	gradeBox.value = pts;
	var resetbox = false;
	if (isNaN(pts) || pts < 0) {
	    alert("A number equal or greater than 0 is expected. Entered value = "+pts);
	    for (var i=0; i<radioButton.length; i++) {
		if (radioButton[i].checked) {
		    gradeBox.value = i;
		    resetbox = true;
		}
	    }
	    if (!resetbox) {
		formtextbox.value = "";
	    }
	    return;
	}

	if (pts > weight) {
	    var resp = confirm("You entered a value ("+pts+
			       ") greater than the weight for the part. Accept?");
	    if (resp == false) {
		gradeBox.value = oldpts;
		return;
	    }
	}

	for (var i=0; i<radioButton.length; i++) {
	    radioButton[i].checked=false;
	    if (pts == i && pts != "") {
		radioButton[i].checked=true;
	    }
	}
	updateSelect(formname,id);
	formname["stores"+id].value = "0";
    }

    function writeBox(formname,id,pts) {
	var gradeBox = formname["GD_BOX"+id];
	if (checkSolved(formname,id) == 'update') {
	    gradeBox.value = pts;
	} else {
	    var oldpts = formname["oldpts"+id].value;
	    gradeBox.value = oldpts;
	    var radioButton = formname["RADVAL"+id];
	    for (var i=0; i<radioButton.length; i++) {
		radioButton[i].checked=false;
		if (i == oldpts) {
		    radioButton[i].checked=true;
		}
	    }
	}
	formname["stores"+id].value = "0";
	updateSelect(formname,id);
	return;
    }

    function clearRadBox(formname,id) {
	if (checkSolved(formname,id) == 'noupdate') {
	    updateSelect(formname,id);
	    return;
	}
	gradeSelect = formname["GD_SEL"+id];
	for (var i=0; i<gradeSelect.length; i++) {
	    if (gradeSelect[i].selected) {
		var selectx=i;
	    }
	}
	var stores = formname["stores"+id];
	if (selectx == stores.value) { return };
	var gradeBox = formname["GD_BOX"+id];
	gradeBox.value = "";
	var radioButton = formname["RADVAL"+id];
	for (var i=0; i<radioButton.length; i++) {
	    radioButton[i].checked=false;
	}
	stores.value = selectx;
    }

    function checkSolved(formname,id) {
	if (formname["solved"+id].value == "correct_by_student" && formname.overRideScore.value == 'no') {
	    var reply = confirm("This problem has been graded correct by the computer. Do you want to change the score?");
	    if (!reply) {return "noupdate";}
	    formname.overRideScore.value = 'yes';
	}
	return "update";
    }

    function updateSelect(formname,id) {
	formname["GD_SEL"+id][0].selected = true;
	return;
    }

//=========== Check that a point is assigned for all the parts  ============
    function checksubmit(formname,val,total,parttot) {
	formname.gradeOpt.value = val;
	if (val == "Save & Next") {
	    for (i=0;i<=total;i++) {
		for (j=0;j<parttot;j++) {
		    var partid = formname["partid"+i+"_"+j].value;
		    if (formname["GD_SEL"+i+"_"+partid][0].selected) {
			var points = formname["GD_BOX"+i+"_"+partid].value;
			if (points == "") {
			    var name = formname["name"+i].value;
			    var studentID = (name != '' ? name : formname["unamedom"+i].value);
			    var resp = confirm("You did not assign a score for "+studentID+
					       ", part "+partid+". Continue?");
			    if (resp == false) {
				formname["GD_BOX"+i+"_"+partid].focus();
				return false;
			    }
			}
		    }
		    
		}
	    }
	    
	}
	if (val == "Grade Student") {
	    formname.showgrading.value = "yes";
	    if (formname.Status.value == "") {
		formname.Status.value = "Active";
	    }
	    formname.studentNo.value = total;
	}
	formname.submit();
    }

//======= Check that a score is assigned for all the problems (page/sequence grading only) =========
    function checkSubmitPage(formname,total) {
	noscore = new Array(100);
	var ptr = 0;
	for (i=1;i<total;i++) {
	    var partid = formname["q_"+i].value;
	    if (formname["GD_SEL"+i+"_"+partid][0].selected) {
		var points = formname["GD_BOX"+i+"_"+partid].value;
		var status = formname["solved"+i+"_"+partid].value;
		if (points == "" && status != "correct_by_student") {
		    noscore[ptr] = i;
		    ptr++;
		}
	    }
	}
	if (ptr != 0) {
	    var sense = ptr == 1 ? ": " : "s: ";
	    var prolist = "";
	    if (ptr == 1) {
		prolist = noscore[0];
	    } else {
		var i = 0;
		while (i < ptr-1) {
		    prolist += noscore[i]+", ";
		    i++;
		}
		prolist += "and "+noscore[i];
	    }
	    var resp = confirm("You did not assign any score for the following problem"+sense+prolist+". Continue?");
	    if (resp == false) {
		return false;
	    }
	}

	formname.submit();
    }
</script>
SUBJAVASCRIPT
}

#--- javascript for essay type problem --
sub sub_page_kw_js {
    my $request = shift;
    my $iconpath = $request->dir_config('lonIconsURL');
    &commonJSfunctions($request);

    my $inner_js_msg_central=<<INNERJS;
    <script text="text/javascript">
    function checkInput() {
      opener.document.SCORE.msgsub.value = opener.checkEntities(document.msgcenter.msgsub.value);
      var nmsg   = opener.document.SCORE.savemsgN.value;
      var usrctr = document.msgcenter.usrctr.value;
      var newval = opener.document.SCORE["newmsg"+usrctr];
      newval.value = opener.checkEntities(document.msgcenter.newmsg.value);

      var msgchk = "";
      if (document.msgcenter.subchk.checked) {
         msgchk = "msgsub,";
      }
      var includemsg = 0;
      for (var i=1; i<=nmsg; i++) {
          var opnmsg = opener.document.SCORE["savemsg"+i];
          var frmmsg = document.msgcenter["msg"+i];
          opnmsg.value = opener.checkEntities(frmmsg.value);
          var showflg = opener.document.SCORE["shownOnce"+i];
          showflg.value = "1";
          var chkbox = document.msgcenter["msgn"+i];
          if (chkbox.checked) {
             msgchk += "savemsg"+i+",";
             includemsg = 1;
          }
      }
      if (document.msgcenter.newmsgchk.checked) {
         msgchk += "newmsg"+usrctr;
         includemsg = 1;
      }
      imgformname = opener.document.SCORE["mailicon"+usrctr];
      imgformname.src = "$iconpath/"+((includemsg) ? "mailto.gif" : "mailbkgrd.gif");
      var includemsg = opener.document.SCORE["includemsg"+usrctr];
      includemsg.value = msgchk;

      self.close()

    }
    </script>
INNERJS

    my $inner_js_highlight_central=<<INNERJS;
 <script type="text/javascript">
    function updateChoice(flag) {
      opener.document.SCORE.kwclr.value = opener.radioSelection(document.hlCenter.kwdclr);
      opener.document.SCORE.kwsize.value = opener.radioSelection(document.hlCenter.kwdsize);
      opener.document.SCORE.kwstyle.value = opener.radioSelection(document.hlCenter.kwdstyle);
      opener.document.SCORE.refresh.value = "on";
      if (opener.document.SCORE.keywords.value!=""){
         opener.document.SCORE.submit();
      }
      self.close()
    }
</script>
INNERJS

    my $start_page_msg_central = 
        &Apache::loncommon::start_page('Message Central',$inner_js_msg_central,
				       {'js_ready'  => 1,
					'only_body' => 1,
					'bgcolor'   =>'#FFFFFF',});
    my $end_page_msg_central = 
	&Apache::loncommon::end_page({'js_ready' => 1});


    my $start_page_highlight_central = 
        &Apache::loncommon::start_page('Highlight Central',
				       $inner_js_highlight_central,
				       {'js_ready'  => 1,
					'only_body' => 1,
					'bgcolor'   =>'#FFFFFF',});
    my $end_page_highlight_central = 
	&Apache::loncommon::end_page({'js_ready' => 1});

    my $docopen=&Apache::lonhtmlcommon::javascript_docopen();
    $docopen=~s/^document\.//;
    $request->print(<<SUBJAVASCRIPT);
<script type="text/javascript" language="javascript">

//===================== Show list of keywords ====================
  function keywords(formname) {
    var nret = prompt("Keywords list, separated by a space. Add/delete to list if desired.",formname.keywords.value);
    if (nret==null) return;
    formname.keywords.value = nret;

    if (formname.keywords.value != "") {
	formname.refresh.value = "on";
	formname.submit();
    }
    return;
  }

//===================== Script to view submitted by ==================
  function viewSubmitter(submitter) {
    document.SCORE.refresh.value = "on";
    document.SCORE.NCT.value = "1";
    document.SCORE.unamedom0.value = submitter;
    document.SCORE.submit();
    return;
  }

//===================== Script to add keyword(s) ==================
  function getSel() {
    if (document.getSelection) txt = document.getSelection();
    else if (document.selection) txt = document.selection.createRange().text;
    else return;
    var cleantxt = txt.replace(new RegExp('([\\f\\n\\r\\t\\v ])+', 'g')," ");
    if (cleantxt=="") {
	alert("Please select a word or group of words from document and then click this link.");
	return;
    }
    var nret = prompt("Add selection to keyword list? Edit if desired.",cleantxt);
    if (nret==null) return;
    document.SCORE.keywords.value = document.SCORE.keywords.value+" "+nret;
    if (document.SCORE.keywords.value != "") {
	document.SCORE.refresh.value = "on";
	document.SCORE.submit();
    }
    return;
  }

//====================== Script for composing message ==============
   // preload images
   img1 = new Image();
   img1.src = "$iconpath/mailbkgrd.gif";
   img2 = new Image();
   img2.src = "$iconpath/mailto.gif";

  function msgCenter(msgform,usrctr,fullname) {
    var Nmsg  = msgform.savemsgN.value;
    savedMsgHeader(Nmsg,usrctr,fullname);
    var subject = msgform.msgsub.value;
    var msgchk = document.SCORE["includemsg"+usrctr].value;
    re = /msgsub/;
    var shwsel = "";
    if (re.test(msgchk)) { shwsel = "checked" }
    subject = (document.SCORE.shownSub.value == 0 ? checkEntities(subject) : subject);
    displaySubject(checkEntities(subject),shwsel);
    for (var i=1; i<=Nmsg; i++) {
	var testmsg = "savemsg"+i+",";
	re = new RegExp(testmsg,"g");
	shwsel = "";
	if (re.test(msgchk)) { shwsel = "checked" }
	var message = document.SCORE["savemsg"+i].value;
	message = (document.SCORE["shownOnce"+i].value == 0 ? checkEntities(message) : message);
	displaySavedMsg(i,message,shwsel); //I do not get it. w/o checkEntities on saved messages,
	                                   //any &lt; is already converted to <, etc. However, only once!!
    }
    newmsg = document.SCORE["newmsg"+usrctr].value;
    shwsel = "";
    re = /newmsg/;
    if (re.test(msgchk)) { shwsel = "checked" }
    newMsg(newmsg,shwsel);
    msgTail(); 
    return;
  }

  function checkEntities(strx) {
    if (strx.length == 0) return strx;
    var orgStr = ["&", "<", ">", '"']; 
    var newStr = ["&amp;", "&lt;", "&gt;", "&quot;"];
    var counter = 0;
    while (counter < 4) {
	strx = strReplace(strx,orgStr[counter],newStr[counter]);
	counter++;
    }
    return strx;
  }

  function strReplace(strx, orgStr, newStr) {
    return strx.split(orgStr).join(newStr);
  }

  function savedMsgHeader(Nmsg,usrctr,fullname) {
    var height = 70*Nmsg+250;
    var scrollbar = "no";
    if (height > 600) {
	height = 600;
	scrollbar = "yes";
    }
    var xpos = (screen.width-600)/2;
    xpos = (xpos < 0) ? '0' : xpos;
    var ypos = (screen.height-height)/2-30;
    ypos = (ypos < 0) ? '0' : ypos;

    pWin = window.open('', 'MessageCenter', 'resizable=yes,toolbar=no,location=no,scrollbars='+scrollbar+',screenx='+xpos+',screeny='+ypos+',width=600,height='+height);
    pWin.focus();
    pDoc = pWin.document;
    pDoc.$docopen;
    pDoc.write('$start_page_msg_central');

    pDoc.write("<form action=\\"inactive\\" name=\\"msgcenter\\">");
    pDoc.write("<input value=\\""+usrctr+"\\" name=\\"usrctr\\" type=\\"hidden\\">");
    pDoc.write("<h3><span class=\\"LC_info\\">&nbsp;Compose Message for \"+fullname+\"</span></h3><br /><br />");

    pDoc.write("<table border=0 width=100%><tr><td bgcolor=\\"#777777\\">");
    pDoc.write("<table border=0 width=100%><tr bgcolor=\\"#ddffff\\">");
    pDoc.write("<td><b>Type</b></td><td><b>Include</b></td><td><b>Message</td></tr>");
}
    function displaySubject(msg,shwsel) {
    pDoc = pWin.document;
    pDoc.write("<tr bgcolor=\\"#ffffdd\\">");
    pDoc.write("<td>Subject</td>");
    pDoc.write("<td align=\\"center\\"><input name=\\"subchk\\" type=\\"checkbox\\"" +shwsel+"></td>");
    pDoc.write("<td><input name=\\"msgsub\\" type=\\"text\\" value=\\""+msg+"\\"size=\\"60\\" maxlength=\\"80\\"></td></tr>");
}

  function displaySavedMsg(ctr,msg,shwsel) {
    pDoc = pWin.document;
    pDoc.write("<tr bgcolor=\\"#ffffdd\\">");
    pDoc.write("<td align=\\"center\\">"+ctr+"</td>");
    pDoc.write("<td align=\\"center\\"><input name=\\"msgn"+ctr+"\\" type=\\"checkbox\\"" +shwsel+"></td>");
    pDoc.write("<td><textarea name=\\"msg"+ctr+"\\" cols=\\"60\\" rows=\\"3\\">"+msg+"</textarea></td></tr>");
}

  function newMsg(newmsg,shwsel) {
    pDoc = pWin.document;
    pDoc.write("<tr bgcolor=\\"#ffffdd\\">");
    pDoc.write("<td align=\\"center\\">New</td>");
    pDoc.write("<td align=\\"center\\"><input name=\\"newmsgchk\\" type=\\"checkbox\\"" +shwsel+"></td>");
    pDoc.write("<td><textarea name=\\"newmsg\\" cols=\\"60\\" rows=\\"3\\" onchange=\\"javascript:this.form.newmsgchk.checked=true\\" >"+newmsg+"</textarea></td></tr>");
}

  function msgTail() {
    pDoc = pWin.document;
    pDoc.write("</table>");
    pDoc.write("</td></tr></table>&nbsp;");
    pDoc.write("<input type=\\"button\\" value=\\"Save\\" onClick=\\"javascript:checkInput()\\">&nbsp;&nbsp;");
    pDoc.write("<input type=\\"button\\" value=\\"Cancel\\" onClick=\\"self.close()\\"><br /><br />");
    pDoc.write("</form>");
    pDoc.write('$end_page_msg_central');
    pDoc.close();
}

//====================== Script for keyword highlight options ==============
  function kwhighlight() {
    var kwclr    = document.SCORE.kwclr.value;
    var kwsize   = document.SCORE.kwsize.value;
    var kwstyle  = document.SCORE.kwstyle.value;
    var redsel = "";
    var grnsel = "";
    var blusel = "";
    if (kwclr=="red")   {var redsel="checked"};
    if (kwclr=="green") {var grnsel="checked"};
    if (kwclr=="blue")  {var blusel="checked"};
    var sznsel = "";
    var sz1sel = "";
    var sz2sel = "";
    if (kwsize=="0")  {var sznsel="checked"};
    if (kwsize=="+1") {var sz1sel="checked"};
    if (kwsize=="+2") {var sz2sel="checked"};
    var synsel = "";
    var syisel = "";
    var sybsel = "";
    if (kwstyle=="")    {var synsel="checked"};
    if (kwstyle=="<i>") {var syisel="checked"};
    if (kwstyle=="<b>") {var sybsel="checked"};
    highlightCentral();
    highlightbody('red','red',redsel,'0','normal',sznsel,'','normal',synsel);
    highlightbody('green','green',grnsel,'+1','+1',sz1sel,'<i>','italic',syisel);
    highlightbody('blue','blue',blusel,'+2','+2',sz2sel,'<b>','bold',sybsel);
    highlightend();
    return;
  }

  function highlightCentral() {
//    if (window.hwdWin) window.hwdWin.close();
    var xpos = (screen.width-400)/2;
    xpos = (xpos < 0) ? '0' : xpos;
    var ypos = (screen.height-330)/2-30;
    ypos = (ypos < 0) ? '0' : ypos;

    hwdWin = window.open('', 'KeywordHighlightCentral', 'resizeable=yes,toolbar=no,location=no,scrollbars=no,width=400,height=300,screenx='+xpos+',screeny='+ypos);
    hwdWin.focus();
    var hDoc = hwdWin.document;
    hDoc.$docopen;
    hDoc.write('$start_page_highlight_central');
    hDoc.write("<form action=\\"inactive\\" name=\\"hlCenter\\">");
    hDoc.write("<h3><span class=\\"LC_info\\">&nbsp;Keyword Highlight Options</span></h3><br /><br />");

    hDoc.write("<table border=0 width=100%><tr><td bgcolor=\\"#777777\\">");
    hDoc.write("<table border=0 width=100%><tr bgcolor=\\"#ddffff\\">");
    hDoc.write("<td><b>Text Color</b></td><td><b>Font Size</b></td><td><b>Font Style</td></tr>");
  }

  function highlightbody(clrval,clrtxt,clrsel,szval,sztxt,szsel,syval,sytxt,sysel) { 
    var hDoc = hwdWin.document;
    hDoc.write("<tr bgcolor=\\"#ffffdd\\">");
    hDoc.write("<td align=\\"left\\">");
    hDoc.write("<input name=\\"kwdclr\\" type=\\"radio\\" value=\\""+clrval+"\\" "+clrsel+">&nbsp;"+clrtxt+"</td>");
    hDoc.write("<td align=\\"left\\">");
    hDoc.write("<input name=\\"kwdsize\\" type=\\"radio\\" value=\\""+szval+"\\" "+szsel+">&nbsp;"+sztxt+"</td>");
    hDoc.write("<td align=\\"left\\">");
    hDoc.write("<input name=\\"kwdstyle\\" type=\\"radio\\" value=\\""+syval+"\\" "+sysel+">&nbsp;"+sytxt+"</td>");
    hDoc.write("</tr>");
  }

  function highlightend() { 
    var hDoc = hwdWin.document;
    hDoc.write("</table>");
    hDoc.write("</td></tr></table>&nbsp;");
    hDoc.write("<input type=\\"button\\" value=\\"Save\\" onClick=\\"javascript:updateChoice(1)\\">&nbsp;&nbsp;");
    hDoc.write("<input type=\\"button\\" value=\\"Cancel\\" onClick=\\"self.close()\\"><br /><br />");
    hDoc.write("</form>");
    hDoc.write('$end_page_highlight_central');
    hDoc.close();
  }

</script>
SUBJAVASCRIPT
}

sub get_increment {
    my $increment = $env{'form.increment'};
    if ($increment != 1 && $increment != .5 && $increment != .25 &&
        $increment != .1) {
        $increment = 1;
    }
    return $increment;
}

#--- displays the grading box, used in essay type problem and grading by page/sequence
sub gradeBox {
    my ($request,$symb,$uname,$udom,$counter,$partid,$record) = @_;
    my $checkIcon = '<img alt="'.&mt('Check Mark').
	'" src="'.$request->dir_config('lonIconsURL').
	'/check.gif" height="16" border="0" />';
    my $wgt    = &Apache::lonnet::EXT('resource.'.$partid.'.weight',$symb,$udom,$uname);
    my $wgtmsg = ($wgt > 0 ? '(problem weight)' : 
		  '<span class="LC_info">problem weight assigned by computer</span>');
    $wgt       = ($wgt > 0 ? $wgt : '1');
    my $score  = ($$record{'resource.'.$partid.'.awarded'} eq '' ?
		  '' : &compute_points($$record{'resource.'.$partid.'.awarded'},$wgt));
    my $result='<input type="hidden" name="WGT'.$counter.'_'.$partid.'" value="'.$wgt.'" />'."\n";
    my $display_part=&get_display_part($partid,$symb);
    my %last_resets = &get_last_resets($symb,$env{'request.course.id'},
				       [$partid]);
    my $aggtries = $$record{'resource.'.$partid.'.tries'};
    if ($last_resets{$partid}) {
        $aggtries = &get_num_tries($record,$last_resets{$partid},$partid);
    }
    $result.='<table border="0"><tr><td>'.
	'<b>Part: </b>'.$display_part.' <b>Points: </b></td><td>'."\n";
    my $ctr = 0;
    my $thisweight = 0;
    my $increment = &get_increment();
    $result.='<table border="0"><tr>'."\n";  # display radio buttons in a nice table 10 across
    while ($thisweight<=$wgt) {
	$result.= '<td><span style="white-space: nowrap;"><label><input type="radio" name="RADVAL'.$counter.'_'.$partid.'" '.
	    'onclick="javascript:writeBox(this.form,\''.$counter.'_'.$partid.'\','.
	    $thisweight.')" value="'.$thisweight.'" '.
	    ($score eq $thisweight ? 'checked':'').' /> '.$thisweight."</label></span></td>\n";
	$result.=(($ctr+1)%10 == 0 ? '</tr><tr>' : '');
        $thisweight += $increment;
	$ctr++;
    }
    $result.='</tr></table>';
    $result.='</td><td>&nbsp;<b>or</b>&nbsp;</td>'."\n";
    $result.='<td><input type="text" name="GD_BOX'.$counter.'_'.$partid.'"'.
	($score ne ''? ' value = "'.$score.'"':'').' size="4" '.
	'onChange="javascript:updateRadio(this.form,\''.$counter.'_'.$partid.'\','.
	$wgt.')" /></td>'."\n";
    $result.='<td>/'.$wgt.' '.$wgtmsg.
	($$record{'resource.'.$partid.'.solved'} eq 'correct_by_student' ? '&nbsp;'.$checkIcon : '').
	' </td><td>'."\n";
    $result.='<select name="GD_SEL'.$counter.'_'.$partid.'" '.
	'onChange="javascript:clearRadBox(this.form,\''.$counter.'_'.$partid.'\')" >'."\n";
    if ($$record{'resource.'.$partid.'.solved'} eq 'excused') {
	$result.='<option></option>'.
	    '<option selected="on">excused</option>';
    } else {
	$result.='<option selected="on"></option>'.
	    '<option>excused</option>';
    }
    $result.='<option>reset status</option></select>'."\n";
    $result.="&nbsp;&nbsp;\n";
    $result.='<input type="hidden" name="stores'.$counter.'_'.$partid.'" value="" />'."\n".
	'<input type="hidden" name="oldpts'.$counter.'_'.$partid.'" value="'.$score.'" />'."\n".
	'<input type="hidden" name="solved'.$counter.'_'.$partid.'" value="'.
	$$record{'resource.'.$partid.'.solved'}.'" />'."\n".
        '<input type="hidden" name="totaltries'.$counter.'_'.$partid.'" value="'.
        $$record{'resource.'.$partid.'.tries'}.'" />'."\n".
        '<input type="hidden" name="aggtries'.$counter.'_'.$partid.'" value="'.
        $aggtries.'" />'."\n";
    $result.='</td></tr></table>'."\n";
    $result.=&handback_box($symb,$uname,$udom,$counter,$partid,$record);
    return $result;
}

sub handback_box {
    my ($symb,$uname,$udom,$counter,$partid,$record) = @_;
    my ($partlist,$handgrade,$responseType) = &response_type($symb);
    my (@respids);
     my @part_response_id = &flatten_responseType($responseType);
    foreach my $part_response_id (@part_response_id) {
    	my ($part,$resp) = @{ $part_response_id };
        if ($part eq $partid) {
            push(@respids,$resp);
        }
    }
    my $result;
    foreach my $respid (@respids) {
	my $prefix = $counter.'_'.$partid.'_'.$respid.'_';
	my $files=&get_submitted_files($udom,$uname,$partid,$respid,$record);
	next if (!@$files);
	my $file_counter = 1;
	foreach my $file (@$files) {
	    if ($file =~ /\/portfolio\//) {
    	        my ($file_path, $file_disp) = ($file =~ m|(.+/)(.+)$|);
    	        my ($name,$version,$ext) = &file_name_version_ext($file_disp);
    	        $file_disp = "$name.$ext";
    	        $file = $file_path.$file_disp;
    	        $result.=&mt('Return commented version of [_1] to student.',
    			 '<span class="LC_filename">'.$file_disp.'</span>');
    	        $result.='<input type="file"   name="'.$prefix.'returndoc'.$file_counter.'" />'."\n";
    	        $result.='<input type="hidden" name="'.$prefix.'origdoc'.$file_counter.'" value="'.$file.'" /><br />';
    	        $result.='(File will be uploaded when you click on Save & Next below.)<br />';
    	        $file_counter++;
	    }
	}
    }
    return $result;    
}

sub show_problem {
    my ($request,$symb,$uname,$udom,$removeform,$viewon,$mode,$form) = @_;
    my $rendered;
    my %form = ((ref($form) eq 'HASH')? %{$form} : ());
    &Apache::lonxml::remember_problem_counter();
    if ($mode eq 'both' or $mode eq 'text') {
	$rendered=&Apache::loncommon::get_student_view($symb,$uname,$udom,
						       $env{'request.course.id'},
						       undef,\%form);
    }
    if ($removeform) {
	$rendered=~s|<form(.*?)>||g;
	$rendered=~s|</form>||g;
	$rendered=~s|(<input[^>]*name\s*=\s*"?)(\w+)("?)|$1would_have_been_$2$3|g;
    }
    my $companswer;
    if ($mode eq 'both' or $mode eq 'answer') {
	&Apache::lonxml::restore_problem_counter();
	$companswer=
	    &Apache::loncommon::get_student_answers($symb,$uname,$udom,
						    $env{'request.course.id'},
						    %form);
    }
    if ($removeform) {
	$companswer=~s|<form(.*?)>||g;
	$companswer=~s|</form>||g;
	$companswer=~s|name="submit"|name="would_have_been_submit"|g;
    }
    my $result.='<table border="0" width="100%"><tr><td bgcolor="#777777">';
    $result.='<table border="0" width="100%">';
    if ($viewon) {
	$result.='<tr><td bgcolor="#e6ffff"><b> ';
	if ($mode eq 'both' or $mode eq 'text') {
	    $result.='View of the problem - ';
	} else {
	    $result.='Correct answer: ';
	}
	$result.=$env{'form.fullname'}.'</b></td></tr>';
    }
    if ($mode eq 'both') {
	$result.='<tr><td bgcolor="#ffffff">'.$rendered.'<br />';
	$result.='<b>Correct answer:</b><br />'.$companswer;
    } elsif ($mode eq 'text') {
	$result.='<tr><td bgcolor="#ffffff">'.$rendered;
    } elsif ($mode eq 'answer') {
	$result.='<tr><td bgcolor="#ffffff">'.$companswer;
    }
    $result.='</td></tr></table>';
    $result.='</td></tr></table><br />';
    return $result;
}

sub files_exist {
    my ($r, $symb) = @_;
    my @students = &Apache::loncommon::get_env_multiple('form.stuinfo');

    foreach my $student (@students) {
        my ($uname,$udom,$fullname) = split(/:/,$student);
        my %record = &Apache::lonnet::restore($symb,$env{'request.course.id'},
					      $udom,$uname);
        my ($string,$timestamp)= &get_last_submission(\%record);
        foreach my $submission (@$string) {
            my ($partid,$respid) =
		($submission =~ /^resource\.([^\.]*)\.([^\.]*)\.submission/);
            my $files=&get_submitted_files($udom,$uname,$partid,$respid,
					   \%record);
            return 1 if (@$files);
        }
    }
    return 0;
}

sub download_all_link {
    my ($r,$symb) = @_;
    my $all_students = 
	join("\n", &Apache::loncommon::get_env_multiple('form.stuinfo'));

    my $parts =
	join("\n",&Apache::loncommon::get_env_multiple('form.vPart'));

    my $identifier = &Apache::loncommon::get_cgi_id();
    &Apache::lonnet::appenv('cgi.'.$identifier.'.students' => $all_students,
                            'cgi.'.$identifier.'.symb' => $symb,
                            'cgi.'.$identifier.'.parts' => $parts,);
    $r->print('<a href="/cgi-bin/multidownload.pl?'.$identifier.'">'.
	      &mt('Download All Submitted Documents').'</a>');
    return
}

# --------------------------- show submissions of a student, option to grade 
sub submission {
    my ($request,$counter,$total) = @_;

    my ($uname,$udom)     = ($env{'form.student'},$env{'form.userdom'});
    $udom = ($udom eq '' ? $env{'user.domain'} : $udom); #has form.userdom changed for a student?
    my $usec = &Apache::lonnet::getsection($udom,$uname,$env{'request.course.id'});
    $env{'form.fullname'} = &Apache::loncommon::plainname($uname,$udom,'lastname') if $env{'form.fullname'} eq '';

    my $symb = &get_symb($request); 
    if ($symb eq '') { $request->print("Unable to handle ambiguous references:."); return ''; }

    if (!&canview($usec)) {
	$request->print('<span class="LC_warning">Unable to view requested student.('.
			$uname.':'.$udom.' in section '.$usec.' in course id '.
			$env{'request.course.id'}.')</span>');
	$request->print(&show_grading_menu_form($symb));
	return;
    }

    if (!$env{'form.lastSub'}) { $env{'form.lastSub'} = 'datesub'; }
    if (!$env{'form.vProb'}) { $env{'form.vProb'} = 'yes'; }
    if (!$env{'form.vAns'}) { $env{'form.vAns'} = 'yes'; }
    my $last = ($env{'form.lastSub'} eq 'last' ? 'last' : '');
    my $checkIcon = '<img alt="'.&mt('Check Mark').
	'" src="'.$request->dir_config('lonIconsURL').
	'/check.gif" height="16" border="0" />';

    # header info
    if ($counter == 0) {
	&sub_page_js($request);
	&sub_page_kw_js($request) if ($env{'form.handgrade'} eq 'yes');
	$env{'form.probTitle'} = $env{'form.probTitle'} eq '' ? 
	    &Apache::lonnet::gettitle($symb) : $env{'form.probTitle'};
	if ($env{'form.handgrade'} eq 'yes' && &files_exist($request, $symb)) {
	    &download_all_link($request, $symb);
	}
	$request->print('<h3>&nbsp;<span class="LC_info">Submission Record</span></h3>'."\n".
			'<h4>&nbsp;<b>Resource: </b>'.$env{'form.probTitle'}.'</h4>'."\n");

	if ($env{'form.handgrade'} eq 'no') {
	    my $checkMark='<br /><br />&nbsp;<b>Note:</b> Part(s) graded correct by the computer is marked with a '.
		$checkIcon.' symbol.'."\n";
	    $request->print($checkMark);
	}

	# option to display problem, only once else it cause problems 
        # with the form later since the problem has a form.
	if ($env{'form.vProb'} eq 'yes' or $env{'form.vAns'} eq 'yes') {
	    my $mode;
	    if ($env{'form.vProb'} eq 'yes' && $env{'form.vAns'} eq 'yes') {
		$mode='both';
	    } elsif ($env{'form.vProb'} eq 'yes') {
		$mode='text';
	    } elsif ($env{'form.vAns'} eq 'yes') {
		$mode='answer';
	    }
	    &Apache::lonxml::clear_problem_counter();
	    $request->print(&show_problem($request,$symb,$uname,$udom,0,1,$mode));
	}
	
	# kwclr is the only variable that is guaranteed to be non blank 
        # if this subroutine has been called once.
	my %keyhash = ();
	if ($env{'form.kwclr'} eq '' && $env{'form.handgrade'} eq 'yes') {
	    %keyhash = &Apache::lonnet::dump('nohist_handgrade',
					     $env{'course.'.$env{'request.course.id'}.'.domain'},
					     $env{'course.'.$env{'request.course.id'}.'.num'});

	    my $loginuser = $env{'user.name'}.':'.$env{'user.domain'};
	    $env{'form.keywords'} = $keyhash{$symb.'_keywords'} ne '' ? $keyhash{$symb.'_keywords'} : '';
	    $env{'form.kwclr'}    = $keyhash{$loginuser.'_kwclr'} ne '' ? $keyhash{$loginuser.'_kwclr'} : 'red';
	    $env{'form.kwsize'}   = $keyhash{$loginuser.'_kwsize'} ne '' ? $keyhash{$loginuser.'_kwsize'} : '0';
	    $env{'form.kwstyle'}  = $keyhash{$loginuser.'_kwstyle'} ne '' ? $keyhash{$loginuser.'_kwstyle'} : '';
	    $env{'form.msgsub'}   = $keyhash{$symb.'_subject'} ne '' ? 
		$keyhash{$symb.'_subject'} : $env{'form.probTitle'};
	    $env{'form.savemsgN'} = $keyhash{$symb.'_savemsgN'} ne '' ? $keyhash{$symb.'_savemsgN'} : '0';
	}
	my $overRideScore = $env{'form.overRideScore'} eq '' ? 'no' : $env{'form.overRideScore'};

	$request->print('<form action="/adm/grades" method="post" name="SCORE" enctype="multipart/form-data">'."\n".
			'<input type="hidden" name="command"    value="handgrade" />'."\n".
			'<input type="hidden" name="saveState"  value="'.$env{'form.saveState'}.'" />'."\n".
			'<input type="hidden" name="Status"     value="'.$env{'form.Status'}.'" />'."\n".
			'<input type="hidden" name="overRideScore" value="'.$overRideScore.'" />'."\n".
			'<input type="hidden" name="probTitle"  value="'.$env{'form.probTitle'}.'" />'."\n".
			'<input type="hidden" name="refresh"    value="off" />'."\n".
			'<input type="hidden" name="studentNo"  value="" />'."\n".
			'<input type="hidden" name="gradeOpt"   value="" />'."\n".
			'<input type="hidden" name="symb"       value="'.$symb.'" />'."\n".
			'<input type="hidden" name="showgrading" value="'.$env{'form.showgrading'}.'" />'."\n".
			'<input type="hidden" name="vProb"      value="'.$env{'form.vProb'}.'" />'."\n".
			'<input type="hidden" name="vAns"       value="'.$env{'form.vAns'}.'" />'."\n".
			'<input type="hidden" name="lastSub"    value="'.$env{'form.lastSub'}.'" />'."\n".
			'<input type="hidden" name="section"    value="'.$env{'form.section'}.'" />'."\n".
			'<input type="hidden" name="submitonly" value="'.$env{'form.submitonly'}.'" />'."\n".
			'<input type="hidden" name="handgrade"  value="'.$env{'form.handgrade'}.'" />'."\n".
			'<input type="hidden" name="NCT"'.
			' value="'.($env{'form.NTSTU'} ne '' ? $env{'form.NTSTU'} : $total+1).'" />'."\n");
	if ($env{'form.handgrade'} eq 'yes') {
	    $request->print('<input type="hidden" name="keywords" value="'.$env{'form.keywords'}.'" />'."\n".
			    '<input type="hidden" name="kwclr"    value="'.$env{'form.kwclr'}.'" />'."\n".
			    '<input type="hidden" name="kwsize"   value="'.$env{'form.kwsize'}.'" />'."\n".
			    '<input type="hidden" name="kwstyle"  value="'.$env{'form.kwstyle'}.'" />'."\n".
			    '<input type="hidden" name="msgsub"   value="'.$env{'form.msgsub'}.'" />'."\n".
			    '<input type="hidden" name="shownSub" value="0" />'."\n".
			    '<input type="hidden" name="savemsgN" value="'.$env{'form.savemsgN'}.'" />'."\n");
	    foreach my $partid (&Apache::loncommon::get_env_multiple('form.vPart')) {
		$request->print('<input type="hidden" name="vPart" value="'.$partid.'" />'."\n");
	    }
	}
	
	my ($cts,$prnmsg) = (1,'');
	while ($cts <= $env{'form.savemsgN'}) {
	    $prnmsg.='<input type="hidden" name="savemsg'.$cts.'" value="'.
		(!exists($keyhash{$symb.'_savemsg'.$cts}) ? 
		 &Apache::lonfeedback::clear_out_html($env{'form.savemsg'.$cts}) :
		 &Apache::lonfeedback::clear_out_html($keyhash{$symb.'_savemsg'.$cts})).
		'" />'."\n".
		'<input type="hidden" name="shownOnce'.$cts.'" value="0" />'."\n";
	    $cts++;
	}
	$request->print($prnmsg);

	if ($env{'form.handgrade'} eq 'yes' && $env{'form.showgrading'} eq 'yes') {
#
# Print out the keyword options line
#
	    $request->print(<<KEYWORDS);
&nbsp;<b>Keyword Options:</b>&nbsp;
<a href="javascript:keywords(document.SCORE)"; TARGET=_self>List</a>&nbsp; &nbsp;
<a href="#" onMouseDown="javascript:getSel(); return false"
 CLASS="page">Paste Selection to List</a>&nbsp; &nbsp;
<a href="javascript:kwhighlight()"; TARGET=_self>Highlight Attribute</a><br /><br />
KEYWORDS
#
# Load the other essays for similarity check
#
            my (undef,undef,$essayurl) = &Apache::lonnet::decode_symb($symb);
	    my ($adom,$aname,$apath)=($essayurl=~/^($LONCAPA::domain_re)\/($LONCAPA::username_re)\/(.*)$/);
	    $apath=&escape($apath);
	    $apath=~s/\W/\_/gs;
	    %oldessays=&Apache::lonnet::dump('nohist_essay_'.$apath,$adom,$aname);
        }
    }

    if ($env{'form.vProb'} eq 'all' or $env{'form.vAns'} eq 'all') {
	$request->print('<br /><br /><br />') if ($counter > 0);
	my $mode;
	if ($env{'form.vProb'} eq 'all' && $env{'form.vAns'} eq 'all') {
	    $mode='both';
	} elsif ($env{'form.vProb'} eq 'all' ) {
	    $mode='text';
	} elsif ($env{'form.vAns'} eq 'all') {
	    $mode='answer';
	}
	&Apache::lonxml::clear_problem_counter();
	$request->print(&show_problem($request,$symb,$uname,$udom,1,1,$mode));
    }

    my %record = &Apache::lonnet::restore($symb,$env{'request.course.id'},$udom,$uname);
    my ($partlist,$handgrade,$responseType) = &response_type($symb);

    # Display student info
    $request->print(($counter == 0 ? '' : '<br />'));
    my $result='<table border="0" width="100%"><tr><td bgcolor="#777777">'."\n".
	'<table border="0" width="100%"><tr bgcolor="#edffff"><td>'."\n";

    $result.='<b>Fullname: </b>'.&nameUserString(undef,$env{'form.fullname'},$uname,$udom).'<br />'."\n";
    $result.='<input type="hidden" name="name'.$counter.
	'" value="'.$env{'form.fullname'}.'" />'."\n";

    # If any part of the problem is an essay-response (handgraded), then check for collaborators
    my @col_fullnames;
    my ($classlist,$fullname);
    if ($env{'form.handgrade'} eq 'yes') {
	($classlist,undef,$fullname) = &getclasslist('all','0');
	for (keys (%$handgrade)) {
	    my $ncol = &Apache::lonnet::EXT('resource.'.$_.
					    '.maxcollaborators',
                                            $symb,$udom,$uname);
	    next if ($ncol <= 0);
            s/\_/\./g;
            next if ($record{'resource.'.$_.'.collaborators'} eq '');
            my @goodcollaborators = ();
            my @badcollaborators  = ();
	    foreach (split(/,?\s+/,$record{'resource.'.$_.'.collaborators'})) { 
		$_ =~ s/[\$\^\(\)]//g;
		next if ($_ eq '');
		my ($co_name,$co_dom) = split /\@|:/,$_;
		$co_dom = $udom if (! defined($co_dom) || $co_dom =~ /^domain$/i);
		next if ($co_name eq $uname && $co_dom eq $udom);
		# Doing this grep allows 'fuzzy' specification
		my @Matches = grep /^$co_name:$co_dom$/i,keys %$classlist;
		if (! scalar(@Matches)) {
		    push @badcollaborators,$_;
		} else {
		    push @goodcollaborators, @Matches;
		}
	    }
            if (scalar(@goodcollaborators) != 0) {
                $result.='<b>Collaborators: </b>';
                foreach (@goodcollaborators) {
		    my ($lastname,$givenn) = split(/,/,$$fullname{$_});
		    push @col_fullnames, $givenn.' '.$lastname;
		    $result.=$$fullname{$_}.'&nbsp; &nbsp; &nbsp;';
		}
                $result.='<br />'."\n";
		my ($part)=split(/\./,$_);
		$result.='<input type="hidden" name="collaborator'.$counter.
		    '" value="'.$part.':'.(join ':',@goodcollaborators).'" />'.
		    "\n";
	    }
	    if (scalar(@badcollaborators) > 0) {
		$result.='<table border="0"><tr bgcolor="#ffbbbb"><td>';
		$result.='This student has submitted ';
		$result.=(scalar(@badcollaborators) == 1) ? 'an invalid collaborator' : 'invalid collaborators';
		$result .= ': '.join(', ',@badcollaborators);
		$result .= '</td></tr></table>';
	    }         
	    if (scalar(@badcollaborators > $ncol)) {
		$result .= '<table border="0"><tr bgcolor="#ffbbbb"><td>';
		$result .= 'This student has submitted too many '.
		    'collaborators.  Maximum is '.$ncol.'.';
		$result .= '</td></tr></table>';
	    }
	}
    }
    $request->print($result."\n");

    # print student answer/submission
    # Options are (1) Handgaded submission only
    #             (2) Last submission, includes submission that is not handgraded 
    #                  (for multi-response type part)
    #             (3) Last submission plus the parts info
    #             (4) The whole record for this student
    if ($env{'form.lastSub'} =~ /^(lastonly|hdgrade)$/) {
	my ($string,$timestamp)= &get_last_submission(\%record);
	my $lastsubonly=''.
	    ($$timestamp eq '' ? '' : '<b>Date Submitted:</b> '.
	     $$timestamp)."</td></tr>\n";
	if ($$timestamp eq '') {
	    $lastsubonly.='<tr><td bgcolor="#ffffe6">'.$$string[0]; 
	} else {
	    my %seenparts;
	    my @part_response_id = &flatten_responseType($responseType);
	    foreach my $part (@part_response_id) {
		next if ($env{'form.lastSub'} eq 'hdgrade' 
			 && $$handgrade{$$part[0].'_'.$$part[1]} ne 'yes');

		my ($partid,$respid) = @{ $part };
		my $display_part=&get_display_part($partid,$symb);
		if ($env{"form.$uname:$udom:$partid:submitted_by"}) {
		    if (exists($seenparts{$partid})) { next; }
		    $seenparts{$partid}=1;
		    my $submitby='<b>Part:</b> '.$display_part.
			' <b>Collaborative submission by:</b> '.
			'<a href="javascript:viewSubmitter(\''.
			$env{"form.$uname:$udom:$partid:submitted_by"}.
			'\')"; TARGET=_self>'.
			$$fullname{$env{"form.$uname:$udom:$partid:submitted_by"}}.'</a><br />';
		    $request->print($submitby);
		    next;
		}
		my $responsetype = $responseType->{$partid}->{$respid};
		if (!exists($record{"resource.$partid.$respid.submission"})) {
		    $lastsubonly.='<tr><td bgcolor="#ffffe6"><b>Part:</b> '.
			$display_part.' <span class="LC_internal_info">( ID '.$respid.
			' )</span>&nbsp; &nbsp;'.
			'<span class="LC_warning">Nothing submitted - no attempts</span><br /><br />';
		    next;
		}
		foreach (@$string) {
		    my ($partid,$respid) = /^resource\.([^\.]*)\.([^\.]*)\.submission/;
		    if (join('_',@{$part}) ne ($partid.'_'.$respid)) { next; }
		    my ($ressub,$subval) = split(/:/,$_,2);
		    # Similarity check
		    my $similar='';
		    if($env{'form.checkPlag'}){
			my ($oname,$odom,$ocrsid,$oessay,$osim)=
			    &most_similar($uname,$udom,$subval);
			if ($osim) {
			    $osim=int($osim*100.0);
			    $similar="<hr /><h3><span class=\"LC_warning\">Essay".
				" is $osim% similar to an essay by ".
				&Apache::loncommon::plainname($oname,$odom).
				'</span></h3><blockquote><i>'.
				&keywords_highlight($oessay).
				'</i></blockquote><hr />';
			}
		    }
		    my $order=&get_order($partid,$respid,$symb,$uname,$udom);
		    if ($env{'form.lastSub'} eq 'lastonly' || 
			($env{'form.lastSub'} eq 'hdgrade' && 
			 $$handgrade{$$part[0].'_'.$$part[1]} eq 'yes')) {
			my $display_part=&get_display_part($partid,$symb);
			$lastsubonly.='<tr><td bgcolor="#ffffe6"><b>Part:</b> '.
			    $display_part.' <span class="LC_internal_info">( ID '.$respid.
			    ' )</span>&nbsp; &nbsp;';
			my $files=&get_submitted_files($udom,$uname,$partid,$respid,\%record);
			if (@$files) {
			    $lastsubonly.='<br /><span class="LC_warning">Like all files provided by users, this file may contain virusses</span><br />';
			    my $file_counter = 0;
			    foreach my $file (@$files) {
			        $file_counter ++;
				&Apache::lonnet::allowuploaded('/adm/grades',$file);
				$lastsubonly.='<br /><a href="'.$file.'?rawmode=1" target="lonGRDs"><img src="'.&Apache::loncommon::icon($file).'" border=0"> '.$file.'</a>';
			    }
			    $lastsubonly.='<br />';
			}
			$lastsubonly.='<b>Submitted Answer: </b>'.
			    &cleanRecord($subval,$responsetype,$symb,$partid,
					 $respid,\%record,$order);
			if ($similar) {$lastsubonly.="<br /><br />$similar\n";}
		    }
		}
	    }
	}
	$lastsubonly.='</td></tr><tr bgcolor="#ffffff"><td>'."\n";
	$request->print($lastsubonly);
    } elsif ($env{'form.lastSub'} eq 'datesub') {
	my (undef,$responseType,undef,$parts) = &showResourceInfo($symb);
	$request->print(&displaySubByDates($symb,\%record,$parts,$responseType,$checkIcon,$uname,$udom));
    } elsif ($env{'form.lastSub'} =~ /^(last|all)$/) {
	$request->print(&Apache::loncommon::get_previous_attempt($symb,$uname,$udom,
								 $env{'request.course.id'},
								 $last,'.submission',
								 'Apache::grades::keywords_highlight'));
    }

    $request->print('<input type="hidden" name="unamedom'.$counter.'" value="'.$uname.':'
	.$udom.'" />'."\n");
    
    # return if view submission with no grading option
    if ($env{'form.showgrading'} eq '' || (!&canmodify($usec))) {
	my $toGrade.='<input type="button" value="Grade Student" '.
	    'onClick="javascript:checksubmit(this.form,\'Grade Student\',\''
	    .$counter.'\');" TARGET=_self> &nbsp;'."\n" if (&canmodify($usec));
	$toGrade.='</td></tr></table></td></tr></table>'."\n";
	if (($env{'form.command'} eq 'submission') || 
	    ($env{'form.command'} eq 'processGroup' && $counter == $total)) {
	    $toGrade.='</form>'.&show_grading_menu_form($symb); 
	}
	$request->print($toGrade);
	return;
    } else {
	$request->print('</td></tr></table></td></tr></table>'."\n");
    }

    # essay grading message center
    if ($env{'form.handgrade'} eq 'yes') {
	my ($lastname,$givenn) = split(/,/,$env{'form.fullname'});
	my $msgfor = $givenn.' '.$lastname;
	if (scalar(@col_fullnames) > 0) {
	    my $lastone = pop @col_fullnames;
	    $msgfor .= ', '.(join ', ',@col_fullnames).' and '.$lastone.'.';
	}
	$msgfor =~ s/\'/\\'/g; #' stupid emacs - no! javascript
	$result='<input type="hidden" name="includemsg'.$counter.'" value="" />'."\n".
	    '<input type="hidden" name="newmsg'.$counter.'" value="" />'."\n";
	$result.='&nbsp;<a href="javascript:msgCenter(document.SCORE,'.$counter.
	    ',\''.$msgfor.'\')"; TARGET=_self>'.
	    &mt('Compose message to student').(scalar(@col_fullnames) >= 1 ? 's' : '').'</a><label> ('.
	    &mt('incl. grades').' <input type="checkbox" name="withgrades'.$counter.'" /></label>)'.
	    '<img src="'.$request->dir_config('lonIconsURL').
	    '/mailbkgrd.gif" width="14" height="10" name="mailicon'.$counter.'" />'."\n".
	    '<br />&nbsp;('.
	    &mt('Message will be sent when you click on Save & Next below.').")\n";
	$request->print($result);
    }
    if ($perm{'vgr'}) {
	$request->print('<br />'.
	    &Apache::loncommon::track_student_link(&mt('View recent activity'),
						   $uname,$udom,'check'));
    }
    if ($perm{'opa'}) {
	$request->print('<br />'.
	    &Apache::loncommon::pprmlink(&mt('Set/Change parameters'),
					 $uname,$udom,$symb,'check'));
    }

    my %seen = ();
    my @partlist;
    my @gradePartRespid;
    my @part_response_id = &flatten_responseType($responseType);
    foreach my $part_response_id (@part_response_id) {
    	my ($partid,$respid) = @{ $part_response_id };
	my $part_resp = join('_',@{ $part_response_id });
	next if ($seen{$partid} > 0);
	$seen{$partid}++;
	next if ($$handgrade{$part_resp} ne 'yes' 
		 && $env{'form.lastSub'} eq 'hdgrade');
	push @partlist,$partid;
	push @gradePartRespid,$partid.'.'.$respid;
	$request->print(&gradeBox($request,$symb,$uname,$udom,$counter,$partid,\%record));
    }
    $result='<input type="hidden" name="partlist'.$counter.
	'" value="'.(join ":",@partlist).'" />'."\n";
    $result.='<input type="hidden" name="gradePartRespid'.
	'" value="'.(join ":",@gradePartRespid).'" />'."\n" if ($counter == 0);
    my $ctr = 0;
    while ($ctr < scalar(@partlist)) {
	$result.='<input type="hidden" name="partid'.$counter.'_'.$ctr.'" value="'.
	    $partlist[$ctr].'" />'."\n";
	$ctr++;
    }
    $request->print($result.'</td></tr></table></td></tr></table>'."\n");

    # print end of form
    if ($counter == $total) {
	my $endform='<table border="0"><tr><td>'."\n";
	$endform.='<input type="button" value="Save & Next" '.
	    'onClick="javascript:checksubmit(this.form,\'Save & Next\','.
	    $total.','.scalar(@partlist).');" TARGET=_self> &nbsp;'."\n";
	my $ntstu ='<select name="NTSTU">'.
	    '<option>1</option><option>2</option>'.
	    '<option>3</option><option>5</option>'.
	    '<option>7</option><option>10</option></select>'."\n";
	my $nsel = ($env{'form.NTSTU'} ne '' ? $env{'form.NTSTU'} : '1');
	$ntstu =~ s/<option>$nsel</<option selected="on">$nsel</;
	$endform.=$ntstu.'student(s) &nbsp;&nbsp;';
	$endform.='<input type="button" value="Previous" '.
	    'onClick="javascript:checksubmit(this.form,\'Previous\');" TARGET=_self> &nbsp;'."\n".
	    '<input type="button" value="Next" '.
	    'onClick="javascript:checksubmit(this.form,\'Next\');" TARGET=_self> &nbsp;';
	$endform.='(Next and Previous (student) do not save the scores.)'."\n" ;
        $endform.="<input type='hidden' value='".&get_increment().
            "' name='increment' />";
	$endform.='</td><tr></table></form>';
	$endform.=&show_grading_menu_form($symb);
	$request->print($endform);
    }
    return '';
}

#--- Retrieve the last submission for all the parts
sub get_last_submission {
    my ($returnhash)=@_;
    my (@string,$timestamp);
    if ($$returnhash{'version'}) {
	my %lasthash=();
	my ($version);
	for ($version=1;$version<=$$returnhash{'version'};$version++) {
	    foreach my $key (sort(split(/\:/,
					$$returnhash{$version.':keys'}))) {
		$lasthash{$key}=$$returnhash{$version.':'.$key};
		$timestamp = 
		    scalar(localtime($$returnhash{$version.':timestamp'}));
	    }
	}
	foreach my $key (keys(%lasthash)) {
	    next if ($key !~ /\.submission$/);

	    my ($partid,$foo) = split(/submission$/,$key);
	    my $draft  = $lasthash{$partid.'awarddetail'} eq 'DRAFT' ?
		'<span class="LC_warning">Draft Copy</span> ' : '';
	    push(@string, join(':', $key, $draft.$lasthash{$key}));
	}
    }
    if (!@string) {
	$string[0] =
	    '<span class="LC_warning">Nothing submitted - no attempts.</span>';
    }
    return (\@string,\$timestamp);
}

#--- High light keywords, with style choosen by user.
sub keywords_highlight {
    my $string    = shift;
    my $size      = $env{'form.kwsize'} eq '0' ? '' : 'size='.$env{'form.kwsize'};
    my $styleon   = $env{'form.kwstyle'} eq ''  ? '' : $env{'form.kwstyle'};
    (my $styleoff = $styleon) =~ s/\</\<\//;
    my @keylist   = split(/[,\s+]/,$env{'form.keywords'});
    foreach my $keyword (@keylist) {
	$string =~ s/\b\Q$keyword\E(\b|\.)/<font color\=$env{'form.kwclr'} $size\>$styleon$keyword$styleoff<\/font>/gi;
    }
    return $string;
}

#--- Called from submission routine
sub processHandGrade {
    my ($request) = shift;
    my $symb   = &get_symb($request);
    my (undef,undef,$url) = &Apache::lonnet::decode_symb($symb);
    my $button = $env{'form.gradeOpt'};
    my $ngrade = $env{'form.NCT'};
    my $ntstu  = $env{'form.NTSTU'};
    my $cdom   = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cnum   = $env{'course.'.$env{'request.course.id'}.'.num'};

    if ($button eq 'Save & Next') {
	my $ctr = 0;
	while ($ctr < $ngrade) {
	    my ($uname,$udom) = split(/:/,$env{'form.unamedom'.$ctr});
	    my ($errorflag,$pts,$wgt) = &saveHandGrade($request,$symb,$uname,$udom,$ctr);
	    if ($errorflag eq 'no_score') {
		$ctr++;
		next;
	    }
	    if ($errorflag eq 'not_allowed') {
		$request->print("<span class=\"LC_warning\">Not allowed to modify grades for $uname:$udom</span>");
		$ctr++;
		next;
	    }
	    my $includemsg = $env{'form.includemsg'.$ctr};
	    my ($subject,$message,$msgstatus) = ('','','');
            my $restitle = &Apache::lonnet::gettitle($symb);
            my $encrypturl=&Apache::lonnet::EXT('resource.0.encrypturl',
                                                $symb,$udom,$uname);
            my ($feedurl,$baseurl,$showsymb,$messagetail);
            $feedurl = &Apache::lonnet::clutter($url);
            if ($encrypturl =~ /^yes$/i) {
                $baseurl = &Apache::lonenc::encrypted($feedurl,1);
                $showsymb = &Apache::lonenc::encrypted($symb,1);
            } else {
                $baseurl = $feedurl;
                $showsymb = $symb;
            }
	    if ($includemsg =~ /savemsg|newmsg\Q$ctr\E/) {
		$subject = $env{'form.msgsub'} if ($includemsg =~ /msgsub/);
		unless ($subject=~/\w/) { $subject=&mt('Grading Feedback'); }
		$subject.=' ['.$restitle.']';
		my (@msgnum) = split(/,/,$includemsg);
		foreach (@msgnum) {
		    $message.=$env{'form.'.$_} if ($_ =~ /savemsg|newmsg/ && $_ ne '');
		}
		$message =&Apache::lonfeedback::clear_out_html($message);
		if ($env{'form.withgrades'.$ctr}) {
		    $message.="\n\nPoint".($pts > 1 ? 's':'').' awarded = '.$pts.' out of '.$wgt;
		    $messagetail = " for <a href=\"".
		                   $baseurl."?symb=$showsymb\">$env{'form.probTitle'}</a>";
		}
		$msgstatus = 
                    &Apache::lonmsg::user_normal_msg($uname,$udom,$subject,
						     $message.$messagetail,
                                                     undef,$baseurl,undef,
                                                     undef,undef,$showsymb,
                                                     $restitle);
		$request->print('<br />'.&mt('Sending message to [_1]:[_2]',$uname,$udom).': '.
				$msgstatus);
	    }
	    if ($env{'form.collaborator'.$ctr}) {
		my @collabstrs=&Apache::loncommon::get_env_multiple("form.collaborator$ctr");
		foreach my $collabstr (@collabstrs) {
		    my ($part,@collaborators) = split(/:/,$collabstr);
		    foreach my $collaborator (@collaborators) {
			my ($errorflag,$pts,$wgt) = 
			    &saveHandGrade($request,$symb,$collaborator,$udom,$ctr,
					   $env{'form.unamedom'.$ctr},$part);
			if ($errorflag eq 'not_allowed') {
			    $request->print("<span class=\"LC_error\">".&mt('Not allowed to modify grades for [_1]',"$collaborator:$udom")."</span>");
			    next;
			} else {
			    if ($message ne '') {
                                $encrypturl=
                                  &Apache::lonnet::EXT('resource.0.encrypturl',
                                                       $symb,$udom,$collaborator);
                                if ($encrypturl =~ /^yes$/i) {
                                    $baseurl = &Apache::lonenc::encrypted($feedurl,1);
                                    $showsymb = &Apache::lonenc::encrypted($symb,1);
                                } else {
                                    $baseurl = $feedurl;
                                    $showsymb = $symb;
                                }
                                if ($env{'form.withgrades'.$ctr}) {
                                    $messagetail = " for <a href=\"".
                                    $baseurl."?symb=$showsymb\">$env{'form.probTitle'}</a>";

                                }
				$msgstatus = 
                                    &Apache::lonmsg::user_normal_msg($collaborator,$udom,$subject,$message.$messagetail,undef,$baseurl,undef,undef,undef,$showsymb,$restitle);
			    }
			}
		    }
		}
	    }
	    $ctr++;
	}
    }

    if ($env{'form.handgrade'} eq 'yes') {
	# Keywords sorted in alphabatical order
	my $loginuser = $env{'user.name'}.':'.$env{'user.domain'};
	my %keyhash = ();
	$env{'form.keywords'}           =~ s/,\s{0,}|\s+/ /g;
	$env{'form.keywords'}           =~ s/^\s+|\s+$//;
	my (@keywords) = sort(split(/\s+/,$env{'form.keywords'}));
	$env{'form.keywords'} = join(' ',@keywords);
	$keyhash{$symb.'_keywords'}     = $env{'form.keywords'};
	$keyhash{$symb.'_subject'}      = $env{'form.msgsub'};
	$keyhash{$loginuser.'_kwclr'}   = $env{'form.kwclr'};
	$keyhash{$loginuser.'_kwsize'}  = $env{'form.kwsize'};
	$keyhash{$loginuser.'_kwstyle'} = $env{'form.kwstyle'};

	# message center - Order of message gets changed. Blank line is eliminated.
	# New messages are saved in env for the next student.
	# All messages are saved in nohist_handgrade.db
	my ($ctr,$idx) = (1,1);
	while ($ctr <= $env{'form.savemsgN'}) {
	    if ($env{'form.savemsg'.$ctr} ne '') {
		$keyhash{$symb.'_savemsg'.$idx} = $env{'form.savemsg'.$ctr};
		$idx++;
	    }
	    $ctr++;
	}
	$ctr = 0;
	while ($ctr < $ngrade) {
	    if ($env{'form.newmsg'.$ctr} ne '') {
		$keyhash{$symb.'_savemsg'.$idx} = $env{'form.newmsg'.$ctr};
		$env{'form.savemsg'.$idx} = $env{'form.newmsg'.$ctr};
		$idx++;
	    }
	    $ctr++;
	}
	$env{'form.savemsgN'} = --$idx;
	$keyhash{$symb.'_savemsgN'} = $env{'form.savemsgN'};
	my $putresult = &Apache::lonnet::put
	    ('nohist_handgrade',\%keyhash,$cdom,$cnum);
    }
    # Called by Save & Refresh from Highlight Attribute Window
    my (undef,undef,$fullname) = &getclasslist($env{'form.section'},'1');
    if ($env{'form.refresh'} eq 'on') {
	my ($ctr,$total) = (0,0);
	while ($ctr < $ngrade) {
	    $total++ if  $env{'form.unamedom'.$ctr} ne '';
	    $ctr++;
	}
	$env{'form.NTSTU'}=$ngrade;
	$ctr = 0;
	while ($ctr < $total) {
	    my $processUser = $env{'form.unamedom'.$ctr};
	    ($env{'form.student'},$env{'form.userdom'}) = split(/:/,$processUser);
	    $env{'form.fullname'} = $$fullname{$processUser};
	    &submission($request,$ctr,$total-1);
	    $ctr++;
	}
	return '';
    }

# Go directly to grade student - from submission or link from chart page
    if ($button eq 'Grade Student') {
	(undef,undef,$env{'form.handgrade'},undef,undef) = &showResourceInfo($symb);
	my $processUser = $env{'form.unamedom'.$env{'form.studentNo'}};
	($env{'form.student'},$env{'form.userdom'}) = split(/:/,$processUser);
	$env{'form.fullname'} = $$fullname{$processUser};
	&submission($request,0,0);
	return '';
    }

    # Get the next/previous one or group of students
    my $firststu = $env{'form.unamedom0'};
    my $laststu = $env{'form.unamedom'.($ngrade-1)};
    my $ctr = 2;
    while ($laststu eq '') {
	$laststu  = $env{'form.unamedom'.($ngrade-$ctr)};
	$ctr++;
	$laststu = $firststu if ($ctr > $ngrade);
    }

    my (@parsedlist,@nextlist);
    my ($nextflg) = 0;
    foreach (sort 
	     {
		 if (lc($$fullname{$a}) ne lc($$fullname{$b})) {
		     return (lc($$fullname{$a}) cmp lc($$fullname{$b}));
		 }
		 return $a cmp $b;
	     } (keys(%$fullname))) {
	if ($nextflg == 1 && $button =~ /Next$/) {
	    push @parsedlist,$_;
	}
	$nextflg = 1 if ($_ eq $laststu);
	if ($button eq 'Previous') {
	    last if ($_ eq $firststu);
	    push @parsedlist,$_;
	}
    }
    $ctr = 0;
    @parsedlist = reverse @parsedlist if ($button eq 'Previous');
    my ($partlist) = &response_type($symb);
    foreach my $student (@parsedlist) {
	my $submitonly=$env{'form.submitonly'};
	my ($uname,$udom) = split(/:/,$student);
	
	if ($submitonly eq 'queued') {
	    my %queue_status = 
		&Apache::bridgetask::get_student_status($symb,$cdom,$cnum,
							$udom,$uname);
	    next if (!defined($queue_status{'gradingqueue'}));
	}

	if ($submitonly =~ /^(yes|graded|incorrect)$/) {
#	    my %record = &Apache::lonnet::restore($symb,$env{'request.course.id'},$udom,$uname);
	    my %status=&student_gradeStatus($symb,$udom,$uname,$partlist);
	    my $submitted = 0;
	    my $ungraded = 0;
	    my $incorrect = 0;
	    foreach (keys(%status)) {
		$submitted = 1 if ($status{$_} ne 'nothing');
		$ungraded = 1 if ($status{$_} =~ /^ungraded/);
		$incorrect = 1 if ($status{$_} =~ /^incorrect/);
		my ($foo,$partid,$foo1) = split(/\./,$_);
		if ($status{'resource.'.$partid.'.submitted_by'} ne '') {
		    $submitted = 0;
		}
	    }
	    next if (!$submitted && ($submitonly eq 'yes' ||
				     $submitonly eq 'incorrect' ||
				     $submitonly eq 'graded'));
	    next if (!$ungraded && ($submitonly eq 'graded'));
	    next if (!$incorrect && $submitonly eq 'incorrect');
	}
	push @nextlist,$student if ($ctr < $ntstu);
	last if ($ctr == $ntstu);
	$ctr++;
    }

    $ctr = 0;
    my $total = scalar(@nextlist)-1;

    foreach (sort @nextlist) {
	my ($uname,$udom,$submitter) = split(/:/);
	$env{'form.student'}  = $uname;
	$env{'form.userdom'}  = $udom;
	$env{'form.fullname'} = $$fullname{$_};
	&submission($request,$ctr,$total);
	$ctr++;
    }
    if ($total < 0) {
	my $the_end = '<h3><span class="LC_info">LON-CAPA User Message</span></h3><br />'."\n";
	$the_end.='<b>Message: </b> No more students for this section or class.<br /><br />'."\n";
	$the_end.='Click on the button below to return to the grading menu.<br /><br />'."\n";
	$the_end.=&show_grading_menu_form($symb);
	$request->print($the_end);
    }
    return '';
}

#---- Save the score and award for each student, if changed
sub saveHandGrade {
    my ($request,$symb,$stuname,$domain,$newflg,$submitter,$part) = @_;
    my @version_parts;
    my $usec = &Apache::lonnet::getsection($domain,$stuname,
					   $env{'request.course.id'});
    if (!&canmodify($usec)) { return('not_allowed'); }
    my %record = &Apache::lonnet::restore($symb,$env{'request.course.id'},$domain,$stuname);
    my @parts_graded;
    my %newrecord  = ();
    my ($pts,$wgt) = ('','');
    my %aggregate = ();
    my $aggregateflag = 0;
    my @parts = split(/:/,$env{'form.partlist'.$newflg});
    foreach my $new_part (@parts) {
	#collaborator ($submi may vary for different parts
	if ($submitter && $new_part ne $part) { next; }
	my $dropMenu = $env{'form.GD_SEL'.$newflg.'_'.$new_part};
	if ($dropMenu eq 'excused') {
	    if ($record{'resource.'.$new_part.'.solved'} ne 'excused') {
		$newrecord{'resource.'.$new_part.'.solved'} = 'excused';
		if (exists($record{'resource.'.$new_part.'.awarded'})) {
		    $newrecord{'resource.'.$new_part.'.awarded'} = '';
		}
	        $newrecord{'resource.'.$new_part.'.regrader'}="$env{'user.name'}:$env{'user.domain'}";
	    }
	} elsif ($dropMenu eq 'reset status'
		 && exists($record{'resource.'.$new_part.'.solved'})) { #don't bother if no old records -> no attempts
	    foreach my $key (keys (%record)) {
		if ($key=~/^resource\.\Q$new_part\E\./) { $newrecord{$key} = ''; }
	    }
	    $newrecord{'resource.'.$new_part.'.regrader'}=
		"$env{'user.name'}:$env{'user.domain'}";
            my $totaltries = $record{'resource.'.$part.'.tries'};

            my %last_resets = &get_last_resets($symb,$env{'request.course.id'},
					       [$new_part]);
            my $aggtries =$totaltries;
            if ($last_resets{$new_part}) {
                $aggtries = &get_num_tries(\%record,$last_resets{$new_part},
					   $new_part);
            }

            my $solvedstatus = $record{'resource.'.$new_part.'.solved'};
            if ($aggtries > 0) {
                &decrement_aggs($symb,$new_part,\%aggregate,$aggtries,$totaltries,$solvedstatus);
                $aggregateflag = 1;
            }
	} elsif ($dropMenu eq '') {
	    $pts = ($env{'form.GD_BOX'.$newflg.'_'.$new_part} ne '' ? 
		    $env{'form.GD_BOX'.$newflg.'_'.$new_part} : 
		    $env{'form.RADVAL'.$newflg.'_'.$new_part});
	    if ($pts eq '' && $env{'form.GD_SEL'.$newflg.'_'.$new_part} eq '') {
		next;
	    }
	    $wgt = $env{'form.WGT'.$newflg.'_'.$new_part} eq '' ? 1 : 
		$env{'form.WGT'.$newflg.'_'.$new_part};
	    my $partial= $pts/$wgt;
	    if ($partial eq $record{'resource.'.$new_part.'.awarded'}) {
		#do not update score for part if not changed.
                &handback_files($request,$symb,$stuname,$domain,$newflg,$new_part,\%newrecord);
		next;
	    } else {
	        push @parts_graded, $new_part;
	    }
	    if ($record{'resource.'.$new_part.'.awarded'} ne $partial) {
		$newrecord{'resource.'.$new_part.'.awarded'}  = $partial;
	    }
	    my $reckey = 'resource.'.$new_part.'.solved';
	    if ($partial == 0) {
		if ($record{$reckey} ne 'incorrect_by_override') {
		    $newrecord{$reckey} = 'incorrect_by_override';
		}
	    } else {
		if ($record{$reckey} ne 'correct_by_override') {
		    $newrecord{$reckey} = 'correct_by_override';
		}
	    }	    
	    if ($submitter && 
		($record{'resource.'.$new_part.'.submitted_by'} ne $submitter)) {
		$newrecord{'resource.'.$new_part.'.submitted_by'} = $submitter;
	    }
	    $newrecord{'resource.'.$new_part.'.regrader'}=
		"$env{'user.name'}:$env{'user.domain'}";
	}
	# unless problem has been graded, set flag to version the submitted files
	unless ($record{'resource.'.$new_part.'.solved'} =~ /^correct_/  || 
	        $record{'resource.'.$new_part.'.solved'} eq 'incorrect_by_override' ||
	        $dropMenu eq 'reset status')
	   {
	    push (@version_parts,$new_part);
	}
    }
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};

    if (%newrecord) {
        if (@version_parts) {
            my @changed_keys = &version_portfiles(\%record, \@parts_graded, 
                                $env{'request.course.id'}, $symb, $domain, $stuname, \@version_parts);
	    @newrecord{@changed_keys} = @record{@changed_keys};
	    foreach my $new_part (@version_parts) {
		&handback_files($request,$symb,$stuname,$domain,$newflg,
				$new_part,\%newrecord);
	    }
        }
	&Apache::lonnet::cstore(\%newrecord,$symb,
				$env{'request.course.id'},$domain,$stuname);
	&check_and_remove_from_queue(\@parts,\%record,\%newrecord,$symb,
				     $cdom,$cnum,$domain,$stuname);
    }
    if ($aggregateflag) {
        &Apache::lonnet::cinc('nohist_resourcetracker',\%aggregate,
			      $cdom,$cnum);
    }
    return ('',$pts,$wgt);
}

sub check_and_remove_from_queue {
    my ($parts,$record,$newrecord,$symb,$cdom,$cnum,$domain,$stuname) = @_;
    my @ungraded_parts;
    foreach my $part (@{$parts}) {
	if (    $record->{   'resource.'.$part.'.awarded'} eq ''
	     && $record->{   'resource.'.$part.'.solved' } ne 'excused'
	     && $newrecord->{'resource.'.$part.'.awarded'} eq ''
	     && $newrecord->{'resource.'.$part.'.solved' } ne 'excused'
		) {
	    push(@ungraded_parts, $part);
	}
    }
    if ( !@ungraded_parts ) {
	&Apache::bridgetask::remove_from_queue('gradingqueue',$symb,$cdom,
					       $cnum,$domain,$stuname);
    }
}

sub handback_files {
    my ($request,$symb,$stuname,$domain,$newflg,$new_part,$newrecord) = @_;
    my $portfolio_root = &propath($domain,$stuname).'/userfiles/portfolio';
    my ($partlist,$handgrade,$responseType) = &response_type($symb);

    my @part_response_id = &flatten_responseType($responseType);
    foreach my $part_response_id (@part_response_id) {
    	my ($part_id,$resp_id) = @{ $part_response_id };
	my $part_resp = join('_',@{ $part_response_id });
            if (($env{'form.'.$newflg.'_'.$part_resp.'_returndoc1'}) && ($new_part == $part_id)) {
                # if multiple files are uploaded names will be 'returndoc2','returndoc3'
                my $file_counter = 1;
		my $file_msg;
                while ($env{'form.'.$newflg.'_'.$part_resp.'_returndoc'.$file_counter}) {
                    my $fname=$env{'form.'.$newflg.'_'.$part_resp.'_returndoc'.$file_counter.'.filename'};
                    my ($directory,$answer_file) = 
                        ($env{'form.'.$newflg.'_'.$part_resp.'_origdoc'.$file_counter} =~ /^(.*?)([^\/]*)$/);
                    my ($answer_name,$answer_ver,$answer_ext) =
		        &file_name_version_ext($answer_file);
		    my ($portfolio_path) = ($directory =~ /^.+$stuname\/portfolio(.*)/);
		    my @dir_list = &Apache::lonnet::dirlist($portfolio_path,$domain,$stuname,$portfolio_root);
		    my $version = &get_next_version($answer_name, $answer_ext, \@dir_list);
                    # fix file name
                    my ($save_file_name) = (($directory.$answer_name.".$version.".$answer_ext) =~ /^.+\/${stuname}\/(.*)/);
                    my $result=&Apache::lonnet::finishuserfileupload($stuname,$domain,
            	                                $newflg.'_'.$part_resp.'_returndoc'.$file_counter,
            	                                $save_file_name);
                    if ($result !~ m|^/uploaded/|) {
                        $request->print('<span class="LC_error">An errror occured ('.$result.
                        ') while trying to upload '.$newflg.'_'.$part_resp.'_returndoc'.$file_counter.'</span><br />');
                    } else {
                        # mark the file as read only
                        my @files = ($save_file_name);
                        my @what = ($symb,$env{'request.course.id'},'handback');
                        &Apache::lonnet::mark_as_readonly($domain,$stuname,\@files,\@what);
			if (exists($$newrecord{"resource.$new_part.$resp_id.handback"})) {
			    $$newrecord{"resource.$new_part.$resp_id.handback"}.=',';
			}
                        $$newrecord{"resource.$new_part.$resp_id.handback"} .= $save_file_name;
			$file_msg.= "\n".'<br /><span class="LC_filename"><a href="/uploaded/'."$domain/$stuname/".$save_file_name.'">'.$save_file_name."</a></span><br />";

                    }
                    $request->print("<br />".$fname." will be the uploaded file name");
                    $request->print(" ".$env{'form.'.$newflg.'_'.$part_resp.'_origdoc'.$file_counter});
                    $file_counter++;
                }
		my $subject = "File Handed Back by Instructor ";
		my $message = "A file has been returned that was originally submitted in reponse to: <br />";
		$message .= "<strong>".&Apache::lonnet::gettitle($symb)."</strong><br />";
		$message .= ' The returned file(s) are named: '. $file_msg;
		$message .= " and can be found in your portfolio space.";
		my $url = (&Apache::lonnet::decode_symb($symb))[2];
		my $feedurl = &Apache::lonnet::clutter($url);
                my $encrypturl=&Apache::lonnet::EXT('resource.0.encrypturl',
                                                    $symb,$domain,$stuname);
                my ($baseurl,$showsymb);
                if ($encrypturl =~ /^yes$/i) {
                    $baseurl = &Apache::lonenc::encrypted($feedurl,1);
                    $showsymb = &Apache::lonenc::encrypted($symb,1);
                } else {
                    $baseurl = $feedurl;
                    $showsymb = $symb;
                }
                my $restitle = &Apache::lonnet::gettitle($symb);
		my $msgstatus = 
                   &Apache::lonmsg::user_normal_msg($stuname,$domain,$subject.
			 ' (File Returned) ['.$restitle.']',$message,undef,
                         $baseurl,undef,undef,undef,$showsymb,$restitle);
            }
        }
    return;
}

sub get_submitted_files {
    my ($udom,$uname,$partid,$respid,$record) = @_;
    my @files;
    if ($$record{"resource.$partid.$respid.portfiles"}) {
        my $file_url = '/uploaded/'.$udom.'/'.$uname.'/portfolio';
        foreach my $file (split(',',$$record{"resource.$partid.$respid.portfiles"})) {
    	    push(@files,$file_url.$file);
        }
    }
    if ($$record{"resource.$partid.$respid.uploadedurl"}) {
        push(@files,$$record{"resource.$partid.$respid.uploadedurl"});
    }
    return (\@files);
}

# ----------- Provides number of tries since last reset.
sub get_num_tries {
    my ($record,$last_reset,$part) = @_;
    my $timestamp = '';
    my $num_tries = 0;
    if ($$record{'version'}) {
        for (my $version=$$record{'version'};$version>=1;$version--) {
            if (exists($$record{$version.':resource.'.$part.'.solved'})) {
                $timestamp = $$record{$version.':timestamp'};
                if ($timestamp > $last_reset) {
                    $num_tries ++;
                } else {
                    last;
                }
            }
        }
    }
    return $num_tries;
}

# ----------- Determine decrements required in aggregate totals 
sub decrement_aggs {
    my ($symb,$part,$aggregate,$aggtries,$totaltries,$solvedstatus) = @_;
    my %decrement = (
                        attempts => 0,
                        users => 0,
                        correct => 0
                    );
    $decrement{'attempts'} = $aggtries;
    if ($solvedstatus =~ /^correct/) {
        $decrement{'correct'} = 1;
    }
    if ($aggtries == $totaltries) {
        $decrement{'users'} = 1;
    }
    foreach my $type (keys (%decrement)) {
        $$aggregate{$symb."\0".$part."\0".$type} = -$decrement{$type};
    }
    return;
}

# ----------- Determine timestamps for last reset of aggregate totals for parts  
sub get_last_resets {
    my ($symb,$courseid,$partids) =@_;
    my %last_resets;
    my $cdom = $env{'course.'.$courseid.'.domain'};
    my $cname = $env{'course.'.$courseid.'.num'};
    my @keys;
    foreach my $part (@{$partids}) {
	push(@keys,"$symb\0$part\0resettime");
    }
    my %results=&Apache::lonnet::get('nohist_resourcetracker',\@keys,
				     $cdom,$cname);
    foreach my $part (@{$partids}) {
	$last_resets{$part}=$results{"$symb\0$part\0resettime"};
    }
    return %last_resets;
}

# ----------- Handles creating versions for portfolio files as answers
sub version_portfiles {
    my ($record, $parts_graded, $courseid, $symb, $domain, $stu_name, $v_flag) = @_;
    my $version_parts = join('|',@$v_flag);
    my @returned_keys;
    my $parts = join('|', @$parts_graded);
    my $portfolio_root = &propath($domain,$stu_name).
	'/userfiles/portfolio';
    foreach my $key (keys(%$record)) {
        my $new_portfiles;
        if ($key =~ /^resource\.($version_parts)\./ && $key =~ /\.portfiles$/ ) {
            my @versioned_portfiles;
            my @portfiles = split(/\s*,\s*/,$$record{$key});
            foreach my $file (@portfiles) {
                &Apache::lonnet::unmark_as_readonly($domain,$stu_name,[$symb,$env{'request.course.id'}],$file);
                my ($directory,$answer_file) =($file =~ /^(.*?)([^\/]*)$/);
		my ($answer_name,$answer_ver,$answer_ext) =
		    &file_name_version_ext($answer_file);
                my @dir_list = &Apache::lonnet::dirlist($directory,$domain,$stu_name,$portfolio_root);
                my $version = &get_next_version($answer_name, $answer_ext, \@dir_list);
                my $new_answer = &version_selected_portfile($domain, $stu_name, $directory, $answer_file, $version);
                if ($new_answer ne 'problem getting file') {
                    push(@versioned_portfiles, $directory.$new_answer);
                    &Apache::lonnet::mark_as_readonly($domain,$stu_name,
                        [$directory.$new_answer],
                        [$symb,$env{'request.course.id'},'graded']);
                }
            }
            $$record{$key} = join(',',@versioned_portfiles);
            push(@returned_keys,$key);
        }
    } 
    return (@returned_keys);   
}

sub get_next_version {
    my ($answer_name, $answer_ext, $dir_list) = @_;
    my $version;
    foreach my $row (@$dir_list) {
        my ($file) = split(/\&/,$row,2);
        my ($file_name,$file_version,$file_ext) =
	    &file_name_version_ext($file);
        if (($file_name eq $answer_name) && 
	    ($file_ext eq $answer_ext)) {
                # gets here if filename and extension match, regardless of version
                if ($file_version ne '') {
                # a versioned file is found  so save it for later
                if ($file_version > $version) {
		    $version = $file_version;
	        }
            }
        }
    } 
    $version ++;
    return($version);
}

sub version_selected_portfile {
    my ($domain,$stu_name,$directory,$file_name,$version) = @_;
    my ($answer_name,$answer_ver,$answer_ext) =
        &file_name_version_ext($file_name);
    my $new_answer;
    $env{'form.copy'} = &Apache::lonnet::getfile("/uploaded/$domain/$stu_name/portfolio$directory$file_name");
    if($env{'form.copy'} eq '-1') {
        &Apache::lonnet::logthis('problem getting file '.$file_name);
        $new_answer = 'problem getting file';
    } else {
        $new_answer = $answer_name.'.'.$version.'.'.$answer_ext;
        my $copy_result = &Apache::lonnet::finishuserfileupload(
                            $stu_name,$domain,'copy',
		        '/portfolio'.$directory.$new_answer);
    }    
    return ($new_answer);
}

sub file_name_version_ext {
    my ($file)=@_;
    my @file_parts = split(/\./, $file);
    my ($name,$version,$ext);
    if (@file_parts > 1) {
	$ext=pop(@file_parts);
	if (@file_parts > 1 && $file_parts[-1] =~ /^\d+$/) {
	    $version=pop(@file_parts);
	}
	$name=join('.',@file_parts);
    } else {
	$name=join('.',@file_parts);
    }
    return($name,$version,$ext);
}

#--------------------------------------------------------------------------------------
#
#-------------------------- Next few routines handles grading by section or whole class
#
#--- Javascript to handle grading by section or whole class
sub viewgrades_js {
    my ($request) = shift;

    $request->print(<<VIEWJAVASCRIPT);
<script type="text/javascript" language="javascript">
   function writePoint(partid,weight,point) {
	var radioButton = document.classgrade["RADVAL_"+partid];
	var textbox = document.classgrade["TEXTVAL_"+partid];
	if (point == "textval") {
	    point = document.classgrade["TEXTVAL_"+partid].value;
	    if (isNaN(point) || parseFloat(point) < 0) {
		alert("A number equal or greater than 0 is expected. Entered value = "+parseFloat(point));
		var resetbox = false;
		for (var i=0; i<radioButton.length; i++) {
		    if (radioButton[i].checked) {
			textbox.value = i;
			resetbox = true;
		    }
		}
		if (!resetbox) {
		    textbox.value = "";
		}
		return;
	    }
	    if (parseFloat(point) > parseFloat(weight)) {
		var resp = confirm("You entered a value ("+parseFloat(point)+
				   ") greater than the weight for the part. Accept?");
		if (resp == false) {
		    textbox.value = "";
		    return;
		}
	    }
	    for (var i=0; i<radioButton.length; i++) {
		radioButton[i].checked=false;
		if (parseFloat(point) == i) {
		    radioButton[i].checked=true;
		}
	    }

	} else {
	    textbox.value = parseFloat(point);
	}
	for (i=0;i<document.classgrade.total.value;i++) {
	    var user = document.classgrade["ctr"+i].value;
	    user = user.replace(new RegExp(':', 'g'),"_");
	    var scorename = document.classgrade["GD_"+user+"_"+partid+"_awarded"];
	    var saveval   = document.classgrade["GD_"+user+"_"+partid+"_solved_s"].value;
	    var selname   = document.classgrade["GD_"+user+"_"+partid+"_solved"];
	    if (saveval != "correct") {
		scorename.value = point;
		if (selname[0].selected != true) {
		    selname[0].selected = true;
		}
	    }
	}
	document.classgrade["SELVAL_"+partid][0].selected = true;
    }

    function writeRadText(partid,weight) {
	var selval   = document.classgrade["SELVAL_"+partid];
	var radioButton = document.classgrade["RADVAL_"+partid];
        var override = document.classgrade["FORCE_"+partid].checked;
	var textbox = document.classgrade["TEXTVAL_"+partid];
	if (selval[1].selected || selval[2].selected) {
	    for (var i=0; i<radioButton.length; i++) {
		radioButton[i].checked=false;

	    }
	    textbox.value = "";

	    for (i=0;i<document.classgrade.total.value;i++) {
		var user = document.classgrade["ctr"+i].value;
		user = user.replace(new RegExp(':', 'g'),"_");
		var scorename = document.classgrade["GD_"+user+"_"+partid+"_awarded"];
		var saveval   = document.classgrade["GD_"+user+"_"+partid+"_solved_s"].value;
		var selname   = document.classgrade["GD_"+user+"_"+partid+"_solved"];
		if ((saveval != "correct") || override) {
		    scorename.value = "";
		    if (selval[1].selected) {
			selname[1].selected = true;
		    } else {
			selname[2].selected = true;
			if (Number(document.classgrade["GD_"+user+"_"+partid+"_tries"].value)) 
			{document.classgrade["GD_"+user+"_"+partid+"_tries"].value = '0';}
		    }
		}
	    }
	} else {
	    for (i=0;i<document.classgrade.total.value;i++) {
		var user = document.classgrade["ctr"+i].value;
		user = user.replace(new RegExp(':', 'g'),"_");
		var scorename = document.classgrade["GD_"+user+"_"+partid+"_awarded"];
		var saveval   = document.classgrade["GD_"+user+"_"+partid+"_solved_s"].value;
		var selname   = document.classgrade["GD_"+user+"_"+partid+"_solved"];
		if ((saveval != "correct") || override) {
		    scorename.value = document.classgrade["GD_"+user+"_"+partid+"_awarded_s"].value;
		    selname[0].selected = true;
		}
	    }
	}	    
    }

    function changeSelect(partid,user) {
	var selval = document.classgrade["GD_"+user+'_'+partid+"_solved"];
	var textbox = document.classgrade["GD_"+user+'_'+partid+"_awarded"];
	var point  = textbox.value;
	var weight = document.classgrade["weight_"+partid].value;

	if (isNaN(point) || parseFloat(point) < 0) {
	    alert("A number equal or greater than 0 is expected. Entered value = "+parseFloat(point));
	    textbox.value = "";
	    return;
	}
	if (parseFloat(point) > parseFloat(weight)) {
	    var resp = confirm("You entered a value ("+parseFloat(point)+
			       ") greater than the weight of the part. Accept?");
	    if (resp == false) {
		textbox.value = "";
		return;
	    }
	}
	selval[0].selected = true;
    }

    function changeOneScore(partid,user) {
	var selval = document.classgrade["GD_"+user+'_'+partid+"_solved"];
	if (selval[1].selected || selval[2].selected) {
	    document.classgrade["GD_"+user+'_'+partid+"_awarded"].value = "";
	    if (selval[2].selected) {
		document.classgrade["GD_"+user+'_'+partid+"_tries"].value = "0";
	    }
        }
    }

    function resetEntry(numpart) {
	for (ctpart=0;ctpart<numpart;ctpart++) {
	    var partid = document.classgrade["partid_"+ctpart].value;
	    var radioButton = document.classgrade["RADVAL_"+partid];
	    var textbox = document.classgrade["TEXTVAL_"+partid];
	    var selval  = document.classgrade["SELVAL_"+partid];
	    for (var i=0; i<radioButton.length; i++) {
		radioButton[i].checked=false;

	    }
	    textbox.value = "";
	    selval[0].selected = true;

	    for (i=0;i<document.classgrade.total.value;i++) {
		var user = document.classgrade["ctr"+i].value;
		user = user.replace(new RegExp(':', 'g'),"_");
		var resetscore = document.classgrade["GD_"+user+"_"+partid+"_awarded"];
		resetscore.value = document.classgrade["GD_"+user+"_"+partid+"_awarded_s"].value;
		var resettries = document.classgrade["GD_"+user+"_"+partid+"_tries"];
		resettries.value = document.classgrade["GD_"+user+"_"+partid+"_tries_s"].value;
		var saveselval   = document.classgrade["GD_"+user+"_"+partid+"_solved_s"].value;
		var selname   = document.classgrade["GD_"+user+"_"+partid+"_solved"];
		if (saveselval == "excused") {
		    if (selname[1].selected == false) { selname[1].selected = true;}
		} else {
		    if (selname[0].selected == false) {selname[0].selected = true};
		}
	    }
	}
    }

</script>
VIEWJAVASCRIPT
}

#--- show scores for a section or whole class w/ option to change/update a score
sub viewgrades {
    my ($request) = shift;
    &viewgrades_js($request);

    my ($symb) = &get_symb($request);
    #need to make sure we have the correct data for later EXT calls, 
    #thus invalidate the cache
    &Apache::lonnet::devalidatecourseresdata(
                 $env{'course.'.$env{'request.course.id'}.'.num'},
                 $env{'course.'.$env{'request.course.id'}.'.domain'});
    &Apache::lonnet::clear_EXT_cache_status();

    my $result='<h3><span class="LC_info">'.&mt('Manual Grading').'</span></h3>';
    $result.='<h4><b>Current Resource: </b>'.$env{'form.probTitle'}.'</h4>'."\n";

    #view individual student submission form - called using Javascript viewOneStudent
    $result.=&jscriptNform($symb);

    #beginning of class grading form
    $result.= '<form action="/adm/grades" method="post" name="classgrade">'."\n".
	'<input type="hidden" name="symb"    value="'.$symb.'" />'."\n".
	'<input type="hidden" name="command" value="editgrades" />'."\n".
	'<input type="hidden" name="section" value="'.$env{'form.section'}.'" />'."\n".
	'<input type="hidden" name="saveState" value="'.$env{'form.saveState'}.'" />'."\n".
	'<input type="hidden" name="Status" value="'.$env{'form.Status'}.'" />'."\n".
	'<input type="hidden" name="probTitle" value="'.$env{'form.probTitle'}.'" />'."\n";

    my $sectionClass;
    if ($env{'form.section'} eq 'all') {
	$sectionClass='Class </h3>';
    } elsif ($env{'form.section'} eq 'none') {
	$sectionClass='Students in no Section </h3>';
    } else {
	$sectionClass='Students in Section '.$env{'form.section'}.'</h3>';
    }
    $result.='<h3>Assign Common Grade To '.$sectionClass;
    $result.= '<table border=0><tr><td bgcolor="#777777">'."\n".
	'<table border=0><tr bgcolor="#ffffdd"><td>';
    #radio buttons/text box for assigning points for a section or class.
    #handles different parts of a problem
    my ($partlist,$handgrade,$responseType) = &response_type($symb);
    my %weight = ();
    my $ctsparts = 0;
    $result.='<table border="0">';
    my %seen = ();
    my @part_response_id = &flatten_responseType($responseType);
    foreach my $part_response_id (@part_response_id) {
    	my ($partid,$respid) = @{ $part_response_id };
	my $part_resp = join('_',@{ $part_response_id });
	next if $seen{$partid};
	$seen{$partid}++;
	my $handgrade=$$handgrade{$part_resp};
	my $wgt = &Apache::lonnet::EXT('resource.'.$partid.'.weight',$symb);
	$weight{$partid} = $wgt eq '' ? '1' : $wgt;

	$result.='<input type="hidden" name="partid_'.
	    $ctsparts.'" value="'.$partid.'" />'."\n";
	$result.='<input type="hidden" name="weight_'.
	    $partid.'" value="'.$weight{$partid}.'" />'."\n";
	my $display_part=&get_display_part($partid,$symb);
	$result.='<tr><td><b>Part:</b> '.$display_part.'&nbsp; &nbsp;<b>Point:</b> </td><td>';
	$result.='<table border="0"><tr>';  
	my $ctr = 0;
	while ($ctr<=$weight{$partid}) { # display radio buttons in a nice table 10 across
	    $result.= '<td><label><input type="radio" name="RADVAL_'.$partid.'" '.
		'onclick="javascript:writePoint(\''.$partid.'\','.$weight{$partid}.
		','.$ctr.')" />'.$ctr."</label></td>\n";
	    $result.=(($ctr+1)%10 == 0 ? '</tr><tr>' : '');
	    $ctr++;
	}
	$result.='</tr></table>';
	$result.= '</td><td><b> or </b><input type="text" name="TEXTVAL_'.
	    $partid.'" size="4" '.'onChange="javascript:writePoint(\''.
		$partid.'\','.$weight{$partid}.',\'textval\')" /> /'.
	    $weight{$partid}.' (problem weight)</td>'."\n";
	$result.= '</td><td><select name="SELVAL_'.$partid.'"'.
	    'onChange="javascript:writeRadText(\''.$partid.'\','.
		$weight{$partid}.')"> '.
	    '<option selected="on"> </option>'.
	    '<option>excused</option>'.
	    '<option>reset status</option></select></td>'.
            '<td><label><input type="checkbox" name="FORCE_'.$partid.'" /> Override "Correct"</label></td></tr>'."\n";
	$ctsparts++;
    }
    $result.='</table>'.'</td></tr></table>'.'</td></tr></table>'."\n".
	'<input type="hidden" name="totalparts" value="'.$ctsparts.'" />';
    $result.='<input type="button" value="Revert to Default" '.
	'onClick="javascript:resetEntry('.$ctsparts.');" TARGET=_self>';

    #table listing all the students in a section/class
    #header of table
    $result.= '<h3>Assign Grade to Specific Students in '.$sectionClass;
    $result.= '<table border=0><tr><td bgcolor="#777777">'."\n".
	'<table border=0><tr bgcolor="#deffff"><td>&nbsp;<b>No.</b>&nbsp;</td>'.
	'<td>'.&nameUserString('header')."</td>\n";
    my (@parts) = sort(&getpartlist($symb));
    my (undef,undef,$url)=&Apache::lonnet::decode_symb($symb);
    my @partids = ();
    foreach my $part (@parts) {
	my $display=&Apache::lonnet::metadata($url,$part.'.display');
	$display =~ s|^Number of Attempts|Tries<br />|; # makes the column narrower
	if  (!$display) { $display = &Apache::lonnet::metadata($url,$part.'.name'); }
	my ($partid) = &split_part_type($part);
        push(@partids, $partid);
	my $display_part=&get_display_part($partid,$symb);
	if ($display =~ /^Partial Credit Factor/) {
	    $result.='<td><b>Score Part:</b> '.$display_part.
		' <br /><b>(weight = '.$weight{$partid}.')</b></td>'."\n";
	    next;
	} else {
	    $display =~s/\[Part: \Q$partid\E\]/Part:<\/b> $display_part/;
	}
	$display =~ s|Problem Status|Grade Status<br />|;
	$result.='<td><b>'.$display.'</td>'."\n";
    }
    $result.='</tr>';

    my %last_resets = 
	&get_last_resets($symb,$env{'request.course.id'},\@partids);

    #get info for each student
    #list all the students - with points and grade status
    my (undef,undef,$fullname) = &getclasslist($env{'form.section'},'1');
    my $ctr = 0;
    foreach (sort 
	     {
		 if (lc($$fullname{$a}) ne lc($$fullname{$b})) {
		     return (lc($$fullname{$a}) cmp lc($$fullname{$b}));
		 }
		 return $a cmp $b;
	     } (keys(%$fullname))) {
	$ctr++;
	$result.=&viewstudentgrade($symb,$env{'request.course.id'},
				   $_,$$fullname{$_},\@parts,\%weight,$ctr,\%last_resets);
    }
    $result.='</table></td></tr></table>';
    $result.='<input type="hidden" name="total" value="'.$ctr.'" />'."\n";
    $result.='<input type="button" value="Save" '.
	'onClick="javascript:submit();" TARGET=_self /></form>'."\n";
    if (scalar(%$fullname) eq 0) {
	my $colspan=3+scalar(@parts);
	$result='<span class="LC_warning">There are no students in section "'.$env{'form.section'}.
	    '" with enrollment status "'.$env{'form.Status'}.'" to modify or grade.</span>';
    }
    $result.=&show_grading_menu_form($symb);
    return $result;
}

#--- call by previous routine to display each student
sub viewstudentgrade {
    my ($symb,$courseid,$student,$fullname,$parts,$weight,$ctr,$last_resets) = @_;
    my ($uname,$udom) = split(/:/,$student);
    my %record=&Apache::lonnet::restore($symb,$courseid,$udom,$uname);
    my %aggregates = (); 
    my $result='<tr bgcolor="#ffffdd"><td align="right">'.
	'<input type="hidden" name="ctr'.($ctr-1).'" value="'.$student.'" />'.
	"\n".$ctr.'&nbsp;</td><td>&nbsp;'.
	'<a href="javascript:viewOneStudent(\''.$uname.'\',\''.$udom.
	'\')"; TARGET=_self>'.$fullname.'</a> '.
	'<span class="LC_internal_info">('.$uname.($env{'user.domain'} eq $udom ? '' : ':'.$udom).')</span></td>'."\n";
    $student=~s/:/_/; # colon doen't work in javascript for names
    foreach my $apart (@$parts) {
	my ($part,$type) = &split_part_type($apart);
	my $score=$record{"resource.$part.$type"};
        $result.='<td align="center">';
        my ($aggtries,$totaltries);
        unless (exists($aggregates{$part})) {
	    $totaltries = $record{'resource.'.$part.'.tries'};

	    $aggtries = $totaltries;
            if ($$last_resets{$part}) {  
                $aggtries = &get_num_tries(\%record,$$last_resets{$part},
					   $part);
            }
            $result.='<input type="hidden" name="'.
                'GD_'.$student.'_'.$part.'_aggtries" value="'.$aggtries.'" />'."\n";
            $result.='<input type="hidden" name="'.
                'GD_'.$student.'_'.$part.'_totaltries" value="'.$totaltries.'" />'."\n";
            $aggregates{$part} = 1;
        }
	if ($type eq 'awarded') {
	    my $pts = $score eq '' ? '' : &compute_points($score,$$weight{$part});
	    $result.='<input type="hidden" name="'.
		'GD_'.$student.'_'.$part.'_awarded_s" value="'.$pts.'" />'."\n";
	    $result.='<input type="text" name="'.
		'GD_'.$student.'_'.$part.'_awarded" '.
		'onChange="javascript:changeSelect(\''.$part.'\',\''.$student.
		'\')" value="'.$pts.'" size="4" /></td>'."\n";
	} elsif ($type eq 'solved') {
	    my ($status,$foo)=split(/_/,$score,2);
	    $status = 'nothing' if ($status eq '');
	    $result.='<input type="hidden" name="'.'GD_'.$student.'_'.
		$part.'_solved_s" value="'.$status.'" />'."\n";
	    $result.='&nbsp;<select name="'.
		'GD_'.$student.'_'.$part.'_solved" '.
		'onChange="javascript:changeOneScore(\''.$part.'\',\''.$student.'\')" >'."\n";
	    $result.= (($status eq 'excused') ? '<option> </option><option selected="on">excused</option>' 
		: '<option selected="on"> </option><option>excused</option>')."\n";
	    $result.='<option>reset status</option>';
	    $result.="</select>&nbsp;</td>\n";
	} else {
	    $result.='<input type="hidden" name="'.
		'GD_'.$student.'_'.$part.'_'.$type.'_s" value="'.$score.'" />'.
		    "\n";
	    $result.='<input type="text" name="'.
		'GD_'.$student.'_'.$part.'_'.$type.'" '.
		'value="'.$score.'" size="4" /></td>'."\n";
	}
    }
    $result.='</tr>';
    return $result;
}

#--- change scores for all the students in a section/class
#    record does not get update if unchanged
sub editgrades {
    my ($request) = @_;

    my $symb=&get_symb($request);
    my $title='<h3><span class="LC_info">Current Grade Status</span></h3>';
    $title.='<h4><b>Current Resource: </b>'.$env{'form.probTitle'}.'</h4><br />'."\n";
    $title.='<h4><b>Section: </b>'.$env{'form.section'}.'</h4>'."\n";

    my $result= '<table border="0"><tr><td bgcolor="#777777">'."\n";
    $result.= '<table border="0"><tr bgcolor="#deffff">'.
	'<td rowspan=2 valign="center">&nbsp;<b>No.</b>&nbsp;</td>'.
	'<td rowspan=2 valign="center">'.&nameUserString('header')."</td>\n";

    my %scoreptr = (
		    'correct'  =>'correct_by_override',
		    'incorrect'=>'incorrect_by_override',
		    'excused'  =>'excused',
		    'ungraded' =>'ungraded_attempted',
		    'nothing'  => '',
		    );
    my ($classlist,undef,$fullname) = &getclasslist($env{'form.section'},'0');

    my (@partid);
    my %weight = ();
    my %columns = ();
    my ($i,$ctr,$count,$rec_update) = (0,0,0,0);

    my (@parts) = sort(&getpartlist($symb));
    my $header;
    while ($ctr < $env{'form.totalparts'}) {
	my $partid = $env{'form.partid_'.$ctr};
	push @partid,$partid;
	$weight{$partid} = $env{'form.weight_'.$partid};
	$ctr++;
    }
    my (undef,undef,$url) = &Apache::lonnet::decode_symb($symb);
    foreach my $partid (@partid) {
	$header .= '<td align="center">&nbsp;<b>Old Score</b>&nbsp;</td>'.
	    '<td align="center">&nbsp;<b>New Score</b>&nbsp;</td>';
	$columns{$partid}=2;
	foreach my $stores (@parts) {
	    my ($part,$type) = &split_part_type($stores);
	    if ($part !~ m/^\Q$partid\E/) { next;}
	    if ($type eq 'awarded' || $type eq 'solved') { next; }
	    my $display=&Apache::lonnet::metadata($url,$stores.'.display');
	    $display =~ s/\[Part: (\w)+\]//;
	    $display =~ s/Number of Attempts/Tries/;
	    $header .= '<td align="center">&nbsp;<b>Old '.$display.'</b>&nbsp;</td>'.
		'<td align="center">&nbsp;<b>New '.$display.'</b>&nbsp;</td>';
	    $columns{$partid}+=2;
	}
    }
    foreach my $partid (@partid) {
	my $display_part=&get_display_part($partid,$symb);
	$result .= '<td colspan="'.$columns{$partid}.
	    '" align="center"><b>Part:</b> '.$display_part.
	    ' (Weight = '.$weight{$partid}.')</td>';

    }
    $result .= '</tr><tr bgcolor="#deffff">';
    $result .= $header;
    $result .= '</tr>'."\n";
    my $noupdate;
    my ($updateCtr,$noupdateCtr) = (1,1);
    for ($i=0; $i<$env{'form.total'}; $i++) {
	my $line;
	my $user = $env{'form.ctr'.$i};
	my ($uname,$udom)=split(/:/,$user);
	my %newrecord;
	my $updateflag = 0;
	$line .= '<td>'.&nameUserString(undef,$$fullname{$user},$uname,$udom).'</td>';
	my $usec=$classlist->{"$uname:$udom"}[5];
	if (!&canmodify($usec)) {
	    my $numcols=scalar(@partid)*4+2;
	    $noupdate.=$line."<td colspan=\"$numcols\"><span class=\"LC_warning\">Not allowed to modify student</span></td></tr>";
	    next;
	}
        my %aggregate = ();
        my $aggregateflag = 0;
	$user=~s/:/_/; # colon doen't work in javascript for names
	foreach (@partid) {
	    my $old_aw    = $env{'form.GD_'.$user.'_'.$_.'_awarded_s'};
	    my $old_part_pcr = $old_aw/($weight{$_} ne '0' ? $weight{$_}:1);
	    my $old_part  = $old_aw eq '' ? '' : $old_part_pcr;
	    my $old_score = $scoreptr{$env{'form.GD_'.$user.'_'.$_.'_solved_s'}};
	    my $awarded   = $env{'form.GD_'.$user.'_'.$_.'_awarded'};
	    my $pcr       = $awarded/($weight{$_} ne '0' ? $weight{$_} : 1);
	    my $partial   = $awarded eq '' ? '' : $pcr;
	    my $score;
	    if ($partial eq '') {
		$score = $scoreptr{$env{'form.GD_'.$user.'_'.$_.'_solved_s'}};
	    } elsif ($partial > 0) {
		$score = 'correct_by_override';
	    } elsif ($partial == 0) {
		$score = 'incorrect_by_override';
	    }
	    my $dropMenu = $env{'form.GD_'.$user.'_'.$_.'_solved'};
	    $score = 'excused' if (($dropMenu eq 'excused') && ($score ne 'excused'));

	    $newrecord{'resource.'.$_.'.regrader'}=
		"$env{'user.name'}:$env{'user.domain'}";
	    if ($dropMenu eq 'reset status' &&
		$old_score ne '') { # ignore if no previous attempts => nothing to reset
		$newrecord{'resource.'.$_.'.tries'} = '';
		$newrecord{'resource.'.$_.'.solved'} = '';
		$newrecord{'resource.'.$_.'.award'} = '';
		$newrecord{'resource.'.$_.'.awarded'} = '';
		$updateflag = 1;
                if ($env{'form.GD_'.$user.'_'.$_.'_aggtries'} > 0) {
                    my $aggtries = $env{'form.GD_'.$user.'_'.$_.'_aggtries'};
                    my $totaltries = $env{'form.GD_'.$user.'_'.$_.'_totaltries'};
                    my $solvedstatus = $env{'form.GD_'.$user.'_'.$_.'_solved_s'};
                    &decrement_aggs($symb,$_,\%aggregate,$aggtries,$totaltries,$solvedstatus);
                    $aggregateflag = 1;
                }
	    } elsif (!($old_part eq $partial && $old_score eq $score)) {
		$updateflag = 1;
		$newrecord{'resource.'.$_.'.awarded'}  = $partial if $partial ne '';
		$newrecord{'resource.'.$_.'.solved'}   = $score;
		$rec_update++;
	    }

	    $line .= '<td align="center">'.$old_aw.'&nbsp;</td>'.
		'<td align="center">'.$awarded.
		($score eq 'excused' ? $score : '').'&nbsp;</td>';


	    my $partid=$_;
	    foreach my $stores (@parts) {
		my ($part,$type) = &split_part_type($stores);
		if ($part !~ m/^\Q$partid\E/) { next;}
		if ($type eq 'awarded' || $type eq 'solved') { next; }
		my $old_aw    = $env{'form.GD_'.$user.'_'.$part.'_'.$type.'_s'};
		my $awarded   = $env{'form.GD_'.$user.'_'.$part.'_'.$type};
		if ($awarded ne '' && $awarded ne $old_aw) {
		    $newrecord{'resource.'.$part.'.'.$type}= $awarded;
		    $newrecord{'resource.'.$part.'.regrader'}="$env{'user.name'}:$env{'user.domain'}";
		    $updateflag=1;
		}
		$line .= '<td align="center">'.$old_aw.'&nbsp;</td>'.
		    '<td align="center">'.$awarded.'&nbsp;</td>';
	    }
	}
	$line.='</tr>'."\n";

	my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
	my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};

	if ($updateflag) {
	    $count++;
	    &Apache::lonnet::cstore(\%newrecord,$symb,$env{'request.course.id'},
				    $udom,$uname);

	    if (&Apache::bridgetask::in_queue('gradingqueue',$symb,$cdom,
					      $cnum,$udom,$uname)) {
		# need to figure out if should be in queue.
		my %record =  
		    &Apache::lonnet::restore($symb,$env{'request.course.id'},
					     $udom,$uname);
		my $all_graded = 1;
		my $none_graded = 1;
		foreach my $part (@parts) {
		    if ( $record{'resource.'.$part.'.awarded'} eq '' ) {
			$all_graded = 0;
		    } else {
			$none_graded = 0;
		    }
		}

		if ($all_graded || $none_graded) {
		    &Apache::bridgetask::remove_from_queue('gradingqueue',
							   $symb,$cdom,$cnum,
							   $udom,$uname);
		}
	    }

	    $result.='<tr bgcolor="#ffffde"><td align="right">&nbsp;'.$updateCtr.'&nbsp;</td>'.$line;
	    $updateCtr++;
	} else {
	    $noupdate.='<tr bgcolor="#ffffde"><td align="right">&nbsp;'.$noupdateCtr.'&nbsp;</td>'.$line;
	    $noupdateCtr++;
	}
        if ($aggregateflag) {
            &Apache::lonnet::cinc('nohist_resourcetracker',\%aggregate,
				  $cdom,$cnum);
        }
    }
    if ($noupdate) {
#	my $numcols=(scalar(@partid)*(scalar(@parts)-1)*2)+3;
	my $numcols=scalar(@partid)*4+2;
	$result .= '<tr bgcolor="#ffffff"><td align="center" colspan="'.$numcols.'">No Changes Occurred For the Students Below</td></tr><tr bgcolor="#ffffde">'.$noupdate;
    }
    $result .= '</table></td></tr></table>'."\n".
	&show_grading_menu_form ($symb);
    my $msg = '<br /><b>Number of records updated = '.$rec_update.
	' for '.$count.' student'.($count <= 1 ? '' : 's').'.</b><br />'.
	'<b>Total number of students = '.$env{'form.total'}.'</b><br />';
    return $title.$msg.$result;
}

sub split_part_type {
    my ($partstr) = @_;
    my ($temp,@allparts)=split(/_/,$partstr);
    my $type=pop(@allparts);
    my $part=join('.',@allparts);
    return ($part,$type);
}

#------------- end of section for handling grading by section/class ---------
#
#----------------------------------------------------------------------------


#----------------------------------------------------------------------------
#
#-------------------------- Next few routines handles grading by csv upload
#
#--- Javascript to handle csv upload
sub csvupload_javascript_reverse_associate {
    my $error1=&mt('You need to specify the username or ID');
    my $error2=&mt('You need to specify at least one grading field');
  return(<<ENDPICK);
  function verify(vf) {
    var foundsomething=0;
    var founduname=0;
    var foundID=0;
    for (i=0;i<=vf.nfields.value;i++) {
      tw=eval('vf.f'+i+'.selectedIndex');
      if (i==0 && tw!=0) { foundID=1; }
      if (i==1 && tw!=0) { founduname=1; }
      if (i!=0 && i!=1 && i!=2 && tw!=0) { foundsomething=1; }
    }
    if (founduname==0 && foundID==0) {
	alert('$error1');
	return;
    }
    if (foundsomething==0) {
	alert('$error2');
	return;
    }
    vf.submit();
  }
  function flip(vf,tf) {
    var nw=eval('vf.f'+tf+'.selectedIndex');
    var i;
    for (i=0;i<=vf.nfields.value;i++) {
      //can not pick the same destination field for both name and domain
      if (((i ==0)||(i ==1)) && 
          ((tf==0)||(tf==1)) && 
          (i!=tf) &&
          (eval('vf.f'+i+'.selectedIndex')==nw)) {
        eval('vf.f'+i+'.selectedIndex=0;')
      }
    }
  }
ENDPICK
}

sub csvupload_javascript_forward_associate {
    my $error1=&mt('You need to specify the username or ID');
    my $error2=&mt('You need to specify at least one grading field');
  return(<<ENDPICK);
  function verify(vf) {
    var foundsomething=0;
    var founduname=0;
    var foundID=0;
    for (i=0;i<=vf.nfields.value;i++) {
      tw=eval('vf.f'+i+'.selectedIndex');
      if (tw==1) { foundID=1; }
      if (tw==2) { founduname=1; }
      if (tw>3) { foundsomething=1; }
    }
    if (founduname==0 && foundID==0) {
	alert('$error1');
	return;
    }
    if (foundsomething==0) {
	alert('$error2');
	return;
    }
    vf.submit();
  }
  function flip(vf,tf) {
    var nw=eval('vf.f'+tf+'.selectedIndex');
    var i;
    //can not pick the same destination field twice
    for (i=0;i<=vf.nfields.value;i++) {
      if ((i!=tf) && (eval('vf.f'+i+'.selectedIndex')==nw)) {
        eval('vf.f'+i+'.selectedIndex=0;')
      }
    }
  }
ENDPICK
}

sub csvuploadmap_header {
    my ($request,$symb,$datatoken,$distotal)= @_;
    my $javascript;
    if ($env{'form.upfile_associate'} eq 'reverse') {
	$javascript=&csvupload_javascript_reverse_associate();
    } else {
	$javascript=&csvupload_javascript_forward_associate();
    }

    my ($result) = &showResourceInfo($symb,$env{'form.probTitle'});
    my $checked=(($env{'form.noFirstLine'})?' checked="checked"':'');
    my $ignore=&mt('Ignore First Line');
    $request->print(<<ENDPICK);
<form method="post" enctype="multipart/form-data" action="/adm/grades" name="gradesupload">
<h3><span class="LC_info">Uploading Class Grades</span></h3>
$result
<hr />
<h3>Identify fields</h3>
Total number of records found in file: $distotal <hr />
Enter as many fields as you can. The system will inform you and bring you back
to this page if the data selected is insufficient to run your class.<hr />
<input type="button" value="Reverse Association" onClick="javascript:this.form.associate.value='Reverse Association';submit(this.form);" />
<label><input type="checkbox" name="noFirstLine" $checked />$ignore</label>
<input type="hidden" name="associate"  value="" />
<input type="hidden" name="phase"      value="three" />
<input type="hidden" name="datatoken"  value="$datatoken" />
<input type="hidden" name="fileupload" value="$env{'form.fileupload'}" />
<input type="hidden" name="upfiletype" value="$env{'form.upfiletype'}" />
<input type="hidden" name="upfile_associate" 
                                       value="$env{'form.upfile_associate'}" />
<input type="hidden" name="symb"       value="$symb" />
<input type="hidden" name="saveState"  value="$env{'form.saveState'}" />
<input type="hidden" name="probTitle"  value="$env{'form.probTitle'}" />
<input type="hidden" name="command"    value="csvuploadoptions" />
<hr />
<script type="text/javascript" language="Javascript">
$javascript
</script>
ENDPICK
    return '';

}

sub csvupload_fields {
    my ($symb) = @_;
    my (@parts) = &getpartlist($symb);
    my @fields=(['ID','Student ID'],
		['username','Student Username'],
		['domain','Student Domain']);
    my (undef,undef,$url) = &Apache::lonnet::decode_symb($symb);
    foreach my $part (sort(@parts)) {
	my @datum;
	my $display=&Apache::lonnet::metadata($url,$part.'.display');
	my $name=$part;
	if  (!$display) { $display = $name; }
	@datum=($name,$display);
	if ($name=~/^stores_(.*)_awarded/) {
	    push(@fields,['stores_'.$1.'_points',"Points [Part: $1]"]);
	}
	push(@fields,\@datum);
    }
    return (@fields);
}

sub csvuploadmap_footer {
    my ($request,$i,$keyfields) =@_;
    $request->print(<<ENDPICK);
</table>
<input type="hidden" name="nfields" value="$i" />
<input type="hidden" name="keyfields" value="$keyfields" />
<input type="button" onClick="javascript:verify(this.form)" value="Assign Grades" /><br />
</form>
ENDPICK
}

sub checkforfile_js {
    my $result =<<CSVFORMJS;
<script type="text/javascript" language="javascript">
    function checkUpload(formname) {
	if (formname.upfile.value == "") {
	    alert("Please use the browse button to select a file from your local directory.");
	    return false;
	}
	formname.submit();
    }
    </script>
CSVFORMJS
    return $result;
}

sub upcsvScores_form {
    my ($request) = shift;
    my ($symb)=&get_symb($request);
    if (!$symb) {return '';}
    my $result=&checkforfile_js();
    $env{'form.probTitle'} = &Apache::lonnet::gettitle($symb);
    my ($table) = &showResourceInfo($symb,$env{'form.probTitle'});
    $result.=$table;
    $result.='<br /><table width="100%" border="0"><tr><td bgcolor="#777777">'."\n";
    $result.='<table width="100%" border="0"><tr bgcolor="#e6ffff"><td>'."\n";
    $result.='&nbsp;<b>'.&mt('Specify a file containing the class scores for current resource').
	'.</b></td></tr>'."\n";
    $result.='<tr bgcolor=#ffffe6><td>'."\n";
    my $upload=&mt("Upload Scores");
    my $upfile_select=&Apache::loncommon::upfile_select_html();
    my $ignore=&mt('Ignore First Line');
    $result.=<<ENDUPFORM;
<form method="post" enctype="multipart/form-data" action="/adm/grades" name="gradesupload">
<input type="hidden" name="symb" value="$symb" />
<input type="hidden" name="command" value="csvuploadmap" />
<input type="hidden" name="probTitle" value="$env{'form.probTitle'}" />
<input type="hidden" name="saveState"  value="$env{'form.saveState'}" />
$upfile_select
<br /><input type="button" onClick="javascript:checkUpload(this.form);" value="$upload" />
<label><input type="checkbox" name="noFirstLine" />$ignore</label>
</form>
ENDUPFORM
    $result.=&Apache::loncommon::help_open_topic("Course_Convert_To_CSV",
                           &mt("How do I create a CSV file from a spreadsheet"))
    .'</td></tr></table>'."\n";
    $result.='</td></tr></table><br /><br />'."\n";
    $result.=&show_grading_menu_form($symb);
    return $result;
}


sub csvuploadmap {
    my ($request)= @_;
    my ($symb)=&get_symb($request);
    if (!$symb) {return '';}

    my $datatoken;
    if (!$env{'form.datatoken'}) {
	$datatoken=&Apache::loncommon::upfile_store($request);
    } else {
	$datatoken=$env{'form.datatoken'};
	&Apache::loncommon::load_tmp_file($request);
    }
    my @records=&Apache::loncommon::upfile_record_sep();
    if ($env{'form.noFirstLine'}) { shift(@records); }
    &csvuploadmap_header($request,$symb,$datatoken,$#records+1);
    my ($i,$keyfields);
    if (@records) {
	my @fields=&csvupload_fields($symb);

	if ($env{'form.upfile_associate'} eq 'reverse') {	
	    &Apache::loncommon::csv_print_samples($request,\@records);
	    $i=&Apache::loncommon::csv_print_select_table($request,\@records,
							  \@fields);
	    foreach (@fields) { $keyfields.=$_->[0].','; }
	    chop($keyfields);
	} else {
	    unshift(@fields,['none','']);
	    $i=&Apache::loncommon::csv_samples_select_table($request,\@records,
							    \@fields);
            foreach my $rec (@records) {
                my %temp = &Apache::loncommon::record_sep($rec);
                if (%temp) {
                    $keyfields=join(',',sort(keys(%temp)));
                    last;
                }
            }
	}
    }
    &csvuploadmap_footer($request,$i,$keyfields);
    $request->print(&show_grading_menu_form($symb));

    return '';
}

sub csvuploadoptions {
    my ($request)= @_;
    my ($symb)=&get_symb($request);
    my $checked=(($env{'form.noFirstLine'})?'1':'0');
    my $ignore=&mt('Ignore First Line');
    $request->print(<<ENDPICK);
<form method="post" enctype="multipart/form-data" action="/adm/grades" name="gradesupload">
<h3><span class="LC_info">Uploading Class Grade Options</span></h3>
<input type="hidden" name="command"    value="csvuploadassign" />
<!--
<p>
<label>
   <input type="checkbox" name="show_full_results" />
   Show a table of all changes
</label>
</p>
-->
<p>
<label>
   <input type="checkbox" name="overwite_scores" checked="checked" />
   Overwrite any existing score
</label>
</p>
ENDPICK
    my %fields=&get_fields();
    if (!defined($fields{'domain'})) {
	my $domform = &Apache::loncommon::select_dom_form($env{'request.role.domain'},'default_domain');
	$request->print("\n<p> Users are in domain: ".$domform."</p>\n");
    }
    foreach my $key (sort(keys(%env))) {
	if ($key !~ /^form\.(.*)$/) { next; }
	my $cleankey=$1;
	if ($cleankey eq 'command') { next; }
	$request->print('<input type="hidden" name="'.$cleankey.
			'"  value="'.$env{$key}.'" />'."\n");
    }
    # FIXME do a check for any duplicated user ids...
    # FIXME do a check for any invalid user ids?...
    $request->print('<input type="submit" value="Assign Grades" /><br />
<hr /></form>'."\n");
    $request->print(&show_grading_menu_form($symb));
    return '';
}

sub get_fields {
    my %fields;
    my @keyfields = split(/\,/,$env{'form.keyfields'});
    for (my $i=0; $i<=$env{'form.nfields'}; $i++) {
	if ($env{'form.upfile_associate'} eq 'reverse') {
	    if ($env{'form.f'.$i} ne 'none') {
		$fields{$keyfields[$i]}=$env{'form.f'.$i};
	    }
	} else {
	    if ($env{'form.f'.$i} ne 'none') {
		$fields{$env{'form.f'.$i}}=$keyfields[$i];
	    }
	}
    }
    return %fields;
}

sub csvuploadassign {
    my ($request)= @_;
    my ($symb)=&get_symb($request);
    if (!$symb) {return '';}
    my $error_msg = '';
    &Apache::loncommon::load_tmp_file($request);
    my @gradedata = &Apache::loncommon::upfile_record_sep();
    if ($env{'form.noFirstLine'}) { shift(@gradedata); }
    my %fields=&get_fields();
    $request->print('<h3>Assigning Grades</h3>');
    my $courseid=$env{'request.course.id'};
    my ($classlist) = &getclasslist('all',0);
    my @notallowed;
    my @skipped;
    my $countdone=0;
    foreach my $grade (@gradedata) {
	my %entries=&Apache::loncommon::record_sep($grade);
	my $domain;
	if ($entries{$fields{'domain'}}) {
	    $domain=$entries{$fields{'domain'}};
	} else {
	    $domain=$env{'form.default_domain'};
	}
	$domain=~s/\s//g;
	my $username=$entries{$fields{'username'}};
	$username=~s/\s//g;
	if (!$username) {
	    my $id=$entries{$fields{'ID'}};
	    $id=~s/\s//g;
	    my %ids=&Apache::lonnet::idget($domain,$id);
	    $username=$ids{$id};
	}
	if (!exists($$classlist{"$username:$domain"})) {
	    my $id=$entries{$fields{'ID'}};
	    $id=~s/\s//g;
	    if ($id) {
		push(@skipped,"$id:$domain");
	    } else {
		push(@skipped,"$username:$domain");
	    }
	    next;
	}
	my $usec=$classlist->{"$username:$domain"}[5];
	if (!&canmodify($usec)) {
	    push(@notallowed,"$username:$domain");
	    next;
	}
	my %points;
	my %grades;
	foreach my $dest (keys(%fields)) {
	    if ($dest eq 'ID' || $dest eq 'username' ||
		$dest eq 'domain') { next; }
	    if ($entries{$fields{$dest}} =~ /^\s*$/) { next; }
	    if ($dest=~/stores_(.*)_points/) {
		my $part=$1;
		my $wgt =&Apache::lonnet::EXT('resource.'.$part.'.weight',
					      $symb,$domain,$username);
                if ($wgt) {
                    $entries{$fields{$dest}}=~s/\s//g;
                    my $pcr=$entries{$fields{$dest}} / $wgt;
                    my $award='correct_by_override';
                    $grades{"resource.$part.awarded"}=$pcr;
                    $grades{"resource.$part.solved"}=$award;
                    $points{$part}=1;
                } else {
                    $error_msg = "<br />" .
                        &mt("Some point values were assigned"
                            ." for problems with a weight "
                            ."of zero. These values were "
                            ."ignored.");
                }
	    } else {
		if ($dest=~/stores_(.*)_awarded/) { if ($points{$1}) {next;} }
		if ($dest=~/stores_(.*)_solved/)  { if ($points{$1}) {next;} }
		my $store_key=$dest;
		$store_key=~s/^stores/resource/;
		$store_key=~s/_/\./g;
		$grades{$store_key}=$entries{$fields{$dest}};
	    }
	}
	if (! %grades) { push(@skipped,"$username:$domain no data to save"); }
	$grades{"resource.regrader"}="$env{'user.name'}:$env{'user.domain'}";
#	&Apache::lonnet::logthis(" storing ".(join('-',%grades)));
	my $result=&Apache::lonnet::cstore(\%grades,$symb,
					   $env{'request.course.id'},
					   $domain,$username);
	if ($result eq 'ok') {
	    $request->print('.');
	} else {
	    $request->print("<p>
                              <span class=\"LC_error\">
                                 Failed to save student $username:$domain.
                                 Message when trying to save was ($result)
                              </span>
                             </p>" );
	}
	$request->rflush();
	$countdone++;
    }
    $request->print("<br />Saved $countdone students\n");
    if (@skipped) {
	$request->print('<p><h4><b>Skipped Students</b></h4></p>');
	foreach my $student (@skipped) { $request->print("$student<br />\n"); }
    }
    if (@notallowed) {
	$request->print('<p><span class="LC_error">Students Not Allowed to Modify</span></p>');
	foreach my $student (@notallowed) { $request->print("$student<br />\n"); }
    }
    $request->print("<br />\n");
    $request->print(&show_grading_menu_form($symb));
    return $error_msg;
}
#------------- end of section for handling csv file upload ---------
#
#-------------------------------------------------------------------
#
#-------------- Next few routines handle grading by page/sequence
#
#--- Select a page/sequence and a student to grade
sub pickStudentPage {
    my ($request) = shift;

    $request->print(<<LISTJAVASCRIPT);
<script type="text/javascript" language="javascript">

function checkPickOne(formname) {
    if (radioSelection(formname.student) == null) {
	alert("Please select the student you wish to grade.");
	return;
    }
    ptr = pullDownSelection(formname.selectpage);
    formname.page.value = formname["page"+ptr].value;
    formname.title.value = formname["title"+ptr].value;
    formname.submit();
}

</script>
LISTJAVASCRIPT
    &commonJSfunctions($request);
    my ($symb) = &get_symb($request);
    my $cdom      = $env{"course.$env{'request.course.id'}.domain"};
    my $cnum      = $env{"course.$env{'request.course.id'}.num"};
    my $getsec    = $env{'form.section'} eq '' ? 'all' : $env{'form.section'};

    my $result='<h3><span class="LC_info">&nbsp;'.
	'Manual Grading by Page or Sequence</span></h3>';

    $result.='<form action="/adm/grades" method="post" name="displayPage">'."\n";
    $result.='&nbsp;<b>Problems from:</b> <select name="selectpage">'."\n";
    my ($titles,$symbx) = &getSymbMap($request);
    my ($curpage) =&Apache::lonnet::decode_symb($symb); 
#    my ($curpage,$mapId) =&Apache::lonnet::decode_symb($symb); 
#    my $type=($curpage =~ /\.(page|sequence)/);
    my $ctr=0;
    foreach (@$titles) {
	my ($minder,$showtitle) = ($_ =~ /(\d+)\.(.*)/);
	$result.='<option value="'.$ctr.'" '.
	    ($$symbx{$_} =~ /$curpage$/ ? 'selected="on"' : '').
	    '>'.$showtitle.'</option>'."\n";
	$ctr++;
    }
    $result.= '</select>'."<br />\n";
    $ctr=0;
    foreach (@$titles) {
	my ($minder,$showtitle) = ($_ =~ /(\d+)\.(.*)/);
	$result.='<input type="hidden" name="page'.$ctr.'" value="'.$$symbx{$_}.'" />'."\n";
	$result.='<input type="hidden" name="title'.$ctr.'" value="'.$showtitle.'" />'."\n";
	$ctr++;
    }
    $result.='<input type="hidden" name="page" />'."\n".
	'<input type="hidden" name="title" />'."\n";

    $result.='&nbsp;<b>View Problems Text: </b><label><input type="radio" name="vProb" value="no" checked="on" /> no </label>'."\n".
	'<label><input type="radio" name="vProb" value="yes" /> yes </label>'."<br />\n";

    $result.='&nbsp;<b>Submission Details: </b>'.
	'<label><input type="radio" name="lastSub" value="none" /> none</label>'."\n".
	'<label><input type="radio" name="lastSub" value="datesub" checked /> by dates and submissions</label>'."\n".
	'<label><input type="radio" name="lastSub" value="all" /> all details</label>'."\n";

    $result.='<input type="hidden" name="section"     value="'.$getsec.'" />'."\n".
	'<input type="hidden" name="Status"  value="'.$env{'form.Status'}.'" />'."\n".
	'<input type="hidden" name="command" value="displayPage" />'."\n".
	'<input type="hidden" name="symb"    value="'.$symb.'" />'."\n".
	'<input type="hidden" name="saveState" value="'.$env{'form.saveState'}.'" />'."<br />\n";

    $result.='&nbsp;<b>'.&mt('Use CODE:').' </b>'.
	'<input type="text" name="CODE" value="" /><br />'."\n";

    $result.='&nbsp;<input type="button" '.
	'onClick="javascript:checkPickOne(this.form);"value="Next->" /><br />'."\n";

    $request->print($result);

    my $studentTable.='&nbsp;<b>Select a student you wish to grade and then click on the Next button.</b><br />'.
	'<table border="0"><tr><td bgcolor="#777777">'.
	'<table border="0"><tr bgcolor="#e6ffff">'.
	'<td align="right">&nbsp;<b>No.</b></td>'.
	'<td>'.&nameUserString('header').'</td>'.
	'<td align="right">&nbsp;<b>No.</b></td>'.
	'<td>'.&nameUserString('header').'</td></tr>';
 
    my (undef,undef,$fullname) = &getclasslist($getsec,'1');
    my $ptr = 1;
    foreach my $student (sort 
			 {
			     if (lc($$fullname{$a}) ne lc($$fullname{$b})) {
				 return (lc($$fullname{$a}) cmp lc($$fullname{$b}));
			     }
			     return $a cmp $b;
			 } (keys(%$fullname))) {
	my ($uname,$udom) = split(/:/,$student);
	$studentTable.=($ptr%2 == 1 ? '<tr bgcolor="#ffffe6">' : '</td>');
	$studentTable.='<td align="right">'.$ptr.'&nbsp;</td>';
	$studentTable.='<td>&nbsp;<label><input type="radio" name="student" value="'.$student.'" /> '
	    .&nameUserString(undef,$$fullname{$student},$uname,$udom)."</label>\n";
	$studentTable.=($ptr%2 == 0 ? '</td></tr>' : '');
	$ptr++;
    }
    $studentTable.='</td><td>&nbsp;</td><td>&nbsp;</td></tr>' if ($ptr%2 == 0);
    $studentTable.='</table></td></tr></table>'."\n";
    $studentTable.='<input type="button" '.
	'onClick="javascript:checkPickOne(this.form);"value="Next->" /></form>'."\n";

    $studentTable.=&show_grading_menu_form($symb);
    $request->print($studentTable);

    return '';
}

sub getSymbMap {
    my ($request) = @_;
    my $navmap = Apache::lonnavmaps::navmap->new();

    my %symbx = ();
    my @titles = ();
    my $minder = 0;

    # Gather every sequence that has problems.
    my @sequences = $navmap->retrieveResources(undef, sub { shift->is_map(); },
					       1,0,1);
    for my $sequence ($navmap->getById('0.0'), @sequences) {
	if ($navmap->hasResource($sequence, sub { shift->is_problem(); }, 0) ) {
	    my $title = $minder.'.'.
		&HTML::Entities::encode($sequence->compTitle(),'"\'&');
	    push(@titles, $title); # minder in case two titles are identical
	    $symbx{$title} = &HTML::Entities::encode($sequence->symb(),'"\'&');
	    $minder++;
	}
    }
    return \@titles,\%symbx;
}

#
#--- Displays a page/sequence w/wo problems, w/wo submissions
sub displayPage {
    my ($request) = shift;

    my ($symb) = &get_symb($request);
    my $cdom      = $env{"course.$env{'request.course.id'}.domain"};
    my $cnum      = $env{"course.$env{'request.course.id'}.num"};
    my $getsec    = $env{'form.section'} eq '' ? 'all' : $env{'form.section'};
    my $pageTitle = $env{'form.page'};
    my ($classlist,undef,$fullname) = &getclasslist($getsec,'1');
    my ($uname,$udom) = split(/:/,$env{'form.student'});
    my $usec=$classlist->{$env{'form.student'}}[5];

    #need to make sure we have the correct data for later EXT calls, 
    #thus invalidate the cache
    &Apache::lonnet::devalidatecourseresdata(
                 $env{'course.'.$env{'request.course.id'}.'.num'},
                 $env{'course.'.$env{'request.course.id'}.'.domain'});
    &Apache::lonnet::clear_EXT_cache_status();

    if (!&canview($usec)) {
	$request->print('<span class="LC_warning">Unable to view requested student.('.$env{'form.student'}.')</span>');
	$request->print(&show_grading_menu_form($symb));
	return;
    }
    my $result='<h3><span class="LC_info">&nbsp;'.$env{'form.title'}.'</span></h3>';
    $result.='<h3>&nbsp;Student: '.&nameUserString(undef,$$fullname{$env{'form.student'}},$uname,$udom).
	'</h3>'."\n";
    if (&Apache::lonnet::validCODE($env{'form.CODE'})) {
	$result.='<h3>&nbsp;CODE: '.$env{'form.CODE'}.'</h3>'."\n";
    } else {
	delete($env{'form.CODE'});
    }
    &sub_page_js($request);
    $request->print($result);

    my $navmap = Apache::lonnavmaps::navmap->new();
    my ($mapUrl, $id, $resUrl)=&Apache::lonnet::decode_symb($env{'form.page'});
    my $map = $navmap->getResourceByUrl($resUrl); # add to navmaps
    if (!$map) {
	$request->print('<span class="LC_warning">Unable to view requested sequence. ('.$resUrl.')</span>');
	$request->print(&show_grading_menu_form($symb));
	return; 
    }
    my $iterator = $navmap->getIterator($map->map_start(),
					$map->map_finish());

    my $studentTable='<form action="/adm/grades" method="post" name="gradePage">'."\n".
	'<input type="hidden" name="command" value="gradeByPage" />'."\n".
	'<input type="hidden" name="fullname" value="'.$$fullname{$env{'form.student'}}.'" />'."\n".
	'<input type="hidden" name="student" value="'.$env{'form.student'}.'" />'."\n".
	'<input type="hidden" name="page"    value="'.$pageTitle.'" />'."\n".
	'<input type="hidden" name="title"   value="'.$env{'form.title'}.'" />'."\n".
	'<input type="hidden" name="symb"    value="'.$symb.'" />'."\n".
	'<input type="hidden" name="overRideScore" value="no" />'."\n".
	'<input type="hidden" name="saveState" value="'.$env{'form.saveState'}.'" />'."\n";

    if (defined($env{'form.CODE'})) {
	$studentTable.=
	    '<input type="hidden" name="CODE" value="'.$env{'form.CODE'}.'" />'."\n";
    }
    my $checkIcon = '<img alt="'.&mt('Check Mark').
	'" src="'.$request->dir_config('lonIconsURL').
	'/check.gif" height="16" border="0" />';

    $studentTable.='&nbsp;<b>Note:</b> Problems graded correct by the computer are marked with a '.$checkIcon.
	' symbol.'."\n".
	'<table border="0"><tr><td bgcolor="#777777">'.
	'<table border="0"><tr bgcolor="#e6ffff">'.
	'<td align="center"><b>&nbsp;Prob.&nbsp;</b></td>'.
	'<td><b>&nbsp;'.($env{'form.vProb'} eq 'no' ? 'Title' : 'Problem Text').'/Grade</b></td></tr>';

    &Apache::lonxml::clear_problem_counter();
    my ($depth,$question,$prob) = (1,1,1);
    $iterator->next(); # skip the first BEGIN_MAP
    my $curRes = $iterator->next(); # for "current resource"
    while ($depth > 0) {
        if($curRes == $iterator->BEGIN_MAP) { $depth++; }
        if($curRes == $iterator->END_MAP) { $depth--; }

        if (ref($curRes) && $curRes->is_problem()) {
	    my $parts = $curRes->parts();
            my $title = $curRes->compTitle();
	    my $symbx = $curRes->symb();
	    $studentTable.='<tr bgcolor="#ffffe6"><td align="center" valign="top" >'.$prob.
		(scalar(@{$parts}) == 1 ? '' : '<br />('.scalar(@{$parts}).'&nbsp;parts)').'</td>';
	    $studentTable.='<td valign="top">';
	    my %form = ('CODE' => $env{'form.CODE'},);
	    if ($env{'form.vProb'} eq 'yes' ) {
		$studentTable.=&show_problem($request,$symbx,$uname,$udom,1,
					     undef,'both',\%form);
	    } else {
		my $companswer = &Apache::loncommon::get_student_answers($symbx,$uname,$udom,$env{'request.course.id'},%form);
		$companswer =~ s|<form(.*?)>||g;
		$companswer =~ s|</form>||g;
#		while ($companswer =~ /(<a href\=\"javascript:newWindow.*?Script Vars<\/a>)/s) { #<a href="javascript:newWindow</a>
#		    $companswer =~ s/$1/ /ms;
#		    $request->print('match='.$1."<br />\n");
#		}
#		$companswer =~ s|<table border=\"1\">|<table border=\"0\">|g;
		$studentTable.='&nbsp;<b>'.$title.'</b>&nbsp;<br />&nbsp;<b>Correct answer:</b><br />'.$companswer;
	    }

	    my %record = &Apache::lonnet::restore($symbx,$env{'request.course.id'},$udom,$uname);

	    if ($env{'form.lastSub'} eq 'datesub') {
		if ($record{'version'} eq '') {
		    $studentTable.='<br />&nbsp;<span class="LC_warning">No recorded submission for this problem</span><br />';
		} else {
		    my %responseType = ();
		    foreach my $partid (@{$parts}) {
			my @responseIds =$curRes->responseIds($partid);
			my @responseType =$curRes->responseType($partid);
			my %responseIds;
			for (my $i=0;$i<=$#responseIds;$i++) {
			    $responseIds{$responseIds[$i]}=$responseType[$i];
			}
			$responseType{$partid} = \%responseIds;
		    }
		    $studentTable.= &displaySubByDates($symbx,\%record,$parts,\%responseType,$checkIcon,$uname,$udom);

		}
	    } elsif ($env{'form.lastSub'} eq 'all') {
		my $last = ($env{'form.lastSub'} eq 'last' ? 'last' : '');
		$studentTable.=&Apache::loncommon::get_previous_attempt($symbx,$uname,$udom,
									$env{'request.course.id'},
									'','.submission');
 
	    }
	    if (&canmodify($usec)) {
		foreach my $partid (@{$parts}) {
		    $studentTable.=&gradeBox($request,$symbx,$uname,$udom,$question,$partid,\%record);
		    $studentTable.='<input type="hidden" name="q_'.$question.'" value="'.$partid.'" />'."\n";
		    $question++;
		}
		$prob++;
	    }
	    $studentTable.='</td></tr>';

	}
        $curRes = $iterator->next();
    }

    $studentTable.='</table></td></tr></table>'."\n".
	'<input type="button" value="Save" '.
	'onClick="javascript:checkSubmitPage(this.form,'.$question.');" />'.
	'</form>'."\n";
    $studentTable.=&show_grading_menu_form($symb);
    $request->print($studentTable);

    return '';
}

sub displaySubByDates {
    my ($symb,$record,$parts,$responseType,$checkIcon,$uname,$udom) = @_;
    my $isCODE=0;
    my $isTask = ($symb =~/\.task$/);
    if (exists($record->{'resource.CODE'})) { $isCODE=1; }
    my $studentTable='<table border="0" width="100%"><tr><td bgcolor="#777777">'.
	'<table border="0" width="100%"><tr bgcolor="#e6ffff">'.
	'<td><b>Date/Time</b></td>'.
	($isCODE?'<td><b>CODE</b></td>':'').
	'<td><b>Submission</b></td>'.
	'<td><b>Status&nbsp;</b></td></tr>';
    my ($version);
    my %mark;
    my %orders;
    $mark{'correct_by_student'} = $checkIcon;
    if (!exists($$record{'1:timestamp'})) {
	return '<br />&nbsp;<span class="LC_warning">Nothing submitted - no attempts</span><br />';
    }

    my $interaction;
    for ($version=1;$version<=$$record{'version'};$version++) {
	my $timestamp = scalar(localtime($$record{$version.':timestamp'}));
	if (exists($$record{$version.':resource.0.version'})) {
	    $interaction = $$record{$version.':resource.0.version'};
	}

	my $where = ($isTask ? "$version:resource.$interaction"
		             : "$version:resource");
	#&Apache::lonnet::logthis(" got $where");
	$studentTable.='<tr bgcolor="#ffffff" valign="top"><td>'.$timestamp.'</td>';
	if ($isCODE) {
	    $studentTable.='<td>'.$record->{$version.':resource.CODE'}.'</td>';
	}
	my @versionKeys = split(/\:/,$$record{$version.':keys'});
	my @displaySub = ();
	foreach my $partid (@{$parts}) {
	    my @matchKey = ($isTask ? sort(grep /^resource\.\d+\.\Q$partid\E\.award$/,@versionKeys)
			            : sort(grep /^resource\.\Q$partid\E\..*?\.submission$/,@versionKeys));
	    

#	    next if ($$record{"$version:resource.$partid.solved"} eq '');
	    my $display_part=&get_display_part($partid,$symb);
	    foreach my $matchKey (@matchKey) {
		if (exists($$record{$version.':'.$matchKey}) &&
		    $$record{$version.':'.$matchKey} ne '') {

		    my ($responseId)= ($isTask ? ($matchKey=~ /^resource\.(.*?)\.\Q$partid\E\.award$/)
				               : ($matchKey=~ /^resource\.\Q$partid\E\.(.*?)\.submission$/));
		    #&Apache::lonnet::logthis("match $matchKey $responseId (".$$record{$version.':'.$matchKey});
		    $displaySub[0].='<b>Part:</b>&nbsp;'.$display_part.'&nbsp;';
		    $displaySub[0].='<span class="LC_internal_info">(ID&nbsp;'.
			$responseId.')</span>&nbsp;<b>';
		    if ($$record{"$where.$partid.tries"} eq '') {
			$displaySub[0].='Trial&nbsp;not&nbsp;counted';
		    } else {
			$displaySub[0].='Trial&nbsp;'.
			    $$record{"$where.$partid.tries"};
		    }
		    my $responseType=($isTask ? 'Task'
                                              : $responseType->{$partid}->{$responseId});
		    if (!exists($orders{$partid})) { $orders{$partid}={}; }
		    if (!exists($orders{$partid}->{$responseId})) {
			$orders{$partid}->{$responseId}=
			    &get_order($partid,$responseId,$symb,$uname,$udom);
		    }
		    $displaySub[0].='</b>&nbsp; '.
			&cleanRecord($$record{$version.':'.$matchKey},$responseType,$symb,$partid,$responseId,$record,$orders{$partid}->{$responseId},"$version:",$uname,$udom).'<br />';
		}
	    }
	    if (exists($$record{"$where.$partid.checkedin"})) {
		$displaySub[1].='Checked in by '.
		    $$record{"$where.$partid.checkedin"}.' into slot '.
		    $$record{"$where.$partid.checkedin.slot"}.
		    '<br />';
	    }
	    if (exists $$record{"$where.$partid.award"}) {
		$displaySub[1].='<b>Part:</b>&nbsp;'.$display_part.' &nbsp;'.
		    lc($$record{"$where.$partid.award"}).' '.
		    $mark{$$record{"$where.$partid.solved"}}.
		    '<br />';
	    }
	    if (exists $$record{"$where.$partid.regrader"}) {
		$displaySub[2].=$$record{"$where.$partid.regrader"}.
		    ' (<b>'.&mt('Part').':</b> '.$display_part.')';
	    } elsif ($$record{"$version:resource.$partid.regrader"} =~ /\S/) {
		$displaySub[2].=
		    $$record{"$version:resource.$partid.regrader"}.
		    ' (<b>'.&mt('Part').':</b> '.$display_part.')';
	    }
	}
	# needed because old essay regrader has not parts info
	if (exists $$record{"$version:resource.regrader"}) {
	    $displaySub[2].=$$record{"$version:resource.regrader"};
	}
	$studentTable.='<td>'.$displaySub[0].'&nbsp;</td><td>'.$displaySub[1];
	if ($displaySub[2]) {
	    $studentTable.='Manually graded by '.$displaySub[2];
	}
	$studentTable.='&nbsp;</td></tr>';
    
    }
    $studentTable.='</table></td></tr></table>';
    return $studentTable;
}

sub updateGradeByPage {
    my ($request) = shift;

    my $cdom      = $env{"course.$env{'request.course.id'}.domain"};
    my $cnum      = $env{"course.$env{'request.course.id'}.num"};
    my $getsec    = $env{'form.section'} eq '' ? 'all' : $env{'form.section'};
    my $pageTitle = $env{'form.page'};
    my ($classlist,undef,$fullname) = &getclasslist($getsec,'1');
    my ($uname,$udom) = split(/:/,$env{'form.student'});
    my $usec=$classlist->{$env{'form.student'}}[5];
    if (!&canmodify($usec)) {
	$request->print('<span class="LC_warning">Unable to modify requested student.('.$env{'form.student'}.'</span>');
	$request->print(&show_grading_menu_form($env{'form.symb'}));
	return;
    }
    my $result='<h3><span class="LC_info">&nbsp;'.$env{'form.title'}.'</span></h3>';
    $result.='<h3>&nbsp;Student: '.&nameUserString(undef,$env{'form.fullname'},$uname,$udom).
	'</h3>'."\n";

    $request->print($result);

    my $navmap = Apache::lonnavmaps::navmap->new();
    my ($mapUrl, $id, $resUrl) = &Apache::lonnet::decode_symb( $env{'form.page'});
    my $map = $navmap->getResourceByUrl($resUrl); # add to navmaps
    if (!$map) {
	$request->print('<span class="LC_warning">Unable to grade requested sequence. ('.$resUrl.')</span>');
	my ($symb)=&get_symb($request);
	$request->print(&show_grading_menu_form($symb));
	return; 
    }
    my $iterator = $navmap->getIterator($map->map_start(),
					$map->map_finish());

    my $studentTable='<table border="0"><tr><td bgcolor="#777777">'.
	'<table border="0"><tr bgcolor="#e6ffff">'.
	'<td align="center"><b>&nbsp;Prob.&nbsp;</b></td>'.
	'<td><b>&nbsp;Title&nbsp;</b></td>'.
	'<td><b>&nbsp;Previous Score&nbsp;</b></td>'.
	'<td><b>&nbsp;New Score&nbsp;</b></td></tr>';

    $iterator->next(); # skip the first BEGIN_MAP
    my $curRes = $iterator->next(); # for "current resource"
    my ($depth,$question,$prob,$changeflag)= (1,1,1,0);
    while ($depth > 0) {
        if($curRes == $iterator->BEGIN_MAP) { $depth++; }
        if($curRes == $iterator->END_MAP) { $depth--; }

        if (ref($curRes) && $curRes->is_problem()) {
	    my $parts = $curRes->parts();
            my $title = $curRes->compTitle();
	    my $symbx = $curRes->symb();
	    $studentTable.='<tr bgcolor="#ffffe6"><td align="center" valign="top" >'.$prob.
		(scalar(@{$parts}) == 1 ? '' : '<br />('.scalar(@{$parts}).'&nbsp;parts)').'</td>';
	    $studentTable.='<td valign="top">&nbsp;<b>'.$title.'</b>&nbsp;</td>';

	    my %newrecord=();
	    my @displayPts=();
            my %aggregate = ();
            my $aggregateflag = 0;
	    foreach my $partid (@{$parts}) {
		my $newpts = $env{'form.GD_BOX'.$question.'_'.$partid};
		my $oldpts = $env{'form.oldpts'.$question.'_'.$partid};

		my $wgt = $env{'form.WGT'.$question.'_'.$partid} != 0 ? 
		    $env{'form.WGT'.$question.'_'.$partid} : 1;
		my $partial = $newpts/$wgt;
		my $score;
		if ($partial > 0) {
		    $score = 'correct_by_override';
		} elsif ($newpts ne '') { #empty is taken as 0
		    $score = 'incorrect_by_override';
		}
		my $dropMenu = $env{'form.GD_SEL'.$question.'_'.$partid};
		if ($dropMenu eq 'excused') {
		    $partial = '';
		    $score = 'excused';
		} elsif ($dropMenu eq 'reset status'
			 && $env{'form.solved'.$question.'_'.$partid} ne '') { #update only if previous record exists
		    $newrecord{'resource.'.$partid.'.tries'} = 0;
		    $newrecord{'resource.'.$partid.'.solved'} = '';
		    $newrecord{'resource.'.$partid.'.award'} = '';
		    $newrecord{'resource.'.$partid.'.awarded'} = 0;
		    $newrecord{'resource.'.$partid.'.regrader'} = "$env{'user.name'}:$env{'user.domain'}";
		    $changeflag++;
		    $newpts = '';
                    
                    my $aggtries =  $env{'form.aggtries'.$question.'_'.$partid};
                    my $totaltries = $env{'form.totaltries'.$question.'_'.$partid};
                    my $solvedstatus = $env{'form.solved'.$question.'_'.$partid};
                    if ($aggtries > 0) {
                        &decrement_aggs($symbx,$partid,\%aggregate,$aggtries,$totaltries,$solvedstatus);
                        $aggregateflag = 1;
                    }
		}
		my $display_part=&get_display_part($partid,$curRes->symb());
		my $oldstatus = $env{'form.solved'.$question.'_'.$partid};
		$displayPts[0].='&nbsp;<b>Part:</b> '.$display_part.' = '.
		    (($oldstatus eq 'excused') ? 'excused' : $oldpts).
		    '&nbsp;<br />';
		$displayPts[1].='&nbsp;<b>Part:</b> '.$display_part.' = '.
		     (($score eq 'excused') ? 'excused' : $newpts).
		    '&nbsp;<br />';
		$question++;
		next if ($dropMenu eq 'reset status' || ($newpts eq $oldpts && $score ne 'excused'));

		$newrecord{'resource.'.$partid.'.awarded'}  = $partial if $partial ne '';
		$newrecord{'resource.'.$partid.'.solved'}   = $score if $score ne '';
		$newrecord{'resource.'.$partid.'.regrader'} = "$env{'user.name'}:$env{'user.domain'}"
		    if (scalar(keys(%newrecord)) > 0);

		$changeflag++;
	    }
	    if (scalar(keys(%newrecord)) > 0) {
		my %record = 
		    &Apache::lonnet::restore($symbx,$env{'request.course.id'},
					     $udom,$uname);

		if (&Apache::lonnet::validCODE($env{'form.CODE'})) {
		    $newrecord{'resource.CODE'} = $env{'form.CODE'};
		} elsif (&Apache::lonnet::validCODE($record{'resource.CODE'})) {
		    $newrecord{'resource.CODE'} = '';
		}
		&Apache::lonnet::cstore(\%newrecord,$symbx,$env{'request.course.id'},
					$udom,$uname);
		%record = &Apache::lonnet::restore($symbx,
						   $env{'request.course.id'},
						   $udom,$uname);
		&check_and_remove_from_queue($parts,\%record,undef,$symbx,
					     $cdom,$cnum,$udom,$uname);
	    }
	    
            if ($aggregateflag) {
                &Apache::lonnet::cinc('nohist_resourcetracker',\%aggregate,
                      $env{'course.'.$env{'request.course.id'}.'.domain'},
                      $env{'course.'.$env{'request.course.id'}.'.num'});
            }

	    $studentTable.='<td valign="top">'.$displayPts[0].'</td>'.
		'<td valign="top">'.$displayPts[1].'</td>'.
		'</tr>';

	    $prob++;
	}
        $curRes = $iterator->next();
    }

    $studentTable.='</td></tr></table></td></tr></table>';
    $studentTable.=&show_grading_menu_form($env{'form.symb'});
    my $grademsg=($changeflag == 0 ? 'No score was changed or updated.' :
		  'The scores were changed for '.
		  $changeflag.' problem'.($changeflag == 1 ? '.' : 's.'));
    $request->print($grademsg.$studentTable);

    return '';
}

#-------- end of section for handling grading by page/sequence ---------
#
#-------------------------------------------------------------------

#--------------------Scantron Grading-----------------------------------
#
#------ start of section for handling grading by page/sequence ---------

sub defaultFormData {
    my ($symb)=@_;
    return '
      <input type="hidden" name="symb"    value="'.$symb.'" />'."\n".
     '<input type="hidden" name="saveState" value="'.$env{'form.saveState'}.'" />'."\n".
     '<input type="hidden" name="probTitle" value="'.$env{'form.probTitle'}.'" />'."\n";
}

sub getSequenceDropDown {
    my ($request,$symb)=@_;
    my $result='<select name="selectpage">'."\n";
    my ($titles,$symbx) = &getSymbMap($request);
    my ($curpage)=&Apache::lonnet::decode_symb($symb); 
    my $ctr=0;
    foreach (@$titles) {
	my ($minder,$showtitle) = ($_ =~ /(\d+)\.(.*)/);
	$result.='<option value="'.$$symbx{$_}.'" '.
	    ($$symbx{$_} =~ /$curpage$/ ? 'selected="on"' : '').
	    '>'.$showtitle.'</option>'."\n";
	$ctr++;
    }
    $result.= '</select>';
    return $result;
}

sub scantron_filenames {
    my $cdom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cname=$env{'course.'.$env{'request.course.id'}.'.num'};
    my @files=&Apache::lonnet::dirlist('userfiles',$cdom,$cname,
				    &propath($cdom,$cname));
    my @possiblenames;
    foreach my $filename (sort(@files)) {
	($filename)=split(/&/,$filename);
	if ($filename!~/^scantron_orig_/) { next ; }
	$filename=~s/^scantron_orig_//;
	push(@possiblenames,$filename);
    }
    return @possiblenames;
}

sub scantron_uploads {
    my ($file2grade) = @_;
    my $result=	'<select name="scantron_selectfile">';
    $result.="<option></option>";
    foreach my $filename (sort(&scantron_filenames())) {
	$result.="<option".($filename eq $file2grade ? ' selected="on"':'').">$filename</option>\n";
    }
    $result.="</select>";
    return $result;
}

sub scantron_scantab {
    my $fh=Apache::File->new($Apache::lonnet::perlvar{'lonTabDir'}.'/scantronformat.tab');
    my $result='<select name="scantron_format">'."\n";
    $result.='<option></option>'."\n";
    foreach my $line (<$fh>) {
	my ($name,$descrip)=split(/:/,$line);
	if ($name =~ /^\#/) { next; }
	$result.='<option value="'.$name.'">'.$descrip.'</option>'."\n";
    }
    $result.='</select>'."\n";

    return $result;
}

sub scantron_CODElist {
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    my @names=&Apache::lonnet::getkeys('CODEs',$cdom,$cnum);
    my $namechoice='<option></option>';
    foreach my $name (sort {uc($a) cmp uc($b)} @names) {
	if ($name =~ /^error: 2 /) { next; }
	if ($name =~ /^type\0/) { next; }
	$namechoice.='<option value="'.$name.'">'.$name.'</option>';
    }
    $namechoice='<select name="scantron_CODElist">'.$namechoice.'</select>';
    return $namechoice;
}

sub scantron_CODEunique {
    my $result='<span style="white-space: nowrap;">
                 <label><input type="radio" name="scantron_CODEunique"
                        value="yes" checked="checked" /> Yes </label>
                </span>
                <span style="white-space: nowrap;">
                 <label><input type="radio" name="scantron_CODEunique"
                        value="no" /> No </label>
                </span>';
    return $result;
}

sub scantron_selectphase {
    my ($r,$file2grade) = @_;
    my ($symb)=&get_symb($r);
    if (!$symb) {return '';}
    my $sequence_selector=&getSequenceDropDown($r,$symb);
    my $default_form_data=&defaultFormData($symb);
    my $grading_menu_button=&show_grading_menu_form($symb);
    my $file_selector=&scantron_uploads($file2grade);
    my $format_selector=&scantron_scantab();
    my $CODE_selector=&scantron_CODElist();
    my $CODE_unique=&scantron_CODEunique();
    my $result;
    #FIXME allow instructor to be able to download the scantron file
    # and to upload it,
    $result.= <<SCANTRONFORM;
    <table width="100%" border="0">
    <tr>
     <form method="post" enctype="multipart/form-data" action="/adm/grades" name="scantron_process">
      <td bgcolor="#777777">
       <input type="hidden" name="command" value="scantron_warning" />
        $default_form_data
        <table width="100%" border="0">
          <tr bgcolor="#e6ffff">
            <td colspan="2">
              &nbsp;<b>Specify file and which Folder/Sequence to grade</b>
            </td>
          </tr>
          <tr bgcolor="#ffffe6">
            <td> Sequence to grade: </td><td> $sequence_selector </td>
          </tr>
          <tr bgcolor="#ffffe6">
            <td> Filename of scoring office file: </td><td> $file_selector </td>
          </tr>
          <tr bgcolor="#ffffe6">
            <td> Format of data file: </td><td> $format_selector </td>
          </tr>
          <tr bgcolor="#ffffe6">
            <td> Saved CODEs to validate against: </td><td> $CODE_selector</td>
          </tr>
          <tr bgcolor="#ffffe6">
            <td> Each CODE is only to be used once:</td><td> $CODE_unique </td>
          </tr>
          <tr bgcolor="#ffffe6">
	    <td> Options: </td>
            <td>
	       <label><input type="checkbox" name="scantron_options_redo" value="redo_skipped"/> Do only previously skipped records</label> <br />
               <label><input type="checkbox" name="scantron_options_ignore" value="ignore_corrections"/> Remove all exisiting corrections</label> <br />
               <label><input type="checkbox" name="scantron_options_hidden" value="ignore_hidden"/> Skip hidden resources when grading</label>
	    </td>
          </tr>
          <tr bgcolor="#ffffe6">
            <td colspan="2">
              <input type="submit" value="Grading: Validate Scantron Records" />
            </td>
          </tr>
        </table>
       </td>
     </form>
    </tr>
SCANTRONFORM
   
    $r->print($result);

    if (&Apache::lonnet::allowed('usc',$env{'request.role.domain'}) ||
        &Apache::lonnet::allowed('usc',$env{'request.course.id'})) {

        $r->print(<<SCANTRONFORM);
    <tr>
      <td bgcolor="#777777">
        <table width="100%" border="0">
          <tr bgcolor="#e6ffff">
            <td>
              &nbsp;<b>Specify a Scantron data file to upload.</b>
            </td>
          </tr>
          <tr bgcolor="#ffffe6">
            <td>
SCANTRONFORM
    my $default_form_data=&defaultFormData(&get_symb($r,1));
    my $cdom= $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cnum= $env{'course.'.$env{'request.course.id'}.'.num'};
    $r->print(<<UPLOAD);
              <script type="text/javascript" language="javascript">
    function checkUpload(formname) {
	if (formname.upfile.value == "") {
	    alert("Please use the browse button to select a file from your local directory.");
	    return false;
	}
	formname.submit();
    }
              </script>

              <form enctype='multipart/form-data' action='/adm/grades' name='rules' method='post'>
                $default_form_data
                <input name='courseid' type='hidden' value='$cnum' />
                <input name='domainid' type='hidden' value='$cdom' />
                <input name='command' value='scantronupload_save' type='hidden' />
                File to upload:<input type="file" name="upfile" size="50" />
                <br />
                <input type="button" onClick="javascript:checkUpload(this.form);" value="Upload Scantron Data" />
              </form>
UPLOAD

        $r->print(<<SCANTRONFORM);
            </td>
          </tr>
        </table>
      </td>
    </tr>
SCANTRONFORM
    }
    $r->print(<<SCANTRONFORM);
    <tr>
      <form action='/adm/grades' name='scantron_download'>
        <td bgcolor="#777777">
	  $default_form_data
          <input type="hidden" name="command" value="scantron_download" />
          <table width="100%" border="0">
            <tr bgcolor="#e6ffff">
              <td colspan="2">
                &nbsp;<b>Download a scoring office file</b>
              </td>
            </tr>
            <tr bgcolor="#ffffe6">
              <td> Filename of scoring office file: </td><td> $file_selector </td>
            </tr>
            <tr bgcolor="#ffffe6">
              <td colspan="2">
                <input type="submit" value="Download: Show List of Associated Files" />
              </td>
            </tr>
          </table>
        </td>
      </form>
    </tr>
SCANTRONFORM

    $r->print(<<SCANTRONFORM);
  </table>
$grading_menu_button
SCANTRONFORM

    return
}

sub get_scantron_config {
    my ($which) = @_;
    my $fh=Apache::File->new($Apache::lonnet::perlvar{'lonTabDir'}.'/scantronformat.tab');
    my %config;
    #FIXME probably should move to XML it has already gotten a bit much now
    foreach my $line (<$fh>) {
	my ($name,$descrip)=split(/:/,$line);
	if ($name ne $which ) { next; }
	chomp($line);
	my @config=split(/:/,$line);
	$config{'name'}=$config[0];
	$config{'description'}=$config[1];
	$config{'CODElocation'}=$config[2];
	$config{'CODEstart'}=$config[3];
	$config{'CODElength'}=$config[4];
	$config{'IDstart'}=$config[5];
	$config{'IDlength'}=$config[6];
	$config{'Qstart'}=$config[7];
	$config{'Qlength'}=$config[8];
	$config{'Qoff'}=$config[9];
	$config{'Qon'}=$config[10];
	$config{'PaperID'}=$config[11];
	$config{'PaperIDlength'}=$config[12];
	$config{'FirstName'}=$config[13];
	$config{'FirstNamelength'}=$config[14];
	$config{'LastName'}=$config[15];
	$config{'LastNamelength'}=$config[16];
	last;
    }
    return %config;
}

sub username_to_idmap {
    my ($classlist)= @_;
    my %idmap;
    foreach my $student (keys(%$classlist)) {
	$idmap{$classlist->{$student}->[&Apache::loncoursedata::CL_ID]}=
	    $student;
    }
    return %idmap;
}

sub scantron_fixup_scanline {
    my ($scantron_config,$scan_data,$line,$whichline,$field,$args)=@_;
    if ($field eq 'ID') {
	if (length($args->{'newid'}) > $$scantron_config{'IDlength'}) {
	    return ($line,1,'New value too large');
	}
	if (length($args->{'newid'}) < $$scantron_config{'IDlength'}) {
	    $args->{'newid'}=sprintf('%-'.$$scantron_config{'IDlength'}.'s',
				     $args->{'newid'});
	}
	substr($line,$$scantron_config{'IDstart'}-1,
	       $$scantron_config{'IDlength'})=$args->{'newid'};
	if ($args->{'newid'}=~/^\s*$/) {
	    &scan_data($scan_data,"$whichline.user",
		       $args->{'username'}.':'.$args->{'domain'});
	}
    } elsif ($field eq 'CODE') {
	if ($args->{'CODE_ignore_dup'}) {
	    &scan_data($scan_data,"$whichline.CODE_ignore_dup",'1');
	}
	&scan_data($scan_data,"$whichline.useCODE",'1');
	if ($args->{'CODE'} ne 'use_unfound') {
	    if (length($args->{'CODE'}) > $$scantron_config{'CODElength'}) {
		return ($line,1,'New CODE value too large');
	    }
	    if (length($args->{'CODE'}) < $$scantron_config{'CODElength'}) {
		$args->{'CODE'}=sprintf('%-'.$$scantron_config{'CODElength'}.'s',$args->{'CODE'});
	    }
	    substr($line,$$scantron_config{'CODEstart'}-1,
		   $$scantron_config{'CODElength'})=$args->{'CODE'};
	}
    } elsif ($field eq 'answer') {
	my $length=$scantron_config->{'Qlength'};
	my $off=$scantron_config->{'Qoff'};
	my $on=$scantron_config->{'Qon'};
	my $answer=${off}x$length;
	if ($args->{'response'} eq 'none') {
	    &scan_data($scan_data,
		       "$whichline.no_bubble.".$args->{'question'},'1');
	} else {
	    if ($on eq 'letter') {
		my @alphabet=('A'..'Z');
		$answer=$alphabet[$args->{'response'}];
	    } elsif ($on eq 'number') {
		$answer=$args->{'response'}+1;
		if ($answer == 10) { $answer = '0'; }
	    } else {
		substr($answer,$args->{'response'},1)=$on;
	    }
	    &scan_data($scan_data,
		       "$whichline.no_bubble.".$args->{'question'},undef,'1');
	}
	my $where=$length*($args->{'question'}-1)+$scantron_config->{'Qstart'};
	substr($line,$where-1,$length)=$answer;
    }
    return $line;
}

sub scan_data {
    my ($scan_data,$key,$value,$delete)=@_;
    my $filename=$env{'form.scantron_selectfile'};
    if (defined($value)) {
	$scan_data->{$filename.'_'.$key} = $value;
    }
    if ($delete) { delete($scan_data->{$filename.'_'.$key}); }
    return $scan_data->{$filename.'_'.$key};
}

sub scantron_parse_scanline {
    my ($line,$whichline,$scantron_config,$scan_data,$justHeader)=@_;
    my %record;
    my $questions=substr($line,$$scantron_config{'Qstart'}-1);
    my $data=substr($line,0,$$scantron_config{'Qstart'}-1);
    if (!($$scantron_config{'CODElocation'} eq 0 ||
	  $$scantron_config{'CODElocation'} eq 'none')) {
	if ($$scantron_config{'CODElocation'} < 0 ||
	    $$scantron_config{'CODElocation'} eq 'letter' ||
	    $$scantron_config{'CODElocation'} eq 'number') {
	    $record{'scantron.CODE'}=substr($data,
					    $$scantron_config{'CODEstart'}-1,
					    $$scantron_config{'CODElength'});
	    if (&scan_data($scan_data,"$whichline.useCODE")) {
		$record{'scantron.useCODE'}=1;
	    }
	    if (&scan_data($scan_data,"$whichline.CODE_ignore_dup")) {
		$record{'scantron.CODE_ignore_dup'}=1;
	    }
	} else {
	    #FIXME interpret first N questions
	}
    }
    $record{'scantron.ID'}=substr($data,$$scantron_config{'IDstart'}-1,
				  $$scantron_config{'IDlength'});
    $record{'scantron.PaperID'}=
	substr($data,$$scantron_config{'PaperID'}-1,
	       $$scantron_config{'PaperIDlength'});
    $record{'scantron.FirstName'}=
	substr($data,$$scantron_config{'FirstName'}-1,
	       $$scantron_config{'FirstNamelength'});
    $record{'scantron.LastName'}=
	substr($data,$$scantron_config{'LastName'}-1,
	       $$scantron_config{'LastNamelength'});
    if ($justHeader) { return \%record; }

    my @alphabet=('A'..'Z');
    my $questnum=0;
    while ($questions) {
	$questnum++;
	my $currentquest=substr($questions,0,$$scantron_config{'Qlength'});
	substr($questions,0,$$scantron_config{'Qlength'})='';
	if (length($currentquest) < $$scantron_config{'Qlength'}) { next; }
	if ($$scantron_config{'Qon'} eq 'letter') {
	    if ($currentquest eq '?'
		|| $currentquest eq '*') {
		push(@{$record{'scantron.doubleerror'}},$questnum);
		$record{"scantron.$questnum.answer"}='';
	    } elsif (!defined($currentquest)
		     || $currentquest eq $$scantron_config{'Qoff'}
		     || $currentquest !~ /^[A-Z]$/) {
		$record{"scantron.$questnum.answer"}='';
		if (!&scan_data($scan_data,"$whichline.no_bubble.$questnum")) {
		    push(@{$record{"scantron.missingerror"}},$questnum);
		}
	    } else {
		$record{"scantron.$questnum.answer"}=$currentquest;
	    }
	} elsif ($$scantron_config{'Qon'} eq 'number') {
	    if ($currentquest eq '?'
		|| $currentquest eq '*') {
		push(@{$record{'scantron.doubleerror'}},$questnum);
		$record{"scantron.$questnum.answer"}='';
	    } elsif (!defined($currentquest)
		     || $currentquest eq $$scantron_config{'Qoff'} 
		     || $currentquest !~ /^\d$/) {
		$record{"scantron.$questnum.answer"}='';
		if (!&scan_data($scan_data,"$whichline.no_bubble.$questnum")) {
		    push(@{$record{"scantron.missingerror"}},$questnum);
		}
	    } else {
		# wrap zero back to J
		if ($currentquest eq '0') {
		    $record{"scantron.$questnum.answer"}=
			$alphabet[9];
		} else {
		    $record{"scantron.$questnum.answer"}=
			$alphabet[$currentquest-1];
		}
	    }
	} else {
	    my @array=split($$scantron_config{'Qon'},$currentquest,-1);
	    if (length($array[0]) eq $$scantron_config{'Qlength'}) {
		$record{"scantron.$questnum.answer"}='';
		if (!&scan_data($scan_data,"$whichline.no_bubble.$questnum")) {
		    push(@{$record{"scantron.missingerror"}},$questnum);
		}
	    } else {
		$record{"scantron.$questnum.answer"}=
		    $alphabet[length($array[0])];
	    }
	    if (scalar(@array) gt 2) {
		push(@{$record{'scantron.doubleerror'}},$questnum);
		my @ans=@array;
		my $i=length($ans[0]);shift(@ans);
		while ($#ans) {
		    $i+=length($ans[0])+1;
		    $record{"scantron.$questnum.answer"}.=$alphabet[$i];
		    shift(@ans);
		}
	    }
	}
    }
    $record{'scantron.maxquest'}=$questnum;
    return \%record;
}

sub scantron_add_delay {
    my ($delayqueue,$scanline,$errormessage,$errorcode)=@_;
    push(@$delayqueue,
	 {'line' => $scanline, 'emsg' => $errormessage,
	  'ecode' => $errorcode }
	 );
}

sub scantron_find_student {
    my ($scantron_record,$scan_data,$idmap,$line)=@_;
    my $scanID=$$scantron_record{'scantron.ID'};
    if ($scanID =~ /^\s*$/) {
 	return &scan_data($scan_data,"$line.user");
    }
    foreach my $id (keys(%$idmap)) {
 	if (lc($id) eq lc($scanID)) {
 	    return $$idmap{$id};
 	}
    }
    return undef;
}

sub scantron_filter {
    my ($curres)=@_;

    if (ref($curres) && $curres->is_problem()) {
	# if the user has asked to not have either hidden
	# or 'randomout' controlled resources to be graded
	# don't include them
	if ($env{'form.scantron_options_hidden'} eq 'ignore_hidden'
	    && $curres->randomout) {
	    return 0;
	}
	return 1;
    }
    return 0;
}

sub scantron_process_corrections {
    my ($r) = @_;
    my %scantron_config=&get_scantron_config($env{'form.scantron_format'});
    my ($scanlines,$scan_data)=&scantron_getfile();
    my $classlist=&Apache::loncoursedata::get_classlist();
    my $which=$env{'form.scantron_line'};
    my $line=&scantron_get_line($scanlines,$scan_data,$which);
    my ($skip,$err,$errmsg);
    if ($env{'form.scantron_skip_record'}) {
	$skip=1;
    } elsif ($env{'form.scantron_corrections'} =~ /^(duplicate|incorrect)ID$/) {
	my $newstudent=$env{'form.scantron_username'}.':'.
	    $env{'form.scantron_domain'};
	my $newid=$classlist->{$newstudent}->[&Apache::loncoursedata::CL_ID];
	($line,$err,$errmsg)=
	    &scantron_fixup_scanline(\%scantron_config,$scan_data,$line,$which,
				     'ID',{'newid'=>$newid,
				    'username'=>$env{'form.scantron_username'},
				    'domain'=>$env{'form.scantron_domain'}});
    } elsif ($env{'form.scantron_corrections'} =~ /^(duplicate|incorrect)CODE$/) {
	my $resolution=$env{'form.scantron_CODE_resolution'};
	my $newCODE;
	my %args;
	if      ($resolution eq 'use_unfound') {
	    $newCODE='use_unfound';
	} elsif ($resolution eq 'use_found') {
	    $newCODE=$env{'form.scantron_CODE_selectedvalue'};
	} elsif ($resolution eq 'use_typed') {
	    $newCODE=$env{'form.scantron_CODE_newvalue'};
	} elsif ($resolution =~ /^use_closest_(\d+)/) {
	    $newCODE=$env{"form.scantron_CODE_closest_$1"};
	}
	if ($env{'form.scantron_corrections'} eq 'duplicateCODE') {
	    $args{'CODE_ignore_dup'}=1;
	}
	$args{'CODE'}=$newCODE;
	($line,$err,$errmsg)=
	    &scantron_fixup_scanline(\%scantron_config,$scan_data,$line,$which,
				     'CODE',\%args);
    } elsif ($env{'form.scantron_corrections'} =~ /^(missing|double)bubble$/) {
	foreach my $question (split(',',$env{'form.scantron_questions'})) {
	    ($line,$err,$errmsg)=
		&scantron_fixup_scanline(\%scantron_config,$scan_data,$line,
					 $which,'answer',
					 { 'question'=>$question,
		       'response'=>$env{"form.scantron_correct_Q_$question"}});
	    if ($err) { last; }
	}
    }
    if ($err) {
	$r->print("<span class=\"LC_warning\">Unable to accept last correction, an error occurred :$errmsg:</span>");
    } else {
	&scantron_put_line($scanlines,$scan_data,$which,$line,$skip);
	&scantron_putfile($scanlines,$scan_data);
    }
}

sub reset_skipping_status {
    my ($scanlines,$scan_data)=&scantron_getfile();
    &scan_data($scan_data,'remember_skipping',undef,1);
    &scantron_putfile(undef,$scan_data);
}

sub start_skipping {
    my ($scan_data,$i)=@_;
    my %remembered=split(':',&scan_data($scan_data,'remember_skipping'));
    if ($env{'form.scantron_options_redo'} =~ /^redo_/) {
	$remembered{$i}=2;
    } else {
	$remembered{$i}=1;
    }
    &scan_data($scan_data,'remember_skipping',join(':',%remembered));
}

sub should_be_skipped {
    my ($scanlines,$scan_data,$i)=@_;
    if ($env{'form.scantron_options_redo'} !~ /^redo_/) {
	# not redoing old skips
	if ($scanlines->{'skipped'}[$i]) { return 1; }
	return 0;
    }
    my %remembered=split(':',&scan_data($scan_data,'remember_skipping'));

    if (exists($remembered{$i}) && $remembered{$i} != 2 ) {
	return 0;
    }
    return 1;
}

sub remember_current_skipped {
    my ($scanlines,$scan_data)=&scantron_getfile();
    my %to_remember;
    for (my $i=0;$i<=$scanlines->{'count'};$i++) {
	if ($scanlines->{'skipped'}[$i]) {
	    $to_remember{$i}=1;
	}
    }

    &scan_data($scan_data,'remember_skipping',join(':',%to_remember));
    &scantron_putfile(undef,$scan_data);
}

sub check_for_error {
    my ($r,$result)=@_;
    if ($result ne 'ok' && $result ne 'not_found' ) {
	$r->print("An error occured ($result) when trying to Remove the existing corrections.");
    }
}

sub scantron_warning_screen {
    my ($button_text)=@_;
    my $title=&Apache::lonnet::gettitle($env{'form.selectpage'});
    my %scantron_config=&get_scantron_config($env{'form.scantron_format'});
    my $CODElist;
    if ($scantron_config{'CODElocation'} &&
	$scantron_config{'CODEstart'} &&
	$scantron_config{'CODElength'}) {
	$CODElist=$env{'form.scantron_CODElist'};
	if ($env{'form.scantron_CODElist'} eq '') { $CODElist='<span class="LC_warning">None</span>'; }
	$CODElist=
	    '<tr><td><b>List of CODES to validate against:</b></td><td><tt>'.
	    $env{'form.scantron_CODElist'}.'</tt></td></tr>';
    }
    return (<<STUFF);
<p>
<span class="LC_warning">Please double check the information
                 below before clicking on '$button_text'</span>
</p>
<table>
<tr><td><b>Sequence to be Graded:</b></td><td>$title</td></tr>
<tr><td><b>Data File that will be used:</b></td><td><tt>$env{'form.scantron_selectfile'}</tt></td></tr>
$CODElist
</table>
<br />
<p> If this information is correct, please click on '$button_text'.</p>
<p> If something is incorrect, please click the 'Grading Menu' button to start over.</p>

<br />
STUFF
}

sub scantron_do_warning {
    my ($r)=@_;
    my ($symb)=&get_symb($r);
    if (!$symb) {return '';}
    my $default_form_data=&defaultFormData($symb);
    $r->print(&scantron_form_start().$default_form_data);
    if ( $env{'form.selectpage'} eq '' ||
	 $env{'form.scantron_selectfile'} eq '' ||
	 $env{'form.scantron_format'} eq '' ) {
	$r->print("<p>You have forgetten to specify some information. Please go Back and try again.</p>");
	if ( $env{'form.selectpage'} eq '') {
	    $r->print('<p><span class="LC_error">You have not selected a Sequence to grade</span></p>');
	} 
	if ( $env{'form.scantron_selectfile'} eq '') {
	    $r->print('<p><span class="LC_error">You have not selected a file that contains the student\'s response data.</span></p>');
	} 
	if ( $env{'form.scantron_format'} eq '') {
	    $r->print('<p><span class="LC_error">You have not selected a the format of the student\'s response data.</span></p>');
	} 
    } else {
	my $warning=&scantron_warning_screen('Grading: Validate Records');
	$r->print(<<STUFF);
$warning
<input type="submit" name="submit" value="Grading: Validate Records" />
<input type="hidden" name="command" value="scantron_validate" />
STUFF
    }
    $r->print("</form><br />".&show_grading_menu_form($symb));
    return '';
}

sub scantron_form_start {
    my ($max_bubble)=@_;
    my $result= <<SCANTRONFORM;
<form method="post" enctype="multipart/form-data" action="/adm/grades" name="scantronupload">
  <input type="hidden" name="selectpage" value="$env{'form.selectpage'}" />
  <input type="hidden" name="scantron_format" value="$env{'form.scantron_format'}" />
  <input type="hidden" name="scantron_selectfile" value="$env{'form.scantron_selectfile'}" />
  <input type="hidden" name="scantron_maxbubble" value="$max_bubble" />
  <input type="hidden" name="scantron_CODElist" value="$env{'form.scantron_CODElist'}" />
  <input type="hidden" name="scantron_CODEunique" value="$env{'form.scantron_CODEunique'}" />
  <input type="hidden" name="scantron_options_redo" value="$env{'form.scantron_options_redo'}" />
  <input type="hidden" name="scantron_options_ignore" value="$env{'form.scantron_options_ignore'}" />
  <input type="hidden" name="scantron_options_hidden" value="$env{'form.scantron_options_hidden'}" />
SCANTRONFORM
    return $result;
}

sub scantron_validate_file {
    my ($r) = @_;
    my ($symb)=&get_symb($r);
    if (!$symb) {return '';}
    my $default_form_data=&defaultFormData($symb);
    
    # do the detection of only doing skipped records first befroe we delete
    # them  when doing the corrections reset
    if ($env{'form.scantron_options_redo'} ne 'redo_skipped_ready') {
	&reset_skipping_status();
    }
    if ($env{'form.scantron_options_redo'} eq 'redo_skipped') {
	&remember_current_skipped();
	$env{'form.scantron_options_redo'}='redo_skipped_ready';
    }

    if ($env{'form.scantron_options_ignore'} eq 'ignore_corrections') {
	&check_for_error($r,&scantron_remove_file('corrected'));
	&check_for_error($r,&scantron_remove_file('skipped'));
	&check_for_error($r,&scantron_remove_scan_data());
	$env{'form.scantron_options_ignore'}='done';
    }

    if ($env{'form.scantron_corrections'}) {
	&scantron_process_corrections($r);
    }
    $r->print("<p>Gathering neccessary info.</p>");$r->rflush();
    #get the student pick code ready
    $r->print(&Apache::loncommon::studentbrowser_javascript());
    my $max_bubble=&scantron_get_maxbubble();
    my $result=&scantron_form_start($max_bubble).$default_form_data;
    $r->print($result);
    
    my @validate_phases=( 'sequence',
			  'ID',
			  'CODE',
			  'doublebubble',
			  'missingbubbles');
    if (!$env{'form.validatepass'}) {
	$env{'form.validatepass'} = 0;
    }
    my $currentphase=$env{'form.validatepass'};

    my $stop=0;
    while (!$stop && $currentphase < scalar(@validate_phases)) {
	$r->print("<p> Validating ".$validate_phases[$currentphase]."</p>");
	$r->rflush();
	my $which="scantron_validate_".$validate_phases[$currentphase];
	{
	    no strict 'refs';
	    ($stop,$currentphase)=&$which($r,$currentphase);
	}
    }
    if (!$stop) {
	my $warning=&scantron_warning_screen('Start Grading');
	$r->print(<<STUFF);
Validation process complete.<br />
$warning
<input type="submit" name="submit" value="Start Grading" />
<input type="hidden" name="command" value="scantron_process" />
STUFF

    } else {
	$r->print('<input type="hidden" name="command" value="scantron_validate" />');
	$r->print("<input type='hidden' name='validatepass' value='".$currentphase."' />");
    }
    if ($stop) {
	if ($validate_phases[$currentphase] eq 'sequence') {
	    $r->print('<input type="submit" name="submit" value="Ignore -> " />');
	    $r->print(' this error <br />');

	    $r->print(" <p>Or click the 'Grading Menu' button to start over.</p>");
	} else {
	    $r->print('<input type="submit" name="submit" value="Continue ->" />');
	    $r->print(' using corrected info <br />');
	    $r->print("<input type='submit' value='Skip' name='scantron_skip_record' />");
	    $r->print(" this scanline saving it for later.");
	}
    }
    $r->print(" </form><br />".&show_grading_menu_form($symb));
    return '';
}

sub scantron_remove_file {
    my ($which)=@_;
    my $cname=$env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    my $file='scantron_';
    if ($which eq 'corrected' || $which eq 'skipped') {
	$file.=$which.'_';
    } else {
	return 'refused';
    }
    $file.=$env{'form.scantron_selectfile'};
    return &Apache::lonnet::removeuserfile($cname,$cdom,$file);
}

sub scantron_remove_scan_data {
    my $cname=$env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    my @keys=&Apache::lonnet::getkeys('nohist_scantrondata',$cdom,$cname);
    my @todelete;
    my $filename=$env{'form.scantron_selectfile'};
    foreach my $key (@keys) {
	if ($key=~/^\Q$filename\E_/) {
	    if ($env{'form.scantron_options_redo'} eq 'redo_skipped_ready' &&
		$key=~/remember_skipping/) {
		next;
	    }
	    push(@todelete,$key);
	}
    }
    my $result;
    if (@todelete) {
	$result=&Apache::lonnet::del('nohist_scantrondata',\@todelete,$cdom,$cname);
    }
    return $result;
}

sub scantron_getfile {
    #FIXME really would prefer a scantron directory
    my $cname=$env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    my $lines;
    $lines=&Apache::lonnet::getfile('/uploaded/'.$cdom.'/'.$cname.'/'.
		       'scantron_orig_'.$env{'form.scantron_selectfile'});
    my %scanlines;
    $scanlines{'orig'}=[(split("\n",$lines,-1))];
    my $temp=$scanlines{'orig'};
    $scanlines{'count'}=$#$temp;

    $lines=&Apache::lonnet::getfile('/uploaded/'.$cdom.'/'.$cname.'/'.
		       'scantron_corrected_'.$env{'form.scantron_selectfile'});
    if ($lines eq '-1') {
	$scanlines{'corrected'}=[];
    } else {
	$scanlines{'corrected'}=[(split("\n",$lines,-1))];
    }
    $lines=&Apache::lonnet::getfile('/uploaded/'.$cdom.'/'.$cname.'/'.
		       'scantron_skipped_'.$env{'form.scantron_selectfile'});
    if ($lines eq '-1') {
	$scanlines{'skipped'}=[];
    } else {
	$scanlines{'skipped'}=[(split("\n",$lines,-1))];
    }
    my @tmp=&Apache::lonnet::dump('nohist_scantrondata',$cdom,$cname);
    if ($tmp[0] =~ /^(error:|no_such_host)/) { @tmp=(); }
    my %scan_data = @tmp;
    return (\%scanlines,\%scan_data);
}

sub lonnet_putfile {
    my ($contents,$filename)=@_;
    my $docuname=$env{'course.'.$env{'request.course.id'}.'.num'};
    my $docudom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    $env{'form.sillywaytopassafilearound'}=$contents;
    &Apache::lonnet::finishuserfileupload($docuname,$docudom,'sillywaytopassafilearound',$filename);

}

sub scantron_putfile {
    my ($scanlines,$scan_data) = @_;
    #FIXME really would prefer a scantron directory
    my $cname=$env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    if ($scanlines) {
	my $prefix='scantron_';
# no need to update orig, shouldn't change
#   &lonnet_putfile(join("\n",@{$scanlines->{'orig'}}),$prefix.'orig_'.
#		    $env{'form.scantron_selectfile'});
	&lonnet_putfile(join("\n",@{$scanlines->{'corrected'}}),
			$prefix.'corrected_'.
			$env{'form.scantron_selectfile'});
	&lonnet_putfile(join("\n",@{$scanlines->{'skipped'}}),
			$prefix.'skipped_'.
			$env{'form.scantron_selectfile'});
    }
    &Apache::lonnet::put('nohist_scantrondata',$scan_data,$cdom,$cname);
}

sub scantron_get_line {
    my ($scanlines,$scan_data,$i)=@_;
    if (&should_be_skipped($scanlines,$scan_data,$i)) { return undef; }
    #if ($scanlines->{'skipped'}[$i]) { return undef; }
    if ($scanlines->{'corrected'}[$i]) {return $scanlines->{'corrected'}[$i];}
    return $scanlines->{'orig'}[$i]; 
}

sub get_todo_count {
    my ($scanlines,$scan_data)=@_;
    my $count=0;
    for (my $i=0;$i<=$scanlines->{'count'};$i++) {
	my $line=&scantron_get_line($scanlines,$scan_data,$i);
	if ($line=~/^[\s\cz]*$/) { next; }
	$count++;
    }
    return $count;
}

sub scantron_put_line {
    my ($scanlines,$scan_data,$i,$newline,$skip)=@_;
    if ($skip) {
	$scanlines->{'skipped'}[$i]=$newline;
	&start_skipping($scan_data,$i);
	return;
    }
    $scanlines->{'corrected'}[$i]=$newline;
}

sub scantron_clear_skip {
    my ($scanlines,$scan_data,$i)=@_;
    if (exists($scanlines->{'skipped'}[$i])) {
	undef($scanlines->{'skipped'}[$i]);
	return 1;
    }
    return 0;
}

sub scantron_filter_not_exam {
    my ($curres)=@_;
    
    if (ref($curres) && $curres->is_problem() && !$curres->is_exam()) {
	# if the user has asked to not have either hidden
	# or 'randomout' controlled resources to be graded
	# don't include them
	if ($env{'form.scantron_options_hidden'} eq 'ignore_hidden'
	    && $curres->randomout) {
	    return 0;
	}
	return 1;
    }
    return 0;
}

sub scantron_validate_sequence {
    my ($r,$currentphase) = @_;

    my $navmap=Apache::lonnavmaps::navmap->new();
    my (undef,undef,$sequence)=
	&Apache::lonnet::decode_symb($env{'form.selectpage'});

    my $map=$navmap->getResourceByUrl($sequence);

    $r->print('<input type="hidden" name="validate_sequence_exam"
                                    value="ignore" />');
    if ($env{'form.validate_sequence_exam'} ne 'ignore') {
	my @resources=
	    $navmap->retrieveResources($map,\&scantron_filter_not_exam,1,0);
	if (@resources) {
	    $r->print("<p>".&mt('Some resources in the sequence currently are not set to exam mode. Grading these resources currently may not work correctly.')."</p>");
	    return (1,$currentphase);
	}
    }

    return (0,$currentphase+1);
}

sub scantron_validate_ID {
    my ($r,$currentphase) = @_;
    
    #get student info
    my $classlist=&Apache::loncoursedata::get_classlist();
    my %idmap=&username_to_idmap($classlist);

    #get scantron line setup
    my %scantron_config=&get_scantron_config($env{'form.scantron_format'});
    my ($scanlines,$scan_data)=&scantron_getfile();

    my %found=('ids'=>{},'usernames'=>{});
    for (my $i=0;$i<=$scanlines->{'count'};$i++) {
	my $line=&scantron_get_line($scanlines,$scan_data,$i);
	if ($line=~/^[\s\cz]*$/) { next; }
	my $scan_record=&scantron_parse_scanline($line,$i,\%scantron_config,
						 $scan_data);
	my $id=$$scan_record{'scantron.ID'};
	my $found;
	foreach my $checkid (keys(%idmap)) {
	    if (lc($checkid) eq lc($id)) { $found=$checkid;last; }
	}
	if ($found) {
	    my $username=$idmap{$found};
	    if ($found{'ids'}{$found}) {
		&scantron_get_correction($r,$i,$scan_record,\%scantron_config,
					 $line,'duplicateID',$found);
		return(1,$currentphase);
	    } elsif ($found{'usernames'}{$username}) {
		&scantron_get_correction($r,$i,$scan_record,\%scantron_config,
					 $line,'duplicateID',$username);
		return(1,$currentphase);
	    }
	    #FIXME store away line we previously saw the ID on to use above
	    $found{'ids'}{$found}++;
	    $found{'usernames'}{$username}++;
	} else {
	    if ($id =~ /^\s*$/) {
		my $username=&scan_data($scan_data,"$i.user");
		if (defined($username) && $found{'usernames'}{$username}) {
		    &scantron_get_correction($r,$i,$scan_record,
					     \%scantron_config,
					     $line,'duplicateID',$username);
		    return(1,$currentphase);
		} elsif (!defined($username)) {
		    &scantron_get_correction($r,$i,$scan_record,
					     \%scantron_config,
					     $line,'incorrectID');
		    return(1,$currentphase);
		}
		$found{'usernames'}{$username}++;
	    } else {
		&scantron_get_correction($r,$i,$scan_record,\%scantron_config,
					 $line,'incorrectID');
		return(1,$currentphase);
	    }
	}
    }

    return (0,$currentphase+1);
}

sub scantron_get_correction {
    my ($r,$i,$scan_record,$scan_config,$line,$error,$arg)=@_;

#FIXME in the case of a duplicated ID the previous line, probaly need
#to show both the current line and the previous one and allow skipping
#the previous one or the current one

    $r->print("<p><b>An error was detected ($error)</b>");
    if ( $$scan_record{'scantron.PaperID'} =~ /\S/) {
	$r->print(" for PaperID <tt>".
		  $$scan_record{'scantron.PaperID'}."</tt> \n");
    } else {
	$r->print(" in scanline $i <pre>".
		  $line."</pre> \n");
    }
    my $message="<p>The ID on the form is  <tt>".
	$$scan_record{'scantron.ID'}."</tt><br />\n".
	"The name on the paper is ".
	$$scan_record{'scantron.LastName'}.",".
	$$scan_record{'scantron.FirstName'}."</p>";

    $r->print('<input type="hidden" name="scantron_corrections" value="'.$error.'" />'."\n");
    $r->print('<input type="hidden" name="scantron_line" value="'.$i.'" />'."\n");
    if ($error =~ /ID$/) {
	if ($error eq 'incorrectID') {
	    $r->print("The encoded ID is not in the classlist</p>\n");
	} elsif ($error eq 'duplicateID') {
	    $r->print("The encoded ID has also been used by a previous paper $arg</p>\n");
	}
	$r->print($message);
	$r->print("<p>How should I handle this? <br /> \n");
	$r->print("\n<ul><li> ");
	#FIXME it would be nice if this sent back the user ID and
	#could do partial userID matches
	$r->print(&Apache::loncommon::selectstudent_link('scantronupload',
				       'scantron_username','scantron_domain'));
	$r->print(": <input type='text' name='scantron_username' value='' />");
	$r->print("\n@".
		 &Apache::loncommon::select_dom_form($env{'request.role.domain'},'scantron_domain'));

	$r->print('</li>');
    } elsif ($error =~ /CODE$/) {
	if ($error eq 'incorrectCODE') {
	    $r->print("</p><p>The encoded CODE is not in the list of possible CODEs</p>\n");
	} elsif ($error eq 'duplicateCODE') {
	    $r->print("</p><p>The encoded CODE has also been used by a previous paper ".join(', ',@{$arg}).", and CODEs are supposed to be unique</p>\n");
	}
	$r->print("<p>The CODE on the form is  <tt>'".
		  $$scan_record{'scantron.CODE'}."'</tt><br />\n");
	$r->print($message);
	$r->print("<p>How should I handle this? <br /> \n");
	$r->print("\n<br /> ");
	my $i=0;
	if ($error eq 'incorrectCODE' 
	    && $$scan_record{'scantron.CODE'}=~/\S/ ) {
	    my ($max,$closest)=&scantron_get_closely_matching_CODEs($arg,$$scan_record{'scantron.CODE'});
	    if ($closest > 0) {
		foreach my $testcode (@{$closest}) {
		    my $checked='';
		    if (!$i) { $checked=' checked="on" '; }
		    $r->print("<label><input type='radio' name='scantron_CODE_resolution' value='use_closest_$i' $checked /> Use the similar CODE <b><tt>".$testcode."</tt></b> instead.</label><input type='hidden' name='scantron_CODE_closest_$i' value='$testcode' />");
		    $r->print("\n<br />");
		    $i++;
		}
	    }
	}
	if ($$scan_record{'scantron.CODE'}=~/\S/ ) {
	    my $checked; if (!$i) { $checked=' checked="on" '; }
	    $r->print("<label><input type='radio' name='scantron_CODE_resolution' value='use_unfound' $checked /> Use the CODE <b><tt>".$$scan_record{'scantron.CODE'}."</tt></b> that is was on the paper, ignoring the error.</label>");
	    $r->print("\n<br />");
	}

	$r->print(<<ENDSCRIPT);
<script type="text/javascript">
function change_radio(field) {
    var slct=document.scantronupload.scantron_CODE_resolution;
    var i;
    for (i=0;i<slct.length;i++) {
        if (slct[i].value==field) { slct[i].checked=true; }
    }
}
</script>
ENDSCRIPT
	my $href="/adm/pickcode?".
	   "form=".&escape("scantronupload").
	   "&scantron_format=".&escape($env{'form.scantron_format'}).
	   "&scantron_CODElist=".&escape($env{'form.scantron_CODElist'}).
	   "&curCODE=".&escape($$scan_record{'scantron.CODE'}).
	   "&scantron_selectfile=".&escape($env{'form.scantron_selectfile'});
	if ($env{'form.scantron_CODElist'} =~ /\S/) { 
	    $r->print("<label><input type='radio' name='scantron_CODE_resolution' value='use_found' /> <a target='_blank' href='$href'>Select</a> a CODE from the list of all CODEs and use it.</label> Selected CODE is <input readonly='true' type='text' size='8' name='scantron_CODE_selectedvalue' onfocus=\"javascript:change_radio('use_found')\" onchange=\"javascript:change_radio('use_found')\" />");
	    $r->print("\n<br />");
	}
	$r->print("<label><input type='radio' name='scantron_CODE_resolution' value='use_typed' /> Use </label><input type='text' size='8' name='scantron_CODE_newvalue' onfocus=\"javascript:change_radio('use_typed')\" onkeypress=\"javascript:change_radio('use_typed')\" /> as the CODE.");
	$r->print("\n<br /><br />");
    } elsif ($error eq 'doublebubble') {
	$r->print("<p>There have been multiple bubbles scanned for a some question(s)</p>\n");
	$r->print('<input type="hidden" name="scantron_questions" value="'.
		  join(',',@{$arg}).'" />');
	$r->print($message);
	$r->print("<p>Please indicate which bubble should be used for grading</p>");
	foreach my $question (@{$arg}) {
	    my $selected=$$scan_record{"scantron.$question.answer"};
	    &scantron_bubble_selector($r,$scan_config,$question,split('',$selected));
	}
    } elsif ($error eq 'missingbubble') {
	$r->print("<p>There have been <b>no</b> bubbles scanned for some question(s)</p>\n");
	$r->print($message);
	$r->print("<p>Please indicate which bubble should be used for grading</p>");
	$r->print("Some questions have no scanned bubbles\n");
	$r->print('<input type="hidden" name="scantron_questions" value="'.
		  join(',',@{$arg}).'" />');
	foreach my $question (@{$arg}) {
	    my $selected=$$scan_record{"scantron.$question.answer"};
	    &scantron_bubble_selector($r,$scan_config,$question);
	}
    } else {
	$r->print("\n<ul>");
    }
    $r->print("\n</li></ul>");

}

sub scantron_bubble_selector {
    my ($r,$scan_config,$quest,@selected)=@_;
    my $max=$$scan_config{'Qlength'};

    my $scmode=$$scan_config{'Qon'};
    if ($scmode eq 'number' || $scmode eq 'letter') { $max=10; }	     

    my @alphabet=('A'..'Z');
    $r->print("<table border='1'><tr><td rowspan='2'>$quest</td>");
    for (my $i=0;$i<$max+1;$i++) {
	$r->print("\n".'<td align="center">');
	if ($selected[0] eq $alphabet[$i]) { $r->print('X'); shift(@selected) }
	else { $r->print('&nbsp;'); }
	$r->print('</td>');
    }
    $r->print('</tr><tr>');
    for (my $i=0;$i<$max;$i++) {
	$r->print("\n".
		  '<td><label><input type="radio" name="scantron_correct_Q_'.
		  $quest.'" value="'.$i.'" />'.$alphabet[$i]."</label></td>");
    }
    $r->print('<td><label><input type="radio" name="scantron_correct_Q_'.
	      $quest.'" value="none" /> No bubble </label></td>');
    $r->print('</tr></table>');
}

sub num_matches {
    my ($orig,$code) = @_;
    my @code=split(//,$code);
    my @orig=split(//,$orig);
    my $same=0;
    for (my $i=0;$i<scalar(@code);$i++) {
	if ($code[$i] eq $orig[$i]) { $same++; }
    }
    return $same;
}

sub scantron_get_closely_matching_CODEs {
    my ($allcodes,$CODE)=@_;
    my @CODEs;
    foreach my $testcode (sort(keys(%{$allcodes}))) {
	push(@{$CODEs[&num_matches($CODE,$testcode)]},$testcode);
    }

    return ($#CODEs,$CODEs[-1]);
}

sub get_codes {
    my ($old_name, $cdom, $cnum) = @_;
    if (!$old_name) {
	$old_name=$env{'form.scantron_CODElist'};
    }
    if (!$cdom) {
	$cdom =$env{'course.'.$env{'request.course.id'}.'.domain'};
    }
    if (!$cnum) {
	$cnum =$env{'course.'.$env{'request.course.id'}.'.num'};
    }
    my %result=&Apache::lonnet::get('CODEs',[$old_name,"type\0$old_name"],
				    $cdom,$cnum);
    my %allcodes;
    if ($result{"type\0$old_name"} eq 'number') {
	%allcodes=map {($_,1)} split(',',$result{$old_name});
    } else {
	%allcodes=map {(&Apache::lonprintout::num_to_letters($_),1)} split(',',$result{$old_name});
    }
    return %allcodes;
}

sub scantron_validate_CODE {
    my ($r,$currentphase) = @_;
    my %scantron_config=&get_scantron_config($env{'form.scantron_format'});
    if ($scantron_config{'CODElocation'} &&
	$scantron_config{'CODEstart'} &&
	$scantron_config{'CODElength'}) {
	if (!defined($env{'form.scantron_CODElist'})) {
	    &FIXME_blow_up()
	}
    } else {
	return (0,$currentphase+1);
    }
    
    my %usedCODEs;

    my %allcodes=&get_codes();

    my ($scanlines,$scan_data)=&scantron_getfile();
    for (my $i=0;$i<=$scanlines->{'count'};$i++) {
	my $line=&scantron_get_line($scanlines,$scan_data,$i);
	if ($line=~/^[\s\cz]*$/) { next; }
	my $scan_record=&scantron_parse_scanline($line,$i,\%scantron_config,
						 $scan_data);
	my $CODE=$$scan_record{'scantron.CODE'};
	my $error=0;
	if (!&Apache::lonnet::validCODE($CODE)) {
	    &scantron_get_correction($r,$i,$scan_record,
				     \%scantron_config,
				     $line,'incorrectCODE',\%allcodes);
	    return(1,$currentphase);
	}
	if (%allcodes && !exists($allcodes{$CODE}) 
	    && !$$scan_record{'scantron.useCODE'}) {
	    &scantron_get_correction($r,$i,$scan_record,
				     \%scantron_config,
				     $line,'incorrectCODE',\%allcodes);
	    return(1,$currentphase);
	}
	if (exists($usedCODEs{$CODE}) 
	    && $env{'form.scantron_CODEunique'} eq 'yes'
	    && !$$scan_record{'scantron.CODE_ignore_dup'}) {
	    &scantron_get_correction($r,$i,$scan_record,
				     \%scantron_config,
				     $line,'duplicateCODE',$usedCODEs{$CODE});
	    return(1,$currentphase);
	}
	push (@{$usedCODEs{$CODE}},$$scan_record{'scantron.PaperID'});
    }
    return (0,$currentphase+1);
}

sub scantron_validate_doublebubble {
    my ($r,$currentphase) = @_;
    #get student info
    my $classlist=&Apache::loncoursedata::get_classlist();
    my %idmap=&username_to_idmap($classlist);

    #get scantron line setup
    my %scantron_config=&get_scantron_config($env{'form.scantron_format'});
    my ($scanlines,$scan_data)=&scantron_getfile();
    for (my $i=0;$i<=$scanlines->{'count'};$i++) {
	my $line=&scantron_get_line($scanlines,$scan_data,$i);
	if ($line=~/^[\s\cz]*$/) { next; }
	my $scan_record=&scantron_parse_scanline($line,$i,\%scantron_config,
						 $scan_data);
	if (!defined($$scan_record{'scantron.doubleerror'})) { next; }
	&scantron_get_correction($r,$i,$scan_record,\%scantron_config,$line,
				 'doublebubble',
				 $$scan_record{'scantron.doubleerror'});
    	return (1,$currentphase);
    }
    return (0,$currentphase+1);
}

sub scantron_get_maxbubble {    
    if (defined($env{'form.scantron_maxbubble'}) &&
	$env{'form.scantron_maxbubble'}) {
	return $env{'form.scantron_maxbubble'};
    }

    my $navmap=Apache::lonnavmaps::navmap->new();
    my (undef,undef,$sequence)=
	&Apache::lonnet::decode_symb($env{'form.selectpage'});

    my $map=$navmap->getResourceByUrl($sequence);
    my @resources=$navmap->retrieveResources($map,\&scantron_filter,1,0);

    &Apache::lonxml::clear_problem_counter();

    foreach my $resource (@resources) {
	my $result=&Apache::lonnet::ssi($resource->src(),
					('symb' => $resource->symb()));
    }
    &Apache::lonnet::delenv('scantron\.');
    $env{'form.scantron_maxbubble'} =
	&Apache::lonxml::get_problem_counter()-1;

    return $env{'form.scantron_maxbubble'};
}

sub scantron_validate_missingbubbles {
    my ($r,$currentphase) = @_;
    #get student info
    my $classlist=&Apache::loncoursedata::get_classlist();
    my %idmap=&username_to_idmap($classlist);

    #get scantron line setup
    my %scantron_config=&get_scantron_config($env{'form.scantron_format'});
    my ($scanlines,$scan_data)=&scantron_getfile();
    my $max_bubble=&scantron_get_maxbubble();
    if (!$max_bubble) { $max_bubble=2**31; }
    for (my $i=0;$i<=$scanlines->{'count'};$i++) {
	my $line=&scantron_get_line($scanlines,$scan_data,$i);
	if ($line=~/^[\s\cz]*$/) { next; }
	my $scan_record=&scantron_parse_scanline($line,$i,\%scantron_config,
						 $scan_data);
	if (!defined($$scan_record{'scantron.missingerror'})) { next; }
	my @to_correct;
	foreach my $missing (@{$$scan_record{'scantron.missingerror'}}) {
	    if ($missing > $max_bubble) { next; }
	    push(@to_correct,$missing);
	}
	if (@to_correct) {
	    &scantron_get_correction($r,$i,$scan_record,\%scantron_config,
				     $line,'missingbubble',\@to_correct);
	    return (1,$currentphase);
	}

    }
    return (0,$currentphase+1);
}

sub scantron_process_students {
    my ($r) = @_;
    my (undef,undef,$sequence)=&Apache::lonnet::decode_symb($env{'form.selectpage'});
    my ($symb)=&get_symb($r);
    if (!$symb) {return '';}
    my $default_form_data=&defaultFormData($symb);

    my %scantron_config=&get_scantron_config($env{'form.scantron_format'});
    my ($scanlines,$scan_data)=&scantron_getfile();
    my $classlist=&Apache::loncoursedata::get_classlist();
    my %idmap=&username_to_idmap($classlist);
    my $navmap=Apache::lonnavmaps::navmap->new();
    my $map=$navmap->getResourceByUrl($sequence);
    my @resources=$navmap->retrieveResources($map,\&scantron_filter,1,0);
#    $r->print("geto ".scalar(@resources)."<br />");
    my $result= <<SCANTRONFORM;
<form method="post" enctype="multipart/form-data" action="/adm/grades" name="scantronupload">
  <input type="hidden" name="command" value="scantron_configphase" />
  $default_form_data
SCANTRONFORM
    $r->print($result);

    my @delayqueue;
    my %completedstudents;
    
    my $count=&get_todo_count($scanlines,$scan_data);
    my %prog_state=&Apache::lonhtmlcommon::Create_PrgWin($r,'Scantron Status',
 				    'Scantron Progress',$count,
				    'inline',undef,'scantronupload');
    &Apache::lonhtmlcommon::Update_PrgWin($r,\%prog_state,
					  'Processing first student');
    my $start=&Time::HiRes::time();
    my $i=-1;
    my ($uname,$udom,$started);
    while ($i<$scanlines->{'count'}) {
 	($uname,$udom)=('','');
 	$i++;
 	my $line=&scantron_get_line($scanlines,$scan_data,$i);
 	if ($line=~/^[\s\cz]*$/) { next; }
	if ($started) {
	    &Apache::lonhtmlcommon::Increment_PrgWin($r,\%prog_state,
						     'last student');
	}
	$started=1;
 	my $scan_record=&scantron_parse_scanline($line,$i,\%scantron_config,
 						 $scan_data);
 	unless ($uname=&scantron_find_student($scan_record,$scan_data,
 					      \%idmap,$i)) {
  	    &scantron_add_delay(\@delayqueue,$line,
 				'Unable to find a student that matches',1);
 	    next;
  	}
 	if (exists $completedstudents{$uname}) {
 	    &scantron_add_delay(\@delayqueue,$line,
 				'Student '.$uname.' has multiple sheets',2);
 	    next;
 	}
  	($uname,$udom)=split(/:/,$uname);

	&Apache::lonxml::clear_problem_counter();
  	&Apache::lonnet::appenv(%$scan_record);

	if (&scantron_clear_skip($scanlines,$scan_data,$i)) {
	    &scantron_putfile($scanlines,$scan_data);
	}
	
	my $i=0;
	foreach my $resource (@resources) {
	    $i++;
	    my %form=('submitted'     =>'scantron',
		      'grade_target'  =>'grade',
		      'grade_username'=>$uname,
		      'grade_domain'  =>$udom,
		      'grade_courseid'=>$env{'request.course.id'},
		      'grade_symb'    =>$resource->symb());
	    if (exists($scan_record->{'scantron.CODE'})
		&& 
		&Apache::lonnet::validCODE($scan_record->{'scantron.CODE'})) {
		$form{'CODE'}=$scan_record->{'scantron.CODE'};
	    } else {
		$form{'CODE'}='';
	    }
	    my $result=&Apache::lonnet::ssi($resource->src(),%form);
	    if ($result ne '') {
		&Apache::lonnet::logthis("scantron grading error -> $result");
		&Apache::lonnet::logthis("scantron grading error info name $uname domain $udom course $env{'request.course.id'} url ".$resource->src());
	    }
	    if (&Apache::loncommon::connection_aborted($r)) { last; }
	}
	$completedstudents{$uname}={'line'=>$line};
	if (&Apache::loncommon::connection_aborted($r)) { last; }
    } continue {
	&Apache::lonxml::clear_problem_counter();
	&Apache::lonnet::delenv('scantron\.');
    }
    &Apache::lonhtmlcommon::Close_PrgWin($r,\%prog_state);
#    my $lasttime = &Time::HiRes::time()-$start;
#    $r->print("<p>took $lasttime</p>");

    $r->print("</form>");
    $r->print(&show_grading_menu_form($symb));
    return '';
}

sub scantron_upload_scantron_data {
    my ($r)=@_;
    $r->print(&Apache::loncommon::coursebrowser_javascript($env{'request.role.domain'}));
    my $select_link=&Apache::loncommon::selectcourse_link('rules','courseid',
							  'domainid',
							  'coursename');
    my $domsel=&Apache::loncommon::select_dom_form($env{'request.role.domain'},
						   'domainid');
    my $default_form_data=&defaultFormData(&get_symb($r,1));
    $r->print(<<UPLOAD);
<script type="text/javascript" language="javascript">
    function checkUpload(formname) {
	if (formname.upfile.value == "") {
	    alert("Please use the browse button to select a file from your local directory.");
	    return false;
	}
	formname.submit();
    }
</script>

<form enctype='multipart/form-data' action='/adm/grades' name='rules' method='post'>
$default_form_data
<table>
<tr><td>$select_link </td></tr>
<tr><td>Course ID:   </td><td><input name='courseid' type='text' />  </td></tr>
<tr><td>Course Name: </td><td><input name='coursename' type='text' /></td></tr>
<tr><td>Domain:      </td><td>$domsel                                </td></tr>
<tr><td>File to upload:</td><td><input type="file" name="upfile" size="50" /></td></tr>
</table>
<input name='command' value='scantronupload_save' type='hidden' />
<input type="button" onClick="javascript:checkUpload(this.form);" value="Upload Scantron Data" />
</form>
UPLOAD
    return '';
}

sub scantron_upload_scantron_data_save {
    my($r)=@_;
    my ($symb)=&get_symb($r,1);
    my $doanotherupload=
	'<br /><form action="/adm/grades" method="post">'."\n".
	'<input type="hidden" name="command" value="scantronupload" />'."\n".
	'<input type="submit" name="submit" value="Do Another Upload" />'."\n".
	'</form>'."\n";
    if (!&Apache::lonnet::allowed('usc',$env{'form.domainid'}) &&
	!&Apache::lonnet::allowed('usc',
			    $env{'form.domainid'}.'_'.$env{'form.courseid'})) {
	$r->print("You are not allowed to upload Scantron data to the requested course.<br />");
	if ($symb) {
	    $r->print(&show_grading_menu_form($symb));
	} else {
	    $r->print($doanotherupload);
	}
	return '';
    }
    my %coursedata=&Apache::lonnet::coursedescription($env{'form.domainid'}.'_'.$env{'form.courseid'});
    $r->print("Doing upload to ".$coursedata{'description'}." <br />");
    my $fname=$env{'form.upfile.filename'};
    #FIXME
    #copied from lonnet::userfileupload()
    #make that function able to target a specified course
    # Replace Windows backslashes by forward slashes
    $fname=~s/\\/\//g;
    # Get rid of everything but the actual filename
    $fname=~s/^.*\/([^\/]+)$/$1/;
    # Replace spaces by underscores
    $fname=~s/\s+/\_/g;
    # Replace all other weird characters by nothing
    $fname=~s/[^\w\.\-]//g;
    # See if there is anything left
    unless ($fname) { return 'error: no uploaded file'; }
    my $uploadedfile=$fname;
    $fname='scantron_orig_'.$fname;
    if (length($env{'form.upfile'}) < 2) {
	$r->print("<span class=\"LC_error\">Error:</span> The file you attempted to upload, <tt>".&HTML::Entities::encode($env{'form.upfile.filename'},'<>&"')."</tt>, contained no information. Please check that you entered the correct filename.");
    } else {
	my $result=&Apache::lonnet::finishuserfileupload($env{'form.courseid'},$env{'form.domainid'},'upfile',$fname);
	if ($result =~ m|^/uploaded/|) {
	    $r->print("<span class=\"LC_success\">Success:</span> Successfully uploaded ".(length($env{'form.upfile'})-1)." bytes of data into location <tt>".$result."</tt>");
	} else {
	    $r->print("<span class=\"LC_error\">Error:</span> An error (".$result.") occurred when attempting to upload the file, <tt>".&HTML::Entities::encode($env{'form.upfile.filename'},'<>&"')."</tt>");
	}
    }
    if ($symb) {
	$r->print(&scantron_selectphase($r,$uploadedfile));
    } else {
	$r->print($doanotherupload);
    }
    return '';
}

sub valid_file {
    my ($requested_file)=@_;
    foreach my $filename (sort(&scantron_filenames())) {
	if ($requested_file eq $filename) { return 1; }
    }
    return 0;
}

sub scantron_download_scantron_data {
    my ($r)=@_;
    my $default_form_data=&defaultFormData(&get_symb($r,1));
    my $cname=$env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    my $file=$env{'form.scantron_selectfile'};
    if (! &valid_file($file)) {
	$r->print(<<ERROR);
	<p>
	    The requested file name was invalid.
        </p>
ERROR
	$r->print(&show_grading_menu_form(&get_symb($r,1)));
	return;
    }
    my $orig='/uploaded/'.$cdom.'/'.$cname.'/scantron_orig_'.$file;
    my $corrected='/uploaded/'.$cdom.'/'.$cname.'/scantron_corrected_'.$file;
    my $skipped='/uploaded/'.$cdom.'/'.$cname.'/scantron_skipped_'.$file;
    &Apache::lonnet::allowuploaded('/adm/grades',$orig);
    &Apache::lonnet::allowuploaded('/adm/grades',$corrected);
    &Apache::lonnet::allowuploaded('/adm/grades',$skipped);
    $r->print(<<DOWNLOAD);
    <p>
	<a href="$orig">Original</a> file as uploaded by the scantron office.
    </p>
    <p>
	<a href="$corrected">Corrections</a>, a file of corrected records that were used in grading.
    </p>
    <p>
	<a href="$skipped">Skipped</a>, a file of records that were skipped.
    </p>
DOWNLOAD
    $r->print(&show_grading_menu_form(&get_symb($r,1)));
    return '';
}

#-------- end of section for handling grading scantron forms -------
#
#-------------------------------------------------------------------

#-------------------------- Menu interface -------------------------
#
#--- Show a Grading Menu button - Calls the next routine ---
sub show_grading_menu_form {
    my ($symb)=@_;
    my $result.='<br /><form action="/adm/grades" method="post">'."\n".
	'<input type="hidden" name="symb" value="'.$symb.'" />'."\n".
	'<input type="hidden" name="saveState"  value="'.$env{'form.saveState'}.'" />'."\n".
	'<input type="hidden" name="command" value="gradingmenu" />'."\n".
	'<input type="submit" name="submit" value="Grading Menu" />'."\n".
	'</form>'."\n";
    return $result;
}

# -- Retrieve choices for grading form
sub savedState {
    my %savedState = ();
    if ($env{'form.saveState'}) {
	foreach (split(/:/,$env{'form.saveState'})) {
	    my ($key,$value) = split(/=/,$_,2);
	    $savedState{$key} = $value;
	}
    }
    return \%savedState;
}

#--- Displays the main menu page -------
sub gradingmenu {
    my ($request) = @_;
    my ($symb)=&get_symb($request);
    if (!$symb) {return '';}
    my $probTitle = &Apache::lonnet::gettitle($symb);

    $request->print(<<GRADINGMENUJS);
<script type="text/javascript" language="javascript">
    function checkChoice(formname,val,cmdx) {
	if (val <= 2) {
	    var cmd = radioSelection(formname.radioChoice);
	    var cmdsave = cmd;
	} else {
	    cmd = cmdx;
	    cmdsave = 'submission';
	}
	formname.command.value = cmd;
	formname.saveState.value = "saveCmd="+cmdsave+":saveSec="+pullDownSelection(formname.section)+
	    ":saveSub="+pullDownSelection(formname.submitonly)+":saveStatus="+pullDownSelection(formname.Status);
	if (val < 5) formname.submit();
	if (val == 5) {
	    if (!checkReceiptNo(formname,'notOK')) { return false;}
	    formname.submit();
	}
	if (val < 7) formname.submit();
    }

    function checkReceiptNo(formname,nospace) {
	var receiptNo = formname.receipt.value;
	var checkOpt = false;
	if (nospace == "OK" && isNaN(receiptNo)) {checkOpt = true;}
	if (nospace == "notOK" && (isNaN(receiptNo) || receiptNo == "")) {checkOpt = true;}
	if (checkOpt) {
	    alert("Please enter a receipt number given by a student in the receipt box.");
	    formname.receipt.value = "";
	    formname.receipt.focus();
	    return false;
	}
	return true;
    }
</script>
GRADINGMENUJS
    &commonJSfunctions($request);
    my $result='<h3>&nbsp;<span class="LC_info">Manual Grading/View Submission</span></h3>';
    my ($table,undef,$hdgrade) = &showResourceInfo($symb,$probTitle);
    $result.=$table;
    my (undef,$sections) = &getclasslist('all','0');
    my $savedState = &savedState();
    my $saveCmd = ($$savedState{'saveCmd'} eq '' ? 'submission' : $$savedState{'saveCmd'});
    my $saveSec = ($$savedState{'saveSec'} eq '' ? 'all' : $$savedState{'saveSec'});
    my $saveSub = ($$savedState{'saveSub'} eq '' ? 'all' : $$savedState{'saveSub'});
    my $saveStatus = ($$savedState{'saveStatus'} eq '' ? 'Active' : $$savedState{'saveStatus'});

    $result.='<form action="/adm/grades" method="post" name="gradingMenu">'."\n".
	'<input type="hidden" name="symb"        value="'.$symb.'" />'."\n".
	'<input type="hidden" name="handgrade"   value="'.$hdgrade.'" />'."\n".
	'<input type="hidden" name="probTitle"   value="'.$probTitle.'" />'."\n".
	'<input type="hidden" name="command"     value="" />'."\n".
	'<input type="hidden" name="saveState"   value="" />'."\n".
	'<input type="hidden" name="gradingMenu" value="1" />'."\n".
	'<input type="hidden" name="showgrading" value="yes" />'."\n";

    $result.='<table width="100%" border="0"><tr><td bgcolor=#777777>'."\n".
	'<table width="100%" border="0"><tr bgcolor="#e6ffff"><td colspan="2">'."\n".
	'&nbsp;<b>Select a Grading/Viewing Option</b></td></tr>'."\n".
	'<tr bgcolor="#ffffe6" valign="top"><td>'."\n";

    $result.='<table width="100%" border="0">';
    $result.='<tr bgcolor="#ffffe6" valign="top"><td>'."\n".
	'&nbsp;'.&mt('Select Section').': <select name="section">'."\n";
    if (ref($sections)) {
	foreach (sort (@$sections)) {
	    $result.='<option value="'.$_.'" '.
		($saveSec eq $_ ? 'selected="on"':'').'>'.$_.'</option>'."\n";
	}
    }
    $result.= '<option value="all" '.($saveSec eq 'all' ? 'selected="on"' : ''). '>all</option></select> &nbsp; ';

    $result.=&mt('Student Status').':</b>'.&Apache::lonhtmlcommon::StatusOptions($saveStatus,undef,1,undef);

    $result.='</td></tr>';

    $result.='<tr bgcolor="#ffffe6"valign="top"><td><label>'.
	'<input type="radio" name="radioChoice" value="submission" '.
	($saveCmd eq 'submission' ? 'checked' : '').' /> '.'<b>'.&mt('Current Resource').':</b> '.&mt('For one or more students').
	'</label> <select name="submitonly">'.
	'<option value="yes" '.
	($saveSub eq 'yes' ? 'selected="on"' : '').' />'.&mt('with submissions').'</option>'.
	'<option value="queued" '.
	($saveSub eq 'queued' ? 'selected="on"' : '').' />'.&mt('in grading queue').'</option>'.
	'<option value="graded" '.
	($saveSub eq 'graded' ? 'selected="on"' : '').' />'.&mt('with ungraded submissions').'</option>'.
	'<option value="incorrect" '.
	($saveSub eq 'incorrect' ? 'selected="on"' : '').' />'.&mt('with incorrect submissions').'</option>'.
	'<option value="all" '.
	($saveSub eq 'all' ? 'selected="on"' : '').' />'.&mt('with any status').'</option></select></td></tr>'."\n";

    $result.='<tr bgcolor="#ffffe6"valign="top"><td>'.
	'<label><input type="radio" name="radioChoice" value="viewgrades" '.
	($saveCmd eq 'viewgrades' ? 'checked' : '').' /> '.
	'<b>Current Resource:</b> For all students in selected section or course</label></td></tr>'."\n";

    $result.='<tr bgcolor="#ffffe6" valign="top"><td>'.
	'<label><input type="radio" name="radioChoice" value="pickStudentPage" '.
	($saveCmd eq 'pickStudentPage' ? 'checked' : '').' /> '.
	'The <b>complete</b> set/page/sequence: For one student</label></td></tr>'."\n";

    $result.='<tr bgcolor="#ffffe6"><td><br />'.
	'<input type="button" onClick="javascript:checkChoice(this.form,\'2\');" value="Next->" />'.
	'</td></tr></table>'."\n";

    $result.='</td><td valign="top">';

    $result.='<table width="100%" border="0">';
    $result.='<tr bgcolor="#ffffe6"><td>'.
	'<input type="button" onClick="javascript:checkChoice(this.form,\'3\',\'csvform\');" value="'.&mt('Upload').'" />'.
	' '.&mt('scores from file').' </td></tr>'."\n";

    $result.='<tr bgcolor="#ffffe6"valign="top"><td colspan="2">'.
	'<input type="button" onClick="javascript:checkChoice(this.form,\'4\',\'scantron_selectphase\');'.
	'" value="'.&mt('Grade').'" /> scantron forms</td></tr>'."\n";

    if ((&Apache::lonnet::allowed('mgr',$env{'request.course.id'})) && ($symb)) {
	$result.='<tr bgcolor="#ffffe6"valign="top"><td>'.
	    '<input type="button" onClick="javascript:checkChoice(this.form,\'5\',\'verify\');" value="'.&mt('Verify').'" />'.
	    ' '.&mt('receipt').': '.
	    &Apache::lonnet::recprefix($env{'request.course.id'}).
	    '-<input type="text" name="receipt" size="4" onChange="javascript:checkReceiptNo(this.form,\'OK\')" />'.
	    '</td></tr>'."\n";
    } 
    $result.='<tr bgcolor="#ffffe6"valign="top"><td colspan="2">'.
	'<input type="button" onClick="javascript:this.form.action=\'/adm/helper/resettimes.helper\';this.form.submit();'.
	'" value="'.&mt('Manage').'" /> access times.</td></tr>'."\n";
    $result.='<tr bgcolor="#ffffe6"valign="top"><td colspan="2">'.
	'<input type="button" onClick="javascript:this.form.command.value=\'codelist\';this.form.action=\'/adm/pickcode\';this.form.submit();'.
	'" value="'.&mt('View').'" /> saved CODEs.</td></tr>'."\n";

    $result.='</form></td></tr></table>'."\n".
	'</td></tr></table>'."\n".
	'</td></tr></table>'."\n";
    return $result;
}

sub reset_perm {
    undef(%perm);
}

sub init_perm {
    &reset_perm();
    foreach my $test_perm ('vgr','mgr','opa') {

	my $scope = $env{'request.course.id'};
	if (!($perm{$test_perm}=&Apache::lonnet::allowed($test_perm,$scope))) {

	    $scope .= '/'.$env{'request.course.sec'};
	    if ( $perm{$test_perm}=
		 &Apache::lonnet::allowed($test_perm,$scope)) {
		$perm{$test_perm.'_section'}=$env{'request.course.sec'};
	    } else {
		delete($perm{$test_perm});
	    }
	}
    }
}

sub handler {
    my $request=$_[0];

    &reset_perm();
    if ($env{'browser.mathml'}) {
	&Apache::loncommon::content_type($request,'text/xml');
    } else {
	&Apache::loncommon::content_type($request,'text/html');
    }
    $request->send_http_header;
    return '' if $request->header_only;
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'});
    my $symb=&get_symb($request,1);
    my @commands=&Apache::loncommon::get_env_multiple('form.command');
    my $command=$commands[0];
    if ($#commands > 0) {
	&Apache::lonnet::logthis("grades got multiple commands ".join(':',@commands));
    }
    $request->print(&Apache::loncommon::start_page('Grading'));
    if ($symb eq '' && $command eq '') {
	if ($env{'user.adv'}) {
	    if (($env{'form.codeone'}) && ($env{'form.codetwo'}) &&
		($env{'form.codethree'})) {
		my $token=$env{'form.codeone'}.'*'.$env{'form.codetwo'}.'*'.
		    $env{'form.codethree'};
		my ($tsymb,$tuname,$tudom,$tcrsid)=
		    &Apache::lonnet::checkin($token);
		if ($tsymb) {
		    my ($map,$id,$url)=&Apache::lonnet::decode_symb($tsymb);
		    if (&Apache::lonnet::allowed('mgr',$tcrsid)) {
			$request->print(&Apache::lonnet::ssi_body('/res/'.$url,
					  ('grade_username' => $tuname,
					   'grade_domain' => $tudom,
					   'grade_courseid' => $tcrsid,
					   'grade_symb' => $tsymb)));
		    } else {
			$request->print('<h3>Not authorized: '.$token.'</h3>');
		    }
		} else {
		    $request->print('<h3>Not a valid DocID: '.$token.'</h3>');
		}
	    } else {
		$request->print(&Apache::lonxml::tokeninputfield());
	    }
	}
    } else {
	&init_perm();
	if ($command eq 'submission' && $perm{'vgr'}) {
	    ($env{'form.student'} eq '' ? &listStudents($request) : &submission($request,0,0));
	} elsif ($command eq 'pickStudentPage' && $perm{'vgr'}) {
	    &pickStudentPage($request);
	} elsif ($command eq 'displayPage' && $perm{'vgr'}) {
	    &displayPage($request);
	} elsif ($command eq 'gradeByPage' && $perm{'mgr'}) {
	    &updateGradeByPage($request);
	} elsif ($command eq 'processGroup' && $perm{'vgr'}) {
	    &processGroup($request);
	} elsif ($command eq 'gradingmenu' && $perm{'vgr'}) {
	    $request->print(&gradingmenu($request));
	} elsif ($command eq 'viewgrades' && $perm{'vgr'}) {
	    $request->print(&viewgrades($request));
	} elsif ($command eq 'handgrade' && $perm{'mgr'}) {
	    $request->print(&processHandGrade($request));
	} elsif ($command eq 'editgrades' && $perm{'mgr'}) {
	    $request->print(&editgrades($request));
	} elsif ($command eq 'verify' && $perm{'vgr'}) {
	    $request->print(&verifyreceipt($request));
	} elsif ($command eq 'csvform' && $perm{'mgr'}) {
	    $request->print(&upcsvScores_form($request));
	} elsif ($command eq 'csvupload' && $perm{'mgr'}) {
	    $request->print(&csvupload($request));
	} elsif ($command eq 'csvuploadmap' && $perm{'mgr'} ) {
	    $request->print(&csvuploadmap($request));
	} elsif ($command eq 'csvuploadoptions' && $perm{'mgr'}) {
	    if ($env{'form.associate'} ne 'Reverse Association') {
		$request->print(&csvuploadoptions($request));
	    } else {
		if ( $env{'form.upfile_associate'} ne 'reverse' ) {
		    $env{'form.upfile_associate'} = 'reverse';
		} else {
		    $env{'form.upfile_associate'} = 'forward';
		}
		$request->print(&csvuploadmap($request));
	    }
	} elsif ($command eq 'csvuploadassign' && $perm{'mgr'} ) {
	    $request->print(&csvuploadassign($request));
	} elsif ($command eq 'scantron_selectphase' && $perm{'mgr'}) {
	    $request->print(&scantron_selectphase($request));
 	} elsif ($command eq 'scantron_warning' && $perm{'mgr'}) {
 	    $request->print(&scantron_do_warning($request));
	} elsif ($command eq 'scantron_validate' && $perm{'mgr'}) {
	    $request->print(&scantron_validate_file($request));
	} elsif ($command eq 'scantron_process' && $perm{'mgr'}) {
	    $request->print(&scantron_process_students($request));
 	} elsif ($command eq 'scantronupload' && 
 		 (&Apache::lonnet::allowed('usc',$env{'request.role.domain'})||
		  &Apache::lonnet::allowed('usc',$env{'request.course.id'}))) {
 	    $request->print(&scantron_upload_scantron_data($request)); 
 	} elsif ($command eq 'scantronupload_save' &&
 		 (&Apache::lonnet::allowed('usc',$env{'request.role.domain'})||
		  &Apache::lonnet::allowed('usc',$env{'request.course.id'}))) {
 	    $request->print(&scantron_upload_scantron_data_save($request));
 	} elsif ($command eq 'scantron_download' &&
		 &Apache::lonnet::allowed('usc',$env{'request.course.id'})) {
 	    $request->print(&scantron_download_scantron_data($request));
	} elsif ($command) {
	    $request->print("Access Denied ($command)");
	}
    }
    $request->print(&Apache::loncommon::end_page());
    return '';
}

1;

__END__;
