# The LearningOnline Network with CAPA 
# definition of tags that give a structure to a document
#
# $Id: structuretags.pm,v 1.382 2007/06/22 20:42:20 albertel Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###


package Apache::structuretags; 

use strict;
use Apache::lonnet;
use Apache::File();
use Apache::lonmenu;
use Apache::lonlocal;
use Apache::lonxml;
use Apache::lonenc();
use Time::HiRes qw( gettimeofday tv_interval );
use lib '/home/httpd/lib/perl/';
use LONCAPA;
 
BEGIN {
    &Apache::lonxml::register('Apache::structuretags',('block','languageblock','instructorcomment','while','randomlist','problem','library','web','tex','part','preduedate','postanswerdate','solved','notsolved','problemtype','startouttext','endouttext','simpleeditbutton','definetag'));
}

sub start_web {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $bodytext=&Apache::lonxml::get_all_text("/web",$parser,$style);
    if ($target eq 'web' || $target eq 'webgrade') {
	return $bodytext;
    }
    return '';
}

sub end_web {
    return '';
}

sub start_tex {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result='';
    my $bodytext=&Apache::lonxml::get_all_text("/tex",$parser,$style);
    if ($target eq 'tex') {
	return $bodytext.' ';
    }
    return $result;;
}

sub end_tex {
    return '';
}

sub page_start {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$name,
	$extra_head)=@_;
    my %found;
    foreach my $taginside (@$tagstack) {
	foreach my $taglookedfor ('html','body','form') {
	    if ($taginside =~ /^$taglookedfor$/i) {$found{$taglookedfor} = 1;}
	}
    }

    if ($target eq 'tex') {
	return
	    &Apache::londefdef::start_html($target,$token,$tagstack,
					   $parstack,$parser,$safeeval);
    }

    $extra_head.= '
<script type="text/javascript">
function setSubmittedPart (part) {
   this.document.lonhomework.submitted.value="part_"+part;
}
</script>
';

    my %body_args;
    if (defined($found{'html'})) {
	$body_args{'skip_phases'}{'head'}=1;
    } else {
	
	$extra_head .= &Apache::lonhtmlcommon::spellheader();

	$extra_head .= &Apache::londefdef::generate_css_links();

	if ($target eq 'edit') {
	    $extra_head.=&Apache::edit::js_change_detection().
		"<script type=\"text/javascript\">\n".
		"if (typeof swmenu != 'undefined') {swmenu.currentURL=null;}\n".
		&Apache::loncommon::browser_and_searcher_javascript().
                "\n</script>\n";
	}
    }

    if (defined($found{'body'})) {
	$body_args{'skip_phases'}{'body'}=1;
    } elsif (!defined($found{'body'}) 
	     && $env{'request.state'} eq 'construct') {
	if ($target eq 'web' || $target eq 'edit') {
	    # no extra args to bodytag
	}
    } elsif (!defined($found{'body'})) {
	my %add_entries;
	my $background=&Apache::lonxml::get_param('background',$parstack,
						  $safeeval);
	if ($background ne '' ) {
	    $add_entries{'background'} = $background;
	}

	my $bgcolor=&Apache::lonxml::get_param('bgcolor',$parstack,
					       $safeeval);
       	if ($bgcolor eq '' ) { $bgcolor = '#FFFFFF'; }

	$body_args{'bgcolor'}        = $bgcolor;
	$body_args{'no_title'}       = 1;
	$body_args{'force_register'} = 1;
	$body_args{'add_entries'}    = \%add_entries;	
    }
    $body_args{'no_auto_mt_title'} = 1;
    my $page_start = &Apache::loncommon::start_page($name,$extra_head,
						    \%body_args);

    if (!defined($found{'body'}) 
	&& $env{'request.state'} ne 'construct'
	&& ($target eq 'web' || $target eq 'webgrade')) {

	my ($symb,undef,undef,undef,$publicuser)= &Apache::lonnet::whichuser();
	if ($symb eq '' && !$publicuser) {
	    my $help = &Apache::loncommon::help_open_topic("Ambiguous_Reference");
	    $help=&mt("Browsing resource, all submissions are temporary.")."<br />";
	    $page_start .= $help;
	}
    }

    if (!defined($found{'body'})) {
	$page_start .= &Apache::lonxml::message_location();
    }
    
    my $form_tag_start;
    if (!defined($found{'form'})) {
	$form_tag_start='<form name="lonhomework" enctype="multipart/form-data" method="post" action="';
	my $uri = &Apache::lonenc::check_encrypt($env{'request.uri'});
	$form_tag_start.=$uri.'" ';
	if ($target eq 'edit') {
	    $form_tag_start.=&Apache::edit::form_change_detection();
	}
	$form_tag_start.='>'."\n";

	my $symb=&Apache::lonnet::symbread();
	if ($symb =~ /\S/) {
	    $symb=
		&HTML::Entities::encode(&Apache::lonenc::check_encrypt($symb));
	    $form_tag_start.=
		"\t".'<input type="hidden" name="symb" value="'.$symb.'" />'."\n";
	}
    }
    return ($page_start,$form_tag_start);
}

#use Time::HiRes();
sub get_resource_name {
    my ($parstack,$safeeval)=@_;
    if (defined($Apache::lonhomework::name)) {
	return $Apache::lonhomework::name;
    }
    my ($symb)=&Apache::lonnet::whichuser();
    my $name=&Apache::lonnet::gettitle($symb);
    if ($name eq '') {
	$name=&Apache::lonnet::EXT('resource.title');
	if ($name eq 'con_lost') { $name = ''; }
    }
    if ($name!~/\S+/) {
	$name=$env{'request.uri'};
	$name=~s-.*/([^/]+)$-$1-;
    }
    $Apache::lonhomework::name=$name;
    return $name;
}

sub setup_rndseed {
    my ($safeeval)=@_;
    my $rndseed;
    my ($symb)=&Apache::lonnet::whichuser();
    if ($env{'request.state'} eq "construct" 
	|| $symb eq '' 
	|| $Apache::lonhomework::type eq 'practice'
	|| $Apache::lonhomework::history{'resource.CODE'}) {
	&Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
						['rndseed']);
	$rndseed=$env{'form.rndseed'};
	if (!$rndseed) {
	    $rndseed=$Apache::lonhomework::history{'rndseed'};
	    if (!$rndseed) {
		$rndseed=time;
	    }
	    $env{'form.rndseed'}=$rndseed;
	}
	if ( ($env{'form.resetdata'} eq &mt('New Problem Variation')
	      && $env{'form.submitted'} eq 'yes')  ||
	    $env{'form.newrandomization'} eq &mt('New Randomization')) {
	    srand(time);
	    $rndseed=int(rand(2100000000));
	    $env{'form.rndseed'}=$rndseed;
	    delete($env{'form.resetdata'});
	    delete($env{'form.newrandomization'});
	}
	if (defined($rndseed) && $rndseed ne int($rndseed)) {
	    $rndseed=join(':',&Apache::lonnet::digest($rndseed));
        }
        if ($Apache::lonhomework::history{'resource.CODE'}) {
	   $rndseed=&Apache::lonnet::rndseed();
	}
	if ($safeeval) {
	    &Apache::lonxml::debug("Setting rndseed to $rndseed");
	    &Apache::run::run('$external::randomseed="'.$rndseed.'";',$safeeval);
	}
    }
    return $rndseed;
}

sub remember_problem_state {
    return '
       <input type="hidden" name="problemstate" value="'.$env{'form.problemstate'}.'" />
       <input type="hidden" name="problemtype" value="'.$env{'form.problemtype'}.'" />
       <input type="hidden" name="problemstatus" value="'.$env{'form.problemstatus'}.'" />';
}

sub problem_edit_header {
    return '<input type="hidden" name="submitted" value="edit" />'.
	&Apache::structuretags::remember_problem_state().'
       <input type="hidden" name="problemmode" value="'.&mt('Edit').'" />
       <input type="submit" name="problemmode" accesskey="d" value="'.&mt('Discard Edits and View').'" />
       <input '.&Apache::edit::submit_ask_anyway().' type="submit" name="problemmode" accesskey="x" value="'.&mt('EditXML').'" />
       <input type="submit" name="Undo" accesskey="u" value="'.&mt('undo').'" /> <hr />
       <input type="submit" name="submit" accesskey="s" value="'.&mt('Submit Changes and Edit').'" />
       <input type="submit" name="submit" accesskey="v" value="'.&mt('Submit Changes and View').'" /><table><tr><td>'.
       &Apache::loncommon::help_open_menu('Problem Editing Help','Problem_Editor_XML_Index',5,'Authoring',undef,undef,undef,'Problem Editing Help')
       .'</td></tr></table>'.
       '<table border="0" width="100%"><tr><td bgcolor="#DDDDDD">';
}

sub problem_edit_footer {
    return '</td></tr></table><br /><input type="submit" name="submit" value="'.&mt('Submit Changes and Edit').'" />
    <input type="submit" name="submit" value="'.&mt('Submit Changes and View').'" />'.
    &Apache::lonhtmlcommon::htmlareaselectactive(&Apache::lonhtmlcommon::get_htmlareafields()).
    "\n</form>\n".&Apache::loncommon::end_page();
}

sub option {
    my ($value,$name) = @_;
    my $result ="<option value='".$value."' ";
    if ($env{'form.'.$name} eq $value) {
	$result.=" selected='on' ";
    }
    $result.='>';
    return $result;
}

sub problem_web_to_edit_header {
    my ($rndseed)=@_;
    my $result.='<input type="hidden" name="problemmode" value="'.&mt('View').'" />';
    $result .= '<input type="submit" name="problemmode" accesskey="e" value="'.&mt('Edit').'" />';
    $result .= '<input type="submit" name="problemmode" accesskey="x" value="'.&mt('EditXML').'" />
             <input type="submit" name="newrandomization" accesskey="a" value="'.&mt('New Randomization').'" />
             <input type="submit" name="resetdata" accesskey="r" value="'.&mt('Reset Submissions').'" />
             <nobr><input type="submit" name="changerandseed" value="'.&mt('Change Random Seed To:').'" />
              <input type="text" name="rndseed" size="10" value="'.
	       $rndseed.'"
           onchange="javascript:document.lonhomework.changerandseed.click()" /></nobr>
             <label><input type="checkbox" name="showallfoils" ';
    my $show_all_foils_text = 
	($Apache::lonhomework::parsing_a_task) ?
	&mt('&nbsp;Show&nbsp;All&nbsp;Instances')
	: &mt('&nbsp;Show&nbsp;All&nbsp;Foils');

    if (defined($env{'form.showallfoils'})) { $result.='checked="on"'; }
    $result.= ' />'.$show_all_foils_text.'</label>'.
	&Apache::loncommon::help_open_topic('Problem_Editor_Testing_Area','Testing Problems').
	'<hr />';
    if (!$Apache::lonhomework::parsing_a_task) {
	$result.="
<nobr>
Problem Status:
<select name='problemstate'>
  <option value=''></option>
  ".&option('CLOSED'               ,'problemstate').&mt("Closed")."</option>
  ".&option('CAN_ANSWER'           ,'problemstate').&mt("Answerable")."</option>
  ".&option('CANNOT_ANSWER_tries'  ,'problemstate').&mt("Open with full tries")."</option>
  ".&option('CANNOT_ANSWER_correct','problemstate').&mt("Open and correct")."</option>
  ".&option('SHOW_ANSWER'          ,'problemstate').&mt("Show Answer")."</option>
</select>
</nobr>
<nobr>
Problem Type:
<select name='problemtype'>
  <option value=''></option>
  ".&option('exam'   ,'problemtype').&mt("Exam Problem")."</option>
  ".&option('problem','problemtype').&mt("Homework problem")."</option>
  ".&option('survey' ,'problemtype').&mt("Survey Question")."</option>
</select>
</nobr>
<nobr>
Feedback Mode:
<select name='problemstatus'>
  <option value=''></option>
  ".&option('yes','problemstatus').&mt("Show Feedback")."</option>
  ".&option('no', 'problemstatus').&mt("Don't Show Feedback")."</option>
</select>
</nobr>
<input type='submit' name='changeproblemmode' value='".&mt("Change")."' />
<hr />";
	my $numtoanalyze=$env{'form.numtoanalyze'};
	if (!$numtoanalyze) { $numtoanalyze=20; }
	$result.= '<input type="submit" name="problemmode" value='.
	    &mt('"Calculate answers').'" /> for
             <input type="text" name="numtoanalyze" value="'.
	     $numtoanalyze.'" size="5" /> '.&mt('versions of this problem').
	     '.'.&Apache::loncommon::help_open_topic("Analyze_Problem",
						     '',undef,undef,300).
						     '<hr />';
    } elsif ($Apache::lonhomework::parsing_a_task) {
	$result.="
<nobr>
Problem Status:
<select name='problemstate'>
  <option value=''></option>
  ".&option('CLOSED'               ,'problemstate').&mt("Closed")."</option>
  ".&option('CAN_ANSWER'           ,'problemstate').&mt("Answerable")."</option>
  ".&option('WEB_GRADE'            ,'problemstate').&mt("Criteria Grading")."</option>
  ".&option('SHOW_ANSWER'          ,'problemstate').&mt("Show Feedback")."</option>
</select>
</nobr>
<input type='submit' name='changeproblemmode' value='".&mt("Change")."' />
<hr />";
    }
    return $result;
}

sub initialize_storage {
    my ($given_symb) = @_;
    undef(%Apache::lonhomework::results);
    undef(%Apache::lonhomework::history);
    my ($symb,$courseid,$domain,$name) = 
	&Apache::lonnet::whichuser($given_symb);
    
    # anonymous users (CODEd exams) have no data
    if ($name eq 'anonymous' 
	&& !defined($domain)) {
	return;
    }

    if ($env{'request.state'} eq 'construct' 
	|| $symb eq ''
	|| $Apache::lonhomework::type eq 'practice') {
	
	my $namespace = $symb || $env{'request.uri'};
	if ($env{'form.resetdata'} eq &mt('Reset Submissions') ||
	    ($env{'form.resetdata'} eq &mt('New Problem Variation')
	     && $env{'form.submitted'} eq 'yes') ||
	    $env{'form.newrandomization'} eq &mt('New Randomization')) {
	    &Apache::lonnet::tmpreset($namespace,'',$domain,$name);
	    &Apache::lonxml::debug("Attempt reset");
	}
	%Apache::lonhomework::history=
	    &Apache::lonnet::tmprestore($namespace,'',$domain,$name);
	my ($temp)=keys %Apache::lonhomework::history ;
	&Apache::lonxml::debug("Return message of $temp");
    } else {
	%Apache::lonhomework::history=
	    &Apache::lonnet::restore($symb,$courseid,$domain,$name);
    }

    #ignore error conditions
    my ($temp)=keys %Apache::lonhomework::history ;
    if ($temp =~ m/^error:.*/) { %Apache::lonhomework::history=(); }
}

# -------------------------------------------------------------finalize_storage
# Stores away the result has to a student's environment
# checks form.grade_ for specific values, other wises stores
# to the running users environment
# Will increment totals for attempts, students, and corrects
# if running user has student role.  
sub finalize_storage {
    my ($given_symb) = @_;
    my $result;
    if (%Apache::lonhomework::results) {
	my @remove = grep(/^INTERNAL_/,keys(%Apache::lonhomework::results));
	delete(@Apache::lonhomework::results{@remove});
	my ($symb,$courseid,$domain,$name) = 
	    &Apache::lonnet::whichuser($given_symb);
	if ($env{'request.state'} eq 'construct' 
	    || $symb eq ''
	    || $Apache::lonhomework::type eq 'practice') {
	    my $namespace = $symb || $env{'request.uri'};
	    $Apache::lonhomework::results{'rndseed'}=$env{'form.rndseed'};
	    $result=&Apache::lonnet::tmpstore(\%Apache::lonhomework::results,
					      $namespace,'',$domain,$name);
	    &Apache::lonxml::debug('Construct Store return message:'.$result);
	} else {
	    $result=&Apache::lonnet::cstore(\%Apache::lonhomework::results,
					    $symb,$courseid,$domain,$name);
	    &Apache::lonxml::debug('Store return message:'.$result);
            if ($env{'request.role'} =~/^st/) {
                &store_aggregates($symb,$courseid);
            }
	}
    } else {
	&Apache::lonxml::debug('Nothing to store');
    }
    return $result;
}
use Data::Dumper;

# -------------------------------------------------------------store_aggregates
# Sends hash of values to be incremented in nohist_resourcetracker.db
# for the course. Increments total number of attempts, unique students 
# and corrects for each part for an instance of a problem, as appropriate.
sub store_aggregates {
    my ($symb,$courseid) = @_;
    my %aggregate;
    my @parts;
    my $cdomain = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cname = $env{'course.'.$env{'request.course.id'}.'.num'};
    foreach my $key (keys(%Apache::lonhomework::results)) {
        if ($key =~ /resource\.([^\.]+)\.tries/) {
            push(@parts, $1);
        }
    }
    foreach my $part (@parts) {
        if ($Apache::lonhomework::results{'resource.'.$part.'.award'}
	    eq 'APPROX_ANS' ||
	    $Apache::lonhomework::results{'resource.'.$part.'.award'}
	    eq 'EXACT_ANS') {
            $aggregate{$symb."\0".$part."\0correct"} = 1;
        }
        if ($Apache::lonhomework::results{'resource.'.$part.'.tries'} == 1) {
            $aggregate{$symb."\0".$part."\0users"} = 1;
        } else {
            my (undef,$last_reset) = &Apache::grades::get_last_resets($symb,$env{'request.course.id'},[$part]); 
            if ($last_reset) {
                if (&Apache::grades::get_num_tries(\%Apache::lonhomework::history,$last_reset,$part) == 0) {
                    $aggregate{$symb."\0".$part."\0users"} = 1;
                }
            }
        }
        $aggregate{$symb."\0".$part."\0attempts"} = 1;
    }
    if (keys (%aggregate) > 0) {
	&Apache::lonnet::cinc('nohist_resourcetracker',\%aggregate,
                            $cdomain,$cname);
    }
}

sub checkout_msg {
    my %lt=&Apache::lonlocal::texthash( 
		'resource'=>'The resource needs to be checked out',
		'id_expln'=>'As a resource gets checked out, a unique timestamped ID is given to it, and a permanent record is left in the system.',
                'warning'=>'Checking out resources is subject to course policies, and may exclude future credit even if done erroneously.',
                'checkout'=>'Check out Exam for Viewing',
		'checkout?'=>'Check out Exam?');
    my $uri = &Apache::lonenc::check_encrypt($env{'request.uri'});
    return (<<ENDCHECKOUT);
<h2>$lt{'resource'}</h2>
    <p>$lt{'id_expln'}</p>
<font color="red">
<p>$lt{'warning'}</p>
</font>
<form name="checkout" method="POST" action="$uri">
<input type="hidden" name="doescheckout" value="yes" />
<input type="button" name="checkoutbutton" value="$lt{'checkout'}" onClick="javascript:if (confirm('$lt{'checkout?'}')) { document.checkout.submit(); }" />
</form>
ENDCHECKOUT
}

sub firstaccess_msg {
    my ($time,$symb)=@_;
    my ($map)=&Apache::lonnet::decode_symb($symb);
    my $foldertitle=&Apache::lonnet::gettitle($map);
    &Apache::lonxml::debug("map is $map title is $foldertitle");
    my $uri = &Apache::lonenc::check_encrypt($env{'request.uri'});
    return (<<ENDCHECKOUT);
<h2>The resources in "$foldertitle" are open for a limited time. Once you click the 'Show Resource' button below you have $time to complete all resources "$foldertitle".</h2>
<form name="markaccess" method="POST" action="$uri">
<input type="hidden" name="markaccess" value="yes" />
<input type="button" name="accessbutton" value="Show Resource" onClick="javascript:if (confirm('Start Timer?')) { document.markaccess.submit(); }" />
</form>
ENDCHECKOUT
}

sub init_problem_globals {
    my ($type)=@_;
    #initialize globals
    #   For problems, we start out in part 0 (outside a <part> tag).
    #   and part 0 is used to describe the main body of the <problem>
    #
    if ($type eq 'problem') {
	$Apache::inputtags::part='0';
	@Apache::inputtags::partlist=('0');
	$Apache::lonhomework::problemstatus=&get_problem_status('0');
	$Apache::lonhomework::ignore_response_errors=0;

    } elsif ($type eq 'library') {
	$Apache::inputtags::part='';
	@Apache::inputtags::partlist=();
	$Apache::lonhomework::problemstatus='';	
	$Apache::lonhomework::ignore_response_errors=1;

    } elsif ($type eq 'Task') {
	$Apache::inputtags::part='0';
	@Apache::inputtags::partlist=('0');
	$Apache::lonhomework::problemstatus='';	
	$Apache::lonhomework::ignore_response_errors=1;
    }
    @Apache::inputtags::responselist = ();
    @Apache::inputtags::importlist = ();
    @Apache::inputtags::previous=();
    @Apache::inputtags::previous_version=();
    $Apache::structuretags::printanswer='No';
    @Apache::structuretags::whileconds=();
    @Apache::structuretags::whilebody=();
    @Apache::structuretags::whileline=();
    $Apache::lonhomework::scantronmode=0;
    undef($Apache::lonhomework::name);
    undef($Apache::lonhomework::default_type);
    undef($Apache::lonhomework::type);
}

sub reset_problem_globals {
    my ($type)=@_;
    undef(%Apache::lonhomework::history);
    undef(%Apache::lonhomework::results);
    undef($Apache::inputtags::part);
#don't undef this, lonhomework.pm takes care of this, we use this to 
#detect if we try to do 2 problems in one file
#   undef($Apache::lonhomework::parsing_a_problem);
    undef($Apache::lonhomework::name);
    undef($Apache::lonhomework::default_type);
    undef($Apache::lonhomework::type);
    undef($Apache::lonhomework::scantronmode);
    undef($Apache::lonhomework::problemstatus);
    undef($Apache::lonhomework::ignore_response_errors);
}

sub set_problem_state {
    my ($part)=@_;
    if ($env{'form.problemstate'} eq 'CANNOT_ANSWER_correct') {
	$Apache::lonhomework::history{"resource.$part.solved"}=
	    'correct_by_student';
    }
}

sub get_problem_status {
    my ($part)=@_;
    my $problem_status;
    if ($env{'request.state'} eq 'construct' &&
	defined($env{'form.problemstatus'})) {
	$problem_status=$env{'form.problemstatus'};
    } else {
	$problem_status=&Apache::lonnet::EXT("resource.$part.problemstatus");
	&Apache::lonxml::debug("problem status for $part is $problem_status");
	&Apache::lonxml::debug("env probstat is ".$env{'form.problemstatus'});
    }
    return $problem_status;
}

sub start_problem {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;

    # We'll use the redirection to fix up printing of duedates.
    if (!$Apache::lonxml::metamode) {
	&Apache::lonxml::startredirection();
    }

    # Problems don't nest and we don't allow more than one <problem> in
    # a .problem file.
    #
    if ( $Apache::inputtags::part ne '' ||
	 $Apache::lonhomework::parsing_a_problem) {
	&Apache::lonxml::error('Only one &lt;problem&gt; allowed in a .problem file');
	#my $bodytext=&Apache::lonxml::get_all_text("/problem",$parser,$style);
	return '';
    }

    $Apache::lonhomework::parsing_a_problem=1;
    &init_problem_globals('problem');

    if (defined($env{'scantron.maxquest'})) {
	$Apache::lonhomework::scantronmode=1;
    }

    if ($target ne 'analyze') {
       	if ($env{'request.state'} eq 'construct') { &set_problem_state('0'); }
	$Apache::lonhomework::type=&Apache::lonnet::EXT('resource.0.type');
	if (($env{'request.state'} eq 'construct') &&
	    defined($env{'form.problemtype'})) {
	    $Apache::lonhomework::type=$env{'form.problemtype'};
	}
	&Apache::lonxml::debug("Found this to be of type :$Apache::lonhomework::type:");
    }
    if ($Apache::lonhomework::type eq '' ) {
	my $uri=$env{'request.uri'};
	if ($uri=~/\.(\w+)$/) {
	    $Apache::lonhomework::type=$1;
	    &Apache::lonxml::debug("Using type of $1");
	} else {
	    $Apache::lonhomework::type='problem';
	    &Apache::lonxml::debug("Using default type, problem, :$uri:");
	}
    }
    $Apache::lonhomework::default_type = $Apache::lonhomework::type;

    &initialize_storage();
    if ($target eq 'web') {
	&Apache::lonxml::debug(" grading history ");
	&Apache::lonhomework::showhash(%Apache::lonhomework::history);
    }

    #added vars to the scripting enviroment
    my $expression='$external::part=\''.$Apache::inputtags::part.'\';';
    $expression.='$external::type=\''.$Apache::lonhomework::type.'\';';
    &Apache::run::run($expression,$safeeval);
    my $status;
    my $accessmsg;

    my $name= &get_resource_name($parstack,$safeeval);
    my ($result,$form_tag_start);
    if ($target eq 'web' || $target eq 'webgrade' || $target eq 'tex'
	|| $target eq 'edit') {
	($result,$form_tag_start) =
	    &page_start($target,$token,$tagstack,$parstack,$parser,$safeeval,
			$name);
    }

    if ($target eq 'tex' and $env{'request.symb'} =~ m/\.page_/) {$result='';}

    if ($target eq 'analyze') { my $rndseed=&setup_rndseed($safeeval); }
    if ($target eq 'web' || $target eq 'grade' || $target eq 'answer' ||
	$target eq 'tex') {
	#handle exam checkout
	if ($Apache::lonhomework::type eq 'exam') {
	    my $token=
		$Apache::lonhomework::history{"resource.0.outtoken"};
	    if (($env{'form.doescheckout'}) && (!$token)) {
		$token=&Apache::lonxml::maketoken();
		$Apache::lonhomework::history{"resource.0.outtoken"}=
		    $token;
	    }
	    $result.=&Apache::lonxml::printtokenheader($target,$token);
	}
	if ($env{'form.markaccess'}) {
	    &Apache::lonnet::set_first_access('map');
	}
	#handle rand seed in construction space
	my $rndseed=&setup_rndseed($safeeval);
	my ($symb)=&Apache::lonnet::whichuser();
	if ($env{'request.state'} ne "construct" && 
	    ($symb eq '' || $Apache::lonhomework::type eq 'practice')) {
	    $form_tag_start.='<input type="hidden" name="rndseed" value="'.
		$rndseed.'" />'.
		    '<input type="submit" name="resetdata"
                             value="'.&mt('New Problem Variation').'" />';
	    if (exists($env{'form.username'})) {
		$form_tag_start.=
		    '<input type="hidden" name="username"
                             value="'.$env{'form.username'}.'" />';
	    }
	    if ($env{'request.role.adv'}) {
		$form_tag_start.=
		    ' <label><input type="checkbox" name="showallfoils" ';
		if (defined($env{'form.showallfoils'})) {
		    $form_tag_start.='checked="on"';
		}
		$form_tag_start.= ' />'.&mt('&nbsp;Show&nbsp;All&nbsp;Foils').
		    '</label>';
	    }
	    $form_tag_start.='<hr />';
	}

	($status,$accessmsg,my $slot_name,my $slot) = 
	    &Apache::lonhomework::check_slot_access('0','problem');
	push (@Apache::inputtags::status,$status);

	my $expression='$external::datestatus="'.$status.'";';
	$expression.='$external::gradestatus="'.$Apache::lonhomework::history{"resource.0.solved"}.'";';
	&Apache::run::run($expression,$safeeval);
	&Apache::lonxml::debug("Got $status");

	if (( $status eq 'CLOSED' ) ||
	    ( $status eq 'UNCHECKEDOUT') ||
	    ( $status eq 'NOT_YET_VIEWED') ||
	    ( $status eq 'BANNED') ||
	    ( $status eq 'UNAVAILABLE') ||
	    ( $status eq 'NOT_IN_A_SLOT') ||
	    ( $status eq 'INVALID_ACCESS')) {
	    my $bodytext=&Apache::lonxml::get_all_text("/problem",$parser,
						       $style);
	    if ( $target eq "web" ) {
		my $msg;
		if ($status eq 'UNAVAILABLE') {
		    $msg.='<h1>'.&mt('Unable to determine if this resource is open due to network problems. Please try again later.').'</h1>';
		} elsif ($status ne 'NOT_YET_VIEWED') {
		    $msg.='<h1>'.&mt('Not open to be viewed').'</h1>';
		}
		if ($status eq 'CLOSED' || $status eq 'INVALID_ACCESS') {
		    $msg.='The problem '.$accessmsg;
		} elsif ($status eq 'UNCHECKEDOUT') {
		    $msg.=&checkout_msg();
		} elsif ($status eq 'NOT_YET_VIEWED') {
		    $msg.=&firstaccess_msg($accessmsg,$symb);
		} elsif ($status eq 'NOT_IN_A_SLOT') {
		    $msg.=&Apache::bridgetask::add_request_another_attempt_button("Sign up for time to work.");
		}
		$result.=$msg.'<br />';
	    } elsif ($target eq 'tex') {
		my $startminipage = ($env{'form.problem_split'}=~/yes/i)? ''
		                    : '\begin{minipage}{\textwidth}';
		$result.='\begin{document}\noindent \vskip 1 mm '.
		    $startminipage.'\vskip 0 mm';
		if ($status eq 'UNAVAILABLE') {
		    $result.=&mt('Unable to determine if this resource is open due to network problems. Please try again later.').'\vskip 0 mm ';
		} else {
		    $result.=&mt('Problem is not open to be viewed. It')." $accessmsg \\vskip 0 mm ";
		}
	    }
	} elsif ($status eq 'NEEDS_CHECKIN') {
	    my $bodytext=&Apache::lonxml::get_all_text("/problem",$parser,
						       $style);
	    if ($target eq 'web') {
		$result .= 
		    &Apache::bridgetask::proctor_validation_screen($slot);
	    } elsif ($target eq 'grade') {
		&Apache::bridgetask::proctor_check_auth($slot_name,$slot,
							'problem');
	    }
	} elsif ($target eq 'web') {
	    if ($status eq 'CAN_ANSWER' 
		&& $slot_name ne ''
		&& $Apache::lonhomework::history{'resource.0.checkedin'} eq '') {
		# unproctored slot access, self checkin
		&Apache::bridgetask::check_in('problem',undef,undef,
					      $slot_name);
	    }
	    $result.="\n $form_tag_start \t".	
	      '<input type="hidden" name="submitted" value="yes" />';
	    # create a page header and exit
	    if ($env{'request.state'} eq "construct") {
		$result.= &problem_web_to_edit_header($env{'form.rndseed'});
	    }
	    # if we are viewing someone else preserve that info
	    if (defined $env{'form.grade_symb'}) {
		foreach my $field ('symb','courseid','domain','username') {
		    $result .= '<input type="hidden" name="grade_'.$field.
			'" value="'.$env{"form.grade_$field"}.'" />'."\n";
		}
	    }
	} elsif ($target eq 'tex') {
	    $result .= 'INSERTTEXFRONTMATTERHERE';

	}
    } elsif ($target eq 'edit') {
	$result .= $form_tag_start.&problem_edit_header();
	$Apache::lonxml::warnings_error_header=
	    &mt("Editor Errors - these errors might not effect the running of the problem, but they will likely cause problems with further use of the Edit mode. Please use the EditXML mode to fix these errors.")."<br />";
	my $temp=&Apache::edit::insertlist($target,$token);
	$result.=$temp;
    } elsif ($target eq 'modified') {
	$result=$token->[4];
    } else {
	# page_start returned a starting result, delete it if we don't need it
	$result = '';
    }
    return $result;
}

sub end_problem {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;

    if (!$Apache::lonxml::metamode) {
	$result = &Apache::lonxml::endredirection(); #started in &start_problem
	$Apache::lonxml::post_evaluate=0;
    }

    if ($target eq 'tex') {
	# Figure out the front matter and replace the
	# INSERTTEXFRONTMATTERHERE in result with it.  note that we do
	# this in end_problem because whether or not we display due
	# dates depends on whether due dates have already been
	# displayed in the problem parts.

	my $frontmatter   = '';
	my $startminipage = '';
	if (not $env{'form.problem_split'}=~/yes/) {
	    $startminipage = '\begin{minipage}{\textwidth}';
	}
	my $id = $Apache::inputtags::part;
	my $weight = &Apache::lonnet::EXT("resource.$id.weight");
	my $packages=&Apache::lonnet::metadata($env{'request.uri'},'packages');
	my @packages = split /,/,$packages;
	my $allow_print_points = 0;
	foreach my $partial_key (@packages) {
	    if ($partial_key=~m/^part_0$/) {
		$allow_print_points=1;
	    }
	}
	my $maxtries = &Apache::lonnet::EXT("resource.$id.maxtries");
	if (defined($maxtries) && $maxtries < 0) { $allow_print_points=0; }
	if (lc($env{'course.'.$env{'request.course.id'}.
			'.disableexampointprint'}) eq 'yes') {
	    $allow_print_points=0;
	}
	my $name_of_resourse= &Apache::lonxml::latex_special_symbols(&get_resource_name($parstack,$safeeval),'header');
	my $begin_doc='\begin{document} \typeout{STAMPOFPASSEDRESOURCESTART Resource <h2>"'.$name_of_resourse.'"</h2> located in <br /><small><b>'.$env{'request.uri'}.'</b></small><br /> STAMPOFPASSEDRESOURCEEND} \noindent ';
	my $toc_line='\vskip 1 mm\noindent '.$startminipage.
	    '\addcontentsline{toc}{subsection}{'.$name_of_resourse.'}';
	
	#  Figure out what the due date is and if we need to print
	#  it in the problem header.  We have been logging the
	#  last due date written to file. 
	
	my $duetime = &Apache::lonnet::EXT("resource.$id.duedate"); 
	my $duedate = POSIX::strftime("%c",localtime($duetime));
	my $temp_file;
	my $filename = "/home/httpd/prtspool/$env{'user.name'}_$env{'user.domain'}_printout.due";
	
	# Figure out what the last printed due date is or set it
	# to the epoch if no duedates have been printed.
	
	my $due_file_content = 0;      #   If the file does not yet exist, time is the epoch.
	if (-e $filename) {
	    $temp_file = Apache::File->new($filename);
	    my @due_file      = <$temp_file>;
	    $due_file_content = $due_file[$#due_file];
	    chomp $due_file_content;
	} 
	
	# We display the due date iff it is not the same as the last
	# duedate in problem header ($due_file_content), and
	# none of our parts displayed a duedate.
	#
	my $parts_with_displayduedate;
	if (defined $Apache::outputtags::showonce{'displayduedate'}) {
	    $parts_with_displayduedate = 
		scalar(@{$Apache::outputtags::showonce{'displayduedate'}});
	} else {
	    $parts_with_displayduedate = 0;
	}
	if (($due_file_content != $duetime) && ($parts_with_displayduedate == 0) ) {
	    $temp_file = Apache::File->new('>'.$filename);
	    print $temp_file "$duetime\n";
	    if (not $env{'request.symb'} =~ m/\.page_/) {
		if(not $duedate=~m/1969/ and $Apache::lonhomework::type ne 'exam') {
		    $frontmatter .= $begin_doc.
			'\textit{Due date: '.$duedate.'} '.$toc_line;
		} else {
		    $frontmatter.= $begin_doc.$toc_line;
		    if ($Apache::lonhomework::type eq 'exam' and $allow_print_points==1) { $frontmatter .= '\fbox{\textit{'.$weight.' pt}}';}
		}
	    } else {
		$frontmatter .= '\vskip 1mm\textit{Due date: '.$duedate.'} \\\\\\\\'.$startminipage;
	    }
	} else {
	    if (not $env{'request.symb'} =~ m/\.page_/) {
		$frontmatter .= $begin_doc.$toc_line;
		if (($Apache::lonhomework::type eq 'exam') and ($allow_print_points==1)) { $frontmatter .= '\fbox{\textit{'.$weight.' pt}}';}
	    } else {
		$frontmatter .= '\vskip 1mm \\\\\\\\'.$startminipage;
	    }
	}
	$result =~ s/INSERTTEXFRONTMATTERHERE/$frontmatter/;
    }

    my $status=$Apache::inputtags::status['-1'];
    if ($target eq 'grade' || $target eq 'web' || $target eq 'answer' ||
	$target eq 'tex') {
	if ( $target eq 'grade' && $Apache::inputtags::part eq '0') {
	    # if part is zero, no <part>s existed, so we need to the grading
	    if ($status eq 'CAN_ANSWER' ||$Apache::lonhomework::scantronmode) {
		&Apache::inputtags::grade;
	    } elsif ($status eq 'NEEDS_CHECKIN') {
		# no need to grade, and don't want to hide data
	    } else {
		# move any submission data to .hidden
		&Apache::inputtags::hidealldata($Apache::inputtags::part);
	    }
	} elsif ( ($target eq 'web' || $target eq 'tex') &&
		  $Apache::inputtags::part eq '0' &&
		  $status ne 'UNCHECKEDOUT' && $status ne 'NOT_YET_VIEWED') {
	    # if part is zero, no <part>s existed, so we need show the current
	    # grading status
	    my $gradestatus = &Apache::inputtags::gradestatus($Apache::inputtags::part,$target);
	    $result.= $gradestatus;
	}
	if (
	    (($target eq 'web') && ($env{'request.state'} ne 'construct')) ||
	    ($target eq 'answer') || ($target eq 'tex')
	   ) {
	    if ($target ne 'tex' &&
		$env{'form.answer_output_mode'} ne 'tex') {
		$result.="</form>";
		$result.= &Apache::lonhtmlcommon::htmlareaselectactive(&Apache::lonhtmlcommon::get_htmlareafields());
	    }
	    if ($target eq 'web') {
		$result.= &Apache::loncommon::end_page({'discussion' => 1});
	    } elsif ($target eq 'tex') {
		my $endminipage = '';
		if (not $env{'form.problem_split'}=~/yes/) {
		    $endminipage = '\end{minipage}';
		}
                if ($env{'form.print_discussions'} eq 'yes') {
		    $result.=&Apache::lonxml::xmlend($target,$parser);
		} else {
		    $result .= '\keephidden{ENDOFPROBLEM}\vskip 0.5mm\noindent\makebox[\textwidth/$number_of_columns][b]{\hrulefill}';
		    if (not $env{'request.symb'} =~ m/\.page_/) {
			$result .= $endminipage.'\end{document} ';
		    } else {
			$result .= $endminipage;
		    }
		}
	    }
	}
	if ($target eq 'grade') {
	    &Apache::lonhomework::showhash(%Apache::lonhomework::results);
	    &finalize_storage();
	}
	if ($target eq 'answer' && ($env{'request.state'} eq 'construct')
	    && $env{'form.answer_output_mode'} ne 'tex') {
	    $result.=&Apache::loncommon::end_page({'discussion' => 1});
	                        # normally we get it from above, but in CSTR
	                        # we always show answer mode too.
	}
    } elsif ($target eq 'meta') {
	if ($Apache::inputtags::part eq '0') {
	    @Apache::inputtags::response=();
	    $result=&Apache::response::mandatory_part_meta;
	}
	$result.=&Apache::response::meta_part_order();
	$result.=&Apache::response::meta_response_order();
    } elsif ($target eq 'edit') {
	&Apache::lonxml::debug("in end_problem with $target, edit");
	$result .= &problem_edit_footer();
    } elsif ($target eq 'modified') {
	 $result .= $token->[2];
    }

    if ($env{'request.state'} eq 'construct' && $target eq 'web') {
	&Apache::inputtags::check_for_duplicate_ids();
    }

    &reset_problem_globals('problem');

    return $result;
}


sub start_library {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my ($result,$form_tag_start);
    if ($#$tagstack eq 0 && $$tagstack[0] eq 'library') {
	&init_problem_globals('library');
	$Apache::lonhomework::type='problem';
    }
    if ($target eq 'edit') {
	($result,$form_tag_start)=
	    &page_start($target,$token,$tagstack,$parstack,$parser,$safeeval,
			'Edit');
	$result.=$form_tag_start.&problem_edit_header();
	my $temp=&Apache::edit::insertlist($target,$token);
	$result.=$temp;
    } elsif ($target eq 'modified') {
	$result=$token->[4];
    } elsif (($target eq 'web' || $target eq 'webgrade')
	     && ($#$tagstack eq 0 && $$tagstack[0] eq 'library')
	     && $env{'request.state'} eq "construct" ) {
	my $name=&get_resource_name($parstack,$safeeval);
	($result,$form_tag_start)=
	    &page_start($target,$token,$tagstack,$parstack,$parser,$safeeval,
			$name);
	my $rndseed=&setup_rndseed($safeeval);
	$result.=" \n $form_tag_start".	
		  '<input type="hidden" name="submitted" value="yes" />';
	$result.=&problem_web_to_edit_header($rndseed);
    }
    return $result;
}

sub end_library {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result='';
    if ($target eq 'edit') {
	$result=&problem_edit_footer();
    } elsif ($target eq 'web' 
	     && ($#$tagstack eq 0 && $$tagstack[0] eq 'library') 
	     && $env{'request.state'} eq "construct") {
	$result.='</form>'.&Apache::loncommon::end_page({'discussion' => 1});
    }
    if ( $#$tagstack eq 0 && $$tagstack[0] eq 'library') {
	&reset_problem_globals('library');
    }
    return $result;
}

sub start_definetag {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;

    my $result;

    my $name = $token->[2]->{'name'};
    my $skip=&Apache::lonxml::get_all_text("/definetag",$parser,$style);
    if ($name=~/^\//) {
	$result=
 '<br /><table bgcolor="#FFBBBB"><tr><th>END <tt>'.$name.'</tt></th></tr>';
    } else {
	$result=
 '<br /><table bgcolor="#BBFFBB"><tr><th>BEGIN <tt>'.$name.'</tt></th></tr>';
    }
    $skip=~s/\</\&lt\;/gs;
    $skip=~s/\>/\&gt\;/gs;
    $result.='<tr><td><pre>'.$skip.'</pre></td></tr></table>';
    return $result;
}

sub end_definetag {
    return '';
}

sub start_block {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;

    my $result;

    if ($target eq 'web' || $target eq 'grade'   || $target eq 'answer'  ||
	$target eq 'tex' || $target eq 'analyze' || $target eq 'webgrade') {
	my $code = $token->[2]->{'condition'};
	if (defined($code)) {
	    if (!$Apache::lonxml::default_homework_loaded) {
		&Apache::lonxml::default_homework_load($safeeval);
	    }
	    $result = &Apache::run::run($code,$safeeval);
	    &Apache::lonxml::debug("block :$code: returned :$result:");
	} else {
	    $result='1';
	}
	if ( ! $result ) {
	    my $skip=&Apache::lonxml::get_all_text("/block",$parser,$style);
	    &Apache::lonxml::debug("skipping ahead :$skip: $$parser[-1]");
	}
	$result='';
    } elsif ($target eq 'edit') {
	$result .=&Apache::edit::tag_start($target,$token);
	$result .=&Apache::edit::text_arg('Test Condition:','condition',
					  $token,40);
	$result .=&Apache::edit::end_row().&Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'condition');
	if ($constructtag) { $result = &Apache::edit::rebuild_tag($token); }
    }
    return $result;
}

sub end_block {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;
    if ($target eq "edit") {
	$result.= &Apache::edit::tag_end($target,$token,'');
    }
    return $result;
}

sub start_languageblock {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;

    my $result;

    if ($target eq 'web' || $target eq 'grade'   || $target eq 'answer' ||
	$target eq 'tex' || $target eq 'analyze' || $target eq 'webgrade') {
	my $include = $token->[2]->{'include'};
	my $exclude = $token->[2]->{'exclude'};
        my %languages=&Apache::loncommon::display_languages();
        $result='1';
	if ($include) {
            $result='';
            foreach (split(/\,/,$include)) {
                if ($languages{$_}) { $result='1'; }
            }
	}
        if ($exclude) {
            foreach (split(/\,/,$exclude)) {
                if ($languages{$_}) { $result='0'; }
            }
	}
	if ( ! $result ) {
	    my $skip=&Apache::lonxml::get_all_text("/languageblock",$parser,
						   $style);
	    &Apache::lonxml::debug("skipping ahead :$skip: $$parser[-1]");
	}
	$result='';
    } elsif ($target eq 'edit') {
	$result .=&Apache::edit::tag_start($target,$token);
	$result .=&Apache::edit::text_arg(&mt('Include Language:'),'include',
					  $token,40);
	$result .=&Apache::edit::text_arg(&mt('Exclude Language:'),'exclude',
					  $token,40);
	$result .=&Apache::edit::end_row().&Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'include',
						     'exclude');
	if ($constructtag) { $result = &Apache::edit::rebuild_tag($token); }
    }
    return $result;
}

sub end_languageblock {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;
    if ($target eq "edit") {
	$result.= &Apache::edit::tag_end($target,$token,'');
    }
    return $result;
}

sub start_instructorcomment {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;

    my $result;

    if ($target eq 'web' || $target eq 'grade'   || $target eq 'answer' ||
	$target eq 'tex' || $target eq 'analyze' || $target eq 'webgrade') {
        $result=($env{'request.role'}=~/^(in|cc|au|ca|li)/);
	if ( (! $result) or ($env{'form.instructor_comments'} eq 'hide')) {
	    my $skip=&Apache::lonxml::get_all_text("/instructorcomment",
						   $parser,$style);
	    &Apache::lonxml::debug("skipping ahead :$skip: $$parser[-1]");
	}
	$result='';
    } elsif ($target eq 'edit') {
	$result .=&Apache::edit::tag_start($target,$token);
	$result .=&Apache::edit::end_row().&Apache::edit::start_spanning_row();
    }
    return $result;
}

sub end_instructorcomment {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;
    if ($target eq "edit") {
	$result.= &Apache::edit::tag_end($target,$token,'');
    }
    return $result;
}

sub start_while {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;

    my $result;
    if ($target eq 'web' || $target eq 'grade'   || $target eq 'answer' ||
	$target eq 'tex' || $target eq 'analyze' || $target eq 'webgrade') {
	my $code = $token->[2]->{'condition'};

	push( @Apache::structuretags::whileconds, $code);
	if (!$Apache::lonxml::default_homework_loaded) {
	    &Apache::lonxml::default_homework_load($safeeval);
	}
	my $result = &Apache::run::run($code,$safeeval);
	my $bodytext=&Apache::lonxml::get_all_text("/while",$parser,$style);
	push( @Apache::structuretags::whilebody, $bodytext);
	push( @Apache::structuretags::whileline, $token->[5]);
	&Apache::lonxml::debug("s code $code got -$result-");
	if ( $result ) {
	    &Apache::lonxml::newparser($parser,\$bodytext);
	}
    } elsif ($target eq 'edit') {
	$result .=&Apache::edit::tag_start($target,$token);
	$result .=&Apache::edit::text_arg(&mt('Test Condition:'),'condition',
					  $token,40);
	$result .=&Apache::edit::end_row().&Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'condition');
	if ($constructtag) { $result = &Apache::edit::rebuild_tag($token); }
    }
    return $result;
}

sub end_while {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;

    if ($target eq 'web' || $target eq 'grade'   || $target eq 'answer' ||
	$target eq 'tex' || $target eq 'analyze' || $target eq 'webgrade') {
	my $code = pop(@Apache::structuretags::whileconds);
	my $bodytext = pop(@Apache::structuretags::whilebody);
	my $line = pop(@Apache::structuretags::whileline);
	my $return = &Apache::run::run($code,$safeeval);
	my $starttime=time;
	my $error=0;
	while ($return) {
	    if (time-$starttime >
		$Apache::lonnet::perlvar{'lonScriptTimeout'}) {
		$return = 0; $error=1; next;
	    }
	    $result.=&Apache::scripttag::xmlparse($bodytext);
	    if ($target eq 'grade' || $target eq 'answer' ||
		$target eq 'analyze') {
		# grade/answer/analyze should produce no output but if we
		# are redirecting, the redirecter should know what to do
		# with the output
		if (!$Apache::lonxml::redirection) { undef($result); }
	    }
	    $return = &Apache::run::run($code,$safeeval);
	}
	if ($error) {
	    &Apache::lonxml::error('<pre>'.&mt('Code ran too long. It ran for more than').' '.$Apache::lonnet::perlvar{'lonScriptTimeout'}.' '.&mt('seconds occured while running &lt;while&gt; on line').' '.$line.'</pre>');
	}
    } elsif ($target eq "edit") {
	$result.= &Apache::edit::tag_end($target,$token,'');
    }
    return $result;
}

# <randomlist show="1">
#  <tag1>..</tag1>
#  <tag2>..</tag2>
#  <tag3>..</tag3>
#  ...
# </randomlist>
sub start_randomlist {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'answer' || $target eq 'grade'   || $target eq 'web' ||
	$target eq 'tex'    || $target eq 'analyze' || $target eq 'webgrade') {
	my $body= &Apache::lonxml::get_all_text("/randomlist",$parser);
	my $b_parser= HTML::LCParser->new(\$body);
	$b_parser->xml_mode(1);
	$b_parser->marked_sections(1);
	my $b_tok;
	my @randomlist;
	my $list_item;
	while($b_tok = $b_parser->get_token() ) {
	    if($b_tok->[0] eq 'S') { # start tag
		# get content of the tag until matching end tag
		# get all text upto the matching tag
		# and push the content into @randomlist
		$list_item = &Apache::lonxml::get_all_text('/'.$b_tok->[1],
							   $b_parser);
		$list_item = "$b_tok->[4]"."$list_item"."</$b_tok->[1]>";
		push(@randomlist,$list_item);
		#  print "<br /><b>START-TAG $b_tok->[1], $b_tok->[4],
                #         $list_item</b>";
	    }
	    if($b_tok->[0] eq 'T') { # text
		# what to do with text in between tags?
		#  print "<b>TEXT $b_tok->[1]</b><br />";
	    }
	    # if($b_tok->[0] eq 'E') { # end tag, should not happen
	    #  print "<b>END-TAG $b_tok->[1]</b><br />";
	    # }
	}
	if (@randomlist) {
	    my @idx_arr = (0 .. $#randomlist);
	    &Apache::structuretags::shuffle(\@idx_arr);
	    my $bodytext = '';
	    my $show=$#randomlist;
	    my $showarg=&Apache::lonxml::get_param('show',$parstack,$safeeval);
	    $showarg--;
	    if ( ($showarg >= 0) && ($showarg < $show) ) { $show = $showarg; }
	    for(0 .. $show) {
		$bodytext .= "$randomlist[ $idx_arr[$_] ]";
	    }
	    &Apache::lonxml::newparser($parser,\$bodytext);
	}
    } elsif ($target eq 'edit' ) {
	$result .=&Apache::edit::tag_start($target,$token);
	$result .=&Apache::edit::text_arg('Maximum Tags to Show:','show',
					   $token,5);
	$result .=&Apache::edit::end_row().&Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified' ) {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'show');
	if ($constructtag) { $result = &Apache::edit::rebuild_tag($token); }
    }
    return $result;
}

sub shuffle {
    my $a=shift;
    my $i;
    if (ref($a) eq 'ARRAY' && @$a) {
	&Apache::response::pushrandomnumber();
	for($i=@$a;--$i;) {
	    my $j=int(&Math::Random::random_uniform() * ($i+1));
	    next if $i == $j;
	    @$a[$i,$j] = @$a[$j,$i];
	}
	&Apache::response::poprandomnumber();
    }
}

sub end_randomlist {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;
    if ($target eq 'edit' ) {
	$result=&Apache::edit::tag_end($target,$token,
				       'End Randomly Parsed Block');
    }
    return $result;
}

sub ordered_show_check {
    my $last_part=$Apache::inputtags::partlist[-2];
    my $in_order=
	&Apache::lonnet::EXT('resource.'.$Apache::inputtags::part.'.ordered');
    my $in_order_show=1;
    if ($last_part ne '0' && lc($in_order) eq 'yes') {
	$in_order_show=&Apache::response::check_status($last_part);
    }
    return $in_order_show;
}

sub start_part {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    if (!$Apache::lonxml::metamode) {
	&Apache::lonxml::startredirection(); # we'll use redirection to fix up 
	                                     # duedates.
    }
    my $result='';
    my $id= &Apache::lonxml::get_param('id',$parstack,$safeeval);
    if ($id =~ /^\s*$/) { $id = $Apache::lonxml::curdepth; }
    $Apache::inputtags::part=$id;
    push(@Apache::inputtags::partlist,$id);
    @Apache::inputtags::response=();
    @Apache::inputtags::previous=();
    @Apache::inputtags::previous_version=();
    $Apache::lonhomework::problemstatus=&get_problem_status($id);
    my $hidden=&Apache::loncommon::check_if_partid_hidden($Apache::inputtags::part);
    my $newtype=&Apache::lonnet::EXT("resource.$id.type");
    if ($newtype) { $Apache::lonhomework::type=$newtype; }
    my $in_order_show=&ordered_show_check();
    my $expression='$external::part=\''.$Apache::inputtags::part.'\';';
    $expression.='$external::type=\''.$Apache::lonhomework::type.'\';';
    &Apache::run::run($expression,$safeeval);

    if ($target eq 'meta') {
	my $display=&Apache::lonxml::get_param('display',$parstack,$safeeval);
	return &Apache::response::mandatory_part_meta.
	       &Apache::response::meta_parameter_write('display','string',$display,'Part Description');
    } elsif ($target eq 'web' || $target eq 'grade' ||
	     $target eq 'answer' || $target eq 'tex') {
	if ($hidden || !$in_order_show) {
	    my $bodytext=&Apache::lonxml::get_all_text("/part",$parser,$style);
	} else {
	    my ($status,$accessmsg) = &Apache::lonhomework::check_access($id);
	    push (@Apache::inputtags::status,$status);
	    my $expression='$external::datestatus="'.$status.'";';
	    $expression.='$external::gradestatus="'.$Apache::lonhomework::history{"resource.$id.solved"}.'";';
	    &Apache::run::run($expression,$safeeval);
	    if ($env{'request.state'} eq 'construct') {
		&set_problem_state($Apache::inputtags::part); 
	    }
	    if (( $status eq 'CLOSED' ) ||
		( $status eq 'UNCHECKEDOUT') ||
		( $status eq 'NOT_YET_VIEWED') ||
		( $status eq 'BANNED') ||
		( $status eq 'UNAVAILABLE') ||
		( $status eq 'INVALID_ACCESS')) {
		my $bodytext=&Apache::lonxml::get_all_text("/part",$parser,
							   $style);
		if ( $target eq "web" ) {
		    $result="<br />".&mt('Part is not open to be viewed. It')." $accessmsg<br />";
		} elsif ( $target eq 'tex' ) {
		    if (not $env{'form.problem_split'}=~/yes/) {
			$result="\\end{minipage}\\vskip 0 mm ".&mt('Part is not open to be viewed. It')." $accessmsg \\\\\\begin{minipage}{\\textwidth}";
		    } else {
			$result="\\vskip 0 mm ".&mt('Part is not open to be viewed. It')." $accessmsg \\\\";
		    }
		}
	    } else {
		if ($target eq 'tex') {
		    if (not $env{'form.problem_split'}=~/yes/) {
			if ($$tagstack[-2] eq 'td') {
			    $result.='\vskip 0 mm \noindent \begin{minipage}{\textwidth}\noindent';
			} else {
			    $result.='\noindent \end{minipage}\vskip 0 mm \noindent \begin{minipage}{\textwidth}\noindent';
			}
		    }
		    my $weight = &Apache::lonnet::EXT("resource.$id.weight");
		    my $allkeys=&Apache::lonnet::metadata($env{'request.uri'},'packages');
		    my @allkeys = split /,/,$allkeys;
		    my $allow_print_points = 0;
		    foreach my $partial_key (@allkeys) {
			if ($partial_key=~m/^part_(.*)$/) {
			    if ($1 ne '0') {$allow_print_points=1;}
			}
		    }
		    my $maxtries = &Apache::lonnet::EXT("resource.$id.maxtries");
		    if (defined($maxtries) && $maxtries < 0) {
			$allow_print_points=0;
		    }
		    if (lc($env{'course.'.$env{'request.course.id'}.
				    '.disableexampointprint'}) eq 'yes') {
			$allow_print_points=0;
		    }
		    if (($Apache::lonhomework::type eq 'exam') && ($allow_print_points)) { $result .= '\fbox{\textit{'.$weight.' pt}}';}
		} elsif ($target eq 'web') {
		    $result.='<a name="'.&escape($Apache::inputtags::part).'" />';
		}
	    }
	}
    } elsif ($target eq 'edit') {
	$result.=&Apache::edit::tag_start($target,$token);
	$result.=&Apache::edit::text_arg('Part ID:','id',$token).
	    &Apache::loncommon::help_open_topic("Part_Tag_Edit_Help").
	    '&nbsp;&nbsp;'.
&Apache::edit::text_arg('Displayed Part Description:','display',$token).
		&Apache::edit::end_row().&Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'id','display');
	if ($constructtag) {
	    #limiting ids to only letters numbers, and space
	    $token->[2]->{'id'}=~s/[^A-Za-z0-9 ]//gs;
	    $result = &Apache::edit::rebuild_tag($token);
	}
    }
    return $result;
}

sub end_part {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    &Apache::lonxml::debug("in end_part $target ");
    my $status=$Apache::inputtags::status['-1'];
    my $hidden=&Apache::loncommon::check_if_partid_hidden($Apache::inputtags::part);
    my $in_order_show=&ordered_show_check();
    my $result;
    if (!$Apache::lonxml::metamode) {
	$result = &Apache::lonxml::endredirection(); # started in &start_part
	$Apache::lonxml::post_evaluate=0;
    }
    if ($target eq 'grade') {
	if (($status eq 'CAN_ANSWER' || $Apache::lonhomework::scantronmode) &&
	    !$hidden && $in_order_show) {
	    $result.=&Apache::inputtags::grade;
	} else {
	    # move any submission data to .hidden
	    &Apache::inputtags::hidealldata($Apache::inputtags::part);
	}
    } elsif (($target eq 'web' || $target eq 'tex') &&
	     !$hidden && $in_order_show) {
	my $gradestatus=&Apache::inputtags::gradestatus($Apache::inputtags::part,
							$target);
	if ($Apache::lonhomework::type eq 'exam' && $target eq 'tex') {
	    $gradestatus='';
	}
	$result.=$gradestatus;
	if ($$tagstack[-2] eq 'td' and $target eq 'tex') {$result.='\end{minipage}';} 
    } elsif ($target eq 'edit') {
	$result.=&Apache::edit::end_table();
    } elsif ($target eq 'modified') {
	 $result .= $token->[2];
    }
    pop @Apache::inputtags::status;
    $Apache::inputtags::part='';
    $Apache::lonhomework::type = $Apache::lonhomework::default_type;
    return $result;
}

sub start_preduedate {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    if ($target eq 'web' || $target eq 'grade'    || $target eq 'answer' ||
	$target eq 'tex' || $target eq 'webgrade') {
	&Apache::lonxml::debug("State in preduedate is ". $Apache::inputtags::status['-1']);
	if (!$Apache::lonhomework::scantronmode &&
	    $Apache::inputtags::status['-1'] ne 'CAN_ANSWER' &&
	    $Apache::inputtags::status['-1'] ne 'CANNOT_ANSWER') {
	    &Apache::lonxml::debug("Wha? ". ($Apache::inputtags::status['-1'] ne 'SHOW_ANSWER'));
	    &Apache::lonxml::get_all_text("/preduedate",$parser,$style);
	}
    }
    return '';
}

sub end_preduedate {
    return '';
}

# In all the modes where <postanswerdate> text is 
# displayable,  all we do is eat up the text between the start/stop
# tags if the conditions are not right to display it.
sub start_postanswerdate {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $pav = &Apache::lonnet::allowed('pav', $env{'request.course.id'}) ||
	&Apache::lonnet::allowed('pav',
			   $env{'request.course.id'}.'/'.$env{'request.course.sec'});
    if ($target eq 'web' || $target eq 'grade' || $target eq 'webgrade' ||
	$target eq 'tex' ) {
	if ($Apache::lonhomework::scantronmode ||
	    $Apache::inputtags::status['-1'] ne 'SHOW_ANSWER' ||
	    (($target eq 'tex') && !$pav)) {
	    &Apache::lonxml::get_all_text("/postanswerdate",$parser,$style);
	}
    }
    return '';
}

sub end_postanswerdate {
    return '';
}

sub start_notsolved {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    if ($target eq 'web' || $target eq 'grade' || $target eq 'answer' ||
	$target eq 'tex' || $target eq 'webgrade') {
	my $gradestatus=$Apache::lonhomework::history{"resource.$Apache::inputtags::part.solved"};
	&Apache::lonxml::debug("not solved has :$gradestatus:");
	if ($gradestatus =~ /^correct/ &&
	    &Apache::response::show_answer()) {
	    &Apache::lonxml::debug("skipping");
	    &Apache::lonxml::get_all_text("/notsolved",$parser,$style);
	}
    }
    return '';
}

sub end_notsolved {
    return '';
}

sub start_solved {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    if ($target eq 'web' || $target eq 'grade' || $target eq 'answer' ||
	$target eq 'tex') {
	my $gradestatus=$Apache::lonhomework::history{"resource.$Apache::inputtags::part.solved"};
	if ($gradestatus !~ /^correct/ ||
	    !&Apache::response::show_answer()) {
	    &Apache::lonxml::get_all_text("/solved",$parser,$style);
	}
    }
    return '';
}

sub end_solved {
    return '';
}

sub start_problemtype {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'web' || $target eq 'grade'   || $target eq 'answer' ||
	$target eq 'tex' || $target eq 'analyze' || $target eq 'webgrade') {
	my $mode=lc(&Apache::lonxml::get_param('mode',$parstack,$safeeval));
	if (!defined($mode)) { $mode='show'; }
	my $for=&Apache::lonxml::get_param('for',$parstack,$safeeval);
	my $found=0;
	foreach my $type (split(',',$for)) {
	    if ($Apache::lonhomework::type eq lc($type)) { $found=1; }
	}
	if ($mode eq 'show' && !$found) {
	    &Apache::lonxml::get_all_text("/problemtype",$parser,$style);
	}
	if ($mode eq 'hide' && $found) {
	    &Apache::lonxml::get_all_text("/problemtype",$parser,$style);
	}
    } elsif ($target eq 'edit') {
	$result .=&Apache::edit::tag_start($target,$token);
	$result.=&Apache::edit::select_arg('Mode:','mode',
					   [['show','Show'],
					    ['hide','Hide']]
					   ,$token);
	$result .=&Apache::edit::checked_arg('When used as type(s):','for',
					     [ ['exam','Exam/Quiz Problem'],
					       ['survey','Survey'],
					       ['problem','Homework Problem'] ]
					     ,$token);
	$result .=&Apache::edit::end_row().&Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'mode','for');
	if ($constructtag) { $result = &Apache::edit::rebuild_tag($token); }
    }
    return $result;
}

sub end_problemtype {
    return '';
}

sub start_startouttext {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my @result=(''.'');
    if ($target eq 'edit' || $target eq 'modified' ) { @result=('','no'); }
    return (@result);
}

sub end_startouttext {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result='';
    my $text='';

    if ($target eq 'edit') {
	$text=&Apache::lonxml::get_all_text("endouttext",$parser,$style);
	$result.=&Apache::edit::start_table($token)."<tr><td>".&mt('Text Block')."</td>
<td>".&mt('Delete:').
                 &Apache::edit::deletelist($target,$token)
		 ."</td>
<td>".
                 &Apache::edit::insertlist($target,$token).
		 '</td><td align="right" valign="top">' .
		 &Apache::loncommon::helpLatexCheatsheet().
		 &Apache::edit::end_row().
                 &Apache::edit::start_spanning_row()."\n".
		 &Apache::edit::editfield($token->[1],$text,"",80,8,1);
    }
    if ($target eq 'modified') {
	$result='<startouttext />'.&Apache::edit::modifiedfield("endouttext",$parser);
    }
    if ($target eq 'tex') {
	$result .= '\noindent ';
    }
    return $result;
}

sub start_endouttext {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result='';
    if ($target eq "edit" ) { $result="</td></tr>".&Apache::edit::end_table()."\n"; }
    if ($target eq "modified") {
	$result='<endouttext />'.
	    &Apache::edit::handle_insertafter('startouttext');
    }
    return $result;
}

sub end_endouttext {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my @result=('','');
    if ($target eq "edit" || $target eq 'modified') { @result=('','no'); }
    return (@result);
}

sub delete_startouttext {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    #  my $text=&Apache::lonxml::get_all_text("endouttext",$parser,$style);
    my $text=$$parser['-1']->get_text("/endouttext");
    my $ntoken=$$parser['-1']->get_token();
    &Apache::lonxml::debug("Deleting :$text: and :$ntoken->[0]:$ntoken->[1]:$ntoken->[2]: for startouttext");
    &Apache::lonxml::end_tag($tagstack,$parstack,$ntoken);
    # Deleting 2 parallel tag pairs, but we need the numbers later to look like
    # they did the last time round
    &Apache::lonxml::increasedepth($ntoken);
    &Apache::lonxml::decreasedepth($ntoken);
    return 1;
}

sub start_simpleeditbutton {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result='';
    if (($env{'form.simple_edit_button'} ne 'off') &&
	($target eq 'web') &&
        (&Apache::lonnet::allowed('mdc',$env{'request.course.id'}))) {
        my $url=$env{'request.noversionuri'};
        $url=~s/\?.*$//;
	my ($symb) = &Apache::lonnet::whichuser();
	$result='<table width="100%" bgcolor="#FFFFAA" border="2"><tr><td>'.
                '<a href="'.$url.'/smpedit?symb='.&escape($symb).'">'.&mt('Edit').'</a> - '.&mt('Note: it can take up to 10 minutes for changes to take effect for all users.').
&Apache::loncommon::help_open_topic('Caching').'</td></tr></table><br />';
    }
    return $result;
}

sub end_simpleeditbutton {
    return '';
}

1;
__END__
