#   This module defines and implements a class that represents
#   a connection to a lond daemon.
#
# $Id: LondTransaction.pm,v 1.9 2005/06/16 22:59:05 albertel Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

=pod

=head1 Synopsis

  LondTransaction objectifies the state of a transaction between lonc and
  lond (loncnew really).  

=cut

use strict;

package LondTransaction;

=pod

=head1 Description

LondTransaction objects hold the state required to manage a 
transaction between lonc and lond, from the loncnew point of view.
The state consists of the following member data:

=item request

The text of the request to send to lond.

=item active

If zero, the request is not active and the londSocket item is not
defined.

=item londSocket

If the request is active,
this member contains the LondConnection object reference that
this request is being processed on.

=item deferred

True if the request is a deferred (delayed) request.
The member data below are either present or not depending on 
whether or not deferred is true.

=item clientSocket
   
If deferred is false, this member data is defined and is the 
handle to the socket that is connected to the apache child that
has requested this transaction.

=item DeferredFile

If deferred is false, this member data is defined and is the name
of the file that contains the deferred request.  When the transaction
is retired, this file will be deleted.

=head1 Member Functions

=head2  Operational functions

=cut

=pod

=item new

Creates a new transaction object.

=cut

sub new {

    my ($class, $Transaction) = @_;

    
    my $self   = {request     => $Transaction,
		  active      => 0,
		  deferred    => 0};
    bless($self, $class);
    return $self;
}

=pod

=item Activate

Activates the transaction by assigning it to a LondConnection object

Parameters:

=over 3

=item  Connection


Reference to the LondConnection object along which the transaction
will be carried.

=back 

=cut

sub Activate {


    my ($self, $Connection) = @_;


    $self->{londSocket} = $Connection; # Store the connection object and
    $self->{active}     = 1;	       # Indicate it's active.

}

=pod

=item Retire


Retires a transaction after successful completion.  If the
transaction is deferred, the deferred file is destroyed.
Otherwise this is a noop.

=cut

sub Retire {
    my $self     = shift;

    if($self->{deferred}) {
	unlink $self->{DeferredFile};
    }

    #  Destroy my member data to release reference counts.

    delete $self->{londSocket};
    delete $self->{clientSocket};
    delete $self->{DeferredFile};

}

=pod

=item SetDeferred

Sets the state of a transaction to deferred, the deferred member
is set true, clientSocket is undefined, and DeferredFile is set.

Parameters:
 
=over 3

=item File

Name of the file that holds the deferred transaction.

=back 

=cut

sub SetDeferred {


    my ($self, $File) = @_;

    $self->{deferred}      = 1;
    $self->{DeferredFile} = $File;
}

=pod

=item  SetClient

Sets the state of a transaction to not deferred.  The deferred member
is set false, clientSocket is set and DeferredFile is undefined.

Parameters:

=over 3

=item Socket

  The socket open on the client.

=back

=cut

sub SetClient {

    my ($self, $Client) = @_;
    
    $self->{deferred}     = 0;
    $self->{clientSocket} = $Client; 
}

=pod

=head2  Selectors


=item isDeferred

Returns the state of the deferred member.

=cut

sub isDeferred {
    my $self   = shift;
    return $self->{deferred};
}

=pod

=item isActive

Returns the value of the active member.

=cut

sub isActive {
    my $self = shift;
    return $self->{active};
}

=pod

=item getClient

If not deferred returns the client socket, else returns undef.

=cut

sub getClient {
    my $self = shift;
    if($self->{deferred}) {
	return undef;
    } else {
	return $self->{clientSocket};
    }
}


=pod

=item getFile

If deferred, returns the name of the deferred queue file else
returns undef.

=cut

sub getFile {
    my $self = shift;
    if($self->{deferred}) {
	return $self->{DeferredFile};
    } else {
	return undef;
    }
}


=pod

=item getServer

  If active returns the lond server socket else undef.

=cut

sub getServer {
    my $self  = shift;

    if($self->{active}) {
	return $self->{londSocket};
    } else {
	return undef;
    }
}

=pod

=item getRequest

  Returns the remaining request text.

=cut

sub getRequest {
    my $self = shift;
    return $self->{request};
}

=pod

=item getLoggableRequest

   Use this top get what the request is when you don't want to spew
   sensitive data into logs

=cut

sub getLoggableRequest {
    my $self = shift;
    my ($cmd,$subcmd)=split(':',$self->{request});
    if ($cmd eq 'encrypt') {
	return "$cmd:$subcmd";
    }
    return $cmd;
}

1;
