# The LearningOnline Network
# Pick a course
#
# $Id: lonpickcourse.pm,v 1.63.4.1 2007/08/02 19:53:54 albertel Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

package Apache::lonpickcourse;

use strict;
use Apache::Constants qw(:common);
use Apache::loncommon;
use Apache::loncoursedata;
use Apache::lonnet;
use Apache::lonlocal;
use Apache::longroup;
use LONCAPA;

sub handler {
    my $r = shift;
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    return OK if $r->header_only;

# ------------------------------------------------------------ Print the screen

    # Get parameters from query string
    &Apache::loncommon::get_unprocessed_cgi
        ($ENV{'QUERY_STRING'},['domainfilter','form','cnumelement',
			       'cdomelement','cnameelement','roleelement',
                               'multiple','type','setroles']);

    my ($type,$title,$jscript,$multelement,$multiple,$roleelement,
        $lastaction,$autosubmit,$submitopener);

    # Get course type - Course or Non-standard Course.
    $type = $env{'form.type'};
    if (!defined($env{'form.type'})) {
        $type = 'Course';
    }
    $title = &mt('Selecting a [_1]',$type);

    # Setup for multiple course selections, if flag for multiples set.
    $multiple = $env{'form.multiple'};
    if ($multiple) {
        ($jscript,$multelement) = &multiples_tag();
        $title = &mt('Selecting [_1](s)',lc($type));
    }

    # if called when a DC is selecting a course 
    my $roledom = $env{'form.roleelement'};
    if ($roledom) {
        $roleelement = '<input type="hidden" name="roleelement" value="'.$roledom.'" />';
        $submitopener = &Apache::lonroles::processpick();
        $autosubmit = 'process_pick("'.$roledom.'")';
    }

    my $onlyown = 0;
    # if called to assign course-based portfolio access control
    if ((($env{'form.form'} eq 'portform') && (!$env{'user.adv'}))) {
        $onlyown = 1;
    }

    my %loaditem;
    if ($env{'form.form'} eq 'cu' && $env{'form.pickedcourse'}) {
        $loaditem{'onload'} ="setSections()";
    }
    $r->print(&Apache::loncommon::start_page($title,undef,
					     {'add_entries' => \%loaditem,
					      'no_nav_bar'  => 1, }));

    if ($env{'form.form'} eq 'portform') {
        $lastaction = 'document.courselist.submit()';
    } elsif ($env{'form.form'} eq 'cu') {
        $lastaction = 
             'document.courselist.pickedcourse.value = cdom+"_"+cname;'."\n".
             'document.courselist.submit();';
    } else {
        $lastaction = 'self.close()';
    }

    # if called to assign a role in a course to a user via CUSR
    if ($env{'form.form'} eq 'cu') {
        $r->print(&create_user_javascript($type)); 
    }

    # print javascript functions for choosing a course 
    if ($env{'form.gosearch'} || $onlyown) {
        $r->print(&gochoose_javascript($type,$multiple,$autosubmit,$lastaction));
    }
    $r->print('<script type="text/javascript">'.$jscript.'</script>');
    $r->print($submitopener);

# ------------------------------------------ Display of filters to limit search
    my $filter = {};
    my $action = '/adm/pickcourse';
    if (!$onlyown) {
        my $filterlist = ['domainfilter','descriptfilter',
                          'instcodefilter','ownerfilter',
                          'ownerdomfilter'];
        # course ID filter for DCs only
        if ($env{'user.adv'} && $env{'form.domainfilter'} &&
            exists($env{'user.role.dc./'.$env{'form.domainfilter'}.'/'})) {
            push(@{$filterlist},'coursefilter');
        }
        $r->print(&build_filters($filterlist,$type,$roleelement,$multelement,
                                 $filter,$action));
    }

# ---------------------------------------------------------------- Get the data
    if ($env{'form.gosearch'} || $onlyown) {
        my %courses = &search_courses($r,$type,$onlyown,$filter);
        &display_matched_courses($r,$type,$multiple,$action,%courses);
    }
    $r->print(&Apache::loncommon::end_page());
    return OK;
}

sub create_user_javascript {
    my ($type) = @_;
    my $output;
    #javascript for reporting sections and groups then closing
    if ($env{'form.pickedcourse'}) {
        my ($cdom,$cnum) = split/_/,$env{'form.pickedcourse'};
        my $sec_element = 'currsec';
        my $grplist_element = 'groups';
        my ($sections,$groups) =
          &Apache::loncommon::get_secgrprole_info($cdom,$cnum,'',$type);
        my $num_sections = scalar(@{$sections});
        my $seclist = join(',',@{$sections});
        my $num_groups = scalar(@{$groups});
        my $groupslist = join(',',@{$groups});
        $output = qq|
<script type="text/javascript">
function setSections() {
    opener.document.$env{"form.form"}.$grplist_element.value='$groupslist';
    window.opener.setSect('$seclist');
    self.close()
}
</script>
|;
;
    }
    return $output;
}

sub display_matched_courses {
    my ($r,$type,$multiple,$action,%courses) = @_;
    if ($env{'form.form'} eq 'portform') {
        $action = '/adm/portfolio';
    }
    $r->print('<form name="courselist" method="post" action="'.$action.'">');
    my %by_descrip;
    my $numcourses = keys(%courses);
    foreach my $course (keys(%courses)) {
        my $descr;
        if ($courses{$course} =~ m/^([^:]*):/i) {
            $descr = &unescape($1);
        } else {
            $descr = &unescape($courses{$course});
        }
        my $description = $descr;
        push (@{$by_descrip{$description}}, $course);
    }
    if ($numcourses > 1 && $multiple) {
        $r->print('<input type="button" value="check all"
                  onclick="javascript:checkAll(document.courselist.course_id)" />
                  &nbsp;&nbsp;<input type="button" value="uncheck all"
                  onclick="javascript:uncheckAll(document.courselist.course_id)" />
                  <br /><br />');
    }
    foreach my $description (sort { lc($a) cmp lc($b) } (keys(%by_descrip))) {
        foreach my $course (@{$by_descrip{$description}}) {
            my $cleandesc=&HTML::Entities::encode($description,'<>&"');
            $cleandesc=~s/'/\\'/g;
            my ($cdom,$cnum)=split(/\_/,$course);
            my ($descr,$instcode,$owner,$ttype) = split/:/,$courses{$course};
            $r->print(&course_chooser($multiple,$cdom,$cnum,$cleandesc));
            $r->print($description.'('.
                      (&Apache::lonnet::domain($cdom,'description')?
                       &Apache::lonnet::domain($cdom,'description'):$cdom).")");
            unless ($instcode eq '') {
                $r->print(" - ".&unescape($instcode));
            }
            unless ($owner eq '') {
                $r->print(", owner - ".&unescape($owner));
            }
            unless ($ttype eq '') {
                $r->print('('.&unescape($ttype).')');
            }
            if ($multiple) { $r->print("</label>\n"); }
            $r->print("<br />\n");
        }
    }
    if (!%courses) {
        $r->print(&mt('None found'));
    } elsif ($multiple) {
        $r->print('<input type="button" value="Submit" onClick="gochoose('."'','','')".'" />');
    }
    $r->print('<input type="hidden" name="form" value="'.$env{'form.form'}.'" />'.
              "\n".'<input type="hidden" name="pickedcourse" value="" />'."\n".
              '<input type="hidden" name="type" value="'.$type.'" />'."\n");
    if ((exists($env{'form.roleelement'})) && ($env{'form.form'} eq 'rolechoice')) {
        $r->print('<input type="hidden" name="roleelement" value="'.
                  $env{'form.roleelement'}.'" />'."\n");
    }
    if ($env{'form.form'} eq 'portform') {
        $r->print('<input type="hidden" name="cnum" value="" />');
        $r->print('<input type="hidden" name="cdom" value="" />');
        $r->print('<input type="hidden" name="setroles" value="'.$env{'form.setroles'}.'" />');
        $r->print('<input type="hidden" name="action" value="rolepicker" />');
    } elsif ($env{'form.form'} eq 'modifycourse') {
        $r->print('<input type="hidden" name="phase" value="menu" />'."\n");
        $r->print(&Apache::lonhtmlcommon::echo_form_input(['phase','pickedcourse','type','form']));
    } else {
        $r->print('<input type="hidden" name="cnumelement" value="'.
                  $env{'form.cnumelement'}.'" />'."\n".  
                  '<input type="hidden" name="cdomelement" value="'.
                  $env{'form.cdomelement'}.'" />'."\n");
    }
    $r->print("</form>\n");
    return;
}


sub multiples_tag {
    my $jscript = &Apache::loncommon::check_uncheck_jscript();
    my $multelement = '<input type="hidden" name="multiple" value="1" />';
    return ($jscript,$multelement);
}

sub build_filters {
    my ($filterlist,$type,$roleelement,$multelement,$filter,$action,
        $caller) = @_;
    my $list;
    my $formname;
    if (defined($env{'form.form'})) {
         $formname = $env{'form.form'};
    } else {
         $formname = $caller;
    }
    my ($domainselectform,$sincefilterform,$ownerdomselectform,
        $instcodeform,$typeselectform);
    foreach my $item (@{$filterlist}) {
        $filter->{$item} = $env{'form.'.$item};
        if ($item ne 'descriptfilter' && $item ne 'instcodefilter') {
	    if ($item eq 'domainfilter' || $item eq 'ownerdomfilter') {
		$filter->{$item} = &LONCAPA::clean_domain($filter->{$item});
	    } elsif ($item eq 'coursefilter') {
		$filter->{$item} = &LONCAPA::clean_courseid($filter->{$item});
	    } elsif ($item eq 'ownerfilter') {
		$filter->{$item} = &LONCAPA::clean_username($filter->{$item});
	    } else {
		$filter->{$item} =~ s/\W//g;
	    }
            if (!$filter->{$item}) {
                $filter->{$item} = '';
            }
        }
        if ($item eq 'domainfilter') {
	    my $allow_blank = 1;
	    if ($formname eq 'portform') {
		$filter->{$item} ||= $env{'user.domain'};
		$allow_blank=0;
	    }
            $domainselectform =
		&Apache::loncommon::select_dom_form($filter->{$item},
						    'domainfilter',
						    $allow_blank);
        } elsif ($item eq 'ownerdomfilter') {
            $ownerdomselectform =
     &Apache::loncommon::select_dom_form($filter->{$item},'ownerdomfilter',1);
        } else {
            $list->{$item} = &HTML::Entities::encode($filter->{$item},'<>&"');
        }
    }

    # last course activity filter and selection
    $filter->{'sincefilter'} = $env{'form.sincefilter'};
    $filter->{'sincefilter'} =~ s/[^\d-]//g;
    if (!$filter->{'sincefilter'}) { $filter->{'sincefilter'}=-1; }
    $sincefilterform=&Apache::loncommon::select_form($filter->{'sincefilter'},
       'sincefilter',('-1'=>'',
                      '86400' => 'today',
                      '604800' => 'last week',
                      '2592000' => 'last month',
                      '7776000' => 'last three months',
                      '15552000' => 'last six months',
                      '31104000' => 'last year',
                      'select_form_order' =>
                      ['-1','86400','604800','2592000','7776000',
                      '15552000','31104000']));

    my %lt = (
              'cac' => &mt('[_1] Activity',$type),
              'cde' => &mt('[_1] Description',$type),
              'cdo' => &mt('[_1] Domain',$type),
              'cin' => &mt('Course Institutional Code'),
              'cow' => &mt("[_1] Owner's Username",$type),
              'cod' => &mt("[_1] Owner's Domain", $type),
              'cog' => &mt('Course Type')
             );

    $typeselectform =  '<select name="type" size="1"';
    $typeselectform .= ">\n";
    if ($type eq 'Course') {
        $instcodeform = 
                '<input type="text" name="instcodefilter" size="10" value="'.
                $list->{'instcodefilter'}.'" />';
    }
    foreach my $posstype ('Course','Non-standard Course') {
        $typeselectform.='<option value="'.$posstype.'" "'.
            ($posstype eq $type ? 'selected="selected" ' : ''). ">$posstype</option>\n";
    }
    $typeselectform.="</select>";
    my $output = qq|
<form method="post" name="filterpicker" action="$action">
<input type="hidden" name="form" value="$formname" />
|;
    if ($formname eq 'modifycourse') {
        $output .= '<input type="hidden" name="phase" value="courselist" />';
    } else {     
        my $name_input;
        if ($env{'form.cnameelement'} ne '') {
            $name_input = '<input type="hidden" name="cnameelement" value="'.
                          $env{'form.cnameelement'}.'" />';
        }
        $output .= qq|
<input type="hidden" name="cnumelement" value="$env{'form.cnumelement'}" />
<input type="hidden" name="cdomelement" value="$env{'form.cdomelement'}" />
$name_input
$roleelement
$multelement
|;
        if ($formname eq 'portform') {
            $output .= '<input type="hidden" name="setroles" value="'.$env{'form.setroles'}.'" />';
        }
    }
    if ($sincefilterform) { 
        $output .= $lt{'cac'}.': '.$sincefilterform.'<br />'."\n";
    }
    if ($domainselectform) {
        $output .= $lt{'cdo'}.': '.$domainselectform.'<br />'."\n";
    }
#    if ($typeselectform) {
#        $output .= $lt{'cog'}.': '.$typeselectform.'<br />'."\n";
#    }
    if ($instcodeform) {
        $output .= $lt{'cin'}.': '.$instcodeform.'<br />'."\n";
    }
    if (exists($filter->{'ownerfilter'})) {
        $output .= $lt{'cow'}.': '.
                   '<input type="text" name="ownerfilter" size="10" value="'.
                   $list->{'ownerfilter'}.'" /><br />'."\n".
                   $lt{'cod'}.': '.$ownerdomselectform.'<br />'."\n";
    }
    if (exists($filter->{'coursefilter'})) {
        $output .= &mt('LON-CAPA course ID').': '.
                   '<input type="text" name="coursefilter" size="25" value="'.
                    $list->{'coursefilter'}.'" /><br />';
    }
    if (exists($filter->{'descriptfilter'})) {
        $output .=$lt{'cde'}.': '.
                  '<input type="text" name="descriptfilter" size="40" value="'.
                  $list->{'descriptfilter'}.'" />'."\n";
    }
    $output .= qq|
<p><input type="submit" name="gosearch" value="Search" /></p>
</form>
<hr />
|;
    return $output; 
}

sub search_courses {
    my ($r,$type,$onlyown,$filter) = @_;
    my %courses;
    if (!$onlyown) {
        $r->print(&mt('Searching').' ...<br />&nbsp;<br />');
        $r->rflush();
        if (($filter->{'ownerfilter'} ne '') ||
            ($filter->{'ownerdomfilter'} ne '')) {
            $filter->{'combownerfilter'} = $filter->{'ownerfilter'}.':'.
                                           $filter->{'ownerdomfilter'};
        }
        foreach my $item ('descriptfilter','instcodefilter','coursefilter',
                          'combownerfilter') {
            if (!$filter->{$item}) {
                $filter->{$item}='.';
            }
        }
        if ($type eq '') { $type = '.'; }
        my $timefilter = 
           ($filter->{'sincefilter'}==-1?1:time-$filter->{'sincefilter'});
        %courses = 
            &Apache::lonnet::courseiddump($filter->{'domainfilter'},
                                          $filter->{'descriptfilter'},
                                          $timefilter,
                                          $filter->{'instcodefilter'},
                                          $filter->{'combownerfilter'},
                                          $filter->{'coursefilter'},
                                          undef,undef,$type);
    } else {
        $r->print('<br />');
        my %coursehash = &Apache::loncommon::findallcourses(); 
        foreach my $cid (sort(keys(%coursehash))) {
            $courses{$cid} = $env{'course.'.$cid.'.description'};
        }
    }
    return %courses;
}

sub course_chooser {
    my ($multiple,$cdom,$cnum,$cleandesc) = @_;
    my $output; 
    if ($multiple) {
        $output = '<label><input type="checkbox" name="course_id" value="'.$cdom.'_'.$cnum.'" />'."\n";
    } else {
        $output = '<input type="button" value="Select" onClick="gochoose('.
                  "'".$cnum."','".$cdom."','".$cleandesc."')".'" />'."\n";
    }
    return $output;
}

sub gochoose_javascript {
    my ($type,$multiple,$autosubmit,$lastaction) = @_;
    my %elements = (
                     'Course' => {
                                 name  => 'coursepick',
                                 total => 'coursetotal',
                                 list  => 'courselist',
                                 },
                     'Non-standard Course' => {
                                 name  => 'grouppick',
                                 total => 'grouptotal',
                                 list  => 'grouplist',
                              },
                    );
    my $output .= qq|
<script type="text/javascript">
function gochoose(cname,cdom,cdesc) {
    var openerForm = "$env{'form.form'}";
    courseCount = 0;
    var courses = '';
|;
    if ($multiple) {
        $output .= <<"ENDSCRIPT";
    courseCount = 0;
    var courses = '';
    if (typeof(document.courselist.course_id.length) == 'undefined') {
        // only 1 course checkbox was created
        if (document.courselist.course_id.checked) {
            courses = courses + document.courselist.course_id.value + "&&";
            courseCount ++;
        }
    } else {
        for (var j=0; j<document.courselist.course_id.length; j++) {
            if (document.courselist.course_id[j].checked) {
                courses = courses + document.courselist.course_id[j].value + "&&";
                courseCount ++;
            }
        }
    }
    opener.document.$env{'form.form'}.$elements{$type}{'total'}.value = courseCount;
    if (typeof(opener.document.$env{'form.form'}.$elements{$type}{'name'}.length) ==
        'undefined') {
        if (opener.document.$env{'form.form'}.$elements{$type}{'name'}.value == 'specific') {
            opener.document.$env{'form.form'}.$elements{$type}{'name'}.checked = true;
        } else {
            opener.document.$env{'form.form'}.$elements{$type}{'name'}.checked = false;
        }
    } else {
        for (var j=0; j<opener.document.$env{'form.form'}.$elements{$type}{'name'}.length; j++) {
            if (opener.document.$env{'form.form'}.$elements{$type}{'name'}\[j].value == 'specific') {
                opener.document.$env{'form.form'}.$elements{$type}{'name'}\[j].checked = true;
            } else {
                opener.document.$env{'form.form'}.$elements{$type}{'name'}\[j].checked = false;
            }
        }
    }
    if (courseCount > 0) {
        courses = courses.substr(0,courses.length-2);
        opener.document.$env{'form.form'}.$elements{$type}{'list'}.value = courses;
    }
ENDSCRIPT
    } else {
        my $name_code;
        if ($env{'form.cnameelement'} ne '') {
            $name_code = 'opener.document.'.$env{'form.form'}.'.'.
                          $env{'form.cnameelement'}.'.value=cdesc;';
        }
        $output .= qq|
        $name_code
        opener.document.$env{'form.form'}.$env{'form.cnumelement'}.value=cname;
        var slct=opener.document.$env{'form.form'}.$env{'form.cdomelement'};
        if (slct.options == undefined) {
            opener.document.$env{'form.form'}.$env{'form.cdomelement'}.value=cdom;
        }
        else {
            var i;
            for (i=0;i<slct.length;i++) {
                if (slct.options[i].value==cdom) { slct.selectedIndex=i; }
            }
        }
|;
    }
    $output .= qq|
    if (openerForm == 'portform') {
        document.courselist.cnum.value = cname;
        document.courselist.cdom.value = cdom;
    }
    $autosubmit
    $lastaction
}
</script>
|;
    return $output;
}

1;
__END__

=pod

=head1 NAME

Apache::lonpickcourse - Search for course(s) based on user-specified criteria.   

=head1 SYNOPSIS

Invoked by other LON-CAPA modules, when course(s) need to be selected by the user. 

=head1 OVERVIEW

Two screens are typically displayed to the user.  The first is a set of criteria which are used to constrain the search for courses.

=head2 Search Criteria (Screen One)

=head3 Criteria:

=over 4

=item *
Course Activity - how recently was course last visited by anyone.

=item *
Course Domain - the domain of the course

=item *
Course Type - Course or Non-standard Course

=item *
Course Institutional Code - the institutional identifier assigned to the course

=item * 
Course Owner's Username - the username of the owner of the course (assigned by the Domain Coordinator and/or when the course was created).

=item *
Course Owner's Domain - the domain of the owner of the course

=item * 
Course Description - text which appears in the Course Description (or title), as set in the Course Parameters.

=item *
Course ID - the internal course number (course ID part after initial 'domain_') used by LON-CAPA (this criterion is only displayed to Domain Coordinators selecting a course in the same domain as their DC role).

=back

The criteria setting screen is not displayed if course picking is done by a user who does not have advanced privileges (as defined by $env{'user.adv'}).

=head2 Course Display (Screen Two)

A list of courses matching the search criteria is displayed.  If the user is not an advanced user, screen one will have been skipped and the courses displayed will be all courses in which the user has currently active roles. The information displayed for each course is:

=over 4

=item *
Course description

=item *
Domain description of course domain

=item *
Course institutional code

=item * 
Course owner (username:domain)   
 
=back

Depending on context, the display may include a single select box for each course, allowing selection of only a single course, or may include checkboxes allowing selection of more than one course.

Following selection, and/or submission, the course description, number and domain are transferred to the browser window from which the course picker window was opened.  In most cases, the child window containing the course picker screens will be closed.  However, in some cases closure will be delayed until a third screen has been displayed (e.g., setting of course-based conditional access controls for portfolio files).  In this case the page is generated via /adm/portfolio and the page features select boxes to allow the user to select roles, access types, sections and groups.

=head1 SUBROUTINES

=over 4

=item *
X<create_user_javascript()>
B<create_user_javascript($type)>:

Input: 1 - $type  - the course type - Course or Non-standard Course

Output: 1 - $output - javascript wrapped in E<lt>scriptE<gt>E<lt>/scriptE<gt> tags 

Side Effects: None 

javascript code for reporting selected sections (as a string of comma separated sections) and groups in the selected course (as a comma separated list) then calling setSect() javscript function in the opener window (to populate section select box) then closing current window.


=item *
X<display_matched_courses()>
B<display_matched_courses($r,$type,$multiple,$action,%courses)>:

Input: 5 - request object, course type, multiple (0 or 1), form action; hash of courses.

Output: 0

Side Effects: prints select buttons (multiple = 0) or checkboxes (multiple = 1) and hidden form elements for selection of one or more courses which met search criteria.

=item *
X<multiples_tag()>
B<multiples_tag()>:


Input: 0

Output: 2 - $jscript - javascript for check all/uncheck all checkboxes; $multelement - hidden form element with multiple set to 1.

Side Effects: None

=item *
X<build_filters()>
B<build_filters($filterlist,$type,$roleelement,$multelement,$filter,$action)>:


Input: 6 - anonymous array of search criteria; course type; $roleelement ; $multelement ; anonymous hash of criteria and their values; form action.

Output: 1 - $output - HTML for display of search criteria, and hidden form elements. 

Side Effects: None

=item *
X<search_courses()>
B<search_courses($r,$type,$onlyown,$filter)>:


Input: 5 -  request object, course type, search scope: only courses in which user has active role (1), or any course (0); anonymous hash of criteria and their values.

Output: 1 - %courses - hash of courses satisfying search criteria, keys = course IDs, values are corresponding colon-separated escaped description, institutional code, owner and type.

Side Effects: None


=item *
X<course_chooser()>
B<course_chooser($multiple,$cdom,$cnum,$cleandesc)>:

Input: 4 - single (0) or multiple (1) courses; course domain, course number; course description. 

Output: 1 - HTML for either checkbox (multiple=1) or select button (multiple=0) for user yo indicate course selection.

Side Effects: None


=item *
X<gochoose_javascript()>
B<gochoose_javascript($type,$multiple,$autosubmit,$lastaction)>:

Input: 4 - course type; single (0) or multiple courses (1); in context of DC selecting a CC role in a course: javascript code from Apache::lonroles::processpick(); final action to take after user chooses course(s):  either close window, or submit form for display of next page etc.

Output: 1  $output - javascript wrapped in E<lt>scriptE<gt>E<lt>/scriptE<gt> tags

Side Effects: None

javascript functions used when user selects a course(s). Different behavior depending on context:

=back

=over 8

=item

(a) Domain Coordinator using MAIL to select recipients of broadcast e-mail - && separated list of selected courses written to hidden form element in opener window. Child window closes.

=item

(b) Domain Coordinator choosing a course for adoption of a CC role from roles screen - write course identifying information to hidden form elements in opener window and automatically submit role selection form in opener window. Child window closes.

=item

(c) Domain Coordinator creating a course, and selecting a course to clone - course number and domain written to visible form elements in opener window. Child window closes.

=item

(d) User selecting a course for course-based conditional access control for a portfolio file - form is submitted, and new page is displayed for selection of roles, access types, sections and groups to be used in conditional ACL. New page is generated by /adm/portfolio. 

=item

(e) Domain Coordinator assigning a role to a user - form is submitted, and new page does an onload call to a javascript function to (a) write lists of sections and groups to hidden elements in opener window, (b) call function in opener window to dynamically populate select box showing current sections.

=item

(f) Author modifying a rights entry in a .rights file - selected course number and domain are witten to visible form elements in opener window.  Child window closes. 

=item

(g) Scantron Operator uploading a scantron file to a course - course number is written to visible form element in opener window. Child window closes.

=back
     
=cut
