#
# The LearningOnline Network
# Printout
#
# $Id: lonprintout.pm,v 1.506 2007/07/25 23:20:38 albertel Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
# http://www.lon-capa.org/
#
#
package Apache::lonprintout;

use strict;
use Apache::Constants qw(:common :http);
use Apache::lonxml;
use Apache::lonnet;
use Apache::loncommon;
use Apache::inputtags;
use Apache::grades;
use Apache::edit;
use Apache::File();
use Apache::lonnavmaps;
use LONCAPA::map();
use POSIX qw(strftime);
use Apache::lonlocal;
use Carp;
use LONCAPA;

my %perm;
my %parmhash;
my $resources_printed;

# Fetch the contents of a resource, uninterpreted.
# This is used here to fetch a latex file to be included
# verbatim into the printout<
# NOTE: Ask Guy if there is a lonnet function similar to this?
#
# Parameters:
#   URL of the file
#
sub fetch_raw_resource {
    my ($url) = @_;

    my $filename  = &Apache::lonnet::filelocation("", $url);
    my $contents  = &Apache::lonnet::getfile($filename);

    if ($contents == -1) {
	return "File open failed for $filename";      # This will bomb the print.
    }
    return $contents;

    
}

#
#   printf_style_subst  item format_string repl
#  
# Does printf style substitution for a format string that
# can have %[n]item in it.. wherever, %[n]item occurs,
# rep is substituted in format_string.  Note that
# [n] is an optional integer length.  If provided,
# repl is truncated to at most [n] characters prior to 
# substitution.
#
sub printf_style_subst {
    my ($item, $format_string, $repl) = @_;
    my $result = "";
    while ($format_string =~ /(%)(\d*)\Q$item\E/g ) {
	my $fmt = $1;
	my $size = $2;
	my $subst = $repl;
	if ($size ne "") {
	    $subst = substr($subst, 0, $size);
	    
	    #  Here's a nice edge case.. supose the end of the
	    #  substring is a \.  In that case may have  just
	    #  chopped off a TeX escape... in that case, we append
	    #   " " for the trailing character, and let the field 
	    #  spill over a bit (sigh).
	    #  We don't just chop off the last character in order to deal
	    #  with one last pathology, and that would be if substr had
	    #  trimmed us to e.g. \\\  


	    if ($subst =~ /\\$/) {
		$subst .= " ";
	    }
	}
	my $item_pos = pos($format_string);
	$result .= substr($format_string, 0, $item_pos - length($size) -2) . $subst;
        $format_string = substr($format_string, pos($format_string));
    }

    # Put the residual format string into the result:

    $result .= $format_string;

    return $result;
}


# Format a header according to a format.  
# 

# Substitutions:
#     %a    - Assignment name.
#     %c    - Course name.
#     %n    - Student name.
#
sub format_page_header {
    my ($width, $format, $assignment, $course, $student) = @_;
    
    $width = &recalcto_mm($width); # Get width in mm.
    #  Default format?

    if ($format eq '') {
	# For the default format, we may need to truncate
	# elements..  To do this we need to get the page width.
	# we assume that each character is about 2mm in width.
	# (correct for the header text size??).  We ignore
	# any formatting (e.g. boldfacing in this).
	# 
	# - Allow the student/course to be one line.
	#   but only truncate the course.
	# - Allow the assignment to be 2 lines (wrapped).
	#
	my $chars_per_line = $width/2; # Character/textline.


	my $firstline = "$student $course";
	if (length($firstline) > $chars_per_line) {
	    my $lastchar = $chars_per_line - length($student) - 1;
	    if ($lastchar > 0) {
		$course = substr($course, 0, $lastchar);
	    } else {		# Nothing left of course:
		$course = '';
	    }
	}
	if (length($assignment) > $chars_per_line) {
	    $assignment = substr($assignment, 0, $chars_per_line);
	}
	
	$format =  "\\textbf{$student} $course \\hfill \\thepage \\\\ \\textit{$assignment}";
	
    } else {
	# An open question is how to handle long user formatted page headers...
	# A possible future is to support e.g. %na so that the user can control
	# the truncation of the elements that can appear in the header.
	#
	$format =  &printf_style_subst("a", $format, $assignment);
	$format =  &printf_style_subst("c", $format, $course);
	$format =  &printf_style_subst("n", $format, $student);

	# If the user put %'s in the format string, they  must be escaped
	# to \% else LaTeX will think they are comments and terminate
	# the line.. which is bad!!!


    }
    

    return $format;
    
}

#
#   Convert a numeric code to letters
#
sub num_to_letters {
    my ($num) = @_;
    my @nums= split('',$num);
    my @num_to_let=('A'..'Z');
    my $word;
    foreach my $digit (@nums) { $word.=$num_to_let[$digit]; }
    return $word;
}
#   Convert a letter code to numeric.
#
sub letters_to_num {
    my ($letters) = @_;
    my @letters = split('', uc($letters));
   my %substitution;
    my $digit = 0;
    foreach my $letter ('A'..'J') {
	$substitution{$letter} = $digit;
	$digit++;
    }
    #  The substitution is done as below to preserve leading
    #  zeroes which are needed to keep the code size exact
    #
    my $result ="";
    foreach my $letter (@letters) {
	$result.=$substitution{$letter};
    }
    return $result;
}

#  Determine if a code is a valid numeric code.  Valid
#  numeric codes must be comprised entirely of digits and
#  have a correct number of digits.
#
#  Parameters:
#     value      - proposed code value.
#     num_digits - Number of digits required.
#
sub is_valid_numeric_code {
    my ($value, $num_digits) = @_;
    #   Remove leading/trailing whitespace;
    $value =~ s/^\s*//g;
    $value =~ s/\s*$//g;
    
    #  All digits?
    if ($value !~ /^[0-9]+$/) {
	return "Numeric code $value has invalid characters - must only be digits";
    }
    if (length($value) != $num_digits) {
	return "Numeric code $value incorrect number of digits (correct = $num_digits)";
    }
    return undef;
}
#   Determines if a code is a valid alhpa code.  Alpha codes
#   are ciphers that map  [A-J,a-j] -> 0..9 0..9.
#   They also have a correct digit count.
# Parameters:
#     value          - Proposed code value.
#     num_letters    - correct number of letters.
# Note:
#    leading and trailing whitespace are ignored.
#
sub is_valid_alpha_code {
    my ($value, $num_letters) = @_;
    
     # strip leading and trailing spaces.

    $value =~ s/^\s*//g;
    $value =~ s/\s*$//g;

    #  All alphas in the right range?
    if ($value !~ /^[A-J,a-j]+$/) {
	return "Invalid letter code $value must only contain A-J";
    }
    if (length($value) != $num_letters) {
	return "Letter code $value has incorrect number of letters (correct = $num_letters)";
    }
    return undef;
}

#   Determine if a code entered by the user in a helper is valid.
#   valid depends on the code type and the type of code selected.
#   The type of code selected can either be numeric or 
#   Alphabetic.  If alphabetic, the code, in fact is a simple
#   substitution cipher for the actual numeric code: 0->A, 1->B ...
#   We'll be nice and be case insensitive for alpha codes.
# Parameters:
#    code_value    - the value of the code the user typed in.
#    code_option   - The code type selected from the set in the scantron format
#                    table.
# Returns:
#    undef         - The code is valid.
#    other         - An error message indicating what's wrong.
#
sub is_code_valid {
    my ($code_value, $code_option) = @_;
    my ($code_type, $code_length) = ('letter', 6);	# defaults.
    open(FG, $Apache::lonnet::perlvar{'lonTabDir'}.'/scantronformat.tab');
    foreach my $line (<FG>) {
	my ($name, $type, $length) = (split(/:/, $line))[0,2,4];
	if($name eq $code_option) {
	    $code_length = $length;
	    if($type eq 'number') {
		$code_type = 'number';
	    }
	}
    }
    my $valid;
    if ($code_type eq 'number') {
	return &is_valid_numeric_code($code_value, $code_length);
    } else {
	return &is_valid_alpha_code($code_value, $code_length);
    }

}

#   Compare two students by name.  The students are in the form
#   returned by the helper:
#      user:domain:section:last,   first:status
#   This is a helper function for the perl sort built-in  therefore:
# Implicit Inputs:
#    $a     - The first element to compare (global)
#    $b     - The second element to compare (global)
# Returns:
#   -1   - $a < $b
#    0   - $a == $b
#   +1   - $a > $b
#   Note that the initial comparison is done on the last names with the
#   first names only used to break the tie.
#
#
sub compare_names {
    #  First split the names up into the primary fields.

    my ($u1, $d1, $s1, $n1, $stat1) = split(/:/, $a);
    my ($u2, $d2, $s2, $n2, $stat2) = split(/:/, $b);

    # Now split the last name and first name of each n:
    #

    my ($l1,$f1) = split(/,/, $n1);
    my ($l2,$f2) = split(/,/, $n2);

    # We don't bother to remove the leading/trailing whitespace from the
    # firstname, unless the last names compare identical.

    if($l1 lt $l2) {
	return -1;
    }
    if($l1 gt $l2) {
	return  1;
    }

    # Break the tie on the first name, but there are leading (possibly trailing
    # whitespaces to get rid of first 
    #
    $f1 =~ s/^\s+//;		# Remove leading...
    $f1 =~ s/\s+$//;		# Trailing spaces from first 1...
    
    $f2 =~ s/^\s+//;
    $f2 =~ s/\s+$//;		# And the same for first 2...

    if($f1 lt $f2) {
	return -1;
    }
    if($f1 gt $f2) {
	return 1;
    }
    
    #  Must be the same name.

    return 0;
}

sub latex_header_footer_remove {
    my $text = shift;
    $text =~ s/\\end{document}//;
    $text =~ s/\\documentclass([^&]*)\\begin{document}//;
    return $text;
}
#
#  If necessary, encapsulate text inside 
#  a minipage env.
#  necessity is determined by the problem_split param.
#
sub encapsulate_minipage {
    my ($text) = @_;
    if (!($env{'form.problem.split'} =~ /yes/i)) {
	$text = '\begin{minipage}{\textwidth}'.$text.'\end{minipage}';
    }
    return $text;
}
#
#  The NUMBER_TO_PRINT and SPLIT_PDFS
#  variables interact, this sub looks at these two parameters
#  and comes up with a final value for NUMBER_TO_PRINT which can be:
#     all     - if SPLIT_PDFS eq 'all'.
#     1       - if SPLIT_PDFS eq 'oneper'
#     section - if SPLIT_PDFS eq 'sections'
#     <unchanged> - if SPLIT_PDFS eq 'usenumber'
#
sub adjust_number_to_print {
    my $helper = shift;

    my $split_pdf = $helper->{'VARS'}->{'SPLIT_PDFS'};
    
    if ($split_pdf eq 'all') {
	$helper->{'VARS'}->{'NUMBER_TO_PRINT'} = 'all';
    } elsif ($split_pdf eq 'oneper') {
	$helper->{'VARS'}->{'NUMBER_TO_PRINT'} = 1;
    } elsif ($split_pdf eq 'sections') {
	$helper->{'VARS'}->{'NUMBER_TO_PRINT'} = 'section';
    } elsif ($split_pdf eq 'usenumber') {
	#  Unmodified.
    } else {
	# Error!!!!

	croak "bad SPLIT_PDFS: $split_pdf in lonprintout::adjust_number_to_print";
    }
}

sub character_chart {
    my $result = shift;	
    $result =~ s/&\#0?0?(7|9);//g;
    $result =~ s/&\#0?(10|13);//g;
    $result =~ s/&\#0?32;/ /g;
    $result =~ s/&\#0?33;/!/g;
    $result =~ s/&(\#0?34|quot);/\"/g;
    $result =~ s/&\#0?35;/\\\#/g;
    $result =~ s/&\#0?36;/\\\$/g;
    $result =~ s/&\#0?37;/\\%/g; 
    $result =~ s/&(\#0?38|amp);/\\&/g; 
    $result =~ s/&\#(0?39|146);/\'/g;
    $result =~ s/&\#0?40;/(/g;
    $result =~ s/&\#0?41;/)/g;
    $result =~ s/&\#0?42;/\*/g;
    $result =~ s/&\#0?43;/\+/g;
    $result =~ s/&\#(0?44|130);/,/g;
    $result =~ s/&\#0?45;/-/g;
    $result =~ s/&\#0?46;/\./g;
    $result =~ s/&\#0?47;/\//g;
    $result =~ s/&\#0?48;/0/g;
    $result =~ s/&\#0?49;/1/g;
    $result =~ s/&\#0?50;/2/g;
    $result =~ s/&\#0?51;/3/g;
    $result =~ s/&\#0?52;/4/g;
    $result =~ s/&\#0?53;/5/g;
    $result =~ s/&\#0?54;/6/g;
    $result =~ s/&\#0?55;/7/g;
    $result =~ s/&\#0?56;/8/g;
    $result =~ s/&\#0?57;/9/g;
    $result =~ s/&\#0?58;/:/g;
    $result =~ s/&\#0?59;/;/g;
    $result =~ s/&(\#0?60|lt|\#139);/\$<\$/g;
    $result =~ s/&\#0?61;/\\ensuremath\{=\}/g;
    $result =~ s/&(\#0?62|gt|\#155);/\\ensuremath\{>\}/g;
    $result =~ s/&\#0?63;/\?/g;
    $result =~ s/&\#0?65;/A/g;
    $result =~ s/&\#0?66;/B/g;
    $result =~ s/&\#0?67;/C/g;
    $result =~ s/&\#0?68;/D/g;
    $result =~ s/&\#0?69;/E/g;
    $result =~ s/&\#0?70;/F/g;
    $result =~ s/&\#0?71;/G/g;
    $result =~ s/&\#0?72;/H/g;
    $result =~ s/&\#0?73;/I/g;
    $result =~ s/&\#0?74;/J/g;
    $result =~ s/&\#0?75;/K/g;
    $result =~ s/&\#0?76;/L/g;
    $result =~ s/&\#0?77;/M/g;
    $result =~ s/&\#0?78;/N/g;
    $result =~ s/&\#0?79;/O/g;
    $result =~ s/&\#0?80;/P/g;
    $result =~ s/&\#0?81;/Q/g;
    $result =~ s/&\#0?82;/R/g;
    $result =~ s/&\#0?83;/S/g;
    $result =~ s/&\#0?84;/T/g;
    $result =~ s/&\#0?85;/U/g;
    $result =~ s/&\#0?86;/V/g;
    $result =~ s/&\#0?87;/W/g;
    $result =~ s/&\#0?88;/X/g;
    $result =~ s/&\#0?89;/Y/g;
    $result =~ s/&\#0?90;/Z/g;
    $result =~ s/&\#0?91;/[/g;
    $result =~ s/&\#0?92;/\\ensuremath\{\\setminus\}/g;
    $result =~ s/&\#0?93;/]/g;
    $result =~ s/&\#(0?94|136);/\\ensuremath\{\\wedge\}/g;
    $result =~ s/&\#(0?95|138|154);/\\underline{\\makebox[2mm]{\\strut}}/g;
    $result =~ s/&\#(0?96|145);/\`/g;
    $result =~ s/&\#0?97;/a/g;
    $result =~ s/&\#0?98;/b/g;
    $result =~ s/&\#0?99;/c/g;
    $result =~ s/&\#100;/d/g;
    $result =~ s/&\#101;/e/g;
    $result =~ s/&\#102;/f/g;
    $result =~ s/&\#103;/g/g;
    $result =~ s/&\#104;/h/g;
    $result =~ s/&\#105;/i/g;
    $result =~ s/&\#106;/j/g;
    $result =~ s/&\#107;/k/g;
    $result =~ s/&\#108;/l/g;
    $result =~ s/&\#109;/m/g;
    $result =~ s/&\#110;/n/g;
    $result =~ s/&\#111;/o/g;
    $result =~ s/&\#112;/p/g;
    $result =~ s/&\#113;/q/g;
    $result =~ s/&\#114;/r/g;
    $result =~ s/&\#115;/s/g;
    $result =~ s/&\#116;/t/g;
    $result =~ s/&\#117;/u/g;
    $result =~ s/&\#118;/v/g;
    $result =~ s/&\#119;/w/g;
    $result =~ s/&\#120;/x/g;
    $result =~ s/&\#121;/y/g;
    $result =~ s/&\#122;/z/g;
    $result =~ s/&\#123;/\\{/g;
    $result =~ s/&\#124;/\|/g;
    $result =~ s/&\#125;/\\}/g;
    $result =~ s/&\#126;/\~/g;
    $result =~ s/&\#131;/\\textflorin /g;
    $result =~ s/&\#132;/\"/g;
    $result =~ s/&\#133;/\\ensuremath\{\\ldots\}/g;
    $result =~ s/&\#134;/\\ensuremath\{\\dagger\}/g;
    $result =~ s/&\#135;/\\ensuremath\{\\ddagger\}/g;
    $result =~ s/&\#137;/\\textperthousand /g;
    $result =~ s/&\#140;/{\\OE}/g;
    $result =~ s/&\#147;/\`\`/g;
    $result =~ s/&\#148;/\'\'/g;
    $result =~ s/&\#149;/\\ensuremath\{\\bullet\}/g;
    $result =~ s/&(\#150|\#8211);/--/g;
    $result =~ s/&\#151;/---/g;
    $result =~ s/&\#152;/\\ensuremath\{\\sim\}/g;
    $result =~ s/&\#153;/\\texttrademark /g;
    $result =~ s/&\#156;/\\oe/g;
    $result =~ s/&\#159;/\\\"Y/g;
    $result =~ s/&(\#160|nbsp);/~/g;
    $result =~ s/&(\#161|iexcl);/!\`/g;
    $result =~ s/&(\#162|cent);/\\textcent /g;
    $result =~ s/&(\#163|pound);/\\pounds /g; 
    $result =~ s/&(\#164|curren);/\\textcurrency /g;
    $result =~ s/&(\#165|yen);/\\textyen /g;
    $result =~ s/&(\#166|brvbar);/\\textbrokenbar /g;
    $result =~ s/&(\#167|sect);/\\textsection /g;
    $result =~ s/&(\#168|uml);/\\texthighdieresis /g;
    $result =~ s/&(\#169|copy);/\\copyright /g;
    $result =~ s/&(\#170|ordf);/\\textordfeminine /g;
    $result =~ s/&(\#172|not);/\\ensuremath\{\\neg\}/g;
    $result =~ s/&(\#173|shy);/ - /g;
    $result =~ s/&(\#174|reg);/\\textregistered /g;
    $result =~ s/&(\#175|macr);/\\ensuremath\{^{-}\}/g;
    $result =~ s/&(\#176|deg);/\\ensuremath\{^{\\circ}\}/g;
    $result =~ s/&(\#177|plusmn);/\\ensuremath\{\\pm\}/g;
    $result =~ s/&(\#178|sup2);/\\ensuremath\{^2\}/g;
    $result =~ s/&(\#179|sup3);/\\ensuremath\{^3\}/g;
    $result =~ s/&(\#180|acute);/\\textacute /g;
    $result =~ s/&(\#181|micro);/\\ensuremath\{\\mu\}/g;
    $result =~ s/&(\#182|para);/\\P/g;
    $result =~ s/&(\#183|middot);/\\ensuremath\{\\cdot\}/g;
    $result =~ s/&(\#184|cedil);/\\c{\\strut}/g;
    $result =~ s/&(\#185|sup1);/\\ensuremath\{^1\}/g;
    $result =~ s/&(\#186|ordm);/\\textordmasculine /g;
    $result =~ s/&(\#188|frac14);/\\textonequarter /g;
    $result =~ s/&(\#189|frac12);/\\textonehalf /g;
    $result =~ s/&(\#190|frac34);/\\textthreequarters /g;
    $result =~ s/&(\#191|iquest);/?\`/g;   
    $result =~ s/&(\#192|Agrave);/\\\`{A}/g;  
    $result =~ s/&(\#193|Aacute);/\\\'{A}/g; 
    $result =~ s/&(\#194|Acirc);/\\^{A}/g;
    $result =~ s/&(\#195|Atilde);/\\~{A}/g;
    $result =~ s/&(\#196|Auml);/\\\"{A}/g; 
    $result =~ s/&(\#197|Aring);/{\\AA}/g;
    $result =~ s/&(\#198|AElig);/{\\AE}/g;
    $result =~ s/&(\#199|Ccedil);/\\c{c}/g;
    $result =~ s/&(\#200|Egrave);/\\\`{E}/g;  
    $result =~ s/&(\#201|Eacute);/\\\'{E}/g;    
    $result =~ s/&(\#202|Ecirc);/\\^{E}/g;
    $result =~ s/&(\#203|Euml);/\\\"{E}/g;
    $result =~ s/&(\#204|Igrave);/\\\`{I}/g;
    $result =~ s/&(\#205|Iacute);/\\\'{I}/g;    
    $result =~ s/&(\#206|Icirc);/\\^{I}/g;
    $result =~ s/&(\#207|Iuml);/\\\"{I}/g;    
    $result =~ s/&(\#209|Ntilde);/\\~{N}/g;
    $result =~ s/&(\#210|Ograve);/\\\`{O}/g;
    $result =~ s/&(\#211|Oacute);/\\\'{O}/g;
    $result =~ s/&(\#212|Ocirc);/\\^{O}/g;
    $result =~ s/&(\#213|Otilde);/\\~{O}/g;
    $result =~ s/&(\#214|Ouml);/\\\"{O}/g;    
    $result =~ s/&(\#215|times);/\\ensuremath\{\\times\}/g;
    $result =~ s/&(\#216|Oslash);/{\\O}/g;
    $result =~ s/&(\#217|Ugrave);/\\\`{U}/g;    
    $result =~ s/&(\#218|Uacute);/\\\'{U}/g;
    $result =~ s/&(\#219|Ucirc);/\\^{U}/g;
    $result =~ s/&(\#220|Uuml);/\\\"{U}/g;
    $result =~ s/&(\#221|Yacute);/\\\'{Y}/g;
    $result =~ s/&(\#223|szlig);/{\\ss}/g;
    $result =~ s/&(\#224|agrave);/\\\`{a}/g;
    $result =~ s/&(\#225|aacute);/\\\'{a}/g;
    $result =~ s/&(\#226|acirc);/\\^{a}/g;
    $result =~ s/&(\#227|atilde);/\\~{a}/g;
    $result =~ s/&(\#228|auml);/\\\"{a}/g;
    $result =~ s/&(\#229|aring);/{\\aa}/g;
    $result =~ s/&(\#230|aelig);/{\\ae}/g;
    $result =~ s/&(\#231|ccedil);/\\c{c}/g;
    $result =~ s/&(\#232|egrave);/\\\`{e}/g;
    $result =~ s/&(\#233|eacute);/\\\'{e}/g;
    $result =~ s/&(\#234|ecirc);/\\^{e}/g;
    $result =~ s/&(\#235|euml);/\\\"{e}/g;
    $result =~ s/&(\#236|igrave);/\\\`{i}/g;
    $result =~ s/&(\#237|iacute);/\\\'{i}/g;
    $result =~ s/&(\#238|icirc);/\\^{i}/g;
    $result =~ s/&(\#239|iuml);/\\\"{i}/g;
    $result =~ s/&(\#240|eth);/\\ensuremath\{\\partial\}/g;
    $result =~ s/&(\#241|ntilde);/\\~{n}/g;
    $result =~ s/&(\#242|ograve);/\\\`{o}/g;
    $result =~ s/&(\#243|oacute);/\\\'{o}/g;
    $result =~ s/&(\#244|ocirc);/\\^{o}/g;
    $result =~ s/&(\#245|otilde);/\\~{o}/g;
    $result =~ s/&(\#246|ouml);/\\\"{o}/g;
    $result =~ s/&(\#247|divide);/\\ensuremath\{\\div\}/g;
    $result =~ s/&(\#248|oslash);/{\\o}/g;
    $result =~ s/&(\#249|ugrave);/\\\`{u}/g; 
    $result =~ s/&(\#250|uacute);/\\\'{u}/g;
    $result =~ s/&(\#251|ucirc);/\\^{u}/g;
    $result =~ s/&(\#252|uuml);/\\\"{u}/g;
    $result =~ s/&(\#253|yacute);/\\\'{y}/g;
    $result =~ s/&(\#255|yuml);/\\\"{y}/g;
    $result =~ s/&\#295;/\\ensuremath\{\\hbar\}/g;
    $result =~ s/&\#952;/\\ensuremath\{\\theta\}/g;
#Greek Alphabet
    $result =~ s/&(alpha|\#945);/\\ensuremath\{\\alpha\}/g;
    $result =~ s/&(beta|\#946);/\\ensuremath\{\\beta\}/g;
    $result =~ s/&(gamma|\#947);/\\ensuremath\{\\gamma\}/g;
    $result =~ s/&(delta|\#948);/\\ensuremath\{\\delta\}/g;
    $result =~ s/&(epsilon|\#949);/\\ensuremath\{\\epsilon\}/g;
    $result =~ s/&(zeta|\#950);/\\ensuremath\{\\zeta\}/g;
    $result =~ s/&(eta|\#951);/\\ensuremath\{\\eta\}/g;
    $result =~ s/&(theta|\#952);/\\ensuremath\{\\theta\}/g;
    $result =~ s/&(iota|\#953);/\\ensuremath\{\\iota\}/g;
    $result =~ s/&(kappa|\#954);/\\ensuremath\{\\kappa\}/g;
    $result =~ s/&(lambda|\#955);/\\ensuremath\{\\lambda\}/g;
    $result =~ s/&(mu|\#956);/\\ensuremath\{\\mu\}/g;
    $result =~ s/&(nu|\#957);/\\ensuremath\{\\nu\}/g;
    $result =~ s/&(xi|\#958);/\\ensuremath\{\\xi\}/g;
    $result =~ s/&(omicron|\#959);/o/g;
    $result =~ s/&(pi|\#960);/\\ensuremath\{\\pi\}/g;
    $result =~ s/&(rho|\#961);/\\ensuremath\{\\rho\}/g;
    $result =~ s/&(sigma|\#963);/\\ensuremath\{\\sigma\}/g;
    $result =~ s/&(tau|\#964);/\\ensuremath\{\\tau\}/g;
    $result =~ s/&(upsilon|\#965);/\\ensuremath\{\\upsilon\}/g;
    $result =~ s/&(phi|\#966);/\\ensuremath\{\\phi\}/g;
    $result =~ s/&(chi|\#967);/\\ensuremath\{\\chi\}/g;
    $result =~ s/&(psi|\#968);/\\ensuremath\{\\psi\}/g;
    $result =~ s/&(omega|\#969);/\\ensuremath\{\\omega\}/g;
    $result =~ s/&(thetasym|\#977);/\\ensuremath\{\\vartheta\}/g;
    $result =~ s/&(piv|\#982);/\\ensuremath\{\\varpi\}/g;
    $result =~ s/&(Alpha|\#913);/A/g;
    $result =~ s/&(Beta|\#914);/B/g;
    $result =~ s/&(Gamma|\#915);/\\ensuremath\{\\Gamma\}/g;
    $result =~ s/&(Delta|\#916);/\\ensuremath\{\\Delta\}/g;
    $result =~ s/&(Epsilon|\#917);/E/g;
    $result =~ s/&(Zeta|\#918);/Z/g;
    $result =~ s/&(Eta|\#919);/H/g;
    $result =~ s/&(Theta|\#920);/\\ensuremath\{\\Theta\}/g;
    $result =~ s/&(Iota|\#921);/I/g;
    $result =~ s/&(Kappa|\#922);/K/g;
    $result =~ s/&(Lambda|\#923);/\\ensuremath\{\\Lambda\}/g;
    $result =~ s/&(Mu|\#924);/M/g;
    $result =~ s/&(Nu|\#925);/N/g;
    $result =~ s/&(Xi|\#926);/\\ensuremath\{\\Xi\}/g;
    $result =~ s/&(Omicron|\#927);/O/g;
    $result =~ s/&(Pi|\#928);/\\ensuremath\{\\Pi\}/g;
    $result =~ s/&(Rho|\#929);/P/g;
    $result =~ s/&(Sigma|\#931);/\\ensuremath\{\\Sigma\}/g;
    $result =~ s/&(Tau|\#932);/T/g;
    $result =~ s/&(Upsilon|\#933);/\\ensuremath\{\\Upsilon\}/g;
    $result =~ s/&(Phi|\#934);/\\ensuremath\{\\Phi\}/g;
    $result =~ s/&(Chi|\#935);/X/g;
    $result =~ s/&(Psi|\#936);/\\ensuremath\{\\Psi\}/g;
    $result =~ s/&(Omega|\#937);/\\ensuremath\{\\Omega\}/g;
#Arrows (extended HTML 4.01)
    $result =~ s/&(larr|\#8592);/\\ensuremath\{\\leftarrow\}/g;
    $result =~ s/&(uarr|\#8593);/\\ensuremath\{\\uparrow\}/g;
    $result =~ s/&(rarr|\#8594);/\\ensuremath\{\\rightarrow\}/g;
    $result =~ s/&(darr|\#8595);/\\ensuremath\{\\downarrow\}/g;
    $result =~ s/&(harr|\#8596);/\\ensuremath\{\\leftrightarrow\}/g;
    $result =~ s/&(lArr|\#8656);/\\ensuremath\{\\Leftarrow\}/g;
    $result =~ s/&(uArr|\#8657);/\\ensuremath\{\\Uparrow\}/g;
    $result =~ s/&(rArr|\#8658);/\\ensuremath\{\\Rightarrow\}/g;
    $result =~ s/&(dArr|\#8659);/\\ensuremath\{\\Downarrow\}/g;
    $result =~ s/&(hArr|\#8660);/\\ensuremath\{\\Leftrightarrow\}/g;
#Mathematical Operators (extended HTML 4.01)
    $result =~ s/&(forall|\#8704);/\\ensuremath\{\\forall\}/g;
    $result =~ s/&(part|\#8706);/\\ensuremath\{\\partial\}/g;
    $result =~ s/&(exist|\#8707);/\\ensuremath\{\\exists\}/g;
    $result =~ s/&(empty|\#8709);/\\ensuremath\{\\emptyset\}/g;
    $result =~ s/&(nabla|\#8711);/\\ensuremath\{\\nabla\}/g;
    $result =~ s/&(isin|\#8712);/\\ensuremath\{\\in\}/g;
    $result =~ s/&(notin|\#8713);/\\ensuremath\{\\notin\}/g;
    $result =~ s/&(ni|\#8715);/\\ensuremath\{\\ni\}/g;
    $result =~ s/&(prod|\#8719);/\\ensuremath\{\\prod\}/g;
    $result =~ s/&(sum|\#8721);/\\ensuremath\{\\sum\}/g;
    $result =~ s/&(minus|\#8722);/\\ensuremath\{-\}/g;
    $result =~ s/–/\\ensuremath\{-\}/g;
    $result =~ s/&(lowast|\#8727);/\\ensuremath\{*\}/g;
    $result =~ s/&(radic|\#8730);/\\ensuremath\{\\surd\}/g;
    $result =~ s/&(prop|\#8733);/\\ensuremath\{\\propto\}/g;
    $result =~ s/&(infin|\#8734);/\\ensuremath\{\\infty\}/g;
    $result =~ s/&(ang|\#8736);/\\ensuremath\{\\angle\}/g;
    $result =~ s/&(and|\#8743);/\\ensuremath\{\\wedge\}/g;
    $result =~ s/&(or|\#8744);/\\ensuremath\{\\vee\}/g;
    $result =~ s/&(cap|\#8745);/\\ensuremath\{\\cap\}/g;
    $result =~ s/&(cup|\#8746);/\\ensuremath\{\\cup\}/g;
    $result =~ s/&(int|\#8747);/\\ensuremath\{\\int\}/g;
    $result =~ s/&(sim|\#8764);/\\ensuremath\{\\sim\}/g;
    $result =~ s/&(cong|\#8773);/\\ensuremath\{\\cong\}/g;
    $result =~ s/&(asymp|\#8776);/\\ensuremath\{\\approx\}/g;
    $result =~ s/&(ne|\#8800);/\\ensuremath\{\\not=\}/g;
    $result =~ s/&(equiv|\#8801);/\\ensuremath\{\\equiv\}/g;
    $result =~ s/&(le|\#8804);/\\ensuremath\{\\leq\}/g;
    $result =~ s/&(ge|\#8805);/\\ensuremath\{\\geq\}/g;
    $result =~ s/&(sub|\#8834);/\\ensuremath\{\\subset\}/g;
    $result =~ s/&(sup|\#8835);/\\ensuremath\{\\supset\}/g;
    $result =~ s/&(nsub|\#8836);/\\ensuremath\{\\not\\subset\}/g;
    $result =~ s/&(sube|\#8838);/\\ensuremath\{\\subseteq\}/g;
    $result =~ s/&(supe|\#8839);/\\ensuremath\{\\supseteq\}/g;
    $result =~ s/&(oplus|\#8853);/\\ensuremath\{\\oplus\}/g;
    $result =~ s/&(otimes|\#8855);/\\ensuremath\{\\otimes\}/g;
    $result =~ s/&(perp|\#8869);/\\ensuremath\{\\perp\}/g;
    $result =~ s/&(sdot|\#8901);/\\ensuremath\{\\cdot\}/g;
#Geometric Shapes (extended HTML 4.01)
    $result =~ s/&(loz|\#9674);/\\ensuremath\{\\Diamond\}/g;
#Miscellaneous Symbols (extended HTML 4.01)
    $result =~ s/&(spades|\#9824);/\\ensuremath\{\\spadesuit\}/g;
    $result =~ s/&(clubs|\#9827);/\\ensuremath\{\\clubsuit\}/g;
    $result =~ s/&(hearts|\#9829);/\\ensuremath\{\\heartsuit\}/g;
    $result =~ s/&(diams|\#9830);/\\ensuremath\{\\diamondsuit\}/g;
#   Chemically useful 'things' contributed by Hon Kie (bug 4652).
    $result =~ s/&\#8636;/\\ensuremath\{\\leftharpoonup\}/g;
    $result =~ s/&\#8637;/\\ensuremath\{\\leftharpoondown\}/g;
    $result =~ s/&\#8640;/\\ensuremath\{\\rightharpoonup\}/g;
    $result =~ s/&\#8641;/\\ensuremath\{\\rightharpoondown\}/g;
    $result =~ s/&\#8652;/\\ensuremath\{\\rightleftharpoons\}/g;
    $result =~ s/&\#8605;/\\ensuremath\{\\leadsto\}/g;
    $result =~ s/&\#8617;/\\ensuremath\{\\hookleftarrow\}/g;
    $result =~ s/&\#8618;/\\ensuremath\{\\hookrightarrow\}/g;
    $result =~ s/&\#8614;/\\ensuremath\{\\mapsto\}/g;
    $result =~ s/&\#8599;/\\ensuremath\{\\nearrow\}/g;
    $result =~ s/&\#8600;/\\ensuremath\{\\searrow\}/g;
    $result =~ s/&\#8601;/\\ensuremath\{\\swarrow\}/g;
    $result =~ s/&\#8598;/\\ensuremath\{\\nwarrow\}/g;
    return $result;
}


                  #width, height, oddsidemargin, evensidemargin, topmargin
my %page_formats=
    ('letter' => {
	 'book' => {
	     '1' => [ '7.1 in','9.8 in', '-0.57 in','-0.57 in','0.275 in'],
	     '2' => ['3.66 in','9.8 in', '-0.57 in','-0.57 in','0.275 in']
	 },
	 'album' => {
	     '1' => [ '8.8 in', '6.8 in','-0.55 in',  '-0.55 in','0.394 in'],
	     '2' => [ '4.8 in', '6.8 in','-0.5 in', '-1.0 in','3.5 in']
	 },
     },
     'legal' => {
	 'book' => {
	     '1' => ['7.1 in','13 in',,'-0.57 in','-0.57 in','-0.5 in'],
	     '2' => ['3.16 in','13 in','-0.57 in','-0.57 in','-0.5 in']
	 },
	 'album' => {
	     '1' => ['12 in','7.1 in',,'-0.57 in','-0.57 in','-0.5 in'],
             '2' => ['6.0 in','7.1 in','-1 in','-1 in','5 in']
          },
     },
     'tabloid' => {
	 'book' => {
	     '1' => ['9.8 in','16 in','-0.57 in','-0.57 in','-0.5 in'],
	     '2' => ['4.9 in','16 in','-0.57 in','-0.57 in','-0.5 in']
	 },
	 'album' => {
	     '1' => ['16 in','9.8 in','-0.57 in','-0.57 in','-0.5 in'],
	     '2' => ['16 in','4.9 in','-0.57 in','-0.57 in','-0.5 in']
          },
     },
     'executive' => {
	 'book' => {
	     '1' => ['6.8 in','9 in','-0.57 in','-0.57 in','1.2 in'],
	     '2' => ['3.1 in','9 in','-0.57 in','-0.57 in','1.2 in']
	 },
	 'album' => {
	     '1' => [],
	     '2' => []
          },
     },
     'a2' => {
	 'book' => {
	     '1' => [],
	     '2' => []
	 },
	 'album' => {
	     '1' => [],
	     '2' => []
          },
     },
     'a3' => {
	 'book' => {
	     '1' => [],
	     '2' => []
	 },
	 'album' => {
	     '1' => [],
	     '2' => []
          },
     },
     'a4' => {
	 'book' => {
	     '1' => ['17.6 cm','27.2 cm','-1.397 cm','-2.11 cm','-1.27 cm'],
	     '2' => [ '9.1 cm','27.2 cm','-1.397 cm','-2.11 cm','-1.27 cm']
	 },
	 'album' => {
	     '1' => ['21.59 cm','19.558 cm','-1.397cm','-2.11 cm','0 cm'],
	     '2' => ['9.91 cm','19.558 cm','-1.397 cm','-2.11 cm','0 cm']
	 },
     },
     'a5' => {
	 'book' => {
	     '1' => [],
	     '2' => []
	 },
	 'album' => {
	     '1' => [],
	     '2' => []
          },
     },
     'a6' => {
	 'book' => {
	     '1' => [],
	     '2' => []
	 },
	 'album' => {
	     '1' => [],
	     '2' => []
          },
     },
     );

sub page_format {
#
#Supported paper format: "Letter [8 1/2x11 in]",      "Legal [8 1/2x14 in]",
#                        "Ledger/Tabloid [11x17 in]", "Executive [7 1/2x10 in]",
#                        "A2 [420x594 mm]",           "A3 [297x420 mm]",
#                        "A4 [210x297 mm]",           "A5 [148x210 mm]",
#                        "A6 [105x148 mm]"
# 
    my ($papersize,$layout,$numberofcolumns) = @_; 
    return @{$page_formats{$papersize}->{$layout}->{$numberofcolumns}};
}


sub get_name {
    my ($uname,$udom)=@_;
    if (!defined($uname)) { $uname=$env{'user.name'}; }
    if (!defined($udom)) { $udom=$env{'user.domain'}; }
    my $plainname=&Apache::loncommon::plainname($uname,$udom);
    if ($plainname=~/^\s*$/) { $plainname=$uname.'@'.$udom; }
   $plainname=&Apache::lonxml::latex_special_symbols($plainname,'header');
    return $plainname;
}

sub get_course {
    my $courseidinfo;
    if (defined($env{'request.course.id'})) {
	$courseidinfo = &Apache::lonxml::latex_special_symbols(&unescape($env{'course.'.$env{'request.course.id'}.'.description'}),'header');
    }
    return $courseidinfo;
}

sub page_format_transformation {
    my ($papersize,$layout,$numberofcolumns,$choice,$text,$assignment,$tableofcontents,$indexlist,$selectionmade) = @_; 
    my ($textwidth,$textheight,$oddoffset,$evenoffset,$topmargin);

    if ($selectionmade eq '4') {
	if ($choice eq 'all_problems') {
	    $assignment='Problems from the Whole Course';
	} else {
	    $assignment='Resources from the Whole Course';
	}
    } else {
	$assignment=&Apache::lonxml::latex_special_symbols($assignment,'header');
    }
    ($textwidth,$textheight,$oddoffset,$evenoffset,$topmargin) = &page_format($papersize,$layout,$numberofcolumns,$topmargin);


    my $name = &get_name();
    my $courseidinfo = &get_course();
    if (defined($courseidinfo)) { $courseidinfo=' - '.$courseidinfo }
    my $header_text  = $parmhash{'print_header_format'};
    $header_text     = &format_page_header($textwidth, $header_text, $assignment,
					   $courseidinfo, $name);
    my $topmargintoinsert = '';
    if ($topmargin ne '0') {$topmargintoinsert='\setlength{\topmargin}{'.$topmargin.'}';}
    my $fancypagestatement='';
    if ($numberofcolumns eq '2') {
	$fancypagestatement="\\fancyhead{}\\fancyhead[LO]{$header_text}";
    } else {
	$fancypagestatement="\\rhead{}\\chead{}\\lhead{$header_text}";
    }
    if ($layout eq 'album') {
	    $text =~ s/\\begin{document}/\\setlength{\\oddsidemargin}{$oddoffset}\\setlength{\\evensidemargin}{$evenoffset}$topmargintoinsert\n\\setlength{\\textwidth}{$textwidth}\\setlength{\\textheight}{$textheight}\\setlength{\\textfloatsep}{8pt plus 2\.0pt minus 4\.0pt}\n\\newlength{\\minipagewidth}\\setlength{\\minipagewidth}{\\textwidth\/\$number_of_columns-0\.2cm}\\usepackage{fancyhdr}\\addtolength{\\headheight}{\\baselineskip}\n\\pagestyle{fancy}$fancypagestatement\\begin{document}\\voffset=-0\.8 cm\\setcounter{page}{1}\n /;
    } elsif ($layout eq 'book') {
	if ($choice ne 'All class print') { 
	    $text =~ s/\\begin{document}/\\textheight $textheight\\oddsidemargin = $evenoffset\\evensidemargin = $evenoffset $topmargintoinsert\n\\textwidth= $textwidth\\newlength{\\minipagewidth}\\setlength{\\minipagewidth}{\\textwidth\/\$number_of_columns-0\.2cm}\n\\renewcommand{\\ref}{\\keephidden\}\\usepackage{fancyhdr}\\addtolength{\\headheight}{\\baselineskip}\\pagestyle{fancy}$fancypagestatement\\begin{document}\n\\voffset=-0\.8 cm\\setcounter{page}{1}\n/;
	} else {
	    $text =~ s/\\pagestyle{fancy}\\rhead{}\\chead{}\s*\\begin{document}/\\textheight = $textheight\\oddsidemargin = $evenoffset\n\\evensidemargin = $evenoffset $topmargintoinsert\\textwidth= $textwidth\\newlength{\\minipagewidth}\n\\setlength{\\minipagewidth}{\\textwidth\/\$number_of_columns-0\.2cm}\\renewcommand{\\ref}{\\keephidden\}\\pagestyle{fancy}\\rhead{}\\chead{}\\begin{document}\\voffset=-0\.8cm\n\\setcounter{page}{1}  \\vskip 5 mm\n /;
	}
	if ($papersize eq 'a4') {
	    $text =~ s/(\\begin{document})/$1\\special{papersize=210mm,297mm}/;
	}
    }
    if ($tableofcontents eq 'yes') {$text=~s/(\\setcounter\{page\}\{1\})/$1 \\tableofcontents\\newpage /;}
    if ($indexlist eq 'yes') {
	$text=~s/(\\begin{document})/\\makeindex $1/;
	$text=~s/(\\end{document})/\\strut\\\\\\strut\\printindex $1/;
    }
    return $text;
}


sub page_cleanup {
    my $result = shift;	
 
    $result =~ m/\\end{document}(\d*)$/;
    my $number_of_columns = $1;
    my $insert = '{';
    for (my $id=1;$id<=$number_of_columns;$id++) { $insert .='l'; }
    $insert .= '}';
    $result =~ s/(\\begin{longtable})INSERTTHEHEADOFLONGTABLE\\endfirsthead\\endhead/$1$insert/g;
    $result =~ s/&\s*REMOVETHEHEADOFLONGTABLE\\\\/\\\\/g;
    return $result,$number_of_columns;
}


sub details_for_menu {
    my ($helper)=@_;
    my $postdata=$env{'form.postdata'};
    if (!$postdata) { $postdata=$helper->{VARS}{'postdata'}; }
    my $name_of_resource = &Apache::lonnet::gettitle($postdata);
    my $symbolic = &Apache::lonnet::symbread($postdata);
    return if ( $symbolic eq '');

    my ($map,$id,$resource)=&Apache::lonnet::decode_symb($symbolic);
    $map=&Apache::lonnet::clutter($map);
    my $name_of_sequence = &Apache::lonnet::gettitle($map);
    if ($name_of_sequence =~ /^\s*$/) {
	$map =~ m|([^/]+)$|;
	$name_of_sequence = $1;
    }
    my $name_of_map = &Apache::lonnet::gettitle($env{'request.course.uri'});
    if ($name_of_map =~ /^\s*$/) {
	$env{'request.course.uri'} =~ m|([^/]+)$|;
	$name_of_map = $1;
    }
    return ($name_of_resource,$name_of_sequence,$name_of_map);
}

sub copyright_line {
    return '\noindent\makebox[\textwidth/$number_of_columns][b]{\hrulefill}\vspace*{-2 mm}\newline\noindent{\tiny Printed from LON-CAPA\copyright MSU{\hfill} Licensed under GNU General Public License } ';
}
my $end_of_student = "\n".'\special{ps:ENDOFSTUDENTSTAMP}'."\n";

sub latex_corrections {
    my ($number_of_columns,$result,$selectionmade,$answer_mode) = @_;
#    $result =~ s/\\includegraphics{/\\includegraphics\[width=\\minipagewidth\]{/g;
    my $copyright = &copyright_line();
    if ($selectionmade eq '1' || $answer_mode eq 'only') {
	$result =~ s/(\\end{document})/\\strut\\vskip 0 mm $copyright $end_of_student $1/;
    } else {
	$result =~ s/(\\end{document})/\\strut\\vspace\*{-4 mm}\\newline $copyright $end_of_student $1/;
    }
    $result =~ s/\$number_of_columns/$number_of_columns/g;
    $result =~ s/(\\end{longtable}\s*)(\\strut\\newline\\noindent\\makebox\[\\textwidth\/$number_of_columns\]\[b\]{\\hrulefill})/$2$1/g;
    $result =~ s/(\\end{longtable}\s*)\\strut\\newline/$1/g;
#-- LaTeX corrections     
    my $first_comment = index($result,'<!--',0);
    while ($first_comment != -1) {
	my $end_comment = index($result,'-->',$first_comment);
	substr($result,$first_comment,$end_comment-$first_comment+3) = '';
	$first_comment = index($result,'<!--',$first_comment);
    }
    $result =~ s/^\s+$//gm; #remove empty lines
    #removes more than one empty space
    $result =~ s|(\s\s+)|($1=~/[\n\r]/)?"\n":" "|ge;
    $result =~ s/\\\\\s*\\vskip/\\vskip/gm;
    $result =~ s/\\\\\s*\\noindent\s*(\\\\)+/\\\\\\noindent /g;
    $result =~ s/{\\par }\s*\\\\/\\\\/gm;
    $result =~ s/\\\\\s+\[/ \[/g;
    #conversion of html characters to LaTeX equivalents
    if ($result =~ m/&(\w+|#\d+);/) {
	$result = &character_chart($result);
    }
    $result =~ s/(\\end{tabular})\s*\\vskip 0 mm/$1/g;
    $result =~ s/(\\begin{enumerate})\s*\\noindent/$1/g;
    return $result;
}


sub index_table {
    my $currentURL = shift;
    my $insex_string='';
    $currentURL=~s/\.([^\/+])$/\.$1\.meta/;
    $insex_string=&Apache::lonnet::metadata($currentURL,'keywords');
    return $insex_string;
}


sub IndexCreation {
    my ($texversion,$currentURL)=@_;
    my @key_words=split(/,/,&index_table($currentURL));
    my $chunk='';
    my $st=index $texversion,'\addcontentsline{toc}{subsection}{';
    if ($st>0) {
	for (my $i=0;$i<3;$i++) {$st=(index $texversion,'}',$st+1);}
	$chunk=substr($texversion,0,$st+1);
	substr($texversion,0,$st+1)=' ';
    }
    foreach my $key_word (@key_words) {
	if ($key_word=~/\S+/) {
	    $texversion=~s/\b($key_word)\b/$1 \\index{$key_word} /i;
	}
    }			
    if ($st>0) {substr($texversion,0,1)=$chunk;}
    return $texversion;
}

sub print_latex_header {
    my $mode=shift;
    my $output='\documentclass[letterpaper,twoside]{article}\raggedbottom';
    if (($mode eq 'batchmode') || (!$perm{'pav'})) {
	$output.='\batchmode';
    }
    $output.='\newcommand{\keephidden}[1]{}\renewcommand{\deg}{$^{\circ}$}'."\n".
   	    '\usepackage{multirow}'."\n".
	     '\usepackage{longtable}\usepackage{textcomp}\usepackage{makeidx}'."\n".
	     '\usepackage[dvips]{graphicx}\usepackage{epsfig}'."\n".
	     '\usepackage{wrapfig}'.
	     '\usepackage{picins}\usepackage{calc}'."\n".
	     '\newenvironment{choicelist}{\begin{list}{}{\setlength{\rightmargin}{0in}'."\n".
	     '\setlength{\leftmargin}{0.13in}\setlength{\topsep}{0.05in}'."\n".
	     '\setlength{\itemsep}{0.022in}\setlength{\parsep}{0in}'."\n".
	     '\setlength{\belowdisplayskip}{0.04in}\setlength{\abovedisplayskip}{0.05in}'."\n".
	     '\setlength{\abovedisplayshortskip}{-0.04in}'."\n".
	     '\setlength{\belowdisplayshortskip}{0.04in}}}{\end{list}}'."\n".
	     '\renewenvironment{theindex}{\begin{list}{}{{\vskip 1mm \noindent \large'."\n".
	     '\textbf{Index}} \newline \setlength{\rightmargin}{0in}'."\n".
	     '\setlength{\leftmargin}{0.13in}\setlength{\topsep}{0.01in}'."\n".
	     '\setlength{\itemsep}{0.1in}\setlength{\parsep}{-0.02in}'."\n".
	     '\setlength{\belowdisplayskip}{0.01in}\setlength{\abovedisplayskip}{0.01in}'."\n".
	     '\setlength{\abovedisplayshortskip}{-0.04in}'."\n".
	     '\setlength{\belowdisplayshortskip}{0.01in}}}{\end{list}}\begin{document}'."\n";
    return $output;	     
}

sub path_to_problem {
    my ($urlp,$colwidth)=@_;
    $urlp=&Apache::lonnet::clutter($urlp);

    my $newurlp = '';
    $colwidth=~s/\s*mm\s*$//;
#characters average about 2 mm in width
    if (length($urlp)*2 > $colwidth) {
	my @elements = split('/',$urlp);
	my $curlength=0;
	foreach my $element (@elements) {
	    if ($element eq '') { next; }
	    if ($curlength+(length($element)*2) > $colwidth) {
		$newurlp .=  '|\vskip -1 mm \verb|';
		$curlength=length($element)*2;
	    } else {
		$curlength+=length($element)*2;
	    }
	    $newurlp.='/'.$element;
	}
    } else {
	$newurlp=$urlp;
    }
    return '{\small\noindent\verb|'.$newurlp.'|\vskip 0 mm}';
}

sub recalcto_mm {
    my $textwidth=shift;
    my $LaTeXwidth;
    if ($textwidth=~/(-?\d+\.?\d*)\s*cm/) {
	$LaTeXwidth = $1*10;
    } elsif ($textwidth=~/(-?\d+\.?\d*)\s*mm/) {
	$LaTeXwidth = $1;
    } elsif ($textwidth=~/(-?\d+\.?\d*)\s*in/) {
	$LaTeXwidth = $1*25.4;
    }
    $LaTeXwidth.=' mm';
    return $LaTeXwidth;
}

sub get_textwidth {
    my ($helper,$LaTeXwidth)=@_;
    my $textwidth=$LaTeXwidth;
    if ($helper->{'VARS'}->{'pagesize.width'}=~/\d+/ &&
	$helper->{'VARS'}->{'pagesize.widthunit'}=~/\w+/) {
	$textwidth=&recalcto_mm($helper->{'VARS'}->{'pagesize.width'}.' '.
				$helper->{'VARS'}->{'pagesize.widthunit'});
    }
    return $textwidth;
}


sub unsupported {
    my ($currentURL,$mode,$symb)=@_;
    if ($mode ne '') {$mode='\\'.$mode}
    my $result.= &print_latex_header($mode);
    if ($currentURL=~m|^(/adm/wrapper/)?ext/|) {
	$currentURL=~s|^(/adm/wrapper/)?ext/|http://|;
	my $title=&Apache::lonnet::gettitle($symb);
	$title = &Apache::lonxml::latex_special_symbols($title);
	$result.=' \strut \\\\ '.$title.' \strut \\\\ '.$currentURL.' ';
    } else {
	$result.=$currentURL;
    }
    $result.= '\vskip 0.5mm\noindent\makebox[\textwidth/$number_of_columns][b]{\hrulefill} \end{document}';
    return $result;
}


#
# List of recently generated print files
#
sub recently_generated {
    my $r=shift;
    my $prtspool=$r->dir_config('lonPrtDir');
    my $zip_result;
    my $pdf_result;
    opendir(DIR,$prtspool);

    my @files = 
	grep(/^$env{'user.name'}_$env{'user.domain'}_printout_(\d+)_.*\.(pdf|zip)$/,readdir(DIR));
    closedir(DIR);

    @files = sort {
	my ($actime) = (stat($prtspool.'/'.$a))[10];
	my ($bctime) = (stat($prtspool.'/'.$b))[10];
	return $bctime <=> $actime;
    } (@files);

    foreach my $filename (@files) {
	my ($ext) = ($filename =~ m/(pdf|zip)$/);
	my ($cdev,$cino,$cmode,$cnlink,
	    $cuid,$cgid,$crdev,$csize,
	    $catime,$cmtime,$cctime,
	    $cblksize,$cblocks)=stat($prtspool.'/'.$filename);
	my $result="<a href='/prtspool/$filename'>".
	    &mt('Generated [_1] ([_2] bytes)',
		&Apache::lonlocal::locallocaltime($cctime),$csize).
		'</a><br />';
	if ($ext eq 'pdf') { $pdf_result .= $result; }
	if ($ext eq 'zip') { $zip_result .= $result; }
    }
    if ($zip_result) {
	$r->print('<h4>'.&mt('Recently generated printout zip files')."</h4>\n"
		  .$zip_result);
    }
    if ($pdf_result) {
	$r->print('<h4>'.&mt('Recently generated printouts')."</h4>\n"
		  .$pdf_result);
    }
}

#
#   Retrieve the hash of page breaks.
#
#  Inputs:
#    helper   - reference to helper object.
#  Outputs
#    A reference to a page break hash.
#
#
#use Data::Dumper;
#sub dump_helper_vars {
#    my ($helper) = @_;
#    my $helpervars = Dumper($helper->{'VARS'});
#    &Apache::lonnet::logthis("Dump of helper vars:\n $helpervars");
#}

sub get_page_breaks  {
    my ($helper) = @_;
    my %page_breaks;

    foreach my $break (split /\|\|\|/, $helper->{'VARS'}->{'FINISHPAGE'}) {
	$page_breaks{$break} = 1;
    }
    return %page_breaks;
}

#  Output a sequence (recursively if neeed)
#  from construction space.
# Parameters:
#    url     = URL of the sequence to print.
#    helper  - Reference to the helper hash.
#    form    - Copy of the format hash.
#    LaTeXWidth
# Returns:
#   Text to add to the printout.
#   NOTE if the first element of the outermost sequence
#   is itself a sequence, the outermost caller may need to
#   prefix the latex with the page headers stuff.
#
sub print_construction_sequence {
    my ($currentURL, $helper, %form, $LaTeXwidth) = @_;
    my $result;
    my $rndseed=time;
    if ($helper->{'VARS'}->{'curseed'}) {
	$rndseed=$helper->{'VARS'}->{'curseed'};
    }
    my $errtext=&LONCAPA::map::mapread($currentURL);
    # 
    #  These make this all support recursing for subsequences.
    #
    my @order    = @LONCAPA::map::order;
    my @resources = @LONCAPA::map::resources; 
    for (my $member=0;$member<=$#order;$member++) {
	$resources[$order[$member]]=~/^([^:]*):([^:]*):/;
	my $urlp=$2;
	if ($urlp=~/\.(problem|exam|quiz|assess|survey|form|library|xml|html|htm|xhtml|xhtm)$/) {
	    my $texversion='';
	    if ($helper->{'VARS'}->{'ANSWER_TYPE'} ne 'only') {
		$form{'problem_split'}=$parmhash{'problem_stream_switch'};
		$form{'suppress_tries'}=$parmhash{'suppress_tries'};
		$form{'latex_type'}=$helper->{'VARS'}->{'LATEX_TYPE'};
		$form{'rndseed'}=$rndseed;
		$resources_printed .=$urlp.':';
		$texversion=&Apache::lonnet::ssi($urlp,%form);
	    }
	    if((($helper->{'VARS'}->{'ANSWER_TYPE'} eq 'no') ||
		($helper->{'VARS'}->{'ANSWER_TYPE'} eq 'only')) && 
	       ($urlp=~/\.(problem|exam|quiz|assess|survey|form|library|page)$/)) {
		#  Don't permanently modify %$form...
		my %answerform = %form;
		$answerform{'grade_target'}='answer';
		$answerform{'answer_output_mode'}='tex';
		$answerform{'rndseed'}=$rndseed;
		$answerform{'problem_split'}=$parmhash{'problem_stream_switch'};
		if ($urlp=~/\/res\//) {$env{'request.state'}='published';}
		$resources_printed .= $urlp.':';
		my $answer=&Apache::lonnet::ssi($urlp,%answerform);
		if ($helper->{'VARS'}->{'ANSWER_TYPE'} eq 'no') {
		    $texversion=~s/(\\keephidden{ENDOFPROBLEM})/$answer$1/;
		} else {
		    # If necessary, encapsulate answer in minipage:
		    
		    $texversion=&print_latex_header($helper->{'VARS'}->{'LATEX_TYPE'});
		    my $title = &Apache::lonnet::gettitle($helper->{'VARS'}->{'symb'});
		    $title = &Apache::lonxml::latex_special_symbols($title);
		    my $body ='\vskip 0 mm \noindent\textbf{'.$title.'}\vskip 0 mm ';
		    $body.=&path_to_problem($urlp,$LaTeXwidth);
		    $body.='\vskip 1 mm '.$answer.'\end{document}';
		    $body = &encapsulate_minipage($body);
		    $texversion.=$body;
		}
	    }
	    $texversion = &latex_header_footer_remove($texversion);

	    if ($helper->{'VARS'}->{'TABLE_INDEX'} eq 'yes') {
		$texversion=&IndexCreation($texversion,$urlp);
	    }
	    if ($helper->{'VARS'}->{'CONSTR_RESOURSE_URL'} eq 'yes') {
		$texversion=~s/(\\addcontentsline\{toc\}\{subsection\}\{[^\}]*\})/$1 URL: \\verb|$urlp| \\strut\\\\\\strut /;
	    }
	    $result.=$texversion;

	} elsif ($urlp=~/\.(sequence|page)$/) {
	    
	    # header:

	    $result.='\strut\newline\noindent Sequence/page '.$urlp.'\strut\newline\noindent\makebox[\textwidth/$number_of_columns][b]{\hrulefill}\newline\noindent ';

	    # IF sequence, recurse:
	    
	    if ($urlp =~ /\.sequence$/) {
		my $sequence_url = $urlp;
		my $domain       = $env{'user.domain'};	# Constr. space only on local
		my $user         = $env{'user.name'};

		$sequence_url    =~ s/^\/res\/$domain/\/home/;
		$sequence_url    =~ s/^(\/home\/$user)/$1\/public_html/;
#		$sequence_url    =~ s|\/~([^\/]+)\/|\/home\/$1\/public_html\/|;
		$result .= &print_construction_sequence($sequence_url, 
							$helper, %form, 
							$LaTeXwidth);
	    }
	}  
    }
    if ($helper->{VARS}->{'construction'} eq '1') {$result=~s/(\\begin{document})/$1 \\fbox\{RANDOM SEED IS $rndseed\} /;}
    return $result;
}

sub output_data {
    my ($r,$helper,$rparmhash) = @_;
    my %parmhash = %$rparmhash;
    $resources_printed = '';
    my $do_postprocessing = 1;
    my $js = <<ENDPART;
<script type="text/javascript">
    var editbrowser;
    function openbrowser(formname,elementname,only,omit) {
        var url = '/res/?';
        if (editbrowser == null) {
            url += 'launch=1&';
        }
        url += 'catalogmode=interactive&';
        url += 'mode=parmset&';
        url += 'form=' + formname + '&';
        if (only != null) {
            url += 'only=' + only + '&';
        } 
        if (omit != null) {
            url += 'omit=' + omit + '&';
        }
        url += 'element=' + elementname + '';
        var title = 'Browser';
        var options = 'scrollbars=1,resizable=1,menubar=0';
        options += ',width=700,height=600';
        editbrowser = open(url,title,options,'1');
        editbrowser.focus();
    }
</script>
ENDPART

    my $start_page  = &Apache::loncommon::start_page('Preparing Printout',$js);
    my $msg = &mt('Please stand by while processing your print request, this may take some time ...');

    $r->print($start_page."\n<p>\n$msg\n</p>\n");

    # fetch the pagebreaks and store them in the course environment
    # The page breaks will be pulled into the hash %page_breaks which is
    # indexed by symb and contains 1's for each break.

    $env{'form.pagebreaks'}  = $helper->{'VARS'}->{'FINISHPAGE'};
    $env{'form.lastprinttype'} = $helper->{'VARS'}->{'PRINT_TYPE'}; 
    &Apache::loncommon::store_course_settings('print',
					      {'pagebreaks'    => 'scalar',
					       'lastprinttype' => 'scalar'});

    my %page_breaks  = &get_page_breaks($helper);

    my $format_from_helper = $helper->{'VARS'}->{'FORMAT'};
    my ($result,$selectionmade) = ('','');
    my $number_of_columns = 1; #used only for pages to determine the width of the cell
    my @temporary_array=split /\|/,$format_from_helper;
    my ($laystyle,$numberofcolumns,$papersize)=@temporary_array;
    if ($laystyle eq 'L') {
	$laystyle='album';
    } else {
	$laystyle='book';
    }
    my ($textwidth,$textheight,$oddoffset,$evenoffset) = &page_format($papersize,$laystyle,$numberofcolumns);
    my $assignment =  $env{'form.assignment'};
    my $LaTeXwidth=&recalcto_mm($textwidth); 
    my @print_array=();
    my @student_names=();

    #  Common settings for the %form has:
    # In some cases these settings get overriddent by specific cases, but the
    # settings are common enough to make it worthwhile factoring them out
    # here.
    #
    my %form;
    $form{'grade_target'} = 'tex';
    $form{'textwidth'}    = &get_textwidth($helper, $LaTeXwidth);

    # If form.showallfoils is set, then request all foils be shown:
    # privilege will be enforced both by not allowing the 
    # check box selecting this option to be presnt unless it's ok,
    # and by lonresponse's priv. check.
    # The if is here because lonresponse.pm only cares that
    # showallfoils is defined, not what the value is.

    if ($helper->{'VARS'}->{'showallfoils'} eq "1") { 
	$form{'showallfoils'} = $helper->{'VARS'}->{'showallfoils'};
    }
    
    if ($helper->{'VARS'}->{'style_file'}=~/\w/) {
	&Apache::lonnet::appenv('construct.style' =>
				$helper->{'VARS'}->{'style_file'});
    } elsif ($env{'construct.style'}) {
	&Apache::lonnet::delenv('construct\\.style');
    }


    if ($helper->{'VARS'}->{'PRINT_TYPE'} eq 'current_document') {
      #-- single document - problem, page, html, xml, ...
	my ($currentURL,$cleanURL);

	if ($helper->{'VARS'}->{'construction'} ne '1') {
            #prints published resource
	    $currentURL=$helper->{'VARS'}->{'postdata'};
	    $cleanURL=&Apache::lonenc::check_decrypt($currentURL);
	} else {
            #prints resource from the construction space
	    $currentURL='/'.$helper->{'VARS'}->{'filename'};
	    if ($currentURL=~/([^?]+)/) {$currentURL=$1;}
	    $cleanURL=$currentURL;
	}
	$selectionmade = 1;
	if ($cleanURL!~m|^/adm/|
	    && $cleanURL=~/\.(problem|exam|quiz|assess|survey|form|library|page|xml|html|htm|xhtml|xhtm)$/) {
	    my $rndseed=time;
	    my $texversion='';
	    if ($helper->{'VARS'}->{'ANSWER_TYPE'} ne 'only') {
		my %moreenv;
		$moreenv{'request.filename'}=$cleanURL;
                if ($helper->{'VARS'}->{'probstatus'} eq 'exam') {$form{'problemtype'}='exam';}
		$form{'problem_split'}=$parmhash{'problem_stream_switch'};
		$form{'suppress_tries'}=$parmhash{'suppress_tries'};
		$form{'latex_type'}=$helper->{'VARS'}->{'LATEX_TYPE'};
		$form{'print_discussions'}=$helper->{'VARS'}->{'PRINT_DISCUSSIONS'};
		if ($helper->{'VARS'}->{'PRINT_DISCUSSIONS'} eq 'yes') {$form{'problem_split'}='yes';}
		if ($helper->{'VARS'}->{'curseed'}) {
		    $rndseed=$helper->{'VARS'}->{'curseed'};
		}
		$form{'rndseed'}=$rndseed;
		&Apache::lonnet::appenv(%moreenv);

		&Apache::lonxml::clear_problem_counter();

		$resources_printed .= $currentURL.':';
		$texversion.=&Apache::lonnet::ssi($currentURL,%form);

		&Apache::lonxml::clear_problem_counter();

		&Apache::lonnet::delenv('request.filename');
	    }
	    # current document with answers.. no need to encap in minipage
	    #  since there's only one answer.

	    if(($helper->{'VARS'}->{'ANSWER_TYPE'} eq 'no') ||
	       ($helper->{'VARS'}->{'ANSWER_TYPE'} eq 'only')) {
		$form{'problem_split'}=$parmhash{'problem_stream_switch'};
		$form{'grade_target'}='answer';
		$form{'answer_output_mode'}='tex';
		$form{'rndseed'}=$rndseed;
                if ($helper->{'VARS'}->{'probstatus'} eq 'exam') {
		    $form{'problemtype'}='exam';
		}
		$resources_printed .= $currentURL.':';
		my $answer=&Apache::lonnet::ssi($currentURL,%form);
		if ($helper->{'VARS'}->{'ANSWER_TYPE'} eq 'no') {
		    $texversion=~s/(\\keephidden{ENDOFPROBLEM})/$answer$1/;
		} else {
		    $texversion=&print_latex_header($helper->{'VARS'}->{'LATEX_TYPE'});
		    if ($helper->{'VARS'}->{'construction'} ne '1') {
			my $title = &Apache::lonnet::gettitle($helper->{'VARS'}->{'symb'});
			$title = &Apache::lonxml::latex_special_symbols($title);
			$texversion.='\vskip 0 mm \noindent\textbf{'.$title.'}\vskip 0 mm ';
			$texversion.=&path_to_problem($cleanURL,$LaTeXwidth);
		    } else {
			$texversion.='\vskip 0 mm \noindent\textbf{Prints from construction space - there is no title.}\vskip 0 mm ';
			my $URLpath=$cleanURL;
			$URLpath=~s/~([^\/]+)/public_html\/$1\/$1/;
			$texversion.=&path_to_problem($URLpath,$LaTeXwidth);
		    }
		    $texversion.='\vskip 1 mm '.$answer.'\end{document}';
		}
	    }
	    if ($helper->{'VARS'}->{'TABLE_INDEX'} eq 'yes') {
		$texversion=&IndexCreation($texversion,$currentURL);
	    }
	    if ($helper->{'VARS'}->{'CONSTR_RESOURSE_URL'} eq 'yes') {
		$texversion=~s/(\\addcontentsline\{toc\}\{subsection\}\{[^\}]*\})/$1 URL: \\verb|$currentURL| \\strut\\\\\\strut /;

	    }
	    $result .= $texversion;
	    if ($currentURL=~m/\.page\s*$/) {
		($result,$number_of_columns) = &page_cleanup($result);
	    }
        } elsif ($cleanURL!~m|^/adm/|
		 && $currentURL=~/\.sequence$/ && $helper->{'VARS'}->{'construction'} eq '1') {
            #printing content of sequence from the construction space	
	    $currentURL=~s|\/~([^\/]+)\/|\/home\/$1\/public_html\/|;
	    $result .= &print_latex_header($helper->{'VARS'}->{'LATEX_TYPE'});
	    $result .= &print_construction_sequence($currentURL, $helper, %form,
						    $LaTeXwidth);
	    $result .= '\end{document}';  
	    if (!($result =~ /\\begin\{document\}/)) {
		$result = &print_latex_header() . $result;
	    }
	    # End construction space sequence.
	} elsif ($cleanURL=~/\/(smppg|syllabus|aboutme|bulletinboard)$/) { 
		$form{'latex_type'}=$helper->{'VARS'}->{'LATEX_TYPE'};
		if ($currentURL=~/\/syllabus$/) {$currentURL=~s/\/res//;}
		$resources_printed .= $currentURL.':';
		my $texversion=&Apache::lonnet::ssi($currentURL,%form);
		$result .= $texversion;
	} elsif ($cleanURL =~/.tex$/) {
	    # For this sort of print of a single LaTeX file,
	    # We can just print the LaTeX file as it is uninterpreted in any way:
	    #

	    $result = &fetch_raw_resource($currentURL);
	    $do_postprocessing = 0; # Don't massage the result.

	} else {
	    $result.=&unsupported($currentURL,$helper->{'VARS'}->{'LATEX_TYPE'},
				  $helper->{'VARS'}->{'symb'});
	}
    } elsif (($helper->{'VARS'}->{'PRINT_TYPE'} eq 'map_problems')       or
             ($helper->{'VARS'}->{'PRINT_TYPE'} eq 'map_problems_pages') or
             ($helper->{'VARS'}->{'PRINT_TYPE'} eq 'all_problems')       or
	     ($helper->{'VARS'}->{'PRINT_TYPE'} eq 'all_resources')      or # BUGBUG
	     ($helper->{'VARS'}->{'PRINT_TYPE'} eq 'select_sequences')) { 
        #-- produce an output string
	if ($helper->{'VARS'}->{'PRINT_TYPE'} eq 'map_problems') {
	    $selectionmade = 2;
	} elsif ($helper->{'VARS'}->{'PRINT_TYPE'} eq 'map_problems_pages') {
	    $selectionmade = 3;
	} elsif ($helper->{'VARS'}->{'PRINT_TYPE'} eq 'all_problems') {
	    $selectionmade = 4;
	} elsif ($helper->{'VARS'}->{'PRINT_TYPE'} eq 'all_resources') {  #BUGBUG
	    $selectionmade = 4;
	} elsif ($helper->{'VARS'}->{'PRINT_TYPE'} eq 'select_sequences') {
	    $selectionmade = 7;
	}
	$form{'problem_split'}=$parmhash{'problem_stream_switch'};
	$form{'suppress_tries'}=$parmhash{'suppress_tries'};
	$form{'latex_type'}=$helper->{'VARS'}->{'LATEX_TYPE'};
	$form{'print_discussions'}=$helper->{'VARS'}->{'PRINT_DISCUSSIONS'};
	if ($helper->{'VARS'}->{'PRINT_DISCUSSIONS'} eq 'yes') {$form{'problem_split'}='yes';}
	my $flag_latex_header_remove = 'NO';
	my $flag_page_in_sequence = 'NO';
	my @master_seq=split /\|\|\|/, $helper->{'VARS'}->{'RESOURCES'};
	my $prevassignment='';

	&Apache::lonxml::clear_problem_counter();

	my $pbreakresources = keys %page_breaks;
	for (my $i=0;$i<=$#master_seq;$i++) {

	    # Note due to document structure, not allowed to put \newpage
	    # prior to the first resource

	    if (defined $page_breaks{$master_seq[$i]}) {
		if($i != 0) {
		    $result.="\\newpage\n";
		}
	    }
	    my ($sequence,undef,$urlp)=&Apache::lonnet::decode_symb($master_seq[$i]);
	    $urlp=&Apache::lonnet::clutter($urlp);
	    $form{'symb'}=$master_seq[$i];

	    my $assignment=&Apache::lonxml::latex_special_symbols(&Apache::lonnet::gettitle($sequence),'header'); #title of the assignment which contains this problem
	    if ($selectionmade==7) {$helper->{VARS}->{'assignment'}=$assignment;}
	    if ($i==0) {$prevassignment=$assignment;}
	    my $texversion='';
	    if ($urlp!~m|^/adm/|
		&& $urlp=~/\.(problem|exam|quiz|assess|survey|form|library|page|xml|html|htm|xhtml|xhtm)$/) {
		$resources_printed .= $urlp.':';

		&Apache::lonxml::remember_problem_counter();
		$texversion.=&Apache::lonnet::ssi($urlp,%form);
		if ($urlp=~/\.page$/) {
		    ($texversion,my $number_of_columns_page) = &page_cleanup($texversion);
		    if ($number_of_columns_page > $number_of_columns) {$number_of_columns=$number_of_columns_page;} 
		    $texversion =~ s/\\end{document}\d*/\\end{document}/;
		    $flag_page_in_sequence = 'YES';
		} 

		if(($helper->{'VARS'}->{'ANSWER_TYPE'} eq 'no') ||
		   ($helper->{'VARS'}->{'ANSWER_TYPE'} eq 'only')) {
		    #  Don't permanently pervert the %form hash
		    my %answerform = %form;
		    $answerform{'grade_target'}='answer';
		    $answerform{'answer_output_mode'}='tex';
		    $resources_printed .= $urlp.':';

		    &Apache::lonxml::restore_problem_counter();
		    my $answer=&Apache::lonnet::ssi($urlp,%answerform);

		    if ($helper->{'VARS'}->{'ANSWER_TYPE'} eq 'no') {
			$texversion=~s/(\\keephidden{ENDOFPROBLEM})/$answer$1/;
		    } else {
			if ($urlp=~/\.(problem|exam|quiz|assess|survey|form|library)$/) {
			    $texversion=&print_latex_header($helper->{'VARS'}->{'LATEX_TYPE'});
			    my $title = &Apache::lonnet::gettitle($master_seq[$i]);
			    $title = &Apache::lonxml::latex_special_symbols($title);
			    my $body ='\vskip 0 mm \noindent\textbf{'.$title.'}\vskip 0 mm ';
			    $body   .= &path_to_problem ($urlp,$LaTeXwidth);
			    $body   .='\vskip 1 mm '.$answer;
			    $body    = &encapsulate_minipage($body);
			    $texversion .= $body;
			} else {
			    $texversion='';
			}
		    }
		}
		if ($flag_latex_header_remove ne 'NO') {
		    $texversion = &latex_header_footer_remove($texversion);
		} else {
		    $texversion =~ s/\\end{document}//;
		}
		if ($helper->{'VARS'}->{'TABLE_INDEX'} eq 'yes') {
		    $texversion=&IndexCreation($texversion,$urlp);
		}
		if (($selectionmade == 4) and ($assignment ne $prevassignment)) {
		    my $name = &get_name();
		    my $courseidinfo = &get_course();
		    if (defined($courseidinfo)) { $courseidinfo=' - '.$courseidinfo }
		    $prevassignment=$assignment;
		    my $header_text = $parmhash{'print_header_format'};
		    $header_text    = &format_page_header($textwidth, $header_text,
							  $assignment, 
							  $courseidinfo, 
							  $name);
		    if ($numberofcolumns eq '1') {
			$result .='\newpage \noindent\parbox{\minipagewidth}{\noindent\\lhead{'.$header_text.'}} \vskip 5 mm ';
		    } else {
			$result .='\newpage \noindent\parbox{\minipagewidth}{\noindent\\fancyhead[LO]{'.$header_text.'}} \vskip 5 mm ';
		    }			
		}
		$result .= $texversion;
		$flag_latex_header_remove = 'YES';   
	    } elsif ($urlp=~/\/(smppg|syllabus|aboutme|bulletinboard)$/) { 
		$form{'latex_type'}=$helper->{'VARS'}->{'LATEX_TYPE'};
		if ($urlp=~/\/syllabus$/) {$urlp=~s/\/res//;}
		$resources_printed .= $urlp.':';
		my $texversion=&Apache::lonnet::ssi($urlp,%form);
		if ($flag_latex_header_remove ne 'NO') {
		    $texversion = &latex_header_footer_remove($texversion);
		} else {
		    $texversion =~ s/\\end{document}/\\vskip 0\.5mm\\noindent\\makebox\[\\textwidth\/\$number_of_columns\]\[b\]\{\\hrulefill\}/;
		}
		$result .= $texversion;
		$flag_latex_header_remove = 'YES'; 
	    } else {
		$texversion=&unsupported($urlp,$helper->{'VARS'}->{'LATEX_TYPE'},
					 $master_seq[$i]);
		if ($flag_latex_header_remove ne 'NO') {
		    $texversion = &latex_header_footer_remove($texversion);
		} else {
		    $texversion =~ s/\\end{document}//;
		}
		$result .= $texversion;
		$flag_latex_header_remove = 'YES';   
	    }	    
	    if (&Apache::loncommon::connection_aborted($r)) { last; }
	}
	&Apache::lonxml::clear_problem_counter();
	if ($flag_page_in_sequence eq 'YES') {
	    $result =~ s/\\usepackage{calc}/\\usepackage{calc}\\usepackage{longtable}/;
	}	
	$result .= '\end{document}';
     } elsif (($helper->{'VARS'}->{'PRINT_TYPE'} eq 'problems_for_students') ||
	      ($helper->{'VARS'}->{'PRINT_TYPE'} eq 'resources_for_students')){


     #-- prints assignments for whole class or for selected students  
	 my $type;
	 if ($helper->{'VARS'}->{'PRINT_TYPE'} eq 'problems_for_students') {
	     $selectionmade=5;
	     $type='problems';
	 } elsif ($helper->{'VARS'}->{'PRINT_TYPE'} eq 'resources_for_students') {
	     $selectionmade=8;
	     $type='resources';
	 }
	 my @students=split /\|\|\|/, $helper->{'VARS'}->{'STUDENTS'};
	 #   The normal sort order is by section then by students within the
	 #   section. If the helper var student_sort is 1, then the user has elected
	 #   to override this and output the students by name.
	 #    Each element of the students array is of the form:
	 #       username:domain:section:last, first:status
	 #    
	 #  Note that student sort is not compatible with printing 
	 #  1 section per pdf...so that setting overrides.
	 #   
	 if (($helper->{'VARS'}->{'student_sort'}    eq 1)  && 
	     ($helper->{'VARS'}->{'SPLIT_PDFS'} ne "sections")) {
	     @students = sort compare_names  @students;
	 }
	 &adjust_number_to_print($helper);

         if ($helper->{'VARS'}->{'NUMBER_TO_PRINT'} eq '0' ||
	     $helper->{'VARS'}->{'NUMBER_TO_PRINT'} eq 'all' ) {
	     $helper->{'VARS'}->{'NUMBER_TO_PRINT'}=$#students+1;
	 }
	 # If we are splitting on section boundaries, we need 
	 # to remember that in split_on_sections and 
	 # print all of the students in the list.
	 #
	 my $split_on_sections = 0;
	 if ($helper->{'VARS'}->{'NUMBER_TO_PRINT'} eq 'section') {
	     $split_on_sections = 1;
	     $helper->{'VARS'}->{'NUMBER_TO_PRINT'} = $#students+1;
	 }
	 my @master_seq=split /\|\|\|/, $helper->{'VARS'}->{'RESOURCES'};

	 #loop over students
	 my $flag_latex_header_remove = 'NO'; 
	 my %moreenv;
         $moreenv{'instructor_comments'}='hide';
	 $moreenv{'textwidth'}=&get_textwidth($helper,$LaTeXwidth);
	 $moreenv{'print_discussions'}=$helper->{'VARS'}->{'PRINT_DISCUSSIONS'};
	 $moreenv{'problem_split'}    = $parmhash{'problem_stream_switch'};
	 $moreenv{'suppress_tries'}   = $parmhash{'suppress_tries'};
	 if ($helper->{'VARS'}->{'PRINT_DISCUSSIONS'} eq 'yes') {$moreenv{'problem_split'}='yes';}
	 my %prog_state=&Apache::lonhtmlcommon::Create_PrgWin($r,'Print Status','Class Print Status',$#students+1,'inline','75');
	 my $student_counter=-1;
	 my $i = 0;
	 my $last_section = (split(/:/,$students[0]))[2];
	 foreach my $person (@students) {

             my $duefile="/home/httpd/prtspool/$env{'user.name'}_$env{'user.domain'}_printout.due";
	     if (-e $duefile) {
		 my $temp_file = Apache::File->new('>>'.$duefile);
		 print $temp_file "1969\n";
	     }
	     $student_counter++;
	     if ($split_on_sections) {
		 my $this_section = (split(/:/,$person))[2];
		 if ($this_section ne $last_section) {
		     $i++;
		     $last_section = $this_section;
		 }
	     } else {
		 $i=int($student_counter/$helper->{'VARS'}{'NUMBER_TO_PRINT'});
	     }
	     my ($output,$fullname, $printed)=&print_resources($r,$helper,
						     $person,$type,
						     \%moreenv,\@master_seq,
						     $flag_latex_header_remove,
						     $LaTeXwidth);
	     $resources_printed .= ":";
	     $print_array[$i].=$output;
	     $student_names[$i].=$person.':'.$fullname.'_END_';
	     &Apache::lonhtmlcommon::Increment_PrgWin($r,\%prog_state,&mt('last student').' '.$fullname);
	     $flag_latex_header_remove = 'YES';
	     if (&Apache::loncommon::connection_aborted($r)) { last; }
	 }
	 &Apache::lonhtmlcommon::Close_PrgWin($r,\%prog_state);
	 $result .= $print_array[0].'  \end{document}';
     } elsif (($helper->{'VARS'}->{'PRINT_TYPE'} eq 'problems_for_anon')     ||
	      ($helper->{'VARS'}->{'PRINT_TYPE'} eq 'resources_for_anon')  ) { 
	 my $cdom =$env{'course.'.$env{'request.course.id'}.'.domain'};
	 my $cnum =$env{'course.'.$env{'request.course.id'}.'.num'};
	 my $num_todo=$helper->{'VARS'}->{'NUMBER_TO_PRINT_TOTAL'};
	 my $code_name=$helper->{'VARS'}->{'ANON_CODE_STORAGE_NAME'};
	 my $old_name=$helper->{'VARS'}->{'REUSE_OLD_CODES'};
	 my $single_code = $helper->{'VARS'}->{'SINGLE_CODE'};
	 my $selected_code = $helper->{'VARS'}->{'CODE_SELECTED_FROM_LIST'};

	 my $code_option=$helper->{'VARS'}->{'CODE_OPTION'};
	 open(FH,$Apache::lonnet::perlvar{'lonTabDir'}.'/scantronformat.tab');
	 my ($code_type,$code_length)=('letter',6);
	 foreach my $line (<FH>) {
	     my ($name,$type,$length) = (split(/:/,$line))[0,2,4];
	     if ($name eq $code_option) {
		 $code_length=$length;
		 if ($type eq 'number') { $code_type = 'number'; }
	     }
	 }
	 my %moreenv = ('textwidth' => &get_textwidth($helper,$LaTeXwidth));
	 $moreenv{'problem_split'}    = $parmhash{'problem_stream_switch'};
         $moreenv{'instructor_comments'}='hide';
	 my $seed=time+($$<<16)+($$);
	 my @allcodes;
	 if ($old_name) {
	     my %result=&Apache::lonnet::get('CODEs',
					     [$old_name,"type\0$old_name"],
					     $cdom,$cnum);
	     $code_type=$result{"type\0$old_name"};
	     @allcodes=split(',',$result{$old_name});
	     $num_todo=scalar(@allcodes);
	 } elsif ($selected_code) { # Selection value is always numeric.
	     $num_todo = 1;
	     @allcodes = ($selected_code);
	 } elsif ($single_code) {

	     $num_todo    = 1;	# Unconditionally one code to do.
	     # If an alpha code have to convert to numbers so it can be
	     # converted back to letters again :-)
	     #
	     if ($code_type ne 'number') {
		 $single_code = &letters_to_num($single_code);
	     }
	     @allcodes = ($single_code);
	 } else {
	     my %allcodes;
	     srand($seed);
	     for (my $i=0;$i<$num_todo;$i++) {
		 $moreenv{'CODE'}=&get_CODE(\%allcodes,$i,$seed,$code_length,
					    $code_type);
	     }
	     if ($code_name) {
		 &Apache::lonnet::put('CODEs',
				      {
					$code_name =>join(',',keys(%allcodes)),
					"type\0$code_name" => $code_type
				      },
				      $cdom,$cnum);
	     }
	     @allcodes=keys(%allcodes);
	 }
	 my @master_seq=split /\|\|\|/, $helper->{'VARS'}->{'RESOURCES'};
	 my ($type) = split(/_/,$helper->{'VARS'}->{'PRINT_TYPE'});
	 &adjust_number_to_print($helper);
	 my $number_per_page=$helper->{'VARS'}->{'NUMBER_TO_PRINT'};
	 if ($number_per_page eq '0' || $number_per_page eq 'all') {
	     $number_per_page=$num_todo;
	 }
	 my $flag_latex_header_remove = 'NO'; 
	 my %prog_state=&Apache::lonhtmlcommon::Create_PrgWin($r,'Print Status','Class Print Status',$num_todo,'inline','75');
	 my $count=0;
	 foreach my $code (sort(@allcodes)) {
	     my $file_num=int($count/$number_per_page);
	     if ($code_type eq 'number') { 
		 $moreenv{'CODE'}=$code;
	     } else {
		 $moreenv{'CODE'}=&num_to_letters($code);
	     }
	     my ($output,$fullname, $printed)=
		 &print_resources($r,$helper,'anonymous',$type,\%moreenv,
				  \@master_seq,$flag_latex_header_remove,
				  $LaTeXwidth);
	     $resources_printed .= ":";
	     $print_array[$file_num].=$output;
	     &Apache::lonhtmlcommon::Increment_PrgWin($r,\%prog_state,
				       &mt('last assignment').' '.$fullname);
	     $flag_latex_header_remove = 'YES';
	     $count++;
	     if (&Apache::loncommon::connection_aborted($r)) { last; }
	 }
	 &Apache::lonhtmlcommon::Close_PrgWin($r,\%prog_state);
	 $result .= $print_array[0].'  \end{document}';
     } elsif ($helper->{'VARS'}->{'PRINT_TYPE'} eq 'problems_from_directory') {      
    #prints selected problems from the subdirectory 
	$selectionmade = 6;
        my @list_of_files=split /\|\|\|/, $helper->{'VARS'}->{'FILES'};
	@list_of_files=sort @list_of_files;
	my $flag_latex_header_remove = 'NO'; 
	my $rndseed=time;
	if ($helper->{'VARS'}->{'curseed'}) {
	    $rndseed=$helper->{'VARS'}->{'curseed'};
	}
	for (my $i=0;$i<=$#list_of_files;$i++) {
	    my $urlp = $list_of_files[$i];
	    $urlp=~s|//|/|;
	    if ($urlp=~/\//) {
		$form{'problem_split'}=$parmhash{'problem_stream_switch'};
		$form{'rndseed'}=$rndseed;
		if ($urlp =~ m|/home/([^/]+)/public_html|) {
		    $urlp =~ s|/home/([^/]*)/public_html|/~$1|;
		} else {
		    $urlp =~ s|^$Apache::lonnet::perlvar{'lonDocRoot'}||;
		}
		$resources_printed .= $urlp.':';
		my $texversion=&Apache::lonnet::ssi($urlp,%form);
		if(($helper->{'VARS'}->{'ANSWER_TYPE'} eq 'no') ||
		   ($helper->{'VARS'}->{'ANSWER_TYPE'} eq 'only')) {
		    #  Don't permanently pervert %form:
		    my %answerform = %form;
		    $answerform{'grade_target'}='answer';
		    $answerform{'answer_output_mode'}='tex';
		    $answerform{'latex_type'}=$helper->{'VARS'}->{'LATEX_TYPE'};
		    $answerform{'rndseed'}=$rndseed;
		    $resources_printed .= $urlp.':';
		    my $answer=&Apache::lonnet::ssi($urlp,%answerform);
		    if ($helper->{'VARS'}->{'ANSWER_TYPE'} eq 'no') {
			$texversion=~s/(\\keephidden{ENDOFPROBLEM})/$answer$1/;
		    } else {
			$texversion=&print_latex_header($helper->{'VARS'}->{'LATEX_TYPE'});
			if ($helper->{'VARS'}->{'construction'} ne '1') {
			    $texversion.='\vskip 0 mm \noindent ';
			    $texversion.=&path_to_problem ($urlp,$LaTeXwidth);
			} else {
			    $texversion.='\vskip 0 mm \noindent\textbf{Prints from construction space - there is no title.}\vskip 0 mm ';
			    my $URLpath=$urlp;
			    $URLpath=~s/~([^\/]+)/public_html\/$1\/$1/;
			    $texversion.=&path_to_problem ($URLpath,$LaTeXwidth);
			}
			$texversion.='\vskip 1 mm '.$answer.'\end{document}';
		    }
		}
                #this chunck is responsible for printing the path to problem
		my $newurlp=$urlp;
		if ($newurlp=~/~/) {$newurlp=~s|\/~([^\/]+)\/|\/home\/$1\/public_html\/|;}
		$newurlp=&path_to_problem($newurlp,$LaTeXwidth);
		$texversion =~ s/(\\begin{minipage}{\\textwidth})/$1 $newurlp/;
		if ($flag_latex_header_remove ne 'NO') {
		    $texversion = &latex_header_footer_remove($texversion);
		} else {
		    $texversion =~ s/\\end{document}//;
		}
		if ($helper->{'VARS'}->{'TABLE_INDEX'} eq 'yes') {
		    $texversion=&IndexCreation($texversion,$urlp);
		}
		if ($helper->{'VARS'}->{'CONSTR_RESOURSE_URL'} eq 'yes') {
		    $texversion=~s/(\\addcontentsline\{toc\}\{subsection\}\{[^\}]*\})/$1 URL: \\verb|$urlp| \\strut\\\\\\strut /;
		    
		}
		$result .= $texversion;
	    }
	    $flag_latex_header_remove = 'YES';  
	}
	if ($helper->{VARS}->{'construction'} eq '1') {$result=~s/(\\typeout)/ RANDOM SEED IS $rndseed $1/;}
	$result .= '\end{document}';      	
    }
#-------------------------------------------------------- corrections for the different page formats

    # Only post process if that has not been turned off e.g. by a raw latex resource.

    if ($do_postprocessing) {
	$result = &page_format_transformation($papersize,$laystyle,$numberofcolumns,$helper->{'VARS'}->{'PRINT_TYPE'},$result,$helper->{VARS}->{'assignment'},$helper->{'VARS'}->{'TABLE_CONTENTS'},$helper->{'VARS'}->{'TABLE_INDEX'},$selectionmade);
	$result = &latex_corrections($number_of_columns,$result,$selectionmade,
				     $helper->{'VARS'}->{'ANSWER_TYPE'});
	#if ($numberofcolumns == 1) {
	$result =~ s/\\textwidth\s*=\s*-?\d*\.?\d*\s*(cm|mm|in)/\\textwidth= $helper->{'VARS'}->{'pagesize.width'} $helper->{'VARS'}->{'pagesize.widthunit'} /;
	$result =~ s/\\textheight\s*=?\s*-?\d*\.?\d*\s*(cm|mm|in)/\\textheight $helper->{'VARS'}->{'pagesize.height'} $helper->{'VARS'}->{'pagesize.heightunit'} /;
	$result =~ s/\\evensidemargin\s*=\s*-?\d*\.?\d*\s*(cm|mm|in)/\\evensidemargin= $helper->{'VARS'}->{'pagesize.lmargin'} $helper->{'VARS'}->{'pagesize.lmarginunit'} /;
	$result =~ s/\\oddsidemargin\s*=\s*-?\d*\.?\d*\s*(cm|mm|in)/\\oddsidemargin= $helper->{'VARS'}->{'pagesize.lmargin'} $helper->{'VARS'}->{'pagesize.lmarginunit'} /;
	#}
    }

#-- writing .tex file in prtspool 
    my $temp_file;
    my $identifier = &Apache::loncommon::get_cgi_id();
    my $filename = "/home/httpd/prtspool/$env{'user.name'}_$env{'user.domain'}_printout_$identifier.tex";
    if (!($#print_array>0)) { 
	unless ($temp_file = Apache::File->new('>'.$filename)) {
	    $r->log_error("Couldn't open $filename for output $!");
	    return SERVER_ERROR; 
	}
	print $temp_file $result;
	my $begin=index($result,'\begin{document}',0);
	my $inc=substr($result,0,$begin+16);
    } else {
	my $begin=index($result,'\begin{document}',0);
	my $inc=substr($result,0,$begin+16);
        for (my $i=0;$i<=$#print_array;$i++) {
	    if ($i==0) {
		$print_array[$i]=$result;
	    } else {
		$print_array[$i].='\end{document}';
		$print_array[$i] = 
		    &latex_corrections($number_of_columns,$print_array[$i],
				       $selectionmade, 
				       $helper->{'VARS'}->{'ANSWER_TYPE'});

		my $anobegin=index($print_array[$i],'\setcounter{page}',0);
		substr($print_array[$i],0,$anobegin)='';
		$print_array[$i]=$inc.$print_array[$i];
	    }
	    my $temp_file;
	    my $newfilename=$filename;
	    my $num=$i+1;
	    $newfilename =~s/\.tex$//; 
	    $newfilename=sprintf("%s_%03d.tex",$newfilename, $num);
	    unless ($temp_file = Apache::File->new('>'.$newfilename)) {
		$r->log_error("Couldn't open $newfilename for output $!");
		return SERVER_ERROR; 
	    }
	    print $temp_file $print_array[$i];
	}
    }
    my $student_names='';
    if ($#print_array>0) {
	for (my $i=0;$i<=$#print_array;$i++) {
	    $student_names.=$student_names[$i].'_ENDPERSON_';
	}
    } else {
	if ($#student_names>-1) {
	    $student_names=$student_names[0].'_ENDPERSON_';
	} else {
	    my $fullname = &get_name($env{'user.name'},$env{'user.domain'});
	    $student_names=join(':',$env{'user.name'},$env{'user.domain'},
				$env{'request.course.sec'},$fullname).
				    '_ENDPERSON_'.'_END_';
	}
    }

    my $URLback=''; #link to original document
    if ($helper->{'VARS'}->{'construction'} ne '1') {
	#prints published resource
	$URLback=&escape('/adm/flip?postdata=return:');
    } else {
	#prints resource from the construction space
	$URLback='/'.$helper->{'VARS'}->{'filename'};
	if ($URLback=~/([^?]+)/) {
	    $URLback=$1;
	    $URLback=~s|^/~|/priv/|;
	}
    }
    # logic for now is too complex to trace if this has been defined
    #  yet.
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    &Apache::lonnet::appenv('cgi.'.$identifier.'.file'   => $filename,
                            'cgi.'.$identifier.'.layout'  => $laystyle,
                            'cgi.'.$identifier.'.numcol'  => $numberofcolumns,
			    'cgi.'.$identifier.'.paper'  => $papersize,
                            'cgi.'.$identifier.'.selection' => $selectionmade,
			    'cgi.'.$identifier.'.tableofcontents' => $helper->{'VARS'}->{'TABLE_CONTENTS'},
			    'cgi.'.$identifier.'.tableofindex' => $helper->{'VARS'}->{'TABLE_INDEX'},
			    'cgi.'.$identifier.'.role' => $perm{'pav'},
                            'cgi.'.$identifier.'.numberoffiles' => $#print_array,
                            'cgi.'.$identifier.'.studentnames' => $student_names,
                            'cgi.'.$identifier.'.backref' => $URLback,);
    &Apache::lonnet::appenv("cgi.$identifier.user"    => $env{'user.name'},
			    "cgi.$identifier.domain"  => $env{'user.domain'},
			    "cgi.$identifier.courseid" => $cnum, 
			    "cgi.$identifier.coursedom" => $cdom, 
			    "cgi.$identifier.resources" => $resources_printed);

    my $end_page = &Apache::loncommon::end_page();
    $r->print(<<FINALEND);
<br />
<meta http-equiv="Refresh" content="0; url=/cgi-bin/printout.pl?$identifier" />
<a href="/cgi-bin/printout.pl?$identifier">Continue</a>
$end_page
FINALEND
}


sub get_CODE {
    my ($all_codes,$num,$seed,$size,$type)=@_;
    my $max='1'.'0'x$size;
    my $newcode;
    while(1) {
	$newcode=sprintf("%0".$size."d",int(rand($max)));
	if (!exists($$all_codes{$newcode})) {
	    $$all_codes{$newcode}=1;
	    if ($type eq 'number' ) {
		return $newcode;
	    } else {
		return &num_to_letters($newcode);
	    }
	}
    }
}

sub print_resources {
    my ($r,$helper,$person,$type,$moreenv,$master_seq,$remove_latex_header,
	$LaTeXwidth)=@_;
    my $current_output = ''; 
    my $printed = '';
    my ($username,$userdomain,$usersection) = split /:/,$person;
    my $fullname = &get_name($username,$userdomain);
    my $namepostfix = "\\\\";	# Both anon and not anon should get the same vspace.
    if ($person =~ 'anon') {
	$namepostfix .="Name: ";
	$fullname = "CODE - ".$moreenv->{'CODE'};
    }
    #  Fullname may have special latex characters that need \ prefixing:
    #

    my $i           = 0;
    #goes through all resources, checks if they are available for 
    #current student, and produces output   

    &Apache::lonxml::clear_problem_counter();
    my %page_breaks  = &get_page_breaks($helper);
    my $columns_in_format = (split(/\|/,$helper->{'VARS'}->{'FORMAT'}))[1];
    #
    #   end each student with a 
    #   Special that allows the post processor to even out the page
    #   counts later.  Nasty problem this... it would be really
    #   nice to put the special in as a postscript comment
    #   e.g. \special{ps:\ENDOFSTUDENTSTAMP}  unfortunately,
    #   The special gets passed the \ and dvips puts it in the output file
    #   so we will just rely on prntout.pl to strip  ENDOFSTUDENTSTAMP from the
    #   postscript.  Each ENDOFSTUDENTSTAMP will go on a line by itself.
    #

    foreach my $curresline (@{$master_seq})  {
	if (defined $page_breaks{$curresline}) {
	    if($i != 0) {
		$current_output.= "\\newpage\n";
	    }
	}
	$i++;
	if ( !($type eq 'problems' && 
	       ($curresline!~ m/\.(problem|exam|quiz|assess|survey|form|library)$/)) ) {
	    my ($map,$id,$res_url) = &Apache::lonnet::decode_symb($curresline);
	    if (&Apache::lonnet::allowed('bre',$res_url)) {
		if ($res_url!~m|^ext/|
		    && $res_url=~/\.(problem|exam|quiz|assess|survey|form|library|page|xml|html|htm|xhtml|xhtm)$/) {
		    $printed .= $curresline.':';

		    &Apache::lonxml::remember_problem_counter();    

		    my $rendered = &Apache::loncommon::get_student_view($curresline,$username,$userdomain,$env{'request.course.id'},'tex',$moreenv);

		    if(($helper->{'VARS'}->{'ANSWER_TYPE'} eq 'no') ||
		       ($helper->{'VARS'}->{'ANSWER_TYPE'} eq 'only')) {
			#   Use a copy of the hash so we don't pervert it on future loop passes.
			my %answerenv = %{$moreenv};
			$answerenv{'answer_output_mode'}='tex';
			$answerenv{'latex_type'}=$helper->{'VARS'}->{'LATEX_TYPE'};
			
			&Apache::lonxml::restore_problem_counter();

			my $ansrendered = &Apache::loncommon::get_student_answers($curresline,$username,$userdomain,$env{'request.course.id'},%answerenv);

			if ($helper->{'VARS'}->{'ANSWER_TYPE'} eq 'no') {
			    $rendered=~s/(\\keephidden{ENDOFPROBLEM})/$ansrendered$1/;
			} else {

			    
			    my $header =&print_latex_header($helper->{'VARS'}->{'LATEX_TYPE'});
			    my $title = &Apache::lonnet::gettitle($curresline);
			    $title = &Apache::lonxml::latex_special_symbols($title);
			    my $body   ='\vskip 0 mm \noindent\textbf{'.$title.'}\vskip 0 mm ';
			    $body     .=&path_to_problem($res_url,$LaTeXwidth);
			    $body     .='\vskip 1 mm '.$ansrendered;
			    $body     = &encapsulate_minipage($body);
			    $rendered = $header.$body;
			}
		    }
		    if ($remove_latex_header eq 'YES') {
			$rendered = &latex_header_footer_remove($rendered);
		    } else {
			$rendered =~ s/\\end{document}//;
		    }
		    $current_output .= $rendered;		    
		} elsif ($res_url=~/\/(smppg|syllabus|aboutme|bulletinboard)$/) {
		    $printed .= $curresline.':';
		    my $rendered = &Apache::loncommon::get_student_view($curresline,$username,$userdomain,$env{'request.course.id'},'tex',$moreenv);

		    if ($remove_latex_header eq 'YES') {
			$rendered = &latex_header_footer_remove($rendered);
		    } else {
			$rendered =~ s/\\end{document}//;
		    }
		    $current_output .= $rendered.'\vskip 0.5mm\noindent\makebox[\textwidth/$number_of_columns][b]{\hrulefill}\strut \vskip 0 mm \strut ';

		} else {
		    my $rendered = &unsupported($res_url,$helper->{'VARS'}->{'LATEX_TYPE'},$curresline);
		    if ($remove_latex_header ne 'NO') {
			$rendered = &latex_header_footer_remove($rendered);
		    } else {
			$rendered =~ s/\\end{document}//;
		    }
		    $current_output .= $rendered;
		}
	    }
	    $remove_latex_header = 'YES';
	}
	if (&Apache::loncommon::connection_aborted($r)) { last; }
    }
    my $courseidinfo = &get_course();
    if (defined($courseidinfo)) { $courseidinfo=' - '.$courseidinfo }
    if ($usersection ne '') {$courseidinfo.=' - Sec. '.$usersection}
    my $currentassignment=&Apache::lonxml::latex_special_symbols($helper->{VARS}->{'assignment'},'header');
    my $header_line =
	&format_page_header($LaTeXwidth, $parmhash{'print_header_format'},
			    $currentassignment, $courseidinfo, $fullname);
    my $header_start = ($columns_in_format == 1) ? '\lhead'
	                                         : '\fancyhead[LO]';
    $header_line = $header_start.'{'.$header_line.'}';

    if ($current_output=~/\\documentclass/) {
	$current_output =~ s/\\begin{document}/\\setlength{\\topmargin}{1cm} \\begin{document}\\noindent\\parbox{\\minipagewidth}{\\noindent$header_line$namepostfix}\\vskip 5 mm /;
    } else {
	my $blankpages = 
	    '\clearpage\strut\clearpage'x$helper->{'VARS'}->{'EMPTY_PAGES'};
	    
	$current_output = '\strut\vspace*{-6 mm}\\newline'.
	    &copyright_line().' \newpage '.$blankpages.$end_of_student.
	    '\setcounter{page}{1}\noindent\parbox{\minipagewidth}{\noindent'.
	    $header_line.$namepostfix.'} \vskip 5 mm '.$current_output;
    }
    #
    #  Close the student bracketing.
    #
    return ($current_output,$fullname, $printed);

}

sub handler {

    my $r = shift;
    
    &init_perm();



    my $helper = printHelper($r);
    if (!ref($helper)) {
	return $helper;
    }
   

    %parmhash=&Apache::lonnet::coursedescription($env{'request.course.id'});
 



    #  If a figure conversion queue file exists for this user.domain
    # we delete it since it can only be bad (if it were good, printout.pl
    # would have deleted it the last time around.

    my $conversion_queuefile = "/home/httpd/prtspool/$env{'user.name'}_$env{'user.domain'}_printout.dat";
    if(-e $conversion_queuefile) {
	unlink $conversion_queuefile;
    }
    &output_data($r,$helper,\%parmhash);
    return OK;
} 

use Apache::lonhelper;

sub addMessage {
    my $text = shift;
    my $paramHash = Apache::lonhelper::getParamHash();
    $paramHash->{MESSAGE_TEXT} = $text;
    Apache::lonhelper::message->new();
}



sub init_perm {
    undef(%perm);
    $perm{'pav'}=&Apache::lonnet::allowed('pav',$env{'request.course.id'});
    if (!$perm{'pav'}) {
	$perm{'pav'}=&Apache::lonnet::allowed('pav',
		  $env{'request.course.id'}.'/'.$env{'request.course.sec'});
    }
    $perm{'pfo'}=&Apache::lonnet::allowed('pfo',$env{'request.course.id'});
    if (!$perm{'pfo'}) {
	$perm{'pfo'}=&Apache::lonnet::allowed('pfo',
		  $env{'request.course.id'}.'/'.$env{'request.course.sec'});
    }
}

sub printHelper {
    my $r = shift;

    if ($r->header_only) {
        if ($env{'browser.mathml'}) {
            &Apache::loncommon::content_type($r,'text/xml');
        } else {
            &Apache::loncommon::content_type($r,'text/html');
        }
        $r->send_http_header;
        return OK;
    }

    # Send header, nocache
    if ($env{'browser.mathml'}) {
        &Apache::loncommon::content_type($r,'text/xml');
    } else {
        &Apache::loncommon::content_type($r,'text/html');
    }
    &Apache::loncommon::no_cache($r);
    $r->send_http_header;
    $r->rflush();

    # Unfortunately, this helper is so complicated we have to
    # write it by hand

    Apache::loncommon::get_unprocessed_cgi($ENV{QUERY_STRING});
    
    my $helper = Apache::lonhelper::helper->new("Printing Helper");
    $helper->declareVar('symb');
    $helper->declareVar('postdata');    
    $helper->declareVar('curseed'); 
    $helper->declareVar('probstatus');   
    $helper->declareVar('filename');
    $helper->declareVar('construction');
    $helper->declareVar('assignment');
    $helper->declareVar('style_file');
    $helper->declareVar('student_sort');
    $helper->declareVar('FINISHPAGE');
    $helper->declareVar('PRINT_TYPE');
    $helper->declareVar("showallfoils");
    $helper->declareVar("STUDENTS");

    #  The page breaks can get loaded initially from the course environment:
    # But we only do this in the initial state so that they are allowed to change.
    #

    # $helper->{VARS}->{FINISHPAGE} = '';
    
    &Apache::loncommon::restore_course_settings('print',
						{'pagebreaks'  => 'scalar',
					         'lastprinttype' => 'scalar'});
    
    # This will persistently load in the data we want from the
    # very first screen.
    
    if($helper->{VARS}->{PRINT_TYPE} eq $env{'form.lastprinttype'}) {
	if (!defined ($env{"form.CURRENT_STATE"})) {
	    
	    $helper->{VARS}->{FINISHPAGE} = $env{'form.pagebreaks'};
	} else {
	    my $state = $env{"form.CURRENT_STATE"};
	    if ($state eq "START") {
		$helper->{VARS}->{FINISHPAGE} = $env{'form.pagebreaks'};
	    }
	}
	
    }


    # Detect whether we're coming from construction space
    if ($env{'form.postdata'}=~/^(?:http:\/\/[^\/]+\/|\/|)\~([^\/]+)\/(.*)$/) {
        $helper->{VARS}->{'filename'} = "~$1/$2";
        $helper->{VARS}->{'construction'} = 1;
    } else {
        if ($env{'form.postdata'}) {
            $helper->{VARS}->{'symb'} = &Apache::lonnet::symbread($env{'form.postdata'});
	    if ( $helper->{VARS}->{'symb'} eq '') {
		$helper->{VARS}->{'postdata'} = $env{'form.postdata'};
	    }
        }
        if ($env{'form.symb'}) {
            $helper->{VARS}->{'symb'} = $env{'form.symb'};
        }
        if ($env{'form.url'}) {
            $helper->{VARS}->{'symb'} = &Apache::lonnet::symbread($helper->{VARS}->{'postdata'});
        }

    }

    if ($env{'form.symb'}) {
        $helper->{VARS}->{'symb'} = $env{'form.symb'};
    }
    if ($env{'form.url'}) {
        $helper->{VARS}->{'symb'} = &Apache::lonnet::symbread($helper->{VARS}->{'postdata'});

    }
    $helper->{VARS}->{'symb'}=
	&Apache::lonenc::check_encrypt($helper->{VARS}->{'symb'});
    my ($resourceTitle,$sequenceTitle,$mapTitle) = &details_for_menu($helper);
    if ($sequenceTitle ne '') {$helper->{VARS}->{'assignment'}=$sequenceTitle;}

    
    # Extract map
    my $symb = $helper->{VARS}->{'symb'};
    my ($map, $id, $url);
    my $subdir;
    my $is_published=0;		# True when printing from resource space.

    # Get the resource name from construction space
    if ($helper->{VARS}->{'construction'}) {
        $resourceTitle = substr($helper->{VARS}->{'filename'}, 
                                rindex($helper->{VARS}->{'filename'}, '/')+1);
        $subdir = substr($helper->{VARS}->{'filename'},
                         0, rindex($helper->{VARS}->{'filename'}, '/') + 1);
    } else {
	if ($symb ne '') {
	    ($map, $id, $url) = &Apache::lonnet::decode_symb($symb);
	    $helper->{VARS}->{'postdata'} = 
		&Apache::lonenc::check_encrypt(&Apache::lonnet::clutter($url));
	} else {
	    $url = $helper->{VARS}->{'postdata'};
	    $is_published=1;	# From resource space.
	}
	$url = &Apache::lonnet::clutter($url);

        if (!$resourceTitle) { # if the resource doesn't have a title, use the filename
            my $postdata = $helper->{VARS}->{'postdata'};
            $resourceTitle = substr($postdata, rindex($postdata, '/') + 1);
        }
        $subdir = &Apache::lonnet::filelocation("", $url);
    }
    if (!$helper->{VARS}->{'curseed'} && $env{'form.curseed'}) {
	$helper->{VARS}->{'curseed'}=$env{'form.curseed'};
    }
    if (!$helper->{VARS}->{'probstatus'} && $env{'form.problemtype'}) {
	$helper->{VARS}->{'probstatus'}=$env{'form.problemtype'};
    }

    my $userCanSeeHidden = Apache::lonnavmaps::advancedUser();

    Apache::lonhelper::registerHelperTags();

    # "Delete everything after the last slash."
    $subdir =~ s|/[^/]+$||;

    # What can be printed is a very dynamic decision based on
    # lots of factors. So we need to dynamically build this list.
    # To prevent security leaks, states are only added to the wizard
    # if they can be reached, which ensures manipulating the form input
    # won't allow anyone to reach states they shouldn't have permission
    # to reach.

    # printChoices is tracking the kind of printing the user can
    # do, and will be used in a choices construction later.
    # In the meantime we will be adding states and elements to
    # the helper by hand.
    my $printChoices = [];
    my $paramHash;

    if ($resourceTitle) {
        push @{$printChoices}, ["<b><i>$resourceTitle</i></b> (".&mt('the resource you just saw on the screen').")", 'current_document', 'PAGESIZE'];
    }

    # Useful filter strings
    my $isProblem = '($res->is_problem()||$res->contains_problem) ';
    $isProblem .= ' && !$res->randomout()' if !$userCanSeeHidden;
    my $isProblemOrMap = '$res->is_problem() || $res->contains_problem() || $res->is_sequence()';
    my $isNotMap = '!$res->is_sequence()';
    $isNotMap .= ' && !$res->randomout()' if !$userCanSeeHidden;
    my $isMap = '$res->is_map()';
    my $symbFilter = '$res->shown_symb()';
    my $urlValue = '$res->link()';

    $helper->declareVar('SEQUENCE');

    # If we're in a sequence...

    my $start_new_option;
    if ($perm{'pav'}) {
	$start_new_option = 
	    "<option text='".&mt('Start new page<br />before selected').
	    "' variable='FINISHPAGE' />";
    }

    if (($helper->{'VARS'}->{'construction'} ne '1' ) &&

	$helper->{VARS}->{'postdata'} &&
	$helper->{VARS}->{'assignment'}) {
        # Allow problems from sequence
        push @{$printChoices}, [&mt('Selected <b>Problems</b> in folder <b><i>[_1]</i></b>',$sequenceTitle), 'map_problems', 'CHOOSE_PROBLEMS'];
        # Allow all resources from sequence
        push @{$printChoices}, [&mt('Selected <b>Resources</b> in folder <b><i>[_1]</i></b>',$sequenceTitle), 'map_problems_pages', 'CHOOSE_PROBLEMS_HTML'];

        my $helperFragment = <<HELPERFRAGMENT;
  <state name="CHOOSE_PROBLEMS" title="Select Problem(s) to print">
    <resource variable="RESOURCES" multichoice="1" toponly='1' addstatus="1"
              closeallpages="1">
      <nextstate>PAGESIZE</nextstate>
      <filterfunc>return $isProblem;</filterfunc>
      <mapurl>$map</mapurl>
      <valuefunc>return $symbFilter;</valuefunc>
      $start_new_option
      </resource>
    </state>

  <state name="CHOOSE_PROBLEMS_HTML" title="Select Resource(s) to print">
    <resource variable="RESOURCES" multichoice="1" toponly='1' addstatus="1"
              closeallpages="1">
      <nextstate>PAGESIZE</nextstate>
      <filterfunc>return $isNotMap;</filterfunc>
      <mapurl>$map</mapurl>
      <valuefunc>return $symbFilter;</valuefunc>
      $start_new_option
      </resource>
    </state>
HELPERFRAGMENT

	&Apache::lonxml::xmlparse($r, 'helper', $helperFragment);
    }

    # If the user has pfo (print for otheres) allow them to print all 
    # problems and resources  in the entier course, optionally for selected students
    if ($perm{'pfo'} &&  !$is_published  &&
        ($helper->{VARS}->{'postdata'}=~/\/res\// || $helper->{VARS}->{'postdata'}=~/\/(syllabus|smppg|aboutme|bulletinboard)$/)) { 

        push @{$printChoices}, ['Selected <b>Problems</b> from <b>entire course</b>', 'all_problems', 'ALL_PROBLEMS'];
	push @{$printChoices}, ['Selected <b>Resources</b> from <b>entire course</b>', 'all_resources', 'ALL_RESOURCES'];
         &Apache::lonxml::xmlparse($r, 'helper', <<ALL_PROBLEMS);
  <state name="ALL_PROBLEMS" title="Select Problem(s) to print">
    <resource variable="RESOURCES" toponly='0' multichoice="1"
	suppressEmptySequences='0' addstatus="1" closeallpages="1">
      <nextstate>PAGESIZE</nextstate>
      <filterfunc>return $isProblemOrMap;</filterfunc>
      <choicefunc>return $isNotMap;</choicefunc>
      <valuefunc>return $symbFilter;</valuefunc>
      $start_new_option
    </resource>
  </state>
  <state name="ALL_RESOURCES" title="Select Resource(s) to print">
    <resource variable="RESOURCES" toponly='0' multichoice='1'
              suppressEmptySequences='0' addstatus='1' closeallpages='1'>
      <nextstate>PAGESIZE</nextstate>
      <filterfunc>return $isNotMap; </filterfunc>
      <valuefunc>return $symbFilter;</valuefunc>
      $start_new_option
    </resource>
  </state>
ALL_PROBLEMS

	if ($helper->{VARS}->{'assignment'}) {
	    push @{$printChoices}, [&mt("Selected <b>Problems</b> from folder <b><i>[_1]</i></b> for <b>selected people</b>",$sequenceTitle), 'problems_for_students', 'CHOOSE_STUDENTS'];
	    push @{$printChoices}, [&mt("Selected <b>Problems</b> from folder <b><i>[_1]</i></b> for <b>CODEd assignments</b>",$sequenceTitle), 'problems_for_anon', 'CHOOSE_ANON1'];
	}

	# resource_selector will hold a few states that:
	#   - Allow resources to be selected for printing.
	#   - Determine pagination between assignments.
	#   - Determine how many assignments should be bundled into a single PDF.
        # TODO:
	#    Probably good to do things like separate this up into several vars, each
	#    with one state, and use REGEXPs at inclusion time to set state names
	#    and next states for better mix and match capability
	#
	my $resource_selector=<<RESOURCE_SELECTOR;
    <state name="SELECT_PROBLEMS" title="Select resources to print">
   <nextstate>PRINT_FORMATTING</nextstate> 
   <message><br /><big><i><b>Select resources for the assignment</b></i></big><br /></message>
    <resource variable="RESOURCES" multichoice="1" addstatus="1" 
              closeallpages="1">
      <filterfunc>return $isProblem;</filterfunc>
      <mapurl>$map</mapurl>
      <valuefunc>return $symbFilter;</valuefunc>
      $start_new_option
      </resource>
    </state>
    <state name="PRINT_FORMATTING" title="How should results be printed?">
    <message><br /><big><i><b>How should the results be printed?</b></i></big><br /></message>
    <choices variable="EMPTY_PAGES">
      <choice computer='0'>Start each student\'s assignment on a new page/column (add a pagefeed after each assignment)</choice>
      <choice computer='1'>Add one empty page/column after each student\'s assignment</choice>
      <choice computer='2'>Add two empty pages/column after each student\'s assignment</choice>
      <choice computer='3'>Add three empty pages/column after each student\'s assignment</choice>
    </choices>
    <nextstate>PAGESIZE</nextstate>
    <message><hr width='33%' /><b>How do you want assignments split into PDF files? </b></message>
    <choices variable="SPLIT_PDFS">
       <choice computer="all">All assignments in a single PDF file</choice>
       <choice computer="sections">Each PDF contains exactly one section</choice>
       <choice computer="oneper">Each PDF contains exactly one assignment</choice>
       <choice computer="usenumber" relatedvalue="NUMBER_TO_PRINT">
            Specify the number of assignments per PDF:</choice>
    </choices>
    </state>
RESOURCE_SELECTOR

        &Apache::lonxml::xmlparse($r, 'helper', <<CHOOSE_STUDENTS);
  <state name="CHOOSE_STUDENTS" title="Select Students and Resources">
      <message><b>Select sorting order of printout</b> </message>
    <choices variable='student_sort'>
      <choice computer='0'>Sort by section then student</choice>
      <choice computer='1'>Sort by students across sections.</choice>
    </choices>
      <message><br /><hr /><br /> </message>
      <student multichoice='1' variable="STUDENTS" nextstate="SELECT_PROBLEMS" coursepersonnel="1"/>
  </state>
    $resource_selector
CHOOSE_STUDENTS

	my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
	my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
        my @names=&Apache::lonnet::getkeys('CODEs',$cdom,$cnum);
	my $namechoice='<choice></choice>';
	foreach my $name (sort {uc($a) cmp uc($b)} @names) {
	    if ($name =~ /^error: 2 /) { next; }
	    if ($name =~ /^type\0/) { next; }
	    $namechoice.='<choice computer="'.$name.'">'.$name.'</choice>';
	}


	my %code_values;
	my %codes_to_print;
	foreach my $key (@names) {
	    %code_values = &Apache::grades::get_codes($key, $cdom, $cnum);
	    foreach my $key (keys(%code_values)) {
		$codes_to_print{$key} = 1;
	    }
	}

	my $code_selection;
	foreach my $code (sort {uc($a) cmp uc($b)} (keys(%codes_to_print))) {
	    my $choice  = $code;
	    if ($code =~ /^[A-Z]+$/) { # Alpha code
		$choice = &letters_to_num($code);
	    }
	    push(@{$helper->{DATA}{ALL_CODE_CHOICES}},[$code,$choice]);
	}
	if (%codes_to_print) {
	    $code_selection .='   
	    <message><b>Choose single CODE from list:</b></message>
		<message></td><td></message>
		<dropdown variable="CODE_SELECTED_FROM_LIST" multichoice="0" allowempty="0">
                  <choice></choice>
                  <exec>
                     push(@{$state->{CHOICES}},@{$helper->{DATA}{ALL_CODE_CHOICES}});
                  </exec>
		</dropdown>
	    <message></td></tr><tr><td></message>
            '.$/;

	}

	
	open(FH,$Apache::lonnet::perlvar{'lonTabDir'}.'/scantronformat.tab');
	my $codechoice='';
	foreach my $line (<FH>) {
	    my ($name,$description,$code_type,$code_length)=
		(split(/:/,$line))[0,1,2,4];
	    if ($code_length > 0 && 
		$code_type =~/^(letter|number|-1)/) {
		$codechoice.='<choice computer="'.$name.'">'.$description.'</choice>';
	    }
	}
	if ($codechoice eq '') {
	    $codechoice='<choice computer="default">Default</choice>';
	}
        &Apache::lonxml::xmlparse($r, 'helper', <<CHOOSE_ANON1);
  <state name="CHOOSE_ANON1" title="Specify CODEd Assignments">
    <nextstate>SELECT_PROBLEMS</nextstate>
    <message><h4>Fill out one of the forms below</h4></message>
    <message><br /><hr /> <br /></message>
    <message><h3>Generate new CODEd Assignments</h3></message>
    <message><table><tr><td><b>Number of CODEd assignments to print:</b></td><td></message>
    <string variable="NUMBER_TO_PRINT_TOTAL" maxlength="5" size="5">
       <validator>
	if (((\$helper->{'VARS'}{'NUMBER_TO_PRINT_TOTAL'}+0) < 1) &&
	    !\$helper->{'VARS'}{'REUSE_OLD_CODES'}                &&
            !\$helper->{'VARS'}{'SINGLE_CODE'}                    &&
	    !\$helper->{'VARS'}{'CODE_SELECTED_FROM_LIST'}) {
	    return "You need to specify the number of assignments to print";
	}
	return undef;
       </validator>
    </string>
    <message></td></tr><tr><td></message>
    <message><b>Names to save the CODEs under for later:</b></message>
    <message></td><td></message>
    <string variable="ANON_CODE_STORAGE_NAME" maxlength="50" size="20" />
    <message></td></tr><tr><td></message>
    <message><b>Bubble sheet type:</b></message>
    <message></td><td></message>
    <dropdown variable="CODE_OPTION" multichoice="0" allowempty="0">
    $codechoice
    </dropdown>
    <message></td></tr><tr><td colspan="2"></td></tr><tr><td></message>
    <message></td></tr><tr><td></table></message>
    <message><br /><hr /><h3>Print a Specific CODE </h3><br /><table></message>
    <message><tr><td><b>Enter a CODE to print:</b></td><td></message>
    <string variable="SINGLE_CODE" size="10">
        <validator>
	   if(!\$helper->{'VARS'}{'NUMBER_TO_PRINT_TOTAL'}           &&
	      !\$helper->{'VARS'}{'REUSE_OLD_CODES'}                 &&
	      !\$helper->{'VARS'}{'CODE_SELECTED_FROM_LIST'}) {
	      return &Apache::lonprintout::is_code_valid(\$helper->{'VARS'}{'SINGLE_CODE'},
						      \$helper->{'VARS'}{'CODE_OPTION'});
	   } else {
	       return undef;	# Other forces control us.
	   }
        </validator>
    </string>
    <message></td></tr><tr><td></message>
        $code_selection
    <message></td></tr></table></message>
    <message><hr /><h3>Reprint a Set of Saved CODEs</h3><table><tr><td></message>
    <message><b>Select saved CODEs:</b></message>
    <message></td><td></message>
    <dropdown variable="REUSE_OLD_CODES">
        $namechoice
    </dropdown>
    <message></td></tr></table></message>
  </state>
  $resource_selector
CHOOSE_ANON1


	if ($helper->{VARS}->{'assignment'}) {
	    push @{$printChoices}, [&mt("Selected <b>Resources</b> from folder <b><i>[_1]</i></b> for <b>selected people</b>",$sequenceTitle), 'resources_for_students', 'CHOOSE_STUDENTS1'];
	    push @{$printChoices}, [&mt("Selected <b>Resources</b> from folder <b><i>[_1]</i></b> for <b>CODEd assignments</b>",$sequenceTitle), 'resources_for_anon', 'CHOOSE_ANON2'];
	}
	    

	$resource_selector=<<RESOURCE_SELECTOR;
    <state name="SELECT_RESOURCES" title="Select Resources">
    <nextstate>PRINT_FORMATTING</nextstate>
    <message><br /><big><i><b>Select resources for the assignment</b></i></big><br /></message>
    <resource variable="RESOURCES" multichoice="1" addstatus="1" 
              closeallpages="1">
      <filterfunc>return $isNotMap;</filterfunc>
      <mapurl>$map</mapurl>
      <valuefunc>return $symbFilter;</valuefunc>
      $start_new_option
      </resource>
    </state>
    <state name="PRINT_FORMATTING" title="Format of the print job">
    <nextstate>NUMBER_PER_PDF</nextstate>
    <message><br /><big><i><b>How should the results be printed?</b></i></big><br /></message>
    <choices variable="EMPTY_PAGES">
      <choice computer='0'>Start each student\'s assignment on a new page/column (add a pagefeed after each assignment)</choice>
      <choice computer='1'>Add one empty page/column after each student\'s assignment</choice>
      <choice computer='2'>Add two empty pages/column after each student\'s assignment</choice>
      <choice computer='3'>Add three empty pages/column after each student\'s assignment</choice>
    </choices>
    <nextstate>PAGESIZE</nextstate>
    <message><hr width='33%' /><b>How do you want assignments split into PDF files? </b></message>
    <choices variable="SPLIT_PDFS">
       <choice computer="all">All assignments in a single PDF file</choice>
       <choice computer="sections">Each PDF contains exactly one section</choice>
       <choice computer="oneper">Each PDF contains exactly one assignment</choice>
       <choice computer="usenumber" relatedvalue="NUMBER_TO_PRINT">
           Specify the number of assignments per PDF:</choice>
    </choices>
    </state>
RESOURCE_SELECTOR

	&Apache::lonxml::xmlparse($r, 'helper', <<CHOOSE_STUDENTS1);
  <state name="CHOOSE_STUDENTS1" title="Select Students and Resources">
    <choices variable='student_sort'>
      <choice computer='0'>Sort by section then student</choice>
      <choice computer='1'>Sort by students across sections.</choice>
    </choices>
    <message><br /><hr /><br /></message>
    <student multichoice='1' variable="STUDENTS" nextstate="SELECT_RESOURCES" coursepersonnel="1" />

    </state>
    $resource_selector
CHOOSE_STUDENTS1

	&Apache::lonxml::xmlparse($r, 'helper', <<CHOOSE_ANON2);
  <state name="CHOOSE_ANON2" title="Select CODEd Assignments">
    <nextstate>SELECT_RESOURCES</nextstate>
    <message><h4>Fill out one of the forms below</h4></message>
    <message><br /><hr /> <br /></message>
    <message><h3>Generate new CODEd Assignments</h3></message>
    <message><table><tr><td><b>Number of CODEd assignments to print:</b></td><td></message>
    <string variable="NUMBER_TO_PRINT_TOTAL" maxlength="5" size="5">
       <validator>
	if (((\$helper->{'VARS'}{'NUMBER_TO_PRINT_TOTAL'}+0) < 1) &&
	    !\$helper->{'VARS'}{'REUSE_OLD_CODES'}                &&
	    !\$helper->{'VARS'}{'SINGLE_CODE'}                   &&
	    !\$helper->{'VARS'}{'CODE_SELECTED_FROM_LIST'}) {
	    return "You need to specify the number of assignments to print";
	}
	return undef;
       </validator>
    </string>
    <message></td></tr><tr><td></message>
    <message><b>Names to save the CODEs under for later:</b></message>
    <message></td><td></message>
    <string variable="ANON_CODE_STORAGE_NAME" maxlength="50" size="20" />
    <message></td></tr><tr><td></message>
    <message><b>Bubble sheet type:</b></message>
    <message></td><td></message>
    <dropdown variable="CODE_OPTION" multichoice="0" allowempty="0">
    $codechoice
    </dropdown>
    <message></td></tr><tr><td></table></message>
    <message><br /><hr /><h3>Print a Specific CODE </h3><br /><table></message>
    <message><tr><td><b>Enter a CODE to print:</b></td><td></message>
    <string variable="SINGLE_CODE" size="10">
        <validator>
	   if(!\$helper->{'VARS'}{'NUMBER_TO_PRINT_TOTAL'}           &&
	      !\$helper->{'VARS'}{'REUSE_OLD_CODES'}                 &&
	      !\$helper->{'VARS'}{'CODE_SELECTED_FROM_LIST'}) {
	      return &Apache::lonprintout::is_code_valid(\$helper->{'VARS'}{'SINGLE_CODE'},
						      \$helper->{'VARS'}{'CODE_OPTION'});
	   } else {
	       return undef;	# Other forces control us.
	   }
        </validator>
    </string>
    <message></td></tr><tr><td></message>
        $code_selection
    <message></td></tr></table></message>
    <message><hr /><h3>Reprint a Set of Saved CODEs</h3><table><tr><td></message>
    <message><b>Select saved CODEs:</b></message>
    <message></td><td></message>
    <dropdown variable="REUSE_OLD_CODES">
        $namechoice
    </dropdown>
    <message></td></tr></table></message>
  </state>
    $resource_selector
CHOOSE_ANON2
    }

    # FIXME: That RE should come from a library somewhere.
    if (($perm{'pav'} 
	&& $subdir ne $Apache::lonnet::perlvar{'lonDocRoot'}.'/res/'
	&& (defined($helper->{'VARS'}->{'construction'})
	    ||
	    (&Apache::lonnet::allowed('bre',$subdir) eq 'F'
	     && 
	     $helper->{VARS}->{'postdata'}=~/\.(problem|exam|quiz|assess|survey|form|library|page|xml|html|htm|xhtml|xhtm)/)
	    )) 
	&& $helper->{VARS}->{'assignment'} eq ""
	) {

	my $pretty_dir = &Apache::lonnet::hreflocation($subdir);
        push @{$printChoices}, [&mt("Selected <b>Problems</b> from current subdirectory <b><i>[_1]</i></b>",$pretty_dir), 'problems_from_directory', 'CHOOSE_FROM_SUBDIR'];
        my $xmlfrag = <<CHOOSE_FROM_SUBDIR;
  <state name="CHOOSE_FROM_SUBDIR" title="Select File(s) from <b><small>$pretty_dir</small></b> to print">

    <files variable="FILES" multichoice='1'>
      <nextstate>PAGESIZE</nextstate>
      <filechoice>return '$subdir';</filechoice>
CHOOSE_FROM_SUBDIR
        
        # this is broken up because I really want interpolation above,
        # and I really DON'T want it below
        $xmlfrag .= <<'CHOOSE_FROM_SUBDIR';
      <filefilter>return Apache::lonhelper::files::not_old_version($filename) &&
	  $filename =~ m/\.(problem|exam|quiz|assess|survey|form|library)$/;
      </filefilter>
      </files>
    </state>
CHOOSE_FROM_SUBDIR
        &Apache::lonxml::xmlparse($r, 'helper', $xmlfrag);
    }

    # Allow the user to select any sequence in the course, feed it to
    # another resource selector for that sequence
    if (!$helper->{VARS}->{'construction'} && !$is_published) {
	push @$printChoices, ["Selected <b>Resources</b> from <b>selected folder</b> in course",
			      'select_sequences', 'CHOOSE_SEQUENCE'];
	my $escapedSequenceName = $helper->{VARS}->{'SEQUENCE'};
	#Escape apostrophes and backslashes for Perl
	$escapedSequenceName =~ s/\\/\\\\/g;
	$escapedSequenceName =~ s/'/\\'/g;
	&Apache::lonxml::xmlparse($r, 'helper', <<CHOOSE_FROM_ANY_SEQUENCE);
  <state name="CHOOSE_SEQUENCE" title="Select Sequence To Print From">
    <message>Select the sequence to print resources from:</message>
    <resource variable="SEQUENCE">
      <nextstate>CHOOSE_FROM_ANY_SEQUENCE</nextstate>
      <filterfunc>return \$res->is_sequence;</filterfunc>
      <valuefunc>return $urlValue;</valuefunc>
      <choicefunc>return \$res->hasResource(\$res,sub { return !\$_[0]->is_sequence() },0,0);
	</choicefunc>
      </resource>
    </state>
  <state name="CHOOSE_FROM_ANY_SEQUENCE" title="Select Resources To Print">
    <message>(mark desired resources then click "next" button) <br /></message>
    <resource variable="RESOURCES" multichoice="1" toponly='1' addstatus="1"
              closeallpages="1">
      <nextstate>PAGESIZE</nextstate>
      <filterfunc>return $isNotMap</filterfunc>
      <mapurl evaluate='1'>return '$escapedSequenceName';</mapurl>
      <valuefunc>return $symbFilter;</valuefunc>
      $start_new_option
      </resource>
    </state>
CHOOSE_FROM_ANY_SEQUENCE
}

    # Generate the first state, to select which resources get printed.
    Apache::lonhelper::state->new("START", "Select Printing Options:");
    $paramHash = Apache::lonhelper::getParamHash();
    $paramHash->{MESSAGE_TEXT} = "";
    Apache::lonhelper::message->new();
    $paramHash = Apache::lonhelper::getParamHash();
    $paramHash->{'variable'} = 'PRINT_TYPE';
    $paramHash->{CHOICES} = $printChoices;
    Apache::lonhelper::choices->new();

    my $startedTable = 0; # have we started an HTML table yet? (need
                          # to close it later)

    if (($perm{'pav'} and &Apache::lonnet::allowed('vgr',$env{'request.course.id'})) or 
	($helper->{VARS}->{'construction'} eq '1')) {
	addMessage("<hr width='33%' /><table><tr><td align='right'>".
                   '<label for="ANSWER_TYPE_forminput">'.
                   &mt('Print').
                   "</label>: </td><td>");
        $paramHash = Apache::lonhelper::getParamHash();
	$paramHash->{'variable'} = 'ANSWER_TYPE';   
	$helper->declareVar('ANSWER_TYPE');         
        $paramHash->{CHOICES} = [
                                   ['Without Answers', 'yes'],
                                   ['With Answers', 'no'],
                                   ['Only Answers', 'only']
                                ];
        Apache::lonhelper::dropdown->new();
	addMessage("</td></tr>");
	$startedTable = 1;
    }

    if ($perm{'pav'}) {
	if (!$startedTable) {
	    addMessage("<hr width='33%' /><table><tr><td align='right'>".
                       '<label for="LATEX_TYPE_forminput">'.
                       &mt('LaTeX mode').
                       "</label>: </td><td>");
	    $startedTable = 1;
	} else {
	    addMessage("<tr><td align='right'>".
                       '<label for="LATEX_TYPE_forminput">'.
                        &mt('LaTeX mode').
                       "</label>: </td><td>");
	}
        $paramHash = Apache::lonhelper::getParamHash();
	$paramHash->{'variable'} = 'LATEX_TYPE';   
	$helper->declareVar('LATEX_TYPE');  
	if ($helper->{VARS}->{'construction'} eq '1') {       
	    $paramHash->{CHOICES} = [
				     ['standard LaTeX mode', 'standard'], 
				     ['LaTeX batchmode', 'batchmode'], ];
	} else {
	    $paramHash->{CHOICES} = [
				     ['LaTeX batchmode', 'batchmode'],
				     ['standard LaTeX mode', 'standard'] ];
	}
        Apache::lonhelper::dropdown->new();
 
	addMessage("</td></tr><tr><td align='right'>".
                   '<label for="TABLE_CONTENTS_forminput">'.
                   &mt('Print Table of Contents').
                   "</label>: </td><td>");
        $paramHash = Apache::lonhelper::getParamHash();
	$paramHash->{'variable'} = 'TABLE_CONTENTS';   
	$helper->declareVar('TABLE_CONTENTS');         
        $paramHash->{CHOICES} = [
                                   ['No', 'no'],
                                   ['Yes', 'yes'] ];
        Apache::lonhelper::dropdown->new();
	addMessage("</td></tr>");
        
	if (not $helper->{VARS}->{'construction'}) {
	    addMessage("<tr><td align='right'>".
                       '<label for="TABLE_INDEX_forminput">'.
                       &mt('Print Index').
                       "</label>: </td><td>");
	    $paramHash = Apache::lonhelper::getParamHash();
	    $paramHash->{'variable'} = 'TABLE_INDEX';   
	    $helper->declareVar('TABLE_INDEX');         
	    $paramHash->{CHOICES} = [
				     ['No', 'no'],
				     ['Yes', 'yes'] ];
	    Apache::lonhelper::dropdown->new();
	    addMessage("</td></tr>");
	    addMessage("<tr><td align='right'>".
                       '<label for="PRINT_DISCUSSIONS_forminput">'.
                       &mt('Print Discussions').
                       "</label>: </td><td>");
	    $paramHash = Apache::lonhelper::getParamHash();
	    $paramHash->{'variable'} = 'PRINT_DISCUSSIONS';   
	    $helper->declareVar('PRINT_DISCUSSIONS');         
	    $paramHash->{CHOICES} = [
				     ['No', 'no'],
				     ['Yes', 'yes'] ];
	    Apache::lonhelper::dropdown->new();
	    addMessage("</td></tr>");

	    addMessage("<tr><td align = 'right'>  </td><td>");
	    $paramHash = Apache::lonhelper::getParamHash();
	    $paramHash->{'multichoice'} = "true";
	    $paramHash->{'allowempty'}  = "true";
	    $paramHash->{'variable'}   = "showallfoils";
	    $paramHash->{'CHOICES'} = [ ["Show all foils", "1"] ];
	    Apache::lonhelper::choices->new();
	    addMessage("</td></tr>");
	}

	if ($helper->{'VARS'}->{'construction'}) { 
	    my $stylevalue='$Apache::lonnet::env{"construct.style"}';
            my $randseedtext=&mt("Use random seed");
            my $stylefiletext=&mt("Use style file");
            my $selectfiletext=&mt("Select style file");

	    my $xmlfrag .= <<"RNDSEED";
	    <message><tr><td align='right'>
            <label for="curseed_forminput">$randseedtext</label>:
            </td><td></message>
	    <string variable="curseed" size="15" maxlength="15">
		<defaultvalue>
	            return $helper->{VARS}->{'curseed'};
	        </defaultvalue>
	    </string>
	     <message></td></tr><tr><td align="right">
             <label for="style_file">$stylefiletext</label>:
             </td><td></message>
             <string variable="style_file" size="40">
		<defaultvalue>
	            return $stylevalue;
	        </defaultvalue>
             </string><message>&nbsp; <a href="javascript:openbrowser('helpform','style_file_forminput','sty')">$selectfiletext</a> </td></tr><tr><td></td><td align="left"></message>
	     <choices allowempty="1" multichoice="true" variable="showallfoils">
                <choice computer="1">Show all foils</choice>
             </choices>
	     <message></td></tr></message>
RNDSEED
            &Apache::lonxml::xmlparse($r, 'helper', $xmlfrag);
	} 
    }




    if ($startedTable) {
	addMessage("</table>");
    }

    Apache::lonprintout::page_format_state->new("FORMAT");

    # Generate the PAGESIZE state which will offer the user the margin
    # choices if they select one column
    Apache::lonhelper::state->new("PAGESIZE", "Set Margins");
    Apache::lonprintout::page_size_state->new('pagesize', 'FORMAT', 'FINAL');


    $helper->process();


    # MANUAL BAILOUT CONDITION:
    # If we're in the "final" state, bailout and return to handler
    if ($helper->{STATE} eq 'FINAL') {
        return $helper;
    }    

    $r->print($helper->display());
    if ($helper->{STATE} eq 'START') {
	&recently_generated($r);
    }
    &Apache::lonhelper::unregisterHelperTags();

    return OK;
}


1;

package Apache::lonprintout::page_format_state;

=pod

=head1 Helper element: page_format_state

See lonhelper.pm documentation for discussion of the helper framework.

Apache::lonprintout::page_format_state is an element that gives the 
user an opportunity to select the page layout they wish to print 
with: Number of columns, portrait/landscape, and paper size. If you 
want to change the paper size choices, change the @paperSize array 
contents in this package.

page_format_state is always directly invoked in lonprintout.pm, so there
is no tag interface. You actually pass parameters to the constructor.

=over 4

=item * B<new>(varName): varName is where the print information will be stored in the format FIXME.

=back

=cut

use Apache::lonhelper;

no strict;
@ISA = ("Apache::lonhelper::element");
use strict;
use Apache::lonlocal;
use Apache::lonnet;

my $maxColumns = 2;
# it'd be nice if these all worked
#my @paperSize = ("letter [8 1/2x11 in]", "legal [8 1/2x14 in]", 
#                 "tabloid (ledger) [11x17 in]", "executive [7 1/2x10 in]",
#                 "a2 [420x594 mm]", "a3 [297x420 mm]", "a4 [210x297 mm]", 
#                 "a5 [148x210 mm]", "a6 [105x148 mm]" );
my @paperSize = ("letter [8 1/2x11 in]", "legal [8 1/2x14 in]", 
		 "a4 [210x297 mm]");

# Tentative format: Orientation (L = Landscape, P = portrait) | Colnum |
#                   Paper type

sub new { 
    my $self = Apache::lonhelper::element->new();

    shift;

    $self->{'variable'} = shift;
    my $helper = Apache::lonhelper::getHelper();
    $helper->declareVar($self->{'variable'});
    bless($self);
    return $self;
}

sub render {
    my $self = shift;
    my $helper = Apache::lonhelper::getHelper();
    my $result = '';
    my $var = $self->{'variable'};
    my $PageLayout=&mt('Page layout');
    my $NumberOfColumns=&mt('Number of columns');
    my $PaperType=&mt('Paper type');
    my $landscape=&mt('Landscape');
    my $portrait=&mt('Portrait');
    $result .= <<STATEHTML;

<hr width="33%" />
<table cellpadding="3">
  <tr>
    <td align="center"><b>$PageLayout</b></td>
    <td align="center"><b>$NumberOfColumns</b></td>
    <td align="center"><b>$PaperType</b></td>
  </tr>
  <tr>
    <td>
      <label><input type="radio" name="${var}.layout" value="L" /> $landscape </label><br />
      <label><input type="radio" name="${var}.layout" value="P" checked='1'  /> $portrait </label>
    </td>
    <td align="center">
      <select name="${var}.cols">
STATEHTML

    my $i;
    for ($i = 1; $i <= $maxColumns; $i++) {
        if ($i == 2) {
            $result .= "<option value='$i' selected>$i</option>\n";
        } else {
            $result .= "<option value='$i'>$i</option>\n";
        }
    }

    $result .= "</select></td><td>\n";
    $result .= "<select name='${var}.paper'>\n";

    my %parmhash=&Apache::lonnet::coursedescription($env{'request.course.id'});
    my $DefaultPaperSize=lc($parmhash{'default_paper_size'});
    $DefaultPaperSize=~s/\s//g;
    if ($DefaultPaperSize eq '') {$DefaultPaperSize='letter';}
    $i = 0;
    foreach (@paperSize) {
	$_=~/(\w+)/;
	my $papersize=$1;
        if ($paperSize[$i]=~/$DefaultPaperSize/) {
            $result .= "<option selected value='$papersize'>" . $paperSize[$i] . "</option>\n";
        } else {
            $result .= "<option value='$papersize'>" . $paperSize[$i] . "</option>\n";
        }
        $i++;
    }
    $result .= "</select></td></tr></table>";
    return $result;
}

sub postprocess {
    my $self = shift;

    my $var = $self->{'variable'};
    my $helper = Apache::lonhelper->getHelper();
    $helper->{VARS}->{$var} = 
        $env{"form.$var.layout"} . '|' . $env{"form.$var.cols"} . '|' .
        $env{"form.$var.paper"};
    return 1;
}

1;

package Apache::lonprintout::page_size_state;

=pod

=head1 Helper element: page_size_state

See lonhelper.pm documentation for discussion of the helper framework.

Apache::lonprintout::page_size_state is an element that gives the 
user the opportunity to further refine the page settings if they
select a single-column page.

page_size_state is always directly invoked in lonprintout.pm, so there
is no tag interface. You actually pass parameters to the constructor.

=over 4

=item * B<new>(varName): varName is where the print information will be stored in the format FIXME.

=back

=cut

use Apache::lonhelper;
use Apache::lonnet;
no strict;
@ISA = ("Apache::lonhelper::element");
use strict;



sub new { 
    my $self = Apache::lonhelper::element->new();

    shift; # disturbs me (probably prevents subclassing) but works (drops
           # package descriptor)... - Jeremy

    $self->{'variable'} = shift;
    my $helper = Apache::lonhelper::getHelper();
    $helper->declareVar($self->{'variable'});

    # The variable name of the format element, so we can look into 
    # $helper->{VARS} to figure out whether the columns are one or two
    $self->{'formatvar'} = shift;


    $self->{NEXTSTATE} = shift;
    bless($self);

    return $self;
}

sub render {
    my $self = shift;
    my $helper = Apache::lonhelper::getHelper();
    my $result = '';
    my $var = $self->{'variable'};



    if (defined $self->{ERROR_MSG}) {
        $result .= '<br /><span class="LC_error">' . $self->{ERROR_MSG} . '</span><br />';
    }

    my $format = $helper->{VARS}->{$self->{'formatvar'}};

    # Use format to get sensible defaults for the margins:


    my ($laystyle, $cols, $papersize) = split(/\|/, $format);
    ($papersize)                      = split(/ /, $papersize);


    if ($laystyle eq 'L') {
	$laystyle = 'album';
    } else {
	$laystyle = 'book';
    }


    my %size;
    ($size{'width_and_units'},
     $size{'height_and_units'},
     $size{'margin_and_units'})=
	 &Apache::lonprintout::page_format($papersize, $laystyle, $cols);
    
    foreach my $dimension ('width','height','margin') {
	($size{$dimension},$size{$dimension.'_unit'}) =
	    split(/ +/, $size{$dimension.'_and_units'},2);
       	
	foreach my $unit ('cm','in') {
	    $size{$dimension.'_options'} .= '<option ';
	    if ($size{$dimension.'_unit'} eq $unit) {
		$size{$dimension.'_options'} .= 'selected="selected" ';
	    }
	    $size{$dimension.'_options'} .= '>'.$unit.'</option>';
	}
    }

    # Adjust margin for LaTeX margin: .. requires units == cm or in.

    if ($size{'margin_unit'} eq 'in') {
	$size{'margin'} += 1;
    }  else {
	$size{'margin'} += 2.54;
    }
    my %text = ('format' => 'How should each column be formatted?',
		'width'  => 'Width:',
		'height' => 'Height:',
		'margin' => 'Left Margin:',);
    %text = &Apache::lonlocal::texthash(%text);

    $result .= <<ELEMENTHTML;

<p>$text{'format'}</p>

<table cellpadding='3'>
  <tr>
    <td align='right'><b>$text{'width'}</b></td>
    <td align='left'><input type='text' name='$var.width' value="$size{'width'}" size='4' /></td>
    <td align='left'>
      <select name='$var.widthunit'>
      $size{'width_options'}
      </select>
    </td>
  </tr>
  <tr>
    <td align='right'><b>$text{'height'}</b></td>
    <td align='left'><input type='text' name="$var.height" value="$size{'height'}" size='4' /></td>
    <td align='left'>
      <select name='$var.heightunit'>
      $size{'height_options'}
      </select>
    </td>
  </tr>
  <tr>
    <td align='right'><b>$text{'margin'}</b></td>
    <td align='left'><input type='text' name='$var.lmargin' value="$size{'margin'}" size='4' /></td>
    <td align='left'>
      <select name='$var.lmarginunit'>
      $size{'margin_options'}
      </select>
    </td>
  </tr>
</table>

<!--<p>Hint: Some instructors like to leave scratch space for the student by
making the width much smaller than the width of the page.</p>-->

ELEMENTHTML

    return $result;
}


sub preprocess {
    my $self = shift;
    my $helper = Apache::lonhelper::getHelper();

    my $format = $helper->{VARS}->{$self->{'formatvar'}};

    #  If the user does not have 'pav' privilege, set default widths and
    #  on to the next state right away.
    #
    if (!$perm{'pav'}) {
	my $var = $self->{'variable'};
	my $format = $helper->{VARS}->{$self->{'formatvar'}};
	
	my ($laystyle, $cols, $papersize) = split(/\|/, $format);
	($papersize)                      = split(/ /, $papersize);
	
	
	if ($laystyle eq 'L') {
	    $laystyle = 'album';
	} else {
	    $laystyle = 'book';
	}
	#  Figure out some good defaults for the print out and set them:
	
	my %size;
	($size{'width'},
	 $size{'height'},
	 $size{'lmargin'})=
	     &Apache::lonprintout::page_format($papersize, $laystyle, $cols);
	
	foreach my $dim ('width', 'height', 'lmargin') {
	    my ($value, $units) = split(/ /, $size{$dim});
	    	    
	    $helper->{VARS}->{"$var.".$dim}      = $value;
	    $helper->{VARS}->{"$var.".$dim.'unit'} = $units;
	    
	}
	

	# Transition to the next state

	$helper->changeState($self->{NEXTSTATE});
    }
   
    return 1;
}

sub postprocess {
    my $self = shift;

    my $var = $self->{'variable'};
    my $helper = Apache::lonhelper->getHelper();
    my $width = $helper->{VARS}->{$var .'.width'} = $env{"form.${var}.width"}; 
    my $height = $helper->{VARS}->{$var .'.height'} = $env{"form.${var}.height"}; 
    my $lmargin = $helper->{VARS}->{$var .'.lmargin'} = $env{"form.${var}.lmargin"}; 
    $helper->{VARS}->{$var .'.widthunit'} = $env{"form.${var}.widthunit"}; 
    $helper->{VARS}->{$var .'.heightunit'} = $env{"form.${var}.heightunit"}; 
    $helper->{VARS}->{$var .'.lmarginunit'} = $env{"form.${var}.lmarginunit"}; 

    my $error = '';

    # /^-?[0-9]+(\.[0-9]*)?$/ -> optional minus, at least on digit, followed 
    # by an optional period, followed by digits, ending the string

    if ($width !~  /^-?[0-9]*(\.[0-9]*)?$/) {
        $error .= "Invalid width; please type only a number.<br />\n";
    }
    if ($height !~  /^-?[0-9]*(\.[0-9]*)?$/) {
        $error .= "Invalid height; please type only a number.<br />\n";
    }
    if ($lmargin !~  /^-?[0-9]*(\.[0-9]*)?$/) {
        $error .= "Invalid left margin; please type only a number.<br />\n";
    } else {
	# Adjust for LaTeX 1.0 inch margin:

	if ($env{"form.${var}.lmarginunit"} eq "in") {
	    $helper->{VARS}->{$var.'.lmargin'} = $lmargin - 1;
	} else {
	    $helper->{VARS}->{$var.'.lmargin'} = $lmargin - 2.54;
	}
    }

    if (!$error) {
        Apache::lonhelper::getHelper()->changeState($self->{NEXTSTATE});
        return 1;
    } else {
        $self->{ERROR_MSG} = $error;
        return 0;
    }
}



__END__

