# The LearningOnline Network with CAPA
# Handler to display the classlist 
#
# $Id: lonviewclasslist.pm,v 1.8 2007/07/20 00:15:06 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###############################################################
##############################################################

package Apache::lonviewclasslist;

use strict;
use Apache::loncoursedata();
use Apache::loncommon();
use Apache::lonhtmlcommon();
use Apache::Constants qw(:common :http REDIRECT);
use Apache::lonlocal;
use Apache::lonnet;


###################################################################
###################################################################

=pod

=item &handler

The typical handler you see in all these modules.  Takes $r, the
http request, as an argument.  

=cut

###################################################################
###################################################################
sub handler {
    my $r=shift;
    if ($r->header_only) {
        &Apache::loncommon::content_type($r,'text/html');
        $r->send_http_header;
        return OK;
    }
#    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
#                                            ['action','state']);
    &Apache::lonhtmlcommon::clear_breadcrumbs();
    &Apache::lonhtmlcommon::add_breadcrumb
        ({href=>"/adm/viewclasslist",
          text=>"View Classlist",
          faq=>9,bug=>'Instructor Interface',});
    #  Needs to be in a course
    if (! ($env{'request.course.fn'})) {
        $env{'user.error.msg'}=
            "/adm/viewclasslist:not in course role";
        return HTTP_NOT_ACCEPTABLE; 
    }
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    #
    my $start_page = &Apache::loncommon::start_page('Classlist');
    my $breadcrumbs= &Apache::lonhtmlcommon::breadcrumbs(undef,
							 'Enrollment Manager');
    $r->print(<<ENDHEADER);
$start_page
$breadcrumbs
ENDHEADER

    # Get classlist view settings
    my %viewsettings = &retrieve_view_settings();

    # Print classlist
    if (keys(%viewsettings) > 0) {
        $r->print(&html_classlist($r,\%viewsettings));
    } else {
        $r->print('<h3>'.
                  &mt("Display of a student-viewable course roster is not currently enabled.").
                  '</h3>');
    }
    #
    # Finish up
    $r->print(&Apache::loncommon::end_page());
    return OK;
}

sub retrieve_view_settings {
    my %viewsettings;
    if (exists($env{'request.course.id'})) {
        my $cid = $env{'request.course.id'};
        my $viewpermission = 'course.'.$cid.'.student_classlist_view';
        my $student_opt_in = 'course.'.$cid.'.student_classlist_opt_in';
        my $portfiles_link = 'course.'.$cid.'.student_classlist_portfiles';
        if (exists($env{$viewpermission}) &&
            $env{$viewpermission} =~ /^(all|section)$/) {
            $viewsettings{'permission'} = $env{$viewpermission};
            if ($viewsettings{'permission'} =~ /^section$/i) {
                $viewsettings{'limit_to_section'} = 1;
            } else {
                $viewsettings{'limit_to_section'} = 0;
            }
            $viewsettings{'student_opt_in'} = $env{$student_opt_in};
            $viewsettings{'portfiles_link'} = $env{$portfiles_link};
        }
    }
    return %viewsettings;
}

sub html_classlist {
    my ($r,$viewsettings) = @_;
    my ($Str,$title,$secdisplay,$cid,$cdom,$cnum,$listtype,%publicroster);
    my $fullroster = &Apache::loncoursedata::get_classlist();
    my $classlist;

    if ($env{'form.action'} eq 'setenv') {
        $Str .= &process_student_prefs();
    }
    $Str .= '<h3>'.&mt('Student-viewable course roster').'</h3>';

    $cid = $env{'request.course.id'};
    $cdom = $env{'course.'.$cid.'.domain'};
    $cnum = $env{'course.'.$cid.'.num'};

    if ($viewsettings->{'limit_to_section'}) {
        if ($env{'request.course.sec'} eq '') {
            $title = '<h4>'.&mt('Students with no section').'</h4>';
            $listtype = 'without a section';
        } else {
            $title ='<h4>'.&mt('Students in section "[_1]"',
                               $env{'request.course.sec'}).'</h4>';
            $listtype = 'in the section';
            $secdisplay = " ($env{'request.course.sec'}) ";
        }
    } else {
        $title .= '<h4>'.&mt('Students in any section').'</h4>';
        $listtype = 'in the course';
    }

    if ($viewsettings->{'student_opt_in'}) {
        if ($env{'request.role'} =~ /^st/)  {
            $Str .= &print_roster_form();
        }
        %publicroster = &Apache::lonnet::dump('publicroster',$cdom,$cnum);
    }

    $Str .= $title;

    my $fullcount = 0;
    my $publiccount = 0;
    my $displaycount = 0;
    my $sectionidx  = &Apache::loncoursedata::CL_SECTION();
    my $statusidx   = &Apache::loncoursedata::CL_STATUS();

    foreach my $student (keys(%{$fullroster})) {
        my $section  = $fullroster->{$student}->[$sectionidx];
        my $status   = $fullroster->{$student}->[$statusidx];
        next if (lc($status) ne 'active');
        if ($viewsettings->{'limit_to_section'}) {
            next if ($section ne $env{'request.course.sec'});
        }
        $fullcount ++;
        if ($viewsettings->{'student_opt_in'}) {
            if ($publicroster{$student}) {
                $classlist->{$student} = $fullroster->{$student};
                $publiccount ++;
            }
        } else {
            $classlist->{$student} = $fullroster->{$student};
        }
    }
    if ($viewsettings->{'student_opt_in'}) {
        $displaycount = $publiccount;
        if ($fullcount > $publiccount) {
            if ($publiccount) {
                $Str .= &mt('Only students who have opted to be listed in the roster ([_1] out of [_2] students) are shown.',$publiccount,$fullcount).'<br />';
            } else {
                if ($fullcount == 1) {
                    $Str .= &mt('The single student '.$listtype.'[_1] has opted not to be listed in the roster.',$secdisplay);
                } else {
                    $Str .= &mt('None of the [_1] students '.$listtype.'[_2] have opted to be listed in the roster.',$fullcount,$secdisplay);
                }
                return $Str;
            }
        } else {
            if ($fullcount > 1) {
                $Str .= &mt('All [_1] students '.$listtype.'[_2] have opted to be listed in the roster.',$fullcount,$secdisplay);
            } elsif ($fullcount == 1) {
                $Str .= &mt('The single student '.$listtype.'[_1] has opted to be listed in the roster.',$secdisplay);
            }
        }
    } else {
        $displaycount = $fullcount;
        if ($fullcount > 1) {
            $Str .= &mt('All [_1] students '.$listtype.'[_2] are listed in the roster.',$fullcount,$secdisplay);
        } elsif ($fullcount == 1) {
            $Str .= &mt('There is only a single student '.$listtype.'[_1]',$secdisplay);
        }
    }
    undef($fullroster);

    if (!$displaycount) {
        $Str .= &mt('There are currently no students to display.');
        return $Str;
    }

    # Set up a couple variables.
    my $usernameidx = &Apache::loncoursedata::CL_SNAME();
    my $domainidx   = &Apache::loncoursedata::CL_SDOM();
    my $fullnameidx = &Apache::loncoursedata::CL_FULLNAME();

    # Sort the students
    my $sortby = $fullnameidx;
    my @Sorted_Students = sort {
        lc($classlist->{$a}->[$sortby])  cmp lc($classlist->{$b}->[$sortby])
        } (keys(%$classlist));
    $Str .= '<br />'.&Apache::loncommon::start_data_table()."\n".
            &Apache::loncommon::start_data_table_header_row()."\n".
        '<th></th>'. # for the count
        '<th>'.&mt('Student').'</th>'.
        '<th>'.&mt('Username').'</th>';
    if (! $viewsettings->{'limit_to_section'}) {
        $Str .= '<th>'.&mt('Section').'</th>';
    }
    if ($viewsettings->{'portfiles_link'}) {
        $Str .= '<th>'.&mt('Available Portfolio files').'</th>';
    }
    $Str .= &Apache::loncommon::end_data_table_header_row();
    my $count ++;
    foreach my $student (@Sorted_Students) {
        my $username = $classlist->{$student}->[$usernameidx];
        my $domain   = $classlist->{$student}->[$domainidx];
        my $fullname = $classlist->{$student}->[$fullnameidx];
        if ($fullname =~ /^\s*$/) {
            $fullname = &mt('Name not given');
        }
        my $section  = $classlist->{$student}->[$sectionidx];
        if ($section eq '') {
            $section = &mt('none');
        }
        $Str .= &Apache::loncommon::start_data_table_row()."\n".
            '<td>'.$count++.'</td>'.
            '<td>'.&Apache::loncommon::aboutmewrapper($fullname,
                                                      $username,
                                                      $domain).'</td>'.
            '<td>'.('&nbsp;'x2).
            &Apache::loncommon::messagewrapper
            ('<img src="/adm/lonIcons/mailto.gif" border="0" />&nbsp;'.
             $username.'@'.$domain,$username,$domain).'</td>';
        if (! $viewsettings->{'limit_to_section'}) {
            $Str .= '<td>'.$section.'</td>';
        }
        if ($viewsettings->{'portfiles_link'}) {
            my $filecounts = &Apache::lonaboutme::portfolio_files($r,'showlink',undef,undef,$domain,$username,$fullname);
            my $link;
            if (ref($filecounts) eq 'HASH') {
                $link = &mt('[quant,_1,file,files,No files]',$filecounts->{'both'});
                if ($filecounts->{'both'} > 0) {
                    $link = '<a href="/adm/'.$domain.'/'.$username.'/aboutme/portfolio?classlist">'.$link.'</a>'; 
                }
            } else {
                $link = &mt("Error retrieving file information.");
            }
            $Str .= '<td>'.$link.'</td>';
        }
        $Str .= &Apache::loncommon::end_data_table_row()."\n";
    }
    $Str .= &Apache::loncommon::end_data_table();
    return $Str;
}

sub print_roster_form {
    my $cid = $env{'request.course.id'};
    my $showinroster = $env{'environment.internal.'.$cid.'.showinroster'};
    my ($showoff,$showon);
    if ($showinroster) {
        $showon = ' checked="checked" ';
        $showoff = ' ';
    } else {
        $showoff = ' checked="checked" ';
        $showon = ' ';
    }
    my $output = '<hr /><h4>'.&mt('Your roster setting').'</h4>';
    if ($showinroster) {
        $output .= &mt('You are currently listed in the student-viewable roster.');
    } else {
        $output .=  &mt('You are currently <b>not</b> listed in the student-viewable roster.');
    }
    $output .= '<br />'.&mt('Include yourself in the roster?').'&nbsp;&nbsp;'.
        '<form name="studentparm" method="post">'.
        '<span class="LC_nobreak"><label><input type="radio" name="showinroster" value="1"'.$showon.'/>'.&mt('Yes').'</label>&nbsp;&nbsp;<label>'.
        '<input type="radio" name="showinroster" value="0"'.$showoff.'/>'.&mt('No').
        '</label></span><br /><br />'.
        '<input type="hidden" name="action" value="setenv" />'.
        '<input type="submit" name="studentsubmit" value="'.&mt('Save').'" /></form><hr />';
    return $output;
}

sub process_student_prefs {
    my $cid = $env{'request.course.id'};
    my $cdom = $env{'course.'.$cid.'.domain'};
    my $cnum = $env{'course.'.$cid.'.num'};
    my $uname = $env{'user.name'};
    my $udom = $env{'user.domain'};
    my $student = $uname.':'.$udom;
    my %pubroster = &Apache::lonnet::get('publicroster',[$student],$cdom,$cnum);
    my $visibility = &mt('off');
    my $showinroster = $env{'form.showinroster'};
    if ($showinroster) {
        $visibility = &mt('on');
    }
    my $sturoster = 0;
    if ($pubroster{$student}) {
        $sturoster = 1;
    }
    my $output;
    if ($sturoster ne $showinroster) {
        my %changeHash = (
            'environment.internal.'.$cid.'.showinroster' => $showinroster,
        );
        my $putresult = &Apache::lonnet::put('environment',
                                             \%changeHash,$udom,$uname);
        if ($putresult eq 'ok') {
            &Apache::lonnet::appenv(%changeHash);
            my $result = &Apache::lonnet::put('publicroster',{$student => $showinroster,},$cdom,$cnum);
            if ($result eq 'ok') {
                $output .= &mt('Display of your name in the student-viewable roster set to <b>[_1]</b>.',$visibility);
            } else {
                $output .= &mt('Error occurred saving display setting.');
            }
        } else {
            $output .= &mt('Error occurred saving display setting.');
        }
    } else {
        $output .= &mt('Display of your name in the student-viewable roster unchanged (set to <b>[_1]</b>).',$visibility);
    }
    return $output;
}




###################################################################
###################################################################

1;
__END__


