# The LearningOnline Network with CAPA
# Handler to set domain-wide configuration settings
#
# $Id: domainprefs.pm,v 1.25 2007/08/26 15:31:03 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#
###############################################################
##############################################################

package Apache::domainprefs;

use strict;
use Apache::Constants qw(:common :http);
use Apache::lonnet;
use Apache::loncommon();
use Apache::lonhtmlcommon();
use Apache::lonlocal;
use LONCAPA();
use LONCAPA::Enrollment;
use File::Copy;

sub handler {
    my $r=shift;
    if ($r->header_only) {
        &Apache::loncommon::content_type($r,'text/html');
        $r->send_http_header;
        return OK;
    }

    my $dom = $env{'request.role.domain'};
    my $domdesc = &Apache::lonnet::domain($dom,'description');
    if (&Apache::lonnet::allowed('mau',$dom)) {
        &Apache::loncommon::content_type($r,'text/html');
        $r->send_http_header;
    } else {
        $env{'user.error.msg'}=
        "/adm/domainprefs:mau:0:0:Cannot modify domain settings";
        return HTTP_NOT_ACCEPTABLE;
    }
    &Apache::lonhtmlcommon::clear_breadcrumbs();
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
                                            ['phase']);
    my $phase = "display";
    if ( exists($env{'form.phase'}) ) {
        $phase = $env{'form.phase'};
    }
    my %domconfig =
      &Apache::lonnet::get_dom('configuration',['login','rolecolors',
                'quotas','autoenroll','autoupdate','directorysrch'],$dom);
    my @prefs = (
      { text => 'Default color schemes',
        help => 'Default_Color_Schemes',
        action => 'rolecolors',
        header => [{col1 => 'Student Settings',
                    col2 => '',},
                   {col1 => 'Coordinator Settings',
                    col2 => '',},
                   {col1 => 'Author Settings',
                    col2 => '',},
                   {col1 => 'Administrator Settings',
                    col2 => '',}],
        },
      { text => 'Log-in page options',
        help => 'Domain_Log-in_Page',
        action => 'login',
        header => [{col1 => 'Item',
                    col2 => '',}],
        },
      { text => 'Default quotas for user portfolios',
        help => 'Default_User_Quota',
        action => 'quotas',
        header => [{col1 => 'User type',
                    col2 => 'Default quota'}],
        },
      { text => 'Auto-enrollment settings',
        help => 'Domain_Auto_Enrollment',
        action => 'autoenroll',
        header => [{col1 => 'Configuration setting',
                    col2 => 'Value(s)'}],
        },
      { text => 'Auto-update settings',
        help => 'Domain_Auto_Update',
        action => 'autoupdate',
        header => [{col1 => 'Setting',
                    col2 => 'Value',},
                   {col1 => 'User Population',
                    col2 => 'Updataeable user data'}],
        },
      { text => 'Institutional directory searches',
        help => 'Domain_Directory_Search',
        action => 'directorysrch',
        header => [{col1 => 'Setting',
                    col2 => 'Value',}],
        },
    );
    my @roles = ('student','coordinator','author','admin');
    &Apache::lonhtmlcommon::add_breadcrumb
    ({href=>"javascript:changePage(document.$phase,'display')",
      text=>"Domain Configuration"});
    my $confname = $dom.'-domainconfig';
    if ($phase eq 'process') {
        &Apache::lonhtmlcommon::add_breadcrumb
          ({href=>"javascript:changePage(document.$phase,'$phase')",
            text=>"Updated"});
        &print_header($r,$phase);
        foreach my $item (@prefs) {
            $r->print('<h3>'.&mt($item->{'text'}).'</h3>'.
                      &process_changes($r,$dom,$confname,
                        $item->{'action'},\@roles,%domconfig));
        }
        $r->print('<p>');
        &print_footer($r,$phase,'display','Back to actions menu');
        $r->print('</p>');
    } else {
        if ($phase eq '') {
            $phase = 'display';
        }
        my %helphash;   
        my $numprefs = @prefs;
        &print_header($r,$phase);
        if (keys(%domconfig) == 0) {
            my $primarylibserv = &Apache::lonnet::domain($dom,'primary');
            my $perlvarref = &LONCAPA::Configuration::read_conf('loncapa.conf');
            my $hostid = $perlvarref->{'lonHostID'};
            if ($hostid ne $primarylibserv) {
                my %designhash = &Apache::loncommon::get_domainconf($dom);
                my @loginimages = ('img','logo','domlogo');
                my $custom_img_count = 0;
                foreach my $img (@loginimages) {
                    if ($designhash{$dom.'.login.'.$img} ne '') {
                        $custom_img_count ++;
                    }
                }
                foreach my $role (@roles) {
                    if ($designhash{$dom.'.'.$role.'.img'} ne '') {
                        $custom_img_count ++;
                    }
                }
                if ($custom_img_count > 0) {
                    my $switch_server = &check_switchserver($dom,$confname);
                    $r->print(&mt('Domain configuration settings have yet to be saved for this domain via the web-based domain preferences interface.').'<br />'.&mt("While this remains so, you must switch to the domain's primary library server in order to update settings.").'<br /><br />'.&mt("Thereafter, you will be able to update settings from this screen when logged in to any server in the LON-CAPA network (with a DC role selected in the domain), although you will still need to switch to the domain's primary library server to upload new images or logos.").'<br /><br />'.$switch_server.' '.&mt('to primary library server for domain: [_1]',$dom));
                    return OK;
                }
            }
        }
        $r->print('<table border="0" width="100%" cellpadding="2" cellspacing="4"><tr><td align="left" valign="top" width="45%">');
        foreach my $item (@prefs) {
            if ($item->{'action'} eq 'login') {
                $r->print('</td><td width="6%">&nbsp;</td><td align="left" valign="top" width="47%">');
            }
            &print_config_box($r,$dom,$confname,$phase,$item->{'action'},
                              $item,$domconfig{$item->{'action'}});
        }
        $r->print('
      </td>
    </tr>
   </table>');
        &print_footer($r,$phase,'process','Save changes');
    }
    return OK;
}

sub process_changes {
    my ($r,$dom,$confname,$action,$roles,%domconfig) = @_;
    my $output;
    if ($action eq 'login') {
        $output = &modify_login($r,$dom,$confname,%domconfig);
    } elsif ($action eq 'rolecolors') {
        $output = &modify_rolecolors($r,$dom,$confname,$roles,
                                     %domconfig);
    } elsif ($action eq 'quotas') {
        $output = &modify_quotas($dom,%domconfig);
    } elsif ($action eq 'autoenroll') {
        $output = &modify_autoenroll($dom,%domconfig);
    } elsif ($action eq 'autoupdate') {
        $output = &modify_autoupdate($dom,%domconfig);
    } elsif ($action eq 'directorysrch') {
        $output = &modify_directorysrch($dom,%domconfig);
    }
    return $output;
}

sub print_config_box {
    my ($r,$dom,$confname,$phase,$action,$item,$settings) = @_;
    $r->print('
         <table class="LC_nested_outer">
          <tr>
           <th>'.&mt($item->{text}).'&nbsp;</th></tr>');
# 
# FIXME - put the help link back in when the help files exist
#           <th>'.&mt($item->{text}).'&nbsp;'.
#           &Apache::loncommon::help_open_topic($item->{'help'}).'</th>
#          </tr>');
    if (($action eq 'autoupdate') || ($action eq 'rolecolors')) {
        my $colspan = ($action eq 'rolecolors')?' colspan="2"':'';
        $r->print('
          <tr>
           <td>
            <table class="LC_nested">
             <tr class="LC_info_row">
              <td class="LC_left_item"'.$colspan.'>'.$item->{'header'}->[0]->{'col1'}.'</td>
              <td class="LC_right_item">'.$item->{'header'}->[0]->{'col2'}.'</td>
             </tr>');
        if ($action eq 'autoupdate') {
            $r->print(&print_autoupdate('top',$dom,$settings));
        } else {
            $r->print(&print_rolecolors($phase,'student',$dom,$confname,$settings));
        }
        $r->print('
           </table>
          </td>
         </tr>
         <tr>
           <td>
            <table class="LC_nested">
             <tr class="LC_info_row">
              <td class="LC_left_item"'.$colspan.'>'.$item->{'header'}->[1]->{'col1'}.'</td>
              <td class="LC_right_item">'.$item->{'header'}->[1]->{'col2'}.'</td>
             </tr>');
        if ($action eq 'autoupdate') {
            $r->print(&print_autoupdate('bottom',$dom,$settings));
        } else {
            $r->print(&print_rolecolors($phase,'coordinator',$dom,$confname,$settings).'
           </table>
          </td>
         </tr>
         <tr>
           <td>
            <table class="LC_nested">
             <tr class="LC_info_row">
              <td class="LC_left_item"'.$colspan.'>'.$item->{'header'}->[2]->{'col1'}.'</td>
              <td class="LC_right_item">'.$item->{'header'}->[2]->{'col2'}.'</td>
             </tr>'.
            &print_rolecolors($phase,'author',$dom,$confname,$settings).'
           </table>
          </td>
         </tr>
         <tr>
           <td>
            <table class="LC_nested">
             <tr class="LC_info_row">
              <td class="LC_left_item"'.$colspan.'>'.$item->{'header'}->[3]->{'col1'}.'</td>
              <td class="LC_right_item">'.$item->{'header'}->[3]->{'col2'}.'</td>
             </tr>'.
            &print_rolecolors($phase,'admin',$dom,$confname,$settings));
        }
    } else {
        $r->print('
          <tr>
           <td>
            <table class="LC_nested">
             <tr class="LC_info_row">');
        if (($action eq 'login') || ($action eq 'directorysrch')) {
            $r->print('  
              <td class="LC_left_item" colspan="2">'.$item->{'header'}->[0]->{'col1'}.'</td>');
        } else {
            $r->print('
              <td class="LC_left_item">'.$item->{'header'}->[0]->{'col1'}.'</td>');
        }
        $r->print('
              <td class="LC_right_item">'.$item->{'header'}->[0]->{'col2'}.'</td>
             </tr>');
        if ($action eq 'login') {
           $r->print(&print_login($dom,$confname,$phase,$settings));
        } elsif ($action eq 'quotas') {
           $r->print(&print_quotas($dom,$settings));
        } elsif ($action eq 'autoenroll') {
           $r->print(&print_autoenroll($dom,$settings));
        } elsif ($action eq 'directorysrch') {
           $r->print(&print_directorysrch($dom,$settings));
        }  
    }
    $r->print('
   </table>
  </td>
 </tr>
</table><br />');
    return;
}

sub print_header {
    my ($r,$phase) = @_;
    my $js = '
<script type="text/javascript">
function changePage(formname,newphase) {
    formname.phase.value = newphase;
    formname.submit();
}
'.
&color_pick_js().'
</script>
';
    $r->print(&Apache::loncommon::start_page('View/Modify Domain Settings',
                                           $js));
    $r->print(&Apache::lonhtmlcommon::breadcrumbs('Domain Settings'));
    $r->print('
<form name="parmform" action="">
<input type="hidden" name="pres_marker" />
<input type="hidden" name="pres_type" />
<input type="hidden" name="pres_value" />
</form>
');
    $r->print('<form method="post" name="'.$phase.'" action="/adm/domainprefs"'.
              ' enctype="multipart/form-data">');
    return;
}

sub print_footer {
    my ($r,$phase,$newphase,$button_text) = @_;
    $button_text = &mt($button_text);
    $r->print('<input type="hidden" name="phase" value="" />');
    my $dest='"javascript:changePage(document.'.$phase.','."'$newphase'".')"';
    if ($phase eq 'process') {
        $r->print('<a href='.$dest.'>'.$button_text.'</a>');
    } else {
        $r->print('<input type="button" name="store" value="'.
                  $button_text.'" onclick='.$dest.' />');
    }
    $r->print('</form>');
    $r->print('<br />'.&Apache::loncommon::end_page());
    return;
}

sub print_login {
    my ($dom,$confname,$phase,$settings) = @_;
    my %choices = &login_choices();
    my ($catalogon,$catalogoff,$adminmailon,$adminmailoff);
    $catalogon = ' checked="checked" ';
    $adminmailoff = ' checked="checked" ';
    my @images = ('img','logo','domlogo');
    my @bgs = ('pgbg','mainbg','sidebg');
    my @links = ('link','alink','vlink');
    my %designhash = &Apache::loncommon::get_domainconf($dom);
    my %defaultdesign = %Apache::loncommon::defaultdesign;
    my (%is_custom,%designs);
    my %defaults = (
                   font => $defaultdesign{'login.font'},
                   );
    foreach my $item (@images) {
        $defaults{$item} = $defaultdesign{'login.'.$item};
    }
    foreach my $item (@bgs) {
        $defaults{'bgs'}{$item} = $defaultdesign{'login.'.$item};
    }
    foreach my $item (@links) {
        $defaults{'links'}{$item} = $defaultdesign{'login.'.$item};
    }
    if (ref($settings) eq 'HASH') {
        if ($settings->{'coursecatalog'} eq '0') {
            $catalogoff = $catalogon;
            $catalogon = ' ';
        }
        if ($settings->{'adminmail'} eq '1') {
            $adminmailon = $adminmailoff;
            $adminmailoff = ' ';
        }
        foreach my $item (@images) {
            if ($settings->{$item} ne '') {
                $designs{$item} = $settings->{$item};
                $is_custom{$item} = 1;
            }
        }
        if ($settings->{'font'} ne '') {
            $designs{'font'} = $settings->{'font'};
            $is_custom{'font'} = 1;
        }
        foreach my $item (@bgs) {
            if ($settings->{$item} ne '') {
                $designs{'bgs'}{$item} = $settings->{$item};
                $is_custom{$item} = 1;
            }
        }
        foreach my $item (@links) {
            if ($settings->{$item} ne '') {
                $designs{'links'}{$item} = $settings->{$item};
                $is_custom{$item} = 1;
            }
        }
    } else {
        if ($designhash{$dom.'.login.font'} ne '') {
            $designs{'font'} = $designhash{$dom.'.login.font'};
            $is_custom{'font'} = 1;
        }
        foreach my $item (@images) {
            if ($designhash{$dom.'.login.'.$item} ne '') {
                $designs{$item} = $designhash{$dom.'.login.'.$item};
                $is_custom{$item} = 1;
            }
        }
        foreach my $item (@bgs) {
            if ($designhash{$dom.'.login.'.$item} ne '') {
                $designs{'bgs'}{$item} = $designhash{$dom.'.login.'.$item};
                $is_custom{$item} = 1;
            }
        }
        foreach my $item (@links) {
            if ($designhash{$dom.'.login.'.$item} ne '') {
                $designs{'links'}{$item} = $designhash{$dom.'.login.'.$item};
                $is_custom{$item} = 1;
            }
        }
    }
    my %alt_text = &Apache::lonlocal::texthash  ( img => 'Log-in banner',
                                                  logo => 'Institution Logo',
                                                  domlogo => 'Domain Logo');
    my $itemcount = 1;
    my $css_class = $itemcount%2?' class="LC_odd_row"':'';
    my $datatable = 
        '<tr'.$css_class.'><td colspan="2">'.$choices{'coursecatalog'}.
        '</td><td>'.
        '<span class="LC_nobreak"><label><input type="radio" name="coursecatalog"'.
        $catalogon.' value="1" />'.&mt('Yes').'</label>&nbsp;'.
        '<label><input type="radio" name="coursecatalog"'.
        $catalogoff.'value="0" />'.&mt('No').'</label></span></td>'.
        '</tr>';
    $itemcount ++;
    $css_class = $itemcount%2?' class="LC_odd_row"':'';
    $datatable .= '<tr'.$css_class.'>'.
        '<td colspan="2">'.$choices{'adminmail'}.'</td>'.
        '<td><span class="LC_nobreak">'.
        '<label><input type="radio" name="adminmail"'.
        $adminmailon.' value="1" />'.&mt('Yes').'</label>&nbsp;'.
        '<label><input type="radio" name="adminmail"'.
        $adminmailoff.'value="0" />'.&mt('No').'</label></span></td></tr>';
    $itemcount ++;
    $datatable .= &display_color_options($dom,$confname,$phase,'login',$itemcount,\%choices,\%is_custom,\%defaults,\%designs,\@images,\@bgs,\@links,\%alt_text);
    $datatable .= '</tr></table></td></tr>';
    return $datatable;
}

sub login_choices {
    my %choices =
        &Apache::lonlocal::texthash (
            coursecatalog => 'Display Course Catalog link?',
            adminmail => "Display Administrator's E-mail Address?",
            img => "Header",
            logo => "Main Logo",
            domlogo => "Domain Logo",
            bgs => "Background colors",
            links => "Link colors",
            font => "Font color",
            pgbg => "Page",
            mainbg => "Main panel",
            sidebg => "Side panel",
            link => "Link",
            alink => "Active link",
            vlink => "Visited link",
        );
    return %choices;
}

sub print_rolecolors {
    my ($phase,$role,$dom,$confname,$settings) = @_;
    my %choices = &color_font_choices();
    my @bgs = ('pgbg','tabbg','sidebg');
    my @links = ('link','alink','vlink');
    my @images = ('img');
    my %alt_text = &Apache::lonlocal::texthash(img => "Banner for $role role");
    my %designhash = &Apache::loncommon::get_domainconf($dom);
    my %defaultdesign = %Apache::loncommon::defaultdesign;
    my (%is_custom,%designs);
    my %defaults = (
                   img => $defaultdesign{$role.'.img'},
                   font => $defaultdesign{$role.'.font'},
                   );
    foreach my $item (@bgs) {
        $defaults{'bgs'}{$item} = $defaultdesign{$role.'.'.$item};
    }
    foreach my $item (@links) {
        $defaults{'links'}{$item} = $defaultdesign{$role.'.'.$item};
    }
    if (ref($settings) eq 'HASH') {
        if (ref($settings->{$role}) eq 'HASH') {
            if ($settings->{$role}->{'img'} ne '') {
                $designs{'img'} = $settings->{$role}->{'img'};
                $is_custom{'img'} = 1;
            }
            if ($settings->{$role}->{'font'} ne '') {
                $designs{'font'} = $settings->{$role}->{'font'};
                $is_custom{'font'} = 1;
            }
            foreach my $item (@bgs) {
                if ($settings->{$role}->{$item} ne '') {
                    $designs{'bgs'}{$item} = $settings->{$role}->{$item};
                    $is_custom{$item} = 1;
                }
            }
            foreach my $item (@links) {
                if ($settings->{$role}->{$item} ne '') {
                    $designs{'links'}{$item} = $settings->{$role}->{$item};
                    $is_custom{$item} = 1;
                }
            }
        }
    } else {
        if ($designhash{$dom.'.'.$role.'.img'} ne '') {
            $designs{img} = $designhash{$dom.'.'.$role.'.img'};
            $is_custom{'img'} = 1;
        }
        if ($designhash{$dom.'.'.$role.'.font'} ne '') {
            $designs{font} = $designhash{$dom.'.'.$role.'.font'};
            $is_custom{'font'} = 1;
        }
        foreach my $item (@bgs) {
            if ($designhash{$dom.'.'.$role.'.'.$item} ne '') {
                $designs{'bgs'}{$item} = $designhash{$dom.'.'.$role.'.'.$item};
                $is_custom{$item} = 1;
            
            }
        }
        foreach my $item (@links) {
            if ($designhash{$dom.'.'.$role.'.'.$item} ne '') {
                $designs{'links'}{$item} = $designhash{$dom.'.'.$role.'.'.$item};
                $is_custom{$item} = 1;
            }
        }
    }
    my $itemcount = 1;
    my $datatable = &display_color_options($dom,$confname,$phase,$role,$itemcount,\%choices,\%is_custom,\%defaults,\%designs,\@images,\@bgs,\@links,\%alt_text);
    $datatable .= '</tr></table></td></tr>';
    return $datatable;
}

sub display_color_options {
    my ($dom,$confname,$phase,$role,$itemcount,$choices,$is_custom,$defaults,$designs,
        $images,$bgs,$links,$alt_text) = @_;
    my $css_class = $itemcount%2?' class="LC_odd_row"':'';
    my $datatable = '<tr'.$css_class.'>'.
        '<td>'.$choices->{'font'}.'</td>';
    if (!$is_custom->{'font'}) {
        $datatable .=  '<td>'.&mt('Default in use:').'&nbsp;'.$defaults->{'font'}.'</td>';
    } else {
        $datatable .= '<td>&nbsp;</td>';
    }
    my $fontlink = &color_pick($phase,$role,'font',$choices->{'font'},$designs->{'font'});
    $datatable .= '<td><span class="LC_nobreak">'.
                  '<input type="text" size="10" name="'.$role.'_font"'.
                  ' value="'.$designs->{'font'}.'" />&nbsp;'.$fontlink.
                  '</span></td></tr>';
    my $switchserver = &check_switchserver($dom,$confname);
    foreach my $img (@{$images}) {
	$itemcount ++;
        $css_class = $itemcount%2?' class="LC_odd_row"':'';
        $datatable .= '<tr'.$css_class.'>'.
                      '<td>'.$choices->{$img}.'</td>';
        my ($imgfile, $img_import);
        if ($designs->{$img} ne '') {
            $imgfile = $designs->{$img};
	    $img_import = ($imgfile =~ m{^/adm/});
        } else {
            $imgfile = $defaults->{$img};
        }
        if ($imgfile) {
            my ($showfile,$fullsize);
            if ($imgfile =~ m-^(/res/\Q$dom\E/\Q$confname\E/\Q$img\E)/([^/]+)$-) {
                my $urldir = $1;
                my $filename = $2;
                my @info = &Apache::lonnet::stat_file($designs->{$img});
                if (@info) {
                    my $thumbfile = 'tn-'.$filename;
                    my @thumb=&Apache::lonnet::stat_file($urldir.'/'.$thumbfile);
                    if (@thumb) {
                        $showfile = $urldir.'/'.$thumbfile;
                    } else {
                        $showfile = $imgfile;
                    }
                } else {
                    $showfile = '';
                }
            } elsif ($imgfile =~ m-^/(adm/[^/]+)/([^/]+)$-) {
                $showfile = $imgfile;
                my $imgdir = $1;
                my $filename = $2;
                if (-e "/home/httpd/html/$imgdir/tn-".$filename) {
                    $showfile = "/$imgdir/tn-".$filename;
                } else {
                    my $input = "/home/httpd/html".$imgfile;
                    my $output = '/home/httpd/html/'.$imgdir.'/tn-'.$filename;
                    if (!-e $output) {
                        my ($width,$height) = &thumb_dimensions();
                        my ($fullwidth,$fullheight) = &check_dimensions($input);
                        if ($fullwidth ne '' && $fullheight ne '') {
                            if ($fullwidth > $width && $fullheight > $height) { 
                                my $size = $width.'x'.$height;
                                system("convert -sample $size $input $output");
                                $showfile = '/'.$imgdir.'/tn-'.$filename;
                            }
                        }
                    }
                }
            }
            if ($showfile) {
                $showfile = &Apache::loncommon::lonhttpdurl($showfile);
                $fullsize =  &Apache::loncommon::lonhttpdurl($imgfile);
                $datatable.= '<td>';
                if (!$is_custom->{$img}) {
                    $datatable .= &mt('Default in use:').'<br />';
                }
		if ($img_import) {
		    $datatable.= '<input type="hidden" name="'.$role.'_import_'.$img.'" value="'.$imgfile.'" />';
		}
                $datatable.= '<a href="'.$fullsize.'" target="_blank"><img src="'.
                             $showfile.'" alt="'.$alt_text->{$img}.
                             '" border="0" /></a></td>';
                if ($is_custom->{$img}) {
                    $datatable.='<td><span class="LC_nobreak"><label><input type="checkbox" name="'.
                                $role.'_del_'.$img.'" value="1" />'.&mt('Delete?').
                                '</label>&nbsp;'.&mt('Replace:').'</span><br />';
                } else {
                    $datatable.='<td valign="bottom">'.&mt('Upload:').'<br />';
                }
            } else {
                $datatable .= '<td colspan="2" class="LC_right_item"><br />'.
                              &mt('Upload:');
            }
        } else {
            $datatable .= '<td colspan="2" class="LC_right_item"><br />'.
                          &mt('Upload:');
        }
        if ($switchserver) {
            $datatable .= &mt('Upload to library server: [_1]',$switchserver);
        } else {
            $datatable .='&nbsp;<input type="file" name="'.$role.'_'.$img.'" />';
        }
        $datatable .= '</td></tr>';
    }
    $itemcount ++;
    $css_class = $itemcount%2?' class="LC_odd_row"':'';
    $datatable .= '<tr'.$css_class.'>'.
                  '<td>'.$choices->{'bgs'}.'</td>';
    my $bgs_def;
    foreach my $item (@{$bgs}) {
        if (!$is_custom->{$item}) {
            $bgs_def .= '<td>'.$choices->{$item}.'<br />'.$defaults->{'bgs'}{$item}.'</td>';
        }
    }
    if ($bgs_def) {
        $datatable .= '<td>'.&mt('Default(s) in use:').'<br /><table border="0"><tr>'.$bgs_def.'</tr></table></td>';
    } else {
        $datatable .= '<td>&nbsp;</td>';
    }
    $datatable .= '<td class="LC_right_item">'.
                  '<table border="0"><tr>';
    foreach my $item (@{$bgs}) {
        my $link = &color_pick($phase,$role,$item,$choices->{$item},$designs->{'bgs'}{$item});
        $datatable .= '<td align="center">'.$link;
        if ($designs->{'bgs'}{$item}) {
            $datatable .= '<span style="background-color:'.$designs->{'bgs'}{$item}.'width: 10px">&nbsp;</span>';
        }
        $datatable .= '<br /><input type="text" size="8" name="'.$role.'_'.$item.'" value="'.$designs->{'bgs'}{$item}.
                      '" /></td>';
    }
    $datatable .= '</tr></table></td></tr>';
    $itemcount ++;
    $css_class = $itemcount%2?' class="LC_odd_row"':'';
    $datatable .= '<tr'.$css_class.'>'.
                  '<td>'.$choices->{'links'}.'</td>';
    my $links_def;
    foreach my $item (@{$links}) {
        if (!$is_custom->{$item}) {
            $links_def .= '<td>'.$choices->{$item}.'<br />'.$defaults->{'links'}{$item}.'</td>';
        }
    }
    if ($links_def) {
        $datatable .= '<td>'.&mt('Default(s) in use:').'<br /><table border="0"><tr>'.$links_def.'</tr></table></td>';
    } else {
        $datatable .= '<td>&nbsp;</td>';
    }
    $datatable .= '<td class="LC_right_item">'.
                  '<table border="0"><tr>';
    foreach my $item (@{$links}) {
        $datatable .= '<td align="center">';
        my $link = &color_pick($phase,$role,$item,$choices->{$item},$designs->{'links'}{$item});
        if ($designs->{'links'}{$item}) {
            $datatable.='<span style="color: '.$designs->{'links'}{$item}.';">'.
                        $link.'</span>';
        } else {
            $datatable .= $link;
        }
        $datatable .= '<br /><input type="text" size="8" name="'.$role.'_'.$item.'" value="'.$designs->{'links'}{$item}.
                      '" /></td>';
    }
    return $datatable;
}

sub color_pick {
    my ($phase,$role,$item,$desc,$curcol) = @_;
    my $link = '<a href="javascript:pjump('."'color_custom','".$desc.
               "','".$curcol."','".$role.'_'.$item."','parmform.pres','psub'".
               ');">'.$desc.'</a>';
    return $link;
}

sub color_pick_js {
    my $pjump_def = &Apache::lonhtmlcommon::pjump_javascript_definition();
    my $output = <<"ENDCOL";
    function pclose() {
        parmwin=window.open("/adm/rat/empty.html","LONCAPAparms","height=350,width=350,scrollbars=no,menubar=no");
        parmwin.close();
    }

    $pjump_def

    function psub() {
        pclose();
        if (document.parmform.pres_marker.value!='') {
            if (document.parmform.pres_type.value!='') {
                eval('document.display.'+
                     document.parmform.pres_marker.value+
                     '.value=document.parmform.pres_value.value;');
            }
        } else {
            document.parmform.pres_value.value='';
            document.parmform.pres_marker.value='';
        }
    }
ENDCOL
    return $output;
}

sub print_quotas {
    my ($dom,$settings) = @_;
    my $datatable;
    my ($othertitle,$usertypes,$types) = &sorted_inst_types($dom);
    my $typecount = 0;
    my $css_class;
    if (@{$types} > 0) {
        foreach my $type (@{$types}) {
            if (defined($usertypes->{$type})) {
                $typecount ++;
                $css_class = $typecount%2?' class="LC_odd_row"':'';
                $datatable .= '<tr'.$css_class.'>'. 
                              '<td>'.$usertypes->{$type}.'</td>'.
                              '<td class="LC_right_item"><span class="LC_nobreak">'.
                              '<input type="text" name="quota_'.$type.
                              '" value="'.$settings->{$type}.
                              '" size="5" /> Mb</span></td></tr>';
            }
        }
    }
    my $defaultquota = '20';
    if (ref($settings) eq 'HASH') {
        if (defined($settings->{'default'})) {
            $defaultquota = $settings->{'default'};
        }
    }
    $typecount ++;
    $css_class = $typecount%2?' class="LC_odd_row"':'';
    $datatable .= '<tr'.$css_class.'>'.
                  '<td>'.&mt($othertitle).'</td>'.
                  '<td class="LC_right_item"><span class="LC_nobreak">'.
                  '<input type="text" name="defaultquota" value="'.
                  $defaultquota.'" size="5" /> Mb</span></td></tr>';
    return $datatable;
}

sub print_autoenroll {
    my ($dom,$settings) = @_;
    my $autorun = &Apache::lonnet::auto_run(undef,$dom),
    my ($defdom,$runon,$runoff);
    if (ref($settings) eq 'HASH') {
        if (exists($settings->{'run'})) {
            if ($settings->{'run'} eq '0') {
                $runoff = ' checked="checked" ';
                $runon = ' ';
            } else {
                $runon = ' checked="checked" ';
                $runoff = ' ';
            }
        } else {
            if ($autorun) {
                $runon = ' checked="checked" ';
                $runoff = ' ';
            } else {
                $runoff = ' checked="checked" ';
                $runon = ' ';
            }
        }
        if (exists($settings->{'sender_domain'})) {
            $defdom = $settings->{'sender_domain'};
        }
    } else {
        if ($autorun) {
            $runon = ' checked="checked" ';
            $runoff = ' ';
        } else {
            $runoff = ' checked="checked" ';
            $runon = ' ';
        }
    }
    my $domform = &Apache::loncommon::select_dom_form($defdom,'sender_domain',1);
    my $datatable='<tr class="LC_odd_row">'.
                  '<td>'.&mt('Auto-enrollment active?').'</td>'.
                  '<td class="LC_right_item"><span class="LC_nobreak"><label>'.
                  '<input type="radio" name="autoenroll_run"'.
                  $runon.' value="1" />'.&mt('Yes').'</label>&nbsp;'.
                  '<label><input type="radio" name="autoenroll_run"'.
                  $runoff.' value="0" />'.&mt('No').'</label></span></td>'.
                  '</tr><tr>'.
                  '<td>'.&mt('Notification messages - sender').
                  '</td><td class="LC_right_item"><span class="LC_nobreak">'.
                  &mt('username').':&nbsp;'.
                  '<input type="text" name="sender_uname" value="'.
                  $settings->{'sender_uname'}.
                  '" size="10" />&nbsp;&nbsp;'.&mt('domain').
                  ':&nbsp;'.$domform.'</span></td></tr>';
    return $datatable;
}

sub print_autoupdate {
    my ($position,$dom,$settings) = @_;
    my $datatable;
    if ($position eq 'top') {
        my $updateon = ' ';
        my $updateoff = ' checked="checked" ';
        my $classlistson = ' ';
        my $classlistsoff = ' checked="checked" ';
        if (ref($settings) eq 'HASH') {
            if ($settings->{'run'} eq '1') {
                $updateon = $updateoff;
                $updateoff = ' ';
            }
            if ($settings->{'classlists'} eq '1') {
                $classlistson = $classlistsoff;
                $classlistsoff = ' ';
            }
        }
        my %title = (
                   run => 'Auto-update active?',
                   classlists => 'Update information in classlists?',
                    );
        $datatable = '<tr class="LC_odd_row">'. 
                  '<td>'.&mt($title{'run'}).'</td>'.
                  '<td class="LC_right_item"><span class="LC_nobreak"><label>'.
                  '<input type="radio" name="autoupdate_run"'.
                  $updateon.' value="1" />'.&mt('Yes').'</label>&nbsp;'.
                  '<label><input type="radio" name="autoupdate_run"'.
                  $updateoff.'value="0" />'.&mt('No').'</label></span></td>'.
                  '</tr><tr>'.
                  '<td>'.&mt($title{'classlists'}).'</td>'.
                  '<td class="LC_right_item"><span class="LC_nobreak">'.
                  '<label><input type="radio" name="classlists"'.
                  $classlistson.' value="1" />'.&mt('Yes').'</label>&nbsp;'.
                  '<label><input type="radio" name="classlists"'.
                  $classlistsoff.'value="0" />'.&mt('No').'</label></span></td>'.
                  '</tr>';
    } else {
        my ($othertitle,$usertypes,$types) = &sorted_inst_types($dom);
        my @fields = ('lastname','firstname','middlename','gen',
                      'permanentemail','id');
        my %fieldtitles = &Apache::lonlocal::texthash (
                            id => 'Student/Employee ID',
                            permanentemail => 'E-mail address',
                            lastname => 'Last Name',
                            firstname => 'First Name',
                            middlename => 'Middle Name',
                            gen => 'Generation',
                      );
        my $numrows = 0;
        if (@{$types} > 0) {
            $datatable = 
                &usertype_update_row($settings,$usertypes,\%fieldtitles,
                                     \@fields,$types,\$numrows);
        }
        $datatable .= 
            &usertype_update_row($settings,{'default' => $othertitle},
                                 \%fieldtitles,\@fields,['default'],
                                 \$numrows);
    }
    return $datatable;
}

sub print_directorysrch {
    my ($dom,$settings) = @_;
    my $srchon = ' ';
    my $srchoff = ' checked="checked" ';
    my ($exacton,$containson,$beginson);
    my $localon = ' ';
    my $localoff = ' checked="checked" ';
    if (ref($settings) eq 'HASH') {
        if ($settings->{'available'} eq '1') {
            $srchon = $srchoff;
            $srchoff = ' ';
        }
        if ($settings->{'localonly'} eq '1') {
            $localon = $localoff;
            $localoff = ' ';
        }
        if (ref($settings->{'searchtypes'}) eq 'ARRAY') {
            foreach my $type (@{$settings->{'searchtypes'}}) {
                if ($type eq 'exact') {
                    $exacton = ' checked="checked" ';
                } elsif ($type eq 'contains') {
                    $containson = ' checked="checked" ';
                } elsif ($type eq 'begins') {
                    $beginson = ' checked="checked" ';
                }
            }
        } else {
            if ($settings->{'searchtypes'} eq 'exact') {
                $exacton = ' checked="checked" ';
            } elsif ($settings->{'searchtypes'} eq 'contains') {
                $containson = ' checked="checked" ';
            } elsif ($settings->{'searchtypes'} eq 'specify') {
                $exacton = ' checked="checked" ';
                $containson = ' checked="checked" ';
            }
        }
    }
    my ($searchtitles,$titleorder) = &sorted_searchtitles();
    my ($othertitle,$usertypes,$types) = &sorted_inst_types($dom);

    my $numinrow = 4;
    my $datatable='<tr class="LC_odd_row">'.
                  '<td colspan="2">'.&mt('Directory search available?').'</td>'.
                  '<td class="LC_right_item"><span class="LC_nobreak"><label>'.
                  '<input type="radio" name="dirsrch_available"'.
                  $srchon.' value="1" />'.&mt('Yes').'</label>&nbsp;'.
                  '<label><input type="radio" name="dirsrch_available"'.
                  $srchoff.' value="0" />'.&mt('No').'</label></span></td>'.
                  '</tr><tr>'.
                  '<td colspan="2">'.&mt('Other domains can search?').'</td>'.
                  '<td class="LC_right_item"><span class="LC_nobreak"><label>'.
                  '<input type="radio" name="dirsrch_localonly"'.
                  $localoff.' value="0" />'.&mt('Yes').'</label>&nbsp;'.
                  '<label><input type="radio" name="dirsrch_localonly"'.
                  $localon.' value="1" />'.&mt('No').'</label></span></td>'.
                  '</tr>';
    $datatable .= &users_cansearch_row($settings,$types,$usertypes,$dom,
                                       $numinrow,$othertitle);
    $datatable .= '<tr>'.
                  '<td>'.&mt('Supported search methods').
                  '</td><td class="LC_left_item" colspan="2"><table><tr>';
    foreach my $title (@{$titleorder}) {
        if (defined($searchtitles->{$title})) {
            my $check = ' ';
            if (ref($settings->{'searchby'}) eq 'ARRAY') {
                if (grep(/^\Q$title\E$/,@{$settings->{'searchby'}})) {
                    $check = ' checked="checked" ';
                }
            }
            $datatable .= '<td class="LC_left_item">'.
                          '<span class="LC_nobreak"><label>'.
                          '<input type="checkbox" name="searchby" '.
                          'value="'.$title.'"'.$check.'/>'.
                          $searchtitles->{$title}.'</label></span></td>';
        }
    }
    $datatable .= '</tr></table></td></tr><tr class="LC_odd_row">'. 
                  '<td>'.&mt('Search latitude').'</td>'.
                  '<td class="LC_left_item"  colspan="2">'.
                  '<span class="LC_nobreak"><label>'.
                  '<input type="checkbox" name="searchtypes" '.
                  $exacton.' value="exact" />'.&mt('Exact match').
                  '</label>&nbsp;'.
                  '<label><input type="checkbox" name="searchtypes" '.
                  $beginson.' value="begins" />'.&mt('Begins with').
                  '</label>&nbsp;'.
                  '<label><input type="checkbox" name="searchtypes" '.
                  $containson.' value="contains" />'.&mt('Contains').
                  '</label></span></td></tr>';
    return $datatable;
}

sub users_cansearch_row {
    my ($settings,$types,$usertypes,$dom,$numinrow,$othertitle) = @_;
    my $output =  '<tr class="LC_odd_row">'.
                  '<td>'.&mt('Users allowed to search').' ('.$dom.')'.
                  '</td><td class="LC_left_item" colspan="2"><table>';
    for (my $i=0; $i<@{$types}; $i++) {
        if (defined($usertypes->{$types->[$i]})) {
            my $rem = $i%($numinrow);
            if ($rem == 0) {
                if ($i > 0) {
                    $output .= '</tr>';
                }
                $output .= '<tr>';
            }
            my $check = ' ';
            if (ref($settings->{'cansearch'}) eq 'ARRAY') {
                if (grep(/^\Q$types->[$i]\E$/,@{$settings->{'cansearch'}})) {
                    $check = ' checked="checked" ';
                }
            }
            $output .= '<td class="LC_left_item">'.
                       '<span class="LC_nobreak"><label>'.
                       '<input type="checkbox" name="cansearch" '.
                       'value="'.$types->[$i].'"'.$check.'/>'.
                       $usertypes->{$types->[$i]}.'</label></span></td>';
        }
    }

    my $rem = @{$types}%($numinrow);
    my $colsleft = $numinrow - $rem;
    if ($colsleft > 1) {
        $output .= '<td colspan="'.$colsleft.'" class="LC_left_item">';
    } else {
        $output .= '<td class="LC_left_item">';
    }
    my $defcheck = ' ';
    if (ref($settings->{'cansearch'}) eq 'ARRAY') {
         if (grep(/^default$/,@{$settings->{'cansearch'}})) {
              $defcheck = ' checked="checked" ';
         }
    }
    $output .= '<span class="LC_nobreak"><label>'.
               '<input type="checkbox" name="cansearch" '.
               'value="default"'.$defcheck.'/>'.
               $othertitle.'</label></span></td>'.
               '</tr></table></td></tr>';
    return $output;
}

sub sorted_inst_types {
    my ($dom) = @_;
    my ($usertypes,$order) = &Apache::lonnet::retrieve_inst_usertypes($dom);
    my $othertitle = "All users";
    my @types;
    if (ref($order) eq 'ARRAY') {
        @types = @{$order};
    }
    if (@types == 0) {
        if (ref($usertypes) eq 'HASH') {
            @types = sort(keys(%{$usertypes}));
        }
    }
    if (keys(%{$usertypes}) > 0) {
        $othertitle = &mt('Other users');
    }
    return ($othertitle,$usertypes,\@types);
}

sub sorted_searchtitles {
    my %searchtitles = &Apache::lonlocal::texthash(
                         'uname' => 'username',
                         'lastname' => 'last name',
                         'lastfirst' => 'last name, first name',
                     );
    my @titleorder = ('uname','lastname','lastfirst');
    return (\%searchtitles,\@titleorder);
}

sub sorted_searchtypes {
    my %srchtypes_desc = (
                           exact    => 'is exact match',
                           contains => 'contains ..',
                           begins   => 'begins with ..',
                         );
    my @srchtypeorder = ('exact','begins','contains');
    return (\%srchtypes_desc,\@srchtypeorder);
}

sub usertype_update_row {
    my ($settings,$usertypes,$fieldtitles,$fields,$types,$rownums) = @_;
    my $datatable;
    my $numinrow = 4;
    foreach my $type (@{$types}) {
        if (defined($usertypes->{$type})) {
            $$rownums ++;
            my $css_class = $$rownums%2?' class="LC_odd_row"':'';
            $datatable .= '<tr'.$css_class.'><td>'.$usertypes->{$type}.
                          '</td><td class="LC_left_item"><table>';
            for (my $i=0; $i<@{$fields}; $i++) {
                my $rem = $i%($numinrow);
                if ($rem == 0) {
                    if ($i > 0) {
                        $datatable .= '</tr>';
                    }
                    $datatable .= '<tr>';
                }
                my $check = ' ';
                if (ref($settings->{'fields'}) eq 'HASH') {
                    if (ref($settings->{'fields'}{$type}) eq 'ARRAY') {
                        if (grep(/^\Q$fields->[$i]\E$/,@{$settings->{'fields'}{$type}})) {
                            $check = ' checked="checked" ';
                        }
                    }
                }

                if ($i == @{$fields}-1) {
                    my $colsleft = $numinrow - $rem;
                    if ($colsleft > 1) {
                        $datatable .= '<td colspan="'.$colsleft.'">';
                    } else {
                        $datatable .= '<td>';
                    }
                } else {
                    $datatable .= '<td>';
                }
                $datatable .= '<span class="LC_nobreak"><label>'.
                              '<input type="checkbox" name="updateable_'.$type.
                              '_'.$fields->[$i].'" value="1"'.$check.'/>'.
                              $fieldtitles->{$fields->[$i]}.'</label></span></td>';
            }
            $datatable .= '</tr></table></td></tr>';
        }
    }
    return $datatable;
}

sub modify_login {
    my ($r,$dom,$confname,%domconfig) = @_;
    my ($resulttext,$errors,$colchgtext,%changes,%colchanges);
    my %title = ( coursecatalog => 'Display course catalog',
                  adminmail => 'Display administrator E-mail address');
    my @offon = ('off','on');
    my %loginhash;
    ($errors,%colchanges) = &modify_colors($r,$dom,$confname,['login'],
                                           \%domconfig,\%loginhash);
    $loginhash{login}{coursecatalog} = $env{'form.coursecatalog'};
    $loginhash{login}{adminmail} = $env{'form.adminmail'};
    if (ref($colchanges{'login'}) eq 'HASH') {  
        $colchgtext = &display_colorchgs($dom,\%colchanges,['login'],
                                         \%loginhash);
    }
    my $putresult = &Apache::lonnet::put_dom('configuration',\%loginhash,
                                             $dom);
    if ($putresult eq 'ok') {
        if (($domconfig{'login'}{'coursecatalog'} eq '0') &&
            ($env{'form.coursecatalog'} eq '1')) {
            $changes{'coursecatalog'} = 1;
        } elsif (($domconfig{'login'}{'coursecatalog'} eq '' ||
                 $domconfig{'login'}{'coursecatalog'} eq '1') &&
                 ($env{'form.coursecatalog'} eq '0')) {
            $changes{'coursecatalog'} = 1;
        }
        if (($domconfig{'login'}{'adminmail'} eq '1') &&
                ($env{'form.adminmail'} eq '0')) {
            $changes{'adminmail'} = 1;
        } elsif (($domconfig{'login'}{'adminmail'} eq '' ||
                 $domconfig{'login'}{'adminmail'} eq '0') &&
                 ($env{'form.adminmail'} eq '1')) {
            $changes{'adminmail'} = 1;
        }
        if (keys(%changes) > 0 || $colchgtext) {
            $resulttext = &mt('Changes made:').'<ul>';
            foreach my $item (sort(keys(%changes))) {
                $resulttext .= '<li>'.&mt("$title{$item} set to $offon[$env{'form.'.$item}]").'</li>';
            }
            $resulttext .= $colchgtext.'</ul>';
        } else {
            $resulttext = &mt('No changes made to log-in page settings');
        }
    } else {
        $resulttext = '<span class="LC_error">'.
	    &mt('An error occurred: [_1]',$putresult).'</span>';
    }
    if ($errors) {
        $resulttext .= '<br />'.&mt('The following errors occurred: ').'<ul>'.
                       $errors.'</ul>';
    }
    return $resulttext;
}

sub color_font_choices {
    my %choices =
        &Apache::lonlocal::texthash (
            img => "Header",
            bgs => "Background colors",
            links => "Link colors",
            font => "Font color",
            pgbg => "Page",
            tabbg => "Header",
            sidebg => "Border",
            link => "Link",
            alink => "Active link",
            vlink => "Visited link",
        );
    return %choices;
}

sub modify_rolecolors {
    my ($r,$dom,$confname,$roles,%domconfig) = @_;
    my ($resulttext,%rolehash);
    $rolehash{'rolecolors'} = {};
    my ($errors,%changes) = &modify_colors($r,$dom,$confname,$roles,
                         $domconfig{'rolecolors'},$rolehash{'rolecolors'});
    my $putresult = &Apache::lonnet::put_dom('configuration',\%rolehash,
                                             $dom);
    if ($putresult eq 'ok') {
        if (keys(%changes) > 0) {
            $resulttext = &display_colorchgs($dom,\%changes,$roles,
                                             $rolehash{'rolecolors'});
        } else {
            $resulttext = &mt('No changes made to default color schemes');
        }
    } else {
        $resulttext = '<span class="LC_error">'.
	    &mt('An error occurred: [_1]',$putresult).'</span>';
    }
    if ($errors) {
        $resulttext .= &mt('The following errors occurred: ').'<ul>'.
                       $errors.'</ul>';
    }
    return $resulttext;
}

sub modify_colors {
    my ($r,$dom,$confname,$roles,$domconfig,$confhash) = @_;
    my (%changes,%choices);
    my @bgs = ('pgbg','mainbg','sidebg');
    my @links = ('link','alink','vlink');
    my @images;
    my $servadm = $r->dir_config('lonAdmEMail');
    my $errors;
    foreach my $role (@{$roles}) {
        if ($role eq 'login') {
            %choices = &login_choices();
        } else {
            %choices = &color_font_choices();
        }
        if ($role eq 'login') {
            @images = ('img','logo','domlogo');
        } else {
            @images = ('img');
        }
        $confhash->{$role}{'font'} = $env{'form.'.$role.'_font'};
        foreach my $item (@bgs,@links) {
            $confhash->{$role}{$item} = $env{'form.'.$role.'_'.$item};
        }
        my ($configuserok,$author_ok,$switchserver,%currroles);
        my $uhome = &Apache::lonnet::homeserver($confname,$dom,1);
        ($configuserok,%currroles) = &check_configuser($uhome,$dom,
                                                       $confname,$servadm);
        if ($configuserok eq 'ok') {
            $switchserver = &check_switchserver($dom,$confname);
            if ($switchserver eq '') {
                $author_ok = &check_authorstatus($dom,$confname,%currroles);
            }
        }
        my ($width,$height) = &thumb_dimensions();
        foreach my $img (@images) {
	    if ( ! $env{'form.'.$role.'_'.$img.'.filename'} 
		 && !defined($domconfig->{$role}{$img})
		 && !$env{'form.'.$role.'_del_'.$img}
		 && $env{'form.'.$role.'_import_'.$img}) {
		# import the old configured image from the .tab setting
		# if they haven't provided a new one 
		$domconfig->{$role}{$img} = 
		    $env{'form.'.$role.'_import_'.$img};
	    }
            if ($env{'form.'.$role.'_'.$img.'.filename'} ne '') {
                my $error;
                if ($configuserok eq 'ok') {
                    if ($switchserver) {
                        $error = &mt("Upload of [_1] image for $role page(s) is not permitted to this server: [_2]",$choices{$img},$switchserver);
                    } else {
                        if ($author_ok eq 'ok') {
                            my ($result,$logourl) = 
                                &publishlogo($r,'upload',$role.'_'.$img,
                                           $dom,$confname,$img,$width,$height);
                            if ($result eq 'ok') {
                                $confhash->{$role}{$img} = $logourl;
                                $changes{$role}{'images'}{$img} = 1;
                            } else {
                                $error = &mt("Upload of [_1] image for $role page(s) failed because an error occurred publishing the file in RES space. Error was: [_2].",$choices{img},$result);
                            }
                        } else {
                            $error = &mt("Upload of [_1] image for $role page(s) failed because an author role could not be assigned to a Domain Configuation user ([_2]) in domain: [_3].  Error was: [_4].",$choices{$img},$confname,$dom,$author_ok);
                        }
                    }
                } else {
                    $error = &mt("Upload of [_1] image for $role page(s) failed because a Domain Configuation user ([_2]) could not be created in domain: [_3].  Error was: [_4].",$choices{$img},$confname,$dom,$configuserok);
                }
                if ($error) {
                    &Apache::lonnet::logthis($error);
                    $errors .= '<li><span class="LC_error">'.$error.'</span></li>';
                }
            } elsif ($domconfig->{$role}{$img} ne '') {
                if ($domconfig->{$role}{$img} !~ m-^(/res/\Q$dom\E/\Q$confname\E/\Q$img\E)/([^/]+)$-) {
                    my $error;
                    if ($configuserok eq 'ok') {
# is confname an author?
                        if ($switchserver eq '') {
                            if ($author_ok eq 'ok') {
                                my ($result,$logourl) = 
                               &publishlogo($r,'copy',$domconfig->{$role}{$img},
                                            $dom,$confname,$img,$width,$height);
                                if ($result eq 'ok') {
                                    $confhash->{$role}{$img} = $logourl;
				    $changes{$role}{'images'}{$img} = 1;
                                }
                            }
                        }
                    }
                }
            }
        }
        if (ref($domconfig) eq 'HASH') {
            if (ref($domconfig->{$role}) eq 'HASH') {
                foreach my $img (@images) {
                    if ($domconfig->{$role}{$img} ne '') {
                        if ($env{'form.'.$role.'_del_'.$img}) {
                            $confhash->{$role}{$img} = '';
                            $changes{$role}{'images'}{$img} = 1;
                        } else {
                            if ($confhash->{$role}{$img} eq '') {
                                $confhash->{$role}{$img} = $domconfig->{$role}{$img};
                            }
                        }
                    } else {
                        if ($env{'form.'.$role.'_del_'.$img}) {
                            $confhash->{$role}{$img} = '';
                            $changes{$role}{'images'}{$img} = 1;
                        } 
                    }
                }  
                if ($domconfig->{$role}{'font'} ne '') {
                    if ($confhash->{$role}{'font'} ne $domconfig->{$role}{'font'}) {
                        $changes{$role}{'font'} = 1;
                    }
                } else {
                    if ($confhash->{$role}{'font'}) {
                        $changes{$role}{'font'} = 1;
                    }
                }
                foreach my $item (@bgs) {
                    if ($domconfig->{$role}{$item} ne '') {
                        if ($confhash->{$role}{$item} ne $domconfig->{$role}{$item}) {
                            $changes{$role}{'bgs'}{$item} = 1;
                        } 
                    } else {
                        if ($confhash->{$role}{$item}) {
                            $changes{$role}{'bgs'}{$item} = 1;
                        }
                    }
                }
                foreach my $item (@links) {
                    if ($domconfig->{$role}{$item} ne '') {
                        if ($confhash->{$role}{$item} ne $domconfig->{$role}{$item}) {
                            $changes{$role}{'links'}{$item} = 1;
                        }
                    } else {
                        if ($confhash->{$role}{$item}) {
                            $changes{$role}{'links'}{$item} = 1;
                        }
                    }
                }
            } else {
                &default_change_checker($role,\@images,\@links,\@bgs,
                                        $confhash,\%changes); 
            }
        } else {
            &default_change_checker($role,\@images,\@links,\@bgs,
                                    $confhash,\%changes); 
        }
    }
    return ($errors,%changes);
}

sub default_change_checker {
    my ($role,$images,$links,$bgs,$confhash,$changes) = @_;
    foreach my $item (@{$links}) {
        if ($confhash->{$role}{$item}) {
            $changes->{$role}{'links'}{$item} = 1;
        }
    }
    foreach my $item (@{$bgs}) {
        if ($confhash->{$role}{$item}) {
            $changes->{$role}{'bgs'}{$item} = 1;
        }
    }
    foreach my $img (@{$images}) {
        if ($env{'form.'.$role.'_del_'.$img}) {
            $confhash->{$role}{$img} = '';
            $changes->{$role}{'images'}{$img} = 1;
        }
    }
    if ($confhash->{$role}{'font'}) {
        $changes->{$role}{'font'} = 1;
    }
} 

sub display_colorchgs {
    my ($dom,$changes,$roles,$confhash) = @_;
    my (%choices,$resulttext);
    &Apache::loncommon::devalidate_domconfig_cache($dom);
    if (!grep(/^login$/,@{$roles})) {
        $resulttext = &mt('Changes made:').'<br />';
    }
    foreach my $role (@{$roles}) {
        if ($role eq 'login') {
            %choices = &login_choices();
        } else {
            %choices = &color_font_choices();
        }
        if (ref($changes->{$role}) eq 'HASH') {
            if ($role ne 'login') {
                $resulttext .= '<h4>'.&mt($role).'</h4>';
            }
            foreach my $key (sort(keys(%{$changes->{$role}}))) {
                if ($role ne 'login') {
                    $resulttext .= '<ul>';
                }
                if (ref($changes->{$role}{$key}) eq 'HASH') {
                    if ($role ne 'login') {
                        $resulttext .= '<li>'.&mt($choices{$key}).':<ul>';
                    }
                    foreach my $item (sort(keys(%{$changes->{$role}{$key}}))) {
                        if ($confhash->{$role}{$item} eq '') {
                            $resulttext .= '<li>'.&mt("$choices{$item} set to default").'</li>';
                        } else {
                            my $newitem = $confhash->{$role}{$item};
                            if ($key eq 'images') {
                                $newitem = '<img src="'.$confhash->{$role}{$item}.'" alt="'.$choices{$item}.'" valign="bottom" />';
                            }
                            $resulttext .= '<li>'.&mt("$choices{$item} set to [_1]",$newitem).'</li>';
                        }
                    }
                    if ($role ne 'login') {
                        $resulttext .= '</ul></li>';
                    }
                } else {
                    if ($confhash->{$role}{$key} eq '') {
                        $resulttext .= '<li>'.&mt("$choices{$key} set to default").'</li>';
                    } else {
                        $resulttext .= '<li>'.&mt("$choices{$key} set to [_1]",$confhash->{$role}{$key}).'</li>';
                    }
                }
                if ($role ne 'login') {
                    $resulttext .= '</ul>';
                }
            }
        }
    }
    return $resulttext;
}

sub thumb_dimensions {
    return ('200','50');
}

sub check_dimensions {
    my ($inputfile) = @_;
    my ($fullwidth,$fullheight);
    if ($inputfile =~ m|^[/\w.\-]+$|) {
        if (open(PIPE,"identify $inputfile 2>&1 |")) {
            my $imageinfo = <PIPE>;
            if (!close(PIPE)) {
                &Apache::lonnet::logthis("Failed to close PIPE opened to retrieve image information for $inputfile");
            }
            chomp($imageinfo);
            my ($fullsize) = 
                ($imageinfo =~ /^\Q$inputfile\E\s+\w+\s+(\d+x\d+)/);
            if ($fullsize) {
                ($fullwidth,$fullheight) = split(/x/,$fullsize);
            }
        }
    }
    return ($fullwidth,$fullheight);
}

sub check_configuser {
    my ($uhome,$dom,$confname,$servadm) = @_;
    my ($configuserok,%currroles);
    if ($uhome eq 'no_host') {
        srand( time() ^ ($$ + ($$ << 15))  ); # Seed rand.
        my $configpass = &LONCAPA::Enrollment::create_password();
        $configuserok = 
            &Apache::lonnet::modifyuser($dom,$confname,'','internal',
                             $configpass,'','','','','',undef,$servadm);
    } else {
        $configuserok = 'ok';
        %currroles = 
            &Apache::lonnet::get_my_roles($confname,$dom,'userroles');
    }
    return ($configuserok,%currroles);
}

sub check_authorstatus {
    my ($dom,$confname,%currroles) = @_;
    my $author_ok;
    if (!$currroles{':'.$dom.':au'}) {
        my $start = time;
        my $end = 0;
        $author_ok = 
            &Apache::lonnet::assignrole($dom,$confname,'/'.$dom.'/',
                                        'au',$end,$start);
    } else {
        $author_ok = 'ok';
    }
    return $author_ok;
}

sub publishlogo {
    my ($r,$action,$formname,$dom,$confname,$subdir,$thumbwidth,$thumbheight) = @_;
    my ($output,$fname,$logourl);
    if ($action eq 'upload') {
        $fname=$env{'form.'.$formname.'.filename'};
        chop($env{'form.'.$formname});
    } else {
        ($fname) = ($formname =~ /([^\/]+)$/);
    }
    $fname=&Apache::lonnet::clean_filename($fname);
# See if there is anything left
    unless ($fname) { return ('error: no uploaded file'); }
    $fname="$subdir/$fname";
    my $filepath='/home/'.$confname.'/public_html';
    my ($fnamepath,$file,$fetchthumb);
    $file=$fname;
    if ($fname=~m|/|) {
        ($fnamepath,$file) = ($fname =~ m|^(.*)/([^/]+)$|);
    }
    my @parts=split(/\//,$filepath.'/'.$fnamepath);
    my $count;
    for ($count=4;$count<=$#parts;$count++) {
        $filepath.="/$parts[$count]";
        if ((-e $filepath)!=1) {
            mkdir($filepath,02770);
        }
    }
    # Check for bad extension and disallow upload
    if ($file=~/\.(\w+)$/ &&
        (&Apache::loncommon::fileembstyle($1) eq 'hdn')) {
        $output = 
            &mt('Invalid file extension ([_1]) - reserved for LONCAPA use.',$1); 
    } elsif ($file=~/\.(\w+)$/ &&
        !defined(&Apache::loncommon::fileembstyle($1))) {
        $output = &mt('Unrecognized file extension ([_1]) - rename the file with a proper extension and re-upload.',$1);
    } elsif ($file=~/\.(\d+)\.(\w+)$/) {
        $output = &mt('File name not allowed a rename the file to remove the number immediately before the file extension([_1]) and re-upload.',$2);
    } elsif (-d "$filepath/$file") {
        $output = &mt('File name is a directory name - rename the file and re-upload');
    } else {
        my $source = $filepath.'/'.$file;
        my $logfile;
        if (!open($logfile,">>$source".'.log')) {
            return (&mt('No write permission to Construction Space'));
        }
        print $logfile
"\n================= Publish ".localtime()." ================\n".
$env{'user.name'}.':'.$env{'user.domain'}."\n";
# Save the file
        if (!open(FH,'>'.$source)) {
            &Apache::lonnet::logthis('Failed to create '.$source);
            return (&mt('Failed to create file'));
        }
        if ($action eq 'upload') {
            if (!print FH ($env{'form.'.$formname})) {
                &Apache::lonnet::logthis('Failed to write to '.$source);
                return (&mt('Failed to write file'));
            }
        } else {
            my $original = &Apache::lonnet::filelocation('',$formname);
            if(!copy($original,$source)) {
                &Apache::lonnet::logthis('Failed to copy '.$original.' to '.$source);
                return (&mt('Failed to write file'));
            }
        }
        close(FH);
        chmod(0660, $source); # Permissions to rw-rw---.

        my $docroot=$r->dir_config('lonDocRoot');
        my $targetdir=$docroot.'/res/'.$dom.'/'.$confname .'/'.$fnamepath;
        my $copyfile=$targetdir.'/'.$file;

        my @parts=split(/\//,$targetdir);
        my $path="/$parts[1]/$parts[2]/$parts[3]/$parts[4]";
        for (my $count=5;$count<=$#parts;$count++) {
            $path.="/$parts[$count]";
            if (!-e $path) {
                print $logfile "\nCreating directory ".$path;
                mkdir($path,02770);
            }
        }
        my $versionresult;
        if (-e $copyfile) {
            $versionresult = &logo_versioning($targetdir,$file,$logfile);
        } else {
            $versionresult = 'ok';
        }
        if ($versionresult eq 'ok') {
            if (copy($source,$copyfile)) {
                print $logfile "\nCopied original source to ".$copyfile."\n";
                $output = 'ok';
                &write_metadata($dom,$confname,$formname,$targetdir,$file,$logfile);
                $logourl = '/res/'.$dom.'/'.$confname.'/'.$fname;
            } else {
                print $logfile "\nUnable to write ".$copyfile.':'.$!."\n";
                $output = &mt('Failed to copy file to RES space').", $!";
            }
            if (($thumbwidth =~ /^\d+$/) && ($thumbheight =~ /^\d+$/)) {
                my $inputfile = $filepath.'/'.$file;
                my $outfile = $filepath.'/'.'tn-'.$file;
                my ($fullwidth,$fullheight) = &check_dimensions($inputfile);
                if ($fullwidth ne '' && $fullheight ne '') { 
                    if ($fullwidth > $thumbwidth && $fullheight > $thumbheight) {
                        my $thumbsize = $thumbwidth.'x'.$thumbheight;
                        system("convert -sample $thumbsize $inputfile $outfile");
                        chmod(0660, $filepath.'/tn-'.$file);
                        if (-e $outfile) {
                            my $copyfile=$targetdir.'/tn-'.$file;
                            if (copy($outfile,$copyfile)) {
                                print $logfile "\nCopied source to ".$copyfile."\n";
                                &write_metadata($dom,$confname,$formname,
                                                $targetdir,'tn-'.$file,$logfile);
                            } else {
                                print $logfile "\nUnable to write ".$copyfile.
                                               ':'.$!."\n";
                            }
                        }
                    }
                }
            }
        } else {
            $output = $versionresult;
        }
    }
    return ($output,$logourl);
}

sub logo_versioning {
    my ($targetdir,$file,$logfile) = @_;
    my $target = $targetdir.'/'.$file;
    my ($maxversion,$fn,$extn,$output);
    $maxversion = 0;
    if ($file =~ /^(.+)\.(\w+)$/) {
        $fn=$1;
        $extn=$2;
    }
    opendir(DIR,$targetdir);
    while (my $filename=readdir(DIR)) {
        if ($filename=~/\Q$fn\E\.(\d+)\.\Q$extn\E$/) {
            $maxversion=($1>$maxversion)?$1:$maxversion;
        }
    }
    $maxversion++;
    print $logfile "\nCreating old version ".$maxversion."\n";
    my $copyfile=$targetdir.'/'.$fn.'.'.$maxversion.'.'.$extn;
    if (copy($target,$copyfile)) {
        print $logfile "Copied old target to ".$copyfile."\n";
        $copyfile=$copyfile.'.meta';
        if (copy($target.'.meta',$copyfile)) {
            print $logfile "Copied old target metadata to ".$copyfile."\n";
            $output = 'ok';
        } else {
            print $logfile "Unable to write metadata ".$copyfile.':'.$!."\n";
            $output = &mt('Failed to copy old meta').", $!, ";
        }
    } else {
        print $logfile "Unable to write ".$copyfile.':'.$!."\n";
        $output = &mt('Failed to copy old target').", $!, ";
    }
    return $output;
}

sub write_metadata {
    my ($dom,$confname,$formname,$targetdir,$file,$logfile) = @_;
    my (%metadatafields,%metadatakeys,$output);
    $metadatafields{'title'}=$formname;
    $metadatafields{'creationdate'}=time;
    $metadatafields{'lastrevisiondate'}=time;
    $metadatafields{'copyright'}='public';
    $metadatafields{'modifyinguser'}=$env{'user.name'}.':'.
                                         $env{'user.domain'};
    $metadatafields{'authorspace'}=$confname.':'.$dom;
    $metadatafields{'domain'}=$dom;
    {
        print $logfile "\nWrite metadata file for ".$targetdir.'/'.$file;
        my $mfh;
        unless (open($mfh,'>'.$targetdir.'/'.$file.'.meta')) {
            $output = &mt('Could not write metadata');
        }
        foreach (sort keys %metadatafields) {
            unless ($_=~/\./) {
                my $unikey=$_;
                $unikey=~/^([A-Za-z]+)/;
                my $tag=$1;
                $tag=~tr/A-Z/a-z/;
                print $mfh "\n\<$tag";
                foreach (split(/\,/,$metadatakeys{$unikey})) {
                    my $value=$metadatafields{$unikey.'.'.$_};
                    $value=~s/\"/\'\'/g;
                    print $mfh ' '.$_.'="'.$value.'"';
                }
                print $mfh '>'.
                    &HTML::Entities::encode($metadatafields{$unikey},'<>&"')
                        .'</'.$tag.'>';
            }
        }
        $output = 'ok';
        print $logfile "\nWrote metadata";
        close($mfh);
    }
}

sub check_switchserver {
    my ($dom,$confname) = @_;
    my ($allowed,$switchserver);
    my $home = &Apache::lonnet::homeserver($confname,$dom);
    if ($home eq 'no_host') {
        $home = &Apache::lonnet::domain($dom,'primary');
    }
    my @ids=&Apache::lonnet::current_machine_ids();
    foreach my $id (@ids) { if ($id eq $home) { $allowed=1; } }
    if (!$allowed) {
	$switchserver='<a href="/adm/switchserver?otherserver='.$home.'&role=dc./'.$dom.'/">'.&mt('Switch Server').'</a>';
    }
    return $switchserver;
}

sub modify_quotas {
    my ($dom,%domconfig) = @_;
    my ($resulttext,%changes);
    my ($usertypes,$order) = 
        &Apache::lonnet::retrieve_inst_usertypes($dom);
    my %formhash;
    foreach my $key (keys(%env)) {
        if ($key =~ /^form\.quota_(.+)$/) {
            $formhash{$1} = $env{$key};
        }
    }
    $formhash{'default'} = $env{'form.defaultquota'};
    if (ref($domconfig{'quotas'}) eq 'HASH') {
        foreach my $key (keys(%{$domconfig{'quotas'}})) {
            if (exists($formhash{$key})) {
                if ($formhash{$key} ne $domconfig{'quotas'}{$key}) {
                    $changes{$key} = 1;
                }
            } else {
                $formhash{$key} = $domconfig{'quotas'}{$key};
            }
        }
    }
    foreach my $key (keys(%formhash)) {
        if ($formhash{$key} ne '') {
            if (!exists($domconfig{'quotas'}{$key})) {
                $changes{$key} = 1;
            }
        }
    }
    my %quotahash = (
                      quotas => {%formhash},
                    );
    my $putresult = &Apache::lonnet::put_dom('configuration',\%quotahash,
                                             $dom);
    if ($putresult eq 'ok') {
        if (keys(%changes) > 0) {
            $resulttext = &mt('Changes made:').'<ul>';
            foreach my $item (sort(keys(%changes))) {
                $resulttext .= '<li>'.&mt('[_1] set to [_2] Mb',$usertypes->{$item},$formhash{$item}).'</li>';
            }
            $resulttext .= '</ul>';
        } else {
            $resulttext = &mt('No changes made to default quotas');
        }
    } else {
        $resulttext = '<span class="LC_error">'.
	    &mt('An error occurred: [_1]',$putresult).'</span>';
    }
    return $resulttext;
}

sub modify_autoenroll {
    my ($dom,%domconfig) = @_;
    my ($resulttext,%changes);
    my %currautoenroll;
    if (ref($domconfig{'autoenroll'}) eq 'HASH') {
        foreach my $key (keys(%{$domconfig{'autoenroll'}})) {
            $currautoenroll{$key} = $domconfig{'autoenroll'}{$key};
        }
    }
    my $autorun = &Apache::lonnet::auto_run(undef,$dom),
    my %title = ( run => 'Auto-enrollment active',
                  sender => 'Sender for notification messages');
    my @offon = ('off','on');
    my $sender_uname = $env{'form.sender_uname'};
    my $sender_domain = $env{'form.sender_domain'};
    if ($sender_domain eq '') {
        $sender_uname = '';
    } elsif ($sender_uname eq '') {
        $sender_domain = '';
    }
    my %autoenrollhash =  (
                       autoenroll => { run => $env{'form.autoenroll_run'},
                                       sender_uname => $sender_uname,
                                       sender_domain => $sender_domain,

                                }
                     );
    my $putresult = &Apache::lonnet::put_dom('configuration',\%autoenrollhash,
                                             $dom);
    if ($putresult eq 'ok') {
        if (exists($currautoenroll{'run'})) {
             if ($currautoenroll{'run'} ne $env{'form.autoenroll_run'}) {
                 $changes{'run'} = 1;
             }
        } elsif ($autorun) {
            if ($env{'form.autoenroll_run'} ne '1') {
                 $changes{'run'} = 1;
            }
        }
        if ($currautoenroll{'sender_uname'} ne $sender_uname) {
            $changes{'sender'} = 1;
        }
        if ($currautoenroll{'sender_domain'} ne $sender_domain) {
            $changes{'sender'} = 1;
        }
        if (keys(%changes) > 0) {
            $resulttext = &mt('Changes made:').'<ul>';
            if ($changes{'run'}) {
                $resulttext .= '<li>'.&mt("$title{'run'} set to $offon[$env{'form.autoenroll_run'}]").'</li>';
            }
            if ($changes{'sender'}) {
                if ($sender_uname eq '' || $sender_domain eq '') {
                    $resulttext .= '<li>'.&mt("$title{'sender'} set to default (course owner).").'</li>';
                } else {
                    $resulttext .= '<li>'.&mt("$title{'sender'} set to [_1]",$sender_uname.':'.$sender_domain).'</li>';
                }
            }
            $resulttext .= '</ul>';
        } else {
            $resulttext = &mt('No changes made to auto-enrollment settings');
        }
    } else {
        $resulttext = '<span class="LC_error">'.
	    &mt('An error occurred: [_1]',$putresult).'</span>';
    }
    return $resulttext;
}

sub modify_autoupdate {
    my ($dom,%domconfig) = @_;
    my ($resulttext,%currautoupdate,%fields,%changes);
    if (ref($domconfig{'autoupdate'}) eq 'HASH') {
        foreach my $key (keys(%{$domconfig{'autoupdate'}})) {
            $currautoupdate{$key} = $domconfig{'autoupdate'}{$key};
        }
    }
    my @offon = ('off','on');
    my %title = &Apache::lonlocal::texthash (
                   run => 'Auto-update:',
                   classlists => 'Updates to user information in classlists?'
                );
    my ($usertypes,$order) = &Apache::lonnet::retrieve_inst_usertypes($dom);
    my %fieldtitles = &Apache::lonlocal::texthash (
                        id => 'Student/Employee ID',
                        permanentemail => 'E-mail address',
                        lastname => 'Last Name',
                        firstname => 'First Name',
                        middlename => 'Middle Name',
                        gen => 'Generation',
                      );
    my $othertitle = &mt('All users');
    if (keys(%{$usertypes}) >  0) {
        $othertitle = "Other users";
    }
    foreach my $key (keys(%env)) {
        if ($key =~ /^form\.updateable_(.+)_([^_]+)$/) {
            push(@{$fields{$1}},$2);
        }
    }
    my %updatehash = (
                      autoupdate => { run => $env{'form.autoupdate_run'},
                                      classlists => $env{'form.classlists'},
                                      fields => {%fields},
                                    }
                     );
    foreach my $key (keys(%currautoupdate)) {
        if (($key eq 'run') || ($key eq 'classlists')) {
            if (exists($updatehash{autoupdate}{$key})) {
                if ($currautoupdate{$key} ne $updatehash{autoupdate}{$key}) {
                    $changes{$key} = 1;
                }
            }
        } elsif ($key eq 'fields') {
            if (ref($currautoupdate{$key}) eq 'HASH') {
                foreach my $item (keys(%{$currautoupdate{$key}})) {
                    if (ref($currautoupdate{$key}{$item}) eq 'ARRAY') {
                        my $change = 0;
                        foreach my $type (@{$currautoupdate{$key}{$item}}) {
                            if (!exists($fields{$item})) {
                                $change = 1;
                            } elsif (ref($fields{$item}) eq 'ARRAY') {
                                if (!grep/^\Q$type\E$/,@{$fields{$item}}) {
                                    $change = 1;
                                }
                            }
                        }
                        if ($change) {
                            push(@{$changes{$key}},$item);
                        }
                    }
                }
            }
        }
    }
    foreach my $key (keys(%fields)) {
        if (ref($currautoupdate{'fields'}) eq 'HASH') {
            if (!exists($currautoupdate{'fields'}{$key})) {
                push(@{$changes{'fields'}},$key);
            }
        } else {
            push(@{$changes{'fields'}},$key);
        }
    }
    my $putresult = &Apache::lonnet::put_dom('configuration',\%updatehash,
                                             $dom);
    if ($putresult eq 'ok') {
        if (keys(%changes) > 0) {
            $resulttext = &mt('Changes made:').'<ul>';
            foreach my $key (sort(keys(%changes))) {
                if (ref($changes{$key}) eq 'ARRAY') {
                    foreach my $item (@{$changes{$key}}) {
                        my @newvalues;
                        foreach my $type (@{$fields{$item}}) {
                            push(@newvalues,$fieldtitles{$type});
                        }
                        my $newvaluestr;
                        if (@newvalues > 0) {
                            $newvaluestr = join(', ',@newvalues);
                        } else {
                            $newvaluestr = &mt('none');
                        }
                        if ($item eq 'default') {
                            $resulttext .= '<li>'.&mt("Updates for $othertitle set to: [_1]",$newvaluestr).'</li>';
                        } else {
                            $resulttext .= '<li>'.&mt("Updates for [_1] set to: [_2]",$usertypes->{$item},$newvaluestr).'</li>';
                        }
                    }
                } else {
                    my $newvalue;
                    if ($key eq 'run') {
                        $newvalue = $offon[$env{'form.autoupdate_run'}];
                    } else {
                        $newvalue = $offon[$env{'form.'.$key}];
                    }
                    $resulttext .= '<li>'.&mt("[_1] set to $newvalue",$title{$key}).'</li>';
                }
            }
            $resulttext .= '</ul>';
        } else {
            $resulttext = &mt('No changes made to autoupdates');
        }
    } else {
        $resulttext = '<span class="LC_error">'.
	    &mt('An error occurred: [_1]',$putresult).'</span>';
    }
    return $resulttext;
}

sub modify_directorysrch {
    my ($dom,%domconfig) = @_;
    my ($resulttext,%changes);
    my %currdirsrch;
    if (ref($domconfig{'directorysrch'}) eq 'HASH') {
        foreach my $key (keys(%{$domconfig{'directorysrch'}})) {
            $currdirsrch{$key} = $domconfig{'directorysrch'}{$key};
        }
    }
    my %title = ( available => 'Directory search available',
                  cansearch => 'Users permitted to search',
                  localonly => 'Other domains can search',
                  searchby => 'Search types',
                  searchtypes => 'Search latitude');
    my @offon = ('off','on');
    my @otherdoms = ('Yes','No');

    my @searchtypes = &Apache::loncommon::get_env_multiple('form.searchtypes');  
    my @cansearch = &Apache::loncommon::get_env_multiple('form.cansearch');
    my @searchby = &Apache::loncommon::get_env_multiple('form.searchby');

    if (ref($currdirsrch{'cansearch'}) eq 'ARRAY') {
        foreach my $type (@{$currdirsrch{'cansearch'}}) {
            if (!grep(/^\Q$type\E$/,@cansearch)) {
                push(@{$changes{'cansearch'}},$type);
            }
        }
        foreach my $type (@cansearch) {
            if (!grep(/^\Q$type\E$/,@{$currdirsrch{'cansearch'}})) {
                push(@{$changes{'cansearch'}},$type);
            }
        }
    } else {
        push(@{$changes{'cansearch'}},@cansearch);
    }

    if (ref($currdirsrch{'searchby'}) eq 'ARRAY') {
        foreach my $by (@{$currdirsrch{'searchby'}}) {
            if (!grep(/^\Q$by\E$/,@searchby)) {
                push(@{$changes{'searchby'}},$by);
            }
        }
        foreach my $by (@searchby) {
            if (!grep(/^\Q$by\E$/,@{$currdirsrch{'searchby'}})) {
                push(@{$changes{'searchby'}},$by);
            }
        }
    } else {
        push(@{$changes{'searchby'}},@searchby);
    }

    if (ref($currdirsrch{'searchtypes'}) eq 'ARRAY') {
        foreach my $type (@{$currdirsrch{'searchtypes'}}) {
            if (!grep(/^\Q$type\E$/,@searchtypes)) {
                push(@{$changes{'searchtypes'}},$type);
            }
        }
        foreach my $type (@searchtypes) {
            if (!grep(/^\Q$type\E$/,@{$currdirsrch{'searchtypes'}})) {
                push(@{$changes{'searchtypes'}},$type);
            }
        }
    } else {
        if (exists($currdirsrch{'searchtypes'})) {
            foreach my $type (@searchtypes) {  
                if ($type ne $currdirsrch{'searchtypes'}) { 
                    push(@{$changes{'searchtypes'}},$type);
                }
            }
            if (!grep(/^\Q$currdirsrch{'searchtypes'}\E/,@searchtypes)) {
                push(@{$changes{'searchtypes'}},$currdirsrch{'searchtypes'});
            }   
        } else {
            push(@{$changes{'searchtypes'}},@searchtypes); 
        }
    }

    my %dirsrch_hash =  (
            directorysrch => { available => $env{'form.dirsrch_available'},
                               cansearch => \@cansearch,
                               localonly => $env{'form.dirsrch_localonly'},
                               searchby => \@searchby,
                               searchtypes => \@searchtypes,
                             }
            );
    my $putresult = &Apache::lonnet::put_dom('configuration',\%dirsrch_hash,
                                             $dom);
    if ($putresult eq 'ok') {
        if (exists($currdirsrch{'available'})) {
             if ($currdirsrch{'available'} ne $env{'form.dirsrch_available'}) {
                 $changes{'available'} = 1;
             }
        } else {
            if ($env{'form.dirsrch_available'} eq '1') {
                $changes{'available'} = 1;
            }
        }
        if (exists($currdirsrch{'localonly'})) {
             if ($currdirsrch{'localonly'} ne $env{'form.dirsrch_localonly'}) {
                 $changes{'localonly'} = 1;
             }
        } else {
            if ($env{'form.dirsrch_localonly'} eq '1') {
                $changes{'localonly'} = 1;
            }
        }
        if (keys(%changes) > 0) {
            $resulttext = &mt('Changes made:').'<ul>';
            if ($changes{'available'}) {
                $resulttext .= '<li>'.&mt("$title{'available'} set to: $offon[$env{'form.dirsrch_available'}]").'</li>';
            }
            if ($changes{'localonly'}) {
                $resulttext .= '<li>'.&mt("$title{'localonly'} set to: $otherdoms[$env{'form.dirsrch_localonly'}]").'</li>';
            }

            if (ref($changes{'cansearch'}) eq 'ARRAY') {
                my ($othertitle,$usertypes,$types) = &sorted_inst_types($dom);
                my $chgtext;
                foreach my $type (@cansearch) {
                    if (defined($usertypes->{$type})) {
                        $chgtext .= $usertypes->{$type}.'; ';
                    }
                }
                if (grep(/^default$/,@cansearch)) {
                    $chgtext .= $othertitle;
                } else {
                    $chgtext =~ s/\; $//;
                }
                $resulttext .= '<li>'.&mt("$title{'cansearch'} ([_1]) set to: [_2]",$dom,$chgtext).'</li>';
            }
            if (ref($changes{'searchby'}) eq 'ARRAY') {
                my ($searchtitles,$titleorder) = &sorted_searchtitles();
                my $chgtext;
                foreach my $type (@{$titleorder}) {
                    if (grep(/^\Q$type\E$/,@searchby)) {
                        if (defined($searchtitles->{$type})) {
                            $chgtext .= $searchtitles->{$type}.'; ';
                        }
                    }
                }
                $chgtext =~ s/\; $//;
                $resulttext .= '<li>'.&mt("$title{'searchby'} set to: [_1]",$chgtext).'</li>';
            }
            if (ref($changes{'searchtypes'}) eq 'ARRAY') {
                my ($srchtypes_desc,$srchtypeorder) = &sorted_searchtypes(); 
                my $chgtext;
                foreach my $type (@{$srchtypeorder}) {
                    if (grep(/^\Q$type\E$/,@searchtypes)) {
                        if (defined($srchtypes_desc->{$type})) {
                            $chgtext .= $srchtypes_desc->{$type}.'; ';
                        }
                    }
                }
                $chgtext =~ s/\; $//;
                $resulttext .= '<li>'.&mt("$title{'searchtypes'} set to: \"[_1]\"",$chgtext).'</li>';
            }
            $resulttext .= '</ul>';
        } else {
            $resulttext = &mt('No changes made to institution directory search settings');
        }
    } else {
        $resulttext = '<span class="LC_error">'.
            &mt('An error occurred: [_1]',$putresult).'</span>';
    }
    return $resulttext;
}

1;
