# The LearningOnline Network with CAPA
# Construct and maintain state and binary representation of course for user
#
# $Id: lonuserstate.pm,v 1.119 2007/06/28 22:16:53 albertel Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###

package Apache::lonuserstate;

# ------------------------------------------------- modules used by this module
use strict;
use HTML::TokeParser;
use Apache::lonnet;
use Apache::lonlocal;
use Apache::loncommon();
use GDBM_File;
use Apache::lonmsg;
use Safe;
use Safe::Hole;
use Opcode;
use Apache::lonenc;
use Fcntl qw(:flock);
use LONCAPA;
 

# ---------------------------------------------------- Globals for this package

my $pc;      # Package counter
my %hash;    # The big tied hash
my %parmhash;# The hash with the parameters
my @cond;    # Array with all of the conditions
my $errtext; # variable with all errors
my $retfrid; # variable with the very first RID in the course
my $retfurl; # first URL
my %randompick; # randomly picked resources
my %randompickseed; # optional seed for randomly picking resources
my %encurl; # URLs in this folder are supposed to be encrypted
my %hiddenurl; # this URL (or complete folder) is supposed to be hidden

# ----------------------------------- Remove version from URL and store in hash

sub versiontrack {
    my $uri=shift;
    if ($uri=~/\.(\d+)\.\w+$/) {
	my $version=$1;
	$uri=~s/\.\d+\.(\w+)$/\.$1/;
        unless ($hash{'version_'.$uri}) {
	    $hash{'version_'.$uri}=$version;
	}
    }
    return $uri;
}

# -------------------------------------------------------------- Put in version

sub putinversion {
    my $uri=shift;
    my $key=$env{'request.course.id'}.'_'.&Apache::lonnet::clutter($uri);
    if ($hash{'version_'.$uri}) {
	my $version=$hash{'version_'.$uri};
	if ($version eq 'mostrecent') { return $uri; }
	if ($version eq &Apache::lonnet::getversion(
			&Apache::lonnet::filelocation('',$uri))) 
	             { return $uri; }
	$uri=~s/\.(\w+)$/\.$version\.$1/;
    }
    &Apache::lonnet::do_cache_new('courseresversion',$key,&Apache::lonnet::declutter($uri),600);
    return $uri;
}

# ----------------------------------------- Processing versions file for course

sub processversionfile {
    my %cenv=@_;
    my %versions=&Apache::lonnet::dump('resourceversions',
				       $cenv{'domain'},
				       $cenv{'num'});
    foreach my $ver (keys(%versions)) {
	if ($ver=~/^error\:/) { return; }
	$hash{'version_'.$ver}=$versions{$ver};
    }
}

# --------------------------------------------------------- Loads map from disk

sub loadmap { 
    my $uri=shift;
    if ($hash{'map_pc_'.$uri}) { 
	$errtext.=&mt('<br />Multiple use of sequence/page <tt>[_1]</tt>! The course will not function properly.',$uri);
	return; 
    }
    $pc++;
    my $lpc=$pc;
    $hash{'map_pc_'.$uri}=$lpc;
    $hash{'map_id_'.$lpc}=$uri;

# Determine and check filename
    my $fn=&Apache::lonnet::filelocation('',&putinversion($uri));

    my $ispage=($fn=~/\.page$/);

    unless (($fn=~/\.sequence$/) ||
            ($fn=~/\.page$/)) { 
	$errtext.=&mt("<br />Invalid map: <tt>[_1]</tt>",$fn);
	return; 
    }

    my $instr=&Apache::lonnet::getfile($fn);

    unless ($instr eq -1) {

# Successfully got file, parse it

        my $parser = HTML::TokeParser->new(\$instr);
	$parser->attr_encoded(1);
        my $token;

        my $linkpc=0;

        $fn=~/\.(\w+)$/;

        $hash{'map_type_'.$lpc}=$1;

        while ($token = $parser->get_token) {
	    if ($token->[0] eq 'S') {
                if ($token->[1] eq 'resource') {
# -------------------------------------------------------------------- Resource
                    if ($token->[2]->{'type'} eq 'zombie') { next; }
                    my $rid=$lpc.'.'.$token->[2]->{'id'};

                    $hash{'kind_'.$rid}='res';
                    $hash{'title_'.$rid}=$token->[2]->{'title'};
                    my $turi=&versiontrack($token->[2]->{'src'});
                    if ($token->[2]->{'version'}) {
			unless ($hash{'version_'.$turi}) {
			    $hash{'version_'.$turi}=$1;
			}
		    }
		    my $title=$token->[2]->{'title'};
		    $title=~s/\&colon\;/\:/gs;
#		    my $symb=&Apache::lonnet::encode_symb($uri,
#							  $token->[2]->{'id'},
#							  $turi);
#		    &Apache::lonnet::do_cache_new('title',$symb,$title);
                    unless ($ispage) {
                        $turi=~/\.(\w+)$/;
                        my $embstyle=&Apache::loncommon::fileembstyle($1);
                        if ($token->[2]->{'external'} eq 'true') { # external
                            $turi=~s/^http\:\/\//\/adm\/wrapper\/ext\//;
                        } elsif ($turi=~/^\/*uploaded\//) { # uploaded
			    if (($embstyle eq 'img') 
				|| ($embstyle eq 'emb')
				|| ($embstyle eq 'wrp')) {
                                $turi='/adm/wrapper'.$turi;
			    } elsif ($embstyle eq 'ssi') {
				#do nothing with these
			    } elsif ($turi!~/\.(sequence|page)$/) {
				$turi='/adm/coursedocs/showdoc'.$turi;
                            }
                        } elsif ($turi=~/\S/) { # normal non-empty internal resource
			    my $mapdir=$uri;
			    $mapdir=~s/[^\/]+$//;
			    $turi=&Apache::lonnet::hreflocation($mapdir,$turi);
			    if (($embstyle eq 'img') 
				|| ($embstyle eq 'emb')
				|| ($embstyle eq 'wrp')) {
				$turi='/adm/wrapper'.$turi;
			    }
                        }
		    }
# Store reverse lookup, remove query string
		    my $idsuri=$turi;
		    $idsuri=~s/\?.+$//;
                    if (defined($hash{'ids_'.$idsuri})) {
                        $hash{'ids_'.$idsuri}.=','.$rid;
                    } else {
                        $hash{'ids_'.$idsuri}=''.$rid;
                    }
               
                    if ($turi=~/\/(syllabus|aboutme|navmaps|smppg|bulletinboard)$/) {
			$turi.='?register=1';
		    }

                    $hash{'src_'.$rid}=$turi;

                    if ($token->[2]->{'external'} eq 'true') {
                        $hash{'ext_'.$rid}='true:';
                    } else {
                        $hash{'ext_'.$rid}='false:';
                    }
                    if ($token->[2]->{'type'}) {
			$hash{'type_'.$rid}=$token->[2]->{'type'};
                        if ($token->[2]->{'type'} eq 'start') {
			    $hash{'map_start_'.$uri}="$rid";
                        }
                        if ($token->[2]->{'type'} eq 'finish') {
			    $hash{'map_finish_'.$uri}="$rid";
                        }
                    }  else {
                        $hash{'type_'.$rid}='normal';
                    }

                    if (($turi=~/\.sequence$/) ||
                        ($turi=~/\.page$/)) {
                        $hash{'is_map_'.$rid}=1;
                        &loadmap($turi);
                    } 
                    
                } elsif ($token->[1] eq 'condition') {
# ------------------------------------------------------------------- Condition

                    my $rid=$lpc.'.'.$token->[2]->{'id'};

                    $hash{'kind_'.$rid}='cond';
                    $cond[$#cond+1]=$token->[2]->{'value'};
                    $hash{'condid_'.$rid}=$#cond;
                    if ($token->[2]->{'type'}) {
                        $cond[$#cond].=':'.$token->[2]->{'type'};
                    }  else {
                        $cond[$#cond].=':normal';
                    }

                } elsif ($token->[1] eq 'link') {
# ----------------------------------------------------------------------- Links

                    $linkpc++;
                    my $linkid=$lpc.'.'.$linkpc;

                    my $goesto=$lpc.'.'.$token->[2]->{'to'};
                    my $comesfrom=$lpc.'.'.$token->[2]->{'from'};
                    my $undercond=0;

                    if ($token->[2]->{'condition'}) {
			$undercond=$lpc.'.'.$token->[2]->{'condition'};
                    }

                    $hash{'goesto_'.$linkid}=$goesto;
                    $hash{'comesfrom_'.$linkid}=$comesfrom;
                    $hash{'undercond_'.$linkid}=$undercond;

                    if (defined($hash{'to_'.$comesfrom})) {
                        $hash{'to_'.$comesfrom}.=','.$linkid;
                    } else {
                        $hash{'to_'.$comesfrom}=''.$linkid;
                    }
                    if (defined($hash{'from_'.$goesto})) {
                        $hash{'from_'.$goesto}.=','.$linkid;
                    } else {
                        $hash{'from_'.$goesto}=''.$linkid;
                    }
                } elsif ($token->[1] eq 'param') {
# ------------------------------------------------------------------- Parameter

                    my $referid=$lpc.'.'.$token->[2]->{'to'};
		    my $name=$token->[2]->{'name'};
		    my $part;
		    if ($name=~/^parameter_(.*)_/) {
			$part=$1;
		    } else {
			$part=0;
		    }
		    $name=~s/^.*_([^_]*)$/$1/;
                    my $newparam=
			&escape($token->[2]->{'type'}).':'.
			&escape($part.'.'.$name).'='.
			&escape($token->[2]->{'value'});
                    if (defined($hash{'param_'.$referid})) {
                        $hash{'param_'.$referid}.='&'.$newparam;
                    } else {
                        $hash{'param_'.$referid}=''.$newparam;
                    }
                    if ($token->[2]->{'name'}=~/^parameter_(0_)*mapalias$/) {
			$hash{'mapalias_'.$token->[2]->{'value'}}=$referid;
                    }
                    if ($token->[2]->{'name'}=~/^parameter_(0_)*randompick$/) {
			$randompick{$referid}=$token->[2]->{'value'};
                    }
                    if ($token->[2]->{'name'}=~/^parameter_(0_)*randompickseed$/) {
			$randompickseed{$referid}=$token->[2]->{'value'};
                    }
                    if ($token->[2]->{'name'}=~/^parameter_(0_)*encrypturl$/) {
			if ($token->[2]->{'value'}=~/^yes$/i) {
			    $encurl{$referid}=1;
			}
                    }
                    if ($token->[2]->{'name'}=~/^parameter_(0_)*hiddenresource$/) {
			if ($token->[2]->{'value'}=~/^yes$/i) {
			    $hiddenurl{$referid}=1;
			}
                    }
                } 
		
            }
        }

    } else {
        $errtext.=&mt('<br />Map not loaded: The file <tt>[_1]</tt> does not exist.',$fn);
    }
}

# --------------------------------------------------------- Simplify expression

sub simplify {
    my $expression=shift;
# (0&1) = 1
    $expression=~s/\(0\&([_\.\d]+)\)/$1/g;
# (8)=8
    $expression=~s/\(([_\.\d]+)\)/$1/g;
# 8&8=8
    $expression=~s/([^_\.\d])([_\.\d]+)\&\2([^_\.\d])/$1$2$3/g;
# 8|8=8
    $expression=~s/([^_\.\d])([_\.\d]+)\|\2([^_\.\d])/$1$2$3/g;
# (5&3)&4=5&3&4
    $expression=~s/\(([_\.\d]+)((?:\&[_\.\d]+)+)\)\&([_\.\d]+[^_\.\d])/$1$2\&$3/g;
# (((5&3)|(4&6)))=((5&3)|(4&6))
    $expression=~
	s/\((\(\([_\.\d]+(?:\&[_\.\d]+)*\)(?:\|\([_\.\d]+(?:\&[_\.\d]+)*\))+\))\)/$1/g;
# ((5&3)|(4&6))|(1&2)=(5&3)|(4&6)|(1&2)
    $expression=~
	s/\((\([_\.\d]+(?:\&[_\.\d]+)*\))((?:\|\([_\.\d]+(?:\&[_\.\d]+)*\))+)\)\|(\([_\.\d]+(?:\&[_\.\d]+)*\))/\($1$2\|$3\)/g;
    return $expression;
}

# -------------------------------------------------------- Build condition hash

sub traceroute {
    my ($sofar,$rid,$beenhere,$encflag,$hdnflag)=@_;
    my $newsofar=$sofar=simplify($sofar);
    unless ($beenhere=~/\&$rid\&/) {
	$beenhere.=$rid.'&';  
	my ($mapid,$resid)=split(/\./,$rid);
	my $symb=&Apache::lonnet::encode_symb($hash{'map_id_'.$mapid},$resid,$hash{'src_'.$rid});
	my $hidden=&Apache::lonnet::EXT('resource.0.hiddenresource',$symb);

	if ($hdnflag || lc($hidden) eq 'yes') {
	    $hiddenurl{$rid}=1;
	}
	if (!$hdnflag && lc($hidden) eq 'no') {
	    delete($hiddenurl{$rid});
	}

	my $encrypt=&Apache::lonnet::EXT('resource.0.encrypturl',$symb);
	if ($encflag || lc($encrypt) eq 'yes') { $encurl{$rid}=1; }
	if (($retfrid eq '') && ($hash{'src_'.$rid})
	    && ($hash{'src_'.$rid}!~/\.sequence$/)) {
	    $retfrid=$rid;
	}
	if (defined($hash{'conditions_'.$rid})) {
	    $hash{'conditions_'.$rid}=simplify(
           '('.$hash{'conditions_'.$rid}.')|('.$sofar.')');
	} else {
	    $hash{'conditions_'.$rid}=$sofar;
	}

	# if the expression is just the 0th condition keep it
	# otherwise leave a pointer to this condition expression
	$newsofar = ($sofar eq '0') ? $sofar : '_'.$rid;

	if (defined($hash{'is_map_'.$rid})) {
	    if (defined($hash{'map_start_'.$hash{'src_'.$rid}})) {
		$sofar=$newsofar=
		    &traceroute($sofar,
				$hash{'map_start_'.$hash{'src_'.$rid}},'&',
				$encflag || $encurl{$rid},
				$hdnflag || $hiddenurl{$rid});
	    }
	}
	if (defined($hash{'to_'.$rid})) {
	    foreach my $id (split(/\,/,$hash{'to_'.$rid})) {
		my $further=$sofar;
                if ($hash{'undercond_'.$id}) {
		    if (defined($hash{'condid_'.$hash{'undercond_'.$id}})) {
			$further=simplify('('.'_'.$rid.')&('.
					  $hash{'condid_'.$hash{'undercond_'.$id}}.')');
		    } else {
			$errtext.=&mt('<br />Undefined condition ID: [_1]',$hash{'undercond_'.$id});
		    }
                }
                $newsofar=&traceroute($further,$hash{'goesto_'.$id},$beenhere,
				      $encflag,$hdnflag);
	    }
	}
    }
    return $newsofar;
}

# ------------------------------ Cascading conditions, quick access, parameters

sub accinit {
    my ($uri,$short,$fn)=@_;
    my %acchash=();
    my %captured=();
    my $condcounter=0;
    $acchash{'acc.cond.'.$short.'.0'}=0;
    foreach my $key (keys(%hash)) {
	if ($key=~/^conditions/) {
	    my $expr=$hash{$key};
	    # try to find and factor out common sub-expressions
	    foreach my $sub ($expr=~m/(\(\([_\.\d]+(?:\&[_\.\d]+)+\)(?:\|\([_\.\d]+(?:\&[_\.\d]+)+\))+\))/g) {
		my $orig=$sub;

		my ($factor) = ($sub=~/\(\(([_\.\d]+\&(:?[_\.\d]+\&)*)(?:[_\.\d]+\&*)+\)(?:\|\(\1(?:[_\.\d]+\&*)+\))+\)/);
		next if (!defined($factor));

		$sub=~s/\Q$factor\E//g;
		$sub=~s/^\(/\($factor\(/;
		$sub.=')';
		$sub=simplify($sub);
		$expr=~s/\Q$orig\E/$sub/;
	    }
	    $hash{$key}=$expr;
	    unless (defined($captured{$expr})) {
		$condcounter++;
		$captured{$expr}=$condcounter;
		$acchash{'acc.cond.'.$short.'.'.$condcounter}=$expr;
	    } 
	} elsif ($key=~/^param_(\d+)\.(\d+)/) {
	    my $prefix=&Apache::lonnet::encode_symb($hash{'map_id_'.$1},$2,
						    $hash{'src_'.$1.'.'.$2});
	    foreach my $param (split(/\&/,$hash{$key})) {
		my ($typename,$value)=split(/\=/,$param);
		my ($type,$name)=split(/\:/,$typename);
		$parmhash{$prefix.'.'.&unescape($name)}=
		    &unescape($value);
		$parmhash{$prefix.'.'.&unescape($name).'.type'}=
		    &unescape($type);
	    }
	}
    }
    foreach my $key (keys(%hash)) {
	if ($key=~/^ids/) {
	    foreach my $resid (split(/\,/,$hash{$key})) {
		my $uri=$hash{'src_'.$resid};
		my ($uripath,$urifile) =
		    &Apache::lonnet::split_uri_for_cond($uri);
		if ($uripath) {
		    my $uricond='0';
		    if (defined($hash{'conditions_'.$resid})) {
			$uricond=$captured{$hash{'conditions_'.$resid}};
		    }
		    if (defined($acchash{'acc.res.'.$short.'.'.$uripath})) {
			if ($acchash{'acc.res.'.$short.'.'.$uripath}=~
			    /(\&\Q$urifile\E\:[^\&]*)/) {
			    my $replace=$1;
			    my $regexp=$replace;
			    #$regexp=~s/\|/\\\|/g;
			    $acchash{'acc.res.'.$short.'.'.$uripath} =~
				s/\Q$regexp\E/$replace\|$uricond/;
			} else {
			    $acchash{'acc.res.'.$short.'.'.$uripath}.=
				$urifile.':'.$uricond.'&';
			}
		    } else {
			$acchash{'acc.res.'.$short.'.'.$uripath}=
			    '&'.$urifile.':'.$uricond.'&';
		    }
		} 
	    }
	}
    }
    $acchash{'acc.res.'.$short.'.'}='&:0&';
    my $courseuri=$uri;
    $courseuri=~s/^\/res\///;
    &Apache::lonnet::delenv('(acc\.|httpref\.)');
    &Apache::lonnet::appenv(%acchash);
}

# ---------------- Selectively delete from randompick maps and hidden url parms

sub hiddenurls {
    my $randomoutentry='';
    foreach my $rid (keys %randompick) {
        my $rndpick=$randompick{$rid};
        my $mpc=$hash{'map_pc_'.$hash{'src_'.$rid}};
# ------------------------------------------- put existing resources into array
        my @currentrids=();
        foreach my $key (sort(keys(%hash))) {
	    if ($key=~/^src_($mpc\.\d+)/) {
		if ($hash{'src_'.$1}) { push @currentrids, $1; }
            }
        }
	# rids are number.number and we want to numercially sort on 
        # the second number
	@currentrids=sort {
	    my (undef,$aid)=split(/\./,$a);
	    my (undef,$bid)=split(/\./,$b);
	    $aid <=> $bid;
	} @currentrids;
        next if ($#currentrids<$rndpick);
# -------------------------------- randomly eliminate the ones that should stay
	my (undef,$id)=split(/\./,$rid);
        if ($randompickseed{$rid}) { $id=$randompickseed{$rid}; }
	my $rndseed=&Apache::lonnet::rndseed($id); # use id instead of symb
	&Apache::lonnet::setup_random_from_rndseed($rndseed);
	my @whichids=&Math::Random::random_permuted_index($#currentrids+1);
        for (my $i=1;$i<=$rndpick;$i++) { $currentrids[$whichids[$i]]=''; }
	#&Apache::lonnet::logthis("$id,$rndseed,".join(':',@whichids));
# -------------------------------------------------------- delete the leftovers
        for (my $k=0; $k<=$#currentrids; $k++) {
            if ($currentrids[$k]) {
		$hash{'randomout_'.$currentrids[$k]}=1;
                my ($mapid,$resid)=split(/\./,$currentrids[$k]);
                $randomoutentry.='&'.
		    &Apache::lonnet::encode_symb($hash{'map_id_'.$mapid},
						 $resid,
						 $hash{'src_'.$currentrids[$k]}
						 ).'&';
            }
        }
    }
# ------------------------------ take care of explicitly hidden urls or folders
    foreach my $rid (keys %hiddenurl) {
	$hash{'randomout_'.$rid}=1;
	my ($mapid,$resid)=split(/\./,$rid);
	$randomoutentry.='&'.
	    &Apache::lonnet::encode_symb($hash{'map_id_'.$mapid},$resid,
					 $hash{'src_'.$rid}).'&';
    }
# --------------------------------------- append randomout entry to environment
    if ($randomoutentry) {
	&Apache::lonnet::appenv('acc.randomout' => $randomoutentry);
    }
}

# ---------------------------------------------------- Read map and all submaps

sub readmap {
    my $short=shift;
    $short=~s/^\///;
    my %cenv=&Apache::lonnet::coursedescription($short,{'freshen_cache'=>1});
    my $fn=$cenv{'fn'};
    my $uri;
    $short=~s/\//\_/g;
    unless ($uri=$cenv{'url'}) { 
	&Apache::lonnet::logthis("<font color=blue>WARNING: ".
				 "Could not load course $short.</font>"); 
	return ('',&mt('No course data available.'));;
    }
    @cond=('true:normal');

    open(LOCKFILE,">$fn.db.lock");
    my $lock=0;
    if (flock(LOCKFILE,LOCK_EX|LOCK_NB)) {
	$lock=1;
	unlink($fn.'.db');
	unlink($fn.'_symb.db');
	unlink($fn.'.state');
	unlink($fn.'parms.db');
    }
    undef %randompick;
    undef %hiddenurl;
    undef %encurl;
    $retfrid='';
    if ($lock && (tie(%hash,'GDBM_File',"$fn.db",&GDBM_WRCREAT(),0640)) &&
	(tie(%parmhash,'GDBM_File',$fn.'_parms.db',&GDBM_WRCREAT(),0640))) {
	%hash=();
	%parmhash=();
	$errtext='';
	$pc=0;
	&processversionfile(%cenv);
	my $furi=&Apache::lonnet::clutter($uri);
	$hash{'src_0.0'}=&versiontrack($furi);
	$hash{'title_0.0'}=&Apache::lonnet::metadata($uri,'title');
	$hash{'ids_'.$furi}='0.0';
	$hash{'is_map_0.0'}=1;
	loadmap($uri);
	if (defined($hash{'map_start_'.$uri})) {
	    &Apache::lonnet::appenv("request.course.id"  => $short,
				    "request.course.fn"  => $fn,
				    "request.course.uri" => $uri);
	    $env{'request.course.id'}=$short;
	    &traceroute('0',$hash{'map_start_'.$uri},'&');
	    &accinit($uri,$short,$fn);
	    &hiddenurls();
	}
# ------------------------------------------------------- Put versions into src
	foreach my $key (keys(%hash)) {
	    if ($key=~/^src_/) {
		$hash{$key}=&putinversion($hash{$key});
	    } elsif ($key =~ /^(map_(?:start|finish|pc)_)(.*)/) {
		my ($type, $url) = ($1,$2);
		my $value = $hash{$key};
		$hash{$type.&putinversion($url)}=$value;
	    }
	}
# ---------------------------------------------------------------- Encrypt URLs
	foreach my $id (keys(%encurl)) {
#	    $hash{'src_'.$id}=&Apache::lonenc::encrypted($hash{'src_'.$id});
	    $hash{'encrypted_'.$id}=1;
	}
# ----------------------------------------------- Close hashes to finally store
# --------------------------------- Routine must pass this point, no early outs
	$hash{'first_rid'}=$retfrid;
	my ($mapid,$resid)=split(/\./,$retfrid);
	$hash{'first_mapurl'}=$hash{'map_id_'.$mapid};
	my $symb=&Apache::lonnet::encode_symb($hash{'map_id_'.$mapid},$resid,$hash{'src_'.$retfrid});
	$retfurl=&add_get_param($hash{'src_'.$retfrid},{ 'symb' => $symb });
	if ($hash{'encrypted_'.$retfrid}) {
	    $retfurl=&Apache::lonenc::encrypted($retfurl,(&Apache::lonnet::allowed('adv') ne 'F'));
	}
	$hash{'first_url'}=$retfurl;
	unless ((untie(%hash)) && (untie(%parmhash))) {
	    &Apache::lonnet::logthis("<font color=blue>WARNING: ".
				     "Could not untie coursemap $fn for $uri.</font>"); 
	}
# ---------------------------------------------------- Store away initial state
	{
	    my $cfh;
	    if (open($cfh,">$fn.state")) {
		print $cfh join("\n",@cond);
	    } else {
		&Apache::lonnet::logthis("<font color=blue>WARNING: ".
					 "Could not write statemap $fn for $uri.</font>"); 
	    }
	}
	flock(LOCKFILE,LOCK_UN);
	close(LOCKFILE);
    } else {
	# if we are here it is likely because we are already trying to 
	# initialize the course in another child, busy wait trying to 
	# tie the hashes for the next 90 seconds, if we succeed forward 
	# them on to navmaps, if we fail, throw up the Could not init 
	# course screen
	if ($lock) {
	    # Got the lock but not the DB files
	    flock(LOCKFILE,LOCK_UN);
	}
	untie(%hash);
	untie(%parmhash);
	&Apache::lonnet::logthis("<font color=blue>WARNING: ".
				 "Could not tie coursemap $fn for $uri.</font>"); 
	my $i=0;
	while($i<90) {
	    $i++;
	    sleep(1);
	    if (flock(LOCKFILE,LOCK_EX|LOCK_NB) &&
		(tie(%hash,'GDBM_File',"$fn.db",&GDBM_READER(),0640))) {
		if (tie(%parmhash,'GDBM_File',$fn.'_parms.db',&GDBM_READER(),0640)) {
		    $retfurl='/adm/navmaps';
		    &Apache::lonnet::appenv("request.course.id"  => $short,
					    "request.course.fn"  => $fn,
					    "request.course.uri" => $uri);
		    untie(%hash);
		    untie(%parmhash);
		    last;
		}
	    }
	    untie(%hash);
	    untie(%parmhash);
	}
	flock(LOCKFILE,LOCK_UN);
	close(LOCKFILE);
    }
    &Apache::lonmsg::author_res_msg($env{'request.course.uri'},$errtext);
# ------------------------------------------------- Check for critical messages

    my @what=&Apache::lonnet::dump('critical',$env{'user.domain'},
				   $env{'user.name'});
    if ($what[0]) {
	if (($what[0] ne 'con_lost') && ($what[0]!~/^error\:/)) {
	    $retfurl='/adm/email?critical=display';
        }
    }
    return ($retfurl,$errtext);
}

# ------------------------------------------------------- Evaluate state string

sub evalstate {
    my $fn=$env{'request.course.fn'}.'.state';
    my $state='';
    if (-e $fn) {
	my @conditions=();
	{
	    open(my $fh,"<$fn");
	    @conditions=<$fh>;
            close($fh);
	}  
	my $safeeval = new Safe;
	my $safehole = new Safe::Hole;
	$safeeval->permit("entereval");
	$safeeval->permit(":base_math");
	$safeeval->deny(":base_io");
	$safehole->wrap(\&Apache::lonnet::EXT,$safeeval,'&EXT');
	foreach my $line (@conditions) {
	    chomp($line);
	    my ($condition,$weight)=split(/\:/,$line);
	    if ($safeeval->reval($condition)) {
		if ($weight eq 'force') {
		    $state.='3';
		} else {
		    $state.='2';
		}
	    } else {
		if ($weight eq 'stop') {
		    $state.='0';
		} else {
		    $state.='1';
		}
	    }
	}
    }
    &Apache::lonnet::appenv('user.state.'.$env{'request.course.id'} => $state);
    return $state;
}

1;
__END__

=head1 NAME

Apache::lonuserstate - Construct and maintain state and binary representation
of course for user

=head1 SYNOPSIS

Invoked by lonroles.pm.

&Apache::lonuserstate::readmap($cdom.'/'.$cnum);

=head1 INTRODUCTION

This module constructs and maintains state and binary representation
of course for user.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 HANDLER SUBROUTINE

There is no handler subroutine.

=head1 OTHER SUBROUTINES

=over 4

=item *

loadmap() : Loads map from disk

=item *

simplify() : Simplify expression

=item *

traceroute() : Build condition hash

=item *

accinit() : Cascading conditions, quick access, parameters

=item *

readmap() : Read map and all submaps

=item *

evalstate() : Evaluate state string

=back

=cut
