# The LearningOnline Network with CAPA
# User Roles Screen
#
# $Id: lonroles.pm,v 1.179 2007/09/10 16:07:32 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###

package Apache::lonroles;

use strict;
use Apache::lonnet;
use Apache::lonuserstate();
use Apache::Constants qw(:common);
use Apache::File();
use Apache::lonmenu;
use Apache::loncommon;
use Apache::lonhtmlcommon;
use Apache::lonannounce;
use Apache::lonlocal;
use Apache::lonpageflip();
use Apache::lonnavdisplay();
use GDBM_File;
use LONCAPA qw(:DEFAULT :match);
 

sub redirect_user {
    my ($r,$title,$url,$msg,$launch_nav) = @_;
    $msg = $title if (! defined($msg));
    &Apache::loncommon::content_type($r,'text/html');
    &Apache::loncommon::no_cache($r);
    $r->send_http_header;
    my $swinfo=&Apache::lonmenu::rawconfig();
    my $navwindow;
    if ($launch_nav eq 'on') {
	$navwindow.=&Apache::lonnavdisplay::launch_win('now',undef,undef,
						       ($url =~ m-^/adm/whatsnew-));
    } else {
	$navwindow.=&Apache::lonnavmaps::close();
    }
    my $start_page = &Apache::loncommon::start_page('Switching Role',undef,
						    {'redirect' => [1,$url],});
    my $end_page   = &Apache::loncommon::end_page();

# Note to style police: 
# This must only replace the spaces, nothing else, or it bombs elsewhere.
    $url=~s/ /\%20/g;
    $r->print(<<ENDREDIR);
$start_page
<script type="text/javascript">
$swinfo
</script>
$navwindow
<h1>$msg</h1>
$end_page
ENDREDIR
    return;
}

sub error_page {
    my ($r,$error,$dest)=@_;
    &Apache::loncommon::content_type($r,'text/html');
    &Apache::loncommon::no_cache($r);
    $r->send_http_header;
    return OK if $r->header_only;
    $r->print(&Apache::loncommon::start_page('Problems during Course Initialization').
	      '<script type="text/javascript">'.
	      &Apache::lonmenu::rawconfig().'</script>'.
	      '<p>'.&mt('The following problems occurred:').
	      $error.
	      '</p><br /><a href="'.$dest.'">'.&mt('Continue').'</a>'.
	      &Apache::loncommon::end_page());
}

sub handler {

    my $r = shift;

    my $now=time;
    my $then=$env{'user.login.time'};
    my $envkey;
    my %dcroles = ();
    my $numdc = &check_fordc(\%dcroles,$then);
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'});

# ================================================================== Roles Init
    if ($env{'form.selectrole'}) {
        if ($env{'form.newrole'}) {
            $env{'form.'.$env{'form.newrole'}}=1;
	}
	if ($env{'request.course.id'}) {
	    my %temp=('logout_'.$env{'request.course.id'} => time);
	    &Apache::lonnet::put('email_status',\%temp);
	    &Apache::lonnet::delenv('user.state.'.$env{'request.course.id'});
	}
	&Apache::lonnet::appenv("request.course.id"   => '',
				"request.course.fn"   => '',
				"request.course.uri"  => '',
				"request.course.sec"  => '',
				"request.role"        => 'cm',
                                "request.role.adv"    => $env{'user.adv'},
				"request.role.domain" => $env{'user.domain'});

# Check if user is a DC trying to enter a course and needs privs to be created
        if ($numdc > 0) {
            foreach my $envkey (keys %env) {
                if (my ($domain,$coursenum) =
		    ($envkey =~ m-^form\.cc\./($match_domain)/($match_courseid)$-)) {
                    if ($dcroles{$domain}) {
                        &check_privs($domain,$coursenum,$then,$now);
                    }
                    last;
                }
            }
        }

        foreach $envkey (keys %env) {
            next if ($envkey!~/^user\.role\./);
            my ($where,$trolecode,$role,$tstatus,$tend,$tstart);
            &role_status($envkey,$then,$now,\$role,\$where,\$trolecode,\$tstatus,\$tstart,\$tend);
            if ($env{'form.'.$trolecode}) {
		if ($tstatus eq 'is') {
		    $where=~s/^\///;
		    my ($cdom,$cnum,$csec)=split(/\//,$where);
# check for course groups
                    my %coursegroups = &Apache::lonnet::get_active_groups(
                          $env{'user.domain'},$env{'user.name'},$cdom, $cnum);
                    my $cgrps = join(':',keys(%coursegroups));

# store role if recent_role list being kept
                    if ($env{'environment.recentroles'}) {
                        my %frozen_roles =
                           &Apache::lonhtmlcommon::get_recent_frozen('roles',$env{'environment.recentrolesn'});
			&Apache::lonhtmlcommon::store_recent('roles',
							     $trolecode,' ',$frozen_roles{$trolecode});
                    }


# check for keyed access
		    if (($role eq 'st') && 
                       ($env{'course.'.$cdom.'_'.$cnum.'.keyaccess'} eq 'yes')) {
# who is key authority?
			my $authdom=$cdom;
			my $authnum=$cnum;
			if ($env{'course.'.$cdom.'_'.$cnum.'.keyauth'}) {
			    ($authnum,$authdom)=
				split(/:/,$env{'course.'.$cdom.'_'.$cnum.'.keyauth'});
			}
# check with key authority
			unless (&Apache::lonnet::validate_access_key(
				     $env{'environment.key.'.$cdom.'_'.$cnum},
					     $authdom,$authnum)) {
# there is no valid key
			     if ($env{'form.newkey'}) {
# student attempts to register a new key
				 &Apache::loncommon::content_type($r,'text/html');
				 &Apache::loncommon::no_cache($r);
				 $r->send_http_header;
				 my $swinfo=&Apache::lonmenu::rawconfig();
				 my $start_page=&Apache::loncommon::start_page
				    ('Verifying Access Key to Unlock this Course');
				 my $end_page=&Apache::loncommon::end_page();
				 my $buttontext=&mt('Enter Course');
				 my $message=&mt('Successfully registered key');
				 my $assignresult=
				     &Apache::lonnet::assign_access_key(
						     $env{'form.newkey'},
						     $authdom,$authnum,
						     $cdom,$cnum,
                                                     $env{'user.domain'},
						     $env{'user.name'},
	      'Assigned from '.$ENV{'REMOTE_ADDR'}.' at '.localtime().' for '.
                                                     $trolecode);
				 unless ($assignresult eq 'ok') {
				     $assignresult=~s/^error\:\s*//;
				     $message=&mt($assignresult).
				     '<br /><a href="/adm/logout">'.
				     &mt('Logout').'</a>';
				     $buttontext=&mt('Re-Enter Key');
				 }
				 $r->print(<<ENDENTEREDKEY);
$start_page
<script type="text/javascript">
$swinfo
</script>
<form method="post">
<input type="hidden" name="selectrole" value="1" />
<input type="hidden" name="$trolecode" value="1" />
<font size="+2">$message</font><br />
<input type="submit" value="$buttontext" />
</form>
$end_page
ENDENTEREDKEY
                                 return OK;
			     } else {
# print form to enter a new key
				 &Apache::loncommon::content_type($r,'text/html');
				 &Apache::loncommon::no_cache($r);
				 $r->send_http_header;
				 my $swinfo=&Apache::lonmenu::rawconfig();
				 my $start_page=&Apache::loncommon::start_page
				    ('Enter Access Key to Unlock this Course');
				 my $end_page=&Apache::loncommon::end_page();
				 $r->print(<<ENDENTERKEY);
$start_page
<script type="text/javascript">
$swinfo
</script>
<form method="post">
<input type="hidden" name="selectrole" value="1" />
<input type="hidden" name="$trolecode" value="1" />
<input type="text" size="20" name="newkey" value="$env{'form.newkey'}" />
<input type="submit" value="Enter key" />
</form>
$end_page
ENDENTERKEY
				 return OK;
			     }
			 }
		     }
		    &Apache::lonnet::log($env{'user.domain'},
					 $env{'user.name'},
					 $env{'user.home'},
					 "Role ".$trolecode);
		    
		    &Apache::lonnet::appenv(
					   'request.role'        => $trolecode,
					   'request.role.domain' => $cdom,
					   'request.course.sec'  => $csec,
                                           'request.course.groups' => $cgrps);
                    my $tadv=0;

		    if (($cnum) && ($role ne 'ca') && ($role ne 'aa')) {
                        my $msg;
			my ($furl,$ferr)=
			    &Apache::lonuserstate::readmap($cdom.'/'.$cnum);
			if (($env{'form.orgurl'}) && 
			    ($env{'form.orgurl'}!~/^\/adm\/flip/)) {
			    my $dest=$env{'form.orgurl'};
			    if (&Apache::lonnet::allowed('adv') eq 'F') { $tadv=1; }
			    &Apache::lonnet::appenv('request.role.adv'=>$tadv);
                            if (($ferr) && ($tadv)) {
				&error_page($r,$ferr,$dest);
			    } else {
				$r->internal_redirect($dest);
			    }
			    return OK;
			} else {
			    if (!$env{'request.course.id'}) {
				&Apache::lonnet::appenv(
				      "request.course.id"  => $cdom.'_'.$cnum);
				$furl='/adm/roles?tryagain=1';
				$msg=
				    '<h1><span class="LC_error">'.
				    &mt('Could not initialize [_1] at this time.',
					$env{'course.'.$cdom.'_'.$cnum.'.description'}).
				    '</span></h1><h3>'.&mt('Please try again.').'</h3>'.$ferr;
			    }
			    if (&Apache::lonnet::allowed('adv') eq 'F') { $tadv=1; }
			    &Apache::lonnet::appenv('request.role.adv'=>$tadv);

			    if (($ferr) && ($tadv)) {
				&error_page($r,$ferr,$furl);
			    } else {
				# Check to see if the user is a CC entering a course 
				# for the first time
				my (undef, undef, $role, $courseid) = split(/\./, $envkey);
				if (substr($courseid, 0, 1) eq '/') {
				    $courseid = substr($courseid, 1);
				}
				$courseid =~ s/\//_/;
				if ($role eq 'cc' && $env{'course.' . $courseid . 
							      '.course.helper.not.run'}) {
				    $furl = "/adm/helper/course.initialization.helper";
				    # Send the user to the course they selected
				} elsif ($env{'request.course.id'}) {
				    if (&Apache::lonnet::allowed('whn',
								 $env{'request.course.id'})
					|| &Apache::lonnet::allowed('whn',
								    $env{'request.course.id'}.'/'
								    .$env{'request.course.sec'})
					) {
					my $startpage = &courseloadpage($courseid);
					unless ($startpage eq 'firstres') {         
					    $msg = &mt('Entering [_1] ....',
						       $env{'course.'.$courseid.'.description'});
					    &redirect_user($r,&mt('New in course'),
							   '/adm/whatsnew?refpage=start',$msg,
							   $env{'environment.remotenavmap'});
					    return OK;
					}
				    }
				}
# Are we allowed to look at the first resource?
				if ($furl !~ m|^/adm/|) {
# Guess not ...
				    $furl=&Apache::lonpageflip::first_accessible_resource();
				}
                                $msg = &mt('Entering [_1] ...',
					   $env{'course.'.$courseid.'.description'});
				&redirect_user($r,&mt('Entering [_1]',
						      $env{'course.'.$courseid.'.description'}),
					       $furl,$msg,
					       $env{'environment.remotenavmap'});
			    }
			    return OK;
			}
		    }
                    #
                    # Send the user to the construction space they selected
                    if ($role =~ /^(au|ca|aa)$/) {
                        my $redirect_url = '/priv/';
                        if ($role eq 'au') {
                            $redirect_url.=$env{'user.name'};
                        } else {
                            $where =~ /\/(.*)$/;
                            $redirect_url .= $1;
                        }
                        $redirect_url .= '/';
                        &redirect_user($r,&mt('Entering Construction Space'),
                                       $redirect_url);
                        return OK;
                    }
                    if ($role eq 'dc') {
                        my $redirect_url = '/adm/menu/';
                        &redirect_user($r,&mt('Loading Domain Coordinator Menu'),
                                       $redirect_url);
                        return OK;
                    }
		}
            }
        }
    }


# =============================================================== No Roles Init

    &Apache::loncommon::content_type($r,'text/html');
    &Apache::loncommon::no_cache($r);
    $r->send_http_header;
    return OK if $r->header_only;

    my $swinfo=&Apache::lonmenu::rawconfig();
    my $start_page=&Apache::loncommon::start_page('User Roles');
    my $standby=&mt('Role selected. Please stand by.');
    $standby=~s/\n/\\n/g;
    my $noscript='<span class="LC_error">'.&mt('Use of LON-CAPA requires Javascript to be enabled in your web browser.').'<br />'.&mt('As this is not the case, most functionality in the system will ba unavailable.').'</span><br />';

    $r->print(<<ENDHEADER);
$start_page
<br />
<noscript>
$noscript
</noscript>
<script type="text/javascript">
$swinfo
window.focus();

active=true;

function enterrole (thisform,rolecode,buttonname) {
    if (active) {
	active=false;
        document.title='$standby';
        window.status='$standby';
	thisform.newrole.value=rolecode;
	thisform.submit();
    } else {
       alert('$standby');
    }   
}
</script>
ENDHEADER

# ------------------------------------------ Get Error Message from Environment

    my ($fn,$priv,$nochoose,$error,$msg)=split(/:/,$env{'user.error.msg'});
    if ($env{'user.error.msg'}) {
	$r->log_reason(
   "$msg for $env{'user.name'} domain $env{'user.domain'} access $priv",$fn);
    }

# ------------------------------------------------- Can this user re-init, etc?

    my $advanced=$env{'user.adv'};
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},['tryagain']);
    my $tryagain=$env{'form.tryagain'};

# -------------------------------------------------------- Generate Page Output
# --------------------------------------------------------------- Error Header?
    if ($error) {
	$r->print("<h1>LON-CAPA Access Control</h1>");
	$r->print("<!-- LONCAPAACCESSCONTROLERRORSCREEN --><hr /><pre>");
	if ($priv ne '') {
	    $r->print("Access  : ".&Apache::lonnet::plaintext($priv)."\n");
	}
	if ($fn ne '') {
	    $r->print("Resource: ".&Apache::lonenc::check_encrypt($fn)."\n");
	}
	if ($msg ne '') {
	    $r->print("Action  : $msg\n");
	}
	$r->print("</pre><hr />");
	my $url=$fn;
	my $last;
	if (tie(my %hash,'GDBM_File',$env{'request.course.fn'}.'_symb.db',
		&GDBM_READER(),0640)) {
	    $last=$hash{'last_known'};
	    untie(%hash);
	}
	if ($last) { $fn.='?symb='.&escape($last); }

	&Apache::londocs::changewarning($r,undef,'You have modified your course recently, [_1] may fix this access problem.',
					&Apache::lonenc::check_encrypt($fn));
    } else {
        if ($env{'user.error.msg'}) {
	    $r->print(
 '<h3><span class="LC_error">'.
 &mt('You need to choose another user role or enter a specific course for this function').'</span></h3>');
	}
    }
# -------------------------------------------------------- Choice or no choice?
    if ($nochoose) {
	$r->print("<h2>".&mt('Sorry ...')."</h2>\n<span class='LC_error'>".
		  &mt('This action is currently not authorized.').'</span>'.
		  &Apache::loncommon::end_page());
	return OK;
    } else {
        if (($ENV{'REDIRECT_QUERY_STRING'}) && ($fn)) {
    	    $fn.='?'.$ENV{'REDIRECT_QUERY_STRING'};
        }
        $r->print('<form method="post" name="rolechoice" action="'.(($fn)?$fn:$r->uri).'">');
        $r->print('<input type="hidden" name="orgurl" value="'.$fn.'" />');
        $r->print('<input type="hidden" name="selectrole" value="1" />');
        $r->print('<input type="hidden" name="newrole" value="" />');
    }
    if ($env{'user.adv'}) {
	$r->print(
	      '<br /><span class="LC_rolesinfo"><label>'.&mt('Show all roles').': <input type="checkbox" name="showall"');
	if ($env{'form.showall'}) { $r->print(' checked="checked" '); }
	$r->print(' /></label><input type="submit" value="'.&mt('Display').'" /></span>');
    }

    my (%roletext,%sortrole,%roleclass);
    my $countactive=0;
    my $inrole=0;
    my $possiblerole='';
    foreach $envkey (sort keys %env) {
        my $button = 1;
        my $switchserver='';
	my $roletext;
	my $sortkey;
        if ($envkey=~/^user\.role\./) {
            my ($role,$where,$trolecode,$tstart,$tend,$tremark,$tstatus,$tpstart,$tpend,$tfont);
            &role_status($envkey,$then,$now,\$role,\$where,\$trolecode,\$tstatus,\$tstart,\$tend);
            next if (!defined($role) || $role eq '' || $role =~ /^gr/);
            $tremark='';
            $tpstart='&nbsp;';
            $tpend='&nbsp;';
            $tfont='#000000';
            if ($tstart) {
                $tpstart=&Apache::lonlocal::locallocaltime($tstart);
            }
            if ($tend) {
                $tpend=&Apache::lonlocal::locallocaltime($tend);
            }
            if ($env{'request.role'} eq $trolecode) {
		$tstatus='selected';
            }
            my $tbg;
            if (($tstatus eq 'is') 
		|| ($tstatus eq 'selected') 
		|| ($tstatus eq 'will') 
		|| ($tstatus eq 'future') 
                || ($env{'form.showall'})) {
                if ($tstatus eq 'is') {
                    $tbg='#77FF77';
                    $tfont='#003300';
		    $possiblerole=$trolecode;
		    $countactive++;
                } elsif ($tstatus eq 'future') {
                    $tbg='#FFFF77';
                    $button=0;
                } elsif ($tstatus eq 'will') {
                    $tbg='#FFAA77';
                    $tremark.=&mt('Active at next login. ');
                } elsif ($tstatus eq 'expired') {
                    $tbg='#FF7777';
                    $tfont='#330000';
                    $button=0;
                } elsif ($tstatus eq 'will_not') {
                    $tbg='#AAFF77';
                    $tremark.=&mt('Expired after logout. ');
                } elsif ($tstatus eq 'selected') {
                    $tbg='#11CC55';
                    $tfont='#002200';
		    $inrole=1;
		    $countactive++;
                    $tremark.=&mt('Currently selected. ');
                }
                my $trole;
                if ($role =~ /^cr\//) {
                    my ($rdummy,$rdomain,$rauthor,$rrole)=split(/\//,$role);
		    if ($tremark) { $tremark.='<br />'; }
                    $tremark.=&mt('Defined by ').$rauthor.
			&mt(' at ').$rdomain.'.';
		}
		$trole=Apache::lonnet::plaintext($role);
                my $ttype;
                my $twhere;
                my ($tdom,$trest,$tsection)=
                    split(/\//,Apache::lonnet::declutter($where));
                # First, Co-Authorship roles
                if (($role eq 'ca') || ($role eq 'aa')) {
                    my $home = &Apache::lonnet::homeserver($trest,$tdom);
		    my $allowed=0;
		    my @ids=&Apache::lonnet::current_machine_ids();
		    foreach my $id (@ids) { if ($id eq $home) { $allowed=1; } }
                    if (!$allowed) {
			$button=0;
                        $switchserver='otherserver='.$home.'&role='.$trolecode;
                    }
                    #next if ($home eq 'no_host');
                    $home = &Apache::lonnet::hostname($home);
                    $ttype='Construction Space';
                    $twhere=&mt('User').': '.$trest.'<br />'.&mt('Domain').
			': '.$tdom.'<br />'.
                        ' '.&mt('Server').':&nbsp;'.$home;
                    $env{'course.'.$tdom.'_'.$trest.'.description'}='ca';
		    $tremark.=&Apache::lonhtmlcommon::authorbombs('/res/'.$tdom.'/'.$trest.'/');
		    $sortkey=$role."$trest:$tdom";
                } elsif ($role eq 'au') {
                    # Authors
                    my $home = &Apache::lonnet::homeserver
                        ($env{'user.name'},$env{'user.domain'});
		    my $allowed=0;
		    my @ids=&Apache::lonnet::current_machine_ids();
		    foreach my $id (@ids) { if ($id eq $home) { $allowed=1; } }
                    if (!$allowed) {
			$button=0;
                        $switchserver='otherserver='.$home.'&role='.$trolecode;
                    }
                    #next if ($home eq 'no_host');
                    $home = &Apache::lonnet::hostname($home);
                    $ttype='Construction Space';
                    $twhere=&mt('Domain').': '.$tdom.'<br />'.&mt('Server').
			':&nbsp;'.$home;
                    $env{'course.'.$tdom.'_'.$trest.'.description'}='ca';
		    $tremark.=&Apache::lonhtmlcommon::authorbombs('/res/'.$tdom.'/'.$env{'user.name'}.'/');
		    $sortkey=$role;
                } elsif ($trest) {
                    my $tcourseid=$tdom.'_'.$trest;
                    $ttype = &Apache::loncommon::course_type($tcourseid);
                    $trole = &Apache::lonnet::plaintext($role,$ttype);
                    if ($env{'course.'.$tcourseid.'.description'}) {
                        $twhere=$env{'course.'.$tcourseid.'.description'};
			$sortkey=$role."\0".$tdom."\0".$twhere."\0".$envkey;
                        unless ($twhere eq &mt('Currently not available')) {
			    $twhere.=' <font size="-2">'.
        &Apache::loncommon::syllabuswrapper(&mt('Syllabus'),$trest,$tdom,$tfont).
                                    '</font>';
			}
                    } else {
                        my %newhash=&Apache::lonnet::coursedescription($tcourseid);
                        if (%newhash) {
			    $sortkey=$role."\0".$tdom."\0".$newhash{'description'}.
				"\0".$envkey;
                            $twhere=$newhash{'description'}.
                              ' <font size="-2">'.
        &Apache::loncommon::syllabuswrapper(&mt('Syllabus'),$trest,$tdom,$tfont).
                              '</font>';
                            $ttype = $newhash{'type'};
                            $trole = &Apache::lonnet::plaintext($role,$ttype);
                        } else {
                            $twhere=&mt('Currently not available');
                            $env{'course.'.$tcourseid.'.description'}=$twhere;
			    $sortkey=$role."\0".$tdom."\0".$twhere."\0".$envkey;
                            $ttype = 'Unavailable';
                        }
                    }
                    if ($tsection) {
                        $twhere.='<br />'.&mt('Section').': '.$tsection;
		    }
		    if ($role ne 'st') { $twhere.="<br />".&mt('Domain').":".$tdom; }
                } elsif ($tdom) {
                    $ttype='Domain';
                    $twhere=$tdom;
		    $sortkey=$role.$twhere;
                } else {
                    $ttype='System';
                    $twhere=&mt('system wide');
		    $sortkey=$role.$twhere;
                }
                $roletext.=&build_roletext($trolecode,$tdom,$trest,$tstatus,$tryagain,$advanced,$tremark,$tbg,$tfont,$trole,$twhere,$tpstart,$tpend,$nochoose,$button,$switchserver);
		$roletext{$envkey}=$roletext;
		if (!$sortkey) {$sortkey=$twhere."\0".$envkey;}
		$sortrole{$sortkey}=$envkey;
		$roleclass{$envkey}=$ttype;
	    }
        }
    }
# No active roles
    if ($countactive==0) {
	if ($inrole) {
	    $r->print('<h2>'.&mt('Currently no additional roles or courses').'</h2>');
	} else {
	    $r->print('<h2>'.&mt('Currently no active roles or courses').'</h2>');
	}
	$r->print('</form>'.&Apache::loncommon::end_page());
	return OK;
# Is there only one choice?
    } elsif (($countactive==1) && ($env{'request.role'} eq 'cm')) {
	$r->print('<h3>'.&mt('Please stand by.').'</h3>'.
	    '<input type="hidden" name="'.$possiblerole.'" value="1" />'.
            '<noscript><br /><input type="submit" name="submit" value="'.&mt('Continue').'" /></noscript>');
	$r->print("</form>\n");
	$r->rflush();
	$r->print('<script type="text/javascript">document.forms.rolechoice.submit();</script>');
	$r->print(&Apache::loncommon::end_page());
	return OK;
    }
# More than one possible role
# ----------------------------------------------------------------------- Table
    unless (($advanced) || ($nochoose)) {
	$r->print("<h2>".&mt('Select a Course to Enter')."</h2>\n");
    }
    $r->print('<br /><table id="LC_rolesmenu"><tr>');
    unless ($nochoose) { $r->print('<th>&nbsp;</th>'); }
    $r->print('<th>'.&mt('User Role').'</th><th>'.&mt('Extent').
         '</th><th>'.&mt('Start').'</th><th>'.&mt('End').'</th></tr>'."\n");
    my $doheaders=-1;
    foreach my $type ('Domain','Construction Space','Course','Unavailable','System') {
	my $haverole=0;
	foreach my $which (sort {uc($a) cmp uc($b)} (keys(%sortrole))) {
	    if ($roleclass{$sortrole{$which}} =~ /^\Q$type\E/) { 
		$haverole=1;
	    }
	}
	if ($haverole) { $doheaders++; }
    }

    if ($env{'environment.recentroles'}) {
        my %recent_roles =
               &Apache::lonhtmlcommon::get_recent('roles',$env{'environment.recentrolesn'});
	my $output='';
	foreach (sort(keys(%recent_roles))) {
	    if (defined($roletext{'user.role.'.$_})) {
		$output.=$roletext{'user.role.'.$_};
                if ($_ =~ m-dc\./($match_domain)/- 
		    && $dcroles{$1}) {
		    $output .= &allcourses_row($1,'recent');
                }
	    } elsif ($numdc > 0) {
                unless ($_ =~/^error\:/) {
                    $output.=&display_cc_role('user.role.'.$_);
                }
            } 
	}
	if ($output) {
	    $r->print("<tr><td align='center' colspan='5'><font face='arial'>".
		      &mt('Recent Roles')."</font></td>");
	    $r->print($output);
	    $r->print("</tr>");
            $doheaders ++;
	}
    }

    if ($numdc > 0) {
        $r->print(&coursepick_jscript());
        $r->print(&Apache::loncommon::coursebrowser_javascript());
    }
    foreach my $type ('Construction Space','Domain','Course','Unavailable','System') {
	my $output;
	foreach my $which (sort {uc($a) cmp uc($b)} (keys(%sortrole))) {
	    if ($roleclass{$sortrole{$which}} =~ /^\Q$type\E/) { 
		$output.=$roletext{$sortrole{$which}};
                if ($sortrole{$which} =~ m-dc\./($match_domain)/-) {
                    if ($dcroles{$1}) {
                        $output .= &allcourses_row($1,'');
                    }
                }
	    }
	}
	if ($output) {
	    if ($doheaders > 0) {
		$r->print("<tr>".
			  "<td align='center' colspan='5'><font face='arial'>".&mt($type)."</font></td></tr>");
	    }
	    $r->print($output);	
	}
    }
    my $tremark='';
    my $tfont='#003300';
    if ($env{'request.role'} eq 'cm') {
	$r->print('<tr bgcolor="#11CC55">');
        $tremark=&mt('Currently selected. ');
        $tfont='#002200';
    } else {
        $r->print('<tr bgcolor="#77FF77">');
    }
    unless ($nochoose) {
	if ($env{'request.role'} ne 'cm') {
	    $r->print('<td><input type="submit" value="'.
		      &mt('Select').'" name="cm"></td>');
	} else {
	    $r->print('<td>&nbsp;</td>');
	}
    }
    $r->print('<td colspan="3"><font color="'.$tfont.'"><span class="LC_rolesinfo">'.&mt('No role specified').
      '</font></span></td><td><font color="'.$tfont.'">'.$tremark.
      '&nbsp;</font></td></tr>'."\n");

    $r->print('</table>');
    unless ($nochoose) {
	$r->print("</form>\n");
    }
# ------------------------------------------------------------ Privileges Info
    if (($advanced) && (($env{'user.error.msg'}) || ($error))) {
	$r->print('<hr /><h2>Current Privileges</h2>');
	$r->print(&privileges_info());
    }
    $r->print(&Apache::lonnet::getannounce());
    if ($advanced) {
	$r->print('<p><small><i>This is LON-CAPA '.
		  $r->dir_config('lonVersion').'</i><br />'.
		  '<a href="/adm/logout">'.&mt('Logout').'</a></small></p>');
    }
    $r->print(&Apache::loncommon::end_page());
    return OK;
}

sub privileges_info {
    my ($which) = @_;
    my $output;

    $which ||= $env{'request.role'};

    foreach my $envkey (sort(keys(%env))) {
	next if ($envkey!~/^user\.priv\.\Q$which\E\.(.*)/);

	my $where=$1;
	my $ttype;
	my $twhere;
	my (undef,$tdom,$trest,$tsec)=split(m{/},$where);
	if ($trest) {
	    if ($env{'course.'.$tdom.'_'.$trest.'.description'} eq 'ca') {
		$ttype='Construction Space';
		$twhere='User: '.$trest.', Domain: '.$tdom;
	    } else {
		$ttype= &Apache::loncommon::course_type($tdom.'_'.$trest);
		$twhere=$env{'course.'.$tdom.'_'.$trest.'.description'};
		if ($tsec) {
		    my $sec_type = 'Section';
		    if (exists($env{"user.role.gr.$where"})) {
			$sec_type = 'Group';
		    }
		    $twhere.=' ('.$sec_type.': '.$tsec.')';
		}
	    }
	} elsif ($tdom) {
	    $ttype='Domain';
	    $twhere=$tdom;
	} else {
	    $ttype='System';
	    $twhere='/';
	}
	$output .= "\n<h3>".$ttype.': '.$twhere.'</h3>'."\n<ul>";
	foreach my $priv (sort(split(/:/,$env{$envkey}))) {
	    next if (!$priv);

	    my ($prv,$restr)=split(/\&/,$priv);
	    my $trestr='';
	    if ($restr ne 'F') {
		$trestr.=' ('.
		    join(', ',
			 map { &Apache::lonnet::plaintext($_) } 
			     (split('',$restr))).') ';
	    }
	    $output .= "\n\t".
		'<li>'.&Apache::lonnet::plaintext($prv).$trestr.'</li>';
	}
	$output .= "\n".'</ul>';
    }
    return $output;
}

sub role_status {
    my ($rolekey,$then,$now,$role,$where,$trolecode,$tstatus,$tstart,$tend) = @_;
    my @pwhere = ();
    if (exists($env{$rolekey}) && $env{$rolekey} ne '') {
        (undef,undef,$$role,@pwhere)=split(/\./,$rolekey);
        unless (!defined($$role) || $$role eq '') {
            $$where=join('.',@pwhere);
            $$trolecode=$$role.'.'.$$where;
            ($$tstart,$$tend)=split(/\./,$env{$rolekey});
            $$tstatus='is';
            if ($$tstart && $$tstart>$then) {
		$$tstatus='future';
		if ($$tstart<$now) { $$tstatus='will'; }
            }
            if ($$tend) {
                if ($$tend<$then) {
                    $$tstatus='expired';
                } elsif ($$tend<$now) {
                    $$tstatus='will_not';
                }
            }
        }
    }
}

sub build_roletext {
    my ($trolecode,$tdom,$trest,$tstatus,$tryagain,$advanced,$tremark,$tbg,$tfont,$trole,$twhere,$tpstart,$tpend,$nochoose,$button,$switchserver) = @_;
    my $roletext='<tr bgcolor="'.$tbg.'" class="LC_rolesmenu_'.$tstatus.'">';
    my $is_dc=($trolecode =~ m/^dc\./);
    my $rowspan=($is_dc) ? ''
                         : ' rowspan="2" ';

    unless ($nochoose) {
        my $buttonname=$trolecode;
        $buttonname=~s/\W//g;
        if (!$button) {
            if ($switchserver) {
                $roletext.='<td'.$rowspan.'><span class="LC_rolesinfo"><a href="/adm/switchserver?'.
                $switchserver.'">'.&mt('Switch Server').'</a></span></td>';
            } else {
                $roletext.=('<td'.$rowspan.'>&nbsp;</td>');
            }
        } elsif ($tstatus eq 'is') {
            $roletext.='<td'.$rowspan.'><input name="'.$buttonname.'" type="button" value="'.
                        &mt('Select').'" onClick="javascript:enterrole(this.form,\''.
                        $trolecode."','".$buttonname.'\');"></td>';
        } elsif ($tryagain) {
            $roletext.=
                '<td'.$rowspan.'><input name="'.$buttonname.'" type="button" value="'.
                &mt('Try Selecting Again').'" onClick="javascript:enterrole(this.form,\''.
                        $trolecode."','".$buttonname.'\');"></td>';
        } elsif ($advanced) {
            $roletext.=
                '<td'.$rowspan.'><input name="'.$buttonname.'" type="button" value="'.
                &mt('Re-Initialize').'" onClick="javascript:enterrole(this.form,\''.
                        $trolecode."','".$buttonname.'\');"></td>';
        } else {
            $roletext.='<td'.$rowspan.'>&nbsp;</td>';
        }
    }
    if ($trolecode !~ m/^(dc|ca|au|aa)\./) {
	$tremark.=&Apache::lonannounce::showday(time,1,
			 &Apache::lonannounce::readcalendar($tdom.'_'.$trest));
    }
    $roletext.='<td><font color="'.$tfont.'">'.$trole.
	       '</font></td><td><font color="'.$tfont.'">'.$twhere.
               '</font></td><td><font color="'.$tfont.'">'.$tpstart.
               '</font></td><td><font color="'.$tfont.'">'.$tpend.
               '</font></td></tr>';
    if (!$is_dc) {
	$roletext.='<tr bgcolor="'.$tbg.'"><td colspan="4"><font color="'.$tfont.'"><span class="LC_rolesinfo">'.$tremark.
	    '</span>&nbsp;</font></td></tr><tr><td colspan="5" height="3"></td></tr>'."\n";
    }
    return $roletext;
}

sub check_privs {
    my ($cdom,$cnum,$then,$now) = @_;
    my $cckey = 'user.role.cc./'.$cdom.'/'.$cnum; 
    if ($env{$cckey}) {
        my ($role,$where,$trolecode,$tstart,$tend,$tremark,$tstatus,$tpstart,$tpend,$tfont);
        &role_status($cckey,$then,$now,\$role,\$where,\$trolecode,\$tstatus,\$tstart,\$tend);
        unless (($tstatus eq 'is') || ($tstatus eq 'will_not')) {
            &set_privileges($cdom,$cnum);
        }
    } else {
        &set_privileges($cdom,$cnum);
    }
}

sub check_fordc {
    my ($dcroles,$then) = @_;
    my $numdc = 0;
    if ($env{'user.adv'}) {
        foreach my $envkey (sort keys %env) {
            if ($envkey=~/^user\.role\.dc\.\/($match_domain)\/$/) {
                my $dcdom = $1;
                my $livedc = 1;
                my ($tstart,$tend)=split(/\./,$env{$envkey});
                if ($tstart && $tstart>$then) { $livedc = 0; }
                if ($tend   && $tend  <$then) { $livedc = 0; }
                if ($livedc) {
                    $$dcroles{$dcdom} = $envkey;
                    $numdc++;
                }
            }
        }
    }
    return $numdc;
}

sub courselink {
    my ($dcdom,$rowtype,$selecttype) = @_;
    my $courseform=&Apache::loncommon::selectcourse_link
                   ('rolechoice','dccourse'.$rowtype.'_'.$dcdom,
                    'dcdomain'.$rowtype.'_'.$dcdom,'coursedesc'.$rowtype.'_'.
                    $dcdom,$dcdom,undef);
    my $hiddenitems = '<input type="hidden" name="dcdomain'.$rowtype.'_'.$dcdom.'" value="'.$dcdom.'" />'.
                      '<input type="hidden" name="origdom'.$rowtype.'_'.$dcdom.'" value="'.$dcdom.'" />'.
                      '<input type="hidden" name="dccourse'.$rowtype.'_'.$dcdom.'" value="" />'.
                      '<input type="hidden" name="coursedesc'.$rowtype.'_'.$dcdom.'" value="" />';
    return $courseform.$hiddenitems;
}

sub coursepick_jscript {
    my $verify_script = <<"END";
<script type="text/javascript">
function verifyCoursePick(caller) {
    var numbutton = getIndex(caller)
    var pickedCourse = document.rolechoice.elements[numbutton+4].value
    var pickedDomain = document.rolechoice.elements[numbutton+2].value
    if (document.rolechoice.elements[numbutton+2].value == document.rolechoice.elements[numbutton+3].value) {
        if (pickedCourse != '') {
            if (numbutton != -1) {
                var courseTarget = "cc./"+pickedDomain+"/"+pickedCourse
                document.rolechoice.elements[numbutton+1].name = courseTarget
                document.rolechoice.submit()
            }
        }
        else {
            alert("Please use the 'Select Course' link to open a separate pick course window where you may select the course you wish to enter.");
        }
    }
    else {
        alert("You can only use this screen to select courses in the current domain")
    }
}
function getIndex(caller) {
    for (var i=0;i<document.rolechoice.elements.length;i++) {
        if (document.rolechoice.elements[i] == caller) {
            return i;
        }
    }
    return -1;
}
</script>
END
    return $verify_script;
}

sub processpick {
    my $process_pick = <<"END";
<script type="text/javascript">
function process_pick(dom) {
    var pickedCourse=opener.document.rolechoice.$env{'form.cnumelement'}.value;
    var pickedDomain=opener.document.rolechoice.$env{'form.cdomelement'}.value;
    var okDomain = 0;

    if (pickedDomain == dom) {
        if (pickedCourse != '') {
            var courseTarget = "cc./"+pickedDomain+"/"+pickedCourse
            opener.document.title='Role selected. Please stand by.';
            opener.status='Role selected. Please stand by.';
	    opener.document.rolechoice.newrole.value=courseTarget
            opener.document.rolechoice.submit()
        }
    } else {
        alert("You may only use this screen to select courses in the current domain: "+dom+"\\nPlease return to the roles page window and click the 'Select Course' link for domain: "+pickedDomain+",\\n if you are a Domain Coordinator in that domain, and wish to become a Course Coordinator in a course in the domain");
    }
}
 
</script>
END
    return $process_pick;
}

sub display_cc_role {
    my $rolekey = shift;
    my $roletext;
    my $advanced = $env{'user.adv'};
    my $tryagain = $env{'form.tryagain'};
    unless ($rolekey =~/^error\:/) {
        if ($rolekey =~ m-^user\.role.cc\./($match_domain)/($match_courseid)$-) {
            my $tcourseid = $1.'_'.$2;
            my $trolecode = 'cc./'.$1.'/'.$2;
            my $twhere;
            my $ttype;
            my $tbg='#77FF77';
            my $tfont='#003300';
            my %newhash=&Apache::lonnet::coursedescription($tcourseid);
            if (%newhash) {
                $twhere=$newhash{'description'}.
                        ' <font size="-2">'.
                        &Apache::loncommon::syllabuswrapper(&mt('Syllabus'),$2,$1,$tfont).
                        '</font>';
                $ttype = $newhash{'type'};
            } else {
                $twhere=&mt('Currently not available');
                $env{'course.'.$tcourseid.'.description'}=$twhere;
            }
            my $trole = &Apache::lonnet::plaintext('cc',$ttype);
            $twhere.="<br />".&mt('Domain').":".$1;
            $roletext = &build_roletext($trolecode,$1,$2,'is',$tryagain,$advanced,'',$tbg,$tfont,$trole,$twhere,'','','',1,'');
        }
    }
    return ($roletext);
}

sub allcourses_row {
    my ($dcdom,$rowtype) = @_;
    my $output = '<tr bgcolor="#77FF77">'.
                 ' <td colspan="5">';
    my $selectlink = &courselink($dcdom,$rowtype);
    my $ccrole = &Apache::lonnet::plaintext('cc');
    $output.= '<span class="LC_rolesinfo">'.
            &mt('[_1]: [_2] from domain [_3]',$ccrole,$selectlink,$dcdom).
            '</span><br /></tr><tr><td colspan="5" height="3"></td></tr>'."\n";
    return $output;
}

sub recent_filename {
    my $area=shift;
    return 'nohist_recent_'.&escape($area);
}

sub set_privileges {
    my ($dcdom,$pickedcourse) = @_;
    my $area = '/'.$dcdom.'/'.$pickedcourse;
    my $role = 'cc';
    my $spec = $role.'.'.$area;
    my %userroles = &Apache::lonnet::set_arearole($role,$area,'','',
						  $env{'user.domain'},
						  $env{'user.name'});
    my %ccrole = ();
    &Apache::lonnet::standard_roleprivs(\%ccrole,$role,$dcdom,$spec,$pickedcourse,$area);
    my ($author,$adv)= &Apache::lonnet::set_userprivs(\%userroles,\%ccrole);
    &Apache::lonnet::appenv(%userroles);
    &Apache::lonnet::log($env{'user.domain'},
                         $env{'user.name'},
                         $env{'user.home'},
                        "Role ".$role);
    &Apache::lonnet::appenv(
                          'request.role'        => $spec,
                          'request.role.domain' => $dcdom,
                          'request.course.sec'  => '');
    my $tadv=0;
    if (&Apache::lonnet::allowed('adv') eq 'F') { $tadv=1; }
    &Apache::lonnet::appenv('request.role.adv'    => $tadv);
}

sub courseloadpage {
    my ($courseid) = @_;
    my $startpage;
    my %entry_settings = &Apache::lonnet::get('nohist_whatsnew',
					      [$courseid.':courseinit']);
    my ($tmp) = %entry_settings;
    unless ($tmp =~ /^error: 2 /) {
        $startpage = $entry_settings{$courseid.':courseinit'};
    }
    if ($startpage eq '') {
        if (exists($env{'environment.course_init_display'})) {
            $startpage = $env{'environment.course_init_display'};
        }
    }
    return $startpage;
}

1;
__END__

=head1 NAME

Apache::lonroles - User Roles Screen

=head1 SYNOPSIS

Invoked by /etc/httpd/conf/srm.conf:

 <Location /adm/roles>
 PerlAccessHandler       Apache::lonacc
 SetHandler perl-script
 PerlHandler Apache::lonroles
 ErrorDocument     403 /adm/login
 ErrorDocument	  500 /adm/errorhandler
 </Location>

=head1 OVERVIEW

=head2 Choosing Roles

C<lonroles> is a handler that allows a user to switch roles in
mid-session. LON-CAPA attempts to work with "No Role Specified", the
default role that a user has before selecting a role, as widely as
possible, but certain handlers for example need specification which
course they should act on, etc. Both in this scenario, and when the
handler determines via C<lonnet>'s C<&allowed> function that a certain
action is not allowed, C<lonroles> is used as error handler. This
allows the user to select another role which may have permission to do
what they were trying to do. C<lonroles> can also be accessed via the
B<CRS> button in the Remote Control. 

=begin latex

\begin{figure}
\begin{center}
\includegraphics[width=0.45\paperwidth,keepaspectratio]{Sample_Roles_Screen}
  \caption{\label{Sample_Roles_Screen}Sample Roles Screen} 
\end{center}
\end{figure}

=end latex

=head2 Role Initialization

The privileges for a user are established at login time and stored in the session environment. As a consequence, a new role does not become active till the next login. Handlers are able to query for privileges using C<lonnet>'s C<&allowed> function. When a user first logs in, their role is the "common" role, which means that they have the sum of all of their privileges. During a session it might become necessary to choose a particular role, which as a consequence also limits the user to only the privileges in that particular role.

=head1 INTRODUCTION

This module enables a user to select what role he wishes to
operate under (instructor, student, teaching assistant, course
coordinator, etc).  These roles are pre-established by the actions
of upper-level users.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 HANDLER SUBROUTINE

This routine is called by Apache and mod_perl.

=over 4

=item *

Roles Initialization (yes/no)

=item *

Get Error Message from Environment

=item *

Who is this?

=item *

Generate Page Output

=item *

Choice or no choice

=item *

Table

=item *

Privileges

=back

=cut
