# The LearningOnline Network with CAPA
# random labelling tool
#
# $Id: randomlabel.pm,v 1.90 2007/10/17 22:11:55 albertel Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
# SYNTAX:
# <randomlabel bgimg="URL" width="12" height="45" texwidth="50">
#    <labelgroup name="GroupOne" type="image">
#      <location x="123" y="456" />
#      <location x="321" y="654" />
#      <location x="213" y="546" />
#      <label description="TEXT-1">IMG-URL</label>
#      <label description="TEXT-2">IMG-URL</label>
#      <label description="TEXT-3">IMG-URL</label>
#    </labelgroup>
#    <labelgroup name="GroupTwo" type="text">
#      <location x="12" y="45" />
#      <location x="32" y="65" />
#      <location x="21" y="54" />
#      <label>TEXT-1</label>
#      <label>TEXT-2</label>
#      <label>TEXT-3</label>
#    </labelgroup>
#   </randomlabel>
#  ===========================================
#  side effect:
#    location (123,456): $GroupOne[0] = 2  # images give out indexes
#             (321,654): $GroupOne[1] = 1
#             (213,546): $GroupOne[2] = 0
#    location (12,45)  : $GroupTwo[0] = "TEXT-3"
#             (32,65)  : $GroupTwo[1] = "TEXT-1"
#             (21,54)  : $GroupTwo[2] = "TEXT-2"
#  ===========================================
package Apache::randomlabel;
use Apache::lonnet;
use strict;
use Apache::edit;
use Apache::File();
use Apache::Constants qw(:common :http);
use Image::Magick;
use Apache::lonplot;
use LONCAPA;
 

my %args;
my $cgi_id;
my $scale_factor;		# image scale factor.
my $label_xscale;                # Label scale factor (needed for gnuplot).
my $label_yscale;
my $dirty_width_adjust = 5;     # Width adjustment for e.g. gnuplot images.

BEGIN {
    &Apache::lonxml::register('Apache::randomlabel',('randomlabel','labelgroup','location','label','bgimg'));
}



sub check_int {
    # utility function to do error checking on a integer.
    my ($num,$default) = @_;
    $default = 100 if (! defined($default));
    $num =~ s/\s+//g;  # We dont need no stinkin white space!
    # If it is a real, just grab the integer part.
    ($num,undef) = split (/\./,$num) if ($num =~ /\./); 
    # set to default if what we have left doesn't match anything...
    $num = $default unless ($num =~/^\d+$/);
    return $num;
}

#  Get width/height from an image tag...
#
#  Parameters:
#      tag         - tag potentially containing height/width attributes.
#      def_width   - Default width.
#      def_height  - Default height.
#  Returns:
#      list containing width/height.
#
sub extract_tag_sizes {
    my ($tag, $dw, $dh) = @_;
    $tag =~ s/\s+/ /g;         # Collapse whitespace.
    $tag =~ s/\s*=\s*/=/g;     # kill space around ='s.
    $tag =~ s/[<>\"]//g;       # Get rid of the <">'s too.

    &Apache::lonxml::debug("Compressed tag: $tag");
    my @taglist = split(/ /,$tag);
    foreach my $attribute (@taglist) {
	if ($attribute =~ /^width/i) {
	    my ($e, $s)= split(/=/,$attribute);
	    $dw = $s;
	}
	if ($attribute =~  /^height/i) {
	    my ($e, $s) = split(/=/,$attribute);
	    $dh = $s;
	}
    } 
    return($dw, $dh);

}

my ($height_param,$width_param);
sub start_randomlabel {

    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result='';
    push (@Apache::lonxml::namespace,'randomlabel');
    ($height_param,$width_param)=(0,0);
    $label_xscale = 1.0;		# Assume image size not overridden.
    $label_yscale = 1.0;
    my $bgimg= &Apache::lonxml::get_param('bgimg',$parstack,$safeeval);
    if ( defined($bgimg) && $bgimg !~ /^http:/ ) {
	$bgimg=&Apache::lonnet::filelocation($Apache::lonxml::pwd[-1],$bgimg);
	if (&Apache::lonnet::repcopy($bgimg) ne 'ok') {
	    $bgimg='/home/httpd/html/adm/lonKaputt/lonlogo_broken.gif';
	}
    }
    $Apache::randomlabel::obj_cnt=0;
    if ($target eq 'web') {
	$cgi_id=&Apache::loncommon::get_cgi_id();
	%args=();
	$args{"cgi.$cgi_id.BGIMG"}=&escape($bgimg);
	$height_param = &Apache::lonxml::get_param('height',$parstack, $safeeval);
	$width_param  = &Apache::lonxml::get_param('width', $parstack, $safeeval);
    } elsif ($target eq 'tex' && defined($bgimg)) {
	$result.=&make_eps_image($bgimg,$parstack,$safeeval);
    } elsif ($target eq 'edit') {
        my $only = join(',',&Apache::loncommon::filecategorytypes('Pictures'));
	$result.=&Apache::edit::tag_start($target,$token);
	$Apache::edit::bgimgsrc=
	    &Apache::lonxml::get_param('bgimg',$parstack,$safeeval);
	$Apache::edit::bgimgsrccurdepth=$Apache::lonxml::curdepth;
	$result.=&Apache::edit::text_arg('Image:','bgimg',$token,75).' ';
	$result.=&Apache::edit::browse('bgimg',undef,undef,$only).' ';
	$result.=&Apache::edit::search('bgimg').'<br />'.
	    &Apache::edit::text_arg('Width(pixel):' ,'width'   ,$token,6).
	    &Apache::edit::text_arg('Height(pixel):','height'  ,$token,6).
	    &Apache::edit::text_arg('TeXWidth(mm):' ,'texwidth',$token,6).
	    &Apache::edit::end_row().&Apache::edit::start_spanning_row();     
    } elsif ($target eq 'modified') {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'bgimg','width',
						     'height','texwidth');
	if ($constructtag) {
	    $result = &Apache::edit::rebuild_tag($token);
	}
    }
    return $result;
}

sub end_randomlabel {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result='';
    my $count;
    pop @Apache::lonxml::namespace;
    if ($target eq 'web') {
	$count = $Apache::randomlabel::obj_cnt;
	if( $count != 0) { $args{"cgi.$cgi_id.OBJCOUNT"}=$count; }
	$result.='<img src="/adm/randomlabel.png?token='.$cgi_id.'" /><br />'."\n";
	&Apache::lonnet::appenv(%args);
    } elsif ($target eq 'tex') {
	$result='\end{picture}\\\\';
	$result.= ' \vskip -'.$height_param.' mm }  \\\\ ';
    } elsif ($target eq 'edit') {
	$result.=&Apache::edit::end_table;
    }
    return $result;
}


sub start_bgimg {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result='';
    if ($target eq 'web' || $target eq 'tex' || $target eq 'analyze') { 
	&Apache::lonxml::startredirection(); 
    }
    return $result;
}

sub end_bgimg {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result='';
    if ($target eq 'web' || $target eq 'tex' || $target eq 'analyze') { 
	my $bgimg=&Apache::lonxml::endredirection(); 
	if ($target eq 'web') {
	    
	    # If the  tag produced has sizes, they override ours.
	    # (for now anyway).
	    #
	    &Apache::lonxml::debug("Base sizes: $width_param x $height_param");
	
	    my ($plot_x, $plot_y) = &extract_tag_sizes($bgimg, 
						       $width_param,
						       $height_param);
	    &Apache::lonxml::debug("Extracted sizes: $plot_x x $plot_y");
	    if ($width_param) {
		$label_xscale     = $plot_x / $width_param;
	    }
	    if ($height_param) {
		$label_yscale     = $plot_y / $height_param;
	    }
	    &Apache::lonxml::debug("Scale factors:   $label_xscale $label_yscale");

	    &Apache::lonxml::debug("Image: $bgimg");
	    $bgimg=&Apache::imageresponse::clean_up_image($bgimg);
	    &Apache::lonxml::debug("Cleaned image: $bgimg");
	    $args{"cgi.$cgi_id.BGIMG"}=&escape($bgimg);
	} elsif ($target eq 'tex') {
	    #   Some bg images can create latex for us... e.g. gnuplot.
	    #   If it looks like we have some latex use that, 
	    #   otherwise, assume this is a resource name that must
	    #   be converted into the latex to create an eps insertion.
	    #
	    my $src = $bgimg;
	    $src =~ s/\s+$//s;
	    $src =~ s/^\s+//s;

	    #If this is a dynamically generated image, it will
	    #be in latex already, with a comment header that
	    #describes the dimensions:

	    if($src =~ /^%DYNAMICIMAGE:/) {
		$Apache::lonxml::debug = 0;
		&Apache::lonxml::debug("Dynamic image");
		my ($commentline, $junk) = split(/\n/, $src);
		&Apache::lonxml::debug("Comment line was: $commentline");
		my $trash;
		my $initial_width;
		($trash, $initial_width, $height_param, $width_param) =
		    split(/:/,$commentline);
		&Apache::lonxml::debug("internal web Width/height: $initial_width $height_param");
		&Apache::lonxml::debug("Texwitdh: $width_param");
		if($initial_width == 0) {
		    $initial_width = $width_param;
		}
		# strip off the comments since output does not always
		# preserve \n's:
                #
		$src =~ s/$commentline//;
		$scale_factor = $width_param / $initial_width;
		$height_param = $height_param*$scale_factor;

		$label_xscale = 1.0; #  $scale_factor;
		$label_yscale = 1.0; #  $scale_factor;
	   
		&Apache::lonxml::debug("height $height_param");
		&Apache::lonxml::debug("Width $width_param");
		&Apache::lonxml::debug("Scale factors: $label_xscale $label_yscale");
		my $dirty_width = $width_param + $dirty_width_adjust;
		my $x_offset    = -$dirty_width_adjust/2.0;
		#
		#  Somewhere here it looks like height_param and
		#  width_param got backwards...
		#
		$result .= '\parbox{'.$dirty_width.'mm}{';
		$result  .= " $src \n";
		$result  .= '\setlength{\unitlength}{1mm}'."\n";
		$result  .= '\begin{picture}('."$width_param,$height_param)";
		$result  .= "($x_offset,-$height_param)";
		$result  .= "\n";
		$Apache::lonxml::debug = 0;

	    } else {
		$result.=&make_eps_image($bgimg,$parstack,$safeeval,-2);
	    }
	}
    }
    return $result;
}
sub make_eps_image {
    my ($bgimg,$parstack,$safeeval,$depth)=@_;
    &Apache::lonxml::debug("image prior to get_eps_image: $bgimg");
    my ($path,$file) = &Apache::londefdef::get_eps_image($bgimg);
    &Apache::lonxml::debug("image after:  $bgimg");
    ($height_param,$width_param)=
	&Apache::londefdef::image_size($bgimg,0.3,$parstack,$safeeval,
				       $depth,1);

    &Apache::lonxml::debug("Image size: $height_param x $width_param");

    my $dirtywidth=$width_param+5;

    my $result ="\n".'\vspace*{2mm}\noindent'."\n".
	'\parbox{'.$dirtywidth.
	' mm}{  \noindent \epsfxsize='.$width_param.
	' mm \epsffile{'.$path.$file.
	'}\setlength{\unitlength}{1mm}'."\n".'  \begin{picture}('.
	$width_param.','.$height_param.')(0,-'.$height_param.')'."\n";
    my $magick = Image::Magick->new;
    $magick->Read($bgimg);
    my $initial_width = $magick->Get('width');
    &Apache::lonxml::debug("ImageMagick thinks width is; $initial_width");
    $scale_factor = $width_param / $initial_width;
    return $result;
}

sub start_labelgroup {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result='';
    my $name = &Apache::lonxml::get_param('name',$parstack,$safeeval);
    my $type = &Apache::lonxml::get_param('type',$parstack,$safeeval);
    $type =~tr/A-Z/a-z/;
    if ($target ne 'modified' && ($name =~ /\W/ || $name =~ /^[0-9]/)) {
	&Apache::lonxml::error("Only _ a-z A-Z and 0-9 are allowed in the name to a labelgroup, and the first character can not be a number.<br />");
    }
    if ($target eq 'web' || $target eq 'tex' ||
	$target eq 'grade' || $target eq 'answer' || $target eq 'analyze') {
	$Apache::randomlabel::groupname=$name;
	$Apache::randomlabel::type=$type;
	@Apache::randomlabel::xcoord = ();
	@Apache::randomlabel::ycoord = ();
	@Apache::randomlabel::value = ();
	@Apache::randomlabel::label_arr  = ();
	@Apache::randomlabel::description  = ();
    } elsif ($target eq 'edit') {
	$result.=&Apache::edit::tag_start($target,$token);
	$result.=&Apache::edit::text_arg('Name:','name',$token).
	    &Apache::edit::select_arg('Type:','type',['text','image'],$token);
	if (!defined($token->[2]{'TeXsize'})) {
	    $token->[2]{'TeXsize'}='\normalsize';
	}
	$result.=&Apache::edit::select_arg('TeX font size:','TeXsize',
					   ['\tiny','\scriptsize',
					    '\footnotesize','\small',
					    '\normalsize','\large','\Large',
					    '\LARGE','\huge','\Huge'],
					   $token);
	$result.=&Apache::edit::end_row().&Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'name','type',
						     'TeXsize');
	if ($constructtag) {
	    $result = &Apache::edit::rebuild_tag($token);
	}
    }
    return $result;
}

#
#   Utility sub to compute the width of a label.
#
sub get_label_width {
    my $label         = shift;
    &Apache::lonxml::debug("image label = $label");
    if (-e $label) {
	&Apache::lonxml::debug("$label exists");
    } else {
	&Apache::lonxml::debug("$label does not exist");
    }
    my $magick        = Image::Magick->new;
    $magick->Read($label);
    my $pixel_width   = $magick->Get('width');
    return $pixel_width * $scale_factor;
    
	
}

sub get_label_height {
    my $label         = shift;
    &Apache::lonxml::debug("image label = $label");
    if (-e $label) {
	&Apache::lonxml::debug("$label exists");
    } else {
	&Apache::lonxml::debug("$label does not exist");
    }
    my $magick        = Image::Magick->new;
    $magick->Read($label);
    my $pixel_height   = $magick->Get('height');
    return $pixel_height * $scale_factor;
}

sub add_vars {
    my ($name,$order,$label,$labelorder,$value,$image,$safeeval) = @_;
    if (!defined($name) || $name eq '') { return; }
    my $code = '${'.$name."}{'".($order+1)."'}='".$label."';";
    my $out=Apache::run::run($code,$safeeval);
    if ($value) {
	$code = '${'.$name."}{'value_".($order+1)."'}='".$value."';";
	$out=Apache::run::run($code,$safeeval);
	$code = '${'.$name."}{'labelvalue_".($labelorder+1)."'}='".$value."';";
	$out=Apache::run::run($code,$safeeval);
    }
    if ($image) {
	my $code = '${'.$name."}{'image_".($order+1)."'}='".$image."';";
	my $out=Apache::run::run($code,$safeeval);
    }
    $code = '${'.$name."}{'numlocations'}='".($order+1)."';";
    $out=Apache::run::run($code,$safeeval);
}

# begin to assign labels to locations
sub end_labelgroup {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $gname = $Apache::randomlabel::groupname;
    my $type  = $Apache::randomlabel::type;
    my $result='';
    if ($target eq 'web' || $target eq 'answer' || $target eq 'grade' ||
	$target eq 'analyze') {
	my @idx_arr = (0 .. $#Apache::randomlabel::label_arr);
	&Apache::structuretags::shuffle(\@idx_arr);
	for(0 .. $#Apache::randomlabel::label_arr) {
	    my $str;
	    my $label = "$Apache::randomlabel::label_arr[ $idx_arr[$_] ]";
	    my $x = $Apache::randomlabel::xcoord[$_];
	    my $y = $Apache::randomlabel::ycoord[$_];
	    my $value = $Apache::randomlabel::value[$_];
	    my $i=$Apache::randomlabel::obj_cnt++;
	    if( $type eq 'text') {
		&add_vars($gname,$_,$label,$idx_arr[$_],$value,'',$safeeval);
		$str = join(':',$x,$y,&escape($label));
		$args{"cgi.$cgi_id.OBJTYPE"}.='LABEL:';
	    } elsif ( $type eq 'image') {
		&add_vars($gname,$_,
			  $Apache::randomlabel::description[$idx_arr[$_]],
			  $idx_arr[$_],$value,$label,$safeeval);
		$str = join(':',$x,$y,&escape($label));
		$args{"cgi.$cgi_id.OBJTYPE"}.='IMAGE:';
	    } else {
		&Apache::lonxml::error('Unknown type of label :'.$type.':');
	    }
	    if ($target eq 'web') { $args{"cgi.$cgi_id.OBJ$i"} =$str; }
	}
    } elsif ($target eq 'tex') {
	my $WX1=0; #  Web x-coord. of upper left corner (ULC)
	my $WY1=0; #  Web y-coord. of (ULC)
	my $wwidth=&Apache::lonxml::get_param('width',$parstack,$safeeval,-2);
	my $wheight=&Apache::lonxml::get_param('height',$parstack,$safeeval,-2);
	my $TeXsize=&Apache::lonxml::get_param('TeXsize',$parstack,$safeeval);
	if (!defined($TeXsize)) { $TeXsize='\\normalsize'; }
	
	my @idx_arr = (0 .. $#Apache::randomlabel::label_arr);
	&Apache::structuretags::shuffle(\@idx_arr);

	&Apache::lonxml::debug("Array is:".$#Apache::randomlabel::label_arr.":");
	$Apache::lonxml::debug = 0;
	for(my $i=0;$i <= $#Apache::randomlabel::label_arr; $i++) {
	    my $label = "$Apache::randomlabel::label_arr[ $idx_arr[$i] ]";
	    my $x = $Apache::randomlabel::xcoord[$i];
	    my $y = $Apache::randomlabel::ycoord[$i];
	    if ( $type eq 'text' ) {
		# FIXME the 3.5 here is the 'height' of the letter in TeX
		$y=$y-3.5;
	    }
	    &Apache::lonxml::debug("initially: x= $x y= $y");
	    my $value = $Apache::randomlabel::value[$i];
	    #x latex coordinate
	    my $tcX=($x)*($width_param/$wwidth);
	    &Apache::lonxml::debug("wparam = $width_param wwidth = $wwidth, texx = $tcX");
	    #y latex coordinate
            #      my $ratio=($wwidth > 0 ? $wheight/$wwidth : 1 );
	    my $tcY=$height_param-$y*($height_param/$wheight);
	    if ( $type eq 'image') {
		my $label_height = &get_label_height($label);
		$tcY=$tcY-$label_height;
	    }

	    &Apache::lonxml::debug("hparam = $height_param wheight = $wheight texy = $tcY");
	    $tcX=sprintf('%.2f',$tcX);
	    $tcY=sprintf('%.2f',$tcY);
	    $result .= '\put('.$tcX.','.$tcY.'){';
	    if( $type eq 'text') {
		$result.= $TeXsize.' \bf '.$label."}\n";
		&add_vars($gname,$i,$label,$idx_arr[$i],$value,'',$safeeval);
	    } elsif ( $type eq 'image') {
		my ($path,$file) = &Apache::londefdef::get_eps_image($label);
		my $image_name = $path.$file;
		#
		#  Note that spaces in e.. \includegraphics cause problems for Latex
		# so they get replaced by _'s by lonprintout/printout and us:
		#
		my $label_width = &get_label_width($label);

		$result .=  '\includegraphics[width='.$label_width.'mm]{'
		            .$image_name."}}\n";
		&add_vars($gname,$i,
			  $Apache::randomlabel::description[$idx_arr[$i]],
			  $idx_arr[$i],$value,$label,$safeeval);
	    } else {
		&Apache::lonxml::error('Unknown type of label :'.$type.':');
	    }
	}
	$Apache::lonxml::debug =0;
    } elsif ($target eq 'edit') {
	$result.=&Apache::edit::end_table;
    }
    return $result;
}

# <location x="123" y="456" value="some value"/>
sub start_location {
    $Apache::lonxml::debug = 0;
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $x= &check_int(&Apache::lonxml::get_param('x',$parstack,$safeeval),50);
    my $y= &check_int(&Apache::lonxml::get_param('y',$parstack,$safeeval),50);
    &Apache::lonxml::debug("x = $x y = $y");
    $x = $x*$label_xscale;
    $y = $y*$label_yscale;
    &Apache::lonxml::debug(" H = $height_param W = $width_param");
    &Apache::lonxml::debug(" XS = $label_xscale YS = $label_yscale");
    &Apache::lonxml::debug(" X  = $x Y = $y");
    my $value= &Apache::lonxml::get_param('value',$parstack,$safeeval);
    my $result='';
    push(@Apache::randomlabel::xcoord,$x);
    push(@Apache::randomlabel::ycoord,$y);
    push(@Apache::randomlabel::value,$value);
    if ($target eq 'edit') {
	$result.=&Apache::edit::tag_start($target,$token);
	$result.=&Apache::edit::text_arg('X:','x',$token,4).
	    &Apache::edit::text_arg('Y:','y',$token,4).
	    &Apache::edit::entercoords('x','y','attribute','width','height').'&nbsp;&nbsp;&nbsp;'.
	    &Apache::edit::text_arg('Value:','value',$token).
	    &Apache::edit::end_row();
	$result.=&Apache::edit::end_table;
    } elsif ($target eq 'modified') {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						    $safeeval,'x','y','value');
	if ($constructtag) {
	    $result = &Apache::edit::rebuild_tag($token);
	}
    }
    $Apache::lonxml::debug = 0;
    return $result;
}

sub end_location {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my @result;
    if ($target eq 'edit') { @result=('','no') }
    return @result;
}

# <label>$var_abc</label>
sub insert_label {
    my ($after) = @_;
    my $depth = scalar(@Apache::lonxml::depthcounter);
    $depth-- if ($after);
    my $inset = "\t"x$depth;
    return "\n$inset<label></label>";
}

sub start_label {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result='';
    my $type = &Apache::lonxml::get_param('type',$parstack,$safeeval,-2);
    if ($target eq 'web' || $target eq 'tex' ||
	$target eq 'grade' || $target eq 'answer' || $target eq 'analyze') {
	&Apache::lonxml::startredirection; 
    } elsif ($target eq 'edit') {
	$result.=&Apache::edit::tag_start($target,$token,"$type Label");
	my $text=&Apache::lonxml::get_all_text("/label",$parser,$style);
	if ($type eq 'image') {
	    $result.=&Apache::edit::end_row().
		&Apache::edit::start_spanning_row();
	    $result.=&Apache::edit::text_arg('Description:','description',
					     $token);
	}
	if ($type eq 'text') { $result.="Label Text:"; }
	if ($type eq 'image') { $result.="Path to image:&nbsp;"; }
	$result.=&Apache::edit::editline('',$text,'',50).
	    &Apache::edit::end_table();
    } elsif ($target eq 'modified') {
	$result = '<label>';
	if ($type eq 'image') {
	    my $constructtag=&Apache::edit::get_new_args($token,$parstack,
							 $safeeval,
							 'description');
	    if ($constructtag) {
		$result = &Apache::edit::rebuild_tag($token);
	    } else {
		$result = $token->[4];
	    }
	}
	$result.=&Apache::edit::modifiedfield("/label",$parser);
    }
    return $result;
}

sub end_label {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my @result;
    if ($target eq 'edit') {
	@result=('','no') ;
    } elsif ($target eq 'web' || $target eq 'tex' || $target eq 'grade' ||
	     $target eq 'answer' || $target eq 'analyze') {
	my $type = &Apache::lonxml::get_param('type',$parstack,$safeeval,-2);
	my $ltext=&Apache::lonxml::endredirection; 
	if ($type eq 'image') {
	    if ($target eq 'tex') {
		# For tex targets, our image url has been potentially corrupted
		# by prepending \'s in front of special latex symbols.
		# For now we only worry about the _ case (most common?)
		# There's a whole host of theim in lonxml::latex_special_symbols
		# that could potentially have to be re-done.

		$ltext =~ s/\\_/_/g;
	    }
	    &Apache::lonxml::debug("Turning $ltext, $Apache::lonxml::pwd[-1]");
	    $ltext=&Apache::imageresponse::clean_up_image($ltext);
#	    $ltext=&Apache::lonnet::filelocation($Apache::lonxml::pwd[-1],
#						 $ltext);
	    &Apache::lonxml::debug("into $ltext");
	    my $description = &Apache::lonxml::get_param('description',
							 $parstack,$safeeval);
	    push(@Apache::randomlabel::description,$description);
	} else {
	    $ltext=~s/[\r\n]*//gs
	}
	push(@Apache::randomlabel::label_arr,$ltext);
    }
    return @result;
}

1;
__END__
 
