# The LearningOnline Network
# Allow access to password changing via a token sent to user's e-mail. 
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#
package Apache::resetpw;

use strict;
use Apache::Constants qw(:common);
use Apache::lonacc;
use Apache::lonnet;
use Apache::loncommon;
use Apache::lonlocal;
use LONCAPA;

sub handler {
    my $r = shift;
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    if ($r->header_only) {
        return OK;
    }
    my $start_page =
        &Apache::loncommon::start_page('Reset password','',
                                           {
                                             'no_inline_link'   => 1,});
    $r->print($start_page);
    my $contact_name = &mt('LON-CAPA helpdesk');
    my $contact_email =  $r->dir_config('lonSupportEMail');
    my $server = $r->dir_config('lonHostID');
    my $defdom = $r->dir_config('lonDefDomain');
    &Apache::lonacc::get_posted_cgi($r);
    &Apache::lonlocal::get_language_handle($r);
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},['token']);
    
    my @emailtypes = ('permanentemail','critnotification','notification');
    my $uname = &unescape($env{'form.uname'});
    my $udom = $env{'form.udom'};
    my $token = $env{'form.token'};
    $r->print(&mt('<h3>Reset forgotten LON-CAPA password</h3>'));
    my $output;
    if ($token) {
        $output = &reset_passwd($r,$token,$contact_name,$contact_email);
    } elsif ($uname && $udom) {
        my $domdesc = &Apache::lonnet::domain($udom,'description');
        my $authtype = &Apache::lonnet::queryauthenticate($uname,$udom);
        if ($authtype =~ /^internal/) {
            my $useremail = $env{'form.useremail'};
            if ($useremail !~ /^[^\@]+\@[^\@]+\.[^\@\.]+$/) {
                $output = &invalid_state('baduseremail',$domdesc,
                                         $contact_name,$contact_email);
            } else {
                my %userinfo = 
		    &Apache::lonnet::get('environment',\@emailtypes,
					 $udom,$uname);
                my $email = '';
                my $emailtarget;
                foreach my $type (@emailtypes) {
                    $email = $userinfo{$type};
                    if ($email =~ /[^\@]+\@[^\@]+/) {
                        $emailtarget = $type; 
                        last;
                    }
                }
                if ($email =~ /^[^\@]+\@[^\@]+\.[^\@\.]+$/) {
                    if ($useremail eq $email) {
                        $output = &send_token($uname,$udom,$email,$server,
                                              $domdesc,$contact_name,
                                              $contact_email);
                    } else {
                        $output = &invalid_state('mismatch',$domdesc,
                                                 $contact_name,
                                                 $contact_email);
                    }
                } else {
                    $output = &invalid_state('missing',$domdesc,
                                             $contact_name,$contact_email);
                }
            }
        } elsif ($authtype =~ /^(krb|unix|local)/) { 
            $output = &invalid_state('authentication',$domdesc,
                                     $contact_name,$contact_email);
        } else {
            $output = &invalid_state('invalid',$domdesc,
                                     $contact_name,$contact_email);
        }
    } else {
        $output = &get_uname($defdom);
    }
    $r->print($output);
    $r->print(&Apache::loncommon::end_page());
    return OK;
}

sub get_uname {
    my ($defdom) = @_;
    my %lt = &Apache::lonlocal::texthash(
                                         unam => 'username',
                                         udom => 'domain',
                                         uemail => 'Email address in LON-CAPA',
                                         proc => 'Proceed');

    my $msg = &mt('If you use the same account for other campus services besides LON-CAPA, (e.g., e-mail, course registration, etc.), a separate centrally managed mechanism likely exists to reset a password.  However, if your account is used for just LON-CAPA access you will probably be able to reset a password from this page.');
    $msg .= '<br /><br />'.&mt('Three conditions must be met:<ul><li>An e-mail address must have previously been associated with your LON-CAPA username.</li><li>You must be able to access e-mail sent to that address.</li><li>Your account must be of a type for which LON-CAPA can reset a password.</ul>');
    $msg .= qq|
<form name="forgotpw" method="post">
<table>
<tr><td>
<tr><td align="left">LON-CAPA $lt{'unam'}:                      </td>
    <td><input type="text" name="uname" size="15"  /> </td></tr>
<tr><td align="left">LON-CAPA $lt{'udom'}:                      </td>
    <td>|;
    $msg .= &Apache::loncommon::select_dom_form($defdom,'udom');
    $msg .= qq|</td></tr>
<tr><td align="left">$lt{'uemail'}:                             </td>
    <td><input type="text" name="useremail" size="30"  /></td></tr>
<tr><td colspan="2" align="left"><br />
    <input type="button" value="$lt{'proc'}" onClick="document.forgotpw.submit()"></td></tr>
</table>
</form>
|;
    return $msg;
}

sub send_token {
    my ($uname,$udom,$email,$server,$domdesc,$contact_name,
        $contact_email) = @_;
    my $msg = &mt('Thank you for your request to reset the password for your
        LON-CAPA account.').'<br /><br />';

    my $now = time;
    my $temppasswd = &create_passwd();
    my %info = ('ip'         => $ENV{'REMOTE_ADDR'},
		'time'       => $now,
		'domain'     => $udom,
		'username'   => $uname,
		'email'      => $email,
		'temppasswd' => $temppasswd);

    my $token = &Apache::lonnet::tmpput(\%info,$server,'resetpw');
    if ($token !~ /^error/ && $token ne 'no_such_host') {
        my $esc_token = &escape($token);
        my $mailmsg = "A request was submitted on ".localtime(time)." for a reset of the ".
             "password for your LON-CAPA account.".
             "To complete this process please open a web browser and enter the following ".
             "URL in the address/location box: ".&Apache::lonnet::absolute_url()."/adm/resetpw?token=$esc_token";
        my $result = &send_mail($domdesc,$email,$mailmsg,$contact_name,
                                $contact_email);
        if ($result eq 'ok') {
            $msg .= &mt("An e-mail message sent to the e-mail address associated with your LON-CAPA account includes the web address for the link you should use to complete the reset process.<br /><br />The link included in the message will be valid for the next <b>two</b> hours.");
        } else {
            $msg .= &mt("An error occurred when sending a message to the e-mail address associated with your LON-CAPA account. Please contact the [_1] ([_2]) for assistance.",$contact_name,$contact_email);
        }
    } else {
        $msg .= &mt("An error occurred creating a token required for the password reset process. Please contact the [_1] ([_2]) for assistance.",$contact_name,$contact_email);
    }
    return $msg;
}

sub send_mail {
    my ($domdesc,$email,$mailmsg,$contact_name,$contact_email) = @_;
    my $outcome;
    my $requestmail = "To: $email\n".
                      "From: $contact_name <$contact_email>\n".
                      "Subject: ".&mt('Your LON-CAPA account')."\n".
                      "\n\n".$mailmsg."\n\n".
                      &mt('[_1] LON-CAPA support team',$domdesc)."\n".
                      "$contact_email\n";
    if (open(MAIL, "|/usr/lib/sendmail -oi -t -odb")) {
        print MAIL $requestmail;
        close(MAIL);
        $outcome = 'ok';
    } else {
        $outcome = 'fail';
    }
    return $outcome;
}

sub invalid_state {
    my ($error,$domdesc,$contact_name,$contact_email) = @_;
    my $msg;
    if ($error eq 'invalid') {
        $msg = &mt('The username you provided was not verified as a valid username in the LON-CAPA system for the [_1] domain.',$domdesc).&mt(' Please <a href="javascript:history.go(-1)"><u>go back</u></a> and try again.');
    } else {
        if ($error eq 'baduseremail') {
            $msg = &mt('The e-mail address you provided does not appear to be a valid address.');
        } elsif ($error eq 'mismatch') {
            $msg = &mt('The e-mail address you provided does not match the address recorded in the LON-CAPA system for the username and domain you provided.');  
        } elsif ($error eq 'missing') {
            $msg = &mt('A valid e-mail address was not located in the LON-CAPA system for the username and domain you provided.');
        } elsif ($error eq 'authentication') {
            $msg = &mt('The username you provided uses an authentication type which can not be reset directly via LON-CAPA.');
        }
        if ($contact_email ne '') {
            my $escuri = &HTML::Entities::encode('/adm/resetpw','&<>"');
            $msg .= '<br />'.&mt(' You may wish to contact the <a href="/adm/helpdesk?origurl=[_1]">LON-CAPA helpdesk</a> for the [_2] domain.',$escuri,$domdesc);
        } else {
            $msg .= '<br />'.&mt(' You may wish to send an e-mail to the server administrator: [_1] for the [_2] domain.',$Apache::lonnet::perlvar{'AdminEmail'},$domdesc);
        }
    }
    return $msg;
}

sub reset_passwd {
    my ($r,$token,$contact_name,$contact_email) = @_;
    my $msg;
    my %data = &Apache::lonnet::tmpget($token);
    my $now = time;
    if (keys(%data) == 0) {
        $msg = &mt('Sorry, the URL you provided to complete the reset of your password was invalid.  Either the token included in the URL has been deleted or the URL you provided was invalid. Please submit a <a href="/adm/resetpw">new request</a> for a password reset, and follow the link to the new URL included in the e-mail that will be sent to you, to allow you to enter a new password.');
        return $msg;
    }
    if (($data{'time'} =~ /^\d+$/) && 
        ($data{'username'} ne '') && 
        ($data{'domain'} ne '') && 
        ($data{'email'}  =~ /^[^\@]+\@[^\@]+\.[^\@\.]+$/) && 
        ($data{'temppasswd'} =~/^\w+$/)) {
        my $reqtime = localtime($data{'time'});
        if ($now - $data{'time'} < 7200) {
            if ($env{'form.action'} eq 'verify_and_change_pass') {
                my $change_failed = 
		    &Apache::lonpreferences::verify_and_change_password($r,'reset_by_email',$token);
                if (!$change_failed) {
                    my $delete = &Apache::lonnet::tmpdel($token);
                    my $now = localtime(time);
                    my $domdesc = 
			&Apache::lonnet::domain($data{'domain'},'description');
                    my $mailmsg = &mt('The password for your LON-CAPA account in the [_1] domain was changed [_2] from IP address: [_3].  If you did not perform this change or authorize it, please contact the [_4] ([_5]).',$domdesc,$now,$ENV{'REMOTE_ADDR'},$contact_name,$contact_email)."\n";
                    my $result = &send_mail($domdesc,$data{'email'},$mailmsg,
                                            $contact_name,$contact_email);
                    if ($result eq 'ok') {
                        $msg .= &mt('An e-mail confirming setting of the password for your LON-CAPA account has been sent to [_1].',$data{'email'});
                    } else {
                        $msg .= &mt('An error occurred when sending e-mail to [_1] confirming setting of your new password.',$data{'email'});
                    }
                    $msg .= '<br /><br />'.&mt('<a href="/adm/login">Go to the login page</a>.');
                } else {
                    $msg .= &mt('A problem occurred when attempting to reset the password for your account.  Please contact the [_1] - (<a href="mailto:[_2]">[_2]</a>) for assistance.',$contact_name,$contact_email);
                }
            } else {
                $r->print(&mt('The token included in an email sent to you [_1] has been verified, so you may now proceed to reset the password for your LON-CAPA account.',$reqtime).'<br /><br />');
                $r->print(&mt('Please enter the username and domain of the LON-CAPA account, and the associated e-mail address, for which you are setting a password. The new password must contain at least 7 characters.').' '.&mt('Your new password will be sent to the LON-CAPA server in an encrypted form.').'<br />');
                &Apache::lonpreferences::passwordchanger($r,'','reset_by_email',$token);
            }
        } else {
            $msg = &mt('Sorry, the token generated when you requested a password reset has expired. Please submit a <a href="/adm/resetpw">new request</a>, and follow the link to the web page included in the new e-mail that will be sent to you, to allow you to enter a new password.');
        }
    } else {
        $msg .= &mt('Sorry, the URL generated when you requested reset of your password contained incomplete information. Please submit a <a href="/adm/resetpw">new request</a> for a password reset, and use the new URL that will be sent to your e-mail account to complete the process.');
    }
    return $msg;
}

sub create_passwd {
    my $passwd = '';
    my @letts = ("a".."z");
    for (my $i=0; $i<8; $i++) {
        my $lettnum = int(rand(2));
        my $item = '';
        if ($lettnum) {
            $item = $letts[int(rand(26))];
            my $uppercase = int(rand(2));
            if ($uppercase) {
                $item =~ tr/a-z/A-Z/;
            }
        } else {
            $item = int(rand(10));
        }
        $passwd .= $item;
    }
    return ($passwd);
}

1;
