#!/usr/bin/perl

=pod

=head1 NAME

make_domain_coordinator.pl - Make a domain coordinator on a LON-CAPA system

=cut

# The LearningOnline Network
# make_domain_coordinator.pl - Make a domain coordinator on a system
#
# $Id: make_domain_coordinator.pl,v 1.14 2008/01/03 21:16:11 raeburn Exp $
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###

=pod

=head1 DESCRIPTION

Automates the steps for domain coordinator creation.  This
program also describes a manual procedure (see below).

These are the steps that are executed on the linux operating system:

=over 4

=item * 

Tests to see if user already exists for linux system or for
LON-CAPA, if so aborts.  A message is output that recommends following
a manual procedure enabling this user if so desired.

=item *

Creates a linux system user

=item *

Sets password

=item *

Creates a LON-CAPA lonUsers directory for user

=item *

Sets LON-CAPA password mechanism to be "unix"

=item *

Set roles.hist and roles.db

=back

=cut

# NOTE: I am interspersing the manual procedure with the automation.
# To see the manual procedure, do perldoc ./make_domain_coordinator.pl

# This is a standalone script.  It *could* alternatively use the
# lcuseradd script, however lcuseradd relies on certain system
# dependencies.  In order to have a focused performance, I am trying
# to avoid system dependencies until the LON-CAPA code base becomes
# more robust and well-boundaried.  make_domain_coordinator.pl should be able
# to run freely as possible, irrespective of the status of a LON-CAPA
# installation.

# ---------------------------------------------------- Configure general values

use lib '/home/httpd/lib/perl/';
use LONCAPA;
use LONCAPA::lonmetadata;
use DBI;

=pod

=head1 OPTIONS

There are no flags to this script.

usage: make_domain_coordinator.pl [USERNAME] [DOMAIN] 

The password is accepted through standard input
and should only consist of printable ASCII
characters and be a string of length greater than 5 characters.

The first argument
specifies the user name of the domain coordinator and
should consist of only alphanumeric characters.
It is recommended that the USERNAME should be institution-specific
as opposed to something like "Sammy" or "Jo".
For example, "dcmsu" or "dcumich" would be good domain coordinator
USERNAMEs for places like Mich State Univ, etc.

The second argument specifies the domain of the computer
coordinator.

=cut

# ----------------------------------------------- So, are we invoked correctly?
# Two arguments or abort
if (@ARGV!=2) {
    die('usage: make_domain_coordinator.pl [USERNAME] [DOMAIN] '."\n".
	'(and password through standard input)'."\n".
	'It is recommended that the USERNAME should be institution-specific '.
	"\n".'as opposed to something like "Sammy" or "Jo".'."\n".
	'For example, "dcmsu" or "dcumich" would be good domain coordinator'.
	"\n".'USERNAMEs for places like Mich State Univ, etc.'."\n");
}
my ($username,$domain)=(@ARGV); shift @ARGV; shift @ARGV;
if ($username=~/$LONCAPA::not_username_re/) {
    die('**** ERROR **** '.
	'Username '.$username.' must consist only of - . and alphanumeric characters'.
	"\n");
}
if ($domain=~/$LONCAPA::not_domain_re/) {
    die('**** ERROR **** '.
	'Domain '.$domain.' must consist only of - . and alphanumeric charaters and '.
	"\n");
}

# Output a warning message.
print('**** NOTE **** '.
      'Generating a domain coordinator is "serious business".'."\n".
      'Choosing a difficult-to-guess (and keeping it a secret) password '."\n".
      'is highly recommended.'."\n");

print("Password: "); $|=1;
my $passwd=<>; # read in password from standard input
chomp($passwd);

if (length($passwd)<6 or length($passwd)>30) {
    die('**** ERROR **** '.'Password is an unreasonable length.'."\n".
	'It should be at least 6 characters in length.'."\n");
}
my $pbad=0;
foreach (split(//,$passwd)) {if ((ord($_)<32)||(ord($_)>126)){$pbad=1;}}
if ($pbad) {
    die('**** ERROR **** '.
	'Password must consist of standard ASCII characters'."\n");
}

# And does user already exist

my $caveat =
    'For security reasons, this script will only automatically generate '."\n".
    'new users, not pre-existing users.'."\n".
    "If you want to make '$username' a domain coordinator, you "."\n".
    'should do so manually by customizing the MANUAL PROCEDURE'."\n".
    'described in the documentation.  To view the documentation '."\n".
    'for this script, type '.
    "'perldoc ./make_domain_coordinator.pl'."."\n";

if (-d "/home/$username") {
    die ('**** ERROR **** '.$username.' is already a linux operating system '.
	 'user.'."\n".$caveat);
}
my $udpath=&propath($domain,$username);
if (-d $udpath) {
    die ('**** ERROR **** '.$username.' is already defined as a LON-CAPA '.
	 'user.'."\n".$caveat);
}

=pod

=head1 MANUAL PROCEDURE

There are 10 steps to manually recreating what this script performs
automatically.

You need to decide on three pieces of information
to create a domain coordinator.

 * USERNAME (kermit, albert, joe, etc)
 * DOMAIN (should be the same as lonDefDomain in /etc/httpd/conf/loncapa.conf)
 * PASSWORD (don't tell me)

The examples in these instructions will be based
on three example pieces of information:

 * USERNAME=dc103
 * DOMAIN=103
 * PASSWORD=sesame

You will also need to know your "root" password
and your "www" password.

=over 4

=item 1.

login as root on your Linux system
 [prompt %] su

=cut

# ------------------------------------------------------------ So, are we root?

if ($< != 0) { # Am I root?
  die 'You must be root in order to generate a domain coordinator.'."\n";
}

=pod

=item 2 (as root). add the user

 Command: [prompt %] /usr/sbin/useradd USERNAME
 Example: [prompt %] /usr/sbin/useradd dc103

=cut

# ----------------------------------------------------------- /usr/sbin/groupadd
# -- Add group
$username=~s/\W//g; # an extra filter, just to be sure

print "adding group: $username \n";
my $status = system('/usr/sbin/groupadd', $username);
if ($status) {
    die "Error.  Something went wrong with the addition of group ".
          "\"$username\".\n";
}
my $gid = getgrnam($username);

# ----------------------------------------------------------- /usr/sbin/useradd
# -- Add user

print "adding user: $username \n";
my $status = system('/usr/sbin/useradd','-c','LON-CAPA user','-g',$gid,$username);
if ($status) {
    system("/usr/sbin/groupdel $username");
    die "Error.  Something went wrong with the addition of user ".
          "\"$username\".\n";
}

print "Done adding user\n";
# Make www a member of that user group.
my $groups=`/usr/bin/groups www`;
# untaint
my ($safegroups)=($groups=~/:\s*([\s\w]+)/);
$groups=$safegroups;
chomp $groups; $groups=~s/^\S+\s+\:\s+//;
my @grouplist=split(/\s+/,$groups);
my @ugrouplist=grep {!/www|$username/} @grouplist;
my $gl=join(',',(@ugrouplist,$username));
print "Putting www in user's group\n";
if (system('/usr/sbin/usermod','-G',$gl,'www')) {
    die "Error. Could not make www a member of the group ".
          "\"$username\".\n";
}

# Check if home directory exists for user
# If not, create one.
if (!-e "/home/$username") {
    if (!mkdir("/home/$username",0710)) {
        print "Error. Could not add home directory for ".
          "\"$username\".\n";
    }
}

if (-d "/home/$username") {
    system('/bin/chown',"$username:$username","/home/$username");
    system('/bin/chmod','-R','0660',"/home/$username");
    system('/bin/chmod','0710',"/home/$username");
}
=pod

=item 3 (as root). enter in a password

 Command: [prompt %] passwd USERNAME
          New UNIX password: PASSWORD
          Retype new UNIX passwd: PASSWORD
 Example: [prompt %] passwd dc103
          New UNIX password: sesame
          Retype new UNIX passwd: sesame

=cut

# Process password (taint-check, then pass to the UNIX passwd command).
$username =~ s/\W//g; # an extra filter, just to be sure
$pbad = 0;
foreach (split(//,$passwd)) {if ((ord($_)<32)||(ord($_)>126)){$pbad=1;}}
if ($pbad) {
    die('Password must consist of standard ASCII characters'."\n");
}
open(OUT,"|passwd --stdin $username");
print(OUT $passwd."\n");
close(OUT);

=pod

=cut

=pod

=item 4. login as user=www

 Command: [prompt %] su www
 Password: WWWPASSWORD

=item 5. (as www). cd /home/httpd/lonUsers

=item 6. (as www) Create user directory for your new user.

 Let U equal first letter of USERNAME
 Let S equal second letter of USERNAME
 Let E equal third letter of USERNAME
 Command: [prompt %] install -d DOMAIN/U/S/E/USERNAME

 Here are three examples of the commands that would be needed
 for different domain coordinator names (dc103, morphy, or ng):

 Example #1 (dc103):  [prompt %] install -d 103/d/c/1/dc103
 Example #2 (morphy): [prompt %] install -d 103/m/o/r/morphy
 Example #3 (ng):     [prompt %] install -d 103/n/g/_/ng

=cut

# Generate the user directory.
`install -o www -g www -d $udpath`; # Must be writeable by httpd process.

=pod

=item 7. (as www) Enter the newly created user directory.

 Command: [prompt %] cd DOMAIN/U/S/E/USERNAME
 Example: [prompt %] cd 103/d/c/1/dc103

=item 8. (as www). Set your password mechanism to 'unix' 

 Command: [prompt %] echo "unix:" > passwd

=cut

# UNIX (/etc/passwd) style authentication is asserted for domain coordinators.
open(OUT, ">$udpath/passwd");
print(OUT 'unix:'."\n");
close(OUT);
`chown www:www $udpath/passwd`; # Must be writeable by httpd process.

=pod

=item 9. (as www). Run CVS:loncapa/doc/rolesmanip.pl:

 Command: [prompt %] perl rolesmanip.pl DOMAIN USERNAME
 Example: [prompt %] perl rolesmanip.pl 103 dc103

=cut

use GDBM_File; # A simplistic key-value pairing database.

my $rolesref=&LONCAPA::locking_hash_tie("$udpath/roles.db",&GDBM_WRCREAT());
if (!$rolesref) {
    die('unable to tie roles db: '."$udpath/roles.db");
}
my $now = time;
$rolesref->{'/'.$domain.'/_dc'}='dc_0_'.$now; # Set the domain coordinator role.
open(OUT, ">$udpath/roles.hist"); # roles.hist is the synchronous plain text.
foreach my $key (keys(%{$rolesref})) {
    print(OUT $key.' : '.$rolesref->{$key}."\n");
}
close(OUT);
&LONCAPA::locking_hash_untie($rolesref);


`chown www:www $udpath/roles.hist`; # Must be writeable by httpd process.
`chown www:www $udpath/roles.db`; # Must be writeable by httpd process.

my %perlvar = %{&LONCAPA::Configuration::read_conf('loncapa.conf')};
my $dompath = $perlvar{'lonUsersDir'}.'/'.$domain;
my $domrolesref = &LONCAPA::locking_hash_tie("$dompath/nohist_domainroles.db",&GDBM_WRCREAT());

if (!$domrolesref) {
    die('unable to tie nohist_domainroles db: '."$dompath/nohist_domainroles.db");
}

# Store in nohist_domainroles.db
my $domkey=&LONCAPA::escape('dc:'.$username.':'.$domain.'::'.$domain.':');
$domrolesref->{$domkey}= &LONCAPA::escape('0:'.$now);
&LONCAPA::locking_hash_untie($domrolesref);

 system('/bin/chown',"www:www","$dompath/nohist_domainroles.db"); # Must be writeable by httpd process.
 system('/bin/chown',"www:www","$dompath/nohist_domainroles.db.lock");

#Update allusers MySQL table

print "Adding new user to allusers table\n";
&allusers_update($username,$domain,\%perlvar);

=pod

=item 10.

You may further define the domain coordinator user (i.e. dc103)
by going to http://MACHINENAME/adm/createuser.

=cut

# Output success message, and inform sysadmin about how to further proceed.
print("\n$username is now a domain coordinator\n"); # Output success message.
my $hostname=`hostname`; chomp($hostname); # Read in hostname.
print("\n".'Once LON-CAPA is running, you should log-in and use: '."\n".
      'http://'.$hostname.'/adm/createuser to further define this user.'."\n\n".
      'From the user management menu, click the link: "Add/Modify a Single User" '."\n".
      'to search for the user and to provide additional information (last name, first name etc.).'."\n");
# Output a suggested URL.

sub allusers_update {
    my ($username,$domain,$perlvar) = @_;
    my %tablenames = (
                       'allusers'   => 'allusers',
                     );
    my $dbh;
    unless ($dbh = DBI->connect("DBI:mysql:loncapa","www",
                            $perlvar->{'lonSqlAccess'},
                            { RaiseError =>0,PrintError=>0})) {
        print "Cannot connect to database!\n";
        return;
    }
    my $tablechk = &allusers_table_exists($dbh);
    if ($tablechk == 0) {
        my $request =
   &LONCAPA::lonmetadata::create_metadata_storage('allusers','allusers');
        $dbh->do($request);
        if ($dbh->err) {
             print "Failed to crate allusers table\n";
             return;
        }
    }
    my %userdata =  (
                username => $username,
                domain   => $domain,
    );
    my %loghash =
        &LONCAPA::lonmetadata::process_allusers_data($dbh,undef,
            \%tablenames,$username,$domain,\%userdata,'update');
    foreach my $key (keys(%loghash)) {
        print $loghash{$key}."\n";
    }
    return;
}

sub allusers_table_exists {
    my ($dbh) = @_;
    my $sth=$dbh->prepare('SHOW TABLES');
    $sth->execute();
    my $aref = $sth->fetchall_arrayref;
    $sth->finish();
    if ($sth->err()) {
        return undef;
    }
    my $result = 0;
    foreach my $table (@{$aref}) {
        if ($table->[0] eq 'allusers') {
            $result = 1;
            last;
        }
    }
    return $result;
}

=pod

=head1 AUTHOR

Written to help the LON-CAPA project.

=cut

