# The LearningOnline Network with CAPA
# a pile of common html routines
#
# $Id: lonhtmlcommon.pm,v 1.173 2008/03/06 22:30:17 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
######################################################################
######################################################################

=pod

=head1 NAME

Apache::lonhtmlcommon - routines to do common html things

=head1 SYNOPSIS

Referenced by other mod_perl Apache modules.

=head1 INTRODUCTION

lonhtmlcommon is a collection of subroutines used to present information
in a consistent html format, or provide other functionality related to
html.

=head2 General Subroutines

=over 4

=cut 

######################################################################
######################################################################

package Apache::lonhtmlcommon;

use strict;
use Time::Local;
use Time::HiRes;
use Apache::lonlocal;
use Apache::lonnet;
use LONCAPA;

##############################################
##############################################

=pod

=item authorbombs

=cut

##############################################
##############################################

sub authorbombs {
    my $url=shift;
    $url=&Apache::lonnet::declutter($url);
    my ($udom,$uname)=($url=~m{^($LONCAPA::domain_re)/($LONCAPA::username_re)/});
    my %bombs=&Apache::lonmsg::all_url_author_res_msg($uname,$udom);
    foreach (keys %bombs) {
	if ($_=~/^$udom\/$uname\//) {
	    return '<a href="/adm/bombs/'.$url.
		'"><img src="'.&Apache::loncommon::lonhttpdurl('/adm/lonMisc/bomb.gif').'" border="0" /></a>'.
		&Apache::loncommon::help_open_topic('About_Bombs');
	}
    }
    return '';
}

##############################################
##############################################

sub recent_filename {
    my $area=shift;
    return 'nohist_recent_'.&escape($area);
}

sub store_recent {
    my ($area,$name,$value,$freeze)=@_;
    my $file=&recent_filename($area);
    my %recent=&Apache::lonnet::dump($file);
    if (scalar(keys(%recent))>20) {
# remove oldest value
	my $oldest=time();
	my $delkey='';
	foreach my $item (keys(%recent)) {
	    my $thistime=(split(/\&/,$recent{$item}))[0];
	    if (($thistime ne "always_include") && ($thistime<$oldest)) {
		$oldest=$thistime;
		$delkey=$item;
	    }
	}
	&Apache::lonnet::del($file,[$delkey]);
    }
# store new value
    my $timestamp;
    if ($freeze) {
        $timestamp = "always_include";
    } else {
        $timestamp = time();
    }   
    &Apache::lonnet::put($file,{ $name => 
				 $timestamp.'&'.&escape($value) });
}

sub remove_recent {
    my ($area,$names)=@_;
    my $file=&recent_filename($area);
    return &Apache::lonnet::del($file,$names);
}

sub select_recent {
    my ($area,$fieldname,$event)=@_;
    my %recent=&Apache::lonnet::dump(&recent_filename($area));
    my $return="\n<select name='$fieldname'".
	($event?" onchange='$event'":'').
	">\n<option value=''>--- ".&mt('Recent')." ---</option>";
    foreach my $value (sort(keys(%recent))) {
	unless ($value =~/^error\:/) {
	    my $escaped = &Apache::loncommon::escape_url($value);
	    &Apache::loncommon::inhibit_menu_check(\$escaped);
	    $return.="\n<option value='$escaped'>".
		&unescape((split(/\&/,$recent{$value}))[1]).
		'</option>';
	}
    }
    $return.="\n</select>\n";
    return $return;
}

sub get_recent {
    my ($area, $n) = @_;
    my %recent=&Apache::lonnet::dump(&recent_filename($area));

# Create hash with key as time and recent as value
# Begin filling return_hash with any 'always_include' option
    my %time_hash = ();
    my %return_hash = ();
    foreach my $item (keys %recent) {
        my ($thistime,$thisvalue)=(split(/\&/,$recent{$item}));
        if ($thistime eq 'always_include') {
            $return_hash{$item} = &unescape($thisvalue);
            $n--;
        } else {
            $time_hash{$thistime} = $item;
        }
    }

# Sort by decreasing time and return key value pairs
    my $idx = 1;
    foreach my $item (reverse(sort(keys(%time_hash)))) {
       $return_hash{$time_hash{$item}} =
                  &unescape((split(/\&/,$recent{$time_hash{$item}}))[1]);
       if ($n && ($idx++ >= $n)) {last;}
    }

    return %return_hash;
}

sub get_recent_frozen {
    my ($area) = @_;
    my %recent=&Apache::lonnet::dump(&recent_filename($area));

# Create hash with all 'frozen' items
    my %return_hash = ();
    foreach my $item (keys(%recent)) {
        my ($thistime,$thisvalue)=(split(/\&/,$recent{$item}));
        if ($thistime eq 'always_include') {
            $return_hash{$item} = &unescape($thisvalue);
        }
    }
    return %return_hash;
}



=pod

=item textbox

=cut

##############################################
##############################################
sub textbox {
    my ($name,$value,$size,$special) = @_;
    $size = 40 if (! defined($size));
    $value = &HTML::Entities::encode($value,'<>&"');
    my $Str = '<input type="text" name="'.$name.'" size="'.$size.'" '.
        'value="'.$value.'" '.$special.' />';
    return $Str;
}

##############################################
##############################################

=pod

=item checkbox

=cut

##############################################
##############################################
sub checkbox {
    my ($name,$checked,$value) = @_;
    my $Str = '<input type="checkbox" name="'.$name.'" ';
    if (defined($value)) {
        $Str .= 'value="'.$value.'"';
    } 
    if ($checked) {
        $Str .= ' checked="1"';
    }
    $Str .= ' />';
    return $Str;
}


=pod

=item radiobutton

=cut

##############################################
##############################################
sub radio {
    my ($name,$checked,$value) = @_;
    my $Str = '<input type="radio" name="'.$name.'" ';
    if (defined($value)) {
        $Str .= 'value="'.$value.'"';
    } 
    if ($checked eq $value) {
        $Str .= ' checked="1"';
    }
    $Str .= ' />';
    return $Str;
}

##############################################
##############################################

=pod

=item &date_setter

&date_setter returns html and javascript for a compact date-setting form.
To retrieve values from it, use &get_date_from_form().

Inputs

=over 4

=item $dname 

The name to prepend to the form elements.  
The form elements defined will be dname_year, dname_month, dname_day,
dname_hour, dname_min, and dname_sec.

=item $currentvalue

The current setting for this time parameter.  A unix format time
(time in seconds since the beginning of Jan 1st, 1970, GMT.  
An undefined value is taken to indicate the value is the current time.
Also, to be explicit, a value of 'now' also indicates the current time.

=item $special

Additional html/javascript to be associated with each element in
the date_setter.  See lonparmset for example usage.

=item $includeempty 

=item $state

Specifies the initial state of the form elements.  Either 'disabled' or empty.
Defaults to empty, which indiciates the form elements are not disabled. 

=back

Bugs

The method used to restrict user input will fail in the year 2400.

=cut

##############################################
##############################################
sub date_setter {
    my ($formname,$dname,$currentvalue,$special,$includeempty,$state,
        $no_hh_mm_ss,$defhour,$defmin,$defsec,$nolink) = @_;
    my $wasdefined=1;
    if (! defined($state) || $state ne 'disabled') {
        $state = '';
    }
    if (! defined($no_hh_mm_ss)) {
        $no_hh_mm_ss = 0;
    }
    if ($currentvalue eq 'now') {
	$currentvalue=time;
    }
    if ((!defined($currentvalue)) || ($currentvalue eq '')) {
	$wasdefined=0;
	if ($includeempty) {
	    $currentvalue = 0;
	} else {
	    $currentvalue = time;
	}
    }
    # other potentially useful values:     wkday,yrday,is_daylight_savings
    my ($sec,$min,$hour,$mday,$month,$year)=('','',undef,'','','');
    if ($currentvalue) {
	($sec,$min,$hour,$mday,$month,$year,undef,undef,undef) = 
	    localtime($currentvalue);
	$year += 1900;
    }
    unless ($wasdefined) {
	if (($defhour) || ($defmin) || ($defsec)) {
	    ($sec,$min,$hour,$mday,$month,$year,undef,undef,undef) = 
		localtime(time);
	    $year += 1900;
	    $sec=($defsec?$defsec:0);
	    $min=($defmin?$defmin:0);
	    $hour=($defhour?$defhour:0);
	} elsif (!$includeempty) {
	    $sec=0;
	    $min=0;
	    $hour=0;
	}
    }
    my $result = "\n<!-- $dname date setting form -->\n";
    $result .= <<ENDJS;
<script type="text/javascript">
    function $dname\_checkday() {
        var day   = document.$formname.$dname\_day.value;
        var month = document.$formname.$dname\_month.value;
        var year  = document.$formname.$dname\_year.value;
        var valid = true;
        if (day < 1) {
            document.$formname.$dname\_day.value = 1;
        } 
        if (day > 31) {
            document.$formname.$dname\_day.value = 31;
        }
        if ((month == 1)  || (month == 3)  || (month == 5)  ||
            (month == 7)  || (month == 8)  || (month == 10) ||
            (month == 12)) {
            if (day > 31) {
                document.$formname.$dname\_day.value = 31;
                day = 31;
            }
        } else if (month == 2 ) {
            if ((year % 4 == 0) && (year % 100 != 0)) {
                if (day > 29) {
                    document.$formname.$dname\_day.value = 29;
                }
            } else if (day > 29) {
                document.$formname.$dname\_day.value = 28;
            }
        } else if (day > 30) {
            document.$formname.$dname\_day.value = 30;
        }
    }
    
    function $dname\_disable() {
        document.$formname.$dname\_month.disabled=true;
        document.$formname.$dname\_day.disabled=true;
        document.$formname.$dname\_year.disabled=true;
        document.$formname.$dname\_hour.disabled=true;
        document.$formname.$dname\_minute.disabled=true;
        document.$formname.$dname\_second.disabled=true;
    }

    function $dname\_enable() {
        document.$formname.$dname\_month.disabled=false;
        document.$formname.$dname\_day.disabled=false;
        document.$formname.$dname\_year.disabled=false;
        document.$formname.$dname\_hour.disabled=false;
        document.$formname.$dname\_minute.disabled=false;
        document.$formname.$dname\_second.disabled=false;        
    }

    function $dname\_opencalendar() {
        if (! document.$formname.$dname\_month.disabled) {
            var calwin=window.open(
"/adm/announcements?pickdate=yes&formname=$formname&element=$dname&month="+
document.$formname.$dname\_month.value+"&year="+
document.$formname.$dname\_year.value,
             "LONCAPAcal",
              "height=350,width=350,scrollbars=yes,resizable=yes,menubar=no");
        }

    }
</script>
ENDJS
    $result .= '  <span style="white-space: nowrap;">';
    my $monthselector = qq{<select name="$dname\_month" $special $state onchange="javascript:$dname\_checkday()" >};
    # Month
    my @Months = qw/January February  March     April   May      June 
                    July    August    September October November December/;
    # Pad @Months with a bogus value to make indexing easier
    unshift(@Months,'If you can read this an error occurred');
    if ($includeempty) { $monthselector.="<option value=''></option>"; }
    for(my $m = 1;$m <=$#Months;$m++) {
        $monthselector .= qq{      <option value="$m" };
        $monthselector .= "selected " if ($m-1 eq $month);
        $monthselector .= '> '.&mt($Months[$m]).' </option>';
    }
    $monthselector.= '  </select>';
    # Day
    my $dayselector = qq{<input type="text" name="$dname\_day" $state value="$mday" size="3" $special onchange="javascript:$dname\_checkday()" />};
    # Year
    my $yearselector = qq{<input type="year" name="$dname\_year" $state value="$year" size="5" $special onchange="javascript:$dname\_checkday()" />};
    #
    my $hourselector = qq{<select name="$dname\_hour" $special $state >};
    if ($includeempty) { 
        $hourselector.=qq{<option value=''></option>};
    }
    for (my $h = 0;$h<24;$h++) {
        $hourselector .= qq{<option value="$h" };
        $hourselector .= "selected " if (defined($hour) && $hour == $h);
        $hourselector .= ">";
        my $timest='';
        if ($h == 0) {
            $timest .= "12 am";
        } elsif($h == 12) {
            $timest .= "12 noon";
        } elsif($h < 12) {
            $timest .= "$h am";
        } else {
            $timest .= $h-12 ." pm";
        }
        $timest=&mt($timest);
        $hourselector .= $timest." </option>\n";
    }
    $hourselector .= "  </select>\n";
    my $minuteselector = qq{<input type="text" name="$dname\_minute" $special $state value="$min" size="3" />};
    my $secondselector= qq{<input type="text" name="$dname\_second" $special $state value="$sec" size="3" />};
    my $cal_link;
    if (!$nolink) {
        $cal_link = qq{<a href="javascript:$dname\_opencalendar()">};
    }
    #
    if ($no_hh_mm_ss) {
        $result .= &mt('[_1] [_2] [_3] ',
                       $monthselector,$dayselector,$yearselector);
        if (!$nolink) {
            $result .= &mt('[_1]Select Date[_2]',$cal_link,'</a>');
        }
    } else {
        $result .= &mt('[_1] [_2] [_3] [_4] [_5]m [_6]s ',
                      $monthselector,$dayselector,$yearselector,
                      $hourselector,$minuteselector,$secondselector);
        if (!$nolink) {
            $result .= &mt('[_1]Select Date[_2]',$cal_link,'</a>');
        }
    }
    $result .= "</span>\n<!-- end $dname date setting form -->\n";
    return $result;
}


sub build_url {
    my ($base, $fields)=@_;
    my $url;
    $url = $base.'?';
    foreach my $key (keys(%$fields)) {
        $url.=&escape($key).'='.&escape($$fields{$key}).'&amp;';
    }
    $url =~ s/&amp;$//;
    return $url;
}


##############################################
##############################################

=pod

=item &get_date_from_form

get_date_from_form retrieves the date specified in an &date_setter form.

Inputs:

=over 4

=item $dname

The name passed to &datesetter, which prefixes the form elements.

=item $defaulttime

The unix time to use as the default in case of poor inputs.

=back

Returns: Unix time represented in the form.

=cut

##############################################
##############################################
sub get_date_from_form {
    my ($dname) = @_;
    my ($sec,$min,$hour,$day,$month,$year);
    #
    if (defined($env{'form.'.$dname.'_second'})) {
        my $tmpsec = $env{'form.'.$dname.'_second'};
        if (($tmpsec =~ /^\d+$/) && ($tmpsec >= 0) && ($tmpsec < 60)) {
            $sec = $tmpsec;
        }
	if (!defined($tmpsec) || $tmpsec eq '') { $sec = 0; }
    } else {
        $sec = 0;
    }
    if (defined($env{'form.'.$dname.'_minute'})) {
        my $tmpmin = $env{'form.'.$dname.'_minute'};
        if (($tmpmin =~ /^\d+$/) && ($tmpmin >= 0) && ($tmpmin < 60)) {
            $min = $tmpmin;
        }
	if (!defined($tmpmin) || $tmpmin eq '') { $min = 0; }
    } else {
        $min = 0;
    }
    if (defined($env{'form.'.$dname.'_hour'})) {
        my $tmphour = $env{'form.'.$dname.'_hour'};
        if (($tmphour =~ /^\d+$/) && ($tmphour >= 0) && ($tmphour < 24)) {
            $hour = $tmphour;
        }
    } else {
        $hour = 0;
    }
    if (defined($env{'form.'.$dname.'_day'})) {
        my $tmpday = $env{'form.'.$dname.'_day'};
        if (($tmpday =~ /^\d+$/) && ($tmpday > 0) && ($tmpday < 32)) {
            $day = $tmpday;
        }
    }
    if (defined($env{'form.'.$dname.'_month'})) {
        my $tmpmonth = $env{'form.'.$dname.'_month'};
        if (($tmpmonth =~ /^\d+$/) && ($tmpmonth > 0) && ($tmpmonth < 13)) {
            $month = $tmpmonth - 1;
        }
    }
    if (defined($env{'form.'.$dname.'_year'})) {
        my $tmpyear = $env{'form.'.$dname.'_year'};
        if (($tmpyear =~ /^\d+$/) && ($tmpyear > 1900)) {
            $year = $tmpyear - 1900;
        }
    }
    if (($year<70) || ($year>137)) { return undef; }
    if (defined($sec) && defined($min)   && defined($hour) &&
        defined($day) && defined($month) && defined($year) &&
        eval('&timelocal($sec,$min,$hour,$day,$month,$year)')) {
        return &timelocal($sec,$min,$hour,$day,$month,$year);
    } else {
        return undef;
    }
}

##############################################
##############################################

=pod

=item &pjump_javascript_definition()

Returns javascript defining the 'pjump' function, which opens up a
parameter setting wizard.

=cut

##############################################
##############################################
sub pjump_javascript_definition {
    my $Str = <<END;
    function pjump(type,dis,value,marker,ret,call,hour,min,sec) {
        parmwin=window.open("/adm/rat/parameter.html?type="+escape(type)
                 +"&value="+escape(value)+"&marker="+escape(marker)
                 +"&return="+escape(ret)
                 +"&call="+escape(call)+"&name="+escape(dis)
                 +"&defhour="+escape(hour)+"&defmin="+escape(min)
                 +"&defsec="+escape(sec),"LONCAPAparms",
                 "height=350,width=350,scrollbars=no,menubar=no");
    }
END
    return $Str;
}

##############################################
##############################################

=pod

=item &javascript_nothing()

Return an appropriate null for the users browser.  This is used
as the first arguement for window.open calls when you want a blank
window that you can then write to.

=cut

##############################################
##############################################
sub javascript_nothing {
    # mozilla and other browsers work with "''", but IE on mac does not.
    my $nothing = "''";
    my $user_browser;
    my $user_os;
    $user_browser = $env{'browser.type'} if (exists($env{'browser.type'}));
    $user_os      = $env{'browser.os'}   if (exists($env{'browser.os'}));
    if (! defined($user_browser) || ! defined($user_os)) {
        (undef,$user_browser,undef,undef,undef,$user_os) = 
                           &Apache::loncommon::decode_user_agent();
    }
    if ($user_browser eq 'explorer' && $user_os =~ 'mac') {
        $nothing = "'javascript:void(0);'";
    }
    return $nothing;
}

##############################################
##############################################
sub javascript_docopen {
    my ($mimetype) = @_;
    $mimetype ||= 'text/html';
    # safari does not understand document.open() and loads "text/html"
    my $nothing = "''";
    my $user_browser;
    my $user_os;
    $user_browser = $env{'browser.type'} if (exists($env{'browser.type'}));
    $user_os      = $env{'browser.os'}   if (exists($env{'browser.os'}));
    if (! defined($user_browser) || ! defined($user_os)) {
        (undef,$user_browser,undef,undef,undef,$user_os) = 
                           &Apache::loncommon::decode_user_agent();
    }
    if ($user_browser eq 'safari' && $user_os =~ 'mac') {
        $nothing = "document.clear()";
    } else {
	$nothing = "document.open('$mimetype','replace')";
    }
    return $nothing;
}


##############################################
##############################################

=pod

=item &StatusOptions()

Returns html for a selection box which allows the user to choose the
enrollment status of students.  The selection box name is 'Status'.

Inputs:

$status: the currently selected status.  If undefined the value of
$env{'form.Status'} is taken.  If that is undefined, a value of 'Active'
is used.

$formname: The name of the form.  If defined the onchange attribute of
the selection box is set to document.$formname.submit().

$size: the size (number of lines) of the selection box.

$onchange: javascript to use when the value is changed.  Enclosed in 
double quotes, ""s, not single quotes.

Returns: a perl string as described.

=cut

##############################################
##############################################
sub StatusOptions {
    my ($status, $formName,$size,$onchange,$mult)=@_;
    $size = 1 if (!defined($size));
    if (! defined($status)) {
        $status = 'Active';
        $status = $env{'form.Status'} if (exists($env{'form.Status'}));
    }

    my $Str = '';
    $Str .= '<select name="Status"';
    if (defined($mult)){
        $Str .= ' multiple="multiple" ';
    }
    if(defined($formName) && $formName ne '' && ! defined($onchange)) {
        $Str .= ' onchange="document.'.$formName.'.submit()"';
    }
    if (defined($onchange)) {
        $Str .= ' onchange="'.$onchange.'"';
    }
    $Str .= ' size="'.$size.'" ';
    $Str .= '>'."\n";
    foreach my $type (['Active',  &mt('Currently Has Access')],
		      ['Future',  &mt('Will Have Future Access')],
		      ['Expired', &mt('Previously Had Access')],
		      ['Any',     &mt('Any Access Status')]) {
	my ($name,$label) = @$type;
	$Str .= '<option value="'.$name.'" ';
	if ($status eq $name) {
	    $Str .= 'selected="selected" ';
	}
	$Str .= '>'.$label.'</option>'."\n";
    }

    $Str .= '</select>'."\n";
}

########################################################
########################################################

=pod

=item Progess Window Handling Routines

These routines handle the creation, update, increment, and closure of 
progress windows.  The progress window reports to the user the number
of items completed and an estimate of the time required to complete the rest.

=over 4


=item &Create_PrgWin

Writes javascript to the client to open a progress window and returns a
data structure used for bookkeeping.

Inputs

=over 4

=item $r Apache request

=item $title The title of the progress window

=item $heading A description (usually 1 line) of the process being initiated.

=item $number_to_do The total number of items being processed.

=item $type Either 'popup' or 'inline' (popup is assumed if nothing is
       specified)

=item $width Specify the width in charaters of the input field.

=item $formname Only useful in the inline case, if a form already exists, this needs to be used and specfiy the name of the form, otherwise the Progress line will be created in a new form of it's own

=item $inputname Only useful in the inline case, if a form and an input of type text exists, use this to specify the name of the input field 

=back

Returns a hash containing the progress state data structure.


=item &Update_PrgWin

Updates the text in the progress indicator.  Does not increment the count.
See &Increment_PrgWin.

Inputs:

=over 4

=item $r Apache request

=item $prog_state Pointer to the data structure returned by &Create_PrgWin

=item $displaystring The string to write to the status indicator

=back

Returns: none


=item Increment_PrgWin

Increment the count of items completed for the progress window by 1.  

Inputs:

=over 4

=item $r Apache request

=item $prog_state Pointer to the data structure returned by Create_PrgWin

=item $extraInfo A description of the items being iterated over.  Typically
'student'.

=back

Returns: none


=item Close_PrgWin

Closes the progress window.

Inputs:

=over 4 

=item $r Apache request

=item $prog_state Pointer to the data structure returned by Create_PrgWin

=back

Returns: none

=back

=cut

########################################################
########################################################

my $uniq=0;
sub get_uniq_name {
    $uniq++;
    return 'uniquename'.$uniq;
}

# Create progress
sub Create_PrgWin {
    my ($r, $title, $heading, $number_to_do,$type,$width,$formname,
	$inputname)=@_;
    if (!defined($type)) { $type='popup'; }
    if (!defined($width)) { $width=55; }
    my %prog_state;
    $prog_state{'type'}=$type;
    if ($type eq 'popup') {
	$prog_state{'window'}='popwin';
	my $start_page =
	    &Apache::loncommon::start_page($title,undef,
					   {'only_body' => 1,
					    'bgcolor'   => '#88DDFF',
					    'js_ready'  => 1});
	my $end_page = &Apache::loncommon::end_page({'js_ready'  => 1});

	#the whole function called through timeout is due to issues
	#in mozilla Read BUG #2665 if you want to know the whole story
	&r_print($r,'<script type="text/javascript">'.
        "var popwin;
         function openpopwin () {
         popwin=open(\'\',\'popwin\',\'width=400,height=100\');".
        "popwin.document.writeln(\'".$start_page.
              "<h4>".&mt("$heading")."<\/h4>".
              "<form action= \"\" name=\"popremain\" method=\"post\">".
              '<input type="text" size="'.$width.'" name="remaining" value="'.
	      &mt('Starting').'" /><\\/form>'.$end_page.
              "\');".
        "popwin.document.close();}".
        "\nwindow.setTimeout(openpopwin,0)</script>");
	$prog_state{'formname'}='popremain';
	$prog_state{'inputname'}="remaining";
    } elsif ($type eq 'inline') {
	$prog_state{'window'}='window';
	if (!$formname) {
	    $prog_state{'formname'}=&get_uniq_name();
	    &r_print($r,'<form action="" name="'.$prog_state{'formname'}.'">');
	} else {
	    $prog_state{'formname'}=$formname;
	}
	if (!$inputname) {
	    $prog_state{'inputname'}=&get_uniq_name();
	    &r_print($r,&mt("$heading [_1]",' <input type="text" name="'.$prog_state{'inputname'}.'" size="'.$width.'" />'));
	} else {
	    $prog_state{'inputname'}=$inputname;
	    
	}
	if (!$formname) { &r_print($r,'</form>'); }
	&Update_PrgWin($r,\%prog_state,&mt('Starting'));
    }

    $prog_state{'done'}=0;
    $prog_state{'firststart'}=&Time::HiRes::time();
    $prog_state{'laststart'}=&Time::HiRes::time();
    $prog_state{'max'}=$number_to_do;
    
    return %prog_state;
}

# update progress
sub Update_PrgWin {
    my ($r,$prog_state,$displayString)=@_;
    &r_print($r,'<script type="text/javascript">'.$$prog_state{'window'}.'.document.'.
	     $$prog_state{'formname'}.'.'.
	     $$prog_state{'inputname'}.'.value="'.
	     $displayString.'";</script>');
    $$prog_state{'laststart'}=&Time::HiRes::time();
}

# increment progress state
sub Increment_PrgWin {
    my ($r,$prog_state,$extraInfo)=@_;
    $$prog_state{'done'}++;
    my $time_est= (&Time::HiRes::time() - $$prog_state{'firststart'})/
        $$prog_state{'done'} *
	($$prog_state{'max'}-$$prog_state{'done'});
    $time_est = int($time_est);
    #
    my $min = int($time_est/60);
    my $sec = $time_est % 60;
    # 
    my $str;
    if ($min == 0 && $sec > 1) {
        $str = '[_2] seconds';
    } elsif ($min == 1 && $sec > 1) {
        $str = '1 minute [_2] seconds';
    } elsif ($min == 1 && $sec < 2) {
        $str = '1 minute';
    } elsif ($min < 10 && $sec > 1) {
        $str = '[_1] minutes, [_2] seconds';
    } elsif ($min >= 10 || $sec < 2) {
        $str = '[_1] minutes';
    }
    $time_est = &mt($str,$min,$sec);
    #
    my $lasttime = &Time::HiRes::time()-$$prog_state{'laststart'};
    if ($lasttime > 9) {
        $lasttime = int($lasttime);
    } elsif ($lasttime < 0.01) {
        $lasttime = 0;
    } else {
        $lasttime = sprintf("%3.2f",$lasttime);
    }
    if ($lasttime == 1) {
        $lasttime = '('.$lasttime.' '.&mt('second for').' '.$extraInfo.')';
    } else {
        $lasttime = '('.$lasttime.' '.&mt('seconds for').' '.$extraInfo.')';
    }
    #
    my $user_browser = $env{'browser.type'} if (exists($env{'browser.type'}));
    my $user_os      = $env{'browser.os'}   if (exists($env{'browser.os'}));
    if (! defined($user_browser) || ! defined($user_os)) {
        (undef,$user_browser,undef,undef,undef,$user_os) = 
                           &Apache::loncommon::decode_user_agent();
    }
    if ($user_browser eq 'explorer' && $user_os =~ 'mac') {
        $lasttime = '';
    }
    &r_print($r,'<script>'.$$prog_state{'window'}.'.document.'.
	     $$prog_state{'formname'}.'.'.
	     $$prog_state{'inputname'}.'.value="'.
	     $$prog_state{'done'}.'/'.$$prog_state{'max'}.
	     ': '.$time_est.' '.&mt('remaining').' '.$lasttime.'";'.'</script>');
    $$prog_state{'laststart'}=&Time::HiRes::time();
}

# close Progress Line
sub Close_PrgWin {
    my ($r,$prog_state)=@_;
    if ($$prog_state{'type'} eq 'popup') {
	&r_print($r,'<script>popwin.close()</script>'."\n");
    } elsif ($$prog_state{'type'} eq 'inline') {
	&Update_PrgWin($r,$prog_state,&mt('Done'));
    }
    undef(%$prog_state);
}

sub r_print {
    my ($r,$to_print)=@_;
    if ($r) {
	$r->print($to_print);
	$r->rflush();
    } else {
	print($to_print);
    }
}

# ------------------------------------------------------- Puts directory header

sub crumbs {
    my ($uri,$target,$prefix,$form,$size,$noformat,$skiplast)=@_;
    if (! defined($size)) {
        $size = '+2';
    }
    if ($target) {
        $target = ' target="'.
                  &Apache::loncommon::escape_single($target).'"';
    }
    my $output='';
    unless ($noformat) { $output.='<br /><tt><b>'; }
    $output.='<font size="'.$size.'">'.$prefix.'/';
    if ($env{'user.adv'}) {
	my $path=$prefix.'/';
	foreach my $dir (split('/',$uri)) {
            if (! $dir) { next; }
            $path .= $dir;
	    if ($path eq $uri) {
		if ($skiplast) {
		    $output.=$dir;
                    last;
		} 
	    } else {
		$path.='/'; 
	    }	    
            my $href_path = &HTML::Entities::encode($path,'<>&"');
	    &Apache::loncommon::inhibit_menu_check(\$href_path);
	    if ($form) {
	        my $href = 'javascript:'.$form.".action='".$href_path."';".$form.'.submit();';
	        $output.=qq{<a href="$href" $target>$dir</a>/};
	    } else {
	        $output.=qq{<a href="$href_path" $target>$dir</a>/};
	    }
	}
    } else {
	foreach my $dir (split('/',$uri)) {
            if (! $dir) { next; }
	    $output.=$dir.'/';
	}
    }
    if ($uri !~ m|/$|) { $output=~s|/$||; }
    return $output.'</font>'.($noformat?'':'</b></tt><br />');
}

# --------------------- A function that generates a window for the spellchecker

sub spellheader {
    my $start_page=
	&Apache::loncommon::start_page('Speller Suggestions',undef,
				       {'only_body'   => 1,
					'js_ready'    => 1,
					'bgcolor'     => '#DDDDDD',
				        'add_entries' => {
					    'onload' => 
                                               'document.forms.spellcheckform.submit()',
                                             }
				        });
    my $end_page=
	&Apache::loncommon::end_page({'js_ready'  => 1}); 

    my $nothing=&javascript_nothing();
    return (<<ENDCHECK);
<script type="text/javascript"> 
//<!-- BEGIN LON-CAPA Internal
var checkwin;

function spellcheckerwindow(string) {
    var esc_string = string.replace(/\"/g,'&quot;');
    checkwin=window.open($nothing,'spellcheckwin','height=320,width=280,resizable=yes,scrollbars=yes,location=no,menubar=no,toolbar=no');
    checkwin.document.writeln('$start_page<form name="spellcheckform" action="/adm/spellcheck" method="post"><input type="hidden" name="text" value="'+esc_string+'" /><\\/form>$end_page');
    checkwin.document.close();
}
// END LON-CAPA Internal -->
</script>
ENDCHECK
}

# ---------------------------------- Generate link to spell checker for a field

sub spelllink {
    my ($form,$field)=@_;
    my $linktext=&mt('Check Spelling');
    return (<<ENDLINK);
<a href="javascript:if (typeof(document.$form.onsubmit)!='undefined') { if (document.$form.onsubmit!=null) { document.$form.onsubmit();}};spellcheckerwindow(this.document.forms.$form.$field.value);">$linktext</a>
ENDLINK
}

# ------------------------------------------------- Output headers for HTMLArea

{
    my @htmlareafields;
    sub init_htmlareafields {
	undef(@htmlareafields);
    }
    
    sub add_htmlareafields {
	my (@newfields) = @_;
	push(@htmlareafields,@newfields);
    }

    sub get_htmlareafields {
	return @htmlareafields;
    }
}

sub htmlareaheaders {
    return if (&htmlareablocked());
    return if (!&htmlareabrowser());
    return (<<ENDHEADERS);
<script type="text/javascript" src="/fckeditor/fckeditor.js"></script>
ENDHEADERS
}

# ----------------------------------------------------------------- Preferences

sub disablelink {
    my @fields=@_;
    if (defined($#fields)) {
	unless ($#fields>=0) { return ''; }
    }
    return '<a href="'.&HTML::Entities::encode('/adm/preferences?action=set_wysiwyg&wysiwyg=off&returnurl=','<>&"').&escape($ENV{'REQUEST_URI'}).'">'.&mt('Disable WYSIWYG Editor').'</a>';
}

sub enablelink {
    my @fields=@_;
    if (defined($#fields)) {
	unless ($#fields>=0) { return ''; }
    }
    return '<a href="'.&HTML::Entities::encode('/adm/preferences?action=set_wysiwyg&wysiwyg=on&returnurl=','<>&"').&escape($ENV{'REQUEST_URI'}).'">'.&mt('Enable WYSIWYG Editor').'</a>';
}

# ------------------------------------------------- lang to use in html editor
sub htmlarea_lang {
    my $lang='en';
    if (&mt('htmlarea_lang') ne 'htmlarea_lang') {
	$lang=&mt('htmlarea_lang');
    }
    return $lang;
}

# ----------------------------------------- Script to activate only some fields

sub htmlareaselectactive {
    my @fields=@_;
    unless (&htmlareabrowser()) { return ''; }
    if (&htmlareablocked()) { return '<br />'.&enablelink(@fields); }
    my $output='<script type="text/javascript" defer="1">';
    my $lang = &htmlarea_lang();
    foreach my $field (@fields) {
	$output.="
{
    var oFCKeditor = new FCKeditor('$field');
    oFCKeditor.Config['CustomConfigurationsPath'] = 
	'/fckeditor/loncapaconfig.js';    
    oFCKeditor.ReplaceTextarea();
    oFCKeditor.Config['AutoDetectLanguage'] = false;
    oFCKeditor.Config['DefaultLanguage'] = '$lang';
}";
    }
    $output.="\nwindow.status='Activated Editfields';\n</script><br />".
	&disablelink(@fields);
    return $output;
}

# --------------------------------------------------------------------- Blocked

sub htmlareablocked {
    unless ($env{'environment.wysiwygeditor'} eq 'on') { return 1; }
    return 0;
}

# ---------------------------------------- Browser capable of running HTMLArea?

sub htmlareabrowser {
    return 1;
}

############################################################
############################################################

=pod

=item breadcrumbs

Compiles the previously registered breadcrumbs into an series of links.
FAQ and BUG links will be placed on the left side of the table if they
are defined for the last registered breadcrumb.  
Additionally supports a 'component', which will be displayed on the
right side of the table (without a link).
A link to help for the component will be included if one is specified.

All inputs can be undef without problems.

Inputs: $component (the large text on the right side of the table),
        $component_help
        $menulink (boolean, controls whether to include a link to /adm/menu)
        $helplink (if 'nohelp' don't include the orange help link)
        $css_class (optional name for the class to apply to the table for CSS)
Returns a string containing breadcrumbs for the current page.

=item clear_breadcrumbs

Clears the previously stored breadcrumbs.

=item add_breadcrumb

Pushes a breadcrumb on the stack of crumbs.

input: $breadcrumb, a hash reference.  The keys 'href','title', and 'text'
are required.  If present the keys 'faq' and 'bug' will be used to provide
links to the FAQ and bug sites. If the key 'no_mt' is present the 'title' 
and 'text' values won't be sent through &mt()

returns: nothing    

=cut

############################################################
############################################################
{
    my @Crumbs;
    
    sub breadcrumbs {
        my ($component,$component_help,$menulink,$helplink,$css_class) = @_;
        #
	$css_class ||= 'LC_breadcrumbs';
        my $Str = "\n".'<table class="'.$css_class.'"><tr><td>';
        #
        # Make the faq and bug data cascade
        my $faq = '';
        my $bug = '';
	my $help='';
        # The last breadcrumb does not have a link, so handle it separately.
        my $last = pop(@Crumbs);
        #
        # The first one should be the course or a menu link
	if (!defined($menulink)) { $menulink=1; }
        if ($menulink) {
            my $description = 'Menu';
            my $no_mt_descr = 0;
            if (exists($env{'request.course.id'}) && 
                $env{'request.course.id'} ne '') {
                $description = 
                    $env{'course.'.$env{'request.course.id'}.'.description'};
                $no_mt_descr = 1;
            }
            unshift(@Crumbs,{
                    href   =>'/adm/menu',
                    title  =>'Go to main menu',
                    target =>'_top',
                    text   =>$description,
                    no_mt  =>$no_mt_descr,
                });
        }
        my $links .= 
            join('-&gt;',
                 map {
                     $faq = $_->{'faq'} if (exists($_->{'faq'}));
                     $bug = $_->{'bug'} if (exists($_->{'bug'}));
                     $help = $_->{'help'} if (exists($_->{'help'}));
                     my $result = '<a href="'.$_->{'href'}.'" ';
                     if (defined($_->{'target'}) && $_->{'target'} ne '') {
                         $result .= 'target="'.$_->{'target'}.'" ';
                     }
		     if ($_->{'no_mt'}) {
			 $result .='title="'.$_->{'title'}.'">'.
			     $_->{'text'}.'</a>';
		     } else {
			 $result .='title="'.&mt($_->{'title'}).'">'.
			     &mt($_->{'text'}).'</a>';
		     }
                     $result;
                     } @Crumbs
                 );
        $links .= '-&gt;' if ($links ne '');
	if ($last->{'no_mt'}) {
	    $links .= '<b>'.$last->{'text'}.'</b>';
	} else {
	    $links .= '<b>'.&mt($last->{'text'}).'</b>';
	}
        #
        my $icons = '';
        $faq = $last->{'faq'} if (exists($last->{'faq'}));
        $bug = $last->{'bug'} if (exists($last->{'bug'}));
        $help = $last->{'help'} if (exists($last->{'help'}));
        $component_help=($component_help?$component_help:$help);
#        if ($faq ne '') {
#            $icons .= &Apache::loncommon::help_open_faq($faq);
#        }
#        if ($bug ne '') {
#            $icons .= &Apache::loncommon::help_open_bug($bug);
#        }
	if ($faq ne '' || $component_help ne '' || $bug ne '') {
	    $icons .= &Apache::loncommon::help_open_menu($component,
							 $component_help,
							 $faq,$bug);
	}
        #
        $Str .= $links.'</td>';
        #
        if (defined($component)) {
            $Str .= '<td class="'.$css_class.'_component">'.
                &mt($component);
	    if ($icons ne '') {
		$Str .= '&nbsp;'.$icons;
	    }
	    $Str .= '</td>';
        }
        $Str .= '</tr></table>'."\n";
        #
        # Return the @Crumbs stack to what we started with
        push(@Crumbs,$last);
        shift(@Crumbs);
        #
        return $Str;
    }

    sub clear_breadcrumbs {
        undef(@Crumbs);
    }

    sub add_breadcrumb {
        push (@Crumbs,@_);
    }

} # End of scope for @Crumbs

############################################################
############################################################

# Nested table routines.
#
# Routines to display form items in a multi-row table with 2 columns.
# Uses nested tables to divide form elements into segments.
# For examples of use see loncom/interface/lonnotify.pm 
#
# Can be used in following order: ...
# &start_pick_box()
# row1
# row2
# row3   ... etc.
# &submit_row()
# &end_pick_box()
#
# where row1, row 2 etc. are chosen from &role_select_row,&course_select_row,
# &status_select_row and &email_default_row
#
# Can also be used in following order:
#
# &start_pick_box()
# &row_title()
# &row_closure()
# &row_title()
# &row_closure()  ... etc.
# &submit_row()
# &end_pick_box()
#
# In general a &submit_row() call should proceed the call to &end_pick_box(),
# as this routine adds a button for form submission.
# &submit_row() does not require a &row_closure after it.
#  
# &start_pick_box() creates a bounding table with 1-pixel wide black border.
# rows should be placed between calls to &start_pick_box() and &end_pick_box.
#
# &row_title() adds a title in the left column for each segment.
# &row_closure() closes a row with a 1-pixel wide black line.
#
# &role_select_row() provides a select box from which to choose 1 or more roles 
# &course_select_row provides ways of picking groups of courses
#    radio buttons: all, by category or by picking from a course picker pop-up
#      note: by category option is only displayed if a domain has implemented 
#                selection by year, semester, department, number etc.
#
# &status_select_row() provides a select box from which to choose 1 or more
#  access types (current access, prior access, and future access)  
#
# &email_default_row() provides text boxes for default e-mail suffixes for
#  different authentication types in a domain.
#
# &row_title() and &row_closure() are called internally by the &*_select_row
# routines, but can also be called directly to start and end rows which have 
# needs that are not accommodated by the *_select_row() routines.    

sub start_pick_box {
    my ($css_class) = @_;
    if (defined($css_class)) {
	$css_class = 'class="'.$css_class.'"';
    } else {
	$css_class= 'class="LC_pick_box"';
    }
    my $output = <<"END";
 <table $css_class>
END
    return $output;
}

sub end_pick_box {
    my $output = <<"END";
       </table>
END
    return $output;
}

sub row_title {
    my ($title,$css_title_class,$css_value_class) = @_;
    $css_title_class ||= 'LC_pick_box_title';
    $css_title_class = 'class="'.$css_title_class.'"';

    $css_value_class ||= 'LC_pick_box_value';
    $css_value_class = 'class="'.$css_value_class.'"';

    if ($title ne '') {
        $title .= ':';
    }
    my $output = <<"ENDONE";
           <tr class="LC_pick_box_row">
            <td $css_title_class>
	       $title
            </td>
            <td $css_value_class>
ENDONE
    return $output;
}

sub row_closure {
    my ($no_separator) =@_;
    my $output = <<"ENDTWO";
            </td>
           </tr>
ENDTWO
    if (!$no_separator) {
        $output .= <<"ENDTWO";
           <tr>
            <td colspan="2" class="LC_pick_box_separator">
            </td>
           </tr>
ENDTWO
    }
    return $output;
}

sub role_select_row {
    my ($roles,$title,$css_class,$show_separate_custom,$cdom,$cnum) = @_;
    my $output;
    if (defined($title)) {
        $output = &row_title($title,$css_class);
    }
    $output .= qq|
                                  <select name="roles" multiple >\n|;
    foreach my $role (@$roles) {
        my $plrole;
        if ($role eq 'ow') {
            $plrole = &mt('Course Owner');
        } elsif ($role eq 'cr') {
            if ($show_separate_custom) {
                if ($cdom ne '' && $cnum ne '') {
                    my %course_customroles = &course_custom_roles($cdom,$cnum);
                    foreach my $crrole (sort(keys(%course_customroles))) {
                        my ($plcrrole) = ($crrole =~ m|^cr/[^/]+/[^/]+/(.+)$|);
                        $output .= '  <option value="'.$crrole.'">'.$plcrrole.
                                   '</option>';
                    }
                }
            } else {
                $plrole = &mt('Custom Role');
            }
        } else {
            $plrole=&Apache::lonnet::plaintext($role);
        }
        if (($role ne 'cr') || (!$show_separate_custom)) {
            $output .= '  <option value="'.$role.'">'.$plrole.'</option>';
        }
    }
    $output .= qq|                </select>\n|;
    if (defined($title)) {
        $output .= &row_closure();
    }
    return $output;
}

sub course_select_row {
    my ($title,$formname,$totcodes,$codetitles,$idlist,$idlist_titles,
	$css_class) = @_;
    my $output = &row_title($title,$css_class);
    $output .= &course_selection($formname,$totcodes,$codetitles,$idlist,$idlist_titles);
    $output .= &row_closure();
    return $output;
}

sub course_selection {
    my ($formname,$totcodes,$codetitles,$idlist,$idlist_titles) = @_;
    my $output = qq|
<script type="text/javascript">
    function coursePick (formname) {
        for  (var i=0; i<formname.coursepick.length; i++) {
            if (formname.coursepick[i].value == 'category') {
                courseSet('');
            }
            if (!formname.coursepick[i].checked) {
                if (formname.coursepick[i].value == 'specific') {
                    formname.coursetotal.value = 0;
                    formname.courselist = '';
                }
            }
        }
    }
    function setPick (formname) {
        for  (var i=0; i<formname.coursepick.length; i++) {
            if (formname.coursepick[i].value == 'category') {
                formname.coursepick[i].checked = true;
            }
            formname.coursetotal.value = 0;
            formname.courselist = '';
        }
    }
</script>
    |;
    my $courseform='<b>'.&Apache::loncommon::selectcourse_link
                     ($formname,'pickcourse','pickdomain','coursedesc','',1).'</b>';
        $output .= '<input type="radio" name="coursepick" value="all" onclick="coursePick(this.form)" />'.&mt('All courses').'<br />';
    if ($totcodes > 0) {
        my $numtitles = @$codetitles;
        if ($numtitles > 0) {
            $output .= '<input type="radio" name="coursepick" value="category" onclick="coursePick(this.form);alert('."'".&mt('Choose categories, from left to right')."'".')" />'.&mt('Pick courses by category:').' <br />';
            $output .= '<table><tr><td>'.$$codetitles[0].'<br />'."\n".
               '<select name="'.$$codetitles[0].
               '" onChange="setPick(this.form);courseSet('."'$$codetitles[0]'".')">'."\n".
               ' <option value="-1" />Select'."\n";
            my @items = ();
            my @longitems = ();
            if ($$idlist{$$codetitles[0]} =~ /","/) {
                @items = split(/","/,$$idlist{$$codetitles[0]});
            } else {
                $items[0] = $$idlist{$$codetitles[0]};
            }
            if (defined($$idlist_titles{$$codetitles[0]})) {
                if ($$idlist_titles{$$codetitles[0]} =~ /","/) {
                    @longitems = split(/","/,$$idlist_titles{$$codetitles[0]});
                } else {
                    $longitems[0] = $$idlist_titles{$$codetitles[0]};
                }
                for (my $i=0; $i<@longitems; $i++) {
                    if ($longitems[$i] eq '') {
                        $longitems[$i] = $items[$i];
                    }
                }
            } else {
                @longitems = @items;
            }
            for (my $i=0; $i<@items; $i++) {
                $output .= ' <option value="'.$items[$i].'">'.$longitems[$i].'</option>';
            }
            $output .= '</select></td>';
            for (my $i=1; $i<$numtitles; $i++) {
                $output .= '<td>'.$$codetitles[$i].'<br />'."\n".
                          '<select name="'.$$codetitles[$i].
                          '" onChange="courseSet('."'$$codetitles[$i]'".')">'."\n".
                          '<option value="-1">&lt;-Pick '.$$codetitles[$i-1].'</option>'."\n".
                          '</select>'."\n".
                          '</td>';
            }
            $output .= '</tr></table><br />';
        }
    }
    $output .= '<input type="radio" name="coursepick" value="specific" onclick="coursePick(this.form);opencrsbrowser('."'".$formname."','dccourse','dcdomain','coursedesc','','1'".')" />'.&mt('Pick specific course(s):').' '.$courseform.'&nbsp;&nbsp;<input type="text" value="0" size="4" name="coursetotal" /><input type="hidden" name="courselist" value="" />selected.<br />'."\n";
    return $output;
}

sub status_select_row {
    my ($types,$title,$css_class) = @_;
    my $output; 
    if (defined($title)) {
        $output = &row_title($title,$css_class,'LC_pick_box_select');
    }
    $output .= qq|
                                    <select name="types" multiple>\n|;
    foreach my $status_type (sort(keys(%{$types}))) {
        $output .= '  <option value="'.$status_type.'">'.$$types{$status_type}.'</option>';
    }
    $output .= qq|                   </select>\n|; 
    if (defined($title)) {
        $output .= &row_closure();
    }
    return $output;
}

sub email_default_row {
    my ($authtypes,$title,$descrip,$css_class) = @_;
    my $output = &row_title($title,$css_class);
    $output .= $descrip.
	&Apache::loncommon::start_data_table().
	&Apache::loncommon::start_data_table_header_row().
	'<th>'.&mt('Authentication Method').'</th>'.
	'<th align="right">'.&mt('Username -> e-mail conversion').'</th>'."\n".
	&Apache::loncommon::end_data_table_header_row();
    my $rownum = 0;
    foreach my $auth (sort(keys(%{$authtypes}))) {
        my ($userentry,$size);
        if ($auth =~ /^krb/) {
            $userentry = '';
            $size = 25;
        } else {
            $userentry = 'username@';
            $size = 15;
        }
        $output .= &Apache::loncommon::start_data_table_row().
	    '<td>  '.$$authtypes{$auth}.'</td>'.
	    '<td align="right">'.$userentry.
	    '<input type="text" name="'.$auth.'" size="'.$size.'" /></td>'.
	    &Apache::loncommon::end_data_table_row();
    }
    $output .= &Apache::loncommon::end_data_table();
    $output .= &row_closure();
    return $output;
}


sub submit_row {
    my ($title,$cmd,$submit_text,$css_class) = @_;
    my $output = &row_title($title,$css_class,'LC_pick_box_submit');
    $output .= qq|
             <br />
             <input type="hidden" name="command" value="$cmd" />
             <input type="submit" value="$submit_text"/> &nbsp;
             <br /><br />
            \n|;
    return $output;
}

sub course_custom_roles {
    my ($cdom,$cnum) = @_;
    my %returnhash=();
    my %coursepersonnel=&Apache::lonnet::dump('nohist_userroles',$cdom,$cnum);
    foreach my $person (sort(keys(%coursepersonnel))) {
        my ($role) = ($person =~ /^([^:]+):/);
        my ($end,$start) = split(/:/,$coursepersonnel{$person});
        if ($end == -1 && $start == -1) {
            next;
        }
        if ($role =~ m|^cr/[^/]+/[^/]+/[^/]|) {
            $returnhash{$role} ++;
        }
    }
    return %returnhash;
}


##############################################
##############################################
                                                                             
# echo_form_input
#
# Generates html markup to add form elements from the referrer page
# as hidden form elements (values encoded) in the new page.
#
# Intended to support two types of use 
# (a) to allow backing up to earlier pages in a multi-page 
# form submission process using a breadcrumb trail.
#
# (b) to allow the current page to be reloaded with form elements
# set on previous page to remain unchanged.  An example would
# be where the a page containing a dynamically-built table of data is 
# is to be redisplayed, with only the sort order of the data changed. 
#  
# Inputs:
# 1. Reference to array of form elements in the submitted form on 
# the referrer page which are to be excluded from the echoed elements.
#
# 2. Reference to array of regular expressions, which if matched in the  
# name of the form element n the referrer page will be omitted from echo. 
#
# Outputs: A scalar containing the html markup for the echoed form
# elements (all as hidden elements, with values encoded). 


sub echo_form_input {
    my ($excluded,$regexps) = @_;
    my $output = '';
    foreach my $key (keys(%env)) {
        if ($key =~ /^form\.(.+)$/) {
            my $name = $1;
            my $match = 0;
            if ((!@{$excluded}) || (!grep/^$name$/,@{$excluded})) {
                if (defined($regexps)) {
                    if (@{$regexps} > 0) {
                        foreach my $regexp (@{$regexps}) {
                            if ($name =~ /\Q$regexp\E/) {
                                $match = 1;
                                last;
                            }
                        }
                    }
                }
                if (!$match) {
                    if (ref($env{$key})) {
                        foreach my $value (@{$env{$key}}) {
                            $value = &HTML::Entities::encode($value,'<>&"');
                            $output .= '<input type="hidden" name="'.$name.
                                             '" value="'.$value.'" />'."\n";
                        }
                    } else {
                        my $value = &HTML::Entities::encode($env{$key},'<>&"');
                        $output .= '<input type="hidden" name="'.$name.
                                             '" value="'.$value.'" />'."\n";
                    }
                }
            }
        }
    }
    return $output;
}

##############################################
##############################################
                                                                             
# set_form_elements
#
# Generates javascript to set form elements to values based on
# corresponding values for the same form elements when the page was
# previously submitted.
#     
# Last submission values are read from hidden form elements in referring 
# page which have the same name, i.e., generated by &echo_form_input(). 
#
# Intended to be called by onload event.
#
# Inputs:
# (a) Reference to hash of echoed form elements to be set.
#
# In the hash, keys are the form element names, and the values are the
# element type (selectbox, radio, checkbox or text -for textbox, textarea or
# hidden).
#
# (b) Optional reference to hash of stored elements to be set.
#
# If the page being displayed is a page which permits modification of
# previously stored data, e.g., the first page in a multi-page submission,
# then if stored is supplied, form elements will be set to the last stored
# values.  If user supplied values are also available for the same elements
# these will replace the stored values. 
#        
# Output:
#  
# javascript function - set_form_elements() which sets form elements,
# expects an argument: formname - the name of the form according to 
# the DOM, e.g., document.compose

sub set_form_elements {
    my ($elements,$stored) = @_;
    my %values;
    my $output .= 'function setFormElements(courseForm) {
';
    if (defined($stored)) {
        foreach my $name (keys(%{$stored})) {
            if (exists($$elements{$name})) {
                if (ref($$stored{$name}) eq 'ARRAY') {
                    $values{$name} = $$stored{$name};
                } else {
                    @{$values{$name}} = ($$stored{$name});
                }
            }
        }
    }

    foreach my $key (keys(%env)) {
        if ($key =~ /^form\.(.+)$/) {
            my $name = $1;
            if (exists($$elements{$name})) {
                @{$values{$name}} = &Apache::loncommon::get_env_multiple($key);
            }
        }
    }

    foreach my $name (keys(%values)) {
        for (my $i=0; $i<@{$values{$name}}; $i++) {
            $values{$name}[$i] = &HTML::Entities::decode($values{$name}[$i],'<>&"');
            $values{$name}[$i] =~ s/([\r\n\f]+)/\\n/g;
            $values{$name}[$i] =~ s/"/\\"/g;
        }
        if ($$elements{$name} eq 'text') {
            my $numvalues = @{$values{$name}};
            if ($numvalues > 1) {
                my $valuestring = join('","',@{$values{$name}});
                $output .= qq|
  var textvalues = new Array ("$valuestring");
  var total = courseForm.elements['$name'].length;
  if (total > $numvalues) {
      total = $numvalues;
  }    
  for (var i=0; i<total; i++) {
      courseForm.elements['$name']\[i].value = textvalues[i];
  }
|;
            } else {
                $output .= qq|
  courseForm.elements['$name'].value = "$values{$name}[0]";
|;
            }
        } else {
            $output .=  qq|
  var elementLength = courseForm.elements['$name'].length;
  if (elementLength==undefined) {
|;
            foreach my $value (@{$values{$name}}) {
                if ($$elements{$name} eq 'selectbox') {
                    $output .=  qq|
      if (courseForm.elements['$name'].options[0].value == "$value") {
          courseForm.elements['$name'].options[0].selected = true;
      }|;
                } elsif (($$elements{$name} eq 'radio') ||
                         ($$elements{$name} eq 'checkbox')) {
                    $output .= qq|
      if (courseForm.elements['$name'].value == "$value") {
          courseForm.elements['$name'].checked = true;
      }|;
                }
            }
            $output .= qq|
  }
  else {
      for (var i=0; i<courseForm.elements['$name'].length; i++) {
|;
            if ($$elements{$name} eq 'selectbox') {
                $output .=  qq|
          courseForm.elements['$name'].options[i].selected = false;|;
            } elsif (($$elements{$name} eq 'radio') || 
                     ($$elements{$name} eq 'checkbox')) {
                $output .= qq|
          courseForm.elements['$name']\[i].checked = false;|; 
            }
            $output .= qq|
      }
      for (var j=0; j<courseForm.elements['$name'].length; j++) {
|;
            foreach my $value (@{$values{$name}}) {
                if ($$elements{$name} eq 'selectbox') {
                    $output .=  qq|
          if (courseForm.elements['$name'].options[j].value == "$value") {
              courseForm.elements['$name'].options[j].selected = true;
          }|;
                } elsif (($$elements{$name} eq 'radio') ||
                         ($$elements{$name} eq 'checkbox')) { 
                      $output .= qq|
          if (courseForm.elements['$name']\[j].value == "$value") {
              courseForm.elements['$name']\[j].checked = true;
          }|;
                }
            }
            $output .= qq|
      }
  }
|;
        }
    }
    $output .= "
}\n";
    return $output;
}

##############################################
##############################################

# javascript_valid_email
#
# Generates javascript to validate an e-mail address.
# Returns a javascript function which accetps a form field as argumnent, and
# returns false if field.value does not satisfy two regular expression matches
# for a valid e-mail address.  Backwards compatible with old browsers without
# support for javascript RegExp (just checks for @ in field.value in this case). 

sub javascript_valid_email {
    my $scripttag .= <<'END';
function validmail(field) {
    var str = field.value;
    if (window.RegExp) {
        var reg1str = "(@.*@)|(\\.\\.)|(@\\.)|(\\.@)|(^\\.)";
        var reg2str = "^.+\\@(\\[?)[a-zA-Z0-9\\-\\.]+\\.([a-zA-Z]{2,3}|[0-9]{1,3})(\\]?)$"; //"
        var reg1 = new RegExp(reg1str);
        var reg2 = new RegExp(reg2str);
        if (!reg1.test(str) && reg2.test(str)) {
            return true;
        }
        return false;
    }
    else
    {
        if(str.indexOf("@") >= 0) {
            return true;
        }
        return false;
    }
}
END
    return $scripttag;
}

1;

__END__
