# The LearningOnline Network with CAPA
# a pile of common routines
#
# $Id: loncommon.pm,v 1.672 2008/07/23 10:07:25 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

# Makes a table out of the previous attempts
# Inputs result_from_symbread, user, domain, course_id
# Reads in non-network-related .tab files

# POD header:

=pod

=head1 NAME

Apache::loncommon - pile of common routines

=head1 SYNOPSIS

Common routines for manipulating connections, student answers,
    domains, common Javascript fragments, etc.

=head1 OVERVIEW

A collection of commonly used subroutines that don't have a natural
home anywhere else. This collection helps remove
redundancy from other modules and increase efficiency of memory usage.

=cut 

# End of POD header
package Apache::loncommon;

use strict;
use Apache::lonnet;
use GDBM_File;
use POSIX qw(strftime mktime);
use Apache::lonmenu();
use Apache::lonenc();
use Apache::lonlocal;
use HTML::Entities;
use Apache::lonhtmlcommon();
use Apache::loncoursedata();
use Apache::lontexconvert();
use Apache::lonclonecourse();
use LONCAPA qw(:DEFAULT :match);
use DateTime::TimeZone;

# ---------------------------------------------- Designs
use vars qw(%defaultdesign);

my $readit;


##
## Global Variables
##


# ----------------------------------------------- SSI with retries:
#

=pod

=head1 Server Side include with retries:

=over 4

=item * &ssi_with_retries(resource,retries form)

Performs an ssi with some number of retries.  Retries continue either
until the result is ok or until the retry count supplied by the
caller is exhausted.  

Inputs:

=over 4

resource   - Identifies the resource to insert.

retries    - Count of the number of retries allowed.

form       - Hash that identifies the rendering options.

=back

Returns:

=over 4

content    - The content of the response.  If retries were exhausted this is empty.

response   - The response from the last attempt (which may or may not have been successful.

=back

=back

=cut

sub ssi_with_retries {
    my ($resource, $retries, %form) = @_;


    my $ok = 0;			# True if we got a good response.
    my $content;
    my $response;

    # Try to get the ssi done. within the retries count:

    do {
	($content, $response) = &Apache::lonnet::ssi($resource, %form);
	$ok      = $response->is_success;
        if (!$ok) {
            &Apache::lonnet::logthis("Failed ssi_with_retries on $resource: ".$response->is_success.', '.$response->code.', '.$response->message);
        }
	$retries--;
    } while (!$ok && ($retries > 0));

    if (!$ok) {
	$content = '';		# On error return an empty content.
    }
    return ($content, $response);

}



# ----------------------------------------------- Filetypes/Languages/Copyright
my %language;
my %supported_language;
my %cprtag;
my %scprtag;
my %fe; my %fd; my %fm;
my %category_extensions;

# ---------------------------------------------- Thesaurus variables
#
# %Keywords:
#      A hash used by &keyword to determine if a word is considered a keyword.
# $thesaurus_db_file 
#      Scalar containing the full path to the thesaurus database.

my %Keywords;
my $thesaurus_db_file;

#
# Initialize values from language.tab, copyright.tab, filetypes.tab,
# thesaurus.tab, and filecategories.tab.
#
BEGIN {
    # Variable initialization
    $thesaurus_db_file = $Apache::lonnet::perlvar{'lonTabDir'}."/thesaurus.db";
    #
    unless ($readit) {
# ------------------------------------------------------------------- languages
    {
        my $langtabfile = $Apache::lonnet::perlvar{'lonTabDir'}.
                                   '/language.tab';
        if ( open(my $fh,"<$langtabfile") ) {
            while (my $line = <$fh>) {
                next if ($line=~/^\#/);
                chomp($line);
                my ($key,$two,$country,$three,$enc,$val,$sup)=(split(/\t/,$line));
                $language{$key}=$val.' - '.$enc;
                if ($sup) {
                    $supported_language{$key}=$sup;
                }
            }
            close($fh);
        }
    }
# ------------------------------------------------------------------ copyrights
    {
        my $copyrightfile = $Apache::lonnet::perlvar{'lonIncludes'}.
                                  '/copyright.tab';
        if ( open (my $fh,"<$copyrightfile") ) {
            while (my $line = <$fh>) {
                next if ($line=~/^\#/);
                chomp($line);
                my ($key,$val)=(split(/\s+/,$line,2));
                $cprtag{$key}=$val;
            }
            close($fh);
        }
    }
# ----------------------------------------------------------- source copyrights
    {
        my $sourcecopyrightfile = $Apache::lonnet::perlvar{'lonIncludes'}.
                                  '/source_copyright.tab';
        if ( open (my $fh,"<$sourcecopyrightfile") ) {
            while (my $line = <$fh>) {
                next if ($line =~ /^\#/);
                chomp($line);
                my ($key,$val)=(split(/\s+/,$line,2));
                $scprtag{$key}=$val;
            }
            close($fh);
        }
    }

# -------------------------------------------------------------- default domain designs
    my $designdir=$Apache::lonnet::perlvar{'lonTabDir'}.'/lonDomColors';
    my $designfile = $designdir.'/default.tab';
    if ( open (my $fh,"<$designfile") ) {
        while (my $line = <$fh>) {
            next if ($line =~ /^\#/);
            chomp($line);
            my ($key,$val)=(split(/\=/,$line));
            if ($val) { $defaultdesign{$key}=$val; }
        }
        close($fh);
    }

# ------------------------------------------------------------- file categories
    {
        my $categoryfile = $Apache::lonnet::perlvar{'lonTabDir'}.
                                  '/filecategories.tab';
        if ( open (my $fh,"<$categoryfile") ) {
	    while (my $line = <$fh>) {
		next if ($line =~ /^\#/);
		chomp($line);
                my ($extension,$category)=(split(/\s+/,$line,2));
                push @{$category_extensions{lc($category)}},$extension;
            }
            close($fh);
        }

    }
# ------------------------------------------------------------------ file types
    {
        my $typesfile = $Apache::lonnet::perlvar{'lonTabDir'}.
               '/filetypes.tab';
        if ( open (my $fh,"<$typesfile") ) {
            while (my $line = <$fh>) {
		next if ($line =~ /^\#/);
		chomp($line);
                my ($ending,$emb,$mime,$descr)=split(/\s+/,$line,4);
                if ($descr ne '') {
                    $fe{$ending}=lc($emb);
                    $fd{$ending}=$descr;
                    if ($mime ne 'unk') { $fm{$ending}=$mime; }
                }
            }
            close($fh);
        }
    }
    &Apache::lonnet::logthis(
              "<font color=yellow>INFO: Read file types</font>");
    $readit=1;
    }  # end of unless($readit) 
    
}

###############################################################
##           HTML and Javascript Helper Functions            ##
###############################################################

=pod 

=head1 HTML and Javascript Functions

=over 4

=item * &browser_and_searcher_javascript()

X<browsing, javascript>X<searching, javascript>Returns a string
containing javascript with two functions, C<openbrowser> and
C<opensearcher>. Returned string does not contain E<lt>scriptE<gt>
tags.

=item * &openbrowser(formname,elementname,only,omit) [javascript]

inputs: formname, elementname, only, omit

formname and elementname indicate the name of the html form and name of
the element that the results of the browsing selection are to be placed in. 

Specifying 'only' will restrict the browser to displaying only files
with the given extension.  Can be a comma separated list.

Specifying 'omit' will restrict the browser to NOT displaying files
with the given extension.  Can be a comma separated list.

=item * &opensearcher(formname,elementname) [javascript]

Inputs: formname, elementname

formname and elementname specify the name of the html form and the name
of the element the selection from the search results will be placed in.

=cut

sub browser_and_searcher_javascript {
    my ($mode)=@_;
    if (!defined($mode)) { $mode='edit'; }
    my $resurl=&escape_single(&lastresurl());
    return <<END;
// <!-- BEGIN LON-CAPA Internal
    var editbrowser = null;
    function openbrowser(formname,elementname,only,omit,titleelement) {
        var url = '$resurl/?';
        if (editbrowser == null) {
            url += 'launch=1&';
        }
        url += 'catalogmode=interactive&';
        url += 'mode=$mode&';
        url += 'inhibitmenu=yes&';
        url += 'form=' + formname + '&';
        if (only != null) {
            url += 'only=' + only + '&';
        } else {
            url += 'only=&';
	}
        if (omit != null) {
            url += 'omit=' + omit + '&';
        } else {
            url += 'omit=&';
	}
        if (titleelement != null) {
            url += 'titleelement=' + titleelement + '&';
        } else {
	    url += 'titleelement=&';
	}
        url += 'element=' + elementname + '';
        var title = 'Browser';
        var options = 'scrollbars=1,resizable=1,menubar=0,toolbar=1,location=1';
        options += ',width=700,height=600';
        editbrowser = open(url,title,options,'1');
        editbrowser.focus();
    }
    var editsearcher;
    function opensearcher(formname,elementname,titleelement) {
        var url = '/adm/searchcat?';
        if (editsearcher == null) {
            url += 'launch=1&';
        }
        url += 'catalogmode=interactive&';
        url += 'mode=$mode&';
        url += 'form=' + formname + '&';
        if (titleelement != null) {
            url += 'titleelement=' + titleelement + '&';
        } else {
	    url += 'titleelement=&';
	}
        url += 'element=' + elementname + '';
        var title = 'Search';
        var options = 'scrollbars=1,resizable=1,menubar=0,toolbar=1,location=1';
        options += ',width=700,height=600';
        editsearcher = open(url,title,options,'1');
        editsearcher.focus();
    }
// END LON-CAPA Internal -->
END
}

sub lastresurl {
    if ($env{'environment.lastresurl'}) {
	return $env{'environment.lastresurl'}
    } else {
	return '/res';
    }
}

sub storeresurl {
    my $resurl=&Apache::lonnet::clutter(shift);
    unless ($resurl=~/^\/res/) { return 0; }
    $resurl=~s/\/$//;
    &Apache::lonnet::put('environment',{'lastresurl' => $resurl});
    &Apache::lonnet::appenv({'environment.lastresurl' => $resurl});
    return 1;
}

sub studentbrowser_javascript {
   unless (
            (($env{'request.course.id'}) && 
             (&Apache::lonnet::allowed('srm',$env{'request.course.id'})
	      || &Apache::lonnet::allowed('srm',$env{'request.course.id'}.
					  '/'.$env{'request.course.sec'})
	      ))
         || ($env{'request.role'}=~/^(au|dc|su)/)
          ) { return ''; }  
   return (<<'ENDSTDBRW');
<script type="text/javascript" language="Javascript" >
    var stdeditbrowser;
    function openstdbrowser(formname,uname,udom,roleflag,ignorefilter) {
        var url = '/adm/pickstudent?';
        var filter;
	if (!ignorefilter) {
	    eval('filter=document.'+formname+'.'+uname+'.value;');
	}
        if (filter != null) {
           if (filter != '') {
               url += 'filter='+filter+'&';
	   }
        }
        url += 'form=' + formname + '&unameelement='+uname+
                                    '&udomelement='+udom;
	if (roleflag) { url+="&roles=1"; }
        var title = 'Student_Browser';
        var options = 'scrollbars=1,resizable=1,menubar=0';
        options += ',width=700,height=600';
        stdeditbrowser = open(url,title,options,'1');
        stdeditbrowser.focus();
    }
</script>
ENDSTDBRW
}

sub selectstudent_link {
   my ($form,$unameele,$udomele)=@_;
   if ($env{'request.course.id'}) {  
       if (!&Apache::lonnet::allowed('srm',$env{'request.course.id'})
	   && !&Apache::lonnet::allowed('srm',$env{'request.course.id'}.
					'/'.$env{'request.course.sec'})) {
	   return '';
       }
       return "<a href='".'javascript:openstdbrowser("'.$form.'","'.$unameele.
        '","'.$udomele.'");'."'>".&mt('Select User')."</a>";
   }
   if ($env{'request.role'}=~/^(au|dc|su)/) {
       return "<a href='".'javascript:openstdbrowser("'.$form.'","'.$unameele.
        '","'.$udomele.'",1);'."'>".&mt('Select User')."</a>";
   }
   return '';
}

sub authorbrowser_javascript {
    return <<"ENDAUTHORBRW";
<script type="text/javascript">
var stdeditbrowser;

function openauthorbrowser(formname,udom) {
    var url = '/adm/pickauthor?';
    url += 'form='+formname+'&roledom='+udom;
    var title = 'Author_Browser';
    var options = 'scrollbars=1,resizable=1,menubar=0';
    options += ',width=700,height=600';
    stdeditbrowser = open(url,title,options,'1');
    stdeditbrowser.focus();
}

</script>
ENDAUTHORBRW
}

sub coursebrowser_javascript {
    my ($domainfilter,$sec_element,$formname)=@_;
    my $crs_or_grp_alert = &mt('Please select the type of LON-CAPA entity - Course or Group - for which you wish to add/modify a user role');
   my $output = '
<script type="text/javascript">
    var stdeditbrowser;'."\n";
   $output .= <<"ENDSTDBRW";
    function opencrsbrowser(formname,uname,udom,desc,extra_element,multflag,crstype) {
        var url = '/adm/pickcourse?';
        var domainfilter = '';
        var formid = getFormIdByName(formname);
        if (formid > -1) {
            var domid = getIndexByName(formid,udom);
            if (domid > -1) {
                if (document.forms[formid].elements[domid].type == 'select-one') {
                    domainfilter=document.forms[formid].elements[domid].options[document.forms[formid].elements[domid].selectedIndex].value;
                }
                if (document.forms[formid].elements[domid].type == 'hidden') {
                    domainfilter=document.forms[formid].elements[domid].value;
                }
            }
        }
        if (domainfilter != null) {
           if (domainfilter != '') {
               url += 'domainfilter='+domainfilter+'&';
	   }
        }
        url += 'form=' + formname + '&cnumelement='+uname+
	                            '&cdomelement='+udom+
                                    '&cnameelement='+desc;
        if (extra_element !=null && extra_element != '') {
            if (formname == 'rolechoice' || formname == 'studentform') {
                url += '&roleelement='+extra_element;
                if (domainfilter == null || domainfilter == '') {
                    url += '&domainfilter='+extra_element;
                }
            }
            else {
                if (formname == 'portform') {
                    url += '&setroles='+extra_element;
                }
            }     
        }
        if (multflag !=null && multflag != '') {
            url += '&multiple='+multflag;
        }
        if (crstype == 'Course/Group') {
            if (formname == 'cu') {
                crstype = document.cu.crstype.options[document.cu.crstype.selectedIndex].value; 
                if (crstype == "") {
                    alert("$crs_or_grp_alert");
                    return;
                }
            }
        }
        if (crstype !=null && crstype != '') {
            url += '&type='+crstype;
        }
        var title = 'Course_Browser';
        var options = 'scrollbars=1,resizable=1,menubar=0';
        options += ',width=700,height=600';
        stdeditbrowser = open(url,title,options,'1');
        stdeditbrowser.focus();
    }

    function getFormIdByName(formname) {
        for (var i=0;i<document.forms.length;i++) {
            if (document.forms[i].name == formname) {
                return i;
            }
        }
        return -1; 
    }

    function getIndexByName(formid,item) {
        for (var i=0;i<document.forms[formid].elements.length;i++) {
            if (document.forms[formid].elements[i].name == item) {
                return i;
            }
        }
        return -1;
    }
ENDSTDBRW
    if ($sec_element ne '') {
        $output .= &setsec_javascript($sec_element,$formname);
    }
    $output .= '
</script>';
    return $output;
}

sub setsec_javascript {
    my ($sec_element,$formname) = @_;
    my $setsections = qq|
function setSect(sectionlist) {
    var sectionsArray = new Array();
    if ((sectionlist != '') && (typeof sectionlist != "undefined")) {
        sectionsArray = sectionlist.split(",");
    }
    var numSections = sectionsArray.length;
    document.$formname.$sec_element.length = 0;
    if (numSections == 0) {
        document.$formname.$sec_element.multiple=false;
        document.$formname.$sec_element.size=1;
        document.$formname.$sec_element.options[0] = new Option('No existing sections','',false,false)
    } else {
        if (numSections == 1) {
            document.$formname.$sec_element.multiple=false;
            document.$formname.$sec_element.size=1;
            document.$formname.$sec_element.options[0] = new Option('Select','',true,true);
            document.$formname.$sec_element.options[1] = new Option('No section','',false,false)
            document.$formname.$sec_element.options[2] = new Option(sectionsArray[0],sectionsArray[0],false,false);
        } else {
            for (var i=0; i<numSections; i++) {
                document.$formname.$sec_element.options[i] = new Option(sectionsArray[i],sectionsArray[i],false,false)
            }
            document.$formname.$sec_element.multiple=true
            if (numSections < 3) {
                document.$formname.$sec_element.size=numSections;
            } else {
                document.$formname.$sec_element.size=3;
            }
            document.$formname.$sec_element.options[0].selected = false
        }
    }
}
|;
    return $setsections;
}


sub selectcourse_link {
   my ($form,$unameele,$udomele,$desc,$extra_element,$multflag,$selecttype)=@_;
   return "<a href='".'javascript:opencrsbrowser("'.$form.'","'.$unameele.
        '","'.$udomele.'","'.$desc.'","'.$extra_element.'","'.$multflag.'","'.$selecttype.'");'."'>".&mt('Select Course')."</a>";
}

sub selectauthor_link {
   my ($form,$udom)=@_;
   return '<a href="javascript:openauthorbrowser('."'$form','$udom'".');">'.
          &mt('Select Author').'</a>';
}

sub check_uncheck_jscript {
    my $jscript = <<"ENDSCRT";
function checkAll(field) {
    if (field.length > 0) {
        for (i = 0; i < field.length; i++) {
            field[i].checked = true ;
        }
    } else {
        field.checked = true
    }
}
 
function uncheckAll(field) {
    if (field.length > 0) {
        for (i = 0; i < field.length; i++) {
            field[i].checked = false ;
        }
    } else {
        field.checked = false ;
    }
}
ENDSCRT
    return $jscript;
}

sub select_timezone {
   my ($name,$selected,$onchange,$includeempty)=@_;
   my $output='<select name="'.$name.'" '.$onchange.'>'."\n";
   if ($includeempty) {
       $output .= '<option value=""';
       if (($selected eq '') || ($selected eq 'local')) {
           $output .= ' selected="selected" ';
       }
       $output .= '> </option>';
   }
   my @timezones = DateTime::TimeZone->all_names;
   foreach my $tzone (@timezones) {
       $output.= '<option value="'.$tzone.'"';
       if ($tzone eq $selected) {
           $output.=' selected="selected"';
       }
       $output.=">$tzone</option>\n";
   }
   $output.="</select>";
   return $output;
}

=pod

=item * &linked_select_forms(...)

linked_select_forms returns a string containing a <script></script> block
and html for two <select> menus.  The select menus will be linked in that
changing the value of the first menu will result in new values being placed
in the second menu.  The values in the select menu will appear in alphabetical
order unless a defined order is provided.

linked_select_forms takes the following ordered inputs:

=over 4

=item * $formname, the name of the <form> tag

=item * $middletext, the text which appears between the <select> tags

=item * $firstdefault, the default value for the first menu

=item * $firstselectname, the name of the first <select> tag

=item * $secondselectname, the name of the second <select> tag

=item * $hashref, a reference to a hash containing the data for the menus.

=item * $menuorder, the order of values in the first menu

=back 

Below is an example of such a hash.  Only the 'text', 'default', and 
'select2' keys must appear as stated.  keys(%menu) are the possible 
values for the first select menu.  The text that coincides with the 
first menu value is given in $menu{$choice1}->{'text'}.  The values 
and text for the second menu are given in the hash pointed to by 
$menu{$choice1}->{'select2'}.  

 my %menu = ( A1 => { text =>"Choice A1" ,
                       default => "B3",
                       select2 => { 
                           B1 => "Choice B1",
                           B2 => "Choice B2",
                           B3 => "Choice B3",
                           B4 => "Choice B4"
                           },
                       order => ['B4','B3','B1','B2'],
                   },
               A2 => { text =>"Choice A2" ,
                       default => "C2",
                       select2 => { 
                           C1 => "Choice C1",
                           C2 => "Choice C2",
                           C3 => "Choice C3"
                           },
                       order => ['C2','C1','C3'],
                   },
               A3 => { text =>"Choice A3" ,
                       default => "D6",
                       select2 => { 
                           D1 => "Choice D1",
                           D2 => "Choice D2",
                           D3 => "Choice D3",
                           D4 => "Choice D4",
                           D5 => "Choice D5",
                           D6 => "Choice D6",
                           D7 => "Choice D7"
                           },
                       order => ['D4','D3','D2','D1','D7','D6','D5'],
                   }
               );

=cut

sub linked_select_forms {
    my ($formname,
        $middletext,
        $firstdefault,
        $firstselectname,
        $secondselectname, 
        $hashref,
        $menuorder,
        ) = @_;
    my $second = "document.$formname.$secondselectname";
    my $first = "document.$formname.$firstselectname";
    # output the javascript to do the changing
    my $result = '';
    $result.="<script type=\"text/javascript\">\n";
    $result.="var select2data = new Object();\n";
    $" = '","';
    my $debug = '';
    foreach my $s1 (sort(keys(%$hashref))) {
        $result.="select2data.d_$s1 = new Object();\n";        
        $result.="select2data.d_$s1.def = new String('".
            $hashref->{$s1}->{'default'}."');\n";
        $result.="select2data.d_$s1.values = new Array(";
        my @s2values = sort(keys( %{ $hashref->{$s1}->{'select2'} } ));
        if (ref($hashref->{$s1}->{'order'}) eq 'ARRAY') {
            @s2values = @{$hashref->{$s1}->{'order'}};
        }
        $result.="\"@s2values\");\n";
        $result.="select2data.d_$s1.texts = new Array(";        
        my @s2texts;
        foreach my $value (@s2values) {
            push @s2texts, $hashref->{$s1}->{'select2'}->{$value};
        }
        $result.="\"@s2texts\");\n";
    }
    $"=' ';
    $result.= <<"END";

function select1_changed() {
    // Determine new choice
    var newvalue = "d_" + $first.value;
    // update select2
    var values     = select2data[newvalue].values;
    var texts      = select2data[newvalue].texts;
    var select2def = select2data[newvalue].def;
    var i;
    // out with the old
    for (i = 0; i < $second.options.length; i++) {
        $second.options[i] = null;
    }
    // in with the nuclear
    for (i=0;i<values.length; i++) {
        $second.options[i] = new Option(values[i]);
        $second.options[i].value = values[i];
        $second.options[i].text = texts[i];
        if (values[i] == select2def) {
            $second.options[i].selected = true;
        }
    }
}
</script>
END
    # output the initial values for the selection lists
    $result .= "<select size=\"1\" name=\"$firstselectname\" onchange=\"select1_changed()\">\n";
    my @order = sort(keys(%{$hashref}));
    if (ref($menuorder) eq 'ARRAY') {
        @order = @{$menuorder};
    }
    foreach my $value (@order) {
        $result.="    <option value=\"$value\" ";
        $result.=" selected=\"selected\" " if ($value eq $firstdefault);
        $result.=">".&mt($hashref->{$value}->{'text'})."</option>\n";
    }
    $result .= "</select>\n";
    my %select2 = %{$hashref->{$firstdefault}->{'select2'}};
    $result .= $middletext;
    $result .= "<select size=\"1\" name=\"$secondselectname\">\n";
    my $seconddefault = $hashref->{$firstdefault}->{'default'};
    
    my @secondorder = sort(keys(%select2));
    if (ref($hashref->{$firstdefault}->{'order'}) eq 'ARRAY') {
        @secondorder = @{$hashref->{$firstdefault}->{'order'}};
    }
    foreach my $value (@secondorder) {
        $result.="    <option value=\"$value\" ";        
        $result.=" selected=\"selected\" " if ($value eq $seconddefault);
        $result.=">".&mt($select2{$value})."</option>\n";
    }
    $result .= "</select>\n";
    #    return $debug;
    return $result;
}   #  end of sub linked_select_forms {

=pod

=item * &help_open_topic($topic,$text,$stayOnPage,$width,$height)

Returns a string corresponding to an HTML link to the given help
$topic, where $topic corresponds to the name of a .tex file in
/home/httpd/html/adm/help/tex, with underscores replaced by
spaces. 

$text will optionally be linked to the same topic, allowing you to
link text in addition to the graphic. If you do not want to link
text, but wish to specify one of the later parameters, pass an
empty string. 

$stayOnPage is a value that will be interpreted as a boolean. If true,
the link will not open a new window. If false, the link will open
a new window using Javascript. (Default is false.) 

$width and $height are optional numerical parameters that will
override the width and height of the popped up window, which may
be useful for certain help topics with big pictures included. 

=cut

sub help_open_topic {
    my ($topic, $text, $stayOnPage, $width, $height) = @_;
    $text = "" if (not defined $text);
    $stayOnPage = 0 if (not defined $stayOnPage);
    if ($env{'browser.interface'} eq 'textual') {
	$stayOnPage=1;
    }
    $width = 350 if (not defined $width);
    $height = 400 if (not defined $height);
    my $filename = $topic;
    $filename =~ s/ /_/g;

    my $template = "";
    my $link;
    
    $topic=~s/\W/\_/g;

    if (!$stayOnPage) {
	$link = "javascript:void(open('/adm/help/${filename}.hlp', 'Help_for_$topic', 'menubar=0,toolbar=1,scrollbars=1,width=$width,height=$height,resizable=yes'))";
    } else {
	$link = "/adm/help/${filename}.hlp";
    }

    # Add the text
    if ($text ne "") {
	$template .= 
            "<table bgcolor='#3333AA' cellspacing='1' cellpadding='1' border='0'><tr>".
            "<td bgcolor='#5555FF'><a target=\"_top\" href=\"$link\"><font color='#FFFFFF' size='2'>$text</font></a>";
    }

    # Add the graphic
    my $title = &mt('Online Help');
    my $helpicon=&lonhttpdurl("/adm/help/help.png");
    $template .= <<"ENDTEMPLATE";
 <a target="_top" href="$link" title="$title"><img src="$helpicon" border="0" alt="(Help: $topic)" /></a>
ENDTEMPLATE
    if ($text ne '') { $template.='</td></tr></table>' };
    return $template;

}

# This is a quicky function for Latex cheatsheet editing, since it 
# appears in at least four places
sub helpLatexCheatsheet {
    my $other = shift;
    my $addOther = '';
    if ($other) {
	$addOther = Apache::loncommon::help_open_topic($other, shift,
						       undef, undef, 600) .
							   '</td><td>';
    }
    return '<table><tr><td>'.
	$addOther .
	&Apache::loncommon::help_open_topic("Greek_Symbols",&mt('Greek Symbols'),
					    undef,undef,600)
	.'</td><td>'.
	&Apache::loncommon::help_open_topic("Other_Symbols",&mt('Other Symbols'),
					    undef,undef,600)
	.'</td></tr></table>';
}

sub general_help {
    my $helptopic='Student_Intro';
    if ($env{'request.role'}=~/^(ca|au)/) {
	$helptopic='Authoring_Intro';
    } elsif ($env{'request.role'}=~/^cc/) {
	$helptopic='Course_Coordination_Intro';
    } elsif ($env{'request.role'}=~/^dc/) {
        $helptopic='Domain_Coordination_Intro';
    }
    return $helptopic;
}

sub update_help_link {
    my ($topic,$component_help,$faq,$bug,$stayOnPage) = @_;
    my $origurl = $ENV{'REQUEST_URI'};
    $origurl=~s|^/~|/priv/|;
    my $timestamp = time;
    foreach my $datum (\$topic,\$component_help,\$faq,\$bug,\$origurl) {
        $$datum = &escape($$datum);
    }

    my $banner_link = "/adm/helpmenu?page=banner&amp;topic=$topic&amp;component_help=$component_help&amp;faq=$faq&amp;bug=$bug&amp;origurl=$origurl&amp;stamp=$timestamp&amp;stayonpage=$stayOnPage";
    my $output .= <<"ENDOUTPUT";
<script type="text/javascript">
banner_link = '$banner_link';
</script>
ENDOUTPUT
    return $output;
}

# now just updates the help link and generates a blue icon
sub help_open_menu {
    my ($topic,$component_help,$faq,$bug,$stayOnPage,$width,$height,$text) 
	= @_;    
    $stayOnPage = 0 if (not defined $stayOnPage);
    # only use pop-up help (stayOnPage == 0)
    # if environment.remote is on (using remote control UI)
    if ($env{'browser.interface'} eq 'textual' ||
    	$env{'environment.remote'} eq 'off' ) {
        $stayOnPage=1;
    }
    my $output;
    if ($component_help) {
	if (!$text) {
	    $output=&help_open_topic($component_help,undef,$stayOnPage,
				       $width,$height);
	} else {
	    my $help_text;
	    $help_text=&unescape($topic);
	    $output='<table><tr><td>'.
		&help_open_topic($component_help,$help_text,$stayOnPage,
				 $width,$height).'</td></tr></table>';
	}
    }
    my $banner_link = &update_help_link($topic,$component_help,$faq,$bug,$stayOnPage);
    return $output.$banner_link;
}

sub top_nav_help {
    my ($text) = @_;
    $text = &mt($text);
    my $stay_on_page = 
	($env{'browser.interface'}  eq 'textual' ||
	 $env{'environment.remote'} eq 'off' );
    my $link = ($stay_on_page) ? "javascript:helpMenu('display')"
	                     : "javascript:helpMenu('open')";
    my $banner_link = &update_help_link(undef,undef,undef,undef,$stay_on_page);

    my $title = &mt('Get help');

    return <<"END";
$banner_link
 <a href="$link" title="$title">$text</a>
END
}

sub help_menu_js {
    my ($text) = @_;

    my $stayOnPage = 
	($env{'browser.interface'}  eq 'textual' ||
	 $env{'environment.remote'} eq 'off' );

    my $width = 620;
    my $height = 600;
    my $helptopic=&general_help();
    my $details_link = '/adm/help/'.$helptopic.'.hlp';
    my $nothing=&Apache::lonhtmlcommon::javascript_nothing();
    my $start_page =
        &Apache::loncommon::start_page('Help Menu', undef,
				       {'frameset'    => 1,
					'js_ready'    => 1,
					'add_entries' => {
					    'border' => '0',
					    'rows'   => "110,*",},});
    my $end_page =
        &Apache::loncommon::end_page({'frameset' => 1,
				      'js_ready' => 1,});

    my $template .= <<"ENDTEMPLATE";
<script type="text/javascript">
// <!-- BEGIN LON-CAPA Internal
// <![CDATA[
var banner_link = '';
function helpMenu(target) {
    var caller = this;
    if (target == 'open') {
        var newWindow = null;
        try {
            newWindow =  window.open($nothing,"helpmenu","HEIGHT=$height,WIDTH=$width,resizable=yes,scrollbars=yes" )
        }
        catch(error) {
            writeHelp(caller);
            return;
        }
        if (newWindow) {
            caller = newWindow;
        }
    }
    writeHelp(caller);
    return;
}
function writeHelp(caller) {
    caller.document.writeln('$start_page<frame name="bannerframe"  src="'+banner_link+'" /><frame name="bodyframe" src="$details_link" /> $end_page')
    caller.document.close()
    caller.focus()
}
// ]]>
// END LON-CAPA Internal -->
</script>
ENDTEMPLATE
    return $template;
}

sub help_open_bug {
    my ($topic, $text, $stayOnPage, $width, $height) = @_;
    unless ($env{'user.adv'}) { return ''; }
    unless ($Apache::lonnet::perlvar{'BugzillaHost'}) { return ''; }
    $text = "" if (not defined $text);
    $stayOnPage = 0 if (not defined $stayOnPage);
    if ($env{'browser.interface'} eq 'textual' ||
	$env{'environment.remote'} eq 'off' ) {
	$stayOnPage=1;
    }
    $width = 600 if (not defined $width);
    $height = 600 if (not defined $height);

    $topic=~s/\W+/\+/g;
    my $link='';
    my $template='';
    my $url=$Apache::lonnet::perlvar{'BugzillaHost'}.'enter_bug.cgi?product=LON-CAPA&amp;bug_file_loc='.
	&escape($ENV{'REQUEST_URI'}).'&amp;component='.$topic;
    if (!$stayOnPage)
    {
	$link = "javascript:void(open('$url', 'Bugzilla', 'menubar=0,toolbar=1,scrollbars=1,width=$width,height=$height,resizable=yes'))";
    }
    else
    {
	$link = $url;
    }
    # Add the text
    if ($text ne "")
    {
	$template .= 
  "<table bgcolor='#AA3333' cellspacing='1' cellpadding='1' border='0'><tr>".
  "<td bgcolor='#FF5555'><a target=\"_top\" href=\"$link\"><font color='#FFFFFF' size='2'>$text</font></a>";
    }

    # Add the graphic
    my $title = &mt('Report a Bug');
    my $bugicon=&lonhttpdurl("/adm/lonMisc/smallBug.gif");
    $template .= <<"ENDTEMPLATE";
 <a target="_top" href="$link" title="$title"><img src="$bugicon" border="0" alt="(Bug: $topic)" /></a>
ENDTEMPLATE
    if ($text ne '') { $template.='</td></tr></table>' };
    return $template;

}

sub help_open_faq {
    my ($topic, $text, $stayOnPage, $width, $height) = @_;
    unless ($env{'user.adv'}) { return ''; }
    unless ($Apache::lonnet::perlvar{'FAQHost'}) { return ''; }
    $text = "" if (not defined $text);
    $stayOnPage = 0 if (not defined $stayOnPage);
    if ($env{'browser.interface'} eq 'textual' ||
	$env{'environment.remote'} eq 'off' ) {
	$stayOnPage=1;
    }
    $width = 350 if (not defined $width);
    $height = 400 if (not defined $height);

    $topic=~s/\W+/\+/g;
    my $link='';
    my $template='';
    my $url=$Apache::lonnet::perlvar{'FAQHost'}.'/fom/cache/'.$topic.'.html';
    if (!$stayOnPage)
    {
	$link = "javascript:void(open('$url', 'FAQ-O-Matic', 'menubar=0,toolbar=1,scrollbars=1,width=$width,height=$height,resizable=yes'))";
    }
    else
    {
	$link = $url;
    }

    # Add the text
    if ($text ne "")
    {
	$template .= 
  "<table bgcolor='#337733' cellspacing='1' cellpadding='1' border='0'><tr>".
  "<td bgcolor='#448844'><a target=\"_top\" href=\"$link\"><font color='#FFFFFF' size='2'>$text</font></a>";
    }

    # Add the graphic
    my $title = &mt('View the FAQ');
    my $faqicon=&lonhttpdurl("/adm/lonMisc/smallFAQ.gif");
    $template .= <<"ENDTEMPLATE";
 <a target="_top" href="$link" title="$title"><img src="$faqicon" border="0" alt="(FAQ: $topic)" /></a>
ENDTEMPLATE
    if ($text ne '') { $template.='</td></tr></table>' };
    return $template;

}

###############################################################
###############################################################

=pod

=item * &change_content_javascript():

This and the next function allow you to create small sections of an
otherwise static HTML page that you can update on the fly with
Javascript, even in Netscape 4.

The Javascript fragment returned by this function (no E<lt>scriptE<gt> tag)
must be written to the HTML page once. It will prove the Javascript
function "change(name, content)". Calling the change function with the
name of the section 
you want to update, matching the name passed to C<changable_area>, and
the new content you want to put in there, will put the content into
that area.

B<Note>: Netscape 4 only reserves enough space for the changable area
to contain room for the original contents. You need to "make space"
for whatever changes you wish to make, and be B<sure> to check your
code in Netscape 4. This feature in Netscape 4 is B<not> powerful;
it's adequate for updating a one-line status display, but little more.
This script will set the space to 100% width, so you only need to
worry about height in Netscape 4.

Modern browsers are much less limiting, and if you can commit to the
user not using Netscape 4, this feature may be used freely with
pretty much any HTML.

=cut

sub change_content_javascript {
    # If we're on Netscape 4, we need to use Layer-based code
    if ($env{'browser.type'} eq 'netscape' &&
	$env{'browser.version'} =~ /^4\./) {
	return (<<NETSCAPE4);
	function change(name, content) {
	    doc = document.layers[name+"___escape"].layers[0].document;
	    doc.open();
	    doc.write(content);
	    doc.close();
	}
NETSCAPE4
    } else {
	# Otherwise, we need to use semi-standards-compliant code
	# (technically, "innerHTML" isn't standard but the equivalent
	# is really scary, and every useful browser supports it
	return (<<DOMBASED);
	function change(name, content) {
	    element = document.getElementById(name);
	    element.innerHTML = content;
	}
DOMBASED
    }
}

=pod

=item * &changable_area($name,$origContent):

This provides a "changable area" that can be modified on the fly via
the Javascript code provided in C<change_content_javascript>. $name is
the name you will use to reference the area later; do not repeat the
same name on a given HTML page more then once. $origContent is what
the area will originally contain, which can be left blank.

=cut

sub changable_area {
    my ($name, $origContent) = @_;

    if ($env{'browser.type'} eq 'netscape' &&
	$env{'browser.version'} =~ /^4\./) {
	# If this is netscape 4, we need to use the Layer tag
	return "<ilayer width='100%' id='${name}___escape' overflow='none'><layer width='100%' id='$name' overflow='none'>$origContent</layer></ilayer>";
    } else {
	return "<span id='$name'>$origContent</span>";
    }
}

=pod

=item * &viewport_geometry_js 

Provides javascript object (Geometry) which can provide information about the viewport geometry for the client browser.

=cut


sub viewport_geometry_js { 
    return <<"GEOMETRY";
var Geometry = {};
function init_geometry() {
    if (Geometry.init) { return };
    Geometry.init=1;
    if (window.innerHeight) {
        Geometry.getViewportHeight   = function() { return window.innerHeight; };
        Geometry.getViewportWidth   = function() { return window.innerWidth; };
        Geometry.getHorizontalScroll = function() { return window.pageXOffset; };
        Geometry.getVerticalScroll   = function() { return window.pageYOffset; };
    }
    else if (document.documentElement && document.documentElement.clientHeight) {
        Geometry.getViewportHeight =
            function() { return document.documentElement.clientHeight; };
        Geometry.getViewportWidth =
            function() { return document.documentElement.clientWidth; };

        Geometry.getHorizontalScroll =
            function() { return document.documentElement.scrollLeft; };
        Geometry.getVerticalScroll =
            function() { return document.documentElement.scrollTop; };
    }
    else if (document.body.clientHeight) {
        Geometry.getViewportHeight =
            function() { return document.body.clientHeight; };
        Geometry.getViewportWidth =
            function() { return document.body.clientWidth; };
        Geometry.getHorizontalScroll =
            function() { return document.body.scrollLeft; };
        Geometry.getVerticalScroll =
            function() { return document.body.scrollTop; };
    }
}

GEOMETRY
}

=pod

=item * &viewport_size_js()

Provides a javascript function to set values of two form elements - width and height (elements are passed in as arguments to the javascript function) to the dimensions of the user's browser window. 

=cut

sub viewport_size_js {
    my $geometry = &viewport_geometry_js();
    return <<"DIMS";

$geometry

function getViewportDims(width,height) {
    init_geometry();
    width.value = Geometry.getViewportWidth();
    height.value = Geometry.getViewportHeight();
    return;
}

DIMS
}

=pod

=item * &resize_textarea_js()

emits the needed javascript to resize a textarea to be as big as possible

creates a function resize_textrea that takes two IDs first should be
the id of the element to resize, second should be the id of a div that
surrounds everything that comes after the textarea, this routine needs
to be attached to the <body> for the onload and onresize events.

=back

=cut

sub resize_textarea_js {
    my $geometry = &viewport_geometry_js();
    return <<"RESIZE";
    <script type="text/javascript">
$geometry

function getX(element) {
    var x = 0;
    while (element) {
	x += element.offsetLeft;
	element = element.offsetParent;
    }
    return x;
}
function getY(element) {
    var y = 0;
    while (element) {
	y += element.offsetTop;
	element = element.offsetParent;
    }
    return y;
}


function resize_textarea(textarea_id,bottom_id) {
    init_geometry();
    var textarea        = document.getElementById(textarea_id);
    //alert(textarea);

    var textarea_top    = getY(textarea);
    var textarea_height = textarea.offsetHeight;
    var bottom          = document.getElementById(bottom_id);
    var bottom_top      = getY(bottom);
    var bottom_height   = bottom.offsetHeight;
    var window_height   = Geometry.getViewportHeight();
    var fudge           = 23;
    var new_height      = window_height-fudge-textarea_top-bottom_height;
    if (new_height < 300) {
	new_height = 300;
    }
    textarea.style.height=new_height+'px';
}
</script>
RESIZE

}

=pod

=head1 Excel and CSV file utility routines

=over 4

=cut

###############################################################
###############################################################

=pod

=item * &csv_translate($text) 

Translate $text to allow it to be output as a 'comma separated values' 
format.

=cut

###############################################################
###############################################################
sub csv_translate {
    my $text = shift;
    $text =~ s/\"/\"\"/g;
    $text =~ s/\n/ /g;
    return $text;
}

###############################################################
###############################################################

=pod

=item * &define_excel_formats()

Define some commonly used Excel cell formats.

Currently supported formats:

=over 4

=item header

=item bold

=item h1

=item h2

=item h3

=item h4

=item i

=item date

=back

Inputs: $workbook

Returns: $format, a hash reference.

=cut

###############################################################
###############################################################
sub define_excel_formats {
    my ($workbook) = @_;
    my $format;
    $format->{'header'} = $workbook->add_format(bold      => 1, 
                                                bottom    => 1,
                                                align     => 'center');
    $format->{'bold'} = $workbook->add_format(bold=>1);
    $format->{'h1'}   = $workbook->add_format(bold=>1, size=>18);
    $format->{'h2'}   = $workbook->add_format(bold=>1, size=>16);
    $format->{'h3'}   = $workbook->add_format(bold=>1, size=>14);
    $format->{'h4'}   = $workbook->add_format(bold=>1, size=>12);
    $format->{'i'}    = $workbook->add_format(italic=>1);
    $format->{'date'} = $workbook->add_format(num_format=>
                                            'mm/dd/yyyy hh:mm:ss');
    return $format;
}

###############################################################
###############################################################

=pod

=item * &create_workbook()

Create an Excel worksheet.  If it fails, output message on the
request object and return undefs.

Inputs: Apache request object

Returns (undef) on failure, 
    Excel worksheet object, scalar with filename, and formats 
    from &Apache::loncommon::define_excel_formats on success

=cut

###############################################################
###############################################################
sub create_workbook {
    my ($r) = @_;
        #
    # Create the excel spreadsheet
    my $filename = '/prtspool/'.
        $env{'user.name'}.'_'.$env{'user.domain'}.'_'.
        time.'_'.rand(1000000000).'.xls';
    my $workbook  = Spreadsheet::WriteExcel->new('/home/httpd'.$filename);
    if (! defined($workbook)) {
        $r->log_error("Error creating excel spreadsheet $filename: $!");
        $r->print('<p>'.&mt("Unable to create new Excel file.  ".
                            "This error has been logged.  ".
                            "Please alert your LON-CAPA administrator").
                  '</p>');
        return (undef);
    }
    #
    $workbook->set_tempdir('/home/httpd/perl/tmp');
    #
    my $format = &Apache::loncommon::define_excel_formats($workbook);
    return ($workbook,$filename,$format);
}

###############################################################
###############################################################

=pod

=item * &create_text_file()

Create a file to write to and eventually make available to the user.
If file creation fails, outputs an error message on the request object and 
return undefs.

Inputs: Apache request object, and file suffix

Returns (undef) on failure, 
    Filehandle and filename on success.

=cut

###############################################################
###############################################################
sub create_text_file {
    my ($r,$suffix) = @_;
    if (! defined($suffix)) { $suffix = 'txt'; };
    my $fh;
    my $filename = '/prtspool/'.
        $env{'user.name'}.'_'.$env{'user.domain'}.'_'.
        time.'_'.rand(1000000000).'.'.$suffix;
    $fh = Apache::File->new('>/home/httpd'.$filename);
    if (! defined($fh)) {
        $r->log_error("Couldn't open $filename for output $!");
        $r->print("Problems occured in creating the output file.  ".
                  "This error has been logged.  ".
                  "Please alert your LON-CAPA administrator.");
    }
    return ($fh,$filename)
}


=pod 

=back

=cut

###############################################################
##        Home server <option> list generating code          ##
###############################################################

# ------------------------------------------

sub domain_select {
    my ($name,$value,$multiple)=@_;
    my %domains=map { 
	$_ => $_.' '. &Apache::lonnet::domain($_,'description') 
    } &Apache::lonnet::all_domains();
    if ($multiple) {
	$domains{''}=&mt('Any domain');
	$domains{'select_form_order'} = [sort {lc($a) cmp lc($b) } (keys(%domains))];
	return &multiple_select_form($name,$value,4,\%domains);
    } else {
	$domains{'select_form_order'} = [sort {lc($a) cmp lc($b) } (keys(%domains))];
	return &select_form($name,$value,%domains);
    }
}

#-------------------------------------------

=pod

=head1 Routines for form select boxes

=over 4

=item * &multiple_select_form($name,$value,$size,$hash,$order)

Returns a string containing a <select> element int multiple mode


Args:
  $name - name of the <select> element
  $value - scalar or array ref of values that should already be selected
  $size - number of rows long the select element is
  $hash - the elements should be 'option' => 'shown text'
          (shown text should already have been &mt())
  $order - (optional) array ref of the order to show the elements in

=cut

#-------------------------------------------
sub multiple_select_form {
    my ($name,$value,$size,$hash,$order)=@_;
    my %selected = map { $_ => 1 } ref($value)?@{$value}:($value);
    my $output='';
    if (! defined($size)) {
        $size = 4;
        if (scalar(keys(%$hash))<4) {
            $size = scalar(keys(%$hash));
        }
    }
    $output.="\n<select name='$name' size='$size' multiple='1'>";
    my @order;
    if (ref($order) eq 'ARRAY')  {
        @order = @{$order};
    } else {
        @order = sort(keys(%$hash));
    }
    if (exists($$hash{'select_form_order'})) {
        @order = @{$$hash{'select_form_order'}};
    }
        
    foreach my $key (@order) {
        $output.='<option value="'.&HTML::Entities::encode($key,'"<>&').'" ';
        $output.='selected="selected" ' if ($selected{$key});
        $output.='>'.$hash->{$key}."</option>\n";
    }
    $output.="</select>\n";
    return $output;
}

#-------------------------------------------

=pod

=item * &select_form($defdom,$name,%hash)

Returns a string containing a <select name='$name' size='1'> form to 
allow a user to select options from a hash option_name => displayed text.  
See lonrights.pm for an example invocation and use.

=cut

#-------------------------------------------
sub select_form {
    my ($def,$name,%hash) = @_;
    my $selectform = "<select name=\"$name\" size=\"1\">\n";
    my @keys;
    if (exists($hash{'select_form_order'})) {
	@keys=@{$hash{'select_form_order'}};
    } else {
	@keys=sort(keys(%hash));
    }
    foreach my $key (@keys) {
        $selectform.=
	    '<option value="'.&HTML::Entities::encode($key,'"<>&').'" '.
            ($key eq $def ? 'selected="selected" ' : '').
                ">".&mt($hash{$key})."</option>\n";
    }
    $selectform.="</select>";
    return $selectform;
}

# For display filters

sub display_filter {
    if (!$env{'form.show'}) { $env{'form.show'}=10; }
    if (!$env{'form.displayfilter'}) { $env{'form.displayfilter'}='currentfolder'; }
    return '<nobr><label>'.&mt('Records [_1]',
			       &Apache::lonmeta::selectbox('show',$env{'form.show'},undef,
							   (&mt('all'),10,20,50,100,1000,10000))).
	   '</label></nobr> <nobr>'.
           &mt('Filter [_1]',
	   &select_form($env{'form.displayfilter'},
			'displayfilter',
			('currentfolder' => 'Current folder/page',
			 'containing' => 'Containing phrase',
			 'none' => 'None'))).
			 '<input type="text" name="containingphrase" size="30" value="'.&HTML::Entities::encode($env{'form.containingphrase'}).'" /></nobr>';
}

sub gradeleveldescription {
    my $gradelevel=shift;
    my %gradelevels=(0 => 'Not specified',
		     1 => 'Grade 1',
		     2 => 'Grade 2',
		     3 => 'Grade 3',
		     4 => 'Grade 4',
		     5 => 'Grade 5',
		     6 => 'Grade 6',
		     7 => 'Grade 7',
		     8 => 'Grade 8',
		     9 => 'Grade 9',
		     10 => 'Grade 10',
		     11 => 'Grade 11',
		     12 => 'Grade 12',
		     13 => 'Grade 13',
		     14 => '100 Level',
		     15 => '200 Level',
		     16 => '300 Level',
		     17 => '400 Level',
		     18 => 'Graduate Level');
    return &mt($gradelevels{$gradelevel});
}

sub select_level_form {
    my ($deflevel,$name)=@_;
    unless ($deflevel) { $deflevel=0; }
    my $selectform = "<select name=\"$name\" size=\"1\">\n";
    for (my $i=0; $i<=18; $i++) {
        $selectform.="<option value=\"$i\" ".
            ($i==$deflevel ? 'selected="selected" ' : '').
                ">".&gradeleveldescription($i)."</option>\n";
    }
    $selectform.="</select>";
    return $selectform;
}

#-------------------------------------------

=pod

=item * &select_dom_form($defdom,$name,$includeempty,$showdomdesc)

Returns a string containing a <select name='$name' size='1'> form to 
allow a user to select the domain to preform an operation in.  
See loncreateuser.pm for an example invocation and use.

If the $includeempty flag is set, it also includes an empty choice ("no domain
selected");

If the $showdomdesc flag is set, the domain name is followed by the domain description. 

=cut

#-------------------------------------------
sub select_dom_form {
    my ($defdom,$name,$includeempty,$showdomdesc) = @_;
    my @domains = sort {lc($a) cmp lc($b)} (&Apache::lonnet::all_domains());
    if ($includeempty) { @domains=('',@domains); }
    my $selectdomain = "<select name=\"$name\" size=\"1\">\n";
    foreach my $dom (@domains) {
        $selectdomain.="<option value=\"$dom\" ".
            ($dom eq $defdom ? 'selected="selected" ' : '').'>'.$dom;
        if ($showdomdesc) {
            if ($dom ne '') {
                my $domdesc = &Apache::lonnet::domain($dom,'description');
                if ($domdesc ne '') {
                    $selectdomain .= ' ('.$domdesc.')';
                }
            } 
        }
        $selectdomain .= "</option>\n";
    }
    $selectdomain.="</select>";
    return $selectdomain;
}

#-------------------------------------------

=pod

=item * &home_server_form_item($domain,$name,$defaultflag)

input: 4 arguments (two required, two optional) - 
    $domain - domain of new user
    $name - name of form element
    $default - Value of 'default' causes a default item to be first 
                            option, and selected by default. 
    $hide - Value of 'hide' causes hiding of the name of the server, 
                            if 1 server found, or default, if 0 found.
output: returns 2 items: 
(a) form element which contains either:
   (i) <select name="$name">
        <option value="$hostid1">$hostid $servers{$hostid}</option>
        <option value="$hostid2">$hostid $servers{$hostid}</option>       
       </select>
       form item if there are multiple library servers in $domain, or
   (ii) an <input type="hidden" name="$name" value="$hostid" /> form item 
       if there is only one library server in $domain.

(b) number of library servers found.

See loncreateuser.pm for example of use.

=cut

#-------------------------------------------
sub home_server_form_item {
    my ($domain,$name,$default,$hide) = @_;
    my %servers = &Apache::lonnet::get_servers($domain,'library');
    my $result;
    my $numlib = keys(%servers);
    if ($numlib > 1) {
        $result .= '<select name="'.$name.'" />'."\n";
        if ($default) {
            $result .= '<option value="default" selected>'.&mt('default').
                       '</option>'."\n";
        }
        foreach my $hostid (sort(keys(%servers))) {
            $result.= '<option value="'.$hostid.'">'.
	              $hostid.' '.$servers{$hostid}."</option>\n";
        }
        $result .= '</select>'."\n";
    } elsif ($numlib == 1) {
        my $hostid;
        foreach my $item (keys(%servers)) {
            $hostid = $item;
        }
        $result .= '<input type="hidden" name="'.$name.'" value="'.
                   $hostid.'" />';
                   if (!$hide) {
                       $result .= $hostid.' '.$servers{$hostid};
                   }
                   $result .= "\n";
    } elsif ($default) {
        $result .= '<input type="hidden" name="'.$name.
                   '" value="default" />';
                   if (!$hide) {
                       $result .= &mt('default');
                   }
                   $result .= "\n";
    }
    return ($result,$numlib);
}

=pod

=back 

=cut

###############################################################
##                  Decoding User Agent                      ##
###############################################################

=pod

=head1 Decoding the User Agent

=over 4

=item * &decode_user_agent()

Inputs: $r

Outputs:

=over 4

=item * $httpbrowser

=item * $clientbrowser

=item * $clientversion

=item * $clientmathml

=item * $clientunicode

=item * $clientos

=back

=back 

=cut

###############################################################
###############################################################
sub decode_user_agent {
    my ($r)=@_;
    my @browsertype=split(/\&/,$Apache::lonnet::perlvar{"lonBrowsDet"});
    my %mathcap=split(/\&/,$$Apache::lonnet::perlvar{"lonMathML"});
    my $httpbrowser=$ENV{"HTTP_USER_AGENT"};
    if (!$httpbrowser && $r) { $httpbrowser=$r->header_in('User-Agent'); }
    my $clientbrowser='unknown';
    my $clientversion='0';
    my $clientmathml='';
    my $clientunicode='0';
    for (my $i=0;$i<=$#browsertype;$i++) {
        my ($bname,$match,$notmatch,$vreg,$minv,$univ)=split(/\:/,$browsertype[$i]);
	if (($httpbrowser=~/$match/i)  && ($httpbrowser!~/$notmatch/i)) {
	    $clientbrowser=$bname;
            $httpbrowser=~/$vreg/i;
	    $clientversion=$1;
            $clientmathml=($clientversion>=$minv);
            $clientunicode=($clientversion>=$univ);
	}
    }
    my $clientos='unknown';
    if (($httpbrowser=~/linux/i) ||
        ($httpbrowser=~/unix/i) ||
        ($httpbrowser=~/ux/i) ||
        ($httpbrowser=~/solaris/i)) { $clientos='unix'; }
    if (($httpbrowser=~/vax/i) ||
        ($httpbrowser=~/vms/i)) { $clientos='vms'; }
    if ($httpbrowser=~/next/i) { $clientos='next'; }
    if (($httpbrowser=~/mac/i) ||
        ($httpbrowser=~/powerpc/i)) { $clientos='mac'; }
    if ($httpbrowser=~/win/i) { $clientos='win'; }
    if ($httpbrowser=~/embed/i) { $clientos='pda'; }
    return ($httpbrowser,$clientbrowser,$clientversion,$clientmathml,
            $clientunicode,$clientos,);
}

###############################################################
##    Authentication changing form generation subroutines    ##
###############################################################
##
## All of the authform_xxxxxxx subroutines take their inputs in a
## hash, and have reasonable default values.
##
##    formname = the name given in the <form> tag.
#-------------------------------------------

=pod

=head1 Authentication Routines

=over 4

=item * &authform_xxxxxx()

The authform_xxxxxx subroutines provide javascript and html forms which 
handle some of the conveniences required for authentication forms.  
This is not an optimal method, but it works.  

=over 4

=item * authform_header

=item * authform_authorwarning

=item * authform_nochange

=item * authform_kerberos

=item * authform_internal

=item * authform_filesystem

=back

See loncreateuser.pm for invocation and use examples.

=cut

#-------------------------------------------
sub authform_header{  
    my %in = (
        formname => 'cu',
        kerb_def_dom => '',
        @_,
    );
    $in{'formname'} = 'document.' . $in{'formname'};
    my $result='';

#---------------------------------------------- Code for upper case translation
    my $Javascript_toUpperCase;
    unless ($in{kerb_def_dom}) {
        $Javascript_toUpperCase =<<"END";
        switch (choice) {
           case 'krb': currentform.elements[choicearg].value =
               currentform.elements[choicearg].value.toUpperCase();
               break;
           default:
        }
END
    } else {
        $Javascript_toUpperCase = "";
    }

    my $radioval = "'nochange'";
    if (defined($in{'curr_authtype'})) {
        if ($in{'curr_authtype'} ne '') {
            $radioval = "'".$in{'curr_authtype'}."arg'";
        }
    }
    my $argfield = 'null';
    if (defined($in{'mode'})) {
        if ($in{'mode'} eq 'modifycourse')  {
            if (defined($in{'curr_autharg'})) {
                if ($in{'curr_autharg'} ne '') {
                    $argfield = "'$in{'curr_autharg'}'";
                }
            }
        }
    }

    $result.=<<"END";
var current = new Object();
current.radiovalue = $radioval;
current.argfield = $argfield;

function changed_radio(choice,currentform) {
    var choicearg = choice + 'arg';
    // If a radio button in changed, we need to change the argfield
    if (current.radiovalue != choice) {
        current.radiovalue = choice;
        if (current.argfield != null) {
            currentform.elements[current.argfield].value = '';
        }
        if (choice == 'nochange') {
            current.argfield = null;
        } else {
            current.argfield = choicearg;
            switch(choice) {
                case 'krb': 
                    currentform.elements[current.argfield].value = 
                        "$in{'kerb_def_dom'}";
                break;
              default:
                break;
            }
        }
    }
    return;
}

function changed_text(choice,currentform) {
    var choicearg = choice + 'arg';
    if (currentform.elements[choicearg].value !='') {
        $Javascript_toUpperCase
        // clear old field
        if ((current.argfield != choicearg) && (current.argfield != null)) {
            currentform.elements[current.argfield].value = '';
        }
        current.argfield = choicearg;
    }
    set_auth_radio_buttons(choice,currentform);
    return;
}

function set_auth_radio_buttons(newvalue,currentform) {
    var i=0;
    while (i < currentform.login.length) {
        if (currentform.login[i].value == newvalue) { break; }
        i++;
    }
    if (i == currentform.login.length) {
        return;
    }
    current.radiovalue = newvalue;
    currentform.login[i].checked = true;
    return;
}
END
    return $result;
}

sub authform_authorwarning{
    my $result='';
    $result='<i>'.
        &mt('As a general rule, only authors or co-authors should be '.
            'filesystem authenticated '.
            '(which allows access to the server filesystem).')."</i>\n";
    return $result;
}

sub authform_nochange{  
    my %in = (
              formname => 'document.cu',
              kerb_def_dom => 'MSU.EDU',
              @_,
          );
    my ($authnum,%can_assign) =  &get_assignable_auth($in{'domain'}); 
    my $result;
    if (keys(%can_assign) == 0) {
        $result = &mt('Under you current role you are not permitted to change login settings for this user');  
    } else {
        $result = '<label>'.&mt('[_1] Do not change login data',
                  '<input type="radio" name="login" value="nochange" '.
                  'checked="checked" onclick="'.
            "javascript:changed_radio('nochange',$in{'formname'});".'" />').
	    '</label>';
    }
    return $result;
}

sub authform_kerberos {
    my %in = (
              formname => 'document.cu',
              kerb_def_dom => 'MSU.EDU',
              kerb_def_auth => 'krb4',
              @_,
              );
    my ($check4,$check5,$krbcheck,$krbarg,$krbver,$result,$authtype,
        $autharg,$jscall);
    my ($authnum,%can_assign) =  &get_assignable_auth($in{'domain'});
    if ($in{'kerb_def_auth'} eq 'krb5') {
       $check5 = ' checked="on"';
    } else {
       $check4 = ' checked="on"';
    }
    $krbarg = $in{'kerb_def_dom'};
    if (defined($in{'curr_authtype'})) {
        if ($in{'curr_authtype'} eq 'krb') {
            $krbcheck = ' checked="on"';
            if (defined($in{'mode'})) {
                if ($in{'mode'} eq 'modifyuser') {
                    $krbcheck = '';
                }
            }
            if (defined($in{'curr_kerb_ver'})) {
                if ($in{'curr_krb_ver'} eq '5') {
                    $check5 = ' checked="on"';
                    $check4 = '';
                } else {
                    $check4 = ' checked="on"';
                    $check5 = '';
                }
            }
            if (defined($in{'curr_autharg'})) {
                $krbarg = $in{'curr_autharg'};
            }
            if (!$can_assign{'krb4'} && !$can_assign{'krb5'}) {
                if (defined($in{'curr_autharg'})) {
                    $result = 
    &mt('Currently Kerberos authenticated with domain [_1] Version [_2].',
        $in{'curr_autharg'},$krbver);
                } else {
                    $result =
    &mt('Currently Kerberos authenticated, Version [_1].',$krbver);
                }
                return $result; 
            }
        }
    } else {
        if ($authnum == 1) {
            $authtype = '<input type="hidden" name="login" value="krb">';
        }
    }
    if (!$can_assign{'krb4'} && !$can_assign{'krb5'}) {
        return;
    } elsif ($authtype eq '') {
        if (defined($in{'mode'})) {
            if ($in{'mode'} eq 'modifycourse') {
                if ($authnum == 1) {
                    $authtype = '<input type="hidden" name="login" value="krb">';
                }
            }
        }
    }
    $jscall = "javascript:changed_radio('krb',$in{'formname'});";
    if ($authtype eq '') {
        $authtype = '<input type="radio" name="login" value="krb" '.
                    'onclick="'.$jscall.'" onchange="'.$jscall.'"'.
                    $krbcheck.' />';
    }
    if (($can_assign{'krb4'} && $can_assign{'krb5'}) ||
        ($can_assign{'krb4'} && !$can_assign{'krb5'} && 
         $in{'curr_authtype'} eq 'krb5') ||
        (!$can_assign{'krb4'} && $can_assign{'krb5'} && 
         $in{'curr_authtype'} eq 'krb4')) {
        $result .= &mt
        ('[_1] Kerberos authenticated with domain [_2] '.
         '[_3] Version 4 [_4] Version 5 [_5]',
         '<label>'.$authtype,
         '</label><input type="text" size="10" name="krbarg" '.
             'value="'.$krbarg.'" '.
             'onchange="'.$jscall.'" />',
         '<label><input type="radio" name="krbver" value="4" '.$check4.' />',
         '</label><label><input type="radio" name="krbver" value="5" '.$check5.' />',
	 '</label>');
    } elsif ($can_assign{'krb4'}) {
        $result .= &mt
        ('[_1] Kerberos authenticated with domain [_2] '.
         '[_3] Version 4 [_4]',
         '<label>'.$authtype,
         '</label><input type="text" size="10" name="krbarg" '.
             'value="'.$krbarg.'" '.
             'onchange="'.$jscall.'" />',
         '<label><input type="hidden" name="krbver" value="4" />',
         '</label>');
    } elsif ($can_assign{'krb5'}) {
        $result .= &mt
        ('[_1] Kerberos authenticated with domain [_2] '.
         '[_3] Version 5 [_4]',
         '<label>'.$authtype,
         '</label><input type="text" size="10" name="krbarg" '.
             'value="'.$krbarg.'" '.
             'onchange="'.$jscall.'" />',
         '<label><input type="hidden" name="krbver" value="5" />',
         '</label>');
    }
    return $result;
}

sub authform_internal{  
    my %in = (
                formname => 'document.cu',
                kerb_def_dom => 'MSU.EDU',
                @_,
                );
    my ($intcheck,$intarg,$result,$authtype,$autharg,$jscall);
    my ($authnum,%can_assign) =  &get_assignable_auth($in{'domain'});
    if (defined($in{'curr_authtype'})) {
        if ($in{'curr_authtype'} eq 'int') {
            if ($can_assign{'int'}) {
                $intcheck = 'checked="on" ';
                if (defined($in{'mode'})) {
                    if ($in{'mode'} eq 'modifyuser') {
                        $intcheck = '';
                    }
                }
                if (defined($in{'curr_autharg'})) {
                    $intarg = $in{'curr_autharg'};
                }
            } else {
                $result = &mt('Currently internally authenticated.');
                return $result;
            }
        }
    } else {
        if ($authnum == 1) {
            $authtype = '<input type="hidden" name="login" value="int">';
        }
    }
    if (!$can_assign{'int'}) {
        return;
    } elsif ($authtype eq '') {
        if (defined($in{'mode'})) {
            if ($in{'mode'} eq 'modifycourse') {
                if ($authnum == 1) {
                    $authtype = '<input type="hidden" name="login" value="int">';
                }
            }
        }
    }
    $jscall = "javascript:changed_radio('int',$in{'formname'});";
    if ($authtype eq '') {
        $authtype = '<input type="radio" name="login" value="int" '.$intcheck.
                    ' onchange="'.$jscall.'" onclick="'.$jscall.'" />';
    }
    $autharg = '<input type="password" size="10" name="intarg" value="'.
               $intarg.'" onchange="'.$jscall.'" />';
    $result = &mt
        ('[_1] Internally authenticated (with initial password [_2])',
         '<label>'.$authtype,'</label>'.$autharg);
    $result.="<label><input type=\"checkbox\" name=\"visible\" onClick='if (this.checked) { this.form.intarg.type=\"text\" } else { this.form.intarg.type=\"password\" }' />".&mt('Visible input').'</label>';
    return $result;
}

sub authform_local{  
    my %in = (
              formname => 'document.cu',
              kerb_def_dom => 'MSU.EDU',
              @_,
              );
    my ($loccheck,$locarg,$result,$authtype,$autharg,$jscall);
    my ($authnum,%can_assign) =  &get_assignable_auth($in{'domain'});
    if (defined($in{'curr_authtype'})) {
        if ($in{'curr_authtype'} eq 'loc') {
            if ($can_assign{'loc'}) {
                $loccheck = 'checked="on" ';
                if (defined($in{'mode'})) {
                    if ($in{'mode'} eq 'modifyuser') {
                        $loccheck = '';
                    }
                }
                if (defined($in{'curr_autharg'})) {
                    $locarg = $in{'curr_autharg'};
                }
            } else {
                $result = &mt('Currently using local (institutional) authentication.');
                return $result;
            }
        }
    } else {
        if ($authnum == 1) {
            $authtype = '<input type="hidden" name="login" value="loc">';
        }
    }
    if (!$can_assign{'loc'}) {
        return;
    } elsif ($authtype eq '') {
        if (defined($in{'mode'})) {
            if ($in{'mode'} eq 'modifycourse') {
                if ($authnum == 1) {
                    $authtype = '<input type="hidden" name="login" value="loc">';
                }
            }
        }
    }
    $jscall = "javascript:changed_radio('loc',$in{'formname'});";
    if ($authtype eq '') {
        $authtype = '<input type="radio" name="login" value="loc" '.
                    $loccheck.' onchange="'.$jscall.'" onclick="'.
                    $jscall.'" />';
    }
    $autharg = '<input type="text" size="10" name="locarg" value="'.
               $locarg.'" onchange="'.$jscall.'" />';
    $result = &mt('[_1] Local Authentication with argument [_2]',
                  '<label>'.$authtype,'</label>'.$autharg);
    return $result;
}

sub authform_filesystem{  
    my %in = (
              formname => 'document.cu',
              kerb_def_dom => 'MSU.EDU',
              @_,
              );
    my ($fsyscheck,$result,$authtype,$autharg,$jscall);
    my ($authnum,%can_assign) =  &get_assignable_auth($in{'domain'});
    if (defined($in{'curr_authtype'})) {
        if ($in{'curr_authtype'} eq 'fsys') {
            if ($can_assign{'fsys'}) {
                $fsyscheck = 'checked="on" ';
                if (defined($in{'mode'})) {
                    if ($in{'mode'} eq 'modifyuser') {
                        $fsyscheck = '';
                    }
                }
            } else {
                $result = &mt('Currently Filesystem Authenticated.');
                return $result;
            }           
        }
    } else {
        if ($authnum == 1) {
            $authtype = '<input type="hidden" name="login" value="fsys">';
        }
    }
    if (!$can_assign{'fsys'}) {
        return;
    } elsif ($authtype eq '') {
        if (defined($in{'mode'})) {
            if ($in{'mode'} eq 'modifycourse') {
                if ($authnum == 1) {
                    $authtype = '<input type="hidden" name="login" value="fsys">';
                }
            }
        }
    }
    $jscall = "javascript:changed_radio('fsys',$in{'formname'});";
    if ($authtype eq '') {
        $authtype = '<input type="radio" name="login" value="fsys" '.
                    $fsyscheck.' onchange="'.$jscall.'" onclick="'.
                    $jscall.'" />';
    }
    $autharg = '<input type="text" size="10" name="fsysarg" value=""'.
               ' onchange="'.$jscall.'" />';
    $result = &mt
        ('[_1] Filesystem Authenticated (with initial password [_2])',
         '<label><input type="radio" name="login" value="fsys" '.
         $fsyscheck.'onchange="'.$jscall.'" onclick="'.$jscall.'" />',
         '</label><input type="password" size="10" name="fsysarg" value="" '.
                  'onchange="'.$jscall.'" />');
    return $result;
}

sub get_assignable_auth {
    my ($dom) = @_;
    if ($dom eq '') {
        $dom = $env{'request.role.domain'};
    }
    my %can_assign = (
                          krb4 => 1,
                          krb5 => 1,
                          int  => 1,
                          loc  => 1,
                     );
    my %domconfig = &Apache::lonnet::get_dom('configuration',['usercreation'],$dom);
    if (ref($domconfig{'usercreation'}) eq 'HASH') {
        if (ref($domconfig{'usercreation'}{'authtypes'}) eq 'HASH') {
            my $authhash = $domconfig{'usercreation'}{'authtypes'};
            my $context;
            if ($env{'request.role'} =~ /^au/) {
                $context = 'author';
            } elsif ($env{'request.role'} =~ /^dc/) {
                $context = 'domain';
            } elsif ($env{'request.course.id'}) {
                $context = 'course';
            }
            if ($context) {
                if (ref($authhash->{$context}) eq 'HASH') {
                   %can_assign = %{$authhash->{$context}}; 
                }
            }
        }
    }
    my $authnum = 0;
    foreach my $key (keys(%can_assign)) {
        if ($can_assign{$key}) {
            $authnum ++;
        }
    }
    if ($can_assign{'krb4'} && $can_assign{'krb5'}) {
        $authnum --;
    }
    return ($authnum,%can_assign);
}

###############################################################
##    Get Kerberos Defaults for Domain                 ##
###############################################################
##
## Returns default kerberos version and an associated argument
## as listed in file domain.tab. If not listed, provides
## appropriate default domain and kerberos version.
##
#-------------------------------------------

=pod

=item * &get_kerberos_defaults()

get_kerberos_defaults($target_domain) returns the default kerberos
version and domain. If not found, it defaults to version 4 and the 
domain of the server.

=over 4

($def_version, $def_krb_domain) = &get_kerberos_defaults($target_domain);

=back

=back

=cut

#-------------------------------------------
sub get_kerberos_defaults {
    my $domain=shift;
    my ($krbdef,$krbdefdom);
    my %domdefaults = &Apache::lonnet::get_domain_defaults($domain);
    if (($domdefaults{'auth_def'} =~/^krb(4|5)$/) && ($domdefaults{'auth_arg_def'} ne '')) {
        $krbdef = $domdefaults{'auth_def'};
        $krbdefdom = $domdefaults{'auth_arg_def'};
    } else {
        $ENV{'SERVER_NAME'}=~/(\w+\.\w+)$/;
        my $krbdefdom=$1;
        $krbdefdom=~tr/a-z/A-Z/;
        $krbdef = "krb4";
    }
    return ($krbdef,$krbdefdom);
}


###############################################################
##                Thesaurus Functions                        ##
###############################################################

=pod

=head1 Thesaurus Functions

=over 4

=item * &initialize_keywords()

Initializes the package variable %Keywords if it is empty.  Uses the
package variable $thesaurus_db_file.

=cut

###################################################

sub initialize_keywords {
    return 1 if (scalar keys(%Keywords));
    # If we are here, %Keywords is empty, so fill it up
    #   Make sure the file we need exists...
    if (! -e $thesaurus_db_file) {
        &Apache::lonnet::logthis("Attempt to access $thesaurus_db_file".
                                 " failed because it does not exist");
        return 0;
    }
    #   Set up the hash as a database
    my %thesaurus_db;
    if (! tie(%thesaurus_db,'GDBM_File',
              $thesaurus_db_file,&GDBM_READER(),0640)){
        &Apache::lonnet::logthis("Could not tie \%thesaurus_db to ".
                                 $thesaurus_db_file);
        return 0;
    } 
    #  Get the average number of appearances of a word.
    my $avecount = $thesaurus_db{'average.count'};
    #  Put keywords (those that appear > average) into %Keywords
    while (my ($word,$data)=each (%thesaurus_db)) {
        my ($count,undef) = split /:/,$data;
        $Keywords{$word}++ if ($count > $avecount);
    }
    untie %thesaurus_db;
    # Remove special values from %Keywords.
    foreach my $value ('total.count','average.count') {
        delete($Keywords{$value}) if (exists($Keywords{$value}));
  }
    return 1;
}

###################################################

=pod

=item * &keyword($word)

Returns true if $word is a keyword.  A keyword is a word that appears more 
than the average number of times in the thesaurus database.  Calls 
&initialize_keywords

=cut

###################################################

sub keyword {
    return if (!&initialize_keywords());
    my $word=lc(shift());
    $word=~s/\W//g;
    return exists($Keywords{$word});
}

###############################################################

=pod 

=item * &get_related_words()

Look up a word in the thesaurus.  Takes a scalar argument and returns
an array of words.  If the keyword is not in the thesaurus, an empty array
will be returned.  The order of the words returned is determined by the
database which holds them.

Uses global $thesaurus_db_file.

=cut

###############################################################
sub get_related_words {
    my $keyword = shift;
    my %thesaurus_db;
    if (! -e $thesaurus_db_file) {
        &Apache::lonnet::logthis("Attempt to access $thesaurus_db_file ".
                                 "failed because the file does not exist");
        return ();
    }
    if (! tie(%thesaurus_db,'GDBM_File',
              $thesaurus_db_file,&GDBM_READER(),0640)){
        return ();
    } 
    my @Words=();
    my $count=0;
    if (exists($thesaurus_db{$keyword})) {
	# The first element is the number of times
	# the word appears.  We do not need it now.
	my (undef,@RelatedWords) = (split(/:/,$thesaurus_db{$keyword}));
	my (undef,$mostfrequentcount)=split(/\,/,$RelatedWords[0]);
	my $threshold=$mostfrequentcount/10;
        foreach my $possibleword (@RelatedWords) {
            my ($word,$wordcount)=split(/\,/,$possibleword);
            if ($wordcount>$threshold) {
		push(@Words,$word);
                $count++;
                if ($count>10) { last; }
	    }
        }
    }
    untie %thesaurus_db;
    return @Words;
}

=pod

=back

=cut

# -------------------------------------------------------------- Plaintext name
=pod

=head1 User Name Functions

=over 4

=item * &plainname($uname,$udom,$first)

Takes a users logon name and returns it as a string in
"first middle last generation" form 
if $first is set to 'lastname' then it returns it as
'lastname generation, firstname middlename' if their is a lastname

=cut


###############################################################
sub plainname {
    my ($uname,$udom,$first)=@_;
    return if (!defined($uname) || !defined($udom));
    my %names=&getnames($uname,$udom);
    my $name=&Apache::lonnet::format_name($names{'firstname'},
					  $names{'middlename'},
					  $names{'lastname'},
					  $names{'generation'},$first);
    $name=~s/^\s+//;
    $name=~s/\s+$//;
    $name=~s/\s+/ /g;
    if ($name !~ /\S/) { $name=$uname.':'.$udom; }
    return $name;
}

# -------------------------------------------------------------------- Nickname
=pod

=item * &nickname($uname,$udom)

Gets a users name and returns it as a string as

"&quot;nickname&quot;"

if the user has a nickname or

"first middle last generation"

if the user does not

=cut

sub nickname {
    my ($uname,$udom)=@_;
    return if (!defined($uname) || !defined($udom));
    my %names=&getnames($uname,$udom);
    my $name=$names{'nickname'};
    if ($name) {
       $name='&quot;'.$name.'&quot;'; 
    } else {
       $name=$names{'firstname'}.' '.$names{'middlename'}.' '.
	     $names{'lastname'}.' '.$names{'generation'};
       $name=~s/\s+$//;
       $name=~s/\s+/ /g;
    }
    return $name;
}

sub getnames {
    my ($uname,$udom)=@_;
    return if (!defined($uname) || !defined($udom));
    if ($udom eq 'public' && $uname eq 'public') {
	return ('lastname' => &mt('Public'));
    }
    my $id=$uname.':'.$udom;
    my ($names,$cached)=&Apache::lonnet::is_cached_new('namescache',$id);
    if ($cached) {
	return %{$names};
    } else {
	my %loadnames=&Apache::lonnet::get('environment',
                    ['firstname','middlename','lastname','generation','nickname'],
					 $udom,$uname);
	&Apache::lonnet::do_cache_new('namescache',$id,\%loadnames);
	return %loadnames;
    }
}

# -------------------------------------------------------------------- getemails

=pod

=item * &getemails($uname,$udom)

Gets a user's email information and returns it as a hash with keys:
notification, critnotification, permanentemail

For notification and critnotification, values are comma-separated lists 
of e-mail addresses; for permanentemail, value is a single e-mail address.
 

=cut


sub getemails {
    my ($uname,$udom)=@_;
    if ($udom eq 'public' && $uname eq 'public') {
	return;
    }
    if (!$udom) { $udom=$env{'user.domain'}; }
    if (!$uname) { $uname=$env{'user.name'}; }
    my $id=$uname.':'.$udom;
    my ($names,$cached)=&Apache::lonnet::is_cached_new('emailscache',$id);
    if ($cached) {
	return %{$names};
    } else {
	my %loadnames=&Apache::lonnet::get('environment',
                    			   ['notification','critnotification',
					    'permanentemail'],
					   $udom,$uname);
	&Apache::lonnet::do_cache_new('emailscache',$id,\%loadnames);
	return %loadnames;
    }
}

sub flush_email_cache {
    my ($uname,$udom)=@_;
    if (!$udom)  { $udom =$env{'user.domain'}; }
    if (!$uname) { $uname=$env{'user.name'};   }
    return if ($udom eq 'public' && $uname eq 'public');
    my $id=$uname.':'.$udom;
    &Apache::lonnet::devalidate_cache_new('emailscache',$id);
}

# ------------------------------------------------------------------ Screenname

=pod

=item * &screenname($uname,$udom)

Gets a users screenname and returns it as a string

=cut

sub screenname {
    my ($uname,$udom)=@_;
    if ($uname eq $env{'user.name'} &&
	$udom eq $env{'user.domain'}) {return $env{'environment.screenname'};}
    my %names=&Apache::lonnet::get('environment',['screenname'],$udom,$uname);
    return $names{'screenname'};
}


# ------------------------------------------------------------- Message Wrapper

sub messagewrapper {
    my ($link,$username,$domain,$subject,$text)=@_;
    return 
        '<a href="/adm/email?compose=individual&amp;'.
        'recname='.$username.'&amp;recdom='.$domain.
	'&amp;subject='.&escape($subject).'&amp;text='.&escape($text).'" '.
        'title="'.&mt('Send message').'">'.$link.'</a>';
}
# --------------------------------------------------------------- Notes Wrapper

sub noteswrapper {
    my ($link,$un,$do)=@_;
    return 
"<a href='/adm/email?recordftf=retrieve&recname=$un&recdom=$do'>$link</a>";
}
# ------------------------------------------------------------- Aboutme Wrapper

sub aboutmewrapper {
    my ($link,$username,$domain,$target)=@_;
    if (!defined($username)  && !defined($domain)) {
        return;
    }
    return '<a href="/adm/'.$domain.'/'.$username.'/aboutme"'.
	($target?' target="$target"':'').' title="'.&mt("View this user's personal page").'">'.$link.'</a>';
}

# ------------------------------------------------------------ Syllabus Wrapper


sub syllabuswrapper {
    my ($linktext,$coursedir,$domain,$fontcolor)=@_;
    if ($fontcolor) { 
        $linktext='<font color="'.$fontcolor.'">'.$linktext.'</font>'; 
    }
    return qq{<a href="/public/$domain/$coursedir/syllabus">$linktext</a>};
}

sub track_student_link {
    my ($linktext,$sname,$sdom,$target,$start) = @_;
    my $link ="/adm/trackstudent?";
    my $title = 'View recent activity';
    if (defined($sname) && $sname !~ /^\s*$/ &&
        defined($sdom)  && $sdom  !~ /^\s*$/) {
        $link .= "selected_student=$sname:$sdom";
        $title .= ' of this student';
    } 
    if (defined($target) && $target !~ /^\s*$/) {
        $target = qq{target="$target"};
    } else {
        $target = '';
    }
    if ($start) { $link.='&amp;start='.$start; }
    $title = &mt($title);
    $linktext = &mt($linktext);
    return qq{<a href="$link" title="$title" $target>$linktext</a>}.
	&help_open_topic('View_recent_activity');
}

# ===================================================== Display a student photo


sub student_image_tag {
    my ($domain,$user)=@_;
    my $imgsrc=&Apache::lonnet::studentphoto($domain,$user,'jpg');
    if (($imgsrc) && ($imgsrc ne '/adm/lonKaputt/lonlogo_broken.gif')) {
	return '<img src="'.$imgsrc.'" align="right" />';
    } else {
	return '';
    }
}

=pod

=back

=head1 Access .tab File Data

=over 4

=item * &languageids() 

returns list of all language ids

=cut

sub languageids {
    return sort(keys(%language));
}

=pod

=item * &languagedescription() 

returns description of a specified language id

=cut

sub languagedescription {
    my $code=shift;
    return  ($supported_language{$code}?'* ':'').
            $language{$code}.
	    ($supported_language{$code}?' ('.&mt('interface available').')':'');
}

sub plainlanguagedescription {
    my $code=shift;
    return $language{$code};
}

sub supportedlanguagecode {
    my $code=shift;
    return $supported_language{$code};
}

=pod

=item * &copyrightids() 

returns list of all copyrights

=cut

sub copyrightids {
    return sort(keys(%cprtag));
}

=pod

=item * &copyrightdescription() 

returns description of a specified copyright id

=cut

sub copyrightdescription {
    return &mt($cprtag{shift(@_)});
}

=pod

=item * &source_copyrightids() 

returns list of all source copyrights

=cut

sub source_copyrightids {
    return sort(keys(%scprtag));
}

=pod

=item * &source_copyrightdescription() 

returns description of a specified source copyright id

=cut

sub source_copyrightdescription {
    return &mt($scprtag{shift(@_)});
}

=pod

=item * &filecategories() 

returns list of all file categories

=cut

sub filecategories {
    return sort(keys(%category_extensions));
}

=pod

=item * &filecategorytypes() 

returns list of file types belonging to a given file
category

=cut

sub filecategorytypes {
    my ($cat) = @_;
    return @{$category_extensions{lc($cat)}};
}

=pod

=item * &fileembstyle() 

returns embedding style for a specified file type

=cut

sub fileembstyle {
    return $fe{lc(shift(@_))};
}

sub filemimetype {
    return $fm{lc(shift(@_))};
}


sub filecategoryselect {
    my ($name,$value)=@_;
    return &select_form($value,$name,
			'' => &mt('Any category'),
			map { $_,$_ } sort(keys(%category_extensions)));
}

=pod

=item * &filedescription() 

returns description for a specified file type

=cut

sub filedescription {
    my $file_description = $fd{lc(shift())};
    $file_description =~ s:([\[\]]):~$1:g;
    return &mt($file_description);
}

=pod

=item * &filedescriptionex() 

returns description for a specified file type with
extra formatting

=cut

sub filedescriptionex {
    my $ex=shift;
    my $file_description = $fd{lc($ex)};
    $file_description =~ s:([\[\]]):~$1:g;
    return '.'.$ex.' '.&mt($file_description);
}

# End of .tab access
=pod

=back

=cut

# ------------------------------------------------------------------ File Types
sub fileextensions {
    return sort(keys(%fe));
}

# ----------------------------------------------------------- Display Languages
# returns a hash with all desired display languages
#

sub display_languages {
    my %languages=();
    foreach my $lang (&preferred_languages()) {
	$languages{$lang}=1;
    }
    &get_unprocessed_cgi($ENV{'QUERY_STRING'},['displaylanguage']);
    if ($env{'form.displaylanguage'}) {
	foreach my $lang (split(/\s*(\,|\;|\:)\s*/,$env{'form.displaylanguage'})) {
	    $languages{$lang}=1;
        }
    }
    return %languages;
}

sub preferred_languages {
    my @languages=();
    if (($env{'request.role.adv'}) && ($env{'form.languages'})) {
        @languages=(@languages,split(/\s*(\,|\;|\:)\s*/,$env{'form.languages'}));
    }
    if ($env{'course.'.$env{'request.course.id'}.'.languages'}) {
	@languages=(@languages,split(/\s*(\,|\;|\:)\s*/,
	         $env{'course.'.$env{'request.course.id'}.'.languages'}));
    }

    if ($env{'environment.languages'}) {
	@languages=(@languages,
		    split(/\s*(\,|\;|\:)\s*/,$env{'environment.languages'}));
    }
    my $browser=$ENV{'HTTP_ACCEPT_LANGUAGE'};
    if ($browser) {
	my @browser = 
	    map { (split(/\s*;\s*/,$_))[0] } (split(/\s*,\s*/,$browser));
	push(@languages,@browser);
    }

    foreach my $domtype ($env{'user.domain'},$env{'request.role.domain'},
                         $Apache::lonnet::perlvar{'lonDefDomain'}) {
        if ($domtype ne '') {
            my %domdefs = &Apache::lonnet::get_domain_defaults($domtype);
            if ($domdefs{'lang_def'} ne '') {
                push(@languages,$domdefs{'lang_def'});
            }
        }
    }
# turn "en-ca" into "en-ca,en"
    my @genlanguages;
    foreach my $lang (@languages) {
	unless ($lang=~/\w/) { next; }
	push(@genlanguages,$lang);
	if ($lang=~/(\-|\_)/) {
	    push(@genlanguages,(split(/(\-|\_)/,$lang))[0]);
	}
    }
    #uniqueify the languages list
    my %count;
    @genlanguages = map { $count{$_}++ == 0 ? $_ : () } @genlanguages;
    return @genlanguages;
}

sub languages {
    my ($possible_langs) = @_;
    my @preferred_langs = &preferred_languages();
    if (!ref($possible_langs)) {
	if( wantarray ) {
	    return @preferred_langs;
	} else {
	    return $preferred_langs[0];
	}
    }
    my %possibilities = map { $_ => 1 } (@$possible_langs);
    my @preferred_possibilities;
    foreach my $preferred_lang (@preferred_langs) {
	if (exists($possibilities{$preferred_lang})) {
	    push(@preferred_possibilities, $preferred_lang);
	}
    }
    if( wantarray ) {
	return @preferred_possibilities;
    }
    return $preferred_possibilities[0];
}

###############################################################
##               Student Answer Attempts                     ##
###############################################################

=pod

=head1 Alternate Problem Views

=over 4

=item * &get_previous_attempt($symb, $username, $domain, $course,
    $getattempt, $regexp, $gradesub)

Return string with previous attempt on problem. Arguments:

=over 4

=item * $symb: Problem, including path

=item * $username: username of the desired student

=item * $domain: domain of the desired student

=item * $course: Course ID

=item * $getattempt: Leave blank for all attempts, otherwise put
    something

=item * $regexp: if string matches this regexp, the string will be
    sent to $gradesub

=item * $gradesub: routine that processes the string if it matches $regexp

=back

The output string is a table containing all desired attempts, if any.

=cut

sub get_previous_attempt {
  my ($symb,$username,$domain,$course,$getattempt,$regexp,$gradesub)=@_;
  my $prevattempts='';
  no strict 'refs';
  if ($symb) {
    my (%returnhash)=
      &Apache::lonnet::restore($symb,$course,$domain,$username);
    if ($returnhash{'version'}) {
      my %lasthash=();
      my $version;
      for ($version=1;$version<=$returnhash{'version'};$version++) {
        foreach my $key (sort(split(/\:/,$returnhash{$version.':keys'}))) {
	  $lasthash{$key}=$returnhash{$version.':'.$key};
        }
      }
      $prevattempts=&start_data_table().&start_data_table_header_row();
      $prevattempts.='<th>'.&mt('History').'</th>';
      foreach my $key (sort(keys(%lasthash))) {
	my ($ign,@parts) = split(/\./,$key);
	if ($#parts > 0) {
	  my $data=$parts[-1];
	  pop(@parts);
	  $prevattempts.='<th>'.&mt('Part ').join('.',@parts).'<br />'.$data.'&nbsp;</th>';
	} else {
	  if ($#parts == 0) {
	    $prevattempts.='<th>'.$parts[0].'</th>';
	  } else {
	    $prevattempts.='<th>'.$ign.'</th>';
	  }
	}
      }
      $prevattempts.=&end_data_table_header_row();
      if ($getattempt eq '') {
	for ($version=1;$version<=$returnhash{'version'};$version++) {
	  $prevattempts.=&start_data_table_row().
	      '<td>'.&mt('Transaction [_1]',$version).'</td>';
	    foreach my $key (sort(keys(%lasthash))) {
		my $value = &format_previous_attempt_value($key,
							   $returnhash{$version.':'.$key});
		$prevattempts.='<td>'.$value.'&nbsp;</td>';   
	    }
	  $prevattempts.=&end_data_table_row();
	 }
      }
      $prevattempts.=&start_data_table_row().'<td>'.&mt('Current').'</td>';
      foreach my $key (sort(keys(%lasthash))) {
	my $value = &format_previous_attempt_value($key,$lasthash{$key});
	if ($key =~/$regexp$/ && (defined &$gradesub)) {$value = &$gradesub($value)}
	$prevattempts.='<td>'.$value.'&nbsp;</td>';
      }
      $prevattempts.= &end_data_table_row().&end_data_table();
    } else {
      $prevattempts=
	  &start_data_table().&start_data_table_row().
	  '<td>'.&mt('Nothing submitted - no attempts.').'</td>'.
	  &end_data_table_row().&end_data_table();
    }
  } else {
    $prevattempts=
	  &start_data_table().&start_data_table_row().
	  '<td>'.&mt('No data.').'</td>'.
	  &end_data_table_row().&end_data_table();
  }
}

sub format_previous_attempt_value {
    my ($key,$value) = @_;
    if ($key =~ /timestamp/) {
	$value = &Apache::lonlocal::locallocaltime($value);
    } elsif (ref($value) eq 'ARRAY') {
	$value = '('.join(', ', @{ $value }).')';
    } else {
	$value = &unescape($value);
    }
    return $value;
}


sub relative_to_absolute {
    my ($url,$output)=@_;
    my $parser=HTML::TokeParser->new(\$output);
    my $token;
    my $thisdir=$url;
    my @rlinks=();
    while ($token=$parser->get_token) {
	if ($token->[0] eq 'S') {
	    if ($token->[1] eq 'a') {
		if ($token->[2]->{'href'}) {
		    $rlinks[$#rlinks+1]=$token->[2]->{'href'};
		}
	    } elsif ($token->[1] eq 'img' || $token->[1] eq 'embed' ) {
		$rlinks[$#rlinks+1]=$token->[2]->{'src'};
	    } elsif ($token->[1] eq 'base') {
		$thisdir=$token->[2]->{'href'};
	    }
	}
    }
    $thisdir=~s-/[^/]*$--;
    foreach my $link (@rlinks) {
	unless (($link=~/^http:\/\//i) ||
		($link=~/^\//) ||
		($link=~/^javascript:/i) ||
		($link=~/^mailto:/i) ||
		($link=~/^\#/)) {
	    my $newlocation=&Apache::lonnet::hreflocation($thisdir,$link);
	    $output=~s/(\"|\'|\=\s*)\Q$link\E(\"|\'|\s|\>)/$1$newlocation$2/;
	}
    }
# -------------------------------------------------- Deal with Applet codebases
    $output=~s/(\<applet[^\>]+)(codebase\=[^\S\>]+)*([^\>]*)\>/$1.($2?$2:' codebase="'.$thisdir.'"').$3.'>'/gei;
    return $output;
}

=pod

=item * &get_student_view()

show a snapshot of what student was looking at

=cut

sub get_student_view {
  my ($symb,$username,$domain,$courseid,$target,$moreenv) = @_;
  my ($map,$id,$feedurl) = &Apache::lonnet::decode_symb($symb);
  my (%form);
  my @elements=('symb','courseid','domain','username');
  foreach my $element (@elements) {
      $form{'grade_'.$element}=eval '$'.$element #'
  }
  if (defined($moreenv)) {
      %form=(%form,%{$moreenv});
  }
  if (defined($target)) { $form{'grade_target'} = $target; }
  $feedurl=&Apache::lonnet::clutter($feedurl);
  my ($userview,$response)=&Apache::lonnet::ssi_body($feedurl,%form);
  $userview=~s/\<body[^\>]*\>//gi;
  $userview=~s/\<\/body\>//gi;
  $userview=~s/\<html\>//gi;
  $userview=~s/\<\/html\>//gi;
  $userview=~s/\<head\>//gi;
  $userview=~s/\<\/head\>//gi;
  $userview=~s/action\s*\=/would_be_action\=/gi;
  $userview=&relative_to_absolute($feedurl,$userview);
  if (wantarray) {
     return ($userview,$response);
  } else {
     return $userview;
  }
}

sub get_student_view_with_retries {
  my ($symb,$retries,$username,$domain,$courseid,$target,$moreenv) = @_;

    my $ok = 0;                 # True if we got a good response.
    my $content;
    my $response;

    # Try to get the student_view done. within the retries count:
    
    do {
         ($content, $response) = &get_student_view($symb,$username,$domain,$courseid,$target,$moreenv);
         $ok      = $response->is_success;
         if (!$ok) {
            &Apache::lonnet::logthis("Failed get_student_view_with_retries on $symb: ".$response->is_success.', '.$response->code.', '.$response->message);
         }
         $retries--;
    } while (!$ok && ($retries > 0));
    
    if (!$ok) {
       $content = '';          # On error return an empty content.
    }
    if (wantarray) {
       return ($content, $response);
    } else {
       return $content;
    }
}

=pod

=item * &get_student_answers() 

show a snapshot of how student was answering problem

=cut

sub get_student_answers {
  my ($symb,$username,$domain,$courseid,%form) = @_;
  my ($map,$id,$feedurl) = &Apache::lonnet::decode_symb($symb);
  my (%moreenv);
  my @elements=('symb','courseid','domain','username');
  foreach my $element (@elements) {
    $moreenv{'grade_'.$element}=eval '$'.$element #'
  }
  $moreenv{'grade_target'}='answer';
  %moreenv=(%form,%moreenv);
  $feedurl = &Apache::lonnet::clutter($feedurl);
  my $userview=&Apache::lonnet::ssi($feedurl,%moreenv);
  return $userview;
}

=pod

=item * &submlink()

Inputs: $text $uname $udom $symb $target

Returns: A link to grades.pm such as to see the SUBM view of a student

=cut

###############################################
sub submlink {
    my ($text,$uname,$udom,$symb,$target)=@_;
    if (!($uname && $udom)) {
	(my $cursymb, my $courseid,$udom,$uname)=
	    &Apache::lonnet::whichuser($symb);
	if (!$symb) { $symb=$cursymb; }
    }
    if (!$symb) { $symb=&Apache::lonnet::symbread(); }
    $symb=&escape($symb);
    if ($target) { $target="target=\"$target\""; }
    return '<a href="/adm/grades?&command=submission&'.
	'symb='.$symb.'&student='.$uname.
	'&userdom='.$udom.'" '.$target.'>'.$text.'</a>';
}
##############################################

=pod

=item * &pgrdlink()

Inputs: $text $uname $udom $symb $target

Returns: A link to grades.pm such as to see the PGRD view of a student

=cut

###############################################
sub pgrdlink {
    my $link=&submlink(@_);
    $link=~s/(&command=submission)/$1&showgrading=yes/;
    return $link;
}
##############################################

=pod

=item * &pprmlink()

Inputs: $text $uname $udom $symb $target

Returns: A link to parmset.pm such as to see the PPRM view of a
student and a specific resource

=cut

###############################################
sub pprmlink {
    my ($text,$uname,$udom,$symb,$target)=@_;
    if (!($uname && $udom)) {
	(my $cursymb, my $courseid,$udom,$uname)=
	    &Apache::lonnet::whichuser($symb);
	if (!$symb) { $symb=$cursymb; }
    }
    if (!$symb) { $symb=&Apache::lonnet::symbread(); }
    $symb=&escape($symb);
    if ($target) { $target="target=\"$target\""; }
    return '<a href="/adm/parmset?command=set&amp;'.
	'symb='.$symb.'&amp;uname='.$uname.
	'&amp;udom='.$udom.'" '.$target.'>'.$text.'</a>';
}
##############################################

=pod

=back

=cut

###############################################


sub timehash {
    my @ltime=localtime(shift);
    return ( 'seconds' => $ltime[0],
             'minutes' => $ltime[1],
             'hours'   => $ltime[2],
             'day'     => $ltime[3],
             'month'   => $ltime[4]+1,
             'year'    => $ltime[5]+1900,
             'weekday' => $ltime[6],
             'dayyear' => $ltime[7]+1,
             'dlsav'   => $ltime[8] );
}

sub utc_string {
    my ($date)=@_;
    return strftime("%Y%m%dT%H%M%SZ",gmtime($date));
}

sub maketime {
    my %th=@_;
    return POSIX::mktime(
        ($th{'seconds'},$th{'minutes'},$th{'hours'},
         $th{'day'},$th{'month'}-1,$th{'year'}-1900,0,0,-1));
}

#########################################

sub findallcourses {
    my ($roles,$uname,$udom) = @_;
    my %roles;
    if (ref($roles)) { %roles = map { $_ => 1 } @{$roles}; }
    my %courses;
    my $now=time;
    if (!defined($uname)) {
        $uname = $env{'user.name'};
    }
    if (!defined($udom)) {
        $udom = $env{'user.domain'};
    }
    if (($uname ne $env{'user.name'}) || ($udom ne $env{'user.domain'})) {
        my %roleshash = &Apache::lonnet::dump('roles',$udom,$uname);
        if (!%roles) {
            %roles = (
                       cc => 1,
                       in => 1,
                       ep => 1,
                       ta => 1,
                       cr => 1,
                       st => 1,
             );
        }
        foreach my $entry (keys(%roleshash)) {
            my ($trole,$tend,$tstart) = split(/_/,$roleshash{$entry});
            if ($trole =~ /^cr/) { 
                next if (!exists($roles{$trole}) && !exists($roles{'cr'}));
            } else {
                next if (!exists($roles{$trole}));
            }
            if ($tend) {
                next if ($tend < $now);
            }
            if ($tstart) {
                next if ($tstart > $now);
            }
            my ($cdom,$cnum,$sec,$cnumpart,$secpart,$role,$realsec);
            (undef,$cdom,$cnumpart,$secpart) = split(/\//,$entry);
            if ($secpart eq '') {
                ($cnum,$role) = split(/_/,$cnumpart); 
                $sec = 'none';
                $realsec = '';
            } else {
                $cnum = $cnumpart;
                ($sec,$role) = split(/_/,$secpart);
                $realsec = $sec;
            }
            $courses{$cdom.'_'.$cnum}{$sec} = $trole.'/'.$cdom.'/'.$cnum.'/'.$realsec;
        }
    } else {
        foreach my $key (keys(%env)) {
	    if ( $key=~m{^user\.role\.(\w+)\./($match_domain)/($match_courseid)/?(\w*)$} ||
                 $key=~m{^user\.role\.(cr/$match_domain/$match_username/\w+)\./($match_domain)/($match_courseid)/?(\w*)$}) {
	        my ($role,$cdom,$cnum,$sec) = ($1,$2,$3,$4);
	        next if ($role eq 'ca' || $role eq 'aa');
	        next if (%roles && !exists($roles{$role}));
	        my ($starttime,$endtime)=split(/\./,$env{$key});
                my $active=1;
                if ($starttime) {
		    if ($now<$starttime) { $active=0; }
                }
                if ($endtime) {
                    if ($now>$endtime) { $active=0; }
                }
                if ($active) {
                    if ($sec eq '') {
                        $sec = 'none';
                    }
                    $courses{$cdom.'_'.$cnum}{$sec} = 
                                     $role.'/'.$cdom.'/'.$cnum.'/'.$sec;
                }
            }
        }
    }
    return %courses;
}

###############################################

sub blockcheck {
    my ($setters,$activity,$uname,$udom) = @_;

    if (!defined($udom)) {
        $udom = $env{'user.domain'};
    }
    if (!defined($uname)) {
        $uname = $env{'user.name'};
    }

    # If uname and udom are for a course, check for blocks in the course.

    if (&Apache::lonnet::is_course($udom,$uname)) {
        my %records = &Apache::lonnet::dump('comm_block',$udom,$uname);
        my ($startblock,$endblock)=&get_blocks($setters,$activity,$udom,$uname);
        return ($startblock,$endblock);
    }

    my $startblock = 0;
    my $endblock = 0;
    my %live_courses = &findallcourses(undef,$uname,$udom);

    # If uname is for a user, and activity is course-specific, i.e.,
    # boards, chat or groups, check for blocking in current course only.

    if (($activity eq 'boards' || $activity eq 'chat' ||
         $activity eq 'groups') && ($env{'request.course.id'})) {
        foreach my $key (keys(%live_courses)) {
            if ($key ne $env{'request.course.id'}) {
                delete($live_courses{$key});
            }
        }
    }

    my $otheruser = 0;
    my %own_courses;
    if ((($uname ne $env{'user.name'})) || ($udom ne $env{'user.domain'})) {
        # Resource belongs to user other than current user.
        $otheruser = 1;
        # Gather courses for current user
        %own_courses = 
            &findallcourses(undef,$env{'user.name'},$env{'user.domain'});
    }

    # Gather active course roles - course coordinator, instructor, 
    # exam proctor, ta, student, or custom role.

    foreach my $course (keys(%live_courses)) {
        my ($cdom,$cnum);
        if ((defined($env{'course.'.$course.'.domain'})) && (defined($env{'course.'.$course.'.num'}))) {
            $cdom = $env{'course.'.$course.'.domain'};
            $cnum = $env{'course.'.$course.'.num'};
        } else {
            ($cdom,$cnum) = split(/_/,$course); 
        }
        my $no_ownblock = 0;
        my $no_userblock = 0;
        if ($otheruser && $activity ne 'com') {
            # Check if current user has 'evb' priv for this
            if (defined($own_courses{$course})) {
                foreach my $sec (keys(%{$own_courses{$course}})) {
                    my $checkrole = 'cm./'.$cdom.'/'.$cnum;
                    if ($sec ne 'none') {
                        $checkrole .= '/'.$sec;
                    }
                    if (&Apache::lonnet::allowed('evb',undef,undef,$checkrole)) {
                        $no_ownblock = 1;
                        last;
                    }
                }
            }
            # if they have 'evb' priv and are currently not playing student
            next if (($no_ownblock) &&
                 ($env{'request.role'} !~ m{^st\./$cdom/$cnum}));
        }
        foreach my $sec (keys(%{$live_courses{$course}})) {
            my $checkrole = 'cm./'.$cdom.'/'.$cnum;
            if ($sec ne 'none') {
                $checkrole .= '/'.$sec;
            }
            if ($otheruser) {
                # Resource belongs to user other than current user.
                # Assemble privs for that user, and check for 'evb' priv.
                my ($trole,$tdom,$tnum,$tsec);
                my $entry = $live_courses{$course}{$sec};
                if ($entry =~ /^cr/) {
                    ($trole,$tdom,$tnum,$tsec) = 
                      ($entry =~ m|^(cr/$match_domain/$match_username/\w+)\./($match_domain)/($match_username)/?(\w*)$|);
                } else {
                    ($trole,$tdom,$tnum,$tsec) = split(/\//,$entry);
                }
                my ($spec,$area,$trest,%allroles,%userroles);
                $area = '/'.$tdom.'/'.$tnum;
                $trest = $tnum;
                if ($tsec ne '') {
                    $area .= '/'.$tsec;
                    $trest .= '/'.$tsec;
                }
                $spec = $trole.'.'.$area;
                if ($trole =~ /^cr/) {
                    &Apache::lonnet::custom_roleprivs(\%allroles,$trole,
                                                      $tdom,$spec,$trest,$area);
                } else {
                    &Apache::lonnet::standard_roleprivs(\%allroles,$trole,
                                                       $tdom,$spec,$trest,$area);
                }
                my ($author,$adv) = &Apache::lonnet::set_userprivs(\%userroles,\%allroles);
                if ($userroles{'user.priv.'.$checkrole} =~ /evb\&([^\:]*)/) {
                    if ($1) {
                        $no_userblock = 1;
                        last;
                    }
                }
            } else {
                # Resource belongs to current user
                # Check for 'evb' priv via lonnet::allowed().
                if (&Apache::lonnet::allowed('evb',undef,undef,$checkrole)) {
                    $no_ownblock = 1;
                    last;
                }
            }
        }
        # if they have the evb priv and are currently not playing student
        next if (($no_ownblock) &&
                 ($env{'request.role'} !~ m{^st\./\Q$cdom\E/\Q$cnum\E}));
        next if ($no_userblock);

        # Retrieve blocking times and identity of blocker for course
        # of specified user, unless user has 'evb' privilege.
        
        my ($start,$end)=&get_blocks($setters,$activity,$cdom,$cnum);
        if (($start != 0) && 
            (($startblock == 0) || ($startblock > $start))) {
            $startblock = $start;
        }
        if (($end != 0)  &&
            (($endblock == 0) || ($endblock < $end))) {
            $endblock = $end;
        }
    }
    return ($startblock,$endblock);
}

sub get_blocks {
    my ($setters,$activity,$cdom,$cnum) = @_;
    my $startblock = 0;
    my $endblock = 0;
    my $course = $cdom.'_'.$cnum;
    $setters->{$course} = {};
    $setters->{$course}{'staff'} = [];
    $setters->{$course}{'times'} = [];
    my %records = &Apache::lonnet::dump('comm_block',$cdom,$cnum);
    foreach my $record (keys(%records)) {
        my ($start,$end) = ($record =~ m/^(\d+)____(\d+)$/);
        if ($start <= time && $end >= time) {
            my ($staff_name,$staff_dom,$title,$blocks) =
                &parse_block_record($records{$record});
            if ($blocks->{$activity} eq 'on') {
                push(@{$$setters{$course}{'staff'}},[$staff_name,$staff_dom]);
                push(@{$$setters{$course}{'times'}}, [$start,$end]);
                if ( ($startblock == 0) || ($startblock > $start) ) {
                    $startblock = $start;
                }
                if ( ($endblock == 0) || ($endblock < $end) ) {
                    $endblock = $end;
                }
            }
        }
    }
    return ($startblock,$endblock);
}

sub parse_block_record {
    my ($record) = @_;
    my ($setuname,$setudom,$title,$blocks);
    if (ref($record) eq 'HASH') {
        ($setuname,$setudom) = split(/:/,$record->{'setter'});
        $title = &unescape($record->{'event'});
        $blocks = $record->{'blocks'};
    } else {
        my @data = split(/:/,$record,3);
        if (scalar(@data) eq 2) {
            $title = $data[1];
            ($setuname,$setudom) = split(/@/,$data[0]);
        } else {
            ($setuname,$setudom,$title) = @data;
        }
        $blocks = { 'com' => 'on' };
    }
    return ($setuname,$setudom,$title,$blocks);
}

sub build_block_table {
    my ($startblock,$endblock,$setters) = @_;
    my %lt = &Apache::lonlocal::texthash(
        'cacb' => 'Currently active communication blocks',
        'cour' => 'Course',
        'dura' => 'Duration',
        'blse' => 'Block set by'
    );
    my $output;
    $output = '<br />'.$lt{'cacb'}.':<br />';
    $output .= &start_data_table();
    $output .= '
<tr>
 <th>'.$lt{'cour'}.'</th>
 <th>'.$lt{'dura'}.'</th>
 <th>'.$lt{'blse'}.'</th>
</tr>
';
    foreach my $course (keys(%{$setters})) {
        my %courseinfo=&Apache::lonnet::coursedescription($course);
        for (my $i=0; $i<@{$$setters{$course}{staff}}; $i++) {
            my ($uname,$udom) = @{$$setters{$course}{staff}[$i]};
            my $fullname = &plainname($uname,$udom);
            if (defined($env{'user.name'}) && defined($env{'user.domain'})
                && $env{'user.name'} ne 'public' 
                && $env{'user.domain'} ne 'public') {
                $fullname = &aboutmewrapper($fullname,$uname,$udom);
            }
            my ($openblock,$closeblock) = @{$$setters{$course}{times}[$i]};
            $openblock = &Apache::lonlocal::locallocaltime($openblock);
            $closeblock= &Apache::lonlocal::locallocaltime($closeblock);
            $output .= &Apache::loncommon::start_data_table_row().
                       '<td>'.$courseinfo{'description'}.'</td>'.
                       '<td>'.$openblock.' to '.$closeblock.'</td>'.
                       '<td>'.$fullname.'</td>'.
                        &Apache::loncommon::end_data_table_row();
        }
    }
    $output .= &end_data_table();
}

sub blocking_status {
    my ($activity,$uname,$udom) = @_;
    my %setters;
    my ($blocked,$output,$ownitem,$is_course);
    my ($startblock,$endblock)=&blockcheck(\%setters,$activity,$uname,$udom);
    if ($startblock && $endblock) {
        $blocked = 1;
        if (wantarray) {
            my $category;
            if ($activity eq 'boards') {
                $category = 'Discussion posts in this course';
            } elsif ($activity eq 'blogs') {
                $category = 'Blogs';
            } elsif ($activity eq 'port') {
                if (defined($uname) && defined($udom)) {
                    if ($uname eq $env{'user.name'} &&
                        $udom eq $env{'user.domain'}) {
                        $ownitem = 1;
                    }
                }
                $is_course = &Apache::lonnet::is_course($udom,$uname);
                if ($ownitem) { 
                    $category = 'Your portfolio files';  
                } elsif ($is_course) {
                    my $coursedesc;
                    foreach my $course (keys(%setters)) {
                        my %courseinfo =
                             &Apache::lonnet::coursedescription($course);
                        $coursedesc = $courseinfo{'description'};
                    }
                    $category = "Group files in the course '$coursedesc'";
                } else {
                    $category = 'Portfolio files belonging to ';
                    if ($env{'user.name'} eq 'public' && 
                        $env{'user.domain'} eq 'public') {
                        $category .= &plainname($uname,$udom);
                    } else {
                        $category .= &aboutmewrapper(&plainname($uname,$udom),$uname,$udom);  
                    }
                }
            } elsif ($activity eq 'groups') {
                $category = 'Groups in this course';
            }
            my $showstart = &Apache::lonlocal::locallocaltime($startblock);
            my $showend = &Apache::lonlocal::locallocaltime($endblock);
            $output = '<br />'.&mt('[_1] will be inaccessible between [_2] and [_3] because communication is being blocked.',$category,$showstart,$showend).'<br />';
            if (!($activity eq 'port' && !($ownitem) && !($is_course))) { 
                $output .= &build_block_table($startblock,$endblock,\%setters);
            }
        }
    }
    if (wantarray) {
        return ($blocked,$output);
    } else {
        return $blocked;
    }
}

###############################################

=pod

=head1 Domain Template Functions

=over 4

=item * &determinedomain()

Inputs: $domain (usually will be undef)

Returns: Determines which domain should be used for designs

=cut

###############################################
sub determinedomain {
    my $domain=shift;
    if (! $domain) {
        # Determine domain if we have not been given one
        $domain = $Apache::lonnet::perlvar{'lonDefDomain'};
        if ($env{'user.domain'}) { $domain=$env{'user.domain'}; }
        if ($env{'request.role.domain'}) { 
            $domain=$env{'request.role.domain'}; 
        }
    }
    return $domain;
}
###############################################

sub devalidate_domconfig_cache {
    my ($udom)=@_;
    &Apache::lonnet::devalidate_cache_new('domainconfig',$udom);
}

# ---------------------- Get domain configuration for a domain
sub get_domainconf {
    my ($udom) = @_;
    my $cachetime=1800;
    my ($result,$cached)=&Apache::lonnet::is_cached_new('domainconfig',$udom);
    if (defined($cached)) { return %{$result}; }

    my %domconfig = &Apache::lonnet::get_dom('configuration',
					     ['login','rolecolors'],$udom);
    my (%designhash,%legacy);
    if (keys(%domconfig) > 0) {
        if (ref($domconfig{'login'}) eq 'HASH') {
            if (keys(%{$domconfig{'login'}})) {
                foreach my $key (keys(%{$domconfig{'login'}})) {
                    $designhash{$udom.'.login.'.$key}=$domconfig{'login'}{$key};
                }
            } else {
                $legacy{'login'} = 1;
            }
        } else {
            $legacy{'login'} = 1;
        }
        if (ref($domconfig{'rolecolors'}) eq 'HASH') {
            if (keys(%{$domconfig{'rolecolors'}})) {
                foreach my $role (keys(%{$domconfig{'rolecolors'}})) {
                    if (ref($domconfig{'rolecolors'}{$role}) eq 'HASH') {
                        foreach my $item (keys(%{$domconfig{'rolecolors'}{$role}})) {
                            $designhash{$udom.'.'.$role.'.'.$item}=$domconfig{'rolecolors'}{$role}{$item};
                        }
                    }
                }
            } else {
                $legacy{'rolecolors'} = 1;
            }
        } else {
            $legacy{'rolecolors'} = 1;
        }
        if (keys(%legacy) > 0) {
            my %legacyhash = &get_legacy_domconf($udom);
            foreach my $item (keys(%legacyhash)) {
                if ($item =~ /^\Q$udom\E\.login/) {
                    if ($legacy{'login'}) { 
                        $designhash{$item} = $legacyhash{$item};
                    }
                } else {
                    if ($legacy{'rolecolors'}) {
                        $designhash{$item} = $legacyhash{$item};
                    }
                }
            }
        }
    } else {
        %designhash = &get_legacy_domconf($udom); 
    }
    &Apache::lonnet::do_cache_new('domainconfig',$udom,\%designhash,
				  $cachetime);
    return %designhash;
}

sub get_legacy_domconf {
    my ($udom) = @_;
    my %legacyhash;
    my $designdir=$Apache::lonnet::perlvar{'lonTabDir'}.'/lonDomColors';
    my $designfile =  $designdir.'/'.$udom.'.tab';
    if (-e $designfile) {
        if ( open (my $fh,"<$designfile") ) {
            while (my $line = <$fh>) {
                next if ($line =~ /^\#/);
                chomp($line);
                my ($key,$val)=(split(/\=/,$line));
                if ($val) { $legacyhash{$udom.'.'.$key}=$val; }
            }
            close($fh);
        }
    }
    if (-e '/home/httpd/html/adm/lonDomLogos/'.$udom.'.gif') {
        $legacyhash{$udom.'.login.domlogo'} = "/adm/lonDomLogos/$udom.gif";
    }
    return %legacyhash;
}

=pod

=item * &domainlogo()

Inputs: $domain (usually will be undef)

Returns: A link to a domain logo, if the domain logo exists.
If the domain logo does not exist, a description of the domain.

=cut

###############################################
sub domainlogo {
    my $domain = &determinedomain(shift);
    my %designhash = &get_domainconf($domain);    
    # See if there is a logo
    if ($designhash{$domain.'.login.domlogo'} ne '') {
        my $imgsrc = $designhash{$domain.'.login.domlogo'};
        if ($imgsrc =~ m{^/(adm|res)/}) {
	    if ($imgsrc =~ m{^/res/}) {
		my $local_name = &Apache::lonnet::filelocation('',$imgsrc);
		&Apache::lonnet::repcopy($local_name);
	    }
	   $imgsrc = &lonhttpdurl($imgsrc);
        } 
        return '<img src="'.$imgsrc.'" alt="'.$domain.'" />';
    } elsif (defined(&Apache::lonnet::domain($domain,'description'))) {
        return &Apache::lonnet::domain($domain,'description');
    } else {
        return '';
    }
}
##############################################

=pod

=item * &designparm()

Inputs: $which parameter; $domain (usually will be undef)

Returns: value of designparamter $which

=cut


##############################################
sub designparm {
    my ($which,$domain)=@_;
    if ($env{'browser.blackwhite'} eq 'on') {
	if ($which=~/\.(font|alink|vlink|link|textcol)$/) {
	    return '#000000';
	}
	if ($which=~/\.(pgbg|sidebg|bgcol)$/) {
	    return '#FFFFFF';
	}
	if ($which=~/\.tabbg$/) {
	    return '#CCCCCC';
	}
    }
    if (exists($env{'environment.color.'.$which})) {
	return $env{'environment.color.'.$which};
    }
    $domain=&determinedomain($domain);
    my %domdesign = &get_domainconf($domain);
    my $output;
    if ($domdesign{$domain.'.'.$which} ne '') {
	$output = $domdesign{$domain.'.'.$which};
    } else {
        $output = $defaultdesign{$which};
    }
    if (($which =~ /^(student|coordinator|author|admin)\.img$/) ||
        ($which =~ /login\.(img|logo|domlogo|login)/)) {
        if ($output =~ m{^/(adm|res)/}) {
	    if ($output =~ m{^/res/}) {
		my $local_name = &Apache::lonnet::filelocation('',$output);
		&Apache::lonnet::repcopy($local_name);
	    }
            $output = &lonhttpdurl($output);
        }
    }
    return $output;
}

###############################################
###############################################

=pod

=back

=head1 HTML Helpers

=over 4

=item * &bodytag()

Returns a uniform header for LON-CAPA web pages.

Inputs: 

=over 4

=item * $title, A title to be displayed on the page.

=item * $function, the current role (can be undef).

=item * $addentries, extra parameters for the <body> tag.

=item * $bodyonly, if defined, only return the <body> tag.

=item * $domain, if defined, force a given domain.

=item * $forcereg, if page should register as content page (relevant for 
            text interface only)

=item * $customtitle, alternate text to use instead of $title
                      in the title box that appears, this text
                      is not auto translated like the $title is

=item * $notopbar, if true, keep the 'what is this' info but remove the
                   navigational links

=item * $bgcolor, used to override the bgcolor on a webpage to a specific value

=item * $notitle, if true keep the nav controls, but remove the title bar

=item * $no_inline_link, if true and in remote mode, don't show the 
         'Switch To Inline Menu' link

=item * $args, optional argument valid values are
            no_auto_mt_title -> prevents &mt()ing the title arg
            inherit_jsmath -> when creating popup window in a page,
                              should it have jsmath forced on by the
                              current page

=back

Returns: A uniform header for LON-CAPA web pages.  
If $bodyonly is nonzero, a string containing a <body> tag will be returned.
If $bodyonly is undef or zero, an html string containing a <body> tag and 
other decorations will be returned.

=cut

sub bodytag {
    my ($title,$function,$addentries,$bodyonly,$domain,$forcereg,$customtitle,
	$notopbar,$bgcolor,$notitle,$no_inline_link,$args)=@_;

    if (!$args->{'no_auto_mt_title'}) { $title = &mt($title); }

    $function = &get_users_function() if (!$function);
    my $img =    &designparm($function.'.img',$domain);
    my $font =   &designparm($function.'.font',$domain);
    my $pgbg   = $bgcolor || &designparm($function.'.pgbg',$domain);

    my %design = ( 'style'   => 'margin-top: 0px',
		   'bgcolor' => $pgbg,
		   'text'    => $font,
                   'alink'   => &designparm($function.'.alink',$domain),
		   'vlink'   => &designparm($function.'.vlink',$domain),
		   'link'    => &designparm($function.'.link',$domain),);
    @design{keys(%$addentries)} = @$addentries{keys(%$addentries)}; 

 # role and realm
    my ($role,$realm) = split(/\./,$env{'request.role'},2);
    if ($role  eq 'ca') {
        my ($rdom,$rname) = ($realm =~ m{^/($match_domain)/($match_username)$});
        $realm = &plainname($rname,$rdom);
    } 
# realm
    if ($env{'request.course.id'}) {
        if ($env{'request.role'} !~ /^cr/) {
            $role = &Apache::lonnet::plaintext($role,&course_type());
        }
	$realm = $env{'course.'.$env{'request.course.id'}.'.description'};
    } else {
        $role = &Apache::lonnet::plaintext($role);
    }

    if (!$realm) { $realm='&nbsp;'; }
# Set messages
    my $messages=&domainlogo($domain);

    my $extra_body_attr = &make_attr_string($forcereg,\%design);

# construct main body tag
    my $bodytag = "<body $extra_body_attr>".
	&Apache::lontexconvert::init_math_support($args->{'inherit_jsmath'});

    if ($bodyonly) {
        return $bodytag;
    } elsif ($env{'browser.interface'} eq 'textual') {
# Accessibility
          
	$bodytag.=&Apache::lonmenu::menubuttons($forcereg,$forcereg);
	if (!$notitle) {
	    $bodytag.='<h1>LON-CAPA: '.$title.'</h1>';
	}
	return $bodytag;
    }

    my $name = &plainname($env{'user.name'},$env{'user.domain'});
    if ($env{'user.name'} eq 'public' && $env{'user.domain'} eq 'public') {
	undef($role);
    } else {
	$name = &aboutmewrapper($name,$env{'user.name'},$env{'user.domain'});
    }
    
    my $roleinfo=(<<ENDROLE);
<td class="LC_title_bar_who">
<div class="LC_title_bar_name">
    $name
    &nbsp;
</div>
<div class="LC_title_bar_role">
$role&nbsp;
</div>
<div class="LC_title_bar_realm">
$realm&nbsp;
</div>
</td>
ENDROLE

    my $titleinfo = '<span class="LC_title_bar_title">'.$title.'</span>';
    if ($customtitle) {
        $titleinfo = $customtitle;
    }
    #
    # Extra info if you are the DC
    my $dc_info = '';
    if ($env{'user.adv'} && exists($env{'user.role.dc./'.
                        $env{'course.'.$env{'request.course.id'}.
                                 '.domain'}.'/'})) {
        my $cid = $env{'request.course.id'};
        $dc_info.= $cid.' '.$env{'course.'.$cid.'.internal.coursecode'};
        $dc_info =~ s/\s+$//;
        $dc_info = '('.$dc_info.')';
    }

    if (($env{'environment.remote'} eq 'off') || ($args->{'suppress_header_logos'})) {
        # No Remote
	if ($env{'request.state'} eq 'construct') {
	    $forcereg=1;
	}

	if (!$customtitle && $env{'request.state'} eq 'construct') {
	    # this is for resources; directories have customtitle, and crumbs
            # and select recent are created in lonpubdir.pm  
	    my ($uname,$thisdisfn)=
		($env{'request.filename'} =~ m|^/home/([^/]+)/public_html/(.*)|);
	    my $formaction='/priv/'.$uname.'/'.$thisdisfn;
	    $formaction=~s/\/+/\//g;

	    my $parentpath = '';
	    my $lastitem = '';
	    if ($thisdisfn =~ m-(.+/)([^/]*)$-) {
		$parentpath = $1;
		$lastitem = $2;
	    } else {
		$lastitem = $thisdisfn;
	    }
	    $titleinfo = 
		&Apache::loncommon::help_open_menu('','',3,'Authoring')
		.'<b>'.&mt('Construction Space').'</b>:&nbsp;'
		.'<form name="dirs" method="post" action="'.$formaction
		.'" target="_top"><tt><b>'
		.&Apache::lonhtmlcommon::crumbs($uname.'/'.$parentpath,'_top','/priv','','+1',1)."<font size=\"+1\">$lastitem</font></b></tt><br />"
		.&Apache::lonhtmlcommon::select_recent('construct','recent','this.form.action=this.form.recent.value;this.form.submit()')
		.'</form>'
		.&Apache::lonmenu::constspaceform();
        }

        my $titletable;
	if (!$notitle) {
	    $titletable =
		'<table id="LC_title_bar">'.
                         "<tr><td> $titleinfo $dc_info</td>".$roleinfo.
			 '</tr></table>';
	}
	if ($notopbar) {
	    $bodytag .= $titletable;
	} else {
	    if ($env{'request.state'} eq 'construct') {
                $bodytag .= &Apache::lonmenu::menubuttons($forcereg,$forcereg,
							  $titletable);
            } else {
                $bodytag .= &Apache::lonmenu::menubuttons($forcereg,$forcereg).
		    $titletable;
            }
        }
        return $bodytag;
    }

#
# Top frame rendering, Remote is up
#

    my $imgsrc = $img;
    if ($img =~ /^\/adm/) {
        $imgsrc = &lonhttpdurl($img);
    }
    my $upperleft='<img src="'.$imgsrc.'" alt="'.$function.'" />';

    # Explicit link to get inline menu
    my $menu= ($no_inline_link?''
	       :'<br /><a href="/adm/remote?action=collapse">'.&mt('Switch to Inline Menu Mode').'</a>');
    #
    if ($notitle) {
	return $bodytag;
    }
    return(<<ENDBODY);
$bodytag
<table id="LC_title_bar" class="LC_with_remote">
<tr><td class="LC_title_bar_role_logo">$upperleft</td>
    <td class="LC_title_bar_domain_logo">$messages&nbsp;</td>
</tr>
<tr><td>$titleinfo $dc_info $menu</td>
$roleinfo
</tr>
</table>
ENDBODY
}

sub make_attr_string {
    my ($register,$attr_ref) = @_;

    if ($attr_ref && !ref($attr_ref)) {
	die("addentries Must be a hash ref ".
	    join(':',caller(1))." ".
	    join(':',caller(0))." ");
    }

    if ($register) {
	my ($on_load,$on_unload);
	foreach my $key (keys(%{$attr_ref})) {
	    if      (lc($key) eq 'onload') {
		$on_load.=$attr_ref->{$key}.';';
		delete($attr_ref->{$key});

	    } elsif (lc($key) eq 'onunload') {
		$on_unload.=$attr_ref->{$key}.';';
		delete($attr_ref->{$key});
	    }
	}
	$attr_ref->{'onload'}  =
	    &Apache::lonmenu::loadevents().  $on_load;
	$attr_ref->{'onunload'}=
	    &Apache::lonmenu::unloadevents().$on_unload;
    }

# Accessibility font enhance
    if ($env{'browser.fontenhance'} eq 'on') {
	my $style;
	foreach my $key (keys(%{$attr_ref})) {
	    if (lc($key) eq 'style') {
		$style.=$attr_ref->{$key}.';';
		delete($attr_ref->{$key});
	    }
	}
	$attr_ref->{'style'}=$style.'; font-size: x-large;';
    }

    if ($env{'browser.blackwhite'} eq 'on') {
	delete($attr_ref->{'font'});
	delete($attr_ref->{'link'});
	delete($attr_ref->{'alink'});
	delete($attr_ref->{'vlink'});
	delete($attr_ref->{'bgcolor'});
	delete($attr_ref->{'background'});
    }

    my $attr_string;
    foreach my $attr (keys(%$attr_ref)) {
	$attr_string .= " $attr=\"".$attr_ref->{$attr}.'" ';
    }
    return $attr_string;
}


###############################################
###############################################

=pod

=item * &endbodytag()

Returns a uniform footer for LON-CAPA web pages.

Inputs: 1 - optional reference to an args hash
If in the hash, key for noredirectlink has a value which evaluates to true,
a 'Continue' link is not displayed if the page contains an
internal redirect in the <head></head> section,
i.e., $env{'internal.head.redirect'} exists   

=cut

sub endbodytag {
    my ($args) = @_;
    my $endbodytag='</body>';
    $endbodytag=&Apache::lontexconvert::jsMath_process()."\n".$endbodytag;
    if ( exists( $env{'internal.head.redirect'} ) ) {
        if (!(ref($args) eq 'HASH' && $args->{'noredirectlink'})) {
	    $endbodytag=
	        "<br /><a href=\"$env{'internal.head.redirect'}\">".
	        &mt('Continue').'</a>'.
	        $endbodytag;
        }
    }
    return $endbodytag;
}

=pod

=item * &standard_css()

Returns a style sheet

Inputs: (all optional)
            domain         -> force to color decorate a page for a specific
                               domain
            function       -> force usage of a specific rolish color scheme
            bgcolor        -> override the default page bgcolor

=cut

sub standard_css {
    my ($function,$domain,$bgcolor) = @_;
    $function  = &get_users_function() if (!$function);
    my $img    = &designparm($function.'.img',   $domain);
    my $tabbg  = &designparm($function.'.tabbg', $domain);
    my $font   = &designparm($function.'.font',  $domain);
    my $sidebg = &designparm($function.'.sidebg',$domain);
    my $pgbg_or_bgcolor =
	         $bgcolor ||
	         &designparm($function.'.pgbg',  $domain);
    my $pgbg   = &designparm($function.'.pgbg',  $domain);
    my $alink  = &designparm($function.'.alink', $domain);
    my $vlink  = &designparm($function.'.vlink', $domain);
    my $link   = &designparm($function.'.link',  $domain);

    my $sans                 = 'Verdana,Arial,Helvetica,sans-serif';
    my $mono                 = 'monospace';
    my $data_table_head      = $tabbg;
    my $data_table_light     = '#EEEEEE';
    my $data_table_dark      = '#DDDDDD';
    my $data_table_darker    = '#CCCCCC';
    my $data_table_highlight = '#FFFF00';
    my $mail_new             = '#FFBB77';
    my $mail_new_hover       = '#DD9955';
    my $mail_read            = '#BBBB77';
    my $mail_read_hover      = '#999944';
    my $mail_replied         = '#AAAA88';
    my $mail_replied_hover   = '#888855';
    my $mail_other           = '#99BBBB';
    my $mail_other_hover     = '#669999';
    my $table_header         = '#DDDDDD';
    my $feedback_link_bg     = '#BBBBBB';

    my $border = ($env{'browser.type'} eq 'explorer' ||
		  $env{'browser.type'} eq 'safari'     ) ? '0px 2px 0px 2px'
	                                                 : '0px 3px 0px 4px';


    return <<END;
h1, h2, h3, th { font-family: $sans }
a:focus { color: red; background: yellow } 
table.thinborder,

table.thinborder tr th {
  border-style: solid;
  border-width: 1px;
  background: $tabbg;
}
table.thinborder tr td {
  border-style: solid;
  border-width: 1px
}

form, .inline { display: inline; }
.center { text-align: center; }
.LC_filename {font-family: $mono; white-space:pre;}
.LC_error {
  color: red;
  font-size: larger;
}
.LC_warning,
.LC_diff_removed {
  color: red;
}

.LC_info,
.LC_success,
.LC_diff_added {
  color: green;
}
.LC_unknown {
  color: yellow;
}

.LC_icon {
  border: 0px;
}
.LC_indexer_icon {
  border: 0px;
  height: 22px;
}
.LC_docs_spacer {
  width: 25px;
  height: 1px;
  border: 0px;
}

.LC_internal_info {
  color: #999;
}

table.LC_pastsubmission {
  border: 1px solid black;
  margin: 2px;
}

table#LC_top_nav, table#LC_menubuttons,table#LC_nav_location {
  width: 100%;
  background: $pgbg;
  border: 2px;
  border-collapse: separate;
  padding: 0px;
}

table#LC_title_bar, table.LC_breadcrumbs, 
table#LC_title_bar.LC_with_remote {
  width: 100%;
  border-color: $pgbg;
  border-style: solid;
  border-width: $border;

  background: $pgbg;
  font-family: $sans;
  border-collapse: collapse;
  padding: 0px;
}

table.LC_docs_path {
  width: 100%;
  border: 0;
  background: $pgbg;
  font-family: $sans;
  border-collapse: collapse;
  padding: 0px;
}

table#LC_title_bar td {
  background: $tabbg;
}
table#LC_title_bar td.LC_title_bar_who {
  background: $tabbg;
  color: $font;
  font: small $sans;
  text-align: right;
}
span.LC_metadata {
    font-family: $sans;
}
span.LC_title_bar_title {
  font: bold x-large $sans;
}
table#LC_title_bar td.LC_title_bar_domain_logo {
  background: $sidebg;
  text-align: right;
  padding: 0px;
}
table#LC_title_bar td.LC_title_bar_role_logo {
  background: $sidebg;
  padding: 0px;
}

table#LC_menubuttons_mainmenu {
  width: 100%;
  border: 0px;
  border-spacing: 1px;
  padding: 0px 1px;
  margin: 0px;
  border-collapse: separate;
}
table#LC_menubuttons img, table#LC_menubuttons_mainmenu img {
  border: 0px;
}
table#LC_top_nav td {
  background: $tabbg;
  border: 0px;
  font-size: small;
}
table#LC_top_nav td a, div#LC_top_nav a {
  color: $font;
  font-family: $sans;
}
table#LC_top_nav td.LC_top_nav_logo {
  background: $tabbg;
  text-align: left;
  white-space: nowrap;
  width: 31px;
}
table#LC_top_nav td.LC_top_nav_logo img {
  border: 0px;
  vertical-align: bottom;
}
table#LC_top_nav td.LC_top_nav_exit,
table#LC_top_nav td.LC_top_nav_help {
  width: 2.0em;
}
table#LC_top_nav td.LC_top_nav_login {
  width: 4.0em;
  text-align: center;
}
table.LC_breadcrumbs td, table.LC_docs_path td  {
  background: $tabbg;
  color: $font;
  font-family: $sans;
  font-size: smaller;
}
table.LC_breadcrumbs td.LC_breadcrumbs_component,
table.LC_docs_path td.LC_docs_path_component {
  background: $tabbg;
  color: $font;
  font-family: $sans;
  font-size: larger;
  text-align: right;
}
td.LC_table_cell_checkbox {
  text-align: center;
}

table#LC_mainmenu td.LC_mainmenu_column {
    vertical-align: top;
}

.LC_menubuttons_inline_text {
  color: $font;
  font-family: $sans;
  font-size: smaller;
}

.LC_menubuttons_link {
  text-decoration: none;
}

.LC_menubuttons_category {
  color: $font;
  background: $pgbg;
  font-family: $sans;
  font-size: larger;
  font-weight: bold;
}

td.LC_menubuttons_text {
  width: 90%;
  color: $font;
  font-family: $sans;
}

td.LC_menubuttons_img {
}

.LC_current_location {
  font-family: $sans;
  background: $tabbg;
}
.LC_new_mail {
  font-family: $sans;
  background: $tabbg;
  font-weight: bold;
}

.LC_rolesmenu_is {
  font-family: $sans;
}

.LC_rolesmenu_selected {
  font-family: $sans;
}

.LC_rolesmenu_future {
  font-family: $sans;
}


.LC_rolesmenu_will {
  font-family: $sans;
}

.LC_rolesmenu_will_not {
  font-family: $sans;
}

.LC_rolesmenu_expired {
  font-family: $sans;
}

.LC_rolesinfo {
  font-family: $sans;
}

.LC_dropadd_labeltext {
  font-family: $sans;
  text-align: right;
}

.LC_preferences_labeltext {
  font-family: $sans;
  text-align: right;
}

.LC_roleslog_note {
  font-size: smaller;
}

table.LC_aboutme_port {
  border: 0px;
  border-collapse: collapse;
  border-spacing: 0px;
}
table.LC_data_table, table.LC_mail_list {
  border: 1px solid #000000;
  border-collapse: separate;
  border-spacing: 1px;
  background: $pgbg;
}
.LC_data_table_dense {
  font-size: small;
}
table.LC_nested_outer {
  border: 1px solid #000000;
  border-collapse: collapse;
  border-spacing: 0px;
  width: 100%;
}
table.LC_nested {
  border: 0px;
  border-collapse: collapse;
  border-spacing: 0px;
  width: 100%;
}
table.LC_data_table tr th, table.LC_calendar tr th, table.LC_mail_list tr th,
table.LC_prior_tries tr th {
  font-weight: bold;
  background-color: $data_table_head;
  font-size: smaller;
}
table.LC_data_table tr.LC_odd_row > td, 
table.LC_aboutme_port tr td {
  background-color: $data_table_light;
  padding: 2px;
}
table.LC_data_table tr.LC_even_row > td,
table.LC_aboutme_port tr.LC_even_row td {
  background-color: $data_table_dark;
}
table.LC_data_table tr.LC_data_table_highlight td {
  background-color: $data_table_darker;
}
table.LC_data_table tr td.LC_leftcol_header {
  background-color: $data_table_head;
  font-weight: bold;
}
table.LC_data_table tr.LC_empty_row td,
table.LC_nested tr.LC_empty_row td {
  background-color: #FFFFFF;
  font-weight: bold;
  font-style: italic;
  text-align: center;
  padding: 8px;
}
table.LC_nested tr.LC_empty_row td {
  padding: 4ex
}
table.LC_nested_outer tr th {
  font-weight: bold;
  background-color: $data_table_head;
  font-size: smaller;
  border-bottom: 1px solid #000000;
}
table.LC_nested_outer tr td.LC_subheader {
  background-color: $data_table_head;
  font-weight: bold;
  font-size: small;
  border-bottom: 1px solid #000000;
  text-align: right;
}
table.LC_nested tr.LC_info_row td {
  background-color: #CCC;
  font-weight: bold;
  font-size: small;
  text-align: center;
}
table.LC_nested tr.LC_info_row td.LC_left_item,
table.LC_nested_outer tr th.LC_left_item {
  text-align: left;
}
table.LC_nested td {
  background-color: #FFF;
  font-size: small;
}
table.LC_nested_outer tr th.LC_right_item,
table.LC_nested tr.LC_info_row td.LC_right_item,
table.LC_nested tr.LC_odd_row td.LC_right_item,
table.LC_nested tr td.LC_right_item {
  text-align: right;
}

table.LC_nested tr.LC_odd_row td {
  background-color: #EEE;
}

table.LC_createuser {
}

table.LC_createuser tr.LC_section_row td {
  font-size: smaller;
}

table.LC_createuser tr.LC_info_row td  {
  background-color: #CCC;
  font-weight: bold;
  text-align: center;
}

table.LC_calendar {
  border: 1px solid #000000;
  border-collapse: collapse;
}
table.LC_calendar_pickdate {
  font-size: xx-small;
}
table.LC_calendar tr td {
  border: 1px solid #000000;
  vertical-align: top;
}
table.LC_calendar tr td.LC_calendar_day_empty {
  background-color: $data_table_dark;
}
table.LC_calendar tr td.LC_calendar_day_current {
  background-color: $data_table_highlight;
}

table.LC_mail_list tr.LC_mail_new {
  background-color: $mail_new;
}
table.LC_mail_list tr.LC_mail_new:hover {
  background-color: $mail_new_hover;
}
table.LC_mail_list tr.LC_mail_read {
  background-color: $mail_read;
}
table.LC_mail_list tr.LC_mail_read:hover {
  background-color: $mail_read_hover;
}
table.LC_mail_list tr.LC_mail_replied {
  background-color: $mail_replied;
}
table.LC_mail_list tr.LC_mail_replied:hover {
  background-color: $mail_replied_hover;
}
table.LC_mail_list tr.LC_mail_other {
  background-color: $mail_other;
}
table.LC_mail_list tr.LC_mail_other:hover {
  background-color: $mail_other_hover;
}
table.LC_mail_list tr.LC_mail_even {
}
table.LC_mail_list tr.LC_mail_odd {
}


table#LC_portfolio_actions {
  width: auto;
  background: $pgbg;
  border: 0px;
  border-spacing: 2px 2px;
  padding: 0px;
  margin: 0px;
  border-collapse: separate;
}
table#LC_portfolio_actions td.LC_label {
  background: $tabbg;
  text-align: right;
}
table#LC_portfolio_actions td.LC_value {
  background: $tabbg;
}

table#LC_cstr_controls {
  width: 100%;
  border-collapse: collapse;
}
table#LC_cstr_controls tr td {
  border: 4px solid $pgbg;
  padding: 4px;
  text-align: center;
  background: $tabbg;
}
table#LC_cstr_controls tr th {
  border: 4px solid $pgbg;
  background: $table_header;
  text-align: center;
  font-family: $sans;
  font-size: smaller;
}

table#LC_browser {
 
}
table#LC_browser tr th {
  background: $table_header;
}
table#LC_browser tr td {
  padding: 2px;
}
table#LC_browser tr.LC_browser_file,
table#LC_browser tr.LC_browser_file_published {
  background: #CCFF88;
}
table#LC_browser tr.LC_browser_file_locked,
table#LC_browser tr.LC_browser_file_unpublished {
  background: #FFAA99;
}
table#LC_browser tr.LC_browser_file_obsolete {
  background: #AAAAAA;
}
table#LC_browser tr.LC_browser_file_modified,
table#LC_browser tr.LC_browser_file_metamodified {
  background: #FFFF77;
}
table#LC_browser tr.LC_browser_folder {
  background: #CCCCFF;
}
span.LC_current_location {
  font-size: x-large;
  background: $pgbg;
}

span.LC_parm_menu_item {
  font-size: larger;
  font-family: $sans;
}
span.LC_parm_scope_all {
  color: red;
}
span.LC_parm_scope_folder {
  color: green;
}
span.LC_parm_scope_resource {
  color: orange;
}
span.LC_parm_part {
  color: blue;
}
span.LC_parm_folder, span.LC_parm_symb {
  font-size: x-small;
  font-family: $mono;
  color: #AAAAAA;
}

td.LC_parm_overview_level_menu, td.LC_parm_overview_map_menu,
td.LC_parm_overview_parm_selectors, td.LC_parm_overview_parm_restrictions {
  border: 1px solid black;
  border-collapse: collapse;
}
table.LC_parm_overview_restrictions td {
  border-width: 1px 4px 1px 4px;
  border-style: solid;
  border-color: $pgbg;
  text-align: center;
}
table.LC_parm_overview_restrictions th {
  background: $tabbg;
  border-width: 1px 4px 1px 4px;
  border-style: solid;
  border-color: $pgbg;
}
table#LC_helpmenu {
  border: 0px;
  height: 55px;
  border-spacing: 0px;
}

table#LC_helpmenu fieldset legend {
  font-size: larger;
  font-weight: bold;
}
table#LC_helpmenu_links {
  width: 100%;
  border: 1px solid black;
  background: $pgbg;
  padding: 0px;
  border-spacing: 1px;
}
table#LC_helpmenu_links tr td {
  padding: 1px;
  background: $tabbg;
  text-align: center;
  font-weight: bold;
}

table#LC_helpmenu_links a:link, table#LC_helpmenu_links a:visited,
table#LC_helpmenu_links a:active {
  text-decoration: none;
  color: $font;
}
table#LC_helpmenu_links a:hover {
  text-decoration: underline;
  color: $vlink;
}

.LC_chrt_popup_exists {
  border: 1px solid #339933;
  margin: -1px;
}
.LC_chrt_popup_up {
  border: 1px solid yellow;
  margin: -1px;
}
.LC_chrt_popup {
  border: 1px solid #8888FF;
  background: #CCCCFF;
}
table.LC_pick_box {
  border-collapse: separate;
  background: white;
  border: 1px solid black;
  border-spacing: 1px;
}
table.LC_pick_box td.LC_pick_box_title {
  background: $tabbg;
  font-weight: bold;
  text-align: right;
  width: 184px;
  padding: 8px;
}
table.LC_pick_box td.LC_selfenroll_pick_box_title {
  background: $tabbg;
  font-weight: bold;
  text-align: right;
  width: 350px;
  padding: 8px;
}

table.LC_pick_box td.LC_pick_box_value {
  text-align: left;
  padding: 8px;
}
table.LC_pick_box td.LC_pick_box_select {
  text-align: left;
  padding: 8px;
}
table.LC_pick_box td.LC_pick_box_separator {
  padding: 0px;
  height: 1px;
  background: black;
}
table.LC_pick_box td.LC_pick_box_submit {
  text-align: right;
}
table.LC_pick_box td.LC_evenrow_value {
  text-align: left;
  padding: 8px;
  background-color: $data_table_light;
}
table.LC_pick_box td.LC_oddrow_value {
  text-align: left;
  padding: 8px;
  background-color: $data_table_light;
}
table.LC_helpform_receipt {
  width: 620px;
  border-collapse: separate;
  background: white;
  border: 1px solid black;
  border-spacing: 1px;
}
table.LC_helpform_receipt td.LC_pick_box_title {
  background: $tabbg;
  font-weight: bold;
  text-align: right;
  width: 184px;
  padding: 8px;
}
table.LC_helpform_receipt td.LC_evenrow_value {
  text-align: left;
  padding: 8px;
  background-color: $data_table_light;
}
table.LC_helpform_receipt td.LC_oddrow_value {
  text-align: left;
  padding: 8px;
  background-color: $data_table_light;
}
table.LC_helpform_receipt td.LC_pick_box_separator {
  padding: 0px;
  height: 1px;
  background: black;
}
span.LC_helpform_receipt_cat {
  font-weight: bold;
}
table.LC_group_priv_box {
  background: white;
  border: 1px solid black;
  border-spacing: 1px;
}
table.LC_group_priv_box td.LC_pick_box_title {
  background: $tabbg;
  font-weight: bold;
  text-align: right;
  width: 184px;
}
table.LC_group_priv_box td.LC_groups_fixed {
  background: $data_table_light;
  text-align: center;
}
table.LC_group_priv_box td.LC_groups_optional {
  background: $data_table_dark;
  text-align: center;
}
table.LC_group_priv_box td.LC_groups_functionality {
  background: $data_table_darker;
  text-align: center;
  font-weight: bold;
}
table.LC_group_priv td {
  text-align: left;
  padding: 0px;
}

table.LC_notify_front_page {
  background: white;
  border: 1px solid black;
  padding: 8px;
}
table.LC_notify_front_page td {
  padding: 8px;
}
.LC_navbuttons {
  margin: 2ex 0ex 2ex 0ex;
}
.LC_topic_bar {
  font-family: $sans;
  font-weight: bold;
  width: 100%;
  background: $tabbg;
  vertical-align: middle;
  margin: 2ex 0ex 2ex 0ex;
}
.LC_topic_bar span {
  vertical-align: middle;
}
.LC_topic_bar img {
  vertical-align: bottom;
}
table.LC_course_group_status {
  margin: 20px;
}
table.LC_status_selector td {
  vertical-align: top;
  text-align: center;
  padding: 4px;
}
table.LC_descriptive_input td.LC_description {
  vertical-align: top;
  text-align: right;
  font-weight: bold;
}
div.LC_feedback_link {
  clear: both;
  background: white;
  width: 100%;  
}
span.LC_feedback_link {
  background: $feedback_link_bg;
  font-size: larger;
}
span.LC_message_link {
  background: $feedback_link_bg;
  font-size: larger;
  position: absolute;
  right: 1em;
}

table.LC_prior_tries {
  border: 1px solid #000000;
  border-collapse: separate;
  border-spacing: 1px;
}

table.LC_prior_tries td {
  padding: 2px;
}

.LC_answer_correct {
  background: #AAFFAA;
  color: black;
}
.LC_answer_charged_try {
  background: #FFAAAA ! important;
  color: black;
}
.LC_answer_not_charged_try, 
.LC_answer_no_grade,
.LC_answer_late {
  background: #FFFFAA;
  color: black;
}
.LC_answer_previous {
  background: #AAAAFF;
  color: black;
}
.LC_answer_no_message {
  background: #FFFFFF;
  color: black;
}
.LC_answer_unknown {
  background: orange;
  color: black;
}


span.LC_prior_numerical,
span.LC_prior_string,
span.LC_prior_custom,
span.LC_prior_reaction,
span.LC_prior_math {
  font-family: monospace;
  white-space: pre;
}

span.LC_prior_string {
  font-family: monospace;
  white-space: pre;
}

table.LC_prior_option {
  width: 100%;
  border-collapse: collapse;
}
table.LC_prior_rank, table.LC_prior_match {
  border-collapse: collapse;
}
table.LC_prior_option tr td,
table.LC_prior_rank tr td,
table.LC_prior_match tr td {
  border: 1px solid #000000;
}

span.LC_nobreak {
  white-space: nowrap;
}

span.LC_cusr_emph {
  font-style: italic;
}

span.LC_cusr_subheading {
  font-weight: normal;
  font-size: 85%;
}

table.LC_docs_documents {
  background: #BBBBBB;
  border-width: 0px;
  border-collapse: collapse;
}

table.LC_docs_documents td.LC_docs_document {
  border: 2px solid black;
  padding: 4px;
}

.LC_docs_course_commands div {
  float: left;
  border: 4px solid #AAAAAA;
  padding: 4px;
  background: #DDDDCC;
}

.LC_docs_entry_move {
  border: 0px;
  border-collapse: collapse;
}

.LC_docs_entry_move td {
  border: 2px solid #BBBBBB;
  background: #DDDDDD;
}

.LC_docs_editor td.LC_docs_entry_commands {
  background: #DDDDDD;
  font-size: x-small;
}
.LC_docs_copy {
  color: #000099;
}
.LC_docs_cut {
  color: #550044;
}
.LC_docs_rename {
  color: #009900;
}
.LC_docs_remove {
  color: #990000;
}

.LC_docs_reinit_warn,
.LC_docs_ext_edit {
  font-size: x-small;
}

.LC_docs_editor td.LC_docs_entry_title,
.LC_docs_editor td.LC_docs_entry_icon {
  background: #FFFFBB;
}
.LC_docs_editor td.LC_docs_entry_parameter {
  background: #BBBBFF;
  font-size: x-small;
  white-space: nowrap;
}

table.LC_docs_adddocs td,
table.LC_docs_adddocs th {
  border: 1px solid #BBBBBB;
  padding: 4px;
  background: #DDDDDD;
}

table.LC_sty_begin {
  background: #BBFFBB;
}
table.LC_sty_end {
  background: #FFBBBB;
}

table.LC_double_column {
  border-width: 0px;
  border-collapse: collapse;
  width: 100%;
  padding: 2px;
}

table.LC_double_column tr td.LC_left_col {
  top: 2px;
  left: 2px;
  width: 47%;
  vertical-align: top;
}

table.LC_double_column tr td.LC_right_col {
  top: 2px;
  right: 2px; 
  width: 47%;
  vertical-align: top;
}

span.LC_role_level {
  font-weight: bold;
}

div.LC_left_float {
  float: left;
  padding-right: 5%;
  padding-bottom: 4px;
}

div.LC_clear_float_header {
  padding-bottom: 2px;
}

div.LC_clear_float_footer {
  padding-top: 10px;
  clear: both;
}


div.LC_grade_select_mode {
  font-family: $sans;
}
div.LC_grade_select_mode div div {
  margin: 5px;
}
div.LC_grade_select_mode_selector {
  margin: 5px;
  float: left;
}
div.LC_grade_select_mode_selector_header {
  font: bold medium $sans;
}
div.LC_grade_select_mode_type {
  clear: left;
}

div.LC_grade_show_user {
  margin-top: 20px;
  border: 1px solid black;
}
div.LC_grade_user_name {
  background: #DDDDEE;
  border-bottom: 1px solid black;
  font: bold large $sans;
}
div.LC_grade_show_user_odd_row div.LC_grade_user_name {
  background: #DDEEDD;
}

div.LC_grade_show_problem,
div.LC_grade_submissions,
div.LC_grade_message_center,
div.LC_grade_info_links,
div.LC_grade_assign {
  margin: 5px;
  width: 99%;
  background: #FFFFFF;
}
div.LC_grade_show_problem_header,
div.LC_grade_submissions_header,
div.LC_grade_message_center_header,
div.LC_grade_assign_header {
  font: bold large $sans;
}
div.LC_grade_show_problem_problem,
div.LC_grade_submissions_body,
div.LC_grade_message_center_body,
div.LC_grade_assign_body {
  border: 1px solid black;
  width: 99%;
  background: #FFFFFF;
}
span.LC_grade_check_note {
  font: normal medium $sans;
  display: inline;
  position: absolute;
  right: 1em;
}

table.LC_scantron_action {
  width: 100%;
}
table.LC_scantron_action tr th {
  font: normal bold $sans;
}

div.LC_edit_problem_header, 
div.LC_edit_problem_footer {
  font: normal medium $sans;
  margin: 2px;
}
div.LC_edit_problem_header,
div.LC_edit_problem_header div,
div.LC_edit_problem_footer,
div.LC_edit_problem_footer div,
div.LC_edit_problem_editxml_header,
div.LC_edit_problem_editxml_header div {
  margin-top: 5px;
}
div.LC_edit_problem_header_edit_row {
  background: $tabbg;
  padding: 3px;
  margin-bottom: 5px;
}
div.LC_edit_problem_header_title {
  font: larger bold $sans;
  background: $tabbg;
  padding: 3px;
}
table.LC_edit_problem_header_title {
  font: larger bold $sans;
  width: 100%;
  border-color: $pgbg;
  border-style: solid;
  border-width: $border;

  background: $tabbg;
  border-collapse: collapse;
  padding: 0px
}

div.LC_edit_problem_discards {
  float: left;
  padding-bottom: 5px;
}
div.LC_edit_problem_saves {
  float: right;
  padding-bottom: 5px;
}
hr.LC_edit_problem_divide {
  clear: both;
  color: $tabbg;
  background-color: $tabbg;
  height: 3px;
  border: 0px;
}
END
}

=pod

=item * &headtag()

Returns a uniform footer for LON-CAPA web pages.

Inputs: $title - optional title for the head
        $head_extra - optional extra HTML to put inside the <head>
        $args - optional arguments
            force_register - if is true call registerurl so the remote is 
                             informed
            redirect       -> array ref of
                                   1- seconds before redirect occurs
                                   2- url to redirect to
                                   3- whether the side effect should occur
                           (side effect of setting 
                               $env{'internal.head.redirect'} to the url 
                               redirected too)
            domain         -> force to color decorate a page for a specific
                               domain
            function       -> force usage of a specific rolish color scheme
            bgcolor        -> override the default page bgcolor
            no_auto_mt_title
                           -> prevent &mt()ing the title arg

=cut

sub headtag {
    my ($title,$head_extra,$args) = @_;
    
    my $function = $args->{'function'} || &get_users_function();
    my $domain   = $args->{'domain'}   || &determinedomain();
    my $bgcolor  = $args->{'bgcolor'}  || &designparm($function.'.pgbg',$domain);
    my $url = join(':',$env{'user.name'},$env{'user.domain'},
		   $Apache::lonnet::perlvar{'lonVersion'},
		   #time(),
		   $env{'environment.color.timestamp'},
		   $function,$domain,$bgcolor);

    $url = '/adm/css/'.&escape($url).'.css';

    my $result =
	'<head>'.
	&font_settings();

    if (!$args->{'frameset'}) {
	$result .= &Apache::lonhtmlcommon::htmlareaheaders();
    }
    if ($args->{'force_register'}) {
	$result .= &Apache::lonmenu::registerurl(1);
    }
    if (!$args->{'no_nav_bar'} 
	&& !$args->{'only_body'}
	&& !$args->{'frameset'}) {
	$result .= &help_menu_js();
    }

    if (ref($args->{'redirect'})) {
	my ($time,$url,$inhibit_continue) = @{$args->{'redirect'}};
	$url = &Apache::lonenc::check_encrypt($url);
	if (!$inhibit_continue) {
	    $env{'internal.head.redirect'} = $url;
	}
	$result.=<<ADDMETA
<meta http-equiv="pragma" content="no-cache" />
<meta http-equiv="Refresh" content="$time; url=$url" />
ADDMETA
    }
    if (!defined($title)) {
	$title = 'The LearningOnline Network with CAPA';
    }
    if (!$args->{'no_auto_mt_title'}) { $title = &mt($title); }
    $result .= '<title> LON-CAPA '.$title.'</title>'
	.'<link rel="stylesheet" type="text/css" href="'.$url.'" />'
	.$head_extra;
    return $result;
}

=pod

=item * &font_settings()

Returns neccessary <meta> to set the proper encoding

Inputs: none

=cut

sub font_settings {
    my $headerstring='';
    if (!$env{'browser.mathml'} && $env{'browser.unicode'}) {
	$headerstring.=
	    '<meta http-equiv="Content-Type" content="text/html; charset=utf-8" />';
    }
    return $headerstring;
}

=pod

=item * &xml_begin()

Returns the needed doctype and <html>

Inputs: none

=cut

sub xml_begin {
    my $output='';

    if ($env{'internal.start_page'}==1) {
	&Apache::lonhtmlcommon::init_htmlareafields();
    }

    if ($env{'browser.mathml'}) {
	$output='<?xml version="1.0"?>'
            #.'<?xml-stylesheet type="text/css" href="/adm/MathML/mathml.css"?>'."\n"
#            .'<!DOCTYPE html SYSTEM "/adm/MathML/mathml.dtd" '
            
#	    .'<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd" [<!ENTITY mathns "http://www.w3.org/1998/Math/MathML">] >'
	    .'<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1 plus MathML 2.0 plus SVG 1.1//EN" "http://www.w3.org/2002/04/xhtml-math-svg/xhtml-math-svg.dtd">'
            .'<html xmlns:math="http://www.w3.org/1998/Math/MathML" ' 
	    .'xmlns="http://www.w3.org/1999/xhtml">';
    } else {
	$output='<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd"><html>';
    }
    return $output;
}

=pod

=item * &endheadtag()

Returns a uniform </head> for LON-CAPA web pages.

Inputs: none

=cut

sub endheadtag {
    return '</head>';
}

=pod

=item * &head()

Returns a uniform complete <head>..</head> section for LON-CAPA web pages.

Inputs:

=over 4

$title - optional title for the page

$head_extra - optional extra HTML to put inside the <head>

=back

=cut

sub head {
    my ($title,$head_extra,$args) = @_;
    return &headtag($title,$head_extra,$args).&endheadtag();
}

=pod

=item * &start_page()

Returns a complete <html> .. <body> section for LON-CAPA web pages.

Inputs:

=over 4

$title - optional title for the page

$head_extra - optional extra HTML to incude inside the <head>

$args - additional optional args supported are:

=over 8

             only_body      -> is true will set &bodytag() onlybodytag
                                    arg on
             no_nav_bar     -> is true will set &bodytag() notopbar arg on
             add_entries    -> additional attributes to add to the  <body>
             domain         -> force to color decorate a page for a 
                                    specific domain
             function       -> force usage of a specific rolish color
                                    scheme
             redirect       -> see &headtag()
             bgcolor        -> override the default page bg color
             js_ready       -> return a string ready for being used in 
                                    a javascript writeln
             html_encode    -> return a string ready for being used in 
                                    a html attribute
             force_register -> if is true will turn on the &bodytag()
                                    $forcereg arg
             body_title     -> alternate text to use instead of $title
                                    in the title box that appears, this text
                                    is not auto translated like the $title is
             frameset       -> if true will start with a <frameset>
                                    rather than <body>
             no_title       -> if true the title bar won't be shown
             skip_phases    -> hash ref of 
                                    head -> skip the <html><head> generation
                                    body -> skip all <body> generation
             no_inline_link -> if true and in remote mode, don't show the 
                                    'Switch To Inline Menu' link
             no_auto_mt_title -> prevent &mt()ing the title arg
             inherit_jsmath -> when creating popup window in a page,
                                    should it have jsmath forced on by the
                                    current page

=back

=back

=cut

sub start_page {
    my ($title,$head_extra,$args) = @_;
    #&Apache::lonnet::logthis("start_page ".join(':',caller(0)));
    my %head_args;
    foreach my $arg ('redirect','force_register','domain','function',
		     'bgcolor','frameset','no_nav_bar','only_body',
		     'no_auto_mt_title') {
	if (defined($args->{$arg})) {
	    $head_args{$arg} = $args->{$arg};
	}
    }

    $env{'internal.start_page'}++;
    my $result;
    if (! exists($args->{'skip_phases'}{'head'}) ) {
	$result.=
	    &xml_begin().
	    &headtag($title,$head_extra,\%head_args).&endheadtag();
    }
    
    if (! exists($args->{'skip_phases'}{'body'}) ) {
	if ($args->{'frameset'}) {
	    my $attr_string = &make_attr_string($args->{'force_register'},
						$args->{'add_entries'});
	    $result .= "\n<frameset $attr_string>\n";
	} else {
	    $result .=
		&bodytag($title, 
			 $args->{'function'},       $args->{'add_entries'},
			 $args->{'only_body'},      $args->{'domain'},
			 $args->{'force_register'}, $args->{'body_title'},
			 $args->{'no_nav_bar'},     $args->{'bgcolor'},
			 $args->{'no_title'},       $args->{'no_inline_link'},
			 $args);
	}
    }

    if ($args->{'js_ready'}) {
	$result = &js_ready($result);
    }
    if ($args->{'html_encode'}) {
	$result = &html_encode($result);
    }
    return $result;
}


=pod

=item * &head()

Returns a complete </body></html> section for LON-CAPA web pages.

Inputs:         $args - additional optional args supported are:
                 js_ready     -> return a string ready for being used in 
                                 a javascript writeln
                 html_encode  -> return a string ready for being used in 
                                 a html attribute
                 frameset     -> if true will start with a <frameset>
                                 rather than <body>
                 dicsussion   -> if true will get discussion from
                                  lonxml::xmlend
                                 (you can pass the target and parser arguments
                                  through optional 'target' and 'parser' args
                                  to this routine)

=cut

sub end_page {
    my ($args) = @_;
    $env{'internal.end_page'}++;
    my $result;
    if ($args->{'discussion'}) {
	my ($target,$parser);
	if (ref($args->{'discussion'})) {
	    ($target,$parser) =($args->{'discussion'}{'target'},
				$args->{'discussion'}{'parser'});
	}
	$result .= &Apache::lonxml::xmlend($target,$parser);
    }

    if ($args->{'frameset'}) {
	$result .= '</frameset>';
    } else {
	$result .= &endbodytag($args);
    }
    $result .= "\n</html>";

    if ($args->{'js_ready'}) {
	$result = &js_ready($result);
    }

    if ($args->{'html_encode'}) {
	$result = &html_encode($result);
    }

    return $result;
}

sub html_encode {
    my ($result) = @_;

    $result = &HTML::Entities::encode($result,'<>&"');
    
    return $result;
}
sub js_ready {
    my ($result) = @_;

    $result =~ s/[\n\r]/ /xmsg;
    $result =~ s/\\/\\\\/xmsg;
    $result =~ s/'/\\'/xmsg;
    $result =~ s{</}{<\\/}xmsg;
    
    return $result;
}

sub validate_page {
    if (  exists($env{'internal.start_page'})
	  &&     $env{'internal.start_page'} > 1) {
	&Apache::lonnet::logthis('start_page called multiple times '.
				 $env{'internal.start_page'}.' '.
				 $ENV{'request.filename'});
    }
    if (  exists($env{'internal.end_page'})
	  &&     $env{'internal.end_page'} > 1) {
	&Apache::lonnet::logthis('end_page called multiple times '.
				 $env{'internal.end_page'}.' '.
				 $env{'request.filename'});
    }
    if (     exists($env{'internal.start_page'})
	&& ! exists($env{'internal.end_page'})) {
	&Apache::lonnet::logthis('start_page called without end_page '.
				 $env{'request.filename'});
    }
    if (   ! exists($env{'internal.start_page'})
	&&   exists($env{'internal.end_page'})) {
	&Apache::lonnet::logthis('end_page called without start_page'.
				 $env{'request.filename'});
    }
}

sub simple_error_page {
    my ($r,$title,$msg) = @_;
    my $page =
	&Apache::loncommon::start_page($title).
	&mt($msg).
	&Apache::loncommon::end_page();
    if (ref($r)) {
	$r->print($page);
	return;
    }
    return $page;
}

{
    my @row_count;
    sub start_data_table {
	my ($add_class) = @_;
	my $css_class = (join(' ','LC_data_table',$add_class));
	unshift(@row_count,0);
	return '<table class="'.$css_class.'">'."\n";
    }

    sub end_data_table {
	shift(@row_count);
	return '</table>'."\n";;
    }

    sub start_data_table_row {
	my ($add_class) = @_;
	$row_count[0]++;
	my $css_class = ($row_count[0] % 2)?'LC_odd_row':'LC_even_row';
	$css_class = (join(' ',$css_class,$add_class));
	return  '<tr class="'.$css_class.'">'."\n";;
    }
    
    sub continue_data_table_row {
	my ($add_class) = @_;
	my $css_class = ($row_count[0] % 2)?'LC_odd_row':'LC_even_row';
	$css_class = (join(' ',$css_class,$add_class));
	return  '<tr class="'.$css_class.'">'."\n";;
    }

    sub end_data_table_row {
	return '</tr>'."\n";;
    }

    sub start_data_table_empty_row {
	$row_count[0]++;
	return  '<tr class="LC_empty_row" >'."\n";;
    }

    sub end_data_table_empty_row {
	return '</tr>'."\n";;
    }

    sub start_data_table_header_row {
	return  '<tr class="LC_header_row">'."\n";;
    }

    sub end_data_table_header_row {
	return '</tr>'."\n";;
    }
}

=pod

=item * &inhibit_menu_check($arg)

Checks for a inhibitmenu state and generates output to preserve it

Inputs:         $arg - can be any of
                     - undef - in which case the return value is a string 
                               to add  into arguments list of a uri
                     - 'input' - in which case the return value is a HTML
                                 <form> <input> field of type hidden to
                                 preserve the value
                     - a url - in which case the return value is the url with
                               the neccesary cgi args added to preserve the
                               inhibitmenu state
                     - a ref to a url - no return value, but the string is
                                        updated to include the neccessary cgi
                                        args to preserve the inhibitmenu state

=cut

sub inhibit_menu_check {
    my ($arg) = @_;
    &get_unprocessed_cgi($ENV{'QUERY_STRING'}, ['inhibitmenu']);
    if ($arg eq 'input') {
	if ($env{'form.inhibitmenu'}) {
	    return '<input type="hidden" name="inhibitmenu" value="'.$env{'form.inhibitmenu'}.'" />';
	} else {
	    return
	}
    }
    if ($env{'form.inhibitmenu'}) {
	if (ref($arg)) {
	    $$arg .= '?inhibitmenu='.$env{'form.inhibitmenu'};
	} elsif ($arg eq '') {
	    $arg .= 'inhibitmenu='.$env{'form.inhibitmenu'};
	} else {
	    $arg .= '?inhibitmenu='.$env{'form.inhibitmenu'};
	}
    }
    if (!ref($arg)) {
	return $arg;
    }
}

###############################################

=pod

=back

=head1 User Information Routines

=over 4

=item * &get_users_function()

Used by &bodytag to determine the current users primary role.
Returns either 'student','coordinator','admin', or 'author'.

=cut

###############################################
sub get_users_function {
    my $function = 'student';
    if ($env{'request.role'}=~/^(cc|in|ta|ep)/) {
        $function='coordinator';
    }
    if ($env{'request.role'}=~/^(su|dc|ad|li)/) {
        $function='admin';
    }
    if (($env{'request.role'}=~/^(au|ca)/) ||
        ($ENV{'REQUEST_URI'}=~/^(\/priv|\~)/)) {
        $function='author';
    }
    return $function;
}

###############################################

=pod

=item * &check_user_status()

Determines current status of supplied role for a
specific user. Roles can be active, previous or future.

Inputs: 
user's domain, user's username, course's domain,
course's number, optional section ID.

Outputs:
role status: active, previous or future. 

=cut

sub check_user_status {
    my ($udom,$uname,$cdom,$crs,$role,$sec) = @_;
    my %userinfo = &Apache::lonnet::dump('roles',$udom,$uname);
    my @uroles = keys %userinfo;
    my $srchstr;
    my $active_chk = 'none';
    my $now = time;
    if (@uroles > 0) {
        if (($role eq 'cc') || ($sec eq '') || (!defined($sec))) {
            $srchstr = '/'.$cdom.'/'.$crs.'_'.$role;
        } else {
            $srchstr = '/'.$cdom.'/'.$crs.'/'.$sec.'_'.$role;
        }
        if (grep/^\Q$srchstr\E$/,@uroles) {
            my $role_end = 0;
            my $role_start = 0;
            $active_chk = 'active';
            if ($userinfo{$srchstr} =~ m/^\Q$role\E_(\d+)/) {
                $role_end = $1;
                if ($userinfo{$srchstr} =~ m/^\Q$role\E_\Q$role_end\E_(\d+)$/) {
                    $role_start = $1;
                }
            }
            if ($role_start > 0) {
                if ($now < $role_start) {
                    $active_chk = 'future';
                }
            }
            if ($role_end > 0) {
                if ($now > $role_end) {
                    $active_chk = 'previous';
                }
            }
        }
    }
    return $active_chk;
}

###############################################

=pod

=item * &get_sections()

Determines all the sections for a course including
sections with students and sections containing other roles.
Incoming parameters: 

1. domain
2. course number 
3. reference to array containing roles for which sections should 
be gathered (optional).
4. reference to array containing status types for which sections 
should be gathered (optional).

If the third argument is undefined, sections are gathered for any role. 
If the fourth argument is undefined, sections are gathered for any status.
Permissible values are 'active' or 'future' or 'previous'.
 
Returns section hash (keys are section IDs, values are
number of users in each section), subject to the
optional roles filter, optional status filter 

=cut

###############################################
sub get_sections {
    my ($cdom,$cnum,$possible_roles,$possible_status) = @_;
    if (!defined($cdom) || !defined($cnum)) {
        my $cid =  $env{'request.course.id'};

	return if (!defined($cid));

        $cdom = $env{'course.'.$cid.'.domain'};
        $cnum = $env{'course.'.$cid.'.num'};
    }

    my %sectioncount;
    my $now = time;

    if (!defined($possible_roles) || (grep(/^st$/,@$possible_roles))) {
	my ($classlist) = &Apache::loncoursedata::get_classlist($cdom,$cnum);
	my $sec_index = &Apache::loncoursedata::CL_SECTION();
	my $status_index = &Apache::loncoursedata::CL_STATUS();
        my $start_index = &Apache::loncoursedata::CL_START();
        my $end_index = &Apache::loncoursedata::CL_END();
        my $status;
	while (my ($student,$data) = each(%$classlist)) {
	    my ($section,$stu_status,$start,$end) = ($data->[$sec_index],
				                     $data->[$status_index],
                                                     $data->[$start_index],
                                                     $data->[$end_index]);
            if ($stu_status eq 'Active') {
                $status = 'active';
            } elsif ($end < $now) {
                $status = 'previous';
            } elsif ($start > $now) {
                $status = 'future';
            } 
	    if ($section ne '-1' && $section !~ /^\s*$/) {
                if ((!defined($possible_status)) || (($status ne '') && 
                    (grep/^\Q$status\E$/,@{$possible_status}))) { 
		    $sectioncount{$section}++;
                }
	    }
	}
    }
    my %courseroles = &Apache::lonnet::dump('nohist_userroles',$cdom,$cnum);
    foreach my $user (sort(keys(%courseroles))) {
	if ($user !~ /^(\w{2})/) { next; }
	my ($role) = ($user =~ /^(\w{2})/);
	if ($possible_roles && !(grep(/^$role$/,@$possible_roles))) { next; }
	my ($section,$status);
	if ($role eq 'cr' &&
	    $user =~ m-^$role/[^/]*/[^/]*/[^/]*:[^:]*:[^:]*:(\w+)-) {
	    $section=$1;
	}
	if ($user =~ /^$role:[^:]*:[^:]*:(\w+)/) { $section=$1; }
	if (!defined($section) || $section eq '-1') { next; }
        my ($end,$start) = ($courseroles{$user} =~ /^([^:]*):([^:]*)$/);
        if ($end == -1 && $start == -1) {
            next; #deleted role
        }
        if (!defined($possible_status)) { 
            $sectioncount{$section}++;
        } else {
            if ((!$end || $end >= $now) && (!$start || $start <= $now)) {
                $status = 'active';
            } elsif ($end < $now) {
                $status = 'future';
            } elsif ($start > $now) {
                $status = 'previous';
            }
            if (($status ne '') && (grep/^\Q$status\E$/,@{$possible_status})) {
                $sectioncount{$section}++;
            }
        }
    }
    return %sectioncount;
}

###############################################

=pod

=item * &get_course_users()

Retrieves usernames:domains for users in the specified course
with specific role(s), and access status. 

Incoming parameters:
1. course domain
2. course number
3. access status: users must have - either active, 
previous, future, or all.
4. reference to array of permissible roles
5. reference to array of section restrictions (optional)
6. reference to results object (hash of hashes).
7. reference to optional userdata hash
8. reference to optional statushash
9. flag if privileged users (except those set to unhide in
   course settings) should be excluded    
Keys of top level results hash are roles.
Keys of inner hashes are username:domain, with 
values set to access type.
Optional userdata hash returns an array with arguments in the 
same order as loncoursedata::get_classlist() for student data.

Optional statushash returns

Entries for end, start, section and status are blank because
of the possibility of multiple values for non-student roles.

=cut

###############################################

sub get_course_users {
    my ($cdom,$cnum,$types,$roles,$sections,$users,$userdata,$statushash,$hidepriv) = @_;
    my %idx = ();
    my %seclists;

    $idx{udom} = &Apache::loncoursedata::CL_SDOM();
    $idx{uname} =  &Apache::loncoursedata::CL_SNAME();
    $idx{end} = &Apache::loncoursedata::CL_END();
    $idx{start} = &Apache::loncoursedata::CL_START();
    $idx{id} = &Apache::loncoursedata::CL_ID();
    $idx{section} = &Apache::loncoursedata::CL_SECTION();
    $idx{fullname} = &Apache::loncoursedata::CL_FULLNAME();
    $idx{status} = &Apache::loncoursedata::CL_STATUS();

    if (grep(/^st$/,@{$roles})) {
        my ($classlist,$keylist)=&Apache::loncoursedata::get_classlist($cdom,$cnum);
        my $now = time;
        foreach my $student (keys(%{$classlist})) {
            my $match = 0;
            my $secmatch = 0;
            my $section = $$classlist{$student}[$idx{section}];
            my $status = $$classlist{$student}[$idx{status}];
            if ($section eq '') {
                $section = 'none';
            }
            if ((ref($sections) eq 'ARRAY') && (@{$sections} > 0)) {
                if (grep(/^all$/,@{$sections})) {
                    $secmatch = 1;
                } elsif ($$classlist{$student}[$idx{section}] eq '') {
                    if (grep(/^none$/,@{$sections})) {
                        $secmatch = 1;
                    }
                } else {  
		    if (grep(/^\Q$section\E$/,@{$sections})) {
		        $secmatch = 1;
                    }
		}
                if (!$secmatch) {
                    next;
                }
            }
            if (defined($$types{'active'})) {
                if ($$classlist{$student}[$idx{status}] eq 'Active') {
                    push(@{$$users{st}{$student}},'active');
                    $match = 1;
                }
            }
            if (defined($$types{'previous'})) {
                if ($$classlist{$student}[$idx{status}] eq 'Expired') {
                    push(@{$$users{st}{$student}},'previous');
                    $match = 1;
                }
            }
            if (defined($$types{'future'})) {
                if ($$classlist{$student}[$idx{status}] eq 'Future') {
                    push(@{$$users{st}{$student}},'future');
                    $match = 1;
                }
            }
            if ($match) {
                push(@{$seclists{$student}},$section);
                if (ref($userdata) eq 'HASH') {
                    $$userdata{$student} = $$classlist{$student};
                }
                if (ref($statushash) eq 'HASH') {
                    $statushash->{$student}{'st'}{$section} = $status;
                }
            }
        }
    }
    if ((@{$roles} > 1) || ((@{$roles} == 1) && ($$roles[0] ne "st"))) {
        my %coursepersonnel = &Apache::lonnet::dump('nohist_userroles',$cdom,$cnum);
        my $now = time;
        my %displaystatus = ( previous => 'Expired',
                              active   => 'Active',
                              future   => 'Future',
                            );
        my %nothide;
        if ($hidepriv) {
            my %coursehash=&Apache::lonnet::coursedescription($cdom.'_'.$cnum);
            foreach my $user (split(/\s*\,\s*/,$coursehash{'nothideprivileged'})) {
                if ($user !~ /:/) {
                    $nothide{join(':',split(/[\@]/,$user))}=1;
                } else {
                    $nothide{$user} = 1;
                }
            }
        }
        foreach my $person (sort(keys(%coursepersonnel))) {
            my $match = 0;
            my $secmatch = 0;
            my $status;
            my ($role,$user,$usec) = ($person =~ /^([^:]*):([^:]+:[^:]+):([^:]*)/);
            $user =~ s/:$//;
            my ($end,$start) = split(/:/,$coursepersonnel{$person});
            if ($end == -1 || $start == -1) {
                next;
            }
            if (($role) && ((grep(/^\Q$role\E$/,@{$roles})) ||
                (grep(/^cr$/,@{$roles}) && $role =~ /^cr\//))) {
                my ($uname,$udom) = split(/:/,$user);
                if ((ref($sections) eq 'ARRAY') && (@{$sections} > 0)) {
                    if (grep(/^all$/,@{$sections})) {
                        $secmatch = 1;
                    } elsif ($usec eq '') {
                        if (grep(/^none$/,@{$sections})) {
                            $secmatch = 1;
                        }
                    } else {
                        if (grep(/^\Q$usec\E$/,@{$sections})) {
                            $secmatch = 1;
                        }
                    }
                    if (!$secmatch) {
                        next;
                    }
                }
                if ($usec eq '') {
                    $usec = 'none';
                }
                if ($uname ne '' && $udom ne '') {
                    if ($hidepriv) {
                        if ((&Apache::lonnet::privileged($uname,$udom)) &&
                            (!$nothide{$uname.':'.$udom})) {
                            next;
                        }
                    }
                    if ($end > 0 && $end < $now) {
                        $status = 'previous';
                    } elsif ($start > $now) {
                        $status = 'future';
                    } else {
                        $status = 'active';
                    }
                    foreach my $type (keys(%{$types})) { 
                        if ($status eq $type) {
                            if (!grep(/^\Q$type\E$/,@{$$users{$role}{$user}})) {
                                push(@{$$users{$role}{$user}},$type);
                            }
                            $match = 1;
                        }
                    }
                    if (($match) && (ref($userdata) eq 'HASH')) {
                        if (!exists($$userdata{$uname.':'.$udom})) {
			    &get_user_info($udom,$uname,\%idx,$userdata);
                        }
                        if (!grep(/^\Q$usec\E$/,@{$seclists{$uname.':'.$udom}})) {
                            push(@{$seclists{$uname.':'.$udom}},$usec);
                        }
                        if (ref($statushash) eq 'HASH') {
                            $statushash->{$uname.':'.$udom}{$role}{$usec} = $displaystatus{$status};
                        }
                    }
                }
            }
        }
        if (grep(/^ow$/,@{$roles})) {
            if ((defined($cdom)) && (defined($cnum))) {
                my %csettings = &Apache::lonnet::get('environment',['internal.courseowner'],$cdom,$cnum);
                if ( defined($csettings{'internal.courseowner'}) ) {
                    my $owner = $csettings{'internal.courseowner'};
                    next if ($owner eq '');
                    my ($ownername,$ownerdom);
                    if ($owner =~ /^([^:]+):([^:]+)$/) {
                        $ownername = $1;
                        $ownerdom = $2;
                    } else {
                        $ownername = $owner;
                        $ownerdom = $cdom;
                        $owner = $ownername.':'.$ownerdom;
                    }
                    @{$$users{'ow'}{$owner}} = 'any';
                    if (defined($userdata) && 
			!exists($$userdata{$owner})) {
			&get_user_info($ownerdom,$ownername,\%idx,$userdata);
                        if (!grep(/^none$/,@{$seclists{$owner}})) {
                            push(@{$seclists{$owner}},'none');
                        }
                        if (ref($statushash) eq 'HASH') {
                            $statushash->{$owner}{'ow'}{'none'} = 'Any';
                        }
		    }
                }
            }
        }
        foreach my $user (keys(%seclists)) {
            @{$seclists{$user}} = (sort {$a <=> $b} @{$seclists{$user}});
            $$userdata{$user}[$idx{section}] = join(',',@{$seclists{$user}});
        }
    }
    return;
}

sub get_user_info {
    my ($udom,$uname,$idx,$userdata) = @_;
    $$userdata{$uname.':'.$udom}[$$idx{fullname}] = 
	&plainname($uname,$udom,'lastname');
    $$userdata{$uname.':'.$udom}[$$idx{uname}] = $uname;
    $$userdata{$uname.':'.$udom}[$$idx{udom}] = $udom;
    my %idhash =  &Apache::lonnet::idrget($udom,($uname));
    $$userdata{$uname.':'.$udom}[$$idx{id}] = $idhash{$uname}; 
    return;
}

###############################################

=pod

=item * &get_user_quota()

Retrieves quota assigned for storage of portfolio files for a user  

Incoming parameters:
1. user's username
2. user's domain

Returns:
1. Disk quota (in Mb) assigned to student.
2. (Optional) Type of setting: custom or default
   (individually assigned or default for user's 
   institutional status).
3. (Optional) - User's institutional status (e.g., faculty, staff
   or student - types as defined in localenroll::inst_usertypes 
   for user's domain, which determines default quota for user.
4. (Optional) - Default quota which would apply to the user.

If a value has been stored in the user's environment, 
it will return that, otherwise it returns the maximal default
defined for the user's instituional status(es) in the domain.

=cut

###############################################


sub get_user_quota {
    my ($uname,$udom) = @_;
    my ($quota,$quotatype,$settingstatus,$defquota);
    if (!defined($udom)) {
        $udom = $env{'user.domain'};
    }
    if (!defined($uname)) {
        $uname = $env{'user.name'};
    }
    if (($udom eq '' || $uname eq '') ||
        ($udom eq 'public') && ($uname eq 'public')) {
        $quota = 0;
        $quotatype = 'default';
        $defquota = 0; 
    } else {
        my $inststatus;
        if ($udom eq $env{'user.domain'} && $uname eq $env{'user.name'}) {
            $quota = $env{'environment.portfolioquota'};
            $inststatus = $env{'environment.inststatus'};
        } else {
            my %userenv = 
                &Apache::lonnet::get('environment',['portfolioquota',
                                     'inststatus'],$udom,$uname);
            my ($tmp) = keys(%userenv);
            if ($tmp !~ /^(con_lost|error|no_such_host)/i) {
                $quota = $userenv{'portfolioquota'};
                $inststatus = $userenv{'inststatus'};
            } else {
                undef(%userenv);
            }
        }
        ($defquota,$settingstatus) = &default_quota($udom,$inststatus);
        if ($quota eq '') {
            $quota = $defquota;
            $quotatype = 'default';
        } else {
            $quotatype = 'custom';
        }
    }
    if (wantarray) {
        return ($quota,$quotatype,$settingstatus,$defquota);
    } else {
        return $quota;
    }
}

###############################################

=pod

=item * &default_quota()

Retrieves default quota assigned for storage of user portfolio files,
given an (optional) user's institutional status.

Incoming parameters:
1. domain
2. (Optional) institutional status(es).  This is a : separated list of 
   status types (e.g., faculty, staff, student etc.)
   which apply to the user for whom the default is being retrieved.
   If the institutional status string in undefined, the domain
   default quota will be returned. 

Returns:
1. Default disk quota (in Mb) for user portfolios in the domain.
2. (Optional) institutional type which determined the value of the
   default quota.

If a value has been stored in the domain's configuration db,
it will return that, otherwise it returns 20 (for backwards 
compatibility with domains which have not set up a configuration
db file; the original statically defined portfolio quota was 20 Mb). 

If the user's status includes multiple types (e.g., staff and student),
the largest default quota which applies to the user determines the
default quota returned.

=cut

###############################################


sub default_quota {
    my ($udom,$inststatus) = @_;
    my ($defquota,$settingstatus);
    my %quotahash = &Apache::lonnet::get_dom('configuration',
                                            ['quotas'],$udom);
    if (ref($quotahash{'quotas'}) eq 'HASH') {
        if ($inststatus ne '') {
            my @statuses = split(/:/,$inststatus);
            foreach my $item (@statuses) {
                if ($quotahash{'quotas'}{$item} ne '') {
                    if ($defquota eq '') {
                        $defquota = $quotahash{'quotas'}{$item};
                        $settingstatus = $item;
                    } elsif ($quotahash{'quotas'}{$item} > $defquota) {
                        $defquota = $quotahash{'quotas'}{$item};
                        $settingstatus = $item;
                    }
                }
            }
        }
        if ($defquota eq '') {
            $defquota = $quotahash{'quotas'}{'default'};
            $settingstatus = 'default';
        }
    } else {
        $settingstatus = 'default';
        $defquota = 20;
    }
    if (wantarray) {
        return ($defquota,$settingstatus);
    } else {
        return $defquota;
    }
}

sub get_secgrprole_info {
    my ($cdom,$cnum,$needroles,$type)  = @_;
    my %sections_count = &get_sections($cdom,$cnum);
    my @sections =  (sort {$a <=> $b} keys(%sections_count));
    my %curr_groups = &Apache::longroup::coursegroups($cdom,$cnum);
    my @groups = sort(keys(%curr_groups));
    my $allroles = [];
    my $rolehash;
    my $accesshash = {
                     active => 'Currently has access',
                     future => 'Will have future access',
                     previous => 'Previously had access',
                  };
    if ($needroles) {
        $rolehash = {'all' => 'all'};
        my %user_roles = &Apache::lonnet::dump('nohist_userroles',$cdom,$cnum);
	if (&Apache::lonnet::error(%user_roles)) {
	    undef(%user_roles);
	}
        foreach my $item (keys(%user_roles)) {
            my ($role)=split(/\:/,$item,2);
            if ($role eq 'cr') { next; }
            if ($role =~ /^cr/) {
                $$rolehash{$role} = (split('/',$role))[3];
            } else {
                $$rolehash{$role} = &Apache::lonnet::plaintext($role,$type);
            }
        }
        foreach my $key (sort(keys(%{$rolehash}))) {
            push(@{$allroles},$key);
        }
        push (@{$allroles},'st');
        $$rolehash{'st'} = &Apache::lonnet::plaintext('st',$type);
    }
    return (\@sections,\@groups,$allroles,$rolehash,$accesshash);
}

sub user_picker {
    my ($dom,$srch,$forcenewuser,$caller,$cancreate,$usertype) = @_;
    my $currdom = $dom;
    my %curr_selected = (
                        srchin => 'dom',
                        srchby => 'lastname',
                      );
    my $srchterm;
    if ((ref($srch) eq 'HASH') && ($env{'form.origform'} ne 'crtusername')) {
        if ($srch->{'srchby'} ne '') {
            $curr_selected{'srchby'} = $srch->{'srchby'};
        }
        if ($srch->{'srchin'} ne '') {
            $curr_selected{'srchin'} = $srch->{'srchin'};
        }
        if ($srch->{'srchtype'} ne '') {
            $curr_selected{'srchtype'} = $srch->{'srchtype'};
        }
        if ($srch->{'srchdomain'} ne '') {
            $currdom = $srch->{'srchdomain'};
        }
        $srchterm = $srch->{'srchterm'};
    }
    my %lt=&Apache::lonlocal::texthash(
                    'usr'       => 'Search criteria',
                    'doma'      => 'Domain/institution to search',
                    'uname'     => 'username',
                    'lastname'  => 'last name',
                    'lastfirst' => 'last name, first name',
                    'crs'       => 'in this course',
                    'dom'       => 'in selected LON-CAPA domain', 
                    'alc'       => 'all LON-CAPA',
                    'instd'     => 'in institutional directory for selected domain',
                    'exact'     => 'is',
                    'contains'  => 'contains',
                    'begins'    => 'begins with',
                    'youm'      => "You must include some text to search for.",
                    'thte'      => "The text you are searching for must contain at least two characters when using a 'begins' type search.",
                    'thet'      => "The text you are searching for must contain at least three characters when using a 'contains' type search.",
                    'yomc'      => "You must choose a domain when using an institutional directory search.",
                    'ymcd'      => "You must choose a domain when using a domain search.",
                    'whus'      => "When using searching by last,first you must include a comma as separator between last name and first name.",
                    'whse'      => "When searching by last,first you must include at least one character in the first name.",
                     'thfo'     => "The following need to be corrected before the search can be run:",
                                       );
    my $domform = &select_dom_form($currdom,'srchdomain',1,1);
    my $srchinsel = ' <select name="srchin">';

    my @srchins = ('crs','dom','alc','instd');

    foreach my $option (@srchins) {
        # FIXME 'alc' option unavailable until 
        #       loncreateuser::print_user_query_page()
        #       has been completed.
        next if ($option eq 'alc');
        next if ($option eq 'crs' && !$env{'request.course.id'});
        if ($curr_selected{'srchin'} eq $option) {
            $srchinsel .= ' 
   <option value="'.$option.'" selected="selected">'.$lt{$option}.'</option>';
        } else {
            $srchinsel .= '
   <option value="'.$option.'">'.$lt{$option}.'</option>';
        }
    }
    $srchinsel .= "\n  </select>\n";

    my $srchbysel =  ' <select name="srchby">';
    foreach my $option ('lastname','lastfirst','uname') {
        if ($curr_selected{'srchby'} eq $option) {
            $srchbysel .= '
   <option value="'.$option.'" selected="selected">'.$lt{$option}.'</option>';
        } else {
            $srchbysel .= '
   <option value="'.$option.'">'.$lt{$option}.'</option>';
         }
    }
    $srchbysel .= "\n  </select>\n";

    my $srchtypesel = ' <select name="srchtype">';
    foreach my $option ('begins','contains','exact') {
        if ($curr_selected{'srchtype'} eq $option) {
            $srchtypesel .= '
   <option value="'.$option.'" selected="selected">'.$lt{$option}.'</option>';
        } else {
            $srchtypesel .= '
   <option value="'.$option.'">'.$lt{$option}.'</option>';
        }
    }
    $srchtypesel .= "\n  </select>\n";

    my ($newuserscript,$new_user_create);

    if ($forcenewuser) {
        if (ref($srch) eq 'HASH') {
            if ($srch->{'srchby'} eq 'uname' && $srch->{'srchtype'} eq 'exact' && $srch->{'srchin'} eq 'dom' && $srch->{'srchdomain'} eq $env{'request.role.domain'}) {
                if ($cancreate) {
                    $new_user_create = '<p> <input type="submit" name="forcenew" value="'.&HTML::Entities::encode(&mt('Make new user "[_1]"',$srchterm),'<>&"').'" onclick="javascript:setSearch(\'1\','.$caller.');" /> </p>';
                } else {
                    my $helplink = ' href="javascript:helpMenu('."'display'".')"';
                    my %usertypetext = (
                        official   => 'institutional',
                        unofficial => 'non-institutional',
                    );
                    $new_user_create = '<br /><span class="LC_warning">'.&mt("You are not authorized to create new $usertypetext{$usertype} users in this domain.").' '.&mt('Contact the <a[_1]>helpdesk</a> for assistance.',$helplink).'</span><br /><br />';
                }
            }
        }

        $newuserscript = <<"ENDSCRIPT";

function setSearch(createnew,callingForm) {
    if (createnew == 1) {
        for (var i=0; i<callingForm.srchby.length; i++) {
            if (callingForm.srchby.options[i].value == 'uname') {
                callingForm.srchby.selectedIndex = i;
            }
        }
        for (var i=0; i<callingForm.srchin.length; i++) {
            if ( callingForm.srchin.options[i].value == 'dom') {
		callingForm.srchin.selectedIndex = i;
            }
        }
        for (var i=0; i<callingForm.srchtype.length; i++) {
            if (callingForm.srchtype.options[i].value == 'exact') {
                callingForm.srchtype.selectedIndex = i;
            }
        }
        for (var i=0; i<callingForm.srchdomain.length; i++) {
            if (callingForm.srchdomain.options[i].value == '$env{'request.role.domain'}') {
                callingForm.srchdomain.selectedIndex = i;
            }
        }
    }
}
ENDSCRIPT

    }

    my $output = <<"END_BLOCK";
<script type="text/javascript">
function validateEntry(callingForm) {

    var checkok = 1;
    var srchin;
    for (var i=0; i<callingForm.srchin.length; i++) {
	if ( callingForm.srchin[i].checked ) {
	    srchin = callingForm.srchin[i].value;
	}
    }

    var srchtype = callingForm.srchtype.options[callingForm.srchtype.selectedIndex].value;
    var srchby = callingForm.srchby.options[callingForm.srchby.selectedIndex].value;
    var srchdomain = callingForm.srchdomain.options[callingForm.srchdomain.selectedIndex].value;
    var srchterm =  callingForm.srchterm.value;
    var srchin = callingForm.srchin.options[callingForm.srchin.selectedIndex].value;
    var msg = "";

    if (srchterm == "") {
        checkok = 0;
        msg += "$lt{'youm'}\\n";
    }

    if (srchtype== 'begins') {
        if (srchterm.length < 2) {
            checkok = 0;
            msg += "$lt{'thte'}\\n";
        }
    }

    if (srchtype== 'contains') {
        if (srchterm.length < 3) {
            checkok = 0;
            msg += "$lt{'thet'}\\n";
        }
    }
    if (srchin == 'instd') {
        if (srchdomain == '') {
            checkok = 0;
            msg += "$lt{'yomc'}\\n";
        }
    }
    if (srchin == 'dom') {
        if (srchdomain == '') {
            checkok = 0;
            msg += "$lt{'ymcd'}\\n";
        }
    }
    if (srchby == 'lastfirst') {
        if (srchterm.indexOf(",") == -1) {
            checkok = 0;
            msg += "$lt{'whus'}\\n";
        }
        if (srchterm.indexOf(",") == srchterm.length -1) {
            checkok = 0;
            msg += "$lt{'whse'}\\n";
        }
    }
    if (checkok == 0) {
        alert("$lt{'thfo'}\\n"+msg);
        return;
    }
    if (checkok == 1) {
        callingForm.submit();
    }
}

$newuserscript

</script>

$new_user_create

<table>
 <tr>
  <td>$lt{'doma'}:</td>
  <td>$domform</td>
  </td>
 </tr>
 <tr>
  <td>$lt{'usr'}:</td>
  <td>$srchbysel
      $srchtypesel 
      <input type="text" size="15" name="srchterm" value="$srchterm" />
      $srchinsel 
  </td>
 </tr>
</table>
<br />
END_BLOCK

    return $output;
}

sub user_rule_check {
    my ($usershash,$checks,$alerts,$rulematch,$inst_results,$curr_rules,$got_rules) = @_;
    my $response;
    if (ref($usershash) eq 'HASH') {
        foreach my $user (keys(%{$usershash})) {
            my ($uname,$udom) = split(/:/,$user);
            next if ($udom eq '' || $uname eq '');
            my ($id,$newuser);
            if (ref($usershash->{$user}) eq 'HASH') {
                $newuser = $usershash->{$user}->{'newuser'};
                $id = $usershash->{$user}->{'id'};
            }
            my $inst_response;
            if (ref($checks) eq 'HASH') {
                if (defined($checks->{'username'})) {
                    ($inst_response,%{$inst_results->{$user}}) = 
                        &Apache::lonnet::get_instuser($udom,$uname);
                } elsif (defined($checks->{'id'})) {
                    ($inst_response,%{$inst_results->{$user}}) =
                        &Apache::lonnet::get_instuser($udom,undef,$id);
                }
            } else {
                ($inst_response,%{$inst_results->{$user}}) =
                    &Apache::lonnet::get_instuser($udom,$uname);
                return;
            }
            if (!$got_rules->{$udom}) {
                my %domconfig = &Apache::lonnet::get_dom('configuration',
                                                  ['usercreation'],$udom);
                if (ref($domconfig{'usercreation'}) eq 'HASH') {
                    foreach my $item ('username','id') {
                        if (ref($domconfig{'usercreation'}{$item.'_rule'}) eq 'ARRAY') {
                            $$curr_rules{$udom}{$item} = 
                                $domconfig{'usercreation'}{$item.'_rule'};
                        }
                    }
                }
                $got_rules->{$udom} = 1;  
            }
            foreach my $item (keys(%{$checks})) {
                if (ref($$curr_rules{$udom}) eq 'HASH') {
                    if (ref($$curr_rules{$udom}{$item}) eq 'ARRAY') {
                        if (@{$$curr_rules{$udom}{$item}} > 0) {
                            my %rule_check = &Apache::lonnet::inst_rulecheck($udom,$uname,$id,$item,$$curr_rules{$udom}{$item});
                            foreach my $rule (@{$$curr_rules{$udom}{$item}}) {
                                if ($rule_check{$rule}) {
                                    $$rulematch{$user}{$item} = $rule;
                                    if ($inst_response eq 'ok') {
                                        if (ref($inst_results) eq 'HASH') {
                                            if (ref($inst_results->{$user}) eq 'HASH') {
                                                if (keys(%{$inst_results->{$user}}) == 0) {
                                                    $$alerts{$item}{$udom}{$uname} = 1;
                                                }
                                            }
                                        }
                                    }
                                    last;
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    return;
}

sub user_rule_formats {
    my ($domain,$domdesc,$curr_rules,$check) = @_;
    my %text = ( 
                 'username' => 'Usernames',
                 'id'       => 'IDs',
               );
    my $output;
    my ($rules,$ruleorder) = &Apache::lonnet::inst_userrules($domain,$check);
    if ((ref($rules) eq 'HASH') && (ref($ruleorder) eq 'ARRAY')) {
        if (@{$ruleorder} > 0) {
            $output = '<br />'.&mt("$text{$check} with the following format(s) may <span class=\"LC_cusr_emph\">only</span> be used for verified users at [_1]:",$domdesc).' <ul>';
            foreach my $rule (@{$ruleorder}) {
                if (ref($curr_rules) eq 'ARRAY') {
                    if (grep(/^\Q$rule\E$/,@{$curr_rules})) {
                        if (ref($rules->{$rule}) eq 'HASH') {
                            $output .= '<li>'.$rules->{$rule}{'name'}.': '.
                                        $rules->{$rule}{'desc'}.'</li>';
                        }
                    }
                }
            }
            $output .= '</ul>';
        }
    }
    return $output;
}

sub instrule_disallow_msg {
    my ($checkitem,$domdesc,$count,$mode) = @_;
    my $response;
    my %text = (
                  item   => 'username',
                  items  => 'usernames',
                  match  => 'matches',
                  do     => 'does',
                  action => 'a username',
                  one    => 'one',
               );
    if ($count > 1) {
        $text{'item'} = 'usernames';
        $text{'match'} ='match';
        $text{'do'} = 'do';
        $text{'action'} = 'usernames',
        $text{'one'} = 'ones';
    }
    if ($checkitem eq 'id') {
        $text{'items'} = 'IDs';
        $text{'item'} = 'ID';
        $text{'action'} = 'an ID';
        if ($count > 1) {
            $text{'item'} = 'IDs';
            $text{'action'} = 'IDs';
        }
    }
    $response = &mt("The $text{'item'} you chose $text{'match'} the format of $text{'items'} defined for <span class=\"LC_cusr_emph\">[_1]</span>, but the $text{'item'} $text{'do'} not exist in the institutional directory.",$domdesc).'<br />';
    if ($mode eq 'upload') {
        if ($checkitem eq 'username') {
            $response .= &mt("You will need to modify your upload file so it will include $text{'action'} with a different format --  $text{'one'} that will not conflict with 'official' institutional $text{'items'}.");
        } elsif ($checkitem eq 'id') {
            $response .= &mt("Either upload a file which includes $text{'action'} with a different format --  $text{'one'} that will not conflict with 'official' institutional $text{'items'}, or when associating fields with data columns, omit an association for the ID/Student Number field.");
        }
    } elsif ($mode eq 'selfcreate') {
        if ($checkitem eq 'id') {
            $response .= &mt("You must either choose $text{'action'} with a different format --  $text{'one'} that will not conflict with 'official' institutional $text{'items'}, or leave the ID field blank.");
        }
    } else {
        if ($checkitem eq 'username') {
            $response .= &mt("You must choose $text{'action'} with a different format --  $text{'one'} that will not conflict with 'official' institutional $text{'items'}.");
        } elsif ($checkitem eq 'id') {
            $response .= &mt("You must either choose $text{'action'} with a different format --  $text{'one'} that will not conflict with 'official' institutional $text{'items'}, or leave the ID field blank.");
        }
    }
    return $response;
}

sub personal_data_fieldtitles {
    my %fieldtitles = &Apache::lonlocal::texthash (
                        id => 'Student/Employee ID',
                        permanentemail => 'E-mail address',
                        lastname => 'Last Name',
                        firstname => 'First Name',
                        middlename => 'Middle Name',
                        generation => 'Generation',
                        gen => 'Generation',
                   );
    return %fieldtitles;
}

sub sorted_inst_types {
    my ($dom) = @_;
    my ($usertypes,$order) = &Apache::lonnet::retrieve_inst_usertypes($dom);
    my $othertitle = &mt('All users');
    if ($env{'request.course.id'}) {
        $othertitle  = &mt('Any users');
    }
    my @types;
    if (ref($order) eq 'ARRAY') {
        @types = @{$order};
    }
    if (@types == 0) {
        if (ref($usertypes) eq 'HASH') {
            @types = sort(keys(%{$usertypes}));
        }
    }
    if (keys(%{$usertypes}) > 0) {
        $othertitle = &mt('Other users');
    }
    return ($othertitle,$usertypes,\@types);
}

sub get_institutional_codes {
    my ($settings,$allcourses,$LC_code) = @_;
# Get complete list of course sections to update
    my @currsections = ();
    my @currxlists = ();
    my $coursecode = $$settings{'internal.coursecode'};

    if ($$settings{'internal.sectionnums'} ne '') {
        @currsections = split(/,/,$$settings{'internal.sectionnums'});
    }

    if ($$settings{'internal.crosslistings'} ne '') {
        @currxlists = split(/,/,$$settings{'internal.crosslistings'});
    }

    if (@currxlists > 0) {
        foreach (@currxlists) {
            if (m/^([^:]+):(\w*)$/) {
                unless (grep/^$1$/,@{$allcourses}) {
                    push @{$allcourses},$1;
                    $$LC_code{$1} = $2;
                }
            }
        }
    }
 
    if (@currsections > 0) {
        foreach (@currsections) {
            if (m/^(\w+):(\w*)$/) {
                my $sec = $coursecode.$1;
                my $lc_sec = $2;
                unless (grep/^$sec$/,@{$allcourses}) {
                    push @{$allcourses},$sec;
                    $$LC_code{$sec} = $lc_sec;
                }
            }
        }
    }
    return;
}

=pod

=back

=head1 HTTP Helpers

=over 4

=item * &get_unprocessed_cgi($query,$possible_names)

Modify the %env hash to contain unprocessed CGI form parameters held in
$query.  The parameters listed in $possible_names (an array reference),
will be set in $env{'form.name'} if they do not already exist.

Typically called with $ENV{'QUERY_STRING'} as the first parameter.  
$possible_names is an ref to an array of form element names.  As an example:
get_unprocessed_cgi($ENV{'QUERY_STRING'},['uname','udom']);
will result in $env{'form.uname'} and $env{'form.udom'} being set.

=cut

sub get_unprocessed_cgi {
  my ($query,$possible_names)= @_;
  # $Apache::lonxml::debug=1;
  foreach my $pair (split(/&/,$query)) {
    my ($name, $value) = split(/=/,$pair);
    $name = &unescape($name);
    if (!defined($possible_names) || (grep {$_ eq $name} @$possible_names)) {
      $value =~ tr/+/ /;
      $value =~ s/%([a-fA-F0-9][a-fA-F0-9])/pack("C",hex($1))/eg;
      unless (defined($env{'form.'.$name})) { &add_to_env('form.'.$name,$value) };
    }
  }
}

=pod

=item * &cacheheader() 

returns cache-controlling header code

=cut

sub cacheheader {
    unless ($env{'request.method'} eq 'GET') { return ''; }
    my $date=strftime("%a, %d %b %Y %H:%M:%S GMT",gmtime);
    my $output .='<meta HTTP-EQUIV="Expires" CONTENT="'.$date.'" />
                <meta HTTP-EQUIV="Cache-control" CONTENT="no-cache" />
                <meta HTTP-EQUIV="Pragma" CONTENT="no-cache" />';
    return $output;
}

=pod

=item * &no_cache($r) 

specifies header code to not have cache

=cut

sub no_cache {
    my ($r) = @_;
    if ($ENV{'REQUEST_METHOD'} ne 'GET' &&
	$env{'request.method'} ne 'GET') { return ''; }
    my $date=strftime("%a, %d %b %Y %H:%M:%S GMT",gmtime(time));
    $r->no_cache(1);
    $r->header_out("Expires" => $date);
    $r->header_out("Pragma" => "no-cache");
}

sub content_type {
    my ($r,$type,$charset) = @_;
    if ($r) {
	#  Note that printout.pl calls this with undef for $r.
	&no_cache($r);
    }
    if ($env{'browser.mathml'} && $type eq 'text/html') { $type='text/xml'; }
    unless ($charset) {
	$charset=&Apache::lonlocal::current_encoding;
    }
    if ($charset) { $type.='; charset='.$charset; }
    if ($r) {
	$r->content_type($type);
    } else {
	print("Content-type: $type\n\n");
    }
}

=pod

=item * &add_to_env($name,$value) 

adds $name to the %env hash with value
$value, if $name already exists, the entry is converted to an array
reference and $value is added to the array.

=cut

sub add_to_env {
  my ($name,$value)=@_;
  if (defined($env{$name})) {
    if (ref($env{$name})) {
      #already have multiple values
      push(@{ $env{$name} },$value);
    } else {
      #first time seeing multiple values, convert hash entry to an arrayref
      my $first=$env{$name};
      undef($env{$name});
      push(@{ $env{$name} },$first,$value);
    }
  } else {
    $env{$name}=$value;
  }
}

=pod

=item * &get_env_multiple($name) 

gets $name from the %env hash, it seemlessly handles the cases where multiple
values may be defined and end up as an array ref.

returns an array of values

=cut

sub get_env_multiple {
    my ($name) = @_;
    my @values;
    if (defined($env{$name})) {
        # exists is it an array
        if (ref($env{$name})) {
            @values=@{ $env{$name} };
        } else {
            $values[0]=$env{$name};
        }
    }
    return(@values);
}

sub ask_for_embedded_content {
    my ($actionurl,$state,$allfiles,$codebase,$args)=@_;
    my $upload_output = '
   <form name="upload_embedded" action="'.$actionurl.'"
                  method="post" enctype="multipart/form-data">';
    $upload_output .= $state;
    $upload_output .= '<b>Upload embedded files</b>:<br />'.&start_data_table();

    my $num = 0;
    foreach my $embed_file (sort {lc($a) cmp lc($b)} keys(%{$allfiles})) {
        $upload_output .= &start_data_table_row().
            '<td>'.$embed_file.'</td><td>';
        if ($args->{'ignore_remote_references'}
            && $embed_file =~ m{^\w+://}) {
            $upload_output.='<span class="LC_warning">'.&mt("URL points to other server.").'</span>';
        } elsif ($args->{'error_on_invalid_names'}
            && $embed_file ne &Apache::lonnet::clean_filename($embed_file,{'keep_path' => 1,})) {

            $upload_output.='<span class="LC_warning">'.&mt("Invalid characters").'</span>';

        } else {
            $upload_output .='
           <input name="embedded_item_'.$num.'" type="file" value="" />
           <input name="embedded_orig_'.$num.'" type="hidden" value="'.&escape($embed_file).'" />';
            my $attrib = join(':',@{$$allfiles{$embed_file}});
            $upload_output .=
                "\n\t\t".
                '<input name="embedded_attrib_'.$num.'" type="hidden" value="'.
                $attrib.'" />';
            if (exists($$codebase{$embed_file})) {
                $upload_output .=
                    "\n\t\t".
                    '<input name="codebase_'.$num.'" type="hidden" value="'.
                    &escape($$codebase{$embed_file}).'" />';
            }
        }
        $upload_output .= '</td>'.&Apache::loncommon::end_data_table_row();
        $num++;
    }
    $upload_output .= &Apache::loncommon::end_data_table().'<br />
   <input type ="hidden" name="number_embedded_items" value="'.$num.'" />
   <input type ="submit" value="'.&mt('Upload Listed Files').'" />
   '.&mt('(only files for which a location has been provided will be uploaded)').'
   </form>';
    return $upload_output;
}

sub upload_embedded {
    my ($context,$dirpath,$uname,$udom,$dir_root,$url_root,$group,$disk_quota,
        $current_disk_usage) = @_;
    my $output;
    for (my $i=0; $i<$env{'form.number_embedded_items'}; $i++) {
        next if (!exists($env{'form.embedded_item_'.$i.'.filename'}));
        my $orig_uploaded_filename =
            $env{'form.embedded_item_'.$i.'.filename'};

        $env{'form.embedded_orig_'.$i} =
            &unescape($env{'form.embedded_orig_'.$i});
        my ($path,$fname) =
            ($env{'form.embedded_orig_'.$i} =~ m{(.*/)([^/]*)});
        # no path, whole string is fname
        if (!$fname) { $fname = $env{'form.embedded_orig_'.$i} };

        $path = $env{'form.currentpath'}.$path;
        $fname = &Apache::lonnet::clean_filename($fname);
        # See if there is anything left
        next if ($fname eq '');

        # Check if file already exists as a file or directory.
        my ($state,$msg);
        if ($context eq 'portfolio') {
            my $port_path = $dirpath;
            if ($group ne '') {
                $port_path = "groups/$group/$port_path";
            }
            ($state,$msg) = &check_for_upload($path,$fname,$group,'embedded_item_'.$i,
                                              $dir_root,$port_path,$disk_quota,
                                              $current_disk_usage,$uname,$udom);
            if ($state eq 'will_exceed_quota'
                || $state eq 'file_locked'
                || $state eq 'file_exists' ) {
                $output .= $msg;
                next;
            }
        } elsif (($context eq 'author') || ($context eq 'testbank')) {
            ($state,$msg) = &check_for_existing($path,$fname,'embedded_item_'.$i);
            if ($state eq 'exists') {
                $output .= $msg;
                next;
            }
        }
        # Check if extension is valid
        if (($fname =~ /\.(\w+)$/) &&
            (&Apache::loncommon::fileembstyle($1) eq 'hdn')) {
            $output .= &mt('Invalid file extension ([_1]) - reserved for LONCAPA use - rename the file with a different extension and re-upload. ',$1);
            next;
        } elsif (($fname =~ /\.(\w+)$/) &&
                 (!defined(&Apache::loncommon::fileembstyle($1)))) {
            $output .= &mt('Unrecognized file extension ([_1]) - rename the file with a proper extension and re-upload.',$1);
            next;
        } elsif ($fname=~/\.(\d+)\.(\w+)$/) {
            $output .= &mt('File name not allowed - rename the file to remove the number immediately before the file extension([_1]) and re-upload.',$2);
            next;
        }

        $env{'form.embedded_item_'.$i.'.filename'}=$fname;
        if ($context eq 'portfolio') {
            my $result=
                &Apache::lonnet::userfileupload('embedded_item_'.$i,'',
                                                $dirpath.$path);
            if ($result !~ m|^/uploaded/|) {
                $output .= '<span class="LC_error">'
                      .&mt('An error occurred ([_1]) while trying to upload [_2] for embedded element [_3].'
                           ,$result,$orig_uploaded_filename,$env{'form.embedded_orig_'.$i})
                      .'</span><br />';
                next;
            } else {
                $output .= '<p>'.&mt('Uploaded [_1]','<span class="LC_filename">'.
                           $path.$fname.'</span>').'</p>';     
            }
        } else {
# Save the file
            my $target = $env{'form.embedded_item_'.$i};
            my $fullpath = $dir_root.$dirpath.'/'.$path;
            my $dest = $fullpath.$fname;
            my $url = $url_root.$dirpath.'/'.$path.$fname;
            my @parts=split(/\//,$fullpath);
            my $count;
            my $filepath = $dir_root;
            for ($count=4;$count<=$#parts;$count++) {
                $filepath .= "/$parts[$count]";
                if ((-e $filepath)!=1) {
                    mkdir($filepath,0770);
                }
            }
            my $fh;
            if (!open($fh,'>'.$dest)) {
                &Apache::lonnet::logthis('Failed to create '.$dest);
                $output .= '<span class="LC_error">'.
                           &mt('An error occurred while trying to upload [_1] for embedded element [_2].',$orig_uploaded_filename,$env{'form.embedded_orig_'.$i}).
                           '</span><br />';
            } else {
                if (!print $fh $env{'form.embedded_item_'.$i}) {
                    &Apache::lonnet::logthis('Failed to write to '.$dest);
                    $output .= '<span class="LC_error">'.
                              &mt('An error occurred while writing the file [_1] for embedded element [_2].',$orig_uploaded_filename,$env{'form.embedded_orig_'.$i}).
                              '</span><br />';
                } else {
                    if ($context eq 'testbank') {
                        $output .= &mt('Embedded file uploaded successfully:').
                                   '&nbsp;<a href="'.$url.'">'.
                                   $orig_uploaded_filename.'</a><br />';
                    } else {
                        $output .= '<font size="+2">'.
                                   &mt('View embedded file: [_1]','<a href="'.$url.'">'.
                                   $orig_uploaded_filename.'</a>').'</font><br />';
                    }
                }
                close($fh);
            }
        }
    }
    return $output;
}

sub check_for_existing {
    my ($path,$fname,$element) = @_;
    my ($state,$msg);
    if (-d $path.'/'.$fname) {
        $state = 'exists';
        $msg = &mt('Unable to upload [_1]. A directory by that name was found in [_2].','<span class="LC_filename">'.$fname.'</span>',$path);
    } elsif (-e $path.'/'.$fname) {
        $state = 'exists';
        $msg = &mt('Unable to upload [_1]. A file by that name was found in [_2].','<span class="LC_filename">'.$fname.'</span>',$path);
    }
    if ($state eq 'exists') {
        $msg = '<span class="LC_error">'.$msg.'</span><br />';
    }
    return ($state,$msg);
}

sub check_for_upload {
    my ($path,$fname,$group,$element,$portfolio_root,$port_path,
        $disk_quota,$current_disk_usage,$uname,$udom) = @_;
    my $filesize = (length($env{'form.'.$element})) / 1000; #express in k (1024?)
    my $getpropath = 1;
    my @dir_list = &Apache::lonnet::dirlist($portfolio_root.$path,$udom,$uname,
                                            $getpropath);
    my $found_file = 0;
    my $locked_file = 0;
    foreach my $line (@dir_list) {
        my ($file_name)=split(/\&/,$line,2);
        if ($file_name eq $fname){
            $file_name = $path.$file_name;
            if ($group ne '') {
                $file_name = $group.$file_name;
            }
            $found_file = 1;
            if (&Apache::lonnet::is_locked($file_name,$udom,$uname) eq 'true') {
                $locked_file = 1;
            }
        }
    }
    if (($current_disk_usage + $filesize) > $disk_quota){
        my $msg = '<span class="LC_error">'.
                &mt('Unable to upload [_1]. (size = [_2] kilobytes). Disk quota will be exceeded.','<span class="LC_filename">'.$fname.'</span>',$filesize).'</span>'.
                  '<br />'.&mt('Disk quota is [_1] kilobytes. Your current disk usage is [_2] kilobytes.',$disk_quota,$current_disk_usage);
        return ('will_exceed_quota',$msg);
    } elsif ($found_file) {
        if ($locked_file) {
            my $msg = '<span class="LC_error">';
            $msg .= &mt('Unable to upload [_1]. A locked file by that name was found in [_2].','<span class="LC_filename">'.$fname.'</span>','<span class="LC_filename">'.$port_path.$env{'form.currentpath'}.'</span>');
            $msg .= '</span><br />';
            $msg .= &mt('You will be able to rename or delete existing [_1] after a grade has been assigned.','<span class="LC_filename">'.$fname.'</span>');
            return ('file_locked',$msg);
        } else {
            my $msg = '<span class="LC_error">';
            $msg .= &mt('Unable to upload [_1]. A file by that name was found in [_2].','<span class="LC_filename">'.$fname.'</span>',$port_path.$env{'form.currentpath'});
            $msg .= '</span>';
            $msg .= '<br />';
            $msg .= &mt('To upload, rename or delete existing [_1] in [_2].','<span class="LC_filename">'.$fname.'</span>', $port_path.$env{'form.currentpath'});
            return ('file_exists',$msg);
        }
    }
}


=pod

=back

=head1 CSV Upload/Handling functions

=over 4

=item * &upfile_store($r)

Store uploaded file, $r should be the HTTP Request object,
needs $env{'form.upfile'}
returns $datatoken to be put into hidden field

=cut

sub upfile_store {
    my $r=shift;
    $env{'form.upfile'}=~s/\r/\n/gs;
    $env{'form.upfile'}=~s/\f/\n/gs;
    $env{'form.upfile'}=~s/\n+/\n/gs;
    $env{'form.upfile'}=~s/\n+$//gs;

    my $datatoken=$env{'user.name'}.'_'.$env{'user.domain'}.
	'_enroll_'.$env{'request.course.id'}.'_'.time.'_'.$$;
    {
        my $datafile = $r->dir_config('lonDaemons').
                           '/tmp/'.$datatoken.'.tmp';
        if ( open(my $fh,">$datafile") ) {
            print $fh $env{'form.upfile'};
            close($fh);
        }
    }
    return $datatoken;
}

=pod

=item * &load_tmp_file($r)

Load uploaded file from tmp, $r should be the HTTP Request object,
needs $env{'form.datatoken'},
sets $env{'form.upfile'} to the contents of the file

=cut

sub load_tmp_file {
    my $r=shift;
    my @studentdata=();
    {
        my $studentfile = $r->dir_config('lonDaemons').
                              '/tmp/'.$env{'form.datatoken'}.'.tmp';
        if ( open(my $fh,"<$studentfile") ) {
            @studentdata=<$fh>;
            close($fh);
        }
    }
    $env{'form.upfile'}=join('',@studentdata);
}

=pod

=item * &upfile_record_sep()

Separate uploaded file into records
returns array of records,
needs $env{'form.upfile'} and $env{'form.upfiletype'}

=cut

sub upfile_record_sep {
    if ($env{'form.upfiletype'} eq 'xml') {
    } else {
	my @records;
	foreach my $line (split(/\n/,$env{'form.upfile'})) {
	    if ($line=~/^\s*$/) { next; }
	    push(@records,$line);
	}
	return @records;
    }
}

=pod

=item * &record_sep($record)

Separate a record into fields $record should be an item from the upfile_record_sep(), needs $env{'form.upfiletype'}

=cut

sub takeleft {
    my $index=shift;
    return substr('0000'.$index,-4,4);
}

sub record_sep {
    my $record=shift;
    my %components=();
    if ($env{'form.upfiletype'} eq 'xml') {
    } elsif ($env{'form.upfiletype'} eq 'space') {
        my $i=0;
        foreach my $field (split(/\s+/,$record)) {
            $field=~s/^(\"|\')//;
            $field=~s/(\"|\')$//;
            $components{&takeleft($i)}=$field;
            $i++;
        }
    } elsif ($env{'form.upfiletype'} eq 'tab') {
        my $i=0;
        foreach my $field (split(/\t/,$record)) {
            $field=~s/^(\"|\')//;
            $field=~s/(\"|\')$//;
            $components{&takeleft($i)}=$field;
            $i++;
        }
    } else {
        my $separator=',';
        if ($env{'form.upfiletype'} eq 'semisv') {
            $separator=';';
        }
        my $i=0;
# the character we are looking for to indicate the end of a quote or a record 
        my $looking_for=$separator;
# do not add the characters to the fields
        my $ignore=0;
# we just encountered a separator (or the beginning of the record)
        my $just_found_separator=1;
# store the field we are working on here
        my $field='';
# work our way through all characters in record
        foreach my $character ($record=~/(.)/g) {
            if ($character eq $looking_for) {
               if ($character ne $separator) {
# Found the end of a quote, again looking for separator
                  $looking_for=$separator;
                  $ignore=1;
               } else {
# Found a separator, store away what we got
                  $components{&takeleft($i)}=$field;
	          $i++;
                  $just_found_separator=1;
                  $ignore=0;
                  $field='';
               }
               next;
            }
# single or double quotation marks after a separator indicate beginning of a quote
# we are now looking for the end of the quote and need to ignore separators
            if ((($character eq '"') || ($character eq "'")) && ($just_found_separator))  {
               $looking_for=$character;
               next;
            }
# ignore would be true after we reached the end of a quote
            if ($ignore) { next; }
            if (($just_found_separator) && ($character=~/\s/)) { next; }
            $field.=$character;
            $just_found_separator=0; 
        }
# catch the very last entry, since we never encountered the separator
        $components{&takeleft($i)}=$field;
    }
    return %components;
}

######################################################
######################################################

=pod

=item * &upfile_select_html()

Return HTML code to select a file from the users machine and specify 
the file type.

=cut

######################################################
######################################################
sub upfile_select_html {
    my %Types = (
                 csv   => &mt('CSV (comma separated values, spreadsheet)'),
                 semisv => &mt('Semicolon separated values'),
                 space => &mt('Space separated'),
                 tab   => &mt('Tabulator separated'),
#                 xml   => &mt('HTML/XML'),
                 );
    my $Str = '<input type="file" name="upfile" size="50" />'.
        '<br />Type: <select name="upfiletype">';
    foreach my $type (sort(keys(%Types))) {
        $Str .= '<option value="'.$type.'" >'.$Types{$type}."</option>\n";
    }
    $Str .= "</select>\n";
    return $Str;
}

sub get_samples {
    my ($records,$toget) = @_;
    my @samples=({});
    my $got=0;
    foreach my $rec (@$records) {
	my %temp = &record_sep($rec);
	if (! grep(/\S/, values(%temp))) { next; }
	if (%temp) {
	    $samples[$got]=\%temp;
	    $got++;
	    if ($got == $toget) { last; }
	}
    }
    return \@samples;
}

######################################################
######################################################

=pod

=item * &csv_print_samples($r,$records)

Prints a table of sample values from each column uploaded $r is an
Apache Request ref, $records is an arrayref from
&Apache::loncommon::upfile_record_sep

=cut

######################################################
######################################################
sub csv_print_samples {
    my ($r,$records) = @_;
    my $samples = &get_samples($records,5);

    $r->print(&mt('Samples').'<br />'.&start_data_table().
              &start_data_table_header_row());
    foreach my $sample (sort({$a <=> $b} keys(%{ $samples->[0] }))) { 
        $r->print('<th>'.&mt('Column&nbsp;[_1]',($sample+1)).'</th>'); }
    $r->print(&end_data_table_header_row());
    foreach my $hash (@$samples) {
	$r->print(&start_data_table_row());
	foreach my $sample (sort({$a <=> $b} keys(%{ $samples->[0] }))) {
	    $r->print('<td>');
	    if (defined($$hash{$sample})) { $r->print($$hash{$sample}); }
	    $r->print('</td>');
	}
	$r->print(&end_data_table_row());
    }
    $r->print(&end_data_table().'<br />'."\n");
}

######################################################
######################################################

=pod

=item * &csv_print_select_table($r,$records,$d)

Prints a table to create associations between values and table columns.

$r is an Apache Request ref,
$records is an arrayref from &Apache::loncommon::upfile_record_sep,
$d is an array of 2 element arrays (internal name, displayed name,defaultcol)

=cut

######################################################
######################################################
sub csv_print_select_table {
    my ($r,$records,$d) = @_;
    my $i=0;
    my $samples = &get_samples($records,1);
    $r->print(&mt('Associate columns with student attributes.')."\n".
	      &start_data_table().&start_data_table_header_row().
              '<th>'.&mt('Attribute').'</th>'.
              '<th>'.&mt('Column').'</th>'.
              &end_data_table_header_row()."\n");
    foreach my $array_ref (@$d) {
	my ($value,$display,$defaultcol)=@{ $array_ref };
	$r->print(&start_data_table_row().'<tr><td>'.$display.'</td>');

	$r->print('<td><select name=f'.$i.
		  ' onchange="javascript:flip(this.form,'.$i.');">');
	$r->print('<option value="none"></option>');
	foreach my $sample (sort({$a <=> $b} keys(%{ $samples->[0] }))) {
	    $r->print('<option value="'.$sample.'"'.
                      ($sample eq $defaultcol ? ' selected="selected" ' : '').
                      '>'.&mt('Column [_1]',($sample+1)).'</option>');
	}
	$r->print('</select></td>'.&end_data_table_row()."\n");
	$i++;
    }
    $r->print(&end_data_table());
    $i--;
    return $i;
}

######################################################
######################################################

=pod

=item * &csv_samples_select_table($r,$records,$d)

Prints a table of sample values from the upload and can make associate samples to internal names.

$r is an Apache Request ref,
$records is an arrayref from &Apache::loncommon::upfile_record_sep,
$d is an array of 2 element arrays (internal name, displayed name)

=cut

######################################################
######################################################
sub csv_samples_select_table {
    my ($r,$records,$d) = @_;
    my $i=0;
    #
    my $max_samples = 5;
    my $samples = &get_samples($records,$max_samples);
    $r->print(&start_data_table().
              &start_data_table_header_row().'<th>'.
              &mt('Field').'</th><th>'.&mt('Samples').'</th>'.
              &end_data_table_header_row());

    foreach my $key (sort(keys(%{ $samples->[0] }))) {
	$r->print(&start_data_table_row().'<td><select name="f'.$i.'"'.
		  ' onchange="javascript:flip(this.form,'.$i.');">');
	foreach my $option (@$d) {
	    my ($value,$display,$defaultcol)=@{ $option };
	    $r->print('<option value="'.$value.'"'.
                      ($i eq $defaultcol ? ' selected="selected" ':'').'>'.
                      $display.'</option>');
	}
	$r->print('</select></td><td>');
	foreach my $line (0..($max_samples-1)) {
	    if (defined($samples->[$line]{$key})) { 
		$r->print($samples->[$line]{$key}."<br />\n"); 
	    }
	}
	$r->print('</td>'.&end_data_table_row());
	$i++;
    }
    $r->print(&end_data_table());
    $i--;
    return($i);
}

######################################################
######################################################

=pod

=item * &clean_excel_name($name)

Returns a replacement for $name which does not contain any illegal characters.

=cut

######################################################
######################################################
sub clean_excel_name {
    my ($name) = @_;
    $name =~ s/[:\*\?\/\\]//g;
    if (length($name) > 31) {
        $name = substr($name,0,31);
    }
    return $name;
}

=pod

=item * &check_if_partid_hidden($id,$symb,$udom,$uname)

Returns either 1 or undef

1 if the part is to be hidden, undef if it is to be shown

Arguments are:

$id the id of the part to be checked
$symb, optional the symb of the resource to check
$udom, optional the domain of the user to check for
$uname, optional the username of the user to check for

=cut

sub check_if_partid_hidden {
    my ($id,$symb,$udom,$uname) = @_;
    my $hiddenparts=&Apache::lonnet::EXT('resource.0.hiddenparts',
					 $symb,$udom,$uname);
    my $truth=1;
    #if the string starts with !, then the list is the list to show not hide
    if ($hiddenparts=~s/^\s*!//) { $truth=undef; }
    my @hiddenlist=split(/,/,$hiddenparts);
    foreach my $checkid (@hiddenlist) {
	if ($checkid =~ /^\s*\Q$id\E\s*$/) { return $truth; }
    }
    return !$truth;
}


############################################################
############################################################

=pod

=back 

=head1 cgi-bin script and graphing routines

=over 4

=item * &get_cgi_id()

Inputs: none

Returns an id which can be used to pass environment variables
to various cgi-bin scripts.  These environment variables will
be removed from the users environment after a given time by
the routine &Apache::lonnet::transfer_profile_to_env.

=cut

############################################################
############################################################
my $uniq=0;
sub get_cgi_id {
    $uniq=($uniq+1)%100000;
    return (time.'_'.$$.'_'.$uniq);
}

############################################################
############################################################

=pod

=item * &DrawBarGraph()

Facilitates the plotting of data in a (stacked) bar graph.
Puts plot definition data into the users environment in order for 
graph.png to plot it.  Returns an <img> tag for the plot.
The bars on the plot are labeled '1','2',...,'n'.

Inputs:

=over 4

=item $Title: string, the title of the plot

=item $xlabel: string, text describing the X-axis of the plot

=item $ylabel: string, text describing the Y-axis of the plot

=item $Max: scalar, the maximum Y value to use in the plot
If $Max is < any data point, the graph will not be rendered.

=item $colors: array ref holding the colors to be used for the data sets when
they are plotted.  If undefined, default values will be used.

=item $labels: array ref holding the labels to use on the x-axis for the bars.

=item @Values: An array of array references.  Each array reference holds data
to be plotted in a stacked bar chart.

=item If the final element of @Values is a hash reference the key/value
pairs will be added to the graph definition.

=back

Returns:

An <img> tag which references graph.png and the appropriate identifying
information for the plot.

=cut

############################################################
############################################################
sub DrawBarGraph {
    my ($Title,$xlabel,$ylabel,$Max,$colors,$labels,@Values)=@_;
    #
    if (! defined($colors)) {
        $colors = ['#33ff00', 
                  '#0033cc', '#990000', '#aaaa66', '#663399', '#ff9933',
                  '#66ccff', '#ff9999', '#cccc33', '#660000', '#33cc66',
                  ]; 
    }
    my $extra_settings = {};
    if (ref($Values[-1]) eq 'HASH') {
        $extra_settings = pop(@Values);
    }
    #
    my $identifier = &get_cgi_id();
    my $id = 'cgi.'.$identifier;        
    if (! @Values || ref($Values[0]) ne 'ARRAY') {
        return '';
    }
    #
    my @Labels;
    if (defined($labels)) {
        @Labels = @$labels;
    } else {
        for (my $i=0;$i<@{$Values[0]};$i++) {
            push (@Labels,$i+1);
        }
    }
    #
    my $NumBars = scalar(@{$Values[0]});
    if ($NumBars < scalar(@Labels)) { $NumBars = scalar(@Labels); }
    my %ValuesHash;
    my $NumSets=1;
    foreach my $array (@Values) {
        next if (! ref($array));
        $ValuesHash{$id.'.data.'.$NumSets++} = 
            join(',',@$array);
    }
    #
    my ($height,$width,$xskip,$bar_width) = (200,120,1,15);
    if ($NumBars < 3) {
        $width = 120+$NumBars*32;
        $xskip = 1;
        $bar_width = 30;
    } elsif ($NumBars < 5) {
        $width = 120+$NumBars*20;
        $xskip = 1;
        $bar_width = 20;
    } elsif ($NumBars < 10) {
        $width = 120+$NumBars*15;
        $xskip = 1;
        $bar_width = 15;
    } elsif ($NumBars <= 25) {
        $width = 120+$NumBars*11;
        $xskip = 5;
        $bar_width = 8;
    } elsif ($NumBars <= 50) {
        $width = 120+$NumBars*8;
        $xskip = 5;
        $bar_width = 4;
    } else {
        $width = 120+$NumBars*8;
        $xskip = 5;
        $bar_width = 4;
    }
    #
    $Max = 1 if ($Max < 1);
    if ( int($Max) < $Max ) {
        $Max++;
        $Max = int($Max);
    }
    $Title  = '' if (! defined($Title));
    $xlabel = '' if (! defined($xlabel));
    $ylabel = '' if (! defined($ylabel));
    $ValuesHash{$id.'.title'}    = &escape($Title);
    $ValuesHash{$id.'.xlabel'}   = &escape($xlabel);
    $ValuesHash{$id.'.ylabel'}   = &escape($ylabel);
    $ValuesHash{$id.'.y_max_value'} = $Max;
    $ValuesHash{$id.'.NumBars'}  = $NumBars;
    $ValuesHash{$id.'.NumSets'}  = $NumSets;
    $ValuesHash{$id.'.PlotType'} = 'bar';
    $ValuesHash{$id.'.Colors'}   = join(',',@{$colors});
    $ValuesHash{$id.'.height'}   = $height;
    $ValuesHash{$id.'.width'}    = $width;
    $ValuesHash{$id.'.xskip'}    = $xskip;
    $ValuesHash{$id.'.bar_width'} = $bar_width;
    $ValuesHash{$id.'.labels'} = join(',',@Labels);
    #
    # Deal with other parameters
    while (my ($key,$value) = each(%$extra_settings)) {
        $ValuesHash{$id.'.'.$key} = $value;
    }
    #
    &Apache::lonnet::appenv(\%ValuesHash);
    return '<img src="/cgi-bin/graph.png?'.$identifier.'" border="1" />';
}

############################################################
############################################################

=pod

=item * &DrawXYGraph()

Facilitates the plotting of data in an XY graph.
Puts plot definition data into the users environment in order for 
graph.png to plot it.  Returns an <img> tag for the plot.

Inputs:

=over 4

=item $Title: string, the title of the plot

=item $xlabel: string, text describing the X-axis of the plot

=item $ylabel: string, text describing the Y-axis of the plot

=item $Max: scalar, the maximum Y value to use in the plot
If $Max is < any data point, the graph will not be rendered.

=item $colors: Array ref containing the hex color codes for the data to be 
plotted in.  If undefined, default values will be used.

=item $Xlabels: Array ref containing the labels to be used for the X-axis.

=item $Ydata: Array ref containing Array refs.  
Each of the contained arrays will be plotted as a separate curve.

=item %Values: hash indicating or overriding any default values which are 
passed to graph.png.  
Possible values are: width, xskip, x_ticks, x_tick_offset, among others.

=back

Returns:

An <img> tag which references graph.png and the appropriate identifying
information for the plot.

=cut

############################################################
############################################################
sub DrawXYGraph {
    my ($Title,$xlabel,$ylabel,$Max,$colors,$Xlabels,$Ydata,%Values)=@_;
    #
    # Create the identifier for the graph
    my $identifier = &get_cgi_id();
    my $id = 'cgi.'.$identifier;
    #
    $Title  = '' if (! defined($Title));
    $xlabel = '' if (! defined($xlabel));
    $ylabel = '' if (! defined($ylabel));
    my %ValuesHash = 
        (
         $id.'.title'  => &escape($Title),
         $id.'.xlabel' => &escape($xlabel),
         $id.'.ylabel' => &escape($ylabel),
         $id.'.y_max_value'=> $Max,
         $id.'.labels'     => join(',',@$Xlabels),
         $id.'.PlotType'   => 'XY',
         );
    #
    if (defined($colors) && ref($colors) eq 'ARRAY') {
        $ValuesHash{$id.'.Colors'}   = join(',',@{$colors});
    }
    #
    if (! ref($Ydata) || ref($Ydata) ne 'ARRAY') {
        return '';
    }
    my $NumSets=1;
    foreach my $array (@{$Ydata}){
        next if (! ref($array));
        $ValuesHash{$id.'.data.'.$NumSets++} = join(',',@$array);
    }
    $ValuesHash{$id.'.NumSets'} = $NumSets-1;
    #
    # Deal with other parameters
    while (my ($key,$value) = each(%Values)) {
        $ValuesHash{$id.'.'.$key} = $value;
    }
    #
    &Apache::lonnet::appenv(\%ValuesHash);
    return '<img src="/cgi-bin/graph.png?'.$identifier.'" border="1" />';
}

############################################################
############################################################

=pod

=item * &DrawXYYGraph()

Facilitates the plotting of data in an XY graph with two Y axes.
Puts plot definition data into the users environment in order for 
graph.png to plot it.  Returns an <img> tag for the plot.

Inputs:

=over 4

=item $Title: string, the title of the plot

=item $xlabel: string, text describing the X-axis of the plot

=item $ylabel: string, text describing the Y-axis of the plot

=item $colors: Array ref containing the hex color codes for the data to be 
plotted in.  If undefined, default values will be used.

=item $Xlabels: Array ref containing the labels to be used for the X-axis.

=item $Ydata1: The first data set

=item $Min1: The minimum value of the left Y-axis

=item $Max1: The maximum value of the left Y-axis

=item $Ydata2: The second data set

=item $Min2: The minimum value of the right Y-axis

=item $Max2: The maximum value of the left Y-axis

=item %Values: hash indicating or overriding any default values which are 
passed to graph.png.  
Possible values are: width, xskip, x_ticks, x_tick_offset, among others.

=back

Returns:

An <img> tag which references graph.png and the appropriate identifying
information for the plot.

=cut

############################################################
############################################################
sub DrawXYYGraph {
    my ($Title,$xlabel,$ylabel,$colors,$Xlabels,$Ydata1,$Min1,$Max1,
                                        $Ydata2,$Min2,$Max2,%Values)=@_;
    #
    # Create the identifier for the graph
    my $identifier = &get_cgi_id();
    my $id = 'cgi.'.$identifier;
    #
    $Title  = '' if (! defined($Title));
    $xlabel = '' if (! defined($xlabel));
    $ylabel = '' if (! defined($ylabel));
    my %ValuesHash = 
        (
         $id.'.title'  => &escape($Title),
         $id.'.xlabel' => &escape($xlabel),
         $id.'.ylabel' => &escape($ylabel),
         $id.'.labels' => join(',',@$Xlabels),
         $id.'.PlotType' => 'XY',
         $id.'.NumSets' => 2,
         $id.'.two_axes' => 1,
         $id.'.y1_max_value' => $Max1,
         $id.'.y1_min_value' => $Min1,
         $id.'.y2_max_value' => $Max2,
         $id.'.y2_min_value' => $Min2,
         );
    #
    if (defined($colors) && ref($colors) eq 'ARRAY') {
        $ValuesHash{$id.'.Colors'}   = join(',',@{$colors});
    }
    #
    if (! ref($Ydata1) || ref($Ydata1) ne 'ARRAY' ||
        ! ref($Ydata2) || ref($Ydata2) ne 'ARRAY'){
        return '';
    }
    my $NumSets=1;
    foreach my $array ($Ydata1,$Ydata2){
        next if (! ref($array));
        $ValuesHash{$id.'.data.'.$NumSets++} = join(',',@$array);
    }
    #
    # Deal with other parameters
    while (my ($key,$value) = each(%Values)) {
        $ValuesHash{$id.'.'.$key} = $value;
    }
    #
    &Apache::lonnet::appenv(\%ValuesHash);
    return '<img src="/cgi-bin/graph.png?'.$identifier.'" border="1" />';
}

############################################################
############################################################

=pod

=back 

=head1 Statistics helper routines?  

Bad place for them but what the hell.

=over 4

=item * &chartlink()

Returns a link to the chart for a specific student.  

Inputs:

=over 4

=item $linktext: The text of the link

=item $sname: The students username

=item $sdomain: The students domain

=back

=back

=cut

############################################################
############################################################
sub chartlink {
    my ($linktext, $sname, $sdomain) = @_;
    my $link = '<a href="/adm/statistics?reportSelected=student_assessment'.
        '&amp;SelectedStudent='.&escape($sname.':'.$sdomain).
        '&amp;chartoutputmode='.HTML::Entities::encode('html, with all links').
       '">'.$linktext.'</a>';
}

#######################################################
#######################################################

=pod

=head1 Course Environment Routines

=over 4

=item * &restore_course_settings()

=item * &store_course_settings()

Restores/Store indicated form parameters from the course environment.
Will not overwrite existing values of the form parameters.

Inputs: 
a scalar describing the data (e.g. 'chart', 'problem_analysis')

a hash ref describing the data to be stored.  For example:
   
%Save_Parameters = ('Status' => 'scalar',
    'chartoutputmode' => 'scalar',
    'chartoutputdata' => 'scalar',
    'Section' => 'array',
    'Group' => 'array',
    'StudentData' => 'array',
    'Maps' => 'array');

Returns: both routines return nothing

=back

=cut

#######################################################
#######################################################
sub store_course_settings {
    return &store_settings($env{'request.course.id'},@_);
}

sub store_settings {
    # save to the environment
    # appenv the same items, just to be safe
    my $udom  = $env{'user.domain'};
    my $uname = $env{'user.name'};
    my ($context,$prefix,$Settings) = @_;
    my %SaveHash;
    my %AppHash;
    while (my ($setting,$type) = each(%$Settings)) {
        my $basename = join('.','internal',$context,$prefix,$setting);
        my $envname = 'environment.'.$basename;
        if (exists($env{'form.'.$setting})) {
            # Save this value away
            if ($type eq 'scalar' &&
                (! exists($env{$envname}) || 
                 $env{$envname} ne $env{'form.'.$setting})) {
                $SaveHash{$basename} = $env{'form.'.$setting};
                $AppHash{$envname}   = $env{'form.'.$setting};
            } elsif ($type eq 'array') {
                my $stored_form;
                if (ref($env{'form.'.$setting})) {
                    $stored_form = join(',',
                                        map {
                                            &escape($_);
                                        } sort(@{$env{'form.'.$setting}}));
                } else {
                    $stored_form = 
                        &escape($env{'form.'.$setting});
                }
                # Determine if the array contents are the same.
                if ($stored_form ne $env{$envname}) {
                    $SaveHash{$basename} = $stored_form;
                    $AppHash{$envname}   = $stored_form;
                }
            }
        }
    }
    my $put_result = &Apache::lonnet::put('environment',\%SaveHash,
                                          $udom,$uname);
    if ($put_result !~ /^(ok|delayed)/) {
        &Apache::lonnet::logthis('unable to save form parameters, '.
                                 'got error:'.$put_result);
    }
    # Make sure these settings stick around in this session, too
    &Apache::lonnet::appenv(\%AppHash);
    return;
}

sub restore_course_settings {
    return &restore_settings($env{'request.course.id'},@_);
}

sub restore_settings {
    my ($context,$prefix,$Settings) = @_;
    while (my ($setting,$type) = each(%$Settings)) {
        next if (exists($env{'form.'.$setting}));
        my $envname = 'environment.internal.'.$context.'.'.$prefix.
            '.'.$setting;
        if (exists($env{$envname})) {
            if ($type eq 'scalar') {
                $env{'form.'.$setting} = $env{$envname};
            } elsif ($type eq 'array') {
                $env{'form.'.$setting} = [ 
                                           map { 
                                               &unescape($_); 
                                           } split(',',$env{$envname})
                                           ];
            }
        }
    }
}

#######################################################
#######################################################

=pod

=head1 Domain E-mail Routines  

=over 4

=item * &build_recipient_list()

Build recipient lists for three types of e-mail:
(a) Error Reports, (b) Package Updates, (c) Help requests, generated by
lonerrorhandler.pm, CHECKRPMS and lonsupportreq.pm respectively.

Inputs:
defmail (scalar - email address of default recipient), 
mailing type (scalar - errormail, packagesmail, or helpdeskmail), 
defdom (domain for which to retrieve configuration settings),
origmail (scalar - email address of recipient from loncapa.conf, 
i.e., predates configuration by DC via domainprefs.pm 

Returns: comma separated list of addresses to which to send e-mail.

=back

=cut

############################################################
############################################################
sub build_recipient_list {
    my ($defmail,$mailing,$defdom,$origmail) = @_;
    my @recipients;
    my $otheremails;
    my %domconfig =
         &Apache::lonnet::get_dom('configuration',['contacts'],$defdom);
    if (ref($domconfig{'contacts'}) eq 'HASH') {
        if (ref($domconfig{'contacts'}{$mailing}) eq 'HASH') {
            my @contacts = ('adminemail','supportemail');
            foreach my $item (@contacts) {
                if ($domconfig{'contacts'}{$mailing}{$item}) {
                    my $addr = $domconfig{'contacts'}{$item}; 
                    if (!grep(/^\Q$addr\E$/,@recipients)) {
                        push(@recipients,$addr);
                    }
                }
                $otheremails = $domconfig{'contacts'}{$mailing}{'others'};
            }
        }
    } elsif ($origmail ne '') {
        push(@recipients,$origmail);
    }
    if ($defmail ne '') {
        push(@recipients,$defmail);
    }
    if ($otheremails) {
        my @others;
        if ($otheremails =~ /,/) {
            @others = split(/,/,$otheremails);
        } else {
            push(@others,$otheremails);
        }
        foreach my $addr (@others) {
            if (!grep(/^\Q$addr\E$/,@recipients)) {
                push(@recipients,$addr);
            }
        }
    }
    my $recipientlist = join(',',@recipients); 
    return $recipientlist;
}

############################################################
############################################################

=pod

=head1 Course Catalog Routines

=over 4

=item * &gather_categories()

Converts category definitions - keys of categories hash stored in  
coursecategories in configuration.db on the primary library server in a 
domain - to an array.  Also generates javascript and idx hash used to 
generate Domain Coordinator interface for editing Course Categories.

Inputs:

categories (reference to hash of category definitions).

cats (reference to array of arrays/hashes which encapsulates hierarchy of
      categories and subcategories).

idx (reference to hash of counters used in Domain Coordinator interface for 
      editing Course Categories).

jsarray (reference to array of categories used to create Javascript arrays for
         Domain Coordinator interface for editing Course Categories).

Returns: nothing

Side effects: populates cats, idx and jsarray. 

=cut

sub gather_categories {
    my ($categories,$cats,$idx,$jsarray) = @_;
    my %counters;
    my $num = 0;
    foreach my $item (keys(%{$categories})) {
        my ($cat,$container,$depth) = map { &unescape($_); } split(/:/,$item);
        if ($container eq '' && $depth == 0) {
            $cats->[$depth][$categories->{$item}] = $cat;
        } else {
            $cats->[$depth]{$container}[$categories->{$item}] = $cat;
        }
        my ($escitem,$tail) = split(/:/,$item,2);
        if ($counters{$tail} eq '') {
            $counters{$tail} = $num;
            $num ++;
        }
        if (ref($idx) eq 'HASH') {
            $idx->{$item} = $counters{$tail};
        }
        if (ref($jsarray) eq 'ARRAY') {
            push(@{$jsarray->[$counters{$tail}]},$item);
        }
    }
    return;
}

=pod

=item * &extract_categories()

Used to generate breadcrumb trails for course categories.

Inputs:

categories (reference to hash of category definitions).

cats (reference to array of arrays/hashes which encapsulates hierarchy of
      categories and subcategories).

trails (reference to array of breacrumb trails for each category).

allitems (reference to hash - key is category key 
         (format: escaped(name):escaped(parent category):depth in hierarchy).

idx (reference to hash of counters used in Domain Coordinator interface for
      editing Course Categories).

jsarray (reference to array of categories used to create Javascript arrays for
         Domain Coordinator interface for editing Course Categories).

subcats (reference to hash of arrays containing all subcategories within each 
         category, -recursive)

Returns: nothing

Side effects: populates trails and allitems hash references.

=cut

sub extract_categories {
    my ($categories,$cats,$trails,$allitems,$idx,$jsarray,$subcats) = @_;
    if (ref($categories) eq 'HASH') {
        &gather_categories($categories,$cats,$idx,$jsarray);
        if (ref($cats->[0]) eq 'ARRAY') {
            for (my $i=0; $i<@{$cats->[0]}; $i++) {
                my $name = $cats->[0][$i];
                my $item = &escape($name).'::0';
                my $trailstr;
                if ($name eq 'instcode') {
                    $trailstr = &mt('Official courses (with institutional codes)');
                } else {
                    $trailstr = $name;
                }
                if ($allitems->{$item} eq '') {
                    push(@{$trails},$trailstr);
                    $allitems->{$item} = scalar(@{$trails})-1;
                }
                my @parents = ($name);
                if (ref($cats->[1]{$name}) eq 'ARRAY') {
                    for (my $j=0; $j<@{$cats->[1]{$name}}; $j++) {
                        my $category = $cats->[1]{$name}[$j];
                        if (ref($subcats) eq 'HASH') {
                            push(@{$subcats->{$item}},&escape($category).':'.&escape($name).':1');
                        }
                        &recurse_categories($cats,2,$category,$trails,$allitems,\@parents,$subcats);
                    }
                } else {
                    if (ref($subcats) eq 'HASH') {
                        $subcats->{$item} = [];
                    }
                }
            }
        }
    }
    return;
}

=pod

=item *&recurse_categories()

Recursively used to generate breadcrumb trails for course categories.

Inputs:

cats (reference to array of arrays/hashes which encapsulates hierarchy of
      categories and subcategories).

depth (current depth in hierarchy of categories and sub-categories - 0 indexed).

category (current course category, for which breadcrumb trail is being generated).

trails (reference to array of breadcrumb trails for each category).

allitems (reference to hash - key is category key
         (format: escaped(name):escaped(parent category):depth in hierarchy).

parents (array containing containers directories for current category, 
         back to top level). 

Returns: nothing

Side effects: populates trails and allitems hash references

=cut

sub recurse_categories {
    my ($cats,$depth,$category,$trails,$allitems,$parents,$subcats) = @_;
    my $shallower = $depth - 1;
    if (ref($cats->[$depth]{$category}) eq 'ARRAY') {
        for (my $k=0; $k<@{$cats->[$depth]{$category}}; $k++) {
            my $name = $cats->[$depth]{$category}[$k];
            my $item = &escape($category).':'.&escape($parents->[-1]).':'.$shallower;
            my $trailstr = join(' -&gt; ',(@{$parents},$category));
            if ($allitems->{$item} eq '') {
                push(@{$trails},$trailstr);
                $allitems->{$item} = scalar(@{$trails})-1;
            }
            my $deeper = $depth+1;
            push(@{$parents},$category);
            if (ref($subcats) eq 'HASH') {
                my $subcat = &escape($name).':'.$category.':'.$depth;
                for (my $j=@{$parents}; $j>=0; $j--) {
                    my $higher;
                    if ($j > 0) {
                        $higher = &escape($parents->[$j]).':'.
                                  &escape($parents->[$j-1]).':'.$j;
                    } else {
                        $higher = &escape($parents->[$j]).'::'.$j;
                    }
                    push(@{$subcats->{$higher}},$subcat);
                }
            }
            &recurse_categories($cats,$deeper,$name,$trails,$allitems,$parents,
                                $subcats);
            pop(@{$parents});
        }
    } else {
        my $item = &escape($category).':'.&escape($parents->[-1]).':'.$shallower;
        my $trailstr = join(' -&gt; ',(@{$parents},$category));
        if ($allitems->{$item} eq '') {
            push(@{$trails},$trailstr);
            $allitems->{$item} = scalar(@{$trails})-1;
        }
    }
    return;
}

=pod

=item *&assign_categories_table()

Create a datatable for display of hierarchical categories in a domain,
with checkboxes to allow a course to be categorized. 

Inputs:

cathash - reference to hash of categories defined for the domain (from
          configuration.db)

currcat - scalar with an & separated list of categories assigned to a course. 

Returns: $output (markup to be displayed) 

=cut

sub assign_categories_table {
    my ($cathash,$currcat) = @_;
    my $output;
    if (ref($cathash) eq 'HASH') {
        my (@cats,@trails,%allitems,%idx,@jsarray,@path,$maxdepth);
        &extract_categories($cathash,\@cats,\@trails,\%allitems,\%idx,\@jsarray);
        $maxdepth = scalar(@cats);
        if (@cats > 0) {
            my $itemcount = 0;
            if (ref($cats[0]) eq 'ARRAY') {
                $output = &Apache::loncommon::start_data_table();
                my @currcategories;
                if ($currcat ne '') {
                    @currcategories = split('&',$currcat);
                }
                for (my $i=0; $i<@{$cats[0]}; $i++) {
                    my $parent = $cats[0][$i];
                    my $css_class = $itemcount%2?' class="LC_odd_row"':'';
                    next if ($parent eq 'instcode');
                    my $item = &escape($parent).'::0';
                    my $checked = '';
                    if (@currcategories > 0) {
                        if (grep(/^\Q$item\E$/,@currcategories)) {
                            $checked = ' checked="checked" ';
                        }
                    }
                    $output .= '<tr '.$css_class.'><td><span class="LC_nobreak">'
                               .'<input type="checkbox" name="usecategory" value="'.
                               $item.'"'.$checked.' />'.$parent.'</span></td>';
                    my $depth = 1;
                    push(@path,$parent);
                    $output .= &assign_category_rows($itemcount,\@cats,$depth,$parent,\@path,\@currcategories);
                    pop(@path);
                    $output .= '</tr><tr><td colspan="'.$maxdepth.'" class="LC_row_separator"></td></tr>';
                    $itemcount ++;
                }
                $output .= &Apache::loncommon::end_data_table();
            }
        }
    }
    return $output;
}

=pod

=item *&assign_category_rows()

Create a datatable row for display of nested categories in a domain,
with checkboxes to allow a course to be categorized,called recursively.

Inputs:

itemcount - track row number for alternating colors

cats - reference to array of arrays/hashes which encapsulates hierarchy of
      categories and subcategories.

depth - current depth in hierarchy of categories and sub-categories - 0 indexed.

parent - parent of current category item

path - Array containing all categories back up through the hierarchy from the
       current category to the top level.

currcategories - reference to array of current categories assigned to the course

Returns: $output (markup to be displayed).

=cut

sub assign_category_rows {
    my ($itemcount,$cats,$depth,$parent,$path,$currcategories) = @_;
    my ($text,$name,$item,$chgstr);
    if (ref($cats) eq 'ARRAY') {
        my $maxdepth = scalar(@{$cats});
        if (ref($cats->[$depth]) eq 'HASH') {
            if (ref($cats->[$depth]{$parent}) eq 'ARRAY') {
                my $numchildren = @{$cats->[$depth]{$parent}};
                my $css_class = $itemcount%2?' class="LC_odd_row"':'';
                $text .= '<td><table class="LC_datatable">';
                for (my $j=0; $j<$numchildren; $j++) {
                    $name = $cats->[$depth]{$parent}[$j];
                    $item = &escape($name).':'.&escape($parent).':'.$depth;
                    my $deeper = $depth+1;
                    my $checked = '';
                    if (ref($currcategories) eq 'ARRAY') {
                        if (@{$currcategories} > 0) {
                            if (grep(/^\Q$item\E$/,@{$currcategories})) {
                                $checked = ' checked="checked" ';
                            }
                        }
                    }
                    $text .= '<tr><td><span class="LC_nobreak"><label>'.
                             '<input type="checkbox" name="usecategory" value="'.
                             $item.'"'.$checked.' />'.$name.'</label></span></td><td>';
                    if (ref($path) eq 'ARRAY') {
                        push(@{$path},$name);
                        $text .= &assign_category_rows($itemcount,$cats,$deeper,$name,$path,$currcategories);
                        pop(@{$path});
                    }
                    $text .= '</td></tr>';
                }
                $text .= '</table></td>';
            }
        }
    }
    return $text;
}

############################################################
############################################################


sub commit_customrole {
    my ($udom,$uname,$url,$three,$four,$five,$start,$end,$context) = @_;
    my $output = &mt('Assigning custom role').' "'.$five.'" by '.$four.':'.$three.' in '.$url.
                         ($start?', '.&mt('starting').' '.localtime($start):'').
                         ($end?', ending '.localtime($end):'').': <b>'.
              &Apache::lonnet::assigncustomrole(
                 $udom,$uname,$url,$three,$four,$five,$end,$start,undef,undef,$context).
                 '</b><br />';
    return $output;
}

sub commit_standardrole {
    my ($udom,$uname,$url,$three,$start,$end,$one,$two,$sec,$context) = @_;
    my ($output,$logmsg,$linefeed);
    if ($context eq 'auto') {
        $linefeed = "\n";
    } else {
        $linefeed = "<br />\n";
    }  
    if ($three eq 'st') {
        my $result = &commit_studentrole(\$logmsg,$udom,$uname,$url,$three,$start,$end,
                                         $one,$two,$sec,$context);
        if (($result =~ /^error/) || ($result eq 'not_in_class') || 
            ($result eq 'unknown_course') || ($result eq 'refused')) {
            $output = $logmsg.' '.&mt('Error: ').$result."\n"; 
        } else {
            $output = $logmsg.$linefeed.&mt('Assigning').' '.$three.' in '.$url.
               ($start?', '.&mt('starting').' '.localtime($start):'').
               ($end?', '.&mt('ending').' '.localtime($end):'').': ';
            if ($context eq 'auto') {
                $output .= $result.$linefeed.&mt('Add to classlist').': ok';
            } else {
               $output .= '<b>'.$result.'</b>'.$linefeed.
               &mt('Add to classlist').': <b>ok</b>';
            }
            $output .= $linefeed;
        }
    } else {
        $output = &mt('Assigning').' '.$three.' in '.$url.
               ($start?', '.&mt('starting').' '.localtime($start):'').
               ($end?', '.&mt('ending').' '.localtime($end):'').': ';
        my $result = &Apache::lonnet::assignrole($udom,$uname,$url,$three,$end,$start,'','',$context);
        if ($context eq 'auto') {
            $output .= $result.$linefeed;
        } else {
            $output .= '<b>'.$result.'</b>'.$linefeed;
        }
    }
    return $output;
}

sub commit_studentrole {
    my ($logmsg,$udom,$uname,$url,$three,$start,$end,$one,$two,$sec,$context) = @_;
    my ($result,$linefeed,$oldsecurl,$newsecurl);
    if ($context eq 'auto') {
        $linefeed = "\n";
    } else {
        $linefeed = '<br />'."\n";
    }
    if (defined($one) && defined($two)) {
        my $cid=$one.'_'.$two;
        my $oldsec=&Apache::lonnet::getsection($udom,$uname,$cid);
        my $secchange = 0;
        my $expire_role_result;
        my $modify_section_result;
        if ($oldsec ne '-1') { 
            if ($oldsec ne $sec) {
                $secchange = 1;
                my $now = time;
                my $uurl='/'.$cid;
                $uurl=~s/\_/\//g;
                if ($oldsec) {
                    $uurl.='/'.$oldsec;
                }
                $oldsecurl = $uurl;
                $expire_role_result = 
                    &Apache::lonnet::assignrole($udom,$uname,$uurl,'st',$now,'','',$context);
                if ($env{'request.course.sec'} ne '') { 
                    if ($expire_role_result eq 'refused') {
                        my @roles = ('st');
                        my @statuses = ('previous');
                        my @roledoms = ($one);
                        my $withsec = 1;
                        my %roleshash = 
                            &Apache::lonnet::get_my_roles($uname,$udom,'userroles',
                                              \@statuses,\@roles,\@roledoms,$withsec);
                        if (defined ($roleshash{$two.':'.$one.':st:'.$oldsec})) {
                            my ($oldstart,$oldend) = 
                                split(':',$roleshash{$two.':'.$one.':st:'.$oldsec});
                            if ($oldend > 0 && $oldend <= $now) {
                                $expire_role_result = 'ok';
                            }
                        }
                    }
                }
                $result = $expire_role_result;
            }
        }
        if (($expire_role_result eq 'ok') || ($secchange == 0)) {
            $modify_section_result = &Apache::lonnet::modify_student_enrollment($udom,$uname,undef,undef,undef,undef,undef,$sec,$end,$start,'','',$cid,'',$context);
            if ($modify_section_result =~ /^ok/) {
                if ($secchange == 1) {
                    if ($sec eq '') {
                        $$logmsg .= &mt('Section for [_1] switched from (possibly expired) old section: [_2] to student role without a section.',$uname,$oldsec).$linefeed;
                    } else {
                        $$logmsg .= &mt('Section for [_1] switched from (possibly expired) old section: [_2] to new section: [_3].',$uname,$oldsec,$sec).$linefeed;
                    }
                } elsif ($oldsec eq '-1') {
                    if ($sec eq '') {
                        $$logmsg .= &mt('New student role without a section for [_1] in course [_2].',$uname,$cid).$linefeed;
                    } else {
                        $$logmsg .= &mt('New student role for [_1] in section [_2] in course [_3].',$uname,$sec,$cid).$linefeed;
                    }
                } else {
                    if ($sec eq '') {
                        $$logmsg .= &mt('Student [_1] assigned to course [_2] without a section.',$uname,$cid).$linefeed;
                    } else {
                        $$logmsg .= &mt('Student [_1] assigned to section [_2] in course [_3].',$uname,$sec,$cid).$linefeed;
                    }
                }
            } else {
                if ($secchange) {       
                    $$logmsg .= &mt('Error when attempting section change for [_1] from old section "[_2]" to new section: "[_3]" in course [_4] -error:',$uname,$oldsec,$sec,$cid).' '.$modify_section_result.$linefeed;
                } else {
                    $$logmsg .= &mt('Error when attempting to modify role for [_1] for section: "[_2]" in course [_3] -error:',$uname,$sec,$cid).' '.$modify_section_result.$linefeed;
                }
            }
            $result = $modify_section_result;
        } elsif ($secchange == 1) {
            if ($oldsec eq '') {
                $$logmsg .= &mt('Error when attempting to expire existing role without a section for [_1] in course [_3] -error: ',$uname,$cid).' '.$expire_role_result.$linefeed;
            } else {
                $$logmsg .= &mt('Error when attempting to expire existing role for [_1] in section [_2] in course [_3] -error: ',$uname,$oldsec,$cid).' '.$expire_role_result.$linefeed;
            }
            if ($expire_role_result eq 'refused') {
                my $newsecurl = '/'.$cid;
                $newsecurl =~ s/\_/\//g;
                if ($sec ne '') {
                    $newsecurl.='/'.$sec;
                }
                if (&Apache::lonnet::allowed('cst',$newsecurl) && !(&Apache::lonnet::allowed('cst',$oldsecurl))) {
                    if ($sec eq '') {
                        $$logmsg .= &mt('Although your current role has privileges to add students to section "[_1]", you do not have privileges to modify existing enrollments unaffiliated with any section.',$sec).$linefeed;
                    } else {
                        $$logmsg .= &mt('Although your current role has privileges to add students to section "[_1]", you do not have privileges to modify existing enrollments in other sections.',$sec).$linefeed;
                    }
                }
            }
        }
    } else {
        $$logmsg .= &mt('Incomplete course id defined.').$linefeed.&mt('Addition of user [_1] from domain [_2] to course [_3], section [_4] not completed.',$uname,$udom,$one.'_'.$two,$sec).$linefeed;
        $result = "error: incomplete course id\n";
    }
    return $result;
}

############################################################
############################################################

sub check_clone {
    my ($args,$linefeed) = @_;
    my $cloneid='/'.$args->{'clonedomain'}.'/'.$args->{'clonecourse'};
    my ($clonecrsudom,$clonecrsunum)= &LONCAPA::split_courseid($cloneid);
    my $clonehome=&Apache::lonnet::homeserver($clonecrsunum,$clonecrsudom);
    my $clonemsg;
    my $can_clone = 0;

    if ($clonehome eq 'no_host') {
        $clonemsg = &mt('No new course created.').$linefeed.&mt('A new course could not be cloned from the specified original - [_1] - because it is a non-existent course.',$args->{'clonecourse'}.':'.$args->{'clonedomain'});     
    } else {
	my %clonedesc = &Apache::lonnet::coursedescription($cloneid,{'one_time' => 1});
	if ($env{'request.role.domain'} eq $args->{'clonedomain'}) {
	    $can_clone = 1;
	} else {
	    my %clonehash = &Apache::lonnet::get('environment',['cloners'],
						 $args->{'clonedomain'},$args->{'clonecourse'});
	    my @cloners = split(/,/,$clonehash{'cloners'});
            if (grep(/^\*$/,@cloners)) {
                $can_clone = 1;
            } elsif (grep(/^\*\:\Q$args->{'ccdomain'}\E$/,@cloners)) {
                $can_clone = 1;
            } else {
	        my %roleshash =
		    &Apache::lonnet::get_my_roles($args->{'ccuname'},
					 $args->{'ccdomain'},
                                         'userroles',['active'],['cc'],
					 [$args->{'clonedomain'}]);
	        if (($roleshash{$args->{'clonecourse'}.':'.$args->{'clonedomain'}.':cc'}) || (grep(/^\Q$args->{'ccuname'}\E:\Q$args->{'ccdomain'}\E$/,@cloners))) {
		    $can_clone = 1;
	        } else {
                    $clonemsg = &mt('No new course created.').$linefeed.&mt('The new course could not be cloned from the existing course because the new course owner ([_1]) does not have cloning rights in the existing course ([_2]).',$args->{'ccuname'}.':'.$args->{'ccdomain'},$clonedesc{'description'});
	        }
	    }
        }
    }
    return ($can_clone, $clonemsg, $cloneid, $clonehome);
}

sub construct_course {
    my ($args,$logmsg,$courseid,$crsudom,$crsunum,$udom,$uname,$context) = @_;
    my $outcome;
    my $linefeed =  '<br />'."\n";
    if ($context eq 'auto') {
        $linefeed = "\n";
    }

#
# Are we cloning?
#
    my ($can_clone, $clonemsg, $cloneid, $clonehome);
    if (($args->{'clonecourse'}) && ($args->{'clonedomain'})) {
	($can_clone, $clonemsg, $cloneid, $clonehome) = &check_clone($args,$linefeed);
	if ($context ne 'auto') {
            if ($clonemsg ne '') {
	        $clonemsg = '<span class="LC_error">'.$clonemsg.'</span>';
            }
	}
	$outcome .= $clonemsg.$linefeed;

        if (!$can_clone) {
	    return (0,$outcome);
	}
    }

#
# Open course
#
    my $crstype = lc($args->{'crstype'});
    my %cenv=();
    $$courseid=&Apache::lonnet::createcourse($args->{'course_domain'},
                                             $args->{'cdescr'},
                                             $args->{'curl'},
                                             $args->{'course_home'},
                                             $args->{'nonstandard'},
                                             $args->{'crscode'},
                                             $args->{'ccuname'}.':'.
                                             $args->{'ccdomain'},
                                             $args->{'crstype'});

    # Note: The testing routines depend on this being output; see 
    # Utils::Course. This needs to at least be output as a comment
    # if anyone ever decides to not show this, and Utils::Course::new
    # will need to be suitably modified.
    $outcome .= &mt('New LON-CAPA [_1] ID: [_2]',$crstype,$$courseid).$linefeed;
#
# Check if created correctly
#
    ($$crsudom,$$crsunum)= &LONCAPA::split_courseid($$courseid);
    my $crsuhome=&Apache::lonnet::homeserver($$crsunum,$$crsudom);
    $outcome .= &mt('Created on').': '.$crsuhome.$linefeed;

#
# Do the cloning
#   
    if ($can_clone && $cloneid) {
	$clonemsg = &mt('Cloning [_1] from [_2]',$crstype,$clonehome);
	if ($context ne 'auto') {
	    $clonemsg = '<span class="LC_success">'.$clonemsg.'</span>';
	}
	$outcome .= $clonemsg.$linefeed;
	my %oldcenv=&Apache::lonnet::dump('environment',$$crsudom,$$crsunum);
# Copy all files
	&Apache::lonclonecourse::copycoursefiles($cloneid,$$courseid,$args->{'datemode'},$args->{'dateshift'});
# Restore URL
	$cenv{'url'}=$oldcenv{'url'};
# Restore title
	$cenv{'description'}=$oldcenv{'description'};
# Mark as cloned
	$cenv{'clonedfrom'}=$cloneid;
# Need to clone grading mode
        my %newenv=&Apache::lonnet::get('environment',['grading'],$$crsudom,$$crsunum);
        $cenv{'grading'}=$newenv{'grading'};
# Do not clone these environment entries
        &Apache::lonnet::del('environment',
                  ['default_enrollment_start_date',
                   'default_enrollment_end_date',
                   'question.email',
                   'policy.email',
                   'comment.email',
                   'pch.users.denied',
                   'plc.users.denied'],
                   $$crsudom,$$crsunum);
    }

#
# Set environment (will override cloned, if existing)
#
    my @sections = ();
    my @xlists = ();
    if ($args->{'crstype'}) {
        $cenv{'type'}=$args->{'crstype'};
    }
    if ($args->{'crsid'}) {
        $cenv{'courseid'}=$args->{'crsid'};
    }
    if ($args->{'crscode'}) {
        $cenv{'internal.coursecode'}=$args->{'crscode'};
    }
    if ($args->{'crsquota'} ne '') {
        $cenv{'internal.coursequota'}=$args->{'crsquota'};
    } else {
        $cenv{'internal.coursequota'}=$args->{'crsquota'} = 20;
    }
    if ($args->{'ccuname'}) {
        $cenv{'internal.courseowner'} = $args->{'ccuname'}.
                                        ':'.$args->{'ccdomain'};
    } else {
        $cenv{'internal.courseowner'} = $args->{'curruser'};
    }
    my @badclasses = (); # Used to accumulate sections/crosslistings that did not pass classlist access check for course owner.
    if ($args->{'crssections'}) {
        $cenv{'internal.sectionnums'} = '';
        if ($args->{'crssections'} =~ m/,/) {
            @sections = split/,/,$args->{'crssections'};
        } else {
            $sections[0] = $args->{'crssections'};
        }
        if (@sections > 0) {
            foreach my $item (@sections) {
                my ($sec,$gp) = split/:/,$item;
                my $class = $args->{'crscode'}.$sec;
                my $addcheck = &Apache::lonnet::auto_new_course($$crsunum,$$crsudom,$class,$cenv{'internal.courseowner'});
                $cenv{'internal.sectionnums'} .= $item.',';
                unless ($addcheck eq 'ok') {
                    push @badclasses, $class;
                }
            }
            $cenv{'internal.sectionnums'} =~ s/,$//;
        }
    }
# do not hide course coordinator from staff listing, 
# even if privileged
    $cenv{'nothideprivileged'}=$args->{'ccuname'}.':'.$args->{'ccdomain'};
# add crosslistings
    if ($args->{'crsxlist'}) {
        $cenv{'internal.crosslistings'}='';
        if ($args->{'crsxlist'} =~ m/,/) {
            @xlists = split/,/,$args->{'crsxlist'};
        } else {
            $xlists[0] = $args->{'crsxlist'};
        }
        if (@xlists > 0) {
            foreach my $item (@xlists) {
                my ($xl,$gp) = split/:/,$item;
                my $addcheck =  &Apache::lonnet::auto_new_course($$crsunum,$$crsudom,$xl,$cenv{'internal.courseowner'});
                $cenv{'internal.crosslistings'} .= $item.',';
                unless ($addcheck eq 'ok') {
                    push @badclasses, $xl;
                }
            }
            $cenv{'internal.crosslistings'} =~ s/,$//;
        }
    }
    if ($args->{'autoadds'}) {
        $cenv{'internal.autoadds'}=$args->{'autoadds'};
    }
    if ($args->{'autodrops'}) {
        $cenv{'internal.autodrops'}=$args->{'autodrops'};
    }
# check for notification of enrollment changes
    my @notified = ();
    if ($args->{'notify_owner'}) {
        if ($args->{'ccuname'} ne '') {
            push(@notified,$args->{'ccuname'}.':'.$args->{'ccdomain'});
        }
    }
    if ($args->{'notify_dc'}) {
        if ($uname ne '') { 
            push(@notified,$uname.':'.$udom);
        }
    }
    if (@notified > 0) {
        my $notifylist;
        if (@notified > 1) {
            $notifylist = join(',',@notified);
        } else {
            $notifylist = $notified[0];
        }
        $cenv{'internal.notifylist'} = $notifylist;
    }
    if (@badclasses > 0) {
        my %lt=&Apache::lonlocal::texthash(
                'tclb' => 'The courses listed below were included as sections or crosslistings affiliated with your new LON-CAPA course.  However, if automated course roster updates are enabled for this class, these particular sections/crosslistings will not contribute towards enrollment, because the user identified as the course owner for this LON-CAPA course',
                'dnhr' => 'does not have rights to access enrollment in these classes',
                'adby' => 'as determined by the policies of your institution on access to official classlists'
        );
        my $badclass_msg = $cenv{'internal.courseowner'}.') - '.$lt{'dnhr'}.
                           ' ('.$lt{'adby'}.')';
        if ($context eq 'auto') {
            $outcome .= $badclass_msg.$linefeed;
            $outcome .= '<div class="LC_warning">'.$badclass_msg.$linefeed.'<ul>'."\n";
            foreach my $item (@badclasses) {
                if ($context eq 'auto') {
                    $outcome .= " - $item\n";
                } else {
                    $outcome .= "<li>$item</li>\n";
                }
            }
            if ($context eq 'auto') {
                $outcome .= $linefeed;
            } else {
                $outcome .= "</ul><br /><br /></div>\n";
            }
        } 
    }
    if ($args->{'no_end_date'}) {
        $args->{'endaccess'} = 0;
    }
    $cenv{'internal.autostart'}=$args->{'enrollstart'};
    $cenv{'internal.autoend'}=$args->{'enrollend'};
    $cenv{'default_enrollment_start_date'}=$args->{'startaccess'};
    $cenv{'default_enrollment_end_date'}=$args->{'endaccess'};
    if ($args->{'showphotos'}) {
      $cenv{'internal.showphotos'}=$args->{'showphotos'};
    }
    $cenv{'internal.authtype'} = $args->{'authtype'};
    $cenv{'internal.autharg'} = $args->{'autharg'}; 
    if ( ($cenv{'internal.authtype'} =~ /^krb/) && ($cenv{'internal.autoadds'} == 1)) {
        if (! defined($cenv{'internal.autharg'}) || $cenv{'internal.autharg'}  eq '') {
            my $krb_msg = &mt('As you did not include the default Kerberos domain to be used for authentication in this class, the institutional data used by the automated enrollment process must include the Kerberos domain for each new student'); 
            if ($context eq 'auto') {
                $outcome .= $krb_msg;
            } else {
                $outcome .= '<span class="LC_error">'.$krb_msg.'</span>';
            }
            $outcome .= $linefeed;
        }
    }
    if (($args->{'ccdomain'}) && ($args->{'ccuname'})) {
       if ($args->{'setpolicy'}) {
           $cenv{'policy.email'}=$args->{'ccuname'}.':'.$args->{'ccdomain'};
       }
       if ($args->{'setcontent'}) {
           $cenv{'question.email'}=$args->{'ccuname'}.':'.$args->{'ccdomain'};
       }
    }
    if ($args->{'reshome'}) {
	$cenv{'reshome'}=$args->{'reshome'}.'/';
	$cenv{'reshome'}=~s/\/+$/\//;
    }
#
# course has keyed access
#
    if ($args->{'setkeys'}) {
       $cenv{'keyaccess'}='yes';
    }
# if specified, key authority is not course, but user
# only active if keyaccess is yes
    if ($args->{'keyauth'}) {
	my ($user,$domain) = split(':',$args->{'keyauth'});
	$user = &LONCAPA::clean_username($user);
	$domain = &LONCAPA::clean_username($domain);
	if ($user ne '' && $domain ne '') {
	    $cenv{'keyauth'}=$user.':'.$domain;
	}
    }

    if ($args->{'disresdis'}) {
        $cenv{'pch.roles.denied'}='st';
    }
    if ($args->{'disablechat'}) {
        $cenv{'plc.roles.denied'}='st';
    }

    # Record we've not yet viewed the Course Initialization Helper for this 
    # course
    $cenv{'course.helper.not.run'} = 1;
    #
    # Use new Randomseed
    #
    $cenv{'rndseed'}=&Apache::lonnet::latest_rnd_algorithm_id();;
    $cenv{'receiptalg'}=&Apache::lonnet::latest_receipt_algorithm_id();;
    #
    # The encryption code and receipt prefix for this course
    #
    $cenv{'internal.encseed'}=$Apache::lonnet::perlvar{'lonReceipt'}.$$.time.int(rand(9999));
    $cenv{'internal.encpref'}=100+int(9*rand(99));
    #
    # By default, use standard grading
    if (!defined($cenv{'grading'})) { $cenv{'grading'} = 'standard'; }

    $outcome .= $linefeed.&mt('Setting environment').': '.                 
          &Apache::lonnet::put('environment',\%cenv,$$crsudom,$$crsunum).$linefeed;
#
# Open all assignments
#
    if ($args->{'openall'}) {
       my $storeunder=$$crsudom.'_'.$$crsunum.'.0.opendate';
       my %storecontent = ($storeunder         => time,
                           $storeunder.'.type' => 'date_start');
       
       $outcome .= &mt('Opening all assignments').': '.&Apache::lonnet::cput
                 ('resourcedata',\%storecontent,$$crsudom,$$crsunum).$linefeed;
   }
#
# Set first page
#
    unless (($args->{'nonstandard'}) || ($args->{'firstres'} eq 'blank')
	    || ($cloneid)) {
	use LONCAPA::map;
	$outcome .= &mt('Setting first resource').': ';

	my $map = '/uploaded/'.$$crsudom.'/'.$$crsunum.'/default.sequence';
        my ($errtext,$fatal)=&LONCAPA::map::mapread($map);

        $outcome .= ($fatal?$errtext:'read ok').' - ';
        my $title; my $url;
        if ($args->{'firstres'} eq 'syl') {
	    $title='Syllabus';
            $url='/public/'.$$crsudom.'/'.$$crsunum.'/syllabus';
        } else {
            $title='Navigate Contents';
            $url='/adm/navmaps';
        }

        $LONCAPA::map::resources[1]=$title.':'.$url.':false:start:res';
	(my $outtext,$errtext) = &LONCAPA::map::storemap($map,1);

	if ($errtext) { $fatal=2; }
        $outcome .= ($fatal?$errtext:'write ok').$linefeed;
    }

    return (1,$outcome);
}

############################################################
############################################################

sub course_type {
    my ($cid) = @_;
    if (!defined($cid)) {
        $cid = $env{'request.course.id'};
    }
    if (defined($env{'course.'.$cid.'.type'})) {
        return $env{'course.'.$cid.'.type'};
    } else {
        return 'Course';
    }
}

sub group_term {
    my $crstype = &course_type();
    my %names = (
                  'Course' => 'group',
                  'Group' => 'team',
                );
    return $names{$crstype};
}

sub icon {
    my ($file)=@_;
    my $curfext = lc((split(/\./,$file))[-1]);
    my $iconname=$Apache::lonnet::perlvar{'lonIconsURL'}.'/unknown.gif';
    my $embstyle = &Apache::loncommon::fileembstyle($curfext);
    if (!(!defined($embstyle) || $embstyle eq 'unk' || $embstyle eq 'hdn')) {
	if (-e  $Apache::lonnet::perlvar{'lonDocRoot'}.'/'.
	          $Apache::lonnet::perlvar{'lonIconsURL'}.'/'.
	            $curfext.".gif") {
	    $iconname=$Apache::lonnet::perlvar{'lonIconsURL'}.'/'.
		$curfext.".gif";
	}
    }
    return &lonhttpdurl($iconname);
} 

sub lonhttpd_port {
    my $lonhttpd_port=$Apache::lonnet::perlvar{'lonhttpdPort'};
    if (!defined($lonhttpd_port)) { $lonhttpd_port='8080'; }
    # IE doesn't like a secure page getting images from a non-secure
    # port (when logging we haven't parsed the browser type so default
    # back to secure
    if ((!exists($env{'browser.type'}) || $env{'browser.type'} eq 'explorer')
	&& $ENV{'SERVER_PORT'} == 443) {
	return 443;
    }
    return $lonhttpd_port;

}

sub lonhttpdurl {
    my ($url)=@_;

    my $lonhttpd_port = &lonhttpd_port();
    if ($lonhttpd_port == 443) {
	return 'https://'.$ENV{'SERVER_NAME'}.$url;
    }
    return 'http://'.$ENV{'SERVER_NAME'}.':'.$lonhttpd_port.$url;
}

sub connection_aborted {
    my ($r)=@_;
    $r->print(" ");$r->rflush();
    my $c = $r->connection;
    return $c->aborted();
}

#    Escapes strings that may have embedded 's that will be put into
#    strings as 'strings'.
sub escape_single {
    my ($input) = @_;
    $input =~ s/\\/\\\\/g;	# Escape the \'s..(must be first)>
    $input =~ s/\'/\\\'/g;	# Esacpe the 's....
    return $input;
}

#  Same as escape_single, but escape's "'s  This 
#  can be used for  "strings"
sub escape_double {
    my ($input) = @_;
    $input =~ s/\\/\\\\/g;	# Escape the /'s..(must be first)>
    $input =~ s/\"/\\\"/g;	# Esacpe the "s....
    return $input;
}
 
#   Escapes the last element of a full URL.
sub escape_url {
    my ($url)   = @_;
    my @urlslices = split(/\//, $url,-1);
    my $lastitem = &escape(pop(@urlslices));
    return join('/',@urlslices).'/'.$lastitem;
}

# -------------------------------------------------------- Initliaze user login
sub init_user_environment {
    my ($r, $username, $domain, $authhost, $form, $args) = @_;
    my $lonids=$Apache::lonnet::perlvar{'lonIDsDir'};

    my $public=($username eq 'public' && $domain eq 'public');

# See if old ID present, if so, remove

    my ($filename,$cookie,$userroles);
    my $now=time;

    if ($public) {
	my $max_public=100;
	my $oldest;
	my $oldest_time=0;
	for(my $next=1;$next<=$max_public;$next++) {
	    if (-e $lonids."/publicuser_$next.id") {
		my $mtime=(stat($lonids."/publicuser_$next.id"))[9];
		if ($mtime<$oldest_time || !$oldest_time) {
		    $oldest_time=$mtime;
		    $oldest=$next;
		}
	    } else {
		$cookie="publicuser_$next";
		last;
	    }
	}
	if (!$cookie) { $cookie="publicuser_$oldest"; }
    } else {
	# if this isn't a robot, kill any existing non-robot sessions
	if (!$args->{'robot'}) {
	    opendir(DIR,$lonids);
	    while ($filename=readdir(DIR)) {
		if ($filename=~/^$username\_\d+\_$domain\_$authhost\.id$/) {
		    unlink($lonids.'/'.$filename);
		}
	    }
	    closedir(DIR);
	}
# Give them a new cookie
	my $id = ($args->{'robot'} ? 'robot'.$args->{'robot'}
		                   : $now);
	$cookie="$username\_$id\_$domain\_$authhost";
    
# Initialize roles

	$userroles=&Apache::lonnet::rolesinit($domain,$username,$authhost);
    }
# ------------------------------------ Check browser type and MathML capability

    my ($httpbrowser,$clientbrowser,$clientversion,$clientmathml,
        $clientunicode,$clientos) = &decode_user_agent($r);

# -------------------------------------- Any accessibility options to remember?
    if (($form->{'interface'}) && ($form->{'remember'} eq 'true')) {
	foreach my $option ('imagesuppress','appletsuppress',
			    'embedsuppress','fontenhance','blackwhite') {
	    if ($form->{$option} eq 'true') {
		&Apache::lonnet::put('environment',{$option => 'on'},
				     $domain,$username);
	    } else {
		&Apache::lonnet::del('environment',[$option],
				     $domain,$username);
	    }
	}
    }
# ------------------------------------------------------------- Get environment

    my %userenv = &Apache::lonnet::dump('environment',$domain,$username);
    my ($tmp) = keys(%userenv);
    if ($tmp !~ /^(con_lost|error|no_such_host)/i) {
	# default remote control to off
	if ($userenv{'remote'} ne 'on') { $userenv{'remote'} = 'off'; }
    } else {
	undef(%userenv);
    }
    if (($userenv{'interface'}) && (!$form->{'interface'})) {
	$form->{'interface'}=$userenv{'interface'};
    }
    $env{'environment.remote'}=$userenv{'remote'};
    if ($userenv{'texengine'} eq 'ttm') { $clientmathml=1; }

# --------------- Do not trust query string to be put directly into environment
    foreach my $option ('imagesuppress','appletsuppress',
			'embedsuppress','fontenhance','blackwhite',
			'interface','localpath','localres') {
	$form->{$option}=~s/[\n\r\=]//gs;
    }
# --------------------------------------------------------- Write first profile

    {
	my %initial_env = 
	    ("user.name"          => $username,
	     "user.domain"        => $domain,
	     "user.home"          => $authhost,
	     "browser.type"       => $clientbrowser,
	     "browser.version"    => $clientversion,
	     "browser.mathml"     => $clientmathml,
	     "browser.unicode"    => $clientunicode,
	     "browser.os"         => $clientos,
	     "server.domain"      => $Apache::lonnet::perlvar{'lonDefDomain'},
	     "request.course.fn"  => '',
	     "request.course.uri" => '',
	     "request.course.sec" => '',
	     "request.role"       => 'cm',
	     "request.role.adv"   => $env{'user.adv'},
	     "request.host"       => $ENV{'REMOTE_ADDR'},);

        if ($form->{'localpath'}) {
	    $initial_env{"browser.localpath"}  = $form->{'localpath'};
	    $initial_env{"browser.localres"}   = $form->{'localres'};
        }
	
	if ($public) {
	    $initial_env{"environment.remote"} = "off";
	}
	if ($form->{'interface'}) {
	    $form->{'interface'}=~s/\W//gs;
	    $initial_env{"browser.interface"} = $form->{'interface'};
	    $env{'browser.interface'}=$form->{'interface'};
	    foreach my $option ('imagesuppress','appletsuppress',
				'embedsuppress','fontenhance','blackwhite') {
		if (($form->{$option} eq 'true') ||
		    ($userenv{$option} eq 'on')) {
		    $initial_env{"browser.$option"} = "on";
		}
	    }
	}

	$env{'user.environment'} = "$lonids/$cookie.id";
	
	if (tie(my %disk_env,'GDBM_File',"$lonids/$cookie.id",
		 &GDBM_WRCREAT(),0640)) {
	    &_add_to_env(\%disk_env,\%initial_env);
	    &_add_to_env(\%disk_env,\%userenv,'environment.');
	    &_add_to_env(\%disk_env,$userroles);
	    if (ref($args->{'extra_env'})) {
		&_add_to_env(\%disk_env,$args->{'extra_env'});
	    }
	    untie(%disk_env);
	} else {
	    &Apache::lonnet::logthis("<font color=\"blue\">WARNING: ".
			   'Could not create environment storage in lonauth: '.$!.'</font>');
	    return 'error: '.$!;
	}
    }
    $env{'request.role'}='cm';
    $env{'request.role.adv'}=$env{'user.adv'};
    $env{'browser.type'}=$clientbrowser;

    return $cookie;

}

sub _add_to_env {
    my ($idf,$env_data,$prefix) = @_;
    while (my ($key,$value) = each(%$env_data)) {
	$idf->{$prefix.$key} = $value;
	$env{$prefix.$key}   = $value;
    }
}


=pod

=back

=cut

1;
__END__;

