# The LearningOnline Network with CAPA
# lonselstudent.pm : Reusable subs for student selection.
#
# $Id: lonselstudent.pm,v 1.12 2008/06/06 19:12:59 bisitz Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

package Apache::lonselstudent;
use     Apache::lonnet;
use     Apache::lonlocal;
use     Apache::loncoursedata();
use     HTML::Entities();

#
#  Utility function used when rendering <student> tags.
#  This function produces a list references to four
#  arrays:
#    (\@course_personel, \@current_members, \@expired_members, \@future_members)
#
#
# Parameters;
#
#  restrict           - Optional.. if present and defined should be a section name.
#                       The *_members arrays will then only contain people
#                       in that section
#
# Where:
#    course_personnel - Each element of this array is itself a reference to an array
#                      containing information about a member of the course staff.
#    current_members  - Each element of this array is itself a reference to an
#                       array that contains information about current students in
#                       the course.
#    expired_members  - Each element of this array is itself a reference to an
#                       array that contains information about students whose 
#                       status has expired.
#    future_members   - Each element of this arrya is itself a reference to an
#                       array that contains information about students who will
#                       become active at a future date.
#
#  Course personnel elements include:
#       [0]    Last, First of the user.
#       [1]    Role held by the user.
#       [2]    Empty.
#       [3]    Empty
#       [4]    username:domain of the user.
#
#  Student member array elements are:
#       [0]    Last, First of the user.
#       [1]    Status of the user one of ("Active", "Future", or "Expired')
#              depending on which array the user was put in.
#       [2]    Section the student is in.
#       [3]    Role of the member (student).
#       [4]    username:domain of the user.
#
sub get_people_in_class {
    my ($section_restriction) = @_;
    my %coursepersonnel = &Apache::lonnet::get_course_adv_roles();
    #
    #  Enumerate the course_personnel.
    #
    my @course_personnel;
    for my $role (sort(keys(%coursepersonnel))) {
	# extract the names so we can sort them
	my @people;
	for my $person (split(/,/, $coursepersonnel{$role})) {
	    my ($uname,$domain) = split(/:/, $person);
	    push(@people, [&Apache::loncommon::plainname($uname,$domain),
			   $uname,$domain]);
	}
	@people = sort { $a->[0] cmp $b->[0] } (@people);
	    
	for my $person (@people) {
	    push(@course_personnel, [join(':', $person->[1],$person->[2]), 
				     $person->[0], '', '', $role]);
	}
    }
    #  Students must be split into the three categories:

    my @current_members;
    my @future_members;
    my @expired_members;

    #   Indices into the coures data elements.

    my $section    = &Apache::loncoursedata::CL_SECTION();
    my $fullname   = &Apache::loncoursedata::CL_FULLNAME();
    my $status     = &Apache::loncoursedata::CL_STATUS();
    my $start_date = &Apache::loncoursedata::CL_START();


    my $classlist = &Apache::loncoursedata::get_classlist();
    my @keys = keys(%{$classlist});
    # Sort by: fullname, username
    @keys = sort {
	lc($classlist->{$a}[$fullname]) cmp lc($classlist->{$b}[$fullname]) ||
	lc($a) cmp lc($b)
    } (@keys);
 



    for my $user (@keys) {
	if (!$section_restriction || 
	    ($section_restriction eq $classlist->{$user}->[$section])) {
	    
	    if ( $classlist->{$user}->[$status] eq
		 'Active') {
		push(@current_members, [$user, $classlist->{$user}->[$fullname], 
					$classlist->{$user}->[$section],
					$classlist->{$user}->[$status], 'Student']);
	    } else {
		#  Need to figure out if this user is future or
		#  Expired... If the start date is in the future
		#  the user is future...else expired.
		
		my $now = time;
		if ($classlist->{$user}->[$start_date] > $now) {
		    push(@future_members, [$user, $classlist->{$user}->[$fullname],
					   $classlist->{$user}->[$section],
					   "Future", "Student"]);
		} else {
		    push(@expired_members, [$user,
					    $classlist->{$user}->[$fullname],
					    $classlist->{$user}->[$section],
					    "Expired", "Student"]);
		}
		
	    }
	}
    }
    return (\@course_personnel, 
	    \@current_members,
	    \@expired_members,
	    \@future_members);

}

#
#  Utility function used when rendering the <student> tag.
#  This function renders a segment of course personel
#  Personel are broken up by the helper into past, current and
#  future...each one gets is own subpage of selection.
#  This sub renders one of these pages.
#  Parameters:
#     $students    - Students in the section. (ref to array of references
#                    to arrays).
#     $formname    - Name of the form in which this stuff gets rendered.
#     $formprefix  - form path prefix for form element names
#                    This is used to make each form element
#                    so that the segments having to do with each
#                    set of students won't collide.
#     $defaultusers - reference to a hash containng
#                     the set of users that should be on or off.
#     $multiselect  - True if multiselect allowed.
#     $resultname   - Name of result variable.
#     $javascript   - If true, the javascript to run this is output
#                     This should be true for the first call for a page
#                     and false for all other calls... only matters if
#                     multiselect is true.
#  Returns:
#     HTML  text to add to the rendering of the helper.
#
sub render_student_list {
    my ($students, $formname, $formprefix, $defaultusers,
	$multiselect, $resultname, $javascript) = @_;

    my $result = "";

    # no students so no output
    return if (!@$students);

    if ($javascript && $multiselect) {
        $result .= <<SCRIPT;
<script type="text/javascript">
// <!--

    function findElement(name) {
	var i;
	var ele;
	for(i =0; i < document.forms.$formname.elements.length; i++) {
	    ele = document.forms.$formname.elements[i];
	    if(ele.name == name) {
		return ele;
	    }
	}
	return null;
    }
    function isStudent(element) {
	if(element.value.indexOf(":Student") != -1) {
	    return 1;
	}
	return 0;
    }
    function section(element) {
	var i;
	var info;
	if (element.value.indexOf(':') != -1) {
	    info = element.value.split(':');
	    return info[2];
	} else {
	    return "";
	}
    }
    function rightSubForm(element, which) {
	if (element.value.indexOf(which) != -1) {
	    return true;
	} else {
	    return false;
	}
    }

    function setAllStudents(value, which) {
	var i;
	var ele;
	for (i =0; i < document.forms.$formname.elements.length; i++) {
	    ele = document.forms.$formname.elements[i];
	    if(isStudent(ele) && rightSubForm(ele, which)) {
		ele.checked=value;
	    }
	}
    }
    function setAllCoursePersonnel(value, which) {
	var i;
	var ele;
	for (i =0; i < document.forms.$formname.elements.length; i++) {
	    ele = document.forms.$formname.elements[i];
	    if(!isStudent(ele) && rightSubForm(ele, which)) {
		ele.checked = value;
	    }
	}
    }
    function setSection(which, value, subform) {
	var i;
	var ele;
	for (i =0; i < document.forms.$formname.elements.length; i++) {
	    ele = document.forms.$formname.elements[i];
	    if (ele.value.indexOf(':') != -1) {
		if ((section(ele) == which) && rightSubForm(ele, subform)) {
		    ele.checked = value;
		}
	    }
	}
    }

    function setCheckboxes(listbox, which, value) {
	var k;
	var elem;
	var what;
        elem = findElement(listbox);
	if (elem != null) {
	    for (k = 0; k < elem.length; k++) {
		if (elem.options[k].selected) {
		    what = elem.options[k].value;
		    if (what == 'allstudents') {
			setAllStudents(value, which);
		    } else if (what == 'allpersonnel') {
			setAllCoursePersonnel(value, which);
		    } else if (what == 'nosection') {
			setSection('',value, which);
		    } else {
			setSection(what, value, which);
		    }
		}
	    }
	}
    }
    function selectSections(listbox, which) {
	setCheckboxes(listbox, which, true);

    }
    function unselectSections(listbox, which) {
	setCheckboxes(listbox, which, false);
    }

// -->
</script>
SCRIPT

    }

    # If multiple selections are allowed, we have a listbox
    # at the top which allows quick selections from each section
    # as well as from categories of personnel.

    if ($multiselect) {
	#  Make a section hash so we can add sections to the choice:

	my %sections;
	for my $student (@$students) {
	    my $sect = $student->[2];
	    if ($sect ne "") {
		$sections{$sect} = 1;
	    }
	}

	$result .= '<table><tr><td>';

	my $size = scalar(keys(%sections));
	$size += 3;		# We have allstudents allpersonel nosection too.
	if ($size > 5) { 
	    $size = 5; 
	}
	$result .= '<select multiple name="'.$formprefix
	    .'.chosensections" size="'.$size.'">'."\n";
	$result .= '<option value="allstudents">'.&mt('All Students').'</option>';
	$result .= '<option value="allpersonnel">'.&mt('All Course Personnel').'</option>';
	$result .= '<option value="nosection">'.&mt('No Section').'</option>';
	$result .= "\n";
	foreach my $sec (sort {lc($a) cmp lc($b)} (keys(%sections))) {
	    $result .= '<option name="'.$sec.'">'.$sec.'</option>'."\n";
	}
	$result .= '</td><td valign="top">';
	$result .= '<input type="button" name="'.$formprefix.'.select" value="'.&mt('Select').'" onclick='
	    ."'selectSections(\"$formprefix.chosensections\", \"$formprefix\")'".' /></td>';
	$result .= '<td valign="top"><input type="button" name="'.$formprefix
	    .'.unselect" value="'.&mt('Unselect').'"  onclick='.
	    "'unselectSections(\"$formprefix.chosensections\", \"$formprefix\")' ".' /></td></tr></table>';
    }

    #  Now we list the students, but the form element type
    #  will depend on whether or not multiselect is true.
    #  True -> checkboxes.
    #  False -> radiobuttons.

    $result .= &Apache::loncommon::start_data_table();
    $result .= &Apache::loncommon::start_data_table_header_row();
    $result .= '<th></th><th>'.&mt('Name').'</th>'."\n";
    $result .= '    <th>'.&mt('Section').'</th>'."\n";
    $result .= '    <th>'.&mt('Status').'</th>'."\n";
    $result .= '    <th>'.&mt('Role').'</th>'."\n";
    $result .= '    <th>'.&mt('Username : Domain').'</th>'."\n";
    $result .= &Apache::loncommon::end_data_table_header_row();

    my $input_type;
    if ($multiselect) {
	$input_type = "checkbox";
    } else {
	$input_type = "radio";
    }

    my $checked = 0;
    for my $student (@$students) {
	$result .= &Apache::loncommon::start_data_table_row().
	    '<td><input type="'.$input_type.'"  name="'.
	    $resultname."_forminput".'"';
	my $user    = $student->[0];

	# Figure out which students are checked by default...
	
	if (%$defaultusers) {
	    if (exists ($defaultusers->{$user})) {
		$result .= ' checked ="checked" ';
		$checked = 1;
	    }
	} elsif (!$multiselect  && !$checked) {
	    $result .= ' checked="checked" ';
	    $checked = 1;	# First one for radio if no default specified.
	}
	$result .= ' value="'.&HTML::Entities::encode($user .          ':'
				 		      .$student->[2] . ':'
						      .$student->[1] . ':'
						      .$student->[3] . ':'
						      .$student->[4] . ":"
						      .$formprefix,   "<>&\"'")
	    ."\" /></td><td>\n";
	$result .= &HTML::Entities::encode($student->[1], '<>&"')
	        . '</td><td align="center" >'."\n";
	$result .= &HTML::Entities::encode($student->[2], '<>&"')
   	        . '</td><td align="center">'."\n";
	$result .= &HTML::Entities::encode($student->[3], '<>&"')
	        . '</td><td align="center">'."\n";
	$result .= &HTML::Entities::encode($student->[4], '<>&"')
  	        . '</td><td align="center">'."\n";
	$result .= &HTML::Entities::encode($student->[0], '<>&"')
	        . '</td>'.&Apache::loncommon::end_data_table_row().
		"\n";
    }
    $result .= &Apache::loncommon::end_data_table().
	" <br /> <hr />\n";

    return $result;
}

1;
