# The LearningOnline Network with CAPA
# Edit Handler for RAT Maps
#
# $Id: lonratedt.pm,v 1.88 2007/07/11 20:32:01 albertel Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#


package Apache::lonratedt;

=pod

=head1 NAME

Apache::lonratedt: simple resource assembly tool

=head1 SYNOPSIS

lonratedt provides the routines and the handler for the Advanced
    Resource Assembly Tool (RAT), and ties the various pieces together
    with Javascript.

=head1 OVERVIEW

=head2 Map Representation

=begin latex

%
\begin{figure}
\begin{center}\includegraphics[%
  width=0.55\paperwidth,bb = 0 0 200 100, draft, type=eps]{Map_Example}\end{center}


\caption{\label{Map_In_Advanced_Editor}Example of a Map in the Advanced Editor}
\end{figure}
%
\begin{figure}
\begin{lyxcode}
<map>

~~<resource~id=\char`\"{}1\char`\"{}

~~~~src=\char`\"{}/res/msu/korte/phy231welcome.html\char`\"{}

~~~~type=\char`\"{}start\char`\"{}

~~~~title=\char`\"{}Start\char`\"{}>

~~~~</resource>

~~<resource~id=\char`\"{}2\char`\"{}

~~~~src=\char`\"{}\char`\"{}~type=\char`\"{}finish\char`\"{}

~~~~title=\char`\"{}Finish\char`\"{}>

~~~~</resource>

~~<resource~id=\char`\"{}6\char`\"{}

~~~~src=\char`\"{}/res/msu/korte/tests/units.problem\char`\"{}

~~~~type=\char`\"{}mandatory\char`\"{}

~~~~title=\char`\"{}Physical~Units~Test\char`\"{}>

~~~~</resource>

~~<resource~id=\char`\"{}9\char`\"{}

~~~~src=\char`\"{}/res/msu/korte/chapters/onedim.sequence\char`\"{}

~~~~title=\char`\"{}Motion~in~One~Dimension\char`\"{}>

~~~~</resource>

~~<resource~id=\char`\"{}11\char`\"{}

~~~~src=\char`\"{}/res/msu/bauer/bridges/units.sequence\char`\"{}

~~~~title=\char`\"{}Physical~Units~Refresher\char`\"{}>

~~~~</resource>

~~<condition~id=\char`\"{}19\char`\"{}

~~~~type=\char`\"{}stop\char`\"{}

~~~~value=\char`\"{}user.assessments{[}this./res/msu/korte/tests/units.problem{]}.status=solved\char`\"{}>

~~~~</condition>

~~<link~from=\char`\"{}1\char`\"{}~to=\char`\"{}6\char`\"{}></link>

~~<link~from=\char`\"{}6\char`\"{}~to=\char`\"{}9\char`\"{}~condition=\char`\"{}19\char`\"{}></link>

~~<link~from=\char`\"{}6\char`\"{}~to=\char`\"{}11\char`\"{}></link>

~~<link~from=\char`\"{}11\char`\"{}~to=\char`\"{}6\char`\"{}></link>

~~</map>
\end{lyxcode}

\caption{\label{XML}XML for Map in Figure \ref{Map_In_Advanced_Editor}}
\end{figure}

=end latex

Fig. "XML for Map in Figure" shows the XML representation of the
resource map shown in Fig. "Example of a Map in the Advanced Editor",
which is the format in which maps are stored. In the figure, however,
additional graphical map layout information generated by the Advanced
Resource Assembly Tool is not displayed. This graphical information is
optional to re-generate the same graphical layout when the map is
brought up again in the Resource Assembly Tool, and is not needed for
any other system functionality.

Maps can be generated by tools other than the Resource Assembly
Tool. In particular, an author might have some other representation of
a course sequence, which can be converted into a map using scripts. If
this map then were to be brought up in the Resource Assembly Tool, the
Tool would automatically generate a graphical layout for it. Each
entry of the map (resources, conditions and links) is stored in a
separate tag.

Resources and conditionsX<conditions> have to have unique ID
numbers. These numbers are automatically generated by the Resource
Assembly Tool when the entry is first created, or added to the entries
when a map generated outside the Resource Assembly Tool is first
retrieved. They can also be assigned by custom scripts or added in by
hand.

In the XML example, entry 1 is the start resource of the map. When
this map is accessed, the source (src) URL of this tag will be the
first resource rendered. Entry 2 is the finish resource of this
map. This resource will be the last resource in the sequence of
resources. Entry 6 is a problem resource with the given URL and title,
as well as the priority "mandatory". Entry 19 is a condition, which is
used by the link between entries 6, the problem, and 9, a
sequence. I<The final syntax for conditions has not yet been
determined.>

=cut

use strict;
use Apache::Constants qw(:common);
use Apache::lonnet;
use Apache::lonsequence();
use Apache::loncommon();
use Apache::lonlocal;
use LONCAPA::map();
use File::Copy;
use LONCAPA;
use HTML::Entities();

# --------------------------------------------------------- Build up RAT screen
sub ratedt {
  my ($r,$url)=@_;
  my %layout = ('border' => "0");
    if ($env{'environment.remote'} eq 'off') {
      $layout{'rows'} = "1,250,*";
  } else {
      $layout{'rows'} = "1,70,*";
  }
  my $js ='
<script type="text/javascript">
    var flag=0;
</script>';

  my $start_page = 
      &Apache::loncommon::start_page('Edit Sequence',$js,
				     {'frameset'    => 1,
				      'add_entries' => \%layout});
  my $end_page = 
      &Apache::loncommon::end_page({'frameset' => 1});

  $r->print(<<ENDDOCUMENT);
$start_page
<frame name="server" src="$url/loadonly/ratserver" noresize="noresize"
       noscroll="noscroll" />
<frame name="code" src="$url/loadonly/adveditmenu" />
<frame name="mapout" src="/adm/rat/map.html" />
$end_page
ENDDOCUMENT
}

# ---------------------------------------------------------------- Make buttons

sub buttons {
    my $adv=shift;
    my $output='<form method=post>';     
    if ($adv==1) {
	$output.='<input type=submit name=forceadv value="'.&mt('Edit').'">'.
	    &Apache::loncommon::help_open_topic('Sequence_Advanced_Editor_Creation');;
    } else {
        unless ($adv==2) {
           $output.='<input type=submit name=forcesmp value="'.&mt('Simple Edit').'">'.
	    &Apache::loncommon::help_open_topic('Sequence_Simple_Editor_Creation');
        }
	$output.='<input type=submit name=forceadv value="'.&mt('Advanced Edit').'">'.
	    &Apache::loncommon::help_open_topic('Sequence_Advanced_Editor_Creation');
    }
    return $output.'</form><hr>';
}

# ----------------------------------------------------------------- Edit script
sub editscript {
    my $mode=shift;
    my $resurl=
	&Apache::loncommon::escape_single(&Apache::loncommon::lastresurl());
    return(<<ENDSCRIPT);
var srch;
var srchflag=-1; // 1 means currently open
                 // 0 means closed (but has been open)
                 // -1 means never yet opened/defined
var srchmode='';

var idx;
var idxflag=-1; // 1 means currently open
                 // 0 means closed (but has been open)
                 // -1 means never yet opened/defined
var idxmode='';

// ------------------------------------------------------ Clears indexer window
function idxclear() {
  idx.document.clear();
}

// ------------------------------------------------------- Clears search window
function srchclear() {
  srch.document.clear();
}

// ------------------------------------------------------ Closes indexer window
function idxclose() {
  if (idx && !idx.closed) {
    idxflag=0;
    idx.close();
  }
}

// ------------------------------------------------------- Closes search window
function srchclose() {
  if (srch && !srch.closed) {
    srchflag=0;
    srch.close();
  }
}

// -------------------------------------------------------- Open indexer window
function idxopen(mode) {
   var options="scrollbars=1,resizable=1,menubar=0,location=1,toolbar=1";
   idxmode=mode;
   idxflag=1;
   idx=open('$resurl/?inhibitmenu=yes&launch=1&mode=$mode&catalogmode='+mode,'idxout',options);
   idx.focus();
}

// ------------------------------------------------------ Open groupsort window
function groupopen(url,recover,bookmarks) {
   var options="scrollbars=1,resizable=1,menubar=0";
   idxflag=1;
   idx=open("/adm/groupsort?inhibitmenu=yes&mode=$mode&recover="+recover+"&readfile="+url+"&bookmarks="+bookmarks,"idxout",options);
   idx.focus();
}

// --------------------------------------------------------- Open search window
function srchopen(mode) {
   var options="scrollbars=1,resizable=1,menubar=0";
   srchmode=mode;
   srchflag=1;
   srch=open("/adm/searchcat?inhibitmenu=yes&launch=1&mode=$mode&catalogmode="+mode,"srchout",options);
   srch.focus();
}
// ----------------------------------------------------- launch indexer browser
function groupsearch() {
   srchcheck('import');
}

function groupimport() {
   idxcheck('import');
}

// ------------------------------------------------------- Do srch status check
function srchcheck(mode) {
   if (!srch || srch.closed || srchmode!=mode) {
      srchopen(mode);
   }
   srch.focus();
}

// -------------------------------------------------------- Do idx status check
function idxcheck(mode) {
   if (!idx || idx.closed || idxmode!=mode) {
      idxopen(mode);
   }
   idx.focus();
}


    var editbrowser;
    function openbrowser(formname,elementname,only,omit) {
        var url = '$resurl/?';
        if (editbrowser == null) {
            url += 'launch=1&';
        }
        url += 'inhibitmenu=yes&';
        url += 'catalogmode=interactive&';
        url += 'mode=edit&';
        url += 'form=' + formname + '&';
        if (only != null) {
            url += 'only=' + only + '&';
        } 
        if (omit != null) {
            url += 'omit=' + omit + '&';
        }
        url += 'element=' + elementname + '';
        var title = 'Browser';
        var options = 'scrollbars=1,resizable=1,menubar=0,toolbar=1,location=1';
        options += ',width=700,height=600';
        editbrowser = open(url,title,options,'1');
        editbrowser.focus();
    }
ENDSCRIPT
}
# ------------------------------------------------------- Simple edit processor

sub smpedt {
   my ($r,$url,$errtext)=@_;
   my $buttons=&buttons(2);
   my $tmpfn=&Apache::lonnet::filelocation('',$url).'.tmp';
   my $targetmsg='';
   if ($env{'form.save'}) {
       copy($tmpfn,&Apache::lonnet::filelocation('',$url));
       unlink($tmpfn);
       my ($errtext,$fatal)=
	   &LONCAPA::map::mapread(&Apache::lonnet::filelocation('',$url),'');
       unless ($fatal) {
	   $targetmsg='<b>'.&mt('Saved.').'</b><br />';
       } else {
	   $targetmsg='<b>'.&mt('An error occured while saving.').'</b><br />';
       }
   }
   if ($env{'form.revert'}) {
       $targetmsg='<b>'.&mt('Reverted.').'</b><br />';
       unlink($tmpfn);
       my ($errtext,$fatal)=
	   &LONCAPA::map::mapread(&Apache::lonnet::filelocation('',$url),'');
   }
   if (-e $tmpfn) {
      $targetmsg=
        '<b><font color="red">'.&mt('You are working with an unsaved version of your map.').'</font></b><br>';
      my ($errtext,$fatal)=&LONCAPA::map::mapread($tmpfn,'');
   }
# ---------------------------------------------------------- Process form input

   my @importselect=&Apache::loncommon::get_env_multiple('form.importsel');
   my @targetselect=&Apache::loncommon::get_env_multiple('form.target');
# ============================================================ Process commands

   my $targetdetail=$env{'form.targetdetail'};
   my $importdetail=$env{'form.curimpdetail'};

# ---------------------------------------------------- Importing from groupsort
   if (($env{'form.importdetail'}) && (!$env{'form.impfortarget'})) {

       $importdetail='';
       my @curimport=split(/\&/,$env{'form.curimpdetail'});

       my $lastsel;

       if (defined($importselect[-1])) {
	   $lastsel=$importselect[-1];
       } else {
           $lastsel=$#curimport;
       }

       for (my $i=0;$i<=$lastsel;$i++) {
           my ($name,$url)=split(/\=/,$curimport[$i]);
           if ($url) {
              $importdetail.='&'.$name.'='.$url;
	   }
       }

      $importdetail.='&'.$env{'form.importdetail'};

       for (my $i=$lastsel+1;$i<=$#curimport;$i++) {
           my ($name,$url)=split(/\=/,$curimport[$i]);
           if ($url) {
              $importdetail.='&'.$name.'='.$url;
	  }
       }
       $importdetail=~s/\&+/\&/g;
       $importdetail=~s/^\&//;

# ------------------------------------------------------------------- Clear all
   } elsif ($env{'form.clear'}) {
       $importdetail='';
# ------------------------------------------------------------ Discard selected
   } elsif ($env{'form.discard'}) {
       $importdetail='';
       my @curimport=split(/\&/,$env{'form.curimpdetail'});
       foreach (@importselect) {
	   $curimport[$_]='';
       }
       for (my $i=0;$i<=$#curimport;$i++) {
           my ($name,$url)=split(/\=/,$curimport[$i]);
           if ($url) {
              $importdetail.='&'.$name.'='.$url;
	   }
       }
# --------------------------------------------------------- Loading another map
   } elsif ($env{'form.loadmap'}) {
       $importdetail='';
       my @curimport=split(/\&/,$env{'form.curimpdetail'});

       my $lastsel;

       if (defined($importselect[-1])) {
	   $lastsel=$importselect[-1];
       } else {
           $lastsel=$#curimport;
       }

       for (my $i=0;$i<=$lastsel;$i++) {
           my ($name,$url)=split(/\=/,$curimport[$i]);
           if ($url) {
              $importdetail.='&'.$name.'='.$url;
	   }
       }

       foreach (
    &Apache::lonsequence::attemptread(&Apache::lonnet::filelocation('',$env{'form.importmap'}))) {
	   my ($name,$url)=split(/\:/,$_);
           if ($url) {
              $importdetail.='&'.&escape($name).'='.
		 	         &escape($url);
	  }
       }

       for (my $i=$lastsel+1;$i<=$#curimport;$i++) {
           my ($name,$url)=split(/\=/,$curimport[$i]);
           if ($url) {
              $importdetail.='&'.$name.'='.$url;
	  }
       }
       $importdetail=~s/\&+/\&/g;
       $importdetail=~s/^\&//;

# ------------------------------------------------ Groupimport/search to target
   } elsif ($env{'form.importdetail'}) {
       my $lastsel;
       if (defined($targetselect[-1])) {
	   $lastsel=$targetselect[-1];
       } else {
           $lastsel=$#LONCAPA::map::order+1;
       }
       &LONCAPA::map::pastetarget($lastsel,split(/\&/,$env{'form.importdetail'}));
       &LONCAPA::map::storemap(&Apache::lonnet::filelocation('',$url));
# ------------------------------------------------------------------------- Cut
   } elsif (($env{'form.cut'}) || ($env{'form.copy'})) {
       $importdetail='';
       my @curimport=split(/\&/,$env{'form.curimpdetail'});

       my $lastsel;

       if (defined($importselect[-1])) {
	   $lastsel=$importselect[-1];
       } else {
           $lastsel=$#curimport;
       }

       for (my $i=0;$i<=$lastsel;$i++) {
           my ($name,$url)=split(/\=/,$curimport[$i]);
           if ($url) {
              $importdetail.='&'.$name.'='.$url;
	   }
       }

       foreach (@targetselect) {
	   my ($name,$url)=split(/\:/,$LONCAPA::map::resources[$LONCAPA::map::order[$_-1]]);
           if ($url) {
              $importdetail.='&'.&escape($name).'='.
		 	         &escape($url);
	  }
       }

       for (my $i=$lastsel+1;$i<=$#curimport;$i++) {
           my ($name,$url)=split(/\=/,$curimport[$i]);
           if ($url) {
              $importdetail.='&'.$name.'='.$url;
	  }
       }
       $importdetail=~s/\&+/\&/g;
       $importdetail=~s/^\&//;

       if ($env{'form.cut'}) {
           my @neworder=();
           for (my $i=0;$i<=$#LONCAPA::map::order;$i++) {
               my $include=1;
               foreach (@targetselect) {
		   if ($_-1==$i) { $include=0; }
               }
               if ($include) { 
		   $neworder[$#neworder+1]=$LONCAPA::map::order[$i]; 
	       } else {
		   &LONCAPA::map::makezombie($LONCAPA::map::order[$i]);
	       }
           }
           @LONCAPA::map::order=@neworder;
           &LONCAPA::map::storemap(&Apache::lonnet::filelocation('',$url));      
       }

# ----------------------------------------------------------------------- Paste
   } elsif ($env{'form.paste'}) {
       my $lastsel;
       if (defined($targetselect[-1])) {
	   $lastsel=$targetselect[-1];
       } else {
           $lastsel=$#LONCAPA::map::order+1;
       }
       my @newsequence;
       my @curimport=split(/\&/,$env{'form.curimpdetail'});
       foreach (@importselect) {
          $newsequence[$#newsequence+1]=$curimport[$_];
       }
       &LONCAPA::map::pastetarget($lastsel,@newsequence);
       &LONCAPA::map::storemap(&Apache::lonnet::filelocation('',$url));
# -------------------------------------------------------------------- Move up 
   } elsif ($env{'form.moveup'}) {
       foreach (sort @targetselect) {
           if ($_-1>0) {
              my $movethis=$LONCAPA::map::order[$_-1];
              $LONCAPA::map::order[$_-1]=$LONCAPA::map::order[$_-2];
              $LONCAPA::map::order[$_-2]=$movethis;
	  }
       }
       &LONCAPA::map::storemap(&Apache::lonnet::filelocation('',$url));
# ------------------------------------------------------------------ Move down
   } elsif ($env{'form.movedown'}) {
       foreach (reverse sort @targetselect) {
           if ($_-1<$#LONCAPA::map::order) {
              my $movethis=$LONCAPA::map::order[$_-1];
              $LONCAPA::map::order[$_-1]=$LONCAPA::map::order[$_];
              $LONCAPA::map::order[$_]=$movethis;
	   }
       }
       &LONCAPA::map::storemap(&Apache::lonnet::filelocation('',$url));
# --------------------------------------------------------------------- Rename
   } elsif ($env{'form.renameres'}) {
       my $residx=$LONCAPA::map::order[$env{'form.renameidx'}-1];
       my ($name,@resrest)=split(/\:/,$LONCAPA::map::resources[$residx]);
       $name=$env{'form.renametitle'};
       $name=~s/\:/\&colon\;/g;
       $LONCAPA::map::resources[$residx]=$name.':'.join(':',@resrest);
       &LONCAPA::map::storemap(&Apache::lonnet::filelocation('',$url));
   }
# ------------------------------------------------------------ Assemble windows
   
   my $idx=-1;
   $importdetail='&'.$importdetail;
   $importdetail=~s/^\&+/\&/;
   my $importwindow=
       '<option value="-1"> ---- '.&mt('Import and Paste Area').' ---- </option>'.
     join("\n",map {
       $idx++;
       if ($_) { 
          my ($name,$url)=split(/\=/,$_);
          unless ($name) { $name=(split(/\//,$url))[-1]; }
          unless ($name) { $name='EMPTY'; }
          '<option value="'.$idx.'">'.&unescape($name).
                                    '</option>';
      }
   } split(/\&/,$importdetail));

   $idx=0;
   $targetdetail='';
   my $targetwindow=       
       '<option value="0"> ------- '.&mt('Target Edit Map').' ------- </option>'.
     join("\n",map { 
       my ($name,$url)=split(/\:/,$LONCAPA::map::resources[$_]);
       unless ($name) {  $name=(split(/\//,$url))[-1]; }
       unless ($name) { $name='EMPTY'; }
       $name = &LONCAPA::map::qtescape($name);
       $url  = &LONCAPA::map::qtescape($url);
       $targetdetail.='&'.&escape($name).'='.
	                  &escape($url);
       $idx++;
       $name = &HTML::Entities::encode($name,'\'"<>&');
       '<option value="'.$idx.'">'.$name.'</option>';
   } @LONCAPA::map::order);

# ----------------------------------------------------- Start simple RAT screen
   my $editscript=&editscript('simple');

   my %lt=&Apache::lonlocal::texthash(
				      'sa' => 'Save',
				      'nt' => 'New Title',
				      'se' => 'Search',
				      'im' => 'Import',
                                      'bk' => 'Import Bookmarks',
				      'vi' => 'View',
				      'lm' => 'Load Map',
				      'ds' => 'Discard Selected',
				      'ca' => 'Clear All',
				      'ta' => 'Temporary Assembly Workspace',
				      'rv' => 'Revert to Last Saved',
                                      'sa' => 'Save',
                                      'mu' => 'Move Up',
                                      'md' => 'Move Down',
                                      're' => 'Rename',
                                      'as' => 'after selected',
                                      'cs' => 'Cut selected',
                                      'ps' => 'Copy selected',
                                      'pas' => 'Paste after selected',
                                      'reco' => 'Recover Deleted'
				      );
   my $js=<<ENDJS;
<script type="text/javascript">

   $editscript

   function openview(entry) {
       var url=unescape((entry.split('='))[1]);
       var parts=new Array;
       if (url) { open(url,'cat'); }
   }

   function viewtarget() {
       openview((document.forms.simpleedit.targetdetail.value.split('&'))
                [document.forms.simpleedit.target.selectedIndex]);
   }

   function viewimport() {
       openview((document.forms.simpleedit.curimpdetail.value.split('&'))
                [document.forms.simpleedit.importsel.selectedIndex]);
   }

   function renametarget() {
       var selidx=document.forms.simpleedit.target.selectedIndex;
       var entry=(document.forms.simpleedit.targetdetail.value.split('&'))
                [selidx];
       var oldname=unescape((entry.split('='))[0]);
       newtitle=prompt('$lt{'nt'}',oldname);
       if (newtitle) {
           document.forms.simpleedit.renameres.value=1;
           document.forms.simpleedit.renameidx.value=selidx;
           document.forms.simpleedit.renametitle.value=newtitle;
	   document.forms.simpleedit.submit();
       }
   }

</script>
ENDJS

    my $start_page = &Apache::loncommon::start_page(undef,$js).
        &Apache::loncommon::help_open_menu('Sequence_Simple_Editor_Creation',
					   'Sequence_Simple_Editor_Creation',
					   6,'RAT');    
    my $end_page = &Apache::loncommon::end_page();

    $r->print(<<ENDSMPHEAD);
$start_page
$buttons
<font color=red>$errtext</font>
<form name=simpleedit method=post>
<input type=hidden name=forcesmp value=1>
<input type=hidden name=renameres value=0>
<input type=hidden name=renametitle value=''>
<input type=hidden name=renameidx value=0>
<table>
    <tr><th width="40%">$lt{'ta'}</th>
<th>&nbsp;</th>
<th width="40%">File: $url</th></tr>
<tr><td bgcolor="#FFFFCC">
<input type=button onClick="javascript:groupsearch()" value="$lt{'se'}">
<input type=button onClick="javascript:groupimport();" value="$lt{'im'}">
$lt{'as'}
<hr>
<input type=text size=20 name=importmap>
<input type=button 
onClick="javascript:openbrowser('simpleedit','importmap','sequence,page','')"
value="Select Map"><input type=submit name=loadmap value="$lt{'lm'}"><hr>
<input type=submit name="discard" value="$lt{'ds'}">
<input type=submit name="clear" value="$lt{'ca'}">
<input type=button onClick="javascript:viewimport()" value="$lt{'vi'}">

    </td><td>&nbsp;</td><td bgcolor="#FFFFCC">

<input type=button onClick=
"javascript:impfortarget.value=1;groupsearch()" value="$lt{'se'}">
<input type=button onClick=
"javascript:impfortarget.value=1;groupimport();" value="$lt{'im'}">
<input type=button onClick=
"javascript:impfortarget.value=1;groupopen(0,1,1);" value="$lt{'bk'}">
<input type=button onClick=
"javascript:impfortarget.value=1;groupopen('$url',1,0);" value="$lt{'reco'}">
$lt{'as'}
<hr>
<input type=submit name="moveup" value="$lt{'mu'}">
<input type=submit name="movedown" value="$lt{'md'}">
<input type=button onClick="javascript:renametarget()" value="$lt{'re'}">
<hr>$targetmsg
<input type=submit name="revert" value="$lt{'rv'}">
<input type=submit name="save" value="$lt{'sa'}">
<input type=button onClick="javascript:viewtarget()" value="$lt{'vi'}">
</td></tr>

<tr><td bgcolor="#FFFFCC"><select name="importsel" size=10 multiple>
$importwindow
</select>
</td>
<td bgcolor="#FFFFAA" align="center">
$lt{'cs'}<br>
<input type=submit name=cut value='<<<'><p>
<hr>
$lt{'ps'}<br>
<input type=submit name=copy value='<--'><p>
<hr>
$lt{'pas'}<br>
<input type=submit name=paste value='-->'>
</td>
<td bgcolor="#FFFFCC"><select name="target" size=10 multiple>
$targetwindow
</select>
</table>
<input type=hidden name=importdetail value="">
<input type=hidden name=curimpdetail value="$importdetail">
<input type=hidden name=targetdetail value="$targetdetail">
<input type=hidden name=impfortarget value="0">
</form>
$end_page
ENDSMPHEAD
}

# ----------------------------------------------------------------- No such dir
sub nodir {
   my ($r,$dir)=@_;
   $dir=~s{^/home/$LONCAPA::username_re/public_html}{};
   $r->print(&Apache::loncommon::start_page(undef,undef,
					    {'only_body' => 1,
					     'bgcolor'   => '#FFFFFF',}).
	     "<h1>No such directory: $dir</h1>".
	     &Apache::loncommon::end_page());
}

# ---------------------------------------------------------------- View Handler

sub viewmap {
    my ($r,$url,$adv,$errtext)=@_;
    $r->print(
	      &Apache::loncommon::start_page('Edit Content of a Map').
	      &Apache::loncommon::help_open_menu('','',6,'RAT').
	      &buttons($adv));
    if ($errtext) {
	$r->print($errtext.'<hr />');
    }
    my $idx=0;
    $r->print('<h1>'.$url.'</h1>');
    if ($adv) {
	$r->print('<p><b><font color="red">'.&mt('Map contents are not shown in order.').'</font></b></p><br />');
    }
    $r->print( '<table border="0" cellspacing="2" cellpadding="4">'.
               '<tr bgcolor="#DDDDDD"><th>'.&mt('Type').'</th>'.
               '<th>'.&mt('Title in map').'</th>'.
               '<th>'.&mt('Filename of resource').'</th>'.
               '<th>'.&mt('Link to published resource').'</th>'.
               '<th>'.&mt('Link to resource in Construction Space').'</th>'.
               '</tr>');
    my @backgroundColors = ("#FFFFFF", "#F6F6F6");
    foreach (&LONCAPA::map::attemptread(&Apache::lonnet::filelocation('',$url))) {
	if (defined($_)) {
            $idx++;
	    my ($title,$url,$cond)=split(/\:/,$_);
	    if ($cond eq 'cond') { next; }
            $title= &LONCAPA::map::qtescape($title);
            $url  = &LONCAPA::map::qtescape($url);
            unless ($title) { $title=(split(/\//,$url))[-1] };
            unless ($title) { $title='<i>'.&mt('Empty').'</i>'; }
            my $resurl = $url;
            my $resfilepath = $Apache::lonnet::perlvar{'lonDocRoot'}.$resurl;
            my $filename; 
            if ($resurl =~ m#/([^/]+)$#) {
                $filename = $1;
            }
            my $cstrurl = $resurl;
            $cstrurl =~ s#^/res/[^/]+/([^/]+)/#/priv/$1/#;
            my $bgcol = $idx%2; 
            $r->print('<tr bgcolor='.$backgroundColors[$bgcol].'><td>'.
                      '<img src="'.&Apache::loncommon::icon($resfilepath).
                      '" /></td><td>'.&HTML::Entities::encode(&LONCAPA::map::qtescape($title)).
                      '</td><td>'.$filename.'</td><td>');
            if ($url) {
		$r->print('<a href="'.$resurl.'">'.&mt('Resource space').'</a>');
            }
            $r->print('</td><td>');
            if ($url) {
               $r->print('<a href="'.$cstrurl.'">'.
                         &mt('Construction space').'</a>');
            }
            $r->print('</td></tr>'."\n");
        }
    }
    $r->print('</table>');
    $r->print(&Apache::loncommon::end_page());
}

# ================================================================ Main Handler

sub handler {
  my $r=shift;
  &Apache::loncommon::content_type($r,'text/html');
  $r->send_http_header;

  return OK if $r->header_only;
  my $target = $env{'form.grade_target'};
  if ($target eq 'meta') {
      &Apache::loncommon::content_type($r,'text/html');
      $r->send_http_header;
      return OK;
  }
  
  my $url=$r->uri;
  my $fn=&Apache::lonnet::filelocation('',$url);

  my ($dir)=($fn=~/^(.+)\/[^\/]+$/);
  unless (-e $dir) {
      &nodir($r,$dir);
      return OK;
  }

# ------------------------------------------- Determine which tools can be used
  my $adv=0;

  unless ($env{'form.forcesmp'}) {
     if ($env{'form.forceadv'}) {
        $adv=1;
     } elsif (my $fh=Apache::File->new($fn)) {
	 my $allmap=join('',<$fh>);
         $adv=($allmap=~/\<map[^\>]+mode\s*\=\s*(\'|\")rat/is);
     }
  }

  my $errtext='';
  my $fatal=0;

# -------------------------------------------------------------------- Load map
  ($errtext,$fatal)=&LONCAPA::map::mapread($fn,$errtext);

  if ($fatal==1) { $adv=1; }

# ----------------------------------- adv==1 now means "graphical MUST be used"

  if ($env{'form.forceadv'}) {
      &ratedt($r,$url);
  } elsif ($env{'form.forcesmp'}) {
      &smpedt($r,$url,$errtext);
  } else {
      &viewmap($r,$url,$adv,$errtext);
  }
  return OK;
}

1;
__END__







