# The LearningOnline Network with CAPA
# Interface routines to MAXIMA CAS
#
# $Id: lonmaxima.pm,v 1.26 2008/09/24 18:01:07 riegler Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
 
package Apache::lonmaxima;
 
use strict;
use IO::Socket;
use Apache::lonnet;
use Apache::response();
use LONCAPA;

sub connect {
   return IO::Socket::UNIX->new(Peer    => $Apache::lonnet::perlvar{'lonSockDir'}.'/maximasock',
				Type    => SOCK_STREAM,
				Timeout => 10);
}

sub disconnect {
    my ($socket)=@_;
    if ($socket) { close($socket); }
}

sub maximareply {
    my ($socket,$cmd)=@_;
    if ($socket) {
	print $socket &escape($cmd)."\n";
        my $reply=<$socket>;
        chomp($reply);
        if ($reply=~/^Incorrect/) { $reply='Error: '.$reply; }
        return &unescape($reply);
    } else {
        return 'Error: no connection.';
    }
}

sub blacklisted {
    my ($cmd)=@_;
    foreach my $forbidden (
        '\? ','\?','%o','batch','block'
       ,'compil','concat','describe','display2d','file','inchar'
       ,'includ','lisp','load','outchar','plot','quit'
       ,'read','reset','save','stin','stout','stringout'
       ,'system','translat','ttyoff','with_stdout','writefile'
     ) {
	if ($cmd=~/$forbidden/s) { return 1; }
    } 
    return 0;
}

sub maxima_allowed_libraries {
   return (
      "absimp","affine","atensor","atrig1","augmented_lagrangian","contrib_ode","ctensor","descriptive","diag",
      "eigen","facexp","fft","fourie","functs","ggf","grobner","impdiff","ineq","interpol","itensor","lapack",
      "lbfgs","lindstedt","linearalgebra","lsquares","makeOrders","mnewton","mchrpl","ntrig","orthopoly",
      "quadpack","rducon","romberg","scifac","simplex","solve_rec","sqdnst","stats","sterling","sym","units",
      "vect","zeilberger");
}

sub maxima_is_allowed_library {
    my ($library)=@_;
    foreach my $allowed_library (&maxima_allowed_libraries()) {
       if ($library eq $allowed_library) { return 1; }
    }
    return 0;
}

sub runscript {
    my ($socket,$fullscript,$libraries)=@_;
    if (&blacklisted($fullscript)) { return 'Error: blacklisted'; }
    my $reply;
    $fullscript=~s/[\n\r\l]//gs;
    if ($libraries) {
       foreach my $library (split(/\s*\,\s*/,$libraries)) {
          unless ($library=~/\w/) { next; }
          if (&maxima_is_allowed_library($library)) {
              $reply=&maximareply($socket,'load('.$library.')$'."\n");
              if ($reply=~/^Error\:/) { return $reply; }
          } else { 
             return 'Error: blacklisted'; 
          }
       }
    }
    foreach my $line (split(/\;/s,$fullscript)) {
	if ($line=~/\w/) { $reply=&maximareply($socket,$line.";\n"); }
	if ($reply=~/^Error\:/) { return $reply; }
    }
    $reply=~s/^\s*//gs;
    $reply=~s/\s*$//gs;
    &Apache::lonxml::debug("maxima $fullscript \n reply $reply");
    return $reply;
}

sub maxima_cas_formula_fix {
   my ($expression)=@_;
   return &Apache::response::implicit_multiplication($expression);
}

sub maxima_run {
    my ($script,$submission,$argument,$libraries) = @_;
    my $socket=&connect();
    my @submissionarray=split(/\s*\,\s*/,$submission);
    for (my $i=0;$i<=$#submissionarray;$i++) {
        my $n=$i+1;
        my $fixedsubmission=&maxima_cas_formula_fix($submissionarray[$i]);
        $script=~s/RESPONSE\[$n\]/$fixedsubmission/gs;
    }
    my @argumentarray=@{$argument};
    for (my $i=0;$i<=$#argumentarray;$i++) {
        my $n=$i+1;
        my $fixedargument=&maxima_cas_formula_fix($argumentarray[$i]);
        $script=~s/LONCAPALIST\[$n\]/$fixedargument/gs;
    }
    my $reply=&runscript($socket,$script,$libraries);
    &disconnect($socket);
    if ($reply=~/^\s*true\s*$/i) { return 'EXACT_ANS'; }
    if ($reply=~/^\s*false\s*$/i) { return 'INCORRECT'; } 
    return 'BAD_FORMULA';
}

sub maxima_eval {
    my ($script,$libraries) = @_;
    my $socket=&connect();
    my $reply=&runscript($socket,$script,$libraries);
    &disconnect($socket);
    return $reply;
}


sub compareterms {
    my ($socket,$terma,$termb)=@_;
    my $difference=$terma.'-('.$termb.')';
    if (&blacklisted($difference)) { return 'Error: blacklisted'; }
    my $reply=&maximareply($socket,'trigsimp(trigreduce('.$difference.'));');
    if ($reply=~/^\s*0\s*$/) { return 'true'; }
    if ($reply=~/^Error\:/) { return $reply; }
    return 'false';
}

sub maxima_check {
    my ($response,$answer,$reterror) = @_;
    my $socket=&connect();
    my $reply=&compareterms($socket,$response,$answer);
    &disconnect($socket);
    # integer to string mappings come from capaParser.h
    # 1 maps to 'EXACT_ANS'
    if ($reply eq 'true') { return 1; }
    # 11 maps to 'BAD_FORMULA'
    if ($reply=~/^Error\:/) { return 11; }
    # 7 maps to 'INCORRECT'
    return 7;
}
 
1;
__END__;
