# The LearningOnline Network with CAPA
#
# $Id: lonproblemstatistics.pm,v 1.113 2008/09/11 14:47:22 bisitz Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
# (Navigate problems for statistical reports
#
###############################################
###############################################

=pod

=head1 NAME

lonproblemstatistics

=head1 SYNOPSIS

Routines to present problem statistics to instructors via tables,
Excel files, and plots.

=over 4

=cut

###############################################
###############################################

package Apache::lonproblemstatistics;

use strict;
use Apache::lonnet;
use Apache::loncommon();
use Apache::lonhtmlcommon;
use Apache::loncoursedata;
use Apache::lonstatistics;
use LONCAPA::lonmetadata();
use Apache::lonlocal;
use Spreadsheet::WriteExcel;
use Apache::lonstathelpers();
use Time::HiRes;
use LONCAPA;
 

my @StatsArray;
my %SeqStat;    # keys are symbs, values are hash refs

##
## Localization notes:
##
## in @Fields[0]->{'long_title'} is placed in Excel files and is used as the
## header for plots created with Graph.pm, both of which more than likely do
## not support localization.
##
#
#
##
## Description of Field attributes
##
## Attribute     Required   Value       Meaning or Use
##
## name            yes      any scalar  Used to uniquely identify field
## title           yes      any scalar  This is what the user sees to identify
##                                      the field.  Passed through &mt().
## long_title      yes      any scalar  Used as graph heading and in excel
##                                      output.  NOT translated
## align           no    (left|right|center)  HTML cell contents alignment
## color           yes      html color  HTML cell background color
##                                      used to visually group statistics
## special         no          (link)   Indicates a link, target is name.link
##                                      Currently set in &get_statistics()
## graphable       no      (yes|no)     Can a bar graph of the field be 
##                                      produced?
## sortable        no      (yes|no)     Should a sort link be put in the
##                                      column header?
## selectable      yes     (yes|no)     Can the column be removed from the
##                                      statistics display?
## selected        yes     (yes|no)     Is the column selected by default?
##
## format          no      sprintf format string
##
## excel_format    no      excel format type 
##                               (see &Apache::loncommon::define_excel_formats
my @Fields = (
           { name => 'problem_num',
             title => 'P#',
             align => 'right',
             color => '#FFFFE6',
             selectable => 'no',
             defaultselected => 'yes',
           },
           { name   => 'container',
             title  => 'Sequence or Folder',
             align  => 'left',
             color  => '#FFFFE6',
             sortable => 'yes',
             selectable => 'no',
             defaultselected => 'yes',
           },
           { name   => 'title',
             title  => 'Title',
             align  => 'left',
             color  => '#FFFFE6',
             special  => 'link',
             sortable => 'yes', 
             selectable => 'no',
             defaultselected => 'yes',
           },
           { name   => 'part', 
             title  => 'Part',
             align  => 'left',
             color  => '#FFFFE6',
             selectable => 'no',
             defaultselected => 'yes',
           },
           { name   => 'num_students',
             title  => '#Stdnts',
             align  => 'right',
             color  => '#EEFFCC',
             format => '%d',
             sortable  => 'yes',
             graphable => 'yes',
             long_title => 'Number of Students Attempting Problem',
             selectable => 'yes',
             defaultselected => 'yes',
           },
           { name   => 'tries',
             title  => 'Tries',
             align  => 'right',
             color  => '#EEFFCC',
             format => '%d',
             sortable  => 'yes',
             graphable => 'yes',
             long_title => 'Total Number of Tries',
             selectable => 'yes',
             defaultselected => 'yes',
           },
           { name   => 'max_tries',
             title  => 'Max Tries',
             align  => 'right',
             color  => '#DDFFFF',
             format => '%d',
             sortable  => 'yes',
             graphable => 'yes',
             long_title => 'Maximum Number of Tries',
             selectable => 'yes',
             defaultselected => 'yes',
           },
           { name   => 'min_tries',
             title  => 'Min Tries',
             align  => 'right',
             color  => '#DDFFFF',
             format => '%d',
             sortable  => 'yes',
             graphable => 'yes',
             long_title => 'Minumum Number of Tries',
             selectable => 'yes',
             defaultselected => 'yes',
           },
           { name   => 'mean_tries',
             title  => 'Mean Tries',
             align  => 'right',
             color  => '#DDFFFF',
             format => '%5.2f',
             sortable  => 'yes',
             graphable => 'yes',
             long_title => 'Average Number of Tries',
             selectable => 'yes',
             defaultselected => 'yes',
           },
           { name   => 'std_tries',
             title  => 'S.D. tries',
             align  => 'right',
             color  => '#DDFFFF',
             format => '%5.2f',
             sortable  => 'yes',
             graphable => 'yes',
             long_title => 'Standard Deviation of Number of Tries',
             selectable => 'yes',
             defaultselected => 'yes',
           },
           { name   => 'skew_tries',
             title  => 'Skew Tries',
             align  => 'right',
             color  => '#DDFFFF',
             format => '%5.2f',
             sortable  => 'yes',
             graphable => 'yes',
             long_title => 'Skew of Number of Tries',
             selectable => 'yes',
             defaultselected => 'no',
           },
           { name   => 'num_solved',
             title  => '#YES',
             align  => 'right',
             color  => '#FFDDDD',
             format => '%4.1f',#             format => '%d',
             sortable  => 'yes',
             graphable => 'yes',
             long_title => 'Number of Students able to Solve',
             selectable => 'yes',
             defaultselected => 'yes',
           },
           { name   => 'num_override',
             title  => '#yes',
             align  => 'right',
             color  => '#FFDDDD',
             format => '%4.1f',#             format => '%d',
             sortable  => 'yes',
             graphable => 'yes',
             long_title => 'Number of Students given Override',
             selectable => 'yes',
             defaultselected => 'yes',
           },
           { name   => 'tries_per_correct',
             title  => 'tries/correct',
             align  => 'right',
             color  => '#FFDDDD',
             format => '%4.1f',
             sortable  => 'yes',
             graphable => 'yes',
             long_title => 'Tries per Correct Answer',
             selectable => 'yes',
             defaultselected => 'yes',
           },
           { name   => 'num_wrong',
             title  => '#Wrng',
             align  => 'right',
             color  => '#FFDDDD',
             format => '%4.1f',
             sortable  => 'yes',
             graphable => 'yes',
             long_title => 'Number of students whose final answer is wrong',
             selectable => 'yes',
             defaultselected => 'yes',
           },
           { name   => 'per_wrong',
             title  => '%Wrng',
             align  => 'right',
             color  => '#FFDDDD',
             format => '%4.1f',
             sortable  => 'yes',
             graphable => 'yes',
             long_title => 'Percent of students whose final answer is wrong',
             selectable => 'yes',
             defaultselected => 'yes',
           },
           { name   => 'deg_of_diff',
             title  => 'DoDiff',
             align  => 'right',
             color  => '#FFFFE6',
             format => '%5.2f',
             sortable  => 'yes',
             graphable => 'yes',
             long_title => 'Degree of Difficulty'.
                           '[ 1 - ((#YES+#yes) / Tries) ]',
             selectable => 'yes',
             defaultselected => 'yes',
           },
           { name   => 'deg_of_disc',
             title  => 'DoDisc',
             align  => 'right',
             color  => '#FFFFE6',
             format => '%4.2f',
             sortable  => 'yes',
             graphable => 'yes',
             long_title => 'Degree of Discrimination',
             selectable => 'yes',
             defaultselected => 'yes',
           },
##   duedate included for research purposes.  Commented out most of the time.
#           { name => 'duedate',
#             title => 'Due Date',
#             align => 'left',
#             color => '#FFFFFF',
#             sortable => 'yes',
#             graphable => 'no',
#             long_title => 'Due date of resource for instructor',
#             selectable => 'no',
#             defaultselected => 'yes',
#            },
##   opendate included for research purposes.  Commented out most of the time.
#           { name => 'opendate',
#             title => 'Open Date',
#             align => 'left',
#             color => '#FFFFFF',
#             sortable => 'yes',
#             graphable => 'no',
#             long_title => 'date resource became answerable',
#             selectable => 'no',
#             defaultselected => 'yes',
#            },
##   symb included for research purposes.  Commented out most of the time.
#           { name => 'symb',
#             title => 'Symb',
#             align => 'left',
#             color => '#FFFFFF',
#             sortable => 'yes',
#             graphable => 'no',
#             long_title => 'Unique LON-CAPA identifier for problem',
#             selectable => 'no',
#             defaultselected => 'yes',
#            },
##   resptypes included for research purposes.  Commented out most of the time.
#           { name => 'resptypes',
#             title => 'Response Types',
#             align => 'left',
#             color => '#FFFFFF',
#             sortable => 'no',
#             graphable => 'no',
#             long_title => 'Response Types used in this problem',
#             selectable => 'no',
#             defaultselected => 'yes',
#            },
##   maxtries included for research purposes.  Commented out most of the time.
#           { name => 'maxtries',
#             title => 'Maxtries',
#             align => 'left',
#             color => '#FFFFFF',
#             sortable => 'no',
#             graphable => 'no',
#             long_title => 'Maximum number of tries',
#             selectable => 'no',
#             defaultselected => 'yes',
#            },
##   hinttries included for research purposes.  Commented out most of the time.
#           { name => 'hinttries',
#             title => 'hinttries',
#             align => 'left',
#             color => '#FFFFFF',
#             sortable => 'no',
#             graphable => 'no',
#             long_title => 'Number of tries before a hint appears',
#             selectable => 'no',
#             defaultselected => 'yes',
#            },
#
##   problem weight for instructor
           { name => 'weight',
             title => 'weight',
             align => 'right',
             color => '#FFFFFF',
             sortable => 'no',
             graphable => 'no',
             long_title => 'Problem weight (for instructor)',
             selectable => 'yes',
             defaultselected => 'yes',
            },
);

my @SeqFields = (
           { name   => 'title',
             title  => 'Sequence',
             align  => 'left',
             color  => '#FFFFE6',
             special  => 'no',
             sortable => 'no', 
             selectable => 'yes',
             defaultselected => 'no',
           },
           { name   => 'items',
             title  => '#Items',
             align  => 'right',
             color  => '#FFFFE6',
             format => '%4d',
             sortable  => 'no',
             graphable => 'no',
             long_title => 'Number of Items in Sequence',
             selectable => 'yes',
             defaultselected => 'no',
           },
           { name   => 'scoremean',
             title  => 'Score Mean',
             align  => 'right',
             color  => '#FFFFE6',
             format => '%4.2f',
             sortable  => 'no',
             graphable => 'no',
             long_title => 'Mean Sequence Score',
             selectable => 'yes',
             defaultselected => 'no',
           },
           { name   => 'scorestd',
             title  => 'Score STD',
             align  => 'right',
             color  => '#FFFFE6',
             format => '%4.2f',
             sortable  => 'no',
             graphable => 'no',
             long_title => 'Standard Deviation of Sequence Scores',
             selectable => 'yes',
             defaultselected => 'no',
           },
           { name   => 'scoremax',
             title  => 'Score Max',
             align  => 'right',
             color  => '#FFFFE6',
             format => '%4.2f',
             sortable  => 'no',
             graphable => 'no',
             long_title => 'Maximum Sequence Score',
             selectable => 'yes',
             defaultselected => 'no',
           },
           { name   => 'scoremin',
             title  => 'Score Min',
             align  => 'right',
             color  => '#FFFFE6',
             format => '%4.2f',
             sortable  => 'no',
             graphable => 'no',
             long_title => 'Minumum Sequence Score',
             selectable => 'yes',
             defaultselected => 'no',
           },
           { name   => 'scorecount',
             title  => 'Score N',
             align  => 'right',
             color  => '#FFFFE6',
             format => '%4d',
             sortable  => 'no',
             graphable => 'no',
             long_title => 'Number of Students in score computations',
             selectable => 'yes',
             defaultselected => 'no',
           },
           { name   => 'countmean',
             title  => 'Count Mean',
             align  => 'right',
             color  => '#FFFFFF',
             format => '%4.2f',
             sortable  => 'no',
             graphable => 'no',
             long_title => 'Mean Sequence Score',
             selectable => 'yes',
             defaultselected => 'no',
           },
           { name   => 'countstd',
             title  => 'Count STD',
             align  => 'right',
             color  => '#FFFFFF',
             format => '%4.2f',
             sortable  => 'no',
             graphable => 'no',
             long_title => 'Standard Deviation of Sequence Scores',
             selectable => 'yes',
             defaultselected => 'no',
           },
           { name   => 'countmax',
             title  => 'Count Max',
             align  => 'right',
             color  => '#FFFFFF',
             format => '%4.2f',
             sortable  => 'no',
             graphable => 'no',
             long_title => 'Maximum Number of Correct Problems',
             selectable => 'yes',
             defaultselected => 'no',
           },
           { name   => 'countmin',
             title  => 'Count Min',
             align  => 'right',
             color  => '#FFFFFF',
             format => '%4.2f',
             sortable  => 'no',
             graphable => 'no',
             long_title => 'Minumum Number of Correct Problems',
             selectable => 'yes',
             defaultselected => 'no',
           },
           { name   => 'count',
             title  => 'Count N',
             align  => 'right',
             color  => '#FFFFFF',
             format => '%4d',
             sortable  => 'no',
             graphable => 'no',
             long_title => 'Number of Students in score computations',
             selectable => 'yes',
             defaultselected => 'no',
           },
           { name   => 'KR-21',
             title  => 'KR-21',
             align  => 'right',
             color  => '#FFAAAA',
             format => '%4.2f',
             sortable  => 'no',
             graphable => 'no',
             long_title => 'KR-21 reliability statistic',
             selectable => 'yes',
             defaultselected => 'no',
           },           
);

my %SelectedFields;

sub parse_field_selection {
    #
    # Pull out the defaults
    if (! defined($env{'form.fieldselections'})) {
        $env{'form.fieldselections'} = [];
        foreach my $field (@Fields) {
            next if ($field->{'selectable'} ne 'yes');
            if ($field->{'defaultselected'} eq 'yes') {
                push(@{$env{'form.fieldselections'}},$field->{'name'});
            }
        }
    }
    #
    # Make sure the data we are plotting is there
    my %NeededFields;
    if (exists($env{'form.plot'}) && $env{'form.plot'} ne '' &&
        $env{'form.plot'} ne 'none') {
        if ($env{'form.plot'} eq 'degrees') {
            $NeededFields{'deg_of_diff'}++;
            $NeededFields{'deg_of_disc'}++;
        } elsif ($env{'form.plot'} eq 'tries statistics') {
            $NeededFields{'mean_tries'}++;
            $NeededFields{'std_tries'}++;
            $NeededFields{'problem_num'}++;
        } else {
            $NeededFields{$env{'form.plot'}}++;
        }
    }
    #
    # This should not happen, but in case it does...
    if (ref($env{'form.fieldselections'}) ne 'ARRAY') {
        $env{'form.fieldselections'} = [$env{'form.fieldselections'}];
    }
    #
    # Set the field data and the selected fields (for easier checking)
    undef(%SelectedFields);
    foreach my $field (@Fields) {
        if ($field->{'selectable'} ne 'yes') {
            $field->{'selected'} = 'yes';
        } else {
            $field->{'selected'} = 'no';
        }
        if (exists($NeededFields{$field->{'name'}})) {
            $field->{'selected'} = 'yes';
            $SelectedFields{$field->{'name'}}++;
        }
        foreach my $selection (@{$env{'form.fieldselections'}}) {
            if ($selection eq $field->{'name'} || $selection eq 'all') {
                $field->{'selected'} = 'yes';
                $SelectedFields{$field->{'name'}}++;
            }
        }
    }
    #
    # Always show all the sequence statistics (for now)
    foreach my $field (@SeqFields) {
        $field->{'selected'} = 'yes';
    }
    return;
}

sub field_selection_input {
    my $Str = '<select name="fieldselections" multiple size="5">'."\n";
    $Str .= '<option value="all">all</option>'."\n";
    foreach my $field (@Fields) {
        next if ($field->{'selectable'} ne 'yes');
        $Str .= '    <option value="'.$field->{'name'}.'" ';
        if ($field->{'selected'} eq 'yes') {
            $Str .= 'selected ';
        }
        $Str .= '>'.$field->{'title'}.'</option>'."\n";
    }
    $Str .= "</select>\n";
}

###############################################
###############################################

=pod 

=item &CreateInterface()

Create the main intereface for the statistics page.  Allows the user to
select sections, maps, and output.

=cut

###############################################
###############################################
sub CreateInterface {
    my ($r) = @_;
    #
    &parse_field_selection();
    #
    my $Str = '';
    $Str .= &Apache::lonhtmlcommon::breadcrumbs('Overall Problem Statistics',
						'Statistics_Overall_Key');
    $Str .= '<table cellspacing="5">'."\n";
    $Str .= '<tr>';
    $Str .= '<td align="center"><b>'.&mt('Sections').'</b></td>';
    $Str .= '<td align="center"><b>'.&mt('Groups').'</b></td>';
    $Str .= '<td align="center"><b>'.&mt('Access Status').'</b></td>';
    $Str .= '<td align="center"><b>'.&mt('Sequences and Folders').'</b></td>';
    $Str .= '<td align="center"><b>'.&mt('Statistics').'</b></td>';
    $Str .= '<td rowspan="2">'.
        &Apache::lonstathelpers::limit_by_time_form().'</td>';
    $Str .= '</tr>'."\n";
    #
    $Str .= '<tr><td align="center">'."\n";
    $Str .= &Apache::lonstatistics::SectionSelect('Section','multiple',5);
    $Str .= '</td><td align="center">';
    $Str .= &Apache::lonstatistics::GroupSelect('Group','multiple',5);
    $Str .= '</td><td align="center">';
    $Str .= &Apache::lonhtmlcommon::StatusOptions(undef,undef,5);
    $Str .= '</td><td align="center">';
    #
    $Str .= &Apache::lonstatistics::map_select('Maps','multiple,all',5);
    $Str .= '</td><td>'.&field_selection_input();
    $Str .= '</td></tr>'."\n";
    $Str .= '</table>'."\n";
    #
    $Str .= '<p>'.&mt('Status: [_1]',
                         '<input type="text" '.
                         'name="stats_status" size="60" value="" />'
                         ).
                         '</nobr></p>';
    #
    $Str .= '<input type="submit" name="GenerateStatistics" value="'.
        &mt('Generate Statistics').'" />';
    $Str .= '&nbsp;'x5;
    $Str .= 'Plot '.&plot_dropdown().('&nbsp;'x10);
    #
    return $Str;
}

###############################################
###############################################

=pod 

=item &BuildProblemStatisticsPage()

Main interface to problem statistics.

=cut

###############################################
###############################################
my $navmap;
my @sequences;

sub clean_up {
    undef($navmap);
    undef(@sequences);
}

sub BuildProblemStatisticsPage {
    my ($r,$c)=@_;
    undef($navmap);
    undef(@sequences);
    #
    my %Saveable_Parameters = ('Status' => 'scalar',
                               'statsoutputmode' => 'scalar',
                               'Section' => 'array',
                               'Groups' => 'array',
                               'StudentData' => 'array',
                               'Maps' => 'array',
                               'fieldselections'=> 'array');
    &Apache::loncommon::store_course_settings('statistics',
                                              \%Saveable_Parameters);
    &Apache::loncommon::restore_course_settings('statistics',
                                                \%Saveable_Parameters);
    #
    &Apache::lonstatistics::PrepareClasslist();
    #
    # Clear the package variables
    undef(@StatsArray);
    undef(%SeqStat);
    #
    # Finally let the user know we are here
    my $interface = &CreateInterface($r);
    $r->print($interface);
    $r->print('<input type="hidden" name="sortby" value="'.$env{'form.sortby'}.
              '" />');
    #
    my @CacheButtonHTML = 
        &Apache::lonstathelpers::manage_caches($r,'Statistics','stats_status');
    my $Str;
    foreach my $html (@CacheButtonHTML) {
        $Str.=$html.('&nbsp;'x5);
    }
    #
    $r->print($Str);
    if (! exists($env{'form.firstrun'})) {
        $r->print('<h3>'.
                  &mt('Press "Generate Statistics" when you are ready.').
                  '</h3><p>'.
                  &mt('It may take some time to update the student data '.
                      'for the first analysis.  Future analysis this session '.
                      'will not have this delay.').
                  '</p>');
	&clean_up();
        return;
    }
    $r->rflush();
    #
    # This probably does not need to be done each time we are called, but
    # it does not slow things down noticably.
    &Apache::loncoursedata::populate_weight_table();
    #
    ($navmap,@sequences) = 
        &Apache::lonstatistics::selected_sequences_with_assessments();
    if (! ref($navmap)) {
        $r->print('<h1>'.&mt('A course-wide error occurred.').'</h1>'.
                  '<h3>'.$navmap.'</h3>');
	&clean_up();
        return;
    }
    if (exists($env{'form.Excel'})) {
        $r->print('<h4>'.
                  &Apache::lonstatistics::section_and_enrollment_description().
                  '</h4>');
        &Excel_output($r);
    } else { 
        $r->print('<input type="submit" name="Excel" value="'.
                  &mt('Produce Excel Output').'" />'.'&nbsp;'x5);
        $r->rflush();
        $r->print('<h4>'.
                  &Apache::lonstatistics::section_and_enrollment_description().
                  '</h4>');
        my $count = 0;
        foreach my $seq (@sequences) {
            my @resources = 
                &Apache::lonstathelpers::get_resources($navmap,$seq);
            $count += scalar(@resources);
        }
        if ($count > 10) {
            $r->print('<h2>'.
                      &mt('Compiling statistics for [_1] problems',$count).
                      '</h2>');
            if ($count > 30) {
                $r->print('<h3>'.&mt('This will take some time.').'</h3>');
            }
            $r->rflush();
        }
        #
        my $sortby = $env{'form.sortby'};
        $sortby = 'container' if (! defined($sortby) || $sortby =~ /^\s*$/);
        my $plot = $env{'form.plot'};
        if ($plot eq '' || $plot eq 'none') {
            undef($plot);
        }
        if ($sortby eq 'container' && ! defined($plot)) {
            &output_sequence_statistics($r);
            &output_html_by_sequence($r);
        } else {
            if (defined($plot)) {
                &make_plot($r,$plot);
            }
            &output_html_stats($r);
            &output_sequence_statistics($r);
        }
    }
    &clean_up();
    return;
}

sub output_sequence_statistics {
    my ($r) = @_;
    my $c=$r->connection();
    $r->print('<h2>'.&mt('Sequence Statistics').
	      &Apache::loncommon::help_open_topic('Statistics_Sequence').
	      '</h2>');
    $r->print('<table border="0"><tr><td bgcolor="#777777">'."\n".
              '<table border="0" cellpadding="3">'."\n".
              '<tr bgcolor="#FFFFE6">');
    $r->print(&sequence_html_header());
    foreach my $seq (@sequences) { 
        last if ($c->aborted);
        &compute_sequence_statistics($seq);
        $r->print(&sequence_html_output($seq));
    }
    $r->print('</table>');
    $r->print('</table>');
    $r->rflush();
    return;
}


##########################################################
##########################################################
##
## HTML output routines
##
##########################################################
##########################################################
sub output_html_by_sequence {
    my ($r) = @_;
    my $c = $r->connection();
    $r->print(&html_preamble());
    #
    foreach my $seq (@sequences) {
        last if ($c->aborted);
        $r->print("<h3>".$seq->compTitle."</h3>".
                  '<table border="0"><tr><td bgcolor="#777777">'."\n".
                  '<table border="0" cellpadding="3">'."\n".
                  '<tr bgcolor="#FFFFE6">'.
                  &statistics_table_header('no container')."</tr>\n");
        my @Data = &compute_statistics_on_sequence($seq);
        foreach my $data (@Data) {
            $r->print('<tr>'.&statistics_html_table_data($data,
                                                         'no container').
                      "</tr>\n");
        }
        $r->print('</table>'."\n".'</table>'."\n");
        $r->rflush();
    }
    return;
}

sub output_html_stats {
    my ($r)=@_;
    &compute_all_statistics($r);
    $r->print(&html_preamble());
    &sort_data($env{'form.sortby'});
    #
    my $count=0;
    foreach my $data (@StatsArray) {
        if ($count++ % 50 == 0) {
            $r->print("</table>\n</table>\n");
            $r->print('<table border="0"><tr><td bgcolor="#777777">'."\n".
                      '<table border="0" cellpadding="3">'."\n".
                      '<tr bgcolor="#FFFFE6">'.
                      '<tr bgcolor="#FFFFE6">'.
                      &statistics_table_header().
                      "</tr>\n");
        }
        $r->print('<tr>'.&statistics_html_table_data($data)."</tr>\n");
    }
    $r->print("</table>\n</table>\n");
    return;
}

sub html_preamble {
    my $Str='';
    $Str .= "<h2>".
        $env{'course.'.$env{'request.course.id'}.'.description'}.
        "</h2>\n";
    my ($starttime,$endtime) = &Apache::lonstathelpers::get_time_limits();
    if (defined($starttime) || defined($endtime)) {
        # Inform the user what the time limits on the data are.
        $Str .= '<h3>'.&mt('Statistics on submissions from [_1] to [_2]',
                           &Apache::lonlocal::locallocaltime($starttime),
                           &Apache::lonlocal::locallocaltime($endtime)
                           ).'</h3>';
    }
    $Str .= "<h3>".&mt('Compiled on [_1]',
                       &Apache::lonlocal::locallocaltime(time))."</h3>";
    return $Str;
}


###############################################
###############################################
##
## Misc HTML output routines
##
###############################################
###############################################
sub statistics_html_table_data {
    my ($data,$options) = @_;
    my $row = '';
    foreach my $field (@Fields) {
        next if ($options =~ /no $field->{'name'}/);
        next if ($field->{'selected'} ne 'yes');
        $row .= '<td bgcolor="'.$field->{'color'}.'"';
        if (exists($field->{'align'})) {
            $row .= ' align="'.$field->{'align'}.'"';
            }
        $row .= '>';
        if (exists($field->{'special'}) && $field->{'special'} eq 'link') {
            $row .= '<a href="'.$data->{$field->{'name'}.'.link'}.'">';
        }
        if (exists($field->{'format'}) && $data->{$field->{'name'}} !~ /[A-Z]/i) {
            $row .= sprintf($field->{'format'},$data->{$field->{'name'}});
        } else {
            $row .= $data->{$field->{'name'}};
        }
        if (exists($field->{'special'}) && $field->{'special'} eq 'link') {
            $row.= '</a>';
        }
        $row .= '</td>';
    }
    return $row;
}

sub statistics_table_header {
    my ($options) = @_;
    my $header_row;
    foreach my $field (@Fields) {
        next if ($options =~ /no $field->{'name'}/);
        next if ($field->{'selected'} ne 'yes');
        $header_row .= '<th>';
        if (exists($field->{'sortable'}) && $field->{'sortable'} eq 'yes') {
            $header_row .= '<a href="javascript:'.
                'document.Statistics.sortby.value='."'".$field->{'name'}."'".
                    ';document.Statistics.submit();">';
        }
        $header_row .= &mt($field->{'title'});
        if ($options =~ /sortable/) {
            $header_row.= '</a>';
        }
        if ($options !~ /no plots/        && 
            exists($field->{'graphable'}) && 
            $field->{'graphable'} eq 'yes') {
            $header_row.=' (';
            $header_row .= '<a href="javascript:'.
                "document.Statistics.plot.value='$field->{'name'}'".
                    ';document.Statistics.submit();">';
            $header_row .= &mt('plot').'</a>)';
        }
        $header_row .= '</th>';
    }
    return $header_row;
}

sub sequence_html_header {
    my $Str .= '<tr>';
    foreach my $field (@SeqFields) {
#        next if ($field->{'selected'} ne 'yes');
        $Str .= '<th bgcolor="'.$field->{'color'}.'"';
        $Str .= '>'.$field->{'title'}.'</th>';
    }
    $Str .= '</tr>';
    return $Str;
}


sub sequence_html_output {
    my ($seq) = @_;
    my $data = $SeqStat{$seq->symb};
    my $row = '<tr>';
    foreach my $field (@SeqFields) {
        next if ($field->{'selected'} ne 'yes');
        $row .= '<td bgcolor="'.$field->{'color'}.'"';
        if (exists($field->{'align'})) {
            $row .= ' align="'.$field->{'align'}.'"';
        }
        $row .= '>';
        if (exists($field->{'format'})) {
            $row .= sprintf($field->{'format'},$data->{$field->{'name'}});
        } else {
            $row .= $data->{$field->{'name'}};
        }
        $row .= '</td>';
    }
    $row .= '</tr>'."\n";
    return $row;
}

####################################################
####################################################
##
##    Plotting Routines
##
####################################################
####################################################
sub make_plot {
    my ($r,$plot) = @_;
    &compute_all_statistics($r);
    &sort_data($env{'form.sortby'});
    if ($plot eq 'degrees') {
        &degrees_plot($r);
    } elsif ($plot eq 'tries statistics') {
        &tries_data_plot($r);
    } else {
        &make_single_stat_plot($r,$plot);
    }
    return;
}

sub make_single_stat_plot {
    my ($r,$datafield) = @_;
    #
    my $title; my $yaxis;
    foreach my $field (@Fields) {
        next if ($field->{'name'} ne $datafield);
        $title = $field->{'long_title'};
        $yaxis = $field->{'title'};
        last;
    }
    if ($title eq '' || $yaxis eq '') {
        # datafield is something we do not know enough about to plot
        $r->print('<h3>'.
                  &mt('Unable to plot the requested statistic.').
                  '</h3>');
        return;
    }
    #
    # Build up the data sets to plot
    my @Labels; 
    my @Data;
    my $max = 1;
    foreach my $data (@StatsArray) {
        push(@Labels,$data->{'problem_num'});
        push(@Data,$data->{$datafield});
        if ($data->{$datafield}>$max) {
            $max = $data->{$datafield};
        }
    }
    foreach (1,2,3,4,5,10,15,20,25,40,50,75,100,150,200,250,300,500,600,750,
             1000,1500,2000,2500,3000,3500,4000,5000,7500,10000,15000,20000) {
        if ($max <= $_) {
            $max = $_;
            last;
        }
    }
    if ($max > 20000) {
        $max = 10000*(int($max/10000)+1);
    }
    #
    $r->print("<p>".&Apache::loncommon::DrawBarGraph($title,
                                                     'Problem Number',
                                                     $yaxis,
                                                     $max,
                                                     undef, # colors
                                                     \@Labels,
                                                     \@Data)."</p>\n");
    return;
}

sub degrees_plot {
    my ($r)=@_;
    my $count = scalar(@StatsArray);
    my $width = 50 + 10*$count;
    $width = 300 if ($width < 300);
    my $height = 300;
    my $plot = '';
    my $ymax = 0;
    my $ymin = 0;
    my @Disc; my @Diff; my @Labels;    
    foreach my $data (@StatsArray) {
        push(@Labels,$data->{'problem_num'});
        my $disc = $data->{'deg_of_disc'};
        my $diff = $data->{'deg_of_diff'};
        push(@Disc,$disc);
        push(@Diff,$diff);
        #
        $ymin = $disc if ($ymin > $disc);
        $ymin = $diff if ($ymin > $diff);
        $ymax = $disc if ($ymax < $disc);
        $ymax = $diff if ($ymax < $diff);
    }
    #
    # Make sure we show relevant information.
    if ($ymin < 0) {
        if (abs($ymin) < 0.05) {
            $ymin = 0;
        } else {
            $ymin = -1;
        }
    }
    if ($ymax > 0) {
        if (abs($ymax) < 0.05) {
            $ymax = 0;
        } else {
            $ymax = 1;
        }
    }
    #
    my $xmax = $Labels[-1];
    if ($xmax > 50) {
        if ($xmax % 10 != 0) {
            $xmax = 10 * (int($xmax/10)+1);
        }
    } else {
        if ($xmax % 5 != 0) {
            $xmax = 5 * (int($xmax/5)+1);
        }
    }
    #
    my $discdata .= '<data>'.join(',',@Labels).'</data>'.$/.
                    '<data>'.join(',',@Disc).'</data>'.$/;
    #
    my $diffdata .= '<data>'.join(',',@Labels).'</data>'.$/.
                    '<data>'.join(',',@Diff).'</data>'.$/;
    #
    my $title = 'Degree of Discrimination\nand Degree of Difficulty';
    if ($xmax > 50) {
        $title = 'Degree of Discrimination and Degree of Difficulty';
    }
    #
    $plot=<<"END";
<gnuplot 
    texfont="10"
    fgcolor="x000000"
    plottype="Cartesian"
    font="large"
    grid="on"
    align="center"
    border="on"
    transparent="on"
    alttag="Degree of Discrimination and Degree of Difficulty Plot"
    samples="100"
    bgcolor="xffffff"
    height="$height"
    width="$width">
    <key 
        pos="top right"
        title=""
        box="off" />
    <title>$title</title>
    <axis xmin="0" ymin="$ymin" xmax="$xmax" ymax="$ymax" color="x000000" />
    <xlabel>Problem Number</xlabel>
    <curve 
        linestyle="linespoints" 
        name="DoDisc" 
        pointtype="0" 
        color="x000000">
        $discdata
    </curve>
    <curve 
        linestyle="linespoints" 
        name="DoDiff" 
        pointtype="0" 
        color="xFF0000">
        $diffdata
    </curve>
</gnuplot>
END
    my $plotresult = 
        '<p>'.&Apache::lonxml::xmlparse($r,'web',$plot).'</p>'.$/;
    $r->print($plotresult);
    return;
}

sub tries_data_plot {
    my ($r)=@_;
    my $count = scalar(@StatsArray);
    my $width = 50 + 10*$count;
    $width = 300 if ($width < 300);
    my $height = 300;
    my $plot = '';
    my @STD;  my @Mean; my @Max; my @Min;
    my @Labels;
    my $ymax = 5;
    foreach my $data (@StatsArray) {
        my $max = $data->{'mean_tries'} + $data->{'std_tries'};
        $ymax = $max if ($ymax < $max);
        $ymax = $max if ($ymax < $max);
        push(@Labels,$data->{'problem_num'});
        push(@STD,$data->{'std_tries'});
        push(@Mean,$data->{'mean_tries'});
    }
    #
    # Make sure we show relevant information.
    my $xmax = $Labels[-1];
    if ($xmax > 50) {
        if ($xmax % 10 != 0) {
            $xmax = 10 * (int($xmax/10)+1);
        }
    } else {
        if ($xmax % 5 != 0) {
            $xmax = 5 * (int($xmax/5)+1);
        }
    }
    $ymax = int($ymax)+1+2;
    #
    my $std_data .= '<data>'.join(',',@Labels).'</data>'.$/.
                    '<data>'.join(',',@Mean).'</data>'.$/;
    #
    my $std_error_data .= '<data>'.join(',',@Labels).'</data>'.$/.
                          '<data>'.join(',',@Mean).'</data>'.$/.
                          '<data>'.join(',',@STD).'</data>'.$/;
    #
    my $title = 'Mean and S.D. of Tries';
    if ($xmax > 25) {
        $title = 'Mean and Standard Deviation of Tries';
    }
    #
    $plot=<<"END";
<gnuplot 
    texfont="10"
    fgcolor="x000000"
    plottype="Cartesian"
    font="large"
    grid="on"
    align="center"
    border="on"
    transparent="on"
    alttag="Mean and S.D of Tries Plot"
    samples="100"
    bgcolor="xffffff"
    height="$height"
    width="$width">
    <title>$title</title>
    <axis xmin="0" ymin="0" xmax="$xmax" ymax="$ymax" color="x000000" />
    <xlabel>Problem Number</xlabel>
    <ylabel>Number of Tries</ylabel>
    <curve 
        linestyle="yerrorbars"
        name="S.D. Tries" 
        pointtype="1" 
        color="x666666">
        $std_error_data
    </curve>
    <curve 
        linestyle="points"
        name="Mean Tries" 
        pointtype="1" 
        color="xCC4444">
        $std_data
    </curve>
</gnuplot>
END
    my $plotresult = 
        '<p>'.&Apache::lonxml::xmlparse($r,'web',$plot).'</p>'.$/;
    $r->print($plotresult);
    return;
}

sub plot_dropdown {
    my $current = '';
    #
    if (defined($env{'form.plot'})) {
        $current = $env{'form.plot'};
    }
    #
    my @Additional_Plots = (
                            { graphable=>'yes',
                              name => 'degrees',
                              title => 'Difficulty Indexes' },
                            { graphable=>'yes',
                              name => 'tries statistics',
                              title => 'Tries Statistics' });
    #
    my $Str= "\n".'<select name="plot" size="1">';
    $Str .= '<option name="none"></option>'."\n";
    $Str .= '<option name="none2">none</option>'."\n";
    foreach my $field (@Additional_Plots,@Fields) {
        if (! exists($field->{'graphable'}) ||
            $field->{'graphable'} ne 'yes') {
            next;
        }
        $Str .= '<option value="'.$field->{'name'}.'"';
        if ($field->{'name'} eq $current) {
            $Str .= ' selected ';
        }
        $Str.= '>'.&mt($field->{'title'}).'</option>'."\n";
    }
    $Str .= '</select>'."\n";
    return $Str;
}

###############################################
###############################################
##
## Excel output routines
##
###############################################
###############################################
sub Excel_output {
    my ($r) = @_;
    $r->print('<h2>'.&mt('Preparing Excel Spreadsheet').'</h2>');
    ##
    ## Compute the statistics
    &compute_all_statistics($r);
    my $c = $r->connection;
    return if ($c->aborted());
    #
    my ($starttime,$endtime) = &Apache::lonstathelpers::get_time_limits();
    ##
    ## Create the excel workbook
    my ($excel_workbook,$filename,$format) =
        &Apache::loncommon::create_workbook($r);
    return if (! defined($excel_workbook));
    #
    # Add a worksheet
    my $sheetname = $env{'course.'.$env{'request.course.id'}.'.description'};
    if (length($sheetname) > 31) {
        $sheetname = substr($sheetname,0,31);
    }
    my $excel_sheet = $excel_workbook->addworksheet(
        &Apache::loncommon::clean_excel_name($sheetname));
    ##
    ## Begin creating excel sheet
    ##
    my ($rows_output,$cols_output) = (0,0);
    #
    # Put the course description in the header
    $excel_sheet->write($rows_output,$cols_output++,
                   $env{'course.'.$env{'request.course.id'}.'.description'},
                        $format->{'h1'});
    $cols_output += 3;
    #
    # Put a description of the sections listed
    my $sectionstring = '';
    $excel_sheet->write($rows_output,$cols_output++,
                        &Apache::lonstatistics::section_and_enrollment_description('plaintext'),
                        $format->{'h3'});
    $cols_output += scalar(&Apache::lonstatistics::get_selected_sections());
    $cols_output += scalar(&Apache::lonstatistics::get_selected_groups());
    #
    # Time restrictions
    my $time_string;
    if (defined($starttime)) {
        # call localtime but not lonlocal:locallocaltime because excel probably
        # cannot handle localized text.  Probably.
        $time_string .= 'Data collected from '.localtime($time_string);
        if (defined($endtime)) {
            $time_string .= ' to '.localtime($endtime);
        }
        $time_string .= '.';
    } elsif (defined($endtime)) {
        # See note above about lonlocal:locallocaltime
        $time_string .= 'Data collected before '.localtime($endtime).'.';
    }
    if (defined($time_string)) {
        $excel_sheet->write($rows_output,$cols_output++,$time_string);
        $cols_output+= 5;
    }
    #
    # Put the date in there too
    $excel_sheet->write($rows_output,$cols_output++,
                        'Compiled on '.localtime(time));
    #
    $rows_output++; 
    $cols_output=0;
    ##
    ## Sequence Statistics
    ## 
    &write_headers($excel_sheet,$format,\$rows_output,\$cols_output,
                   \@SeqFields);
    foreach my $seq (@sequences) {
        my $data = $SeqStat{$seq->symb};
        $cols_output=0;
        foreach my $field (@SeqFields) {
            next if ($field->{'selected'} ne 'yes');
            my $fieldformat = undef;
            if (exists($field->{'excel_format'})) {
                $fieldformat = $format->{$field->{'excel_format'}};
            }
            $excel_sheet->write($rows_output,$cols_output++,
                                $data->{$field->{'name'}},$fieldformat);
        }
        $rows_output++;
        $cols_output=0;
    }
    ##
    ## Resource Statistics
    ##
    $rows_output++;
    $cols_output=0;
    &write_headers($excel_sheet,$format,\$rows_output,\$cols_output,
                   \@Fields);
    #
    foreach my $data (@StatsArray) {
        $cols_output=0;
        foreach my $field (@Fields) {
            next if ($field->{'selected'} ne 'yes');
            next if ($field->{'name'} eq 'problem_num');
            my $fieldformat = undef;
            if (exists($field->{'excel_format'})) {
                $fieldformat = $format->{$field->{'excel_format'}};
            }
            $excel_sheet->write($rows_output,$cols_output++,
                                $data->{$field->{'name'}},$fieldformat);
        }
        $rows_output++;
        $cols_output=0;
    }
    #
    $excel_workbook->close();
    #
    # Tell the user where to get their excel file
    $r->print('<br />'.
              '<a href="'.$filename.'">'.
              &mt('Your Excel Spreadsheet').'</a>'."\n");
    $r->rflush();
    return;
}

##
## &write_headers
##
sub write_headers {
    my ($excel_sheet,$format,$rows_output,$cols_output,$Fields) = @_;
    ##
    ## First the long titles
    foreach my $field (@{$Fields}) {
        next if ($field->{'name'} eq 'problem_num');
        next if ($field->{'selected'} ne 'yes');
        if (exists($field->{'long_title'})) {
            $excel_sheet->write($$rows_output,${$cols_output},
                                $field->{'long_title'},
                                $format->{'bold'});
        } else {
            $excel_sheet->write($$rows_output,${$cols_output},'');
        }
        ${$cols_output}+= 1;
    }
    ${$cols_output} =0;
    ${$rows_output}+=1;
    ##
    ## Then the short titles
    foreach my $field (@{$Fields}) {
        next if ($field->{'selected'} ne 'yes');
        next if ($field->{'name'} eq 'problem_num');
        # Use english for excel as I am not sure how well excel handles 
        # other character sets....
        $excel_sheet->write($$rows_output,$$cols_output,
                            $field->{'title'},
                            $format->{'bold'});
        $$cols_output+=1;
    }
    ${$cols_output} =0;
    ${$rows_output}+=1;
    return;
}

##################################################
##################################################
##
## Statistics Gathering and Manipulation Routines
##
##################################################
##################################################
sub compute_statistics_on_sequence {
    my ($seq) = @_;
    my @Data;
    foreach my $res (&Apache::lonstathelpers::get_resources($navmap,$seq)) {
        foreach my $part (@{$res->parts}) {
            next if ($res->is_survey($part));
            #
            # This is where all the work happens
            my $data = &get_statistics($seq,$res,$part,scalar(@StatsArray)+1);
            push (@Data,$data);
            push (@StatsArray,$data);
        }
    }
    return @Data;
}

sub compute_all_statistics {
    my ($r) = @_;
    if (@StatsArray > 0) {
        # Assume we have already computed the statistics
        return;
    }
    my $c = $r->connection;
    foreach my $seq (@sequences) {
        last if ($c->aborted);
        &compute_sequence_statistics($seq);
        &compute_statistics_on_sequence($seq);
    }
}

sub sort_data {
    my ($sortkey) = @_;
    return if (! @StatsArray);
    #
    # Sort the data
    my $sortby = undef;
    foreach my $field (@Fields) {
        if ($sortkey eq $field->{'name'}) {
            $sortby = $field->{'name'};
        }
    }
    if (! defined($sortby) || $sortby eq '' || $sortby eq 'problem_num') {
        $sortby = 'container';
    }
    if ($sortby ne 'container') {
        # $sortby is already defined, so we can charge ahead
        if ($sortby =~ /^(title|part)$/i) {
            # Alpha comparison
            @StatsArray = sort {
                lc($a->{$sortby}) cmp lc($b->{$sortby}) ||
                lc($a->{'title'}) cmp lc($b->{'title'}) ||
                lc($a->{'part'}) cmp lc($b->{'part'});
            } @StatsArray;
        } else {
            # Numerical comparison
            @StatsArray = sort {
                my $retvalue = 0;
                if ($b->{$sortby} eq 'nan') {
                    if ($a->{$sortby} ne 'nan') {
                        $retvalue = -1;
                    } else {
                        $retvalue = 0;
                    }
                }
                if ($a->{$sortby} eq 'nan') {
                    if ($b->{$sortby} ne 'nan') {
                        $retvalue = 1;
                    }
                }
                if ($retvalue eq '0') {
                    $retvalue = $b->{$sortby} <=> $a->{$sortby}     ||
                            lc($a->{'title'}) <=> lc($b->{'title'}) ||
                            lc($a->{'part'})  <=> lc($b->{'part'});
                }
                $retvalue;
            } @StatsArray;
        }
    }
    #
    # Renumber the data set
    my $count;
    foreach my $data (@StatsArray) {
        $data->{'problem_num'} = ++$count;
    }
    return;
}

########################################################
########################################################

=pod

=item &get_statistics()

Wrapper routine from the call to loncoursedata::get_problem_statistics.  
Calls lonstathelpers::get_time_limits() to limit the data set by time
and &compute_discrimination_factor

Inputs: $sequence, $resource, $part, $problem_num

Returns: Hash reference with statistics data from 
loncoursedata::get_problem_statistics.

=cut

########################################################
########################################################
sub get_statistics {
    my ($sequence,$resource,$part,$problem_num) = @_;
    #
    my ($starttime,$endtime) = &Apache::lonstathelpers::get_time_limits();
    my $symb = $resource->symb;
    my $courseid = $env{'request.course.id'};
    #
    my $data = &Apache::loncoursedata::get_problem_statistics
                        ([&Apache::lonstatistics::get_selected_sections()],
                         [&Apache::lonstatistics::get_selected_groups()],
                         $Apache::lonstatistics::enrollment_status,
                         $symb,$part,$courseid,$starttime,$endtime);
    $data->{'symb'}        = $symb;
    $data->{'part'}        = $part;
    $data->{'problem_num'} = $problem_num;
    $data->{'container'}   = $sequence->compTitle;
    $data->{'title'}       = $resource->compTitle;
    $data->{'title.link'}  = $resource->src.'?symb='.
        &escape($resource->symb);
    #
    if ($SelectedFields{'deg_of_disc'}) {
        $data->{'deg_of_disc'} = 
            &compute_discrimination_factor($resource,$part,$sequence);
    }
    #
    # Store in metadata if computations were done for all students
    if ($data->{'num_students'} > 1) {
        my @Sections = &Apache::lonstatistics::get_selected_sections();
        my $sections = '"'.join(' ',@Sections).'"';
        $sections =~ s/&+/_/g;  # Ensure no special characters
        $data->{'sections'}=$sections;
        $data->{'course'} = $env{'request.course.id'};
        my $urlres=(&Apache::lonnet::decode_symb($resource->symb))[2];
        $data->{'urlres'}=$urlres;
        my %storestats = 
            &LONCAPA::lonmetadata::dynamic_metadata_storage($data);
        my ($dom,$user) = ($urlres=~m{^($LONCAPA::domain_re)/($LONCAPA::username_re)}); 
        &Apache::lonnet::put('nohist_resevaldata',\%storestats,$dom,$user);
    }
    #
    $data->{'tries_per_correct'} = $data->{'tries'} / 
        ($data->{'num_solved'}+0.1);
    #
    # Get the due date for research purposes (commented out most of the time)
#    my $duedate = &Apache::lonnet::EXT('resource.'.$part.'.duedate',$symb);;
#    my $opendate = &Apache::lonnet::EXT('resource.'.$part.'.opendate',$symb);
#    my $maxtries = &Apache::lonnet::EXT('resource.'.$part.'.maxtries',$symb);
#    my $hinttries = &Apache::lonnet::EXT('resource.'.$part.'.hinttries',$symb);
    my $weight = &Apache::lonnet::EXT('resource.'.$part.'.weight',$symb);
    $data->{'weight'} = $weight;
#    $data->{'duedate'} = $duedate; 
#    $data->{'opendate'} = $opendate;
#    $data->{'maxtries'} = $maxtries;
#    $data->{'hinttries'} = $hinttries;
#    $data->{'resptypes'} = join(',',@{$resource->{'partdata'}->{$part}->{'ResponseTypes'}});
    return $data;
}

###############################################
###############################################

=pod

=item &compute_discrimination_factor()

Inputs: $Resource, $Sequence

Returns: integer between -1 and 1

=cut

###############################################
###############################################
sub compute_discrimination_factor {
    my ($resource,$part,$seq) = @_;
    my $symb = $resource->symb;
    my @Resources;
    foreach my $res (&Apache::lonstathelpers::get_resources($navmap,$seq)){
        next if ($res->symb eq $symb);
        push (@Resources,$res->symb);
    }
    #
    # rank
    my ($starttime,$endtime) = &Apache::lonstathelpers::get_time_limits();
    my $ranking = 
        &Apache::loncoursedata::rank_students_by_scores_on_resources
        (\@Resources,
         [&Apache::lonstatistics::get_selected_sections()],
         [&Apache::lonstatistics::get_selected_groups()],
         $Apache::lonstatistics::enrollment_status,undef,
         $starttime,$endtime, $symb);
    #
    # compute their percent scores on the problems in the sequence,
    my $number_to_grab = int(scalar(@{$ranking})/4);
    my $num_students = scalar(@{$ranking});
    my @BottomSet = map { $_->[&Apache::loncoursedata::RNK_student()]; 
                      } @{$ranking}[0..$number_to_grab];
    my @TopSet    = 
        map { 
            $_->[&Apache::loncoursedata::RNK_student()]; 
          } @{$ranking}[-$number_to_grab..0];
    if (! @BottomSet || (@BottomSet == 1 && $BottomSet[0] eq '') ||
        ! @TopSet    || (@TopSet    == 1 && $TopSet[0]    eq '')) {
        return 'nan';
    }
    my ($bottom_sum,$bottom_max) = 
        &Apache::loncoursedata::get_sum_of_scores($symb,$part,\@BottomSet,
                                                  undef,$starttime,$endtime);
    my ($top_sum,$top_max) = 
        &Apache::loncoursedata::get_sum_of_scores($symb,$part,\@TopSet,
                                                  undef,$starttime,$endtime);
    my $deg_of_disc;
    if ($top_max == 0 || $bottom_max==0) {
        $deg_of_disc = 'nan';
    } else {
        $deg_of_disc = ($top_sum/$top_max) - ($bottom_sum/$bottom_max);
    }
    #&Apache::lonnet::logthis('    '.$top_sum.'/'.$top_max.
    #                         ' - '.$bottom_sum.'/'.$bottom_max);
    return $deg_of_disc;
}

###############################################
###############################################
##
## Compute KR-21
##
## To compute KR-21, you need the following information:
##
## K=the number of items in your test
## M=the mean score on the test
## s=the standard deviation of the scores on your test 
##
## then:
## 
## KR-21 rk= [K/(K-1)] * [1- (M*(K-M))/(K*s^2))]
##
###############################################
###############################################
sub compute_sequence_statistics {
    my ($seq) = @_;
    my $symb = $seq->symb;
    my @Resources;
    my $part_count;
    foreach my $res (&Apache::lonstathelpers::get_resources($navmap,$seq)) {
        push (@Resources,$res->symb);
        $part_count += scalar(@{$res->parts});
    }
    my ($starttime,$endtime) = &Apache::lonstathelpers::get_time_limits();
    #
    # First compute statistics based on student scores
    my ($smin,$smax,$sMean,$sSTD,$scount,$sMAX) = 
        &Apache::loncoursedata::score_stats
                    ([&Apache::lonstatistics::get_selected_sections()],
                     [&Apache::lonstatistics::get_selected_groups()],
                     $Apache::lonstatistics::enrollment_status,
                     \@Resources,$starttime,$endtime,undef);
    $SeqStat{$symb}->{'title'}  = $seq->compTitle;
    $SeqStat{$symb}->{'scoremax'}  = $smax;
    $SeqStat{$symb}->{'scoremin'}  = $smin;
    $SeqStat{$symb}->{'scoremean'} = $sMean;
    $SeqStat{$symb}->{'scorestd'}  = $sSTD;
    $SeqStat{$symb}->{'scorecount'} = $scount;
    $SeqStat{$symb}->{'max_possible'} = $sMAX;
    #
    # Compute statistics based on the number of correct problems
    # 'correct' is taken to mean 
    my ($cmin,$cmax,$cMean,$cSTD,$ccount)=
        &Apache::loncoursedata::count_stats
        ([&Apache::lonstatistics::get_selected_sections()],
         [&Apache::lonstatistics::get_selected_groups()],
         $Apache::lonstatistics::enrollment_status,
         \@Resources,$starttime,$endtime,undef);
    my $K = $part_count;
    my $kr_21;
    if ($K > 1 && $cSTD > 0) {
        $kr_21 =  ($K/($K-1)) * (1 - $cMean*($K-$cMean)/($K*$cSTD**2));
    } else {
        $kr_21 = 'nan';
    }
    $SeqStat{$symb}->{'countmax'} = $cmax;
    $SeqStat{$symb}->{'countmin'} = $cmin;
    $SeqStat{$symb}->{'countstd'} = $cSTD;
    $SeqStat{$symb}->{'countmean'} = $cMean;
    $SeqStat{$symb}->{'count'} = $ccount;
    $SeqStat{$symb}->{'items'} = $K;
    $SeqStat{$symb}->{'KR-21'}=$kr_21;
    return;
}



=pod 

=item ProblemStatisticsLegend

=over 4

=item #Stdnts
Total number of students attempted the problem.

=item Tries
Total number of tries for solving the problem.

=item Max Tries
Largest number of tries for solving the problem by a student.

=item Mean
Average number of tries. [ Tries / #Stdnts ]

=item #YES
Number of students solved the problem correctly.

=item #yes
Number of students solved the problem by override.

=item %Wrong
Percentage of students who tried to solve the problem 
but is still incorrect. [ 100*((#Stdnts-(#YES+#yes))/#Stdnts) ]

=item DoDiff
Degree of Difficulty of the problem.  
[ 1 - ((#YES+#yes) / Tries) ]

=item S.D.
Standard Deviation of the tries.  
[ sqrt(sum((Xi - Mean)^2)) / (#Stdnts-1) 
where Xi denotes every student\'s tries ]

=item Skew.
Skewness of the students tries.
[(sqrt( sum((Xi - Mean)^3) / #Stdnts)) / (S.D.^3)]

=item Dis.F.
Discrimination Factor: A Standard for evaluating the 
problem according to a Criterion<br>

=item [Criterion to group students into %27 Upper Students - 
and %27 Lower Students]
1st Criterion for Sorting the Students: 
Sum of Partial Credit Awarded / Total Number of Tries
2nd Criterion for Sorting the Students: 
Total number of Correct Answers / Total Number of Tries

=item Disc.
Number of Students had at least one discussion.

=back

=cut

############################################################
############################################################

1;
__END__
