# The LearningOnline Network with CAPA
# Construction Space Directory Lister
#
# $Id: lonpubdir.pm,v 1.110 2008/08/08 10:08:48 bisitz Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###

package Apache::lonpubdir;

use strict;
use Apache::File;
use File::Copy;
use Apache::Constants qw(:common :http :methods);
use Apache::loncacc;
use Apache::loncommon();
use Apache::lonhtmlcommon();
use Apache::londiff();
use Apache::lonlocal;
use Apache::lonmsg;
use Apache::lonmenu;
use Apache::lonnet;
use LONCAPA;

sub handler {

  my $r=shift;

  my $fn;



  $fn = getEffectiveUrl($r);

  # Validate access to the construction space and get username@domain.

  my $uname;
  my $udom;

  ($uname,$udom)=
    &Apache::loncacc::constructaccess(
             $fn,$r->dir_config('lonDefDomain')); 
  unless (($uname) && ($udom)) {
     $r->log_reason($uname.' at '.$udom.
         ' trying to list directory '.$env{'form.filename'}.
         ' ('.$fn.') - not authorized', 
         $r->filename); 
     return HTTP_NOT_ACCEPTABLE;
  }

  # Remove trailing / from directory name.

  $fn=~s/\/$//;

  unless ($fn) { 
     $r->log_reason($env{'user.name'}.' at '.$env{'user.domain'}.
         ' trying to list empty directory', $r->filename); 
     return HTTP_NOT_FOUND;
  } 

# ----------------------------------------------------------- Start page output

  my $thisdisfn=$fn;
  $thisdisfn=~s/^\/home\/$uname\/public_html//;	# subdirectory part of
                                                # construction space. 
  my $docroot=$r->dir_config('lonDocRoot');     # Apache  londocument root.

  my $resdir=$docroot.'/res/'.$udom.'/'.$uname.$thisdisfn; # Resource directory
  my $targetdir=$udom.'/'.$uname.$thisdisfn; # Publiction target directory.
  my $linkdir='/priv/'.$uname.$thisdisfn;      # Full URL name of constr space.

  my %bombs=&Apache::lonmsg::all_url_author_res_msg($uname,$udom);

  &startpage($r, $uname, $udom, $thisdisfn);   # Put out the start of page.
  if ($env{'environment.remote'} eq 'off') {
      &dircontrols($r,$uname,$udom,$thisdisfn);    # Put out actions for directory, 
                                               # browse/upload + new file page.
  } else {
      &pubbuttons($r,$uname,$thisdisfn);
  }
  &resourceactions($r,$uname,$udom,$thisdisfn); #Put out form used for printing/deletion etc.

  my $numdir = 0;
  my $numres = 0;
  
  # Start off the directory table.
  $r->print('<h3>'.&mt('Directory Contents:').'</h3>');
  $r->print('<table id="LC_browser"><tr>'.
            '<th>'.&mt('Type').'</th>'.
            '<th>'.&mt('Actions').'</th>'.
            '<th>'.&mt('Name').'</th>'.
            '<th>'.&mt('Title').'</th>'.
	    '<th>'.&mt('Status').'</th>'.
            '<th>'.&mt('Last Modified').
	    '</th></tr>'."\n");

  my $filename;
  my $dirptr=16384;		# Mask indicating a directory in stat.cmode.

  opendir(DIR,$fn);
  my @files=sort {uc($a) cmp uc($b)} (readdir(DIR));
  foreach my $filename (@files) {
     my ($cdev,$cino,$cmode,$cnlink,
         $cuid,$cgid,$crdev,$csize,
         $catime,$cmtime,$cctime,
         $cblksize,$cblocks)=stat($fn.'/'.$filename);

     my $extension='';
     if ($filename=~/\.(\w+)$/) { $extension=$1; }
     if ($cmode&$dirptr) {
	 putdirectory($r, $thisdisfn, $linkdir, $filename, $cmtime,$targetdir,\%bombs,\$numdir);
     } elsif (&Apache::loncommon::fileembstyle($extension) ne 'hdn') {
	 putresource($r, $udom, $uname, $filename, $thisdisfn, $resdir, 
		     $targetdir, $linkdir, $cmtime,\%bombs,\$numres);
     } else {
	# "hidden" extension and not a directory, so hide it away.
     }
  }
  closedir(DIR);

  $r->print('</table>'.&Apache::loncommon::end_page());
  return OK;  
}
#
#  Gets the effective URL of the request and returns it:
#    $effn = getEffectiveUrl($r);
#       $r  - The Apache Request object.
sub getEffectiveUrl {
    my $r = shift;
    my $fn;
    
    if ($env{'form.filename'}) {	# If a form filename is defined.
	$fn=$env{'form.filename'};
	#
	#   Replace the ~username of the URL with /home/username/public_html
	#   so that we don't have to worry about ~ expansion internally.
	#
	$fn=~s/^http\:\/\/[^\/]+\///;
        $fn=~s/^\///;
        $fn=~s{~($LONCAPA::username_re)}{/home/$1/public_html};
	
	#  Remove trailing / strings (?) 
	
	$fn=~s/\/[^\/]+$//;
    } else {
	#   If no form is defined, use request filename.
	$fn = $r->filename();
	my $lonDocRoot=$r->dir_config('lonDocRoot');
	if ( $fn =~ /$lonDocRoot/ ) {
	    #internal authentication, needs fixup.
	    $fn = $r->uri(); # non users do not get the full path request
                             # through SCRIPT_FILENAME
	    $fn=~s{^/~($LONCAPA::username_re)}{/home/$1/public_html};
	}
    }
    $fn=~s/\/+/\//g;
    return $fn;
}
#
#   Output the header of the page.  This includes:
#   - The HTML header 
#   - The H1/H3  stuff which includes the directory.
#
#     startpage($r, $uame, $udom, $thisdisfn);
#      $r     - The apache request object.
#      $uname - User name.
#      $udom  - Domain name the user is logged in under.
#      $thisdisfn - Displayable version of the filename.

sub startpage {
    my ($r, $uname, $udom, $thisdisfn) = @_;
    my $currdir = '/priv/'.$uname.$thisdisfn;
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;

    my $formaction='/priv/'.$uname.$thisdisfn.'/';
    $formaction=~s|/+|/|g;
    my $pagetitle .= &Apache::loncommon::help_open_menu('','',3,'Authoring').
        '<font face="Arial, Helvetica, sans-serif" size="+1"><b>'.&mt('Construction Space').'</b>:</font>&nbsp;'.
        '<form name="dirs" method="post" action="'.$formaction.
        '" target="_parent"><tt><b>'.
        &Apache::lonhtmlcommon::crumbs($uname.$thisdisfn.'/','_top','/priv','','+1',1)."</b></tt><br />".
        &Apache::lonhtmlcommon::select_recent('construct','recent',
                 'this.form.action=this.form.recent.value;this.form.submit()').
              '</form>';
    &Apache::lonhtmlcommon::store_recent('construct',$formaction,$formaction);
    if ($env{'environment.remote'} eq 'off') {
	$env{'request.noversionuri'}=$currdir.'/';
	$r->print(&Apache::loncommon::start_page('Construction Space',undef,
						 {'body_title' =>
						      $pagetitle,}));
    } else {
	$r->print(&Apache::loncommon::start_page('Construction Space',undef,
						 { 'only_body' => 1,}));
	$r->print($pagetitle);
    }

    my $esc_thisdisfn = &Apache::loncommon::escape_single($thisdisfn);
    my $doctitle = 'LON-CAPA '.&mt('Construction Space');
    my $newname = &mt('New Name');
    my $pubdirscript=(<<ENDPUBDIRSCRIPT);
<script type="text/javascript">
top.document.title = '$esc_thisdisfn/ - $doctitle';
// Store directory location for menu bar to find

parent.lastknownpriv='/~$uname$esc_thisdisfn/';

// Confirmation dialogues

    function currdiract(theform) {
        if (theform.dirtask.options[theform.dirtask.selectedIndex].value == 'publish') {
            document.publishdir.filename.value = theform.filename.value;
	    document.publishdir.submit();
        }
        if (theform.dirtask.options[theform.dirtask.selectedIndex].value == 'editcat') {
            top.location=theform.filename.value+'default.meta'
        }
        if (theform.dirtask.options[theform.dirtask.selectedIndex].value == 'printdir' ) {
            document.printdir.postdata.value=theform.filename.value
            document.printdir.submit();
        }
        if (theform.dirtask.options[theform.dirtask.selectedIndex].value == "delete") {
              var delform = document.delresource
              delform.filename.value = theform.filename.value
              delform.submit()
        }
    }
  
    function checkUpload(theform) {
        if (theform.file == '') {
            alert("Please use 'Browse..' to choose a file first, before uploading")
            return 
        }
        theform.submit()  
    }

    function SetPubDir(theform,printForm) {
        if (theform.diraction.options[theform.diraction.selectedIndex].value == "open") {
            top.location = theform.openname.value
            return
        }
        if (theform.diraction.options[theform.diraction.selectedIndex].value == "publish") {
            theform.submit();
        }
        if (theform.diraction.options[theform.diraction.selectedIndex].value == "editcat") {
            top.location=theform.filename.value+'default.meta'
        }
        if (theform.diraction.options[theform.diraction.selectedIndex].value == "printdir") {
            theform.action = '/adm/printout'
            theform.postdata.value = theform.filename.value
            theform.submit()
        }
        if (theform.diraction.options[theform.diraction.selectedIndex].value == "delete") {
              var delform = document.delresource
              delform.filename.value = theform.filename.value
              delform.submit()
        }
        return
    }
    function SetResChoice(theform) {
      var activity = theform.reschoice.options[theform.reschoice.selectedIndex].value
      if ((activity == 'rename') || (activity == 'copy') || (activity == 'move')) {
          changename(theform,activity)
      }
      if (activity == 'publish') {
          var pubform = document.pubresource
          pubform.filename.value = theform.filename.value
          pubform.submit()
      }
      if (activity == 'delete') {
          var delform = document.delresource
          delform.filename.value = theform.filename.value
          delform.submit()
      }
      if (activity == 'obsolete') {
          var pubform = document.pubresource
          pubform.filename.value = theform.filename.value
          pubform.makeobsolete.value=1;
          pubform.submit()
      }
      if (activity == 'print') {
          document.printresource.postdata.value = theform.filename.value
          document.printresource.submit()
      }
      if (activity == 'retrieve') {
          document.retrieveres.filename.value = theform.filename.value
          document.retrieveres.submit()
      }
      if (activity == 'cleanup') {
          document.cleanup.filename.value = theform.filename.value
          document.cleanup.submit()
      }
      return
    }
    function changename(theform,activity) {
        var oldname=theform.dispfilename.value;
        var newname=prompt('$newname',oldname);
        if (newname == "" || !newname || newname == oldname)  {
            return
        }
        document.moveresource.newfilename.value = newname
        document.moveresource.filename.value = theform.filename.value
        document.moveresource.action.value = activity
        document.moveresource.submit();
    }
</script>
ENDPUBDIRSCRIPT
    $r->print($pubdirscript);

    if ((($uname ne $env{'user.name'}) || ($udom ne $env{'user.domain'})) &&
	$env{'environment.remote'} ne 'off') {
	$r->print('<h3>'.&mt('Co-Author').': '.$uname.' at '.$udom.
		  '</h3>');
    }
}

sub dircontrols {
    my ($r,$uname,$udom,$thisdisfn) = @_;
    my %lt=&Apache::lonlocal::texthash(
                                       cnpd => 'Cannot publish directory',
                                       cnrd => 'Cannot retrieve directory',
                                       mcdi => 'Must create new subdirectory inside a directory',
                                       pubr => 'Publish this Resource',
                                       pubd => 'Publish this Directory',
                                       dedr => 'Delete Directory',
                                       rtrv => 'Retrieve Old Version',
                                       list => 'List Directory',
                                       uplo => 'Upload file',  
                                       dele => 'Delete',
                                       edit => 'Edit Catalog Information', 
                                       sela => 'Select Action',
                                       nfil => 'New file',
                                       nhtm => 'New HTML file',
                                       nprb => 'New problem',
                                       npag => 'New assembled page',
                                       nseq => 'New assembled sequence',
                                       ncrf => 'New custom rights file',
                                       nsty => 'New style file',
                                       nlib => 'New library file',
                                       nbt  => 'New bridgetask file',
                                       nsub => 'New subdirectory',
                                       renm => 'Rename current file to',
                                       move => 'Move current file to',
                                       copy => 'Copy current file to',
                                       type => 'Type Name Here',
                                       go   => 'Go',
                                       prnt => 'Print contents of directory',
                                       crea => 'Create a new directory or LON-CAPA document',
				       acti => 'Actions for current directory',
				       updc => 'Upload a new document',
				       pick => 'Please select an action to perform using the new filename',
                                      );
    my $mytype = $lt{'type'}; # avoid conflict with " and ' in javascript
    $r->print(<<END);
        <table id="LC_cstr_controls">
         <tr>
          <th>$lt{'acti'}</th>
          <th>$lt{'updc'}</th>
          <th>$lt{'crea'}</th>
        </tr>
        <tr>
         <td>
          <form name="curractions" method="post" action="">
           <select name="dirtask" onchange="currdiract(this.form)">
            <option>$lt{'sela'}</option>
            <option value="publish">$lt{'pubd'}</option>
            <option value="editcat">$lt{'edit'}</option>
            <option value="printdir">$lt{'prnt'}</option>
            <option value="delete">$lt{'dedr'}</option>
           </select>
           <input type="hidden" name="filename" value="/~$uname$thisdisfn/" />
          </form>
          <form name="publishdir" method="post" action="/adm/publish" target="_parent">
           <input type="hidden" name="pubrec" value="" />
           <input type="hidden" name="filename" value="" />
          </form>
          <form name="printdir" method="post" action="/adm/printout" target="_parent">
           <input type="hidden" name="postdata" value="" />
          </form>
         </td>
         <td>
	    <form name="upublisher" enctype="multipart/form-data" method="post" action="/adm/upload" target="_parent">
	      <input type="hidden" name="filename" value="/~$uname$thisdisfn/" />
	      <input type="file" name="upfile" size="20" />
	      <input type="button" value="$lt{'uplo'}"  onclick="checkUpload(this.form)" />
	    </form>
	 </td>
	 <td>
	    <form name="fileaction" method="post" action="/adm/cfile" target="_parent">
	      <span style="white-space: nowrap">
		<input type="hidden" name="filename" value="/~$uname$thisdisfn/" />
                  <script type="text/javascript">
                    function validate_go() {
                        var selected = document.fileaction.action.selectedIndex;
                        if (selected == 0) {
                            alert('$lt{'pick'}');
                        } else {
                            document.fileaction.submit();
                        }
                    }
                  </script>
		  <select name="action">
		    <option value="none">$lt{'sela'}</option>
		    <option value="newfile">$lt{'nfil'}:</option>
		    <option value="newhtmlfile">$lt{'nhtm'}:</option>
		    <option value="newproblemfile">$lt{'nprb'}:</option>
                    <option value="newpagefile">$lt{'npag'}:</option>
                    <option value="newsequencefile">$lt{'nseq'}:</option>
                    <option value="newrightsfile">$lt{'ncrf'}:</option>
                    <option value="newstyfile">$lt{'nsty'}:</option>
                    <option value="newtaskfile">$lt{'nbt'}:</option>
                    <option value="newlibraryfile">$lt{'nlib'}:</option>
	            <option value="newdir">$lt{'nsub'}:</option>
		  </select>&nbsp;<input type="text" name="newfilename" value="$lt{'type'}" onfocus="if (this.value == '$mytype') this.value=''" />&nbsp;<input type="button" value="Go" onclick="validate_go();" />
		 </span>
		</form>
	  </td>
         </tr>
        </table>
END
}

sub pubbuttons {
    my ($r,$uname,$thisdisfn) = @_;
    $r->print('<form method="post" action="/adm/publish" target="_parent">'.
              '<table><tr><td><input type="hidden" name="filename" value="/~'.
               $uname.$thisdisfn.'/" />'.
              '<input type="submit" value="'.&mt('Publish Directory').'" /></td><td>'.
'<input type="button" onclick="window.location='."'/~".
               $uname.$thisdisfn."/default.meta'".'" value="'.
&mt('Edit Directory Catalog Information').'" /></td></tr></table></form>');
}

sub resourceactions {
    my ($r,$uname,$udom,$thisdisfn) = @_;
    $r->print(<<END);
       <form name="moveresource" action="/adm/cfile" target="_parent" method="post">
         <input type="hidden" name="filename" value="" />
         <input type="hidden" name="newfilename" value="" />
         <input type="hidden" name="action" value="" />
       </form>
       <form name="delresource" action="/adm/cfile" target="_parent" method="post">
         <input type="hidden" name="filename" value="" />
         <input type="hidden" name="action" value="delete" />
       </form>
       <form name="pubresource" action="/adm/publish" target="_parent" method="post">
         <input type="hidden" name="filename" value="" />
         <input type="hidden" name="makeobsolete" value="0" />
       </form>
       <form name="printresource" action="/adm/printout" target="_parent" method="post">
           <input type="hidden" name="postdata" value="" />
       </form>
       <form name="retrieveres" action="/adm/retrieve" target="_parent" method="post">
           <input type="hidden" name="filename" value="" />
       </form>
       <form name="cleanup" action="/adm/cleanup" target="_parent" method="post">
           <input type="hidden" name="filename" value="" />
       </form>
END
}

#
#   Get the title string or "[untitled]" if the file has no title metadata:
#   Without the latter substitution, it's impossible to examine metadata for
#   untitled resources.  Resources may be legitimately untitled, to prevent
#   searches from locating them.
#
#   $str = getTitleString($fullname);
#       $fullname - Fully qualified filename to check.
#
sub getTitleString {
    my $fullname = shift;
    my $title    = &Apache::lonnet::metadata($fullname, 'title');

    unless ($title) {
	$title = "[".&mt('untitled')."]";
    }
    return $title;
}

sub getCopyRightString {
    my $fullname = shift;
    return &Apache::lonnet::metadata($fullname, 'copyright');
}

sub getSourceRightString {
    my $fullname = shift;
    return &Apache::lonnet::metadata($fullname, 'sourceavail');
}
#
#  Put out a directory table row:
#    putdirectory(r, base, here, dirname, modtime)
#      r       - Apache request object.
#      reqfile - File in request.
#      here    - Where we are in directory tree.
#      dirname - Name of directory special file.
#      modtime - Encoded modification time.
# 
sub putdirectory {
    my ($r, $reqfile, $here, $dirname, $modtime, $resdir, $bombs, $numdir) = @_;
    # construct the display filename: the directory name unless ..:
    
    my $disfilename = $dirname;
    if ($dirname eq '..') {
	$disfilename = '<i>'.&mt('Parent Directory').'</i>';
    }
    unless ( (($dirname eq '..') && ($reqfile eq '')) || ($dirname eq '.')) {
	my $kaputt=0;
	foreach (keys %{$bombs}) {
	    if ($_=~m:^\Q$resdir\E/\Q$disfilename\E/:) { $kaputt=1; last; }
	}
	%Apache::lonpublisher::metadatafields=();
	%Apache::lonpublisher::metadatakeys=();
	my $construct=$here;
	$construct=~s{^/priv/($LONCAPA::username_re)$}{/home/$1/public_html};
        my $dirpath = $here;
        $dirpath=~s{^/priv/}{/~};
	&Apache::lonpublisher::metaeval(&Apache::lonnet::getfile(
       				 $construct.'/'.$dirname.'/default.meta'
								 ));
        my $actionitem = '';
        if ($dirname eq '..') {
            $actionitem = &mt('Go to ...');
        } else {
            $actionitem = 
                    '<form name="dirselect_'.$$numdir.
                    '" action="/adm/publish" target="_parent">'.
                    '<select name="diraction" onchange="SetPubDir(this.form,document)">'.
                      '<option selected="selected">'.&mt('Select action').'</option>'.
                      '<option value="open">'.&mt('Open').'</option>'.
                      '<option value="publish">'.&mt('Publish').'</option>'.
                      '<option value="editcat">'.&mt('Edit catalog information').'</option>'.
                      '<option value="printdir">'.&mt('Print directory').'</option>'.
                      '<option value="delete">'.&mt('Delete directory').'</option>'.
                    '</select>'.
                     '<input type="hidden" name="filename" value="'.&HTML::Entities::encode($dirpath.'/'.$dirname,'<>&"').'/" />'.
                     '<input type="hidden" name="openname" value="'.$here.'/'.$dirname.'/" />'.
                     '<input type="hidden" name="postdata" value="" />'.
                   '</form>';
            $$numdir ++;
        }
	$r->print('<tr class="LC_browser_folder">'.
		  '<td><img src="'.
		  $Apache::lonnet::perlvar{'lonIconsURL'}.'/folder_closed.gif" alt="folder" /></td>'.
		  '<td>'.$actionitem.'</td>'.
		  '<td><span class="LC_filename"><a href="'.&HTML::Entities::encode($here.'/'.$dirname,'<>&"').'/" target="_parent">'.
		  $disfilename.'</a></span></td>'.
		        '<td colspan="2">'.($kaputt?&Apache::lonhtmlcommon::authorbombs($resdir.'/'.$disfilename.'/'):'').$Apache::lonpublisher::metadatafields{'title'});
	if ($Apache::lonpublisher::metadatafields{'subject'} ne '') {
	    $r->print(' <i>'.
		      $Apache::lonpublisher::metadatafields{'subject'}.
		      '</i> ');
	}
	$r->print($Apache::lonpublisher::metadatafields{'keywords'}.'</td>'.
		  '<td>'.&Apache::lonlocal::locallocaltime($modtime).'</td>'.
		  "</tr>\n");
    }
    return OK;
}
#
#   Put a table row for a file resource.
#
sub putresource {
    my ($r, $udom, $uname, $filename, $thisdisfn, 
	$resdir, $targetdir, $linkdir,
	$cmtime,$bombs,$numres) = @_;
    &Apache::lonnet::devalidate_cache_new('meta',$targetdir.'/'.$filename);
    my $pubstatus = 'unpublished';
    my $status=&mt('Unpublished');
    my $css_class='LC_browser_file';
    my $title='&nbsp;';
    my $publish_button=&mt('Publish');
    my $cstr_dir = '/home/'.$uname.'/public_html/'.$thisdisfn.'/';
#    my $action_buttons=
#        '<br /><a target="_parent" href="/adm/cfile?action=delete&filename=/~'.
#	$uname.'/'.$thisdisfn.'/'.$filename.'">'.
#	&mt('Delete').'</a>';
    if (-e $resdir.'/'.$filename) {
        my $same=0;
	my ($rdev,$rino,$rmode,$rnlink,
	    $ruid,$rgid,$rrdev,$rsize,
	    $ratime,$rmtime,$rctime,
	    $rblksize,$rblocks)=stat($resdir.'/'.$filename);
        if ($rmtime>=$cmtime) {
           $same=1;
        } else {
           if (&Apache::londiff::are_different_files($resdir.'/'.$filename,
						     $cstr_dir.'/'.$filename)) {
              $same=0;
           } else {
              $same=1;
           }
        }
	my $meta_cmtime = (stat($cstr_dir.'/'.$filename.'.meta'))[9];
	my $meta_rmtime = (stat($resdir.'/'.$filename.'.meta'))[9];
	my $meta_same = 1;
	if ($meta_rmtime < $meta_cmtime
	    && &Apache::londiff::are_different_files($resdir.'/'.$filename.'.meta',
						     $cstr_dir.'/'.$filename.'.meta')) {
	    $meta_same = 0;
	}
	$publish_button=&mt('Re-publish');
	my $rights_status =
	    &mt(&getCopyRightString($targetdir.'/'.$filename)).' '.
	    &mt(&getSourceRightString($targetdir.'/'.$filename));
	$title = '<a href="/res/'.$targetdir.'/'.$filename.
	    '.meta" target="cat">'.
	    &getTitleString($targetdir.'/'.$filename).'</a>';
	if ($same) {
	    if (&Apache::lonnet::metadata($targetdir.'/'.$filename,'obsolete')) {
                $pubstatus = 'obsolete';
		$status=&mt('Obsolete');
            } else {
		if (!$meta_same) {
		    $pubstatus = 'metamodified';
		} else {
		    $pubstatus = 'published';
		}
		$status=&mt('Published').
		    '<br />'. $rights_status;
	    }
#	    } else {
#		$action_buttons='';
#	    }
	} else {
            $pubstatus = 'modified';
	    $status=&mt('Modified').
		'<br />'. $rights_status;
#	    $action_buttons='';
	    if (&Apache::loncommon::fileembstyle(($filename=~/\.(\w+)$/)) eq 'ssi') {
		$status.='<br /><a href="/adm/diff?filename=/~'.$uname.
		    $thisdisfn.'/'.$filename.
		    '&amp;versiontwo=priv" target="cat">'.&mt('Diffs').'</a>';
	    }
	} 

	$title.="\n".'<br /><a href="/~'.$uname.$thisdisfn.'/'.$filename.'.meta">'. 
	    ($$bombs{$targetdir.'/'.$filename}?'<img src="/adm/lonMisc/bomb.gif" border="0" alt="'.&mt('bomb').'" />':&mt('Edit Metadata')).'</a>';

	if (!$meta_same) {
	    $title = &mt('Metadata Modified').'<br />'.$title.
		'<br /><a href="/adm/diff?filename=/~'.$uname.
		$thisdisfn.'/'.$filename.'.meta'.
		'&amp;versiontwo=priv" target="cat">'.&mt('Metadata Diffs').'</a>';
	    $title.="\n".'<br /><a href="/adm/retrieve?filename=/~'.$uname.
		$thisdisfn.'/'.$filename.'.meta" target="_parent">'.&mt('Retrieve Metadata').'</a>';
	}
	$status.="\n".'<br /><a href="/adm/retrieve?filename=/~'.$uname.
	    $thisdisfn.'/'.$filename.'" target="_parent">'.&mt('Retrieve').'</a>';
    }
    my $editlink='';
    my $editlink2='';
    if ($filename=~/\.(xml|html|htm|xhtml|xhtm|sty)$/) {
	$editlink=' <br />(<a href="'.$linkdir.'/'.$filename.'?forceedit=1" target="_parent">'.&mt('Edit').'</a>)';
    }
    if ($filename=~/\.(problem|exam|quiz|assess|survey|form|library)$/) {
	$editlink=' (<a href="'.$linkdir.'/'.$filename.'?forceedit=1" target="_parent">'.&mt('EditXML').'</a>)';
	$editlink2=' <br />(<a href="'.$linkdir.'/'.$filename.'?forceColoredit=1" target="_parent">'.&mt('Edit').'</a>)';
    }
    if ($filename=~/\.(problem|exam|quiz|assess|survey|form|library|xml|html|htm|xhtml|xhtm|sty)$/) {
	$editlink.=' (<a href="/adm/cleanup?filename=/~'.$uname.
	    $thisdisfn.'/'.$filename.'" target="_parent">'.&mt('Clean Up').')</a>';
    }
    if ($filename=~/\.(zip|tar|bz2|gz|tar.gz|tar.bz2|tgz)$/) {
	$editlink=' (<a target="_parent" href="/adm/cfile?decompress=/~'.
	      $uname.$thisdisfn.'/'.$filename.'">'.&mt('Decompress').'</a>)';
    }
    my $pub_select = '';
    &create_pubselect($r,\$pub_select,$udom,$uname,$thisdisfn,$filename,$resdir,$pubstatus,$publish_button,$numres);
    $r->print('<tr class="LC_browser_file_'.$pubstatus.'">'.
	      '<td>'.($filename=~/[\#\~]$/?'&nbsp;':
		      '<img src="'.&Apache::loncommon::icon($filename).'" alt="" />').'</td>'.
              '<td>'.$pub_select.'</td>'.
	      '<td><span class="LC_filename">'.
	      '<a href="'.$linkdir.'/'.$filename.'" target="_parent">'.
               $filename.'</a></span>'.$editlink2.$editlink.
	      '</td>'.
	      '<td>'.$title.'</td>'.
	      '<td>'.$status.'</td>'.
	      '<td>'.&Apache::lonlocal::locallocaltime($cmtime).'</td>'.
	      "</tr>\n");
    return OK;
}

sub create_pubselect {
    my ($r,$pub_select,$udom,$uname,$thisdisfn,$filename,$resdir,$pubstatus,$publish_button,$numres) = @_;
    $$pub_select = '
<form name="resselect_'.$$numres.'" action="">
<select name="reschoice"  onchange="SetResChoice(this.form)">
<option>'.&mt('Select action').'</option>'.
'<option value="copy">'.&mt('Copy').'</option>';
    if ($pubstatus eq 'obsolete' || $pubstatus eq 'unpublished') {
        $$pub_select .= 
'<option value="rename">'.&mt('Rename').'</option>'.
'<option value="move">'.&mt('Move').'</option>'.
'<option value="delete">'.&mt('Delete').'</option>';
    } else {
        $$pub_select .= '
<option value="obsolete">'.&mt('Mark obsolete').'</option>';
    }
# check for versions
    my $versions = &check_for_versions($r,'/'.$filename,$udom,$uname);
    if ($versions > 0) {
        $$pub_select .='
<option value="retrieve">'.&mt('Retrieve old version').'</option>';
    }
    $$pub_select .= '
<option value="publish">'.$publish_button.'</option>'.
'<option value="cleanup">'.&mt('Clean up').'</option>'.
'<option value="print">'.&mt('Print').'</option>'.
'</select>
<input type="hidden" name="filename" value="/~'.
 &HTML::Entities::encode($uname.$thisdisfn.'/'.$filename,'<>&"').'" />
 <input type="hidden" name="dispfilename" value="'.
 &HTML::Entities::encode($filename).'" /></form>';
    $$numres ++;
}

sub check_for_versions {
    my ($r,$fn,$udom,$uname) = @_;
    my $versions = 0;
    my $docroot=$r->dir_config('lonDocRoot');
    my $resfn=$docroot.'/res/'.$udom.'/'.$uname.$fn;
    my $resdir=$resfn;
    $resdir=~s/\/[^\/]+$/\//;
    $fn=~/\/([^\/]+)\.(\w+)$/;
    my $main=$1;
    my $suffix=$2;
    opendir(DIR,$resdir);
    while (my $filename=readdir(DIR)) {
        if ($filename=~/^\Q$main\E\.(\d+)\.\Q$suffix\E$/) {
            $versions ++;        
        }
    }
    return $versions;
}

#
#   Categorize files in the directory.
#   For each file in a list of files in a file directory, 
#   the  file categorized as one of:
#    - directory  
#    - sequence
#    - problem 
#    - Other resource.
#
#   For each file the modification date is determined as well.
#   Returned is a list of sublists:
#    (directories, sequences, problems, other)
#   each of the sublists contains entries of the following form (sorted by
#   filename):
#     (filename, typecode, lastmodtime)
#
#   $list = CategorizeFiles($location, $files)
#       $location   - Directory in which the files live (relative to our
#                     execution.
#       $files      - list of files.
#
sub CategorizeFiles {
    my $location = shift;
    my $files    = shift;
}

1;
__END__

=head1 NAME

Apache::lonpubdir - Construction space directory lister

=head1 SYNOPSIS

Invoked (for various locations) by /etc/httpd/conf/srm.conf:

 <LocationMatch "^/\~.*/$">
 PerlAccessHandler       Apache::loncacc
 SetHandler perl-script
 PerlHandler Apache::lonpubdir
 ErrorDocument     403 /adm/login
 ErrorDocument     404 /adm/notfound.html
 ErrorDocument     406 /adm/unauthorized.html
 ErrorDocument	  500 /adm/errorhandler
 </LocationMatch>

 <Location /adm/pubdir>
 PerlAccessHandler       Apache::lonacc
 SetHandler perl-script
 PerlHandler Apache::lonpubdir
 ErrorDocument     403 /adm/login
 ErrorDocument     404 /adm/notfound.html
 ErrorDocument     406 /adm/unauthorized.html
 ErrorDocument	  500 /adm/errorhandler
 </Location>

=head1 INTRODUCTION

This module publishes a directory of files.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 HANDLER SUBROUTINE

This routine is called by Apache and mod_perl.

=over 4

=item *

read in information

=item *

start page output

=item *

run through list of files and attempt to publish unhidden files

=back

=cut
