#
# LON-CAPA authorization for pages generated by server-status reports 
#
# $Id: lonstatusacc.pm,v 1.5 2008/12/25 05:49:20 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#############################################
#############################################

package Apache::lonstatusacc;

use strict;
use Apache::Constants qw(:common :http :remotehost);
use Apache::lonnet;
use LONCAPA::loncgi;
use LONCAPA::lonauthcgi;

sub handler {
    my $r = shift;
    my $reqhost = $r->get_remote_host(REMOTE_NOLOOKUP);
    my $page = 'server-status';
    if (($r->uri eq '/adm/domainstatus') ||
        ($r->uri eq '/adm/test')) {
        if (&LONCAPA::loncgi::check_cookie_and_load_env($r)) {
            if ($r->uri eq '/adm/domainstatus') {
                return OK;
            } elsif ($r->uri eq '/adm/test') { 
                $page = 'showenv'; 
                if (&LONCAPA::lonauthcgi::can_view($page)) {
                    return OK; 
                } elsif (&LONCAPA::lonauthcgi::check_ipbased_access($page,$reqhost)) {
                    return OK;
                } else {
                    $Apache::lonnet::env{'user.error.msg'} =
                        $r->uri.":bre:1:1:Access Denied";
                    return HTTP_NOT_ACCEPTABLE;
                }
            }
        } else {
            return FORBIDDEN;
        }
    } elsif ($r->uri ne '/server-status') {
        $page = 'lonstatus';
        if (!-e $r->filename) {
            return NOT_FOUND;
        }
    }
    if ($reqhost eq '127.0.0.1') {
        return OK;
    }
    my @hostids= &Apache::lonnet::get_hosts_from_ip($reqhost);
    my @poss_domains = &Apache::lonnet::current_machine_domains();
    if (@hostids > 0) {
        foreach my $id (@hostids) {
            if ($id ne '') {
                my $dom = &Apache::lonnet::host_domain($id);
                if ($dom ne '') {
                    if (grep(/^\Q$dom\E$/,@poss_domains)) {
                        return OK;
                    }
                }
            }
        }
    } elsif (&LONCAPA::lonauthcgi::check_ipbased_access($page,$reqhost)) {
        return OK;
    } else {
        if (&LONCAPA::loncgi::check_cookie_and_load_env($r)) {
            if (&LONCAPA::lonauthcgi::can_view($page)) {
                return OK;
            }
        }
    }
    $r->log_reason("Invalid request for server status from $reqhost",
                   $r->uri);
    return FORBIDDEN;
}

1;

__END__

=head1 NAME

Apache::lonstatusacc - Access Handler for Apache's server-status page 
and also pages in lon-status directory.

=head1 SYNOPSIS

Invoked (for appropriate locations) by /etc/httpd/conf/loncapa_apache.conf

 PerlAccessHandler       Apache::lonstatusacc

=head1 INTRODUCTION

This module can support access control based on IP 
address, or based on Domain Configuration settings 
for authenticated users (via cookie).

The module is used for control of access to
(a) Apache's server-status page
(b) Status pages in the /home/httpd/html/lon-status directory
    which were generated as follows:
    (i) when loncron was last run 
        (index.html, loncron_simple.txt, loncstatus.txt, and londstatus.txt),
    (ii) when lonsql was last started 
         (mysql.txt - only on connection failure),
    (iii) when /usr/local/loncapa/bin/CHECKRPMS was last run 
         (checkrpms.txt),
    (iv) when ./UPDATE was run to install/update 
         (version.txt). 
(c) User environment information reported by /adm/test
 
This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 HANDLER SUBROUTINE

This routine is called by Apache and mod_perl.

The check for whether access is allowed for a specific page proceeds as follows:

(a) Access allowed for request from loopback address for any page.

(b) For any page except /adm/test, access allowed if at least one of the following applies:
    (a) If request is from a LON-CAPA server, if at least one domain hosted on 
        requesting machine is also a domain hosted on this server. 
    (b) IP address of requesting server is listed in domain configuration list
        of allowed machines for any of the domains hosted on this server
    (c) If requestor has an active LON-CAPA session -- checked using
        LONCAPA::loncgi::check_cookie_and_load_env() -- access allowed 
        AND one of the following is true:
        (i) Requestor has LON-CAPA superuser role
        (ii) Requestor's role is Domain Coordinator in one of the domains
             hosted on this server
        (iii) Domain configurations for domains hosted on this server include
              the requestor as one of the named users (username:domain) with access
              to the page.

(c) /adm/test
    Access requires a valid session - checked using 
    LONCAPA::loncgi::check_cookie_and_load_env(). 
    If so, access is allowed if one of the following is true:
    (i) Requestor has LON-CAPA superuser role, or
    (ii) Requestor's role is Domain Coordinator in one of the domains
         hosted on this server
    (iii) Domain configurations for domains hosted on this server include
          the requestor as one of the named users (username:domain) with access
          to the page.
    (iv) IP address of requestor is listed in domain configuration list
         of allowed machines for any of the domains hosted on this server

=cut 

