# The LearningOnline Network with CAPA 
# definition of tags that give a structure to a document
#
# $Id: bridgetask.pm,v 1.243 2008/12/05 10:23:50 bisitz Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###


package Apache::bridgetask; 

use strict;
use Apache::lonnet;
use Apache::File();
use Apache::lonmenu;
use Apache::lonlocal;
use Apache::lonxml;
use Apache::slotrequest();
use Time::HiRes qw( gettimeofday tv_interval );
use LONCAPA;
 

BEGIN {
    &Apache::lonxml::register('Apache::bridgetask',('Task','IntroParagraph','Dimension','Question','QuestionText','Setup','Instance','InstanceText','Criteria','CriteriaText','GraderNote','ClosingParagraph'));
}

my %dimension;
my $top = 'top';

sub initialize_bridgetask {
    # id of current Dimension, 0 means that no dimension is current 
    # (inside <Task> only)
    @Apache::bridgetask::dimension=();
    # list of all current Instance ids
    %Apache::bridgetask::instance=();
    # list of all Instance ids seen in this problem
    @Apache::bridgetask::instancelist=();
    # key of queud user data that we are currently grading
    $Apache::bridgetask::queue_key='';
    undef(%dimension);
}

sub proctor_check_auth {
    my ($slot_name,$slot,$type)=@_;
    my $user=$env{'form.proctorname'};
    my $domain=$env{'form.proctordomain'};
    
    my @allowed=split(",",$slot->{'proctor'});
    foreach my $possible (@allowed) {
	my ($puser,$pdom)=(split(':',$possible));
	if ($puser eq $user && $pdom eq $domain) {
	    my $authenticated=0;
	    if ( $slot->{'secret'} =~ /\S/ &&
		 $env{'form.proctorpassword'} eq $slot->{'secret'} ) {
		$authenticated=1;
	    } else {
		
		my $authhost=&Apache::lonnet::authenticate($puser,$env{'form.proctorpassword'},$pdom);
		if ($authhost ne 'no_host') {
		    $authenticated=1;
		}
	    }
	    if ($authenticated) {
		&check_in($type,$user,$domain,$slot_name);
		return 1;
	    }
	}
    }
    return 0;
}

sub check_in {
    my ($type,$user,$domain,$slot_name) = @_;
    my $useslots = &Apache::lonnet::EXT("resource.0.useslots");
    if ( $useslots eq 'map_map') {
	&check_in_sequence($user,$domain,$slot_name);
    } else {
	&create_new_version($type,$user,$domain,$slot_name);
	&Apache::structuretags::finalize_storage();
    }
    return 1;
}

sub check_in_sequence {
    my ($user,$domain,$slot_name) = @_;
    my $navmap = Apache::lonnavmaps::navmap->new();
    my ($symb) = &Apache::lonnet::whichuser();
    my ($map)  = &Apache::lonnet::decode_symb($symb);
    my @resources = 
	$navmap->retrieveResources($map, sub { $_[0]->is_problem() },0,0);
    my %old_history = %Apache::lonhomework::history;
    my %old_results = %Apache::lonhomework::results;

    foreach my $res (@resources) {
	&Apache::lonxml::debug("doing ".$res->src);
	&Apache::structuretags::initialize_storage($res->symb);
	my $type = ($res->is_task()) ? 'Task' : 'problem';
	&create_new_version($type,$user,$domain,$slot_name);
	&Apache::structuretags::finalize_storage($res->symb);
    }
    
    %Apache::lonhomework::history = %old_history;
    %Apache::lonhomework::results = %old_results;
}

sub create_new_version {
    my ($type,$user,$domain,$slot_name) = @_;
    
    my $id = '0';
    if ($type eq 'Task') {
	# increment version
	my $version=
	    $Apache::lonhomework::history{'resource.0.version'};
	$version++;
	&Apache::lonxml::debug("Making version $version");
	#clean out all current results
	foreach my $key (keys(%Apache::lonhomework::history)) {
	    if ($key=~/^resource\.0\./) {
		$Apache::lonhomework::results{$key}='';
	    }
	}
	
	#setup new version and who did it
       	$Apache::lonhomework::results{'resource.0.version'}=$version;
	$id = "$version.0";
	if (!defined($user) || !defined($domain)) {
	    $user = $env{'user.name'};
	    $domain = $env{'user.domain'};
	}
	
    } elsif ($type eq 'problem') {
	&Apache::lonxml::debug("authed $slot_name");
    }
    if (!defined($user) || !defined($domain)) {
	$user = $env{'user.name'};
	$domain = $env{'user.domain'};
    }

    $Apache::lonhomework::results{"resource.$id.checkedin"}=
	$user.':'.$domain;

    if (defined($slot_name)) {
	$Apache::lonhomework::results{"resource.$id.checkedin.slot"}=
	    $slot_name;
    }
}

sub get_version {
    my ($version,$previous);
    if ($env{'form.previousversion'} && 
	$env{'form.previousversion'} ne 'current' &&
	defined($Apache::lonhomework::history{'resource.'.$env{'form.previousversion'}.'.0.status'})) {
	$version=$env{'form.previousversion'};
	$previous=1;
    } else {
	if (defined($Apache::lonhomework::results{'resource.0.version'})) {
	    $version=$Apache::lonhomework::results{'resource.0.version'};
	} elsif (defined($Apache::lonhomework::history{'resource.0.version'})) {
	    $version=$Apache::lonhomework::history{'resource.0.version'};
	}
	$previous=0;
    }
    if (wantarray) {
	return ($version,$previous);
    }
    return $version;
}

sub add_previous_version_button {
    my ($status)=@_;
    my $result;
    if ($Apache::lonhomework::history{'resource.0.version'} eq '') {
	return '';
    }
    if ($Apache::lonhomework::history{'resource.0.version'} < 2 &&
	$status ne 'NEEDS_CHECKIN') {
	return '';
    }
    my $version=&get_version();
    if ($env{'form.previousversion'} ne '' &&
	$env{'form.previousversion'} eq $version) {
	$result.="<h3>".&mt("Showing previous version [_1]",$version).
	    "</h3>\n";
    }
    my @to_show;
    foreach my $test_version (1..$Apache::lonhomework::history{'resource.0.version'}) {
	if (defined($Apache::lonhomework::history{'resource.'.$test_version.'.0.status'})) {
	    push(@to_show,$test_version);
	}
    }
    my $list='<option>'.
	join("</option>\n<option>",@to_show).
	     "</option>\n";
    $list.='<option value="current">'.&mt('Current').'</option>';
    $result.='<form name="getprevious" method="post" action="';
    my $uri=$env{'request.uri'};
    if ($env{'request.enc'}) { $uri=&Apache::lonenc::encrypted($uri); }
    $result.=$uri.'">'.
	&mt(' Show a previously done version: [_1]','<select onchange="this.form.submit()" name="previousversion">
<option>'.&mt('Pick one').'</option>
'.$list.'
</select>')."</form>";
    return $result;
}

sub add_grading_button {
    my (undef,$cid)=&Apache::lonnet::whichuser();
    my $cnum=$env{'course.'.$cid.'.num'};
    my $cdom=$env{'course.'.$cid.'.domain'};
    my %sections = &Apache::loncommon::get_sections($cdom,$cnum);

    my $size=5;
    if (scalar(keys(%sections)) < 3) {
	$size=scalar(keys(%sections))+2;
    }
    my $sec_select = "\n".'<select multiple="multiple" name="chosensections" size="'.$size.'">'."\n";
    $sec_select .= "\t<option value='all' selected='selected'>all</option>\n";
    foreach my $sec (sort {lc($a) cmp lc($b)} (keys(%sections))) {
	$sec_select .= "\t<option value=\"$sec\">$sec</option>\n";
    }
    $sec_select .= "\t<option value='none'>none</option>\n</select>\n";
    
    my $result="\n\t".'<input type="submit" name="gradeasubmission" value="'.
	&mt("Get a submission to grade").'" />';
    $result.="\n\t".'<input type="hidden" name="grade_target" value="webgrade" />';
    my $see_all = &Apache::lonnet::allowed('mgq',$env{'request.course.id'});
    my $see_sec = &Apache::lonnet::allowed('mgq',$env{'request.course.id'}.
					   '/'.$env{'request.course.sec'});

    if ($see_all || $see_sec) {
	my ($entries,$ready,$locks)=&get_queue_counts('gradingqueue');
	$result.="\n\t".'<table>'."\n\t\t".'<tr>';
	if ($see_all || (!&section_restricted())) {
	    $result.="\n\t\t\t".'<td rowspan="4">'.&mt('Specify a section:').' </td>'.
		"\n\t\t\t".'<td rowspan="4">'.$sec_select."\n\t\t\t".'</td>';
	} else {
	    $result.="\n\t\t\t".'<td rowspan="4">'.&mt('Grading section:').' </td>'.
		"\n\t\t\t".'<td rowspan="4">'.$env{'request.course.sec'}."\n\t\t\t".'</td>';
	}
	$result.="\n\t\t\t".'<td>'.'<input type="submit" name="reviewagrading" value="'.
	    &mt("Select an entry from the grading queue:").'" /> ';

	$result.= "\n\t\t\t\t".&mt("[_1] entries, [_2] ready, [_3] being graded",$entries,$ready,$locks).'</td>'."\n\t\t".'</tr>'."\n";

	($entries,$ready,$locks)=&get_queue_counts('reviewqueue');
	$result.="\n\t\t".'<tr>'.
	    "\n\t\t\t".'<td>'.
	    "\n\t\t\t\t".'<input type="submit" name="reviewasubmission" value="'.
	    &mt("Select an entry from the review queue:").'" /> ';
	$result.=&mt("[_1] entries, [_2] ready, [_3] being graded",
		     $entries,$ready,$locks).'</td>'."\n\t\t".'</tr>'."\n";
	$result.="\n\t\t".'<tr>'.
	    "\n\t\t\t".'<td>'.
	    "\n\t\t\t\t".'<input type="submit" name="regradeasubmission" value="'.
	    &mt("List of user's grade status").'" /> </td>'
	    ."\n\t\t".'</tr>'
	    ."\n\t".'</table>'."\n";
	$result.="\n\t".'<p>'.
	    "\n\t\t".'<input type="submit" name="regradeaspecificsubmission" value="'.
	    &mt("Regrade specific user:").'" />';
	$result.= "\n\t\t".'<input type="text" size="12" name="gradinguser" />';
	$result.=&Apache::loncommon::select_dom_form($env{'user.domain'},
						     'gradingdomain');
	$result.=' '.
	    &Apache::loncommon::selectstudent_link('gradesubmission',
						   'gradinguser',
						   'gradingdomain');
	$result.=&Apache::loncommon::studentbrowser_javascript();
	$result.= '</p>'."\n";
    }
    return $result;
}

sub add_request_another_attempt_button {
    my ($text)=@_;
    if (!$text) { $text=&mt('Request another attempt'); }
    my $result;
    my $symb=&Apache::lonnet::symbread();
    # not a slot access based resource
    my $useslots = &Apache::lonnet::EXT("resource.0.useslots",$symb);
    if ($useslots =~ /^\s*no\s*$/i) {
	return '';
    }

    my ($slot_name,$slot)=&Apache::slotrequest::check_for_reservation($symb);
    my $action='get_reservation';
    if ($slot_name) {
	$text=&mt('Change reservation.');
	$action='change_reservation';
	my $description=&Apache::slotrequest::get_description($slot_name,
							      $slot);
	$result.='<p>'
                .&mt('Will be next available:')
                .' '.$description
                .'</p>';
    }
    
    if ($env{'request.enc'}) { $symb=&Apache::lonenc::encrypted($symb); }
    $symb=&escape($symb);
    $result.=
        "\n\t".'<form method="post" action="/adm/slotrequest">'."\n\t\t".
	'<input type="hidden" name="symb" value="'.$symb.'" />'."\n\t\t".
	'<input type="hidden" name="command" value="'.$action.'" />'."\n\t\t".
	'<input type="submit" name="requestattempt" value="'.
	$text.'" />'."\n\t".
	'</form>'."\n";
    return $result;
}

sub preserve_grade_info {
    my $result;
    # if we are viewing someone else preserve that info
    if (defined $env{'form.grade_symb'}) {
	foreach my $field ('symb','courseid','domain','username') {
	    $result .= '<input type="hidden" name="grade_'.$field.
		'" value="'.$env{"form.grade_$field"}.'" />'."\n";
	}
    }
    return $result;
}

sub style {
    my ($target) = @_;
    if ($target eq 'web'
	|| $target eq 'webgrade') {
	my $style = (<<STYLE);
<link rel="stylesheet" type="text/css" href="/res/adm/includes/task.css" />
STYLE
        if ($env{'browser.type'} eq 'explorer'
	    && $env{'browser.os'} eq 'win' ) {
	    if ($env{'browser.version'} < 7) {
		$style .= (<<STYLE);
<link rel="stylesheet" type="text/css" href="/res/adm/includes/task_ie.css" />
STYLE
            } else {
		$style .= (<<STYLE);
<link rel="stylesheet" type="text/css" href="/res/adm/includes/task_ie7.css" />
STYLE
	    }
	}
	return $style;
    }
    return;
}

sub show_task {
    my ($status,$previous)=@_;
    if (!$previous && (
		       ( $status eq 'CLOSED' ) ||
		       ( $status eq 'BANNED') ||
		       ( $status eq 'UNAVAILABLE') ||
		       ( $status eq 'NOT_IN_A_SLOT') ||
		       ( $status eq 'NEEDS_CHECKIN') ||
		       ( $status eq 'WAITING_FOR_GRADE') ||
		       ( $status eq 'INVALID_ACCESS') ||
		       ( &get_version() eq ''))) {
	return 0;
    }
    if ($env{'form.donescreen'}) { return 0; }
    return 1;
}

my @delay;
sub nest { 
    if (@delay) {
	return $delay[-1];
    } else {
	return;
    }
}

sub start_delay {
    push(@delay,1);
}
sub end_delay {
    pop(@delay);
}

sub nested_parse {
    my ($str,$env,$args) = @_;
    my @old_env = @Apache::scripttag::parser_env;
    @Apache::scripttag::parser_env = @$env;
    if (exists($args->{'set_dim_id'})) {
	&enable_dimension_parsing($args->{'set_dim_id'});
    }
    push(@delay,(($args->{'delayed_dim_results'})? 1 : 0));
    my $result = &Apache::scripttag::xmlparse($$str);
    pop(@delay);
    if (exists($args->{'set_dim_id'})) {
	&disable_dimension_parsing();
    }
    @Apache::scripttag::parser_env = @old_env;
    if ($args->{'delayed_dim_results'}) {
	my $dim = &get_dim_id();
	&Apache::lonxml::debug(" tossing out $result ");
	&Apache::lonxml::debug(" usining out $dim 's  ". $dimension{$dim}{'result'});
	return $dimension{$dim}{'result'};
    }
    return $result;
}

sub internal_location {
    my ($id)=@_;
    return '<!-- LONCAPA_INTERNAL_ADD_TASK_STATUS'.$id.' -->';
}

sub submission_time_stamp {
    my ($symb,$courseid,$udom,$uname)=&Apache::lonnet::whichuser();
    my $submissiontime;
    my $version=$Apache::lonhomework::history{'resource.0.version'};
    for (my $v=$Apache::lonhomework::history{'version'};$v>0;$v--) {
	if (defined($Apache::lonhomework::history{$v.':resource.'.$version.'.0.bridgetask.portfiles'})
	    && defined($Apache::lonhomework::history{$v.':resource.'.$version.'.0.tries'})) {
	    $submissiontime=$Apache::lonhomework::history{$v.':timestamp'};
	    last;
	}
    }
    my $result;
    if ($submissiontime) {
	my $slot_name=$Apache::lonhomework::history{'resource.'.$version.'.0.checkedin.slot'};
	my %slot=&Apache::lonnet::get_slot($slot_name);
	my $diff = $slot{'endtime'} - $submissiontime;
	my ($color,$when)=('#FF6666','after');
	if ($diff > 0) { ($color,$when)=('#336600','before'); }
	my $info;
	$diff = abs($diff);
	if ($diff%60) { $info=($diff%60).' seconds'; }
	$diff=int($diff/60);
	if ($diff%60) { $info=($diff%60).' minutes '.$info; }
	$diff=int($diff/60);
	if ($diff) {    $info=$diff.' hours '.$info; }
	$result='<p><font color="'.$color.'">'.
	    &mt('Student submitted [_1] [_2] the deadline. '.
		'(Submission was at [_3], end of period was [_4].)',
		$info,$when,
		&Apache::lonlocal::locallocaltime($submissiontime),
		&Apache::lonlocal::locallocaltime($slot{'endtime'})).
		'</font></p>';
    }
    return $result;
}

sub file_list {
    my ($files,$uname,$udom) = @_;
    if (!defined($uname) || !defined($udom)) {
	(undef,undef,$udom,$uname) = &Apache::lonnet::whichuser();
    }
    my $file_url = '/uploaded/'.$udom.'/'.$uname.'/portfolio/';

    my $file_list="<ul class=\"LC_GRADING_handininfo\">\n";
    foreach my $partial_file (split(',',$files)) {
	my $file=$file_url.$partial_file;
	$file=~s|/+|/|g;
	&Apache::lonnet::allowuploaded('/adm/bridgetask',$file);
	$file_list.='<li><span class="LC_nobreak"><a href="'.$file.'?rawmode=1" target="lonGRDs"><img src="'.
	    &Apache::loncommon::icon($file).'" alt="file icon" border="0" /> '.$file.
	    '</a></span></li>'."\n";
    }
    $file_list.="</ul>\n";
    return $file_list;
}

sub grade_mode {
    if ($env{'form.regrade'} || $env{'form.regradeaspecificsubmission'}) {
	return 'regrade';
    }
    return 'queue_grade';
}

sub webgrade_standard_info {
    my ($version)=&get_version();

    my $file_list = &file_list($Apache::lonhomework::history{"resource.$version.0.bridgetask.portfiles"});

    my %lt=('done'   => 'Next Item',
	    'stop'   => 'Quit Grading',
	    'fail'   => 'Fail Rest',
	    'cancel' => 'Cancel',
	    'submit' => 'Submit Grades',
	    );
    %lt=&Apache::lonlocal::texthash(%lt);

    my $result=<<INFO;
  <div class="LC_GRADING_maincontrols">
INFO

    if ($env{'request.state'} eq 'construct') {
	$result.=<<INFO;
    <input type="submit" name="next" value="$lt{'submit'}" />
INFO
    } else {
	if (&grade_mode() eq 'regrade' && $env{'request.state'} ne 'construct') {
	    $result.=<<INFO;
    <input type="submit" name="cancel" value="$lt{'cancel'}" />
INFO
        }

	$result.=<<INFO;
    <input type="submit" name="next" value="$lt{'done'}" />
    <input type="submit" name="stop" value="$lt{'stop'}" />
INFO
    }
    $result.=<<INFO;
    <input type="button" name="fail" value="$lt{'fail'}" 
           onclick="javascript:onFailRest()" />
  </div>
  $file_list
INFO
    return $result;

}

sub done_screen {
    my ($version) = @_;
    my $title=&Apache::lonnet::gettitle($env{'request.uri'});
    my @files=split(',',$Apache::lonhomework::history{'resource.'.$version.'.0.bridgetask.portfiles'});
    my (undef,undef,$domain,$user)= &Apache::lonnet::whichuser();
    my $files = '<ul>';
    my $msg;
    foreach my $file (@files) {
	my $url="/uploaded/$domain/$user/portfolio$file";
	if (! &Apache::lonnet::stat_file($url)) {
	    $file = '<span class="LC_error">'
                   .&mt('[_1]Nonexistent file:[_2]'
                       ,'<span class="LC_error"> '
                       ,'</span> <span class="LC_filename">'.$file.'</span>');
	    $msg .= "<p>".&mt('Submitted non-existent file [_1]',$file)."</p>\n";
	} else {
	    $file = '<span class="LC_filename">'.$file.'</span>';
	    $msg .= "<p>".&mt('Submitted file [_1]',$file)."</p>\n";
	}
	$files .= '<li>'.$file.'</li>';
    }
    $files.='</ul>';
    my $subject = &mt('Submission message for [_1]',$title);
    my ($message_status,$comment_status);
    my $setting = $env{'course.'.$env{'request.course.id'}.'.task_messages'};
    $setting =~ s/^\s*(\S*)\s*$/$1/;
    $setting = lc($setting);
    if ($setting eq 'only_student'
	|| $setting eq 'student_and_user_notes_screen') {
	$message_status =
	    &Apache::lonmsg::user_normal_msg($user,$domain,$subject,$msg);
	$message_status = '<p>'.&mt('Message sent to user: [_1]',
				    $message_status).' </p>';
    }
    if ($setting eq 'student_and_user_notes_screen') {
	$comment_status = 
	    &Apache::lonmsg::store_instructor_comment($subject.'<br />'.
						      $msg,$user,$domain);
	$comment_status = '<p>'.&mt('Message sent to instructor: [_1]',
				    $comment_status).' </p>';
    }
    return "<h2>$title</h2>"
          .'<p>'.&mt('Files submitted: [_1]',$files).'</p>'
          .'<p>'.&mt('You are now done with this Bridge Task').'</p>'
          .'<hr />'
          .'<p><a href="/adm/logout">'.&mt('Logout').'</a></p>'
.'<p><a href="/adm/roles">'.&mt('Change to a different course').'</a></p>'
.$message_status
.$comment_status;

}

sub start_Task {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;

    my ($status,$accessmsg,$slot);
    &Apache::structuretags::init_problem_globals('Task');
    if ($target ne 'webgrade') {
	&Apache::structuretags::initialize_storage();
	&Apache::lonhomework::showhash(%Apache::lonhomework::history);
	if ($env{'request.state'} eq 'construct') {
	    &Apache::structuretags::setup_rndseed($safeeval);
	}
    } 

    $Apache::lonhomework::parsing_a_task=1;

    my $name;
    if ($target eq 'web' || $target eq 'webgrade') {
	$name = &Apache::structuretags::get_resource_name($parstack,$safeeval);
    }

    my ($result,$form_tag_start);
    if ($target eq 'web' || $target eq 'webgrade' || $target eq 'tex'
	|| $target eq 'edit') {
	($result,$form_tag_start) =
	    &Apache::structuretags::page_start($target,$token,$tagstack,
					       $parstack,$parser,$safeeval,
					       $name,&style($target));
    }

    if ($target eq 'web' && $env{'request.state'} ne 'construct') {
	if ($Apache::lonhomework::queuegrade
	    || $Apache::lonhomework::modifygrades) {
	    $result.='<form name="gradesubmission" method="post" action="';
	    my $uri=$env{'request.uri'};
	    if ($env{'request.enc'}) { $uri=&Apache::lonenc::encrypted($uri); }
	    $result.=$uri.'">'.&add_grading_button()."</form>\n";
	    my $symb=&Apache::lonnet::symbread();
	    if (&Apache::lonnet::allowed('mgq',$env{'request.course.id'})
		|| &Apache::lonnet::allowed('mgq',$env{'request.course.id'}.'/'.$env{'request.course.sec'})) {
		$result.='<form method="post" name="slotrequest" action="/adm/slotrequest">'.
		    '<input type="hidden" name="symb" value="'.$symb.'" />'.
		    '<input type="hidden" name="command" value="showslots" />'.
		    '<input type="submit" name="requestattempt" value="'.
		    &mt('Show Slot list').'" />'.
		    '</form>';
		my $target_id = 
		    &Apache::lonstathelpers::make_target_id({symb => $symb,
							     part => '0'});
		if (!&section_restricted()) {
		    $result.='<form method="post" name="gradingstatus" action="/adm/statistics">'.
			'<input type="hidden" name="problemchoice" value="'.$target_id.'" />'.
			'<input type="hidden" name="reportSelected" value="grading_analysis" />'.
			'<input type="submit" name="grading" value="'.
			&mt('Show Grading Status').'" />'.
			'</form>';
		}
	    }
	}
    }
    if ($target =~/(web|webgrade)/ && $env{'request.state'} eq 'construct') {
	$form_tag_start.=&Apache::structuretags::problem_web_to_edit_header($env{'form.rndseed'});
    }
    if ($target eq 'web' 
	|| ($target eq 'grade' && !$env{'form.webgrade'}) 
	|| $target eq 'answer' 
	|| $target eq 'tex') {
	my ($version,$previous)=&get_version();
	($status,$accessmsg,my $slot_name,$slot) = 
	    &Apache::lonhomework::check_slot_access('0','Task');
	if ($status eq 'CAN_ANSWER' && $version eq '') {
	    # CAN_ANSWER mode, and no current version, unproctored access
	    # thus self-checkedin
	    &check_in('Task',undef,undef,$slot_name);
	    &add_to_queue('gradingqueue',{'type' => 'Task',
					  'time' => time,
					  'slot' => $slot_name});
	    ($version,$previous)=&get_version();
	}
	
	my $status_id = 
	    ($previous || $status eq 'SHOW_ANSWER') ? 'LC_task_feedback'
	                                            : 'LC_task_take';
	$result .= '<div class="LC_task" id="'.$status_id.'">'."\n";

	push(@Apache::inputtags::status,$status);
	$Apache::inputtags::slot_name=$slot_name;
	my $expression='$external::datestatus="'.$status.'";';
	$expression.='$external::gradestatus="'.$Apache::lonhomework::history{"resource.$version.0.solved"}.'";';
	&Apache::run::run($expression,$safeeval);
	&Apache::lonxml::debug("Got $status");
	$result.=&add_previous_version_button($status);
	if (!&show_task($status,$previous)) {
	    my $bodytext=&Apache::lonxml::get_all_text("/task",$parser,$style);
	    if ( $target eq "web" ) {
		if ($env{'request.state'} eq 'construct') {
		    $result.=$form_tag_start;
		}
		my $msg;
		if ($status eq 'UNAVAILABLE') {
		    $msg.='<h1>'.&mt('Unable to determine if this resource is open due to network problems. Please try again later.').'</h1>';
		} elsif ($status eq 'NOT_IN_A_SLOT') {
		    $msg.='<h1>'.&mt('You are not currently signed up to work at this time and/or place.').'</h1>';
		    $msg.=&add_request_another_attempt_button("Sign up for time to work.");
		} elsif ($status eq 'NEEDS_CHECKIN') {
		    $msg.='<h1>'.&mt('You need the Proctor to validate you.').
			'</h1>'.&proctor_validation_screen($slot);
		} elsif ($status eq 'WAITING_FOR_GRADE') {
		    $msg.='<h1>'.&mt('Your submission is in the grading queue.').'</h1>';
		} elsif ($env{'form.donescreen'}) {
		    $result .= &done_screen($version);
		} elsif ($status ne 'NOT_YET_VIEWED') {
		    $msg.='<h1>'.&mt('Not open to be viewed').'</h1>';
		}
		if ($status eq 'CLOSED' || $status eq 'INVALID_ACCESS') {
		    $msg.='The problem '.$accessmsg;
		}
		$result.=$msg.'<br />';
	    } elsif ($target eq 'tex') {
		$result.='\begin{document}\noindent \vskip 1 mm  \begin{minipage}{\textwidth}\vskip 0 mm';
		if ($status eq 'UNAVAILABLE') {
		    $result.=&mt('Unable to determine if this resource is open due to network problems. Please try again later.').'\vskip 0 mm ';
		} else {
		    $result.=&mt('Problem is not open to be viewed. It')." $accessmsg \\vskip 0 mm ";
		}
	    } elsif ($target eq 'grade' && !$env{'form.webgrade'}) {
		if ($status eq 'NEEDS_CHECKIN') {
		    if(&proctor_check_auth($slot_name,$slot,'Task')
		       && defined($Apache::inputtags::slot_name)) {
			my $result=
			    &add_to_queue('gradingqueue',
					  {'type' => 'Task',
					   'time' => time,
					   'slot' => 
					       $Apache::inputtags::slot_name});
			&Apache::lonxml::debug("add_to_queue said $result");
		    }
		}
	    }
	} elsif ($target eq 'web') {

	    $result.=&preserve_grade_info();
	    $result.=&internal_location(); 
	    $result.=$form_tag_start."\t".
		'<input type="hidden" name="submitted" value="yes" />';
	    &Apache::lonxml::startredirection();
	}
    } elsif ( ($target eq 'grade' && $env{'form.webgrade'}) ||
	      $target eq 'webgrade') {
	my $webgrade='yes';
	if ($target eq 'webgrade') {
	    $result .= '<div class="LC_task">'."\n";
	    $result.= "\n".'<div class="LC_GRADING_task">'."\n".
		'<script type="text/javascript" 
                         src="/res/adm/includes/task_grading.js"></script>';
	    #$result.='<br />Review'.&show_queue('reviewqueue');
	    #$result.='<br />Grade'.&show_queue('gradingqueue');
	}

	my ($todo,$status_code,$msg)=&get_key_todo($target);

	if ($todo) {
	    &setup_env_for_other_user($todo,$safeeval);
	    my ($symb,$uname,$udom)=&decode_queue_key($todo);
	    if ($env{'request.state'} eq 'construct') {
		$symb = $env{'request.uri'};
	    }
	    $result.="\n".'<p>'.
		&mt('Grading [_1] for [_2] at [_3]',
		    &Apache::lonnet::gettitle($symb),$uname,$udom).'</p>';
	    $form_tag_start.=
		'<input type="hidden" name="gradingkey" value="'.
		&escape($todo).'" />';
	    $Apache::bridgetask::queue_key=$todo;
	    &Apache::structuretags::initialize_storage();
	    &Apache::lonhomework::showhash(%Apache::lonhomework::history);
	    if ($target eq 'webgrade' && $status_code eq 'selected') {
		$form_tag_start.=
		    '<input type="hidden" name="queuemode" value="selected" />';
	    }
	} else {
	    if ($target eq 'webgrade') {
		$result.="\n";
		my $back='<p><a href="/adm/flip?postdata=return:">'.
		    &mt('Return to resource').'</a></p>';
		if      ($status_code eq 'stop') {
		    $result.='<b>'.&mt("Stopped grading.").'</b>'.$back;
		} elsif ($status_code eq 'cancel') {
		    $result.='<b>'.&mt("Cancelled grading.").'</b>'.$back;
		} elsif ($status_code eq 'never_versioned') {
		    $result.='<b>'.
			&mt("Requested user has never accessed the task.").
			'</b>'.$back;
		} elsif ($status_code =~ /still_open:(.*)/) {
		    my $date = &Apache::lonlocal::locallocaltime($1);
		    $result.='<b>'.
			&mt("Task is still open, will close at [_1].",$date).
			'</b>'.$back;
		} elsif ($status_code eq 'lock_failed') {
		    $result.='<b>'.&mt("Failed to lock the requested record.")
			.'</b>'.$back;
		} elsif ($status_code eq 'unlock') {
		    $result.='<b>'.&mt("Unlocked the requested record.")
			.'</b>'.$back;
		    $result.=&show_queue($env{'form.queue'},1);
		} elsif ($status_code eq 'show_list') {
		    $result.=&show_queue($env{'form.queue'},1);
		} elsif ($status_code eq 'select_user') {
		    $result.=&select_user();
		} elsif ($status_code eq 'unable') {
		    $result.='<b>'.&mt("Unable to aqcuire a user to grade.").'</b>'.$back;
		} elsif ($status_code eq 'not_allowed') {
		    $result.='<b>'.&mt('Not allowed to grade the requested user.').' '.$msg.'</b>'.$back;
		} else {
		    $result.='<b>'.&mt("No user to be graded.").'</b>'.$back;
		}
	    }
	    $webgrade='no';
	}
	if (!$todo || $env{'form.cancel'}) {
	    my $bodytext=&Apache::lonxml::get_all_text("/task",$parser,$style);
	}
	if ($target eq 'webgrade' && defined($env{'form.queue'})) {
	    if ($webgrade eq 'yes') {
		$result.=&submission_time_stamp();
	    }
	    $result.=$form_tag_start;
	    $result.='<input type="hidden" name="webgrade" value="'.
		$webgrade.'" />';
	    $result.='<input type="hidden" name="queue" value="'.
		$env{'form.queue'}.'" />';
	    if ($env{'form.regrade'}) {
		$result.='<input type="hidden" name="regrade" value="'.
		    $env{'form.regrade'}.'" />';
	    }
	    if ($env{'form.chosensections'} || &section_restricted()) {
		my @chosen_sections = &get_allowed_sections();
		foreach my $sec (@chosen_sections) {
		    $result.='<input type="hidden" name="chosensections" 
                               value="'.$sec.'" />';
		}
	    }
	    if ($webgrade eq 'yes') { $result.=&webgrade_standard_info(); }
	} elsif ($target eq 'webgrade' 
		 && $env{'request.state'} eq 'construct') {
	    $result.=$form_tag_start;
	    $result.='<input type="hidden" name="webgrade" value="'.
		$webgrade.'" />';
	    $result.=&webgrade_standard_info();
	}
	if ($target eq 'webgrade') {
	    $result.="\n".'<div id="LC_GRADING_criterialist">';
	    &Apache::lonxml::startredirection();
	    &start_delay();
	    $dimension{$top}{'result'}=$result;
	    undef($result);
	}
    } elsif ($target eq 'edit') {
	$result.=$form_tag_start.
	    &Apache::structuretags::problem_edit_header();
	$Apache::lonxml::warnings_error_header=
	    &mt("Editor Errors - these errors might not effect the running of the problem, but they will likely cause problems with further use of the Edit mode. Please use the EditXML mode to fix these errors.")."<br />";
	$result.= &Apache::edit::text_arg('Required number of passed optional elements to pass the Task:','OptionalRequired',$token,10)." <br />\n";
	$result.= &Apache::edit::insertlist($target,$token);
    } elsif ($target eq 'modified') {
	my $constructtag=
	    &Apache::edit::get_new_args($token,$parstack,$safeeval,
					'OptionalRequired');
	if ($constructtag) {
	    $result = &Apache::edit::rebuild_tag($token);
	}
    } else {
	# page_start returned a starting result, delete it if we don't need it
	$result = '';
    }
    return $result;
}

sub get_task_end_time {
    my ($queue_entry,$symb,$udom,$uname) = @_;

    my $end_time;
    if (my $slot = &slotted_access($queue_entry)) {
	my %slot_data=&Apache::lonnet::get_slot($slot);
	$end_time = $slot_data{'endtime'};
    } else {
	$end_time = &Apache::lonhomework::due_date('0',$symb,
						   $udom,$uname);
    }
    return $end_time;
}

sub get_key_todo {
    my ($target)=@_;
    my $todo;

    if ($env{'request.state'} eq 'construct') {
	my ($symb,$cid,$udom,$uname) = &Apache::lonnet::whichuser();
	my $gradingkey=&encode_queue_key($symb,$udom,$uname);
	return ($gradingkey);
    }

    if (defined($env{'form.reviewasubmission'})) {
	&Apache::lonxml::debug("review a submission....");
	$env{'form.queue'}='reviewqueue';
	return (undef,'show_list');
    }

    if (defined($env{'form.reviewagrading'})) {
	&Apache::lonxml::debug("review a grading....");
	$env{'form.queue'}='gradingqueue';
	return (undef,'show_list');
    }

    if (defined($env{'form.regradeasubmission'})) {
	&Apache::lonxml::debug("regrade a grading....");
	$env{'form.queue'}='none';
	return (undef,'select_user');
    }


    my $me=$env{'user.name'}.':'.$env{'user.domain'};

    #need to try both queues..
    if (defined($env{'form.regradeaspecificsubmission'}) &&
	defined($env{'form.gradinguser'})               &&
	defined($env{'form.gradingdomain'})               ) {
	my ($symb,$cid)=&Apache::lonnet::whichuser();
	my $cnum  = $env{'course.'.$cid.'.num'};
	my $cdom  = $env{'course.'.$cid.'.domain'};
	my $uname = &LONCAPA::clean_username($env{'form.gradinguser'});
	my $udom  = &LONCAPA::clean_domain($env{'form.gradingdomain'});
	
	if (&section_restricted()) {
	    my $classlist=&get_limited_classlist();
	    if (!&allow_grade_user($classlist->{$uname.':'.$udom})) {
		return (undef,'not_allowed',
			&mt('Requested student ([_1]) is in a section you aren\'t allowed to grade.',$uname.':'.$udom));
	    }
	}
	my $gradingkey=&encode_queue_key($symb,$udom,$uname);

	my $queue;

	if      (&in_queue('gradingqueue',$symb,$cdom,$cnum,$udom,$uname)) {
	    $env{'form.queue'} = $queue = 'gradingqueue';
	} elsif (&in_queue('reviewqueue' ,$symb,$cdom,$cnum,$udom,$uname)) {
	    $env{'form.queue'} = $queue = 'reviewqueue';
	}
	
	if (!$queue) {
	    $env{'form.queue'} = $queue = 'none';
	    #not queued so doing either a re or pre grade
	    my %status = &Apache::lonnet::restore($symb,$cid,$udom,$uname);
	    if ($status{'resource.0.version'} < 1) {
		return (undef,'never_versioned');
	    }
	    return ($gradingkey);
	}

	if ($queue) {
	    my $queue_entry = &get_queue_data($queue,$udom,$uname);
	
	    my $end_time = &get_task_end_time($queue_entry,$symb,
					      $udom,$uname);
	    if ($end_time > time) {
		return (undef,"still_open:$end_time");
	    }
	}

	my $who=&queue_key_locked($queue,$gradingkey);
	if ($who eq $me) {
	    #already have the lock
	    $env{'form.gradingkey'}=&escape($gradingkey);
	    &Apache::lonxml::debug("already locked");
	    return ($gradingkey);
	}
	
	if (!defined($who)) {
	    if (&lock_key($queue,$gradingkey)) {
		&Apache::lonxml::debug("newly locked");
		return ($gradingkey);
	    } else {
		return (undef,'lock_failed');
	    }
	}

	#otherwise (defined($who) && $who ne $me) some else has it...
	return (undef,'not_allowed',
		&mt('Another user ([_1]) currently has the record for [_2] locked.',
		    $who,$env{'form.gradinguser'}.':'.$env{'form.gradingdomain'}));
    }


    my $queue=$env{'form.queue'};

    if (!defined($queue)) {
	$env{'form.queue'}=$queue='gradingqueue';
    }

    my $gradingkey=&unescape($env{'form.gradingkey'});

    if ($env{'form.queue'} eq 'none') {
	if (defined($env{'form.gradingkey'})) {
	    if ($target eq 'webgrade') {
		if ($env{'form.stop'}) {
		    return (undef,'stop');
		} elsif ($env{'form.cancel'}) {
		    return (undef,'cancel');
		} elsif ($env{'form.next'}) {
		    return (undef,'select_user');
		}
	    }
	    return ($gradingkey,'selected');
	} else {
	    return (undef,'select_user');
	}
    }
    if (defined($env{'form.queue'}) && defined($env{'form.gradingkey'})
	&& !defined($env{'form.gradingaction'}) 
	&& $env{'form.queuemode'} eq 'selected') {
	
	my $who=&queue_key_locked($queue,$gradingkey);
	if ($who eq $me) {
	    &Apache::lonxml::debug("Found a key was given to me");
	    return ($gradingkey,'selected');
	} else {
	    return (undef,'show_list');
	}

    }

    if ($target eq 'webgrade' && $env{'form.queuemode'} eq 'selected') {
	if ($env{'form.gradingaction'} eq 'resume') {
	    delete($env{'form.gradingaction'});
	    &Apache::lonxml::debug("Resuming a key");
	    return ($gradingkey);
	} elsif ($env{'form.gradingaction'} eq 'unlock') {
	    &Apache::lonxml::debug("Unlocking a key ".
				     &check_queue_unlock($queue,$gradingkey,1));
	    return (undef,'unlock');
	} elsif ($env{'form.gradingaction'} eq 'select') {
	    &Apache::lonxml::debug("Locking a key");
	    if (&lock_key($queue,$gradingkey)) {
		&Apache::lonxml::debug("Success $queue");
		return ($gradingkey);
	    }
	    &Apache::lonxml::debug("Failed $queue");
	    return (undef,'lock_failed');
	}
    }

    if ($env{'form.queuemode'} ne 'selected') {
	# don't get something new from the queue if they hit the stop button
    	if (!(($env{'form.cancel'} || $env{'form.stop'}) 
	      && $target eq 'webgrade') 
	    && !$env{'form.gradingaction'}) {
	    &Apache::lonxml::debug("Getting anew $queue");
	    return (&get_from_queue($queue));
	} else {
	    return (undef,'stop');
	}
    }
    return (undef,undef)
}

sub minimize_storage {
    foreach my $key (keys(%Apache::lonhomework::results)) {
	if ($key =~ /regrader$/) { next; }
	if ($Apache::lonhomework::results{$key} eq
	    $Apache::lonhomework::history{$key}) {
	    delete($Apache::lonhomework::results{$key});
	}
    }
}

sub end_Task {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result='';
    my $status=$Apache::inputtags::status['-1'];
    my ($version,$previous)=&get_version();
    if ($target eq 'grade' || $target eq 'web' || $target eq 'answer' ||
	$target eq 'tex') {
	if ($target eq 'web' || $target eq 'answer' || $target eq 'tex') {
	    if ($target eq 'web') {
		if (&show_task($status,$previous)) {
		    $result.=&Apache::lonxml::endredirection();
		}
		if ($status eq 'CAN_ANSWER' && !$previous && 
		    !$env{'form.donescreen'}) {
		    $result.="\n".'<table border="1">'.
			&Apache::inputtags::file_selector("$version.0",
							  "bridgetask","*",
							  'portfolioonly',
							  '
<h2>'.&mt('Submit Portfolio Files for Grading').'</h2>
<p>'.&mt('Indicate the files from your portfolio to be evaluated in grading this task.').'</p>').
							  "</table>";
		}
		if (!$previous && $status ne 'SHOW_ANSWER' &&
		    &show_task($status,$previous)) {
		    $result.=&Apache::inputtags::gradestatus('0',$target,1);
		}
		
		$result.='</form>';

		if (!$previous && $status ne 'SHOW_ANSWER' &&
		    &show_task($status,$previous)) {
		    my $action = &Apache::lonenc::check_encrypt($env{'request.uri'});
                    my $donetext = &mt('Done');
		    $result.=<<DONEBUTTON;
<form name="done" method="post" action="$action">
   <input type="hidden" name="donescreen" value="1" />
   <input type="submit" value="$donetext" />
</form>
DONEBUTTON
                }
		if (&show_task($status,$previous) &&
		    $Apache::lonhomework::history{"resource.$version.0.status"} =~ /^(pass|fail)$/) {
		    my $bt_status=$Apache::lonhomework::history{"resource.$version.0.status"};
		    my $title=&Apache::lonnet::gettitle($env{'request.uri'});
		    my $start_time;

		    my $slot_name=
			$Apache::lonhomework::history{"resource.$version.0.checkedin.slot"};
		    if ($slot_name) {
			my %slot=&Apache::lonnet::get_slot($slot_name);

			$start_time=$slot{'starttime'}
		    } else {
			$start_time= 
			    &Apache::lonnet::EXT('resource.0.opendate');
		    }
		    $start_time=&Apache::lonlocal::locallocaltime($start_time);

		    my $status = 
			"\n<div class='LC_$bt_status LC_criteria LC_task_overall_status'>\n\t";
		    
		    my $dim = $top;
		    my %counts = &get_counts($dim,undef,$parstack,
					     $safeeval);
		    my $question_status ="\n\t<p>".
			&question_status_message(\%counts,-1).
			"</p>\n";

		    if ($bt_status eq 'pass')  {
			$status.='<h2>'
                                .&mt('You passed the [_1] given on [_2].',$title,$start_time)
                                .'</h2>';
			$status.=$question_status;
		    }
		    if ($bt_status eq 'fail')  {
			$status.='<h2>'
                                .&mt('You did not pass the [_1] given on [_2].',$title,$start_time)
                                .'</h2>';
			$status.=$question_status;
			if (!$previous) {
			    $status.=&add_request_another_attempt_button();
			}
		    }
		    
		    $status.="\n".'</div>'."\n";

		    foreach my $id (@{$dimension{$dim}{'criterias'}}) {
			my $type = $dimension{$dim}{'criteria.'.$id.'.type'};
			if ($type eq 'dimension') {
			    $result.=$dimension{$id}{'result'};
			    next;
			}
			my $criteria = 
			    &nested_parse(\$dimension{$dim}{'criteria.'.$id},
					  [@_]);
			$status .= &layout_web_Criteria($dim,$id,$criteria);
		    }

		    my $internal_location=&internal_location();
		    $result=~s/\Q$internal_location\E/$status/;
		}
		$result.="\n</div>\n".
		    &Apache::loncommon::end_page({'discussion' => 1});
	    }
	}

	my $useslots = &Apache::lonnet::EXT("resource.0.useslots");
	my %queue_data = ('type' => 'Task',
			  'time' => time,);
	if (defined($Apache::inputtags::slot_name)) {
	    $queue_data{'slot'} = $Apache::inputtags::slot_name;
	} elsif (defined($Apache::lonhomework::history{"resource.$version.0.checkedin.slot"})) {
	    $queue_data{'slot'} = $Apache::lonhomework::history{"resource.$version.0.checkedin.slot"};
	}
	

	if ($target eq 'grade' && !$env{'form.webgrade'} && !$previous
	    && $status eq 'CAN_ANSWER') {
	    my $award='SUBMITTED';
	    &Apache::essayresponse::file_submission("$version.0",'bridgetask',
						    'portfiles',\$award);
	    if ($award eq 'SUBMITTED' &&
		$Apache::lonhomework::results{"resource.$version.0.bridgetask.portfiles"}) {
		$Apache::lonhomework::results{"resource.0.tries"}=
		    $Apache::lonhomework::results{"resource.$version.0.tries"}=
		    1+$Apache::lonhomework::history{"resource.$version.0.tries"};

		$Apache::lonhomework::results{"resource.0.award"}=
		    $Apache::lonhomework::results{"resource.$version.0.award"}=
		    $award;
		$Apache::lonhomework::results{"resource.0.submission"}=
		    $Apache::lonhomework::results{"resource.$version.0.submission"}='';
	    } else {
		delete($Apache::lonhomework::results{"resource.$version.0.bridgetask.portfiles"});
		$award = '';
	    }
	    &Apache::lonhomework::showhash(%Apache::lonhomework::results);
	    &Apache::structuretags::finalize_storage();
	    if ($award eq 'SUBMITTED') {
		&add_to_queue('gradingqueue',\%queue_data);
	    }
	}
	if ($target eq 'grade' && $env{'form.webgrade'} eq 'yes' 
	    && exists($env{'form.cancel'})) {
	    &check_queue_unlock($env{'form.queue'});
	    &Apache::lonxml::debug(" cancelled grading .".$env{'form.queue'});
	} elsif ($target eq 'grade' && $env{'form.webgrade'} eq 'yes' 
		 && !exists($env{'form.cancel'})) {
	    my $optional_required=
		&Apache::lonxml::get_param('OptionalRequired',$parstack,
					   $safeeval);
	    my $optional_passed=0;
	    my $mandatory_failed=0;
	    my $ungraded=0;
	    my $review=0;   
	    &Apache::lonhomework::showhash(%Apache::lonhomework::results);
	    my $dim = $top;
	    foreach my $id (@{$dimension{$dim}{'criterias'}}) {
		my $link=&link($id);

		my $type = $dimension{$dim}{'criteria.'.$id.'.type'};

		if ($type eq 'criteria') {
		    # dimensional 'criteria' don't get assigned grades
		    $Apache::lonhomework::results{"resource.$version.0.$id.status"}=$env{'form.HWVAL_'.$link};
		    $Apache::lonhomework::results{"resource.$version.0.$id.comment"}=$env{'form.HWVAL_comment_'.$link};
		} 
		my $status=
		    $Apache::lonhomework::results{"resource.$version.0.$id.status"};
		my $mandatory=($dimension{$dim}{'criteria.'.$id.'.mandatory'} ne 'N');

		if ($status eq 'pass') {
		    if (!$mandatory) { $optional_passed++; }
		} elsif ($status eq 'fail') {
		    if ($mandatory) { $mandatory_failed++; }
		} elsif ($status eq 'review') {
		    $review++;
		} elsif ($status eq 'ungraded') {
		    $ungraded++;
		} else {
		    $ungraded++;
		}
	    }
	    if ($optional_passed < $optional_required) {
		$mandatory_failed++;
	    }
	    &Apache::lonxml::debug(" task results -> m_f $mandatory_failed o_p $optional_passed u $ungraded r $review");
	    $Apache::lonhomework::results{'resource.0.regrader'}=
		$env{'user.name'}.':'.$env{'user.domain'};
	    if ($review) {
		$Apache::lonhomework::results{"resource.$version.0.status"}='review';
	    } elsif ($ungraded) {
		$Apache::lonhomework::results{"resource.$version.0.status"}='ungraded';
	    } elsif ($mandatory_failed) {
		$Apache::lonhomework::results{"resource.$version.0.status"}='fail';
		$Apache::lonhomework::results{"resource.$version.0.solved"}='incorrect_by_override';
		$Apache::lonhomework::results{"resource.$version.0.award"}='INCORRECT';
		$Apache::lonhomework::results{"resource.$version.0.awarded"}='0';
		my ($symb,$courseid,$udom,$uname)=&Apache::lonnet::whichuser();
		
		if ($env{'form.regrade'} ne 'yes') {
		    $Apache::lonhomework::results{"resource.$version.0.bridgetask.portfiles"}=
			$Apache::lonhomework::history{"resource.$version.0.bridgetask.portfiles"};
		    &Apache::grades::version_portfiles(
						       \%Apache::lonhomework::results,
						       ["$version.0.bridgetask"],$courseid,
						       $symb,$udom,$uname,
						       ["$version.0.bridgetask"]);
		}
	    } else {
		$Apache::lonhomework::results{"resource.$version.0.status"}='pass';
		$Apache::lonhomework::results{"resource.$version.0.solved"}='correct_by_override';
		$Apache::lonhomework::results{"resource.$version.0.award"}='EXACT_ANS';
		$Apache::lonhomework::results{"resource.$version.0.awarded"}='1';
		my ($symb,$courseid,$udom,$uname)=&Apache::lonnet::whichuser();
		if ($env{'form.regrade'} ne 'yes') {
		    $Apache::lonhomework::results{"resource.$version.0.bridgetask.portfiles"}=
			$Apache::lonhomework::history{"resource.$version.0.bridgetask.portfiles"};
		    &Apache::grades::version_portfiles(
						       \%Apache::lonhomework::results,
						       ["$version.0.bridgetask"],$courseid,
						       $symb,$udom,$uname,
						       ["$version.0.bridgetask"]);
		}
	    }
	    $Apache::lonhomework::results{"resource.0.status"}=
		$Apache::lonhomework::results{"resource.$version.0.status"};
	    if (defined($Apache::lonhomework::results{"resource.$version.0.awarded"})) {
		$Apache::lonhomework::results{"resource.0.award"}=
		    $Apache::lonhomework::results{"resource.$version.0.award"};
		$Apache::lonhomework::results{"resource.0.awarded"}=
		    $Apache::lonhomework::results{"resource.$version.0.awarded"};
		$Apache::lonhomework::results{"resource.0.solved"}=
		    $Apache::lonhomework::results{"resource.$version.0.solved"};
	    }
	    &minimize_storage();
	    &Apache::structuretags::finalize_storage();
	    
	    # data stored, now handle queue
	    if ($review) {
		if ($env{'form.queue'} eq 'reviewqueue') {
		    &check_queue_unlock($env{'form.queue'});
		    &Apache::lonxml::debug(" still needs review not changing status.");
		} else {
		    if ($env{'form.queue'} ne 'none') {
			&move_between_queues($env{'form.queue'},'reviewqueue');
		    } else {
			&add_to_queue('reviewqueue',\%queue_data);
		    }
		}
	    } elsif ($ungraded) {
		if ($env{'form.queue'} eq 'reviewqueue') {
		    &Apache::lonxml::debug("moving back.");
		    &move_between_queues($env{'form.queue'},
					 'gradingqueue');
		} elsif ($env{'form.queue'} eq 'none' ) {
		    &add_to_queue('gradingqueue',\%queue_data);	
		} else {
		    &check_queue_unlock($env{'form.queue'});
		}
	    } elsif ($mandatory_failed) {
		&remove_from_queue($env{'form.queue'}); 
	    } else {
		&remove_from_queue($env{'form.queue'});
	    }
	}
	if (exists($Apache::lonhomework::results{'INTERNAL_store'})) {
	    # instance generation occurred and hasn't yet been stored
	    &Apache::structuretags::finalize_storage();
	}
    } elsif ($target eq 'webgrade') {
	if (&nest()) {
	    &Apache::lonxml::endredirection();
	    &end_delay();
	    $result.=$dimension{$top}{'result'};
	} else {
	    $result.=&Apache::lonxml::endredirection();
	}
	my $dim = $top;
	foreach my $id (@{$dimension{$dim}{'criterias'}} ) {
	    my $type = $dimension{$dim}{'criteria.'.$id.'.type'};
	    if ($type eq 'dimension') {
		# dimensional 'criteria' don't get assigned grades
		next;
	    } else {
		my $criteria =&nested_parse(\$dimension{$dim}{'criteria.'.$id},
					     [@_]);
		$criteria = &layout_webgrade_Criteria($dim,$id,$criteria);
		my $internal_location=&internal_location($id);
		if ($result =~ m/\Q$internal_location\E/) {
		    $result=~s/\Q$internal_location\E/$criteria/;
		} else {
		    $result.=$criteria;
		}

	    }
	}
        $result.="</div>";
	#$result.='<input type="submit" name="next" value="'.
	#    &mt('Save &amp; Next').'" /> ';
	#$result.='<input type="submit" name="end" value="'.
	#    &mt('Save &amp; Stop Grading').'" /> ';
	#$result.='<input type="submit" name="throwaway" value="'.
	#    &mt('Throw Away &amp; Stop Grading').'" /> ';
	#$result.='<input type="submit" name="save" value="'.
	#    &mt('Save Partial Grade and Continue Grading').'" /> ';
	$result.='</form>'."\n</div>\n</div>\n".
	    &Apache::loncommon::end_page();
    } elsif ($target eq 'meta') {
	$result.=&Apache::response::meta_package_write('Task');
        $result.=&Apache::response::meta_stores_write('solved','string',
						      'Problem Status');
	$result.=&Apache::response::meta_stores_write('tries','int_zeropos',
						      'Number of Attempts');
	$result.=&Apache::response::meta_stores_write('awarded','float',
						      'Partial Credit Factor');
	$result.=&Apache::response::meta_stores_write('status','string',
						      'Bridge Task Status');
    } elsif ($target eq 'edit') {
	$result.= &Apache::structuretags::problem_edit_footer();
    }
    &Apache::structuretags::reset_problem_globals('Task');
    undef($Apache::lonhomework::parsing_a_task);
    return $result;
}

sub move_between_queues {
    my ($src_queue,$dest_queue)=@_;
    my $cur_data;
    if ($src_queue ne 'none') {
	$cur_data=&get_queue_data($src_queue);
	if (!$cur_data) { return 'not_exist'; }
    } else {
	$cur_data = ['none'];
    }
    my $result=&add_to_queue($dest_queue,$cur_data);
    if ($result ne 'ok') {
	return $result;
    }
    &check_queue_unlock($src_queue);
    return &remove_from_queue($src_queue);
}

sub check_queue_unlock {
    my ($queue,$key,$allow_not_me)=@_;
    if ($queue eq 'none') { return 'ok'; }
    my ($symb,$cid,$udom,$uname)=&Apache::lonnet::whichuser();
    if (!defined($key)) {
	$key="$symb\0queue\0$uname:$udom";
    }
    my $cnum=$env{'course.'.$cid.'.num'};
    my $cdom=$env{'course.'.$cid.'.domain'};
    my $me=$env{'user.name'}.':'.$env{'user.domain'};
    my $who=&queue_key_locked($queue,$key,$cdom,$cnum);
    if  ($who eq $me) {
	&Apache::lonxml::debug("unlocking my own $who");
	return &Apache::lonnet::del($queue,["$key\0locked"],$cdom,$cnum);
    } elsif ($allow_not_me) {
	&Apache::lonxml::debug("unlocking $who by $me");
	return &Apache::lonnet::del($queue,["$key\0locked"],$cdom,$cnum);
    }
    return 'not_owner';
}

sub in_queue {
    my ($queue,$symb,$cdom,$cnum,$udom,$uname)=@_;
    if ($queue eq 'none') { return 0; }
    if (!defined($symb) || !defined($cdom) || !defined($cnum)
	|| !defined($udom) || !defined($uname)) {
	($symb,my $cid,$udom,$uname)=&Apache::lonnet::whichuser();
	$cnum=$env{'course.'.$cid.'.num'};
	$cdom=$env{'course.'.$cid.'.domain'};
    }

    my $key=&encode_queue_key($symb,$udom,$uname);
    my %results = &Apache::lonnet::get($queue,[$key],$cdom,$cnum);

    if (defined($results{$key})) {
	return 1;
    }
    return 0;
}

sub remove_from_queue {
    my ($queue,$symb,$cdom,$cnum,$udom,$uname)=@_;
    if ($queue eq 'none') { return 'ok'; }
    if (!defined($symb) || !defined($cdom) || !defined($cnum)
	|| !defined($udom) || !defined($uname)) {
	($symb,my $cid,$udom,$uname)=&Apache::lonnet::whichuser();
	$cnum=$env{'course.'.$cid.'.num'};
	$cdom=$env{'course.'.$cid.'.domain'};
    }
    if (!&in_queue($queue,$symb,$cdom,$cnum,$udom,$uname)) {
	return 'ok';
    }
    my $key=&encode_queue_key($symb,$udom,$uname);
    my @keys=($key,"$key\0locked");
    return &Apache::lonnet::del($queue,\@keys,$cdom,$cnum);
}

sub setup_env_for_other_user {
    my ($queue_key,$safeeval)=@_;
    my ($symb,$uname,$udom)=&decode_queue_key($queue_key);
    &Apache::lonxml::debug("setup_env for $queue_key");
    $env{'form.grade_symb'}=$symb;
    $env{'form.grade_domain'}=$udom;
    $env{'form.grade_username'}=$uname;
    $env{'form.grade_courseid'}=$env{'request.course.id'};
    &Apache::lonxml::initialize_rndseed($safeeval);
}

sub get_queue_data {
    my ($queue,$udom,$uname)=@_;
    my ($symb,$cid,$other_udom,$other_uname)=&Apache::lonnet::whichuser();
    if (!$uname || !$udom) {
	$uname=$other_uname;
	$udom =$other_udom;
    }
    my $cnum=$env{'course.'.$cid.'.num'};
    my $cdom=$env{'course.'.$cid.'.domain'};
    my $todo="$symb\0queue\0$uname:$udom";
    my ($key,$value)=&Apache::lonnet::get($queue,[$todo],$cdom,$cnum);
    if ($key eq $todo && ref($value)) {
	return $value;
    }
    return undef;
}


sub check_queue_for_key {
    my ($cdom,$cnum,$queue,$todo)=@_;

    my %results=
	&Apache::lonnet::get($queue,[$todo,"$todo\0locked"],$cdom,$cnum);
    
    if (exists($results{$todo}) && ref($results{$todo})) {
	if (defined($results{"$todo\0locked"})) {
	    return 'locked';
	}
	if (my $slot=&slotted_access($results{$todo})) {
	    my %slot_data=&Apache::lonnet::get_slot($slot);
	    if ($slot_data{'endtime'} > time) { 
		return 'in_progress';
	    }
	} else {
	    my ($symb) = &decode_queue_key($todo);
	    my $due_date = &Apache::lonhomework::due_date('0',$symb);
	    if ($due_date > time) {
		return 'in_progress';
	    }
	}
	return 'enqueued';
    }
    return undef;
}

sub add_to_queue {
    my ($queue,$user_data)=@_;
    if ($queue eq 'none') { return 'ok'; }
    my ($symb,$cid,$udom,$uname)=&Apache::lonnet::whichuser();
    if (!$cid || $env{'request.state'} eq 'construct') {
	return 'no_queue';
    }
    my $cnum=$env{'course.'.$cid.'.num'};
    my $cdom=$env{'course.'.$cid.'.domain'};
    my %data;
    $data{"$symb\0queue\0$uname:$udom"}=$user_data;
    return &Apache::lonnet::cput($queue,\%data,$cdom,$cnum);
}

sub get_limited_classlist {
    my ($sections) = @_;

    my $classlist = &Apache::loncoursedata::get_classlist();
    foreach my $student (keys(%$classlist)) {
	if ( $classlist->{$student}[&Apache::loncoursedata::CL_STATUS()]
	     ne 'Active') {
	    delete($classlist->{$student});
       	}
    }

    if (ref($sections) && !grep {$_ eq 'all'} (@{ $sections })) {
	foreach my $student (keys(%$classlist)) {
	    my $section  = 
		$classlist->{$student}[&Apache::loncoursedata::CL_SECTION()];
	    if (! grep {$_ eq $section} (@{ $sections })) {
		delete($classlist->{$student});
	    }
	}
    }
    return $classlist;
}


sub show_queue {
    my ($queue,$with_selects)=@_;
    my $result;
    my ($symb,$cid,$udom,$uname)=&Apache::lonnet::whichuser();
    my $cnum=$env{'course.'.$cid.'.num'};
    my $cdom=$env{'course.'.$cid.'.domain'};

    my @chosen_sections = &get_allowed_sections();

    my $classlist = &get_limited_classlist(\@chosen_sections);

    if (!(grep(/^all$/,@chosen_sections))) {
	$result.='<p>'
                .&mt('Showing only sections [_1].'
                    ,'<tt>'.join(', ',@chosen_sections).'</tt>')
                ."</p>\n";
    }

    my ($view,$view_section);
    my $scope = $env{'request.course.id'};
    if (!($view=&Apache::lonnet::allowed('vgr',$scope))) {
	$scope .= '/'.$env{'request.course.sec'};
	if ( $view = &Apache::lonnet::allowed('vgr',$scope)) {
	    $view_section=$env{'request.course.sec'};
	} else {
	    undef($view);
	}
    }

    $result .= 
	'<p><a href="/adm/flip?postdata=return:">'.
	&mt('Return to resource').'</a></p><hr />'.
	"\n<h3>".&mt('Current Queue - [_1]',$queue)."</h3>";
    my $regexp="^$symb\0";
    my %queue=&Apache::lonnet::dump($queue,$cdom,$cnum,$regexp);
    my ($tmp)=%queue;
    if ($tmp=~/^error: 2 /) {
	$result.=
	    &Apache::loncommon::start_data_table().
	    &Apache::loncommon::start_data_table_row().
	    '<td>'.&mt('Empty').'</td>'.
	    &Apache::loncommon::end_data_table_row().
	    &Apache::loncommon::end_data_table();
	return $result;
    }
    my $title=&Apache::lonnet::gettitle($symb);
    $result.=
	&Apache::loncommon::start_data_table().
	&Apache::loncommon::start_data_table_header_row();
    if ($with_selects) { $result.='<th>'.&mt('Status').'</th><th></th>'; }
    $result.='<th>'.&mt('User').'</th><th>'.&mt('Data').'</th>'.
	&Apache::loncommon::end_data_table_header_row();
    foreach my $key (sort(keys(%queue))) {
	my ($symb,$uname,$udom) = &decode_queue_key($key);
	next if (!defined($classlist->{$uname.':'.$udom}));
	next if (!&allow_grade_user($classlist->{$uname.':'.$udom}));
	
	my $section = $classlist->{$uname.':'.$udom}[&Apache::loncoursedata::CL_SECTION()];

	my $can_view=1;
	if (!$view
	    || ($view_section && !$section)
	    || ($view_section && $section && ($view_section ne $section))) {
	    $can_view=0;
	}

	if ($key=~/locked$/ && !$with_selects) {
	    $result.= &Apache::loncommon::start_data_table_row().
		"<td>$uname</td>";
	    $result.='<td>'.$queue{$key}.'</td></tr>';
	} elsif ($key=~/timestamp$/ && !$with_selects) {
	    $result.=&Apache::loncommon::start_data_table_row()."<td></td>";
	    $result.='<td>'.
		&Apache::lonlocal::locallocaltime($queue{$key})."</td></tr>";
	} elsif ($key!~/(timestamp|locked)$/) {
	    $result.= &Apache::loncommon::start_data_table_row();
	    my ($end_time,$slot_text);
	    if (my $slot=&slotted_access($queue{$key})) {
		my %slot_data=&Apache::lonnet::get_slot($slot);
		$end_time = $slot_data{'endtime'};
		$slot_text = &mt('Slot: [_1]',$slot);
	    } else {
		$end_time = &Apache::lonhomework::due_date('0',$symb);
		$slot_text = '';
	    }
	    if ($with_selects) {
		my $ekey=&escape($key);
		my ($action,$description,$status)=('select',&mt('Select'));
		if (exists($queue{"$key\0locked"})) {
		    my ($locker,$time) = 
			&get_lock_info($queue{"$key\0locked"});
		    if ($time) {
			$time = 
			    &Apache::lonnavmaps::timeToHumanString($time,
								   'start');
		    }
		    my $me=$env{'user.name'}.':'.$env{'user.domain'};
		    $status=&mt('Locked by [_1] [_2]','<tt>'.$locker.'</tt>',$time);
		    if ($me eq $locker) {
			($action,$description)=('resume',&mt('Resume'));
		    } else {
			($action,$description)=('unlock',&mt('Unlock'));
		    }
		}
		my $seclist;
		foreach my $sec (@chosen_sections) {
		    $seclist.='<input type="hidden" name="chosensections" 
                               value="'.$sec.'" />';
		}
		if ($can_view && ($end_time ne '' && time > $end_time)) {
		    $result.=(<<FORM);
<td>$status</td>
<td>
<form style="display: inline" method="post">
 <input type="hidden" name="gradingkey" value="$ekey" />
 <input type="hidden" name="queue" value="$queue" />
 <input type="hidden" name="gradingaction" value="$action" />
 <input type="hidden" name="webgrade" value="no" />
 <input type="hidden" name="queuemode" value="selected" />
 <input type="submit" name="submit" value="$description" />
 $seclist
</form>
</td>
FORM
                } elsif (!$can_view && ($end_time ne '' && time > $end_time)) {
		    $result.='<td>'.&mt("Not gradable").'</td><td>&nbsp;</td>'
                } else {
		    $result.='<td>'.&mt("In Progress").'</td><td>&nbsp;</td>'
		}
	    }
	    $result.= "<td>".$classlist->{$uname.':'.$udom}[&Apache::loncoursedata::CL_FULLNAME()].
		" <tt>($uname:$udom)</tt> </td>";
            $result.='<td>'.$slot_text.' '
                    .&mt('End time: [_1]'
                        ,&Apache::lonlocal::locallocaltime($end_time))
                    .'</td>'
                    .&Apache::loncommon::end_data_table_row();
	}
    }
    $result.= &Apache::loncommon::end_data_table()."<hr />\n";
    return $result;
}

sub get_allowed_sections {
    my @chosen_sections;
    if (&section_restricted()) {
	@chosen_sections = ($env{'request.course.sec'});
    } else {
	@chosen_sections =
	    &Apache::loncommon::get_env_multiple('form.chosensections');
    }

    return @chosen_sections;
}

sub section_restricted {
    my $cid =(&Apache::lonnet::whichuser())[1];
    return (lc($env{'course.'.$cid.'.task_grading'}) eq 'section'
	    && $env{'request.course.sec'} ne '' );
}

sub allow_grade_user {
    my ($classlist_entry) = @_;

    if (&section_restricted()
	&& $env{'request.course.sec'} ne
	      $classlist_entry->[&Apache::loncoursedata::CL_SECTION()]) {
	return 0;
    }
    return 1;
}

sub get_queue_counts {
    my ($queue)=@_;
    my $result;
    my ($symb,$cid,$udom,$uname)=&Apache::lonnet::whichuser();
    my $cnum=$env{'course.'.$cid.'.num'};
    my $cdom=$env{'course.'.$cid.'.domain'};

    my $classlist=&get_limited_classlist();

    my $regexp="^$symb\0";
    my %queue=&Apache::lonnet::dump($queue,$cdom,$cnum,$regexp);
    my ($tmp)=%queue;
    if ($tmp=~/^error: 2 /) {
	return (0,0,0);
    }

    my ($entries,$ready_to_grade,$locks)=(0,0,0);
    my %slot_cache;
    foreach my $key (sort(keys(%queue))) {
	my ($symb,$uname,$udom) = &decode_queue_key($key);
	next if (!defined($classlist->{$uname.':'.$udom}));
	next if (!&allow_grade_user($classlist->{$uname.':'.$udom}));

	if ($key=~/locked$/) {
	    $locks++;
	} elsif ($key=~/timestamp$/) {
	    #ignore
	} elsif ($key!~/(timestamp|locked)$/) {
	    $entries++;
	    if (my $slot=&slotted_access($queue{$key})) {
		if (!exists($slot_cache{$slot})) {
		    my %slot_data=&Apache::lonnet::get_slot($slot);
		    $slot_cache{$slot} = \%slot_data;
		}
		if (time > $slot_cache{$slot}{'endtime'}) {
		    $ready_to_grade++;
		}
	    } else {
		my $due_date = &Apache::lonhomework::due_date('0',$symb);
		if ($due_date ne '' && time > $due_date) {
		    $ready_to_grade++;
		}
	    }
	}
    }
    return ($entries,$ready_to_grade,$locks);
}

sub encode_queue_key {
    my ($symb,$udom,$uname)=@_;
    return "$symb\0queue\0$uname:$udom";
}

sub decode_queue_key {
    my ($key)=@_;
    my ($symb,undef,$user) = split("\0",$key);
    my ($uname,$udom) = split(':',$user);
    return ($symb,$uname,$udom);
}

sub queue_key_locked {
    my ($queue,$key,$cdom,$cnum)=@_;
    if (!defined($cdom) || !defined($cnum)) {
	my (undef,$cid)=&Apache::lonnet::whichuser();
	$cnum=$env{'course.'.$cid.'.num'};
	$cdom=$env{'course.'.$cid.'.domain'};
    }
    my ($key_locked,$value)=
	&Apache::lonnet::get($queue,["$key\0locked"],$cdom,$cnum);
    if ($key_locked eq "$key\0locked") {
	return &get_lock_info($value);
    }
    return undef;
}

sub slotted_access {
    my ($queue_entry) = @_;
    if (ref($queue_entry) eq 'ARRAY') {
	if (defined($queue_entry->[0])) {
	    return $queue_entry->[0];
	}
	return undef;
    } elsif (ref($queue_entry) eq 'HASH') {
	if (defined($queue_entry->{'slot'})) {
	    return $queue_entry->{'slot'};
	}
	return undef;
    }
    return undef;
}

sub pick_from_queue_data {
    my ($queue,$check_section,$queuedata,$cdom,$cnum,$classlist)=@_;
    my @possible; # will hold queue entries that are valid to be selected
    foreach my $key (keys(%$queuedata)) {
	if ($key =~ /\0locked$/) { next; }
	if ($key =~ /\0timestamp$/) { next; }

	my ($symb,$uname,$udom)=&decode_queue_key($key);
	next if (!defined($classlist->{$uname.':'.$udom}));
	next if (!&allow_grade_user($classlist->{$uname.':'.$udom}));

	if ($check_section) {
	    my $section =
		$classlist->{$uname.':'.$udom}[&Apache::loncoursedata::CL_SECTION()];
	    if ($section eq $check_section) {
		&Apache::lonxml::debug("my sec");
		next;
	    }
	}
	my $end_time;
	if (my $slot=&slotted_access($queuedata->{$key})) {
	    &Apache::lonxml::debug("looking at slot $slot");
	    my %slot_data=&Apache::lonnet::get_slot($slot);
	    if ($slot_data{'endtime'} < time) { 
		$end_time = $slot_data{'endtime'};
	    } else {
		&Apache::lonxml::debug("not time ".$slot_data{'endtime'});
		next;
	    }
	} else {
	    my $due_date = &Apache::lonhomework::due_date('0',$symb);
	    if ($due_date < time) {
		$end_time = $due_date;
	    } else {
		&Apache::lonxml::debug("not time $due_date");
		next;
	    }
	}
	
	if (exists($queuedata->{"$key\0locked"})) {
	    &Apache::lonxml::debug("someone already has um.");
	    next;
	}
	push(@possible,[$key,$end_time]);
    }
    if (@possible) {
        # sort entries in order by slot end time
	@possible = sort { $a->[1] <=> $b->[1] } @possible;
	# pick one of the entries in the top 10% in small queues and one
	# of the first ten entries in large queues
	#my $ten_percent = int($#possible * 0.1);
	#if ($ten_percent < 1 ) { $ten_percent = 1;  }
	#if ($ten_percent > 10) { $ten_percent = 10; }
	#my $max=($#possible < $ten_percent) ? $#possible : $ten_percent;
	
	#return $possible[int(rand($max))][0];
	return $possible[0][0];
    }
    return undef;
}

sub get_lock_info {
    my ($lock_info) = @_;
    if (wantarray) {
	if (ref($lock_info) eq 'ARRAY') {
	    return @{$lock_info};
	} else {
	    return ($lock_info);
	}
    } else {
	if (ref($lock_info) eq 'ARRAY') {
	    return $lock_info->[0];
	} else {
	    return $lock_info;
	}
    }
    return;
}

sub find_mid_grade {
    my ($queue,$symb,$cdom,$cnum)=@_;
    my $todo=&unescape($env{'form.gradingkey'});
    my $me=$env{'user.name'}.':'.$env{'user.domain'};
    if ($todo) {
	my $who=&queue_key_locked($queue,$todo,$cdom,$cnum);
	if ($who eq $me) { return $todo; }
    }
    my $regexp="^$symb\0.*\0locked\$";
    my %locks=&Apache::lonnet::dump($queue,$cdom,$cnum,$regexp);
    foreach my $key (keys(%locks)) {
	my $who= &get_lock_info($locks{$key});
	if ($who eq $me) {
	    $todo=$key;
	    $todo=~s/\0locked$//;
	    return $todo;
	}
    }
    return undef;
}

sub lock_key {
    my ($queue,$todo)=@_;
    my $me=$env{'user.name'}.':'.$env{'user.domain'};
    my (undef,$cid)=&Apache::lonnet::whichuser();
    my $cnum=$env{'course.'.$cid.'.num'};
    my $cdom=$env{'course.'.$cid.'.domain'};
    my $success=&Apache::lonnet::newput($queue,{"$todo\0locked"=> [$me,time]},
					$cdom,$cnum);
    &Apache::lonxml::debug("success $success $todo");
    if ($success eq 'ok') {
	return 1;
    }
    return 0;
}

sub get_queue_symb_status {
    my ($queue,$symb,$cdom,$cnum) = @_;
    if (!defined($cdom) || !defined($cnum)) {
	my (undef,$cid) =&Apache::lonnet::whichuser();
	$cnum=$env{'course.'.$cid.'.num'};
	$cdom=$env{'course.'.$cid.'.domain'};
    }
    my $classlist=&get_limited_classlist();

    my $regexp="^$symb\0";
    my %queue=&Apache::lonnet::dump($queue,$cdom,$cnum,$regexp);
    my ($tmp)=%queue;
    if ($tmp=~/^error: 2 /) { return; }
    my @users;
    foreach my $key (sort(keys(%queue))) {
	next if ($key=~/locked$/);
	next if ($key=~/timestamp$/);
	my ($symb,$uname,$udom) = &decode_queue_key($key);
	next if (!defined($classlist->{$uname.':'.$udom}));
	next if (!&allow_grade_user($classlist->{$uname.':'.$udom}));
	push(@users,"$uname:$udom");
    }
    return @users;
}

sub get_from_queue {
    my ($queue)=@_;
    my $result;
    my ($symb,$cid,$udom,$uname)=&Apache::lonnet::whichuser();
    my $cnum=$env{'course.'.$cid.'.num'};
    my $cdom=$env{'course.'.$cid.'.domain'};
    my $todo=&find_mid_grade($queue,$symb,$cdom,$cnum);
    &Apache::lonxml::debug("found ".join(':',&decode_queue_key($todo)));
    if ($todo) { return $todo; }
    my $attempts=0;

    my $classlist=&get_limited_classlist();

    while (1) {
	if ($attempts > 2) {
	    # tried twice to get a queue entry, giving up
	    return (undef,'unable');
	}
	my $starttime=time;
	&Apache::lonnet::cput($queue,{"$symb\0timestamp"=>$starttime},
			      $cdom,$cnum);
	&Apache::lonxml::debug("$starttime");
	my $regexp="^$symb\0queue\0";
	#my $range= ($attempts < 1 ) ? '0-100' : '0-400';

	my %queue=&Apache::lonnet::dump($queue,$cdom,$cnum,$regexp);
	#make a pass looking for a user _not_ in my section
	if ($env{'request.course.sec'}) {
	    &Apache::lonxml::debug("sce");
	    $todo=&pick_from_queue_data($queue,$env{'request.course.sec'},
					\%queue,$cdom,$cnum,$classlist);
	    &Apache::lonxml::debug("sce $todo");
	}
	# no one _not_ in our section so look for any user that is
	# ready for grading
	if (!$todo) {
	    &Apache::lonxml::debug("no sce");
	    $todo=&pick_from_queue_data($queue,undef,\%queue,$cdom,$cnum,
					$classlist);
	    &Apache::lonxml::debug("no sce $todo");
	}
	# no user to grade 
	if (!$todo) { last; }
	&Apache::lonxml::debug("got $todo");
	# otherwise found someone so lets try to lock them
	# unless someone else already picked them
	if (!&lock_key($queue,$todo)) {
	    $attempts++;
	    next;
	}
	my (undef,$endtime)=
	    &Apache::lonnet::get($queue,["$symb\0timestamp"],
				 $cdom,$cnum);
	&Apache::lonxml::debug("emd  $endtime");
	# someone else already modified the queue, 
	# perhaps our picked user wass already fully graded between
	# when we picked him and when we locked his record? so lets
	# double check.
	if ($endtime != $starttime) {
	    my ($key,$value)=
		&Apache::lonnet::get($queue,["$todo"],
				     $cdom,$cnum);
	    &Apache::lonxml::debug("check  $key .. $value");
	    if ($key eq $todo && ref($value)) {
	    } else {
		&Apache::lonnet::del($queue,["$todo\0locked"],
				     $cdom,$cnum);
		&Apache::lonxml::debug("del");
		$attempts++;
		next;
	    }
	}
	&Apache::lonxml::debug("last $todo");
	last;
    }
    return $todo;
}

sub select_user {
    my ($symb,$cid)=&Apache::lonnet::whichuser();

    my @chosen_sections = &get_allowed_sections();
    my $classlist = &get_limited_classlist(\@chosen_sections);
    
    my $result;
    if (!(grep(/^all$/,@chosen_sections))) {
        $result.='<p>'
                .&mt('Showing only sections [_1].'
                    ,'<tt>'.join(', ',@chosen_sections).'</tt>')
                .'</p> '."\n";
    }
    $result.=&Apache::loncommon::start_data_table();

    foreach my $student (sort {lc($classlist->{$a}[&Apache::loncoursedata::CL_FULLNAME()]) cmp lc($classlist->{$b}[&Apache::loncoursedata::CL_FULLNAME()]) } (keys(%$classlist))) {
	my ($uname,$udom) = split(/:/,$student);
	
	my $cnum=$env{'course.'.$cid.'.num'};
	my $cdom=$env{'course.'.$cid.'.domain'};
	my %status = &get_student_status($symb,$cdom,$cnum,$udom,$uname,
					 'Task');
	my $queue = 'none';
	my $cannot_grade;
	if ($status{'reviewqueue'} =~ /^(in_progress|enqueue)$/) {
	    $queue = 'reviewqueue';
	    if ($status{'reviewqueue'} eq 'in_progress') {
		$cannot_grade=1;
	    }
	} elsif ($status{'gradingqueue'} =~ /^(in_progress|enqueue)$/) {
	    $queue = 'gradingqueue';
	    if ($status{'gradingqueue'} eq 'in_progress') {
		$cannot_grade=1;
	    }
	}
	my $todo = 
	    &escape(&encode_queue_key($symb,$udom,$uname));
	if ($cannot_grade) {
	    $result.=&Apache::loncommon::start_data_table_row().
		'<td>&nbsp;</td><td>'.$classlist->{$student}[&Apache::loncoursedata::CL_FULLNAME()].
		'</td><td>';
	} else {
	    my $seclist;
	    foreach my $sec (@chosen_sections) {
		$seclist.='<input type="hidden" name="chosensections" 
                               value="'.$sec.'" />';
	    }
            my $buttontext=&mt('Regrade');
	    $result.=&Apache::loncommon::start_data_table_row();
	    $result.=<<RESULT;
  <td>
    <form style="display: inline" method="post">
      <input type="hidden" name="gradingkey" value="$todo" />
      <input type="hidden" name="queue" value="$queue" />
      <input type="hidden" name="webgrade" value="no" />
      <input type="hidden" name="regrade" value="yes" />
      <input type="submit" name="submit" value="$buttontext" />
      $seclist
    </form>
  <td>$classlist->{$student}[&Apache::loncoursedata::CL_FULLNAME()] <tt>($student)</tt> Sec: $classlist->{$student}[&Apache::loncoursedata::CL_SECTION()]</td>
  <td>
RESULT
        }
        if ($status{'status'} eq 'pass') {
	    $result .= '<font color="green">'.&mt('Passed').'</font>';
	} elsif ($status{'status'} eq 'fail') {
	    $result .= '<font color="red">'.&mt('Failed').'</font>';
	} elsif ($status{'status'} eq 'review') {
	    $result .= '<font color="blue">'.&mt('Under Review').'</font>';
	} elsif ($status{'status'} eq 'ungraded') {
	    $result .= &mt('Ungraded');
	} elsif ($status{'status'} ne '') {
	    $result .= '<font color="orange">'.&mt('Unknown Status').'</font>';
	} else {
	    $result.="&nbsp;";
	}
	if ($status{'version'}) {
	    $result .= ' '.&mt('Version').' '.$status{'version'};
	}
	if ($status{'grader'}) {
	    $result .= ' '.&mt('(Graded by [_1])',$status{'grader'}).' ';
	}
	$result.= '</td><td>';
	if ($status{'reviewqueue'} eq 'enqueued') {
	    $result .= &mt('Awaiting Review');
	} elsif ($status{'reviewqueue'} eq 'locked') {
	    $result .= &mt('Under Review');
	} elsif ($status{'reviewqueue'} eq 'in_progress') {
	    $result .= &mt('Still being worked on.');
	} elsif ($status{'gradingqueue'} eq 'enqueued') {
	    $result .= &mt('Awaiting Grading');
	} elsif ($status{'gradingqueue'} eq 'locked') {
	    $result .= &mt('Being Graded');
	} elsif ($status{'gradingqueue'} eq 'in_progress') {
	    $result .= &mt('Still being worked on.');
	} else {
	    $result.="&nbsp;";
	}
	$result.= '</td>'.&Apache::loncommon::end_data_table_row();
    }
    $result.=&Apache::loncommon::end_data_table();
    return $result;
}

sub get_student_status {
    my ($symb,$cdom,$cnum,$udom,$uname,$type)=@_;

    my %status;

    if ($type eq 'Task') {
	my %record = &Apache::lonnet::restore($symb,$env{'request.course.id'},
					  $udom,$uname);
	$status{'status'}=$record{'resource.0.status'};
	$status{'version'}=$record{'resource.0.version'};
	$status{'grader'}=$record{'resource.0.regrader'};
    }
    $status{'reviewqueue'}=
	&check_queue_for_key($cdom,$cnum,'reviewqueue',
			     &encode_queue_key($symb,$udom,$uname));
    $status{'gradingqueue'}=
	&check_queue_for_key($cdom,$cnum,'gradingqueue',
			     &encode_queue_key($symb,$udom,$uname));
    return %status;
}

sub start_ClosingParagraph {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;
    if ($target eq 'web') {
    } elsif ($target eq 'webgrade') {
	&Apache::lonxml::startredirection();
    } elsif ($target eq 'edit') {
	$result = &Apache::edit::tag_start($target,$token);
    } elsif ($target eq 'modified') {
    }
    return $result;
}

sub end_ClosingParagraph {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;
    if ($target eq 'web') {
    } elsif ($target eq 'webgrade') {
	&Apache::lonxml::endredirection();
    }
    return $result;
}

sub insert_ClosingParagraph {
    return '
<ClosingParagraph>
    <startouttext />
    <endouttext />
</ClosingParagraph>';
}

sub get_dim_id {
    if (@Apache::bridgetask::dimension) {
	return $Apache::bridgetask::dimension[-1];
    } else {
	return $top;
    }
}

sub get_id {
    my ($parstack,$safeeval)=@_;
    return &Apache::lonxml::get_id($parstack,$safeeval);
}

sub start_Setup {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;
    my $dim = &get_id($parstack,$safeeval);
    push(@Apache::bridgetask::dimension,$dim);
    if ($target eq 'web' || $target eq 'webgrade' || $target eq 'grade') {
	&Apache::lonxml::startredirection();
    } elsif ($target eq 'edit') {
	$result = &Apache::edit::tag_start($target,$token);
	$result.= &Apache::edit::text_arg('Id:','id',$token,10).
	    &Apache::edit::end_row().
	    &Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
	my $constructtag=
	    &Apache::edit::get_new_args($token,$parstack,$safeeval,'id');
	if ($constructtag) {
	    $result = &Apache::edit::rebuild_tag($token);
	}
    }
    return $result;
}

{
    my @allowed;
    sub enable_dimension_parsing {
	my ($id) = @_;
	push(@allowed,$id);
    }
    sub disable_dimension_parsing {
	pop(@allowed);
    }
    sub skip_dimension_parsing {
	my ($check) = @_;
	if (!@allowed) { return 0;}
	# if unspecified allow any id
	if ($allowed[-1] eq undef) { return 0;}

	return ($check ne $allowed[-1]);
    }
}

sub start_Question { return &start_Dimension(@_); }
sub start_Dimension {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $dim = &get_id($parstack,$safeeval);
    my $previous_dim;
    my $result;
    if ($target eq 'grade' || $target eq 'web' || $target eq 'webgrade') {
	if (@Apache::bridgetask::dimension) {
	    $previous_dim = $Apache::bridgetask::dimension[-1];
	    push(@{$Apache::bridgetask::dimension{$previous_dim}{'contains'}},
		 $dim);
	    if(&skip_dimension_parsing($dim)) {
		$dimension{$previous_dim}{'criteria.'.$dim} =
		    $token->[4]
		    .&Apache::lonxml::get_all_text('/'.$tagstack->[-1],$parser,
						   $style)
		    .'</'.$tagstack->[-1].'>';
	    }
	    $dimension{$previous_dim}{'criteria.'.$dim.'.type'}='dimension';
	    $dimension{$previous_dim}{'criteria.'.$dim.'.mandatory'}=
		&Apache::lonxml::get_param('Mandatory',$parstack,$safeeval);
	    push(@{$dimension{$previous_dim}{'criterias'}},$dim);
	    $dimension{$dim}{'nested'}=$previous_dim;
	    $dimension{$dim}{'depth'} = 1 + $dimension{$previous_dim}{'depth'};
	    
	    &Apache::lonxml::debug("adding $dim as criteria to $previous_dim");
	} else {
	    $dimension{$top}{'depth'}=0;
	    $dimension{$top}{'criteria.'.$dim.'.type'}='dimension';
	    $dimension{$top}{'criteria.'.$dim.'.mandatory'}=
		&Apache::lonxml::get_param('Mandatory',$parstack,$safeeval);
	    push(@{$dimension{$top}{'criterias'}},$dim);
	    $dimension{$dim}{'nested'}=$top;
	}
        push(@Apache::bridgetask::dimension,$dim);
	&Apache::lonxml::startredirection();
	if (!&skip_dimension_parsing($dim)) {
	    &enable_dimension_parsing($dim);
	}
    } elsif ($target eq 'edit') {
  	$result = &Apache::edit::tag_start($target,$token);
	$result.=  
	    &Apache::edit::text_arg('Id:','id',$token,10).' '.
	    &Apache::edit::select_arg('Passing is Mandatory:','Mandatory',
				      [['Y', 'Yes'],
				       ['N','No'],],
				      $token).' <br /> '.
	    &Apache::edit::text_arg('Required number of passed optional elements to pass the '.$token->[1].':',
				    'OptionalRequired',$token,4).
	    &Apache::edit::end_row().
	    &Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
	my $constructtag=
	    &Apache::edit::get_new_args($token,$parstack,$safeeval,
					'id','Mandatory','OptionalRequired');
	if ($constructtag) {
	    $result = &Apache::edit::rebuild_tag($token);
	}
    }
    return $result;# &internal_location($dim);
}

sub start_QuestionText {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'grade' || $target eq 'web' || $target eq 'webgrade') {
	my $text=&Apache::lonxml::get_all_text('/questiontext',$parser,$style);
    my $dim = &get_dim_id();
	$dimension{$dim}{'questiontext'}=$text;
    } elsif ($target eq 'edit') {
	$result = &Apache::edit::tag_start($target,$token);
    } elsif ($target eq 'modified') {
    }
    return $result;
}

sub end_QuestionText {
    return '';
}

sub insert_QuestionText {
    return '
<QuestionText>
    <startouttext />
    <endouttext />
</QuestionText>';
}

sub get_instance {
    my ($dim)=@_;
    my $rand_alg=&Apache::lonnet::get_rand_alg();
    if (!$rand_alg || $rand_alg eq '32bit' || $rand_alg eq '64bit' ||
	$rand_alg eq '64bit2' || $rand_alg eq '64bit3' ||
	$rand_alg eq '64bit4' ) {
	&Apache::response::pushrandomnumber();
	my @order=&Math::Random::random_permutation(@{$dimension{$dim}{'instances'}});
	my $num=@order;
	my $version=&get_version();
	my $which=($version-1)%$num;
	return $order[$which];
    } else {
	my ($version,$previous) = &get_version();
	my $instance = 
	    $Apache::lonhomework::history{"resource.$version.0.$dim.instance"};
	if (defined($instance)) { return $instance; }

	&Apache::response::pushrandomnumber();
	if (ref($dimension{$dim}{'instances'}) eq 'ARRAY') {
	    my @instances = @{$dimension{$dim}{'instances'}};
	    # remove disabled instances
	    for (my $i=0; $i < $#instances; $i++) {
		if ($dimension{$dim}{$instances[$i].'.disabled'}) {
		    splice(@instances,$i,1);
		    $i--;
		}
	    }
	    @instances = &Math::Random::random_permutation(@instances);
	    $instance  = $instances[($version-1)%scalar(@instances)];
	    if ($version =~ /^\d$/) {
		$Apache::lonhomework::results{"resource.$version.0.$dim.instance"} = 
		    $instance;
		$Apache::lonhomework::results{'INTERNAL_store'} = 1; 
	    }
	}
	&Apache::response::poprandomnumber();
	return $instance;
    }
}

sub get_criteria {
    my ($what,$version,$dim,$id) = @_;
    my $type = $dimension{$dim}{'criteria.'.$id.'.type'};
    my $prefix = ($type eq 'criteria' && $dim ne $top) ? "$dim.$id"
	                                               : "$id";
    my $entry = "resource.$version.0.$prefix.$what";
    if (exists($Apache::lonhomework::results{$entry})) {
	return $Apache::lonhomework::results{$entry};
    }
    return $Apache::lonhomework::history{$entry};
}

sub link {
    my ($id) = @_;
    $id =~ s/\./_/g;
    return 'LC_GRADING_criteria_'.$id;
}
sub end_Question { return &end_Dimension(@_); }
sub end_Dimension {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;
    my $dim=&get_id($parstack,$safeeval);
    if ($target eq 'grade' || $target eq 'web' || $target eq 'webgrade') {
	$result=&Apache::lonxml::endredirection();
	if (&skip_dimension_parsing($dim)) {
	    pop(@Apache::bridgetask::dimension);
	    return;
	}
    }
    my $instance=&get_instance($dim);
    my $version=&get_version();
    if ($target eq 'web') {
	$result .= &nested_parse(\$dimension{$dim}{'intro'},[@_]);
	my @instances = $instance;
	if (&Apache::response::showallfoils()) {
	    @instances = @{$dimension{$dim}{'instances'}};
	}
	my $shown_question_text;
	foreach my $instance (@instances) {
	    $result .= &nested_parse(\$dimension{$dim}{$instance.'.text'},
				     [@_]);
	    $result .= &nested_parse(\$dimension{$dim}{'questiontext'},
				     [@_],{'set_dim_id' => undef});
	    my $task_status = 
		$Apache::lonhomework::history{"resource.$version.0.status"};
	    if ($task_status ne 'pass' && $task_status ne 'fail') {
		
		foreach my $id (@{$dimension{$dim}{$instance.'.criterias'}},
				@{$dimension{$dim}{'criterias'}}) {
		    my $type = $dimension{$dim}{'criteria.'.$id.'.type'};
		    &Apache::lonxml::debug("$id is $type");
		    if ($type eq 'dimension') {
			$result.=
			    &nested_parse(\$dimension{$dim}{'criteria.'.$id},
					  [@_],{'set_dim_id' => $id});
		    }
		}
	    } else {
		my $dim_status=$Apache::lonhomework::history{"resource.$version.0.$dim.status"};
		my $mandatory='Mandatory';
		if (&Apache::lonxml::get_param('Mandatory',$parstack,$safeeval) eq 'N') {
		    $mandatory='Optional';
		}
		my $dim_info=
		    "\n<div class='LC_$dim_status LC_question_grade'>\n\t";
		my $ucquestion = 
		    my $question = 
		    ('sub' x $dimension{$dim}{'depth'}).'question';
		$ucquestion =~ s/^(.)/uc($1)/e;
		if ($dim_status eq 'pass') {
                    $dim_info.='<h3>'.$ucquestion.' : '
                              .&mt('you passed this [_1] [_2]',$mandatory,$question)
                              .'</h3>';
		}
		if ($dim_status eq 'fail') {
                    $dim_info.='<h3>'.$ucquestion.' : '
                              .&mt('you did not pass this [_1] [_2]',$mandatory,$question)
                              .'</h3>';
		}
		my %counts = &get_counts($dim,$instance,$parstack,
					 $safeeval);

		$dim_info.="\n\t<p>"
		    .&question_status_message(\%counts,
					      $dimension{$dim}{'depth'})
		    ."</p>\n</div>\n";
		
		foreach my $id (@{$dimension{$dim}{$instance.'.criterias'}},
				@{$dimension{$dim}{'criterias'}}) {
		    my $type = $dimension{$dim}{'criteria.'.$id.'.type'};
		    if ($type eq 'dimension') {
			if (defined($dimension{$id}{'result'})) {
			    $result.=$dimension{$id}{'result'};
			    next;
			} else {
			    $dim_info .=
				&nested_parse(\$dimension{$dim}{'criteria.'.$id},
					      [@_],{'set_dim_id' => $id});
			}
		    } else {
			my $criteria =
			    &nested_parse(\$dimension{$dim}{'criteria.'.$id},
					  [@_]);
			$dim_info .= &layout_web_Criteria($dim,$id,$criteria);
		    }
		}
		# puts the results at the end of the dimension
		if ($result =~m{<QuestionGradeInfo\s*/>}) {
		    $result=~s{<QuestionGradeInfo\s*/>}{$dim_info};
		} else {
		    $result .= $dim_info;
		}
		# puts the results at the beginning of the dimension
		# my $internal_location=&internal_location($dim);
		# $result=~s/\Q$internal_location\E/$dim_info/;
	    }
	}
	if ($result !~ /^\s*$/s) {
	    # FIXME? this maybe unneccssary in the future, (CSE101 BT
	    # from Fall 2006 geenrate a div that attempts to hide some
	    # of the output in an odd way, this is a workaround so
	    # those old ones will continue to work.  # It puts the
	    # LC_question div to come after any starting closie div
	    # that the dimension produces
	    if ($result =~ m{^\s*</div>}) {
		$result =~ s{^(\s*</div>)}
		            {$1\n<div id="$dim" class="LC_question">};
	    } else {
		$result = "\n".'<div id="'.$dim.'" class="LC_question">'.
		    "\n".$result;
	    }
	    $result .= "\n</div>\n";
	}
    } elsif ($target eq 'webgrade') {
	# in case of any side effects that we need
	&nested_parse(\$dimension{$dim}{'intro'},[@_]);
	&nested_parse(\$dimension{$dim}{$instance.'.text'},[@_]);
	$result.=
	    &nested_parse(\$dimension{$dim}{'questiontext'},[@_],
			  {'set_dim_id'          => undef,
			   'delayed_dim_results' => 1});
	foreach my $id (@{$dimension{$dim}{$instance.'.criterias'}},
			@{$dimension{$dim}{'criterias'}} ) {
	    my $type = $dimension{$dim}{'criteria.'.$id.'.type'};
	    if ($type eq 'dimension') {
		# dimensional 'criteria' don't get assigned grades
		$result.=
		    &nested_parse(\$dimension{$dim}{'criteria.'.$id},
				  [@_],{'set_dim_id' => $id});
		next;
	    } else {
		my $criteria =&nested_parse(\$dimension{$dim}{'criteria.'.$id},
					     [@_]);
		$criteria = &layout_webgrade_Criteria($dim,$id,$criteria);
		my $internal_location=&internal_location($id);
		if ($result =~ m/\Q$internal_location\E/) {
		    $result =~ s/\Q$internal_location\E/$criteria/;
		} else {
		    $result.=$criteria ;
		}
	    }
	}
	if (&nest()) {
	    &Apache::lonxml::debug(" for $dim stashing results into ".$dimension{$dim}{'nested'});
	    $dimension{$dimension{$dim}{'nested'}}{'result'}.=$result;
	    undef($result);
	}
    } elsif ($target eq 'grade' && $env{'form.webgrade'}) {
	my $optional_passed=0;
	my $mandatory_failed=0;
	my $ungraded=0;
	my $review=0;
	
	$result .= &nested_parse(\$dimension{$dim}{'intro'},[@_]);
	$result .= &nested_parse(\$dimension{$dim}{$instance.'.text'},
				 [@_]);
	$result .= &nested_parse(\$dimension{$dim}{'questiontext'},
				 [@_],{'set_dim_id' => undef});
	
	foreach my $id (@{$dimension{$dim}{$instance.'.criterias'}},
			@{$dimension{$dim}{'criterias'}}) {
	    my $link=&link($id);
	    
	    my $type = $dimension{$dim}{'criteria.'.$id.'.type'};
	    if ($type eq 'criteria') {
		# dimensional 'criteria' don't get assigned grades
		$Apache::lonhomework::results{"resource.$version.0.$dim.$id.status"}=$env{'form.HWVAL_'.$link};
		$Apache::lonhomework::results{"resource.$version.0.$dim.$id.comment"}=$env{'form.HWVAL_comment_'.$link};
	    } else {
		$result .=
		    &nested_parse(\$dimension{$dim}{'criteria.'.$id},
				  [@_],{'set_dim_id' => $id});
	    }
	    my $status= &get_criteria('status',$version,$dim,$id);
	    
	    my $mandatory=($dimension{$dim}{'criteria.'.$id.'.mandatory'} ne 'N');
	    if ($status eq 'pass') {
		if (!$mandatory) { $optional_passed++; }
	    } elsif ($status eq 'fail') {
		if ($mandatory) { $mandatory_failed++; }
	    } elsif ($status eq 'review') {
		$review++;
	    } elsif ($status eq 'ungraded') {
		$ungraded++;
	    } else {
		$ungraded++;
	    }
	}

	my $opt_req=$dimension{$dim}{$instance.'.optionalrequired'};
	if ($opt_req !~ /\S/) {
	    $opt_req=
		&Apache::lonxml::get_param('OptionalRequired',
					   $parstack,$safeeval);
	    if ($opt_req !~ /\S/) { $opt_req = 0; }
	}
	if ($optional_passed < $opt_req) {
	    $mandatory_failed++;
	}
	&Apache::lonxml::debug("all instance ".join(':',@{$dimension{$dim}{$instance.'.criterias'}})." results -> m_f $mandatory_failed o_p $optional_passed u $ungraded r $review");
	if ($review) {
	    $Apache::lonhomework::results{"resource.$version.0.$dim.status"}=
		'review';
	} elsif ($ungraded) {
	    $Apache::lonhomework::results{"resource.$version.0.$dim.status"}=
		'ungraded';
	} elsif ($mandatory_failed) {
	    $Apache::lonhomework::results{"resource.$version.0.$dim.status"}=
		'fail';
	} else {
	    $Apache::lonhomework::results{"resource.$version.0.$dim.status"}=
		'pass';
	}
    } elsif ($target eq 'edit') {
    } elsif ($target eq 'modified') {
    } else {
	# any other targets no output
	undef($result);
    }
    if ($target eq 'grade' || $target eq 'web' || $target eq 'webgrade') {
	&disable_dimension_parsing();
	pop(@Apache::bridgetask::dimension);
    }
    return $result;
}

sub question_status_message {
    my ($counts,$depth) = @_;
    my %req  = ('man' => 'mandatory',
		'opt' => 'optional',);
    my %type = ('cri' => 'criteria',
		'dim' => ('sub'x($depth+1)).'questions',);
    my @sections;
    foreach my $req ('man','opt') {
	foreach my $type ('cri','dim') {
	    if ($counts->{$req.'_'.$type}) {
		push(@sections,
		     $counts->{$req.'_'.$type.'_passed'}.' of the '.
		     $counts->{$req.'_'.$type}.' '.
		     $req{$req}.' '.$type{$type});
	    }
	}
    }

    my $status = 'You passed ';
    if (@sections == -1) {
    } elsif (@sections == 1) {
	$status .= $sections[0];
    } elsif (@sections == 2) {
	$status .= $sections[0].' and '.$sections[1];
    } else {
	my $last = pop(@sections);
	$status .= join(', ',@sections).', and '.$last;
    }
    $status .= '.';
    if ($counts->{'opt'}) {
        if ($counts->{'opt_dim'} + $counts->{'man_dim'} < 1) {
            $status .= ' '.&mt('You were required to pass [quant,_1,optional criterion,optional criteria].',$counts->{'opt_req'});
        } else { 
            $status .= ' '.&mt('You were required to pass [quant,_1,optional component].',$counts->{'opt_req'});
        }
    }
    return $status;
}

sub get_counts {
    my ($dim,$instance,$parstack,$safeeval) = @_;
    my %counts;
    my @possible = ('man_cri','man_dim',
		    'opt_cri','opt_dim',
		    'man_cri_passed', 'man_dim_passed',
		    'opt_cri_passed', 'opt_dim_passed',
		    'man_passed',
		    'opt_passed',
		    'opt_req');
    foreach my $which (@possible) { $counts{$which} = 0; }

    my $version = &get_version();

    foreach my $id ( @{$dimension{$dim}{$instance.'.criterias'}},
		     @{$dimension{$dim}{'criterias'}} ) {
	my $status = &get_criteria('status',$version,$dim,$id);
	my $which;
	if ($dimension{$dim}{'criteria.'.$id.'.mandatory'} 
	    eq 'N') {
	    $which = 'opt';
	} else {
	    $which = 'man';
	}
	$counts{$which}++;
	if ($status eq 'pass') { $counts{$which.'_passed'}++; }
	if ($dimension{$dim}{'criteria.'.$id.'.type'}
	    eq 'dimension') {
	    $which .= '_dim';
	} else {
	    $which .= '_cri';
	}
	$counts{$which}++;
	if ($status eq 'pass') { $counts{$which.'_passed'}++; }


    }
    if ($counts{'man_dim_passed'} eq $counts{'man_dim'}) {
	$counts{'man_dim_passed'}='all';
    }
    if ($counts{'man_cri_passed'} eq $counts{'man_cri'}) {
	$counts{'man_cri_passed'}='all';
    }
    
    $counts{'opt_req'}=$dimension{$dim}{$instance.'.optionalrequired'};
    if ($counts{'opt_req'} !~ /\S/) {
	$counts{'opt_req'}= &Apache::lonxml::get_param('OptionalRequired',
						       $parstack,$safeeval);
	if ($counts{'opt_req'} !~ /\S/) { $counts{'opt_req'} = 0; }
    }
    return %counts;
}

sub end_Setup {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;
    my $dim=&get_id($parstack,$safeeval);
    my $instance=&get_instance($dim);
    my $version=&get_version();
    if ($target eq 'web' || $target eq 'webgrade' || $target eq 'grade') {
	$result=&Apache::lonxml::endredirection();
    }
    if ($target eq 'web') {
	@Apache::scripttag::parser_env = @_;
	$result.=&Apache::scripttag::xmlparse($dimension{$dim}{'intro'});
	my @instances = $instance;
	if (&Apache::response::showallfoils()) {
	    @instances = @{$dimension{$dim}{'instances'}};
	}
	foreach my $instance (@instances) {
	    @Apache::scripttag::parser_env = @_;
	    $result.=&Apache::scripttag::xmlparse($dimension{$dim}{$instance.'.text'});
	    @Apache::scripttag::parser_env = @_;
	    $result.=&Apache::scripttag::xmlparse($dimension{$dim}{'questiontext'});
	}
    } elsif ($target eq 'webgrade' 
	     || $target eq 'grade' && $env{'form.webgrade'}) {
	# in case of any side effects that we need
	@Apache::scripttag::parser_env = @_;
	&Apache::scripttag::xmlparse($dimension{$dim}{'intro'});
	@Apache::scripttag::parser_env = @_;
	&Apache::scripttag::xmlparse($dimension{$dim}{$instance.'.text'});
	@Apache::scripttag::parser_env = @_;
	&Apache::scripttag::xmlparse($dimension{$dim}{'questiontext'});
    } else {
	# any other targets no output
	undef($result);
    }
    pop(@Apache::bridgetask::dimension);
    return $result;
}

sub grading_history {
    my ($version,$dim,$id) = @_;
    if (!&Apache::lonnet::allowed('mgq',$env{'request.course.id'})
	&& !&Apache::lonnet::allowed('mgq',$env{'request.course.id'}.'/'.$env{'request.course.sec'})) {
	return '';
    }
    my ($result,$grader);
    my $scope="resource.$version.0.";
    $scope .= ($dim ne $top) ? "$dim.$id"
	                     : "$id";
    foreach my $t (1..$Apache::lonhomework::history{'version'}) {
	if (exists($Apache::lonhomework::history{$t.':resource.0.regrader'})) {
	    my ($gname,$gdom) = 
		split(':',$Apache::lonhomework::history{$t.':resource.0.regrader'});
	    my $fullname = &Apache::loncommon::plainname($gname,$gdom);
	    $grader = &Apache::loncommon::aboutmewrapper($fullname,
							 $gname,$gdom);
	}
	my $entry;
	if (exists($Apache::lonhomework::history{"$t:$scope.status"})) {
	    $entry.="<tt>".$Apache::lonhomework::history{"$t:$scope.status"}.'</tt>';
	}
	if (exists($Apache::lonhomework::history{"$t:$scope.comment"})) {
	    $entry.=' comment: "'.$Apache::lonhomework::history{"$t:$scope.comment"}.'"';
	}
	if ($entry) {
	    $result.= "\n\t\t<li>\n\t\t\t$grader :\n\t\t\t $entry \n\t\t</li>";
	}
    }
    if ($result) {
	return "\n\t".'<ul class="LC_GRADING_pastgrading">'.$result.
	    "\n\t".'</ul>'."\n";
    }
    return '';
}

sub start_IntroParagraph {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    my $dim = &get_dim_id();
    if ($target eq 'grade' || $target eq 'web' || $target eq 'webgrade') {
	if ($tagstack->[-2] eq 'Dimension' || $tagstack->[-2] eq 'Question' ) {
	    $dimension{$dim}{'intro'}=
		&Apache::lonxml::get_all_text('/introparagraph',
					      $parser,$style);
       	} elsif ($tagstack->[-2] eq 'Task' && $target eq 'webgrade') {
	    &Apache::lonxml::startredirection();
	}
	
    } elsif ($target eq 'edit') {
	$result = &Apache::edit::tag_start($target,$token);
    } elsif ($target eq 'modified') {
    }
    return $result;
}

sub end_IntroParagraph {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    if ($tagstack->[-2] eq 'Task' && $target eq 'webgrade') {
	my $result = &Apache::lonxml::endredirection();
    }
}

sub insert_IntroParagraph {
    return '
<IntroParagraph>
    <startouttext />
    <endouttext />
</IntroParagraph>';
}

sub start_Instance {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $dim = &get_dim_id();
    my $id  = &get_id($parstack,$safeeval);
    push(@{$dimension{$dim}{'instances'}},$id);
    push(@{$Apache::bridgetask::instance{$dim}},$id);
    push(@Apache::bridgetask::instancelist,$id);
    $dimension{$dim}{$id.'.optionalrequired'}=
	&Apache::lonxml::get_param('OptionalRequired',$parstack,$safeeval);
    my $disabled = &Apache::lonxml::get_param('Disabled',$parstack,$safeeval);
    if (lc($disabled) eq 'yes') {
	$dimension{$dim}{$id.'.disabled'}='1';
    }
    my $result;
    if ($target eq 'edit') {
	$result = &Apache::edit::tag_start($target,$token);
	$result.=  
	    &Apache::edit::text_arg('Id:','id',$token,10).' '.
	    &Apache::edit::select_arg('Instance is Disabled:','Disabled',
				      [['no', 'No'],
				       ['yes','Yes'],],
				      $token)
	    .' <br /> '.
	    &Apache::edit::text_arg('Required number of passed optional elements to pass the Instance:',
				    'OptionalRequired',$token,4)
	    .&Apache::edit::end_row().
	    &Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
	my $constructtag=
	    &Apache::edit::get_new_args($token,$parstack,$safeeval,
					'id','OptionalRequired','Disabled');
	if ($constructtag) {
	    $result = &Apache::edit::rebuild_tag($token);
	}
    }
    return $result;
}

sub end_Instance {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;
    if ($target eq 'edit') {
	$result = &Apache::edit::tag_end($target,$token);
    }
    return $result;
}

sub start_InstanceText {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'grade' || $target eq 'web' || $target eq 'webgrade') {
	my $text=&Apache::lonxml::get_all_text('/instancetext',$parser,$style);
	my $dim = &get_dim_id();
	my $instance_id=$Apache::bridgetask::instance{$dim}[-1];
	$dimension{$dim}{$instance_id.'.text'}=$text;
    } elsif ($target eq 'edit') {
	$result = &Apache::edit::tag_start($target,$token);
    } elsif ($target eq 'modified') {
    }
    return $result;
}

sub end_InstanceText {
    return '';
}

sub insert_InstanceText {
    return '
<InstanceText>
    <startouttext />
    <endouttext />
</InstanceText>';
}

sub start_Criteria {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result = '';
    if ($target eq 'web' || $target eq 'webgrade' || $target eq 'grade') {
	my $criteria=&Apache::lonxml::get_all_text('/criteria',$parser,$style);
	my $dim = &get_dim_id();
	my $id=&get_id($parstack,$safeeval);
	if ($target eq 'web' || $target eq 'webgrade') {
	    if ($target eq 'webgrade') {
		&Apache::lonxml::debug(" for $dim $id stashing results into $dim ");
		$dimension{$dim}{'result'} .= &internal_location($id);
	    } else {
		&Apache::lonxml::debug(" not stashing $dim $id");
		#$result .= &internal_location($id);
	    }
	}
	&Apache::lonxml::debug("Criteria $id with $dim");
	if (&Apache::londefdef::is_inside_of($tagstack,'Instance')) {
	    my $instance_id=$Apache::bridgetask::instance{$dim}[-1];
	    $dimension{$dim}{"criteria.$instance_id.$id"}=$criteria;
	    $dimension{$dim}{"criteria.$instance_id.$id.type"}='criteria';
	    $dimension{$dim}{"criteria.$instance_id.$id.mandatory"}=
		&Apache::lonxml::get_param('Mandatory',$parstack,$safeeval);
	    push(@{$dimension{$dim}{$instance_id.'.criterias'}},"$instance_id.$id");
	} else {
	    $dimension{$dim}{'criteria.'.$id}=$criteria;
	    $dimension{$dim}{'criteria.'.$id.'.type'}='criteria';
	    $dimension{$dim}{'criteria.'.$id.'.mandatory'}=
		&Apache::lonxml::get_param('Mandatory',$parstack,$safeeval);
	    push(@{$dimension{$dim}{'criterias'}},$id);
	}
    } elsif ($target eq 'edit') {
	$result .=&Apache::edit::tag_start($target,$token);
	$result.=  
	    &Apache::edit::text_arg('Id:','id',$token,10).' '.
	    &Apache::edit::select_arg('Passing is Mandatory:','Mandatory',
				      [['Y', 'Yes'],
				       ['N','No'],],
				      $token)
	    .' <br /> '.&Apache::edit::end_row().
	    &Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
	my $constructtag=
	    &Apache::edit::get_new_args($token,$parstack,$safeeval,
					'id','Mandatory');
	if ($constructtag) { $result = &Apache::edit::rebuild_tag($token); }
    }
    return $result;
}

sub layout_web_Criteria {
    my ($dim,$id,$criteria) = @_;

    my $version = &get_version();
    my $status= &get_criteria('status', $version,$dim,$id);
    my $comment=&get_criteria('comment',$version,$dim,$id);
    my $mandatory=($dimension{$dim}{'criteria.'.$id.'.mandatory'} ne 'N');
    if ($mandatory) {
	$mandatory='Mandatory';
    } else {
	$mandatory='Optional';
    }
    my $status_display=$status;
    $status_display=~s/^([a-z])/uc($1)/e;
    my $criteria_info.=
	'<div class="LC_'.$status.' LC_criteria">'."\n\t".'<h4>'
	.$mandatory.' Criteria</h4>'."\n\t".'<p class="LC_criteria_text">'
	."\n";
    $criteria =~ s/^\s*//s;
    $criteria =~ s/\s*$//s;
    $criteria_info.= $criteria;
    $criteria_info.="\n\t".'</p>'.
	"\n\t".'<p class="LC_grade">'.$status_display.'</p>';
    if ($comment =~ /\w/) {
	$criteria_info.=
	    "\n\t".
	    '<p class="LC_comment">'.&mt('Comment: [_1]',$comment).'</p>';
    }
    $criteria_info.="\n".'</div>'."\n";
    
    return $criteria_info;
}

sub layout_webgrade_Criteria {
    my ($dim,$id,$criteria) = @_;
    my $link=&link($id);
    my $version = &get_version();
    my $status  = &get_criteria('status',$version,$dim,$id);
    my %lt = ( 'ungraded' => 'Ungraded',
	       'fail'     => 'Fail',
	       'pass'     => 'Pass',
	       'review'   => 'Review',
	       'comment'  => 'Additional Comment for Student',
	       );
    %lt = &Apache::lonlocal::texthash(%lt);
    my $comment = &get_criteria('comment',$version,$dim,$id);
    $comment = &HTML::Entities::encode($comment,'<>"&');
    my %checked;
    foreach my $which ('ungraded','fail','pass','review') {
	if ($status eq $which) { $checked{$which} = 'checked="checked"'; }
    }
    if (!%checked) { $checked{'ungraded'} = 'checked="checked"'; }
    my $buttons;
    foreach my $which  ('ungraded','fail','pass','review') {
	$buttons .= <<END_BUTTON;
		<label class="LC_GRADING_$which">
			<input type="radio" name="HWVAL_$link" value="$which" $checked{$which} />
			$lt{$which}
		</label>
END_BUTTON
    }
    $criteria =~ s/^\s*//s;
    $criteria =~ s/\s*$//s;
    my $result = <<END_CRITERIA;
<div class="LC_GRADING_criteria">
	<div class="LC_GRADING_criteriatext">
		$criteria
	</div>
	<div class="LC_GRADING_grade">
$buttons
	</div>
	<label class="LC_GRADING_comment">
		$lt{'comment'}
		<textarea class="LC_GRADING_comment_area" name="HWVAL_comment_$link">$comment</textarea>
	</label>
</div>
END_CRITERIA
    $result .= &grading_history($version,$dim,$id);
    return $result;
}

sub end_Criteria {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    if ($target eq 'edit') {
    } elsif ($target eq 'modified') {
    }
}
sub insert_Criteria {
    return '
<Criteria>
    <CriteriaText>
        <startouttext />
        <endouttext />
    </CriteriaText>
</Criteria>';
}

sub start_CriteriaText {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'grade' || $target eq 'web' || $target eq 'webgrade') {
	
    } elsif ($target eq 'edit') {
	$result = &Apache::edit::tag_start($target,$token);
    } elsif ($target eq 'modified') {
    }
    return $result;
}

sub end_CriteriaText {
    return '';
}

sub insert_CriteriaText {
    return '
<CriteriaText>
    <startouttext />
    <endouttext />
</CriteriaText>';
}

sub start_GraderNote {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'webgrade') {
	$result = '<div class="LC_GRADING_gradernote"><b>'.
	    &mt('Note to graders:').'</b>';
    } elsif ($target eq 'edit') {
	$result = &Apache::edit::tag_start($target,$token);
    } elsif ($target eq 'modified') {
    } elsif ($target eq 'web' || $target eq 'grade') {
	my $note=&Apache::lonxml::get_all_text('/gradernote',$parser,$style); 
    }
    return $result;
}

sub end_GraderNote {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;

    if ($target eq 'webgrade') {
	return '</div>';
    }
    return;
}

sub insert_GraderNote {
    return '
<GraderNote>
    <startouttext />
    <endouttext />
</GraderNote>';
}


sub proctor_validation_screen {
    my ($slot) = @_;
    my (undef,undef,$domain,$user) = &Apache::lonnet::whichuser();
    my $url=&Apache::lonnet::studentphoto($domain,$user,'jpg');
    if ($url ne '/adm/lonKaputt/lonlogo_broken.gif') {
	$url = "<tr><td colspan=\"2\"><img src=\"$url\" /></td></tr>";
    } else {
	undef($url);
    }

    my $name=&Apache::loncommon::plainname($user,$domain);
    
    my $msg;
    if ($env{'form.proctorpassword'}) {
	$msg.='<p><span class="LC_warning">'
	    .&mt("Failed to authenticate the proctor.")
	    .'</span></p>';
    }

    my $valid;
    my @possible_proctors=split(",",$slot->{'proctor'});
    foreach my $proctor (@possible_proctors) {
	if ($proctor =~ /$LONCAPA::username_re:$LONCAPA::domain_re/) {
	    $valid = 1;
	    last;
	}
    }
    if (!$valid) {
	$msg.='<p><span class="LC_error">'
	    .&mt("No valid proctors are defined.")
	    .'</span></p>';
    }
    
    if (!$env{'form.proctordomain'}) { $env{'form.proctordomain'}=$domain; }
    my $uri = &Apache::lonenc::check_encrypt($env{'request.uri'});
    $uri = &HTML::Entities::encode($uri,'<>&"');
    my %lt = &Apache::lonlocal::texthash(
                            'prva' => "Proctor Validation",
                            'yoro' => "Your room's proctor needs to validate your access to this resource.",
                            'prus'  => "Proctor's Username:",
                            'pasw'  => "Password:",
                            'prdo'  => "Proctor's Domain:",
                            'vali'  => 'Validate',
                            'stui'  => "Student who should be logged in is:",
                            'name'  => "Name:",
                            'sid'   => "Student ID:",
                            'unam'  => "Username:",
                           );
    my $result= (<<ENDCHECKOUT);
<h2>$lt{'prva'}</h2>
    <p>$lt{'yoro'}</p>
    $msg
<form name="checkout" method="post" action="$uri">
<input type="hidden" name="validate" value="yes" />
<input type="hidden" name="submitted" value="yes" />
<table>
  <tr><td>$lt{'prus'}</td><td><input type="string" name="proctorname" value="$env{'form.proctorname'}" /></td></tr>
  <tr><td>$lt{'pasw'}</td><td><input type="password" name="proctorpassword" value="" /></td></tr>
  <tr><td>$lt{'prdo'}</td><td><input type="string" name="proctordomain" value="$env{'form.proctordomain'}" /></td></tr>
</table>
<input type="submit" name="checkoutbutton" value="$lt{'vali'}"  /><br />
<table border="1">
  <tr><td>
    <table>
      <tr><td colspan="2">$lt{'stui'}</td></tr>
      <tr><td>$lt{'name'}</td><td>$name</td></tr>
      <tr><td>$lt{'sid'}</td><td>$env{'environment.id'}</td></tr>
      <tr><td>$lt{'unam'}</td><td>$user:$domain</td></tr>
      $url
    </table>
  </tr></td>
</table>
</form>
ENDCHECKOUT

    return $result;
}

1;
__END__
