# The LearningOnline Network with CAPA
# Create a user
#
# $Id: loncreateuser.pm,v 1.295.2.34 2010/05/22 13:30:43 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###

package Apache::loncreateuser;

=pod

=head1 NAME

Apache::loncreateuser.pm

=head1 SYNOPSIS

    Handler to create users and custom roles

    Provides an Apache handler for creating users,
    editing their login parameters, roles, and removing roles, and
    also creating and assigning custom roles.

=head1 OVERVIEW

=head2 Custom Roles

In LON-CAPA, roles are actually collections of privileges. "Teaching
Assistant", "Course Coordinator", and other such roles are really just
collection of privileges that are useful in many circumstances.

Custom roles can be defined by a Domain Coordinator, Course Coordinator
or Community Coordinator via the Manage User functionality.
The custom role editor screen will show all privileges which can be
assigned to users. For a complete list of privileges, please see 
C</home/httpd/lonTabs/rolesplain.tab>.

Custom role definitions are stored in the C<roles.db> file of the creator
of the role.

=cut

use strict;
use Apache::Constants qw(:common :http);
use Apache::lonnet;
use Apache::loncommon;
use Apache::lonlocal;
use Apache::longroup;
use Apache::lonuserutils;
use Apache::loncoursequeueadmin;
use LONCAPA qw(:DEFAULT :match);

my $loginscript; # piece of javascript used in two separate instances
my $authformnop;
my $authformkrb;
my $authformint;
my $authformfsys;
my $authformloc;

sub initialize_authen_forms {
    my ($dom,$formname,$curr_authtype,$mode) = @_;
    my ($krbdef,$krbdefdom) = &Apache::loncommon::get_kerberos_defaults($dom);
    my %param = ( formname => $formname,
                  kerb_def_dom => $krbdefdom,
                  kerb_def_auth => $krbdef,
                  domain => $dom,
                );
    my %abv_auth = &auth_abbrev();
    if ($curr_authtype =~ /^(krb4|krb5|internal|localauth|unix):(.*)$/) {
        my $long_auth = $1;
        my $curr_autharg = $2;
        my %abv_auth = &auth_abbrev();
        $param{'curr_authtype'} = $abv_auth{$long_auth};
        if ($long_auth =~ /^krb(4|5)$/) {
            $param{'curr_kerb_ver'} = $1;
            $param{'curr_autharg'} = $curr_autharg;
        }
        if ($mode eq 'modifyuser') {
            $param{'mode'} = $mode;
        }
    }
    $loginscript  = &Apache::loncommon::authform_header(%param);
    $authformkrb  = &Apache::loncommon::authform_kerberos(%param);
    $authformnop  = &Apache::loncommon::authform_nochange(%param);
    $authformint  = &Apache::loncommon::authform_internal(%param);
    $authformfsys = &Apache::loncommon::authform_filesystem(%param);
    $authformloc  = &Apache::loncommon::authform_local(%param);
}

sub auth_abbrev {
    my %abv_auth = (
                     krb5     => 'krb',
                     krb4     => 'krb',
                     internal => 'int',
                     localuth => 'loc',
                     unix     => 'fsys',
                   );
    return %abv_auth;
}

# ====================================================

sub portfolio_quota {
    my ($ccuname,$ccdomain) = @_;
    my %lt = &Apache::lonlocal::texthash(
                   'usrt'      => "User Tools",
                   'disk'      => "Disk space allocated to user's portfolio files",
                   'cuqu'      => "Current quota",
                   'cust'      => "Custom quota",
                   'defa'      => "Default",
                   'chqu'      => "Change quota",
    );
    my ($currquota,$quotatype,$inststatus,$defquota) = 
        &Apache::loncommon::get_user_quota($ccuname,$ccdomain);
    my ($usertypes,$order) = &Apache::lonnet::retrieve_inst_usertypes($ccdomain);
    my ($longinsttype,$showquota,$custom_on,$custom_off,$defaultinfo);
    if ($inststatus ne '') {
        if ($usertypes->{$inststatus} ne '') {
            $longinsttype = $usertypes->{$inststatus};
        }
    }
    $custom_on = ' ';
    $custom_off = ' checked="checked" ';
    my $quota_javascript = <<"END_SCRIPT";
<script type="text/javascript">
// <![CDATA[
function quota_changes(caller) {
    if (caller == "custom") {
        if (document.cu.customquota[0].checked) {
            document.cu.portfolioquota.value = "";
        }
    }
    if (caller == "quota") {
        document.cu.customquota[1].checked = true;
    }
}
// ]]>
</script>
END_SCRIPT
    if ($quotatype eq 'custom') {
        $custom_on = $custom_off;
        $custom_off = ' ';
        $showquota = $currquota;
        if ($longinsttype eq '') {
            $defaultinfo = &mt('For this user, the default quota would be [_1]'
                            .' Mb.',$defquota);
        } else {
            $defaultinfo = &mt("For this user, the default quota would be [_1]".
                               " Mb, as determined by the user's institutional".
                               " affiliation ([_2]).",$defquota,$longinsttype);
        }
    } else {
        if ($longinsttype eq '') {
            $defaultinfo = &mt('For this user, the default quota is [_1]'
                            .' Mb.',$defquota);
        } else {
            $defaultinfo = &mt("For this user, the default quota of [_1]".
                               " Mb, is determined by the user's institutional".
                               " affiliation ([_2]).",$defquota,$longinsttype);
        }
    }

    my $output = $quota_javascript."\n".
                 '<h3>'.$lt{'usrt'}.'</h3>'."\n".
                 &Apache::loncommon::start_data_table();

    if (&Apache::lonnet::allowed('mut',$ccdomain)) {
        $output .= &build_tools_display($ccuname,$ccdomain,'tools');
    }
    if (&Apache::lonnet::allowed('mpq',$ccdomain)) {
        $output .= '<tr class="LC_info_row">'."\n".
                   '    <td>'.$lt{'disk'}.'</td>'."\n".
                   '  </tr>'."\n".
                   &Apache::loncommon::start_data_table_row()."\n".
                   '  <td>'.$lt{'cuqu'}.': '.
                   $currquota.'&nbsp;Mb.&nbsp;&nbsp;'.
                   $defaultinfo.'</td>'."\n".
                   &Apache::loncommon::end_data_table_row()."\n".
                   &Apache::loncommon::start_data_table_row()."\n".
                   '  <td><span class="LC_nobreak">'.$lt{'chqu'}.
                   ': <label>'.
                   '<input type="radio" name="customquota" value="0" '.
                   $custom_off.' onchange="javascript:quota_changes('."'custom'".')"'.
                   ' />'.$lt{'defa'}.'&nbsp;('.$defquota.' Mb).</label>&nbsp;'.
                   '&nbsp;<label><input type="radio" name="customquota" value="1" '. 
                   $custom_on.'  onchange="javascript:quota_changes('."'custom'".')" />'.
                   $lt{'cust'}.':</label>&nbsp;'.
                   '<input type="text" name="portfolioquota" size ="5" value="'.
                   $showquota.'" onfocus="javascript:quota_changes('."'quota'".')" '.
                   '/>&nbsp;Mb</span></td>'."\n".
                   &Apache::loncommon::end_data_table_row()."\n";
    }  
    $output .= &Apache::loncommon::end_data_table();
    return $output;
}

sub build_tools_display {
    my ($ccuname,$ccdomain,$context) = @_;
    my (@usertools,%userenv,$output,@options,%validations,%reqtitles,%reqdisplay,
        $colspan,$isadv,%domconfig);
    my %lt = &Apache::lonlocal::texthash (
                   'blog'       => "Personal User Blog",
                   'aboutme'    => "Personal Information Page",
                   'portfolio'  => "Personal User Portfolio",
                   'avai'       => "Available",
                   'cusa'       => "availability",
                   'chse'       => "Change setting",
                   'usde'       => "Use default",
                   'uscu'       => "Use custom",
                   'official'   => 'Can request creation of official courses',
                   'unofficial' => 'Can request creation of unofficial courses',
                   'community'  => 'Can request creation of communities',
    );
    if ($context eq 'requestcourses') {
        %userenv = &Apache::lonnet::userenvironment($ccdomain,$ccuname,
                      'requestcourses.official','requestcourses.unofficial',
                      'requestcourses.community');
        @usertools = ('official','unofficial','community');
        @options =('norequest','approval','autolimit','validate');
        %validations = &Apache::lonnet::auto_courserequest_checks($ccdomain);
        %reqtitles = &courserequest_titles();
        %reqdisplay = &courserequest_display();
        $colspan = ' colspan="2"';
        %domconfig =
            &Apache::lonnet::get_dom('configuration',['requestcourses'],$ccdomain);
        $isadv = &Apache::lonnet::is_advanced_user($ccuname,$ccdomain);
    } else {
        %userenv = &Apache::lonnet::userenvironment($ccdomain,$ccuname,
                          'tools.aboutme','tools.portfolio','tools.blog');
        @usertools = ('aboutme','blog','portfolio');
    }
    foreach my $item (@usertools) {
        my ($custom_access,$curr_access,$cust_on,$cust_off,$tool_on,$tool_off,
            $currdisp,$custdisp,$custradio);
        $cust_off = 'checked="checked" ';
        $tool_on = 'checked="checked" ';
        $curr_access =  
            &Apache::lonnet::usertools_access($ccuname,$ccdomain,$item,undef,
                                              $context);
        if ($userenv{$context.'.'.$item} ne '') {
            $cust_on = ' checked="checked" ';
            $cust_off = '';
        }
        if ($context eq 'requestcourses') {
            if ($userenv{$context.'.'.$item} eq '') {
                $custom_access = &mt('Currently from default setting.');
            } else {
                $custom_access = &mt('Currently from custom setting.');
            }
        } else {
            if ($userenv{$context.'.'.$item} eq '') {
                $custom_access =
                    &mt('Availability determined currently from default setting.');
                if (!$curr_access) {
                    $tool_off = 'checked="checked" ';
                    $tool_on = '';
                }
            } else {
                $custom_access =
                    &mt('Availability determined currently from custom setting.');
                if ($userenv{$context.'.'.$item} == 0) {
                    $tool_off = 'checked="checked" ';
                    $tool_on = '';
                }
            }
        }
        $output .= '  <tr class="LC_info_row">'."\n".
                   '   <td'.$colspan.'>'.$lt{$item}.'</td>'."\n".
                   '  </tr>'."\n".
                   &Apache::loncommon::start_data_table_row()."\n";
        if ($context eq 'requestcourses') {
            my ($curroption,$currlimit);
            if ($userenv{$context.'.'.$item} ne '') {
                $curroption = $userenv{$context.'.'.$item};
            } else {
                my (@inststatuses);
                $curroption =
                    &Apache::loncoursequeueadmin::get_processtype($ccuname,$ccdomain,$isadv,$ccdomain,
                                                               $item,\@inststatuses,\%domconfig);
            }
            if (!$curroption) {
                $curroption = 'norequest';
            }
            if ($curroption =~ /^autolimit=(\d*)$/) {
                $currlimit = $1;
                if ($currlimit eq '') {
                    $currdisp = &mt('Yes, automatic creation');
                } else {
                    $currdisp = &mt('Yes, up to [quant,_1,request]/user',$currlimit);
                }
            } else {
                $currdisp = $reqdisplay{$curroption};
            }
            $custdisp = '<table>';
            foreach my $option (@options) {
                my $val = $option;
                if ($option eq 'norequest') {
                    $val = 0;
                }
                if ($option eq 'validate') {
                    my $canvalidate = 0;
                    if (ref($validations{$item}) eq 'HASH') {
                        if ($validations{$item}{'_custom_'}) {
                            $canvalidate = 1;
                        }
                    }
                    next if (!$canvalidate);
                }
                my $checked = '';
                if ($option eq $curroption) {
                    $checked = ' checked="checked"';
                } elsif ($option eq 'autolimit') {
                    if ($curroption =~ /^autolimit/) {
                        $checked = ' checked="checked"';
                    }
                }
                $custdisp .= '<tr><td><span class="LC_nobreak"><label>'.
                             '<input type="radio" name="crsreq_'.$item.
                             '" value="'.$val.'"'.$checked.' />'.
                             $reqtitles{$option}.'</label>&nbsp;';
                if ($option eq 'autolimit') {
                    $custdisp .= '<input type="text" name="crsreq_'.
                                 $item.'_limit" size="1" '.
                                 'value="'.$currlimit.'" /></span><br />'.
                                 $reqtitles{'unlimited'};
                 } else {
                     $custdisp .= '</span>';
                 }
                 $custdisp .= '</td></tr>';
            }
            $custdisp .= '</table>';
            $custradio = '</span></td><td>'.&mt('Custom setting').'<br />'.$custdisp;
        } else {
            $currdisp = ($curr_access?&mt('Yes'):&mt('No'));
            $custdisp = '<span class="LC_nobreak"><label>'.
                        '<input type="radio" name="'.$context.'_'.$item.'"'.
                        ' value="1"'. $tool_on.'/>'.&mt('On').'</label>&nbsp;<label>'.
                        '<input type="radio" name="'.$context.'_'.$item.'" value="0" '.
                        $tool_off.'/>'.&mt('Off').'</label></span>';
            $custradio = ('&nbsp;'x2).'--'.$lt{'cusa'}.':&nbsp;'.$custdisp.
                          '</span>';
        }
        $output .= '  <td'.$colspan.'>'.$custom_access.('&nbsp;'x4).
                   $lt{'avai'}.': '.$currdisp.'</td>'."\n".
                   &Apache::loncommon::end_data_table_row()."\n".
                   &Apache::loncommon::start_data_table_row()."\n".
                   '  <td style="vertical-align:top;"><span class="LC_nobreak">'.
                   $lt{'chse'}.': <label>'.
                   '<input type="radio" name="custom'.$item.'" value="0" '.
                   $cust_off.'/>'.$lt{'usde'}.'</label>'.('&nbsp;' x3).
                   '<label><input type="radio" name="custom'.$item.'" value="1" '.
                   $cust_on.'/>'.$lt{'uscu'}.'</label>'.$custradio.'</td>'.
                   &Apache::loncommon::end_data_table_row()."\n";
    }
    return $output;
}

sub coursereq_externaluser {
    my ($ccuname,$ccdomain,$cdom) = @_;
    my (@usertools,@options,%validations,%userenv,$output);
    my %lt = &Apache::lonlocal::texthash (
                   'official'   => 'Can request creation of official courses',
                   'unofficial' => 'Can request creation of unofficial courses',
                   'community'  => 'Can request creation of communities',
    );

    %userenv = &Apache::lonnet::userenvironment($ccdomain,$ccuname,
                      'reqcrsotherdom.official','reqcrsotherdom.unofficial',
                      'reqcrsotherdom.community');
    @usertools = ('official','unofficial','community');
    @options = ('approval','validate','autolimit');
    %validations = &Apache::lonnet::auto_courserequest_checks($cdom);
    my $optregex = join('|',@options);
    my %reqtitles = &courserequest_titles();
    foreach my $item (@usertools) {
        my ($curroption,$currlimit,$tooloff);
        if ($userenv{'reqcrsotherdom.'.$item} ne '') {
            my @curr = split(',',$userenv{'reqcrsotherdom.'.$item});
            foreach my $req (@curr) {
                if ($req =~ /^\Q$cdom\E\:($optregex)=?(\d*)$/) {
                    $curroption = $1;
                    $currlimit = $2;
                    last;
                }
            }
            if (!$curroption) {
                $curroption = 'norequest';
                $tooloff = ' checked="checked"';
            }
        } else {
            $curroption = 'norequest';
            $tooloff = ' checked="checked"';
        }
        $output.= &Apache::loncommon::start_data_table_row()."\n".
                  '  <td><span class="LC_nobreak">'.$lt{$item}.': </span></td><td>'.
                  '<table><tr><td valign="top">'."\n".
                  '<label><input type="radio" name="reqcrsotherdom_'.$item.
                  '" value=""'.$tooloff.' />'.$reqtitles{'norequest'}.
                  '</label></td>';
        foreach my $option (@options) {
            if ($option eq 'validate') {
                my $canvalidate = 0;
                if (ref($validations{$item}) eq 'HASH') {
                    if ($validations{$item}{'_external_'}) {
                        $canvalidate = 1;
                    }
                }
                next if (!$canvalidate);
            }
            my $checked = '';
            if ($option eq $curroption) {
                $checked = ' checked="checked"';
            }
            $output .= '<td valign="top"><span class="LC_nobreak"><label>'.
                       '<input type="radio" name="reqcrsotherdom_'.$item.
                       '" value="'.$option.'"'.$checked.' />'.
                       $reqtitles{$option}.'</label>';
            if ($option eq 'autolimit') {
                $output .= '&nbsp;<input type="text" name="reqcrsotherdom_'.
                           $item.'_limit" size="1" '.
                           'value="'.$currlimit.'" /></span>'.
                           '<br />'.$reqtitles{'unlimited'};
            } else {
                $output .= '</span>';
            }
            $output .= '</td>';
        }
        $output .= '</td></tr></table></td>'."\n".
                   &Apache::loncommon::end_data_table_row()."\n";
    }
    return $output;
}

sub courserequest_titles {
    my %titles = &Apache::lonlocal::texthash (
                                   official   => 'Official',
                                   unofficial => 'Unofficial',
                                   community  => 'Communities',
                                   norequest  => 'Not allowed',
                                   approval   => 'Approval by Dom. Coord.',
                                   validate   => 'With validation',
                                   autolimit  => 'Numerical limit',
                                   unlimited  => '(blank for unlimited)',
                 );
    return %titles;
}

sub courserequest_display {
    my %titles = &Apache::lonlocal::texthash (
                                   approval   => 'Yes, need approval',
                                   validate   => 'Yes, with validation',
                                   norequest  => 'No',
   );
   return %titles;
}

# =================================================================== Phase one

sub print_username_entry_form {
    my ($r,$context,$response,$srch,$forcenewuser,$crstype) = @_;
    my $defdom=$env{'request.role.domain'};
    my $formtoset = 'crtuser';
    if (exists($env{'form.startrolename'})) {
        $formtoset = 'docustom';
        $env{'form.rolename'} = $env{'form.startrolename'};
    } elsif ($env{'form.origform'} eq 'crtusername') {
        $formtoset =  $env{'form.origform'};
    }

    my ($jsback,$elements) = &crumb_utilities();

    my $jscript = &Apache::loncommon::studentbrowser_javascript()."\n".
        '<script type="text/javascript">'."\n".
        '// <![CDATA['."\n".
        &Apache::lonhtmlcommon::set_form_elements($elements->{$formtoset})."\n".
        '// ]]>'."\n".
        '</script>'."\n";

    my %existingroles=&Apache::lonuserutils::my_custom_roles($crstype);
    if (($env{'form.action'} eq 'custom') && (keys(%existingroles) > 0)
        && (&Apache::lonnet::allowed('mcr','/'))) {
        $jscript .= &customrole_javascript();
    }
    my %loaditems = (
                'onload' => "javascript:setFormElements(document.$formtoset)",
                    );
    my %breadcrumb_text = &singleuser_breadcrumb($crstype);
    my $start_page =
	&Apache::loncommon::start_page('User Management',
				       $jscript,{'add_entries' => \%loaditems,});
    if ($env{'form.action'} eq 'custom') {
        &Apache::lonhtmlcommon::add_breadcrumb
          ({href=>"javascript:backPage(document.crtuser)",
            text=>"Pick custom role",});
    } else {
        &Apache::lonhtmlcommon::add_breadcrumb
          ({href=>"javascript:backPage(document.crtuser)",
            text=>$breadcrumb_text{'search'},
            faq=>282,bug=>'Instructor Interface',});
    }
    my $helpitem = 'Course_Change_Privileges';
    if ($env{'form.action'} eq 'custom') {
        $helpitem = 'Course_Editing_Custom_Roles';
    } elsif ($env{'form.action'} eq 'singlestudent') {
        $helpitem = 'Course_Add_Student';
    }
    my $crumbs = &Apache::lonhtmlcommon::breadcrumbs('User Management',
                                                     $helpitem);
    my %lt=&Apache::lonlocal::texthash(
                    'srst' => 'Search for a user and enroll as a student',
                    'srme' => 'Search for a user and enroll as a member',
                    'srad' => 'Search for a user and modify/add user information or roles',
		    'usr'  => "Username",
                    'dom'  => "Domain",
                    'ecrp' => "Define or Edit Custom Role",
                    'nr'   => "role name",
                    'cre'  => "Next",
				       );
    $r->print($start_page."\n".$crumbs);
    if ($env{'form.action'} eq 'custom') {
        if (&Apache::lonnet::allowed('mcr','/')) {
            my $newroletext = &mt('Define new custom role:');
            $r->print('<form action="/adm/createuser" method="post" name="docustom">'.
                      '<input type="hidden" name="action" value="'.$env{'form.action'}.'" />'.
                      '<input type="hidden" name="phase" value="selected_custom_edit" />'.
                      '<h3>'.$lt{'ecrp'}.'</h3>'.
                      &Apache::loncommon::start_data_table().
                      &Apache::loncommon::start_data_table_row().
                      '<td>');
            if (keys(%existingroles) > 0) {
                $r->print('<br /><label><input type="radio" name="customroleaction" value="new" checked="checked" onclick="setCustomFields();" /><b>'.$newroletext.'</b></label>');
            } else {
                $r->print('<br /><input type="hidden" name="customroleaction" value="new" /><b>'.$newroletext.'</b>');
            }
            $r->print('</td><td align="center">'.$lt{'nr'}.'<br /><input type="text" size="15" name="newrolename" onfocus="setCustomAction('."'new'".');" /></td>'.
                      &Apache::loncommon::end_data_table_row());
            if (keys(%existingroles) > 0) {
                $r->print(&Apache::loncommon::start_data_table_row().'<td><br />'.
                          '<label><input type="radio" name="customroleaction" value="edit" onclick="setCustomFields();"/><b>'.
                          &mt('View/Modify existing role:').'</b></label></td>'.
                          '<td align="center"><br />'.
                          '<select name="rolename" onchange="setCustomAction('."'edit'".');">'.
                          '<option value="" selected="selected">'.
                          &mt('Select'));
                foreach my $role (sort(keys(%existingroles))) {
                    $r->print('<option value="'.$role.'">'.$role.'</option>');
                }
                $r->print('</select>'.
                          '</td>'.
                          &Apache::loncommon::end_data_table_row());
            }
            $r->print(&Apache::loncommon::end_data_table().'<p>'.
                      '<input name="customeditor" type="submit" value="'.
                      $lt{'cre'}.'" /></p>'.
                      '</form>');
        }
    } else {
        my $actiontext = $lt{'srad'};
        if ($env{'form.action'} eq 'singlestudent') {
            if ($crstype eq 'Community') {
                $actiontext = $lt{'srme'};
            } else {
                $actiontext = $lt{'srst'};
            }
        }
        $r->print("<h3>$actiontext</h3>");
        if ($env{'form.origform'} ne 'crtusername') {
            $r->print("\n".$response);
        }
        $r->print(&entry_form($defdom,$srch,$forcenewuser,$context,$response,$crstype));
    }
    $r->print(&Apache::loncommon::end_page());
}

sub customrole_javascript {
    my $js = <<"END";
<script type="text/javascript">
// <![CDATA[

function setCustomFields() {
    if (document.docustom.customroleaction.length > 0) {
        for (var i=0; i<document.docustom.customroleaction.length; i++) {
            if (document.docustom.customroleaction[i].checked) {
                if (document.docustom.customroleaction[i].value == 'new') {
                    document.docustom.rolename.selectedIndex = 0;
                } else {
                    document.docustom.newrolename.value = '';
                }
            }
        }
    }
    return;
}

function setCustomAction(caller) {
    if (document.docustom.customroleaction.length > 0) {
        for (var i=0; i<document.docustom.customroleaction.length; i++) {
            if (document.docustom.customroleaction[i].value == caller) {
                document.docustom.customroleaction[i].checked = true;
            }
        }
    }
    setCustomFields();
    return;
}

// ]]>
</script>
END
    return $js;
}

sub entry_form {
    my ($dom,$srch,$forcenewuser,$context,$responsemsg,$crstype) = @_;
    my %domconf = &Apache::lonnet::get_dom('configuration',['usercreation'],$dom);
    my ($usertype,$inexact);
    if (ref($srch) eq 'HASH') {
        if (($srch->{'srchin'} eq 'dom') &&
            ($srch->{'srchby'} eq 'uname') &&
            ($srch->{'srchtype'} eq 'exact') &&
            ($srch->{'srchdomain'} ne '') &&
            ($srch->{'srchterm'} ne '')) {
            my ($rules,$ruleorder) =
                &Apache::lonnet::inst_userrules($srch->{'srchdomain'},'username');
            $usertype = &Apache::lonuserutils::check_usertype($srch->{'srchdomain'},$srch->{'srchterm'},$rules);
        } else {
            $inexact = 1;
        }
    }
    my $cancreate =
        &Apache::lonuserutils::can_create_user($dom,$context,$usertype);
    my $userpicker = 
       &Apache::loncommon::user_picker($dom,$srch,$forcenewuser,
                                       'document.crtuser',$cancreate,$usertype);
    my $srchbutton = &mt('Search');
    if ($env{'form.action'} eq 'singlestudent') {
        $srchbutton = &mt('Search and Enroll');
    } elsif ($cancreate && $responsemsg ne '' && $inexact) {
        $srchbutton = &mt('Search or Add New User');
    }
    my $output = <<"ENDBLOCK";
<form action="/adm/createuser" method="post" name="crtuser">
<input type="hidden" name="action" value="$env{'form.action'}" />
<input type="hidden" name="phase" value="get_user_info" />
$userpicker
<input name="userrole" type="button" value="$srchbutton" onclick="javascript:validateEntry(document.crtuser)" />
</form>
ENDBLOCK
    if ($env{'form.phase'} eq '') {
        my $defdom=$env{'request.role.domain'};
        my $domform = &Apache::loncommon::select_dom_form($defdom,'srchdomain');
        my %lt=&Apache::lonlocal::texthash(
                  'enro' => 'Enroll one student',
                  'enrm' => 'Enroll one member',
                  'admo' => 'Add/modify a single user',
                  'crea' => 'create new user if required',
                  'uskn' => "username is known",
                  'crnu' => 'Create a new user',
                  'usr'  => 'Username',
                  'dom'  => 'in domain',
                  'enrl' => 'Enroll',
                  'cram'  => 'Create/Modify user',
        );
        my $sellink=&Apache::loncommon::selectstudent_link('crtusername','srchterm','srchdomain');
        my ($title,$buttontext,$showresponse);
        if ($env{'form.action'} eq 'singlestudent') {   
            if ($crstype eq 'Community') {
                $title = $lt{'enrm'};
            } else {
                $title = $lt{'enro'};
            }
            $buttontext = $lt{'enrl'};
        } else {
            $title = $lt{'admo'};
            $buttontext = $lt{'cram'};
        }
        if ($cancreate) {
            $title .= ' <span class="LC_cusr_subheading">('.$lt{'crea'}.')</span>';
        } else {
            $title .= ' <span class="LC_cusr_subheading">('.$lt{'uskn'}.')</span>';
        }
        if ($env{'form.origform'} eq 'crtusername') {
            $showresponse = $responsemsg;
        }
        $output .= <<"ENDDOCUMENT";
<br />
<form action="/adm/createuser" method="post" name="crtusername">
<input type="hidden" name="action" value="$env{'form.action'}" />
<input type="hidden" name="phase" value="createnewuser" />
<input type="hidden" name="srchtype" value="exact" />
<input type="hidden" name="srchby" value="uname" />
<input type="hidden" name="srchin" value="dom" />
<input type="hidden" name="forcenewuser" value="1" />
<input type="hidden" name="origform" value="crtusername" />
<h3>$title</h3>
$showresponse
<table>
 <tr>
  <td>$lt{'usr'}:</td>
  <td><input type="text" size="15" name="srchterm" /></td>
  <td>&nbsp;$lt{'dom'}:</td><td>$domform</td>
  <td>&nbsp;$sellink&nbsp;</td>
  <td>&nbsp;<input name="userrole" type="submit" value="$buttontext" /></td>
 </tr>
</table>
</form>
ENDDOCUMENT
    }
    return $output;
}

sub user_modification_js {
    my ($pjump_def,$dc_setcourse_code,$nondc_setsection_code,$groupslist)=@_;
    
    return <<END;
<script type="text/javascript" language="Javascript">
// <![CDATA[

    function pclose() {
        parmwin=window.open("/adm/rat/empty.html","LONCAPAparms",
                 "height=350,width=350,scrollbars=no,menubar=no");
        parmwin.close();
    }

    $pjump_def
    $dc_setcourse_code

    function dateset() {
        eval("document.cu."+document.cu.pres_marker.value+
            ".value=document.cu.pres_value.value");
        pclose();
    }

    $nondc_setsection_code
// ]]>
</script>
END
}

# =================================================================== Phase two
sub print_user_selection_page {
    my ($r,$response,$srch,$srch_results,$srcharray,$context,$opener_elements,$crstype) = @_;
    my @fields = ('username','domain','lastname','firstname','permanentemail');
    my $sortby = $env{'form.sortby'};

    if (!grep(/^\Q$sortby\E$/,@fields)) {
        $sortby = 'lastname';
    }

    my ($jsback,$elements) = &crumb_utilities();

    my $jscript = (<<ENDSCRIPT);
<script type="text/javascript">
// <![CDATA[
function pickuser(uname,udom) {
    document.usersrchform.seluname.value=uname;
    document.usersrchform.seludom.value=udom;
    document.usersrchform.phase.value="userpicked";
    document.usersrchform.submit();
}

$jsback
// ]]>
</script>
ENDSCRIPT

    my %lt=&Apache::lonlocal::texthash(
                                       'usrch'          => "User Search to add/modify roles",
                                       'stusrch'        => "User Search to enroll student",
                                       'memsrch'        => "User Search to enroll member",
                                       'usel'           => "Select a user to add/modify roles",
                                       'stusel'         => "Select a user to enroll as a student",
                                       'memsel'         => "Select a user to enroll as a member",
                                       'username'       => "username",
                                       'domain'         => "domain",
                                       'lastname'       => "last name",
                                       'firstname'      => "first name",
                                       'permanentemail' => "permanent e-mail",
                                      );
    if ($context eq 'requestcrs') {
        $r->print('<div>');
    } else {
        $r->print(&Apache::loncommon::start_page('User Management',$jscript));

        my %breadcrumb_text = &singleuser_breadcrumb($crstype);
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>"javascript:backPage(document.usersrchform,'','')",
              text=>$breadcrumb_text{'search'},
              faq=>282,bug=>'Instructor Interface',},
             {href=>"javascript:backPage(document.usersrchform,'get_user_info','select')",
              text=>$breadcrumb_text{'userpicked'},
              faq=>282,bug=>'Instructor Interface',});
        if ($env{'form.action'} eq 'singleuser') {
            $r->print(&Apache::lonhtmlcommon::breadcrumbs('User Management',
                                                          'Course_Change_Privileges'));
            $r->print("<b>$lt{'usrch'}</b><br />");
            $r->print(&entry_form($srch->{'srchdomain'},$srch,undef,$context,undef,$crstype));
            $r->print('<h3>'.$lt{'usel'}.'</h3>');
        } elsif ($env{'form.action'} eq 'singlestudent') {
            $r->print(&Apache::lonhtmlcommon::breadcrumbs('User Management',
                                                          'Course_Add_Student'));
            $r->print($jscript."<b>");
            if ($crstype eq 'Community') {
                $r->print($lt{'memsrch'});
            } else {
                $r->print($lt{'stusrch'});
            }
            $r->print("</b><br />");
            $r->print(&entry_form($srch->{'srchdomain'},$srch,undef,$context,undef,$crstype));
            $r->print('</form><h3>');
            if ($crstype eq 'Community') {
                $r->print($lt{'memsel'});
            } else {
                $r->print($lt{'stusel'});
            }
            $r->print('</h3>');
        }
    }
    $r->print('<form name="usersrchform" method="post">'.
              &Apache::loncommon::start_data_table()."\n".
              &Apache::loncommon::start_data_table_header_row()."\n".
              ' <th> </th>'."\n");
    foreach my $field (@fields) {
        $r->print(' <th><a href="javascript:document.usersrchform.sortby.value='.
                  "'".$field."'".';document.usersrchform.submit();">'.
                  $lt{$field}.'</a></th>'."\n");
    }
    $r->print(&Apache::loncommon::end_data_table_header_row());

    my @sorted_users = sort {
        lc($srch_results->{$a}->{$sortby})   cmp lc($srch_results->{$b}->{$sortby})
            ||
        lc($srch_results->{$a}->{lastname})  cmp lc($srch_results->{$b}->{lastname})
            ||
        lc($srch_results->{$a}->{firstname}) cmp lc($srch_results->{$b}->{firstname})
	    ||
	lc($a) cmp lc($b)
        } (keys(%$srch_results));

    foreach my $user (@sorted_users) {
        my ($uname,$udom) = split(/:/,$user);
        my $onclick;
        if ($context eq 'requestcrs') {
            $onclick =
                'onclick="javascript:gochoose('."'$uname','$udom',".
                                               "'$srch_results->{$user}->{firstname}',".
                                               "'$srch_results->{$user}->{lastname}',".
                                               "'$srch_results->{$user}->{permanentemail}'".');"';
        } else {
            $onclick =
                ' onclick="javascript:pickuser('."'".$uname."'".','."'".$udom."'".');"';
        }
        $r->print(&Apache::loncommon::start_data_table_row().
                  '<td><input type="button" name="seluser" value="'.&mt('Select').'" '.
                  $onclick.' /></td>'.
                  '<td><tt>'.$uname.'</tt></td>'.
                  '<td><tt>'.$udom.'</tt></td>');
        foreach my $field ('lastname','firstname','permanentemail') {
            $r->print('<td>'.$srch_results->{$user}->{$field}.'</td>');
        }
        $r->print(&Apache::loncommon::end_data_table_row());
    }
    $r->print(&Apache::loncommon::end_data_table().'<br /><br />');
    if (ref($srcharray) eq 'ARRAY') {
        foreach my $item (@{$srcharray}) {
            $r->print('<input type="hidden" name="'.$item.'" value="'.$env{'form.'.$item}.'" />'."\n");
        }
    }
    $r->print(' <input type="hidden" name="sortby" value="'.$sortby.'" />'."\n".
              ' <input type="hidden" name="seluname" value="" />'."\n".
              ' <input type="hidden" name="seludom" value="" />'."\n".
              ' <input type="hidden" name="currstate" value="select" />'."\n".
              ' <input type="hidden" name="phase" value="get_user_info" />'."\n".
              ' <input type="hidden" name="action" value="'.$env{'form.action'}.'" />'."\n");
    if ($context eq 'requestcrs') {
        $r->print($opener_elements.'</form></div>');
    } else {
        $r->print($response.'</form>'.&Apache::loncommon::end_page());
    }
}

sub print_user_query_page {
    my ($r,$caller) = @_;
# FIXME - this is for a network-wide name search (similar to catalog search)
# To use frames with similar behavior to catalog/portfolio search.
# To be implemented. 
    return;
}

sub print_user_modification_page {
    my ($r,$ccuname,$ccdomain,$srch,$response,$context,$permission,$crstype) = @_;
    if (($ccuname eq '') || ($ccdomain eq '')) {
        my $usermsg = &mt('No username and/or domain provided.');
        $env{'form.phase'} = '';
	&print_username_entry_form($r,$context,$usermsg,'','',$crstype);
        return;
    }
    my ($form,$formname);
    if ($env{'form.action'} eq 'singlestudent') {
        $form = 'document.enrollstudent';
        $formname = 'enrollstudent';
    } else {
        $form = 'document.cu';
        $formname = 'cu';
    }
    my %abv_auth = &auth_abbrev();
    my (%rulematch,%inst_results,$newuser,%alerts,%curr_rules,%got_rules);
    my $uhome=&Apache::lonnet::homeserver($ccuname,$ccdomain);
    if ($uhome eq 'no_host') {
        my $usertype;
        my ($rules,$ruleorder) =
            &Apache::lonnet::inst_userrules($ccdomain,'username');
            $usertype =
                &Apache::lonuserutils::check_usertype($ccdomain,$ccuname,$rules);
        my $cancreate =
            &Apache::lonuserutils::can_create_user($ccdomain,$context,
                                                   $usertype);
        if (!$cancreate) {
            my $helplink = 'javascript:helpMenu('."'display'".')';
            my %usertypetext = (
                official   => 'institutional',
                unofficial => 'non-institutional',
            );
            my $response;
            if ($env{'form.origform'} eq 'crtusername') {
                $response =  '<span class="LC_warning">'.&mt('No match found for the username [_1] in LON-CAPA domain: [_2]','<b>'.$ccuname.'</b>',$ccdomain).
                            '</span><br />';
            }
            $response .= '<p class="LC_warning">'
                        .&mt("You are not authorized to create new $usertypetext{$usertype} users in this domain.")
                        .' '
                        .&mt('Please contact the [_1]helpdesk[_2] for assistance.'
                            ,'<a href="'.$helplink.'">','</a>')
                        .'</p><br />';
            $env{'form.phase'} = '';
            &print_username_entry_form($r,$context,$response,undef,undef,$crstype);
            return;
        }
        $newuser = 1;
        my $checkhash;
        my $checks = { 'username' => 1 };
        $checkhash->{$ccuname.':'.$ccdomain} = { 'newuser' => $newuser };
        &Apache::loncommon::user_rule_check($checkhash,$checks,
            \%alerts,\%rulematch,\%inst_results,\%curr_rules,\%got_rules);
        if (ref($alerts{'username'}) eq 'HASH') {
            if (ref($alerts{'username'}{$ccdomain}) eq 'HASH') {
                my $domdesc =
                    &Apache::lonnet::domain($ccdomain,'description');
                if ($alerts{'username'}{$ccdomain}{$ccuname}) {
                    my $userchkmsg;
                    if (ref($curr_rules{$ccdomain}) eq 'HASH') {  
                        $userchkmsg = 
                            &Apache::loncommon::instrule_disallow_msg('username',
                                                                 $domdesc,1).
                        &Apache::loncommon::user_rule_formats($ccdomain,
                            $domdesc,$curr_rules{$ccdomain}{'username'},
                            'username');
                    }
                    $env{'form.phase'} = '';
                    &print_username_entry_form($r,$context,$userchkmsg,undef,undef,$crstype);
                    return;
                }
            }
        }
    } else {
        $newuser = 0;
    }
    if ($response) {
        $response = '<br />'.$response;
    }

    my $pjump_def = &Apache::lonhtmlcommon::pjump_javascript_definition();
    my $dc_setcourse_code = '';
    my $nondc_setsection_code = '';                                        
    my %loaditem;

    my $groupslist = &Apache::lonuserutils::get_groupslist();

    my $js = &validation_javascript($context,$ccdomain,$pjump_def,
                               $groupslist,$newuser,$formname,\%loaditem);
    my $args = {'add_entries' => \%loaditem};  
    if ($env{'form.popup'}) {
       $args->{'no_nav_bar'} = 1; 
    }
    my $start_page = 
	&Apache::loncommon::start_page('User Management',$js,$args);
    my %breadcrumb_text = &singleuser_breadcrumb($crstype);
    &Apache::lonhtmlcommon::add_breadcrumb
     ({href=>"javascript:backPage($form)",
       text=>$breadcrumb_text{'search'},
       faq=>282,bug=>'Instructor Interface',});

    if ($env{'form.phase'} eq 'userpicked') {
        &Apache::lonhtmlcommon::add_breadcrumb
     ({href=>"javascript:backPage($form,'get_user_info','select')",
       text=>$breadcrumb_text{'userpicked'},
       faq=>282,bug=>'Instructor Interface',});
    }
    &Apache::lonhtmlcommon::add_breadcrumb
      ({href=>"javascript:backPage($form,'$env{'form.phase'}','modify')",
        text=>$breadcrumb_text{'modify'},
        faq=>282,bug=>'Instructor Interface',});
    my $helpitem = 'Course_Change_Privileges';
    if ($env{'form.action'} eq 'singlestudent') {
        $helpitem = 'Course_Add_Student';
    }
    my $crumbs = &Apache::lonhtmlcommon::breadcrumbs('User Management',
                                                     $helpitem);

    my $forminfo =<<"ENDFORMINFO";
<form action="/adm/createuser" method="post" name="$formname">
<input type="hidden" name="phase" value="update_user_data" />
<input type="hidden" name="ccuname" value="$ccuname" />
<input type="hidden" name="ccdomain" value="$ccdomain" />
<input type="hidden" name="pres_value"  value="" />
<input type="hidden" name="pres_type"   value="" />
<input type="hidden" name="pres_marker" value="" />
ENDFORMINFO
    my (%inccourses,$roledom);
    if ($context eq 'course') {
        $inccourses{$env{'request.course.id'}}=1;
        $roledom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    } elsif ($context eq 'author') {
        $roledom = $env{'request.role.domain'};
    } elsif ($context eq 'domain') {
        foreach my $key (keys(%env)) {
            $roledom = $env{'request.role.domain'};
            if ($key=~/^user\.priv\.cm\.\/($roledom)\/($match_username)/) {
                $inccourses{$1.'_'.$2}=1;
            }
        }
    } else {
        foreach my $key (keys(%env)) {
            if ($key=~/^user\.priv\.cm\.\/($match_domain)\/($match_username)/) {
                $inccourses{$1.'_'.$2}=1;
            }
        }
    }
    if ($newuser) {
        my $portfolioform;
        if ((&Apache::lonnet::allowed('mpq',$env{'request.role.domain'})) ||
            (&Apache::lonnet::allowed('mut',$env{'request.role.domain'}))) {
            # Current user has quota or user tools modification privileges
            $portfolioform = '<br />'.&portfolio_quota($ccuname,$ccdomain);
        }
        &initialize_authen_forms($ccdomain,$formname);
        my %lt=&Apache::lonlocal::texthash(
                'cnu'            => 'Create New User',
                'ast'            => 'as a student',
                'ame'            => 'as a member',
                'ind'            => 'in domain',
                'lg'             => 'Login Data',
                'hs'             => "Home Server",
        );
	$r->print(<<ENDTITLE);
$start_page
$crumbs
$response
$forminfo
<script type="text/javascript" language="Javascript">
// <![CDATA[
$loginscript
// ]]>
</script>
<input type='hidden' name='makeuser' value='1' />
<h2>$lt{'cnu'} "$ccuname" $lt{'ind'} $ccdomain
ENDTITLE
        if ($env{'form.action'} eq 'singlestudent') {
            if ($crstype eq 'Community') {
                $r->print(' ('.$lt{'ame'}.')');
            } else {
                $r->print(' ('.$lt{'ast'}.')');
            }
        }
        $r->print('</h2>'."\n".'<div class="LC_left_float">');
        my $personal_table = 
            &personal_data_display($ccuname,$ccdomain,$newuser,$context,
                                   $inst_results{$ccuname.':'.$ccdomain});
        $r->print($personal_table);
        my ($home_server_pick,$numlib) = 
            &Apache::loncommon::home_server_form_item($ccdomain,'hserver',
                                                      'default','hide');
        if ($numlib > 1) {
            $r->print("
<br />
$lt{'hs'}: $home_server_pick
<br />");
        } else {
            $r->print($home_server_pick);
        }
        if (&Apache::lonnet::allowed('ccc',$env{'request.role.domain'})) {
            $r->print('<br /><h3>'.&mt('User Can Request Creation of Courses/Communities in this Domain?').'</h3>'.
                      &Apache::loncommon::start_data_table().
                      &build_tools_display($ccuname,$ccdomain,
                                           'requestcourses').
                      &Apache::loncommon::end_data_table());
        }
        $r->print('</div>'."\n".'<div class="LC_left_float"><h3>'.
                  $lt{'lg'}.'</h3>');
        my ($fixedauth,$varauth,$authmsg); 
        if (ref($rulematch{$ccuname.':'.$ccdomain}) eq 'HASH') {
            my $matchedrule = $rulematch{$ccuname.':'.$ccdomain}{'username'};
            my ($rules,$ruleorder) = 
                &Apache::lonnet::inst_userrules($ccdomain,'username');
            if (ref($rules) eq 'HASH') {
                if (ref($rules->{$matchedrule}) eq 'HASH') {
                    my $authtype = $rules->{$matchedrule}{'authtype'};
                    if ($authtype !~ /^(krb4|krb5|int|fsys|loc)$/) {
                        $r->print(&Apache::lonuserutils::set_login($ccdomain,$authformkrb,$authformint,$authformloc));
                    } else { 
                        my $authparm = $rules->{$matchedrule}{'authparm'};
                        $authmsg = $rules->{$matchedrule}{'authmsg'};
                        if ($authtype =~ /^krb(4|5)$/) {
                            my $ver = $1;
                            if ($authparm ne '') {
                                $fixedauth = <<"KERB"; 
<input type="hidden" name="login" value="krb" />
<input type="hidden" name="krbver" value="$ver" />
<input type="hidden" name="krbarg" value="$authparm" />
KERB
                            }
                        } else {
                            $fixedauth = 
'<input type="hidden" name="login" value="'.$authtype.'" />'."\n";
                            if ($rules->{$matchedrule}{'authparmfixed'}) {
                                $fixedauth .=    
'<input type="hidden" name="'.$authtype.'arg" value="'.$authparm.'" />'."\n";
                            } else {
                                if ($authtype eq 'int') {
                                    $varauth = '<br />'.
&mt('[_1] Internally authenticated (with initial password [_2])','','<input type="password" size="10" name="intarg" value="" />')."<label><input type=\"checkbox\" name=\"visible\" onclick='if (this.checked) { this.form.intarg.type=\"text\" } else { this.form.intarg.type=\"password\" }' />".&mt('Visible input').'</label>';
                                } elsif ($authtype eq 'loc') {
                                    $varauth = '<br />'.
&mt('[_1] Local Authentication with argument [_2]','','<input type="text" name="'.$authtype.'arg" value="" />')."\n";
                                } else {
                                    $varauth =
'<input type="text" name="'.$authtype.'arg" value="" />'."\n";
                                }
                            }
                        }
                    }
                } else {
                    $r->print(&Apache::lonuserutils::set_login($ccdomain,$authformkrb,$authformint,$authformloc));
                }
            }
            if ($authmsg) {
                $r->print(<<ENDAUTH);
$fixedauth
$authmsg
$varauth
ENDAUTH
            }
        } else {
            $r->print(&Apache::lonuserutils::set_login($ccdomain,$authformkrb,$authformint,$authformloc)); 
        }
        $r->print($portfolioform);
        if ($env{'form.action'} eq 'singlestudent') {
            $r->print(&date_sections_select($context,$newuser,$formname,
                                            $permission));
        }
        $r->print('</div><div class="LC_clear_float_footer"></div>');
    } else { # user already exists
	my %lt=&Apache::lonlocal::texthash(
                    'cup'  => "Modify existing user: ",
                    'ens'  => "Enroll one student: ",
                    'enm'  => "Enroll one member: ",
                    'id'   => "in domain",
				       );
	$r->print(<<ENDCHANGEUSER);
$start_page
$crumbs
$forminfo
<h2>
ENDCHANGEUSER
        if ($env{'form.action'} eq 'singlestudent') {
            if ($crstype eq 'Community') {
                $r->print($lt{'enm'});
            } else {
                $r->print($lt{'ens'});
            }
        } else {
            $r->print($lt{'cup'});
        }
        $r->print(' "'.$ccuname.'" '.$lt{'id'}.' "'.$ccdomain.'"</h2>'.
                  "\n".'<div class="LC_left_float">');
        my ($personal_table,$showforceid) = 
            &personal_data_display($ccuname,$ccdomain,$newuser,$context,
                                   $inst_results{$ccuname.':'.$ccdomain});
        $r->print($personal_table);
        if ($showforceid) {
            $r->print(&Apache::lonuserutils::forceid_change($context));
        }
        if (&Apache::lonnet::allowed('ccc',$env{'request.role.domain'})) {
            $r->print('<h3>'.&mt('User Can Request Creation of Courses/Communities in this Domain?').'</h3>'.
                      &Apache::loncommon::start_data_table());
            if ($env{'request.role.domain'} eq $ccdomain) {
                $r->print(&build_tools_display($ccuname,$ccdomain,'requestcourses'));
            } else {
                $r->print(&coursereq_externaluser($ccuname,$ccdomain,
                                                  $env{'request.role.domain'}));
            }
            $r->print(&Apache::loncommon::end_data_table());
        }
        $r->print('</div>');
        my $user_auth_text =  &user_authentication($ccuname,$ccdomain,$formname);
        my ($user_quota_text,$user_tools_text,$user_reqcrs_text);
        if ((&Apache::lonnet::allowed('mpq',$ccdomain)) ||
            (&Apache::lonnet::allowed('mut',$ccdomain))) {
            # Current user has quota modification privileges
            $user_quota_text = &portfolio_quota($ccuname,$ccdomain);
        }
        if (!&Apache::lonnet::allowed('mpq',$ccdomain)) {
            if (&Apache::lonnet::allowed('mpq',$env{'request.role.domain'})) {
                # Get the user's portfolio information
                my %portq = &Apache::lonnet::get('environment',['portfolioquota'],
                                                 $ccdomain,$ccuname);
                my %lt=&Apache::lonlocal::texthash(
                    'dska'  => "Disk space allocated to user's portfolio files",
                    'youd'  => "You do not have privileges to modify the portfolio quota for this user.",
                    'ichr'  => "If a change is required, contact a domain coordinator for the domain",
                );
                $user_quota_text = <<ENDNOPORTPRIV;
<h3>$lt{'dska'}</h3>
$lt{'youd'} $lt{'ichr'}: $ccdomain
ENDNOPORTPRIV
            }
        }
        if (!&Apache::lonnet::allowed('mut',$ccdomain)) {
            if (&Apache::lonnet::allowed('mut',$env{'request.role.domain'})) {
                my %lt=&Apache::lonlocal::texthash(
                    'utav'  => "User Tools Availability",
                    'yodo'  => "You do not have privileges to modify Portfolio, Blog or Personal Information Page settings for this user.",
                    'ifch'  => "If a change is required, contact a domain coordinator for the domain",
                );
                $user_tools_text = <<ENDNOTOOLSPRIV;
<h3>$lt{'utav'}</h3>
$lt{'yodo'} $lt{'ifch'}: $ccdomain
ENDNOTOOLSPRIV
            }
        }
        if ($user_auth_text ne '') {
            $r->print('<div class="LC_left_float">'.$user_auth_text);
            if ($user_quota_text ne '') {
                $r->print($user_quota_text);
            }
            if ($user_tools_text ne '') {
                $r->print($user_tools_text);
            }
            if ($env{'form.action'} eq 'singlestudent') {
                $r->print(&date_sections_select($context,$newuser,$formname));
            }
        } elsif ($user_quota_text ne '') {
            $r->print('<div class="LC_left_float">'.$user_quota_text);
            if ($user_tools_text ne '') {
                $r->print($user_tools_text);
            }
            if ($env{'form.action'} eq 'singlestudent') {
                $r->print(&date_sections_select($context,$newuser,$formname));
            }
        } elsif ($user_tools_text ne '') {
            $r->print('<div class="LC_left_float">'.$user_tools_text);
            if ($env{'form.action'} eq 'singlestudent') {
                $r->print(&date_sections_select($context,$newuser,$formname));
            }
        } else {
            if ($env{'form.action'} eq 'singlestudent') {
                $r->print('<div class="LC_left_float">'.
                          &date_sections_select($context,$newuser,$formname));
            }
        }
        $r->print('</div><div class="LC_clear_float_footer"></div>');
        if ($env{'form.action'} ne 'singlestudent') {
            &display_existing_roles($r,$ccuname,$ccdomain,\%inccourses,$context,
                                    $roledom,$crstype);
        }
    } ## End of new user/old user logic
    if ($env{'form.action'} eq 'singlestudent') {
        my $btntxt;
        if ($crstype eq 'Community') {
            $btntxt = &mt('Enroll Member');
        } else {
            $btntxt = &mt('Enroll Student');
        }
        $r->print('<br /><input type="button" value="'.$btntxt.'" onclick="setSections(this.form)" />'."\n");
    } else {
        $r->print('<h3>'.&mt('Add Roles').'</h3>');
        my $addrolesdisplay = 0;
        if ($context eq 'domain' || $context eq 'author') {
            $addrolesdisplay = &new_coauthor_roles($r,$ccuname,$ccdomain);
        }
        if ($context eq 'domain') {
            my $add_domainroles = &new_domain_roles($r);
            if (!$addrolesdisplay) {
                $addrolesdisplay = $add_domainroles;
            }
            $r->print(&course_level_dc($env{'request.role.domain'},'Course'));
            $r->print('<br /><input type="button" value="'.&mt('Save').'" onclick="setCourse()" />'."\n");
        } elsif ($context eq 'author') {
            if ($addrolesdisplay) {
                $r->print('<br /><input type="button" value="'.&mt('Save').'"');
                if ($newuser) {
                    $r->print(' onclick="auth_check()" \>'."\n");
                } else {
                    $r->print('onclick="this.form.submit()" \>'."\n");
                }
            } else {
                $r->print('<br /><a href="javascript:backPage(document.cu)">'.
                          &mt('Back to previous page').'</a>');
            }
        } else {
            $r->print(&course_level_table(%inccourses));
            $r->print('<br /><input type="button" value="'.&mt('Save').'" onclick="setSections(this.form)" />'."\n");
        }
    }
    $r->print(&Apache::lonhtmlcommon::echo_form_input(['phase','userrole','ccdomain','prevphase','currstate','ccuname','ccdomain']));
    $r->print('<input type="hidden" name="currstate" value="" />');
    $r->print('<input type="hidden" name="prevphase" value="'.$env{'form.phase'}.'" />');
    $r->print("</form>".&Apache::loncommon::end_page());
    return;
}

sub singleuser_breadcrumb {
    my ($crstype) = @_;
    my %breadcrumb_text;
    if ($env{'form.action'} eq 'singlestudent') {
        if ($crstype eq 'Community') {
            $breadcrumb_text{'search'} = 'Enroll a member';
        } else {
            $breadcrumb_text{'search'} = 'Enroll a student';
        }
        $breadcrumb_text{'userpicked'} = 'Select a user',
        $breadcrumb_text{'modify'} = 'Set section/dates',
    } else {
        $breadcrumb_text{'search'} = 'Create/modify a user';
        $breadcrumb_text{'userpicked'} = 'Select a user',
        $breadcrumb_text{'modify'} = 'Set user role',
    }
    return %breadcrumb_text;
}

sub date_sections_select {
    my ($context,$newuser,$formname,$permission) = @_;
    my $cid = $env{'request.course.id'};
    my ($cnum,$cdom) = &Apache::lonuserutils::get_course_identity($cid);
    my $date_table = '<h3>'.&mt('Starting and Ending Dates').'</h3>'."\n".
        &Apache::lonuserutils::date_setting_table(undef,undef,$context,
                                                  undef,$formname,$permission);
    my $rowtitle = 'Section';
    my $secbox = '<h3>'.&mt('Section').'</h3>'."\n".
        &Apache::lonuserutils::section_picker($cdom,$cnum,'st',$rowtitle,
                                              $permission);
    my $output = $date_table.$secbox;
    return $output;
}

sub validation_javascript {
    my ($context,$ccdomain,$pjump_def,$groupslist,$newuser,$formname,
        $loaditem) = @_;
    my $dc_setcourse_code = '';
    my $nondc_setsection_code = '';
    if ($context eq 'domain') {
        my $dcdom = $env{'request.role.domain'};
        $loaditem->{'onload'} = "document.cu.coursedesc.value='';";
        $dc_setcourse_code = 
            &Apache::lonuserutils::dc_setcourse_js('cu','singleuser',$context);
    } else {
        my $checkauth; 
        if (($newuser) || (&Apache::lonnet::allowed('mau',$ccdomain))) {
            $checkauth = 1;
        }
        if ($context eq 'course') {
            $nondc_setsection_code =
                &Apache::lonuserutils::setsections_javascript($formname,$groupslist,
                                                              undef,$checkauth);
        }
        if ($checkauth) {
            $nondc_setsection_code .= 
                &Apache::lonuserutils::verify_authen($formname,$context);
        }
    }
    my $js = &user_modification_js($pjump_def,$dc_setcourse_code,
                                   $nondc_setsection_code,$groupslist);
    my ($jsback,$elements) = &crumb_utilities();
    $js .= "\n".
           '<script type="text/javascript">'."\n".
           '// <![CDATA['."\n".
           $jsback."\n".
           '// ]]>'."\n".
           '</script>'."\n";
    return $js;
}

sub display_existing_roles {
    my ($r,$ccuname,$ccdomain,$inccourses,$context,$roledom,$crstype) = @_;
    my $now=time;
    my %lt=&Apache::lonlocal::texthash(
                    'rer'  => "Existing Roles",
                    'rev'  => "Revoke",
                    'del'  => "Delete",
                    'ren'  => "Re-Enable",
                    'rol'  => "Role",
                    'ext'  => "Extent",
                    'sta'  => "Start",
                    'end'  => "End",
                                       );
    my (%rolesdump,%roletext,%sortrole,%roleclass,%rolepriv);
    if ($context eq 'course' || $context eq 'author') {
        my @roles = &Apache::lonuserutils::roles_by_context($context,1,$crstype);
        my %roleshash =
            &Apache::lonnet::get_my_roles($ccuname,$ccdomain,'userroles',
                              ['active','previous','future'],\@roles,$roledom,1);
        foreach my $key (keys(%roleshash)) {
            my ($start,$end) = split(':',$roleshash{$key});
            next if ($start eq '-1' || $end eq '-1');
            my ($rnum,$rdom,$role,$sec) = split(':',$key);
            if ($context eq 'course') {
                next unless (($rnum eq $env{'course.'.$env{'request.course.id'}.'.num'})
                             && ($rdom eq $env{'course.'.$env{'request.course.id'}.'.domain'}));
            } elsif ($context eq 'author') {
                next unless (($rnum eq $env{'user.name'}) && ($rdom eq $env{'request.role.domain'}));
            }
            my ($newkey,$newvalue,$newrole);
            $newkey = '/'.$rdom.'/'.$rnum;
            if ($sec ne '') {
                $newkey .= '/'.$sec;
            }
            $newvalue = $role;
            if ($role =~ /^cr/) {
                $newrole = 'cr';
            } else {
                $newrole = $role;
            }
            $newkey .= '_'.$newrole;
            if ($start ne '' && $end ne '') {
                $newvalue .= '_'.$end.'_'.$start;
            } elsif ($end ne '') {
                $newvalue .= '_'.$end;
            }
            $rolesdump{$newkey} = $newvalue;
        }
    } else {
        %rolesdump=&Apache::lonnet::dump('roles',$ccdomain,$ccuname);
    }
    # Build up table of user roles to allow revocation and re-enabling of roles.
    my ($tmp) = keys(%rolesdump);
    return if ($tmp =~ /^(con_lost|error)/i);
    foreach my $area (sort { my $a1=join('_',(split('_',$a))[1,0]);
                                my $b1=join('_',(split('_',$b))[1,0]);
                                return $a1 cmp $b1;
                            } keys(%rolesdump)) {
        next if ($area =~ /^rolesdef/);
        my $envkey=$area;
        my $role = $rolesdump{$area};
        my $thisrole=$area;
        $area =~ s/\_\w\w$//;
        my ($role_code,$role_end_time,$role_start_time) =
            split(/_/,$role);
# Is this a custom role? Get role owner and title.
        my ($croleudom,$croleuname,$croletitle)=
            ($role_code=~m{^cr/($match_domain)/($match_username)/(\w+)$});
        my $allowed=0;
        my $delallowed=0;
        my $sortkey=$role_code;
        my $class='Unknown';
        if ($area =~ m{^/($match_domain)/($match_courseid)} ) {
            $class='Course';
            my ($coursedom,$coursedir) = ($1,$2);
            my $cid = $1.'_'.$2;
            # $1.'_'.$2 is the course id (eg. 103_12345abcef103l3).
            my %coursedata=
                &Apache::lonnet::coursedescription($cid);
            if ($coursedir =~ /^$match_community$/) {
                $class='Community';
            }
            $sortkey.="\0$coursedom";
            my $carea;
            if (defined($coursedata{'description'})) {
                $carea=$coursedata{'description'}.
                    '<br />'.&mt('Domain').': '.$coursedom.('&nbsp;'x8).
     &Apache::loncommon::syllabuswrapper(&mt('Syllabus'),$coursedir,$coursedom);
                $sortkey.="\0".$coursedata{'description'};
            } else {
                if ($class eq 'Community') {
                    $carea=&mt('Unavailable community').': '.$area;
                    $sortkey.="\0".&mt('Unavailable community').': '.$area;
                } else {
                    $carea=&mt('Unavailable course').': '.$area;
                    $sortkey.="\0".&mt('Unavailable course').': '.$area;
                }
            }
            $sortkey.="\0$coursedir";
            $inccourses->{$cid}=1;
            if ((&Apache::lonnet::allowed('c'.$role_code,$coursedom.'/'.$coursedir)) ||
                (&Apache::lonnet::allowed('c'.$role_code,$ccdomain))) {
                $allowed=1;
            }
            unless ($allowed) {
                my $isowner = &is_courseowner($cid,$coursedata{'internal.courseowner'});
                if ($isowner) {
                    if (($role_code eq 'co') && ($class eq 'Community')) {
                        $allowed = 1;
                    } elsif (($role_code eq 'cc') && ($class eq 'Course')) {
                        $allowed = 1;
                    }
                }
            }
            if ((&Apache::lonnet::allowed('dro',$coursedom)) ||
                (&Apache::lonnet::allowed('dro',$ccdomain))) {
               $delallowed=1;
            }
# - custom role. Needs more info, too
            if ($croletitle) {
                if (&Apache::lonnet::allowed('ccr',$coursedom.'/'.$coursedir)) {
                    $allowed=1;
                    $thisrole.='.'.$role_code;
                }
            }
            if ($area=~m{^/($match_domain)/($match_courseid)/(\w+)}) {
                $carea.='<br />Section: '.$3;
                $sortkey.="\0$3";
                if (!$allowed) {
                   if ($env{'request.course.sec'} eq $3) {
                       if (&Apache::lonnet::allowed('c'.$role_code,$1.'/'.$2.'/'.$3)) {
                           $allowed = 1;
                       }
                   }
                }
            }
            $area=$carea;
        } else {
            $sortkey.="\0".$area;
            # Determine if current user is able to revoke privileges
            if ($area=~m{^/($match_domain)/}) {
                if ((&Apache::lonnet::allowed('c'.$role_code,$1)) ||
                    (&Apache::lonnet::allowed('c'.$role_code,$ccdomain))) {
                    $allowed=1;
                }
                if (((&Apache::lonnet::allowed('dro',$1))  ||
                    (&Apache::lonnet::allowed('dro',$ccdomain))) &&
                    ($role_code ne 'dc')) {
                    $delallowed=1;
                }
            } else {
                if (&Apache::lonnet::allowed('c'.$role_code,'/')) {
                    $allowed=1;
                }
            }
            if ($role_code eq 'ca' || $role_code eq 'au') {
                $class='Construction Space';
            } elsif ($role_code eq 'su') {
                $class='System';
            } else {
                $class='Domain';
            }
        }
        if (($role_code eq 'ca') || ($role_code eq 'aa')) {
            $area=~m{/($match_domain)/($match_username)};
            if (&Apache::lonuserutils::authorpriv($2,$1)) {
                $allowed=1;
            } else {
                $allowed=0;
            }
        }
        my $row = '';
        $row.= '<td>';
        my $active=1;
        $active=0 if (($role_end_time) && ($now>$role_end_time));
        if (($active) && ($allowed)) {
            $row.= '<input type="checkbox" name="rev:'.$thisrole.'" />';
        } else {
            if ($active) {
               $row.='&nbsp;';
            } else {
               $row.=&mt('expired or revoked');
            }
        }
        $row.='</td><td>';
        if ($allowed && !$active) {
            $row.= '<input type="checkbox" name="ren:'.$thisrole.'" />';
        } else {
            $row.='&nbsp;';
        }
        $row.='</td><td>';
        if ($delallowed) {
            $row.= '<input type="checkbox" name="del:'.$thisrole.'" />';
        } else {
            $row.='&nbsp;';
        }
        my $plaintext='';
        if (!$croletitle) {
            $plaintext=&Apache::lonnet::plaintext($role_code,$class)
        } else {
            $plaintext=
        "Customrole '$croletitle'<br />defined by $croleuname\@$croleudom";
        }
        $row.= '</td><td>'.$plaintext.
               '</td><td>'.$area.
               '</td><td>'.($role_start_time?&Apache::lonlocal::locallocaltime($role_start_time)
                                            : '&nbsp;' ).
              '</td><td>'.($role_end_time  ?&Apache::lonlocal::locallocaltime($role_end_time)
                                            : '&nbsp;' )
               ."</td>";
        $sortrole{$sortkey}=$envkey;
        $roletext{$envkey}=$row;
        $roleclass{$envkey}=$class;
        $rolepriv{$envkey}=$allowed;
    } # end of foreach        (table building loop)

    my $rolesdisplay = 0;
    my %output = ();
    foreach my $type ('Construction Space','Course','Community','Domain','System','Unknown') {
        $output{$type} = '';
        foreach my $which (sort {uc($a) cmp uc($b)} (keys(%sortrole))) {
            if ( ($roleclass{$sortrole{$which}} =~ /^\Q$type\E/ ) && ($rolepriv{$sortrole{$which}}) ) {
                $output{$type}.=
                      &Apache::loncommon::start_data_table_row().
                      $roletext{$sortrole{$which}}.
                      &Apache::loncommon::end_data_table_row();
            }
        }
        unless($output{$type} eq '') {
            $output{$type} = '<tr class="LC_info_row">'.
                      "<td align='center' colspan='7'>".&mt($type)."</td></tr>".
                      $output{$type};
            $rolesdisplay = 1;
        }
    }
    if ($rolesdisplay == 1) {
        my $contextrole='';
        if ($env{'request.course.id'}) {
            if (&Apache::loncommon::course_type() eq 'Community') {
                $contextrole = &mt('Existing Roles in this Community');
            } else {
                $contextrole = &mt('Existing Roles in this Course');
            }
        } elsif ($env{'request.role'} =~ /^au\./) {
            $contextrole = &mt('Existing Co-Author Roles in your Construction Space');
        } else {
            $contextrole = &mt('Existing Roles in this Domain');
        }
        $r->print('
<h3>'.$lt{'rer'}.'</h3>'.
'<div>'.$contextrole.'</div>'.
&Apache::loncommon::start_data_table("LC_createuser").
&Apache::loncommon::start_data_table_header_row().
'<th>'.$lt{'rev'}.'</th><th>'.$lt{'ren'}.'</th><th>'.$lt{'del'}.
'</th><th>'.$lt{'rol'}.'</th><th>'.$lt{'ext'}.
'</th><th>'.$lt{'sta'}.'</th><th>'.$lt{'end'}.'</th>'.
&Apache::loncommon::end_data_table_header_row());
        foreach my $type ('Construction Space','Course','Community','Domain','System','Unknown') {
            if ($output{$type}) {
                $r->print($output{$type}."\n");
            }
        }
        $r->print(&Apache::loncommon::end_data_table());
    }
    return;
}

sub new_coauthor_roles {
    my ($r,$ccuname,$ccdomain) = @_;
    my $addrolesdisplay = 0;
    #
    # Co-Author
    #
    if (&Apache::lonuserutils::authorpriv($env{'user.name'},
                                          $env{'request.role.domain'}) &&
        ($env{'user.name'} ne $ccuname || $env{'user.domain'} ne $ccdomain)) {
        # No sense in assigning co-author role to yourself
        $addrolesdisplay = 1;
        my $cuname=$env{'user.name'};
        my $cudom=$env{'request.role.domain'};
        my %lt=&Apache::lonlocal::texthash(
                    'cs'   => "Construction Space",
                    'act'  => "Activate",
                    'rol'  => "Role",
                    'ext'  => "Extent",
                    'sta'  => "Start",
                    'end'  => "End",
                    'cau'  => "Co-Author",
                    'caa'  => "Assistant Co-Author",
                    'ssd'  => "Set Start Date",
                    'sed'  => "Set End Date"
                                       );
        $r->print('<h4>'.$lt{'cs'}.'</h4>'."\n".
                  &Apache::loncommon::start_data_table()."\n".
                  &Apache::loncommon::start_data_table_header_row()."\n".
                  '<th>'.$lt{'act'}.'</th><th>'.$lt{'rol'}.'</th>'.
                  '<th>'.$lt{'ext'}.'</th><th>'.$lt{'sta'}.'</th>'.
                  '<th>'.$lt{'end'}.'</th>'."\n".
                  &Apache::loncommon::end_data_table_header_row()."\n".
                  &Apache::loncommon::start_data_table_row().'
           <td>
            <input type="checkbox" name="act_'.$cudom.'_'.$cuname.'_ca" />
           </td>
           <td>'.$lt{'cau'}.'</td>
           <td>'.$cudom.'_'.$cuname.'</td>
           <td><input type="hidden" name="start_'.$cudom.'_'.$cuname.'_ca" value="" />
             <a href=
"javascript:pjump('."'date_start','Start Date Co-Author',document.cu.start_$cudom\_$cuname\_ca.value,'start_$cudom\_$cuname\_ca','cu.pres','dateset'".')">'.$lt{'ssd'}.'</a></td>
<td><input type="hidden" name="end_'.$cudom.'_'.$cuname.'_ca" value="" />
<a href=
"javascript:pjump('."'date_end','End Date Co-Author',document.cu.end_$cudom\_$cuname\_ca.value,'end_$cudom\_$cuname\_ca','cu.pres','dateset'".')">'.$lt{'sed'}.'</a></td>'."\n".
              &Apache::loncommon::end_data_table_row()."\n".
              &Apache::loncommon::start_data_table_row()."\n".
'<td><input type="checkbox" name="act_'.$cudom.'_'.$cuname.'_aa" /></td>
<td>'.$lt{'caa'}.'</td>
<td>'.$cudom.'_'.$cuname.'</td>
<td><input type="hidden" name="start_'.$cudom.'_'.$cuname.'_aa" value="" />
<a href=
"javascript:pjump('."'date_start','Start Date Assistant Co-Author',document.cu.start_$cudom\_$cuname\_aa.value,'start_$cudom\_$cuname\_aa','cu.pres','dateset'".')">'.$lt{'ssd'}.'</a></td>
<td><input type="hidden" name="end_'.$cudom.'_'.$cuname.'_aa" value="" />
<a href=
"javascript:pjump('."'date_end','End Date Assistant Co-Author',document.cu.end_$cudom\_$cuname\_aa.value,'end_$cudom\_$cuname\_aa','cu.pres','dateset'".')">'.$lt{'sed'}.'</a></td>'."\n".
             &Apache::loncommon::end_data_table_row()."\n".
             &Apache::loncommon::end_data_table());
    } elsif ($env{'request.role'} =~ /^au\./) {
        if (!(&Apache::lonuserutils::authorpriv($env{'user.name'},
                                                $env{'request.role.domain'}))) {
            $r->print('<span class="LC_error">'.
                      &mt('You do not have privileges to assign co-author roles.').
                      '</span>');
        } elsif (($env{'user.name'} eq $ccuname) &&
             ($env{'user.domain'} eq $ccdomain)) {
            $r->print(&mt('Assigning yourself a co-author or assistant co-author role in your own author area in Construction Space is not permitted'));
        }
    }
    return $addrolesdisplay;;
}

sub new_domain_roles {
    my ($r) = @_;
    my $addrolesdisplay = 0;
    #
    # Domain level
    #
    my $num_domain_level = 0;
    my $domaintext =
    '<h4>'.&mt('Domain Level').'</h4>'.
    &Apache::loncommon::start_data_table().
    &Apache::loncommon::start_data_table_header_row().
    '<th>'.&mt('Activate').'</th><th>'.&mt('Role').'</th><th>'.
    &mt('Extent').'</th>'.
    '<th>'.&mt('Start').'</th><th>'.&mt('End').'</th>'.
    &Apache::loncommon::end_data_table_header_row();
    my @allroles = &Apache::lonuserutils::roles_by_context('domain');
    foreach my $thisdomain (sort(&Apache::lonnet::all_domains())) {
        foreach my $role (@allroles) {
            next if ($role eq 'ad');
            if (&Apache::lonnet::allowed('c'.$role,$thisdomain)) {
               my $plrole=&Apache::lonnet::plaintext($role);
               my %lt=&Apache::lonlocal::texthash(
                    'ssd'  => "Set Start Date",
                    'sed'  => "Set End Date"
                                       );
               $num_domain_level ++;
               $domaintext .=
&Apache::loncommon::start_data_table_row().
'<td><input type="checkbox" name="act_'.$thisdomain.'_'.$role.'" /></td>
<td>'.$plrole.'</td>
<td>'.$thisdomain.'</td>
<td><input type="hidden" name="start_'.$thisdomain.'_'.$role.'" value="" />
<a href=
"javascript:pjump('."'date_start','Start Date $plrole',document.cu.start_$thisdomain\_$role.value,'start_$thisdomain\_$role','cu.pres','dateset'".')">'.$lt{'ssd'}.'</a></td>
<td><input type="hidden" name="end_'.$thisdomain.'_'.$role.'" value="" />
<a href=
"javascript:pjump('."'date_end','End Date $plrole',document.cu.end_$thisdomain\_$role.value,'end_$thisdomain\_$role','cu.pres','dateset'".')">'.$lt{'sed'}.'</a></td>'.
&Apache::loncommon::end_data_table_row();
            }
        }
    }
    $domaintext.= &Apache::loncommon::end_data_table();
    if ($num_domain_level > 0) {
        $r->print($domaintext);
        $addrolesdisplay = 1;
    }
    return $addrolesdisplay;
}

sub user_authentication {
    my ($ccuname,$ccdomain,$formname) = @_;
    my $currentauth=&Apache::lonnet::queryauthenticate($ccuname,$ccdomain);
    my $outcome;
    # Check for a bad authentication type
    if ($currentauth !~ /^(krb4|krb5|unix|internal|localauth):/) {
        # bad authentication scheme
        my %lt=&Apache::lonlocal::texthash(
                       'err'   => "ERROR",
                       'uuas'  => "This user has an unrecognized authentication scheme",
                       'adcs'  => "Please alert a domain coordinator of this situation",
                       'sldb'  => "Please specify login data below",
                       'ld'    => "Login Data"
        );
        if (&Apache::lonnet::allowed('mau',$ccdomain)) {
            &initialize_authen_forms($ccdomain,$formname);

            my $choices = &Apache::lonuserutils::set_login($ccdomain,$authformkrb,$authformint,$authformloc);
            $outcome = <<ENDBADAUTH;
<script type="text/javascript" language="Javascript">
// <![CDATA[
$loginscript
// ]]>
</script>
<span class="LC_error">$lt{'err'}:
$lt{'uuas'} ($currentauth). $lt{'sldb'}.</span>
<h3>$lt{'ld'}</h3>
$choices
ENDBADAUTH
        } else {
            # This user is not allowed to modify the user's
            # authentication scheme, so just notify them of the problem
            $outcome = <<ENDBADAUTH;
<span class="LC_error"> $lt{'err'}: 
$lt{'uuas'} ($currentauth). $lt{'adcs'}.
</span>
ENDBADAUTH
        }
    } else { # Authentication type is valid
        &initialize_authen_forms($ccdomain,$formname,$currentauth,'modifyuser');
        my ($authformcurrent,$can_modify,@authform_others) =
            &modify_login_block($ccdomain,$currentauth);
        if (&Apache::lonnet::allowed('mau',$ccdomain)) {
            # Current user has login modification privileges
            my %lt=&Apache::lonlocal::texthash (
                           'ld'    => "Login Data",
                           'ccld'  => "Change Current Login Data",
                           'enld'  => "Enter New Login Data"
                                               );
            $outcome =
                       '<script type="text/javascript" language="Javascript">'."\n".
                       '// <![CDATA['."\n".
                       $loginscript."\n".
                       '// ]]>'."\n".
                       '</script>'."\n".
                       '<h3>'.$lt{'ld'}.'</h3>'.
                       &Apache::loncommon::start_data_table().
                       &Apache::loncommon::start_data_table_row().
                       '<td>'.$authformnop;
            if ($can_modify) {
                $outcome .= '</td>'."\n".
                            &Apache::loncommon::end_data_table_row().
                            &Apache::loncommon::start_data_table_row().
                            '<td>'.$authformcurrent.'</td>'.
                            &Apache::loncommon::end_data_table_row()."\n";
            } else {
                $outcome .= '&nbsp;('.$authformcurrent.')</td>'.
                            &Apache::loncommon::end_data_table_row()."\n";
            }
            foreach my $item (@authform_others) { 
                $outcome .= &Apache::loncommon::start_data_table_row().
                            '<td>'.$item.'</td>'.
                            &Apache::loncommon::end_data_table_row()."\n";
            }
            $outcome .= &Apache::loncommon::end_data_table();
        } else {
            if (&Apache::lonnet::allowed('mau',$env{'request.role.domain'})) {
                my %lt=&Apache::lonlocal::texthash(
                           'ccld'  => "Change Current Login Data",
                           'yodo'  => "You do not have privileges to modify the authentication configuration for this user.",
                           'ifch'  => "If a change is required, contact a domain coordinator for the domain",
                );
                $outcome .= <<ENDNOPRIV;
<h3>$lt{'ccld'}</h3>
$lt{'yodo'} $lt{'ifch'}: $ccdomain
<input type="hidden" name="login" value="nochange" />
ENDNOPRIV
            }
        }
    }  ## End of "check for bad authentication type" logic
    return $outcome;
}

sub modify_login_block {
    my ($dom,$currentauth) = @_;
    my %domconfig = &Apache::lonnet::get_dom('configuration',['usercreation'],$dom);
    my ($authnum,%can_assign) =
        &Apache::loncommon::get_assignable_auth($dom);
    my ($authformcurrent,@authform_others,$show_override_msg);
    if ($currentauth=~/^krb(4|5):/) {
        $authformcurrent=$authformkrb;
        if ($can_assign{'int'}) {
            push(@authform_others,$authformint);
        }
        if ($can_assign{'loc'}) {
            push(@authform_others,$authformloc);
        }
        if (($can_assign{'krb4'}) || ($can_assign{'krb5'})) {
            $show_override_msg = 1;
        }
    } elsif ($currentauth=~/^internal:/) {
        $authformcurrent=$authformint;
        if (($can_assign{'krb4'}) || ($can_assign{'krb5'})) {
            push(@authform_others,$authformkrb);
        }
        if ($can_assign{'loc'}) {
            push(@authform_others,$authformloc);
        }
        if ($can_assign{'int'}) {
            $show_override_msg = 1;
        }
    } elsif ($currentauth=~/^unix:/) {
        $authformcurrent=$authformfsys;
        if (($can_assign{'krb4'}) || ($can_assign{'krb5'})) {
            push(@authform_others,$authformkrb);
        }
        if ($can_assign{'int'}) {
            push(@authform_others,$authformint);
        }
        if ($can_assign{'loc'}) {
            push(@authform_others,$authformloc);
        }
        if ($can_assign{'fsys'}) {
            $show_override_msg = 1;
        }
    } elsif ($currentauth=~/^localauth:/) {
        $authformcurrent=$authformloc;
        if (($can_assign{'krb4'}) || ($can_assign{'krb5'})) {
            push(@authform_others,$authformkrb);
        }
        if ($can_assign{'int'}) {
            push(@authform_others,$authformint);
        }
        if ($can_assign{'loc'}) {
            $show_override_msg = 1;
        }
    }
    if ($show_override_msg) {
        $authformcurrent = '<table><tr><td colspan="3">'.$authformcurrent.
                           '</td></tr>'."\n".
                           '<tr><td>&nbsp;&nbsp;&nbsp;</td>'.
                           '<td><b>'.&mt('Currently in use').'</b></td>'.
                           '<td align="right"><span class="LC_cusr_emph">'.
                            &mt('will override current values').
                            '</span></td></tr></table>';
    }
    return ($authformcurrent,$show_override_msg,@authform_others); 
}

sub personal_data_display {
    my ($ccuname,$ccdomain,$newuser,$context,$inst_results,$rolesarray) = @_;
    my ($output,$showforceid,%userenv,%canmodify,%canmodify_status);
    my @userinfo = ('firstname','middlename','lastname','generation',
                    'permanentemail','id');
    my $rowcount = 0;
    my $editable = 0;
    %canmodify_status = 
        &Apache::lonuserutils::can_modify_userinfo($context,$ccdomain,
                                                   ['inststatus'],$rolesarray);
    if (!$newuser) {
        # Get the users information
        %userenv = &Apache::lonnet::get('environment',
                   ['firstname','middlename','lastname','generation',
                    'permanentemail','id','inststatus'],$ccdomain,$ccuname);
        %canmodify =
            &Apache::lonuserutils::can_modify_userinfo($context,$ccdomain,
                                                       \@userinfo,$rolesarray);
    } elsif ($context eq 'selfcreate') {
        %canmodify = &selfcreate_canmodify($context,$ccdomain,\@userinfo,
                                           $inst_results,$rolesarray);
    }
    my %lt=&Apache::lonlocal::texthash(
                'pd'             => "Personal Data",
                'firstname'      => "First Name",
                'middlename'     => "Middle Name",
                'lastname'       => "Last Name",
                'generation'     => "Generation",
                'permanentemail' => "Permanent e-mail address",
                'id'             => "Student/Employee ID",
                'lg'             => "Login Data",
                'inststatus'     => "Affiliation",
    );
    my %textboxsize = (
                       firstname      => '15',
                       middlename     => '15',
                       lastname       => '15',
                       generation     => '5',
                       permanentemail => '25',
                       id             => '15',
                      );
    my $genhelp=&Apache::loncommon::help_open_topic('Generation');
    $output = '<h3>'.$lt{'pd'}.'</h3>'.
              &Apache::lonhtmlcommon::start_pick_box();
    foreach my $item (@userinfo) {
        my $rowtitle = $lt{$item};
        my $hiderow = 0;
        if ($item eq 'generation') {
            $rowtitle = $genhelp.$rowtitle;
        }
        my $row = &Apache::lonhtmlcommon::row_title($rowtitle,undef,'LC_oddrow_value')."\n";
        if ($newuser) {
            if (ref($inst_results) eq 'HASH') {
                if ($inst_results->{$item} ne '') {
                    $row .= '<input type="hidden" name="c'.$item.'" value="'.$inst_results->{$item}.'" />'.$inst_results->{$item};
                } else {
                    if ($context eq 'selfcreate') {
                        if ($canmodify{$item}) { 
                            $row .= '<input type="text" name="c'.$item.'" size="'.$textboxsize{$item}.'" value="" />';
                            $editable ++;
                        } else {
                            $hiderow = 1;
                        }
                    } else {
                        $row .= '<input type="text" name="c'.$item.'" size="'.$textboxsize{$item}.'" value="" />';
                    }
                }
            } else {
                if ($context eq 'selfcreate') {
                    if (($item eq 'permanentemail') && ($newuser eq 'email')) {
                        $row .= $ccuname;
                    } else {
                        if ($canmodify{$item}) {
                            $row .= '<input type="text" name="c'.$item.'" size="'.$textboxsize{$item}.'" value="" />';
                            $editable ++;
                        } else {
                            $hiderow = 1;
                        }
                    }
                } else {
                    $row .= '<input type="text" name="c'.$item.'" size="'.$textboxsize{$item}.'" value="" />';
                }
            }
        } else {
            if ($canmodify{$item}) {
                $row .= '<input type="text" name="c'.$item.'" size="'.$textboxsize{$item}.'" value="'.$userenv{$item}.'" />';
            } else {
                $row .= $userenv{$item};
            }
            if ($item eq 'id') {
                $showforceid = $canmodify{$item};
            }
        }
        $row .= &Apache::lonhtmlcommon::row_closure(1);
        if (!$hiderow) {
            $output .= $row;
            $rowcount ++;
        }
    }
    if (($canmodify_status{'inststatus'}) || ($context ne 'selfcreate')) {
        my ($othertitle,$usertypes,$types) = &Apache::loncommon::sorted_inst_types($ccdomain);
        if (ref($types) eq 'ARRAY') {
            if (@{$types} > 0) {
                my ($hiderow,$shown);
                if ($canmodify_status{'inststatus'}) {
                    $shown = &pick_inst_statuses($userenv{'inststatus'},$usertypes,$types);
                } else {
                    if ($userenv{'inststatus'} eq '') {
                        $hiderow = 1;
                    } else {
                        my @showitems;
                        foreach my $item ( map { &unescape($_); } split(':',$userenv{'inststatus'})) {
                            if (exists($usertypes->{$item})) {
                                push(@showitems,$usertypes->{$item});
                            } else {
                                push(@showitems,$item);
                            }
                        }
                        if (@showitems) {
                            $shown = join(', ',@showitems);
                        } else {
                            $hiderow = 1;
                        }
                    }
                }
                if (!$hiderow) {
                    my $row = &Apache::lonhtmlcommon::row_title(&mt('Affliations'),undef,'LC_oddrow_value')."\n".
                              $shown.&Apache::lonhtmlcommon::row_closure(1); 
                    if ($context eq 'selfcreate') {
                        $rowcount ++;
                    }
                    $output .= $row;
                }
            }
        }
    }
    $output .= &Apache::lonhtmlcommon::end_pick_box();
    if (wantarray) {
        if ($context eq 'selfcreate') {
            return($output,$rowcount,$editable);
        } else {
            return ($output,$showforceid);
        }
    } else {
        return $output;
    }
}

sub pick_inst_statuses {
    my ($curr,$usertypes,$types) = @_;
    my ($output,$rem,@currtypes);
    if ($curr ne '') {
        @currtypes = map { &unescape($_); } split(/:/,$curr);
    }
    my $numinrow = 2;
    if (ref($types) eq 'ARRAY') {
        $output = '<table>';
        my $lastcolspan; 
        for (my $i=0; $i<@{$types}; $i++) {
            if (defined($usertypes->{$types->[$i]})) {
                my $rem = $i%($numinrow);
                if ($rem == 0) {
                    if ($i<@{$types}-1) {
                        if ($i > 0) { 
                            $output .= '</tr>';
                        }
                        $output .= '<tr>';
                    }
                } elsif ($i==@{$types}-1) {
                    my $colsleft = $numinrow - $rem;
                    if ($colsleft > 1) {
                        $lastcolspan = ' colspan="'.$colsleft.'"';
                    }
                }
                my $check = ' ';
                if (grep(/^\Q$types->[$i]\E$/,@currtypes)) {
                    $check = ' checked="checked" ';
                }
                $output .= '<td class="LC_left_item"'.$lastcolspan.'>'.
                           '<span class="LC_nobreak"><label>'.
                           '<input type="checkbox" name="inststatus" '.
                           'value="'.$types->[$i].'"'.$check.'/>'.
                           $usertypes->{$types->[$i]}.'</label></span></td>';
            }
        }
        $output .= '</tr></table>';
    }
    return $output;
}

sub selfcreate_canmodify {
    my ($context,$dom,$userinfo,$inst_results,$rolesarray) = @_;
    if (ref($inst_results) eq 'HASH') {
        my @inststatuses = &get_inststatuses($inst_results);
        if (@inststatuses == 0) {
            @inststatuses = ('default');
        }
        $rolesarray = \@inststatuses;
    }
    my %canmodify =
        &Apache::lonuserutils::can_modify_userinfo($context,$dom,$userinfo,
                                                   $rolesarray);
    return %canmodify;
}

sub get_inststatuses {
    my ($insthashref) = @_;
    my @inststatuses = ();
    if (ref($insthashref) eq 'HASH') {
        if (ref($insthashref->{'inststatus'}) eq 'ARRAY') {
            @inststatuses = @{$insthashref->{'inststatus'}};
        }
    }
    return @inststatuses;
}

# ================================================================= Phase Three
sub update_user_data {
    my ($r,$context,$crstype) = @_; 
    my $uhome=&Apache::lonnet::homeserver($env{'form.ccuname'},
                                          $env{'form.ccdomain'});
    # Error messages
    my $error     = '<span class="LC_error">'.&mt('Error').': ';
    my $end       = '</span><br /><br />';
    my $rtnlink   = '<a href="javascript:backPage(document.userupdate,'.
                    "'$env{'form.prevphase'}','modify')".'" />'.
                    &mt('Return to previous page').'</a>'.
                    &Apache::loncommon::end_page();
    my $now = time;
    my $title;
    if (exists($env{'form.makeuser'})) {
	$title='Set Privileges for New User';
    } else {
        $title='Modify User Privileges';
    }
    my $newuser = 0;
    my ($jsback,$elements) = &crumb_utilities();
    my $jscript = '<script type="text/javascript">'."\n".
                  '// <![CDATA['."\n".
                  $jsback."\n".
                  '// ]]>'."\n".
                  '</script>'."\n";
    my %breadcrumb_text = &singleuser_breadcrumb($crstype);
    my $args;
    if ($env{'form.popup'}) {
        $args->{'no_nav_bar'} = 1;
    } else {
        $args = undef;
    }
    $r->print(&Apache::loncommon::start_page($title,$jscript,$args));
    &Apache::lonhtmlcommon::add_breadcrumb
       ({href=>"javascript:backPage(document.userupdate)",
         text=>$breadcrumb_text{'search'},
         faq=>282,bug=>'Instructor Interface',});
    if ($env{'form.prevphase'} eq 'userpicked') {
        &Apache::lonhtmlcommon::add_breadcrumb
           ({href=>"javascript:backPage(document.userupdate,'get_user_info','select')",
             text=>$breadcrumb_text{'userpicked'},
             faq=>282,bug=>'Instructor Interface',});
    }
    &Apache::lonhtmlcommon::add_breadcrumb
       ({href=>"javascript:backPage(document.userupdate,'$env{'form.prevphase'}','modify')",
         text=>$breadcrumb_text{'modify'},
         faq=>282,bug=>'Instructor Interface',},
        {href=>"/adm/createuser",
         text=>"Result",
         faq=>282,bug=>'Instructor Interface',});
    my $helpitem = 'Course_Change_Privileges';
    if ($env{'form.action'} eq 'singlestudent') {
        $helpitem = 'Course_Add_Student';
    }
    $r->print(&Apache::lonhtmlcommon::breadcrumbs('User Management',
                                                 $helpitem));
    $r->print(&update_result_form($uhome));
    # Check Inputs
    if (! $env{'form.ccuname'} ) {
	$r->print($error.&mt('No login name specified').'.'.$end.$rtnlink);
	return;
    }
    if (  $env{'form.ccuname'} ne 
	  &LONCAPA::clean_username($env{'form.ccuname'}) ) {
	$r->print($error.&mt('Invalid login name.').'  '.
		  &mt('Only letters, numbers, periods, dashes, @, and underscores are valid.').
		  $end.$rtnlink);
	return;
    }
    if (! $env{'form.ccdomain'}       ) {
	$r->print($error.&mt('No domain specified').'.'.$end.$rtnlink);
	return;
    }
    if (  $env{'form.ccdomain'} ne
	  &LONCAPA::clean_domain($env{'form.ccdomain'}) ) {
	$r->print($error.&mt('Invalid domain name.').'  '.
		  &mt('Only letters, numbers, periods, dashes, and underscores are valid.').
		  $end.$rtnlink);
	return;
    }
    if ($uhome eq 'no_host') {
        $newuser = 1;
    }
    if (! exists($env{'form.makeuser'})) {
        # Modifying an existing user, so check the validity of the name
        if ($uhome eq 'no_host') {
            $r->print($error.&mt('Unable to determine home server for ').
                      $env{'form.ccuname'}.&mt(' in domain ').
                      $env{'form.ccdomain'}.'.');
            return;
        }
    }
    # Determine authentication method and password for the user being modified
    my $amode='';
    my $genpwd='';
    if ($env{'form.login'} eq 'krb') {
	$amode='krb';
	$amode.=$env{'form.krbver'};
	$genpwd=$env{'form.krbarg'};
    } elsif ($env{'form.login'} eq 'int') {
	$amode='internal';
	$genpwd=$env{'form.intarg'};
    } elsif ($env{'form.login'} eq 'fsys') {
	$amode='unix';
	$genpwd=$env{'form.fsysarg'};
    } elsif ($env{'form.login'} eq 'loc') {
	$amode='localauth';
	$genpwd=$env{'form.locarg'};
	$genpwd=" " if (!$genpwd);
    } elsif (($env{'form.login'} eq 'nochange') ||
             ($env{'form.login'} eq ''        )) { 
        # There is no need to tell the user we did not change what they
        # did not ask us to change.
        # If they are creating a new user but have not specified login
        # information this will be caught below.
    } else {
	    $r->print($error.&mt('Invalid login mode or password').$end.$rtnlink);    
	    return;
    }

    $r->print('<h3>'.&mt('User [_1] in domain [_2]',
			 $env{'form.ccuname'}, $env{'form.ccdomain'}).'</h3>');
    my (%alerts,%rulematch,%inst_results,%curr_rules);
    my @userinfo = ('firstname','middlename','lastname','generation','permanentemail','id');
    my @usertools = ('aboutme','blog','portfolio');
    my @requestcourses = ('official','unofficial','community');
    my ($othertitle,$usertypes,$types) = 
        &Apache::loncommon::sorted_inst_types($env{'form.ccdomain'});
    my %canmodify_status =
        &Apache::lonuserutils::can_modify_userinfo($context,$env{'form.ccdomain'},
                                                   ['inststatus']);
    if ($env{'form.makeuser'}) {
	$r->print('<h3>'.&mt('Creating new account.').'</h3>');
        # Check for the authentication mode and password
        if (! $amode || ! $genpwd) {
	    $r->print($error.&mt('Invalid login mode or password').$end.$rtnlink);    
	    return;
	}
        # Determine desired host
        my $desiredhost = $env{'form.hserver'};
        if (lc($desiredhost) eq 'default') {
            $desiredhost = undef;
        } else {
            my %home_servers = 
		&Apache::lonnet::get_servers($env{'form.ccdomain'},'library');
            if (! exists($home_servers{$desiredhost})) {
                $r->print($error.&mt('Invalid home server specified').$end.$rtnlink);
                return;
            }
        }
        # Check ID format
        my %checkhash;
        my %checks = ('id' => 1);
        %{$checkhash{$env{'form.ccuname'}.':'.$env{'form.ccdomain'}}} = (
            'newuser' => $newuser, 
            'id' => $env{'form.cid'},
        );
        if ($env{'form.cid'} ne '') {
            &Apache::loncommon::user_rule_check(\%checkhash,\%checks,\%alerts,
                                          \%rulematch,\%inst_results,\%curr_rules);
            if (ref($alerts{'id'}) eq 'HASH') {
                if (ref($alerts{'id'}{$env{'form.ccdomain'}}) eq 'HASH') {
                    my $domdesc =
                        &Apache::lonnet::domain($env{'form.ccdomain'},'description');
                    if ($alerts{'id'}{$env{'form.ccdomain'}}{$env{'form.cid'}}) {
                        my $userchkmsg;
                        if (ref($curr_rules{$env{'form.ccdomain'}}) eq 'HASH') {
                            $userchkmsg  = 
                                &Apache::loncommon::instrule_disallow_msg('id',
                                                                    $domdesc,1).
                                &Apache::loncommon::user_rule_formats($env{'form.ccdomain'},
                                    $domdesc,$curr_rules{$env{'form.ccdomain'}}{'id'},'id');
                        }
                        $r->print($error.&mt('Invalid ID format').$end.
                                  $userchkmsg.$rtnlink);
                        return;
                    }
                }
            }
        }
	# Call modifyuser
	my $result = &Apache::lonnet::modifyuser
	    ($env{'form.ccdomain'},$env{'form.ccuname'},$env{'form.cid'},
             $amode,$genpwd,$env{'form.cfirstname'},
             $env{'form.cmiddlename'},$env{'form.clastname'},
             $env{'form.cgeneration'},undef,$desiredhost,
             $env{'form.cpermanentemail'});
	$r->print(&mt('Generating user').': '.$result);
        $uhome = &Apache::lonnet::homeserver($env{'form.ccuname'},
                                               $env{'form.ccdomain'});
        my (%changeHash,%newcustom,%changed,%changedinfo);
        if ($uhome ne 'no_host') {
            if ($context eq 'domain') {
                if ($env{'form.customquota'} == 1) {
                    if ($env{'form.portfolioquota'} eq '') {
                        $newcustom{'quota'} = 0;
                    } else {
                        $newcustom{'quota'} = $env{'form.portfolioquota'};
                        $newcustom{'quota'} =~ s/[^\d\.]//g;
                    }
                    $changed{'quota'} = &quota_admin($newcustom{'quota'},\%changeHash);
                }
                foreach my $item (@usertools) {
                    if ($env{'form.custom'.$item} == 1) {
                        $newcustom{$item} = $env{'form.tools_'.$item};
                        $changed{$item} = &tool_admin($item,$newcustom{$item},
                                                      \%changeHash,'tools');
                    }
                }
                foreach my $item (@requestcourses) {
                    if ($env{'form.custom'.$item} == 1) {
                        $newcustom{$item} = $env{'form.crsreq_'.$item};
                        if ($env{'form.crsreq_'.$item} eq 'autolimit') {
                            $newcustom{$item} .= '=';
                            unless ($env{'form.crsreq_'.$item.'_limit'} =~ /\D/) {
                                $newcustom{$item} .= $env{'form.crsreq_'.$item.'_limit'};
                            }
                        }
                        $changed{$item} = &tool_admin($item,$newcustom{$item},
                                                      \%changeHash,'requestcourses');
                    }
                }
            }
            if ($canmodify_status{'inststatus'}) {
                if (exists($env{'form.inststatus'})) {
                    my @inststatuses = &Apache::loncommon::get_env_multiple('form.inststatus');
                    if (@inststatuses > 0) {
                        $changeHash{'inststatus'} = join(',',@inststatuses);
                        $changed{'inststatus'} = $changeHash{'inststatus'};
                    }
                }
            }
            if (keys(%changed)) {
                foreach my $item (@userinfo) {
                    $changeHash{$item}  = $env{'form.c'.$item};
                }
                my $chgresult =
                     &Apache::lonnet::put('environment',\%changeHash,
                                          $env{'form.ccdomain'},$env{'form.ccuname'});
            } 
        }
        $r->print('<br />'.&mt('Home server').': '.$uhome.' '.
                  &Apache::lonnet::hostname($uhome));
    } elsif (($env{'form.login'} ne 'nochange') &&
             ($env{'form.login'} ne ''        )) {
	# Modify user privileges
        if (! $amode || ! $genpwd) {
	    $r->print($error.'Invalid login mode or password'.$end.$rtnlink);    
	    return;
	}
	# Only allow authentification modification if the person has authority
	if (&Apache::lonnet::allowed('mau',$env{'form.ccdomain'})) {
	    $r->print('Modifying authentication: '.
                      &Apache::lonnet::modifyuserauth(
		       $env{'form.ccdomain'},$env{'form.ccuname'},
                       $amode,$genpwd));
            $r->print('<br />'.&mt('Home server').': '.&Apache::lonnet::homeserver
		  ($env{'form.ccuname'},$env{'form.ccdomain'}));
	} else {
	    # Okay, this is a non-fatal error.
	    $r->print($error.&mt('You do not have the authority to modify this users authentification information').'.'.$end);    
	}
    }
    ##
    my (@userroles,%userupdate,$cnum,$cdom,%namechanged);
    if ($context eq 'course') {
        ($cnum,$cdom) = &Apache::lonuserutils::get_course_identity();
        $crstype = &Apache::loncommon::course_type($cdom.'_'.$cnum);
    }
    if (! $env{'form.makeuser'} ) {
        # Check for need to change
        my %userenv = &Apache::lonnet::get
            ('environment',['firstname','middlename','lastname','generation',
             'id','permanentemail','portfolioquota','inststatus','tools.aboutme',
             'tools.blog','tools.portfolio','requestcourses.official',
             'requestcourses.unofficial','requestcourses.community',
             'reqcrsotherdom.official','reqcrsotherdom.unofficial',
             'reqcrsotherdom.community'],
              $env{'form.ccdomain'},$env{'form.ccuname'});
        my ($tmp) = keys(%userenv);
        if ($tmp =~ /^(con_lost|error)/i) { 
            %userenv = ();
        }
        my $no_forceid_alert;
        # Check to see if user information can be changed
        my %domconfig =
            &Apache::lonnet::get_dom('configuration',['usermodification'],
                                     $env{'form.ccdomain'});
        my @statuses = ('active','future');
        my %roles = &Apache::lonnet::get_my_roles($env{'form.ccuname'},$env{'form.ccdomain'},'userroles',\@statuses,undef,$env{'request.role.domain'});
        my ($auname,$audom);
        if ($context eq 'author') {
            $auname = $env{'user.name'};
            $audom = $env{'user.domain'};     
        }
        foreach my $item (keys(%roles)) {
            my ($rolenum,$roledom,$role) = split(/:/,$item,-1);
            if ($context eq 'course') {
                if ($cnum ne '' && $cdom ne '') {
                    if ($rolenum eq $cnum && $roledom eq $cdom) {
                        if (!grep(/^\Q$role\E$/,@userroles)) {
                            push(@userroles,$role);
                        }
                    }
                }
            } elsif ($context eq 'author') {
                if ($rolenum eq $auname && $roledom eq $audom) {
                    if (!grep(/^\Q$role\E$/,@userroles)) { 
                        push(@userroles,$role);
                    }
                }
            }
        }
        if ($env{'form.action'} eq 'singlestudent') {
            if (!grep(/^st$/,@userroles)) {
                push(@userroles,'st');
            }
        } else {
            # Check for course or co-author roles being activated or re-enabled
            if ($context eq 'author' || $context eq 'course') {
                foreach my $key (keys(%env)) {
                    if ($context eq 'author') {
                        if ($key=~/^form\.act_\Q$audom\E_\Q$auname\E_([^_]+)/) {
                            if (!grep(/^\Q$1\E$/,@userroles)) {
                                push(@userroles,$1);
                            }
                        } elsif ($key =~/^form\.ren\:\Q$audom\E\/\Q$auname\E_([^_]+)/) {
                            if (!grep(/^\Q$1\E$/,@userroles)) {
                                push(@userroles,$1);
                            }
                        }
                    } elsif ($context eq 'course') {
                        if ($key=~/^form\.act_\Q$cdom\E_\Q$cnum\E_([^_]+)/) {
                            if (!grep(/^\Q$1\E$/,@userroles)) {
                                push(@userroles,$1);
                            }
                        } elsif ($key =~/^form\.ren\:\Q$cdom\E\/\Q$cnum\E(\/?\w*)_([^_]+)/) {
                            if (!grep(/^\Q$1\E$/,@userroles)) {
                                push(@userroles,$1);
                            }
                        }
                    }
                }
            }
        }
        #Check to see if we can change personal data for the user 
        my (@mod_disallowed,@longroles);
        foreach my $role (@userroles) {
            if ($role eq 'cr') {
                push(@longroles,'Custom');
            } else {
                push(@longroles,&Apache::lonnet::plaintext($role,$crstype)); 
            }
        }
        my %canmodify = &Apache::lonuserutils::can_modify_userinfo($context,$env{'form.ccdomain'},\@userinfo,\@userroles);
        foreach my $item (@userinfo) {
            # Strip leading and trailing whitespace
            $env{'form.c'.$item} =~ s/(\s+$|^\s+)//g;
            if (!$canmodify{$item}) {
                if (defined($env{'form.c'.$item})) {
                    if ($env{'form.c'.$item} ne $userenv{$item}) {
                        push(@mod_disallowed,$item);
                    }
                }
                $env{'form.c'.$item} = $userenv{$item};
            }
        }
        # Check to see if we can change the Student/Employee ID
        my $forceid = $env{'form.forceid'};
        my $recurseid = $env{'form.recurseid'};
        my (%alerts,%rulematch,%idinst_results,%curr_rules,%got_rules);
        my %uidhash = &Apache::lonnet::idrget($env{'form.ccdomain'},
                                            $env{'form.ccuname'});
        if (($uidhash{$env{'form.ccuname'}}) && 
            ($uidhash{$env{'form.ccuname'}}!~/error\:/) && 
            (!$forceid)) {
            if ($env{'form.cid'} ne $uidhash{$env{'form.ccuname'}}) {
                $env{'form.cid'} = $userenv{'id'};
                $no_forceid_alert = &mt('New student/employee ID does not match existing ID for this user.')
                                   .'<br />'
                                   .&mt("Change is not permitted without checking the 'Force ID change' checkbox on the previous page.")
                                   .'<br />'."\n";
            }
        }
        if ($env{'form.cid'} ne $userenv{'id'}) {
            my $checkhash;
            my $checks = { 'id' => 1 };
            $checkhash->{$env{'form.ccuname'}.':'.$env{'form.ccdomain'}} = 
                   { 'newuser' => $newuser,
                     'id'  => $env{'form.cid'}, 
                   };
            &Apache::loncommon::user_rule_check($checkhash,$checks,
                \%alerts,\%rulematch,\%idinst_results,\%curr_rules,\%got_rules);
            if (ref($alerts{'id'}) eq 'HASH') {
                if (ref($alerts{'id'}{$env{'form.ccdomain'}}) eq 'HASH') {
                   $env{'form.cid'} = $userenv{'id'};
                }
            }
        }
        my ($quotachanged,$oldportfolioquota,$newportfolioquota,$oldinststatus,
            $newinststatus,$oldisdefault,$newisdefault,%oldsettings,
            %oldsettingstext,%newsettings,%newsettingstext,@disporder,
            $olddefquota,$oldsettingstatus,$newdefquota,$newsettingstatus);
        @disporder = ('inststatus');
        if ($env{'request.role.domain'} eq $env{'form.ccdomain'}) {
            push(@disporder,'requestcourses');
        } else {
            push(@disporder,'reqcrsotherdom');
        }
        push(@disporder,('quota','tools'));
        $oldinststatus = $userenv{'inststatus'};
        ($olddefquota,$oldsettingstatus) =
            &Apache::loncommon::default_quota($env{'form.ccdomain'},$oldinststatus);
        ($newdefquota,$newsettingstatus) = ($olddefquota,$oldsettingstatus);
        my %canshow;
        if (&Apache::lonnet::allowed('mpq',$env{'form.ccdomain'})) {
            $canshow{'quota'} = 1;
        }
        if (&Apache::lonnet::allowed('mut',$env{'form.ccdomain'})) {
            $canshow{'tools'} = 1;
        }
        if (&Apache::lonnet::allowed('ccc',$env{'form.ccdomain'})) {
            $canshow{'requestcourses'} = 1;
        } elsif (&Apache::lonnet::allowed('ccc',$env{'request.role.domain'})) {
            $canshow{'reqcrsotherdom'} = 1;
        }
        if (&Apache::lonnet::allowed('mau',$env{'form.ccdomain'})) {
            $canshow{'inststatus'} = 1;
        }
        my (%changeHash,%changed);
        if ($oldinststatus eq '') {
            $oldsettings{'inststatus'} = $othertitle;
        } else {
            if (ref($usertypes) eq 'HASH') {
                $oldsettings{'inststatus'} = join(', ',map{ $usertypes->{ &unescape($_) }; } (split(/:/,$userenv{'inststatus'})));
            } else {
                $oldsettings{'inststatus'} = join(', ',map{ &unescape($_); } (split(/:/,$userenv{'inststatus'})));
            }
        }
        $changeHash{'inststatus'} = $userenv{'inststatus'};
        if ($canmodify_status{'inststatus'}) {
            $canshow{'inststatus'} = 1;
            if (exists($env{'form.inststatus'})) {
                my @inststatuses = &Apache::loncommon::get_env_multiple('form.inststatus');
                if (@inststatuses > 0) {
                    $newinststatus = join(':',map { &escape($_); } @inststatuses);
                    $changeHash{'inststatus'} = $newinststatus;
                    if ($newinststatus ne $oldinststatus) {
                        $changed{'inststatus'} = $newinststatus;
                        ($newdefquota,$newsettingstatus) =
                            &Apache::loncommon::default_quota($env{'form.ccdomain'},$newinststatus);
                    }
                    if (ref($usertypes) eq 'HASH') {
                        $newsettings{'inststatus'} = join(', ',map{ $usertypes->{$_}; } (@inststatuses));
                    } else {
                        $newsettings{'inststatus'} = join(', ',@inststatuses);
                    }
                }
            } else {
                $newinststatus = '';
                $changeHash{'inststatus'} = $newinststatus;
                $newsettings{'inststatus'} = $othertitle;
                if ($newinststatus ne $oldinststatus) {
                    $changed{'inststatus'} = $changeHash{'inststatus'};
                    ($newdefquota,$newsettingstatus) =
                        &Apache::loncommon::default_quota($env{'form.ccdomain'},$newinststatus);
                }
            }
        } elsif ($context ne 'selfcreate') {
            $canshow{'inststatus'} = 1;
            $newsettings{'inststatus'} = $oldsettings{'inststatus'};
        }
        $changeHash{'portfolioquota'} = $userenv{'portfolioquota'};
        if ($context eq 'domain') {
            if ($userenv{'portfolioquota'} ne '') {
                $oldportfolioquota = $userenv{'portfolioquota'};
                if ($env{'form.customquota'} == 1) {
                    if ($env{'form.portfolioquota'} eq '') {
                        $newportfolioquota = 0;
                    } else {
                        $newportfolioquota = $env{'form.portfolioquota'};
                        $newportfolioquota =~ s/[^\d\.]//g;
                    }
                    if ($newportfolioquota != $oldportfolioquota) {
                        $changed{'quota'} = &quota_admin($newportfolioquota,\%changeHash);
                    }
                } else {
                    $changed{'quota'} = &quota_admin('',\%changeHash);
                    $newportfolioquota = $newdefquota;
                    $newisdefault = 1;
                }
            } else {
                $oldisdefault = 1;
                $oldportfolioquota = $olddefquota;
                if ($env{'form.customquota'} == 1) {
                    if ($env{'form.portfolioquota'} eq '') {
                        $newportfolioquota = 0;
                    } else {
                        $newportfolioquota = $env{'form.portfolioquota'};
                        $newportfolioquota =~ s/[^\d\.]//g;
                    }
                    $changed{'quota'} = &quota_admin($newportfolioquota,\%changeHash);
                } else {
                    $newportfolioquota = $newdefquota;
                    $newisdefault = 1;
                }
            }
            if ($oldisdefault) {
                $oldsettingstext{'quota'} = &get_defaultquota_text($oldsettingstatus);
            }
            if ($newisdefault) {
                $newsettingstext{'quota'} = &get_defaultquota_text($newsettingstatus);
            }
            &tool_changes('tools',\@usertools,\%oldsettings,\%oldsettingstext,\%userenv,
                          \%changeHash,\%changed,\%newsettings,\%newsettingstext);
            if ($env{'form.ccdomain'} eq $env{'request.role.domain'}) {
                &tool_changes('requestcourses',\@requestcourses,\%oldsettings,\%oldsettingstext,
                              \%userenv,\%changeHash,\%changed,\%newsettings,\%newsettingstext);
            } else {
                &tool_changes('reqcrsotherdom',\@requestcourses,\%oldsettings,\%oldsettingstext,
                              \%userenv,\%changeHash,\%changed,\%newsettings,\%newsettingstext);
            }
        }
        foreach my $item (@userinfo) {
            if ($env{'form.c'.$item} ne $userenv{$item}) {
                $namechanged{$item} = 1;
            }
        }
        $oldsettings{'quota'} = $oldportfolioquota.' Mb';
        $newsettings{'quota'} = $newportfolioquota.' Mb';
        if ((keys(%namechanged) > 0) || (keys(%changed) > 0)) {
            my ($chgresult,$namechgresult);
            if (keys(%changed) > 0) {
                $chgresult = 
                    &Apache::lonnet::put('environment',\%changeHash,
                                  $env{'form.ccdomain'},$env{'form.ccuname'});
                if ($chgresult eq 'ok') {
                    if (($env{'user.name'} eq $env{'form.ccuname'}) &&
                        ($env{'user.domain'} eq $env{'form.ccdomain'})) {
                        my %newenvhash;
                        foreach my $key (keys(%changed)) {
                            if (($key eq 'official') || ($key eq 'unofficial')
                                || ($key eq 'community')) {
                                $newenvhash{'environment.requestcourses.'.$key} =
                                    $changeHash{'requestcourses.'.$key};
                                if ($changeHash{'requestcourses.'.$key} ne '') {
                                    $newenvhash{'environment.canrequest.'.$key} = 1;
                                } else {
                                    $newenvhash{'environment.canrequest.'.$key} =
          &Apache::lonnet::usertools_access($env{'user.name'},$env{'user.domain'},
                                            $key,'reload','requestcourses');
                                }
                            } elsif ($key ne 'quota') {
                                $newenvhash{'environment.tools.'.$key} = 
                                    $changeHash{'tools.'.$key};
                                if ($changeHash{'tools.'.$key} ne '') {
                                    $newenvhash{'environment.availabletools.'.$key} =
                                        $changeHash{'tools.'.$key};
                                } else {
                                    $newenvhash{'environment.availabletools.'.$key} =
          &Apache::lonnet::usertools_access($env{'user.name'},$env{'user.domain'},                                            $key,'reload','tools');
                                }
                            }
                        }
                        if (keys(%newenvhash)) {
                            &Apache::lonnet::appenv(\%newenvhash);
                        }
                    }
                }
            }
            if (keys(%namechanged) > 0) {
                foreach my $field (@userinfo) {
                    $changeHash{$field}  = $env{'form.c'.$field};
                }
# Make the change
                $namechgresult =
                    &Apache::lonnet::modifyuser($env{'form.ccdomain'},
                        $env{'form.ccuname'},$changeHash{'id'},undef,undef,
                        $changeHash{'firstname'},$changeHash{'middlename'},
                        $changeHash{'lastname'},$changeHash{'generation'},
                        $changeHash{'id'},undef,$changeHash{'permanentemail'},undef,\@userinfo);
                %userupdate = (
                               lastname   => $env{'form.clastname'},
                               middlename => $env{'form.cmiddlename'},
                               firstname  => $env{'form.cfirstname'},
                               generation => $env{'form.cgeneration'},
                               id         => $env{'form.cid'},
                             );
            }
            if (((keys(%namechanged) > 0) && $namechgresult eq 'ok') ||
                ((keys(%changed) > 0) && $chgresult eq 'ok')) {
            # Tell the user we changed the name
                &display_userinfo($r,1,\@disporder,\%canshow,\@requestcourses,
                                  \@usertools,\%userenv,\%changed,\%namechanged,
                                  \%oldsettings, \%oldsettingstext,\%newsettings,
                                  \%newsettingstext);

                if ($env{'form.cid'} ne $userenv{'id'}) {
                    &Apache::lonnet::idput($env{'form.ccdomain'},
                         ($env{'form.ccuname'} => $env{'form.cid'}));
                    if (($recurseid) &&
                        (&Apache::lonnet::allowed('mau',$env{'form.ccdomain'}))) {
                        my $idresult = 
                            &Apache::lonuserutils::propagate_id_change(
                                $env{'form.ccuname'},$env{'form.ccdomain'},
                                \%userupdate);
                        $r->print('<br />'.$idresult.'<br />');
                    }
                }
                if (($env{'form.ccdomain'} eq $env{'user.domain'}) && 
                    ($env{'form.ccuname'} eq $env{'user.name'})) {
                    my %newenvhash;
                    foreach my $key (keys(%changeHash)) {
                        $newenvhash{'environment.'.$key} = $changeHash{$key};
                    }
                    &Apache::lonnet::appenv(\%newenvhash);
                }
            } else { # error occurred
                $r->print('<span class="LC_error">'.&mt('Unable to successfully change environment for').' '.
                      $env{'form.ccuname'}.' '.&mt('in domain').' '.
                      $env{'form.ccdomain'}.'</span><br />');
            }
        } else { # End of if ($env ... ) logic
            # They did not want to change the users name, quota, tool availability,
            # or ability to request creation of courses, 
            # but we can still tell them what the name and quota and availabilities are  
            &display_userinfo($r,undef,\@disporder,\%canshow,\@requestcourses,
                              \@usertools,\%userenv,\%changed,\%namechanged,\%oldsettings,
                              \%oldsettingstext,\%newsettings,\%newsettingstext);
        }
        if (@mod_disallowed) {
            my ($rolestr,$contextname);
            if (@longroles > 0) {
                $rolestr = join(', ',@longroles);
            } else {
                $rolestr = &mt('No roles');
            }
            if ($context eq 'course') {
                $contextname = &mt('course');
            } elsif ($context eq 'author') {
                $contextname = &mt('co-author');
            }
            $r->print(&mt('The following fields were not updated: ').'<ul>');
            my %fieldtitles = &Apache::loncommon::personal_data_fieldtitles();
            foreach my $field (@mod_disallowed) {
                $r->print('<li>'.$fieldtitles{$field}.'</li>'."\n"); 
            }
            $r->print('</ul>');
            if (@mod_disallowed == 1) {
                $r->print(&mt("You do not have the authority to change this field given the user's current set of active/future [_1] roles:",$contextname));
            } else {
                $r->print(&mt("You do not have the authority to change these fields given the user's current set of active/future [_1] roles:",$contextname));
            }
            my $helplink = 'javascript:helpMenu('."'display'".')';
            $r->print('<span class="LC_cusr_emph">'.$rolestr.'</span><br />'
                     .&mt('Please contact your [_1]helpdesk[_2] for more information.'
                         ,'<a href="'.$helplink.'">','</a>')
                      .'<br />');
        }
        $r->print('<span class="LC_warning">'
                  .$no_forceid_alert
                  .&Apache::lonuserutils::print_namespacing_alerts($env{'form.ccdomain'},\%alerts,\%curr_rules)
                  .'</span>');
    }
    if ($env{'form.action'} eq 'singlestudent') {
        &enroll_single_student($r,$uhome,$amode,$genpwd,$now,$newuser,$context,$crstype);
        $r->print('<p><a href="javascript:backPage(document.userupdate)">');
        if ($crstype eq 'Community') {
            $r->print(&mt('Enroll Another Member'));
        } else {
            $r->print(&mt('Enroll Another Student'));
        }
        $r->print('</a></p>');
    } else {
        my @rolechanges = &update_roles($r,$context);
        if (keys(%namechanged) > 0) {
            if ($context eq 'course') {
                if (@userroles > 0) {
                    if ((@rolechanges == 0) || 
                        (!(grep(/^st$/,@rolechanges)))) {
                        if (grep(/^st$/,@userroles)) {
                            my $classlistupdated =
                                &Apache::lonuserutils::update_classlist($cdom,
                                              $cnum,$env{'form.ccdomain'},
                                       $env{'form.ccuname'},\%userupdate);
                        }
                    }
                }
            }
        }
        my $userinfo = &Apache::loncommon::plainname($env{'form.ccuname'},
                                                     $env{'form.ccdomain'});
        if ($env{'form.popup'}) {
            $r->print('<p><a href="javascript:window.close()">'.&mt('Close window').'</a></p>');
        } else {
            $r->print('<p><a href="javascript:backPage(document.userupdate,'."'$env{'form.prevphase'}','modify'".')">'
                     .&mt('Modify this user: [_1]','<span class="LC_cusr_emph">'.$env{'form.ccuname'}.':'.$env{'form.ccdomain'}.' ('.$userinfo.')</span>').'</a>'
                     .('&nbsp;'x5).'<a href="javascript:backPage(document.userupdate)">'
                     .&mt('Create/Modify Another User').'</a></p>');
        }
    }
    $r->print(&Apache::loncommon::end_page());
}

sub display_userinfo {
    my ($r,$changed,$order,$canshow,$requestcourses,$usertools,$userenv,
        $changedhash,$namechangedhash,$oldsetting,$oldsettingtext,
        $newsetting,$newsettingtext) = @_;
    return unless (ref($order) eq 'ARRAY' &&
                   ref($canshow) eq 'HASH' &&
                   ref($requestcourses) eq 'ARRAY' &&
                   ref($usertools) eq 'ARRAY' &&
                   ref($userenv) eq 'HASH' &&
                   ref($changedhash) eq 'HASH' &&
                   ref($oldsetting) eq 'HASH' &&
                   ref($oldsettingtext) eq 'HASH' &&
                   ref($newsetting) eq 'HASH' &&
                   ref($newsettingtext) eq 'HASH');
    my %lt=&Apache::lonlocal::texthash(
         'ui'             => 'User Information (unchanged)',
         'uic'            => 'User Information Changed',
         'firstname'      => 'First Name',
         'middlename'     => 'Middle Name',
         'lastname'       => 'Last Name',
         'generation'     => 'Generation',
         'id'             => 'Student/Employee ID',
         'permanentemail' => 'Permanent e-mail address',
         'quota'          => 'Disk space allocated to portfolio files',
         'blog'           => 'Blog Availability',
         'aboutme'        => 'Personal Information Page Availability',
         'portfolio'      => 'Portfolio Availability',
         'official'       => 'Can Request Official Courses',
         'unofficial'     => 'Can Request Unofficial Courses',
         'community'      => 'Can Request Communities',
         'inststatus'     => "Affiliation",
         'prvs'           => 'Previous Value:',
         'chto'           => 'Changed To:'
    );
    my $title = $lt{'ui'};
    if ($changed) {
        $title = $lt{'uic'};
    }
    $r->print('<h4>'.$title.'</h4>'.
              &Apache::loncommon::start_data_table().
              &Apache::loncommon::start_data_table_header_row());
    if ($changed) {
        $r->print("<th>&nbsp;</th>\n");
    }
    my @userinfo = ('firstname','middlename','lastname','generation','permanentemail','id');
    foreach my $item (@userinfo) {
        $r->print("<th>$lt{$item}</th>\n");
    }
    foreach my $entry (@{$order}) {
        if ($canshow->{$entry}) {
            if (($entry eq 'requestcourses') || ($entry eq 'reqcrsotherdom')) {
                foreach my $item (@{$requestcourses}) {
                    $r->print("<th>$lt{$item}</th>\n");
                }
            } elsif ($entry eq 'tools') {
                foreach my $item (@{$usertools}) {
                    $r->print("<th>$lt{$item}</th>\n");
                }
            } else {
                $r->print("<th>$lt{$entry}</th>\n");
            }
        }
    }
    $r->print(&Apache::loncommon::end_data_table_header_row().
             &Apache::loncommon::start_data_table_row());
    if ($changed) {
        $r->print('<td><b>'.$lt{'prvs'}.'</b></td>'."\n");
    }
    foreach my $item (@userinfo) {
        $r->print('<td>'.$userenv->{$item}.' </td>'."\n");
    }
    foreach my $entry (@{$order}) {
        if ($canshow->{$entry}) {
            if (($entry eq 'requestcourses') || ($entry eq 'reqcrsotherdom')) {
                foreach my $item (@{$requestcourses}) {
                    $r->print("<td>$oldsetting->{$item} $oldsettingtext->{$item}</td>\n");
                }
            } elsif ($entry eq 'tools') {
                foreach my $item (@{$usertools}) {
                    $r->print("<td>$oldsetting->{$item} $oldsettingtext->{$item}</td>\n");
                }
            } else {
                $r->print("<td>$oldsetting->{$entry} $oldsettingtext->{$entry} </td>\n");
            }
        }
    }
    $r->print(&Apache::loncommon::end_data_table_row());
    if ($changed) {
        $r->print(&Apache::loncommon::start_data_table_row().
                  '<td><span class="LC_nobreak"><b>'.$lt{'chto'}.'</b></span></td>'."\n");
        foreach my $item (@userinfo) {
            my $value = $env{'form.c'.$item};
            if ($namechangedhash->{$item}) {
                $value = '<span class="LC_cusr_emph">'.$value.'</span>';
            }
            $r->print("<td>$value </td>\n");
        }
        foreach my $entry (@{$order}) {
            if ($canshow->{$entry}) {
                if (($entry eq 'requestcourses') || ($entry eq 'reqcrsotherdom')) {
                    foreach my $item (@{$requestcourses}) {
                        my $value = $newsetting->{$item}.' '.$newsettingtext->{$item};
                        if ($changedhash->{$item}) {
                            $value = '<span class="LC_cusr_emph">'.$value.'</span>';
                        }
                        $r->print("<td>$value </td>\n");
                    }
                } elsif ($entry eq 'tools') {
                    foreach my $item (@{$usertools}) {
                        my $value = $newsetting->{$item}.' '.$newsettingtext->{$item};
                        if ($changedhash->{$item}) {
                            $value = '<span class="LC_cusr_emph">'.$value.'</span>';
                        }
                        $r->print("<td>$value </td>\n");
                    }
                } else {
                    my $value = $newsetting->{$entry}.' '.$newsettingtext->{$entry};
                    if ($changedhash->{$entry}) {
                        $value = '<span class="LC_cusr_emph">'.$value.'</span>';
                    }
                    $r->print("<td>$value </td>\n");
                }
            }
        }
        $r->print(&Apache::loncommon::end_data_table_row());
    }
    $r->print(&Apache::loncommon::end_data_table().'<br />');
    return;
}

sub tool_changes {
    my ($context,$usertools,$oldaccess,$oldaccesstext,$userenv,$changeHash,
        $changed,$newaccess,$newaccesstext) = @_;
    if (!((ref($usertools) eq 'ARRAY') && (ref($oldaccess) eq 'HASH') &&
          (ref($oldaccesstext) eq 'HASH') && (ref($userenv) eq 'HASH') &&
          (ref($changeHash) eq 'HASH') && (ref($changed) eq 'HASH') &&
          (ref($newaccess) eq 'HASH') && (ref($newaccesstext) eq 'HASH'))) {
        return;
    }
    if ($context eq 'reqcrsotherdom') {
        my @options = ('approval','validate','autolimit');
        my $optregex = join('|',@options);
        my %reqdisplay = &courserequest_display();
        my $cdom = $env{'request.role.domain'};
        foreach my $tool (@{$usertools}) {
            $oldaccesstext->{$tool} = &mt('No');
            $newaccesstext->{$tool} = $oldaccesstext->{$tool};
            $changeHash->{$context.'.'.$tool} = $userenv->{$context.'.'.$tool};
            my $newop;
            if ($env{'form.'.$context.'_'.$tool}) {
                $newop = $env{'form.'.$context.'_'.$tool};
                if ($newop eq 'autolimit') {
                    my $limit = $env{'form.'.$context.'_'.$tool.'_limit'};
                    $limit =~ s/\D+//g;
                    $newop .= '='.$limit;
                }
            }
            if ($userenv->{$context.'.'.$tool} eq '') {
                if ($newop) {
                    $changed->{$tool}=&tool_admin($tool,$cdom.':'.$newop,
                                                  $changeHash,$context);
                    if ($changed->{$tool}) {
                        $newaccesstext->{$tool} = &mt('Yes');
                    } else {
                        $newaccesstext->{$tool} = $oldaccesstext->{$tool};
                    }
                }
            } else {
                my @curr = split(',',$userenv->{$context.'.'.$tool});
                my @new;
                my $changedoms;
                foreach my $req (@curr) {
                    if ($req =~ /^\Q$cdom\E\:($optregex\=?\d*)$/) {
                        $oldaccesstext->{$tool} = &mt('Yes');
                        my $oldop = $1;
                        if ($oldop ne $newop) {
                            $changedoms = 1;
                            foreach my $item (@curr) {
                                my ($reqdom,$option) = split(':',$item);
                                unless ($reqdom eq $cdom) {
                                    push(@new,$item);
                                }
                            }
                            if ($newop) {
                                push(@new,$cdom.':'.$newop);
                            }
                            @new = sort(@new);
                        }
                        last;
                    }
                }
                if ((!$changedoms) && ($newop)) {
                    $changedoms = 1;
                    @new = sort(@curr,$cdom.':'.$newop);
                }
                if ($changedoms) {
                    my $newdomstr;
                    if (@new) {
                        $newdomstr = join(',',@new);
                    }
                    $changed->{$tool}=&tool_admin($tool,$newdomstr,$changeHash,
                                                  $context);
                    if ($changed->{$tool}) {
                        if ($env{'form.'.$context.'_'.$tool}) {
                            if ($env{'form.'.$context.'_'.$tool} eq 'autolimit') {
                                my $limit = $env{'form.'.$context.'_'.$tool.'_limit'};
                                $limit =~ s/\D+//g;
                                if ($limit) {
                                    $newaccesstext->{$tool} = &mt('Yes, up to limit of [quant,_1,request] per user.',$limit);
                                } else {
                                    $newaccesstext->{$tool} = &mt('Yes, processed automatically');
                                }
                            } else {
                                $newaccesstext->{$tool} = $reqdisplay{$env{'form.'.$context.'_'.$tool}};
                            }
                        } else {
                            $newaccesstext->{$tool} = &mt('No');
                        }
                    }
                }
            }
        }
        return;
    }
    foreach my $tool (@{$usertools}) {
        my $newval;
        if ($context eq 'requestcourses') {
            $newval = $env{'form.crsreq_'.$tool};
            if ($newval eq 'autolimit') {
                $newval .= '='.$env{'form.crsreq_'.$tool.'_limit'};
            }
        } else {
            $newval = $env{'form.'.$context.'_'.$tool};
        }
        if ($userenv->{$context.'.'.$tool} ne '') {
            $oldaccess->{$tool} = &mt('custom');
            if ($userenv->{$context.'.'.$tool}) {
                $oldaccesstext->{$tool} = &mt("availability set to 'on'");
            } else {
                $oldaccesstext->{$tool} = &mt("availability set to 'off'");
            }
            $changeHash->{$context.'.'.$tool} = $userenv->{$context.'.'.$tool};
            if ($env{'form.custom'.$tool} == 1) {
                if ($newval ne $userenv->{$context.'.'.$tool}) {
                    $changed->{$tool} = &tool_admin($tool,$newval,$changeHash,
                                                    $context);
                    if ($changed->{$tool}) {
                        $newaccess->{$tool} = &mt('custom');
                        if ($newval) {
                            $newaccesstext->{$tool} = &mt("availability set to 'on'");
                        } else {
                            $newaccesstext->{$tool} = &mt("availability set to 'off'");
                        }
                    } else {
                        $newaccess->{$tool} = $oldaccess->{$tool};
                        if ($userenv->{$context.'.'.$tool}) {
                            $newaccesstext->{$tool} = &mt("availability set to 'on'");
                        } else {
                            $newaccesstext->{$tool} = &mt("availability set to 'off'");
                        }
                    }
                } else {
                    $newaccess->{$tool} = $oldaccess->{$tool};
                    $newaccesstext->{$tool} = $oldaccesstext->{$tool};
                }
            } else {
                $changed->{$tool} = &tool_admin($tool,'',$changeHash,$context);
                if ($changed->{$tool}) {
                    $newaccess->{$tool} = &mt('default');
                } else {
                    $newaccess->{$tool} = $oldaccess->{$tool};
                    if ($userenv->{$context.'.'.$tool}) {
                        $newaccesstext->{$tool} = &mt("availability set to 'on'");
                    } else {
                        $newaccesstext->{$tool} = &mt("availability set to 'off'");
                    }
                }
            }
        } else {
            $oldaccess->{$tool} = &mt('default');
            if ($env{'form.custom'.$tool} == 1) {
                $changed->{$tool} = &tool_admin($tool,$newval,$changeHash,
                                                $context);
                if ($changed->{$tool}) {
                    $newaccess->{$tool} = &mt('custom');
                    if ($newval) {
                        $newaccesstext->{$tool} = &mt("availability set to 'on'");
                    } else {
                        $newaccesstext->{$tool} = &mt("availability set to 'off'");
                    }
                } else {
                    $newaccess->{$tool} = $oldaccess->{$tool};
                }
            } else {
                $newaccess->{$tool} = $oldaccess->{$tool};
            }
        }
    }
    return;
}

sub update_roles {
    my ($r,$context) = @_;
    my $now=time;
    my @rolechanges;
    my %disallowed;
    $r->print('<h3>'.&mt('Modifying Roles').'</h3>');
    foreach my $key (keys (%env)) {
	next if (! $env{$key});
        next if ($key eq 'form.action');
	# Revoke roles
	if ($key=~/^form\.rev/) {
	    if ($key=~/^form\.rev\:([^\_]+)\_([^\_\.]+)$/) {
# Revoke standard role
		my ($scope,$role) = ($1,$2);
		my $result =
		    &Apache::lonnet::revokerole($env{'form.ccdomain'},
						$env{'form.ccuname'},
						$scope,$role,'','',$context);
	        $r->print(&mt('Revoking [_1] in [_2]: [_3]',
			      $role,$scope,'<b>'.$result.'</b>').'<br />');
		if ($role eq 'st') {
		    my $result = 
                        &Apache::lonuserutils::classlist_drop($scope,
                            $env{'form.ccuname'},$env{'form.ccdomain'},
			    $now);
		    $r->print($result);
		}
                if (!grep(/^\Q$role\E$/,@rolechanges)) {
                    push(@rolechanges,$role);
                }
	    }
	    if ($key=~m{^form\.rev\:([^_]+)_cr\.cr/($match_domain)/($match_username)/(\w+)$}s) {
# Revoke custom role
		$r->print(&mt('Revoking custom role:').
                      ' '.$4.' by '.$3.':'.$2.' in '.$1.': <b>'.
                      &Apache::lonnet::revokecustomrole($env{'form.ccdomain'},
				  $env{'form.ccuname'},$1,$2,$3,$4,'','',$context).
		'</b><br />');
                if (!grep(/^cr$/,@rolechanges)) {
                    push(@rolechanges,'cr');
                }
	    }
	} elsif ($key=~/^form\.del/) {
	    if ($key=~/^form\.del\:([^\_]+)\_([^\_\.]+)$/) {
# Delete standard role
		my ($scope,$role) = ($1,$2);
		my $result =
		    &Apache::lonnet::assignrole($env{'form.ccdomain'},
						$env{'form.ccuname'},
						$scope,$role,$now,0,1,'',
                                                $context);
	        $r->print(&mt('Deleting [_1] in [_2]: [_3]',$role,$scope,
			      '<b>'.$result.'</b>').'<br />');
		if ($role eq 'st') {
		    my $result = 
                        &Apache::lonuserutils::classlist_drop($scope,
                            $env{'form.ccuname'},$env{'form.ccdomain'},
			    $now);
		    $r->print($result);
		}
                if (!grep(/^\Q$role\E$/,@rolechanges)) {
                    push(@rolechanges,$role);
                }
            }
	    if ($key=~m{^form\.del\:([^_]+)_cr\.cr/($match_domain)/($match_username)/(\w+)$}) {
                my ($url,$rdom,$rnam,$rolename) = ($1,$2,$3,$4);
# Delete custom role
                $r->print(&mt('Deleting custom role [_1] by [_2] in [_3]',
                      $rolename,$rnam.':'.$rdom,$url).': <b>'.
                      &Apache::lonnet::assigncustomrole($env{'form.ccdomain'},
                         $env{'form.ccuname'},$url,$rdom,$rnam,$rolename,$now,
                         0,1,$context).'</b><br />');
                if (!grep(/^cr$/,@rolechanges)) {
                    push(@rolechanges,'cr');
                }
            }
	} elsif ($key=~/^form\.ren/) {
            my $udom = $env{'form.ccdomain'};
            my $uname = $env{'form.ccuname'};
# Re-enable standard role
	    if ($key=~/^form\.ren\:([^\_]+)\_([^\_\.]+)$/) {
                my $url = $1;
                my $role = $2;
                my $logmsg;
                my $output;
                if ($role eq 'st') {
                    if ($url =~ m-^/($match_domain)/($match_courseid)/?(\w*)$-) {
                        my $result = &Apache::loncommon::commit_studentrole(\$logmsg,$udom,$uname,$url,$role,$now,0,$1,$2,$3);
                        if (($result =~ /^error/) || ($result eq 'not_in_class') || ($result eq 'unknown_course') || ($result eq 'refused')) {
                            if ($result eq 'refused' && $logmsg) {
                                $output = $logmsg;
                            } else { 
                                $output = "Error: $result\n";
                            }
                        } else {
                            $output = &mt('Assigning').' '.$role.' in '.$url.
                                      &mt('starting').' '.localtime($now).
                                      ': <br />'.$logmsg.'<br />'.
                                      &mt('Add to classlist').': <b>ok</b><br />';
                        }
                    }
                } else {
		    my $result=&Apache::lonnet::assignrole($env{'form.ccdomain'},
                               $env{'form.ccuname'},$url,$role,0,$now,'','',
                               $context);
		    $output = &mt('Re-enabling [_1] in [_2]: [_3]',
			      $role,$url,'<b>'.$result.'</b>').'<br />';
		}
                $r->print($output);
                if (!grep(/^\Q$role\E$/,@rolechanges)) {
                    push(@rolechanges,$role);
                }
	    }
# Re-enable custom role
	    if ($key=~m{^form\.ren\:([^_]+)_cr\.cr/($match_domain)/($match_username)/(\w+)$}) {
                my ($url,$rdom,$rnam,$rolename) = ($1,$2,$3,$4);
                my $result = &Apache::lonnet::assigncustomrole(
                               $env{'form.ccdomain'}, $env{'form.ccuname'},
                               $url,$rdom,$rnam,$rolename,0,$now,undef,$context);
                $r->print(&mt('Re-enabling custom role [_1] by [_2] in [_3]: [_4]',
                          $rolename,$rnam.':'.$rdom,$url,'<b>'.$result.'</b>').'<br />');
                if (!grep(/^cr$/,@rolechanges)) {
                    push(@rolechanges,'cr');
                }
            }
	} elsif ($key=~/^form\.act/) {
            my $udom = $env{'form.ccdomain'};
            my $uname = $env{'form.ccuname'};
	    if ($key=~/^form\.act\_($match_domain)\_($match_courseid)\_cr_cr_($match_domain)_($match_username)_([^\_]+)$/) {
                # Activate a custom role
		my ($one,$two,$three,$four,$five)=($1,$2,$3,$4,$5);
		my $url='/'.$one.'/'.$two;
		my $full=$one.'_'.$two.'_cr_cr_'.$three.'_'.$four.'_'.$five;

                my $start = ( $env{'form.start_'.$full} ?
                              $env{'form.start_'.$full} :
                              $now );
                my $end   = ( $env{'form.end_'.$full} ?
                              $env{'form.end_'.$full} :
                              0 );
                                                                                     
                # split multiple sections
                my %sections = ();
                my $num_sections = &build_roles($env{'form.sec_'.$full},\%sections,$5);
                if ($num_sections == 0) {
                    $r->print(&Apache::loncommon::commit_customrole($udom,$uname,$url,$three,$four,$five,$start,$end,$context));
                } else {
		    my %curr_groups =
			&Apache::longroup::coursegroups($one,$two);
                    foreach my $sec (sort {$a cmp $b} keys %sections) {
                        if (($sec eq 'none') || ($sec eq 'all') || 
                            exists($curr_groups{$sec})) {
                            $disallowed{$sec} = $url;
                            next;
                        }
                        my $securl = $url.'/'.$sec;
		        $r->print(&Apache::loncommon::commit_customrole($udom,$uname,$securl,$three,$four,$five,$start,$end,$context));
                    }
                }
                if (!grep(/^cr$/,@rolechanges)) {
                    push(@rolechanges,'cr');
                }
	    } elsif ($key=~/^form\.act\_($match_domain)\_($match_name)\_([^\_]+)$/) {
		# Activate roles for sections with 3 id numbers
		# set start, end times, and the url for the class
		my ($one,$two,$three)=($1,$2,$3);
		my $start = ( $env{'form.start_'.$one.'_'.$two.'_'.$three} ? 
			      $env{'form.start_'.$one.'_'.$two.'_'.$three} : 
			      $now );
		my $end   = ( $env{'form.end_'.$one.'_'.$two.'_'.$three} ? 
			      $env{'form.end_'.$one.'_'.$two.'_'.$three} :
			      0 );
		my $url='/'.$one.'/'.$two;
                my $type = 'three';
                # split multiple sections
                my %sections = ();
                my $num_sections = &build_roles($env{'form.sec_'.$one.'_'.$two.'_'.$three},\%sections,$three);
                if ($num_sections == 0) {
                    $r->print(&Apache::loncommon::commit_standardrole($udom,$uname,$url,$three,$start,$end,$one,$two,'',$context));
                } else {
                    my %curr_groups = 
			&Apache::longroup::coursegroups($one,$two);
                    my $emptysec = 0;
                    foreach my $sec (sort {$a cmp $b} keys %sections) {
                        $sec =~ s/\W//g;
                        if ($sec ne '') {
                            if (($sec eq 'none') || ($sec eq 'all') || 
                                exists($curr_groups{$sec})) {
                                $disallowed{$sec} = $url;
                                next;
                            }
                            my $securl = $url.'/'.$sec;
                            $r->print(&Apache::loncommon::commit_standardrole($udom,$uname,$securl,$three,$start,$end,$one,$two,$sec,$context));
                        } else {
                            $emptysec = 1;
                        }
                    }
                    if ($emptysec) {
                        $r->print(&Apache::loncommon::commit_standardrole($udom,$uname,$url,$three,$start,$end,$one,$two,'',$context));
                    }
                }
                if (!grep(/^\Q$three\E$/,@rolechanges)) {
                    push(@rolechanges,$three);
                }
	    } elsif ($key=~/^form\.act\_([^\_]+)\_([^\_]+)$/) {
		# Activate roles for sections with two id numbers
		# set start, end times, and the url for the class
		my $start = ( $env{'form.start_'.$1.'_'.$2} ? 
			      $env{'form.start_'.$1.'_'.$2} : 
			      $now );
		my $end   = ( $env{'form.end_'.$1.'_'.$2} ? 
			      $env{'form.end_'.$1.'_'.$2} :
			      0 );
                my $one = $1;
                my $two = $2;
		my $url='/'.$one.'/';
                # split multiple sections
                my %sections = ();
                my $num_sections = &build_roles($env{'form.sec_'.$one.'_'.$two},\%sections,$two);
                if ($num_sections == 0) {
                    $r->print(&Apache::loncommon::commit_standardrole($udom,$uname,$url,$two,$start,$end,$one,undef,'',$context));
                } else {
                    my $emptysec = 0;
                    foreach my $sec (sort {$a cmp $b} keys %sections) {
                        if ($sec ne '') {
                            my $securl = $url.'/'.$sec;
                            $r->print(&Apache::loncommon::commit_standardrole($udom,$uname,$securl,$two,$start,$end,$one,undef,$sec,$context));
                        } else {
                            $emptysec = 1;
                        }
                    }
                    if ($emptysec) {
                        $r->print(&Apache::loncommon::commit_standardrole($udom,$uname,$url,$two,$start,$end,$one,undef,'',$context));
                    }
                }
                if (!grep(/^\Q$two\E$/,@rolechanges)) {
                    push(@rolechanges,$two);
                }
	    } else {
		$r->print('<p><span class="LC_error">'.&mt('ERROR').': '.&mt('Unknown command').' <tt>'.$key.'</tt></span></p><br />');
            }
            foreach my $key (sort(keys(%disallowed))) {
                $r->print('<p class="LC_warning">');
                if (($key eq 'none') || ($key eq 'all')) {  
                    $r->print(&mt('[_1] may not be used as the name for a section, as it is a reserved word.','<tt>'.$key.'</tt>'));
                } else {
                    $r->print(&mt('[_1] may not be used as the name for a section, as it is the name of a course group.','<tt>'.$key.'</tt>'));
                }
                $r->print('</p><p>'
                         .&mt('Please [_1]go back[_2] and choose a different section name.'
                             ,'<a href="javascript:history.go(-1)'
                             ,'</a>')
                         .'</p><br />'
                );
            }
	}
    } # End of foreach (keys(%env))
# Flush the course logs so reverse user roles immediately updated
    &Apache::lonnet::flushcourselogs();
    if (@rolechanges == 0) {
        $r->print(&mt('No roles to modify'));
    }
    return @rolechanges;
}

sub enroll_single_student {
    my ($r,$uhome,$amode,$genpwd,$now,$newuser,$context,$crstype) = @_;
    $r->print('<h3>');
    if ($crstype eq 'Community') {
        $r->print(&mt('Enrolling Member'));
    } else {
        $r->print(&mt('Enrolling Student'));
    }
    $r->print('</h3>');

    # Remove non alphanumeric values from section
    $env{'form.sections'}=~s/\W//g;

    # Clean out any old student roles the user has in this class.
    &Apache::lonuserutils::modifystudent($env{'form.ccdomain'},
         $env{'form.ccuname'},$env{'request.course.id'},undef,$uhome);
    my ($startdate,$enddate) = &Apache::lonuserutils::get_dates_from_form();
    my $enroll_result =
        &Apache::lonnet::modify_student_enrollment($env{'form.ccdomain'},
            $env{'form.ccuname'},$env{'form.cid'},$env{'form.cfirstname'},
            $env{'form.cmiddlename'},$env{'form.clastname'},
            $env{'form.generation'},$env{'form.sections'},$enddate,
            $startdate,'manual',undef,$env{'request.course.id'},'',$context);
    if ($enroll_result =~ /^ok/) {
        $r->print(&mt('<b>[_1]</b> enrolled',$env{'form.ccuname'}.':'.$env{'form.ccdomain'}));
        if ($env{'form.sections'} ne '') {
            $r->print(' '.&mt('in section [_1]',$env{'form.sections'}));
        }
        my ($showstart,$showend);
        if ($startdate <= $now) {
            $showstart = &mt('Access starts immediately');
        } else {
            $showstart = &mt('Access starts: ').&Apache::lonlocal::locallocaltime($startdate);
        }
        if ($enddate == 0) {
            $showend = &mt('ends: no ending date');
        } else {
            $showend = &mt('ends: ').&Apache::lonlocal::locallocaltime($enddate);
        }
        $r->print('.<br />'.$showstart.'; '.$showend);
        if ($startdate <= $now && !$newuser) {
            $r->print('<p> ');
            if ($crstype eq 'Community') {
                $r->print(&mt('If the member is currently logged-in to LON-CAPA, the new role will be available when the member next logs in.'));
            } else {
                $r->print(&mt('If the student is currently logged-in to LON-CAPA, the new role will be available when the student next logs in.'));
            }
            $r->print('</p>');
        }
    } else {
        $r->print(&mt('unable to enroll').": ".$enroll_result);
    }
    return;
}

sub get_defaultquota_text {
    my ($settingstatus) = @_;
    my $defquotatext; 
    if ($settingstatus eq '') {
        $defquotatext = &mt('(default)');
    } else {
        my ($usertypes,$order) =
            &Apache::lonnet::retrieve_inst_usertypes($env{'form.ccdomain'});
        if ($usertypes->{$settingstatus} eq '') {
            $defquotatext = &mt('(default)');
        } else {
            $defquotatext = &mt('(default for [_1])',$usertypes->{$settingstatus});
        }
    }
    return $defquotatext;
}

sub update_result_form {
    my ($uhome) = @_;
    my $outcome = 
    '<form name="userupdate" method="post" />'."\n";
    foreach my $item ('srchby','srchin','srchtype','srchterm','srchdomain','ccuname','ccdomain') {
        $outcome .= '<input type="hidden" name="'.$item.'" value="'.$env{'form.'.$item}.'" />'."\n";
    }
    if ($env{'form.origname'} ne '') {
        $outcome .= '<input type="hidden" name="origname" value="'.$env{'form.origname'}.'" />'."\n";
    }
    foreach my $item ('sortby','seluname','seludom') {
        if (exists($env{'form.'.$item})) {
            $outcome .= '<input type="hidden" name="'.$item.'" value="'.$env{'form.'.$item}.'" />'."\n";
        }
    }
    if ($uhome eq 'no_host') {
        $outcome .= '<input type="hidden" name="forcenewuser" value="1" />'."\n";
    }
    $outcome .= '<input type="hidden" name="phase" value="" />'."\n".
                '<input type ="hidden" name="currstate" value="" />'."\n".
                '<input type ="hidden" name="action" value="'.$env{'form.action'}.'" />'."\n".
                '</form>';
    return $outcome;
}

sub quota_admin {
    my ($setquota,$changeHash) = @_;
    my $quotachanged;
    if (&Apache::lonnet::allowed('mpq',$env{'form.ccdomain'})) {
        # Current user has quota modification privileges
        if (ref($changeHash) eq 'HASH') {
            $quotachanged = 1;
            $changeHash->{'portfolioquota'} = $setquota;
        }
    }
    return $quotachanged;
}

sub tool_admin {
    my ($tool,$settool,$changeHash,$context) = @_;
    my $canchange = 0; 
    if ($context eq 'requestcourses') {
        if (&Apache::lonnet::allowed('ccc',$env{'form.ccdomain'})) {
            $canchange = 1;
        }
    } elsif ($context eq 'reqcrsotherdom') {
        if (&Apache::lonnet::allowed('ccc',$env{'request.role.domain'})) {
            $canchange = 1;
        }
    } elsif (&Apache::lonnet::allowed('mut',$env{'form.ccdomain'})) {
        # Current user has quota modification privileges
        $canchange = 1;
    }
    my $toolchanged;
    if ($canchange) {
        if (ref($changeHash) eq 'HASH') {
            $toolchanged = 1;
            $changeHash->{$context.'.'.$tool} = $settool;
        }
    }
    return $toolchanged;
}

sub build_roles {
    my ($sectionstr,$sections,$role) = @_;
    my $num_sections = 0;
    if ($sectionstr=~ /,/) {
        my @secnums = split/,/,$sectionstr;
        if ($role eq 'st') {
            $secnums[0] =~ s/\W//g;
            $$sections{$secnums[0]} = 1;
            $num_sections = 1;
        } else {
            foreach my $sec (@secnums) {
                $sec =~ ~s/\W//g;
                if (!($sec eq "")) {
                    if (exists($$sections{$sec})) {
                        $$sections{$sec} ++;
                    } else {
                        $$sections{$sec} = 1;
                        $num_sections ++;
                    }
                }
            }
        }
    } else {
        $sectionstr=~s/\W//g;
        unless ($sectionstr eq '') {
            $$sections{$sectionstr} = 1;
            $num_sections ++;
        }
    }

    return $num_sections;
}

# ========================================================== Custom Role Editor

sub custom_role_editor {
    my ($r) = @_;
    my $action = $env{'form.customroleaction'};
    my $rolename;
    if ($action eq 'new') {
        $rolename=$env{'form.newrolename'};
    } else {
        $rolename=$env{'form.rolename'};
    }

    $rolename=~s/[^A-Za-z0-9]//gs;
    if (!$rolename || $env{'form.phase'} eq 'pickrole') {
	&print_username_entry_form($r);
        return;
    }
    my ($crstype,$context);
    if ($env{'request.course.id'}) {
        $crstype = &Apache::loncommon::course_type();
        $context = 'course';
    } else {
        $context = 'domain';
        $crstype = $env{'form.templatecrstype'};
    }
# ------------------------------------------------------- What can be assigned?
    my %full=();
    my %courselevel=();
    my %courselevelcurrent=();
    my $syspriv='';
    my $dompriv='';
    my $coursepriv='';
    my $body_top;
    my ($rdummy,$roledef)=
			 &Apache::lonnet::get('roles',["rolesdef_$rolename"]);
# ------------------------------------------------------- Does this role exist?
    $body_top .= '<h2>';
    if (($rdummy ne 'con_lost') && ($roledef ne '')) {
	$body_top .= &mt('Existing Role').' "';
# ------------------------------------------------- Get current role privileges
	($syspriv,$dompriv,$coursepriv)=split(/\_/,$roledef);
        if ($crstype eq 'Community') {
            $syspriv =~ s/bre\&S//;
        }
    } else {
	$body_top .= &mt('New Role').' "';
	$roledef='';
    }
    $body_top .= $rolename.'"</h2>';
    foreach my $item (split(/\:/,$Apache::lonnet::pr{'cr:c'})) {
	my ($priv,$restrict)=split(/\&/,$item);
        if (!$restrict) { $restrict='F'; }
        $courselevel{$priv}=$restrict;
        if ($coursepriv=~/\:$priv/) {
	    $courselevelcurrent{$priv}=1;
	}
	$full{$priv}=1;
    }
    my %domainlevel=();
    my %domainlevelcurrent=();
    foreach my $item (split(/\:/,$Apache::lonnet::pr{'cr:d'})) {
	my ($priv,$restrict)=split(/\&/,$item);
        if (!$restrict) { $restrict='F'; }
        $domainlevel{$priv}=$restrict;
        if ($dompriv=~/\:$priv/) {
	    $domainlevelcurrent{$priv}=1;
	}
	$full{$priv}=1;
    }
    my %systemlevel=();
    my %systemlevelcurrent=();
    foreach my $item (split(/\:/,$Apache::lonnet::pr{'cr:s'})) {
	my ($priv,$restrict)=split(/\&/,$item);
        if (!$restrict) { $restrict='F'; }
        $systemlevel{$priv}=$restrict;
        if ($syspriv=~/\:$priv/) {
	    $systemlevelcurrent{$priv}=1;
	}
	$full{$priv}=1;
    }
    my ($jsback,$elements) = &crumb_utilities();
    my $button_code = "\n";
    my $head_script = "\n";
    $head_script .= '<script type="text/javascript">'."\n".
                    '// <![CDATA['."\n";
    my @template_roles = ("in","ta","ep");
    if ($context eq 'domain') {
        push(@template_roles,"ad");
    }
    push(@template_roles,"st");
    if ($crstype eq 'Community') {
        unshift(@template_roles,'co');
    } else {
        unshift(@template_roles,'cc');
    }
    foreach my $role (@template_roles) {
        $head_script .= &make_script_template($role,$crstype);
        $button_code .= &make_button_code($role,$crstype).' ';
    }
    my $context_code;
    if ($context eq 'domain') {
        my $checkedCommunity = '';
        my $checkedCourse = ' checked="checked"';
        if ($env{'form.templatecrstype'} eq 'Community') {
            $checkedCommunity = $checkedCourse;
            $checkedCourse = '';
        }
        $context_code = '<label>'.
                        '<input type="radio" name="templatecrstype" value="Course"'.$checkedCourse.' onclick="this.form.submit();">'.
                        &mt('Course').
                        '</label>'.('&nbsp;' x2).
                        '<label>'.
                        '<input type="radio" name="templatecrstype" value="Community"'.$checkedCommunity.' onclick="this.form.submit();">'.
                        &mt('Community').
                        '</label>'.
                        '</fieldset>'.
                        '<input type="hidden" name="customroleaction" value="'.
                        $action.'" />';
        if ($env{'form.customroleaction'} eq 'new') {
            $context_code .= '<input type="hidden" name="newrolename" value="'.
                             $rolename.'" />';
        } else {
            $context_code .= '<input type="hidden" name="rolename" value="'.
                             $rolename.'" />';
        }
        $context_code .= '<input type="hidden" name="action" value="custom" />'.
                         '<input type="hidden" name="phase" value="selected_custom_edit" />';
    }
    $head_script .= "\n".$jsback."\n".
                    '// ]]>'."\n".
                    '</script>'."\n";
    $r->print(&Apache::loncommon::start_page('Custom Role Editor',$head_script));
    &Apache::lonhtmlcommon::add_breadcrumb
     ({href=>"javascript:backPage(document.form1,'pickrole','')",
       text=>"Pick custom role",
       faq=>282,bug=>'Instructor Interface',},
      {href=>"javascript:backPage(document.form1,'','')",
         text=>"Edit custom role",
         faq=>282,bug=>'Instructor Interface',});
    $r->print(&Apache::lonhtmlcommon::breadcrumbs('User Management',
                                                  'Course_Editing_Custom_Roles'));

    $r->print($body_top);
    my %lt=&Apache::lonlocal::texthash(
		    'prv'  => "Privilege",
		    'crl'  => "Course Level",
                    'dml'  => "Domain Level",
                    'ssl'  => "System Level");

    $r->print('<div class="LC_left_float">'
             .'<form action=""><fieldset>'
             .'<legend>'.&mt('Select a Template').'</legend>'
             .$button_code
             .'</fieldset></form></div>');
    if ($context_code) {
        $r->print('<div class="LC_left_float">'
                 .'<form action="/adm/createuser" method="post"><fieldset>'
                 .'<legend>'.&mt('Context').'</legend>'
                 .$context_code
                 .'</form>'
                 .'</div>'
        );
    }
    $r->print('<br clear="all" />');

    $r->print(<<ENDCCF);
<form name="form1" method="post">
<input type="hidden" name="phase" value="set_custom_roles" />
<input type="hidden" name="rolename" value="$rolename" />
ENDCCF
    $r->print(&Apache::loncommon::start_data_table().
              &Apache::loncommon::start_data_table_header_row(). 
'<th>'.$lt{'prv'}.'</th><th>'.$lt{'crl'}.'</th><th>'.$lt{'dml'}.
'</th><th>'.$lt{'ssl'}.'</th>'.
              &Apache::loncommon::end_data_table_header_row());
    foreach my $priv (sort(keys(%full))) {
        my $privtext = &Apache::lonnet::plaintext($priv,$crstype);
        $r->print(&Apache::loncommon::start_data_table_row().
	          '<td>'.$privtext.'</td><td>'.
    ($courselevel{$priv}?'<input type="checkbox" name="'.$priv.'_c"'.
    ($courselevelcurrent{$priv}?' checked="checked"':'').' />':'&nbsp;').
    '</td><td>'.
    ($domainlevel{$priv}?'<input type="checkbox" name="'.$priv.'_d"'.
    ($domainlevelcurrent{$priv}?' checked="checked"':'').' />':'&nbsp;').
    '</td><td>');
        if ($priv eq 'bre' && $crstype eq 'Community') {
            $r->print('&nbsp;');
        } else {
            $r->print($systemlevel{$priv}?'<input type="checkbox" name="'.$priv.'_s"'.
                      ($systemlevelcurrent{$priv}?' checked="checked"':'').' />':'&nbsp;');
        }
        $r->print('</td>'.
                  &Apache::loncommon::end_data_table_row());
    }
    $r->print(&Apache::loncommon::end_data_table().
   '<input type="hidden" name="action" value="'.$env{'form.action'}.'" />'.
   '<input type="hidden" name="startrolename" value="'.$env{'form.rolename'}.
   '" />'."\n".'<input type="hidden" name="currstate" value="" />'."\n".   
   '<input type="reset" value="'.&mt("Reset").'" />'."\n".
   '<input type="submit" value="'.&mt('Save').'" /></form>'.
	      &Apache::loncommon::end_page());
}
# --------------------------------------------------------
sub make_script_template {
    my ($role,$crstype) = @_;
    my %full_c=();
    my %full_d=();
    my %full_s=();
    my $return_script;
    foreach my $item (split(/\:/,$Apache::lonnet::pr{'cr:c'})) {
        my ($priv,$restrict)=split(/\&/,$item);
        $full_c{$priv}=1;
    }
    foreach my $item (split(/\:/,$Apache::lonnet::pr{'cr:d'})) {
        my ($priv,$restrict)=split(/\&/,$item);
        $full_d{$priv}=1;
    }
    foreach my $item (split(/\:/,$Apache::lonnet::pr{'cr:s'})) {
        next if (($crstype eq 'Community') && ($item eq 'bre&S'));
        my ($priv,$restrict)=split(/\&/,$item);
        $full_s{$priv}=1;
    }
    $return_script .= 'function set_'.$role.'() {'."\n";
    my @temp = split(/:/,$Apache::lonnet::pr{$role.':c'});
    my %role_c;
    foreach my $priv (@temp) {
        my ($priv_item, $dummy) = split(/\&/,$priv);
        $role_c{$priv_item} = 1;
    }
    my %role_d;
    @temp = split(/:/,$Apache::lonnet::pr{$role.':d'});
    foreach my $priv(@temp) {
        my ($priv_item, $dummy) = split(/\&/,$priv);
        $role_d{$priv_item} = 1;
    }
    my %role_s;
    @temp = split(/:/,$Apache::lonnet::pr{$role.':s'});
    foreach my $priv(@temp) {
        my ($priv_item, $dummy) = split(/\&/,$priv);
        $role_s{$priv_item} = 1;
    }
    foreach my $priv_item (keys(%full_c)) {
        my ($priv, $dummy) = split(/\&/,$priv_item);
        if ((exists($role_c{$priv})) || (exists($role_d{$priv})) || 
            (exists($role_s{$priv}))) {
            $return_script .= "document.form1.$priv"."_c.checked = true;\n";
        } else {
            $return_script .= "document.form1.$priv"."_c.checked = false;\n";
        }
    }
    foreach my $priv_item (keys(%full_d)) {
        my ($priv, $dummy) = split(/\&/,$priv_item);
        if ((exists($role_d{$priv})) || (exists($role_s{$priv}))) {
            $return_script .= "document.form1.$priv"."_d.checked = true;\n";
        } else {
            $return_script .= "document.form1.$priv"."_d.checked = false;\n";
        }
    }
    foreach my $priv_item (keys(%full_s)) {
        my ($priv, $dummy) = split(/\&/,$priv_item);
        if (exists($role_s{$priv})) {
            $return_script .= "document.form1.$priv"."_s.checked = true;\n";
        } else {
            $return_script .= "document.form1.$priv"."_s.checked = false;\n";
        }
    }
    $return_script .= '}'."\n";
    return ($return_script);
}
# ----------------------------------------------------------
sub make_button_code {
    my ($role,$crstype) = @_;
    my $label = &Apache::lonnet::plaintext($role,$crstype);
    my $button_code = '<input type="button" onclick="set_'.$role.'()" value="'.$label.'" />';
    return ($button_code);
}
# ---------------------------------------------------------- Call to definerole
sub set_custom_role {
    my ($r,$context) = @_;
    my $rolename=$env{'form.rolename'};
    $rolename=~s/[^A-Za-z0-9]//gs;
    if (!$rolename) {
	&custom_role_editor($r);
        return;
    }
    my ($jsback,$elements) = &crumb_utilities();
    my $jscript = '<script type="text/javascript">'.
                  '// <![CDATA['."\n".
                  $jsback."\n".
                  '// ]]>'."\n".
                  '</script>';

    $r->print(&Apache::loncommon::start_page('Save Custom Role'),$jscript);
    &Apache::lonhtmlcommon::add_breadcrumb
        ({href=>"javascript:backPage(document.customresult,'pickrole','')",
          text=>"Pick custom role",
          faq=>282,bug=>'Instructor Interface',},
         {href=>"javascript:backPage(document.customresult,'selected_custom_edit','')",
          text=>"Edit custom role",
          faq=>282,bug=>'Instructor Interface',},
         {href=>"javascript:backPage(document.customresult,'set_custom_roles','')",
          text=>"Result",
          faq=>282,bug=>'Instructor Interface',});
    $r->print(&Apache::lonhtmlcommon::breadcrumbs('User Management',
                                                  'Course_Editing_Custom_Roles'));

    my ($rdummy,$roledef)=
	&Apache::lonnet::get('roles',["rolesdef_$rolename"]);

# ------------------------------------------------------- Does this role exist?
    $r->print('<h3>');
    if (($rdummy ne 'con_lost') && ($roledef ne '')) {
	$r->print(&mt('Existing Role').' "');
    } else {
	$r->print(&mt('New Role').' "');
	$roledef='';
    }
    $r->print($rolename.'"</h3>');
# ------------------------------------------------------- What can be assigned?
    my $sysrole='';
    my $domrole='';
    my $courole='';

    foreach my $item (split(/\:/,$Apache::lonnet::pr{'cr:c'})) {
	my ($priv,$restrict)=split(/\&/,$item);
        if (!$restrict) { $restrict=''; }
        if ($env{'form.'.$priv.'_c'}) {
	    $courole.=':'.$item;
	}
    }

    foreach my $item (split(/\:/,$Apache::lonnet::pr{'cr:d'})) {
	my ($priv,$restrict)=split(/\&/,$item);
        if (!$restrict) { $restrict=''; }
        if ($env{'form.'.$priv.'_d'}) {
	    $domrole.=':'.$item;
	}
    }

    foreach my $item (split(/\:/,$Apache::lonnet::pr{'cr:s'})) {
	my ($priv,$restrict)=split(/\&/,$item);
        if (!$restrict) { $restrict=''; }
        if ($env{'form.'.$priv.'_s'}) {
	    $sysrole.=':'.$item;
	}
    }
    $r->print('<br />Defining Role: '.
	   &Apache::lonnet::definerole($rolename,$sysrole,$domrole,$courole));
    if ($env{'request.course.id'}) {
        my $url='/'.$env{'request.course.id'};
        $url=~s/\_/\//g;
	$r->print('<br />'.&mt('Assigning Role to Self').': '.
	      &Apache::lonnet::assigncustomrole($env{'user.domain'},
						$env{'user.name'},
						$url,
						$env{'user.domain'},
						$env{'user.name'},
						$rolename,undef,undef,undef,$context));
    }
    $r->print('<p><a href="javascript:backPage(document.customresult,'."'pickrole'".')">'.&mt('Create or edit another custom role').'</a></p><form name="customresult" method="post">');
    $r->print(&Apache::lonhtmlcommon::echo_form_input([]).'</form>');
    $r->print(&Apache::loncommon::end_page());
}

# ================================================================ Main Handler
sub handler {
    my $r = shift;
    if ($r->header_only) {
       &Apache::loncommon::content_type($r,'text/html');
       $r->send_http_header;
       return OK;
    }
    my ($context,$crstype);
    if ($env{'request.course.id'}) {
        $context = 'course';
        $crstype = &Apache::loncommon::course_type();
    } elsif ($env{'request.role'} =~ /^au\./) {
        $context = 'author';
    } else {
        $context = 'domain';
    }
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
        ['action','state','callingform','roletype','showrole','bulkaction','popup','phase',
         'username','domain','srchterm','srchdomain','srchin','srchby','srchtype']);
    &Apache::lonhtmlcommon::clear_breadcrumbs();
    if ($env{'form.action'} ne 'dateselect') {
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>"/adm/createuser",
              text=>"User Management",
              help=>'Course_Create_Class_List,Course_Change_Privileges,Course_View_Class_List,Course_Editing_Custom_Roles,Course_Add_Student,Course_Drop_Student,Course_Automated_Enrollment,Course_Self_Enrollment,Course_Manage_Group'});
    }
    #SD Following files not added to help, because the corresponding .tex-files seem to
    #be missing: Course_Approve_Selfenroll,Course_User_Logs,
    my ($permission,$allowed) = 
        &Apache::lonuserutils::get_permission($context,$crstype);
    if (!$allowed) {
        $env{'user.error.msg'}=
            "/adm/createuser:cst:0:0:Cannot create/modify user data ".
                                 "or view user status.";
        return HTTP_NOT_ACCEPTABLE;
    }

    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;

    # Main switch on form.action and form.state, as appropriate
    if (! exists($env{'form.action'})) {
        $r->print(&header());
        $r->print(&Apache::lonhtmlcommon::breadcrumbs('User Management'));
        $r->print(&print_main_menu($permission,$context,$crstype));
        $r->print(&Apache::loncommon::end_page());
    } elsif ($env{'form.action'} eq 'upload' && $permission->{'cusr'}) {
        $r->print(&header());
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/createuser?action=upload&state=',
              text=>"Upload Users List"});
        $r->print(&Apache::lonhtmlcommon::breadcrumbs('Upload Users List',
                                                   'Course_Create_Class_List'));
        $r->print('<form name="studentform" method="post" '.
                  'enctype="multipart/form-data" '.
                  ' action="/adm/createuser">'."\n");
        if (! exists($env{'form.state'})) {
            &Apache::lonuserutils::print_first_users_upload_form($r,$context);
        } elsif ($env{'form.state'} eq 'got_file') {
            &Apache::lonuserutils::print_upload_manager_form($r,$context,
                                                             $permission,$crstype);
        } elsif ($env{'form.state'} eq 'enrolling') {
            if ($env{'form.datatoken'}) {
                &Apache::lonuserutils::upfile_drop_add($r,$context,$permission);
            }
        } else {
            &Apache::lonuserutils::print_first_users_upload_form($r,$context);
        }
        $r->print('</form>'.&Apache::loncommon::end_page());
    } elsif ((($env{'form.action'} eq 'singleuser') || ($env{'form.action'}
             eq 'singlestudent')) && ($permission->{'cusr'})) {
        my $phase = $env{'form.phase'};
        my @search = ('srchterm','srchby','srchin','srchtype','srchdomain');
	&Apache::loncreateuser::restore_prev_selections();
	my $srch;
	foreach my $item (@search) {
	    $srch->{$item} = $env{'form.'.$item};
	}
        if (($phase eq 'get_user_info') || ($phase eq 'userpicked') ||
            ($phase eq 'createnewuser')) {
            if ($env{'form.phase'} eq 'createnewuser') {
                my $response;
                if ($env{'form.srchterm'} !~ /^$match_username$/) {
                    my $response = &mt('You must specify a valid username. Only the following are allowed: letters numbers - . @');
                    $env{'form.phase'} = '';
                    &print_username_entry_form($r,$context,$response,$srch,undef,$crstype);
                } else {
                    my $ccuname =&LONCAPA::clean_username($srch->{'srchterm'});
                    my $ccdomain=&LONCAPA::clean_domain($srch->{'srchdomain'});
                    &print_user_modification_page($r,$ccuname,$ccdomain,
                                                  $srch,$response,$context,
                                                  $permission,$crstype);
                }
            } elsif ($env{'form.phase'} eq 'get_user_info') {
                my ($currstate,$response,$forcenewuser,$results) = 
                    &user_search_result($context,$srch);
                if ($env{'form.currstate'} eq 'modify') {
                    $currstate = $env{'form.currstate'};
                }
                if ($currstate eq 'select') {
                    &print_user_selection_page($r,$response,$srch,$results,
                                               \@search,$context,undef,$crstype);
                } elsif ($currstate eq 'modify') {
                    my ($ccuname,$ccdomain);
                    if (($srch->{'srchby'} eq 'uname') && 
                        ($srch->{'srchtype'} eq 'exact')) {
                        $ccuname = $srch->{'srchterm'};
                        $ccdomain= $srch->{'srchdomain'};
                    } else {
                        my @matchedunames = keys(%{$results});
                        ($ccuname,$ccdomain) = split(/:/,$matchedunames[0]);
                    }
                    $ccuname =&LONCAPA::clean_username($ccuname);
                    $ccdomain=&LONCAPA::clean_domain($ccdomain);
                    if ($env{'form.forcenewuser'}) {
                        $response = '';
                    }
                    &print_user_modification_page($r,$ccuname,$ccdomain,
                                                  $srch,$response,$context,
                                                  $permission,$crstype);
                } elsif ($currstate eq 'query') {
                    &print_user_query_page($r,'createuser');
                } else {
                    $env{'form.phase'} = '';
                    &print_username_entry_form($r,$context,$response,$srch,
                                               $forcenewuser,$crstype);
                }
            } elsif ($env{'form.phase'} eq 'userpicked') {
                my $ccuname = &LONCAPA::clean_username($env{'form.seluname'});
                my $ccdomain = &LONCAPA::clean_domain($env{'form.seludom'});
                &print_user_modification_page($r,$ccuname,$ccdomain,$srch,'',
                                              $context,$permission,$crstype);
            }
        } elsif ($env{'form.phase'} eq 'update_user_data') {
            &update_user_data($r,$context,$crstype);
        } else {
            &print_username_entry_form($r,$context,undef,$srch,undef,$crstype);
        }
    } elsif ($env{'form.action'} eq 'custom' && $permission->{'custom'}) {
        if ($env{'form.phase'} eq 'set_custom_roles') {
            &set_custom_role($r,$context);
        } else {
            &custom_role_editor($r);
        }
    } elsif (($env{'form.action'} eq 'listusers') && 
             ($permission->{'view'} || $permission->{'cusr'})) {
        if ($env{'form.phase'} eq 'bulkchange') {
            &Apache::lonhtmlcommon::add_breadcrumb
                ({href=>'/adm/createuser?action=listusers',
                  text=>"List Users"},
                {href=>"/adm/createuser",
                  text=>"Result"});
            my $setting = $env{'form.roletype'};
            my $choice = $env{'form.bulkaction'};
            $r->print(&header());
            $r->print(&Apache::lonhtmlcommon::breadcrumbs("Update Users",
                                                          'Course_View_Class_List'));
            if ($permission->{'cusr'}) {
                &Apache::lonuserutils::update_user_list($r,$context,$setting,$choice,$crstype);
                $r->print(&Apache::loncommon::end_page());
            } else {
                $r->print(&mt('You are not authorized to make bulk changes to user roles'));
                $r->print('<p><a href="/adm/createuser?action=listusers">'.&mt('Display User Lists').'</a>');
                $r->print(&Apache::loncommon::end_page());
            }
        } else {
            &Apache::lonhtmlcommon::add_breadcrumb
                ({href=>'/adm/createuser?action=listusers',
                  text=>"List Users"});
            my ($cb_jscript,$jscript,$totcodes,$codetitles,$idlist,$idlist_titles);
            my $formname = 'studentform';
            if (($context eq 'domain') && (($env{'form.roletype'} eq 'course') ||
                ($env{'form.roletype'} eq 'community'))) {
                if ($env{'form.roletype'} eq 'course') {
                    ($cb_jscript,$jscript,$totcodes,$codetitles,$idlist,$idlist_titles) =
                        &Apache::lonuserutils::courses_selector($env{'request.role.domain'},
                                                                $formname);
                } elsif ($env{'form.roletype'} eq 'community') {
                    $cb_jscript =
                        &Apache::loncommon::coursebrowser_javascript($env{'request.role.domain'});
                    my %elements = (
                                      coursepick => 'radio',
                                      coursetotal => 'text',
                                      courselist => 'text',
                                   );
                    $jscript = &Apache::lonhtmlcommon::set_form_elements(\%elements);
                }
                $jscript .= &verify_user_display();
                my $js = &add_script($jscript).$cb_jscript;
                my $loadcode = 
                    &Apache::lonuserutils::course_selector_loadcode($formname);
                if ($loadcode ne '') {
                    $r->print(&header($js,{'onload' => $loadcode,}));
                } else {
                    $r->print(&header($js));
                }
            } else {
                $r->print(&header(&add_script(&verify_user_display())));
            }
            $r->print(&Apache::lonhtmlcommon::breadcrumbs("List Users",
                                                          'Course_View_Class_List'));
            &Apache::lonuserutils::print_userlist($r,undef,$permission,$context,
                         $formname,$totcodes,$codetitles,$idlist,$idlist_titles);
            $r->print(&Apache::loncommon::end_page());
        }
    } elsif ($env{'form.action'} eq 'drop' && $permission->{'cusr'}) {
        $r->print(&header());
        my $brtext;
        if ($crstype eq 'Community') {
            $brtext = 'Drop Members';
        } else {
            $brtext = 'Drop Students';
        }
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/createuser?action=drop',
              text=>$brtext});
        if (!exists($env{'form.state'})) {
            $r->print(&Apache::lonhtmlcommon::breadcrumbs($brtext,
                                                          'Course_Drop_Student'));

            &Apache::lonuserutils::print_drop_menu($r,$context,$permission,$crstype);
        } elsif ($env{'form.state'} eq 'done') {
            &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/createuser?action=drop',
              text=>"Result"});
            $r->print(&Apache::lonhtmlcommon::breadcrumbs($brtext,
                                                          'Course_Drop_Student'));
            &Apache::lonuserutils::update_user_list($r,$context,undef,
                                                    $env{'form.action'});
        }
        $r->print(&Apache::loncommon::end_page());
    } elsif ($env{'form.action'} eq 'dateselect') {
        if ($permission->{'cusr'}) {
            $r->print(&header(undef,undef,{'no_nav_bar' => 1}).
                      &Apache::lonuserutils::date_section_selector($context,
                                                                   $permission,$crstype).
                      &Apache::loncommon::end_page());
        } else {
            $r->print(&header().
                     '<span class="LC_error">'.&mt('You do not have permission to modify dates or sections for users').'</span>'. 
                     &Apache::loncommon::end_page());
        }
    } elsif ($env{'form.action'} eq 'selfenroll') {
        $r->print(&header());
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/createuser?action=selfenroll',
              text=>"Configure Self-enrollment"});
        if (!exists($env{'form.state'})) {
            $r->print(&Apache::lonhtmlcommon::breadcrumbs('Configure Self-enrollment',
                                                          'Course_Self_Enrollment'));
            $r->print('<h3>'.&mt('Self-enrollment with a student role').'</h3>'."\n");
            &print_selfenroll_menu($r,$context,$permission);
        } elsif ($env{'form.state'} eq 'done') {
            &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/createuser?action=selfenroll',
              text=>"Result"});
            $r->print(&Apache::lonhtmlcommon::breadcrumbs('Self-enrollment result',
                                                          'Course_Self_Enrollment'));
            $r->print('<h3>'.&mt('Self-enrollment with a student role').'</h3>'."\n");
            &update_selfenroll_config($r,$context,$permission);
        }
        $r->print(&Apache::loncommon::end_page());
    } elsif ($env{'form.action'} eq 'selfenrollqueue') {
        $r->print(&header());
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/createuser?action=selfenrollqueue',
              text=>"Enrollment requests"});
        my $cid = $env{'request.course.id'};
        my $cdom = $env{'course.'.$cid.'.domain'};
        my $cnum = $env{'course.'.$cid.'.num'};
        my $coursedesc = $env{'course.'.$cid.'.description'};
        if (!exists($env{'form.state'})) {
            $r->print(&Apache::lonhtmlcommon::breadcrumbs('Enrollment requests',
                                                          'Course_SelfEnrollment_Approval'));
            $r->print('<h3>'.&mt('Pending enrollment requests').'</h3>'."\n");
            $r->print(&Apache::loncoursequeueadmin::display_queued_requests($context,
                                                                       $cdom,$cnum));

        } elsif ($env{'form.state'} eq 'done') {
            &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/createuser?action=selfenrollqueue',
              text=>"Result"});
            $r->print(&Apache::lonhtmlcommon::breadcrumbs('Enrollment result',
                                                          'Course_Self_Enrollment'));
            $r->print('<h3>'.&mt('Enrollment request processing').'</h3>'."\n");
            $r->print(&Apache::loncoursequeueadmin::update_request_queue($context,
                          $cdom,$cnum,$coursedesc));
        }
        $r->print(&Apache::loncommon::end_page());
    } elsif ($env{'form.action'} eq 'changelogs') {
        $r->print(&header());
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/createuser?action=changelogs',
              text=>"User Management Logs"});
        $r->print(&Apache::lonhtmlcommon::breadcrumbs('User Changes',
                                                      'Course_User_Logs'));
            &print_userchangelogs_display($r,$context,$permission);
        $r->print(&Apache::loncommon::end_page());        
    } else {
        $r->print(&header());
        $r->print(&Apache::lonhtmlcommon::breadcrumbs('User Management'));
        $r->print(&print_main_menu($permission,$context,$crstype));
        $r->print(&Apache::loncommon::end_page());
    }
    return OK;
}

sub header {
    my ($jscript,$loaditems,$args) = @_;
    my $start_page;
    if (ref($loaditems) eq 'HASH') {
        $start_page=&Apache::loncommon::start_page('User Management',$jscript,{'add_entries' => $loaditems});
    } else {
        $start_page=&Apache::loncommon::start_page('User Management',$jscript,$args);
    }
    return $start_page;
}

sub add_script {
    my ($js) = @_;
    return '<script type="text/javascript">'."\n".
           '// <![CDATA['."\n".
           "\n".$js."\n".
           '// ]]>'."\n".
           '</script>'."\n";
}

sub verify_user_display {
    my $output = <<"END";

function display_update() {
    document.studentform.action.value = 'listusers';
    document.studentform.phase.value = 'display';
    document.studentform.submit();
}

END
    return $output;

}

###############################################################
###############################################################
#  Menu Phase One
sub print_main_menu {
    my ($permission,$context,$crstype) = @_;
    my $linkcontext = $context;
    my $stuterm = lc(&Apache::lonnet::plaintext('st',$crstype));
    if (($context eq 'course') && ($crstype eq 'Community')) {
        $linkcontext = lc($crstype);
        $stuterm = 'Members';
    }

    my %links = (
                       domain => {
                                   upload => 'Upload a File of Users',
                                   singleuser => 'Add/Modify a User',
                                   listusers => 'Manage Users',
                                 },
                       author => {
                                   upload => 'Upload a File of Co-authors',
                                   singleuser => 'Add/Modify a Co-author',
                                   listusers => 'Manage Co-authors',
                                 },
                       course => {
                                   upload => 'Upload a File of Course Users',
                                   singleuser => 'Add/Modify a Course User',
                                   listusers => 'Display Class Lists and Manage Multiple Users',
                                 },
                       community => {
                                   upload     => 'Upload a File of Community Users',
                                   singleuser => 'Add/Modify a Community User',
                                   listusers  => 'Manage Community Users',
                                 },
                );
    my @menu =
        (
          { text => $links{$context}{'upload'},
            help => 'Course_Create_Class_List',
            action => 'upload',
            permission => $permission->{'cusr'},
            },
          { text => $links{$context}{'singleuser'},
            help => 'Course_Change_Privileges',
            action => 'singleuser',
            permission => $permission->{'cusr'},
            },
          { text => $links{$context}{'listusers'},
            help => 'Course_View_Class_List',
            action => 'listusers',
            permission => ($permission->{'view'} || $permission->{'cusr'}),
          },
        );
    if ($context eq 'domain' || $context eq 'course') {
        my $customlink =  { text => 'Edit Custom Roles',
                            help => 'Course_Editing_Custom_Roles',
                            action => 'custom',
                            permission => $permission->{'custom'},
                          };
        push(@menu,$customlink);
    }
    if ($context eq 'course') {
        my ($cnum,$cdom) = &Apache::lonuserutils::get_course_identity();
        my @courselinks =
            (
              { text => 'Enroll a Single Student',
                 help => 'Course_Add_Student',
                 action => 'singlestudent',
                 permission => $permission->{'cusr'},
                 },
              { text => 'Drop Students',
                help => 'Course_Drop_Student',
                action => 'drop',
                permission => $permission->{'cusr'},
              });
        if (!exists($permission->{'cusr_section'})) {
            if ($crstype ne 'Community') {
                push(@courselinks,
                   { text => 'Automated Enrollment Manager',
                     help => 'Course_Automated_Enrollment',
                     permission => (&Apache::lonnet::auto_run($cnum,$cdom)
                                   && $permission->{'cusr'}),
                     url  => '/adm/populate',
                   });
             }
             push(@courselinks,
                 { text  => 'Configure User Self-enrollment',
                   help  => 'Course_Self_Enrollment',
                   action => 'selfenroll',
                   permission => $permission->{'cusr'},
                 });
        }
        if ($env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_approval'}) {
            push(@courselinks,
                    {   text => 'Enrollment Requests',
                        help => 'Course_Approve_Selfenroll',
                        action => 'selfenrollqueue',
                        permission => $permission->{'cusr'},
                    });
        }
        push(@courselinks,
               { text => 'Manage Course Groups',
                 help => 'Course_Manage_Group',
                 permission => $permission->{'grp_manage'},
                 url => '/adm/coursegroups?refpage=cusr',
               },
               { text => 'View Change Logs',
                 help => 'Course_User_Logs',
                 action => 'changelogs',
                 permission => $permission->{'cusr'},
               },);
#               { text => 'View Log-in History',
#                 help => 'Course_User_Logins',
#                 action => 'logins',
#                 permission => $permission->{'cusr'},
#               });
        push(@menu,@courselinks);
    }
    my $menu_html = '';
    foreach my $menu_item (@menu) {
        next if (! $menu_item->{'permission'});
        $menu_html.='<p>';
        if (exists($menu_item->{'help'})) {
            $menu_html.=
                &Apache::loncommon::help_open_topic($menu_item->{'help'});
        }
        $menu_html.='<font size="+1">';
        if (exists($menu_item->{'url'})) {
            $menu_html.=qq{<a href="$menu_item->{'url'}">};
        } else {
            $menu_html.=
                qq{<a href="/adm/createuser?action=$menu_item->{'action'}">};      }
        $menu_html.= &mt($menu_item->{'text'}).'</a></font>';
        $menu_html.='</p>';
    }
    return $menu_html;
}

sub restore_prev_selections {
    my %saveable_parameters = ('srchby'   => 'scalar',
			       'srchin'   => 'scalar',
			       'srchtype' => 'scalar',
			       );
    &Apache::loncommon::store_settings('user','user_picker',
				       \%saveable_parameters);
    &Apache::loncommon::restore_settings('user','user_picker',
					 \%saveable_parameters);
}

sub print_selfenroll_menu {
    my ($r,$context,$permission) = @_;
    my $crstype = &Apache::loncommon::course_type();
    my $formname = 'enrollstudent';
    my $nolink = 1;
    my ($row,$lt) = &get_selfenroll_titles();
    my $groupslist = &Apache::lonuserutils::get_groupslist();
    my $setsec_js = 
        &Apache::lonuserutils::setsections_javascript($formname,$groupslist);
    my %alerts = &Apache::lonlocal::texthash(
        acto => 'Activation of self-enrollment was selected for the following domain(s)',
        butn => 'but no user types have been checked.',
        wilf => "Please uncheck 'activate' or check at least one type.",
    );
    my $selfenroll_js = <<"ENDSCRIPT";
function update_types(caller,num) {
    var delidx = getIndexByName('selfenroll_delete');
    var actidx = getIndexByName('selfenroll_activate');
    if (caller == 'selfenroll_all') {
        var selall;
        for (var i=0; i<document.$formname.selfenroll_all.length; i++) {
            if (document.$formname.selfenroll_all[i].checked) {
                selall = document.$formname.selfenroll_all[i].value;
            }
        }
        if (selall == 1) {
            if (delidx != -1) {
                if (document.$formname.selfenroll_delete.length) {
                    for (var j=0; j<document.$formname.selfenroll_delete.length; j++) {
                        document.$formname.selfenroll_delete[j].checked = true;
                    }
                } else {
                    document.$formname.elements[delidx].checked = true;
                }
            }
            if (actidx != -1) {
                if (document.$formname.selfenroll_activate.length) {
                    for (var j=0; j<document.$formname.selfenroll_activate.length; j++) {
                        document.$formname.selfenroll_activate[j].checked = false;
                    }
                } else {
                    document.$formname.elements[actidx].checked = false;
                }
            }
            document.$formname.selfenroll_newdom.selectedIndex = 0; 
        }
    }
    if (caller == 'selfenroll_activate') {
        if (document.$formname.selfenroll_activate.length) {
            for (var j=0; j<document.$formname.selfenroll_activate.length; j++) {
                if (document.$formname.selfenroll_activate[j].value == num) {
                    if (document.$formname.selfenroll_activate[j].checked) {
                        for (var i=0; i<document.$formname.selfenroll_all.length; i++) {
                            if (document.$formname.selfenroll_all[i].value == '1') {
                                document.$formname.selfenroll_all[i].checked = false;
                            }
                            if (document.$formname.selfenroll_all[i].value == '0') {
                                document.$formname.selfenroll_all[i].checked = true;
                            }
                        }
                    }
                }
            }
        } else {
            for (var i=0; i<document.$formname.selfenroll_all.length; i++) {
                if (document.$formname.selfenroll_all[i].value == '1') {
                    document.$formname.selfenroll_all[i].checked = false;
                }
                if (document.$formname.selfenroll_all[i].value == '0') {
                    document.$formname.selfenroll_all[i].checked = true;
                }
            }
        }
    }
    if (caller == 'selfenroll_delete') {
        if (document.$formname.selfenroll_delete.length) {
            for (var j=0; j<document.$formname.selfenroll_delete.length; j++) {
                if (document.$formname.selfenroll_delete[j].value == num) {
                    if (document.$formname.selfenroll_delete[j].checked) {
                        var delindex = getIndexByName('selfenroll_types_'+num);
                        if (delindex != -1) { 
                            if (document.$formname.elements[delindex].length) {
                                for (var k=0; k<document.$formname.elements[delindex].length; k++) {
                                    document.$formname.elements[delindex][k].checked = false;
                                }
                            } else {
                                document.$formname.elements[delindex].checked = false;
                            }
                        }
                    }
                }
            }
        } else {
            if (document.$formname.selfenroll_delete.checked) {
                var delindex = getIndexByName('selfenroll_types_'+num);
                if (delindex != -1) {
                    if (document.$formname.elements[delindex].length) {
                        for (var k=0; k<document.$formname.elements[delindex].length; k++) {
                            document.$formname.elements[delindex][k].checked = false;
                        }
                    } else {
                        document.$formname.elements[delindex].checked = false;
                    }
                }
            }
        }
    }
    return;
}

function validate_types(form) {
    var needaction = new Array();
    var countfail = 0;
    var actidx = getIndexByName('selfenroll_activate');
    if (actidx != -1) {
        if (document.$formname.selfenroll_activate.length) {
            for (var j=0; j<document.$formname.selfenroll_activate.length; j++) {
                var num = document.$formname.selfenroll_activate[j].value;
                if (document.$formname.selfenroll_activate[j].checked) {
                    countfail = check_types(num,countfail,needaction)
                }
            }
        } else {
            if (document.$formname.selfenroll_activate.checked) {
                var num = document.enrollstudent.selfenroll_activate.value;
                countfail = check_types(num,countfail,needaction)
            }
        }
    }
    if (countfail > 0) {
        var msg = "$alerts{'acto'}\\n";
        var loopend = needaction.length -1;
        if (loopend > 0) {
            for (var m=0; m<loopend; m++) {
                msg += needaction[m]+", ";
            }
        }
        msg += needaction[loopend]+"\\n$alerts{'butn'}\\n$alerts{'wilf'}";
        alert(msg);
        return; 
    }
    setSections(form);
}

function check_types(num,countfail,needaction) {
    var typeidx = getIndexByName('selfenroll_types_'+num);
    var count = 0;
    if (typeidx != -1) {
        if (document.$formname.elements[typeidx].length) {
            for (var k=0; k<document.$formname.elements[typeidx].length; k++) {
                if (document.$formname.elements[typeidx][k].checked) {
                    count ++;
                }
            }
        } else {
            if (document.$formname.elements[typeidx].checked) {
                count ++;
            }
        }
        if (count == 0) {
            var domidx = getIndexByName('selfenroll_dom_'+num);
            if (domidx != -1) {
                var domname = document.$formname.elements[domidx].value;
                needaction[countfail] = domname;
                countfail ++;
            }
        }
    }
    return countfail;
}

function getIndexByName(item) {
    for (var i=0;i<document.$formname.elements.length;i++) {
        if (document.$formname.elements[i].name == item) {
            return i;
        }
    }
    return -1;
}
ENDSCRIPT
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};

    my $output = '<script type="text/javascript">'."\n".
                 '// <![CDATA['."\n".
                 $setsec_js."\n".$selfenroll_js."\n".
                 '// ]]>'."\n".
                 '</script>'."\n".
                 '<h3>'.$lt->{'selfenroll'}.'</h3>'."\n";
    my ($visible,$cansetvis,$vismsgs,$visactions) = &visible_in_cat($cdom,$cnum);
    if (ref($visactions) eq 'HASH') {
        if ($visible) {
            $output .= '<p class="LC_info">'.$visactions->{'vis'}.'</p>';
        } else {
            $output .= '<p class="LC_warning">'.$visactions->{'miss'}.'</p>'
                      .$visactions->{'yous'}.
                       '<p>'.$visactions->{'gen'}.'<br />'.$visactions->{'coca'};
            if (ref($vismsgs) eq 'ARRAY') {
                $output .= '<br />'.$visactions->{'make'}.'<ul>';
                foreach my $item (@{$vismsgs}) {
                    $output .= '<li>'.$visactions->{$item}.'</li>';
                }
                $output .= '</ul>';
            }
            $output .= '</p>';
        }
    }
    $output .= '<form name="'.$formname.'" method="post" action="/adm/createuser">'."\n".
               &Apache::lonhtmlcommon::start_pick_box();
    if (ref($row) eq 'ARRAY') {
        foreach my $item (@{$row}) {
            my $title = $item; 
            if (ref($lt) eq 'HASH') {
                $title = $lt->{$item};
            }
            $output .= 
                &Apache::lonhtmlcommon::row_title($title,
                             'LC_selfenroll_pick_box_title','LC_oddrow_value')."\n";
            if ($item eq 'types') {
                my $curr_types = $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_types'};
                my $showdomdesc = 1;
                my $includeempty = 1;
                my $num = 0;
                $output .= &Apache::loncommon::start_data_table().
                           &Apache::loncommon::start_data_table_row()
                           .'<td colspan="2"><span class="LC_nobreak"><label>'
                           .&mt('Any user in any domain:')
                           .'&nbsp;<input type="radio" name="selfenroll_all" value="1" ';
                if ($curr_types eq '*') {
                    $output .= ' checked="checked" '; 
                }
                $output .= 'onchange="javascript:update_types('.
                           "'selfenroll_all'".');" />'.&mt('Yes').'</label>'.
                           '&nbsp;&nbsp;<input type="radio" name="selfenroll_all" value="0" ';
                if ($curr_types ne '*') {
                    $output .= ' checked="checked" ';
                }
                $output .= ' onchange="javascript:update_types('.
                           "'selfenroll_all'".');"/>'.&mt('No').'</label></td>'.
                           &Apache::loncommon::end_data_table_row().
                           &Apache::loncommon::end_data_table().
                           &mt('Or').'<br />'.
                           &Apache::loncommon::start_data_table();
                my %currdoms;
                if ($curr_types eq '') {
                    $output .= &new_selfenroll_dom_row($cdom,'0');
                } elsif ($curr_types ne '*') {
                    my @entries = split(/;/,$curr_types);
                    if (@entries > 0) {
                        foreach my $entry (@entries) {
                            my ($currdom,$typestr) = split(/:/,$entry);
                            $currdoms{$currdom} = 1;
                            my $domdesc = &Apache::lonnet::domain($currdom);
                            my @currinsttypes = split(',',$typestr);
                            $output .= &Apache::loncommon::start_data_table_row()
                                       .'<td valign="top"><span class="LC_nobreak">'.&mt('Domain:').'<b>'
                                       .'&nbsp;'.$domdesc.' ('.$currdom.')'
                                       .'</b><input type="hidden" name="selfenroll_dom_'.$num
                                       .'" value="'.$currdom.'" /></span><br />'
                                       .'<span class="LC_nobreak"><label><input type="checkbox" '
                                       .'name="selfenroll_delete" value="'.$num.'" onchange="javascript:update_types('."'selfenroll_delete','$num'".');" />'
                                       .&mt('Delete').'</label></span></td>';
                            $output .= '<td valign="top">&nbsp;&nbsp;'.&mt('User types:').'<br />'
                                       .&selfenroll_inst_types($num,$currdom,\@currinsttypes).'</td>'
                                       .&Apache::loncommon::end_data_table_row();
                            $num ++;
                        }
                    }
                }
                my $add_domtitle = &mt('Users in additional domain:');
                if ($curr_types eq '*') { 
                    $add_domtitle = &mt('Users in specific domain:');
                } elsif ($curr_types eq '') {
                    $add_domtitle = &mt('Users in other domain:');
                }
                $output .= &Apache::loncommon::start_data_table_row()
                           .'<td colspan="2"><span class="LC_nobreak">'.$add_domtitle.'</span><br />'
                           .&Apache::loncommon::select_dom_form('','selfenroll_newdom',
                                                                $includeempty,$showdomdesc)
                           .'<input type="hidden" name="selfenroll_types_total" value="'.$num.'" />'
                           .'</td>'.&Apache::loncommon::end_data_table_row()
                           .&Apache::loncommon::end_data_table();
            } elsif ($item eq 'registered') {
                my ($regon,$regoff);
                if ($env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_registered'}) {
                    $regon = ' checked="checked" ';
                    $regoff = ' ';
                } else {
                    $regon = ' ';
                    $regoff = ' checked="checked" ';
                }
                $output .= '<label>'.
                           '<input type="radio" name="selfenroll_registered" value="1"'.$regon.'/>'.
                           &mt('Yes').'</label>&nbsp;&nbsp;<label>'.
                           '<input type="radio" name="selfenroll_registered" value="0"'.$regoff.'/>'.
                           &mt('No').'</label>';
            } elsif ($item eq 'enroll_dates') {
                my $starttime = $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_start_date'};
                my $endtime = $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_end_date'};
                if ($starttime eq '') {
                    $starttime = $env{'course.'.$env{'request.course.id'}.'.default_enrollment_start_date'};
                }
                if ($endtime eq '') {
                    $endtime = $env{'course.'.$env{'request.course.id'}.'.default_enrollment_end_date'};
                }
                my $startform =
                    &Apache::lonhtmlcommon::date_setter($formname,'selfenroll_start_date',$starttime,
                                      undef,undef,undef,undef,undef,undef,undef,$nolink);
                my $endform =
                    &Apache::lonhtmlcommon::date_setter($formname,'selfenroll_end_date',$endtime,
                                      undef,undef,undef,undef,undef,undef,undef,$nolink);
                $output .= &selfenroll_date_forms($startform,$endform);
            } elsif ($item eq 'access_dates') {
                my $starttime = $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_start_access'};
                my $endtime = $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_end_access'};
                if ($starttime eq '') {
                    $starttime = $env{'course.'.$env{'request.course.id'}.'.default_enrollment_start_date'};
                }
                if ($endtime eq '') {
                    $endtime = $env{'course.'.$env{'request.course.id'}.'.default_enrollment_end_date'};
                }
                my $startform =
                    &Apache::lonhtmlcommon::date_setter($formname,'selfenroll_start_access',$starttime,
                                      undef,undef,undef,undef,undef,undef,undef,$nolink);
                my $endform =
                    &Apache::lonhtmlcommon::date_setter($formname,'selfenroll_end_access',$endtime,
                                      undef,undef,undef,undef,undef,undef,undef,$nolink);
                $output .= &selfenroll_date_forms($startform,$endform);
            } elsif ($item eq 'section') {
                my $currsec = $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_section'}; 
                my %sections_count = &Apache::loncommon::get_sections($cdom,$cnum);
                my $newsecval;
                if ($currsec ne 'none' && $currsec ne '') {
                    if (!defined($sections_count{$currsec})) {
                        $newsecval = $currsec;
                    }
                }
                my $sections_select = 
                    &Apache::lonuserutils::course_sections(\%sections_count,'st',$currsec);
                $output .= '<table class="LC_createuser">'."\n".
                           '<tr class="LC_section_row">'."\n".
                           '<td align="center">'.&mt('Existing sections')."\n".
                           '<br />'.$sections_select.'</td><td align="center">'.
                           &mt('New section').'<br />'."\n".
                           '<input type="text" name="newsec" size="15" value="'.$newsecval.'" />'."\n".
                           '<input type="hidden" name="sections" value="" />'."\n".
                           '<input type="hidden" name="state" value="done" />'."\n".
                           '</td></tr></table>'."\n";
            } elsif ($item eq 'approval') {
                my ($appon,$appoff);
                my $cid = $env{'request.course.id'};
                my $currnotified = $env{'course.'.$cid.'.internal.selfenroll_notifylist'};
                if ($env{'course.'.$cid.'.internal.selfenroll_approval'}) {
                    $appon = ' checked="checked" ';
                    $appoff = ' ';
                } else {
                    $appon = ' ';
                    $appoff = ' checked="checked" ';
                }
                $output .= '<label>'.
                           '<input type="radio" name="selfenroll_approval" value="1"'.$appon.'/>'.
                           &mt('Yes').'</label>&nbsp;&nbsp;<label>'.
                           '<input type="radio" name="selfenroll_approval" value="0"'.$appoff.'/>'.
                           &mt('No').'</label>';
                my %advhash = &Apache::lonnet::get_course_adv_roles($cid,1);
                my (@ccs,%notified);
                my $ccrole = 'cc';
                if ($crstype eq 'Community') {
                    $ccrole = 'co';
                }
                if ($advhash{$ccrole}) {
                    @ccs = split(/,/,$advhash{$ccrole});
                }
                if ($currnotified) {
                    foreach my $current (split(/,/,$currnotified)) {
                        $notified{$current} = 1;
                        if (!grep(/^\Q$current\E$/,@ccs)) {
                            push(@ccs,$current);
                        }
                    }
                }
                if (@ccs) {
                    $output .= '<br />'.&mt('Personnel to be notified when an enrollment request needs approval, or has been approved:').'&nbsp;'.&Apache::loncommon::start_data_table().
                               &Apache::loncommon::start_data_table_row();
                    my $count = 0;
                    my $numcols = 4;
                    foreach my $cc (sort(@ccs)) {
                        my $notifyon;
                        my ($ccuname,$ccudom) = split(/:/,$cc);
                        if ($notified{$cc}) {
                            $notifyon = ' checked="checked" ';
                        }
                        if ($count && !$count%$numcols) {
                            $output .= &Apache::loncommon::end_data_table_row().
                                       &Apache::loncommon::start_data_table_row()
                        }
                        $output .= '<td><span class="LC_nobreak"><label>'.
                                   '<input type="checkbox" name="selfenroll_notify"'.$notifyon.' value="'.$cc.'" />'.
                                   &Apache::loncommon::plainname($ccuname,$ccudom).
                                   '</label></span></td>';
                        $count ++;
                    }
                    my $rem = $count%$numcols;
                    if ($rem) {
                        my $emptycols = $numcols - $rem;
                        for (my $i=0; $i<$emptycols; $i++) { 
                            $output .= '<td>&nbsp;</td>';
                        }
                    }
                    $output .= &Apache::loncommon::end_data_table_row().
                               &Apache::loncommon::end_data_table();
                }
            } elsif ($item eq 'limit') {
                my ($crslimit,$selflimit,$nolimit);
                my $cid = $env{'request.course.id'};
                my $currlim = $env{'course.'.$cid.'.internal.selfenroll_limit'};
                my $currcap = $env{'course.'.$cid.'.internal.selfenroll_cap'};
                $nolimit = ' checked="checked" ';
                if ($currlim eq 'allstudents') {
                    $crslimit = ' checked="checked" ';
                    $selflimit = ' ';
                    $nolimit = ' ';
                } elsif ($currlim eq 'selfenrolled') {
                    $crslimit = ' ';
                    $selflimit = ' checked="checked" ';
                    $nolimit = ' '; 
                } else {
                    $crslimit = ' ';
                    $selflimit = ' ';
                }
                $output .= '<table><tr><td><label>'.
                           '<input type="radio" name="selfenroll_limit" value="none"'.$nolimit.'/>'.
                           &mt('No limit').'</label></td><td><label>'.
                           '<input type="radio" name="selfenroll_limit" value="allstudents"'.$crslimit.'/>'.
                           &mt('Limit by total students').'</label></td><td><label>'.
                           '<input type="radio" name="selfenroll_limit" value="selfenrolled"'.$selflimit.'/>'.
                           &mt('Limit by total self-enrolled students').
                           '</td></tr><tr>'.
                           '<td>&nbsp;</td><td colspan="2"><span class="LC_nobreak">'.
                           ('&nbsp;'x3).&mt('Maximum number allowed: ').
                           '<input type="text" name="selfenroll_cap" size = "5" value="'.$currcap.'" /></td></tr></table>';
            }
            $output .= &Apache::lonhtmlcommon::row_closure(1);
        }
    }
    $output .= &Apache::lonhtmlcommon::end_pick_box().
               '<br /><input type="button" name="selfenrollconf" value="'
               .&mt('Save').'" onclick="validate_types(this.form);" />'
               .'<input type="hidden" name="action" value="selfenroll" /></form>';
    $r->print($output);
    return;
}

sub visible_in_cat {
    my ($cdom,$cnum) = @_;
    my %domconf = &Apache::lonnet::get_dom('configuration',['coursecategories'],$cdom);
    my ($cathash,%settable,@vismsgs,$cansetvis);
    my %visactions = &Apache::lonlocal::texthash(
                   vis => 'Your course/community currently appears in the Course/Community Catalog for this domain.',
                   gen => 'Courses can be both self-cataloging, based on an institutional code (e.g., fs08phy231), or can be assigned categories from a hierarchy defined for the domain.',
                   miss => 'Your course/community does not currently appear in the Course/Community Catalog for this domain.',
                   yous => 'You should remedy this if you plan to allow self-enrollment, otherwise students will have difficulty finding your course.',
                   coca => 'Courses can be absent from the Catalog, because they do not have an institutional code, have no assigned category, or have been specifically excluded.',
                   make => 'Make any changes to self-enrollment settings below, click "Save", then take action to include the course in the Catalog:',
                   take => 'Take the following action to ensure the course appears in the Catalog:',
                   dc_unhide  => 'Ask a domain coordinator to change the "Exclude from course catalog" setting.',
                   dc_addinst => 'Ask a domain coordinator to enable display the catalog of "Official courses (with institutional codes)".',
                   dc_instcode => 'Ask a domain coordinator to assign an institutional code (if this is an official course).',
                   dc_catalog  => 'Ask a domain coordinator to enable or create at least one course category in the domain.',
                   dc_categories => 'Ask a domain coordinator to create a hierarchy of categories and sub categories for courses in the domain.',
                   dc_chgcat => 'Ask a domain coordinator to change the category assigned to the course, as the one currently assigned is no longer used in the domain',
                   dc_addcat => 'Ask a domain coordinator to assign a category to the course.',
    );
    $visactions{'unhide'} = &mt('Use [_1]Set course environment[_2] to change the "Exclude from course catalog" setting.','"<a href="/adm/parmset?action=crsenv">','</a>"');
    $visactions{'chgcat'} = &mt('Use [_1]Set course environment[_2] to change the category assigned to the course, as the one currently assigned is no longer used in the domain.','"<a href="/adm/parmset?action=crsenv">','</a>"');
    $visactions{'addcat'} = &mt('Use [_1]Set course environment[_2] to assign a category to the course.','"<a href="/adm/parmset?action=crsenv">','</a>"');
    if (ref($domconf{'coursecategories'}) eq 'HASH') {
        if ($domconf{'coursecategories'}{'togglecats'} eq 'crs') {
            $settable{'togglecats'} = 1;
        }
        if ($domconf{'coursecategories'}{'categorize'} eq 'crs') {
            $settable{'categorize'} = 1;
        }
        $cathash = $domconf{'coursecategories'}{'cats'};
    }
    if ($settable{'togglecats'} && $settable{'categorize'}) {
        $cansetvis = &mt('You are able to both assign a course category and choose to exclude this course from the catalog.');   
    } elsif ($settable{'togglecats'}) {
        $cansetvis = &mt('You are able to choose to exclude this course from the catalog, but only a Domain Coordinator may assign a course category.'); 
    } elsif ($settable{'categorize'}) {
        $cansetvis = &mt('You may assign a course category, but only a Domain Coordinator may choose to exclude this course from the catalog.');  
    } else {
        $cansetvis = &mt('Only a Domain Coordinator may assign a course category or choose to exclude this course from the catalog.'); 
    }
     
    my %currsettings =
        &Apache::lonnet::get('environment',['hidefromcat','categories','internal.coursecode'],
                             $cdom,$cnum);
    my $visible = 0;
    if ($currsettings{'internal.coursecode'} ne '') {
        if (ref($domconf{'coursecategories'}) eq 'HASH') {
            $cathash = $domconf{'coursecategories'}{'cats'};
            if (ref($cathash) eq 'HASH') {
                if ($cathash->{'instcode::0'} eq '') {
                    push(@vismsgs,'dc_addinst'); 
                } else {
                    $visible = 1;
                }
            } else {
                $visible = 1;
            }
        } else {
            $visible = 1;
        }
    } else {
        if (ref($cathash) eq 'HASH') {
            if ($cathash->{'instcode::0'} ne '') {
                push(@vismsgs,'dc_instcode');
            }
        } else {
            push(@vismsgs,'dc_instcode');
        }
    }
    if ($currsettings{'categories'} ne '') {
        my $cathash;
        if (ref($domconf{'coursecategories'}) eq 'HASH') {
            $cathash = $domconf{'coursecategories'}{'cats'};
            if (ref($cathash) eq 'HASH') {
                if (keys(%{$cathash}) == 0) {
                    push(@vismsgs,'dc_catalog');
                } elsif ((keys(%{$cathash}) == 1) && ($cathash->{'instcode::0'} ne '')) {
                    push(@vismsgs,'dc_categories');
                } else {
                    my @currcategories = split('&',$currsettings{'categories'});
                    my $matched = 0;
                    foreach my $cat (@currcategories) {
                        if ($cathash->{$cat} ne '') {
                            $visible = 1;
                            $matched = 1;
                            last;
                        }
                    }
                    if (!$matched) {
                        if ($settable{'categorize'}) { 
                            push(@vismsgs,'chgcat');
                        } else {
                            push(@vismsgs,'dc_chgcat');
                        }
                    }
                }
            }
        }
    } else {
        if (ref($cathash) eq 'HASH') {
            if ((keys(%{$cathash}) > 1) || 
                (keys(%{$cathash}) == 1) && ($cathash->{'instcode::0'} eq '')) {
                if ($settable{'categorize'}) {
                    push(@vismsgs,'addcat');
                } else {
                    push(@vismsgs,'dc_addcat');
                }
            }
        }
    }
    if ($currsettings{'hidefromcat'} eq 'yes') {
        $visible = 0;
        if ($settable{'togglecats'}) {
            unshift(@vismsgs,'unhide');
        } else {
            unshift(@vismsgs,'dc_unhide')
        }
    }
    return ($visible,$cansetvis,\@vismsgs,\%visactions);
}

sub new_selfenroll_dom_row {
    my ($newdom,$num) = @_;
    my $domdesc = &Apache::lonnet::domain($newdom);
    my $output;
    if ($domdesc ne '') {
        $output .= &Apache::loncommon::start_data_table_row()
                   .'<td valign="top"><span class="LC_nobreak">'.&mt('Domain:').'&nbsp;<b>'.$domdesc
                   .' ('.$newdom.')</b><input type="hidden" name="selfenroll_dom_'.$num
                   .'" value="'.$newdom.'" /></span><br />'
                   .'<span class="LC_nobreak"><label><input type="checkbox" '
                   .'name="selfenroll_activate" value="'.$num.'" '
                   .'onchange="javascript:update_types('
                   ."'selfenroll_activate','$num'".');" />'
                   .&mt('Activate').'</label></span></td>';
        my @currinsttypes;
        $output .= '<td>'.&mt('User types:').'<br />'
                   .&selfenroll_inst_types($num,$newdom,\@currinsttypes).'</td>'
                   .&Apache::loncommon::end_data_table_row();
    }
    return $output;
}

sub selfenroll_inst_types {
    my ($num,$currdom,$currinsttypes) = @_;
    my $output;
    my $numinrow = 4;
    my $count = 0;
    my ($othertitle,$usertypes,$types) = &Apache::loncommon::sorted_inst_types($currdom);
    my $othervalue = 'any';
    if ((ref($types) eq 'ARRAY') && (ref($usertypes) eq 'HASH')) {
        if (keys(%{$usertypes}) > 0) {
            $othervalue = 'other';
        }
        $output .= '<table><tr>';
        foreach my $type (@{$types}) {
            if (($count > 0) && ($count%$numinrow == 0)) {
                $output .= '</tr><tr>';
            }
            if (defined($usertypes->{$type})) {
                my $esc_type = &escape($type);
                $output .= '<td><span class="LC_nobreak"><label><input type = "checkbox" value="'.
                           $esc_type.'" ';
                if (ref($currinsttypes) eq 'ARRAY') {
                    if (@{$currinsttypes} > 0) {
                        if (grep(/^any$/,@{$currinsttypes})) {
                            $output .= 'checked="checked"';
                        } elsif (grep(/^\Q$esc_type\E$/,@{$currinsttypes})) {
                            $output .= 'checked="checked"';
                        }
                    } else {
                        $output .= 'checked="checked"';
                    }
                }
                $output .= ' name="selfenroll_types_'.$num.'" />'.$usertypes->{$type}.'</label></span></td>';
            }
            $count ++;
        }
        if (($count > 0) && ($count%$numinrow == 0)) {
            $output .= '</tr><tr>';
        }
        $output .= '<td><span class="LC_nobreak"><label><input type = "checkbox" value="'.$othervalue.'"';
        if (ref($currinsttypes) eq 'ARRAY') {
            if (@{$currinsttypes} > 0) {
                if (grep(/^any$/,@{$currinsttypes})) { 
                    $output .= ' checked="checked"';
                } elsif ($othervalue eq 'other') {
                    if (grep(/^\Q$othervalue\E$/,@{$currinsttypes})) {
                        $output .= ' checked="checked"';
                    }
                }
            } else {
                $output .= ' checked="checked"';
            }
        } else {
            $output .= ' checked="checked"';
        }
        $output .= ' name="selfenroll_types_'.$num.'" />'.$othertitle.'</label></span></td></tr></table>';
    }
    return $output;
}

sub selfenroll_date_forms {
    my ($startform,$endform) = @_;
    my $output .= &Apache::lonhtmlcommon::start_pick_box()."\n".
                  &Apache::lonhtmlcommon::row_title(&mt('Start date'),
                                                    'LC_oddrow_value')."\n".
                  $startform."\n".
                  &Apache::lonhtmlcommon::row_closure(1).
                  &Apache::lonhtmlcommon::row_title(&mt('End date'),
                                                   'LC_oddrow_value')."\n".
                  $endform."\n".
                  &Apache::lonhtmlcommon::row_closure(1).
                  &Apache::lonhtmlcommon::end_pick_box();
    return $output;
}

sub print_userchangelogs_display {
    my ($r,$context,$permission) = @_;
    my $formname = 'roleslog';
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    my $crstype = &Apache::loncommon::course_type();
    my %roleslog=&Apache::lonnet::dump('nohist_rolelog',$cdom,$cnum);
    if ((keys(%roleslog))[0]=~/^error\:/) { undef(%roleslog); }

    $r->print('<form action="/adm/createuser" method="post" name="'.$formname.'">');
    my %saveable_parameters = ('show' => 'scalar',);
    &Apache::loncommon::store_course_settings('roles_log',
                                              \%saveable_parameters);
    &Apache::loncommon::restore_course_settings('roles_log',
                                                \%saveable_parameters);
    # set defaults
    my $now = time();
    my $defstart = $now - (7*24*3600); #7 days ago 
    my %defaults = (
                     page               => '1',
                     show               => '10',
                     role               => 'any',
                     chgcontext         => 'any',
                     rolelog_start_date => $defstart,
                     rolelog_end_date   => $now,
                   );
    my $more_records = 0;

    # set current
    my %curr;
    foreach my $item ('show','page','role','chgcontext') {
        $curr{$item} = $env{'form.'.$item};
    }
    my ($startdate,$enddate) = 
        &Apache::lonuserutils::get_dates_from_form('rolelog_start_date','rolelog_end_date');
    $curr{'rolelog_start_date'} = $startdate;
    $curr{'rolelog_end_date'} = $enddate;
    foreach my $key (keys(%defaults)) {
        if ($curr{$key} eq '') {
            $curr{$key} = $defaults{$key};
        }
    }
    my (%whodunit,%changed,$version);
    ($version) = ($r->dir_config('lonVersion') =~ /^([\d\.]+)\-/);
    $r->print(&role_display_filter($formname,$cdom,$cnum,\%curr,$version,$crstype));
    my $showntablehdr = 0;
    my $tablehdr = &Apache::loncommon::start_data_table().
                   &Apache::loncommon::start_data_table_header_row().
                   '<th>&nbsp;</th><th>'.&mt('When').'</th><th>'.&mt('Who made the change').
                   '</th><th>'.&mt('Changed User').'</th><th>'.&mt('Role').'</th><th>'.&mt('Section').'</th><th>'.
                   &mt('Context').'</th><th>'.&mt('Start').'</th><th>'.&mt('End').'</th>'.
                   &Apache::loncommon::end_data_table_header_row();
    my ($minshown,$maxshown);
    $minshown = 1;
    my $count = 0;
    if ($curr{'show'} ne &mt('all')) { 
        $maxshown = $curr{'page'} * $curr{'show'};
        if ($curr{'page'} > 1) {
            $minshown = 1 + ($curr{'page'} - 1) * $curr{'show'};
        }
    }
    foreach my $id (sort { $roleslog{$b}{'exe_time'}<=>$roleslog{$a}{'exe_time'} } (keys(%roleslog))) {
        next if (($roleslog{$id}{'exe_time'} < $curr{'rolelog_start_date'}) ||
                 ($roleslog{$id}{'exe_time'} > $curr{'rolelog_end_date'}));
        if ($curr{'show'} ne &mt('all')) {
            if ($count >= $curr{'page'} * $curr{'show'}) {
                $more_records = 1;
                last;
            }
        }
        if ($curr{'role'} ne 'any') {
            next if ($roleslog{$id}{'logentry'}{'role'} ne $curr{'role'}); 
        }
        if ($curr{'chgcontext'} ne 'any') {
            if ($curr{'chgcontext'} eq 'selfenroll') {
                next if (!$roleslog{$id}{'logentry'}{'selfenroll'});
            } else {
                next if ($roleslog{$id}{'logentry'}{'context'} ne $curr{'chgcontext'});
            }
        }
        $count ++;
        next if ($count < $minshown);
        if (!$showntablehdr) {
            $r->print($tablehdr);
            $showntablehdr = 1;
        }
        if ($whodunit{$roleslog{$id}{'exe_uname'}.':'.$roleslog{$id}{'exe_udom'}} eq '') {
            $whodunit{$roleslog{$id}{'exe_uname'}.':'.$roleslog{$id}{'exe_udom'}} =
                &Apache::loncommon::plainname($roleslog{$id}{'exe_uname'},$roleslog{$id}{'exe_udom'});
        }
        if ($changed{$roleslog{$id}{'uname'}.':'.$roleslog{$id}{'udom'}} eq '') {
            $changed{$roleslog{$id}{'uname'}.':'.$roleslog{$id}{'udom'}} =
                &Apache::loncommon::plainname($roleslog{$id}{'uname'},$roleslog{$id}{'udom'});
        }
        my $sec = $roleslog{$id}{'logentry'}{'section'};
        if ($sec eq '') {
            $sec = &mt('None');
        }
        my ($rolestart,$roleend);
        if ($roleslog{$id}{'delflag'}) {
            $rolestart = &mt('deleted');
            $roleend = &mt('deleted');
        } else {
            $rolestart = $roleslog{$id}{'logentry'}{'start'};
            $roleend = $roleslog{$id}{'logentry'}{'end'};
            if ($rolestart eq '' || $rolestart == 0) {
                $rolestart = &mt('No start date'); 
            } else {
                $rolestart = &Apache::lonlocal::locallocaltime($rolestart);
            }
            if ($roleend eq '' || $roleend == 0) { 
                $roleend = &mt('No end date');
            } else {
                $roleend = &Apache::lonlocal::locallocaltime($roleend);
            }
        }
        my $chgcontext = $roleslog{$id}{'logentry'}{'context'};
        if ($roleslog{$id}{'logentry'}{'selfenroll'}) {
            $chgcontext = 'selfenroll';
        }
        my %lt = &rolechg_contexts($crstype);
        if ($chgcontext ne '' && $lt{$chgcontext} ne '') {
            $chgcontext = $lt{$chgcontext};
        }
        $r->print(&Apache::loncommon::start_data_table_row().'<td>'.$count.'</td><td>'.&Apache::lonlocal::locallocaltime($roleslog{$id}{'exe_time'}).'</td><td>'.$whodunit{$roleslog{$id}{'exe_uname'}.':'.$roleslog{$id}{'exe_udom'}}.'</td><td>'.$changed{$roleslog{$id}{'uname'}.':'.$roleslog{$id}{'udom'}}.'</td><td>'.&Apache::lonnet::plaintext($roleslog{$id}{'logentry'}{'role'},$crstype).'</td><td>'.$sec.'</td><td>'.$chgcontext.'</td><td>'.$rolestart.'</td><td>'.$roleend.'</td>'.&Apache::loncommon::end_data_table_row()."\n");
    }
    if ($showntablehdr) {
        $r->print(&Apache::loncommon::end_data_table().'<br />');
        if (($curr{'page'} > 1) || ($more_records)) {
            $r->print('<table><tr>');
            if ($curr{'page'} > 1) {
                $r->print('<td><a href="javascript:chgPage('."'previous'".');">'.&mt('Previous [_1] changes',$curr{'show'}).'</a></td>');
            }
            if ($more_records) {
                $r->print('<td><a href="javascript:chgPage('."'next'".');">'.&mt('Next [_1] changes',$curr{'show'}).'</a></td>');
            }
            $r->print('</tr></table>');
            $r->print(<<"ENDSCRIPT");
<script type="text/javascript">
// <![CDATA[
function chgPage(caller) {
    if (caller == 'previous') {
        document.$formname.page.value --;
    }
    if (caller == 'next') {
        document.$formname.page.value ++;
    }
    document.$formname.submit(); 
    return;
}
// ]]>
</script>
ENDSCRIPT
        }
    } else {
        $r->print('<p class="LC_info">'.
                  &mt('There are no records to display').
                  '</p>');
    }
    $r->print('<input type="hidden" name="page" value="'.$curr{'page'}.'" />'.
              '<input type="hidden" name="action" value="changelogs" /></form>');
    return;
}

sub role_display_filter {
    my ($formname,$cdom,$cnum,$curr,$version,$crstype) = @_;
    my $context = 'course';
    my $lctype = lc($crstype);
    my $nolink = 1;
    my $output = '<table><tr><td valign="top">'.
                 '<span class="LC_nobreak"><b>'.&mt('Changes/page:').'</b></span><br />'.
                 &Apache::lonmeta::selectbox('show',$curr->{'show'},undef,
                                              (&mt('all'),5,10,20,50,100,1000,10000)).
                 '</td><td>&nbsp;&nbsp;</td>';
    my $startform =
        &Apache::lonhtmlcommon::date_setter($formname,'rolelog_start_date',
                                            $curr->{'rolelog_start_date'},undef,
                                            undef,undef,undef,undef,undef,undef,$nolink);
    my $endform =
        &Apache::lonhtmlcommon::date_setter($formname,'rolelog_end_date',
                                            $curr->{'rolelog_end_date'},undef,
                                            undef,undef,undef,undef,undef,undef,$nolink);
    my %lt = &rolechg_contexts($crstype);
    $output .= '<td valign="top"><b>'.&mt('Window during which changes occurred:').'</b><br /><table><tr><td>'.&mt('After:').
               '</td><td>'.$startform.'</td></tr><tr><td>'.&mt('Before:').'</td><td>'.
               $endform.'</td></tr></table></td><td>&nbsp;&nbsp;</td>'.
               '<td valign="top"><b>'.&mt('Role:').'</b><br />'.
               '<select name="role"><option value="any"';
    if ($curr->{'role'} eq 'any') {
        $output .= ' selected="selected"';
    }
    $output .=  '>'.&mt('Any').'</option>'."\n";
    my @roles = &Apache::lonuserutils::course_roles($context,undef,1,$lctype);
    foreach my $role (@roles) {
        my $plrole;
        if ($role eq 'cr') {
            $plrole = &mt('Custom Role');
        } else {
            $plrole=&Apache::lonnet::plaintext($role,$crstype);
        }
        my $selstr = '';
        if ($role eq $curr->{'role'}) {
            $selstr = ' selected="selected"';
        }
        $output .= '  <option value="'.$role.'"'.$selstr.'>'.$plrole.'</option>';
    }
    $output .= '</select></td><td>&nbsp;&nbsp;</td><td valign="top"><b>'.
               &mt('Context:').'</b><br /><select name="chgcontext">';
    foreach my $chgtype ('any','auto','updatenow','createcourse','course','domain','selfenroll','requestcourses') {
        my $selstr = '';
        if ($curr->{'chgcontext'} eq $chgtype) {
            $selstr = ' selected="selected"';
        }
        if (($chgtype eq 'auto') || ($chgtype eq 'updatenow')) {
            next if (!&Apache::lonnet::auto_run($cnum,$cdom));
        }
        $output .= '<option value="'.$chgtype.'"'.$selstr.'>'.$lt{$chgtype}.'</option>'."\n";
    }
    $output .= '</select></td>'
              .'</tr></table>';

    # Update Display button
    $output .= '<p>'
              .'<input type="submit" value="'.&mt('Update Display').'" />'
              .'</p>';

    # Server version info
    $output .= '<p style="font-size: smaller;" class="LC_info">'
              .&mt('Only changes made from servers running LON-CAPA [_1] or later are displayed.'
                  ,'2.6.99.0');
    if ($version) {
        $output .= ' '.&mt('This LON-CAPA server is version [_1]',$version);
    }
    $output .= '</p><hr /><br />';
    return $output;
}

sub rolechg_contexts {
    my ($crstype) = @_;
    my %lt = &Apache::lonlocal::texthash (
                                             any          => 'Any',
                                             auto         => 'Automated enrollment',
                                             updatenow    => 'Roster Update',
                                             createcourse => 'Course Creation',
                                             course       => 'User Management in course',
                                             domain       => 'User Management in domain',
                                             selfenroll   => 'Self-enrolled',
                                             requestcourses => 'Course Request',
                                         );
    if ($crstype eq 'Community') {
        $lt{'createcourse'} = &mt('Community Creation');
        $lt{'course'} = &mt('User Management in community');
        $lt{'requestcourses'} = &mt('Community Request');
    }
    return %lt;
}

#-------------------------------------------------- functions for &phase_two
sub user_search_result {
    my ($context,$srch) = @_;
    my %allhomes;
    my %inst_matches;
    my %srch_results;
    my ($response,$currstate,$forcenewuser,$dirsrchres);
    $srch->{'srchterm'} =~ s/\s+/ /g;
    if ($srch->{'srchby'} !~ /^(uname|lastname|lastfirst)$/) {
        $response = &mt('Invalid search.');
    }
    if ($srch->{'srchin'} !~ /^(crs|dom|alc|instd)$/) {
        $response = &mt('Invalid search.');
    }
    if ($srch->{'srchtype'} !~ /^(exact|contains|begins)$/) {
        $response = &mt('Invalid search.');
    }
    if ($srch->{'srchterm'} eq '') {
        $response = &mt('You must enter a search term.');
    }
    if ($srch->{'srchterm'} =~ /^\s+$/) {
        $response = &mt('Your search term must contain more than just spaces.');
    }
    if (($srch->{'srchin'} eq 'dom') || ($srch->{'srchin'} eq 'instd')) {
        if (($srch->{'srchdomain'} eq '') || 
	    ! (&Apache::lonnet::domain($srch->{'srchdomain'}))) {
            $response = &mt('You must specify a valid domain when searching in a domain or institutional directory.')
        }
    }
    if (($srch->{'srchin'} eq 'dom') || ($srch->{'srchin'} eq 'crs') ||
        ($srch->{'srchin'} eq 'alc')) {
        if ($srch->{'srchby'} eq 'uname') {
            my $unamecheck = $srch->{'srchterm'};
            if ($srch->{'srchtype'} eq 'contains') {
                if ($unamecheck !~ /^\w/) {
                    $unamecheck = 'a'.$unamecheck; 
                }
            }
            if ($unamecheck !~ /^$match_username$/) {
                $response = &mt('You must specify a valid username. Only the following are allowed: letters numbers - . @');
            }
        }
    }
    if ($response ne '') {
        $response = '<span class="LC_warning">'.$response.'</span>';
    }
    if ($srch->{'srchin'} eq 'instd') {
        my $instd_chk = &directorysrch_check($srch);
        if ($instd_chk ne 'ok') {
            $response = '<span class="LC_warning">'.$instd_chk.'</span>'.
                        '<br />'.&mt('You may want to search in the LON-CAPA domain instead of the institutional directory.').'<br /><br />';
        }
    }
    if ($response ne '') {
        return ($currstate,$response);
    }
    if ($srch->{'srchby'} eq 'uname') {
        if (($srch->{'srchin'} eq 'dom') || ($srch->{'srchin'} eq 'crs')) {
            if ($env{'form.forcenew'}) {
                if ($srch->{'srchdomain'} ne $env{'request.role.domain'}) {
                    my $uhome=&Apache::lonnet::homeserver($srch->{'srchterm'},$srch->{'srchdomain'});
                    if ($uhome eq 'no_host') {
                        my $domdesc = &Apache::lonnet::domain($env{'request.role.domain'},'description');
                        my $showdom = &display_domain_info($env{'request.role.domain'});
                        $response = &mt('New users can only be created in the domain to which your current role belongs - [_1].',$showdom);
                    } else {
                        $currstate = 'modify';
                    }
                } else {
                    $currstate = 'modify';
                }
            } else {
                if ($srch->{'srchin'} eq 'dom') {
                    if ($srch->{'srchtype'} eq 'exact') {
                        my $uhome=&Apache::lonnet::homeserver($srch->{'srchterm'},$srch->{'srchdomain'});
                        if ($uhome eq 'no_host') {
                            ($currstate,$response,$forcenewuser) =
                                &build_search_response($context,$srch,%srch_results);
                        } else {
                            $currstate = 'modify';
                            my $uname = $srch->{'srchterm'};
                            my $udom = $srch->{'srchdomain'};
                            $srch_results{$uname.':'.$udom} =
                                { &Apache::lonnet::get('environment',
                                                       ['firstname',
                                                        'lastname',
                                                        'permanentemail'],
                                                         $udom,$uname)
                                };
                        }
                    } else {
                        %srch_results = &Apache::lonnet::usersearch($srch);
                        ($currstate,$response,$forcenewuser) =
                            &build_search_response($context,$srch,%srch_results);
                    }
                } else {
                    my $courseusers = &get_courseusers();
                    if ($srch->{'srchtype'} eq 'exact') {
                        if (exists($courseusers->{$srch->{'srchterm'}.':'.$srch->{'srchdomain'}})) {
                            $currstate = 'modify';
                        } else {
                            ($currstate,$response,$forcenewuser) =
                                &build_search_response($context,$srch,%srch_results);
                        }
                    } else {
                        foreach my $user (keys(%$courseusers)) {
                            my ($cuname,$cudomain) = split(/:/,$user);
                            if ($cudomain eq $srch->{'srchdomain'}) {
                                my $matched = 0;
                                if ($srch->{'srchtype'} eq 'begins') {
                                    if ($cuname =~ /^\Q$srch->{'srchterm'}\E/i) {
                                        $matched = 1;
                                    }
                                } else {
                                    if ($cuname =~ /\Q$srch->{'srchterm'}\E/i) {
                                        $matched = 1;
                                    }
                                }
                                if ($matched) {
                                    $srch_results{$user} = 
					{&Apache::lonnet::get('environment',
							     ['firstname',
							      'lastname',
							      'permanentemail'],
							      $cudomain,$cuname)};
                                }
                            }
                        }
                        ($currstate,$response,$forcenewuser) =
                            &build_search_response($context,$srch,%srch_results);
                    }
                }
            }
        } elsif ($srch->{'srchin'} eq 'alc') {
            $currstate = 'query';
        } elsif ($srch->{'srchin'} eq 'instd') {
            ($dirsrchres,%srch_results) = &Apache::lonnet::inst_directory_query($srch);
            if ($dirsrchres eq 'ok') {
                ($currstate,$response,$forcenewuser) = 
                    &build_search_response($context,$srch,%srch_results);
            } else {
                my $showdom = &display_domain_info($srch->{'srchdomain'});
                $response = '<span class="LC_warning">'.
                    &mt('Institutional directory search is not available in domain: [_1]',$showdom).
                    '</span><br />'.
                    &mt('You may want to search in the LON-CAPA domain instead of the institutional directory.').
                    '<br /><br />'; 
            }
        }
    } else {
        if ($srch->{'srchin'} eq 'dom') {
            %srch_results = &Apache::lonnet::usersearch($srch);
            ($currstate,$response,$forcenewuser) = 
                &build_search_response($context,$srch,%srch_results); 
        } elsif ($srch->{'srchin'} eq 'crs') {
            my $courseusers = &get_courseusers(); 
            foreach my $user (keys(%$courseusers)) {
                my ($uname,$udom) = split(/:/,$user);
                my %names = &Apache::loncommon::getnames($uname,$udom);
                my %emails = &Apache::loncommon::getemails($uname,$udom);
                if ($srch->{'srchby'} eq 'lastname') {
                    if ((($srch->{'srchtype'} eq 'exact') && 
                         ($names{'lastname'} eq $srch->{'srchterm'})) || 
                        (($srch->{'srchtype'} eq 'begins') &&
                         ($names{'lastname'} =~ /^\Q$srch->{'srchterm'}\E/i)) ||
                        (($srch->{'srchtype'} eq 'contains') &&
                         ($names{'lastname'} =~ /\Q$srch->{'srchterm'}\E/i))) {
                        $srch_results{$user} = {firstname => $names{'firstname'},
                                            lastname => $names{'lastname'},
                                            permanentemail => $emails{'permanentemail'},
                                           };
                    }
                } elsif ($srch->{'srchby'} eq 'lastfirst') {
                    my ($srchlast,$srchfirst) = split(/,/,$srch->{'srchterm'});
                    $srchlast =~ s/\s+$//;
                    $srchfirst =~ s/^\s+//;
                    if ($srch->{'srchtype'} eq 'exact') {
                        if (($names{'lastname'} eq $srchlast) &&
                            ($names{'firstname'} eq $srchfirst)) {
                            $srch_results{$user} = {firstname => $names{'firstname'},
                                                lastname => $names{'lastname'},
                                                permanentemail => $emails{'permanentemail'},

                                           };
                        }
                    } elsif ($srch->{'srchtype'} eq 'begins') {
                        if (($names{'lastname'} =~ /^\Q$srchlast\E/i) &&
                            ($names{'firstname'} =~ /^\Q$srchfirst\E/i)) {
                            $srch_results{$user} = {firstname => $names{'firstname'},
                                                lastname => $names{'lastname'},
                                                permanentemail => $emails{'permanentemail'},
                                               };
                        }
                    } else {
                        if (($names{'lastname'} =~ /\Q$srchlast\E/i) && 
                            ($names{'firstname'} =~ /\Q$srchfirst\E/i)) {
                            $srch_results{$user} = {firstname => $names{'firstname'},
                                                lastname => $names{'lastname'},
                                                permanentemail => $emails{'permanentemail'},
                                               };
                        }
                    }
                }
            }
            ($currstate,$response,$forcenewuser) = 
                &build_search_response($context,$srch,%srch_results); 
        } elsif ($srch->{'srchin'} eq 'alc') {
            $currstate = 'query';
        } elsif ($srch->{'srchin'} eq 'instd') {
            ($dirsrchres,%srch_results) = &Apache::lonnet::inst_directory_query($srch); 
            if ($dirsrchres eq 'ok') {
                ($currstate,$response,$forcenewuser) = 
                    &build_search_response($context,$srch,%srch_results);
            } else {
                my $showdom = &display_domain_info($srch->{'srchdomain'});                $response = '<span class="LC_warning">'.
                    &mt('Institutional directory search is not available in domain: [_1]',$showdom).
                    '</span><br />'.
                    &mt('You may want to search in the LON-CAPA domain instead of the institutional directory.').
                    '<br /><br />';
            }
        }
    }
    return ($currstate,$response,$forcenewuser,\%srch_results);
}

sub directorysrch_check {
    my ($srch) = @_;
    my $can_search = 0;
    my $response;
    my %dom_inst_srch = &Apache::lonnet::get_dom('configuration',
                                             ['directorysrch'],$srch->{'srchdomain'});
    my $showdom = &display_domain_info($srch->{'srchdomain'});
    if (ref($dom_inst_srch{'directorysrch'}) eq 'HASH') {
        if (!$dom_inst_srch{'directorysrch'}{'available'}) {
            return &mt('Institutional directory search is not available in domain: [_1]',$showdom); 
        }
        if ($dom_inst_srch{'directorysrch'}{'localonly'}) {
            if ($env{'request.role.domain'} ne $srch->{'srchdomain'}) {
                return &mt('Institutional directory search in domain: [_1] is only allowed for users with a current role in the domain.',$showdom); 
            }
            my @usertypes = split(/:/,$env{'environment.inststatus'});
            if (!@usertypes) {
                push(@usertypes,'default');
            }
            if (ref($dom_inst_srch{'directorysrch'}{'cansearch'}) eq 'ARRAY') {
                foreach my $type (@usertypes) {
                    if (grep(/^\Q$type\E$/,@{$dom_inst_srch{'directorysrch'}{'cansearch'}})) {
                        $can_search = 1;
                        last;
                    }
                }
            }
            if (!$can_search) {
                my ($insttypes,$order) = &Apache::lonnet::retrieve_inst_usertypes($srch->{'srchdomain'});
                my @longtypes; 
                foreach my $item (@usertypes) {
                    if (defined($insttypes->{$item})) { 
                        push (@longtypes,$insttypes->{$item});
                    } elsif ($item eq 'default') {
                        push (@longtypes,&mt('other')); 
                    }
                }
                my $insttype_str = join(', ',@longtypes); 
                return &mt('Institutional directory search in domain: [_1] is not available to your user type: ',$showdom).$insttype_str;
            }
        } else {
            $can_search = 1;
        }
    } else {
        return &mt('Institutional directory search has not been configured for domain: [_1]',$showdom);
    }
    my %longtext = &Apache::lonlocal::texthash (
                       uname     => 'username',
                       lastfirst => 'last name, first name',
                       lastname  => 'last name',
                       contains  => 'contains',
                       exact     => 'as exact match to',
                       begins    => 'begins with',
                   );
    if ($can_search) {
        if (ref($dom_inst_srch{'directorysrch'}{'searchby'}) eq 'ARRAY') {
            if (!grep(/^\Q$srch->{'srchby'}\E$/,@{$dom_inst_srch{'directorysrch'}{'searchby'}})) {
                return &mt('Institutional directory search in domain: [_1] is not available for searching by "[_2]"',$showdom,$longtext{$srch->{'srchby'}});
            }
        } else {
            return &mt('Institutional directory search in domain: [_1] is not available.', $showdom);
        }
    }
    if ($can_search) {
        if (ref($dom_inst_srch{'directorysrch'}{'searchtypes'}) eq 'ARRAY') {
            if (grep(/^\Q$srch->{'srchtype'}\E/,@{$dom_inst_srch{'directorysrch'}{'searchtypes'}})) {
                return 'ok';
            } else {
                return &mt('Institutional directory search in domain [_1] is not available for the requested search type: "[_2]"',$showdom,$longtext{$srch->{'srchtype'}});
            }
        } else {
            if ((($dom_inst_srch{'directorysrch'}{'searchtypes'} eq 'specify') &&
                 ($srch->{'srchtype'} eq 'exact' || $srch->{'srchtype'} eq 'contains')) ||
                ($dom_inst_srch{'directorysrch'}{'searchtypes'} eq $srch->{'srchtype'})) {
                return 'ok';
            } else {
                return &mt('Institutional directory search in domain [_1] is not available for the requested search type: "[_2]"',$showdom,$longtext{$srch->{'srchtype'}});
            }
        }
    }
}

sub get_courseusers {
    my %advhash;
    my $classlist = &Apache::loncoursedata::get_classlist();
    my %coursepersonnel=&Apache::lonnet::get_course_adv_roles();
    foreach my $role (sort(keys(%coursepersonnel))) {
        foreach my $user (split(/\,/,$coursepersonnel{$role})) {
	    if (!exists($classlist->{$user})) {
		$classlist->{$user} = [];
	    }
        }
    }
    return $classlist;
}

sub build_search_response {
    my ($context,$srch,%srch_results) = @_;
    my ($currstate,$response,$forcenewuser);
    my %names = (
          'uname' => 'username',
          'lastname' => 'last name',
          'lastfirst' => 'last name, first name',
          'crs' => 'this course',
          'dom' => 'LON-CAPA domain: ',
          'instd' => 'the institutional directory for domain: ',
    );

    my %single = (
                   begins   => 'A match',
                   contains => 'A match',
                   exact    => 'An exact match',
                 );
    my %nomatch = (
                   begins   => 'No match',
                   contains => 'No match',
                   exact    => 'No exact match',
                  );
    if (keys(%srch_results) > 1) {
        $currstate = 'select';
    } else {
        if (keys(%srch_results) == 1) {
            $currstate = 'modify';
            $response = &mt("$single{$srch->{'srchtype'}} was found for the $names{$srch->{'srchby'}} ([_1]) in $names{$srch->{'srchin'}}.",$srch->{'srchterm'});
            if ($srch->{'srchin'} eq 'dom' || $srch->{'srchin'} eq 'instd') {
                $response .= &display_domain_info($srch->{'srchdomain'});
            }
        } else {
            $response = '<span class="LC_warning">'.&mt("$nomatch{$srch->{'srchtype'}} found for the $names{$srch->{'srchby'}} ([_1]) in $names{$srch->{'srchin'}}",$srch->{'srchterm'});
            if ($srch->{'srchin'} eq 'dom' || $srch->{'srchin'} eq 'instd') {
                $response .= &display_domain_info($srch->{'srchdomain'});
            }
            $response .= '</span>';
            if ($srch->{'srchin'} ne 'alc') {
                $forcenewuser = 1;
                my $cansrchinst = 0; 
                if ($srch->{'srchdomain'}) {
                    my %domconfig = &Apache::lonnet::get_dom('configuration',['directorysrch'],$srch->{'srchdomain'});
                    if (ref($domconfig{'directorysrch'}) eq 'HASH') {
                        if ($domconfig{'directorysrch'}{'available'}) {
                            $cansrchinst = 1;
                        } 
                    }
                }
                if ((($srch->{'srchby'} eq 'lastfirst') || 
                     ($srch->{'srchby'} eq 'lastname')) &&
                    ($srch->{'srchin'} eq 'dom')) {
                    if ($cansrchinst) {
                        $response .= '<br />'.&mt('You may want to broaden your search to a search of the institutional directory for the domain.');
                    }
                }
                if ($srch->{'srchin'} eq 'crs') {
                    $response .= '<br />'.&mt('You may want to broaden your search to the selected LON-CAPA domain.');
                }
            }
            my $createdom = $env{'request.role.domain'};
            if ($context eq 'requestcrs') {
                if ($env{'form.coursedom'} ne '') {
                    $createdom = $env{'form.coursedom'};
                }
            }
            if (!($srch->{'srchby'} eq 'uname' && $srch->{'srchin'} eq 'dom' && $srch->{'srchtype'} eq 'exact' && $srch->{'srchdomain'} eq $createdom)) {
                my $cancreate =
                    &Apache::lonuserutils::can_create_user($createdom,$context);
                my $targetdom = '<span class="LC_cusr_emph">'.$createdom.'</span>';
                if ($cancreate) {
                    my $showdom = &display_domain_info($createdom);
                    $response .= '<br /><br />'
                                .'<b>'.&mt('To add a new user:').'</b>'
                                .'<br />';
                    if ($context eq 'requestcrs') {
                        $response .= &mt("(You can only define new users in the new course's domain - [_1])",$targetdom);
                    } else {
                        $response .= &mt("(You can only create new users in your current role's domain - [_1])",$targetdom);
                    }
                    $response .= '<ul><li>'
                                .&mt("Set 'Domain/institution to search' to: [_1]",'<span class="LC_cusr_emph">'.$showdom.'</span>')
                                .'</li><li>'
                                .&mt("Set 'Search criteria' to: [_1]username is ..... in selected LON-CAPA domain[_2]",'<span class="LC_cusr_emph">','</span>')
                                .'</li><li>'
                                .&mt('Provide the proposed username')
                                .'</li><li>'
                                .&mt("Click 'Search'")
                                .'</li></ul><br />';
                } else {
                    my $helplink = ' href="javascript:helpMenu('."'display'".')"';
                    $response .= '<br /><br />';
                    if ($context eq 'requestcrs') {
                        $response .= &mt("You are not authorized to define new users in the new course's domain - [_1].",$targetdom);
                    } else {
                        $response .= &mt("You are not authorized to create new users in your current role's domain - [_1].",$targetdom);
                    }
                    $response .= '<br />'
                                .&mt('Please contact the [_1]helpdesk[_2] if you need to create a new user.'
                                    ,' <a'.$helplink.'>'
                                    ,'</a>')
                                .'<br /><br />';
                }
            }
        }
    }
    return ($currstate,$response,$forcenewuser);
}

sub display_domain_info {
    my ($dom) = @_;
    my $output = $dom;
    if ($dom ne '') { 
        my $domdesc = &Apache::lonnet::domain($dom,'description');
        if ($domdesc ne '') {
            $output .= ' <span class="LC_cusr_emph">('.$domdesc.')</span>';
        }
    }
    return $output;
}

sub crumb_utilities {
    my %elements = (
       crtuser => {
           srchterm => 'text',
           srchin => 'selectbox',
           srchby => 'selectbox',
           srchtype => 'selectbox',
           srchdomain => 'selectbox',
       },
       crtusername => {
           srchterm => 'text',
           srchdomain => 'selectbox',
       },
       docustom => {
           rolename => 'selectbox',
           newrolename => 'textbox',
       },
       studentform => {
           srchterm => 'text',
           srchin => 'selectbox',
           srchby => 'selectbox',
           srchtype => 'selectbox',
           srchdomain => 'selectbox',
       },
    );

    my $jsback .= qq|
function backPage(formname,prevphase,prevstate) {
    if (typeof prevphase == 'undefined') {
        formname.phase.value = '';
    }
    else {  
        formname.phase.value = prevphase;
    }
    if (typeof prevstate == 'undefined') {
        formname.currstate.value = '';
    }
    else {
        formname.currstate.value = prevstate;
    }
    formname.submit();
}
|;
    return ($jsback,\%elements);
}

sub course_level_table {
    my (%inccourses) = @_;
    my $table = '';
# Custom Roles?

    my %customroles=&Apache::lonuserutils::my_custom_roles();
    my %lt=&Apache::lonlocal::texthash(
            'exs'  => "Existing sections",
            'new'  => "Define new section",
            'ssd'  => "Set Start Date",
            'sed'  => "Set End Date",
            'crl'  => "Course Level",
            'act'  => "Activate",
            'rol'  => "Role",
            'ext'  => "Extent",
            'grs'  => "Section",
            'sta'  => "Start",
            'end'  => "End"
    );

    foreach my $protectedcourse (sort(keys(%inccourses))) {
	my $thiscourse=$protectedcourse;
	$thiscourse=~s:_:/:g;
	my %coursedata=&Apache::lonnet::coursedescription($thiscourse);
        my $isowner = &is_courseowner($protectedcourse,$coursedata{'internal.courseowner'});
	my $area=$coursedata{'description'};
        my $crstype=$coursedata{'type'};
	if (!defined($area)) { $area=&mt('Unavailable course').': '.$protectedcourse; }
	my ($domain,$cnum)=split(/\//,$thiscourse);
        my %sections_count;
        if (defined($env{'request.course.id'})) {
            if ($env{'request.course.id'} eq $domain.'_'.$cnum) {
                %sections_count = 
		    &Apache::loncommon::get_sections($domain,$cnum);
            }
        }
        my @roles = &Apache::lonuserutils::roles_by_context('course','',$crstype);
	foreach my $role (@roles) {
            my $plrole=&Apache::lonnet::plaintext($role,$crstype);
            if ((&Apache::lonnet::allowed('c'.$role,$thiscourse)) ||
                ((($role eq 'cc') || ($role eq 'co')) && ($isowner))) {
                $table .= &course_level_row($protectedcourse,$role,$area,$domain,
                                            $plrole,\%sections_count,\%lt);
            } elsif ($env{'request.course.sec'} ne '') {
                if (&Apache::lonnet::allowed('c'.$role,$thiscourse.'/'.
                                             $env{'request.course.sec'})) {
                    $table .= &course_level_row($protectedcourse,$role,$area,$domain,
                                                $plrole,\%sections_count,\%lt);
                }
            }
        }
        if (&Apache::lonnet::allowed('ccr',$thiscourse)) {
            foreach my $cust (sort(keys(%customroles))) {
                next if ($crstype eq 'Community' && $customroles{$cust} =~ /bre\&S/);
                my $role = 'cr_cr_'.$env{'user.domain'}.'_'.$env{'user.name'}.'_'.$cust;
                $table .= &course_level_row($protectedcourse,$role,$area,$domain,
                                            $cust,\%sections_count,\%lt);
            }
	}
    }
    return '' if ($table eq ''); # return nothing if there is nothing 
                                 # in the table
    my $result;
    if (!$env{'request.course.id'}) {
        $result = '<h4>'.$lt{'crl'}.'</h4>'."\n";
    }
    $result .= 
&Apache::loncommon::start_data_table().
&Apache::loncommon::start_data_table_header_row().
'<th>'.$lt{'act'}.'</th><th>'.$lt{'rol'}.'</th><th>'.$lt{'ext'}.'</th>
<th>'.$lt{'grs'}.'</th><th>'.$lt{'sta'}.'</th><th>'.$lt{'end'}.'</th>'.
&Apache::loncommon::end_data_table_header_row().
$table.
&Apache::loncommon::end_data_table();
    return $result;
}

sub course_level_row {
    my ($protectedcourse,$role,$area,$domain,$plrole,$sections_count,$lt) = @_;
    my $row = &Apache::loncommon::start_data_table_row().
              ' <td><input type="checkbox" name="act_'.
              $protectedcourse.'_'.$role.'" /></td>'."\n".
              ' <td>'.$plrole.'</td>'."\n".
              ' <td>'.$area.'<br />Domain: '.$domain.'</td>'."\n";
    if (($role eq 'cc') || ($role eq 'co')) {
        $row .= '<td>&nbsp;</td>';
    } elsif ($env{'request.course.sec'} ne '') {
        $row .= ' <td><input type="hidden" value="'.
                $env{'request.course.sec'}.'" '.
                'name="sec_'.$protectedcourse.'_'.$role.'" />'.
                $env{'request.course.sec'}.'</td>';
    } else {
        if (ref($sections_count) eq 'HASH') {
            my $currsec = 
                &Apache::lonuserutils::course_sections($sections_count,
                                                       $protectedcourse.'_'.$role);
            $row .= '<td><table class="LC_createuser">'."\n".
                    '<tr class="LC_section_row">'."\n".
                    ' <td valign="top">'.$lt->{'exs'}.'<br />'.
                       $currsec.'</td>'."\n".
                     ' <td>&nbsp;&nbsp;</td>'."\n".
                     ' <td valign="top">&nbsp;'.$lt->{'new'}.'<br />'.
                     '<input type="text" name="newsec_'.$protectedcourse.'_'.$role.
                     '" value="" />'.
                     '<input type="hidden" '.
                     'name="sec_'.$protectedcourse.'_'.$role.'" /></td>'."\n".
                     '</tr></table></td>'."\n";
        } else {
            $row .= '<td><input type="text" size="10" '.
                      'name="sec_'.$protectedcourse.'_'.$role.'" /></td>'."\n";
        }
    }
    $row .= <<ENDTIMEENTRY;
<td><input type="hidden" name="start_$protectedcourse\_$role" value="" />
<a href=
"javascript:pjump('date_start','Start Date $plrole',document.cu.start_$protectedcourse\_$role.value,'start_$protectedcourse\_$role','cu.pres','dateset')">$lt->{'ssd'}</a></td>
<td><input type="hidden" name="end_$protectedcourse\_$role" value="" />
<a href=
"javascript:pjump('date_end','End Date $plrole',document.cu.end_$protectedcourse\_$role.value,'end_$protectedcourse\_$role','cu.pres','dateset')">$lt->{'sed'}</a></td>
ENDTIMEENTRY
    $row .= &Apache::loncommon::end_data_table_row();
    return $row;
}

sub course_level_dc {
    my ($dcdom) = @_;
    my %customroles=&Apache::lonuserutils::my_custom_roles();
    my @roles = &Apache::lonuserutils::roles_by_context('course');
    my $hiddenitems = '<input type="hidden" name="dcdomain" value="'.$dcdom.'" />'.
                      '<input type="hidden" name="origdom" value="'.$dcdom.'" />'.
                      '<input type="hidden" name="dccourse" value="" />';
    my $courseform='<b>'.&Apache::loncommon::selectcourse_link
            ('cu','dccourse','dcdomain','coursedesc',undef,undef,'Course/Community','crstype').'</b>';
    my $cb_jscript = &Apache::loncommon::coursebrowser_javascript($dcdom,'currsec','cu','role','Course/Community Browser');
    my %lt=&Apache::lonlocal::texthash(
                    'rol'  => "Role",
                    'grs'  => "Section",
                    'exs'  => "Existing sections",
                    'new'  => "Define new section", 
                    'sta'  => "Start",
                    'end'  => "End",
                    'ssd'  => "Set Start Date",
                    'sed'  => "Set End Date"
                  );
    my $header = '<h4>'.&mt('Course/Community Level').'</h4>'.
                 &Apache::loncommon::start_data_table().
                 &Apache::loncommon::start_data_table_header_row().
                 '<th>'.$courseform.'</th><th>'.$lt{'rol'}.'</th><th>'.$lt{'grs'}.'</th><th>'.$lt{'sta'}.'</th><th>'.$lt{'end'}.'</th>'.
                 &Apache::loncommon::end_data_table_header_row();
    my $otheritems = &Apache::loncommon::start_data_table_row()."\n".
                     '<td><br /><input type="text" name="coursedesc" value="" onfocus="this.blur();opencrsbrowser('."'cu','dccourse','dcdomain','coursedesc','','','','crstype'".')" /></td>'."\n".
                     '<td valign><br /><select name="role">'."\n";
    foreach my $role (@roles) {
        my $plrole=&Apache::lonnet::plaintext($role);
        $otheritems .= '  <option value="'.$role.'">'.$plrole;
    }
    if ( keys %customroles > 0) {
        foreach my $cust (sort keys %customroles) {
            my $custrole='cr_cr_'.$env{'user.domain'}.
                    '_'.$env{'user.name'}.'_'.$cust;
            $otheritems .= '  <option value="'.$custrole.'">'.$cust;
        }
    }
    $otheritems .= '</select></td><td>'.
                     '<table border="0" cellspacing="0" cellpadding="0">'.
                     '<tr><td valign="top"><b>'.$lt{'exs'}.'</b><br /><select name="currsec">'.
                     ' <option value=""><--'.&mt('Pick course first').'</select></td>'.
                     '<td>&nbsp;&nbsp;</td>'.
                     '<td valign="top">&nbsp;<b>'.$lt{'new'}.'</b><br />'.
                     '<input type="text" name="newsec" value="" />'.
                     '<input type="hidden" name="section" value="" />'.
                     '<input type="hidden" name="groups" value="" />'.
                     '<input type="hidden" name="crstype" value="" /></td>'.
                     '</tr></table></td>';
    $otheritems .= <<ENDTIMEENTRY;
<td><br /><input type="hidden" name="start" value='' />
<a href=
"javascript:pjump('date_start','Start Date',document.cu.start.value,'start','cu.pres','dateset')">$lt{'ssd'}</a></td>
<td><br /><input type="hidden" name="end" value='' />
<a href=
"javascript:pjump('date_end','End Date',document.cu.end.value,'end','cu.pres','dateset')">$lt{'sed'}</a></td>
ENDTIMEENTRY
    $otheritems .= &Apache::loncommon::end_data_table_row().
                   &Apache::loncommon::end_data_table()."\n";
    return $cb_jscript.$header.$hiddenitems.$otheritems;
}

sub update_selfenroll_config {
    my ($r,$context,$permission) = @_;
    my ($row,$lt) = &get_selfenroll_titles();
    my %curr_groups = &Apache::longroup::coursegroups();
    my (%changes,%warning);
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    my $curr_types;
    if (ref($row) eq 'ARRAY') {
        foreach my $item (@{$row}) {
            if ($item eq 'enroll_dates') {
                my (%currenrolldate,%newenrolldate);
                foreach my $type ('start','end') {
                    $currenrolldate{$type} = $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_'.$type.'_date'};
                    $newenrolldate{$type} = &Apache::lonhtmlcommon::get_date_from_form('selfenroll_'.$type.'_date');
                    if ($newenrolldate{$type} ne $currenrolldate{$type}) {
                        $changes{'internal.selfenroll_'.$type.'_date'} = $newenrolldate{$type};
                    }
                }
            } elsif ($item eq 'access_dates') {
                my (%currdate,%newdate);
                foreach my $type ('start','end') {
                    $currdate{$type} = $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_'.$type.'_access'};
                    $newdate{$type} = &Apache::lonhtmlcommon::get_date_from_form('selfenroll_'.$type.'_access');
                    if ($newdate{$type} ne $currdate{$type}) {
                        $changes{'internal.selfenroll_'.$type.'_access'} = $newdate{$type};
                    }
                }
            } elsif ($item eq 'types') {
                $curr_types =
                    $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_'.$item};
                if ($env{'form.selfenroll_all'}) {
                    if ($curr_types ne '*') {
                        $changes{'internal.selfenroll_types'} = '*';
                    } else {
                        next;
                    }
                } else {
                    my %currdoms;
                    my @entries = split(/;/,$curr_types);
                    my @deletedoms = &Apache::loncommon::get_env_multiple('form.selfenroll_delete');
                    my @activations = &Apache::loncommon::get_env_multiple('form.selfenroll_activate');
                    my $newnum = 0;
                    my @latesttypes;
                    foreach my $num (@activations) {
                        my @types = &Apache::loncommon::get_env_multiple('form.selfenroll_types_'.$num);
                        if (@types > 0) {
                            @types = sort(@types);
                            my $typestr = join(',',@types);
                            my $typedom = $env{'form.selfenroll_dom_'.$num};
                            $latesttypes[$newnum] = $typedom.':'.$typestr;
                            $currdoms{$typedom} = 1;
                            $newnum ++;
                        }
                    }
                    for (my $j=0; $j<$env{'form.selfenroll_types_total'}; $j++) {
                        if ((!grep(/^$j$/,@deletedoms)) && (!grep(/^$j$/,@activations))) {
                            my @types = &Apache::loncommon::get_env_multiple('form.selfenroll_types_'.$j);
                            if (@types > 0) {
                                @types = sort(@types);
                                my $typestr = join(',',@types);
                                my $typedom = $env{'form.selfenroll_dom_'.$j};
                                $latesttypes[$newnum] = $typedom.':'.$typestr;
                                $currdoms{$typedom} = 1;
                                $newnum ++;
                            }
                        }
                    }
                    if ($env{'form.selfenroll_newdom'} ne '') {
                        my $typedom = $env{'form.selfenroll_newdom'};
                        if ((!defined($currdoms{$typedom})) && 
                            (&Apache::lonnet::domain($typedom) ne '')) {
                            my $typestr;
                            my ($othertitle,$usertypes,$types) = 
                                &Apache::loncommon::sorted_inst_types($typedom);
                            my $othervalue = 'any';
                            if ((ref($types) eq 'ARRAY') && (ref($usertypes) eq 'HASH')) {
                                if (@{$types} > 0) {
                                    my @esc_types = map { &escape($_); } @{$types};
                                    $othervalue = 'other';
                                    $typestr = join(',',(@esc_types,$othervalue));
                                }
                                $typestr = $othervalue;
                            } else {
                                $typestr = $othervalue;
                            } 
                            $latesttypes[$newnum] = $typedom.':'.$typestr;
                            $newnum ++ ;
                        }
                    }
                    my $selfenroll_types = join(';',@latesttypes);
                    if ($selfenroll_types ne $curr_types) {
                        $changes{'internal.selfenroll_types'} = $selfenroll_types;
                    }
                }
            } elsif ($item eq 'limit') {
                my $newlimit = $env{'form.selfenroll_limit'};
                my $newcap = $env{'form.selfenroll_cap'};
                $newcap =~s/\s+//g;
                my $currlimit =  $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_limit'};
                $currlimit = 'none' if ($currlimit eq '');
                my $currcap = $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_cap'};
                if ($newlimit ne $currlimit) {
                    if ($newlimit ne 'none') {
                        if ($newcap =~ /^\d+$/) {
                            if ($newcap ne $currcap) {
                                $changes{'internal.selfenroll_cap'} = $newcap;
                            }
                            $changes{'internal.selfenroll_limit'} = $newlimit;
                        } else {
                            $warning{$item} = &mt('Maximum enrollment setting unchanged.').'<br />'.&mt('The value provided was invalid - it must be a positive integer if enrollment is being limited.'); 
                        }
                    } elsif ($currcap ne '') {
                        $changes{'internal.selfenroll_cap'} = '';
                        $changes{'internal.selfenroll_limit'} = $newlimit; 
                    }
                } elsif ($currlimit ne 'none') {
                    if ($newcap =~ /^\d+$/) {
                        if ($newcap ne $currcap) {
                            $changes{'internal.selfenroll_cap'} = $newcap;
                        }
                    } else {
                        $warning{$item} = &mt('Maximum enrollment setting unchanged.').'<br />'.&mt('The value provided was invalid - it must be a positive integer if enrollment is being limited.');
                    }
                }
            } elsif ($item eq 'approval') {
                my (@currnotified,@newnotified);
                my $currapproval = $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_approval'};
                my $currnotifylist = $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_notifylist'};
                if ($currnotifylist ne '') {
                    @currnotified = split(/,/,$currnotifylist);
                    @currnotified = sort(@currnotified);
                }
                my $newapproval = $env{'form.selfenroll_approval'};
                @newnotified = &Apache::loncommon::get_env_multiple('form.selfenroll_notify');
                @newnotified = sort(@newnotified);
                if ($newapproval ne $currapproval) {
                    $changes{'internal.selfenroll_approval'} = $newapproval;
                    if (!$newapproval) {
                        if ($currnotifylist ne '') {
                            $changes{'internal.selfenroll_notifylist'} = '';
                        }
                    } else {
                        my @differences =  
                            &Apache::loncommon::compare_arrays(\@currnotified,\@newnotified);
                        if (@differences > 0) {
                            if (@newnotified > 0) {
                                $changes{'internal.selfenroll_notifylist'} = join(',',@newnotified);
                            } else {
                                $changes{'internal.selfenroll_notifylist'} = join(',',@newnotified);
                            }
                        }
                    }
                } else {
                    my @differences = &Apache::loncommon::compare_arrays(\@currnotified,\@newnotified);
                    if (@differences > 0) {
                        if (@newnotified > 0) {
                            $changes{'internal.selfenroll_notifylist'} = join(',',@newnotified);
                        } else {
                            $changes{'internal.selfenroll_notifylist'} = '';
                        }
                    }
                }
            } else {
                my $curr_val = 
                    $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_'.$item};
                my $newval = $env{'form.selfenroll_'.$item};
                if ($item eq 'section') {
                    $newval = $env{'form.sections'};
                    if (defined($curr_groups{$newval})) {
                        $newval = $curr_val;
                        $warning{$item} = &mt('Section for self-enrolled users unchanged as the proposed section is a group').'<br />'.&mt('Group names and section names must be distinct');
                    } elsif ($newval eq 'all') {
                        $newval = $curr_val;
                        $warning{$item} = &mt('Section for self-enrolled users unchanged, as "all" is a reserved section name.');
                    }
                    if ($newval eq '') {
                        $newval = 'none';
                    }
                }
                if ($newval ne $curr_val) {
                    $changes{'internal.selfenroll_'.$item} = $newval;
                }
            }
        }
        if (keys(%warning) > 0) {
            foreach my $item (@{$row}) {
                if (exists($warning{$item})) {
                    $r->print($warning{$item}.'<br />');
                }
            } 
        }
        if (keys(%changes) > 0) {
            my $putresult = &Apache::lonnet::put('environment',\%changes,$cdom,$cnum);
            if ($putresult eq 'ok') {
                if ((exists($changes{'internal.selfenroll_types'})) ||
                    (exists($changes{'internal.selfenroll_start_date'}))  ||
                    (exists($changes{'internal.selfenroll_end_date'}))) {
                    my %crsinfo = &Apache::lonnet::courseiddump($cdom,'.',1,'.','.',
                                                                $cnum,undef,undef,'Course');
                    my $chome = &Apache::lonnet::homeserver($cnum,$cdom);
                    if (ref($crsinfo{$env{'request.course.id'}}) eq 'HASH') {
                        foreach my $item ('selfenroll_types','selfenroll_start_date','selfenroll_end_date') {
                            if (exists($changes{'internal.'.$item})) {
                                $crsinfo{$env{'request.course.id'}}{$item} = 
                                    $changes{'internal.'.$item};
                            }
                        }
                        my $crsputresult =
                            &Apache::lonnet::courseidput($cdom,\%crsinfo,
                                                         $chome,'notime');
                    }
                }
                $r->print(&mt('The following changes were made to self-enrollment settings:').'<ul>');
                foreach my $item (@{$row}) {
                    my $title = $item;
                    if (ref($lt) eq 'HASH') {
                        $title = $lt->{$item};
                    }
                    if ($item eq 'enroll_dates') {
                        foreach my $type ('start','end') {
                            if (exists($changes{'internal.selfenroll_'.$type.'_date'})) {
                                my $newdate = &Apache::lonlocal::locallocaltime($changes{'internal.selfenroll_'.$type.'_date'});
                                $r->print('<li>'.&mt('[_1]: "[_2]" set to "[_3]".',
                                          $title,$type,$newdate).'</li>');
                            }
                        }
                    } elsif ($item eq 'access_dates') {
                        foreach my $type ('start','end') {
                            if (exists($changes{'internal.selfenroll_'.$type.'_access'})) {
                                my $newdate = &Apache::lonlocal::locallocaltime($changes{'internal.selfenroll_'.$type.'_access'});
                                $r->print('<li>'.&mt('[_1]: "[_2]" set to "[_3]".',
                                          $title,$type,$newdate).'</li>');
                            }
                        }
                    } elsif ($item eq 'limit') {
                        if ((exists($changes{'internal.selfenroll_limit'})) ||
                            (exists($changes{'internal.selfenroll_cap'}))) {
                            my ($newval,$newcap);
                            if ($changes{'internal.selfenroll_cap'} ne '') {
                                $newcap = $changes{'internal.selfenroll_cap'}
                            } else {
                                $newcap = $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_cap'};
                            }
                            if ($changes{'internal.selfenroll_limit'} eq 'none') {
                                $newval = &mt('No limit');
                            } elsif ($changes{'internal.selfenroll_limit'} eq 
                                     'allstudents') {
                                $newval = &mt('New self-enrollment no longer allowed when total (all students) reaches [_1].',$newcap);
                            } elsif ($changes{'internal.selfenroll_limit'} eq 'selfenrolled') {
                                $newval = &mt('New self-enrollment no longer allowed when total number of self-enrolled students reaches [_1].',$newcap);
                            } else {
                                my $currlimit =  $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_limit'};
                                if ($currlimit eq 'allstudents') {
                                    $newval = &mt('New self-enrollment no longer allowed when total (all students) reaches [_1].',$newcap);
                                } elsif ($changes{'internal.selfenroll_limit'} eq 'selfenrolled') {
                                    $newval =  &mt('New self-enrollment no longer allowed when total number of self-enrolled students reaches [_1].',$newcap);
                                }
                            }
                            $r->print('<li>'.&mt('"[_1]" set to "[_2]".',$title,$newval).'</li>'."\n");
                        }
                    } elsif ($item eq 'approval') {
                        if ((exists($changes{'internal.selfenroll_approval'})) ||
                            (exists($changes{'internal.selfenroll_notifylist'}))) {
                            my ($newval,$newnotify);
                            if (exists($changes{'internal.selfenroll_notifylist'})) {
                                $newnotify = $changes{'internal.selfenroll_notifylist'};
                            } else {   
                                $newnotify = $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_notifylist'};
                            }
                            if ($changes{'internal.selfenroll_approval'}) {
                                $newval = &mt('Yes');
                            } elsif ($changes{'internal.selfenroll_approval'} eq '0') {
                                $newval = &mt('No');
                            } else {
                                my $currapproval = 
                                    $env{'course.'.$env{'request.course.id'}.'.internal.selfenroll_approval'};
                                if ($currapproval) {
                                    $newval = &mt('Yes');
                                } else {
                                    $newval = &mt('No');
                                }
                            }
                            $r->print('<li>'.&mt('"[_1]" set to "[_2]".',$title,$newval));
                            if ($newnotify) {
                                $r->print('<br />'.&mt('The following will be notified when an enrollment request needs approval, or has been approved: [_1].',$newnotify));
                            } else {
                                $r->print('<br />'.&mt('No notifications sent when an enrollment request needs approval, or has been approved.'));
                            }
                            $r->print('</li>'."\n");
                        }
                    } else {
                        if (exists($changes{'internal.selfenroll_'.$item})) {
                            my $newval = $changes{'internal.selfenroll_'.$item};
                            if ($item eq 'types') {
                                if ($newval eq '') {
                                    $newval = &mt('None');
                                } elsif ($newval eq '*') {
                                    $newval = &mt('Any user in any domain');
                                }
                            } elsif ($item eq 'registered') {
                                if ($newval eq '1') {
                                    $newval = &mt('Yes');
                                } elsif ($newval eq '0') {
                                    $newval = &mt('No');
                                }
                            }
                            $r->print('<li>'.&mt('"[_1]" set to "[_2]".',$title,$newval).'</li>'."\n");
                        }
                    }
                }
                $r->print('</ul>');
                my %newenvhash;
                foreach my $key (keys(%changes)) {
                    $newenvhash{'course.'.$env{'request.course.id'}.'.'.$key} = $changes{$key};
                }
                &Apache::lonnet::appenv(\%newenvhash);
            } else {
                $r->print(&mt('An error occurred when saving changes to self-enrollment settings in this course.').'<br />'.&mt('The error was: [_1].',$putresult));
            }
        } else {
            $r->print(&mt('No changes were made to the existing self-enrollment settings in this course.'));
        }
    } else {
        $r->print(&mt('No changes were made to the existing self-enrollment settings in this course.'));
    }
    my ($visible,$cansetvis,$vismsgs,$visactions) = &visible_in_cat($cdom,$cnum);
    if (ref($visactions) eq 'HASH') {
        if (!$visible) {
            $r->print('<br />'.$visactions->{'miss'}.'<br />'.$visactions->{'yous'}.
                      '<br />');
            if (ref($vismsgs) eq 'ARRAY') {
                $r->print('<br />'.$visactions->{'take'}.'<ul>');
                foreach my $item (@{$vismsgs}) {
                    $r->print('<li>'.$visactions->{$item}.'</li>');
                }
                $r->print('</ul>');
            }
            $r->print($cansetvis);
        }
    } 
    return;
}

sub get_selfenroll_titles {
    my @row = ('types','registered','enroll_dates','access_dates','section',
               'approval','limit');
    my %lt = &Apache::lonlocal::texthash (
                types        => 'Users allowed to self-enroll in this course',
                registered   => 'Restrict self-enrollment to students officially registered for the course',
                enroll_dates => 'Dates self-enrollment available',
                access_dates => 'Course access dates assigned to self-enrolling users',
                section      => 'Section assigned to self-enrolling users',
                approval     => 'Self-enrollment requests need approval?',
                limit        => 'Enrollment limit',
             );
    return (\@row,\%lt);
}

sub is_courseowner {
    my ($thiscourse,$courseowner) = @_;
    if ($courseowner eq '') {
        if ($env{'request.course.id'} eq $thiscourse) {
            $courseowner = $env{'course.'.$env{'request.course.id'}.'.internal.courseowner'};
        }
    }
    if ($courseowner ne '') {
        if ($courseowner eq $env{'user.name'}.':'.$env{'user.domain'}) {
            return 1;
        }
    }
    return;
}

#---------------------------------------------- end functions for &phase_two

#--------------------------------- functions for &phase_two and &phase_three

#--------------------------end of functions for &phase_two and &phase_three

1;
__END__


