# The LearningOnline Network with CAPA
# handler for DC-only modifiable course settings
#
# $Id: lonmodifycourse.pm,v 1.51 2010/03/15 19:10:59 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
package Apache::lonmodifycourse;

use strict;
use Apache::Constants qw(:common :http);
use Apache::lonnet;
use Apache::loncommon;
use Apache::lonhtmlcommon;
use Apache::lonlocal;
use Apache::lonuserutils;
use Apache::lonpickcourse;
use lib '/home/httpd/lib/perl';
use LONCAPA;

sub get_dc_settable {
    my ($type) = @_;
    if ($type eq 'Community') {
        return ('courseowner');
    } else {
        return ('courseowner','coursecode','authtype','autharg');
    }
}

sub autoenroll_keys {
    my $internals = ['coursecode','courseowner','authtype','autharg','autoadds','autodrops',
                         'autostart','autoend','sectionnums','crosslistings',
                         'co-owners'];
    my $accessdates = ['default_enrollment_start_date','default_enrollment_end_date'];
    return ($internals,$accessdates);
}

sub catalog_settable {
    my ($confhash,$type) = @_;
    my @settable;
    if (ref($confhash) eq 'HASH') {
        if ($type eq 'Community') {
            if ($confhash->{'togglecatscomm'} ne 'comm') {
                push(@settable,'togglecats');
            }
            if ($confhash->{'categorizecomm'} ne 'comm') {
                push(@settable,'categorize');
            }
        } else {
            if ($confhash->{'togglecats'} ne 'crs') {
                push(@settable,'togglecats');
            }
            if ($confhash->{'categorize'} ne 'crs') {
                push(@settable,'categorize');
            }
        }
    } else {
        push(@settable,('togglecats','categorize'));
    }
    return @settable;
}

sub get_enrollment_settings {
    my ($cdom,$cnum) = @_;
    my ($internals,$accessdates) = &autoenroll_keys();
    my @items;
    if ((ref($internals) eq 'ARRAY') && (ref($accessdates) eq 'ARRAY')) { 
        @items = map { 'internal.'.$_; } (@{$internals});
        push(@items,@{$accessdates});
    }
    my %settings = &Apache::lonnet::get('environment',\@items,$cdom,$cnum);
    my %enrollvar;
    $enrollvar{'autharg'} = '';
    $enrollvar{'authtype'} = '';
    foreach my $item (keys(%settings)) {
        if ($item =~ m/^internal\.(.+)$/) {
            my $type = $1;
            if ( ($type eq "autoadds") || ($type eq "autodrops") ) {
                if ($settings{$item} == 1) {
                    $enrollvar{$type} = "ON";
                } else {
                    $enrollvar{$type} = "OFF";
                }
            } elsif ( ($type eq "autostart") || ($type eq "autoend") ) {
                if ( ($type eq "autoend") && ($settings{$item} == 0) ) {
                    $enrollvar{$type} = &mt('No end date');
                } else {
                    $enrollvar{$type} = &Apache::lonlocal::locallocaltime($settings{$item});
                }
            } elsif (($type eq 'sectionnums') || ($type eq 'co-owners')) {
                $enrollvar{$type} = $settings{$item};
                $enrollvar{$type} =~ s/,/, /g;
            } elsif ($type eq "authtype"
                     || $type eq "autharg"    || $type eq "coursecode"
                     || $type eq "crosslistings") {
                $enrollvar{$type} = $settings{$item};
            } elsif ($type eq 'courseowner') {
                if ($settings{$item} =~ /^[^:]+:[^:]+$/) {
                    $enrollvar{$type} = $settings{$item};
                } else {
                    if ($settings{$item} ne '') {
                        $enrollvar{$type} = $settings{$item}.':'.$cdom;
                    }
                }
            }
        } elsif ($item =~ m/^default_enrollment_(start|end)_date$/) {
            my $type = $1;
            if ( ($type eq 'end') && ($settings{$item} == 0) ) {
                $enrollvar{$item} = &mt('No end date');
            } elsif ( ($type eq 'start') && ($settings{$item} eq '') ) {
                $enrollvar{$item} = 'When enrolled';
            } else {
                $enrollvar{$item} = &Apache::lonlocal::locallocaltime($settings{$item});
            }
        }
    }
    return %enrollvar;
}

sub print_course_search_page {
    my ($r,$dom,$domdesc) = @_;
    my $action = '/adm/modifycourse';
    my $type = $env{'form.type'};
    if (!defined($env{'form.type'})) {
        $type = 'Course';
    }
    &print_header($r,$type);
    my $filterlist = ['descriptfilter',
                      'instcodefilter','ownerfilter',
                      'coursefilter'];
    my $filter = {};
    my ($numtitles,$cctitle,$dctitle);
    my $ccrole = 'cc';
    if ($type eq 'Community') {
        $ccrole = 'co';
    }
    $cctitle = &Apache::lonnet::plaintext($ccrole,$type);
    $dctitle = &Apache::lonnet::plaintext('dc');
    $r->print(&Apache::lonpickcourse::js_changer());
    if ($type eq 'Community') {
        $r->print('<h3>'.&mt('Search for a community in the [_1] domain',$domdesc).'</h3>');
    } else {
        $r->print('<h3>'.&mt('Search for a course in the [_1] domain',$domdesc).'</h3>');
    }   
    $r->print(&Apache::lonpickcourse::build_filters($filterlist,$type,
                             undef,undef,$filter,$action,\$numtitles,'modifycourse'));
    if ($type eq 'Community') {
        $r->print(&mt('Actions available after searching for a community:').'<ul>'.
                  '<li>'.&mt('Enter the community with the role of [_1]',$cctitle).'</li>'."\n".
                  '<li>'.&mt('View or modify community settings which only a [_1] may modify.',$dctitle).
                  '</li>'."\n".'</ul>');
    } else {
        $r->print(&mt('Actions available after searching for a course:').'<ul>'.
                  '<li>'.&mt('Enter the course with the role of [_1]',$cctitle).'</li>'."\n".
                  '<li>'.&mt('View or modify course settings which only a [_1] may modify.',$dctitle).
                  '</li>'."\n".'</ul>');
    }
}

sub print_course_selection_page {
    my ($r,$dom,$domdesc) = @_;
    my $type = $env{'form.type'};
    if (!defined($type)) {
        $type = 'Course';
    }
    &print_header($r,$type);

# Criteria for course search 
    my $filterlist = ['descriptfilter',
                      'instcodefilter','ownerfilter',
                      'ownerdomfilter','coursefilter'];
    my %filter;
    my $action = '/adm/modifycourse';
    my $dctitle = &Apache::lonnet::plaintext('dc');
    my $numtitles;
    $r->print(&Apache::lonpickcourse::js_changer());
    $r->print(&mt('Revise your search criteria for this domain').' ('.$domdesc.').<br />');
    $r->print(&Apache::lonpickcourse::build_filters($filterlist,$type,
                                       undef,undef,\%filter,$action,\$numtitles));
    $filter{'domainfilter'} = $dom;
    my %courses = &Apache::lonpickcourse::search_courses($r,$type,0,
                                                         \%filter,$numtitles);
    &Apache::lonpickcourse::display_matched_courses($r,$type,0,$action,undef,undef,undef,
                                                    %courses);
    return;
}

sub print_modification_menu {
    my ($r,$cdesc,$domdesc,$dom,$type) = @_;
    &print_header($r,$type);
    my ($ccrole,$setquota_text,$setparams_text,$cat_text);
    if ($type eq 'Community') {
        $ccrole = 'co';
    } else {
        $ccrole = 'cc';
    } 
    my $action = '/adm/modifycourse';
    if ($type eq 'Community') {
        $setquota_text = &mt('Total disk space allocated for storage of portfolio files in all groups in a community.');
        $setparams_text = 'View/Modify community owner';
        $cat_text = 'View/Modify catalog settings for community';
    } else {
        $setquota_text = &mt('Total disk space allocated for storage of portfolio files in all groups in a course.');
        $setparams_text = 'View/Modify course owner, institutional code, and default authentication';
        $cat_text = 'View/Modify catalog settings for course'; 
    }
    my @menu =
        (
          { text => $setparams_text,
             phase => 'setparms',
          },
          { text  => 'View/Modify quota for group portfolio files',
            phase => 'setquota',
          }
    );
    my %domconf = &Apache::lonnet::get_dom('configuration',['coursecategories'],$dom);
    my @additional_params = &catalog_settable($domconf{'coursecategories'},$type);
    if (@additional_params > 0) {
        push (@menu, { text => $cat_text,
                       phase => 'catsettings',
                     });
    }
    unless ($type eq 'Community') {
        push(@menu,
           { text  => 'Display current settings for automated enrollment',
            phase => 'viewparms',
           }
        );
    }
    my $menu_html = '<h3>'.&mt('View/Modify settings for: ').
                    ' <span class="LC_nobreak">'.$cdesc.'</span></h3>'."\n";
    if ($type eq 'Community') {
        $menu_html .= &mt('Although almost all community settings can be modified by a Coordinator, the following may only be set or modified by a Domain Coordinator:');
    } else {
        $menu_html .= &mt('Although almost all course settings can be modified by a Course Coordinator, the following may only be set or modified by a Domain Coordinator:');
    }
    $menu_html .= '<ul>';
    if ($type eq 'Community') {
        $menu_html .= '<li>'.&mt('Community owner (permitted to assign Coordinator roles in the community).').'</li>';
    } else {
        $menu_html .=  '<li>'.&mt('Course owner (permitted to assign Course Coordinator roles in the course).').'</li>'.
                       '<li>'.&mt("Institutional code and default authentication (both required for auto-enrollment of students from institutional datafeeds).").'</li>';
    }
    $menu_html .= '<li>'.$setquota_text.'</li>'."\n";
    foreach my $item (@additional_params) {
        if ($type eq 'Community') {
            if ($item eq 'togglecats') {
                $menu_html .= '  <li>'.&mt('Hiding/unhiding a community from the catalog (although can be [_1]configured[_2] to be modifiable by a Coordinator in community context).','<a href="/adm/domainprefs?actions=coursecategories&phase=display">','</a>').'</li>'."\n";
            } elsif ($item eq 'categorize') {
                $menu_html .= '  <li>'.&mt('Manual cataloging of a community (although can be [_1]configured[_2] to be modifiable by a Coordinator in community context).','<a href="/adm/domainprefs?actions=coursecategories&phase=display">','</a>').'</li>'."\n";
            }
        } else {
            if ($item eq 'togglecats') {
                $menu_html .= '  <li>'.&mt('Hiding/unhiding a course from the course catalog (although can be [_1]configured[_2] to be modifiable by a Course Coordinator in course context).','<a href="/adm/domainprefs?actions=coursecategories&phase=display">','</a>').'</li>'."\n";
            } elsif ($item eq 'categorize') {
                $menu_html .= '  <li>'.&mt('Manual cataloging of a course (although can be [_1]configured[_2] to be modifiable by a Course Coordinator in course context).','<a href="/adm/domainprefs?actions=coursecategories&phase=display">','</a>').'</li>'."\n";
            }
        }
    }
    $menu_html .= ' </ul>
<form name="menu" method="post" action="'.$action.'" />'."\n".
    &hidden_form_elements();
    foreach my $menu_item (@menu) {
        $menu_html.='<h3>';
        $menu_html.=
                qq|<a href="javascript:changePage(document.menu,'$menu_item->{'phase'}')">|;
        $menu_html.= &mt($menu_item->{'text'}).'</a>';
        $menu_html.='</h3>';
    }
    
    $r->print($menu_html);
    return;
}

sub print_ccrole_selected {
    my ($r,$type) = @_;
    &print_header($r,$type);
    my ($cdom,$cnum) = split(/_/,$env{'form.pickedcourse'});
    $r->print('<form name="ccrole" method="post" action="/adm/roles">
<input type="hidden" name="selectrole" value="1" />
<input type="hidden" name="newrole" value="cc./'.$cdom.'/'.$cnum.'" />
</form>');
}

sub print_settings_display {
    my ($r,$cdom,$cnum,$cdesc,$type) = @_;
    my %enrollvar = &get_enrollment_settings($cdom,$cnum);
    my %longtype = &course_settings_descrip($type);
    my %lt = &Apache::lonlocal::texthash(
            'valu' => 'Current value',
            'cour' => 'Current settings are:',
            'cose' => "Settings which control auto-enrollment using classlists from your institution's student information system fall into two groups:",
            'dcon' => 'Modifiable only by Domain Coordinator',
            'back' => 'Pick another action',
    );
    my $ccrole = 'cc';
    if ($type eq 'Community') {
       $ccrole = 'co';
    }
    my $cctitle = &Apache::lonnet::plaintext($ccrole,$type);
    my $dctitle = &Apache::lonnet::plaintext('dc');
    my @modifiable_params = &get_dc_settable($type);
    my ($internals,$accessdates) = &autoenroll_keys();
    my @items;
    if ((ref($internals) eq 'ARRAY') && (ref($accessdates) eq 'ARRAY')) {
        @items =  (@{$internals},@{$accessdates});
    }
    my $disp_table = &Apache::loncommon::start_data_table()."\n".
                     &Apache::loncommon::start_data_table_header_row()."\n".
                     "<th>&nbsp;</th>\n".
                     "<th>$lt{'valu'}</th>\n".
                     "<th>$lt{'dcon'}</th>\n".
                     &Apache::loncommon::end_data_table_header_row()."\n";
    foreach my $item (@items) {
        $disp_table .= &Apache::loncommon::start_data_table_row()."\n".
                       "<td><b>$longtype{$item}</b></td>\n".
                       "<td>$enrollvar{$item}</td>\n";
        if (grep(/^\Q$item\E$/,@modifiable_params)) {
            $disp_table .= '<td align="right">'.&mt('Yes').'</td>'."\n";
        } else {
            $disp_table .= '<td align="right">'.&mt('No').'</td>'."\n";
        }
        $disp_table .= &Apache::loncommon::end_data_table_row()."\n";
    }
    $disp_table .= &Apache::loncommon::end_data_table()."\n";
    &print_header($r,$type);
    my $newrole = $ccrole.'./'.$cdom.'/'.$cnum;
    my $escuri = &HTML::Entities::encode('/adm/roles?selectrole=1&'.$newrole.
                                         '=1&destinationurl=/adm/populate','&<>"'); 
    $r->print('<h3>'.&mt('Current automated enrollment settings for:').
              ' <span class="LC_nobreak">'.$cdesc.'</span></h3>'.
              '<form action="/adm/modifycourse" method="post" name="viewparms">'."\n".
              '<p>'.$lt{'cose'}.'<ul>'.
              '<li>'.&mt('Settings modifiable by a [_1] via the [_2]Automated Enrollment Manager[_3] in a course.',$cctitle,'<a href="'.$escuri.'">','</a>').'</li>'.
              '<li>'.&mt('Settings modifiable by a [_1] via [_2]View/Modify course owner, institutional code, and default authentication[_3].',$dctitle,'<a href="javascript:changePage(document.viewparms,'."'setparms'".');">','</a>')."\n".
              '</li></ul></p>'.
              '<p>'.$lt{'cour'}.'</p><p>'.$disp_table.'</p><p>'.
              '<a href="javascript:changePage(document.viewparms,'."'menu'".')">'.$lt{'back'}.'</a>'."\n".
              &hidden_form_elements().
              '</p></form>'
     );
}

sub print_setquota {
    my ($r,$cdom,$cnum,$cdesc,$type) = @_;
    my %lt = &Apache::lonlocal::texthash(
                'cquo' => 'Disk space for storage of group portfolio files for:',
                'gpqu' => 'Course portfolio files disk space',
                'modi' => 'Save',
                'back' => 'Pick another action',
    );
    if ($type eq 'Community') {
        $lt{'gpqu'} = &mt('Community portfolio files disk space');
    }
    my %settings = &Apache::lonnet::get('environment',['internal.coursequota'],$cdom,$cnum);
    my $coursequota = $settings{'internal.coursequota'};
    if ($coursequota eq '') {
        $coursequota = 20;
    }
    &print_header($r,$type);
    my $hidden_elements = &hidden_form_elements();
    my $helpitem = &Apache::loncommon::help_open_topic('Modify_Course_Quota');
    $r->print(<<ENDDOCUMENT);
<form action="/adm/modifycourse" method="post" name="setquota">
<h3>$lt{'cquo'} <span class="LC_nobreak">$cdesc</span></h3>
<p>
$helpitem $lt{'gpqu'}: <input type="text" size="4" name="coursequota" value="$coursequota" /> Mb &nbsp;&nbsp;&nbsp;&nbsp;
<input type="button" onClick="javascript:verify_quota(this.form)" value="$lt{'modi'}" />
</p>
$hidden_elements
<a href="javascript:changePage(document.setquota,'menu')">$lt{'back'}</a>
</form>
ENDDOCUMENT
    return;
}

sub print_catsettings {
    my ($r,$cdom,$cnum,$cdesc,$type) = @_;
    &print_header($r,$type);
    my %lt = &Apache::lonlocal::texthash(
                                         'back'    => 'Pick another action',
                                         'catset'  => 'Catalog Settings for Course',
                                         'visi'    => 'Visibility in Course/Community Catalog',
                                         'exclude' => 'Exclude from course catalog:',
                                         'categ'   => 'Categorize Course',
                                         'assi'    => 'Assign one or more categories and/or subcategories to this course.'
                                        );
    if ($type eq 'Community') {
        $lt{'catset'} = &mt('Catalog Settings for Community');
        $lt{'exclude'} = &mt('Exclude from course catalog');
        $lt{'categ'} = &mt('Categorize Community');
        $lt{'assi'} = &mt('Assign one or more subcategories to this community.');
    }
    $r->print('<form action="/adm/modifycourse" method="post" name="catsettings">'.
              '<h3>'.$lt{'catset'}.' <span class="LC_nobreak">'.$cdesc.'</span></h3>');
    my %domconf = &Apache::lonnet::get_dom('configuration',['coursecategories'],$cdom);
    my @cat_params = &catalog_settable($domconf{'coursecategories'},$type);
    if (@cat_params > 0) {
        my %currsettings = 
            &Apache::lonnet::get('environment',['hidefromcat','categories'],$cdom,$cnum);
        if (grep(/^togglecats$/,@cat_params)) {
            my $excludeon = '';
            my $excludeoff = ' checked="checked" ';
            if ($currsettings{'hidefromcat'} eq 'yes') {
                $excludeon = $excludeoff;
                $excludeoff = ''; 
            }
            $r->print('<br /><h4>'.$lt{'visi'}.'</h4>'.
                      $lt{'exclude'}.
                      '&nbsp;<label><input name="hidefromcat" type="radio" value="yes" '.$excludeon.' />'.&mt('Yes').'</label>&nbsp;&nbsp;&nbsp;<label><input name="hidefromcat" type="radio" value="" '.$excludeoff.' />'.&mt('No').'</label><br /><p>');
            if ($type eq 'Community') {
                $r->print(&mt("If a community has been categorized using at least one of the categories defined for communities in the domain, it will be listed in the domain's publicly accessible Course/Community Catalog, unless excluded."));
            } else {
                $r->print(&mt("Unless excluded, a course will be listed in the domain's publicly accessible Course/Community Catalog, if at least one of the following applies").':<ul>'.
                          '<li>'.&mt('Auto-cataloging is enabled and the course is assigned an institutional code.').'</li>'.
                          '<li>'.&mt('The course has been categorized using at least one of the course categories defined for the domain.').'</li></ul>');
            }
            $r->print('</ul></p>');
        }
        if (grep(/^categorize$/,@cat_params)) {
            $r->print('<br /><h4>'.$lt{'categ'}.'</h4>');
            if (ref($domconf{'coursecategories'}) eq 'HASH') {
                my $cathash = $domconf{'coursecategories'}{'cats'};
                if (ref($cathash) eq 'HASH') {
                    $r->print($lt{'assi'}.'<br /><br />'.
                              &Apache::loncommon::assign_categories_table($cathash,
                                                     $currsettings{'categories'},$type));
                } else {
                    $r->print(&mt('No categories defined for this domain'));
                }
            } else {
                $r->print(&mt('No categories defined for this domain'));
            }
            unless ($type eq 'Community') { 
                $r->print('<p>'.&mt('If auto-cataloging based on institutional code is enabled in the domain, a course will continue to be listed in the catalog of official courses, in addition to receiving a listing under any manually assigned categor(ies).').'</p>');
            }
        }
        $r->print('<p><input type="button" name="chgcatsettings" value="'.
                  &mt('Save').'" onclick="javascript:changePage(document.catsettings,'."'processcat'".');" /></p>');
    } else {
        $r->print('<span class="LC_warning">');
        if ($type eq 'Community') {
            $r->print(&mt('Catalog settings in this domain are set in community context via "Community Configuration".'));
        } else {
            $r->print(&mt('Catalog settings in this domain are set in course context via "Course Configuration".'));
        }
        $r->print('</span><br /><br />'."\n".
                  '<a href="javascript:changePage(document.catsettings,'."'menu'".');">'.
                  $lt{'back'}.'</a>');
    }
    $r->print(&hidden_form_elements().'</form>'."\n");
    return;
}

sub print_course_modification_page {
    my ($r,$cdom,$cnum,$cdesc,$type) = @_;
    my %lt=&Apache::lonlocal::texthash(
            'actv' => "Active",
            'inac' => "Inactive",
            'ownr' => "Owner",
            'name' => "Name",
            'unme' => "Username:Domain",
            'stus' => "Status",
            'nocc' => 'There is currently no owner set for this course.',
            'gobt' => "Save",
    );
    my ($ownertable,$ccrole,$javascript_validations,$authenitems,$ccname);
    my %enrollvar = &get_enrollment_settings($cdom,$cnum);
    if ($type eq 'Community') {
        $ccrole = 'co';
        $lt{'nocc'} = &mt('There is currently no owner set for this community.');
    } else {
        $ccrole ='cc';
        ($javascript_validations,$authenitems) = &gather_authenitems($cdom,\%enrollvar);
    }
    $ccname = &Apache::lonnet::plaintext($ccrole,$type);
    my %roleshash = &Apache::lonnet::get_my_roles($cnum,$cdom,'','',[$ccrole]);
    my (@local_ccs,%cc_status,%pname);
    foreach my $item (keys(%roleshash)) {
        my ($uname,$udom) = split(/:/,$item);
        if (!grep(/^\Q$uname\E:\Q$udom\E$/,@local_ccs)) {
            push(@local_ccs,$uname.':'.$udom);
            $pname{$uname.':'.$udom} = &Apache::loncommon::plainname($uname,$udom);
            $cc_status{$uname.':'.$udom} = $lt{'actv'};
        }
    }
    if (($enrollvar{'courseowner'} ne '') && 
        (!grep(/^$enrollvar{'courseowner'}$/,@local_ccs))) {
        push(@local_ccs,$enrollvar{'courseowner'});
        my ($owneruname,$ownerdom) = split(/:/,$enrollvar{'courseowner'});
        $pname{$enrollvar{'courseowner'}} = 
                         &Apache::loncommon::plainname($owneruname,$ownerdom);
        my $active_cc = &Apache::loncommon::check_user_status($ownerdom,$owneruname,
                                                              $cdom,$cnum,$ccrole);
        if ($active_cc eq 'active') {
            $cc_status{$enrollvar{'courseowner'}} = $lt{'actv'};
        } else {
            $cc_status{$enrollvar{'courseowner'}} = $lt{'inac'};
        }
    }
    @local_ccs = sort(@local_ccs);
    if (@local_ccs == 0) {
        $ownertable = $lt{'nocc'};
    } else {
        my $numlocalcc = scalar(@local_ccs);
        $ownertable = '<input type="hidden" name="numlocalcc" value="'.$numlocalcc.'" />'.
                      &Apache::loncommon::start_data_table()."\n".
                      &Apache::loncommon::start_data_table_header_row()."\n".
                      '<th>'.$lt{'ownr'}.'</th>'.
                      '<th>'.$lt{'name'}.'</th>'.
                      '<th>'.$lt{'unme'}.'</th>'.
                      '<th>'.$lt{'stus'}.'</th>'.
                      &Apache::loncommon::end_data_table_header_row()."\n";
        foreach my $cc (@local_ccs) {
            $ownertable .= &Apache::loncommon::start_data_table_row()."\n";
            if ($cc eq $enrollvar{'courseowner'}) {
                  $ownertable .= '<td><input type="radio" name="courseowner" value="'.$cc.'" checked="checked" /></td>'."\n";
            } else {
                $ownertable .= '<td><input type="radio" name="courseowner" value="'.$cc.'" /></td>'."\n";
            }
            $ownertable .= 
                 '<td>'.$pname{$cc}.'</td>'."\n".
                 '<td>'.$cc.'</td>'."\n".
                 '<td>'.$cc_status{$cc}.' '.$ccname.'</td>'."\n".
                 &Apache::loncommon::end_data_table_row()."\n";
        }
        $ownertable .= &Apache::loncommon::end_data_table();
    }
    &print_header($r,$type,$javascript_validations);
    my $dctitle = &Apache::lonnet::plaintext('dc');
    my $mainheader = &modifiable_only_title($type);
    my $hidden_elements = &hidden_form_elements();
    $r->print('<form action="/adm/modifycourse" method="post" name="'.$env{'form.phase'}.'">'."\n".
              '<h3>'.$mainheader.' <span class="LC_nobreak">'.$cdesc.'</span></h3><p>'.
              &Apache::lonhtmlcommon::start_pick_box());
    if ($type eq 'Community') {
        $r->print(&Apache::lonhtmlcommon::row_title(
                  &Apache::loncommon::help_open_topic('Modify_Community_Owner').
                  '&nbsp;'.&mt('Community Owner'))."\n");
    } else {
        $r->print(&Apache::lonhtmlcommon::row_title(
                      &Apache::loncommon::help_open_topic('Modify_Course_Instcode').
                      '&nbsp;'.&mt('Course Code'))."\n".
                  '<input type="text" size="10" name="coursecode" value="'.$enrollvar{'coursecode'}.'" />'.
                  &Apache::lonhtmlcommon::row_closure().
                  &Apache::lonhtmlcommon::row_title(
                     &Apache::loncommon::help_open_topic('Modify_Course_Defaultauth').
                     '&nbsp;'.&mt('Default Authentication method'))."\n".
                  $authenitems."\n".
                  &Apache::lonhtmlcommon::row_closure().
                  &Apache::lonhtmlcommon::row_title(
                      &Apache::loncommon::help_open_topic('Modify_Course_Owner').
                      '&nbsp;'.&mt('Course Owner'))."\n");
    }
    $r->print($ownertable."\n".&Apache::lonhtmlcommon::row_closure(1).
              &Apache::lonhtmlcommon::end_pick_box().'</p><p>'.$hidden_elements.
              '<input type="button" onclick="javascript:changePage(this.form,'."'processparms'".');');
    if ($type eq 'Community') {
        $r->print('this.form.submit();"');
    } else {
        $r->print('javascript:verify_message(this.form);"');
    }
    $r->print('value="'.$lt{'gobt'}.'" /></p></form>');
    return;
}

sub modifiable_only_title {
    my ($type) = @_;
    my $dctitle = &Apache::lonnet::plaintext('dc');
    if ($type eq 'Community') {
        return &mt('Community settings modifiable only by [_1] for:',$dctitle);
    } else {
        return &mt('Course settings modifiable only by [_1] for:',$dctitle);
    }
}

sub gather_authenitems {
    my ($cdom,$enrollvar) = @_;
    my ($krbdef,$krbdefdom)=&Apache::loncommon::get_kerberos_defaults($cdom);
    my $curr_authtype = '';
    my $curr_authfield = '';
    if (ref($enrollvar) eq 'HASH') {
        if ($enrollvar->{'authtype'} =~ /^krb/) {
            $curr_authtype = 'krb';
        } elsif ($enrollvar->{'authtype'} eq 'internal' ) {
            $curr_authtype = 'int';
        } elsif ($enrollvar->{'authtype'} eq 'localauth' ) {
            $curr_authtype = 'loc';
        }
    }
    unless ($curr_authtype eq '') {
        $curr_authfield = $curr_authtype.'arg';
    }
    my $javascript_validations = 
        &Apache::lonuserutils::javascript_validations('modifycourse',$krbdefdom,
                                                      $curr_authtype,$curr_authfield);
    my %param = ( formname => 'document.'.$env{'form.phase'},
           kerb_def_dom => $krbdefdom,
           kerb_def_auth => $krbdef,
           mode => 'modifycourse',
           curr_authtype => $curr_authtype,
           curr_autharg => $enrollvar->{'autharg'}
        );
    my (%authform,$authenitems);
    $authform{'krb'} = &Apache::loncommon::authform_kerberos(%param);
    $authform{'int'} = &Apache::loncommon::authform_internal(%param);
    $authform{'loc'} = &Apache::loncommon::authform_local(%param);
    foreach my $item ('krb','int','loc') {
        if ($authform{$item} ne '') {
            $authenitems .= $authform{$item}.'<br />';
        }
    }
    return($javascript_validations,$authenitems);
}

sub modify_course {
    my ($r,$cdom,$cnum,$cdesc,$domdesc,$type) = @_;
    my %longtype = &course_settings_descrip($type);
    my @items = ('internal.courseowner','description','internal.co-owners',
                 'internal.pendingco-owners');
    unless ($type eq 'Community') {
        push(@items,('internal.coursecode','internal.authtype','internal.autharg',
                     'internal.sectionnums','internal.crosslistings'));
    }
    my %settings = &Apache::lonnet::get('environment',\@items,$cdom,$cnum);
    my $description = $settings{'description'};
    my ($ccrole,$response,$chgresponse,$nochgresponse,$reply,%currattr,%newattr,%cenv,%changed,
        @changes,@nochanges,@sections,@xlists,@warnings);
    my @modifiable_params = &get_dc_settable($type);
    foreach my $param (@modifiable_params) {
        $currattr{$param} = $settings{'internal.'.$param};
    }
    if ($type eq 'Community') {
        %changed = ( owner  => 0 );
        $ccrole = 'co';
    } else {
        %changed = ( code  => 0,
                     owner => 0,
                   );
        $ccrole = 'cc';
        unless ($settings{'internal.sectionnums'} eq '') {
            if ($settings{'internal.sectionnums'} =~ m/,/) {
                @sections = split/,/,$settings{'internal.sectionnums'};
            } else {
                $sections[0] = $settings{'internal.sectionnums'};
            }
        }
        unless ($settings{'internal.crosslistings'} eq'') {
            if ($settings{'internal.crosslistings'} =~ m/,/) {
                @xlists = split/,/,$settings{'internal.crosslistings'};
            } else {
                $xlists[0] = $settings{'internal.crosslistings'};
            }
        }
        if ($env{'form.login'} eq 'krb') {
            $newattr{'authtype'} = $env{'form.login'};
            $newattr{'authtype'} .= $env{'form.krbver'};
            $newattr{'autharg'} = $env{'form.krbarg'};
        } elsif ($env{'form.login'} eq 'int') {
            $newattr{'authtype'} ='internal';
            if ((defined($env{'form.intarg'})) && ($env{'form.intarg'})) {
                $newattr{'autharg'} = $env{'form.intarg'};
            }
        } elsif ($env{'form.login'} eq 'loc') {
            $newattr{'authtype'} = 'localauth';
            if ((defined($env{'form.locarg'})) && ($env{'form.locarg'})) {
                $newattr{'autharg'} = $env{'form.locarg'};
            }
        }
        if ( $newattr{'authtype'}=~ /^krb/) {
            if ($newattr{'autharg'}  eq '') {
                push(@warnings,
                           &mt('As you did not include the default Kerberos domain'
                          .' to be used for authentication in this class, the'
                          .' institutional data used by the automated'
                          .' enrollment process must include the Kerberos'
                          .' domain for each new student.'));
            }
        }

        if ( exists($env{'form.coursecode'}) ) {
            $newattr{'coursecode'}=$env{'form.coursecode'};
            unless ( $newattr{'coursecode'} eq $currattr{'coursecode'} ) {
                $changed{'code'} = 1;
            }
        }
    }

    if ( exists($env{'form.courseowner'}) ) {
        $newattr{'courseowner'}=$env{'form.courseowner'};
        unless ( $newattr{'courseowner'} eq $currattr{'courseowner'} ) {
            $changed{'owner'} = 1;
        } 
    }

    if ($changed{'owner'} || $changed{'code'}) {
        my %crsinfo = &Apache::lonnet::courseiddump($cdom,'.',1,'.','.',$cnum,
                                                    undef,undef,'.');
        if (ref($crsinfo{$env{'form.pickedcourse'}}) eq 'HASH') {
            if ($changed{'code'}) {
                $crsinfo{$env{'form.pickedcourse'}}{'inst_code'} = $env{'form.coursecode'};
            }
            if ($changed{'owner'}) {
                $crsinfo{$env{'form.pickedcourse'}}{'owner'} = $env{'form.courseowner'};
            }
            my $chome = &Apache::lonnet::homeserver($cnum,$cdom);
            my $putres = &Apache::lonnet::courseidput($cdom,\%crsinfo,$chome,'notime');
            if ($putres eq 'ok') {
                &update_coowners($cdom,$cnum,$chome,\%settings,\%newattr);
            }
        }
    }
    foreach my $param (@modifiable_params) {
        if ($currattr{$param} eq $newattr{$param}) {
            push(@nochanges,$param);
        } else {
            $cenv{'internal.'.$param} = $newattr{$param};
            push(@changes,$param);
        }
    }
    if (@changes > 0) {
        $chgresponse = &mt("The following settings have been changed:<br/><ul>");
    }
    if (@nochanges > 0) {
        $nochgresponse = &mt("The following settings remain unchanged:<br/><ul>");
    }
    if (@changes > 0) {
        my $putreply = &Apache::lonnet::put('environment',\%cenv,$cdom,$cnum);
        if ($putreply !~ /^ok$/) {
            $response = '<p class="LC_error">'.
                        &mt('There was a problem processing your requested changes.').'<br />';
            if ($type eq 'Community') {
                $response .= &mt('Settings for this community have been left unchanged.');
            } else {
                $response .= &mt('Settings for this course have been left unchanged.');
            }
            $response .= '<br/>'.&mt('Error: ').$putreply.'</p>';
        } else {
            foreach my $attr (@modifiable_params) {
                if (grep/^\Q$attr\E$/,@changes) {
	            $chgresponse .= '<li>'.$longtype{$attr}.' '.&mt('now set to:').' "'.$newattr{$attr}.'".</li>';
                } else {
	            $nochgresponse .= '<li>'.$longtype{$attr}.' '.&mt('still set to:').' "'.$currattr{$attr}.'".</li>';
                }
            }
            if (($type ne 'Community') && ($changed{'code'} || $changed{'owner'})) {
                if ( $newattr{'courseowner'} eq '') {
	            push(@warnings,&mt('There is no owner associated with this LON-CAPA course.').
                                   '<br />'.&mt('If automated enrollment at your institution requires validation of course owners, automated enrollment will fail.'));
                } else {
	            if (@sections > 0) {
                        if ($changed{'code'}) {
	                    foreach my $sec (@sections) {
		                if ($sec =~ m/^(.+):/) {
                                    my $instsec = $1;
		                    my $inst_course_id = $newattr{'coursecode'}.$1;
                                    my $course_check = &Apache::lonnet::auto_validate_courseID($cnum,$cdom,$inst_course_id);
			            if ($course_check eq 'ok') {
                                        my $outcome = &Apache::lonnet::auto_new_course($cnum,$cdom,$inst_course_id,$newattr{'courseowner'});
			                unless ($outcome eq 'ok') {
                               
				            push(@warnings,&mt('If automatic enrollment is enabled for "[_1]" automated enrollment may fail for "[_2]" - section: [_3], for the following reason: "[_4]"',$description,$newattr{'coursecode'},$instsec,$outcome).'<br/>');
			                }
			            } else {
                                        push(@warnings,&mt('If automatic enrollment is enabled for "[_1]" automated enrollment may fail for "[_2]" - section: [_3], for the following reason: "[_4]"',$description,$newattr{'coursecode'},$instsec,$course_check));
			            }
		                } else {
			            push(@warnings,&mt('If automatic enrollment is enabled for "[_1]", automated enrollment may fail for "[_2]" - section: [_3], because this is not a valid section entry.',$description,$newattr{'coursecode'},$sec));
		                }
		            }
	                } elsif ($changed{'owner'}) {
                            foreach my $sec (@sections) {
                                if ($sec =~ m/^(.+):/) {
                                    my $instsec = $1;
                                    my $inst_course_id = $newattr{'coursecode'}.$instsec;
                                    my $outcome = &Apache::lonnet::auto_new_course($cnum,$cdom,$inst_course_id,$newattr{'courseowner'});
                                    unless ($outcome eq 'ok') {
                                        push(@warnings,&mt('If automatic enrollment is enabled for "[_1]", automated enrollment may fail for "[_2]" - section [_3] for the following reason: "[_4]".',$description,$newattr{'coursecode'},$instsec,$outcome));
                                    }
                                } else {
                                    push(@warnings,&mt('If automatic enrollment is enabled for "[_1]", automated enrollment may fail for "[_2]" - section [_3] because this is not a valid section entry.',$description,$newattr{'coursecode'},$sec));
                                }
                            }
                        }
	            } else {
	                push(@warnings,&mt('As no section numbers are currently listed for "[_1]" automated enrollment will not occur for any sections of institutional course code: "[_2]".',$description,$newattr{'coursecode'}));
	            }
	            if ( (@xlists > 0) && ($changed{'owner'}) ) {
	                foreach my $xlist (@xlists) {
		            if ($xlist =~ m/^(.+):/) {
                                my $instxlist = $1;
                                my $outcome = &Apache::lonnet::auto_new_course($cnum,$cdom,$instxlist,$newattr{'courseowner'});
		                unless ($outcome eq 'ok') {
			            push(@warnings,&mt('If automatic enrollment is enabled for "[_1]", automated enrollment may fail for crosslisted class "[_2]" for the following reason: "[_3]".',$description,$instxlist,$outcome));
		                }
		            }
	                }
	            }
                }
            }
        }
    } else {
        foreach my $attr (@modifiable_params) {
            $nochgresponse .= '<li>'.$longtype{$attr}.' '.&mt('still set to').' "'.$currattr{$attr}.'".</li>';
        }
    }

    if (@changes > 0) {
        $chgresponse .= "</ul><br/><br/>";
    }
    if (@nochanges > 0) {
        $nochgresponse .=  "</ul><br/><br/>";
    }
    my ($warning,$numwarnings);
    my $numwarnings = scalar(@warnings); 
    if ($numwarnings) {
        $warning = &mt('The following [quant,_1,warning was,warnings were] generated when applying your changes to automated enrollment:',$numwarnings).'<p><ul>';
        foreach my $warn (@warnings) {
            $warning .= '<li><span class="LC_warning">'.$warn.'</span></li>';
        }
        $warning .= '</ul></p>';
    }
    if ($response) {
        $reply = $response;
    } else {
        $reply = $chgresponse.$nochgresponse.$warning;
    }
    &print_header($r,$type);
    my $mainheader = &modifiable_only_title($type);
    $reply = '<h3>'.$mainheader.' <span class="LC_nobreak">'.$cdesc.'</span></h3>'."\n".
             '<p>'.$reply.'</p>'."\n".
             '<form action="/adm/modifycourse" method="post" name="processparms">'.
             &hidden_form_elements().
             '<a href="javascript:changePage(document.processparms,'."'menu'".')">'.
             &mt('Pick another action').'</a>';
    if ($numwarnings) {
        my $newrole = $ccrole.'./'.$cdom.'/'.$cnum;
        my $escuri = &HTML::Entities::encode('/adm/roles?selectrole=1&'.$newrole.
                                             '=1&destinationurl=/adm/populate','&<>"');

        $reply .= '<br /><a href="'.$escuri.'">'.
                  &mt('Go to Automated Enrollment Manager for course').'</a>';
    }
    $reply .= '</form>';
    $r->print($reply);
    return;
}

sub update_coowners {
    my ($cdom,$cnum,$chome,$settings,$newattr) = @_;
    return unless ((ref($settings) eq 'HASH') && (ref($newattr) eq 'HASH'));
    my %designhash = &Apache::loncommon::get_domainconf($cdom);
    my (%cchash,$autocoowners);
    if ($designhash{$cdom.'.autoassign.co-owners'}) {
        $autocoowners = 1;
        %cchash = &Apache::lonnet::get_my_roles($cnum,$cdom,undef,undef,['cc']);
    }
    if ($settings->{'internal.courseowner'} ne $newattr->{'courseowner'}) {
        my $oldowner_to_coowner;
        my @types = ('co-owners');
        if (($newattr->{'coursecode'}) && ($autocoowners)) {
            my $oldowner = $settings->{'internal.courseowner'};
            if ($cchash{$oldowner.':cc'}) {
                my ($result,$desc) = &Apache::lonnet::auto_validate_instcode($cnum,$cdom,$newattr->{'coursecode'},$oldowner);
                if ($result eq 'valid') {
                    if ($settings->{'internal.co-owner'}) {
                        my @current = split(',',$settings->{'internal.co-owners'});
                        unless (grep(/^\Q$oldowner\E$/,@current)) {
                            $oldowner_to_coowner = 1;
                        }
                    } else {
                        $oldowner_to_coowner = 1;
                    }
                }
            }
        } else {
            push(@types,'pendingco-owners');
        }
        foreach my $type (@types) {
            if ($settings->{'internal.'.$type}) {
                my @current = split(',',$settings->{'internal.'.$type});
                my $newowner = $newattr->{'courseowner'};
                my @newvalues = ();
                if (($newowner ne '') && (grep(/^\Q$newowner\E$/,@current))) {
                    foreach my $person (@current) {
                        unless ($person eq $newowner) {
                            push(@newvalues,$person);
                        }
                    }
                } else {
                    @newvalues = @current;
                }
                if ($oldowner_to_coowner) {
                    push(@newvalues,$settings->{'internal.courseowner'});
                    @newvalues = sort(@newvalues);
                }
                my $newownstr = join(',',@newvalues);
                if ($newownstr ne $settings->{'internal.'.$type}) {
                    if ($type eq 'co-owners') {
                        my $deleted = '';
                        unless (@newvalues) {
                            $deleted = 1;
                        }
                        &Apache::lonnet::store_coowners($cdom,$cnum,$chome,
                                                        $deleted,@newvalues);
                    } else {
                        my $pendingcoowners;
                        my $cid = $cdom.'_'.$cnum;
                        if (@newvalues) {
                            $pendingcoowners = join(',',@newvalues);
                            my %pendinghash = (
                                'internal.pendingco-owners' => $pendingcoowners,
                            );
                            my $putresult = &put('environment',\%pendinghash,$cdom,$cnum);
                            if ($putresult eq 'ok') {
                                if ($env{'course.'.$cid.'.num'} eq $cnum) {
                                    &appenv({'course.'.$cid.'.internal.pendingco-owners' => $pendingcoowners});
                                }
                            }
                        } else {
                            my $delresult = &Apache::lonnet::del('environment',['internal.pendingco-owners'],$cdom,$cnum);
                            if ($delresult eq 'ok') {
                                if ($env{'course.'.$cid.'.internal.pendingco-owners'}) {
                                    &Apache::lonnet::delenv('course.'.$cid.'.internal.pendingco-owners');
                                }
                            }
                        }
                    }
                } elsif ($oldowner_to_coowner) {
                    &Apache::lonnet::store_coowners($cdom,$cnum,$chome,'',
                                         $settings->{'internal.courseowner'});

                }
            } elsif ($oldowner_to_coowner) {
                &Apache::lonnet::store_coowners($cdom,$cnum,$chome,'',
                                     $settings->{'internal.courseowner'});
            }
        }
    }
    if ($settings->{'internal.coursecode'} ne $newattr->{'coursecode'}) {
        if ($newattr->{'coursecode'} ne '') {
            my %designhash = &Apache::loncommon::get_domainconf($cdom);
            if ($designhash{$cdom.'.autoassign.co-owners'}) {
                my @newcoowners = ();
                if ($settings->{'internal.co-owners'}) {
                    my @currcoown = split(',',$settings->{'internal.coowners'});
                    my ($updatecoowners,$delcoowners);
                    foreach my $person (@currcoown) {
                        my ($result,$desc) = &Apache::lonnet::auto_validate_instcode($cnum,$cdom,$newattr->{'coursecode'},$person);
                        if ($result eq 'valid') {
                            push(@newcoowners,$person);
                        }
                    }
                    foreach my $item (sort(keys(%cchash))) {
                        my ($uname,$udom,$urole) = split(':',$item);
                        next if ($uname.':'.$udom eq $newattr->{'courseowner'});
                        unless (grep(/^\Q$uname\E:\Q$udom\E$/,@newcoowners)) {
                            my ($result,$desc) = &Apache::lonnet::auto_validate_instcode($cnum,$cdom,$newattr->{'coursecode'},$uname.':'.$udom);
                            if ($result eq 'valid') {
                                push(@newcoowners,$uname.':'.$udom);
                            }
                        }
                    }
                    if (@newcoowners) {
                        my $coowners = join(',',sort(@newcoowners));
                        unless ($coowners eq $settings->{'internal.co-owners'}) {
                            $updatecoowners = 1;
                        }
                    } else {
                        $delcoowners = 1;
                    }
                    if ($updatecoowners || $delcoowners) {
                        &Apache::lonnet::store_coowners($cdom,$cnum,$chome,
                                                        $delcoowners,@newcoowners);
                    }
                } else {
                    foreach my $item (sort(keys(%cchash))) {
                        my ($uname,$udom,$urole) = split(':',$item);
                        push(@newcoowners,$uname.':'.$udom);
                    }
                    if (@newcoowners) {
                        &Apache::lonnet::store_coowners($cdom,$cnum,$chome,'',
                                                        @newcoowners);
                    }
                }
            }
        }
    }
    return;
}

sub modify_quota {
    my ($r,$cdom,$cnum,$cdesc,$domdesc,$type) = @_;
    &print_header($r,$type);
    $r->print('<form action="/adm/modifycourse" method="post" name="processquota">'."\n".
              '<h3>'.&mt('Disk space for storage of group portfolio files for:').
              ' <span class="LC_nobreak">'.$cdesc.'</span></h3><br />');
    my %oldsettings = &Apache::lonnet::get('environment',['internal.coursequota'],$cdom,$cnum);
    my $defaultquota = 20;
    if ($env{'form.coursequota'} ne '') {
        my $newquota = $env{'form.coursequota'};
        if ($newquota =~ /^\s*(\d+\.?\d*|\.\d+)\s*$/) {
            $newquota = $1;
            if ($oldsettings{'internal.coursequota'} eq $env{'form.coursequota'}) {
                $r->print(&mt('The disk space allocated for group portfolio files remains unchanged as [_1] Mb.',$env{'form.coursequota'}));
            } else {
                my %cenv = (
                           'internal.coursequota' => $env{'form.coursequota'},
                           );
                my $putreply = &Apache::lonnet::put('environment',\%cenv,$cdom,
                                                    $cnum);
                if (($oldsettings{'internal.coursequota'} eq '') && 
                    ($env{'form.coursequota'} == $defaultquota)) {
                    if ($type eq 'Community') {
                         $r->print(&mt('The disk space allocated for group portfolio files in this community is the default quota for this domain: [_1] Mb.',$defaultquota));
                    } else {
                         $r->print(&mt('The disk space allocated for group portfolio files in this course is the default quota for this domain: [_1] Mb.',$defaultquota));
                    }
                } else {
                    if ($putreply eq 'ok') {
                        my %updatedsettings = &Apache::lonnet::get('environment',['internal.coursequota'],$cdom,$cnum);
                        $r->print(&mt('The disk space allocated for group portfolio files is now: [_1] Mb.',$updatedsettings{'internal.coursequota'}));
                        my $usage = &Apache::longroup::sum_quotas($cdom.'_'.$cnum);
                        if ($usage >= $updatedsettings{'internal.coursequota'}) {
                            my $newoverquota;
                            if ($usage < $oldsettings{'internal.coursequota'}) {
                                $newoverquota = 'now';
                            }
                            $r->print('<p>');
                            if ($type eq 'Community') {
                                $r->print(&mt('Disk usage [_1] exceeds the quota for this community.',$newoverquota).' '.
                                          &mt('Upload of new portfolio files and assignment of a non-zero Mb quota to new groups in the community will not be possible until some files have been deleted, and total usage is below community quota.'));
                            } else {
                                $r->print(&mt('Disk usage [_1] exceeds the quota for this course.',$newoverquota).' '.
                                          &mt('Upload of new portfolio files and assignment of a non-zero Mb quota to new groups in the course will not be possible until some files have been deleted, and total usage is below course quota.'));
                            }
                            $r->print('</p>');
                        }
                    } else {
                        $r->print(&mt('An error occurred storing the quota for group portfolio files: ').
                                  $putreply);
                    }
                }
            }
        } else {
            $r->print(&mt('The new quota requested contained invalid characters, so the quota is unchanged.'));
        }
    }
    $r->print('<p>'.
              '<a href="javascript:changePage(document.processquota,'."'menu'".')">'.
              &mt('Pick another action').'</a>');
    $r->print(&hidden_form_elements().'</form>');
    return;
}

sub modify_catsettings {
    my ($r,$cdom,$cnum,$cdesc,$domdesc,$type) = @_;
    &print_header($r,$type);
    my ($ccrole,%desc);
    if ($type eq 'Community') {
        $desc{'hidefromcat'} = &mt('Excluded from community catalog');
        $desc{'categories'} = &mt('Assigned categories for this community');
        $ccrole = 'co';
    } else {
        $desc{'hidefromcat'} = &mt('Excluded from course catalog');
        $desc{'categories'} = &mt('Assigned categories for this course');
        $ccrole = 'cc';
    }
    $r->print('
<form action="/adm/modifycourse" method="post" name="processcat">
<h3>'.&mt('Category settings').'</h3>');
    my %domconf = &Apache::lonnet::get_dom('configuration',['coursecategories'],$cdom);
    my @cat_params = &catalog_settable($domconf{'coursecategories'},$type);
    if (@cat_params > 0) {
        my (%cenv,@changes,@nochanges);
        my %currsettings =
            &Apache::lonnet::get('environment',['hidefromcat','categories'],$cdom,$cnum);
        my (@newcategories,%showitem); 
        if (grep(/^togglecats$/,@cat_params)) {
            if ($currsettings{'hidefromcat'} ne $env{'form.hidefromcat'}) {
                push(@changes,'hidefromcat');
                $cenv{'hidefromcat'} = $env{'form.hidefromcat'};
            } else {
                push(@nochanges,'hidefromcat');
            }
            if ($env{'form.hidefromcat'} eq 'yes') {
                $showitem{'hidefromcat'} = '"'.&mt('Yes')."'";
            } else {
                $showitem{'hidefromcat'} = '"'.&mt('No').'"';
            }
        }
        if (grep(/^categorize$/,@cat_params)) {
            my (@cats,@trails,%allitems,%idx,@jsarray);
            if (ref($domconf{'coursecategories'}) eq 'HASH') {
                my $cathash = $domconf{'coursecategories'}{'cats'};
                if (ref($cathash) eq 'HASH') {
                    &Apache::loncommon::extract_categories($cathash,\@cats,\@trails,
                                                           \%allitems,\%idx,\@jsarray);
                }
            }
            @newcategories =  &Apache::loncommon::get_env_multiple('form.usecategory');
            if (@newcategories == 0) {
                $showitem{'categories'} = '"'.&mt('None').'"';
            } else {
                $showitem{'categories'} = '<ul>';
                foreach my $item (@newcategories) {
                    $showitem{'categories'} .= '<li>'.$trails[$allitems{$item}].'</li>';
                }
                $showitem{'categories'} .= '</ul>';
            }
            my $catchg = 0;
            if ($currsettings{'categories'} ne '') {
                my @currcategories = split('&',$currsettings{'categories'});
                foreach my $cat (@currcategories) {
                    if (!grep(/^\Q$cat\E$/,@newcategories)) {
                        $catchg = 1;
                        last;
                    }
                }
                if (!$catchg) {
                    foreach my $cat (@newcategories) {
                        if (!grep(/^\Q$cat\E$/,@currcategories)) {
                            $catchg = 1;
                            last;                     
                        } 
                    } 
                }
            } else {
                if (@newcategories > 0) {
                    $catchg = 1;
                }
            }
            if ($catchg) {
                $cenv{'categories'} = join('&',@newcategories);
                push(@changes,'categories');
            } else {
                push(@nochanges,'categories');
            }
            if (@changes > 0) {
                my $putreply = &Apache::lonnet::put('environment',\%cenv,$cdom,$cnum);
                if ($putreply eq 'ok') {
                    my %crsinfo = &Apache::lonnet::courseiddump($cdom,'.',1,'.','.',
                                                                $cnum,undef,undef,'.');
                    if (ref($crsinfo{$env{'form.pickedcourse'}}) eq 'HASH') {
                        if (grep(/^hidefromcat$/,@changes)) {
                            $crsinfo{$env{'form.pickedcourse'}}{'hidefromcat'} = $env{'form.hidefromcat'};
                        }
                        if (grep(/^categories$/,@changes)) {
                            $crsinfo{$env{'form.pickedcourse'}}{'categories'} = $cenv{'categories'};
                        }
                        my $chome = &Apache::lonnet::homeserver($cnum,$cdom);
                        my $putres = &Apache::lonnet::courseidput($cdom,\%crsinfo,$chome,'notime');
                    }
                    $r->print(&mt('The following changes occurred:').'<ul>');
                    foreach my $item (@changes) {
                        $r->print('<li>'.&mt('[_1] now set to: [_2]',$desc{$item},$showitem{$item}).'</li>');
                    }
                    $r->print('</ul><br />');
                }
            }
            if (@nochanges > 0) {
                $r->print(&mt('The following were unchanged:').'<ul>');
                foreach my $item (@nochanges) {
                    $r->print('<li>'.&mt('[_1] still set to: [_2]',$desc{$item},$showitem{$item}).'</li>');
                }
                $r->print('</ul>');
            }
        }
    } else {
        my $newrole = $ccrole.'./'.$cdom.'/'.$cnum;
        my $escuri = &HTML::Entities::encode('/adm/roles?selectrole=1&'.$newrole.
                                             '=1&destinationurl=/adm/courseprefs','&<>"');
        if ($type eq 'Community') {
            $r->print(&mt('Category settings for communities in this domain should be modified in community context (via "[_1]Community Configuration[_2]").','<a href="$escuri">','</a>').'<br />');
        } else {
            $r->print(&mt('Category settings for courses in this domain should be modified in course context (via "[_1]Course Configuration[_2]").','<a href="$escuri">','</a>').'<br />');
        }
    }
    $r->print('<br />'."\n".
              '<a href="javascript:changePage(document.processcat,'."'menu'".')">'.
              &mt('Pick another action').'</a>');
    $r->print(&hidden_form_elements().'</form>');
    return;
}

sub print_header {
    my ($r,$type,$javascript_validations) = @_;
    my $phase = "start";
    if ( exists($env{'form.phase'}) ) {
        $phase = $env{'form.phase'};
    }
    my $js = qq|
<script type="text/javascript">
function changePage(formname,newphase) {
    formname.phase.value = newphase;
    if (newphase == 'processparms') {
        return;
    }
    formname.submit();
}
</script>
|;
    if ($phase eq 'setparms') {
	$js .= qq|
<script  type="text/javascript">
$javascript_validations
</script>
|;
    } elsif ($phase eq 'courselist') {
        $js .= qq|
<script type="text/javascript">
function gochoose(cname,cdom,cdesc) {
    document.courselist.pickedcourse.value = cdom+'_'+cname;
    document.courselist.submit();
}
</script>
|;
    } elsif ($phase eq 'setquota') {
        $js .= <<'ENDSCRIPT';
<script type="text/javascript">
function verify_quota(formname) {
    var newquota = formname.coursequota.value; 
    var num_reg = /^\s*(\d+\.?\d*|\.\d+)\s*$/;
    if (num_reg.test(newquota)) {
        changePage(formname,'processquota');
    } else {
        alert("The quota you entered contained invalid characters.\nYou must enter a number");
    }
    return;
}
</script>
ENDSCRIPT
    }
    my $starthash;
    if ($env{'form.phase'} eq 'ccrole') {
        $starthash = {
           add_entries => {'onload' => "javascript:document.ccrole.submit();"},
                     };
    }
    $r->print(&Apache::loncommon::start_page('View/Modify Course/Community Settings',
					     $js,$starthash));
    my $bread_text = "View/Modify Courses/Communities";
    if ($type eq 'Community') {
        $bread_text = 'Community Settings';
    } else {
        $bread_text = 'Course Settings';
    }
    $r->print(&Apache::lonhtmlcommon::breadcrumbs($bread_text));
    return;
}

sub print_footer {
    my ($r) = @_;
    $r->print('<br />'.&Apache::loncommon::end_page());
    return;
}

sub check_course {
    my ($r,$dom,$domdesc) = @_;
    my ($ok_course,$description,$instcode,$owner);
    my %args = (
                 one_time => 1,
               );
    my %coursehash = 
        &Apache::lonnet::coursedescription($env{'form.pickedcourse'},\%args);
    my $cnum = $coursehash{'num'};
    my $cdom = $coursehash{'domain'};
    if ($cdom eq $dom) {
        my $description;
        my %courseIDs = &Apache::lonnet::courseiddump($cdom,'.',1,'.','.',
                                                      $cnum,undef,undef,'.');
        if (keys(%courseIDs) > 0) {
            $ok_course = 'ok';
            my ($instcode,$owner);
            if (ref($courseIDs{$cdom.'_'.$cnum}) eq 'HASH') {
                $description = $courseIDs{$cdom.'_'.$cnum}{'description'};
                $instcode = $courseIDs{$cdom.'_'.$cnum}{'inst_code'};
                $owner = $courseIDs{$cdom.'_'.$cnum}{'owner'};          
            } else {
                ($description,$instcode,$owner) = 
                                   split(/:/,$courseIDs{$cdom.'_'.$cnum});
            }
            $description = &unescape($description);
            $instcode = &unescape($instcode);
            if ($instcode) {
                $description .= " ($instcode)";
            }
            return ($ok_course,$description);
        }
    }
}

sub course_settings_descrip {
    my ($type) = @_;
    my %longtype;
    if ($type eq 'Community') {
         %longtype = &Apache::lonlocal::texthash(
                      'courseowner' => "Username:domain of community owner",
                      'co-owners'   => "Username:domain of each co-owner",
         );
    } else {
         %longtype = &Apache::lonlocal::texthash(
                      'authtype' => 'Default authentication method',
                      'autharg'  => 'Default authentication parameter',
                      'autoadds' => 'Automated adds',
                      'autodrops' => 'Automated drops',
                      'autostart' => 'Date of first automated enrollment',
                      'autoend' => 'Date of last automated enrollment',
                      'default_enrollment_start_date' => 'Date of first student access',
                      'default_enrollment_end_date' => 'Date of last student access',
                      'coursecode' => 'Official course code',
                      'courseowner' => "Username:domain of course owner",
                      'co-owners'   => "Username:domain of each co-owner",
                      'notifylist' => 'Course Coordinators to be notified of enrollment changes',
                      'sectionnums' => 'Course section number:LON-CAPA section',
                      'crosslistings' => 'Crosslisted class:LON-CAPA section',
         );
    }
    return %longtype;
}

sub hidden_form_elements {
    my $hidden_elements = 
      &Apache::lonhtmlcommon::echo_form_input(['gosearch','updater','coursecode',
          'prevphase','numlocalcc','courseowner','login','coursequota','intarg',
          'locarg','krbarg','krbver','counter','hidefromcat','usecategory'])."\n".
          '<input type="hidden" name="prevphase" value="'.$env{'form.phase'}.'" />';
    return $hidden_elements;
}

sub handler {
    my $r = shift;
    if ($r->header_only) {
        &Apache::loncommon::content_type($r,'text/html');
        $r->send_http_header;
        return OK;
    }
    my $dom = $env{'request.role.domain'};
    my $domdesc = &Apache::lonnet::domain($dom,'description');
    if (&Apache::lonnet::allowed('ccc',$dom)) {
        &Apache::loncommon::content_type($r,'text/html');
        $r->send_http_header;

        &Apache::lonhtmlcommon::clear_breadcrumbs();

        my $phase = $env{'form.phase'};
        if ($env{'form.updater'}) {
            $phase = '';
        }
        if ($phase eq '') {
            &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>"/adm/modifycourse",
              text=>"Course/Community search"});
            &print_course_search_page($r,$dom,$domdesc);
        } else {
            my $firstform = $phase;
            if ($phase eq 'courselist') {
                $firstform = 'filterpicker';
            }
            my $choose_text;
            my $type = $env{'form.type'};
            if ($type eq '') {
                $type = 'Course';
            }
            if ($type eq 'Community') {
                $choose_text = "Choose a community";
            } else {
                $choose_text = "Choose a course";
            } 
            &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>"javascript:changePage(document.$firstform,'')",
              text=>"Course/Community search"},
              {href=>"javascript:changePage(document.$phase,'courselist')",
              text=>$choose_text});
            if ($phase eq 'courselist') {
                &print_course_selection_page($r,$dom,$domdesc);
            } else {
                my ($checked,$cdesc) = &check_course($r,$dom,$domdesc);
                if ($checked eq 'ok') {
                    my $enter_text;
                    if ($type eq 'Community') {
                        $enter_text = 'Enter community';
                    } else {
                        $enter_text = 'Enter course';
                    }
                    if ($phase eq 'menu') {
                        &Apache::lonhtmlcommon::add_breadcrumb
                        ({href=>"javascript:changePage(document.$phase,'menu')",
                          text=>"Pick action"});
                        &print_modification_menu($r,$cdesc,$domdesc,$dom,$type);
                    } elsif ($phase eq 'ccrole') {
                        &Apache::lonhtmlcommon::add_breadcrumb
                         ({href=>"javascript:changePage(document.$phase,'ccrole')",
                           text=>$enter_text});
                        &print_ccrole_selected($r,$type);
                    } else {
                        &Apache::lonhtmlcommon::add_breadcrumb
                        ({href=>"javascript:changePage(document.$phase,'menu')",
                          text=>"Pick action"});
                        my ($cdom,$cnum) = split(/_/,$env{'form.pickedcourse'});
                        if ($phase eq 'setquota') {
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Set quota"});
                            &print_setquota($r,$cdom,$cnum,$cdesc,$type);
                        } elsif ($phase eq 'processquota') { 
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'setquota')",
                              text=>"Set quota"});
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Result"});
                            &modify_quota($r,$cdom,$cnum,$cdesc,$domdesc,$type);
                        } elsif ($phase eq 'viewparms') {  
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'viewparms')",
                              text=>"Display settings"});
                            &print_settings_display($r,$cdom,$cnum,$cdesc,$type);
                        } elsif ($phase eq 'setparms') {
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Change settings"});
                            &print_course_modification_page($r,$cdom,$cnum,$cdesc,$type);
                        } elsif ($phase eq 'processparms') {
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'setparms')",
                              text=>"Change settings"});
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Result"});
                            &modify_course($r,$cdom,$cnum,$cdesc,$domdesc,$type);
                        } elsif ($phase eq 'catsettings') {
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Catalog settings"});
                            &print_catsettings($r,$cdom,$cnum,$cdesc,$type);
                        } elsif ($phase eq 'processcat') {
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'catsettings')",
                              text=>"Catalog settings"});
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Result"});
                            &modify_catsettings($r,$cdom,$cnum,$cdesc,$domdesc,$type);
                        }
                    }
                } else {
                    $r->print('<span class="LC_error">');
                    if ($type eq 'Community') {
                        $r->print(&mt('The course you selected is not a valid course in this domain'));
                    } else {
                        $r->print(&mt('The community you selected is not a valid community in this domain'));
                    }
                    $r->print(" ($domdesc)</span>");
                }
            }
        }
        &print_footer($r);
    } else {
        $env{'user.error.msg'}=
        "/adm/modifycourse:ccc:0:0:Cannot modify course/community settings";
        return HTTP_NOT_ACCEPTABLE;
    }
    return OK;
}

1;
__END__
