# The LearningOnline Network with CAPA
# Routines for messaging display
#
# $Id: lonmsgdisplay.pm,v 1.124.4.10 2010/04/14 06:36:26 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#


package Apache::lonmsgdisplay;

=pod

=head1 NAME

Apache::lonmsgdisplay: supports internal messaging

=head1 SYNOPSIS

lonmsgdisplay provides a handler to allow users to read, send, 
and delete messages, and to create and delete message folders,
and to move messages between folders.

=head1 OVERVIEW

=head2 Messaging Overview

X<messages>LON-CAPA provides an internal messaging system similar to
email, but customized for LON-CAPA's usage. LON-CAPA implements its
own messaging system, rather then building on top of email, because of
the features LON-CAPA messages can offer that conventional e-mail can
not:

=over 4

=item * B<Critical messages>: A message the recipient B<must>
acknowlegde receipt of before they are allowed to continue using the
system, preventing a user from claiming they never got a message

=item * B<Receipts>: LON-CAPA can reliably send reciepts informing the
sender that it has been read; again, useful for preventing students
from claiming they did not see a message. (While conventional e-mail
has some reciept support, it's sporadic, e-mail client-specific, and
generally the receiver can opt to not send one, making it useless in
this case.)

=item * B<Context>: LON-CAPA knows about the sender, such as where
they are in a course. When a student mails an instructor asking for
help on the problem, the instructor receives not just the student's
question, but all submissions the student has made up to that point,
the user's rendering of the problem, and the complete view the student
saw of the resource, including discussion up to that point. Finally,
the instructor is reading all of this inside of LON-CAPA, not their
email program, so they have full access to LON-CAPA's grading
interface, or other features they may wish to use in response to the
student's query.

=item * B<Blocking>: LON-CAPA can block selected communication 
features for students during an online exam. A course coordinator or
instructor can set an open and close date/time for scheduled online
exams in a course. If a user uses the LON-CAPA internal messaging 
system to display e-mails during the scheduled blocking event,  
display of all e-mail sent during the blocking period will be 
suppressed, and a message of explanation, including details of the 
currently active blocking periods will be displayed instead. A user 
who has a course coordinator or instructor role in a course will be
unaffected by any blocking periods for the course, unless the user
also has a student role in the course, AND has selected the student role.

=back

Users can ask LON-CAPA to forward messages to conventional e-mail
addresses on their B<PREF> screen, but generally, LON-CAPA messages
are much more useful than traditional email can be made to be, even
with HTML support.

=cut

use strict;
use Apache::lonnet;
use HTML::TokeParser();
use Apache::Constants qw(:common);
use Apache::loncommon();
use Apache::lonhtmlcommon();
use Apache::lontexconvert();
use HTML::Entities();
use Apache::lonlocal;
use Apache::loncommunicate;
use Apache::lonfeedback;
use Apache::lonrss();
use Apache::lonselstudent();
use lib '/home/httpd/lib/perl/';
use LONCAPA;

# Querystring component with sorting type
my $sqs;
my $startdis;

# ============================================================ List all folders

sub folderlist {
    my ($folder,$msgstatus) = @_;
    my %lt = &Apache::lonlocal::texthash(
                actn => 'Action',
                fold => 'Folder',
                show => 'Show',
                status => 'Message Status',
                go   => 'Go',
                nnff => 'New Name for Folder',
                newn => 'New Name',
                thfm => 'The folder may not be renamed',
                fmnb => 'folder may not be renamed as it is a folder provided by the system.',
                asth => 'as this name is already in use for a system-provided or user-defined folder.',
                the => 'The',
                tnfm => 'The new folder may not be named',

    );

	# set se lastvisit for the new mail check in the toplevel menu
	&Apache::lonnet::appenv({'user.mailcheck.lastvisit'=>time});

    my %actions = &Apache::lonlocal::texthash(
                                view => 'View Folder',
                                rename => 'Rename Folder',
                                delete => 'Delete Folder',
    );
    $actions{'select_form_order'} = ['view','rename','delete'];

    my %statushash = &Apache::lonlocal::texthash(&get_msgstatus_types());

    $statushash{'select_form_order'} = ['','new','read','replied','forwarded'];

    my %permfolders = &get_permanent_folders();
    my $permlist = join("','",sort(keys(%permfolders)));
    my ($permlistkeys,$permlistvals);
    foreach my $key (sort(keys(%permfolders))) {
        $permlistvals .= $permfolders{$key}."','";
        $permlistkeys .= $key."','";
    }
    $permlistvals =~ s/','$//;
    $permlistkeys =~ s/','$//;
    my %gotfolders = &Apache::lonmsg::get_user_folders();
    my %userfolders;

    foreach my $key (keys(%gotfolders)) {
        $key =~ s/(['"])/\$1/g; #' stupid emacs
        $userfolders{$key} = $key;
    }
    my @userorder = sort(keys(%userfolders));
    my %formhash = (%permfolders,%userfolders);
    my $folderlist = join("','",@userorder);
    $folderlist .= "','".$permlistvals;

    $formhash{'select_form_order'} = ['','critical',@userorder,'sent','trash'];
    my $output = qq|<script type="text/javascript">
// <![CDATA[
function folder_choice(targetform,caller) {
    var permfolders_keys = new Array('$permlistkeys');
    var permfolders_vals = new Array('$permlistvals');
    var allfolders = new Array('$folderlist');
    if (caller == 'change') {
        if (targetform.folderaction.options[targetform.folderaction.selectedIndex].value == 'rename') {
            for (var i=0; i<permfolders_keys.length; i++) {
                if (permfolders_keys[i] == targetform.folder.value) {
                    alert("$lt{'the'} '"+permfolders_vals[i]+"' $lt{'fmnb'}");
                    return;
                }
            }
            var foldername=prompt('$lt{'nnff'}','$lt{'newn'}');
            if (foldername) {
                targetform.renamed.value=foldername;
                for (var i=0; i<allfolders.length; i++) {
                    if (allfolders[i] == foldername) {
                        alert("$lt{'thfm'} '"+foldername+"' $lt{'asth'}");
                        return;
                    }
                }
                targetform.submit();
            }
        }
        else {
            targetform.submit();
        }
    }
    if (caller == 'new') {
        var newname=targetform.newfolder.value;
        if (newname) {
            for (var i=0; i<allfolders.length; i++) {
                if (allfolders[i] == newname) {
                    alert("$lt{'tnfm'} '"+newname+"' $lt{'asth'}");
                    return;
                }
            }
            targetform.submit();
        }
    }
}
// ]]>
</script>|;
    my %show = ('select_form_order' => [10,20,50,100,200],
		map {$_=>$_} (10,20,50,100,200));

    $output .= '
<form method="post" action="/adm/email" name="folderlist">
<table border="0" cellspacing="2" cellpadding="2">
 <tr>
  <td align="left">
   <table border="0" cellspacing="2" cellpadding="2">
    <tr>
     <td align="center"><b>'.$lt{'fold'}.'</b><br />'."\n".
         &Apache::loncommon::select_form($folder,'folder',%formhash).'
     </td>
     <td align="center"><b>'.$lt{'show'}.'</b><br />'."\n".
         &Apache::loncommon::select_form($env{'form.interdis'},'interdis',
					 %show).'
     </td>
     <td align="center"><b>'.$lt{'status'}.'</b><br />'."\n".
       &Apache::loncommon::select_form($msgstatus,'msgstatus',%statushash).'
     </td>
     <td align="center"><b>'.$lt{'actn'}.'</b><br />'.
         &Apache::loncommon::select_form('view','folderaction',%actions).'
     </td><td><br />'.
    '<input type="button" value="'.$lt{'go'}.'" onclick="javascript:folder_choice(this.form,'."'change'".');" />
     </td>
    </tr>
   </table>
  </td><td>&nbsp;</td><td>&nbsp;</td><td>&nbsp;</td><td>&nbsp;</td>
  <td align="right">
   <table><tr><td><br />
    <input type="button" value="'.&mt('New Folder').
    '" onclick="javascript:folder_choice(this.form,'."'new'".');" /></td>'.
    '<td align="center"><b>'.&mt('Name').'</b><br />'.
    '<input type="text" size="15" name="newfolder" value="" />
    </td></tr></table>
  </td>
 </tr>
</table>'."\n".
    '<input type="hidden" name="sortedby" value="'.$env{'form.sortedby'}.'" />'.
    '<input type="hidden" name="renamed" value="" />'.
        ($folder=~/^critical/?'</form>':'');
    return $output;
}

sub get_permanent_folders {
    my %permfolders = 
	&Apache::lonlocal::texthash(''         => 'INBOX',
				    'trash'    => 'TRASH',
				    'critical' => 'Critical',
				    'sent'     => 'Sent Messages',
				    );
    return %permfolders;
}

sub get_msgstatus_types {
    # Don't translate here!
    my %statushash = (
        ''          => 'Any',
        'new'       => 'Unread',
        'read'      => 'Read',
        'replied'   => 'Replied to',
        'forwarded' => 'Forwarded',
    );
    return %statushash;
}

sub scrollbuttons {
    my ($start,$maxdis,$first,$finish,$total,$msgstatus)=@_;
    unless ($total>0) { return ''; }
    $start++; $maxdis++;$first++;$finish++;

    my %statushash = &get_msgstatus_types();
    my $status;
    if ($msgstatus eq '') {
        $status = 'All'; # Don't translate here!
    } else {
        $status = $statushash{$msgstatus};
    }
    return
   '<b>'.&mt('Page').'</b>: '. 
   '<input type="submit" name="firstview" value="|&lt;" />'.
   '<input type="submit" name="prevview" value="&lt;" />'.
   '<input type="text" size="5" name="startdis" value="'.$start.'" onchange="this.form.submit()" /> of '.$maxdis.
   '<input type="submit" name="nextview" value="&gt;" />'.
   '<input type="submit" name="lastview" value="&gt;|" /><br />'.
   &mt('<b>[_1] messages</b>: showing messages [_2] through [_3] of [_4].',$status,$first,$finish,$total).'</form>';
}
# =============================================================== Status Change

sub statuschange {
    my ($msgid,$newstatus,$folder)=@_;
    my $suffix=&Apache::lonmsg::foldersuffix($folder);
    my %status=&Apache::lonnet::get('email_status'.$suffix,[$msgid]);
    if ($status{$msgid}=~/^error\:/) { $status{$msgid}=''; }
    unless ($status{$msgid}) { $status{$msgid}='new'; }
    unless (($status{$msgid} eq 'replied') || 
            ($status{$msgid} eq 'forwarded')) {
	&Apache::lonnet::put('email_status'.$suffix,{$msgid => $newstatus});
    }
    if (($newstatus eq 'deleted') || ($newstatus eq 'new')) {
	&Apache::lonnet::put('email_status'.$suffix,{$msgid => $newstatus});
    }
    if ($newstatus eq 'deleted') {
	return &movemsg($msgid,$folder,'trash');
    }
    return ;
}

# ============================================================= Make new folder

sub makefolder {
    my ($newfolder) = @_;
    my %permfolders = &get_permanent_folders();
    my %userfolders = &Apache::lonmsg::get_user_folders();
    my ($outcome,$warning);
    if (defined($userfolders{$newfolder})) {
        return &mt('The folder name: "[_1]" is already in use for an existing folder.',$newfolder);
    }
    foreach my $perm (keys(%permfolders)) {
        if ($permfolders{$perm} eq $newfolder) {
            return &mt('The folder name: "[_1]" is already used for one of the folders automatically generated by the system.',$newfolder);
        }
    } 
    if (&get_msgfolder_lock() eq 'ok') {
        my %counter_hash = &Apache::lonnet::get('email_folders',["\0".'idcount']);
        my $lastcount = $counter_hash{"\0".'idcount'};
        my $folder_id = $lastcount + 1;
        while (defined($userfolders{$folder_id})) {
            $folder_id ++;
        }
        my %folderinfo = ( id      => $folder_id,
                           created => time, );
        $outcome =  
	    &Apache::lonnet::put('email_folders',{$newfolder => \%folderinfo,
						  "\0".'idcount' => $folder_id});
        my $releaseresult = &release_msgfolder_lock();
        if ($releaseresult ne 'ok') {
            $warning = $releaseresult;
        }
    } else {
        $outcome = 
	    &mt('Error - could not obtain lock on message folders record.');
    }
    return ($outcome,$warning);
}

# ============================================================= Delete folder

sub deletefolder {
    my ($folder)=@_;
    my %permfolders = &get_permanent_folders();
    if (defined($permfolders{$folder})) {
        return &mt('The folder "[_1]" may not be deleted',$folder); 
    }
    my %userfolders = &Apache::lonmsg::get_user_folders();
    if (!defined($userfolders{$folder})) {
        return &mt('The folder "[_1]" does not exist so deletion is not required.',
                   $folder);
    }
    # check folder is empty;
    my $suffix=&Apache::lonmsg::foldersuffix($folder);
    my @messages = &Apache::lonnet::getkeys('nohist_email'.$suffix);
    if (@messages > 0) {
        return &mt('The folder "[_1]" contains messages so it may not be deleted.',$folder).
               '<br />'.
               &mt('Delete or move the messages to a different folder first.');
    }
    my $delresult = &Apache::lonnet::del('email_folders',[$folder]);
    return $delresult;
}

sub renamefolder {
    my ($folder) = @_;
    my $newname = $env{'form.renamed'};
    my %permfolders = &get_permanent_folders();
    if ($env{'form.renamed'} eq '') {
        return &mt('The folder "[_1]" may not be renamed to "[_2]" as the new name you requested is an invalid name.',$folder,$newname);
    }
    if (defined($permfolders{$folder})) {
        return &mt('The folder "[_1]" may not be renamed as it is a folder provided by the system.',$folder);
    }
    if (defined($permfolders{$newname})) {
        return &mt('The folder "[_1]" may not be renamed to "[_2]" as the new name you requested is reserved for folders provided automatically by the system.',$folder,$newname);
    }
    my %userfolders = &Apache::lonmsg::get_user_folders();
    if (defined($userfolders{$newname})) {
        return &mt('The folder "[_1]" may not be renamed to "[_2]" because the new name you requested is already being used for an existing folder.',$folder,$newname);
    }
    if (!defined($userfolders{$folder})) {
        return &mt('The folder "[_1]" could not be renamed to "[_2]" because the folder does not exist.',$folder,$newname);
    }
    my %folderinfo;
    if (ref($userfolders{$folder}) eq 'HASH') {
        %folderinfo = %{$userfolders{$folder}};
    } else {
        %folderinfo = ( id      => $folder,
                        created => $userfolders{$folder},);
    }
    my $outcome =
     &Apache::lonnet::put('email_folders',{$newname => \%folderinfo});
    if ($outcome eq 'ok') {
        $outcome = &Apache::lonnet::del('email_folders',[$folder]);
    }
    return $outcome;
}

sub get_msgfolder_lock {
    # get lock for mail folder counter.
    my $lockhash = { "\0".'lock_counter' => time, };
    my $tries = 0;
    my $gotlock = &Apache::lonnet::newput('email_folders',$lockhash);
    while (($gotlock ne 'ok') && $tries <3) {
        $tries ++;
        sleep(1);
        $gotlock = &Apache::lonnet::newput('email_folders',$lockhash);
    }
    return $gotlock;
}

sub release_msgfolder_lock {
    #  remove lock
    my @del_lock = ("\0".'lock_counter');
    my $dellockoutcome=&Apache::lonnet::del('email_folders',\@del_lock);
    if ($dellockoutcome ne 'ok') {
        return ('<br />'.&mt('Warning: failed to release lock for counter').'<br />');
    } else {
        return 'ok';
    }
}


# ======================================================== Move between folders

sub movemsg {
    my ($msgid,$srcfolder,$trgfolder)=@_;
    if ($srcfolder eq 'new') { $srcfolder=''; }
    my $srcsuffix=&Apache::lonmsg::foldersuffix($srcfolder);
    my $trgsuffix=&Apache::lonmsg::foldersuffix($trgfolder);
    if ($srcsuffix eq $trgsuffix) {
	return (0,&mt('Message not moved, Attempted to move message to the same folder as it already is in.'));
    }

# Copy message
    my %message=&Apache::lonnet::get('nohist_email'.$srcsuffix,[$msgid]);
    if (!exists($message{$msgid}) || $message{$msgid} eq '') {
	if (&Apache::lonnet::error(%message)) {
	    return (0,&mt('Message not moved, A network error occurred.'));
	} else {
	    return (0,&mt('Message not moved as the message is no longer in the source folder.'));
	}
    }

    my $result =&Apache::lonnet::put('nohist_email'.$trgsuffix,
				     {$msgid => $message{$msgid}});
    if (&Apache::lonnet::error($result)) {
	return (0,&mt('Message not moved, A network error occurred.'));
    }

# Copy status
    unless ($trgfolder eq 'trash') {
       	my %status=&Apache::lonnet::get('email_status'.$srcsuffix,[$msgid]);
	# a non-existant status is the mark of an unread msg
	if (&Apache::lonnet::error(%status)) {
	    return (0,&mt('Message copied to new folder but status was not, A network error occurred.'));
	}
	my $result=&Apache::lonnet::put('email_status'.$trgsuffix,
					{$msgid => $status{$msgid}});
	if (&Apache::lonnet::error($result)) {
	    return (0,&mt('Message copied to new folder but status was not, A network error occurred.'));
	}
    }

# Delete orginals
    my $result_del_msg = 
	&Apache::lonnet::del('nohist_email'.$srcsuffix,[$msgid]);
    my $result_del_stat =
	&Apache::lonnet::del('email_status'.$srcsuffix,[$msgid]);
    if (&Apache::lonnet::error($result_del_msg)) {
	return (0,&mt('Message copied, but unable to delete the original from the source folder.'));
    }
    if (&Apache::lonnet::error($result_del_stat)) {
	return (0,&mt('Message copied, but unable to delete the original status from the source folder.'));
    }

    return (1);
}

# ======================================================= Display a course list

sub discourse {
    my ($statushash) = @_;
    my ($result,$active,$previous,$future);
    my $crstype = &Apache::loncommon::course_type();
    my ($course_personnel,
	$current_members,
	$expired_members,
	$future_members) = 
	    &Apache::lonselstudent::get_people_in_class($env{'request.course.sec'});
    unshift @$current_members, (@$course_personnel);
    my %defaultUsers;
    
    my $tmptext;
    if ($tmptext = &Apache::lonselstudent::render_student_list($current_members,
                                                               "activeusers",
                                                               "current",
                                                               \%defaultUsers,
                                                               1,"selectedusers",1,'email')
       ) {
        my $bcc_curr_hdr;
        if ($crstype eq 'Community') {
            $bcc_curr_hdr = &mt('Bcc: community participants with current access');
        } else {
            $bcc_curr_hdr = &mt('Bcc: course members with current access');
        }
        $result .= '<fieldset id="LC_activeusers"><legend>'
                   .'<b>'.$bcc_curr_hdr.'</b>'
                   .'</legend>'
                   .'<form name="activeusers">';
        $result .= $tmptext.'</form></fieldset><br />';
        if (ref($statushash) eq 'HASH') {
            $statushash->{'active'} = 1;
        }
    }
    if ($tmptext = &Apache::lonselstudent::render_student_list($expired_members,
                                                               "previoususers",
                                                               "expired",
                                                               \%defaultUsers,
                                                               1, "selectedusers",0,'email')
       ) {
        my $bcc_prev_hdr;
        if ($crstype eq 'Community') {
            $bcc_prev_hdr = &mt('Bcc: community participants with expired access');
        } else {
            $bcc_prev_hdr = &mt('Bcc: course members with expired access');
        }
        $result .= '<fieldset id="LC_previoususers"><legend>'
                   .'<b>'.$bcc_prev_hdr.'</b>'
                   .'</legend>'
                   .'<form name="previoususers">';
       $result .= $tmptext.'</form></fieldset><br />';
       if (ref($statushash) eq 'HASH') {
           $statushash->{'previous'} = 1;
       }
    }
    if ($tmptext = &Apache::lonselstudent::render_student_list($future_members,
                                                               "futureusers",
                                                               "future",
                                                               \%defaultUsers,
                                                               1, "selectedusers",0,'email')
       ) {
        my $bcc_future_hdr;
        if ($crstype eq 'Community') {
            $bcc_future_hdr = &mt('Bcc: community participants with future access');
        } else {
            $bcc_future_hdr = &mt('Bcc: course members with future access');
        }
        $result .= '<fieldset id="LC_futureusers"><legend>'
                   .'<b>'.$bcc_future_hdr.'</b>'
                   .'</legend>'
                   .'<form name="futureusers">';
       $result .= $tmptext.'</form></fieldset>';
       if (ref($statushash) eq 'HASH') {
           $statushash->{'future'} = 1;
       }
    }
    return $result;
}

sub disgroup {
    my ($r,$cdom,$cnum,$group,$access_status) = @_;
    my $hasfloat;
    #  Needs to be in a course
    if (!($env{'request.course.fn'})) {
        $r->print('<span class="LC_error">'.&mt('Error: you must have a course role selected to be able to send a broadcast message to a group in the course.').'</span>');
        return;
    }
    if ($cdom eq '' || $cnum eq '') {
        $r->print('<span class="LC_error">'.&mt('Error: could not determine domain or number of course').'</span>');
        return;
    }
    my ($memberinfo,$numitems) =
                 &Apache::longroup::group_memberlist($cdom,$cnum,$group,{},[]);
    my @statustypes = ('active');
    my $viewgrps = &Apache::lonnet::allowed('vcg',$env{'request.course.id'}.
                   ($env{'request.course.sec'}?'/'.$env{'request.course.sec'}:''));
    my $editgrps = &Apache::lonnet::allowed('mdg',$env{'request.course.id'}.
                   ($env{'request.course.sec'}?'/'.$env{'request.course.sec'}:''));
    if ($viewgrps || $editgrps) {
        push(@statustypes,('future','previous'));
    }
    if (keys(%{$memberinfo}) == 0) {
        $r->print('<span class="LC_warning">'.
                  &mt('As this group has no members, there are no recipients to select').
                  '</span>');
        return;
    } else {
        $hasfloat = 1;
        unless($env{'environment.wysiwygeditor'} eq 'on') {
            $r->print('<div class="LC_left_float">');
        }
        my %Sortby = (
                         active   => {},
                         previous => {},
                         future   => {},
                     );
        my %lt = &Apache::lonlocal::texthash(
                                     'name'     => 'Name',
                                     'usnm'     => 'Username',
                                     'doma'     => 'Domain',
                                     'active'   => 'Broadcast to Active Members',
                                     'previous' => 'Broadcast (Bcc) to Former Members',
                                     'future'   => 'Broadcast (Bcc) to Future Members',
                                    );
        foreach my $user (sort(keys(%{$memberinfo}))) {
            my $status = $$memberinfo{$user}{status};
            if ($env{'form.'.$status.'.sortby'} eq 'fullname') {
                push(@{$Sortby{$status}{$$memberinfo{$user}{fullname}}},$user);
            } elsif ($env{'form.'.$status.'.sortby'} eq 'username') {
                push(@{$Sortby{$status}{$$memberinfo{$user}{uname}}},$user);
            } elsif ($env{'form.'.$status.'.sortby'} eq 'domain') {
                push(@{$Sortby{$status}{$$memberinfo{$user}{udom}}},$user);
            } else {
                push(@{$Sortby{$status}{$$memberinfo{$user}{fullname}}},$user);
            }
        }
        $r->print(&group_check_uncheck());
        foreach my $status (@statustypes)  {
            if (ref($numitems) eq 'HASH') {
                if ((defined($$numitems{$status})) && ($$numitems{$status})) {
                    my $formname = $status.'users';
                    if (ref($access_status) eq 'HASH') {
                        $access_status->{$status} = $$numitems{$status};
                    }
                    $r->print('<fieldset><legend><b>'.$lt{$status}.
                              '</b></legend><form name="'.$formname.'">'.
                              '<span class="LC_nobreak">'.
                              '<input type="button" value="'.&mt('Check All').'" '.
                              'onclick="javascript:toggleAll('."this.form,'check'".')" />'.
                              '&nbsp;&nbsp;'.
                              '<input type="button" value="'.&mt('Uncheck All').'" '.
                              'onclick="javascript:toggleAll('."this.form,'uncheck'".')" />'.
                              '</span>');
                    if ($status eq 'active') {
                        $r->print(('&nbsp;'x3).'<select name="groupmail">'.
                                 '<option value="bcc" selected="selected">'.&mt('Bcc').'</option>'.
                                 '<option value="cc">'.&mt('Cc').'</option>'.
                               '</select>');
                    }
                    $r->print('<br />'.&Apache::loncommon::start_data_table().
                               &Apache::loncommon::start_data_table_header_row().
                               "<th>$lt{'name'}</th>".
                               "<th>$lt{'usnm'}</th>".
                               "<th>$lt{'doma'}</th>".
                               &Apache::loncommon::end_data_table_header_row());
                    foreach my $key (sort(keys(%{$Sortby{$status}}))) {
                        foreach my $user (@{$Sortby{$status}{$key}}) {
                            $r->print(&Apache::loncommon::start_data_table_row().
                                '<td><span class="LC_nobreak"><input type="checkbox" '.
                                'name="selectedusers_forminput" value="'.
                                $user.':'.$status.'" />'.
                                $$memberinfo{$user}{'fullname'}.'</span></td>'.
                                '<td>'.$$memberinfo{$user}{'uname'}.'</td>'.
                                '<td>'.$$memberinfo{$user}{'udom'}.'</td>'.
                                &Apache::loncommon::end_data_table_row());
                        }
                    }
                    $r->print(&Apache::loncommon::end_data_table().'</form>'.
                              '</fieldset><br />');
                }
            }
        }
        unless($env{'environment.wysiwygeditor'} eq 'on') {
            $r->print('</div>');
        }
    }
    return $hasfloat;
}

sub group_check_uncheck {
    my $output = qq|
<script type="text/javascript">
/ <![CDATA[
function toggleAll(form,action) {
    if (typeof(form.selectedusers_forminput.length)=="undefined") {
         if (action == 'check') {
            form.selectedusers_forminput.checked = true;
         } else {
            form.selectedusers_forminput.checked = false;
        }
    } else {
        for (var i=0; i<form.selectedusers_forminput.length; i++) {
            if (action == 'check') {
                form.selectedusers_forminput[i].checked = true;
            } else {
                form.selectedusers_forminput[i].checked = false;
            }
        }
    }
}
// ]]>
</script>
    |;
}

sub groupmail_header {
    my ($action,$group,$cdom,$cnum) = @_;
    my ($description,$refarg);
    if (!$cdom || !$cnum) {
        $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
        $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    }
    if (exists($env{'form.ref'})) {
        $refarg = 'ref='.$env{'form.ref'};
    }
    if (!$group) {
        $group = $env{'form.group'};
    }
    if ($group eq '') {
        return  '';
    } else {
        my %curr_groups = &Apache::longroup::coursegroups($cdom,$cnum,$group);
        if (defined($curr_groups{$group})) {
            my %groupinfo =
                    &Apache::longroup::get_group_settings($curr_groups{$group});
            $description = &unescape($groupinfo{'description'});
        }
    }
    &Apache::lonhtmlcommon::clear_breadcrumbs();
    if ($refarg) {
        my $brtitle;
        if (&Apache::loncommon::course_type() eq 'Community') {
            $brtitle = 'View community groups';  
        } else {
            $brtitle = 'View course groups';
        }
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>"/adm/coursegroups",
              text=>"Groups",
              title=>$brtitle});
    }
    &Apache::lonhtmlcommon::add_breadcrumb
        ({href=>"/adm/$cdom/$cnum/$group/smppg?$refarg",
          text=>"Group: $description",
          title=>"Go to group's home page"},
         {href=>"/adm/email?compose=group&amp;group=".
                "$env{'form.group'}&amp;$refarg",
          text=>"Send a Message in a Group",
          title=>"Compose Group Message"},);
    if ($action eq 'sending') {
            &Apache::lonhtmlcommon::add_breadcrumb
                         ({text=>"Messages being sent.",
                           title=>"E-mails sent"},);
    }
    my $groupheader = &Apache::loncommon::start_page('Group Message');
    $groupheader .= &Apache::lonhtmlcommon::breadcrumbs
                ('Group - '.$env{'form.group'}.' Email');
    return $groupheader;
}

sub groupmail_sent {
    my ($group,$cdom,$cnum) = @_;
    my $refarg;
    if (exists($env{'form.ref'})) {
        $refarg = 'ref='.$env{'form.ref'};
    }
    my $output .= '<br /><br /><a href="/adm/email?compose=group&amp;group='.
                  $group.'&amp;'.$refarg.'">'.
                  &mt('Send another group message').'</a>'.'&nbsp;&nbsp;&nbsp;'.
                  '<a href="/adm/'.$cdom.'/'.$cnum.'/'.$group.'/smppg?'.
                  $refarg.'">'. &mt('Return to group page').'</a>';
    return $output;
}

# ==================================================== Display Critical Message

sub discrit {
    my $r=shift;
    my $header = '<h1>'.&mt('Critical Messages').'</h1>'
                .'<div class="LC_warning">'
                .&mt('Access to other pages will be prevented until you have moved all critical messages to your inbox.')
                .'</div><br />'
                .'<form action="/adm/email" method="post">'
                .'<input type="hidden" name="confirm" value="true" />';
    my %what=&Apache::lonnet::dump('critical');
    my $result = '';
    foreach my $key (sort(keys(%what))) {
        my %content=&Apache::lonmsg::unpackagemsg($what{$key});
        next if ($content{'senderdomain'} eq '');
        $result .= &Apache::lonhtmlcommon::start_pick_box()
                  .&Apache::lonhtmlcommon::row_title(&mt('From'),undef,'LC_oddrow_value')
                  .'<b>'.&Apache::loncommon::aboutmewrapper(
                   &Apache::loncommon::plainname($content{'sendername'},$content{'senderdomain'}),$content{'sendername'},$content{'senderdomain'}).'</b>'
                  .' ('.$content{'sendername'}.':'.$content{'senderdomain'}.')'
                  .&Apache::lonhtmlcommon::row_closure(1)
                  .&Apache::lonhtmlcommon::row_title(&mt('Date'),undef,'LC_evenrow_value')
                  .$content{'time'}
                  .&Apache::lonhtmlcommon::row_closure(1)
                  .&Apache::lonhtmlcommon::row_title(&mt('Subject'),undef,'LC_oddrow_value')
                  .$content{'subject'}
                  .&Apache::lonhtmlcommon::row_closure(1)
                  .&Apache::lonhtmlcommon::row_title(&mt('Message'),undef,'LC_evenrow_value')
                  .'<pre>'.&Apache::lontexconvert::msgtexconverted($content{'message'}).'</pre>'
                  .&Apache::lonhtmlcommon::row_closure()
                  .&Apache::lonhtmlcommon::row_title('',undef,'LC_oddrow_value')
                  .'<div class="LC_warning">';
        my ($rec_button,$reprec_button);
        $rec_button = &mt('Move to Inbox');
        if (!$content{'noreplies'}) {
            $reprec_button = &mt('Move to Inbox/Compose reply');
        }
        if ($content{'sendback'}) {
            $rec_button = &mt('Confirm Receipt');
            if (!$content{'noreplies'}) {
                $reprec_button = &mt('Confirm Receipt and Reply');
            }
            $result .= &mt('You have to confirm that you have received this message before you can view other pages. After confirmation, this message will be moved to your regular inbox');
        } else {
            $result .= &mt('Access to other pages will be prevented until you have moved the message to your inbox.'); 
        }
        $result .= '</div>'
                  .&Apache::lonhtmlcommon::row_closure(1)
                  .&Apache::lonhtmlcommon::row_title('',undef,'LC_evenrow_value')
                  .'<input type="submit" name="rec_'.$key.'" value="'.$rec_button.'" />';
        if (!$content{'noreplies'}) {
            $result .= '<input type="submit" name="reprec_'.$key.'" '
                      .'value="'.$reprec_button.'" />'
        }
        $result .= &Apache::lonhtmlcommon::row_closure(1)
                  .&Apache::lonhtmlcommon::end_pick_box()
                  .'<br />';
    }
    # Check to see if there were any messages.
    if ($result eq '') {
        $result =
            '<p class="LC_info">'.
            &mt('You have no critical messages.').
            '</p>'.
            '<a href="/adm/roles">'.&mt('Select a course').'</a><br />'.
            '<a href="/adm/email">'.&mt('Communicate').'</a>';
    } else {
        $r->print($header);
    }
    $r->print($result);
    $r->print('<input type="hidden" name="displayedcrit" value="true" /></form>');
}

sub sortedmessages {
    my ($blocked,$startblock,$endblock,$numblocked,$folder,$msgstatus) = @_;
    my $suffix=&Apache::lonmsg::foldersuffix($folder);
    my @messages = &Apache::lonnet::getkeys('nohist_email'.$suffix);
    #unpack the varibles and repack into temp for sorting
    my @temp;
    my %descriptions;
    my %status_cache = 
	&Apache::lonnet::get('email_status'.&Apache::lonmsg::foldersuffix($folder),\@messages);

    my $get_received;
    if ($folder eq 'sent' 
	&& ($env{'form.sortedby'} =~ m/^(rev)?(user|domain)$/)) {
	$get_received = 1;
    }

    foreach my $msgid (@messages) {
        next if ($msgid eq '');
	my $esc_msgid=&escape($msgid);
	my ($sendtime,$shortsubj,$fromname,$fromdomain,$status,$fromcid,$processid,$symb,$error) =
	    &Apache::lonmsg::unpackmsgid($esc_msgid,$folder,undef,
					 \%status_cache);
        next if ($msgstatus ne '' && $msgstatus ne $status);
        my $description = &get_course_desc($fromcid,\%descriptions);
	my @temp1 = ($sendtime,$shortsubj,$fromname,$fromdomain,$status,
		     $esc_msgid,$description);
	if ($get_received) {
	    my %message = &Apache::lonnet::get('nohist_email'.$suffix,
					       [$msgid]);
	    my %content = &Apache::lonmsg::unpackagemsg($message{$msgid});
	    push(@temp1,$content{'recuser'},$content{'recdomain'});
	}
        # Check whether message was sent during blocking period.
        if ($sendtime >= $startblock && ($sendtime <= $endblock && $endblock > 0) ) {
            $$blocked{$msgid} = 'ON';
            $$numblocked ++;
        } else { 
            push @temp ,\@temp1;
        }
    }
    #default sort
    @temp = sort  {$a->[0] <=> $b->[0]} @temp;    
    if ($env{'form.sortedby'} eq "date"){
        @temp = sort  {$a->[0] <=> $b->[0]} @temp;    
    }
    if ($env{'form.sortedby'} eq "revdate"){
    	@temp = sort  {$b->[0] <=> $a->[0]} @temp; 
    }
    if ($env{'form.sortedby'} eq "user"){
	if ($get_received) {
	    @temp = sort  {lc($a->[7][0]) cmp lc($b->[7][0])} @temp;
	} else {
	    @temp = sort  {lc($a->[2])    cmp lc($b->[2])}    @temp;
	}
    }
    if ($env{'form.sortedby'} eq "revuser"){
	if ($get_received) {
	    @temp = sort  {lc($b->[7][0]) cmp lc($a->[7][0])} @temp;
	} else {
	    @temp = sort  {lc($b->[2])    cmp lc($a->[2])}    @temp;
	}
    }
    if ($env{'form.sortedby'} eq "domain"){
	if ($get_received) {
	    @temp = sort  {$a->[8][0] cmp $b->[8][0]} @temp;
	} else {
	    @temp = sort  {$a->[3]    cmp $b->[3]}    @temp;
	}
    }
    if ($env{'form.sortedby'} eq "revdomain"){
	if ($get_received) {
	    @temp = sort  {$b->[8][0] cmp $a->[8][0]} @temp;
	} else {
	    @temp = sort  {$b->[3]    cmp $a->[3]}    @temp;
	}
    }
    if ($env{'form.sortedby'} eq "subject"){
        @temp = sort  {lc($a->[1]) cmp lc($b->[1])} @temp;
    }
    if ($env{'form.sortedby'} eq "revsubject"){
        @temp = sort  {lc($b->[1]) cmp lc($a->[1])} @temp;
    }
    if ($env{'form.sortedby'} eq "course"){
        @temp = sort  {lc($a->[6]) cmp lc($b->[6])} @temp;
    }
    if ($env{'form.sortedby'} eq "revcourse"){
        @temp = sort  {lc($b->[6]) cmp lc($a->[6])} @temp;
    }
    if ($env{'form.sortedby'} eq "status"){
        @temp = sort  {$a->[4] cmp $b->[4]} @temp;
    }
    if ($env{'form.sortedby'} eq "revstatus"){
        @temp = sort  {$b->[4] cmp $a->[4]} @temp;
    }
    return @temp;
}

sub get_course_desc {
    my ($fromcid,$descriptions) = @_;
    my $description;
    if (!$fromcid) {
        return $description;
    } else {
        if (defined($$descriptions{$fromcid})) {
            $description = $$descriptions{$fromcid};
        } else {
            if (defined($env{'course.'.$fromcid.'.description'})) {
                $description = $env{'course.'.$fromcid.'.description'};
            } else {
                my %courseinfo=&Apache::lonnet::coursedescription($fromcid);
                $description = $courseinfo{'description'};
            }
            $$descriptions{$fromcid} = $description;
        }
        return $description;
    }
}

# ======================================================== Display all messages

sub disall {
    my ($r,$folder,$msgstatus)=@_;
    my %saveable = ('msgstatus' => 'scalar',
		    'sortedby'  => 'scalar',
		    'interdis'  => 'scalar',
		    );
    &Apache::loncommon::store_settings('user','mail',\%saveable);
    &Apache::loncommon::restore_settings('user','mail',\%saveable);
    $folder    ||= $env{'form.folder'};
    $msgstatus ||= $env{'form.msgstatus'};
    $env{'form.interdis'} ||= 20;

    $r->print(&folderlist($folder,$msgstatus));
    if ($folder eq 'critical') {
	&discrit($r);
    } else {
	&disfolder($r,$folder,$msgstatus);
    }
}

# ============================================================ Display a folder

sub disfolder {
    my ($r,$folder,$msgstatus)=@_;
    my %statushash = &get_msgstatus_types();
    my %blocked = ();
    my %setters = ();
    my $numblocked = 0;
    my ($startblock,$endblock) = &Apache::loncommon::blockcheck(\%setters,'com');
    my %lt = &Apache::lonlocal::texthash(
                      sede => 'Select a destination folder to which the messages will be moved.',
                      nome => 'No messages have been selected to apply ths action to.',
                      chec => 'Check the checkbox for at least one message.',  
    );
    my $jscript = &Apache::loncommon::check_uncheck_jscript();
    $r->print(<<ENDDISHEADER);
<script type="text/javascript">
// <![CDATA[
    $jscript

    function validate_checkedaction() {
        document.disall.markedaction.value = document.disall.checkedaction.options[document.disall.checkedaction.selectedIndex].value;
        if (document.disall.checkedaction.options[document.disall.checkedaction.selectedIndex].value == 'markedmove') {
            if (document.disall.movetofolder.options[document.disall.movetofolder.selectedIndex].value == "") {
                alert("$lt{'sede'}");
                return;
            } 
        }
        var checktotal = 0;
        if (document.forms.disall.delmark.length > 0) {
            for (var i=0; i<document.forms.disall.delmark.length; i++) {
                if (document.forms.disall.delmark[i].checked) {
                    checktotal ++;
                }
            }
        } else {
            if (document.forms.disall.delmark.checked) {
                checktotal ++;
            }
        }   
        if (checktotal == 0) {
            alert("$lt{'nome'}\\n$lt{'chec'}");
            return;
        }
        document.disall.submit();
    }
// ]]>
</script>
ENDDISHEADER

    my $fsqs='&amp;folder='.$folder;
    my @temp=&sortedmessages(\%blocked,$startblock,$endblock,\$numblocked,$folder,$msgstatus);
    my $totalnumber=$#temp+1;

    if ($totalnumber < 1) {
        $r->print('<p class="LC_info">');
        if ($msgstatus eq '') {
            $r->print(&mt('There are no messages in this folder.'));
        } elsif ($msgstatus eq 'replied') {
            $r->print(&mt('You have not replied to any messages in this folder.'));
        } else { 
            $r->print(&mt('There are no '.lc($statushash{$msgstatus}).' messages in this folder.'));
        }
        $r->print('</p>');
        if ($numblocked > 0) {
            $r->print(&blocked_in_folder($numblocked,$startblock,$endblock,
                                         \%setters));
        }
        return;
    }
    my $interdis = $env{'form.interdis'};
    my $number=int($totalnumber/$interdis);
    if ($totalnumber%$interdis == 0) {
	$number--; 
    }

    if (($startdis<0) || ($startdis>$number)) { $startdis=$number; }
    my $firstdis=$interdis*$startdis;
    if ($firstdis>$#temp) { $firstdis=$#temp-$interdis+1; }
    my $lastdis=$firstdis+$interdis-1;
    if ($lastdis>$#temp) { $lastdis=$#temp; }
    $r->print(&scrollbuttons($startdis,$number,$firstdis,$lastdis,$totalnumber,$msgstatus));
    $r->print('<form method="post" name="disall" action="/adm/email">'.
	      '<table class="LC_mail_list"><tr><th colspan="1">&nbsp;</th><th>');
    if ($env{'form.sortedby'} eq "revdate") {
	$r->print('<a href = "/adm/email?sortedby=date'.$fsqs.'">'.&mt('Date').'</a></th>');
    } else {
	$r->print('<a href = "/adm/email?sortedby=revdate'.$fsqs.'">'.&mt('Date').'</a></th>');
    }
    $r->print('<th>');
    if ($env{'form.sortedby'} eq "revuser") {
	$r->print('<a href = "/adm/email?sortedby=user'.$fsqs.'">'.&mt('Username').'</a>');
    } else {
	$r->print('<a href = "/adm/email?sortedby=revuser'.$fsqs.'">'.&mt('Username').'</a>');
    }
    $r->print('</th><th>');
    if ($env{'form.sortedby'} eq "revdomain") {
	$r->print('<a href = "/adm/email?sortedby=domain'.$fsqs.'">'.&mt('Domain').'</a>');
    } else {
	$r->print('<a href = "/adm/email?sortedby=revdomain'.$fsqs.'">'.&mt('Domain').'</a>');
    }
    $r->print('</th><th>');
    if ($env{'form.sortedby'} eq "revsubject") {
	$r->print('<a href = "/adm/email?sortedby=subject'.$fsqs.'">'.&mt('Subject').'</a>');
    } else {
    	$r->print('<a href = "/adm/email?sortedby=revsubject'.$fsqs.'">'.&mt('Subject').'</a>');
    }
    $r->print('</th><th>');
    if ($env{'form.sortedby'} eq "revcourse") {
        $r->print('<a href = "/adm/email?sortedby=course'.$fsqs.'">'.&mt('Course').'</a>');
    } else {
        $r->print('<a href = "/adm/email?sortedby=revcourse'.$fsqs.'">'.&mt('Course').'</a>');
    }
    $r->print('</th><th>');
    if ($env{'form.sortedby'} eq "revstatus") {
	$r->print('<a href = "/adm/email?sortedby=status'.$fsqs.'">'.&mt('Status').'</a></th>');
    } else {
     	$r->print('<a href = "/adm/email?sortedby=revstatus'.$fsqs.'">'.&mt('Status').'</a></th>');
    }
    $r->print("</tr>\n");

    my $suffix = &Apache::lonmsg::foldersuffix($folder);
    my $count = $firstdis;
    for (my $n=$firstdis;$n<=$lastdis;$n++) {
	my ($sendtime,$shortsubj,$fromname,$fromdomain,$status,$origID,
	    $description,$recv_name,$recv_domain)= 
		@{$temp[$n]};
	if (($status ne 'deleted') && defined($sendtime) && $sendtime!~/error/) {
            $count ++;
	    if ($status eq 'new') {
		$r->print('<tr class="LC_mail_new">');
	    } elsif ($status eq 'read') {
		$r->print('<tr class="LC_mail_read">');
	    } elsif ($status eq 'replied') {
		$r->print('<tr class="LC_mail_replied">'); 
	    } else {
		$r->print('<tr class="LC_mail_other">');
	    }
	    my ($dis_name,$dis_domain) = ($fromname,$fromdomain);
	    if ($folder eq 'sent') {
                if (defined($recv_name) && defined($recv_domain)) {
		    if (ref($recv_name) eq 'ARRAY' && 
                        ref($recv_domain) eq 'ARRAY') {
		        $dis_name   = join('<br />',@{$recv_name});
		        $dis_domain = join('<br />',@{$recv_domain});
                    }
		} else {
		    my $msg_id  = &unescape($origID);
		    my %message = &Apache::lonnet::get('nohist_email'.$suffix,
						       [$msg_id]);
		    my %content = &Apache::lonmsg::unpackagemsg($message{$msg_id});
                    if (ref($content{'recuser'}) eq 'ARRAY') {
		        $dis_name   = join('<br />',@{$content{'recuser'}});
                    }
                    if (ref($content{'recdomain'}) eq 'ARRAY') {
		        $dis_domain = join('<br />',@{$content{'recdomain'}});
                    }
		}
	    }
            my $localsenttime = &Apache::lonlocal::locallocaltime($sendtime);
	    $r->print('<td align="right"><span class="LC_nobreak">'.(($status eq 'new')?'<b>':'').
                      $count.'.'.(($status eq 'new')?'</b>':'').'&nbsp;'.
                      '<input type="checkbox" name="delmark"'. 
                      ' value="'.$origID.'" /></span></td>');
            foreach my $item ($localsenttime,$dis_name,$dis_domain,$shortsubj) {
                $r->print('<td>'.(($status eq 'new')?'<b>':'').
                          '<a href="/adm/email?display='.$origID.$sqs.'">'.
                          $item.(($status eq 'new')?'</b>':'').'</a></td>');
            }
            my $showstatus;
            my %statushash = &get_msgstatus_types();
            if ($status eq '') {
                $showstatus = '';
            } else {
                $showstatus = $statushash{$status};
            }
	    $r->print('<td>'.(($status eq 'new')?'<b>':'').$description.
                      (($status eq 'new')?'</b>':'').'</td><td>'.
                      (($status eq 'new')?'<b>':'').$showstatus.
                      (($status eq 'new')?'</b>':'').'</td></tr>'."\n");
	} elsif ($status eq 'deleted') {
# purge
	    my ($result,$msg) = 
		&movemsg(&unescape($origID),$folder,'trash');
	    
	}
    }   
    $r->print("</table>\n");
    $r->print('<table border="0" cellspacing="2" cellpadding="2">
 <tr>
  <td>'.
  '<input type="button" onclick="javascript:checkAll(document.disall.delmark)" value="'.&mt('Check All').'" /><br />'."\n".
  '<input type="button" onclick="javascript:uncheckAll(document.disall.delmark)" value="'.&mt('Uncheck All').'" />'."\n".
  '<input type="hidden" name="sortedby" value="'.$env{'form.sortedby'}.'" /></td><td>&nbsp;</td>'."\n".
  '<td align="center"><b>'.&mt('Action').'</b><br />'."\n".
  '  <select name="checkedaction">'."\n");

    if ($folder ne 'trash') {
        $r->print('    <option value="markeddel">'.&mt('Delete').'</option>'."\n");
    }
    if ($msgstatus ne 'read') {
        $r->print('    <option value="markedread">'.&mt('Mark Read').'</option>."\n"');
    }
    if ($msgstatus ne 'unread') {
        $r->print('    <option value="markedunread">'.&mt('Mark Unread').'</option>'."\n");
    }
    $r->print('   <option value="markedforward">'.&mt('Forward').'</option>'."\n");

    my %gotfolders = &Apache::lonmsg::get_user_folders();
    if (keys(%gotfolders) > 0) {
        $r->print('   <option value="markedmove">'.&mt('Move to Folder ->').
                  '</option>');
    }
    $r->print("\n".'</select></td>'."\n");

    if (keys(%gotfolders) > 0) {
        $r->print('<td align="center"><b>'.&mt('Destination folder').'<b><br />');
	my %userfolders;
        foreach my $key (keys(%gotfolders)) {
            $userfolders{$key} = $key;
        }
        $userfolders{''} = "";
        $r->print(&Apache::loncommon::select_form('','movetofolder',%userfolders).
                  '</td>');
    }
    $r->print('<td>&nbsp;</td><td>&nbsp;&nbsp;'.
              '<input type="button" name="go" value="'.&mt('Go').
              '" onclick="javascript:validate_checkedaction()"/></td>'."\n".
              '</tr></table>');
    my $postedstartdis=$startdis+1;
    $r->print('<input type="hidden" name="folder" value="'.$folder.'" /><input type="hidden" name="startdis" value="'.$postedstartdis.'" /><input type="hidden" name="interdis" value="'.$env{'form.interdis'}.'" /><input type="hidden" name="msgstatus" value="'.$msgstatus.'" /><input type="hidden" name="markedaction" value="" /></form>');
    if ($numblocked > 0) {
        $r->print(&blocked_in_folder($numblocked,$startblock,$endblock,
                                     \%setters));
    }
}

sub blocked_in_folder {
    my ($numblocked,$startblock,$endblock,$setters) = @_;
    my $beginblock = &Apache::lonlocal::locallocaltime($startblock);
    my $finishblock = &Apache::lonlocal::locallocaltime($endblock);
    my $output = '<br /><br />'.
                  &mt('[quant,_1,message is, messages are] not viewable because display of LON-CAPA messages sent to you by other students between [_2] and [_3] is currently being blocked because of online exams.',$numblocked,$beginblock,$finishblock);
    $output .= &Apache::loncommon::build_block_table($startblock,$endblock,
                                                     $setters);
    return $output;
}

# ============================================================== Compose output

sub compout {
    my ($r,$forwarding,$replying,$broadcast,$replycrit,$folder,$dismode,
        $multiforward)=@_;
    my $suffix=&Apache::lonmsg::foldersuffix($folder);
    my ($cdom,$cnum,$group,$refarg);
    if (exists($env{'form.group'})) {
        $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
        $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
        $group = $env{'form.group'};
        my $action = 'composing';
        $r->print(&groupmail_header($action,$group,$cdom,$cnum));
    } elsif ($broadcast eq 'individual') {
	&printheader($r,'/adm/email?compose=individual',
	     'Send a Message');
    } elsif ($broadcast) {
	&printheader($r,'/adm/email?compose=group',
	     'Broadcast Message');
    } elsif ($forwarding) {
	&Apache::lonhtmlcommon::add_breadcrumb
        ({href=>"/adm/email?display=".&escape($forwarding),
          text=>"Display Message"});
	&printheader($r,'/adm/email?forward='.&escape($forwarding),
	     'Forwarding a Message');
    } elsif ($replying) {
	&Apache::lonhtmlcommon::add_breadcrumb
        ({href=>"/adm/email?display=".&escape($replying),
          text=>"Display Message"});
	&printheader($r,'/adm/email?replyto='.&escape($replying),
	     'Replying to a Message');
    } elsif ($replycrit) {
	$r->print('<h3>'.&mt('Replying to a Critical Message').'</h3>');
	$replying=$replycrit;
    } elsif ($multiforward) {
        &Apache::lonhtmlcommon::add_breadcrumb
        ({href=>"/adm/email?folder=".&escape($folder),
          text=>"Display All Messages"});
        &printheader($r,'/adm/email?compose=multiforward',
             'Forwarding Multiple Messages');
        if ($multiforward > 1) {
            $r->print(&mt('Each of the[_1] [quant,_2,message] [_3]you checked'
                         .' will be forwarded to the recipient(s) you select below.','<b>',$multiforward,'</b>')
                    .'<br />');
        } else {
            $r->print(&mt('The message you checked will be forwarded to the recipient(s) you select below.').'<br />');
        }

    } else {
	&printheader($r,'/adm/email?compose=upload',
	     'Distribute from Uploaded File');
    }

    my $dispcrit='';
    my $dissub='';
    my $dismsg='';
    my $disbase='';
    my $attachrow;
    my $func1='Send'; # do not translate here!
    my %func2=( # do not translate here!
               'ma'  => 'Message',
               'msg' => 'Messages',
              );
    my %lt=&Apache::lonlocal::texthash('us'  => 'Username',
				       'do'  => 'Domain',
				       'ad'  => 'Additional Recipients',
                                       'rt'  => 'Reply to',
                                       'ar'  => 'Allow replies',
				       'sb'  => 'Subject',
				       'ca'  => 'Cancel',
                                       'gen' => 'Generate messages from a file',
                                       'gmt' => 'General message text',
                                       'tff' => 'The file format for the uploaded portion of the message is',
                                       'uas' => 'Upload and Send',
                                       'atta' => 'Attachment',
                                       'to'   => 'To:',
                                      );
    my %attachmax = (
                     text => &mt('(128 KB max size)'),
                     num  => 131072,
                    );
    if (!$forwarding && !$multiforward) {
        $attachrow = '<tr><td colspan="3"><b>'.$lt{'atta'}.'</b> '.$attachmax{'text'}.': <input type="file" name="attachment" /></td></tr>';
    }
    if (&Apache::lonnet::allowed('srm',$env{'request.course.id'})
	|| &Apache::lonnet::allowed('srm',$env{'request.course.id'}.
				    '/'.$env{'request.course.sec'})) {
         my $crstype = &Apache::loncommon::course_type();
	 my $crithelp = Apache::loncommon::help_open_topic("Course_Critical_Message");
         my $rsstxt;
         if (&Apache::loncommon::course_type() eq 'Community') {
             $rsstxt = &mt('Include in community RSS newsfeed');
         } else {
             $rsstxt = &mt('Include in course RSS newsfeed');
         }
         $dispcrit=
 '<span class="LC_nobreak"><label><input type="checkbox" name="critmsg" /> '.&mt('Send as critical message').'.</label>'.$crithelp.'&nbsp;&nbsp;'.&mt('Require return receipt?').'<label><input type="radio" name="sendbck" value="1" />'.&mt('Yes').'</label>&nbsp;&nbsp;<label><input type="radio" name="sendbck" value="" checked="checked" />'.&mt('No').'</label></span><br />'.
 '<label><input type="checkbox" name="permanent" /> '.
&mt('Send copy to permanent e-mail address (if known)').'</label><br />'.
'<label><input type="checkbox" name="rsspost" /> '.
		  $rsstxt.'</label><br />';
    }
    if ($broadcast ne 'group') {
        if (&Apache::lonnet::allowed('dff',$env{'request.course.id'}) ||
            &Apache::lonnet::allowed('dff',$env{'request.course.id'}.
                                     '/'.$env{'request.course.sec'})) {
            my $rectxt;
            if (&Apache::loncommon::course_type() eq 'Community') {
                $rectxt = &mt("Include in community's 'User records' for recipient(s)");
            } else {
                $rectxt = &mt("Include in course's 'User records' for recipient(s)");
            }
            $dispcrit.='<label>'.
                       '<input type="checkbox" name="courserecord" value="1" /> '.
                       $rectxt.
                       '</label><br />';
        }
    }

    my %message;
    my %content;
    my ($hasfloat,$broadcast_js,$sendmode,$can_grp_broadcast);
    my $defdom=$env{'user.domain'};
    if ($broadcast eq 'group') {
        my %access_status = (
                           active => 0,
                           previous => 0,
                           future => 0,
                      );
 
        if ($group eq '') {
            my $studentsel = &discourse(\%access_status);
            if ($studentsel) {
                if ($env{'environment.wysiwygeditor'} eq 'on') {
                    $r->print($studentsel);
                } else {
                    $r->print('<div class="LC_left_float">'.$studentsel.'</div>');
                }
                $hasfloat = 1;
            }
        } else {
            $can_grp_broadcast = &check_group_priv($group);
            if ($can_grp_broadcast) {
                $hasfloat = &disgroup($r,$cdom,$cnum,$group,\%access_status);
            }
        }
        if ($hasfloat) {
            $sendmode = '<input type="hidden" name="sendmode" value="group" />'."\n";
            $broadcast_js = qq|
<script type="text/javascript">
// <![CDATA[
function courseRecipients() {
|;
        foreach my $type (keys(%access_status)) {
            if ($access_status{$type}) {
                my $formname = $type.'users';
                if ($type eq 'active' && $group ne '') {
                    $broadcast_js .= qq|
                    document.compemail.groupmail.value = document.$formname.groupmail[document.$formname.groupmail.selectedIndex].value;
|;
                }
                $broadcast_js .= qq|
    if (typeof(document.$formname.selectedusers_forminput.length)=="undefined") {
        document.compemail.courserecips.value += '_&&&_'+document.$formname.selectedusers_forminput.value;
    } else {
        for (var i=0; i<document.$formname.selectedusers_forminput.length; i++) {
            if (document.$formname.selectedusers_forminput[i].checked) {
                document.compemail.courserecips.value += '_&&&_'+document.$formname.selectedusers_forminput[i].value;  
            }
        }
    }
                |;
            }
        }
        $broadcast_js .= qq|
    return;
}
// ]]>
</script>

|;
        }
    }
    if ($forwarding) {
	%message=&Apache::lonnet::get('nohist_email'.$suffix,[$forwarding]);
	%content=&Apache::lonmsg::unpackagemsg($message{$forwarding},$folder);
	$dispcrit.='<input type="hidden" name="forwid" value="'.
	    $forwarding.'" />';
	$func1='Forward'; # do not translate here!
	
	$dissub=&mt('Forwarding').': '.$content{'subject'};
	$dismsg=&mt('Forwarded message from').' '.
	    $content{'sendername'}.' '.&mt('at').' '.$content{'senderdomain'};
	if ($content{'baseurl'}) {
	    $disbase='<input type="hidden" name="baseurl" value="'.&escape($content{'baseurl'}).'" />';
	}
    }
    if ($replying) {
	%message=&Apache::lonnet::get('nohist_email'.$suffix,[$replying]);
	%content=&Apache::lonmsg::unpackagemsg($message{$replying},$folder);
	$dispcrit.='<input type="hidden" name="replyid" value="'.
	    $replying.'" />';
	$func1='Send Reply to'; # do not translate here!
	
	$dissub=&mt('Reply').': '.$content{'subject'};       
	$dismsg='> '.$content{'message'};
	$dismsg=~s/\r/\n/g;
	$dismsg=~s/\f/\n/g;
	$dismsg=~s/\n+/\n\> /g;
	if ($content{'baseurl'}) {
	    $disbase='<input type="hidden" name="baseurl" value="'.&escape($content{'baseurl'}).'" />';
	    if ($env{'user.adv'}) {
		$disbase.='<label><input type="checkbox" name="storebasecomment" />'.&mt('Save message for re-use').
		    '</label> <a href="/adm/email?showcommentbaseurl='.
		    &escape($content{'baseurl'}).'" target="comments">'.
		    &mt('Show re-usable messages').'</a><br />';
	    }
	}
        my $jscript = &Apache::loncommon::check_uncheck_jscript();
        $r->print(<<"ENDREPSCRIPT");
<script type="text/javascript">
// <![CDATA[
$jscript
// ]]>
</script>
ENDREPSCRIPT
    }
    my $citation=&displayresource(%content);
    my $onsubmit;
    if ($env{'form.recdom'}) { $defdom=$env{'form.recdom'}; }
    if ($env{'form.text'}) { $dismsg=$env{'form.text'}; }
    if ($env{'form.subject'}) { $dissub=$env{'form.subject'}; }
    if ($hasfloat) {
        if ($env{'environment.wysiwygeditor'} eq 'on') {
            $r->print($broadcast_js);
        } else {
            $r->print($broadcast_js.'<div class="LC_left_float">');
        }
        $onsubmit = ' onsubmit="javascript:courseRecipients();" ';
    }
    $r->print(
                '<form action="/adm/email"  name="compemail" method="post"'.
                ' enctype="multipart/form-data"'.$onsubmit.'>'."\n".
                '<input type="hidden" name="sendmail" value="on" />'."\n".
                '<table>');
    if (($broadcast eq 'group') && ($group ne '') && (!$can_grp_broadcast)) {
        $r->print(&recipient_input_row($cdom,%lt));
    }
    if (($broadcast ne 'group') && ($broadcast ne 'upload')) {
	if ($replying) {
            if ($content{'noreplies'}) {
                $r->print('<tr><td>'.&mt('This message was designated by the sender not to allow replies.').'</td></tr></table></form>');
                return;
            }
            $r->print('<tr><td colspan="3"><b>'.&mt('Replying to').'</b> ');
            if ($content{'replytoaddr'}) {
                my ($replytoname,$replytodom) = split(/:/,$content{'replytoaddr'});
                if ($replytoname ne '' && $replytodom ne '') {
                    $r->print(&Apache::loncommon::plainname($replytoname,
                                 $replytodom).' ('.$replytoname.':'.
                                 $replytodom.')');
                    $r->print('<input type="hidden" name="recuname" value="'.
                          $replytoname.'" />'.
                          '<input type="hidden" name="recdomain" value="'.
                          $replytodom.'" /></td></tr>');

                } else {
                    $r->print(&mt('The sender did not designate a reply to address for this message.').'</td></tr></table>');
                    return;
                }
            } else {
	        $r->print(&Apache::loncommon::aboutmewrapper(
							 &Apache::loncommon::plainname($content{'sendername'},$content{'senderdomain'}),$content{'sendername'},$content{'senderdomain'}).' ('.
		      $content{'sendername'}.':'.
		      $content{'senderdomain'}.')');
                $r->print('<input type="hidden" name="recuname" value="'.
		          $content{'sendername'}.'" />'.
		          '<input type="hidden" name="recdomain" value="'.
                          $content{'senderdomain'}.'" /></td></tr>');
            }
            if ($content{'recipid'}) {
                my %recips;
                &retrieve_recips('replying',\%content,\%recips);
                if (ref($recips{'to'}) eq 'ARRAY') {
                    if (@{$recips{'to'}} > 0) {
                        my $replyall;
                        if (@{$recips{'to'}} > 1) {
                            $replyall = qq|
 <span class="LC_nobreak">
      <input type="button" value="check all"
        onclick="javascript:checkAll(document.compemail.replying_to)" />
        &nbsp;&nbsp;
      <input type="button" value="uncheck all"
        onclick="javascript:uncheckAll(document.compemail.replying_to)" />
 </span>
|;
                        }
                        my $tolist = join(' ',@{$recips{'to'}});
                        $r->print('<tr><td colspan="3"><table><tr><td>'.&mt('[_1]Send reply[_2] to other recipients','<b>','</b>').':<br />'.$replyall.'</td><td>'.$tolist.'</td></tr></table></td></tr>');
                    }
                }
                if (ref($recips{'cc'}) eq 'ARRAY') {
                    if (@{$recips{'cc'}} > 0) {
                        my $replyall;
                        if (@{$recips{'cc'}} > 1) {
                            $replyall = qq|
 <span class="LC_nobreak">
      <input type="button" value="check all"
        onclick="javascript:checkAll(document.compemail.replying_cc)" />
        &nbsp;&nbsp;
      <input type="button" value="uncheck all"
        onclick="javascript:uncheckAll(document.compemail.replying_cc)" />
 </span>
|;
                        }
                        my $cclist = join(' ',@{$recips{'cc'}});
                        $r->print('<tr><td colspan="3"><table><tr><td>'.&mt('[_1]Cc[_2] to other copied recipients','<b>','</b>').':<br />'.$replyall.'</td><td>'.$cclist.'</td></tr></table></td></tr>');
                    }
                }
                if ($content{'group'} ne '') {
                    if (&check_group_priv($content{'group'})) {
                        if (ref($recips{'group_cc_broadcast'}) eq 'ARRAY') {
                            if (@{$recips{'group_cc_broadcast'}} > 0) {
                                my $replyall;
                                if (@{$recips{'group_cc_broadcast'}} > 1) {
                                    $replyall = qq|
 <span class="LC_nobreak">
      <input type="button" value="check all"
        onclick="javascript:checkAll(document.compemail.replying_groupcc)" />
        &nbsp;&nbsp;
      <input type="button" value="uncheck all"
        onclick="javascript:uncheckAll(document.compemail.replying_groupcc)" />
 </span>
|;
                                }
                                my $groupcclist = join(' ',@{$recips{'group_cc_broadcast'}});
                                $r->print('<tr><td colspan="3"><table><tr><td>'.&mt('[_1]Cc[_2] to other copied group members','<b>','</b>').':<br />'.$replyall.'</td><td>'.$groupcclist.'<input type="hidden" name="group" value="'.$content{'group'}.'" /><input type="hidden" name="sendmode" value="group" /><input type="hidden" name="groupmail" value="cc" /></td></tr></table></td></tr>');
                            }
                        }
                    }
                }
            }
	} else {
            $r->print(&recipient_input_row($defdom,%lt));
        }
    }
    my $latexHelp = &Apache::loncommon::helpLatexCheatsheet(undef,undef,1);
    my $wysiwyglink=&Apache::lonhtmlcommon::htmlareaselectactive('message').'<br />';
    my $subj_size;
    if ($multiforward) {
        $r->print(&additional_rec_row(\%lt));
        $r->print('<tr><td colspan="2">'.
                  &mt('Unless you choose otherwise:').'<ul><li>'.
        &mt("The subject in each forwarded message will be <i>'Forwarding:'</i> followed by the original subject.").'</li><li>'.
        &mt("The message itself will begin with a first line: <i>'Forwarded message from'</i> followed by the original sender's name.").'</li></ul></td></tr>');
        $func1='Forward'; # do not translate here!
        $dissub = &mt('Forwarding').': ';
        $subj_size = '10';
        my $extra = '&lt;'.&mt('original subject').'&gt;&nbsp;&nbsp;&nbsp;'.
                    '<input type="radio" name="showorigsubj" value="1" checked="checked" />'.&mt('Yes').'&nbsp;<input type="radio" name="showorigsubj" value="0" />'.&mt('No');
        $dismsg = &mt('Forwarded message from ').' ';
        my $sender = &mt("sender's name");
        $r->print(&msg_subject_row($dissub,\%lt,$subj_size,$extra));
        $r->print('<tr><td>'.&mt('Message begins with:').'</td><td><input type="text" name="msgheader" value="'.$dismsg.'" />&nbsp;'.$sender.'&nbsp;&nbsp;&nbsp;<input type="radio" name="showorigsender" value="1" checked="checked" />'.&mt('Yes').'&nbsp;<input type="radio" name="showorigsender" value="0" />'.&mt('No').'<input type="hidden" name="multiforward" value="'.$multiforward.'" /></td></tr>
</table>
<br /><table>
<tr><td align="left">'."\n".
$latexHelp."<br />\n".
&mt("Any new text to display before the text of the original messages:").'<br />'."\n".
'<textarea name="message" id="message" cols="80" rows="5" wrap="hard"></textarea>'.
$wysiwyglink);
        my @to_forward = &Apache::loncommon::get_env_multiple('form.delmark');
        foreach my $msg (@to_forward) {
            $r->print('<input type="hidden" name="delmark" value="'.$msg.'" />');
        }
        $r->print(&submit_button_row($folder,$dismode,&mt($func1.' '.$func2{'msg'}),
                                     \%lt));
    } elsif ($broadcast ne 'upload') {
        $subj_size = '50';
        $r->print(&additional_rec_row(\%lt));
        if (&Apache::lonnet::allowed('srm',$env{'request.course.id'})
            || &Apache::lonnet::allowed('srm',$env{'request.course.id'}.
                                        '/'.$env{'request.course.sec'})) {
            $r->print(&reply_to_row(\%lt));
        }
        $r->print(&msg_subject_row($dissub,\%lt,$subj_size));
        $r->print(<<"ENDCOMP");
$attachrow
</table><br />
$latexHelp<br />
<textarea name="message" id="message" cols="80" rows="15" wrap="hard">$dismsg
</textarea>$wysiwyglink
$sendmode
$dispcrit
$disbase
ENDCOMP
        $r->print(&submit_button_row($folder,$dismode,&mt($func1.' '.$func2{'ma'}),
                                     \%lt,$hasfloat,$group));
        $r->print($citation);
        if (exists($env{'form.ref'})) {
            $r->print('<input type="hidden" name="ref" value="'.
                      $env{'form.ref'}.'" />');
        }
        if (exists($env{'form.group'})) {
            $r->print('<input type="hidden" name="group" value="'.
                      $env{'form.group'}.'" />');
        }
    } else { # $broadcast is 'upload'
	$r->print(<<ENDBLOCK);
<input type="hidden" name="sendmode" value="upload" />
<input type="hidden" name="send" value="on" />
<h3>$lt{'gen'}</h3>
<p>
Subject: <input type="text" size="50" name="subject" />
</p>
<p>$lt{'gmt'}:<br />
<textarea name="message" id="message" cols="60" rows="10" wrap="hard">$dismsg
</textarea>$wysiwyglink</p>
<p>
$lt{'tff'}:
ENDBLOCK
       $r->print('
<pre>'."\n".
&mt('username1:domain1: text')."\n".
&mt('username2:domain2: text')."\n".
&mt('username3:domain1: text')."\n".
'</pre>
</p>
<p>
'.&mt('The messages will be assembled from all lines with the respective'."\n".'<tt>username:domain</tt>, and appended to the general message text.'));
        $r->print(<<ENDUPLOAD);
</p>
<p>
<input type="file" name="upfile" size="40" /></p><p>
$dispcrit
<input type="submit" value="$lt{'uas'}" /></p>
ENDUPLOAD
    }
    if ($env{'form.displayedcrit'}) {
	$r->print('<input type="hidden" name="displayedcrit" value="true" />');
    }
    $r->print('</form>');
    if ($hasfloat) {
        unless($env{'environment.wysiwygeditor'} eq 'on') {
            $r->print('</div><div class="LC_clear_float_footer"></div>');
        }
    }
    $r->print(&generate_preview_form);
}

sub check_group_priv {
    my ($group) = @_;
    my $cid = $env{'request.course.id'};
    my $sec = $env{'request.course.sec'};
    return if !$cid;
    my $can_broadcast = &Apache::lonnet::allowed('sgb',$cid.'/'.$group);
    my $viewgrps = &Apache::lonnet::allowed('vcg',$cid.($sec?'/'.$sec:''));
    my $editgrps = &Apache::lonnet::allowed('mdg',$cid.($sec?'/'.$sec:''));
    if ($viewgrps || $editgrps || $can_broadcast) {
        return 1;
    }
    return;
}

sub recipient_input_row {
    my ($dom,%lt) = @_;
    my $domform = &Apache::loncommon::select_dom_form($dom,'recdomain');
    my $selectlink=
      &Apache::loncommon::selectstudent_link('compemail','recuname',
                                             'recdomain');
    my $output = <<"ENDREC";
<tr><td colspan="3"><span class="LC_nobreak"><b>$lt{'to'}</b> $lt{'us'}:&nbsp;<input type="text" size="12" name="recuname" value="$env{'form.recname'}" />&nbsp;$lt{'do'}:&nbsp;$domform&nbsp;&nbsp;$selectlink</span></td></tr>
ENDREC
    return $output;
}

sub reply_to_row {
    my ($lt) = @_;
    my $radioyes = &mt('Yes');
    my $radiono = &mt('No');
    my $output = <<"ENDREP";
<tr><td colspan="3"><span class="LC_nobreak"><b>$lt->{'ar'}</b>:<label><input type="radio" name="can_reply" value="Y" checked="checked" />$radioyes</label>&nbsp;&nbsp;<label><input type="radio" name="can_reply" value="N" />$radiono</label></span>&nbsp;&nbsp;&nbsp;&nbsp;<span class="LC_nobreak">$lt->{'rt'}:&nbsp;<input type="text" size="25" name="reply_to_addr" value="$env{'user.name'}:$env{'user.domain'}" /></span></td></tr>
ENDREP
    return $output;
}

sub additional_rec_row {
    my ($lt) = @_;
    my $cc = &mt('Cc:');
    my $bcc = &mt('Bcc:');
    my $exmpl = &mt('username:domain,username:domain,...'); 
    my $output = <<"ENDADD";
<tr><td colspan="3"><fieldset id="LC_additionalrecips"><legend><b>$lt->{'ad'}</b> <tt>($exmpl)</tt>:</legend><table>
<tr><td>&nbsp;</td><td>$lt->{'to'}</td><td><input type="text" size="50" name="additionalrec_to" /></td></tr>
<tr><td>&nbsp;</td><td>$cc</td><td><input type="text" size="50" name="additionalrec_cc" /></td></tr> 
<tr><td>&nbsp;</td><td>$bcc</td><td><input type="text" size="50" name="additionalrec_bcc" /></td></tr></table></fieldset></td></tr>
ENDADD
    return $output;
}

sub submit_button_row {
    my ($folder,$dismode,$sendtext,$lt,$is_crsform,$group) = @_;
    my $pre=&mt("Show Preview and Check Spelling");
    my $value=&mt('Send');
    my $prevbutton = '<input type="button" name="preview" value="'.$pre.'" onclick="if (typeof(document.compemail.onsubmit)=='."'function'".') {document.compemail.onsubmit();};document.preview.comment.value=document.compemail.message.value;document.preview.subject.value=document.compemail.subject.value;document.preview.submit();" />';
    my $output = qq|
<input type="hidden" name="folder" value="$folder" />
<input type="hidden" name="dismode" value="$dismode" />|;
    if ($is_crsform) {
        $output .= '<input type="hidden" name="courserecips" value="" />'."\n";
        if ($group ne '') {
            $output .= '<input type="hidden" name="groupmail" value="" />'."\n";
        }
    }
    $output .= qq|
<table><tr><td align="left">
<input type="submit" name="send" value="$value" title="$sendtext" />
<input type="submit" name="cancel" value="$lt->{'ca'}" />
</td><td width="60">&nbsp;</td><td align="right">$prevbutton</td></tr></table>
|;
    return $output;
}

sub msg_subject_row {
    my ($dissub,$lt,$subj_size,$extra) = @_;
    my $output = '<tr><td colspan="3"><b>'.$lt->{'sb'}.'</b>:&nbsp;<input type="text" size="'.
                 $subj_size.'" name="subject" value="'.$dissub.'" />'.$extra.
                 '</td></tr>';
    return $output;
}

sub generate_preview_form {
    my $prevbutton = (<<ENDPREVIEW);
<form name="preview" action="/adm/feedback?preview=1" method="post" target="preview">
<input type="hidden" name="subject" />
<input type="hidden" name="comment" />
</form>
ENDPREVIEW
}

# ---------------------------------------------------- Display all face to face

sub retrieve_instructor_comments {
    my ($user,$domain)=@_;
    my $target=$env{'form.grade_target'};
    if (! $env{'request.course.id'}) { return; }
    if (! &Apache::lonnet::allowed('dff',$env{'request.course.id'})
	&& ! &Apache::lonnet::allowed('dff',$env{'request.course.id'}.
				      '/'.$env{'request.course.sec'})) {
	return;
    }
    my %records=&Apache::lonnet::dump('nohist_email',
			 $env{'course.'.$env{'request.course.id'}.'.domain'},
			 $env{'course.'.$env{'request.course.id'}.'.num'},
                         '%255b'.$user.'%253a'.$domain.'%255d');
    my $result='';
    foreach my $key (sort(keys(%records))) {
        my %content=&Apache::lonmsg::unpackagemsg($records{$key});
        next if ($content{'senderdomain'} eq '');
        next if ($content{'subject'} !~ /^Record/);
	# &Apache::lonfeedback::newline_to_br(\$content{'message'});
	$result.='Recorded by '.
            $content{'sendername'}.':'.$content{'senderdomain'}."\n";
        $result.=
            &Apache::lontexconvert::msgtexconverted($content{'message'})."\n";
     }
    return $result;
}

sub disfacetoface {
    my ($r,$user,$domain)=@_;
    my $target=$env{'form.grade_target'};
    unless ($env{'request.course.id'}) { return; }
    if  (!&Apache::lonnet::allowed('dff',$env{'request.course.id'})
	 && ! &Apache::lonnet::allowed('dff',$env{'request.course.id'}.
				       '/'.$env{'request.course.sec'})) {
	$r->print(&mt('Not allowed'));
	return;
    }
    my %records=&Apache::lonnet::dump('nohist_email',
			 $env{'course.'.$env{'request.course.id'}.'.domain'},
			 $env{'course.'.$env{'request.course.id'}.'.num'},
                         '%255b'.$user.'%253a'.$domain.'%255d');
    my $result='';
    foreach my $key (sort(keys(%records))) {
        my %content=&Apache::lonmsg::unpackagemsg($records{$key});
        next if ($content{'senderdomain'} eq '');
        if (!&Apache::lonfeedback::contains_block_html($content{'message'})) {
            &Apache::lonfeedback::newline_to_br(\$content{'message'});
        }
        if ($content{'subject'}=~/^Record/) {
	    $result.='<h3>'.&mt('Record').'</h3>';
        } elsif ($content{'subject'}=~/^Broadcast/) {
            $result .='<h3>'.&mt('Broadcast Message').'</h3>';
            if ($content{'subject'}=~/^Broadcast\./) {
                if (defined($content{'coursemsgid'})) {
                    my $crsmsgid = &escape($content{'coursemsgid'});
                    my $broadcast_message = &general_message($crsmsgid);
                    $content{'message'} = '<b>'.&mt('Subject').': '.$content{'message'}.'</b><br />'.$broadcast_message;
                } else {
                    %content=&Apache::lonmsg::unpackagemsg($content{'message'});
                    $content{'message'} =
                    '<b>'.&mt('Subject').': '.$content{'subject'}.'</b><br />'.
                    $content{'message'};
                }
            }
        } elsif ($content{'subject'}=~/^Archive/) {
            $result.='<h3>'.&mt('Archived Message').'</h3>';
            if (defined($content{'coursemsgid'})) {
                my $crsmsgid = &escape($content{'coursemsgid'});
                my $archive_message = &general_message($crsmsgid);
                $content{'message'} = '<b>'.&mt('Subject').': '.$content{'message'}.'</b><br />'.$archive_message;
            } else {
                %content=&Apache::lonmsg::unpackagemsg($content{'message'});
                $content{'message'} =
                '<b>'.&mt('Subject').': '.$content{'subject'}.'</b><br
'.
                $content{'message'};
            }
        } else {
            $result.='<h3>'.&mt('Critical Message').'</h3>';
            if (defined($content{'coursemsgid'})) {
                my $crsmsgid=&escape($content{'coursemsgid'});
                my $critical_message = &general_message($crsmsgid);
                $content{'message'} = '<b>'.&mt('Subject').': '.$content{'message'}.'</b><br />'.$critical_message;
            } else {
                %content=&Apache::lonmsg::unpackagemsg($content{'message'});
                $content{'message'}=
                '<b>'.&mt('Subject').': '.$content{'subject'}.'</b><br />'.
		$content{'message'};
            }
        }
        $result.=&mt('By').': <b>'.
&Apache::loncommon::aboutmewrapper(
 &Apache::loncommon::plainname($content{'sendername'},$content{'senderdomain'}),$content{'sendername'},$content{'senderdomain'}).'</b> ('.
$content{'sendername'}.':'.
            $content{'senderdomain'}.') '.$content{'time'}.
            '<br /><pre>'.
              &Apache::lontexconvert::msgtexconverted($content{'message'}).
	      '</pre>';
     }
    # Check to see if there were any messages.
    if ($result eq '') {
        my $lctype = &mt(lc(&Apache::loncommon::course_type()));
	if ($target ne 'tex') { 
	    $r->print("<p><b>".&mt('No notes, face-to-face discussion records, critical messages, or broadcast messages in this [_1].',$lctype)."</b></p>");
	} else {
	    $r->print('\textbf{'.&mt('No notes, face-to-face discussion records, critical messages or broadcast messages in this [_1].',$lctype).'}\\\\');
	}
    } else {
       $r->print($result);
    }
}

sub general_message {
    my ($crsmsgid) = @_;
    my %general_content;
    if ($crsmsgid) { 
        my %course_content = &Apache::lonnet::get('nohist_email',[$crsmsgid],
                           $env{'course.'.$env{'request.course.id'}.'.domain'},
                           $env{'course.'.$env{'request.course.id'}.'.num'});
        %general_content = &Apache::lonmsg::unpackagemsg($course_content{$crsmsgid});
    }
    return $general_content{'message'};
}

# ---------------------------------------------------------------- Face to face

sub facetoface {
    my ($r,$stage)=@_;
    if (!&Apache::lonnet::allowed('dff',$env{'request.course.id'})
	&& ! &Apache::lonnet::allowed('dff',$env{'request.course.id'}.
				      '/'.$env{'request.course.sec'})) {
	$r->print(&mt('Not allowed'));
	return;
    }
    my $crstype = &Apache::loncommon::course_type();
    my $leaders = ($crstype eq 'Community') ? 'coordinators and leaders'
                                        : 'faculty and staff';
    &printheader($r,
		 '/adm/email?recordftf=query',
		 "User Notes, Face-to-Face, Critical Messages, Broadcast Messages, Archived Messages");
# from query string

    if ($env{'form.recname'}) { $env{'form.recuname'}=$env{'form.recname'}; }
    if ($env{'form.recdom'}) { $env{'form.recdomain'}=$env{'form.recdom'}; }

    my $defdom=$env{'user.domain'};
# already filled in
    if ($env{'form.recdomain'}) { $defdom=$env{'form.recdomain'}; }
# generate output
    my $domform = &Apache::loncommon::select_dom_form($defdom,'recdomain');
    my $stdbrws = &Apache::loncommon::selectstudent_link
	('stdselect','recuname','recdomain');
    my %lt=&Apache::lonlocal::texthash('user' => 'Username',
				       'dom' => 'Domain',
				       'head' => "User Notes, Records of Face-To-Face Discussions, Critical Messages, Broadcast Messages and Archived Messages in $crstype",
				       'subm' => 'Retrieve discussion and message records',
				       'newr' => 'New Record (record is visible to '.lc($crstype).' '.$leaders.')',
				       'post' => 'Post this Record');

    $r->print('<h2>'.$lt{'head'}.'</h2>'
             .'<form method="post" action="/adm/email" name="stdselect">'
             .'<input type="hidden" name="recordftf" value="retrieve" />'
             .&Apache::lonhtmlcommon::start_pick_box()
             .&Apache::lonhtmlcommon::row_title($lt{'user'})
             .'<input type="text" size="12" name="recuname" value="'.$env{'form.recuname'}.'" />'
             .' '.$stdbrws
             .&Apache::lonhtmlcommon::row_closure()
             .&Apache::lonhtmlcommon::row_title($lt{'dom'})
             .$domform
             .&Apache::lonhtmlcommon::row_closure(1)
             .&Apache::lonhtmlcommon::end_pick_box()
             .'<br />'
             .'<input type="submit" value="'.$lt{'subm'}.'" />'
             .'</form>'
    );

    if (($stage ne 'query') &&
        ($env{'form.recdomain'}) && ($env{'form.recuname'})) {
        chomp($env{'form.newrecord'});
        if ($env{'form.newrecord'}) {
	    &Apache::lonmsg::store_instructor_comment($env{'form.newrecord'},
						      $env{'form.recuname'},
						      $env{'form.recdomain'});
        }
        my $aboutmelink=&Apache::loncommon::aboutmewrapper(
                            &Apache::loncommon::plainname($env{'form.recuname'}
                                                         ,$env{'form.recdomain'})
                           ,$env{'form.recuname'},$env{'form.recdomain'});
        $r->print('<hr />'
                 .'<h2>'
                 .&mt('Discussion and message records for [_1] ([_2])'
                     ,$aboutmelink
                     ,$env{'form.recuname'}.':'.$env{'form.recdomain'})
                 .'<h2>'
       );
        &disfacetoface($r,$env{'form.recuname'},$env{'form.recdomain'});
	$r->print(<<ENDRHEAD);
<form method="post" action="/adm/email">
<input name="recdomain" value="$env{'form.recdomain'}" type="hidden" />
<input name="recuname" value="$env{'form.recuname'}" type="hidden" />
ENDRHEAD
        $r->print(<<ENDBFORM);
<hr />$lt{'newr'}<br />
<textarea name="newrecord" cols="80" rows="10" wrap="hard"></textarea>
<br />
<input type="hidden" name="recordftf" value="post" />
<input type="submit" value="$lt{'post'}" />
</form>
ENDBFORM
    }
}

# ----------------------------------------------------------- Blocking during exams

sub examblock {
    my ($r,$action) = @_;
    unless ($env{'request.course.id'}) { return;}
    if (!&Apache::lonnet::allowed('dcm',$env{'request.course.id'})
	&& ! &Apache::lonnet::allowed('dcm',$env{'request.course.id'}.
				      '/'.$env{'request.course.sec'})) {
	$r->print('Not allowed');
	return;
    }
    my $usertype;
    my $crstype = &Apache::loncommon::course_type();
    if ($crstype eq 'Community') {
        $usertype = 'members';
    } else {
	$usertype = 'students';
    }
    my $lctype = lc($crstype);
    my %lt=&Apache::lonlocal::texthash(
            'comb' => 'Communication Blocking',
            'cbds' => 'Communication blocking during scheduled exams',
            'desc' => "You can use communication blocking to prevent $usertype enrolled in this $lctype from displaying LON-CAPA messages sent by other $usertype during an online exam. As blocking of communication could potentially interrupt legitimate communication between $usertype who are also both enrolled in a different LON-CAPA course or community, please be careful that you select the correct start and end times for your scheduled exam when setting or modifying these parameters.",
             'mecb' => 'Modify existing communication blocking periods',
             'ncbc' => 'No communication blocks currently saved',
             'stor' => 'Save',
    );

    my %ltext = &Apache::lonlocal::texthash(
            'dura' => 'Duration',
            'setb' => 'Set by',
            'even' => 'Event',
            'blck' => 'Blocked?',
            'actn' => 'Action',
            'star' => 'Start',
            'endd' => 'End'
    );

    &printheader($r,'/adm/email?block=display',$lt{'comb'});
    $r->print('<h3>'.$lt{'cbds'}.'</h3>');

    if ($action eq 'store') {
        &blockstore($r);
    }

    $r->print($lt{'desc'}.'<br /><br />
               <form name="blockform" method="post" action="/adm/email?block=store">
             ');

    $r->print('<h4>'.$lt{'mecb'}.'</h4>');
    my %records = ();
    my $blockcount = 0;
    my $parmcount = 0;
    &get_blockdates(\%records,\$blockcount);
    if ($blockcount > 0) {
        $parmcount = &display_blocker_status($r,\%records,\%ltext);
    } else {
        $r->print($lt{'ncbc'}.'<br /><br />');
    }
    &display_addblocker_table($r,$parmcount,\%ltext);
    my $end_page=&Apache::loncommon::end_page();
    $r->print(<<"END");
<br />
<input type="hidden" name="blocktotal" value="$blockcount" />
<input type ="submit" value="$lt{'stor'}" />
</form>
$end_page
END
    return;
}

sub blockstore {
    my $r = shift;
    my %lt=&Apache::lonlocal::texthash(
            'tfcm' => 'The following changes were made',
            'ncwm' => 'No changes were made.' 
    );
    my %adds = ();
    my %removals = ();
    my %cancels = ();
    my $modtotal = 0;
    my $canceltotal = 0;
    my $addtotal = 0;
    my %blocking = ();
    $r->print('<h3>'.$lt{'head'}.'</h3>');
    foreach my $envkey (keys(%env)) {
        if ($envkey =~ m/^form\.modify_(\d+)$/) {
            $adds{$1} = $1;
            $removals{$1} = $1;
            $modtotal ++;
        } elsif ($envkey =~ m/^form\.cancel_(\d+)$/) {
            $cancels{$1} = $1;
            unless ( defined($removals{$1}) ) {
                $removals{$1} = $1;
                $canceltotal ++;
            }
        } elsif ($envkey =~ m/^form\.add_(\d+)$/) {
            $adds{$1} = $1;
            $addtotal ++;
        } 
    }

    foreach my $key (keys(%removals)) {
        my $hashkey = $env{'form.key_'.$key};
        &Apache::lonnet::del('comm_block',["$hashkey"],
                         $env{'course.'.$env{'request.course.id'}.'.domain'},
                         $env{'course.'.$env{'request.course.id'}.'.num'}
                         );
    }
    foreach my $key (keys(%adds)) {
        unless ( defined($cancels{$key}) ) {
            my ($newstart,$newend) = &get_dates_from_form($key);
            my $newkey = $newstart.'____'.$newend;
            my $blocktypes = &get_block_choices($key);
            $blocking{$newkey} = {
                          setter => $env{'user.name'}.':'.$env{'user.domain'},
                          event  => &escape($env{'form.title_'.$key}),
                          blocks => $blocktypes,
                        };
        }
    }
    if ($addtotal + $modtotal > 0) {
        &Apache::lonnet::put('comm_block',\%blocking,
                     $env{'course.'.$env{'request.course.id'}.'.domain'},
                     $env{'course.'.$env{'request.course.id'}.'.num'}
                     );
    }
    my $chgestotal = $canceltotal + $modtotal + $addtotal;
    if ($chgestotal > 0) {
        $r->print($lt{'tfcm'}.'<ul>');
        if ($canceltotal > 0) {
            $r->print('<li>'.&mt('[quant,_1,communication blocking period was,communication blocking periods were] removed.',$canceltotal).'</li>');
        }
        if ($modtotal > 0) {
            $r->print('<li>'.&mt('[quant,_1,communication blocking period was,communication blocking periods were] modified.',$modtotal).'</li>');
        }
        if ($addtotal > 0) {
            $r->print('<li>'.&mt('[quant,_1,communication blocking period was,communication blocking periods were] added.',$addtotal).'</li>');
        }
        $r->print('</ul>');
    } else {
        $r->print($lt{'ncwm'});
    }
    $r->print('<br />');
    return;
}

sub get_dates_from_form {
    my $item = shift;
    my $startdate = &Apache::lonhtmlcommon::get_date_from_form('startdate_'.$item);
    my $enddate   = &Apache::lonhtmlcommon::get_date_from_form('enddate_'.$item);
    return ($startdate,$enddate);
}

sub get_blockdates {
    my ($records,$blockcount) = @_;
    $$blockcount = 0;
    %{$records} = &Apache::lonnet::dump('comm_block',
                         $env{'course.'.$env{'request.course.id'}.'.domain'},
                         $env{'course.'.$env{'request.course.id'}.'.num'}
                         );
    $$blockcount = keys(%{$records});

    if ((keys(%{$records}))[0] =~ /^error: 2 /) {
	$records = {};
	$$blockcount = 0;
    }
}

sub get_block_choices {
    my $item = shift;
    my $blocklist;
    my ($typeorder,$types) = &blocktype_text();
    foreach my $type (@{$typeorder}) {
        if ($env{'form.'.$type.'_'.$item}) {
            $blocklist->{$type} = 'on'; 
        } else {
            $blocklist->{$type} = 'off';
        }
    }
    return $blocklist;
}

sub display_blocker_status {
    my ($r,$records,$ltext) = @_;
    my $parmcount = 0;
  
    my %lt = &Apache::lonlocal::texthash(
        'modi' => 'Modify',
        'canc' => 'Cancel',
    );
    my ($typeorder,$types) = &blocktype_text();
    $r->print(&Apache::loncommon::start_data_table());
    $r->print(<<"END");
  <tr>
    <th>$ltext->{'dura'}</th>
    <th>$ltext->{'setb'}</th>
    <th>$ltext->{'even'}</th>
    <th>$ltext->{'blck'}</th>
    <th>$ltext->{'actn'}</th>
  </tr>
END
    foreach my $record (sort(keys(%{$records}))) {
        my $onchange = 'onFocus="javascript:window.document.forms['.
                       "'blockform'].elements['modify_".$parmcount."'].".
                       'checked=true;"';
        my ($start,$end) = split(/____/,$record);
        my $startform = &Apache::lonhtmlcommon::date_setter('blockform','startdate_'.$parmcount,$start,$onchange);
        my $endform = &Apache::lonhtmlcommon::date_setter('blockform','enddate_'.$parmcount,$end,$onchange);
	
	my ($setuname,$setudom,$title,$blocks) = 
	    &Apache::loncommon::parse_block_record($$records{$record});
	$title = &HTML::Entities::encode($title,'"<>&');
        my $settername = 
           &Apache::loncommon::aboutmewrapper(
                           &Apache::loncommon::plainname($setuname,$setudom),
                           $setuname,$setudom);
        $r->print(&Apache::loncommon::start_data_table_row());
        $r->print(<<"END");
        <td>$ltext->{'star'}:&nbsp;$startform<br />$ltext->{'endd'}:&nbsp;&nbsp;$endform</td>
        <td>$settername</td>
        <td><input type="text" name="title_$parmcount" size="15" value="$title" /><input type="hidden" name="key_$parmcount" value="$record" /></td>
        <td>
END
        foreach my $block (@{$typeorder}) {
            my $blockstatus = '';
            if ($blocks->{$block} eq 'on') {
                $blockstatus = 'checked="checked"';
            }
            $r->print('<span class="LC_nobreak"><label><input type="checkbox" name="'.$block.'_'.$parmcount.'" '.$blockstatus.' value="1" />'.$types->{$block}.'</label></span><br />');
        }
        $r->print(<<"END");
        </td>
        <td><span class="LC_nobreak"><label>
        <input type="checkbox" name="modify_$parmcount" />$lt{'modi'}
        </label></span><br /><span class="LC_nobreak">
        <label>
        <input type="checkbox" name="cancel_$parmcount" />$lt{'canc'}
        </label></span>
END
        $r->print(&Apache::loncommon::end_data_table_row());
        $parmcount++;
    }
    $r->print(<<"END");
</table>
<br />
<br />
END
    return $parmcount;
}

sub display_addblocker_table {
    my ($r,$parmcount,$ltext) = @_;
    my $start = time;
    my $end = $start + (60 * 60 * 2); #Default is an exam of 2 hours duration.
    my $onchange = 'onFocus="javascript:window.document.forms['.
                   "'blockform'].elements['add_".$parmcount."'].".
                   'checked=true;"';
    my $startform = &Apache::lonhtmlcommon::date_setter('blockform','startdate_'.$parmcount,$start,$onchange);
    my $endform = &Apache::lonhtmlcommon::date_setter('blockform','enddate_'.$parmcount,$end,$onchange);
    my %lt = &Apache::lonlocal::texthash(
        'addb' => 'Add block',
        'exam' => 'e.g., Exam 1',
        'addn' => 'Add new communication blocking periods'
    );
    my ($typeorder,$types) = &blocktype_text();
    $r->print(<<"END");
<h4>$lt{'addn'}</h4> 
END
    $r->print(&Apache::loncommon::start_data_table());
    $r->print(<<"END");
   <tr>
     <th>$ltext->{'dura'}</th>
     <th>$ltext->{'even'} $lt{'exam'}</th>
     <th>$ltext->{'blck'}</th>
     <th>$ltext->{'actn'}</th>
   </tr>
END
    $r->print(&Apache::loncommon::start_data_table_row());
    $r->print(<<"END");
     <td>$ltext->{'star'}:&nbsp;$startform<br />$ltext->{'endd'}:&nbsp;&nbsp;$endform</td>
     <td><input type="text" name="title_$parmcount" size="15" value="" /></td>
     <td>
END
    foreach my $block (@{$typeorder}) {
        $r->print('<span class="LC_nobreak"><label><input type="checkbox" name="'.$block.'_'.$parmcount.'" value="1" />'.$types->{$block}.'</label></span><br />');
     }
     $r->print(<<"END");
     </td>
     <td><span class="LC_nobreak"><label>
     <input type="checkbox" name="add_$parmcount" value="1" />$lt{'addb'}
     </label></span></td>
END
    $r->print(&Apache::loncommon::end_data_table_row());
    $r->print(&Apache::loncommon::end_data_table());
    return;
}

sub blocktype_text {
    my %types = &Apache::lonlocal::texthash(
        'com' => 'Messaging',
        'chat' => 'Chat Room',
        'boards' => 'Discussion',
        'port' => 'Portfolio',
        'groups' => 'Groups',
        'blogs' => 'Blogs',
    );
    my $typeorder = ['com','chat','boards','port','groups','blogs'];
    return ($typeorder,\%types);
}

# ----------------------------------------------------------- Display a message

sub displaymessage {
    my ($r,$msgid,$folder,$msgstatus)=@_;
    my $suffix=&Apache::lonmsg::foldersuffix($folder);
    my %blocked = ();
    my %setters = ();
    my $numblocked = 0;
    my $crstype = &Apache::loncommon::course_type();

# info to generate "next" and "previous" buttons and check if message is blocked
    my ($startblock,$endblock) = &Apache::loncommon::blockcheck(\%setters,'com');
    my @messages=&sortedmessages(\%blocked,$startblock,$endblock,\$numblocked,$folder,$msgstatus);
    if ( $blocked{$msgid} eq 'ON' ) {
        &printheader($r,'/adm/email',&mt('Display a Message'));
        $r->print(&mt('You attempted to display a message that is currently blocked because you are enrolled in one or more courses for which there is an ongoing online exam.'));
        $r->print(&Apache::loncommon::build_block_table($startblock,$endblock,\%setters));
        return;
    }
    if ($msgstatus eq '') {
        &statuschange($msgid,'read',$folder);
    }
    my %message=&Apache::lonnet::get('nohist_email'.$suffix,[$msgid]);
    my %content=&Apache::lonmsg::unpackagemsg($message{$msgid});
    my $counter=0;
    my $escmsgid=&escape($msgid);
    foreach (@messages) {
	if ($_->[5] eq $escmsgid){
	    last;
	}
	$counter++;
    }

    my $see_anonymous;
    my $from_student = 0;
    if ($env{'request.course.id'} eq $content{'courseid'}) {
	my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
	my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
	my $username = $content{'sendername'}.':'.$content{'senderdomain'};
	my %classlist_entry =
	    &Apache::lonnet::get('classlist',[$username],$cdom,$cnum);
	if (exists($classlist_entry{$username})) {
	    $from_student = 1;
	    $see_anonymous = &Apache::lonnet::allowed('rin',$env{'request.course.id'}.($env{'request.course.sec'}?'/'.$env{'request.course.sec'}:''));
	}
    }


    my $number_of_messages = scalar(@messages); #subtract 1 for last index
# start output
    &printheader($r,'/adm/email?display='.&escape($msgid),'Display a Message','',$content{'baseurl'});
    my %courseinfo=&Apache::lonnet::coursedescription($content{'courseid'});

# Functions and Actions
    my $li_start='<span class="LC_nobreak">&bull; ';
    my $li_end='</span> ';

# Prepare available functions
    my @functionlist;
    if (!$content{'noreplies'}) {
        push(@functionlist,'<a href="/adm/email?replyto='.&escape($msgid).$sqs.'">'
                     .&mt('Reply')
                     .'</a>');
    }
    push(@functionlist,'<a href="/adm/email?forward='.&escape($msgid).$sqs.'">'
                  .&mt('Forward')
                  .'</a>');
    push(@functionlist,'<a href="/adm/email?markunread='.&escape($msgid).$sqs.'">'
                  .&mt('Mark Unread')
                  .'</a>');
    push(@functionlist,'<a href="/adm/email?markdel='.&escape($msgid).$sqs.'">'
                  .&mt('Delete')
                  .'</a>');
    push(@functionlist,'<a href="/adm/email?'.$sqs.'">'
                  .&mt('Back to Folder Display')
                  .'</a>');
    if ($counter > 0){
        push(@functionlist,'<a href="/adm/email?display='.$messages[$counter-1]->[5].$sqs.'">'
                          .&mt('Previous')
                          .'</a>');
    }
    if ($counter < $number_of_messages - 1){
        push(@functionlist,'<a href="/adm/email?display='.$messages[$counter+1]->[5].$sqs.'">'
                          .&mt('Next')
                          .'</a>');
    }
# Print functions
    my $legendtext='<span class="LC_mail_functions">'
                   .&mt('Functions')
                   .'</span>';
    my $showfunctionlist = $li_start.join($li_end.$li_start,@functionlist).$li_end;
    $r->print('<div class="LC_left_float">'
             .'<fieldset>'
             .'<legend>'
             .$legendtext
             .'</legend>'
             .$showfunctionlist
             .'</fieldset>'
             .'</div>'
    );

# Prepare available actions
    my $symb;
    if (defined($content{'symb'})) {
        $symb = $content{'symb'};
    } elsif (defined($content{'baseurl'})) {
        $symb=&Apache::lonnet::symbread($content{'baseurl'});
    }
    if ($env{'user.adv'}) {
        my @actionlist;

	if (&Apache::lonnet::allowed('vgr',$env{'request.course.id'})) {
		push(@actionlist,&Apache::loncommon::track_student_link(
                                      &mt('View recent activity')
                                     ,$content{'sendername'}
                                     ,$content{'senderdomain'}
                                     ,'check'));
	}
	if (&Apache::lonnet::allowed('opa',$env{'request.course.id'}) && $symb) {
	    push(@actionlist,&Apache::loncommon::pprmlink(
                                  &mt('Set/Change parameters')
                                     ,$content{'sendername'}
                                     ,$content{'senderdomain'}
                                     ,$symb
                                     ,'check'));
	}
	if (&Apache::lonnet::allowed('mgr',$env{'request.course.id'}) && $symb) {
	    push(@actionlist,&Apache::loncommon::pgrdlink(
                                 &mt('Set/Change grades')
                                     ,$content{'sendername'}
                                     ,$content{'senderdomain'}
                                     ,$symb
                                     ,'check'));
	}

# Print actions
        if (@actionlist) {
            my $showactionlist = $li_start.join($li_end.$li_start,@actionlist).$li_end;
            $legendtext=&mt('[_1]Currently available actions[_2] (will open extra window):'
                           ,'<span class="LC_mail_functions">','</span>');
            $r->print('<div class="LC_left_float">'
                     .'<fieldset>'
                     .'<legend>'
                     .$legendtext
                     .'</legend>'
                     .$showactionlist
                     .'</fieldset>'
                     .'</div>'
            );
        }
    }

    my ($tonum,$tolist,$cclist,$bcclist,$groupcclist,%recipients);
    if ($content{'recipid'}) {
        $tonum = &retrieve_recips('display',\%content,\%recipients);
        if (ref($recipients{'cc'}) eq 'ARRAY') {
            $cclist = join(', ',@{$recipients{'cc'}});
        }
        if (ref($recipients{'to'}) eq 'ARRAY') {
            $tolist = join(', ',@{$recipients{'to'}});
        }
        if (ref($recipients{'bcc'}) eq 'ARRAY') {
            $bcclist = join(', ',@{$recipients{'bcc'}});
        }
    }

    my $broadcast_link;
    if (($content{'courseid'}) && ($content{'recipid'} &&
        (ref($recipients{'course_broadcast'}) eq 'ARRAY') ||
        (ref($recipients{'group_cc_broadcast'}) eq 'ARRAY') ||
        (ref($recipients{'group_bcc_broadcast'}) eq 'ARRAY'))) {
        $broadcast_link = &recipients_link($r,\%content,\%recipients);
    }

    if (((!$tolist) && (!$broadcast_link)) && ref($content{'recuser'}) eq 'ARRAY') {
        my @recipients;
        for (my $i=0; $i<@{$content{'recuser'}}; $i++) {
            $recipients[$i] =  &Apache::loncommon::aboutmewrapper(
               &Apache::loncommon::plainname($content{'recuser'}[$i],
                                      $content{'recdomain'}[$i]),
                  $content{'recuser'}[$i],$content{'recdomain'}[$i]).
           ' ('.$content{'recuser'}[$i].':'.$content{'recdomain'}[$i].') ';
        }
        $tolist = join(', ',@recipients);
    }
    my ($restitle,$baseurl,$refers_to);
    if (defined($content{'resource_title'})) {
        $restitle = $content{'resource_title'};
    } else {
        if (defined($content{'baseurl'})) {
            $restitle = &Apache::lonnet::gettitle($content{'baseurl'});
        }
    }
    if (defined($content{'baseurl'})) {
        $baseurl = &Apache::lonenc::check_encrypt($content{'baseurl'});
    }
    $r->print('<div class="LC_clear_float_footer">');
    if ($from_student && $see_anonymous ) {
	$r->print(&Apache::loncommon::student_image_tag($content{'senderdomain'},$content{'sendername'}).'</br>');
    }

    # Display LON-CAPA Message (Start)
    # Subject
    $r->print('</div>'
             .&Apache::lonhtmlcommon::start_pick_box()
             .&Apache::lonhtmlcommon::row_title(&mt('Subject'))
             .$content{'subject'}
             .&Apache::lonhtmlcommon::row_closure()
    );
    if ($folder eq 'sent') {
        # To
        if ($tolist) {
            $r->print(&Apache::lonhtmlcommon::row_title(&mt('To'))
                     .$tolist
                     .&Apache::lonhtmlcommon::row_closure()
            );
        }
        if ($cclist) {
            $r->print(&Apache::lonhtmlcommon::row_title(&mt('Cc'))
                     .$cclist
                     .&Apache::lonhtmlcommon::row_closure()
            );
        }
        if ($bcclist) {
            $r->print(&Apache::lonhtmlcommon::row_title(&mt('Bcc'))
                     .$bcclist
                     .&Apache::lonhtmlcommon::row_closure()
            );
        }
        if (($content{'courseid'}) && ($content{'recipid'})) {
            my %broadcast_types = 
                &Apache::lonlocal::texthash (
                    course_broadcast    => 'Broadcast to', 
                    group_cc_broadcast  => 'Cc to group',
                    group_bcc_broadcast => 'Bcc to group',
                );                   
            foreach my $type (sort(keys(%broadcast_types))) {
                if (ref($recipients{$type}) eq 'ARRAY') {
                    my $num = @{$recipients{$type}};
                    my $broadcastlist = join(', ',@{$recipients{$type}});
                    if ($broadcastlist && $broadcast_link) {
                        if ($type eq 'group_cc_broadcast') {
                            $groupcclist = $broadcastlist;
                        }
                        $r->print(&Apache::lonhtmlcommon::row_title(
                                            $broadcast_types{$type})
                                  .&mt('[quant,_1,recipient]',$num)
                                  .' <a href="javascript:showBroadcastList();">['
                                  .&mt('Show').']</a>' 
                                  .&Apache::lonhtmlcommon::row_closure());
                    }
                }
            }
        }
        if ($content{'replytoaddr'}) {
            my ($replytoname,$replytodom) = split(/:/,$content{'replytoaddr'});
            if ($replytoname ne '' && $replytodom ne '') {
                $r->print(&Apache::lonhtmlcommon::row_title(&mt('Reply To'))
                         .$replytoname.':'.$replytodom
                         .&Apache::lonhtmlcommon::row_closure()
                );
            }
        }
    } else {
        # From, Reply
        $r->print(&Apache::lonhtmlcommon::row_title(&mt('From'))
                 .&Apache::loncommon::aboutmewrapper(
                     &Apache::loncommon::plainname($content{'sendername'},$content{'senderdomain'}),
                                                 $content{'sendername'},$content{'senderdomain'})
        );
        if ($content{'noreplies'}) {
            $r->print(' ('.&mt('No replies to sender').')'
                     .&Apache::lonhtmlcommon::row_closure()
            );
        } else {
            if ($content{'replytoaddr'}) {
                my ($replytoname,$replytodom) = split(/:/,$content{'replytoaddr'});
                if ($replytoname ne '' && $replytodom ne '') {
                    $r->print(&Apache::lonhtmlcommon::row_closure()
                             .&Apache::lonhtmlcommon::row_title(&mt('Reply To'))
                             .$replytoname.':'.$replytodom
                             .&Apache::lonhtmlcommon::row_closure()
                    );
                } else {
                    $r->print(&Apache::lonhtmlcommon::row_closure());
                }
            } else {
                $r->print(' ('.$content{'sendername'}.':'.$content{'senderdomain'}.') '
                         .&Apache::lonhtmlcommon::row_closure()
                );
            }
            if ($tonum && $tolist) {
                $r->print(&Apache::lonhtmlcommon::row_title(&mt('To'))
                         .$tolist
                         .&Apache::lonhtmlcommon::row_closure()
                    );
            }
            if ($cclist) { 
                $r->print(&Apache::lonhtmlcommon::row_title(&mt('Cc'))
                         .$cclist
                         .&Apache::lonhtmlcommon::row_closure()
                    );
            }
            if ($content{'group'} ne '') {
                if (&check_group_priv($content{'group'})) {
                    $groupcclist = join(', ',@{$recipients{'group_cc_broadcast'}});
                    if ($groupcclist) {
                        $r->print(&Apache::lonhtmlcommon::row_title(&mt('Group Cc'))
                                 .$groupcclist
                                 .&Apache::lonhtmlcommon::row_closure()
                        );
                    }
                }
            }
        }
    }

    # Course
    if ($content{'courseid'}) {
        $r->print(&Apache::lonhtmlcommon::row_title(&mt($crstype))
                 .$courseinfo{'description'}
        );
        if ($content{'coursesec'}) {
            $r->print(' ('.&mt('Section').': '.$content{'coursesec'}.')');
        }
        $r->print(&Apache::lonhtmlcommon::row_closure());
    }
    $r->print(&Apache::lonhtmlcommon::row_title(&mt('Time'))
             .$content{'time'}
             .&Apache::lonhtmlcommon::row_closure()
    );

    # Refers to
    if ($baseurl) {
        if (defined($content{'courseid'}) && defined($env{'request.course.id'})) {
            if ($content{'courseid'} eq $env{'request.course.id'}) {
                my $symblink;
                my $showsymb = &Apache::lonenc::check_decrypt($symb);
                my $showurl = &Apache::lonenc::check_decrypt($baseurl);
                my $encrypturl = &Apache::lonnet::EXT('resource.0.encrypturl',
                              $showsymb,$env{'user.domain'},$env{'user.name'});
                if ($symb) {
                    if ($encrypturl =~ /^yes$/i && !$env{'request.role.adv'}) {
                        $showsymb = &Apache::lonenc::check_encrypt($symb);
                    }
                    $symblink = '?symb='.$showsymb;
                }
                if ($encrypturl =~ /^yes$/i && !$env{'request.role.adv'}) {
                    $showurl = $baseurl;
                }
                $r->print(&Apache::lonhtmlcommon::row_title(&mt('Refers to'))
                         .'<a href="'.$showurl.$symblink.'">'.$restitle.'</a>'
                         .&Apache::lonhtmlcommon::row_closure()
                );
                $refers_to = 1;
            }
        }
        if (!$refers_to) {
            if ($baseurl =~ m-^/enc/-) {
                if (defined($content{'courseid'})) {
                    if (!$env{'request.course.id'}) {
                        my $unencurl =
                           &Apache::lonenc::unencrypted($baseurl,
                                                        $content{'courseid'});
                        if ($unencurl ne '') {
                            if (&Apache::lonnet::allowed('bre',$unencurl)) {
                                $r->print(&Apache::lonhtmlcommon::row_title(&mt('Refers to'))
                                         .'<a href="'.$unencurl.'">'.$restitle.'</a>'
                                         .&Apache::lonhtmlcommon::row_closure()
                                );
                            }
                        }
                    }
                }
            } else {
                if (&Apache::lonnet::allowed('bre',$baseurl)) {
                    $r->print(&Apache::lonhtmlcommon::row_title(&mt('Refers to'))
                             .'<a href="'.$baseurl.'">'.$restitle.'</a>'
                             .&Apache::lonhtmlcommon::row_closure()
                    );

                }
            }
        }
    }

    # Message
    $r->print(&Apache::lonhtmlcommon::row_title(&mt('Message'))
             .'<pre>'
	     .&Apache::lontexconvert::msgtexconverted($content{'message'},1)
	     .'</pre>'
    );
    if (&displayresource(%content)) {
        $r->print(&Apache::lonhtmlcommon::row_closure()
                 .&Apache::lonhtmlcommon::row_title(&mt('Resource Details'))
                 .&displayresource(%content)
        );
    } 
    $r->print(&Apache::lonhtmlcommon::row_closure(1).
              &Apache::lonhtmlcommon::end_pick_box());
    # Display LON-CAPA Message (End)
    return;
}

sub retrieve_recips {
    my ($context,$content,$recips)= @_;
    my $tonum = 0;
    if (ref($content) eq 'HASH') {
        my %reciphash =
            &Apache::lonnet::get('nohist_emailrecip',[$content->{'recipid'}],
                                 $content->{'senderdomain'},$content->{'sendername'});
        my $recipinfo = $reciphash{$content->{'recipid'}};
        if (ref($recipinfo) eq 'HASH') {
            foreach my $type ('to','cc','course_broadcast','group_cc_broadcast','group_bcc_broadcast') {
                if (ref($recipinfo->{$type}) eq 'HASH') {
                    if ($type eq 'to') {
                        $tonum = keys(%{$recipinfo->{$type}});
                    }
                    foreach my $user (sort(keys(%{$recipinfo->{$type}}))) {
                        my ($uname,$udom) = split(/:/,$user);
                        next if (($context eq 'replying') && ($uname eq $env{'user.name'})
                                 && ($udom eq $env{'user.domain'}));
                        my $showuser ='<span class="LC_nobreak">';
                        if ($context eq 'replying') {
                            if (($type eq 'to') || ($type eq 'cc')) { 
                                $showuser = '<label><input type="checkbox" name="replying_'.$type.'" value="'.$user.'" />';
                            } elsif ($type eq 'group_cc_broadcast') {
                                $showuser = '<label><input type="checkbox" name="replying_groupcc" value="'.$user.'" />';
                            }
                        }
                        $showuser .= &Apache::loncommon::aboutmewrapper(
                                           &Apache::loncommon::plainname($uname,
                                           $udom),$uname,$udom);
		        if ($context eq 'replying') {
		            $showuser .='</label>';
		        }
		        $showuser .= '</span>';
                        if (ref($recips) eq 'HASH') {
		            push(@{$recips->{$type}},$showuser);
                        }
                    }
                }
            }
        }
    }
    return $tonum;
}

sub recipients_link {
    my ($r,$content,$recipients) = @_;
    my ($broadcast_link,$show);
    if ((ref($content) eq 'HASH') && (ref($recipients) eq 'HASH')) {
        if (ref($recipients->{'course_broadcast'}) eq 'ARRAY') {
            if (@{$recipients->{'course_broadcast'}} > 0) {
                $show = 'course';
            }
        } elsif (ref($recipients->{'group_cc_broadcast'}) eq 'ARRAY') {
            if (@{$recipients->{'group_cc_broadcast'}} > 0) {
                $show = 'group_cc';
            }
        } elsif (ref($recipients->{'group_bcc_broadcast'}) eq 'ARRAY') {
            if (@{$recipients->{'group_bcc_broadcast'}} > 0) {
                $show = 'group_bcc';
            }
        }
        if ($show) {
            my ($nothing,$height,$width,$start_page,$end_page,$body);
            $nothing=&Apache::lonhtmlcommon::javascript_nothing();
            $height = 400;
            $width = 600;
            $start_page =
                &Apache::loncommon::start_page('Broadcast List', undef,
                                               {only_body => 1,
                                                js_ready  => 1,});
            $end_page = &Apache::loncommon::end_page({js_ready => 1,});
            $body = '<h3>'.&mt("Recipients of broadcast message").'</h3>'.
                    &Apache::loncommon::start_data_table();
            my $cell = 0;
            $body .= &Apache::loncommon::start_data_table_row();
            foreach my $item (@{$recipients->{$show.'_broadcast'}}) {
                $item =~ s/'/\\'/g;
                if (!($cell%2) && $cell > 0) {
                    $body .= &Apache::loncommon::end_data_table_row().
                             &Apache::loncommon::start_data_table_row();
                }
                $cell ++;
                $body .= '<td>'.$cell.'&nbsp;'.$item.'&nbsp;&nbsp;</td>';
            }
            if ($cell%2) {
                $body .= '<td>&nbsp;</td>';
            }
            $body .= &Apache::loncommon::end_data_table_row().
                     &Apache::loncommon::end_data_table();
            $body =~ s{</}{<\\/}g;
            $body =~ s{\n}{}g;
            $r->print(<<ENDJS);
<script type="text/javascript">
// <![CDATA[
function showBroadcastList() {
    var caller = this;
    var newWindow = null;
    try {
        newWindow =  window.open($nothing,"broadcast","HEIGHT=$height,WIDTH=$width,resizable=yes,scrollbars=yes" );
    }
    catch(error) {
        writeWin(caller);
        return;
    }
    if (newWindow) {
        caller = newWindow;
    }
    writeWin(caller);
    return;
}

function writeWin(caller) {
    caller.document.writeln('$start_page $body $end_page');
    caller.document.close();
    caller.focus();
}
// ]]>
</script>

ENDJS
            $broadcast_link = 1;
        }
    }
    return $broadcast_link;
}

# =========================================================== Show the citation

sub displayresource {
    my %content=@_;
#
# If the recipient is in the same course that the message was sent from and
# has sufficient privileges, show "all details," else show citation
#
    if (($env{'request.course.id'} eq $content{'courseid'})
     && (&Apache::lonnet::allowed('vgr',$content{'courseid'}))) {
        my $symb;
        if (defined($content{'symb'})) {
            $symb = $content{'symb'};
        } else { 
	    $symb=&Apache::lonnet::symbread($content{'baseurl'});
        }
# Could not get a symb, give up
	unless ($symb) { return $content{'citation'}; }
# Have a symb, can render
	return '<h2>'.&mt('Current attempts of student (if applicable)').'</h2>'.
	    &Apache::loncommon::get_previous_attempt($symb,
						     $content{'sendername'},
						     $content{'senderdomain'},
						     $content{'courseid'}).
	    '<hr /><h2>'.&mt('Current screen output (if applicable)').'</h2>'.
	    &Apache::loncommon::get_student_view($symb,
						 $content{'sendername'},
						 $content{'senderdomain'},
						 $content{'courseid'}).
	    '<h2>'.&mt('Correct Answer(s) (if applicable)').'</h2>'.
	    &Apache::loncommon::get_student_answers($symb,
						    $content{'sendername'},
						    $content{'senderdomain'},
						    $content{'courseid'});
    } elsif ($env{'user.adv'}) {
	return $content{'citation'};
    }
    return '';
}

# ================================================================== The Header

sub header {
    my ($r,$title,$baseurl)=@_;
    my $extra = &Apache::loncommon::studentbrowser_javascript();
    if ($baseurl) {
	$extra .= "<base href=\"".&Apache::lonnet::absolute_url()."/$baseurl\" />";
    }
    $r->print(&Apache::loncommon::start_page('Messages',
 					$extra));
    $r->print(&Apache::lonhtmlcommon::breadcrumbs
     		(($title?$title:'Send and display messages')));
}

# ---------------------------------------------------------------- Print header

sub printheader {
    my ($r,$url,$desc,$title,$baseurl)=@_;
    &Apache::lonhtmlcommon::add_breadcrumb
	({href=>$url,
	  text=>$desc});
    &header($r,$title,$baseurl);
}

# ------------------------------------------------------------ Store the comment

sub storecomment {
    my ($r)=@_;
    my $msgtxt=&Apache::lonfeedback::clear_out_html($env{'form.message'});
    my $cleanmsgtxt='';
    foreach my $line (split(/[\n\r]/,$msgtxt)) {
	unless ($line=~/^\s*(\>|\&gt\;)/) {
	    $cleanmsgtxt.=$line."\n";
	}
    }
    my $key=&escape($env{'form.baseurl'}).'___'.time;
    &Apache::lonnet::put('nohist_stored_comments',{ $key => $cleanmsgtxt });
}

sub storedcommentlisting {
    my ($r)=@_;
    my %msgs=&Apache::lonnet::dump('nohist_stored_comments',undef,undef,
       '^'.&escape(&escape($env{'form.showcommentbaseurl'})));
    $r->print(&Apache::loncommon::start_page('Saved Comment Listing',undef,
					     {'onlybody' => 1}));
    if ((keys %msgs)[0]=~/^error\:/) {
	$r->print(&mt('No saved comments yet.'));
    } else {
	my $found=0;
	foreach my $key (sort(keys(%msgs))) {
	    $r->print("\n".$msgs{$key}."<hr />");
	    $found=1;
	}
	unless ($found) {
	    $r->print(&mt('No saved comments yet for this resource.'));
	}
    }
}

# ---------------------------------------------------------------- Send an email

sub sendoffmail {
    my ($r,$folder)=@_;
    my $suffix=&Apache::lonmsg::foldersuffix($folder);
    my $sendstatus='';
    my %msg_status;
    my $numsent = 0;
    my $nosentstore = 1;
    my $attachmenturl;
    my $now = time;
    my ($cdom,$cnum,$group);
    if (exists($env{'form.group'})) {
        $group = $env{'form.group'};
    }
    if (exists($env{'request.course.id'})) {
        $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
        $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    }
    if ($env{'form.send'}) {
        if (!$env{'form.multiforward'}) { 
            if ($group eq '') {
	        &printheader($r,'','Messages being sent.');
            } else {
                $r->print(&groupmail_header('sending',$group));
            }
        }
	$r->rflush();
	my %content=();
	undef %content;
	if ($env{'form.forwid'}) {
	    my $msgid=$env{'form.forwid'};
	    my %message=&Apache::lonnet::get('nohist_email'.$suffix,[$msgid]);
	    %content=&Apache::lonmsg::unpackagemsg($message{$msgid},1,1);
	    &statuschange($msgid,'forwarded',$folder);
            if ($content{'attachmenturl'} ne '') {
                $attachmenturl = $content{'attachmenturl'};
            }
	    $env{'form.message'} .= "\n\n-- Forwarded message --\n\n".
		                    $content{'message'};
	}
	if ($env{'form.replyid'}) {
	    my $msgid=$env{'form.replyid'};
	    my %message=&Apache::lonnet::get('nohist_email'.$suffix,[$msgid]);
	    %content=&Apache::lonmsg::unpackagemsg($message{$msgid},1);
	    &statuschange($msgid,'replied',$folder);
	}

	my $mode = $env{'form.sendmode'};
	my (%toaddr,$tos,$cc,$bcc,$broadcast);

	if ($mode eq 'group') {
            if (defined($env{'form.courserecips'})) {
                my $courseusers = $env{'form.courserecips'};
                $courseusers =~ s/^_\&\&\&_//;
                my @to = split('_&&&_',$courseusers);
                foreach my $dest (@to) {
                    my ($user,$domain) = split(/:/, $dest);
                    if (($user ne '') && ($domain ne '')) {
                        my $rec = $user.":".$domain;
                        $toaddr{$rec} = '';
                        $broadcast->{$rec} = '';
                    }
                }
            }
	} elsif ($mode eq 'upload') {
            $nosentstore = 0;
	    foreach my $line (split(/[\n\r\f]+/,$env{'form.upfile'})) {
                my ($rec,$txt) = ($line =~ /^([^:]+:[^:]+):(.*)$/);
		if ($txt) {
                    $rec =~ s/^\s+//;
                    $rec =~ s/\s+$//;
		    $toaddr{$rec}.=$txt."\n";
                    $broadcast->{$rec} = '';
		}
	    }
	} else {
	    if (($env{'form.recuname'} ne '') && ($env{'form.recdomain'} ne '')) {
		$toaddr{$env{'form.recuname'}.':'.$env{'form.recdomain'}}='';
                $tos->{$env{'form.recuname'}.':'.$env{'form.recdomain'}}='';
	    }
	}
        if ($env{'form.additionalrec_to'}) {
            foreach my $rec (split(/\s*,\s*/,$env{'form.additionalrec_to'})) {
                my ($auname,$audom)=split(/:/,$rec);
                if (($auname ne "") && ($audom ne "")) {
                    $toaddr{$auname.':'.$audom}='';
                    $tos->{$auname.':'.$audom}='';
                }
            }
        }
        if ($env{'form.replying_to'}) {
            my @toreplies =
                &Apache::loncommon::get_env_multiple('form.replying_to');
            foreach my $rec (@toreplies) {
                my ($auname,$audom)=split(/:/,$rec);
                if (($auname ne "") && ($audom ne "")) {
                    $toaddr{$auname.':'.$audom}='';
                    $tos->{$auname.':'.$audom}='';
                }
            }
        }
	if ($env{'form.additionalrec_cc'}) {
	    foreach my $rec (split(/\s*,\s*/,$env{'form.additionalrec_cc'})) {
		my ($auname,$audom)=split(/:/,$rec);
		if (($auname ne "") && ($audom ne "")) {
                    $toaddr{$auname.':'.$audom}='';
                    if (!defined($tos->{$auname.':'.$audom})) {
		        $cc->{$auname.':'.$audom}='';
                    }
		}
	    }
	}
        if ($env{'form.replying_cc'}) {
            my @ccreplies =
                &Apache::loncommon::get_env_multiple('form.replying_cc');
            foreach my $rec (@ccreplies) {
                my ($auname,$audom)=split(/:/,$rec);
                if (($auname ne "") && ($audom ne "")) {
                    $toaddr{$auname.':'.$audom}='';
                    if (!defined($tos->{$auname.':'.$audom})) {
                        $cc->{$auname.':'.$audom}='';
                    }
                }
            }
        }
        if ($env{'form.replying_groupcc'}) {
            my @groupreplies =
                &Apache::loncommon::get_env_multiple('form.replying_groupcc');
            foreach my $rec (@groupreplies) {
                my ($auname,$audom)=split(/:/,$rec);
                if (($auname ne "") && ($audom ne "")) {
                    $toaddr{$auname.':'.$audom}='';
                    if (!defined($tos->{$auname.':'.$audom})) {
                        $broadcast->{$auname.':'.$audom}='';
                    }
                }
            }
        }
        if ($env{'form.additionalrec_bcc'}) {
            foreach my $rec (split(/\s*,\s*/,$env{'form.additionalrec_bcc'})) {
                my ($auname,$audom)=split(/:/,$rec);
                if (($auname ne "") && ($audom ne "")) {
                    $toaddr{$auname.':'.$audom}='';
                    if ((!defined($tos->{$auname.':'.$audom})) && 
                        (!defined($cc->{$auname.':'.$audom}))) {
                        $bcc->{$auname.':'.$audom}='';
                    }
                }
            }
        }
        my $savemsg;
        my $msgtype;
        my %sentmessage;
        my $msgsubj=&Apache::lonfeedback::clear_out_html($env{'form.subject'},
							 undef,1);
        if ((($env{'form.critmsg'}) || ($env{'form.sendbck'})) &&
            (&Apache::lonnet::allowed('srm',$env{'request.course.id'})
	     || &Apache::lonnet::allowed('srm',$env{'request.course.id'}.
					 '/'.$env{'request.course.sec'})
	     )) {
            $savemsg=&Apache::lonfeedback::clear_out_html($env{'form.message'},1);
            $msgtype = 'critical';
        } else {
            $savemsg=&Apache::lonfeedback::clear_out_html($env{'form.message'});
        }
        my %reciphash = (
                           to => $tos,
                           cc => $cc,
                           bcc => $bcc,
                        );
        if ($mode eq 'group') {
            if ($group eq '') {
                $reciphash{'course_broadcast'} = $broadcast;
            } else {
                if ($env{'form.groupmail'} eq 'cc') {
                    $reciphash{'group_cc_broadcast'} = $broadcast;
                } else {
                    $reciphash{'group_bcc_broadcast'} = $broadcast;
                }
            }
        }
        my ($recipid,$recipstatus) = 
            &Apache::lonmsg::store_recipients($msgsubj,$env{'user.name'},
                                       $env{'user.domain'},\%reciphash);
        if ($recipstatus ne 'ok') {
            &Apache::lonnet::logthis('Failed to store To, Bcc and Cc recipients for '.$env{'user.name'}.':'.$env{'user.domain'});
        }
        if ($env{'form.attachment'}) {
            if (length($env{'form.attachment'})<131072) {
                $attachmenturl=&Apache::lonnet::userfileupload('attachment',undef,'feedback/'.$now);
            } else {
                $r->print('<p><span class="LC_warning">'.&mt('Attachment not included - exceeded permitted length').'</span><br /></p>');
            }
        } elsif ($env{'form.multiforward'}) {
            if ($env{'form.attachmenturl'} ne '') {
                $attachmenturl = $env{'form.attachmenturl'};
            }
        }
        my @recusers;
        my @recudoms;
	foreach my $address (sort(keys(%toaddr))) {
	    my ($recuname,$recdomain)=split(/\:/,$address);
            my $msgtxt = $savemsg;
            if ($toaddr{$address}) {
	        $msgtxt.='<hr />'.$toaddr{$address};
            }
	    my @thismsg;
	    if ($msgtype eq 'critical') {
		$r->print(&mt('Sending critical message').' '.
                              $recuname.':'.$recdomain.': ');
		@thismsg=
		    &Apache::lonmsg::user_crit_msg($recuname,$recdomain,
						   $msgsubj,$msgtxt,
						   $env{'form.sendbck'},
						   $env{'form.permanent'},
						   \$sentmessage{$address},
                                                   $nosentstore,$recipid);
	    } else {
		$r->print(&mt('Sending').' '.$recuname.':'.$recdomain.': ');
		@thismsg=
		    &Apache::lonmsg::user_normal_msg($recuname,$recdomain,
						     $msgsubj,$msgtxt,
						     $content{'citation'},
						     undef,$attachmenturl,
						     $env{'form.permanent'},
						     \$sentmessage{$address},
                                                     undef,undef,undef,
                                                     $nosentstore,$recipid);
            }
	    $msg_status{$recuname.':'.$recdomain}=join(' ',@thismsg);
	    if ($msg_status{$recuname.':'.$recdomain} =~ /(ok|con_delayed)/) {  
	        $numsent++;
                push(@recusers,$recuname);
                push(@recudoms,$recdomain);
	    }
	    $sendstatus.=' '.join(' ',@thismsg);
	}
        my $subj_prefix;
        if ($numsent > 0) {
            if (($env{'request.course.id'}) && 
                (($mode eq 'group') ||
                 ($env{'form.courserecord'}) ||
                 ($msgtype eq 'critical')) ||
                ($env{'form.replyid'} && 
                 (($content{'courseid'} ne '') && 
                  ($mode eq 'group')))) {
                if ($msgtype eq 'critical') {
                    $subj_prefix = 'Critical.';
                } elsif ($mode eq 'group') {
                    $subj_prefix = 'Broadcast.';
                } else {
                    $subj_prefix = 'Archive';
                }
                my ($specialmsgid,$specialresult);
                my $course_str;
                if ($env{'form.replyid'}) {
                    if ($content{'courseid'} ne '') {
                        my %crsdesc = 
                            &Apache::lonnet::coursedescription($content{'courseid'},
                                                               {'one_time' => 1});
                        $course_str = &escape('['.$crsdesc{'num'}.':'.$crsdesc{'domain'}.']');
                    }
                } elsif ($env{'request.course.id'}) {
                    $course_str = &escape('['.$cnum.':'.$cdom.']');
                }
                $specialresult = 
                    &Apache::lonmsg::user_normal_msg_raw($cnum,$cdom,
                        $subj_prefix.' '.$course_str,$savemsg,undef,undef,
                        $attachmenturl,undef,undef,\$specialmsgid,undef,undef,undef,
                        undef,undef,1);
                $specialmsgid = &unescape($specialmsgid);
                if ($specialresult eq 'ok') {
                    my ($stamp,$crssubj,$msgname,$msgdom,$msgcount,$context,$pid) =
		        split(/\:/,&unescape($specialmsgid));

                    foreach my $recipient (sort(keys(%toaddr))) {
                        if ($msg_status{$recipient} =~ /\s*(ok|con_delayed)\s*/) {
                            my $usersubj = $subj_prefix.'['.$recipient.']';
                            my $usermsgid = 
			        &Apache::lonmsg::buildmsgid($stamp,$usersubj,
							    $msgname,$msgdom,
							    $msgcount,$context,
							    $pid);
                            &Apache::lonmsg::user_normal_msg_raw($cnum,$cdom,
                                $subj_prefix.' ['.$recipient.']',$msgsubj,
                                undef,undef,$attachmenturl,undef,$usermsgid,undef,
                                undef,$specialmsgid,undef,undef,undef,1);
                        }
                    }
                    if (($mode ne 'upload') && (@recusers > 0)) {
                        &Apache::lonmsg::process_sent_mail($msgsubj,
                           $subj_prefix,$numsent,$stamp,$msgname,$msgdom,
                           $msgcount,$context,$pid,$savemsg,\@recusers,
                           \@recudoms,undef,$attachmenturl,'','','','',$recipid);
                    }
                } else {
                    &Apache::lonnet::logthis('Failed to create record of critical, broadcast or archived message in '.$env{'course.'.$env{'request.course.id'}.'.num'}.' '&mt('at').' '.$env{'course.'.$env{'request.course.id'}.'.domain'}.' - no msgid generated');
                }
            } else {
                my $stamp = time;
                my $msgcount = &Apache::lonmsg::get_uniq();
                my $context = &Apache::lonmsg::get_course_context();
                &Apache::lonmsg::process_sent_mail($msgsubj,$subj_prefix,
                       $numsent,$stamp,$env{'user.name'},
                       $env{'user.domain'},$msgcount,$context,
                       $$,$savemsg,\@recusers,\@recudoms,undef,$attachmenturl,
                       '','','','',$recipid);
            }
        }
        if (!$env{'form.multiforward'}) { 
            if ($sendstatus=~/^(\s*(?:ok|con_delayed)\s*)*$/) {
                my $message = &Apache::lonhtmlcommon::confirm_success(&mt('Completed.'));
                $message = &Apache::loncommon::confirmwrapper($message);
                $r->print($message);
	        if ($env{'form.displayedcrit'}) {
	            &discrit($r);
                }
                if ($group ne '') {
                    $r->print(&groupmail_sent($group,$cdom,$cnum)); 
	        } else {
	            &Apache::loncommunicate::menu($r);
	        }
            } else {
                my $message = &Apache::lonhtmlcommon::confirm_success(&mt('Could not deliver message'),1);
                $message .= '<br />'.&mt('Please use the browser "Back" button and correct the recipient addresses ([_1]).',$sendstatus);
                $message = &Apache::loncommon::confirmwrapper($message);
                $r->print($message);
            }
        }
    }
    return $sendstatus;
}

# ===================================================================== Handler

sub handler {
    my $r=shift;

# ----------------------------------------------------------- Set document type
    
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    
    return OK if $r->header_only;
    
# --------------------------- Get query string for limited number of parameters
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
        ['display','replyto','forward','markread','markdel','markunread',
         'sendreply','compose','sendmail','critical','recname','recdom',
         'recordftf','sortedby','block','folder','startdis','interdis',
	 'showcommentbaseurl','dismode','group','subject','text','ref',
         'msgstatus']);
    $sqs='&amp;sortedby='.$env{'form.sortedby'};

# ------------------------------------------------------ They checked for email
    unless ($env{'form.block'}) {
        &Apache::lonnet::put('email_status',{'recnewemail'=>0});
    }

# ----------------------------------------------------------------- Breadcrumbs

    &Apache::lonhtmlcommon::clear_breadcrumbs();
    &Apache::lonhtmlcommon::add_breadcrumb
        ({href=>"/adm/communicate",
          text=>"Messages",
          faq=>12,bug=>'Communication Tools',});

# ------------------------------------------------------------------ Get Folder

    my $folder=$env{'form.folder'};
    unless ($folder) { 
	$folder=''; 
    } else {
	$sqs.='&amp;folder='.&escape($folder);
    }
# ------------------------------------------------------------ Get Display Mode

    my $dismode=$env{'form.dismode'};
    unless ($dismode) { 
	$dismode=''; 
    } else {
	$sqs.='&amp;dismode='.&escape($dismode);
    }

# --------------------------------------------------------------------- Display
    my $msgstatus = $env{'form.msgstatus'};
    $startdis=$env{'form.startdis'};
    if ($startdis ne '') {
        $startdis--;
    }
    unless ($startdis) { $startdis=0; }

    if ($env{'form.firstview'}) {
	$startdis=0;
    }
    if ($env{'form.lastview'}) {
	$startdis=-1;
    }
    if ($env{'form.prevview'}) {
	$startdis--;
    }
    if ($env{'form.nextview'}) {
	$startdis++;
    }
    my $postedstartdis=$startdis+1;
    $sqs.='&amp;startdis='.$postedstartdis;

# --------------------------------------------------------------- Render Output

    if ($env{'form.display'}) {
	&displaymessage($r,$env{'form.display'},$folder,$msgstatus);
    } elsif ($env{'form.replyto'}) {
	&compout($r,'',$env{'form.replyto'},undef,undef,$folder,$dismode);
    } elsif ($env{'form.confirm'}) {
	&printheader($r,'','Confirmed Receipt');
	my $replying = 0;
	foreach my $envkey (keys(%env)) {
	    if ($envkey=~/^form\.(rep)?rec\_(.*)$/) {
                my $repchk = $1;
       		my $msgid = $2;
		$r->print('<b>'.&mt('Confirming Receipt').':</b> ');
		my $result = &Apache::lonmsg::user_crit_received($msgid);
                if ($result =~ /trans:\s+ok/) {
                    &statuschange($msgid,'read');
                }
                $r->print($result.'<br />');
                if ($repchk eq 'rep') {
		    &compout($r,'','','',$msgid);
		    $replying = 1;
                }
	    }
	}
	if (!$replying) {
	    &discrit($r);
	}
    } elsif ($env{'form.critical'}) {
	&printheader($r,'','Displaying Critical Messages');
	&discrit($r);
    } elsif ($env{'form.forward'}) {
	&compout($r,$env{'form.forward'},undef,undef,undef,$folder);
    } elsif ($env{'form.markdel'}) {
	&printheader($r,'','Deleted Message');
	my ($result,$msg) = 
	    &statuschange($env{'form.markdel'},'deleted',$folder);
	if (!$result) {
            my $message = &Apache::lonhtmlcommon::confirm_success(&mt('Failed to delete the message.'),1);
            $message .= '<p class="LC_error">'.$msg.'</p>';
            $message = &Apache::loncommon::confirmwrapper($message);
            $r->print($message);
	}
	&Apache::loncommunicate::menu($r);
	&disall($r,($folder?$folder:$dismode),$msgstatus);
    } elsif ($env{'form.markedaction'} eq 'markedforward') {
        my $total = 0;
        my @to_forward = &Apache::loncommon::get_env_multiple('form.delmark');
        foreach my $msgid (@to_forward) {
            &statuschange(&unescape($msgid),'forwarded',$folder);
            $total ++;
        }
        if ($total > 0) {
            &compout($r,undef,undef,undef,undef,$folder,$dismode,$total);
        }
    } elsif ($env{'form.markedaction'} eq 'markedread') {
        my $total = 0;
        my @to_markread = &Apache::loncommon::get_env_multiple('form.delmark');
        foreach my $msgid (@to_markread) {
            &statuschange(&unescape($msgid),'read',$folder);
            $total ++;
        }
        &printheader($r,'','Marked Messages Read');
        my $message = &Apache::lonhtmlcommon::confirm_success(&mt('Marked [quant,_1,message] read',$total));
        $message = &Apache::loncommon::confirmwrapper($message);
        $r->print($message);
        &Apache::loncommunicate::menu($r);
        &disall($r,($folder?$folder:$dismode),$msgstatus);
    } elsif ($env{'form.markedaction'} eq 'markedunread') {
        my $total = 0;
        my @to_markunread = &Apache::loncommon::get_env_multiple('form.delmark');
        foreach my $msgid (@to_markunread) {
            &statuschange(&unescape($msgid),'new',$folder);
            $total ++;
        }
        &printheader($r,'','Marked Messages Unread');
        my $message = &Apache::lonhtmlcommon::confirm_success(&mt('Marked [quant,_1,message] unread',$total));
        $message = &Apache::loncommon::confirmwrapper($message);
        $r->print($message);
        &Apache::loncommunicate::menu($r);
        &disall($r,($folder?$folder:$dismode),$msgstatus);
    } elsif ($env{'form.markedaction'} eq 'markedmove') {
        my $destfolder = $env{'form.movetofolder'};
        my %gotfolders = &Apache::lonmsg::get_user_folders();
        &printheader($r,'','Moved Messages');
        if (!defined($gotfolders{$destfolder})) {
            $r->print(&mt('Destination folder [_1] is not a valid folder',
                      $destfolder));
        } else {
	    my ($total,$failed,@failed_msg)=(0,0);
            my @to_move = &Apache::loncommon::get_env_multiple('form.delmark');
            foreach my $msgid (@to_move) {
	        my ($result,$msg) = &movemsg(&unescape($msgid),$folder,
			                     $env{'form.movetofolder'});
	        if ($result) {
		    $total++;
	        } else {
		    $failed++;
		    push(@failed_msg,$msg);
	        }
	    }
            my $message;
            if ($failed) {
                $message = &Apache::lonhtmlcommon::confirm_success(&mt('Failed to move [quant,_1,message]',$failed),1);
                $message .= '<p class="LC_error">'.
		            join("</p>\n<p class=\"LC_error\">",@failed_msg).
		            "</p>\n";
            }
            $message .= &Apache::lonhtmlcommon::confirm_success(&mt('Moved [quant,_1,message]',$total));
            $message = &Apache::loncommon::confirmwrapper($message);
            $r->print($message);
        }
	&Apache::loncommunicate::menu($r);
	&disall($r,($folder?$folder:$dismode),$msgstatus);
    } elsif ($env{'form.markedaction'} eq 'markeddel') {
	my ($total,$failed,@failed_msg)=(0,0);
        my @to_delete = &Apache::loncommon::get_env_multiple('form.delmark');
        foreach my $msgid (@to_delete) {
	    my ($result,$msg) = &statuschange(&unescape($msgid),'deleted', 
				              $folder);
	    if ($result) {
	        $total++;
	    } else {
	        $failed++;
		push(@failed_msg,$msg);
	    }
	}
	&printheader($r,'','Deleted Messages');
        my $message;
	if ($failed) {
            $message = &Apache::lonhtmlcommon::confirm_success(&mt('Failed to delete [quant,_1,message]',$failed),1);
            $message .= '<p class="LC_error">'.
		      join("</p>\n<p class=\"LC_error\">",@failed_msg).
		      "</p>\n";
	}
        $message .= &Apache::lonhtmlcommon::confirm_success(&mt('Deleted [quant,_1,message]',$total));
        $message = &Apache::loncommon::confirmwrapper($message);
        $r->print($message);
	&Apache::loncommunicate::menu($r);
	&disall($r,($folder?$folder:$dismode),$msgstatus);
    } elsif ($env{'form.markunread'}) {
	&printheader($r,'','Marked Message as Unread');
	&statuschange($env{'form.markunread'},'new');
	&Apache::loncommunicate::menu($r);
	&disall($r,($folder?$folder:$dismode),$msgstatus);
    } elsif ($env{'form.compose'}) {
	&compout($r,'','',$env{'form.compose'});
    } elsif ($env{'form.recordftf'}) {
	&facetoface($r,$env{'form.recordftf'});
    } elsif ($env{'form.block'}) {
        &examblock($r,$env{'form.block'});
    } elsif ($env{'form.sendmail'}) {
        if ($env{'form.multiforward'}) {
            &printheader($r,'','Messages being sent.');
            my $fixed_subj = $env{'form.subject'};
            my $suffix=&Apache::lonmsg::foldersuffix($folder);
            my (%sendresult,%forwardok,%forwardfail,$fwdcount);
            my @to_forward = &Apache::loncommon::get_env_multiple('form.delmark');
            foreach my $item (@to_forward) {
                my $msgid=&unescape($item);
                my %message=&Apache::lonnet::get('nohist_email'.$suffix,[$msgid]);
                my %content=&Apache::lonmsg::unpackagemsg($message{$msgid},1,1);
                if ($env{'form.showorigsubj'}) {
                    $env{'form.subject'} = $fixed_subj.$content{'subject'};
                } else {
                    $env{'form.subject'} = '';
                }
                my $uname = $content{'sendername'};
                my $udom = $content{'senderdomain'};
                &statuschange($msgid,'forwarded',$folder);
                if ($env{'form.showorigsender'}) {
                    $env{'form.message'} = $env{'form.msgheader'}.' '.
                        &Apache::loncommon::plainname($uname,$udom).' ('.
                                           $uname.':'.$udom.')';
                }
                $env{'form.message'}.="\n\n-- Forwarded message --\n\n".
                                      $content{'message'};
                $env{'form.attachmenturl'} = $content{'attachmenturl'};
                $env{'form.multiforwid'} = $item;
                $fwdcount ++;
                $r->print($fwdcount.': '); 
                $sendresult{$msgid} = &sendoffmail($r,$folder);
                $r->print('<br />');
            }
            foreach my $key (keys(%sendresult)) {
                if ($sendresult{$key} =~/^(\s*(?:ok|con_delayed)\s*)*$/) {
                    $forwardok{$key} = $sendresult{$key};
                } else {
                    $forwardfail{$key} = $sendresult{$key}; 
                }
            }
            if (keys(%forwardok) > 0) {
                my $count = keys(%forwardok);
                my $message = &Apache::lonhtmlcommon::confirm_success(&mt('[quant,_1,message] forwarded.',$count));
                $message = &Apache::loncommon::confirmwrapper($message);
                $r->print($message);
            }
            if (keys(%forwardfail) > 0) {
                my $count = keys(%forwardfail);
                my $message = &Apache::lonhtmlcommon::confirm_success(&mt('Could not forward [quant,_1,message].',$count),1);
                foreach my $key (keys(%forwardfail)) {
                    $message .= '<br />'.&mt('Could not deliver forwarded message.').'</span> '.
                                &mt('The recipient addresses may need to be corrected').' ('.$forwardfail{$key}.')';
                }
                $message = &Apache::loncommon::confirmwrapper($message);
                $r->print($message);
            }
            &Apache::loncommunicate::menu($r);
        } else {
	    &sendoffmail($r,$folder);
        }
	if ($env{'form.storebasecomment'}) {
	    &storecomment($r);
        }
	if (($env{'form.rsspost'}) && ($env{'request.course.id'})) {
	        &Apache::lonrss::addentry($env{'course.'.$env{'request.course.id'}.'.num'},
				      $env{'course.'.$env{'request.course.id'}.'.domain'},
				      'Course_Announcements',
				      $env{'form.subject'},
				      $env{'form.message'},'/adm/communicate','public');
	}
	if ((!exists($env{'form.group'})) && (!$env{'form.displayedcrit'})) {
	    &disall($r,($folder?$folder:$dismode),$msgstatus);
	}
    } elsif ($env{'form.newfolder'}) {
	&printheader($r,'','New Folder');
        my $showfolder = $env{'form.newfolder'};
	my ($makeresult,$warning) = &makefolder($env{'form.newfolder'});
        if ($makeresult eq 'ok') {
            my $message = &Apache::lonhtmlcommon::confirm_success(&mt('Folder "[_1]" created.',$showfolder));
            $message = &Apache::loncommon::confirmwrapper($message);
            $r->print($message);
        } else {
            my $message = &Apache::lonhtmlcommon::confirm_success(&mt('Creation failed.').' '.$makeresult.'<br />'.$warning,1);
            $message = &Apache::loncommon::confirmwrapper($message);
            $r->print($message);
            $showfolder = $folder;
        }
        &Apache::loncommunicate::menu($r);
	&disall($r,$showfolder,$msgstatus);
    } elsif ($env{'form.showcommentbaseurl'}) {
	&storedcommentlisting($r);
    } elsif ($env{'form.folderaction'} eq 'delete') {
        &printheader($r,'','Deleted Folder');
        my $showfolder = '';
        my $delresult = &deletefolder($folder);
        if ($delresult eq 'ok') {
            my $message = &Apache::lonhtmlcommon::confirm_success(&mt('Folder "[_1]" deleted.',$folder));
            $message = &Apache::loncommon::confirmwrapper($message);
            $r->print($message);
            $env{'form.folder'} = '';
        } else {
            my $message = &Apache::lonhtmlcommon::confirm_success(&mt('Deletion failed.').' '.$delresult,1);
            $message = &Apache::loncommon::confirmwrapper($message);
            $r->print($message);
            $showfolder = $folder;
        }
        &Apache::loncommunicate::menu($r);
        &disall($r,$showfolder,$msgstatus);
    } elsif ($env{'form.folderaction'} eq 'rename') {
        &printheader($r,'','Renamed Folder');
        my $showfolder = $env{'form.renamed'};
        my $renresult = &renamefolder($folder);
        if ($renresult eq 'ok') {
            my $message = &Apache::lonhtmlcommon::confirm_success(&mt('Folder "[_1]" renamed to "[_2]".',$folder,$showfolder));
            $message = &Apache::loncommon::confirmwrapper($message);
            $r->print($message);
        } else {
            my $message = &Apache::lonhtmlcommon::confirm_success(&mt('Renaming failed.').' '.$renresult,1);
            $message = &Apache::loncommon::confirmwrapper($message);
            $r->print($message);
            $showfolder = $folder;
        }
        &Apache::loncommunicate::menu($r);
        &disall($r,$showfolder,$msgstatus);
    } else {
	&printheader($r,'','Display All Messages');
	&Apache::loncommunicate::menu($r);
	&disall($r,($folder?$folder:$dismode),$msgstatus);
    }
    $r->print(&Apache::loncommon::end_page());
    return OK;
}
# ================================================= Main program, reset counter

=pod

=cut

1; 

__END__

