# The LearningOnline Network
# Pick a course
#
# $Id: lonpickcourse.pm,v 1.93 2010/03/08 14:43:46 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

package Apache::lonpickcourse;

use strict;
use Apache::Constants qw(:common);
use Apache::loncommon;
use Apache::loncoursedata;
use Apache::lonnet;
use Apache::lonlocal;
use Apache::longroup;
use Apache::courseclassifier;
use LONCAPA qw(:DEFAULT :match);

sub handler {
    my $r = shift;
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    return OK if $r->header_only;

# ------------------------------------------------------------ Print the screen

    # Get parameters from query string
    &Apache::loncommon::get_unprocessed_cgi
        ($ENV{'QUERY_STRING'},['domainfilter','form','cnumelement',
			       'cdomelement','cnameelement','roleelement',
                               'multiple','type','setroles','fixeddom','cloner']);
    my ($type,$title,$jscript,$multelement,$multiple,$roleelement,$typeelement,
        $lastaction,$autosubmit,$submitopener,$cloneruname,$clonerudom);

    # Get course type - Course or Community.
    $type = $env{'form.type'};
    if (!defined($env{'form.type'})) {
        $type = 'Course';
    }
    $title = 'Selecting a '.$type;

    # Setup for multiple course selections, if flag for multiples set.
    $multiple = $env{'form.multiple'};
    if ($multiple) {
        ($jscript,$multelement) = &multiples_tag();
        $title = 'Selecting '.$type.'(s)';
    }

    # if called when a DC is selecting a course 
    my $roledom = $env{'form.roleelement'};
    if ($roledom) {
        $roleelement = '<input type="hidden" name="roleelement" value="'.$roledom.'" />';
        $submitopener = &processpick();
        $autosubmit = 'process_pick("'.$roledom.'")';
    }
    if ($env{'form.typeelement'} ne '') {
        $typeelement = '<input type="hidden" name="typeelement" value="'.$env{'form.typeelement'}.'" />';
    }

    # if called when a DC is creating a course for another user.
    if ($env{'form.form'} eq 'ccrs') {
        ($cloneruname,$clonerudom) = ($env{'form.cloner'} =~ /^($match_username):($match_domain)$/);
    }

    # if called when requesting a course
    if ($env{'form.form'} eq 'requestcrs') {
        $cloneruname = $env{'user.name'};
        $clonerudom =  $env{'user.domain'};
    }

    my $onlyown = 0;
    # if called to assign course-based portfolio access control
    if ((($env{'form.form'} eq 'portform') && (!$env{'user.adv'}))) {
        $onlyown = 1;
    }

    my %loaditem;
    if (($env{'form.type'} eq 'Course') && ($env{'form.numtitles'})) {
        if (($env{'form.official'} eq 'on') && ($env{'form.state'} eq 'listing')) {
            $loaditem{'onload'} = 'setElements(document.filterpicker); ';
        }
    }

    if ((($env{'form.form'} eq 'cu') || ($env{'form.form'} eq 'studentform')) && 
        ($env{'form.pickedcourse'})) {
        $loaditem{'onload'} .= 'setRoles();setSections();';
    }
    my $js = &js_changer();
    $r->print(&Apache::loncommon::start_page($title,$js,
					     {'add_entries' => \%loaditem,
					      'no_nav_bar'  => 1, }));

    if ($env{'form.form'} eq 'portform') {
        $lastaction = 'document.courselist.submit()';
    } elsif ($env{'form.form'} eq 'cu' || ($env{'form.form'} eq 'studentform' &&
        !$multiple)) {
        $lastaction = 
             'document.courselist.pickedcourse.value = cdom+"_"+cname;'."\n".
             'document.courselist.submit();';
    } else {
        $lastaction = 'self.close()';
    }

    # if called to assign a role in a course to a user via CUSR
    if ($env{'form.form'} eq 'cu' || $env{'form.form'} eq 'studentform') {
        $r->print(&create_user_javascript($type));
    }

    # print javascript functions for choosing a course 
    if ((($env{'form.gosearch'}) && ($env{'form.updater'} eq '')) || 
        $onlyown) {
        $r->print(&gochoose_javascript($type,$multiple,$autosubmit,$lastaction));
    }
    $r->print('<script type="text/javascript" language="Javascript">'.$jscript.
              '</script>');
    $r->print($submitopener);

# ------------------------------------------ Display of filters to limit search
    my $filter = {};
    my $action = '/adm/pickcourse';
    my ($numtitles,$showroles,$nohost);
    if (!$onlyown) {
        my $filterlist = ['domainfilter','descriptfilter',
                          'instcodefilter'];
        if ($env{'form.form'} eq 'rules') {
            push(@{$filterlist},'personfilter');
            if (($env{'form.personfilter'} ne '') && ($env{'form.persondomfilter'} ne '')) {
                if (&Apache::lonnet::homeserver($env{'form.personfilter'},
                                                 $env{'form.persondomfilter'}) eq 'no_host') {
                    $nohost = 1;
                } else {
                    $showroles = 1;
                } 
            }
        } else {
            push(@{$filterlist},'ownerfilter');
        }
        # course ID filter for DCs only
        if ($env{'user.adv'} && $env{'form.domainfilter'} &&
            exists($env{'user.role.dc./'.$env{'form.domainfilter'}.'/'})) {
            push(@{$filterlist},'coursefilter');
        }
        if ($cloneruname ne '' && $clonerudom ne '') {
            push(@{$filterlist},'cloneableonly');
        }

        $r->print(&build_filters($filterlist,$type,$roleelement,$multelement,
                                 $filter,$action,\$numtitles,undef,$cloneruname,
                                 $clonerudom,$typeelement));
    }

# ---------------------------------------------------------------- Get the data
    if ((($env{'form.gosearch'}) && ($env{'form.updater'} eq '')) || 
         $onlyown) {
        my %courses = &search_courses($r,$type,$onlyown,$filter,$numtitles,
                                      $cloneruname,$clonerudom);
        if ($nohost) {
            $r->print ('<span class="LC_warning">'.
                       &mt('User does not exist - username: [_1], domain: [_2].',
                           '<b>'.$filter->{'personfilter'}.'</b>',
                           '<b>'.$filter->{'persondomfilter'}.'</b>').'</span>');
        } else {
            &display_matched_courses($r,$type,$multiple,$action,$showroles,$cloneruname,
                                     $clonerudom,%courses);
        }
    }
    $r->print(&Apache::loncommon::end_page());
    return OK;
}

sub js_changer {
    return <<"ENDJS";
<script type="text/javascript">

function updateFilters(caller) {
    if (typeof(caller) != "undefined") {
        document.filterpicker.updater.value = caller.name;
    }
    document.filterpicker.submit();
}

</script>

ENDJS

}

sub processpick {
    my $openerform = 'rolechoice';
    if ($env{'form.form'} eq 'studentform') {
        $openerform = $env{'form.form'};
    }
    my $process_pick = <<"ENDONE";
<script type="text/javascript">
function process_pick(dom) {
    var pickedCourse=opener.document.$openerform.$env{'form.cnumelement'}.value;
    var pickedDomain=opener.document.$openerform.$env{'form.cdomelement'}.value;
    var okDomain = 0;
ENDONE
    if ($openerform eq 'rolechoice') {
        $process_pick .= <<"ENDTWO";
    if (pickedDomain == dom) {
        if (pickedCourse != '') {
            var ccrole = "cc";
            var pickedType = "$env{'form.type'}";
            if (pickedType == "Community") {
                ccrole = "co";
            }
            var courseTarget = ccrole+"./"+pickedDomain+"/"+pickedCourse
            opener.document.title='Role selected. Please stand by.';
            opener.status='Role selected. Please stand by.';
            opener.document.rolechoice.newrole.value=courseTarget
            opener.document.rolechoice.submit();
        }
    } 
    else {
        alert("You may only use this screen to select courses in the current domain: "+dom+"\\nPlease return to the roles page window and click the 'Select Course' link for domain: "+pickedDomain+",\\n if you are a Domain Coordinator in that domain, and wish to become a Course Coordinator in a course in the domain");
    }
ENDTWO
    } else {
        $process_pick .= <<"ENDTHREE";
    if (pickedDomain != dom) {
        alert("You may only use this screen to select courses in the current domain: "+dom+"\\nPlease return to the roles page window and click the 'Select Course' link for domain: "+pickedDomain+",\\n if you are a Domain Coordinator in that domain, and wish to become a Course Coordinator in a course in the domain");
        return;
    }
ENDTHREE
    }
    $process_pick .= "
}

</script>
";
    return $process_pick;
}

sub create_user_javascript {
    my ($type) = @_;
    my $output;
    #javascript for reporting sections and groups then closing
    if ($env{'form.pickedcourse'}) {
        my %coursedescription = 
            &Apache::lonnet::coursedescription($env{'form.pickedcourse'},
                                               {'one_time' => '1'});
        my $cdom = $coursedescription{'domain'};
        my $cnum = $coursedescription{'num'};
        my $crstype = $coursedescription{'type'};
        my $sec_element = 'currsec';
        my $grplist_element = 'groups';
        my ($sections,$groups) =
          &Apache::loncommon::get_secgrprole_info($cdom,$cnum,'',$type);
        my $num_sections = scalar(@{$sections});
        my $seclist = join(',',@{$sections});
        my $num_groups = scalar(@{$groups});
        my $groupslist = join(',',@{$groups});
        $output = qq|
<script type="text/javascript">
function setSections() {
    opener.document.$env{"form.form"}.$grplist_element.value='$groupslist';
    window.opener.setSect('$seclist');
    self.close();
}
function setRoles() {
    window.opener.setRole('$crstype');
}
</script>
|;
;
    }
    return $output;
}

sub display_matched_courses {
    my ($r,$type,$multiple,$action,$showroles,$cloneruname,$clonerudom,%courses) = @_;
    if ($env{'form.form'} eq 'portform') {
        $action = '/adm/portfolio';
    }
    my $numcourses = keys(%courses);
    $r->print('<form name="courselist" method="post" action="'.$action.'">');
    if ($env{'form.form'} eq 'modifycourse') {
        if ($numcourses > 0) {
            my $ccrole = 'cc';
            if ($type eq 'Community') {
                $ccrole = 'co';
            }
            my $cctitle = &Apache::lonnet::plaintext($ccrole,$type);
            my $dctitle = &Apache::lonnet::plaintext('dc');
            my $ccrolechk = ' ';
            my $menuchk = ' checked="checked" ';
            $r->print(
                '<fieldset>'
               .'<legend>'.&mt('Action').'</legend>'
               .'<div class="LC_nobreak"><label>'
               .'<input type="radio" name="phase" value="ccrole"'.$ccrolechk.'/>'
               .'&nbsp;');
            if ($type eq 'Community') {
                $r->print(&mt('Enter the community with the role of [_1].',$cctitle));
            } else {
                $r->print(&mt('Enter the course with the role of [_1].',$cctitle));
            }
            $r->print('</label></div>'
               .'<div class="LC_nobreak"><label>'
               .'<input type="radio" name="phase" value="menu"'.$menuchk.'/>&nbsp;');
            if ($type eq 'Community') {
                $r->print(&mt('View or modify community settings which only a [_1] may modify.',$dctitle));
            } else {
                $r->print(&mt('View or modify course settings which only a [_1] may modify.',$dctitle));
            }
            $r->print('</label></div>'
               .'</fieldset>'
               .'<br />'
            );
        }
    }
    my %by_descrip;
    foreach my $course (keys(%courses)) {
        my $descr;
        if (ref($courses{$course}) eq 'HASH') {
            $descr = $courses{$course}{'description'};
        } elsif ($courses{$course} =~ m/^([^:]*):/i) {
            $descr = &unescape($1);
        } else {
            $descr = &unescape($courses{$course});
        }
        my $description = $descr;
        push (@{$by_descrip{$description}}, $course);
    }
 
    if ($numcourses > 1 && $multiple) {
        $r->print('<input type="button" value="check all"
                  onclick="javascript:checkAll(document.courselist.course_id)" />
                  &nbsp;&nbsp;<input type="button" value="uncheck all"
                  onclick="javascript:uncheckAll(document.courselist.course_id)" />
                  <br /><br />');
    }

    if (%courses) {
        $r->print(&Apache::loncommon::start_data_table());
        $r->print(&Apache::loncommon::start_data_table_header_row());
        my $titlehdr = &mt('Course Title');
        if ($type eq 'Community') {
            $titlehdr = &mt('Community Title');
        }
        $r->print('<th>'.&mt('Select').'</th>'
                 .'<th>'.$titlehdr.'</th>'
                 .'<th>'.&mt('Domain').'</th>');
        unless ($type eq 'Community') {
            $r->print('<th>'.&mt('Course Code').'</th>');
        }
        $r->print('<th>'.&mt('Owner/Co-owner(s)').'</th>');
        if ($showroles) {
            $r->print('<th>'.&mt("Role(s) for [_1]",
                &Apache::loncommon::plainname($env{'form.personfilter'},
                                              $env{'form.persondomfilter'},'firstname')).'</th>');
        }
        $r->print(&Apache::loncommon::end_data_table_header_row());
    }
    my %cc_cloneable;
    if (($env{'form.form'} eq 'ccrs') || ($env{'form.form'} eq 'requestcrs')) {
        my ($coord_cloneable,$warning) =
            &get_coordinator_cloneable($cloneruname,$clonerudom,$type);
        if ($coord_cloneable) {
            map {$cc_cloneable{$_} = 1;} split('&',$coord_cloneable);
        }
    }
    foreach my $description (sort { lc($a) cmp lc($b) } (keys(%by_descrip))) {
        foreach my $course (@{$by_descrip{$description}}) {
            $r->print(&Apache::loncommon::start_data_table_row());
            my $cleandesc=&HTML::Entities::encode($description,'<>&"');
            $cleandesc=~s/'/\\'/g;
            my ($cdom,$cnum)=split(/\_/,$course);
            my ($descr,$instcode,$ttype,$canclone,@owners);
            if (ref($courses{$course}) eq 'HASH') {
                $descr = $courses{$course}{'description'};
                $instcode = $courses{$course}{'inst_code'};
                $ttype = $courses{$course}{'type'};
                if (($env{'form.form'} eq 'ccrs') || ($env{'form.form'} eq 'requestcrs')) {
                    if ($cc_cloneable{$courses{$course}{'cnum'}.':'.$courses{$course}{'domain'}}) {
                        $canclone = 1;
                    }
                    unless ($canclone) {
                        my $cloners = $courses{$course}{'cloners'};
                        if ($cloners ne '') { 
                            my @cloneable = split(',',$cloners);
                            if (grep(/^\*$/,@cloneable)) {
                                $canclone = 1;
                            }
                            if (grep(/^\*:\Q$env{'form.clonerudom'}\E$/,@cloneable)) {
                                $canclone = 1;
                            }
                            if (grep(/^\Q$cloneruname\E:\Q$clonerudom\E$/,@cloneable)) {
                                $canclone = 1;
                            }
                        }
                    }
                }
                push(@owners,&unescape($courses{$course}{'owner'}));
                if (ref($courses{$course}{'co-owners'}) eq 'ARRAY') {
                    foreach my $item (@{$courses{$course}{'co-owners'}}) {
                        push(@owners,&unescape($item));
                    }
                }
            } else {
                my $singleowner; 
                ($descr,$instcode,$singleowner,$ttype)=split(/:/,$courses{$course});
                push(@owners,&unescape($singleowner));
            }
            my $ownerstr = join(', ',map { &Apache::loncommon::plainname(split(':',$_)); } @owners);
            $r->print('<td>'.&course_chooser($multiple,$cdom,$cnum,$cleandesc,$canclone).'</td>');
            $r->print('<td>'.$description.'</td>');
            $r->print('<td>');
            $r->print(&Apache::lonnet::domain($cdom,'description')?
                      $cdom.' ('.&Apache::lonnet::domain($cdom,'description').')':$cdom);
            $r->print('</td>');
            unless ($type eq 'Community') { 
                $r->print('<td>');
                if ($instcode ne '') {
                    $r->print(&unescape($instcode));
                } else {
                    $r->print('&nbsp;');
                }
                $r->print('</td>');
            }
            $r->print('<td>'.$ownerstr.'</td>');
            if ($showroles) {
                $r->print('<td>');
                my $rolestr;
                if (ref($courses{$course}{'roles'}) eq 'ARRAY') {
                    my @roles = sort(@{$courses{$course}{'roles'}});
                    foreach my $role (@roles) {
                        if ($role =~ /^cr/) {
                            my (undef,$crdom,$crname,$crtitle) = split('/',$role);
                            $rolestr .= $crtitle.', ';
                        } else {
                            $rolestr .= &Apache::lonnet::plaintext($role,$type).', ';
                        }
                    }
                    $rolestr =~ s/\, $//;
                }
                $r->print($rolestr.'</td>');
            }
            if ($multiple) { $r->print("</label>\n"); }
            $r->print(&Apache::loncommon::end_data_table_row());
            # $r->print("<br />\n");
        }
    }
    if (%courses) {
        $r->print(&Apache::loncommon::end_data_table());
    }

    if (!%courses) {
        $r->print(&mt('None found'));
    } elsif ($multiple) {
        $r->print('<input type="button" value="Submit" onClick="gochoose('."'','','')".'" />');
    }
    $r->print('<input type="hidden" name="form" value="'.$env{'form.form'}.'" />'.
              "\n".'<input type="hidden" name="pickedcourse" value="" />'."\n".
              '<input type="hidden" name="type" value="'.$type.'" />'."\n");
    if ((exists($env{'form.roleelement'})) && ($env{'form.form'} eq 'rolechoice')) {
        $r->print('<input type="hidden" name="roleelement" value="'.
                  $env{'form.roleelement'}.'" />'."\n");
    }
    if ($env{'form.form'} eq 'portform') {
        $r->print('<input type="hidden" name="cnum" value="" />');
        $r->print('<input type="hidden" name="cdom" value="" />');
        $r->print('<input type="hidden" name="setroles" value="'.$env{'form.setroles'}.'" />');
        $r->print('<input type="hidden" name="action" value="rolepicker" />');
    } elsif ($env{'form.form'} eq 'modifycourse') {
        $r->print(&Apache::lonhtmlcommon::echo_form_input(['phase','pickedcourse','type','form','numtitles','state']));
    } else {
        $r->print('<input type="hidden" name="cnumelement" value="'.
                  $env{'form.cnumelement'}.'" />'."\n".  
                  '<input type="hidden" name="cdomelement" value="'.
                  $env{'form.cdomelement'}.'" />'."\n");
        if ($env{'form.typeelement'} ne '') {
            $r->print('<input type="hidden" name="typeelement" value="'.
                      $env{'form.typeelement'}.'" />'."\n");

        }
    }
    if ((exists($env{'form.fixeddom'})) && ($env{'form.form'} eq 'rules')) {
        $r->print('<input type="hidden" name="fixeddom" value="'.
                  $env{'form.fixeddom'}.'" />');
    }
    if ($env{'form.numtitles'}) {
        $r->print('<input type="hidden" name="numtitles" value="'.
                  $env{'form.numtitles'}.'" />');
    }
    $r->print("</form>\n");
    return;
}

sub multiples_tag {
    my $jscript = &Apache::loncommon::check_uncheck_jscript();
    my $multelement = '<input type="hidden" name="multiple" value="1" />';
    return ($jscript,$multelement);
}

sub build_filters {
    my ($filterlist,$type,$roleelement,$multelement,$filter,$action,
        $numtitlesref,$caller,$cloneruname,$clonerudom,$typeelement) = @_;
    my ($list,$formname,$fixeddom,$codedom,$jscript);
    $codedom = $env{'request.role.domain'};
    if (defined($env{'form.form'})) {
        $formname = $env{'form.form'};
    } else {
        $formname = $caller;
    }
    my $onchange;
    unless ($env{'form.interface'} eq 'textual') {
        $onchange = 'javascript:updateFilters(this)';
    }
    my ($domainselectform,$sincefilterform,$ownerdomselectform,$persondomselectform,
        $instcodeform,$typeselectform,$instcodetitle);
    foreach my $item (@{$filterlist}) {
        $filter->{$item} = $env{'form.'.$item};
        if ($item ne 'descriptfilter' && $item ne 'instcodefilter') {
	    if ($item eq 'domainfilter') {
		$filter->{$item} = &LONCAPA::clean_domain($filter->{$item});
	    } elsif ($item eq 'coursefilter') {
		$filter->{$item} = &LONCAPA::clean_courseid($filter->{$item});
	    } elsif ($item eq 'ownerfilter') {
		$filter->{$item} = &LONCAPA::clean_username($filter->{$item});
                $filter->{'ownerdomfilter'} = 
                    &LONCAPA::clean_domain($env{'form.ownerdomfilter'});
                $ownerdomselectform =
                    &Apache::loncommon::select_dom_form($filter->{'ownerdomfilter'},
                                                        'ownerdomfilter',1);
            } elsif ($item eq 'personfilter') {
                $filter->{$item} = &LONCAPA::clean_username($filter->{$item});
                if ($env{'form.persondomfilter'} eq '') {
                    unless ($env{'form.gosearch'}) {
                        $filter->{'persondomfilter'} = $env{'request.role.domain'};
                    }
                } else {
                    $filter->{'persondomfilter'} =
                        &LONCAPA::clean_domain($env{'form.persondomfilter'});
                }
                $persondomselectform =
                    &Apache::loncommon::select_dom_form($filter->{'persondomfilter'},
                                                        'persondomfilter',1);
	    } else {
		$filter->{$item} =~ s/\W//g;
	    }
            if (!$filter->{$item}) {
                $filter->{$item} = '';
            }
        }
        if ($item eq 'domainfilter') {
	    my $allow_blank = 1;
	    if ($formname eq 'portform') {
		$filter->{$item} ||= $env{'user.domain'};
		$allow_blank=0;
	    } elsif ($formname eq 'studentform') {
                $filter->{$item} ||= $env{'request.role.domain'};
                $allow_blank=0;
            }
            if ($env{'form.fixeddom'}) {
                $domainselectform = '<input type="hidden" name="domainfilter"'.
                                    'value="'.$env{'request.role.domain'}.'" />'.
                                    &Apache::lonnet::domain($env{'request.role.domain'},
                                                            'description');
                $codedom = $env{'request.role.domain'};
            } else {
                $domainselectform =
		    &Apache::loncommon::select_dom_form($filter->{$item},
		    				        'domainfilter',
						        $allow_blank,'',$onchange);
                $codedom = $filter->{'domainfilter'};
            }
        } else {
            $list->{$item} = &HTML::Entities::encode($filter->{$item},'<>&"');
        }
    }

    # last course activity filter and selection
    $filter->{'sincefilter'} = $env{'form.sincefilter'};
    $filter->{'sincefilter'} =~ s/[^\d-]//g;
    if (!$filter->{'sincefilter'}) { $filter->{'sincefilter'}=-1; }
    $sincefilterform=&Apache::loncommon::select_form(
        $filter->{'sincefilter'},
        'sincefilter',
        (      '-1' => '',
            '86400' => &mt('today'),
           '604800' => &mt('last week'),
          '2592000' => &mt('last month'),
          '7776000' => &mt('last three months'),
         '15552000' => &mt('last six months'),
         '31104000' => &mt('last year'),
         'select_form_order' =>
            ['-1','86400','604800','2592000','7776000',
             '15552000','31104000']));

    my %lt = &Apache::lonlocal::texthash(
                'cac' => "$type Activity",
                'cde' => "$type Title",
                'cdo' => "$type Domain",
                'ins' => 'Institutional Code',
                'inc' => 'Institutional Categorization',
                'cow' => "$type Owner/Co-owner",
                'cop' => "$type Personnel Includes",
                'cog' => 'Type',
             );

    if (($env{'form.form'} eq 'ccrs') || ($env{'form.form'} eq 'requestcrs')) {
        my $typeval = 'Course';
        if ($type eq 'Community') {
            $typeval = 'Community';
        }
        $typeselectform = '<input type="hidden" name="type" value="'.$typeval.'" />';
    } else {
        $typeselectform =  '<select name="type" size="1"';
        if ($onchange) {
            $typeselectform .= 'onchange="'.$onchange.'"';
        }
        $typeselectform .= '>'."\n";
        foreach my $posstype ('Course','Community') {
            $typeselectform.='<option value="'.$posstype.'"'.
                ($posstype eq $type ? ' selected="selected" ' : ''). ">".&mt($posstype)."</option>\n";
        }
        $typeselectform.="</select>";
    }
    
    my ($cloneableonlyform,$cloneabletitle);
    if (exists($filter->{'cloneableonly'})) {
        my $cloneableon = '';
        my $cloneableoff = ' checked="checked"';
        if ($filter->{'cloneableonly'}) {
            $cloneableon = $cloneableoff;
            $cloneableoff = '';
        }
        $cloneableonlyform = '<span class="LC_nobreak"><label><input type="radio" name="cloneableonly" value="1" '.$cloneableon.'/>&nbsp;'.&mt('Required').'</label>'.('&nbsp;'x3).'<label><input type="radio" name="cloneableonly" value="" '.$cloneableoff.' />&nbsp;'.&mt('No restriction').'</label></span>';
        if ($env{'form.form'} eq 'ccrs') {
            $cloneabletitle = &mt('Cloneable for').' '.$cloneruname.':'.$clonerudom;
        } else {
            $cloneabletitle = &mt('Cloneable by you');
        }
    }
    my $officialjs;
    if ($type eq 'Course') {
        if (exists($filter->{'instcodefilter'})) {
            if (($env{'form.fixeddom'}) || ($formname eq 'requestcrs')
                || ($formname eq 'modifycourse')) {
                $officialjs = 1;
                ($instcodeform,$jscript,$$numtitlesref) = 
                    &instcode_selectors($codedom,'filterpicker',$officialjs);
                if ($jscript) {
                    $jscript = '<script type="text/javascript" language="Javascript">'.
                               $jscript.'</script>'."\n";
                }
            }
            if ($instcodeform eq '') {
                $instcodeform = 
                    '<input type="text" name="instcodefilter" size="10" value="'.
                    $list->{'instcodefilter'}.'" />';
                $instcodetitle = $lt{'ins'};
            } else {
                $instcodetitle = $lt{'inc'};
            }
            if ($env{'form.fixeddom'}) {
                $instcodetitle .= '<br />('.$codedom.')';
            }
        }
    }

    my $output = qq|
<form method="post" name="filterpicker" action="$action">
<input type="hidden" name="form" value="$formname" />
|;
    if ($formname eq 'modifycourse') {
        $output .= '<input type="hidden" name="phase" value="courselist" />'."\n".
                   '<input type="hidden" name="prevphase" value="'.
                   $env{'form.prevphase'}.'" />'."\n";
    } else {
        my $name_input;
        if ($env{'form.cnameelement'} ne '') {
            $name_input = '<input type="hidden" name="cnameelement" value="'.
                          $env{'form.cnameelement'}.'" />';
        }
        $output .= qq|
<input type="hidden" name="cnumelement" value="$env{'form.cnumelement'}" />
<input type="hidden" name="cdomelement" value="$env{'form.cdomelement'}" />
$name_input
$roleelement
$multelement
$typeelement
|;
        if ($formname eq 'portform') {
            $output .= '<input type="hidden" name="setroles" value="'.$env{'form.setroles'}.'" />'."\n";
        }
    }
    if ($env{'form.fixeddom'}) { 
        $output .= '<input type="hidden" name="fixeddom" value="'.$env{'form.fixeddom'}.'" />'."\n";
    }
    $output .= "<br />\n".&Apache::lonhtmlcommon::start_pick_box();
    if ($sincefilterform) { 
        $output .= &Apache::lonhtmlcommon::row_title($lt{'cac'})
                  .$sincefilterform
                  .&Apache::lonhtmlcommon::row_closure();
    }
    if ($domainselectform) {
        $output .= &Apache::lonhtmlcommon::row_title($lt{'cdo'})
                  .$domainselectform
                  .&Apache::lonhtmlcommon::row_closure();
    } 
    if ($typeselectform) {
        if (($env{'form.form'} eq 'ccrs') || ($env{'form.form'} eq 'requestcrs')) {
            $output .= $typeselectform;
        } else {
            $output .= &Apache::lonhtmlcommon::row_title($lt{'cog'})
                      .$typeselectform
                      .&Apache::lonhtmlcommon::row_closure();
        }
    }
    if ($instcodeform) {
        $output .= &Apache::lonhtmlcommon::row_title($instcodetitle)
                  .$instcodeform
                  .&Apache::lonhtmlcommon::row_closure();
    }
    if (exists($filter->{'ownerfilter'})) {
        $output .= &Apache::lonhtmlcommon::row_title($lt{'cow'}).
                   '<table><tr><td>'.&mt('Username').'<br />'.
                   '<input type="text" name="ownerfilter" size="20" value="'.
                   $list->{'ownerfilter'}.'" /></td><td>'.&mt('Domain').'<br />'.
                   $ownerdomselectform.'</td></tr></table>'.
                   &Apache::lonhtmlcommon::row_closure();
    }
    if (exists($filter->{'personfilter'})) {
        $output .= &Apache::lonhtmlcommon::row_title($lt{'cop'}).
                   '<table><tr><td>'.&mt('Username').'<br />'.
                   '<input type="text" name="personfilter" size="20" value="'.
                   $list->{'personfilter'}.'" /></td><td>'.&mt('Domain').'<br />'.
                   $persondomselectform.'</td></tr></table>'.
                   &Apache::lonhtmlcommon::row_closure();
    }
    if (exists($filter->{'coursefilter'})) {
        $output .= &Apache::lonhtmlcommon::row_title(&mt('LON-CAPA course ID'))
                  .'<input type="text" name="coursefilter" size="25" value="'
                  .$list->{'coursefilter'}.'" />'
                  .&Apache::lonhtmlcommon::row_closure();
    }
    if ($cloneableonlyform) {
        $output .= &Apache::lonhtmlcommon::row_title($cloneabletitle).
                   $cloneableonlyform.&Apache::lonhtmlcommon::row_closure();
    }
    if (exists($filter->{'descriptfilter'})) {
        $output .= &Apache::lonhtmlcommon::row_title($lt{'cde'})
                  .'<input type="text" name="descriptfilter" size="40" value="'
                  .$list->{'descriptfilter'}.'" />'
                  .&Apache::lonhtmlcommon::row_closure(1);
    }
    $output .= &Apache::lonhtmlcommon::end_pick_box().'<p>';
    my ($coord_cloneable,$warning);
    if (($env{'form.form'} eq 'ccrs') || ($env{'form.form'} eq 'requestcrs')) {
        ($coord_cloneable,$warning) = 
            &get_coordinator_cloneable($cloneruname,$clonerudom,$type);
        if ($env{'form.form'} eq 'ccrs') {
            $output .= '<input type="hidden" name="cloner" value="'.$env{'form.cloner'}.'" />'."\n";
        }
        if ($coord_cloneable) {
            $output .= '<input type="hidden" name="cc_clone" value="'.$coord_cloneable.'" />';
        }
    }
    $output .= '<input type="hidden" name="updater" value="">'."\n".
               '<input type="submit" name="gosearch" value="'. 
               &mt('Search').'" /></p>'."\n".'</form>'."\n".'<hr />'."\n";
    return $jscript.$warning.$output;
}

sub get_coordinator_cloneable {
    my ($cloneruname,$clonerudom,$type) = @_;
    if (&Apache::lonnet::homeserver($cloneruname,$clonerudom) eq 'no_host') {
        my $warning = '<div class="LC_error">'.&mt('Intended course owner does not exist').
                   '</div>';
        return ('',$warning);
    } else {
        my ($cc_clone,$ccrole);
        if ($type eq 'Community') {
            $ccrole = 'co';
        } elsif ($type eq 'Course') {
            $ccrole = 'cc';
        }
        my %ccroles = &Apache::lonnet::get_my_roles($cloneruname,$clonerudom,
                                                    'userroles',['active'], [$ccrole]);
        foreach my $key (sort(keys(%ccroles))) {
            my ($cnum,$cdom,$role) = split(':',$key);
            $cc_clone .= $cdom.':'.$cnum.'&';
        }
        $cc_clone =~ s/\&$//;
        return ($cc_clone);
    }
}

sub instcode_selectors {
    my ($codedom,$formname,$officialjs) = @_;
    my ($output,@codetitles,%cat_titles,%cat_order,%cat_items);
    my ($jscript,$totcodes,$numtitles,$lasttitle) =
        &Apache::courseclassifier::instcode_selectors_data($codedom,$formname,
                           \%cat_items,\@codetitles,\%cat_titles,\%cat_order,$officialjs);
    if ($numtitles > 0) {
        my $official = ' checked="checked" ';
        my $unofficial = '';
        if ($env{'form.official'} eq 'off') {
            $unofficial = $official;
            $official = '';
        }
        $output .= '<span class="LC_nobreak">'.&mt('Official course:').'&nbsp;<label>'.
                   '<input type="radio" name="official" value="on"'.$official.' />'.
                   &mt('Yes').'</label>'.('&nbsp;'x3).'<label>'.
                   '<input type="radio" name="official" value="off"'.$unofficial.
                   ' onclick="toggleOfficial();" />'.&mt('No').'</label></span><br />'.
                   &Apache::courseclassifier::build_instcode_selectors($numtitles,
                   $lasttitle,\%cat_items,\@codetitles,\%cat_titles,\%cat_order)."\n".
                   '<input type="hidden" name="numtitles" value="'.$numtitles.'" />'."\n".
                   '<input type="hidden" name="state" value="listing" />'."\n";
 
    }
    return ($output,$jscript,$numtitles);
}

sub search_courses {
    my ($r,$type,$onlyown,$filter,$numtitles,$cloneruname,$clonerudom) = @_;
    my (%courses,%showcourses,$cloner,$ccrole);
    if ($type eq 'Community') {
        $ccrole = 'co';
    } else {
        $ccrole = 'cc';
    }
    if (!$onlyown) {
        $r->print(&mt('Searching ...').'<br />&nbsp;<br />');
        $r->rflush();
        if (($filter->{'ownerfilter'} ne '') ||
            ($filter->{'ownerdomfilter'} ne '')) {
            $filter->{'combownerfilter'} = $filter->{'ownerfilter'}.':'.
                                           $filter->{'ownerdomfilter'};
        }
        foreach my $item ('descriptfilter','coursefilter','combownerfilter') {
            if (!$filter->{$item}) {
                $filter->{$item}='.';
            }
        }
        my $timefilter = 
           ($filter->{'sincefilter'}==-1?1:time-$filter->{'sincefilter'});
        my ($instcodefilter,$regexpok);
        if ($numtitles) {
            if ($env{'form.official'} eq 'on') {
                $instcodefilter = 
                    &Apache::courseclassifier::instcode_search_str($filter->{'domainfilter'},$numtitles);
                $regexpok = 1;
            } elsif ($env{'form.official'} eq 'off') {
                $instcodefilter = &Apache::courseclassifier::instcode_search_str($filter->{'domainfilter'},$numtitles);
                unless ($instcodefilter eq '') {
                    $regexpok = -1;
                }
            }
        } else {
            $instcodefilter = $filter->{'instcodefilter'};
        }
        if ($instcodefilter eq '') { $instcodefilter = '.'; }
        if ($type eq '') { $type = '.'; }
 
        if (($clonerudom ne '') && ($cloneruname ne '')) {
            $cloner = $cloneruname.':'.$clonerudom;  
        }
        %courses = 
            &Apache::lonnet::courseiddump($filter->{'domainfilter'},
                                          $filter->{'descriptfilter'},
                                          $timefilter,
                                          $instcodefilter,
                                          $filter->{'combownerfilter'},
                                          $filter->{'coursefilter'},
                                          undef,undef,$type,$regexpok,undef,undef,
                                          undef,undef,$cloner,$env{'form.cc_clone'},
                                          $filter->{'cloneableonly'});
        if (($filter->{'personfilter'} ne '') && ($filter->{'persondomfilter'} ne '')) {
            my %rolehash = &Apache::lonnet::get_my_roles($filter->{'personfilter'},
                                                         $filter->{'persondomfilter'},
                                                         'userroles',undef,
                                                         [$ccrole,'in','ad','ep','ta','cr'],
                                                         $filter->{'domainfilter'});
            foreach my $role (keys(%rolehash)) {
                my ($cnum,$cdom,$courserole) = split(':',$role);
                my $cid = $cdom.'_'.$cnum;
                if (exists($courses{$cid})) {
                    if (ref($courses{$cid}) eq 'HASH') {
                        if (ref($courses{$cid}{roles}) eq 'ARRAY') {
                            if (!grep(/^\Q$courserole\E$/,@{$courses{$cid}{roles}})) { 
                                push (@{$courses{$cid}{roles}},$courserole);
                            }
                        } else {
                            $courses{$cid}{roles} = [$courserole];
                        }
                        $showcourses{$cid} = $courses{$cid};
                    }
                }
            }
            %courses = %showcourses;
        }
    } else {
        $r->print('<br />');
        my %coursehash = &Apache::loncommon::findallcourses(); 
        foreach my $cid (sort(keys(%coursehash))) {
            $courses{$cid}{'description'} = $env{'course.'.$cid.'.description'};
        }
    }
    return %courses;
}

sub course_chooser {
    my ($multiple,$cdom,$cnum,$cleandesc,$canclone) = @_;
    my $output; 
    if ($multiple) {
        $output = '<label><input type="checkbox" name="course_id" value="'.$cdom.'_'.$cnum.'" />'."\n";
    } elsif ((($env{'form.form'} eq 'ccrs') || ($env{'form.form'} eq 'requestcrs')) && (!$canclone))  {
        if ($env{'form.form'} eq 'ccrs') {
            $output = &mt('No cloning for ').$env{'form.cloner'}."\n";
        } else {
            $output = &mt('No rights to clone')."\n";
        }
    } else {
        $output = '<input type="button" value="'.&mt('Select').'" onClick="gochoose('.
                  "'".$cnum."','".$cdom."','".$cleandesc."')".'" />'."\n";
    }
    return $output;
}

sub gochoose_javascript {
    my ($type,$multiple,$autosubmit,$lastaction) = @_;
    my %elements = (
                     'Course' => {
                                 name  => 'coursepick',
                                 total => 'coursetotal',
                                 list  => 'courselist',
                                 },
                     'Community' => {
                                 name  => 'coursepick',
                                 total => 'coursetotal',
                                 list  => 'courselist',
                              },
                    );
    my $output .= qq|
<script type="text/javascript">
function gochoose(cname,cdom,cdesc) {
    var openerForm = "$env{'form.form'}";
    courseCount = 0;
    var courses = '';
|;
    if ($multiple) {
        $output .= <<"ENDSCRIPT";
    courseCount = 0;
    var courses = '';
    if (typeof(document.courselist.course_id.length) == 'undefined') {
        // only 1 course checkbox was created
        if (document.courselist.course_id.checked) {
            courses = courses + document.courselist.course_id.value + "&&";
            courseCount ++;
        }
    } else {
        for (var j=0; j<document.courselist.course_id.length; j++) {
            if (document.courselist.course_id[j].checked) {
                courses = courses + document.courselist.course_id[j].value + "&&";
                courseCount ++;
            }
        }
    }
    opener.document.$env{'form.form'}.$elements{$type}{'total'}.value = courseCount;
    if (typeof(opener.document.$env{'form.form'}.$elements{$type}{'name'}.length) ==
        'undefined') {
        if (opener.document.$env{'form.form'}.$elements{$type}{'name'}.value == 'specific') {
            opener.document.$env{'form.form'}.$elements{$type}{'name'}.checked = true;
        } else {
            opener.document.$env{'form.form'}.$elements{$type}{'name'}.checked = false;
        }
    } else {
        for (var j=0; j<opener.document.$env{'form.form'}.$elements{$type}{'name'}.length; j++) {
            if (opener.document.$env{'form.form'}.$elements{$type}{'name'}\[j].value == 'specific') {
                opener.document.$env{'form.form'}.$elements{$type}{'name'}\[j].checked = true;
            } else {
                opener.document.$env{'form.form'}.$elements{$type}{'name'}\[j].checked = false;
            }
        }
    }
    if (courseCount > 0) {
        courses = courses.substr(0,courses.length-2);
        opener.document.$env{'form.form'}.$elements{$type}{'list'}.value = courses;
    }
ENDSCRIPT
    } else {
        my ($name_code,$type_code);
        if ($env{'form.cnameelement'} ne '') {
            $name_code = 'opener.document.'.$env{'form.form'}.'.'.
                          $env{'form.cnameelement'}.'.value=cdesc;';
        }
        if ($env{'form.typeelement'} ne '') {
            $type_code = 'opener.document.'.$env{'form.form'}.'.'.
                          $env{'form.typeelement'}.'.value=document.courselist.type;';
        }

        $output .= qq|
        $name_code
        $type_code
        opener.document.$env{'form.form'}.$env{'form.cnumelement'}.value=cname;
        var slct=opener.document.$env{'form.form'}.$env{'form.cdomelement'};
        if (slct.options == undefined) {
            opener.document.$env{'form.form'}.$env{'form.cdomelement'}.value=cdom;
        }
        else {
            var i;
            for (i=0;i<slct.length;i++) {
                if (slct.options[i].value==cdom) { slct.selectedIndex=i; }
            }
        }
|;
    }
    $output .= qq|
    if (openerForm == 'portform') {
        document.courselist.cnum.value = cname;
        document.courselist.cdom.value = cdom;
    }
    $autosubmit
    $lastaction
}
</script>
|;
    return $output;
}

1;
__END__

=pod

=head1 NAME

Apache::lonpickcourse - Search for course(s) based on user-specified criteria.   

=head1 SYNOPSIS

Invoked by other LON-CAPA modules, when course(s) need to be selected by the user. 

=head1 OVERVIEW

Two screens are typically displayed to the user.  The first is a set of criteria which are used to constrain the search for courses.

=head2 Search Criteria (Screen One)

=head3 Criteria:

=over 4

=item *
Course Activity - how recently was course last visited by anyone.

=item *
Course Domain - the domain of the course

=item *
Type - Course or Community

=item *
Course Institutional Code - the institutional identifier assigned to the course

=item * 
Course Owner's Username - the username of the owner of the course (assigned by the Domain Coordinator and/or when the course was created).

=item *
Course Owner's Domain - the domain of the owner of the course

=item * 
Course Title - text which appears in the Course Title, as set in the Course Parameters.

=item *
Course ID - the internal course number (course ID part after initial 'domain_') used by LON-CAPA (this criterion is only displayed to Domain Coordinators selecting a course in the same domain as their DC role).

=back

The criteria setting screen is not displayed if course picking is done by a user who does not have advanced privileges (as defined by $env{'user.adv'}).

=head2 Course Display (Screen Two)

A list of courses matching the search criteria is displayed.  If the user is not an advanced user, screen one will have been skipped and the courses displayed will be all courses in which the user has currently active roles. The information displayed for each course is:

=over 4

=item *
Course description

=item *
Domain description of course domain

=item *
Course institutional code

=item * 
Course owner (username:domain)   
 
=back

Depending on context, the display may include a single select box for each course, allowing selection of only a single course, or may include checkboxes allowing selection of more than one course.

Following selection, and/or submission, the course description, number and domain are transferred to the browser window from which the course picker window was opened.  In most cases, the child window containing the course picker screens will be closed.  However, in some cases closure will be delayed until a third screen has been displayed (e.g., setting of course-based conditional access controls for portfolio files).  In this case the page is generated via /adm/portfolio and the page features select boxes to allow the user to select roles, access types, sections and groups.

=head1 SUBROUTINES

=over 4

=item *
X<create_user_javascript()>
B<create_user_javascript($type)>:

Input: 1 - $type  - the course type - Course or Community

Output: 1 - $output - javascript wrapped in E<lt>scriptE<gt>E<lt>/scriptE<gt> tags 

Side Effects: None 

javascript code for reporting selected sections (as a string of comma separated sections) and groups in the selected course (as a comma separated list) then calling setSect() javscript function in the opener window (to populate section select box) then closing current window.


=item *
X<display_matched_courses()>
B<display_matched_courses($r,$type,$multiple,$action,$showroles,$cloneruname,$clonerudom,%courses)>:

Input: 7 - request object, course type, multiple (0 or 1), form action, whether to show roles (for course personnel filter), username of new course owner, domain of new course owner, hash of courses.

Output: 0

Side Effects: prints select buttons (multiple = 0) or checkboxes (multiple = 1) and hidden form elements for selection of one or more courses which met search criteria.

=item *
X<multiples_tag()>
B<multiples_tag()>:


Input: 0

Output: 2 - $jscript - javascript for check all/uncheck all checkboxes; $multelement - hidden form element with multiple set to 1.

Side Effects: None

=item *
X<build_filters()>
B<build_filters($filterlist,$type,$roleelement,$multelement,$filter,$action,$numfiltersref,$caller,$cloneruname,$clonerudom,$typeelem)>:


Input: 11 - anonymous array of search criteria; course type; $roleelement ; $multelement ; anonymous hash of criteria and their values; form action; ref to scalar (count of number of elements in institutional codes -- e.g., 4 for year, semester, department, and number); caller context (e.g., set to 'modifycourse' when routine is called from lonmodifycourse.pm), username and domain of new course owner who wants to clone, $tyepelem 

Output: 1 - $output - HTML for display of search criteria, and hidden form elements. 

Side Effects: None

=item *
X<search_courses()>
B<search_courses($r,$type,$onlyown,$filter,$numtitles,$cloneruname,$clonerudom)>:


Input: 7 -  request object, course type, search scope: only courses in which user has active role (1), or any course (0); anonymous hash of criteria and their values; for institutional codes - number of categories; optional username of new course owner, optional domain of new course owner). Last two needed when search is for courses to clone from course request form, or course creation form (DC).   

Output: 1 - %courses - hash of courses satisfying search criteria, keys = course IDs, values are corresponding colon-separated escaped description, institutional code, owner and type.

Side Effects: None


=item *
X<course_chooser()>
B<course_chooser($multiple,$cdom,$cnum,$cleandesc,$canclone)>:

Input: 5 - single (0) or multiple (1) courses; course domain, course number; course description; can clone course (1 if new course owner has cloning rights). 

Output: 1 - HTML for either checkbox (multiple=1) or select button (multiple=0) for user to indicate course selection.

Side Effects: None


=item *
X<gochoose_javascript()>
B<gochoose_javascript($type,$multiple,$autosubmit,$lastaction)>:

Input: 4 - course type; single (0) or multiple courses (1); in context of DC selecting a CC role in a course: javascript code from &processpick(); final action to take after user chooses course(s):  either close window, or submit form for display of next page etc.

Output: 1  $output - javascript wrapped in E<lt>scriptE<gt>E<lt>/scriptE<gt> tags

Side Effects: None

javascript functions used when user selects a course(s). Different behavior depending on context:

=back

=over 8

=item

(a) Domain Coordinator using MAIL to select recipients of broadcast e-mail - && separated list of selected courses written to hidden form element in opener window. Child window closes.

=item

(b) Domain Coordinator choosing a course for adoption of a CC role from roles screen - write course identifying information to hidden form elements in opener window and automatically submit role selection form in opener window. Child window closes.

=item

(c) Domain Coordinator creating a course, and selecting a course to clone - course number and domain written to visible form elements in opener window. Child window closes.

=item

(d) User selecting a course for course-based conditional access control for a portfolio file - form is submitted, and new page is displayed for selection of roles, access types, sections and groups to be used in conditional ACL. New page is generated by /adm/portfolio. 

=item

(e) Domain Coordinator assigning a role to a user - form is submitted, and new page does an onload call to a javascript function to (a) write lists of sections and groups to hidden elements in opener window, (b) call function in opener window to dynamically populate select box showing current sections.

=item

(f) Author modifying a rights entry in a .rights file - selected course number and domain are witten to visible form elements in opener window.  Child window closes. 

=item

(g) Scantron Operator uploading a scantron file to a course - course number is written to visible form element in opener window. Child window closes.

=item

(h) User requesting creation of a course, and selecting a course to clone - course number and domain written to visible form elements in opener window. Child window closes.

=back
     
=cut
