# The LearningOnline Network
# TCP networking package
#
# $Id: lonnet.pm,v 1.1056.2.4 2010/05/22 13:50:19 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###

=pod

=head1 NAME

Apache::lonnet.pm

=head1 SYNOPSIS

This file is an interface to the lonc processes of
the LON-CAPA network as well as set of elaborated functions for handling information
necessary for navigating through a given cluster of LON-CAPA machines within a
domain. There are over 40 specialized functions in this module which handle the
reading and transmission of metadata, user information (ids, names, environments, roles,
logs), file information (storage, reading, directories, extensions, replication, embedded
styles and descriptors), educational resources (course descriptions, section names and
numbers), url hashing (to assign roles on a url basis), and translating abbreviated symbols to
and from more descriptive phrases or explanations.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 Package Variables

These are largely undocumented, so if you decipher one please note it here.

=over 4

=item $processmarker

Contains the time this process was started and this servers host id.

=item $dumpcount

Counts the number of times a message log flush has been attempted (regardless
of success) by this process.  Used as part of the filename when messages are
delayed.

=back

=cut

package Apache::lonnet;

use strict;
use LWP::UserAgent();
use HTTP::Date;
use Image::Magick;

use vars qw(%perlvar %spareid %pr %prp $memcache %packagetab $tmpdir
            $_64bit %env %protocol);

my (%badServerCache, $memcache, %courselogs, %accesshash, %domainrolehash,
    %userrolehash, $processmarker, $dumpcount, %coursedombuf,
    %coursenumbuf, %coursehombuf, %coursedescrbuf, %courseinstcodebuf,
    %courseownerbuf, %coursetypebuf,$locknum);

use IO::Socket;
use GDBM_File;
use HTML::LCParser;
use Fcntl qw(:flock);
use Storable qw(thaw nfreeze);
use Time::HiRes qw( gettimeofday tv_interval );
use Cache::Memcached;
use Digest::MD5;
use Math::Random;
use File::MMagic;
use LONCAPA qw(:DEFAULT :match);
use LONCAPA::Configuration;

my $readit;
my $max_connection_retries = 10;     # Or some such value.

require Exporter;

our @ISA = qw (Exporter);
our @EXPORT = qw(%env);


# --------------------------------------------------------------------- Logging
{
    my $logid;
    sub instructor_log {
	my ($hash_name,$storehash,$delflag,$uname,$udom,$cnum,$cdom)=@_;
        if (($cnum eq '') || ($cdom eq '')) {
            $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
            $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
        }
	$logid++;
        my $now = time();
	my $id=$now.'00000'.$$.'00000'.$logid;
	return &Apache::lonnet::put('nohist_'.$hash_name,
				    { $id => {
					'exe_uname' => $env{'user.name'},
					'exe_udom'  => $env{'user.domain'},
					'exe_time'  => $now,
					'exe_ip'    => $ENV{'REMOTE_ADDR'},
					'delflag'   => $delflag,
					'logentry'  => $storehash,
					'uname'     => $uname,
					'udom'      => $udom,
				    }
				  },$cdom,$cnum);
    }
}

sub logtouch {
    my $execdir=$perlvar{'lonDaemons'};
    unless (-e "$execdir/logs/lonnet.log") {	
	open(my $fh,">>$execdir/logs/lonnet.log");
	close $fh;
    }
    my ($wwwuid,$wwwgid)=(getpwnam('www'))[2,3];
    chown($wwwuid,$wwwgid,$execdir.'/logs/lonnet.log');
}

sub logthis {
    my $message=shift;
    my $execdir=$perlvar{'lonDaemons'};
    my $now=time;
    my $local=localtime($now);
    if (open(my $fh,">>$execdir/logs/lonnet.log")) {
	my $logstring = $local. " ($$): ".$message."\n"; # Keep any \'s in string.
	print $fh $logstring;
	close($fh);
    }
    return 1;
}

sub logperm {
    my $message=shift;
    my $execdir=$perlvar{'lonDaemons'};
    my $now=time;
    my $local=localtime($now);
    if (open(my $fh,">>$execdir/logs/lonnet.perm.log")) {
	print $fh "$now:$message:$local\n";
	close($fh);
    }
    return 1;
}

sub create_connection {
    my ($hostname,$lonid) = @_;
    my $client=IO::Socket::UNIX->new(Peer    => $perlvar{'lonSockCreate'},
				     Type    => SOCK_STREAM,
				     Timeout => 10);
    return 0 if (!$client);
    print $client (join(':',$hostname,$lonid,&machine_ids($hostname))."\n");
    my $result = <$client>;
    chomp($result);
    return 1 if ($result eq 'done');
    return 0;
}

sub get_server_timezone {
    my ($cnum,$cdom) = @_;
    my $home=&homeserver($cnum,$cdom);
    if ($home ne 'no_host') {
        my $cachetime = 24*3600;
        my ($timezone,$cached)=&is_cached_new('servertimezone',$home);
        if (defined($cached)) {
            return $timezone;
        } else {
            my $timezone = &reply('servertimezone',$home);
            return &do_cache_new('servertimezone',$home,$timezone,$cachetime);
        }
    }
}

sub get_server_loncaparev {
    my ($dom,$lonhost) = @_;
    if (defined($lonhost)) {
        if (!defined(&hostname($lonhost))) {
            undef($lonhost);
        }
    }
    if (!defined($lonhost)) {
        if (defined(&domain($dom,'primary'))) {
            $lonhost=&domain($dom,'primary');
            if ($lonhost eq 'no_host') {
                undef($lonhost);
            }
        }
    }
    if (defined($lonhost)) {
        my $cachetime = 24*3600;
        my ($loncaparev,$cached)=&is_cached_new('serverloncaparev',$lonhost);
        if (defined($cached)) {
            return $loncaparev;
        } else {
            my $loncaparev = &reply('serverloncaparev',$lonhost);
            return &do_cache_new('serverloncaparev',$lonhost,$loncaparev,$cachetime);
        }
    }
}

# -------------------------------------------------- Non-critical communication
sub subreply {
    my ($cmd,$server)=@_;
    my $peerfile="$perlvar{'lonSockDir'}/".&hostname($server);
    #
    #  With loncnew process trimming, there's a timing hole between lonc server
    #  process exit and the master server picking up the listen on the AF_UNIX
    #  socket.  In that time interval, a lock file will exist:

    my $lockfile=$peerfile.".lock";
    while (-e $lockfile) {	# Need to wait for the lockfile to disappear.
	sleep(1);
    }
    # At this point, either a loncnew parent is listening or an old lonc
    # or loncnew child is listening so we can connect or everything's dead.
    #
    #   We'll give the connection a few tries before abandoning it.  If
    #   connection is not possible, we'll con_lost back to the client.
    #   
    my $client;
    for (my $retries = 0; $retries < $max_connection_retries; $retries++) {
	$client=IO::Socket::UNIX->new(Peer    =>"$peerfile",
				      Type    => SOCK_STREAM,
				      Timeout => 10);
	if ($client) {
	    last;		# Connected!
	} else {
	    &create_connection(&hostname($server),$server);
	}
        sleep(1);		# Try again later if failed connection.
    }
    my $answer;
    if ($client) {
	print $client "sethost:$server:$cmd\n";
	$answer=<$client>;
	if (!$answer) { $answer="con_lost"; }
	chomp($answer);
    } else {
	$answer = 'con_lost';	# Failed connection.
    }
    return $answer;
}

sub reply {
    my ($cmd,$server)=@_;
    unless (defined(&hostname($server))) { return 'no_such_host'; }
    my $answer=subreply($cmd,$server);
    if (($answer=~/^refused/) || ($answer=~/^rejected/)) {
       &logthis("<font color=\"blue\">WARNING:".
                " $cmd to $server returned $answer</font>");
    }
    return $answer;
}

# ----------------------------------------------------------- Send USR1 to lonc

sub reconlonc {
    my ($lonid) = @_;
    my $hostname = &hostname($lonid);
    if ($lonid) {
	my $peerfile="$perlvar{'lonSockDir'}/$hostname";
	if ($hostname && -e $peerfile) {
	    &logthis("Trying to reconnect lonc for $lonid ($hostname)");
	    my $client=IO::Socket::UNIX->new(Peer    => $peerfile,
					     Type    => SOCK_STREAM,
					     Timeout => 10);
	    if ($client) {
		print $client ("reset_retries\n");
		my $answer=<$client>;
		#reset just this one.
	    }
	}
	return;
    }

    &logthis("Trying to reconnect lonc");
    my $loncfile="$perlvar{'lonDaemons'}/logs/lonc.pid";
    if (open(my $fh,"<$loncfile")) {
	my $loncpid=<$fh>;
        chomp($loncpid);
        if (kill 0 => $loncpid) {
	    &logthis("lonc at pid $loncpid responding, sending USR1");
            kill USR1 => $loncpid;
            sleep 1;
         } else {
	    &logthis(
               "<font color=\"blue\">WARNING:".
               " lonc at pid $loncpid not responding, giving up</font>");
        }
    } else {
	&logthis('<font color="blue">WARNING: lonc not running, giving up</font>');
    }
}

# ------------------------------------------------------ Critical communication

sub critical {
    my ($cmd,$server)=@_;
    unless (&hostname($server)) {
        &logthis("<font color=\"blue\">WARNING:".
               " Critical message to unknown server ($server)</font>");
        return 'no_such_host';
    }
    my $answer=reply($cmd,$server);
    if ($answer eq 'con_lost') {
	&reconlonc("$perlvar{'lonSockDir'}/$server");
	my $answer=reply($cmd,$server);
        if ($answer eq 'con_lost') {
            my $now=time;
            my $middlename=$cmd;
            $middlename=substr($middlename,0,16);
            $middlename=~s/\W//g;
            my $dfilename=
      "$perlvar{'lonSockDir'}/delayed/$now.$dumpcount.$$.$middlename.$server";
            $dumpcount++;
            {
		my $dfh;
		if (open($dfh,">$dfilename")) {
		    print $dfh "$cmd\n"; 
		    close($dfh);
		}
            }
            sleep 2;
            my $wcmd='';
            {
		my $dfh;
		if (open($dfh,"<$dfilename")) {
		    $wcmd=<$dfh>; 
		    close($dfh);
		}
            }
            chomp($wcmd);
            if ($wcmd eq $cmd) {
		&logthis("<font color=\"blue\">WARNING: ".
                         "Connection buffer $dfilename: $cmd</font>");
                &logperm("D:$server:$cmd");
	        return 'con_delayed';
            } else {
                &logthis("<font color=\"red\">CRITICAL:"
                        ." Critical connection failed: $server $cmd</font>");
                &logperm("F:$server:$cmd");
                return 'con_failed';
            }
        }
    }
    return $answer;
}

# ------------------------------------------- check if return value is an error

sub error {
    my ($result) = @_;
    if ($result =~ /^(con_lost|no_such_host|error: (\d+) (.*))/) {
	if ($2 == 2) { return undef; }
	return $1;
    }
    return undef;
}

sub convert_and_load_session_env {
    my ($lonidsdir,$handle)=@_;
    my @profile;
    {
	my $opened = open(my $idf,'+<',"$lonidsdir/$handle.id");
	if (!$opened) {
	    return 0;
	}
	flock($idf,LOCK_SH);
	@profile=<$idf>;
	close($idf);
    }
    my %temp_env;
    foreach my $line (@profile) {
	if ($line !~ m/=/) {
	    return 0;
	}
	chomp($line);
	my ($envname,$envvalue)=split(/=/,$line,2);
	$temp_env{&unescape($envname)} = &unescape($envvalue);
    }
    unlink("$lonidsdir/$handle.id");
    if (tie(my %disk_env,'GDBM_File',"$lonidsdir/$handle.id",&GDBM_WRCREAT(),
	    0640)) {
	%disk_env = %temp_env;
	@env{keys(%temp_env)} = @disk_env{keys(%temp_env)};
	untie(%disk_env);
    }
    return 1;
}

# ------------------------------------------- Transfer profile into environment
my $env_loaded;
sub transfer_profile_to_env {
    my ($lonidsdir,$handle,$force_transfer) = @_;
    if (!$force_transfer && $env_loaded) { return; } 

    if (!defined($lonidsdir)) {
	$lonidsdir = $perlvar{'lonIDsDir'};
    }
    if (!defined($handle)) {
        ($handle) = ($env{'user.environment'} =~m|/([^/]+)\.id$| );
    }

    my $convert;
    {
    	my $opened = open(my $idf,'+<',"$lonidsdir/$handle.id");
	if (!$opened) {
	    return;
	}
	flock($idf,LOCK_SH);
	if (tie(my %disk_env,'GDBM_File',"$lonidsdir/$handle.id",
		&GDBM_READER(),0640)) {
	    @env{keys(%disk_env)} = @disk_env{keys(%disk_env)};
	    untie(%disk_env);
	} else {
	    $convert = 1;
	}
    }
    if ($convert) {
	if (!&convert_and_load_session_env($lonidsdir,$handle)) {
	    &logthis("Failed to load session, or convert session.");
	}
    }

    my %remove;
    while ( my $envname = each(%env) ) {
        if (my ($key,$time) = ($envname =~ /^(cgi\.(\d+)_\d+\.)/)) {
            if ($time < time-300) {
                $remove{$key}++;
            }
        }
    }

    $env{'user.environment'} = "$lonidsdir/$handle.id";
    $env_loaded=1;
    foreach my $expired_key (keys(%remove)) {
        &delenv($expired_key);
    }
}

# ---------------------------------------------------- Check for valid session 
sub check_for_valid_session {
    my ($r) = @_;
    my %cookies=CGI::Cookie->parse($r->header_in('Cookie'));
    my $lonid=$cookies{'lonID'};
    return undef if (!$lonid);

    my $handle=&LONCAPA::clean_handle($lonid->value);
    my $lonidsdir=$r->dir_config('lonIDsDir');
    return undef if (!-e "$lonidsdir/$handle.id");

    my $opened = open(my $idf,'+<',"$lonidsdir/$handle.id");
    return undef if (!$opened);

    flock($idf,LOCK_SH);
    my %disk_env;
    if (!tie(%disk_env,'GDBM_File',"$lonidsdir/$handle.id",
	    &GDBM_READER(),0640)) {
	return undef;	
    }

    if (!defined($disk_env{'user.name'})
	|| !defined($disk_env{'user.domain'})) {
	return undef;
    }
    return $handle;
}

sub timed_flock {
    my ($file,$lock_type) = @_;
    my $failed=0;
    eval {
	local $SIG{__DIE__}='DEFAULT';
	local $SIG{ALRM}=sub {
	    $failed=1;
	    die("failed lock");
	};
	alarm(13);
	flock($file,$lock_type);
	alarm(0);
    };
    if ($failed) {
	return undef;
    } else {
	return 1;
    }
}

# ---------------------------------------------------------- Append Environment

sub appenv {
    my ($newenv,$roles) = @_;
    if (ref($newenv) eq 'HASH') {
        foreach my $key (keys(%{$newenv})) {
            my $refused = 0;
	    if (($key =~ /^user\.role/) || ($key =~ /^user\.priv/)) {
                $refused = 1;
                if (ref($roles) eq 'ARRAY') {
                    my ($type,$role) = ($key =~ /^user\.(role|priv)\.([^.]+)\./);
                    if (grep(/^\Q$role\E$/,@{$roles})) {
                        $refused = 0;
                    }
                }
            }
            if ($refused) {
                &logthis("<font color=\"blue\">WARNING: ".
                         "Attempt to modify environment ".$key." to ".$newenv->{$key}
                         .'</font>');
	        delete($newenv->{$key});
            } else {
                $env{$key}=$newenv->{$key};
            }
        }
        my $opened = open(my $env_file,'+<',$env{'user.environment'});
        if ($opened
	    && &timed_flock($env_file,LOCK_EX)
	    &&
	    tie(my %disk_env,'GDBM_File',$env{'user.environment'},
	        (&GDBM_WRITER()|&GDBM_NOLOCK()),0640)) {
	    while (my ($key,$value) = each(%{$newenv})) {
	        $disk_env{$key} = $value;
	    }
	    untie(%disk_env);
        }
    }
    return 'ok';
}
# ----------------------------------------------------- Delete from Environment

sub delenv {
    my ($delthis,$regexp) = @_;
    if (($delthis=~/user\.role/) || ($delthis=~/user\.priv/)) {
        &logthis("<font color=\"blue\">WARNING: ".
                "Attempt to delete from environment ".$delthis);
        return 'error';
    }
    my $opened = open(my $env_file,'+<',$env{'user.environment'});
    if ($opened
	&& &timed_flock($env_file,LOCK_EX)
	&&
	tie(my %disk_env,'GDBM_File',$env{'user.environment'},
	    (&GDBM_WRITER()|&GDBM_NOLOCK()),0640)) {
	foreach my $key (keys(%disk_env)) {
	    if ($regexp) {
                if ($key=~/^$delthis/) {
                    delete($env{$key});
                    delete($disk_env{$key});
                } 
            } else {
                if ($key=~/^\Q$delthis\E/) {
		    delete($env{$key});
		    delete($disk_env{$key});
	        }
            }
	}
	untie(%disk_env);
    }
    return 'ok';
}

sub get_env_multiple {
    my ($name) = @_;
    my @values;
    if (defined($env{$name})) {
        # exists is it an array
        if (ref($env{$name})) {
            @values=@{ $env{$name} };
        } else {
            $values[0]=$env{$name};
        }
    }
    return(@values);
}

# ------------------------------------------------------------------- Locking

sub set_lock {
    my ($text)=@_;
    $locknum++;
    my $id=$$.'-'.$locknum;
    &appenv({'session.locks' => $env{'session.locks'}.','.$id,
             'session.lock.'.$id => $text});
    return $id;
}

sub get_locks {
    my $num=0;
    my %texts=();
    foreach my $lock (split(/\,/,$env{'session.locks'})) {
       if ($lock=~/\w/) {
          $num++;
          $texts{$lock}=$env{'session.lock.'.$lock};
       }
   }
   return ($num,%texts);
}

sub remove_lock {
    my ($id)=@_;
    my $newlocks='';
    foreach my $lock (split(/\,/,$env{'session.locks'})) {
       if (($lock=~/\w/) && ($lock ne $id)) {
          $newlocks.=','.$lock;
       }
    }
    &appenv({'session.locks' => $newlocks});
    &delenv('session.lock.'.$id);
}

sub remove_all_locks {
    my $activelocks=$env{'session.locks'};
    foreach my $lock (split(/\,/,$env{'session.locks'})) {
       if ($lock=~/\w/) {
          &remove_lock($lock);
       }
    }
}


# ------------------------------------------ Find out current server userload
sub userload {
    my $numusers=0;
    {
	opendir(LONIDS,$perlvar{'lonIDsDir'});
	my $filename;
	my $curtime=time;
	while ($filename=readdir(LONIDS)) {
	    next if ($filename eq '.' || $filename eq '..');
	    next if ($filename =~ /publicuser_\d+\.id/);
	    my ($mtime)=(stat($perlvar{'lonIDsDir'}.'/'.$filename))[9];
	    if ($curtime-$mtime < 1800) { $numusers++; }
	}
	closedir(LONIDS);
    }
    my $userloadpercent=0;
    my $maxuserload=$perlvar{'lonUserLoadLim'};
    if ($maxuserload) {
	$userloadpercent=100*$numusers/$maxuserload;
    }
    $userloadpercent=sprintf("%.2f",$userloadpercent);
    return $userloadpercent;
}

# ------------------------------------------ Fight off request when overloaded

sub overloaderror {
    my ($r,$checkserver)=@_;
    unless ($checkserver) { $checkserver=$perlvar{'lonHostID'}; }
    my $loadavg;
    if ($checkserver eq $perlvar{'lonHostID'}) {
       open(my $loadfile,'/proc/loadavg');
       $loadavg=<$loadfile>;
       $loadavg =~ s/\s.*//g;
       $loadavg = 100*$loadavg/$perlvar{'lonLoadLim'};
       close($loadfile);
    } else {
       $loadavg=&reply('load',$checkserver);
    }
    my $overload=$loadavg-100;
    if ($overload>0) {
	$r->err_headers_out->{'Retry-After'}=$overload;
        $r->log_error('Overload of '.$overload.' on '.$checkserver);
        return 413;
    }    
    return '';
}

# ------------------------------ Find server with least workload from spare.tab

sub spareserver {
    my ($loadpercent,$userloadpercent,$want_server_name) = @_;
    my $spare_server;
    if ($userloadpercent !~ /\d/) { $userloadpercent=0; }
    my $lowest_load=($loadpercent > $userloadpercent) ? $loadpercent 
                                                     :  $userloadpercent;
    
    foreach my $try_server (@{ $spareid{'primary'} }) {
	($spare_server, $lowest_load) =
	    &compare_server_load($try_server, $spare_server, $lowest_load);
    }

    my $found_server = ($spare_server ne '' && $lowest_load < 100);

    if (!$found_server) {
	foreach my $try_server (@{ $spareid{'default'} }) {
	    ($spare_server, $lowest_load) =
		&compare_server_load($try_server, $spare_server, $lowest_load);
	}
    }

    if (!$want_server_name) {
        my $protocol = 'http';
        if ($protocol{$spare_server} eq 'https') {
            $protocol = $protocol{$spare_server};
        }
        if (defined($spare_server)) {
            my $hostname = &hostname($spare_server);
            if (defined($hostname)) {  
	        $spare_server = $protocol.'://'.$hostname;
            }
        }
    }
    return $spare_server;
}

sub compare_server_load {
    my ($try_server, $spare_server, $lowest_load) = @_;

    my $loadans     = &reply('load',    $try_server);
    my $userloadans = &reply('userload',$try_server);

    if ($loadans !~ /\d/ && $userloadans !~ /\d/) {
	next; #didn't get a number from the server
    }

    my $load;
    if ($loadans =~ /\d/) {
	if ($userloadans =~ /\d/) {
	    #both are numbers, pick the bigger one
	    $load = ($loadans > $userloadans) ? $loadans 
		                              : $userloadans;
	} else {
	    $load = $loadans;
	}
    } else {
	$load = $userloadans;
    }

    if (($load =~ /\d/) && ($load < $lowest_load)) {
	$spare_server = $try_server;
	$lowest_load  = $load;
    }
    return ($spare_server,$lowest_load);
}

# --------------------------- ask offload servers if user already has a session
sub find_existing_session {
    my ($udom,$uname) = @_;
    foreach my $try_server (@{ $spareid{'primary'} },
			    @{ $spareid{'default'} }) {
	return $try_server if (&has_user_session($try_server, $udom, $uname));
    }
    return;
}

# -------------------------------- ask if server already has a session for user
sub has_user_session {
    my ($lonid,$udom,$uname) = @_;
    my $result = &reply(join(':','userhassession',
			     map {&escape($_)} ($udom,$uname)),$lonid);
    return 1 if ($result eq 'ok');

    return 0;
}

# --------------------------------------------- Try to change a user's password

sub changepass {
    my ($uname,$udom,$currentpass,$newpass,$server,$context)=@_;
    $currentpass = &escape($currentpass);
    $newpass     = &escape($newpass);
    my $lonhost = $perlvar{'lonHostID'};
    my $answer = reply("encrypt:passwd:$udom:$uname:$currentpass:$newpass:$context:$lonhost",
		       $server);
    if (! $answer) {
	&logthis("No reply on password change request to $server ".
		 "by $uname in domain $udom.");
    } elsif ($answer =~ "^ok") {
        &logthis("$uname in $udom successfully changed their password ".
		 "on $server.");
    } elsif ($answer =~ "^pwchange_failure") {
	&logthis("$uname in $udom was unable to change their password ".
		 "on $server.  The action was blocked by either lcpasswd ".
		 "or pwchange");
    } elsif ($answer =~ "^non_authorized") {
        &logthis("$uname in $udom did not get their password correct when ".
		 "attempting to change it on $server.");
    } elsif ($answer =~ "^auth_mode_error") {
        &logthis("$uname in $udom attempted to change their password despite ".
		 "not being locally or internally authenticated on $server.");
    } elsif ($answer =~ "^unknown_user") {
        &logthis("$uname in $udom attempted to change their password ".
		 "on $server but were unable to because $server is not ".
		 "their home server.");
    } elsif ($answer =~ "^refused") {
	&logthis("$server refused to change $uname in $udom password because ".
		 "it was sent an unencrypted request to change the password.");
    } elsif ($answer =~ "invalid_client") {
        &logthis("$server refused to change $uname in $udom password because ".
                 "it was a reset by e-mail originating from an invalid server.");
    }
    return $answer;
}

# ----------------------- Try to determine user's current authentication scheme

sub queryauthenticate {
    my ($uname,$udom)=@_;
    my $uhome=&homeserver($uname,$udom);
    if (!$uhome) {
	&logthis("User $uname at $udom is unknown when looking for authentication mechanism");
	return 'no_host';
    }
    my $answer=reply("encrypt:currentauth:$udom:$uname",$uhome);
    if ($answer =~ /^(unknown_user|refused|con_lost)/) {
	&logthis("User $uname at $udom threw error $answer when checking authentication mechanism");
    }
    return $answer;
}

# --------- Try to authenticate user from domain's lib servers (first this one)

sub authenticate {
    my ($uname,$upass,$udom,$checkdefauth)=@_;
    $upass=&escape($upass);
    $uname= &LONCAPA::clean_username($uname);
    my $uhome=&homeserver($uname,$udom,1);
    my $newhome;
    if ((!$uhome) || ($uhome eq 'no_host')) {
# Maybe the machine was offline and only re-appeared again recently?
        &reconlonc();
# One more
	$uhome=&homeserver($uname,$udom,1);
        if (($uhome eq 'no_host') && $checkdefauth) {
            if (defined(&domain($udom,'primary'))) {
                $newhome=&domain($udom,'primary');
            }
            if ($newhome ne '') {
                $uhome = $newhome;
            }
        }
	if ((!$uhome) || ($uhome eq 'no_host')) {
	    &logthis("User $uname at $udom is unknown in authenticate");
	    return 'no_host';
        }
    }
    my $answer=reply("encrypt:auth:$udom:$uname:$upass:$checkdefauth",$uhome);
    if ($answer eq 'authorized') {
        if ($newhome) {
            &logthis("User $uname at $udom authorized by $uhome, but needs account");
            return 'no_account_on_host'; 
        } else {
            &logthis("User $uname at $udom authorized by $uhome");
            return $uhome;
        }
    }
    if ($answer eq 'non_authorized') {
	&logthis("User $uname at $udom rejected by $uhome");
	return 'no_host'; 
    }
    &logthis("User $uname at $udom threw error $answer when checking authentication mechanism");
    return 'no_host';
}

# ---------------------- Find the homebase for a user from domain's lib servers

my %homecache;
sub homeserver {
    my ($uname,$udom,$ignoreBadCache)=@_;
    my $index="$uname:$udom";

    if (exists($homecache{$index})) { return $homecache{$index}; }

    my %servers = &get_servers($udom,'library');
    foreach my $tryserver (keys(%servers)) {
        next if ($ignoreBadCache ne 'true' && 
		 exists($badServerCache{$tryserver}));

	my $answer=reply("home:$udom:$uname",$tryserver);
	if ($answer eq 'found') {
	    delete($badServerCache{$tryserver}); 
	    return $homecache{$index}=$tryserver;
	} elsif ($answer eq 'no_host') {
	    $badServerCache{$tryserver}=1;
	}
    }    
    return 'no_host';
}

# ------------------------------------- Find the usernames behind a list of IDs

sub idget {
    my ($udom,@ids)=@_;
    my %returnhash=();
    
    my %servers = &get_servers($udom,'library');
    foreach my $tryserver (keys(%servers)) {
	my $idlist=join('&',@ids);
	$idlist=~tr/A-Z/a-z/; 
	my $reply=&reply("idget:$udom:".$idlist,$tryserver);
	my @answer=();
	if (($reply ne 'con_lost') && ($reply!~/^error\:/)) {
	    @answer=split(/\&/,$reply);
	}                    ;
	my $i;
	for ($i=0;$i<=$#ids;$i++) {
	    if ($answer[$i]) {
		$returnhash{$ids[$i]}=$answer[$i];
	    } 
	}
    } 
    return %returnhash;
}

# ------------------------------------- Find the IDs behind a list of usernames

sub idrget {
    my ($udom,@unames)=@_;
    my %returnhash=();
    foreach my $uname (@unames) {
        $returnhash{$uname}=(&userenvironment($udom,$uname,'id'))[1];
    }
    return %returnhash;
}

# ------------------------------- Store away a list of names and associated IDs

sub idput {
    my ($udom,%ids)=@_;
    my %servers=();
    foreach my $uname (keys(%ids)) {
	&cput('environment',{'id'=>$ids{$uname}},$udom,$uname);
        my $uhom=&homeserver($uname,$udom);
        if ($uhom ne 'no_host') {
            my $id=&escape($ids{$uname});
            $id=~tr/A-Z/a-z/;
            my $esc_unam=&escape($uname);
	    if ($servers{$uhom}) {
		$servers{$uhom}.='&'.$id.'='.$esc_unam;
            } else {
                $servers{$uhom}=$id.'='.$esc_unam;
            }
        }
    }
    foreach my $server (keys(%servers)) {
        &critical('idput:'.$udom.':'.$servers{$server},$server);
    }
}

# ------------------------------dump from db file owned by domainconfig user
sub dump_dom {
    my ($namespace,$udom,$regexp,$range)=@_;
    if (!$udom) {
        $udom=$env{'user.domain'};
    }
    my %returnhash;
    if ($udom) {
        my $uname = &get_domainconfiguser($udom);
        %returnhash = &dump($namespace,$udom,$uname,$regexp,$range);
    }
    return %returnhash;
}

# ------------------------------------------ get items from domain db files   

sub get_dom {
    my ($namespace,$storearr,$udom,$uhome)=@_;
    my $items='';
    foreach my $item (@$storearr) {
        $items.=&escape($item).'&';
    }
    $items=~s/\&$//;
    if (!$udom) {
        $udom=$env{'user.domain'};
        if (defined(&domain($udom,'primary'))) {
            $uhome=&domain($udom,'primary');
        } else {
            undef($uhome);
        }
    } else {
        if (!$uhome) {
            if (defined(&domain($udom,'primary'))) {
                $uhome=&domain($udom,'primary');
            }
        }
    }
    if ($udom && $uhome && ($uhome ne 'no_host')) {
        my $rep=&reply("getdom:$udom:$namespace:$items",$uhome);
        my %returnhash;
        if ($rep eq '' || $rep =~ /^error: 2 /) {
            return %returnhash;
        }
        my @pairs=split(/\&/,$rep);
        if ( $#pairs==0 && $pairs[0] =~ /^(con_lost|error|no_such_host)/i) {
            return @pairs;
        }
        my $i=0;
        foreach my $item (@$storearr) {
            $returnhash{$item}=&thaw_unescape($pairs[$i]);
            $i++;
        }
        return %returnhash;
    } else {
        &logthis("get_dom failed - no homeserver and/or domain ($udom) ($uhome)");
    }
}

# -------------------------------------------- put items in domain db files 

sub put_dom {
    my ($namespace,$storehash,$udom,$uhome)=@_;
    if (!$udom) {
        $udom=$env{'user.domain'};
        if (defined(&domain($udom,'primary'))) {
            $uhome=&domain($udom,'primary');
        } else {
            undef($uhome);
        }
    } else {
        if (!$uhome) {
            if (defined(&domain($udom,'primary'))) {
                $uhome=&domain($udom,'primary');
            }
        }
    } 
    if ($udom && $uhome && ($uhome ne 'no_host')) {
        my $items='';
        foreach my $item (keys(%$storehash)) {
            $items.=&escape($item).'='.&freeze_escape($$storehash{$item}).'&';
        }
        $items=~s/\&$//;
        return &reply("putdom:$udom:$namespace:$items",$uhome);
    } else {
        &logthis("put_dom failed - no homeserver and/or domain");
    }
}

# --------------------- newput for items in db file owned by domainconfig user
sub newput_dom {
    my ($namespace,$storehash,$udom) = @_;
    my $result;
    if (!$udom) {
        $udom=$env{'user.domain'};
    }
    if ($udom) {
        my $uname = &get_domainconfiguser($udom);
        $result = &newput($namespace,$storehash,$udom,$uname);
    }
    return $result;
}

# --------------------- delete for items in db file owned by domainconfig user
sub del_dom {
    my ($namespace,$storearr,$udom)=@_;
    if (ref($storearr) eq 'ARRAY') {
        if (!$udom) {
            $udom=$env{'user.domain'};
        }
        if ($udom) {
            my $uname = &get_domainconfiguser($udom); 
            return &del($namespace,$storearr,$udom,$uname);
        }
    }
}

# ----------------------------------construct domainconfig user for a domain 
sub get_domainconfiguser {
    my ($udom) = @_;
    return $udom.'-domainconfig';
}

sub retrieve_inst_usertypes {
    my ($udom) = @_;
    my (%returnhash,@order);
    my %domdefs = &Apache::lonnet::get_domain_defaults($udom);
    if ((ref($domdefs{'inststatustypes'}) eq 'HASH') && 
        (ref($domdefs{'inststatusorder'}) eq 'ARRAY')) {
        %returnhash = %{$domdefs{'inststatustypes'}};
        @order = @{$domdefs{'inststatusorder'}};
    } else {
        if (defined(&domain($udom,'primary'))) {
            my $uhome=&domain($udom,'primary');
            my $rep=&reply("inst_usertypes:$udom",$uhome);
            if ($rep =~ /^(con_lost|error|no_such_host|refused)/) {
                &logthis("get_dom failed - $rep returned from $uhome in domain: $udom");
                return (\%returnhash,\@order);
            }
            my ($hashitems,$orderitems) = split(/:/,$rep); 
            my @pairs=split(/\&/,$hashitems);
            foreach my $item (@pairs) {
                my ($key,$value)=split(/=/,$item,2);
                $key = &unescape($key);
                next if ($key =~ /^error: 2 /);
                $returnhash{$key}=&thaw_unescape($value);
            }
            my @esc_order = split(/\&/,$orderitems);
            foreach my $item (@esc_order) {
                push(@order,&unescape($item));
            }
        } else {
            &logthis("get_dom failed - no primary domain server for $udom");
        }
    }
    return (\%returnhash,\@order);
}

sub is_domainimage {
    my ($url) = @_;
    if ($url=~m-^/+res/+($match_domain)/+\1\-domainconfig/+(img|logo|domlogo)/+-) {
        if (&domain($1) ne '') {
            return '1';
        }
    }
    return;
}

sub inst_directory_query {
    my ($srch) = @_;
    my $udom = $srch->{'srchdomain'};
    my %results;
    my $homeserver = &domain($udom,'primary');
    my $outcome;
    if ($homeserver ne '') {
	my $queryid=&reply("querysend:instdirsearch:".
			   &escape($srch->{'srchby'}).':'.
			   &escape($srch->{'srchterm'}).':'.
			   &escape($srch->{'srchtype'}),$homeserver);
	my $host=&hostname($homeserver);
	if ($queryid !~/^\Q$host\E\_/) {
	    &logthis('instituional directory search invalid queryid: '.$queryid.' for host: '.$homeserver.'in domain '.$udom);
	    return;
	}
	my $response = &get_query_reply($queryid);
	my $maxtries = 5;
	my $tries = 1;
	while (($response=~/^timeout/) && ($tries < $maxtries)) {
	    $response = &get_query_reply($queryid);
	    $tries ++;
	}

        if (!&error($response) && $response ne 'refused') {
            if ($response eq 'unavailable') {
                $outcome = $response;
            } else {
                $outcome = 'ok';
                my @matches = split(/\n/,$response);
                foreach my $match (@matches) {
                    my ($key,$value) = split(/=/,$match);
                    $results{&unescape($key).':'.$udom} = &thaw_unescape($value);
                }
            }
        }
    }
    return ($outcome,%results);
}

sub usersearch {
    my ($srch) = @_;
    my $dom = $srch->{'srchdomain'};
    my %results;
    my %libserv = &all_library();
    my $query = 'usersearch';
    foreach my $tryserver (keys(%libserv)) {
        if (&host_domain($tryserver) eq $dom) {
            my $host=&hostname($tryserver);
            my $queryid=
                &reply("querysend:".&escape($query).':'.
                       &escape($srch->{'srchby'}).':'.
                       &escape($srch->{'srchtype'}).':'.
                       &escape($srch->{'srchterm'}),$tryserver);
            if ($queryid !~/^\Q$host\E\_/) {
                &logthis('usersearch: invalid queryid: '.$queryid.' for host: '.$host.'in domain '.$dom.' and server: '.$tryserver);
                next;
            }
            my $reply = &get_query_reply($queryid);
            my $maxtries = 1;
            my $tries = 1;
            while (($reply=~/^timeout/) && ($tries < $maxtries)) {
                $reply = &get_query_reply($queryid);
                $tries ++;
            }
            if ( ($reply =~/^timeout/) || ($reply =~/^error/) ) {
                &logthis('usersrch error: '.$reply.' for '.$dom.' - searching for : '.$srch->{'srchterm'}.' by '.$srch->{'srchby'}.' ('.$srch->{'srchtype'}.') -  maxtries: '.$maxtries.' tries: '.$tries);
            } else {
                my @matches;
                if ($reply =~ /\n/) {
                    @matches = split(/\n/,$reply);
                } else {
                    @matches = split(/\&/,$reply);
                }
                foreach my $match (@matches) {
                    my ($uname,$udom,%userhash);
                    foreach my $entry (split(/:/,$match)) {
                        my ($key,$value) =
                            map {&unescape($_);} split(/=/,$entry);
                        $userhash{$key} = $value;
                        if ($key eq 'username') {
                            $uname = $value;
                        } elsif ($key eq 'domain') {
                            $udom = $value;
                        }
                    }
                    $results{$uname.':'.$udom} = \%userhash;
                }
            }
        }
    }
    return %results;
}

sub get_instuser {
    my ($udom,$uname,$id) = @_;
    my $homeserver = &domain($udom,'primary');
    my ($outcome,%results);
    if ($homeserver ne '') {
        my $queryid=&reply("querysend:getinstuser:".&escape($uname).':'.
                           &escape($id).':'.&escape($udom),$homeserver);
        my $host=&hostname($homeserver);
        if ($queryid !~/^\Q$host\E\_/) {
            &logthis('get_instuser invalid queryid: '.$queryid.' for host: '.$homeserver.'in domain '.$udom);
            return;
        }
        my $response = &get_query_reply($queryid);
        my $maxtries = 5;
        my $tries = 1;
        while (($response=~/^timeout/) && ($tries < $maxtries)) {
            $response = &get_query_reply($queryid);
            $tries ++;
        }
        if (!&error($response) && $response ne 'refused') {
            if ($response eq 'unavailable') {
                $outcome = $response;
            } else {
                $outcome = 'ok';
                my @matches = split(/\n/,$response);
                foreach my $match (@matches) {
                    my ($key,$value) = split(/=/,$match);
                    $results{&unescape($key)} = &thaw_unescape($value);
                }
            }
        }
    }
    my %userinfo;
    if (ref($results{$uname}) eq 'HASH') {
        %userinfo = %{$results{$uname}};
    } 
    return ($outcome,%userinfo);
}

sub inst_rulecheck {
    my ($udom,$uname,$id,$item,$rules) = @_;
    my %returnhash;
    if ($udom ne '') {
        if (ref($rules) eq 'ARRAY') {
            @{$rules} = map {&escape($_);} (@{$rules});
            my $rulestr = join(':',@{$rules});
            my $homeserver=&domain($udom,'primary');
            if (($homeserver ne '') && ($homeserver ne 'no_host')) {
                my $response;
                if ($item eq 'username') {                
                    $response=&unescape(&reply('instrulecheck:'.&escape($udom).
                                              ':'.&escape($uname).':'.$rulestr,
                                              $homeserver));
                } elsif ($item eq 'id') {
                    $response=&unescape(&reply('instidrulecheck:'.&escape($udom).
                                              ':'.&escape($id).':'.$rulestr,
                                              $homeserver));
                } elsif ($item eq 'selfcreate') {
                    $response=&unescape(&reply('instselfcreatecheck:'.
                                               &escape($udom).':'.&escape($uname).
                                              ':'.$rulestr,$homeserver));
                }
                if ($response ne 'refused') {
                    my @pairs=split(/\&/,$response);
                    foreach my $item (@pairs) {
                        my ($key,$value)=split(/=/,$item,2);
                        $key = &unescape($key);
                        next if ($key =~ /^error: 2 /);
                        $returnhash{$key}=&thaw_unescape($value);
                    }
                }
            }
        }
    }
    return %returnhash;
}

sub inst_userrules {
    my ($udom,$check) = @_;
    my (%ruleshash,@ruleorder);
    if ($udom ne '') {
        my $homeserver=&domain($udom,'primary');
        if (($homeserver ne '') && ($homeserver ne 'no_host')) {
            my $response;
            if ($check eq 'id') {
                $response=&reply('instidrules:'.&escape($udom),
                                 $homeserver);
            } elsif ($check eq 'email') {
                $response=&reply('instemailrules:'.&escape($udom),
                                 $homeserver);
            } else {
                $response=&reply('instuserrules:'.&escape($udom),
                                 $homeserver);
            }
            if (($response ne 'refused') && ($response ne 'error') && 
                ($response ne 'unknown_cmd') && 
                ($response ne 'no_such_host')) {
                my ($hashitems,$orderitems) = split(/:/,$response);
                my @pairs=split(/\&/,$hashitems);
                foreach my $item (@pairs) {
                    my ($key,$value)=split(/=/,$item,2);
                    $key = &unescape($key);
                    next if ($key =~ /^error: 2 /);
                    $ruleshash{$key}=&thaw_unescape($value);
                }
                my @esc_order = split(/\&/,$orderitems);
                foreach my $item (@esc_order) {
                    push(@ruleorder,&unescape($item));
                }
            }
        }
    }
    return (\%ruleshash,\@ruleorder);
}

# ------------- Get Authentication, Language and User Tools Defaults for Domain

sub get_domain_defaults {
    my ($domain) = @_;
    my $cachetime = 60*60*24;
    my ($result,$cached)=&is_cached_new('domdefaults',$domain);
    if (defined($cached)) {
        if (ref($result) eq 'HASH') {
            return %{$result};
        }
    }
    my %domdefaults;
    my %domconfig =
         &Apache::lonnet::get_dom('configuration',['defaults','quotas',
                                  'requestcourses','inststatus',
                                  'coursedefaults'],$domain);
    if (ref($domconfig{'defaults'}) eq 'HASH') {
        $domdefaults{'lang_def'} = $domconfig{'defaults'}{'lang_def'}; 
        $domdefaults{'auth_def'} = $domconfig{'defaults'}{'auth_def'};
        $domdefaults{'auth_arg_def'} = $domconfig{'defaults'}{'auth_arg_def'};
        $domdefaults{'timezone_def'} = $domconfig{'defaults'}{'timezone_def'};
        $domdefaults{'datelocale_def'} = $domconfig{'defaults'}{'datelocale_def'};
    } else {
        $domdefaults{'lang_def'} = &domain($domain,'lang_def');
        $domdefaults{'auth_def'} = &domain($domain,'auth_def');
        $domdefaults{'auth_arg_def'} = &domain($domain,'auth_arg_def');
    }
    if (ref($domconfig{'quotas'}) eq 'HASH') {
        if (ref($domconfig{'quotas'}{'defaultquota'}) eq 'HASH') {
            $domdefaults{'defaultquota'} = $domconfig{'quotas'}{'defaultquota'};
        } else {
            $domdefaults{'defaultquota'} = $domconfig{'quotas'};
        } 
        my @usertools = ('aboutme','blog','portfolio');
        foreach my $item (@usertools) {
            if (ref($domconfig{'quotas'}{$item}) eq 'HASH') {
                $domdefaults{$item} = $domconfig{'quotas'}{$item};
            }
        }
    }
    if (ref($domconfig{'requestcourses'}) eq 'HASH') {
        foreach my $item ('official','unofficial','community') {
            $domdefaults{$item} = $domconfig{'requestcourses'}{$item};
        }
    }
    if (ref($domconfig{'inststatus'}) eq 'HASH') {
        foreach my $item ('inststatustypes','inststatusorder') {
            $domdefaults{$item} = $domconfig{'inststatus'}{$item};
        }
    }
    if (ref($domconfig{'coursedefaults'}) eq 'HASH') {
        foreach my $item ('canuse_pdfforms') {
            $domdefaults{$item} = $domconfig{'coursedefaults'}{$item};
        }
    }
    &Apache::lonnet::do_cache_new('domdefaults',$domain,\%domdefaults,
                                  $cachetime);
    return %domdefaults;
}

# --------------------------------------------------- Assign a key to a student

sub assign_access_key {
#
# a valid key looks like uname:udom#comments
# comments are being appended
#
    my ($ckey,$kdom,$knum,$cdom,$cnum,$udom,$uname,$logentry)=@_;
    $kdom=
   $env{'course.'.$env{'request.course.id'}.'.domain'} unless (defined($kdom));
    $knum=
   $env{'course.'.$env{'request.course.id'}.'.num'} unless (defined($knum));
    $cdom=
   $env{'course.'.$env{'request.course.id'}.'.domain'} unless (defined($cdom));
    $cnum=
   $env{'course.'.$env{'request.course.id'}.'.num'} unless (defined($cnum));
    $udom=$env{'user.name'} unless (defined($udom));
    $uname=$env{'user.domain'} unless (defined($uname));
    my %existing=&get('accesskeys',[$ckey],$kdom,$knum);
    if (($existing{$ckey}=~/^\#(.*)$/) || # - new key
        ($existing{$ckey}=~/^\Q$uname\E\:\Q$udom\E\#(.*)$/)) { 
                                                  # assigned to this person
                                                  # - this should not happen,
                                                  # unless something went wrong
                                                  # the first time around
# ready to assign
        $logentry=$1.'; '.$logentry;
        if (&put('accesskeys',{$ckey=>$uname.':'.$udom.'#'.$logentry},
                                                 $kdom,$knum) eq 'ok') {
# key now belongs to user
	    my $envkey='key.'.$cdom.'_'.$cnum;
            if (&put('environment',{$envkey => $ckey}) eq 'ok') {
                &appenv({'environment.'.$envkey => $ckey});
                return 'ok';
            } else {
                return 
  'error: Count not permanently assign key, will need to be re-entered later.';
	    }
        } else {
            return 'error: Could not assign key, try again later.';
        }
    } elsif (!$existing{$ckey}) {
# the key does not exist
	return 'error: The key does not exist';
    } else {
# the key is somebody else's
	return 'error: The key is already in use';
    }
}

# ------------------------------------------ put an additional comment on a key

sub comment_access_key {
#
# a valid key looks like uname:udom#comments
# comments are being appended
#
    my ($ckey,$cdom,$cnum,$logentry)=@_;
    $cdom=
   $env{'course.'.$env{'request.course.id'}.'.domain'} unless (defined($cdom));
    $cnum=
   $env{'course.'.$env{'request.course.id'}.'.num'} unless (defined($cnum));
    my %existing=&get('accesskeys',[$ckey],$cdom,$cnum);
    if ($existing{$ckey}) {
        $existing{$ckey}.='; '.$logentry;
# ready to assign
        if (&put('accesskeys',{$ckey=>$existing{$ckey}},
                                                 $cdom,$cnum) eq 'ok') {
	    return 'ok';
        } else {
	    return 'error: Count not store comment.';
        }
    } else {
# the key does not exist
	return 'error: The key does not exist';
    }
}

# ------------------------------------------------------ Generate a set of keys

sub generate_access_keys {
    my ($number,$cdom,$cnum,$logentry)=@_;
    $cdom=
   $env{'course.'.$env{'request.course.id'}.'.domain'} unless (defined($cdom));
    $cnum=
   $env{'course.'.$env{'request.course.id'}.'.num'} unless (defined($cnum));
    unless (&allowed('mky',$cdom)) { return 0; }
    unless (($cdom) && ($cnum)) { return 0; }
    if ($number>10000) { return 0; }
    sleep(2); # make sure don't get same seed twice
    srand(time()^($$+($$<<15))); # from "Programming Perl"
    my $total=0;
    for (my $i=1;$i<=$number;$i++) {
       my $newkey=sprintf("%lx",int(100000*rand)).'-'.
                  sprintf("%lx",int(100000*rand)).'-'.
                  sprintf("%lx",int(100000*rand));
       $newkey=~s/1/g/g; # folks mix up 1 and l
       $newkey=~s/0/h/g; # and also 0 and O
       my %existing=&get('accesskeys',[$newkey],$cdom,$cnum);
       if ($existing{$newkey}) {
           $i--;
       } else {
	  if (&put('accesskeys',
              { $newkey => '# generated '.localtime().
                           ' by '.$env{'user.name'}.'@'.$env{'user.domain'}.
                           '; '.$logentry },
		   $cdom,$cnum) eq 'ok') {
              $total++;
	  }
       }
    }
    &log($env{'user.domain'},$env{'user.name'},$env{'user.home'},
         'Generated '.$total.' keys for '.$cnum.' at '.$cdom);
    return $total;
}

# ------------------------------------------------------- Validate an accesskey

sub validate_access_key {
    my ($ckey,$cdom,$cnum,$udom,$uname)=@_;
    $cdom=
   $env{'course.'.$env{'request.course.id'}.'.domain'} unless (defined($cdom));
    $cnum=
   $env{'course.'.$env{'request.course.id'}.'.num'} unless (defined($cnum));
    $udom=$env{'user.domain'} unless (defined($udom));
    $uname=$env{'user.name'} unless (defined($uname));
    my %existing=&get('accesskeys',[$ckey],$cdom,$cnum);
    return ($existing{$ckey}=~/^\Q$uname\E\:\Q$udom\E\#/);
}

# ------------------------------------- Find the section of student in a course
sub devalidate_getsection_cache {
    my ($udom,$unam,$courseid)=@_;
    my $hashid="$udom:$unam:$courseid";
    &devalidate_cache_new('getsection',$hashid);
}

sub courseid_to_courseurl {
    my ($courseid) = @_;
    #already url style courseid
    return $courseid if ($courseid =~ m{^/});

    if (exists($env{'course.'.$courseid.'.num'})) {
	my $cnum = $env{'course.'.$courseid.'.num'};
	my $cdom = $env{'course.'.$courseid.'.domain'};
	return "/$cdom/$cnum";
    }

    my %courseinfo=&Apache::lonnet::coursedescription($courseid);
    if (exists($courseinfo{'num'})) {
	return "/$courseinfo{'domain'}/$courseinfo{'num'}";
    }

    return undef;
}

sub getsection {
    my ($udom,$unam,$courseid)=@_;
    my $cachetime=1800;

    my $hashid="$udom:$unam:$courseid";
    my ($result,$cached)=&is_cached_new('getsection',$hashid);
    if (defined($cached)) { return $result; }

    my %Pending; 
    my %Expired;
    #
    # Each role can either have not started yet (pending), be active, 
    #    or have expired.
    #
    # If there is an active role, we are done.
    #
    # If there is more than one role which has not started yet, 
    #     choose the one which will start sooner
    # If there is one role which has not started yet, return it.
    #
    # If there is more than one expired role, choose the one which ended last.
    # If there is a role which has expired, return it.
    #
    $courseid = &courseid_to_courseurl($courseid);
    my %roleshash = &dump('roles',$udom,$unam,$courseid);
    foreach my $key (keys(%roleshash)) {
        next if ($key !~/^\Q$courseid\E(?:\/)*(\w+)*\_st$/);
        my $section=$1;
        if ($key eq $courseid.'_st') { $section=''; }
        my ($dummy,$end,$start)=split(/\_/,&unescape($roleshash{$key}));
        my $now=time;
        if (defined($end) && $end && ($now > $end)) {
            $Expired{$end}=$section;
            next;
        }
        if (defined($start) && $start && ($now < $start)) {
            $Pending{$start}=$section;
            next;
        }
        return &do_cache_new('getsection',$hashid,$section,$cachetime);
    }
    #
    # Presumedly there will be few matching roles from the above
    # loop and the sorting time will be negligible.
    if (scalar(keys(%Pending))) {
        my ($time) = sort {$a <=> $b} keys(%Pending);
        return &do_cache_new('getsection',$hashid,$Pending{$time},$cachetime);
    } 
    if (scalar(keys(%Expired))) {
        my @sorted = sort {$a <=> $b} keys(%Expired);
        my $time = pop(@sorted);
        return &do_cache_new('getsection',$hashid,$Expired{$time},$cachetime);
    }
    return &do_cache_new('getsection',$hashid,'-1',$cachetime);
}

sub save_cache {
    &purge_remembered();
    #&Apache::loncommon::validate_page();
    undef(%env);
    undef($env_loaded);
}

my $to_remember=-1;
my %remembered;
my %accessed;
my $kicks=0;
my $hits=0;
sub make_key {
    my ($name,$id) = @_;
    if (length($id) > 65 
	&& length(&escape($id)) > 200) {
	$id=length($id).':'.&Digest::MD5::md5_hex($id);
    }
    return &escape($name.':'.$id);
}

sub devalidate_cache_new {
    my ($name,$id,$debug) = @_;
    if ($debug) { &Apache::lonnet::logthis("deleting $name:$id"); }
    $id=&make_key($name,$id);
    $memcache->delete($id);
    delete($remembered{$id});
    delete($accessed{$id});
}

sub is_cached_new {
    my ($name,$id,$debug) = @_;
    $id=&make_key($name,$id);
    if (exists($remembered{$id})) {
	if ($debug) { &Apache::lonnet::logthis("Earyl return $id of $remembered{$id} "); }
	$accessed{$id}=[&gettimeofday()];
	$hits++;
	return ($remembered{$id},1);
    }
    my $value = $memcache->get($id);
    if (!(defined($value))) {
	if ($debug) { &Apache::lonnet::logthis("getting $id is not defined"); }
	return (undef,undef);
    }
    if ($value eq '__undef__') {
	if ($debug) { &Apache::lonnet::logthis("getting $id is __undef__"); }
	$value=undef;
    }
    &make_room($id,$value,$debug);
    if ($debug) { &Apache::lonnet::logthis("getting $id is $value"); }
    return ($value,1);
}

sub do_cache_new {
    my ($name,$id,$value,$time,$debug) = @_;
    $id=&make_key($name,$id);
    my $setvalue=$value;
    if (!defined($setvalue)) {
	$setvalue='__undef__';
    }
    if (!defined($time) ) {
	$time=600;
    }
    if ($debug) { &Apache::lonnet::logthis("Setting $id to $value"); }
    my $result = $memcache->set($id,$setvalue,$time);
    if (! $result) {
	&logthis("caching of id -> $id  failed");
	$memcache->disconnect_all();
    }
    # need to make a copy of $value
    &make_room($id,$value,$debug);
    return $value;
}

sub make_room {
    my ($id,$value,$debug)=@_;

    $remembered{$id}= (ref($value)) ? &Storable::dclone($value)
                                    : $value;
    if ($to_remember<0) { return; }
    $accessed{$id}=[&gettimeofday()];
    if (scalar(keys(%remembered)) <= $to_remember) { return; }
    my $to_kick;
    my $max_time=0;
    foreach my $other (keys(%accessed)) {
	if (&tv_interval($accessed{$other}) > $max_time) {
	    $to_kick=$other;
	    $max_time=&tv_interval($accessed{$other});
	}
    }
    delete($remembered{$to_kick});
    delete($accessed{$to_kick});
    $kicks++;
    if ($debug) { &logthis("kicking $to_kick $max_time $kicks\n"); }
    return;
}

sub purge_remembered {
    #&logthis("Tossing ".scalar(keys(%remembered)));
    #&logthis(sprintf("%-20s is %s",'%remembered',length(&freeze(\%remembered))));
    undef(%remembered);
    undef(%accessed);
}
# ------------------------------------- Read an entry from a user's environment

sub userenvironment {
    my ($udom,$unam,@what)=@_;
    my $items;
    foreach my $item (@what) {
        $items.=&escape($item).'&';
    }
    $items=~s/\&$//;
    my %returnhash=();
    my $uhome = &homeserver($unam,$udom);
    unless ($uhome eq 'no_host') {
        my @answer=split(/\&/, 
            &reply('get:'.$udom.':'.$unam.':environment:'.$items,$uhome));
        if ($#answer==0 && $answer[0] =~ /^(con_lost|error:|no_such_host)/i) {
            return %returnhash;
        }
        my $i;
        for ($i=0;$i<=$#what;$i++) {
	    $returnhash{$what[$i]}=&unescape($answer[$i]);
        }
    }
    return %returnhash;
}

# ---------------------------------------------------------- Get a studentphoto
sub studentphoto {
    my ($udom,$unam,$ext) = @_;
    my $home=&Apache::lonnet::homeserver($unam,$udom);
    if (defined($env{'request.course.id'})) {
        if ($env{'course.'.$env{'request.course.id'}.'.internal.showphoto'}) {
            if ($udom eq $env{'course.'.$env{'request.course.id'}.'.domain'}) {
                return(&retrievestudentphoto($udom,$unam,$ext)); 
            } else {
                my ($result,$perm_reqd)=
		    &Apache::lonnet::auto_photo_permission($unam,$udom);
                if ($result eq 'ok') {
                    if (!($perm_reqd eq 'yes')) {
                        return(&retrievestudentphoto($udom,$unam,$ext));
                    }
                }
            }
        }
    } else {
        my ($result,$perm_reqd) = 
	    &Apache::lonnet::auto_photo_permission($unam,$udom);
        if ($result eq 'ok') {
            if (!($perm_reqd eq 'yes')) {
                return(&retrievestudentphoto($udom,$unam,$ext));
            }
        }
    }
    return '/adm/lonKaputt/lonlogo_broken.gif';
}

sub retrievestudentphoto {
    my ($udom,$unam,$ext,$type) = @_;
    my $home=&Apache::lonnet::homeserver($unam,$udom);
    my $ret=&Apache::lonnet::reply("studentphoto:$udom:$unam:$ext:$type",$home);
    if ($ret eq 'ok') {
        my $url="/uploaded/$udom/$unam/internal/studentphoto.$ext";
        if ($type eq 'thumbnail') {
            $url="/uploaded/$udom/$unam/internal/studentphoto_tn.$ext"; 
        }
        my $tokenurl=&Apache::lonnet::tokenwrapper($url);
        return $tokenurl;
    } else {
        if ($type eq 'thumbnail') {
            return '/adm/lonKaputt/genericstudent_tn.gif';
        } else { 
            return '/adm/lonKaputt/lonlogo_broken.gif';
        }
    }
}

# -------------------------------------------------------------------- New chat

sub chatsend {
    my ($newentry,$anon,$group)=@_;
    my $cnum=$env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    my $chome=$env{'course.'.$env{'request.course.id'}.'.home'};
    &reply('chatsend:'.$cdom.':'.$cnum.':'.
	   &escape($env{'user.domain'}.':'.$env{'user.name'}.':'.$anon.':'.
		   &escape($newentry)).':'.$group,$chome);
}

# ------------------------------------------ Find current version of a resource

sub getversion {
    my $fname=&clutter(shift);
    unless ($fname=~/^\/res\//) { return -1; }
    return &currentversion(&filelocation('',$fname));
}

sub currentversion {
    my $fname=shift;
    my ($result,$cached)=&is_cached_new('resversion',$fname);
    if (defined($cached)) { return $result; }
    my $author=$fname;
    $author=~s/\/home\/httpd\/html\/res\/([^\/]*)\/([^\/]*).*/$1\/$2/;
    my ($udom,$uname)=split(/\//,$author);
    my $home=homeserver($uname,$udom);
    if ($home eq 'no_host') { 
        return -1; 
    }
    my $answer=reply("currentversion:$fname",$home);
    if (($answer eq 'con_lost') || ($answer eq 'rejected')) {
	return -1;
    }
    return &do_cache_new('resversion',$fname,$answer,600);
}

# ----------------------------- Subscribe to a resource, return URL if possible

sub subscribe {
    my $fname=shift;
    if ($fname=~/\/(aboutme|syllabus|bulletinboard|smppg)$/) { return ''; }
    $fname=~s/[\n\r]//g;
    my $author=$fname;
    $author=~s/\/home\/httpd\/html\/res\/([^\/]*)\/([^\/]*).*/$1\/$2/;
    my ($udom,$uname)=split(/\//,$author);
    my $home=homeserver($uname,$udom);
    if ($home eq 'no_host') {
        return 'not_found';
    }
    my $answer=reply("sub:$fname",$home);
    if (($answer eq 'con_lost') || ($answer eq 'rejected')) {
	$answer.=' by '.$home;
    }
    return $answer;
}
    
# -------------------------------------------------------------- Replicate file

sub repcopy {
    my $filename=shift;
    $filename=~s/\/+/\//g;
    if ($filename=~m|^/home/httpd/html/adm/|) { return 'ok'; }
    if ($filename=~m|^/home/httpd/html/lonUsers/|) { return 'ok'; }
    if ($filename=~m|^/home/httpd/html/userfiles/| or
	$filename=~m -^/*(uploaded|editupload)/-) { 
	return &repcopy_userfile($filename);
    }
    $filename=~s/[\n\r]//g;
    my $transname="$filename.in.transfer";
# FIXME: this should flock
    if ((-e $filename) || (-e $transname)) { return 'ok'; }
    my $remoteurl=subscribe($filename);
    if ($remoteurl =~ /^con_lost by/) {
	   &logthis("Subscribe returned $remoteurl: $filename");
           return 'unavailable';
    } elsif ($remoteurl eq 'not_found') {
	   #&logthis("Subscribe returned not_found: $filename");
	   return 'not_found';
    } elsif ($remoteurl =~ /^rejected by/) {
	   &logthis("Subscribe returned $remoteurl: $filename");
           return 'forbidden';
    } elsif ($remoteurl eq 'directory') {
           return 'ok';
    } else {
        my $author=$filename;
        $author=~s/\/home\/httpd\/html\/res\/([^\/]*)\/([^\/]*).*/$1\/$2/;
        my ($udom,$uname)=split(/\//,$author);
        my $home=homeserver($uname,$udom);
        unless ($home eq $perlvar{'lonHostID'}) {
           my @parts=split(/\//,$filename);
           my $path="/$parts[1]/$parts[2]/$parts[3]/$parts[4]";
           if ($path ne "$perlvar{'lonDocRoot'}/res") {
               &logthis("Malconfiguration for replication: $filename");
	       return 'bad_request';
           }
           my $count;
           for ($count=5;$count<$#parts;$count++) {
               $path.="/$parts[$count]";
               if ((-e $path)!=1) {
		   mkdir($path,0777);
               }
           }
           my $ua=new LWP::UserAgent;
           my $request=new HTTP::Request('GET',"$remoteurl");
           my $response=$ua->request($request,$transname);
           if ($response->is_error()) {
	       unlink($transname);
               my $message=$response->status_line;
               &logthis("<font color=\"blue\">WARNING:"
                       ." LWP get: $message: $filename</font>");
               return 'unavailable';
           } else {
	       if ($remoteurl!~/\.meta$/) {
                  my $mrequest=new HTTP::Request('GET',$remoteurl.'.meta');
                  my $mresponse=$ua->request($mrequest,$filename.'.meta');
                  if ($mresponse->is_error()) {
		      unlink($filename.'.meta');
                      &logthis(
                     "<font color=\"yellow\">INFO: No metadata: $filename</font>");
                  }
	       }
               rename($transname,$filename);
               return 'ok';
           }
       }
    }
}

# ------------------------------------------------ Get server side include body
sub ssi_body {
    my ($filelink,%form)=@_;
    if (! exists($form{'LONCAPA_INTERNAL_no_discussion'})) {
        $form{'LONCAPA_INTERNAL_no_discussion'}='true';
    }
    my $output='';
    my $response;
    if ($filelink=~/^https?\:/) {
       ($output,$response)=&externalssi($filelink);
    } else {
       $filelink .= $filelink=~/\?/ ? '&' : '?';
       $filelink .= 'inhibitmenu=yes';
       ($output,$response)=&ssi($filelink,%form);
    }
    $output=~s|//(\s*<!--)? BEGIN LON-CAPA Internal.+?// END LON-CAPA Internal\s*(-->)?\s||gs;
    $output=~s/^.*?\<body[^\>]*\>//si;
    $output=~s/\<\/body\s*\>.*?$//si;
    if (wantarray) {
        return ($output, $response);
    } else {
        return $output;
    }
}

# --------------------------------------------------------- Server Side Include

sub absolute_url {
    my ($host_name) = @_;
    my $protocol = ($ENV{'SERVER_PORT'} == 443?'https://':'http://');
    if ($host_name eq '') {
	$host_name = $ENV{'SERVER_NAME'};
    }
    return $protocol.$host_name;
}

#
#   Server side include.
# Parameters:
#  fn     Possibly encrypted resource name/id.
#  form   Hash that describes how the rendering should be done
#         and other things.
# Returns:
#   Scalar context: The content of the response.
#   Array context:  2 element list of the content and the full response object.
#     
sub ssi {

    my ($fn,%form)=@_;
    my $ua=new LWP::UserAgent;
    my $request;

    $form{'no_update_last_known'}=1;
    &Apache::lonenc::check_encrypt(\$fn);
    if (%form) {
      $request=new HTTP::Request('POST',&absolute_url().$fn);
      $request->content(join('&',map { &escape($_).'='.&escape($form{$_}) } keys(%form)));
    } else {
      $request=new HTTP::Request('GET',&absolute_url().$fn);
    }

    $request->header(Cookie => $ENV{'HTTP_COOKIE'});
    my $response=$ua->request($request);

    if (wantarray) {
	return ($response->content, $response);
    } else {
	return $response->content;
    }
}

sub externalssi {
    my ($url)=@_;
    my $ua=new LWP::UserAgent;
    my $request=new HTTP::Request('GET',$url);
    my $response=$ua->request($request);
    if (wantarray) {
        return ($response->content, $response);
    } else {
        return $response->content;
    }
}

# -------------------------------- Allow a /uploaded/ URI to be vouched for

sub allowuploaded {
    my ($srcurl,$url)=@_;
    $url=&clutter(&declutter($url));
    my $dir=$url;
    $dir=~s/\/[^\/]+$//;
    my %httpref=();
    my $httpurl=&hreflocation('',$url);
    $httpref{'httpref.'.$httpurl}=$srcurl;
    &Apache::lonnet::appenv(\%httpref);
}

# --------- File operations in /home/httpd/html/userfiles/$domain/1/2/3/$course
# input: action, courseID, current domain, intended
#        path to file, source of file, instruction to parse file for objects,
#        ref to hash for embedded objects,
#        ref to hash for codebase of java objects.
#
# output: url to file (if action was uploaddoc), 
#         ok if successful, or diagnostic message otherwise (if action was propagate or copy)
#
# Allows directory structure to be used within lonUsers/../userfiles/ for a 
# course.
#
# action = propagate - /home/httpd/html/userfiles/$domain/1/2/3/$course/$file
#          will be copied to /home/httpd/lonUsers/1/2/3/$course/userfiles in
#          course's home server.
#
# action = copy - /home/httpd/html/userfiles/$domain/1/2/3/$course/$file will
#          be copied from $source (current location) to 
#          /home/httpd/html/userfiles/$domain/1/2/3/$course/$file
#         and will then be copied to
#          /home/httpd/lonUsers/$domain/1/2/3/$course/userfiles/$file in
#         course's home server.
#
# action = uploaddoc - /home/httpd/html/userfiles/$domain/1/2/3/$course/$file
#         will be retrived from $env{form.uploaddoc} (from DOCS interface) to
#         /home/httpd/html/userfiles/$domain/1/2/3/$course/$file
#         and will then be copied to /home/httpd/lonUsers/1/2/3/$course/userfiles/$file
#         in course's home server.
#

sub process_coursefile {
    my ($action,$docuname,$docudom,$file,$source,$parser,$allfiles,$codebase)=@_;
    my $fetchresult;
    my $home=&homeserver($docuname,$docudom);
    if ($action eq 'propagate') {
        $fetchresult= &reply('fetchuserfile:'.$docudom.'/'.$docuname.'/'.$file,
			     $home);
    } else {
        my $fpath = '';
        my $fname = $file;
        ($fpath,$fname) = ($file =~ m|^(.*)/([^/]+)$|);
        $fpath=$docudom.'/'.$docuname.'/'.$fpath;
        my $filepath = &build_filepath($fpath);
        if ($action eq 'copy') {
            if ($source eq '') {
                $fetchresult = 'no source file';
                return $fetchresult;
            } else {
                my $destination = $filepath.'/'.$fname;
                rename($source,$destination);
                $fetchresult= &reply('fetchuserfile:'.$docudom.'/'.$docuname.'/'.$file,
                                 $home);
            }
        } elsif ($action eq 'uploaddoc') {
            open(my $fh,'>'.$filepath.'/'.$fname);
            print $fh $env{'form.'.$source};
            close($fh);
            if ($parser eq 'parse') {
                my $mm = new File::MMagic;
                my $mime_type = $mm->checktype_filename($filepath.'/'.$fname);
                if ($mime_type eq 'text/html') {
                    my $parse_result = &extract_embedded_items($filepath.'/'.$fname,$allfiles,$codebase);
                    unless ($parse_result eq 'ok') {
                        &logthis('Failed to parse '.$filepath.'/'.$fname.' for embedded media: '.$parse_result);
                    }
                }
            }
            $fetchresult= &reply('fetchuserfile:'.$docudom.'/'.$docuname.'/'.$file,
                                 $home);
            if ($fetchresult eq 'ok') {
                return '/uploaded/'.$fpath.'/'.$fname;
            } else {
                &logthis('Failed to transfer '.$docudom.'/'.$docuname.'/'.$file.
                        ' to host '.$home.': '.$fetchresult);
                return '/adm/notfound.html';
            }
        }
    }
    unless ( $fetchresult eq 'ok') {
        &logthis('Failed to transfer '.$docudom.'/'.$docuname.'/'.$file.
             ' to host '.$home.': '.$fetchresult);
    }
    return $fetchresult;
}

sub build_filepath {
    my ($fpath) = @_;
    my $filepath=$perlvar{'lonDocRoot'}.'/userfiles';
    unless ($fpath eq '') {
        my @parts=split('/',$fpath);
        foreach my $part (@parts) {
            $filepath.= '/'.$part;
            if ((-e $filepath)!=1) {
                mkdir($filepath,0777);
            }
        }
    }
    return $filepath;
}

sub store_edited_file {
    my ($primary_url,$content,$docudom,$docuname,$fetchresult) = @_;
    my $file = $primary_url;
    $file =~ s#^/uploaded/$docudom/$docuname/##;
    my $fpath = '';
    my $fname = $file;
    ($fpath,$fname) = ($file =~ m|^(.*)/([^/]+)$|);
    $fpath=$docudom.'/'.$docuname.'/'.$fpath;
    my $filepath = &build_filepath($fpath);
    open(my $fh,'>'.$filepath.'/'.$fname);
    print $fh $content;
    close($fh);
    my $home=&homeserver($docuname,$docudom);
    $$fetchresult= &reply('fetchuserfile:'.$docudom.'/'.$docuname.'/'.$file,
			  $home);
    if ($$fetchresult eq 'ok') {
        return '/uploaded/'.$fpath.'/'.$fname;
    } else {
        &logthis('Failed to transfer '.$docudom.'/'.$docuname.'/'.$file.
		 ' to host '.$home.': '.$$fetchresult);
        return '/adm/notfound.html';
    }
}

sub clean_filename {
    my ($fname,$args)=@_;
# Replace Windows backslashes by forward slashes
    $fname=~s/\\/\//g;
    if (!$args->{'keep_path'}) {
        # Get rid of everything but the actual filename
	$fname=~s/^.*\/([^\/]+)$/$1/;
    }
# Replace spaces by underscores
    $fname=~s/\s+/\_/g;
# Replace all other weird characters by nothing
    $fname=~s{[^/\w\.\-]}{}g;
# Replace all .\d. sequences with _\d. so they no longer look like version
# numbers
    $fname=~s/\.(\d+)(?=\.)/_$1/g;
    return $fname;
}
# This Function checks if an Image's dimensions exceed either $resizewidth (width) 
# or $resizeheight (height) - both pixels. If so, the image is scaled to produce an 
# image with the same aspect ratio as the original, but with dimensions which do 
# not exceed $resizewidth and $resizeheight.
 
sub resizeImage {
    my ($img_path,$resizewidth,$resizeheight) = @_;
    my $ima = Image::Magick->new;
    my $resized;
    if (-e $img_path) {
        $ima->Read($img_path);
        if (($resizewidth =~ /^\d+$/) && ($resizeheight > 0)) {
            my $width = $ima->Get('width');
            my $height = $ima->Get('height');
            if ($width > $resizewidth) {
	        my $factor = $width/$resizewidth;
                my $newheight = $height/$factor;
                $ima->Scale(width=>$resizewidth,height=>$newheight);
                $resized = 1;
            }
        }
        if (($resizeheight =~ /^\d+$/) && ($resizeheight > 0)) {
            my $width = $ima->Get('width');
            my $height = $ima->Get('height');
            if ($height > $resizeheight) {
                my $factor = $height/$resizeheight;
                my $newwidth = $width/$factor;
                $ima->Scale(width=>$newwidth,height=>$resizeheight);
                $resized = 1;
            }
        }
        if ($resized) {
            $ima->Write($img_path);
        }
    }
    return;
}

# --------------- Take an uploaded file and put it into the userfiles directory
# input: $formname - the contents of the file are in $env{"form.$formname"}
#                    the desired filenam is in $env{"form.$formname.filename"}
#        $coursedoc - if true up to the current course
#                     if false
#        $subdir - directory in userfile to store the file into
#        $parser - instruction to parse file for objects ($parser = parse)    
#        $allfiles - reference to hash for embedded objects
#        $codebase - reference to hash for codebase of java objects
#        $desuname - username for permanent storage of uploaded file
#        $dsetudom - domain for permanaent storage of uploaded file
#        $thumbwidth - width (pixels) of thumbnail to make for uploaded image 
#        $thumbheight - height (pixels) of thumbnail to make for uploaded image
#        $resizewidth - width (pixels) to which to resize uploaded image
#        $resizeheight - height (pixels) to which to resize uploaded image
# 
# output: url of file in userspace, or error: <message> 
#             or /adm/notfound.html if failure to upload occurse

sub userfileupload {
    my ($formname,$coursedoc,$subdir,$parser,$allfiles,$codebase,$destuname,
        $destudom,$thumbwidth,$thumbheight,$resizewidth,$resizeheight)=@_;
    if (!defined($subdir)) { $subdir='unknown'; }
    my $fname=$env{'form.'.$formname.'.filename'};
    $fname=&clean_filename($fname);
# See if there is anything left
    unless ($fname) { return 'error: no uploaded file'; }
    chop($env{'form.'.$formname});
    if (($formname eq 'screenshot') && ($subdir eq 'helprequests')) { #files uploaded to help request form are handled differently
        my $now = time;
        my $filepath = 'tmp/helprequests/'.$now;
        my @parts=split(/\//,$filepath);
        my $fullpath = $perlvar{'lonDaemons'};
        for (my $i=0;$i<@parts;$i++) {
            $fullpath .= '/'.$parts[$i];
            if ((-e $fullpath)!=1) {
                mkdir($fullpath,0777);
            }
        }
        open(my $fh,'>'.$fullpath.'/'.$fname);
        print $fh $env{'form.'.$formname};
        close($fh);
        return $fullpath.'/'.$fname;
    } elsif (($formname eq 'coursecreatorxml') && ($subdir eq 'batchupload')) { #files uploaded to create course page are handled differently
        my $filepath = 'tmp/addcourse/'.$destudom.'/web/'.$env{'user.name'}.
                       '_'.$env{'user.domain'}.'/pending';
        my @parts=split(/\//,$filepath);
        my $fullpath = $perlvar{'lonDaemons'};
        for (my $i=0;$i<@parts;$i++) {
            $fullpath .= '/'.$parts[$i];
            if ((-e $fullpath)!=1) {
                mkdir($fullpath,0777);
            }
        }
        open(my $fh,'>'.$fullpath.'/'.$fname);
        print $fh $env{'form.'.$formname};
        close($fh);
        return $fullpath.'/'.$fname;
    }
    if ($subdir eq 'scantron') {
        $fname = 'scantron_orig_'.$fname;
    } else {   
# Create the directory if not present
        $fname="$subdir/$fname";
    }
    if ($coursedoc) {
	my $docuname=$env{'course.'.$env{'request.course.id'}.'.num'};
	my $docudom=$env{'course.'.$env{'request.course.id'}.'.domain'};
        if ($env{'form.folder'} =~ m/^(default|supplemental)/) {
            return &finishuserfileupload($docuname,$docudom,
					 $formname,$fname,$parser,$allfiles,
					 $codebase,$thumbwidth,$thumbheight,
                                         $resizewidth,$resizeheight);
        } else {
            $fname=$env{'form.folder'}.'/'.$fname;
            return &process_coursefile('uploaddoc',$docuname,$docudom,
				       $fname,$formname,$parser,
				       $allfiles,$codebase);
        }
    } elsif (defined($destuname)) {
        my $docuname=$destuname;
        my $docudom=$destudom;
	return &finishuserfileupload($docuname,$docudom,$formname,$fname,
				     $parser,$allfiles,$codebase,
                                     $thumbwidth,$thumbheight,
                                     $resizewidth,$resizeheight);
        
    } else {
        my $docuname=$env{'user.name'};
        my $docudom=$env{'user.domain'};
        if (exists($env{'form.group'})) {
            $docuname=$env{'course.'.$env{'request.course.id'}.'.num'};
            $docudom=$env{'course.'.$env{'request.course.id'}.'.domain'};
        }
	return &finishuserfileupload($docuname,$docudom,$formname,$fname,
				     $parser,$allfiles,$codebase,
                                     $thumbwidth,$thumbheight,
                                     $resizewidth,$resizeheight);
    }
}

sub finishuserfileupload {
    my ($docuname,$docudom,$formname,$fname,$parser,$allfiles,$codebase,
        $thumbwidth,$thumbheight,$resizewidth,$resizeheight) = @_;
    my $path=$docudom.'/'.$docuname.'/';
    my $filepath=$perlvar{'lonDocRoot'};
  
    my ($fnamepath,$file,$fetchthumb);
    $file=$fname;
    if ($fname=~m|/|) {
        ($fnamepath,$file) = ($fname =~ m|^(.*)/([^/]+)$|);
	$path.=$fnamepath.'/';
    }
    my @parts=split(/\//,$filepath.'/userfiles/'.$path);
    my $count;
    for ($count=4;$count<=$#parts;$count++) {
        $filepath.="/$parts[$count]";
        if ((-e $filepath)!=1) {
	    mkdir($filepath,0777);
        }
    }

# Save the file
    {
	if (!open(FH,'>'.$filepath.'/'.$file)) {
	    &logthis('Failed to create '.$filepath.'/'.$file);
	    print STDERR ('Failed to create '.$filepath.'/'.$file."\n");
	    return '/adm/notfound.html';
	}
	if (!print FH ($env{'form.'.$formname})) {
	    &logthis('Failed to write to '.$filepath.'/'.$file);
	    print STDERR ('Failed to write to '.$filepath.'/'.$file."\n");
	    return '/adm/notfound.html';
	}
	close(FH);
        if ($resizewidth && $resizeheight) {
            my $mm = new File::MMagic;
            my $mime_type = $mm->checktype_filename($filepath.'/'.$file);
            if ($mime_type =~ m{^image/}) {
	        &resizeImage($filepath.'/'.$file,$resizewidth,$resizeheight);
            }  
	}
    }
    if ($parser eq 'parse') {
        my $mm = new File::MMagic;
        my $mime_type = $mm->checktype_filename($filepath.'/'.$file);
        if ($mime_type eq 'text/html') {
            my $parse_result = &extract_embedded_items($filepath.'/'.$file,
                                                       $allfiles,$codebase);
            unless ($parse_result eq 'ok') {
                &logthis('Failed to parse '.$filepath.$file.
	   	         ' for embedded media: '.$parse_result); 
            }
        }
    }
    if (($thumbwidth =~ /^\d+$/) && ($thumbheight =~ /^\d+$/)) {
        my $input = $filepath.'/'.$file;
        my $output = $filepath.'/'.'tn-'.$file;
        my $thumbsize = $thumbwidth.'x'.$thumbheight;
        system("convert -sample $thumbsize $input $output");
        if (-e $filepath.'/'.'tn-'.$file) {
            $fetchthumb  = 1; 
        }
    }
 
# Notify homeserver to grep it
#
    my $docuhome=&homeserver($docuname,$docudom);	
    my $fetchresult= &reply('fetchuserfile:'.$path.$file,$docuhome);
    if ($fetchresult eq 'ok') {
        if ($fetchthumb) {
            my $thumbresult= &reply('fetchuserfile:'.$path.'tn-'.$file,$docuhome);
            if ($thumbresult ne 'ok') {
                &logthis('Failed to transfer '.$path.'tn-'.$file.' to host '.
                         $docuhome.': '.$thumbresult);
            }
        }
#
# Return the URL to it
        return '/uploaded/'.$path.$file;
    } else {
        &logthis('Failed to transfer '.$path.$file.' to host '.$docuhome.
		 ': '.$fetchresult);
        return '/adm/notfound.html';
    }
}

sub extract_embedded_items {
    my ($fullpath,$allfiles,$codebase,$content) = @_;
    my @state = ();
    my %javafiles = (
                      codebase => '',
                      code => '',
                      archive => ''
                    );
    my %mediafiles = (
                      src => '',
                      movie => '',
                     );
    my $p;
    if ($content) {
        $p = HTML::LCParser->new($content);
    } else {
        $p = HTML::LCParser->new($fullpath);
    }
    while (my $t=$p->get_token()) {
	if ($t->[0] eq 'S') {
	    my ($tagname, $attr) = ($t->[1],$t->[2]);
	    push(@state, $tagname);
            if (lc($tagname) eq 'allow') {
                &add_filetype($allfiles,$attr->{'src'},'src');
            }
	    if (lc($tagname) eq 'img') {
		&add_filetype($allfiles,$attr->{'src'},'src');
	    }
	    if (lc($tagname) eq 'a') {
		&add_filetype($allfiles,$attr->{'href'},'href');
	    }
            if (lc($tagname) eq 'script') {
                if ($attr->{'archive'} =~ /\.jar$/i) {
                    &add_filetype($allfiles,$attr->{'archive'},'archive');
                } else {
                    &add_filetype($allfiles,$attr->{'src'},'src');
                }
            }
            if (lc($tagname) eq 'link') {
                if (lc($attr->{'rel'}) eq 'stylesheet') { 
                    &add_filetype($allfiles,$attr->{'href'},'href');
                }
            }
	    if (lc($tagname) eq 'object' ||
		(lc($tagname) eq 'embed' && lc($state[-2]) ne 'object')) {
		foreach my $item (keys(%javafiles)) {
		    $javafiles{$item} = '';
		}
	    }
	    if (lc($state[-2]) eq 'object' && lc($tagname) eq 'param') {
		my $name = lc($attr->{'name'});
		foreach my $item (keys(%javafiles)) {
		    if ($name eq $item) {
			$javafiles{$item} = $attr->{'value'};
			last;
		    }
		}
		foreach my $item (keys(%mediafiles)) {
		    if ($name eq $item) {
			&add_filetype($allfiles, $attr->{'value'}, 'value');
			last;
		    }
		}
	    }
	    if (lc($tagname) eq 'embed' || lc($tagname) eq 'applet') {
		foreach my $item (keys(%javafiles)) {
		    if ($attr->{$item}) {
			$javafiles{$item} = $attr->{$item};
			last;
		    }
		}
		foreach my $item (keys(%mediafiles)) {
		    if ($attr->{$item}) {
			&add_filetype($allfiles,$attr->{$item},$item);
			last;
		    }
		}
	    }
	} elsif ($t->[0] eq 'E') {
	    my ($tagname) = ($t->[1]);
	    if ($javafiles{'codebase'} ne '') {
		$javafiles{'codebase'} .= '/';
	    }  
	    if (lc($tagname) eq 'applet' ||
		lc($tagname) eq 'object' ||
		(lc($tagname) eq 'embed' && lc($state[-2]) ne 'object')
		) {
		foreach my $item (keys(%javafiles)) {
		    if ($item ne 'codebase' && $javafiles{$item} ne '') {
			my $file=$javafiles{'codebase'}.$javafiles{$item};
			&add_filetype($allfiles,$file,$item);
		    }
		}
	    } 
	    pop @state;
	}
    }
    return 'ok';
}

sub add_filetype {
    my ($allfiles,$file,$type)=@_;
    if (exists($allfiles->{$file})) {
	unless (grep/^\Q$type\E$/, @{$allfiles->{$file}}) {
	    push(@{$allfiles->{$file}}, &escape($type));
	}
    } else {
	@{$allfiles->{$file}} = (&escape($type));
    }
}

sub removeuploadedurl {
    my ($url)=@_;	
    my (undef,undef,$udom,$uname,$fname)=split('/',$url,5);    
    return &removeuserfile($uname,$udom,$fname);
}

sub removeuserfile {
    my ($docuname,$docudom,$fname)=@_;
    my $home=&homeserver($docuname,$docudom);    
    my $result = &reply("removeuserfile:$docudom/$docuname/$fname",$home);
    if ($result eq 'ok') {	
        if (($fname !~ /\.meta$/) && (&is_portfolio_file($fname))) {
            my $metafile = $fname.'.meta';
            my $metaresult = &removeuserfile($docuname,$docudom,$metafile); 
	    my $url = "/uploaded/$docudom/$docuname/$fname";
            my ($file,$group) = (&parse_portfolio_url($url))[3,4];	   
            my $sqlresult = 
                &update_portfolio_table($docuname,$docudom,$file,
                                        'portfolio_metadata',$group,
                                        'delete');
        }
    }
    return $result;
}

sub mkdiruserfile {
    my ($docuname,$docudom,$dir)=@_;
    my $home=&homeserver($docuname,$docudom);
    return &reply("mkdiruserfile:".&escape("$docudom/$docuname/$dir"),$home);
}

sub renameuserfile {
    my ($docuname,$docudom,$old,$new)=@_;
    my $home=&homeserver($docuname,$docudom);
    my $result = &reply("renameuserfile:$docudom:$docuname:".
                        &escape("$old").':'.&escape("$new"),$home);
    if ($result eq 'ok') {
        if (($old !~ /\.meta$/) && (&is_portfolio_file($old))) {
            my $oldmeta = $old.'.meta';
            my $newmeta = $new.'.meta';
            my $metaresult = 
                &renameuserfile($docuname,$docudom,$oldmeta,$newmeta);
	    my $url = "/uploaded/$docudom/$docuname/$old";
            my ($file,$group) = (&parse_portfolio_url($url))[3,4];
            my $sqlresult = 
                &update_portfolio_table($docuname,$docudom,$file,
                                        'portfolio_metadata',$group,
                                        'delete');
        }
    }
    return $result;
}

# ------------------------------------------------------------------------- Log

sub log {
    my ($dom,$nam,$hom,$what)=@_;
    return critical("log:$dom:$nam:$what",$hom);
}

# ------------------------------------------------------------------ Course Log
#
# This routine flushes several buffers of non-mission-critical nature
#

sub flushcourselogs {
    &logthis('Flushing log buffers');
#
# course logs
# This is a log of all transactions in a course, which can be used
# for data mining purposes
#
# It also collects the courseid database, which lists last transaction
# times and course titles for all courseids
#
    my %courseidbuffer=();
    foreach my $crsid (keys(%courselogs)) {
        if (&reply('log:'.$coursedombuf{$crsid}.':'.$coursenumbuf{$crsid}.':'.
		          &escape($courselogs{$crsid}),
		          $coursehombuf{$crsid}) eq 'ok') {
	    delete $courselogs{$crsid};
        } else {
            &logthis('Failed to flush log buffer for '.$crsid);
            if (length($courselogs{$crsid})>40000) {
               &logthis("<font color=\"blue\">WARNING: Buffer for ".$crsid.
                        " exceeded maximum size, deleting.</font>");
               delete $courselogs{$crsid};
            }
        }
        $courseidbuffer{$coursehombuf{$crsid}}{$crsid} = {
            'description' => $coursedescrbuf{$crsid},
            'inst_code'    => $courseinstcodebuf{$crsid},
            'type'        => $coursetypebuf{$crsid},
            'owner'       => $courseownerbuf{$crsid},
        };
    }
#
# Write course id database (reverse lookup) to homeserver of courses 
# Is used in pickcourse
#
    foreach my $crs_home (keys(%courseidbuffer)) {
        my $response = &courseidput(&host_domain($crs_home),
                                    $courseidbuffer{$crs_home},
                                    $crs_home,'timeonly');
    }
#
# File accesses
# Writes to the dynamic metadata of resources to get hit counts, etc.
#
    foreach my $entry (keys(%accesshash)) {
        if ($entry =~ /___count$/) {
            my ($dom,$name);
            ($dom,$name,undef)=
		($entry=~m{___($match_domain)/($match_name)/(.*)___count$});
            if (! defined($dom) || $dom eq '' || 
                ! defined($name) || $name eq '') {
                my $cid = $env{'request.course.id'};
                $dom  = $env{'request.'.$cid.'.domain'};
                $name = $env{'request.'.$cid.'.num'};
            }
            my $value = $accesshash{$entry};
            my (undef,$url,undef) = ($entry =~ /^(.*)___(.*)___count$/);
            my %temphash=($url => $value);
            my $result = &inc('nohist_accesscount',\%temphash,$dom,$name);
            if ($result eq 'ok') {
                delete $accesshash{$entry};
            } elsif ($result eq 'unknown_cmd') {
                # Target server has old code running on it.
                my %temphash=($entry => $value);
                if (&put('nohist_resevaldata',\%temphash,$dom,$name) eq 'ok') {
                    delete $accesshash{$entry};
                }
            }
        } else {
            my ($dom,$name) = ($entry=~m{___($match_domain)/($match_name)/(.*)___(\w+)$});
            my %temphash=($entry => $accesshash{$entry});
            if (&put('nohist_resevaldata',\%temphash,$dom,$name) eq 'ok') {
                delete $accesshash{$entry};
            }
        }
    }
#
# Roles
# Reverse lookup of user roles for course faculty/staff and co-authorship
#
    foreach my $entry (keys(%userrolehash)) {
        my ($role,$uname,$udom,$runame,$rudom,$rsec)=
	    split(/\:/,$entry);
        if (&Apache::lonnet::put('nohist_userroles',
             { $role.':'.$uname.':'.$udom.':'.$rsec => $userrolehash{$entry} },
                $rudom,$runame) eq 'ok') {
	    delete $userrolehash{$entry};
        }
    }
#
# Reverse lookup of domain roles (dc, ad, li, sc, au)
#
    my %domrolebuffer = ();
    foreach my $entry (keys(%domainrolehash)) {
        my ($role,$uname,$udom,$runame,$rudom,$rsec)=split(/:/,$entry);
        if ($domrolebuffer{$rudom}) {
            $domrolebuffer{$rudom}.='&'.&escape($entry).
                      '='.&escape($domainrolehash{$entry});
        } else {
            $domrolebuffer{$rudom}.=&escape($entry).
                      '='.&escape($domainrolehash{$entry});
        }
        delete $domainrolehash{$entry};
    }
    foreach my $dom (keys(%domrolebuffer)) {
	my %servers = &get_servers($dom,'library');
	foreach my $tryserver (keys(%servers)) {
	    unless (&reply('domroleput:'.$dom.':'.
			   $domrolebuffer{$dom},$tryserver) eq 'ok') {
		&logthis('Put of domain roles failed for '.$dom.' and  '.$tryserver);
	    }
        }
    }
    $dumpcount++;
}

sub courselog {
    my $what=shift;
    $what=time.':'.$what;
    unless ($env{'request.course.id'}) { return ''; }
    $coursedombuf{$env{'request.course.id'}}=
       $env{'course.'.$env{'request.course.id'}.'.domain'};
    $coursenumbuf{$env{'request.course.id'}}=
       $env{'course.'.$env{'request.course.id'}.'.num'};
    $coursehombuf{$env{'request.course.id'}}=
       $env{'course.'.$env{'request.course.id'}.'.home'};
    $coursedescrbuf{$env{'request.course.id'}}=
       $env{'course.'.$env{'request.course.id'}.'.description'};
    $courseinstcodebuf{$env{'request.course.id'}}=
       $env{'course.'.$env{'request.course.id'}.'.internal.coursecode'};
    $courseownerbuf{$env{'request.course.id'}}=
       $env{'course.'.$env{'request.course.id'}.'.internal.courseowner'};
    $coursetypebuf{$env{'request.course.id'}}=
       $env{'course.'.$env{'request.course.id'}.'.type'};
    if (defined $courselogs{$env{'request.course.id'}}) {
	$courselogs{$env{'request.course.id'}}.='&'.$what;
    } else {
	$courselogs{$env{'request.course.id'}}.=$what;
    }
    if (length($courselogs{$env{'request.course.id'}})>4048) {
	&flushcourselogs();
    }
}

sub courseacclog {
    my $fnsymb=shift;
    unless ($env{'request.course.id'}) { return ''; }
    my $what=$fnsymb.':'.$env{'user.name'}.':'.$env{'user.domain'};
    if ($fnsymb=~/(problem|exam|quiz|assess|survey|form|task|page)$/) {
        $what.=':POST';
        # FIXME: Probably ought to escape things....
	foreach my $key (keys(%env)) {
            if ($key=~/^form\.(.*)/) {
                my $formitem = $1;
                if ($formitem =~ /^HWFILE(?:SIZE|TOOBIG)/) {
                    $what.=':'.$formitem.'='.$env{$key};
                } elsif ($formitem !~ /^HWFILE(?:[^.]+)$/) {
                    $what.=':'.$formitem.'='.$env{$key};
                }
            }
        }
    } elsif ($fnsymb =~ m:^/adm/searchcat:) {
        # FIXME: We should not be depending on a form parameter that someone
        # editing lonsearchcat.pm might change in the future.
        if ($env{'form.phase'} eq 'course_search') {
            $what.= ':POST';
            # FIXME: Probably ought to escape things....
            foreach my $element ('courseexp','crsfulltext','crsrelated',
                                 'crsdiscuss') {
                $what.=':'.$element.'='.$env{'form.'.$element};
            }
        }
    }
    &courselog($what);
}

sub countacc {
    my $url=&declutter(shift);
    return if (! defined($url) || $url eq '');
    unless ($env{'request.course.id'}) { return ''; }
    $accesshash{$env{'request.course.id'}.'___'.$url.'___course'}=1;
    my $key=$$.$processmarker.'_'.$dumpcount.'___'.$url.'___count';
    $accesshash{$key}++;
}

sub linklog {
    my ($from,$to)=@_;
    $from=&declutter($from);
    $to=&declutter($to);
    $accesshash{$from.'___'.$to.'___comefrom'}=1;
    $accesshash{$to.'___'.$from.'___goto'}=1;
}
  
sub userrolelog {
    my ($trole,$username,$domain,$area,$tstart,$tend)=@_;
    if (($trole=~/^ca/) || ($trole=~/^aa/) ||
        ($trole=~/^in/) || ($trole=~/^cc/) ||
        ($trole=~/^ep/) || ($trole=~/^cr/) ||
        ($trole=~/^ta/) || ($trole=~/^co/)) {
       my (undef,$rudom,$runame,$rsec)=split(/\//,$area);
       $userrolehash
         {$trole.':'.$username.':'.$domain.':'.$runame.':'.$rudom.':'.$rsec}
                    =$tend.':'.$tstart;
    }
    if (($env{'request.role'} =~ /dc\./) &&
	(($trole=~/^au/) || ($trole=~/^in/) ||
	 ($trole=~/^cc/) || ($trole=~/^ep/) ||
	 ($trole=~/^cr/) || ($trole=~/^ta/) ||
         ($trole=~/^co/))) {
       $userrolehash
         {$trole.':'.$username.':'.$domain.':'.$env{'user.name'}.':'.$env{'user.domain'}.':'}
                    =$tend.':'.$tstart;
    }
    if (($trole=~/^dc/) || ($trole=~/^ad/) ||
        ($trole=~/^li/) || ($trole=~/^li/) ||
        ($trole=~/^au/) || ($trole=~/^dg/) ||
        ($trole=~/^sc/)) {
       my (undef,$rudom,$runame,$rsec)=split(/\//,$area);
       $domainrolehash
         {$trole.':'.$username.':'.$domain.':'.$runame.':'.$rudom.':'.$rsec}
                    = $tend.':'.$tstart;
    }
}

sub courserolelog {
    my ($trole,$username,$domain,$area,$tstart,$tend,$delflag,$selfenroll,$context)=@_;
    if (($trole eq 'cc') || ($trole eq 'in') ||
        ($trole eq 'ep') || ($trole eq 'ad') ||
        ($trole eq 'ta') || ($trole eq 'st') ||
        ($trole=~/^cr/) || ($trole eq 'gr') ||
        ($trole eq 'co')) {
        if ($area =~ m-^/($match_domain)/($match_courseid)/?([^/]*)-) {
            my $cdom = $1;
            my $cnum = $2;
            my $sec = $3;
            my $namespace = 'rolelog';
            my %storehash = (
                               role    => $trole,
                               start   => $tstart,
                               end     => $tend,
                               selfenroll => $selfenroll,
                               context    => $context,
                            );
            if ($trole eq 'gr') {
                $namespace = 'groupslog';
                $storehash{'group'} = $sec;
            } else {
                $storehash{'section'} = $sec;
            }
            &instructor_log($namespace,\%storehash,$delflag,$username,$domain,$cnum,$cdom);
            if (($trole ne 'st') || ($sec ne '')) {
                &devalidate_cache_new('getcourseroles',$cdom.'_'.$cnum);
            }
        }
    }
    return;
}

sub get_course_adv_roles {
    my ($cid,$codes) = @_;
    $cid=$env{'request.course.id'} unless (defined($cid));
    my %coursehash=&coursedescription($cid);
    my $crstype = &Apache::loncommon::course_type($cid);
    my %nothide=();
    foreach my $user (split(/\s*\,\s*/,$coursehash{'nothideprivileged'})) {
        if ($user !~ /:/) {
	    $nothide{join(':',split(/[\@]/,$user))}=1;
        } else {
            $nothide{$user}=1;
        }
    }
    my %returnhash=();
    my %dumphash=
            &dump('nohist_userroles',$coursehash{'domain'},$coursehash{'num'});
    my $now=time;
    my %privileged;
    foreach my $entry (keys(%dumphash)) {
	my ($tend,$tstart)=split(/\:/,$dumphash{$entry});
        if (($tstart) && ($tstart<0)) { next; }
        if (($tend) && ($tend<$now)) { next; }
        if (($tstart) && ($now<$tstart)) { next; }
        my ($role,$username,$domain,$section)=split(/\:/,$entry);
	if ($username eq '' || $domain eq '') { next; }
        unless (ref($privileged{$domain}) eq 'HASH') {
            my %dompersonnel =
                &Apache::lonnet::get_domain_roles($domain,['dc'],$now,$now);
            $privileged{$domain} = {};
            foreach my $server (keys(%dompersonnel)) {
                if (ref($dompersonnel{$server}) eq 'HASH') {
                    foreach my $user (keys(%{$dompersonnel{$server}})) {
                        my ($trole,$uname,$udom) = split(/:/,$user);
                        $privileged{$udom}{$uname} = 1;
                    }
                }
            }
        }
        if ((exists($privileged{$domain}{$username})) && 
            (!$nothide{$username.':'.$domain})) { next; }
	if ($role eq 'cr') { next; }
        if ($codes) {
            if ($section) { $role .= ':'.$section; }
            if ($returnhash{$role}) {
                $returnhash{$role}.=','.$username.':'.$domain;
            } else {
                $returnhash{$role}=$username.':'.$domain;
            }
        } else {
            my $key=&plaintext($role,$crstype);
            if ($section) { $key.=' ('.&Apache::lonlocal::mt('Section [_1]',$section).')'; }
            if ($returnhash{$key}) {
	        $returnhash{$key}.=','.$username.':'.$domain;
            } else {
                $returnhash{$key}=$username.':'.$domain;
            }
        }
    }
    return %returnhash;
}

sub get_my_roles {
    my ($uname,$udom,$context,$types,$roles,$roledoms,$withsec,$hidepriv)=@_;
    unless (defined($uname)) { $uname=$env{'user.name'}; }
    unless (defined($udom)) { $udom=$env{'user.domain'}; }
    my (%dumphash,%nothide);
    if ($context eq 'userroles') { 
        %dumphash = &dump('roles',$udom,$uname);
    } else {
        %dumphash=
            &dump('nohist_userroles',$udom,$uname);
        if ($hidepriv) {
            my %coursehash=&coursedescription($udom.'_'.$uname);
            foreach my $user (split(/\s*\,\s*/,$coursehash{'nothideprivileged'})) {
                if ($user !~ /:/) {
                    $nothide{join(':',split(/[\@]/,$user))} = 1;
                } else {
                    $nothide{$user} = 1;
                }
            }
        }
    }
    my %returnhash=();
    my $now=time;
    my %privileged;
    foreach my $entry (keys(%dumphash)) {
        my ($role,$tend,$tstart);
        if ($context eq 'userroles') {
	    ($role,$tend,$tstart)=split(/_/,$dumphash{$entry});
        } else {
            ($tend,$tstart)=split(/\:/,$dumphash{$entry});
        }
        if (($tstart) && ($tstart<0)) { next; }
        my $status = 'active';
        if (($tend) && ($tend<=$now)) {
            $status = 'previous';
        } 
        if (($tstart) && ($now<$tstart)) {
            $status = 'future';
        }
        if (ref($types) eq 'ARRAY') {
            if (!grep(/^\Q$status\E$/,@{$types})) {
                next;
            } 
        } else {
            if ($status ne 'active') {
                next;
            }
        }
        my ($rolecode,$username,$domain,$section,$area);
        if ($context eq 'userroles') {
            ($area,$rolecode) = split(/_/,$entry);
            (undef,$domain,$username,$section) = split(/\//,$area);
        } else {
            ($role,$username,$domain,$section) = split(/\:/,$entry);
        }
        if (ref($roledoms) eq 'ARRAY') {
            if (!grep(/^\Q$domain\E$/,@{$roledoms})) {
                next;
            }
        }
        if (ref($roles) eq 'ARRAY') {
            if (!grep(/^\Q$role\E$/,@{$roles})) {
                if ($role =~ /^cr\//) {
                    if (!grep(/^cr$/,@{$roles})) {
                        next;
                    }
                } else {
                    next;
                }
            }
        }
        if ($hidepriv) {
            if ($context eq 'userroles') {
                if ((&privileged($username,$domain)) &&
                    (!$nothide{$username.':'.$domain})) {
                    next;
                }
            } else {
                unless (ref($privileged{$domain}) eq 'HASH') {
                    my %dompersonnel =
                        &Apache::lonnet::get_domain_roles($domain,['dc'],$now,$now);
                    $privileged{$domain} = {};
                    if (keys(%dompersonnel)) {
                        foreach my $server (keys(%dompersonnel)) {
                            if (ref($dompersonnel{$server}) eq 'HASH') {
                                foreach my $user (keys(%{$dompersonnel{$server}})) {
                                    my ($trole,$uname,$udom) = split(/:/,$user);
                                    $privileged{$udom}{$uname} = $trole;
                                }
                            }
                        }
                    }
                }
                if (exists($privileged{$domain}{$username})) {
                    if (!$nothide{$username.':'.$domain}) {
                        next;
                    }
                }
            }
        }
        if ($withsec) {
            $returnhash{$username.':'.$domain.':'.$role.':'.$section} =
                $tstart.':'.$tend;
        } else {
            $returnhash{$username.':'.$domain.':'.$role}=$tstart.':'.$tend;
        }
    }
    return %returnhash;
}

# ----------------------------------------------------- Frontpage Announcements
#
#

sub postannounce {
    my ($server,$text)=@_;
    unless (&allowed('psa',&host_domain($server))) { return 'refused'; }
    unless ($text=~/\w/) { $text=''; }
    return &reply('setannounce:'.&escape($text),$server);
}

sub getannounce {

    if (open(my $fh,$perlvar{'lonDocRoot'}.'/announcement.txt')) {
	my $announcement='';
	while (my $line = <$fh>) { $announcement .= $line; }
	close($fh);
	if ($announcement=~/\w/) { 
	    return 
   '<table bgcolor="#FF5555" cellpadding="5" cellspacing="3">'.
   '<tr><td bgcolor="#FFFFFF"><tt>'.$announcement.'</tt></td></tr></table>'; 
	} else {
	    return '';
	}
    } else {
	return '';
    }
}

# ---------------------------------------------------------- Course ID routines
# Deal with domain's nohist_courseid.db files
#

sub courseidput {
    my ($domain,$storehash,$coursehome,$caller) = @_;
    return unless (ref($storehash) eq 'HASH');
    my $outcome;
    if ($caller eq 'timeonly') {
        my $cids = '';
        foreach my $item (keys(%$storehash)) {
            $cids.=&escape($item).'&';
        }
        $cids=~s/\&$//;
        $outcome = &reply('courseidputhash:'.$domain.':'.$caller.':'.$cids,
                          $coursehome);       
    } else {
        my $items = '';
        foreach my $item (keys(%$storehash)) {
            $items.= &escape($item).'='.
                     &freeze_escape($$storehash{$item}).'&';
        }
        $items=~s/\&$//;
        $outcome = &reply('courseidputhash:'.$domain.':'.$caller.':'.$items,
                          $coursehome);
    }
    if ($outcome eq 'unknown_cmd') {
        my $what;
        foreach my $cid (keys(%$storehash)) {
            $what .= &escape($cid).'=';
            foreach my $item ('description','inst_code','owner','type') {
                $what .= &escape($storehash->{$cid}{$item}).':';
            }
            $what =~ s/\:$/&/;
        }
        $what =~ s/\&$//;  
        return &reply('courseidput:'.$domain.':'.$what,$coursehome);
    } else {
        return $outcome;
    }
}

sub courseiddump {
    my ($domfilter,$descfilter,$sincefilter,$instcodefilter,$ownerfilter,
        $coursefilter,$hostidflag,$hostidref,$typefilter,$regexp_ok,
        $selfenrollonly,$catfilter,$showhidden,$caller,$cloner,$cc_clone,
        $cloneonly,$createdbefore,$createdafter,$creationcontext)=@_;
    my $as_hash = 1;
    my %returnhash;
    if (!$domfilter) { $domfilter=''; }
    my %libserv = &all_library();
    foreach my $tryserver (keys(%libserv)) {
        if ( (  $hostidflag == 1 
	        && grep(/^\Q$tryserver\E$/,@{$hostidref}) ) 
	     || (!defined($hostidflag)) ) {

	    if (($domfilter eq '') ||
		(&host_domain($tryserver) eq $domfilter)) {
                my $rep = 
                  &reply('courseiddump:'.&host_domain($tryserver).':'.
                         $sincefilter.':'.&escape($descfilter).':'.
                         &escape($instcodefilter).':'.&escape($ownerfilter).
                         ':'.&escape($coursefilter).':'.&escape($typefilter).
                         ':'.&escape($regexp_ok).':'.$as_hash.':'.
                         &escape($selfenrollonly).':'.&escape($catfilter).':'.
                         $showhidden.':'.$caller.':'.&escape($cloner).':'.
                         &escape($cc_clone).':'.$cloneonly.':'.
                         &escape($createdbefore).':'.&escape($createdafter).':'.
                         &escape($creationcontext),$tryserver);
                my @pairs=split(/\&/,$rep);
                foreach my $item (@pairs) {
                    my ($key,$value)=split(/\=/,$item,2);
                    $key = &unescape($key);
                    next if ($key =~ /^error: 2 /);
                    my $result = &thaw_unescape($value);
                    if (ref($result) eq 'HASH') {
                        $returnhash{$key}=$result;
                    } else {
                        my @responses = split(/:/,$value);
                        my @items = ('description','inst_code','owner','type');
                        for (my $i=0; $i<@responses; $i++) {
                            $returnhash{$key}{$items[$i]} = &unescape($responses[$i]);
                        }
                    }
                }
            }
        }
    }
    return %returnhash;
}

sub courselastaccess {
    my ($cdom,$cnum,$hostidref) = @_;
    my %returnhash;
    if ($cdom && $cnum) {
        my $chome = &homeserver($cnum,$cdom);
        if ($chome ne 'no_host') {
            my $rep = &reply('courselastaccess:'.$cdom.':'.$cnum,$chome);
            &extract_lastaccess(\%returnhash,$rep);
        }
    } else {
        if (!$cdom) { $cdom=''; }
        my %libserv = &all_library();
        foreach my $tryserver (keys(%libserv)) {
            if (ref($hostidref) eq 'ARRAY') {
                next unless (grep(/^\Q$tryserver\E$/,@{$hostidref}));
            } 
            if (($cdom eq '') || (&host_domain($tryserver) eq $cdom)) {
                my $rep = &reply('courselastaccess:'.&host_domain($tryserver).':',$tryserver);
                &extract_lastaccess(\%returnhash,$rep);
            }
        }
    }
    return %returnhash;
}

sub extract_lastaccess {
    my ($returnhash,$rep) = @_;
    if (ref($returnhash) eq 'HASH') {
        unless ($rep eq 'unknown_command' || $rep eq 'no_such_host' || 
                $rep eq 'con_lost' || $rep eq 'rejected' || $rep eq 'refused' ||
                 $rep eq '') {
            my @pairs=split(/\&/,$rep);
            foreach my $item (@pairs) {
                my ($key,$value)=split(/\=/,$item,2);
                $key = &unescape($key);
                next if ($key =~ /^error: 2 /);
                $returnhash->{$key} = &thaw_unescape($value);
            }
        }
    }
    return;
}

# ---------------------------------------------------------- DC e-mail

sub dcmailput {
    my ($domain,$msgid,$message,$server)=@_;
    my $status = &Apache::lonnet::critical(
       'dcmailput:'.$domain.':'.&escape($msgid).'='.
       &escape($message),$server);
    return $status;
}

sub dcmaildump {
    my ($dom,$startdate,$enddate,$senders) = @_;
    my %returnhash=();

    if (defined(&domain($dom,'primary'))) {
        my $cmd='dcmaildump:'.$dom.':'.&escape($startdate).':'.
                                                         &escape($enddate).':';
	my @esc_senders=map { &escape($_)} @$senders;
	$cmd.=&escape(join('&',@esc_senders));
	foreach my $line (split(/\&/,&reply($cmd,&domain($dom,'primary')))) {
            my ($key,$value) = split(/\=/,$line,2);
            if (($key) && ($value)) {
                $returnhash{&unescape($key)} = &unescape($value);
            }
        }
    }
    return %returnhash;
}
# ---------------------------------------------------------- Domain roles

sub get_domain_roles {
    my ($dom,$roles,$startdate,$enddate)=@_;
    if ((!defined($startdate)) || ($startdate eq '')) {
        $startdate = '.';
    }
    if ((!defined($enddate)) || ($enddate eq '')) {
        $enddate = '.';
    }
    my $rolelist;
    if (ref($roles) eq 'ARRAY') {
        $rolelist = join(':',@{$roles});
    }
    my %personnel = ();

    my %servers = &get_servers($dom,'library');
    foreach my $tryserver (keys(%servers)) {
	%{$personnel{$tryserver}}=();
	foreach my $line (split(/\&/,&reply('domrolesdump:'.$dom.':'.
					    &escape($startdate).':'.
					    &escape($enddate).':'.
					    &escape($rolelist), $tryserver))) {
	    my ($key,$value) = split(/\=/,$line,2);
	    if (($key) && ($value)) {
		$personnel{$tryserver}{&unescape($key)} = &unescape($value);
	    }
	}
    }
    return %personnel;
}

# ----------------------------------------------------------- Check out an item

sub get_first_access {
    my ($type,$argsymb)=@_;
    my ($symb,$courseid,$udom,$uname)=&whichuser();
    if ($argsymb) { $symb=$argsymb; }
    my ($map,$id,$res)=&decode_symb($symb);
    if ($type eq 'course') {
	$res='course';
    } elsif ($type eq 'map') {
	$res=&symbread($map);
    } else {
	$res=$symb;
    }
    my %times=&get('firstaccesstimes',["$courseid\0$res"],$udom,$uname);
    return $times{"$courseid\0$res"};
}

sub set_first_access {
    my ($type)=@_;
    my ($symb,$courseid,$udom,$uname)=&whichuser();
    my ($map,$id,$res)=&decode_symb($symb);
    if ($type eq 'course') {
	$res='course';
    } elsif ($type eq 'map') {
	$res=&symbread($map);
    } else {
	$res=$symb;
    }
    my $firstaccess=&get_first_access($type,$symb);
    if (!$firstaccess) {
	return &put('firstaccesstimes',{"$courseid\0$res"=>time},$udom,$uname);
    }
    return 'already_set';
}

sub checkout {
    my ($symb,$tuname,$tudom,$tcrsid)=@_;
    my $now=time;
    my $lonhost=$perlvar{'lonHostID'};
    my $infostr=&escape(
                 'CHECKOUTTOKEN&'.
                 $tuname.'&'.
                 $tudom.'&'.
                 $tcrsid.'&'.
                 $symb.'&'.
		 $now.'&'.$ENV{'REMOTE_ADDR'});
    my $token=&reply('tmpput:'.$infostr,$lonhost);
    if ($token=~/^error\:/) { 
        &logthis("<font color=\"blue\">WARNING: ".
                "Checkout tmpput failed ".$tudom.' - '.$tuname.' - '.$symb.
                 "</font>");
        return ''; 
    }

    $token=~s/^(\d+)\_.*\_(\d+)$/$1\*$2\*$lonhost/;
    $token=~tr/a-z/A-Z/;

    my %infohash=('resource.0.outtoken' => $token,
                  'resource.0.checkouttime' => $now,
                  'resource.0.outremote' => $ENV{'REMOTE_ADDR'});

    unless (&cstore(\%infohash,$symb,$tcrsid,$tudom,$tuname) eq 'ok') {
       return '';
    } else {
        &logthis("<font color=\"blue\">WARNING: ".
                "Checkout cstore failed ".$tudom.' - '.$tuname.' - '.$symb.
                 "</font>");
    }    

    if (&log($tudom,$tuname,&homeserver($tuname,$tudom),
                         &escape('Checkout '.$infostr.' - '.
                                                 $token)) ne 'ok') {
	return '';
    } else {
        &logthis("<font color=\"blue\">WARNING: ".
                "Checkout log failed ".$tudom.' - '.$tuname.' - '.$symb.
                 "</font>");
    }
    return $token;
}

# ------------------------------------------------------------ Check in an item

sub checkin {
    my $token=shift;
    my $now=time;
    my ($ta,$tb,$lonhost)=split(/\*/,$token);
    $lonhost=~tr/A-Z/a-z/;
    my $dtoken=$ta.'_'.&hostname($lonhost).'_'.$tb;
    $dtoken=~s/\W/\_/g;
    my ($dummy,$tuname,$tudom,$tcrsid,$symb,$chtim,$rmaddr)=
                 split(/\&/,&unescape(&reply('tmpget:'.$dtoken,$lonhost)));

    unless (($tuname) && ($tudom)) {
        &logthis('Check in '.$token.' ('.$dtoken.') failed');
        return '';
    }
    
    unless (&allowed('mgr',$tcrsid)) {
        &logthis('Check in '.$token.' ('.$dtoken.') unauthorized: '.
                 $env{'user.name'}.' - '.$env{'user.domain'});
        return '';
    }

    my %infohash=('resource.0.intoken' => $token,
                  'resource.0.checkintime' => $now,
                  'resource.0.inremote' => $ENV{'REMOTE_ADDR'});

    unless (&cstore(\%infohash,$symb,$tcrsid,$tudom,$tuname) eq 'ok') {
       return '';
    }    

    if (&log($tudom,$tuname,&homeserver($tuname,$tudom),
                         &escape('Checkin - '.$token)) ne 'ok') {
	return '';
    }

    return ($symb,$tuname,$tudom,$tcrsid);    
}

# --------------------------------------------- Set Expire Date for Spreadsheet

sub expirespread {
    my ($uname,$udom,$stype,$usymb)=@_;
    my $cid=$env{'request.course.id'}; 
    if ($cid) {
       my $now=time;
       my $key=$uname.':'.$udom.':'.$stype.':'.$usymb;
       return &reply('put:'.$env{'course.'.$cid.'.domain'}.':'.
                            $env{'course.'.$cid.'.num'}.
	        	    ':nohist_expirationdates:'.
                            &escape($key).'='.$now,
                            $env{'course.'.$cid.'.home'})
    }
    return 'ok';
}

# ----------------------------------------------------- Devalidate Spreadsheets

sub devalidate {
    my ($symb,$uname,$udom)=@_;
    my $cid=$env{'request.course.id'}; 
    if ($cid) {
        # delete the stored spreadsheets for
        # - the student level sheet of this user in course's homespace
        # - the assessment level sheet for this resource 
        #   for this user in user's homespace
	# - current conditional state info
	my $key=$uname.':'.$udom.':';
        my $status=
	    &del('nohist_calculatedsheets',
		 [$key.'studentcalc:'],
		 $env{'course.'.$cid.'.domain'},
		 $env{'course.'.$cid.'.num'})
		.' '.
	    &del('nohist_calculatedsheets_'.$cid,
		 [$key.'assesscalc:'.$symb],$udom,$uname);
        unless ($status eq 'ok ok') {
           &logthis('Could not devalidate spreadsheet '.
                    $uname.' at '.$udom.' for '.
		    $symb.': '.$status);
        }
	&delenv('user.state.'.$cid);
    }
}

sub get_scalar {
    my ($string,$end) = @_;
    my $value;
    if ($$string =~ s/^([^&]*?)($end)/$2/) {
	$value = $1;
    } elsif ($$string =~ s/^([^&]*?)&//) {
	$value = $1;
    }
    return &unescape($value);
}

sub array2str {
  my (@array) = @_;
  my $result=&arrayref2str(\@array);
  $result=~s/^__ARRAY_REF__//;
  $result=~s/__END_ARRAY_REF__$//;
  return $result;
}

sub arrayref2str {
  my ($arrayref) = @_;
  my $result='__ARRAY_REF__';
  foreach my $elem (@$arrayref) {
    if(ref($elem) eq 'ARRAY') {
      $result.=&arrayref2str($elem).'&';
    } elsif(ref($elem) eq 'HASH') {
      $result.=&hashref2str($elem).'&';
    } elsif(ref($elem)) {
      #print("Got a ref of ".(ref($elem))." skipping.");
    } else {
      $result.=&escape($elem).'&';
    }
  }
  $result=~s/\&$//;
  $result .= '__END_ARRAY_REF__';
  return $result;
}

sub hash2str {
  my (%hash) = @_;
  my $result=&hashref2str(\%hash);
  $result=~s/^__HASH_REF__//;
  $result=~s/__END_HASH_REF__$//;
  return $result;
}

sub hashref2str {
  my ($hashref)=@_;
  my $result='__HASH_REF__';
  foreach my $key (sort(keys(%$hashref))) {
    if (ref($key) eq 'ARRAY') {
      $result.=&arrayref2str($key).'=';
    } elsif (ref($key) eq 'HASH') {
      $result.=&hashref2str($key).'=';
    } elsif (ref($key)) {
      $result.='=';
      #print("Got a ref of ".(ref($key))." skipping.");
    } else {
	if ($key) {$result.=&escape($key).'=';} else { last; }
    }

    if(ref($hashref->{$key}) eq 'ARRAY') {
      $result.=&arrayref2str($hashref->{$key}).'&';
    } elsif(ref($hashref->{$key}) eq 'HASH') {
      $result.=&hashref2str($hashref->{$key}).'&';
    } elsif(ref($hashref->{$key})) {
       $result.='&';
      #print("Got a ref of ".(ref($hashref->{$key}))." skipping.");
    } else {
      $result.=&escape($hashref->{$key}).'&';
    }
  }
  $result=~s/\&$//;
  $result .= '__END_HASH_REF__';
  return $result;
}

sub str2hash {
    my ($string)=@_;
    my ($hash)=&str2hashref('__HASH_REF__'.$string.'__END_HASH_REF__');
    return %$hash;
}

sub str2hashref {
  my ($string) = @_;

  my %hash;

  if($string !~ /^__HASH_REF__/) {
      if (! ($string eq '' || !defined($string))) {
	  $hash{'error'}='Not hash reference';
      }
      return (\%hash, $string);
  }

  $string =~ s/^__HASH_REF__//;

  while($string !~ /^__END_HASH_REF__/) {
      #key
      my $key='';
      if($string =~ /^__HASH_REF__/) {
          ($key, $string)=&str2hashref($string);
          if(defined($key->{'error'})) {
              $hash{'error'}='Bad data';
              return (\%hash, $string);
          }
      } elsif($string =~ /^__ARRAY_REF__/) {
          ($key, $string)=&str2arrayref($string);
          if($key->[0] eq 'Array reference error') {
              $hash{'error'}='Bad data';
              return (\%hash, $string);
          }
      } else {
          $string =~ s/^(.*?)=//;
	  $key=&unescape($1);
      }
      $string =~ s/^=//;

      #value
      my $value='';
      if($string =~ /^__HASH_REF__/) {
          ($value, $string)=&str2hashref($string);
          if(defined($value->{'error'})) {
              $hash{'error'}='Bad data';
              return (\%hash, $string);
          }
      } elsif($string =~ /^__ARRAY_REF__/) {
          ($value, $string)=&str2arrayref($string);
          if($value->[0] eq 'Array reference error') {
              $hash{'error'}='Bad data';
              return (\%hash, $string);
          }
      } else {
	  $value=&get_scalar(\$string,'__END_HASH_REF__');
      }
      $string =~ s/^&//;

      $hash{$key}=$value;
  }

  $string =~ s/^__END_HASH_REF__//;

  return (\%hash, $string);
}

sub str2array {
    my ($string)=@_;
    my ($array)=&str2arrayref('__ARRAY_REF__'.$string.'__END_ARRAY_REF__');
    return @$array;
}

sub str2arrayref {
  my ($string) = @_;
  my @array;

  if($string !~ /^__ARRAY_REF__/) {
      if (! ($string eq '' || !defined($string))) {
	  $array[0]='Array reference error';
      }
      return (\@array, $string);
  }

  $string =~ s/^__ARRAY_REF__//;

  while($string !~ /^__END_ARRAY_REF__/) {
      my $value='';
      if($string =~ /^__HASH_REF__/) {
          ($value, $string)=&str2hashref($string);
          if(defined($value->{'error'})) {
              $array[0] ='Array reference error';
              return (\@array, $string);
          }
      } elsif($string =~ /^__ARRAY_REF__/) {
          ($value, $string)=&str2arrayref($string);
          if($value->[0] eq 'Array reference error') {
              $array[0] ='Array reference error';
              return (\@array, $string);
          }
      } else {
	  $value=&get_scalar(\$string,'__END_ARRAY_REF__');
      }
      $string =~ s/^&//;

      push(@array, $value);
  }

  $string =~ s/^__END_ARRAY_REF__//;

  return (\@array, $string);
}

# -------------------------------------------------------------------Temp Store

sub tmpreset {
  my ($symb,$namespace,$domain,$stuname) = @_;
  if (!$symb) {
    $symb=&symbread();
    if (!$symb) { $symb= $env{'request.url'}; }
  }
  $symb=escape($symb);

  if (!$namespace) { $namespace=$env{'request.state'}; }
  $namespace=~s/\//\_/g;
  $namespace=~s/\W//g;

  if (!$domain) { $domain=$env{'user.domain'}; }
  if (!$stuname) { $stuname=$env{'user.name'}; }
  if ($domain eq 'public' && $stuname eq 'public') {
      $stuname=$ENV{'REMOTE_ADDR'};
  }
  my $path=$perlvar{'lonDaemons'}.'/tmp';
  my %hash;
  if (tie(%hash,'GDBM_File',
	  $path.'/tmpstore_'.$stuname.'_'.$domain.'_'.$namespace.'.db',
	  &GDBM_WRCREAT(),0640)) {
    foreach my $key (keys(%hash)) {
      if ($key=~ /:$symb/) {
	delete($hash{$key});
      }
    }
  }
}

sub tmpstore {
  my ($storehash,$symb,$namespace,$domain,$stuname) = @_;

  if (!$symb) {
    $symb=&symbread();
    if (!$symb) { $symb= $env{'request.url'}; }
  }
  $symb=escape($symb);

  if (!$namespace) {
    # I don't think we would ever want to store this for a course.
    # it seems this will only be used if we don't have a course.
    #$namespace=$env{'request.course.id'};
    #if (!$namespace) {
      $namespace=$env{'request.state'};
    #}
  }
  $namespace=~s/\//\_/g;
  $namespace=~s/\W//g;
  if (!$domain) { $domain=$env{'user.domain'}; }
  if (!$stuname) { $stuname=$env{'user.name'}; }
  if ($domain eq 'public' && $stuname eq 'public') {
      $stuname=$ENV{'REMOTE_ADDR'};
  }
  my $now=time;
  my %hash;
  my $path=$perlvar{'lonDaemons'}.'/tmp';
  if (tie(%hash,'GDBM_File',
	  $path.'/tmpstore_'.$stuname.'_'.$domain.'_'.$namespace.'.db',
	  &GDBM_WRCREAT(),0640)) {
    $hash{"version:$symb"}++;
    my $version=$hash{"version:$symb"};
    my $allkeys=''; 
    foreach my $key (keys(%$storehash)) {
      $allkeys.=$key.':';
      $hash{"$version:$symb:$key"}=&freeze_escape($$storehash{$key});
    }
    $hash{"$version:$symb:timestamp"}=$now;
    $allkeys.='timestamp';
    $hash{"$version:keys:$symb"}=$allkeys;
    if (untie(%hash)) {
      return 'ok';
    } else {
      return "error:$!";
    }
  } else {
    return "error:$!";
  }
}

# -----------------------------------------------------------------Temp Restore

sub tmprestore {
  my ($symb,$namespace,$domain,$stuname) = @_;

  if (!$symb) {
    $symb=&symbread();
    if (!$symb) { $symb= $env{'request.url'}; }
  }
  $symb=escape($symb);

  if (!$namespace) { $namespace=$env{'request.state'}; }

  if (!$domain) { $domain=$env{'user.domain'}; }
  if (!$stuname) { $stuname=$env{'user.name'}; }
  if ($domain eq 'public' && $stuname eq 'public') {
      $stuname=$ENV{'REMOTE_ADDR'};
  }
  my %returnhash;
  $namespace=~s/\//\_/g;
  $namespace=~s/\W//g;
  my %hash;
  my $path=$perlvar{'lonDaemons'}.'/tmp';
  if (tie(%hash,'GDBM_File',
	  $path.'/tmpstore_'.$stuname.'_'.$domain.'_'.$namespace.'.db',
	  &GDBM_READER(),0640)) {
    my $version=$hash{"version:$symb"};
    $returnhash{'version'}=$version;
    my $scope;
    for ($scope=1;$scope<=$version;$scope++) {
      my $vkeys=$hash{"$scope:keys:$symb"};
      my @keys=split(/:/,$vkeys);
      my $key;
      $returnhash{"$scope:keys"}=$vkeys;
      foreach $key (@keys) {
	$returnhash{"$scope:$key"}=&thaw_unescape($hash{"$scope:$symb:$key"});
	$returnhash{"$key"}=&thaw_unescape($hash{"$scope:$symb:$key"});
      }
    }
    if (!(untie(%hash))) {
      return "error:$!";
    }
  } else {
    return "error:$!";
  }
  return %returnhash;
}

# ----------------------------------------------------------------------- Store

sub store {
    my ($storehash,$symb,$namespace,$domain,$stuname) = @_;
    my $home='';

    if ($stuname) { $home=&homeserver($stuname,$domain); }

    $symb=&symbclean($symb);
    if (!$symb) { unless ($symb=&symbread()) { return ''; } }

    if (!$domain) { $domain=$env{'user.domain'}; }
    if (!$stuname) { $stuname=$env{'user.name'}; }

    &devalidate($symb,$stuname,$domain);

    $symb=escape($symb);
    if (!$namespace) { 
       unless ($namespace=$env{'request.course.id'}) { 
          return ''; 
       } 
    }
    if (!$home) { $home=$env{'user.home'}; }

    $$storehash{'ip'}=$ENV{'REMOTE_ADDR'};
    $$storehash{'host'}=$perlvar{'lonHostID'};

    my $namevalue='';
    foreach my $key (keys(%$storehash)) {
        $namevalue.=&escape($key).'='.&freeze_escape($$storehash{$key}).'&';
    }
    $namevalue=~s/\&$//;
    &courselog($symb.':'.$stuname.':'.$domain.':STORE:'.$namevalue);
    return reply("store:$domain:$stuname:$namespace:$symb:$namevalue","$home");
}

# -------------------------------------------------------------- Critical Store

sub cstore {
    my ($storehash,$symb,$namespace,$domain,$stuname) = @_;
    my $home='';

    if ($stuname) { $home=&homeserver($stuname,$domain); }

    $symb=&symbclean($symb);
    if (!$symb) { unless ($symb=&symbread()) { return ''; } }

    if (!$domain) { $domain=$env{'user.domain'}; }
    if (!$stuname) { $stuname=$env{'user.name'}; }

    &devalidate($symb,$stuname,$domain);

    $symb=escape($symb);
    if (!$namespace) { 
       unless ($namespace=$env{'request.course.id'}) { 
          return ''; 
       } 
    }
    if (!$home) { $home=$env{'user.home'}; }

    $$storehash{'ip'}=$ENV{'REMOTE_ADDR'};
    $$storehash{'host'}=$perlvar{'lonHostID'};

    my $namevalue='';
    foreach my $key (keys(%$storehash)) {
        $namevalue.=&escape($key).'='.&freeze_escape($$storehash{$key}).'&';
    }
    $namevalue=~s/\&$//;
    &courselog($symb.':'.$stuname.':'.$domain.':CSTORE:'.$namevalue);
    return critical
                ("store:$domain:$stuname:$namespace:$symb:$namevalue","$home");
}

# --------------------------------------------------------------------- Restore

sub restore {
    my ($symb,$namespace,$domain,$stuname) = @_;
    my $home='';

    if ($stuname) { $home=&homeserver($stuname,$domain); }

    if (!$symb) {
      unless ($symb=escape(&symbread())) { return ''; }
    } else {
      $symb=&escape(&symbclean($symb));
    }
    if (!$namespace) { 
       unless ($namespace=$env{'request.course.id'}) { 
          return ''; 
       } 
    }
    if (!$domain) { $domain=$env{'user.domain'}; }
    if (!$stuname) { $stuname=$env{'user.name'}; }
    if (!$home) { $home=$env{'user.home'}; }
    my $answer=&reply("restore:$domain:$stuname:$namespace:$symb","$home");

    my %returnhash=();
    foreach my $line (split(/\&/,$answer)) {
	my ($name,$value)=split(/\=/,$line);
        $returnhash{&unescape($name)}=&thaw_unescape($value);
    }
    my $version;
    for ($version=1;$version<=$returnhash{'version'};$version++) {
       foreach my $item (split(/\:/,$returnhash{$version.':keys'})) {
          $returnhash{$item}=$returnhash{$version.':'.$item};
       }
    }
    return %returnhash;
}

# ---------------------------------------------------------- Course Description

sub coursedescription {
    my ($courseid,$args)=@_;
    $courseid=~s/^\///;
    $courseid=~s/\_/\//g;
    my ($cdomain,$cnum)=split(/\//,$courseid);
    my $chome=&homeserver($cnum,$cdomain);
    my $normalid=$cdomain.'_'.$cnum;
    # need to always cache even if we get errors otherwise we keep 
    # trying and trying and trying to get the course description.
    my %envhash=();
    my %returnhash=();
    
    my $expiretime=600;
    if ($env{'request.course.id'} eq $normalid) {
	$expiretime=120;
    }

    my $prefix='course.'.$cdomain.'_'.$cnum.'.';
    if (!$args->{'freshen_cache'}
	&& ((time-$env{$prefix.'last_cache'}) < $expiretime) ) {
	foreach my $key (keys(%env)) {
	    next if ($key !~ /^\Q$prefix\E(.*)/);
	    my ($setting) = $1;
	    $returnhash{$setting} = $env{$key};
	}
	return %returnhash;
    }

    # get the data agin
    if (!$args->{'one_time'}) {
	$envhash{'course.'.$normalid.'.last_cache'}=time;
    }

    if ($chome ne 'no_host') {
       %returnhash=&dump('environment',$cdomain,$cnum);
       if (!exists($returnhash{'con_lost'})) {
           $returnhash{'home'}= $chome;
	   $returnhash{'domain'} = $cdomain;
	   $returnhash{'num'} = $cnum;
           if (!defined($returnhash{'type'})) {
               $returnhash{'type'} = 'Course';
           }
           while (my ($name,$value) = each %returnhash) {
               $envhash{'course.'.$normalid.'.'.$name}=$value;
           }
           $returnhash{'url'}=&clutter($returnhash{'url'});
           $returnhash{'fn'}=$perlvar{'lonDaemons'}.'/tmp/'.
	       $env{'user.name'}.'_'.$cdomain.'_'.$cnum;
           $envhash{'course.'.$normalid.'.home'}=$chome;
           $envhash{'course.'.$normalid.'.domain'}=$cdomain;
           $envhash{'course.'.$normalid.'.num'}=$cnum;
       }
    }
    if (!$args->{'one_time'}) {
	&appenv(\%envhash);
    }
    return %returnhash;
}

# -------------------------------------------------See if a user is privileged

sub privileged {
    my ($username,$domain)=@_;
    my $rolesdump=&reply("dump:$domain:$username:roles",
			&homeserver($username,$domain));
    if (($rolesdump eq 'con_lost') || ($rolesdump eq '') || 
        ($rolesdump =~ /^error:/)) {
        return 0;
    }
    my $now=time;
    if ($rolesdump ne '') {
        foreach my $entry (split(/&/,$rolesdump)) {
	    if ($entry!~/^rolesdef_/) {
		my ($area,$role)=split(/=/,$entry);
		$area=~s/\_\w\w$//;
		my ($trole,$tend,$tstart)=split(/_/,$role);
		if (($trole eq 'dc') || ($trole eq 'su')) {
		    my $active=1;
		    if ($tend) {
			if ($tend<$now) { $active=0; }
		    }
		    if ($tstart) {
			if ($tstart>$now) { $active=0; }
		    }
		    if ($active) { return 1; }
		}
	    }
	}
    }
    return 0;
}

# -------------------------------------------------------- Get user privileges

sub rolesinit {
    my ($domain,$username,$authhost)=@_;
    my $now=time;
    my %userroles = ('user.login.time' => $now);
    my $rolesdump=reply("dump:$domain:$username:roles",$authhost);
    if (($rolesdump eq 'con_lost') || ($rolesdump eq '') || 
        ($rolesdump =~ /^error:/)) { 
        return \%userroles;
    }
    my %allroles=();
    my %allgroups=();   
    my $group_privs;

    if ($rolesdump ne '') {
        foreach my $entry (split(/&/,$rolesdump)) {
	  if ($entry!~/^rolesdef_/) {
            my ($area,$role)=split(/=/,$entry);
	    $area=~s/\_\w\w$//;
            my ($trole,$tend,$tstart,$group_privs);
	    if ($role=~/^cr/) { 
		if ($role=~m|^(cr/$match_domain/$match_username/[a-zA-Z0-9]+)_(.*)$|) {
		    ($trole,my $trest)=($role=~m|^(cr/$match_domain/$match_username/[a-zA-Z0-9]+)_(.*)$|);
		    ($tend,$tstart)=split('_',$trest);
		} else {
		    $trole=$role;
		}
            } elsif ($role =~ m|^gr/|) {
                ($trole,$tend,$tstart) = split(/_/,$role);
                ($trole,$group_privs) = split(/\//,$trole);
                $group_privs = &unescape($group_privs);
	    } else {
		($trole,$tend,$tstart)=split(/_/,$role);
	    }
	    my %new_role = &set_arearole($trole,$area,$tstart,$tend,$domain,
					 $username);
	    @userroles{keys(%new_role)} = @new_role{keys(%new_role)};
            if (($tend!=0) && ($tend<$now)) { $trole=''; }
            if (($tstart!=0) && ($tstart>$now)) { $trole=''; }
            if (($area ne '') && ($trole ne '')) {
		my $spec=$trole.'.'.$area;
		my ($tdummy,$tdomain,$trest)=split(/\//,$area);
		if ($trole =~ /^cr\//) {
                    &custom_roleprivs(\%allroles,$trole,$tdomain,$trest,$spec,$area);
                } elsif ($trole eq 'gr') {
                    &group_roleprivs(\%allgroups,$area,$group_privs,$tend,$tstart);
		} else {
                    &standard_roleprivs(\%allroles,$trole,$tdomain,$spec,$trest,$area);
		}
            }
          }
        }
        my ($author,$adv) = &set_userprivs(\%userroles,\%allroles,\%allgroups);
        $userroles{'user.adv'}    = $adv;
	$userroles{'user.author'} = $author;
        $env{'user.adv'}=$adv;
    }
    return \%userroles;  
}

sub set_arearole {
    my ($trole,$area,$tstart,$tend,$domain,$username) = @_;
# log the associated role with the area
    &userrolelog($trole,$username,$domain,$area,$tstart,$tend);
    return ('user.role.'.$trole.'.'.$area => $tstart.'.'.$tend);
}

sub custom_roleprivs {
    my ($allroles,$trole,$tdomain,$trest,$spec,$area) = @_;
    my ($rdummy,$rdomain,$rauthor,$rrole)=split(/\//,$trole);
    my $homsvr=homeserver($rauthor,$rdomain);
    if (&hostname($homsvr) ne '') {
        my ($rdummy,$roledef)=
            &get('roles',["rolesdef_$rrole"],$rdomain,$rauthor);
        if (($rdummy ne 'con_lost') && ($roledef ne '')) {
            my ($syspriv,$dompriv,$coursepriv)=split(/\_/,$roledef);
            if (defined($syspriv)) {
                if ($trest =~ /^$match_community$/) {
                    $syspriv =~ s/bre\&S//; 
                }
                $$allroles{'cm./'}.=':'.$syspriv;
                $$allroles{$spec.'./'}.=':'.$syspriv;
            }
            if ($tdomain ne '') {
                if (defined($dompriv)) {
                    $$allroles{'cm./'.$tdomain.'/'}.=':'.$dompriv;
                    $$allroles{$spec.'./'.$tdomain.'/'}.=':'.$dompriv;
                }
                if (($trest ne '') && (defined($coursepriv))) {
                    $$allroles{'cm.'.$area}.=':'.$coursepriv;
                    $$allroles{$spec.'.'.$area}.=':'.$coursepriv;
                }
            }
        }
    }
}

sub group_roleprivs {
    my ($allgroups,$area,$group_privs,$tend,$tstart) = @_;
    my $access = 1;
    my $now = time;
    if (($tend!=0) && ($tend<$now)) { $access = 0; }
    if (($tstart!=0) && ($tstart>$now)) { $access=0; }
    if ($access) {
        my ($course,$group) = ($area =~ m|(/$match_domain/$match_courseid)/([^/]+)$|);
        $$allgroups{$course}{$group} .=':'.$group_privs;
    }
}

sub standard_roleprivs {
    my ($allroles,$trole,$tdomain,$spec,$trest,$area) = @_;
    if (defined($pr{$trole.':s'})) {
        $$allroles{'cm./'}.=':'.$pr{$trole.':s'};
        $$allroles{$spec.'./'}.=':'.$pr{$trole.':s'};
    }
    if ($tdomain ne '') {
        if (defined($pr{$trole.':d'})) {
            $$allroles{'cm./'.$tdomain.'/'}.=':'.$pr{$trole.':d'};
            $$allroles{$spec.'./'.$tdomain.'/'}.=':'.$pr{$trole.':d'};
        }
        if (($trest ne '') && (defined($pr{$trole.':c'}))) {
            $$allroles{'cm.'.$area}.=':'.$pr{$trole.':c'};
            $$allroles{$spec.'.'.$area}.=':'.$pr{$trole.':c'};
        }
    }
}

sub set_userprivs {
    my ($userroles,$allroles,$allgroups,$groups_roles) = @_; 
    my $author=0;
    my $adv=0;
    my %grouproles = ();
    if (keys(%{$allgroups}) > 0) {
        my @groupkeys;
        foreach my $role (keys(%{$allroles})) {
            push(@groupkeys,$role);
        }
        if (ref($groups_roles) eq 'HASH') {
            foreach my $key (keys(%{$groups_roles})) {
                unless (grep(/^\Q$key\E$/,@groupkeys)) {
                    push(@groupkeys,$key);
                }
            }
        }
        if (@groupkeys > 0) {
            foreach my $role (@groupkeys) {
                my ($trole,$area,$sec,$extendedarea);
                if ($role =~ m-^(\w+|cr/$match_domain/$match_username/\w+)\.(/$match_domain/$match_courseid)(/?\w*)\.-) {
                    $trole = $1;
                    $area = $2;
                    $sec = $3;
                    $extendedarea = $area.$sec;
                    if (exists($$allgroups{$area})) {
                        foreach my $group (keys(%{$$allgroups{$area}})) {
                            my $spec = $trole.'.'.$extendedarea;
                            $grouproles{$spec.'.'.$area.'/'.$group} = 
                                                  $$allgroups{$area}{$group};
                        }
                    }
                }
            }
        }
    }
    foreach my $group (keys(%grouproles)) {
        $$allroles{$group} = $grouproles{$group};
    }
    foreach my $role (keys(%{$allroles})) {
        my %thesepriv;
        if (($role=~/^au/) || ($role=~/^ca/) || ($role=~/^aa/)) { $author=1; }
        foreach my $item (split(/:/,$$allroles{$role})) {
            if ($item ne '') {
                my ($privilege,$restrictions)=split(/&/,$item);
                if ($restrictions eq '') {
                    $thesepriv{$privilege}='F';
                } elsif ($thesepriv{$privilege} ne 'F') {
                    $thesepriv{$privilege}.=$restrictions;
                }
                if ($thesepriv{'adv'} eq 'F') { $adv=1; }
            }
        }
        my $thesestr='';
        foreach my $priv (keys(%thesepriv)) {
	    $thesestr.=':'.$priv.'&'.$thesepriv{$priv};
	}
        $userroles->{'user.priv.'.$role} = $thesestr;
    }
    return ($author,$adv);
}

sub role_status {
    my ($rolekey,$then,$refresh,$now,$role,$where,$trolecode,$tstatus,$tstart,$tend) = @_;
    my @pwhere = ();
    if (exists($env{$rolekey}) && $env{$rolekey} ne '') {
        (undef,undef,$$role,@pwhere)=split(/\./,$rolekey);
        unless (!defined($$role) || $$role eq '') {
            $$where=join('.',@pwhere);
            $$trolecode=$$role.'.'.$$where;
            ($$tstart,$$tend)=split(/\./,$env{$rolekey});
            $$tstatus='is';
            if ($$tstart && $$tstart>$then) {
                $$tstatus='future';
                if ($$tstart<$now) {
                    if ($$tstart && $$tstart>$refresh) {
                        if (($$where ne '') && ($$role ne '')) {
                            my (%allroles,%allgroups,$group_privs,
                                %groups_roles,@rolecodes);
                            my %userroles = (
                                'user.role.'.$$role.'.'.$$where => $$tstart.'.'.$$tend
                            );
                            @rolecodes = ('cm');
                            my $spec=$$role.'.'.$$where;
                            my ($tdummy,$tdomain,$trest)=split(/\//,$$where);
                            if ($$role =~ /^cr\//) {
                                &custom_roleprivs(\%allroles,$$role,$tdomain,$trest,$spec,$$where);
                                push(@rolecodes,'cr');
                            } elsif ($$role eq 'gr') {
                                push(@rolecodes,$$role);
                                my %rolehash = &get('roles',[$$where.'_'.$$role],$env{'user.domain'},
                                                    $env{'user.name'});
                                my ($trole) = split('_',$rolehash{$$where.'_'.$$role},2);
                                (undef,my $group_privs) = split(/\//,$trole);
                                $group_privs = &unescape($group_privs);
                                &group_roleprivs(\%allgroups,$$where,$group_privs,$$tend,$$tstart);
                                my %course_roles = &get_my_roles($env{'user.name'},$env{'user.domain'},'userroles',['active'],['cc','co','in','ta','ep','ad','st','cr'],[$tdomain],1);
                                if (keys(%course_roles) > 0) {
                                    my ($tnum) = ($trest =~ /^($match_courseid)/);
                                    if ($tdomain ne '' && $tnum ne '') {
                                        foreach my $key (keys(%course_roles)) {
                                            if ($key =~ /^\Q$tnum\E:\Q$tdomain\E:([^:]+):?([^:]*)/) {
                                                my $crsrole = $1;
                                                my $crssec = $2;
                                                if ($crsrole =~ /^cr/) {
                                                    unless (grep(/^cr$/,@rolecodes)) {
                                                        push(@rolecodes,'cr');
                                                    }
                                                } else {
                                                    unless(grep(/^\Q$crsrole\E$/,@rolecodes)) {
                                                        push(@rolecodes,$crsrole);
                                                    }
                                                }
                                                my $rolekey = $crsrole.'./'.$tdomain.'/'.$tnum;
                                                if ($crssec ne '') {
                                                    $rolekey .= '/'.$crssec;
                                                }
                                                $rolekey .= './';
                                                $groups_roles{$rolekey} = \@rolecodes;
                                            }
                                        }
                                    }
                                }
                            } else {
                                push(@rolecodes,$$role);
                                &standard_roleprivs(\%allroles,$$role,$tdomain,$spec,$trest,$$where);
                            }
                            my ($author,$adv)= &set_userprivs(\%userroles,\%allroles,\%allgroups,\%groups_roles);
                            &appenv(\%userroles,\@rolecodes);
                            &log($env{'user.domain'},$env{'user.name'},$env{'user.home'},"Role ".$role);
                        }
                    }
                    $$tstatus = 'is';
                }
            }
            if ($$tend) {
                if ($$tend<$then) {
                    $$tstatus='expired';
                } elsif ($$tend<$now) {
                    $$tstatus='will_not';
                }
            }
        }
    }
}

sub check_adhoc_privs {
    my ($cdom,$cnum,$then,$refresh,$now,$checkrole) = @_;
    my $cckey = 'user.role.'.$checkrole.'./'.$cdom.'/'.$cnum;
    if ($env{$cckey}) {
        my ($role,$where,$trolecode,$tstart,$tend,$tremark,$tstatus,$tpstart,$tpend);
        &role_status($cckey,$then,$refresh,$now,\$role,\$where,\$trolecode,\$tstatus,\$tstart,\$tend);
        unless (($tstatus eq 'is') || ($tstatus eq 'will_not')) {
            &set_adhoc_privileges($cdom,$cnum,$checkrole);
        }
    } else {
        &set_adhoc_privileges($cdom,$cnum,$checkrole);
    }
}

sub set_adhoc_privileges {
# role can be cc or ca
    my ($dcdom,$pickedcourse,$role) = @_;
    my $area = '/'.$dcdom.'/'.$pickedcourse;
    my $spec = $role.'.'.$area;
    my %userroles = &set_arearole($role,$area,'','',$env{'user.domain'},
                                  $env{'user.name'});
    my %ccrole = ();
    &standard_roleprivs(\%ccrole,$role,$dcdom,$spec,$pickedcourse,$area);
    my ($author,$adv)= &set_userprivs(\%userroles,\%ccrole);
    &appenv(\%userroles,[$role,'cm']);
    &log($env{'user.domain'},$env{'user.name'},$env{'user.home'},"Role ".$role);
    &appenv( {'request.role'        => $spec,
              'request.role.domain' => $dcdom,
              'request.course.sec'  => ''
             }
           );
    my $tadv=0;
    if (&allowed('adv') eq 'F') { $tadv=1; }
    &appenv({'request.role.adv'    => $tadv});
}

# --------------------------------------------------------------- get interface

sub get {
   my ($namespace,$storearr,$udomain,$uname)=@_;
   my $items='';
   foreach my $item (@$storearr) {
       $items.=&escape($item).'&';
   }
   $items=~s/\&$//;
   if (!$udomain) { $udomain=$env{'user.domain'}; }
   if (!$uname) { $uname=$env{'user.name'}; }
   my $uhome=&homeserver($uname,$udomain);

   my $rep=&reply("get:$udomain:$uname:$namespace:$items",$uhome);
   my @pairs=split(/\&/,$rep);
   if ( $#pairs==0 && $pairs[0] =~ /^(con_lost|error|no_such_host)/i) {
     return @pairs;
   }
   my %returnhash=();
   my $i=0;
   foreach my $item (@$storearr) {
      $returnhash{$item}=&thaw_unescape($pairs[$i]);
      $i++;
   }
   return %returnhash;
}

# --------------------------------------------------------------- del interface

sub del {
   my ($namespace,$storearr,$udomain,$uname)=@_;
   my $items='';
   foreach my $item (@$storearr) {
       $items.=&escape($item).'&';
   }

   $items=~s/\&$//;
   if (!$udomain) { $udomain=$env{'user.domain'}; }
   if (!$uname) { $uname=$env{'user.name'}; }
   my $uhome=&homeserver($uname,$udomain);
   return &reply("del:$udomain:$uname:$namespace:$items",$uhome);
}

# -------------------------------------------------------------- dump interface

sub dump {
    my ($namespace,$udomain,$uname,$regexp,$range)=@_;
    if (!$udomain) { $udomain=$env{'user.domain'}; }
    if (!$uname) { $uname=$env{'user.name'}; }
    my $uhome=&homeserver($uname,$udomain);
    if ($regexp) {
	$regexp=&escape($regexp);
    } else {
	$regexp='.';
    }
    my $rep=&reply("dump:$udomain:$uname:$namespace:$regexp:$range",$uhome);
    my @pairs=split(/\&/,$rep);
    my %returnhash=();
    foreach my $item (@pairs) {
	my ($key,$value)=split(/=/,$item,2);
	$key = &unescape($key);
	next if ($key =~ /^error: 2 /);
	$returnhash{$key}=&thaw_unescape($value);
    }
    return %returnhash;
}

# --------------------------------------------------------- dumpstore interface

sub dumpstore {
   my ($namespace,$udomain,$uname,$regexp,$range)=@_;
   if (!$udomain) { $udomain=$env{'user.domain'}; }
   if (!$uname) { $uname=$env{'user.name'}; }
   my $uhome=&homeserver($uname,$udomain);
   if ($regexp) {
       $regexp=&escape($regexp);
   } else {
       $regexp='.';
   }
   my $rep=&reply("dump:$udomain:$uname:$namespace:$regexp:$range",$uhome);
   my @pairs=split(/\&/,$rep);
   my %returnhash=();
   foreach my $item (@pairs) {
       my ($key,$value)=split(/=/,$item,2);
       next if ($key =~ /^error: 2 /);
       $returnhash{$key}=&thaw_unescape($value);
   }
   return %returnhash;
}

# -------------------------------------------------------------- keys interface

sub getkeys {
   my ($namespace,$udomain,$uname)=@_;
   if (!$udomain) { $udomain=$env{'user.domain'}; }
   if (!$uname) { $uname=$env{'user.name'}; }
   my $uhome=&homeserver($uname,$udomain);
   my $rep=reply("keys:$udomain:$uname:$namespace",$uhome);
   my @keyarray=();
   foreach my $key (split(/\&/,$rep)) {
      next if ($key =~ /^error: 2 /);
      push(@keyarray,&unescape($key));
   }
   return @keyarray;
}

# --------------------------------------------------------------- currentdump
sub currentdump {
   my ($courseid,$sdom,$sname)=@_;
   $courseid = $env{'request.course.id'} if (! defined($courseid));
   $sdom     = $env{'user.domain'}       if (! defined($sdom));
   $sname    = $env{'user.name'}         if (! defined($sname));
   my $uhome = &homeserver($sname,$sdom);
   my $rep=reply('currentdump:'.$sdom.':'.$sname.':'.$courseid,$uhome);
   return if ($rep =~ /^(error:|no_such_host)/);
   #
   my %returnhash=();
   #
   if ($rep eq "unknown_cmd") { 
       # an old lond will not know currentdump
       # Do a dump and make it look like a currentdump
       my @tmp = &dumpstore($courseid,$sdom,$sname,'.');
       return if ($tmp[0] =~ /^(error:|no_such_host)/);
       my %hash = @tmp;
       @tmp=();
       %returnhash = %{&convert_dump_to_currentdump(\%hash)};
   } else {
       my @pairs=split(/\&/,$rep);
       foreach my $pair (@pairs) {
           my ($key,$value)=split(/=/,$pair,2);
           my ($symb,$param) = split(/:/,$key);
           $returnhash{&unescape($symb)}->{&unescape($param)} = 
                                                        &thaw_unescape($value);
       }
   }
   return %returnhash;
}

sub convert_dump_to_currentdump{
    my %hash = %{shift()};
    my %returnhash;
    # Code ripped from lond, essentially.  The only difference
    # here is the unescaping done by lonnet::dump().  Conceivably
    # we might run in to problems with parameter names =~ /^v\./
    while (my ($key,$value) = each(%hash)) {
        my ($v,$symb,$param) = split(/:/,$key);
	$symb  = &unescape($symb);
	$param = &unescape($param);
        next if ($v eq 'version' || $symb eq 'keys');
        next if (exists($returnhash{$symb}) &&
                 exists($returnhash{$symb}->{$param}) &&
                 $returnhash{$symb}->{'v.'.$param} > $v);
        $returnhash{$symb}->{$param}=$value;
        $returnhash{$symb}->{'v.'.$param}=$v;
    }
    #
    # Remove all of the keys in the hashes which keep track of
    # the version of the parameter.
    while (my ($symb,$param_hash) = each(%returnhash)) {
        # use a foreach because we are going to delete from the hash.
        foreach my $key (keys(%$param_hash)) {
            delete($param_hash->{$key}) if ($key =~ /^v\./);
        }
    }
    return \%returnhash;
}

# ------------------------------------------------------ critical inc interface

sub cinc {
    return &inc(@_,'critical');
}

# --------------------------------------------------------------- inc interface

sub inc {
    my ($namespace,$store,$udomain,$uname,$critical) = @_;
    if (!$udomain) { $udomain=$env{'user.domain'}; }
    if (!$uname) { $uname=$env{'user.name'}; }
    my $uhome=&homeserver($uname,$udomain);
    my $items='';
    if (! ref($store)) {
        # got a single value, so use that instead
        $items = &escape($store).'=&';
    } elsif (ref($store) eq 'SCALAR') {
        $items = &escape($$store).'=&';        
    } elsif (ref($store) eq 'ARRAY') {
        $items = join('=&',map {&escape($_);} @{$store});
    } elsif (ref($store) eq 'HASH') {
        while (my($key,$value) = each(%{$store})) {
            $items.= &escape($key).'='.&escape($value).'&';
        }
    }
    $items=~s/\&$//;
    if ($critical) {
	return &critical("inc:$udomain:$uname:$namespace:$items",$uhome);
    } else {
	return &reply("inc:$udomain:$uname:$namespace:$items",$uhome);
    }
}

# --------------------------------------------------------------- put interface

sub put {
   my ($namespace,$storehash,$udomain,$uname)=@_;
   if (!$udomain) { $udomain=$env{'user.domain'}; }
   if (!$uname) { $uname=$env{'user.name'}; }
   my $uhome=&homeserver($uname,$udomain);
   my $items='';
   foreach my $item (keys(%$storehash)) {
       $items.=&escape($item).'='.&freeze_escape($$storehash{$item}).'&';
   }
   $items=~s/\&$//;
   return &reply("put:$udomain:$uname:$namespace:$items",$uhome);
}

# ------------------------------------------------------------ newput interface

sub newput {
   my ($namespace,$storehash,$udomain,$uname)=@_;
   if (!$udomain) { $udomain=$env{'user.domain'}; }
   if (!$uname) { $uname=$env{'user.name'}; }
   my $uhome=&homeserver($uname,$udomain);
   my $items='';
   foreach my $key (keys(%$storehash)) {
       $items.=&escape($key).'='.&freeze_escape($$storehash{$key}).'&';
   }
   $items=~s/\&$//;
   return &reply("newput:$udomain:$uname:$namespace:$items",$uhome);
}

# ---------------------------------------------------------  putstore interface

sub putstore {
   my ($namespace,$symb,$version,$storehash,$udomain,$uname)=@_;
   if (!$udomain) { $udomain=$env{'user.domain'}; }
   if (!$uname) { $uname=$env{'user.name'}; }
   my $uhome=&homeserver($uname,$udomain);
   my $items='';
   foreach my $key (keys(%$storehash)) {
       $items.= &escape($key).'='.&freeze_escape($storehash->{$key}).'&';
   }
   $items=~s/\&$//;
   my $esc_symb=&escape($symb);
   my $esc_v=&escape($version);
   my $reply =
       &reply("putstore:$udomain:$uname:$namespace:$esc_symb:$esc_v:$items",
	      $uhome);
   if ($reply eq 'unknown_cmd') {
       # gfall back to way things use to be done
       return &old_putstore($namespace,$symb,$version,$storehash,$udomain,
			    $uname);
   }
   return $reply;
}

sub old_putstore {
    my ($namespace,$symb,$version,$storehash,$udomain,$uname)=@_;
    if (!$udomain) { $udomain=$env{'user.domain'}; }
    if (!$uname) { $uname=$env{'user.name'}; }
    my $uhome=&homeserver($uname,$udomain);
    my %newstorehash;
    foreach my $item (keys(%$storehash)) {
	my $key = $version.':'.&escape($symb).':'.$item;
	$newstorehash{$key} = $storehash->{$item};
    }
    my $items='';
    my %allitems = ();
    foreach my $item (keys(%newstorehash)) {
	if ($item =~ m/^([^\:]+):([^\:]+):([^\:]+)$/) {
	    my $key = $1.':keys:'.$2;
	    $allitems{$key} .= $3.':';
	}
	$items.=$item.'='.&freeze_escape($newstorehash{$item}).'&';
    }
    foreach my $item (keys(%allitems)) {
	$allitems{$item} =~ s/\:$//;
	$items.= $item.'='.$allitems{$item}.'&';
    }
    $items=~s/\&$//;
    return &reply("put:$udomain:$uname:$namespace:$items",$uhome);
}

# ------------------------------------------------------ critical put interface

sub cput {
   my ($namespace,$storehash,$udomain,$uname)=@_;
   if (!$udomain) { $udomain=$env{'user.domain'}; }
   if (!$uname) { $uname=$env{'user.name'}; }
   my $uhome=&homeserver($uname,$udomain);
   my $items='';
   foreach my $item (keys(%$storehash)) {
       $items.=&escape($item).'='.&freeze_escape($$storehash{$item}).'&';
   }
   $items=~s/\&$//;
   return &critical("put:$udomain:$uname:$namespace:$items",$uhome);
}

# -------------------------------------------------------------- eget interface

sub eget {
   my ($namespace,$storearr,$udomain,$uname)=@_;
   my $items='';
   foreach my $item (@$storearr) {
       $items.=&escape($item).'&';
   }
   $items=~s/\&$//;
   if (!$udomain) { $udomain=$env{'user.domain'}; }
   if (!$uname) { $uname=$env{'user.name'}; }
   my $uhome=&homeserver($uname,$udomain);
   my $rep=&reply("eget:$udomain:$uname:$namespace:$items",$uhome);
   my @pairs=split(/\&/,$rep);
   my %returnhash=();
   my $i=0;
   foreach my $item (@$storearr) {
      $returnhash{$item}=&thaw_unescape($pairs[$i]);
      $i++;
   }
   return %returnhash;
}

# ------------------------------------------------------------ tmpput interface
sub tmpput {
    my ($storehash,$server,$context)=@_;
    my $items='';
    foreach my $item (keys(%$storehash)) {
	$items.=&escape($item).'='.&freeze_escape($$storehash{$item}).'&';
    }
    $items=~s/\&$//;
    if (defined($context)) {
        $items .= ':'.&escape($context);
    }
    return &reply("tmpput:$items",$server);
}

# ------------------------------------------------------------ tmpget interface
sub tmpget {
    my ($token,$server)=@_;
    if (!defined($server)) { $server = $perlvar{'lonHostID'}; }
    my $rep=&reply("tmpget:$token",$server);
    my %returnhash;
    foreach my $item (split(/\&/,$rep)) {
	my ($key,$value)=split(/=/,$item);
        next if ($key =~ /^error: 2 /);
	$returnhash{&unescape($key)}=&thaw_unescape($value);
    }
    return %returnhash;
}

# ------------------------------------------------------------ tmpget interface
sub tmpdel {
    my ($token,$server)=@_;
    if (!defined($server)) { $server = $perlvar{'lonHostID'}; }
    return &reply("tmpdel:$token",$server);
}

# -------------------------------------------------- portfolio access checking

sub portfolio_access {
    my ($requrl) = @_;
    my (undef,$udom,$unum,$file_name,$group) = &parse_portfolio_url($requrl);
    my $result = &get_portfolio_access($udom,$unum,$file_name,$group);
    if ($result) {
        my %setters;
        if ($env{'user.name'} eq 'public' && $env{'user.domain'} eq 'public') {
            my ($startblock,$endblock) =
                &Apache::loncommon::blockcheck(\%setters,'port',$unum,$udom);
            if ($startblock && $endblock) {
                return 'B';
            }
        } else {
            my ($startblock,$endblock) =
                &Apache::loncommon::blockcheck(\%setters,'port');
            if ($startblock && $endblock) {
                return 'B';
            }
        }
    }
    if ($result eq 'ok') {
       return 'F';
    } elsif ($result =~ /^[^:]+:guest_/) {
       return 'A';
    }
    return '';
}

sub get_portfolio_access {
    my ($udom,$unum,$file_name,$group,$access_hash) = @_;

    if (!ref($access_hash)) {
	my $current_perms = &get_portfile_permissions($udom,$unum);
	my %access_controls = &get_access_controls($current_perms,$group,
						   $file_name);
	$access_hash = $access_controls{$file_name};
    }

    my ($public,$guest,@domains,@users,@courses,@groups);
    my $now = time;
    if (ref($access_hash) eq 'HASH') {
        foreach my $key (keys(%{$access_hash})) {
            my ($num,$scope,$end,$start) = ($key =~ /^([^:]+):([a-z]+)_(\d*)_?(\d*)$/);
            if ($start > $now) {
                next;
            }
            if ($end && $end<$now) {
                next;
            }
            if ($scope eq 'public') {
                $public = $key;
                last;
            } elsif ($scope eq 'guest') {
                $guest = $key;
            } elsif ($scope eq 'domains') {
                push(@domains,$key);
            } elsif ($scope eq 'users') {
                push(@users,$key);
            } elsif ($scope eq 'course') {
                push(@courses,$key);
            } elsif ($scope eq 'group') {
                push(@groups,$key);
            }
        }
        if ($public) {
            return 'ok';
        }
        if ($env{'user.name'} eq 'public' && $env{'user.domain'} eq 'public') {
            if ($guest) {
                return $guest;
            }
        } else {
            if (@domains > 0) {
                foreach my $domkey (@domains) {
                    if (ref($access_hash->{$domkey}{'dom'}) eq 'ARRAY') {
                        if (grep(/^\Q$env{'user.domain'}\E$/,@{$access_hash->{$domkey}{'dom'}})) {
                            return 'ok';
                        }
                    }
                }
            }
            if (@users > 0) {
                foreach my $userkey (@users) {
                    if (ref($access_hash->{$userkey}{'users'}) eq 'ARRAY') {
                        foreach my $item (@{$access_hash->{$userkey}{'users'}}) {
                            if (ref($item) eq 'HASH') {
                                if (($item->{'uname'} eq $env{'user.name'}) &&
                                    ($item->{'udom'} eq $env{'user.domain'})) {
                                    return 'ok';
                                }
                            }
                        }
                    } 
                }
            }
            my %roleshash;
            my @courses_and_groups = @courses;
            push(@courses_and_groups,@groups); 
            if (@courses_and_groups > 0) {
                my (%allgroups,%allroles); 
                my ($start,$end,$role,$sec,$group);
                foreach my $envkey (%env) {
                    if ($envkey =~ m-^user\.role\.(gr|cc|co|in|ta|ep|ad|st)\./($match_domain)/($match_courseid)/?([^/]*)$-) {
                        my $cid = $2.'_'.$3; 
                        if ($1 eq 'gr') {
                            $group = $4;
                            $allgroups{$cid}{$group} = $env{$envkey};
                        } else {
                            if ($4 eq '') {
                                $sec = 'none';
                            } else {
                                $sec = $4;
                            }
                            $allroles{$cid}{$1}{$sec} = $env{$envkey};
                        }
                    } elsif ($envkey =~ m-^user\.role\./cr/($match_domain/$match_username/\w*)./($match_domain)/($match_courseid)/?([^/]*)$-) {
                        my $cid = $2.'_'.$3;
                        if ($4 eq '') {
                            $sec = 'none';
                        } else {
                            $sec = $4;
                        }
                        $allroles{$cid}{$1}{$sec} = $env{$envkey};
                    }
                }
                if (keys(%allroles) == 0) {
                    return;
                }
                foreach my $key (@courses_and_groups) {
                    my %content = %{$$access_hash{$key}};
                    my $cnum = $content{'number'};
                    my $cdom = $content{'domain'};
                    my $cid = $cdom.'_'.$cnum;
                    if (!exists($allroles{$cid})) {
                        next;
                    }    
                    foreach my $role_id (keys(%{$content{'roles'}})) {
                        my @sections = @{$content{'roles'}{$role_id}{'section'}};
                        my @groups = @{$content{'roles'}{$role_id}{'group'}};
                        my @status = @{$content{'roles'}{$role_id}{'access'}};
                        my @roles = @{$content{'roles'}{$role_id}{'role'}};
                        foreach my $role (keys(%{$allroles{$cid}})) {
                            if ((grep/^all$/,@roles) || (grep/^\Q$role\E$/,@roles)) {
                                foreach my $sec (keys(%{$allroles{$cid}{$role}})) {
                                    if (&course_group_datechecker($allroles{$cid}{$role}{$sec},$now,\@status) eq 'ok') {
                                        if (grep/^all$/,@sections) {
                                            return 'ok';
                                        } else {
                                            if (grep/^$sec$/,@sections) {
                                                return 'ok';
                                            }
                                        }
                                    }
                                }
                                if (keys(%{$allgroups{$cid}}) == 0) {
                                    if (grep/^none$/,@groups) {
                                        return 'ok';
                                    }
                                } else {
                                    if (grep/^all$/,@groups) {
                                        return 'ok';
                                    } 
                                    foreach my $group (keys(%{$allgroups{$cid}})) {
                                        if (grep/^$group$/,@groups) {
                                            return 'ok';
                                        }
                                    }
                                } 
                            }
                        }
                    }
                }
            }
            if ($guest) {
                return $guest;
            }
        }
    }
    return;
}

sub course_group_datechecker {
    my ($dates,$now,$status) = @_;
    my ($start,$end) = split(/\./,$dates);
    if (!$start && !$end) {
        return 'ok';
    }
    if (grep/^active$/,@{$status}) {
        if (((!$start) || ($start && $start <= $now)) && ((!$end) || ($end && $end >= $now))) {
            return 'ok';
        }
    }
    if (grep/^previous$/,@{$status}) {
        if ($end > $now ) {
            return 'ok';
        }
    }
    if (grep/^future$/,@{$status}) {
        if ($start > $now) {
            return 'ok';
        }
    }
    return; 
}

sub parse_portfolio_url {
    my ($url) = @_;

    my ($type,$udom,$unum,$group,$file_name);
    
    if ($url =~  m-^/*(?:uploaded|editupload)/($match_domain)/($match_username)/portfolio(/.+)$-) {
	$type = 1;
        $udom = $1;
        $unum = $2;
        $file_name = $3;
    } elsif ($url =~ m-^/*(?:uploaded|editupload)/($match_domain)/($match_courseid)/groups/([^/]+)/portfolio/(.+)$-) {
	$type = 2;
        $udom = $1;
        $unum = $2;
        $group = $3;
        $file_name = $3.'/'.$4;
    }
    if (wantarray) {
	return ($type,$udom,$unum,$file_name,$group);
    }
    return $type;
}

sub is_portfolio_url {
    my ($url) = @_;
    return scalar(&parse_portfolio_url($url));
}

sub is_portfolio_file {
    my ($file) = @_;
    if (($file =~ /^portfolio/) || ($file =~ /^groups\/\w+\/portfolio/)) {
        return 1;
    }
    return;
}

sub usertools_access {
    my ($uname,$udom,$tool,$action,$context) = @_;
    my ($access,%tools);
    if ($context eq '') {
        $context = 'tools';
    }
    if ($context eq 'requestcourses') {
        %tools = (
                      official   => 1,
                      unofficial => 1,
                      community  => 1,
                 );
    } else {
        %tools = (
                      aboutme   => 1,
                      blog      => 1,
                      portfolio => 1,
                 );
    }
    return if (!defined($tools{$tool}));

    if ((!defined($udom)) || (!defined($uname))) {
        $udom = $env{'user.domain'};
        $uname = $env{'user.name'};
    }

    if (($udom eq $env{'user.domain'}) && ($uname eq $env{'user.name'})) {
        if ($action ne 'reload') {
            if ($context eq 'requestcourses') {
                return $env{'environment.canrequest.'.$tool};
            } else {
                return $env{'environment.availabletools.'.$tool};
            }
        }
    }

    my ($toolstatus,$inststatus);

    if (($udom eq $env{'user.domain'}) && ($uname eq $env{'user.name'}) &&
         ($action ne 'reload')) {
        $toolstatus = $env{'environment.'.$context.'.'.$tool};
        $inststatus = $env{'environment.inststatus'};
    } else {
        my %userenv = &userenvironment($udom,$uname,$context.'.'.$tool,'inststatus');
        $toolstatus = $userenv{$context.'.'.$tool};
        $inststatus = $userenv{'inststatus'};
    }

    if ($toolstatus ne '') {
        if ($toolstatus) {
            $access = 1;
        } else {
            $access = 0;
        }
        return $access;
    }

    my $is_adv = &is_advanced_user($udom,$uname);
    my %domdef = &get_domain_defaults($udom);
    if (ref($domdef{$tool}) eq 'HASH') {
        if ($is_adv) {
            if ($domdef{$tool}{'_LC_adv'} ne '') {
                if ($domdef{$tool}{'_LC_adv'}) { 
                    $access = 1;
                } else {
                    $access = 0;
                }
                return $access;
            }
        }
        if ($inststatus ne '') {
            my ($hasaccess,$hasnoaccess);
            foreach my $affiliation (split(/:/,$inststatus)) {
                if ($domdef{$tool}{$affiliation} ne '') { 
                    if ($domdef{$tool}{$affiliation}) {
                        $hasaccess = 1;
                    } else {
                        $hasnoaccess = 1;
                    }
                }
            }
            if ($hasaccess || $hasnoaccess) {
                if ($hasaccess) {
                    $access = 1;
                } elsif ($hasnoaccess) {
                    $access = 0; 
                }
                return $access;
            }
        } else {
            if ($domdef{$tool}{'default'} ne '') {
                if ($domdef{$tool}{'default'}) {
                    $access = 1;
                } elsif ($domdef{$tool}{'default'} == 0) {
                    $access = 0;
                }
                return $access;
            }
        }
    } else {
        if ($context eq 'tools') {
            $access = 1;
        } else {
            $access = 0;
        }
        return $access;
    }
}

sub is_course_owner {
    my ($cdom,$cnum,$udom,$uname) = @_;
    if (($udom eq '') || ($uname eq '')) {
        $udom = $env{'user.domain'};
        $uname = $env{'user.name'};
    }
    unless (($udom eq '') || ($uname eq '')) {
        if (exists($env{'course.'.$cdom.'_'.$cnum.'.internal.courseowner'})) {
            if ($env{'course.'.$cdom.'_'.$cnum.'.internal.courseowner'} eq $uname.':'.$udom) {
                return 1;
            } else {
                my %courseinfo = &Apache::lonnet::coursedescription($cdom.'/'.$cnum);
                if ($courseinfo{'internal.courseowner'} eq $uname.':'.$udom) {
                    return 1;
                }
            }
        }
    }
    return;
}

sub is_advanced_user {
    my ($udom,$uname) = @_;
    my %roleshash = &get_my_roles($uname,$udom,'userroles',undef,undef,undef,1);
    my %allroles;
    my $is_adv;
    foreach my $role (keys(%roleshash)) {
        my ($trest,$tdomain,$trole,$sec) = split(/:/,$role);
        my $area = '/'.$tdomain.'/'.$trest;
        if ($sec ne '') {
            $area .= '/'.$sec;
        }
        if (($area ne '') && ($trole ne '')) {
            my $spec=$trole.'.'.$area;
            if ($trole =~ /^cr\//) {
                &custom_roleprivs(\%allroles,$trole,$tdomain,$trest,$spec,$area);
            } elsif ($trole ne 'gr') {
                &standard_roleprivs(\%allroles,$trole,$tdomain,$spec,$trest,$area);
            }
        }
    }
    foreach my $role (keys(%allroles)) {
        last if ($is_adv);
        foreach my $item (split(/:/,$allroles{$role})) {
            if ($item ne '') {
                my ($privilege,$restrictions)=split(/&/,$item);
                if ($privilege eq 'adv') {
                    $is_adv = 1;
                    last;
                }
            }
        }
    }
    return $is_adv;
}

sub check_can_request {
    my ($dom,$can_request,$request_domains) = @_;
    my $canreq = 0;
    my ($types,$typename) = &Apache::loncommon::course_types();
    my @options = ('approval','validate','autolimit');
    my $optregex = join('|',@options);
    if ((ref($can_request) eq 'HASH') && (ref($types) eq 'ARRAY')) {
        foreach my $type (@{$types}) {
            if (&usertools_access($env{'user.name'},
                                  $env{'user.domain'},
                                  $type,undef,'requestcourses')) {
                $canreq ++;
                if (ref($request_domains) eq 'HASH') {
                    push(@{$request_domains->{$type}},$env{'user.domain'});
                }
                if ($dom eq $env{'user.domain'}) {
                    $can_request->{$type} = 1;
                }
            }
            if ($env{'environment.reqcrsotherdom.'.$type} ne '') {
                my @curr = split(',',$env{'environment.reqcrsotherdom.'.$type});
                if (@curr > 0) {
                    foreach my $item (@curr) {
                        if (ref($request_domains) eq 'HASH') {
                            my ($otherdom) = ($item =~ /^($match_domain):($optregex)(=?\d*)$/);
                            if ($otherdom ne '') {
                                if (ref($request_domains->{$type}) eq 'ARRAY') {
                                    unless (grep(/^\Q$otherdom\E$/,@{$request_domains->{$type}})) {
                                        push(@{$request_domains->{$type}},$otherdom);
                                    }
                                } else {
                                    push(@{$request_domains->{$type}},$otherdom);
                                }
                            }
                        }
                    }
                    unless($dom eq $env{'user.domain'}) {
                        $canreq ++;
                        if (grep(/^\Q$dom\E:($optregex)(=?\d*)$/,@curr)) {
                            $can_request->{$type} = 1;
                        }
                    }
                }
            }
        }
    }
    return $canreq;
}

# ---------------------------------------------- Custom access rule evaluation

sub customaccess {
    my ($priv,$uri)=@_;
    my ($urole,$urealm)=split(/\./,$env{'request.role'},2);
    my (undef,$udom,$ucrs,$usec)=split(/\//,$urealm);
    $udom = &LONCAPA::clean_domain($udom);
    $ucrs = &LONCAPA::clean_username($ucrs);
    my $access=0;
    foreach my $right (split(/\s*\,\s*/,&metadata($uri,'rule_rights'))) {
	my ($effect,$realm,$role,$type)=split(/\:/,$right);
	if ($type eq 'user') {
	    foreach my $scope (split(/\s*\,\s*/,$realm)) {
		my ($tdom,$tuname)=split(m{/},$scope);
		if ($tdom) {
		    if ($tdom ne $env{'user.domain'}) { next; }
		}
		if ($tuname) {
		    if ($tuname ne $env{'user.name'}) { next; }
		}
		$access=($effect eq 'allow');
		last;
	    }
	} else {
	    if ($role) {
		if ($role ne $urole) { next; }
	    }
	    foreach my $scope (split(/\s*\,\s*/,$realm)) {
		my ($tdom,$tcrs,$tsec)=split(/\_/,$scope);
		if ($tdom) {
		    if ($tdom ne $udom) { next; }
		}
		if ($tcrs) {
		    if ($tcrs ne $ucrs) { next; }
		}
		if ($tsec) {
		    if ($tsec ne $usec) { next; }
		}
		$access=($effect eq 'allow');
		last;
	    }
	    if ($realm eq '' && $role eq '') {
		$access=($effect eq 'allow');
	    }
	}
    }
    return $access;
}

# ------------------------------------------------- Check for a user privilege

sub allowed {
    my ($priv,$uri,$symb,$role)=@_;
    my $ver_orguri=$uri;
    $uri=&deversion($uri);
    my $orguri=$uri;
    $uri=&declutter($uri);

    if ($priv eq 'evb') {
# Evade communication block restrictions for specified role in a course
        if ($env{'user.priv.'.$role} =~/evb\&([^\:]*)/) {
            return $1;
        } else {
            return;
        }
    }

    if (defined($env{'allowed.'.$priv})) { return $env{'allowed.'.$priv}; }
# Free bre access to adm and meta resources
    if (((($uri=~/^adm\//) && ($uri !~ m{/(?:smppg|bulletinboard)$})) 
	 || (($uri=~/\.meta$/) && ($uri!~m|^uploaded/|) )) 
	&& ($priv eq 'bre')) {
	return 'F';
    }

# Free bre access to user's own portfolio contents
    my ($space,$domain,$name,@dir)=split('/',$uri);
    if (($space=~/^(uploaded|editupload)$/) && ($env{'user.name'} eq $name) && 
	($env{'user.domain'} eq $domain) && ('portfolio' eq $dir[0])) {
        my %setters;
        my ($startblock,$endblock) = 
            &Apache::loncommon::blockcheck(\%setters,'port');
        if ($startblock && $endblock) {
            return 'B';
        } else {
            return 'F';
        }
    }

# bre access to group portfolio for rgf priv in group, or mdg or vcg in course.
    if (($space=~/^(uploaded|editupload)$/) && ($dir[0] eq 'groups') 
         && ($dir[2] eq 'portfolio') && ($priv eq 'bre')) {
        if (exists($env{'request.course.id'})) {
            my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
            my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
            if (($domain eq $cdom) && ($name eq $cnum)) {
                my $courseprivid=$env{'request.course.id'};
                $courseprivid=~s/\_/\//;
                if ($env{'user.priv.'.$env{'request.role'}.'./'.$courseprivid
                    .'/'.$dir[1]} =~/rgf\&([^\:]*)/) {
                    return $1; 
                } else {
                    if ($env{'request.course.sec'}) {
                        $courseprivid.='/'.$env{'request.course.sec'};
                    }
                    if ($env{'user.priv.'.$env{'request.role'}.'./'.
                        $courseprivid} =~/(mdg|vcg)\&([^\:]*)/) {
                        return $2;
                    }
                }
            }
        }
    }

# Free bre to public access

    if ($priv eq 'bre') {
        my $copyright=&metadata($uri,'copyright');
	if (($copyright eq 'public') && (!$env{'request.course.id'})) { 
           return 'F'; 
        }
        if ($copyright eq 'priv') {
            $uri=~/([^\/]+)\/([^\/]+)\//;
	    unless (($env{'user.name'} eq $2) && ($env{'user.domain'} eq $1)) {
		return '';
            }
        }
        if ($copyright eq 'domain') {
            $uri=~/([^\/]+)\/([^\/]+)\//;
	    unless (($env{'user.domain'} eq $1) ||
                 ($env{'course.'.$env{'request.course.id'}.'.domain'} eq $1)) {
		return '';
            }
        }
        if ($env{'request.role'}=~ /li\.\//) {
            # Library role, so allow browsing of resources in this domain.
            return 'F';
        }
        if ($copyright eq 'custom') {
	    unless (&customaccess($priv,$uri)) { return ''; }
        }
    }
    # Domain coordinator is trying to create a course
    if (($priv eq 'ccc') && ($env{'request.role'} =~ /^dc\./)) {
        # uri is the requested domain in this case.
        # comparison to 'request.role.domain' shows if the user has selected
        # a role of dc for the domain in question.
        return 'F' if ($uri eq $env{'request.role.domain'});
    }

    my $thisallowed='';
    my $statecond=0;
    my $courseprivid='';

    my $ownaccess;
    # Community Coordinator or Assistant Co-author browsing resource space.
    if (($priv eq 'bro') && ($env{'user.author'})) {
        if ($uri eq '') {
            $ownaccess = 1;
        } else {
            if (($env{'user.domain'} ne '') && ($env{'user.name'} ne '')) {
                my $udom = $env{'user.domain'};
                my $uname = $env{'user.name'};
                if ($uri =~ m{^\Q$udom\E/?$}) {
                    $ownaccess = 1;
                } elsif ($uri =~ m{^\Q$udom\E/\Q$uname\E/?}) {
                    unless ($uri =~ m{\.\./}) {
                        $ownaccess = 1;
                    }
                } elsif (($udom ne 'public') && ($uname ne 'public')) {
                    my $now = time;
                    if ($uri =~ m{^([^/]+)/?$}) {
                        my $adom = $1;
                        foreach my $key (keys(%env)) {
                            if ($key =~ m{^user\.role\.(ca|aa)/\Q$adom\E}) {
                                my ($start,$end) = split('.',$env{$key});
                                if (($now >= $start) && (!$end || $end < $now)) {
                                    $ownaccess = 1;
                                    last;
                                }
                            }
                        }
                    } elsif ($uri =~ m{^([^/]+)/([^/]+)/?}) {
                        my $adom = $1;
                        my $aname = $2;
                        foreach my $role ('ca','aa') { 
                            if ($env{"user.role.$role./$adom/$aname"}) {
                                my ($start,$end) =
                                    split('.',$env{"user.role.$role./$adom/$aname"});
                                if (($now >= $start) && (!$end || $end < $now)) {
                                    $ownaccess = 1;
                                    last;
                                }
                            }
                        }
                    }
                }
            }
        }
    }

# Course

    if ($env{'user.priv.'.$env{'request.role'}.'./'}=~/\Q$priv\E\&([^\:]*)/) {
        unless (($priv eq 'bro') && (!$ownaccess)) {
            $thisallowed.=$1;
        }
    }

# Domain

    if ($env{'user.priv.'.$env{'request.role'}.'./'.(split(/\//,$uri))[0].'/'}
       =~/\Q$priv\E\&([^\:]*)/) {
        unless (($priv eq 'bro') && (!$ownaccess)) {
            $thisallowed.=$1;
        }
    }

# Course: uri itself is a course
    my $courseuri=$uri;
    $courseuri=~s/\_(\d)/\/$1/;
    $courseuri=~s/^([^\/])/\/$1/;

    if ($env{'user.priv.'.$env{'request.role'}.'.'.$courseuri}
       =~/\Q$priv\E\&([^\:]*)/) {
        unless (($priv eq 'bro') && (!$ownaccess)) {
            $thisallowed.=$1;
        }
    }

# URI is an uploaded document for this course, default permissions don't matter
# not allowing 'edit' access (editupload) to uploaded course docs
    if (($priv eq 'bre') && ($uri=~m|^uploaded/|)) {
	$thisallowed='';
        my ($match)=&is_on_map($uri);
        if ($match) {
            if ($env{'user.priv.'.$env{'request.role'}.'./'}
                  =~/\Q$priv\E\&([^\:]*)/) {
                $thisallowed.=$1;
            }
        } else {
            my $refuri = $env{'httpref.'.$orguri} || $env{'httpref.'.$ver_orguri};
            if ($refuri) {
                if ($refuri =~ m|^/adm/|) {
                    $thisallowed='F';
                } else {
                    $refuri=&declutter($refuri);
                    my ($match) = &is_on_map($refuri);
                    if ($match) {
                        $thisallowed='F';
                    }
                }
            }
        }
    }

    if ($priv eq 'bre'
	&& $thisallowed ne 'F' 
	&& $thisallowed ne '2'
	&& &is_portfolio_url($uri)) {
	$thisallowed = &portfolio_access($uri);
    }
    
# Full access at system, domain or course-wide level? Exit.
    if ($thisallowed=~/F/) {
	return 'F';
    }

# If this is generating or modifying users, exit with special codes

    if (':csu:cdc:ccc:cin:cta:cep:ccr:cst:cad:cli:cau:cdg:cca:caa:'=~/\:\Q$priv\E\:/) {
	if (($priv eq 'cca') || ($priv eq 'caa')) {
	    my ($audom,$auname)=split('/',$uri);
# no author name given, so this just checks on the general right to make a co-author in this domain
	    unless ($auname) { return $thisallowed; }
# an author name is given, so we are about to actually make a co-author for a certain account
	    if (($auname ne $env{'user.name'} && $env{'request.role'} !~ /^dc\./) ||
		(($audom ne $env{'user.domain'} && $env{'request.role'} !~ /^dc\./) &&
		 ($audom ne $env{'request.role.domain'}))) { return ''; }
	}
	return $thisallowed;
    }
#
# Gathered so far: system, domain and course wide privileges
#
# Course: See if uri or referer is an individual resource that is part of 
# the course

    if ($env{'request.course.id'}) {

       $courseprivid=$env{'request.course.id'};
       if ($env{'request.course.sec'}) {
          $courseprivid.='/'.$env{'request.course.sec'};
       }
       $courseprivid=~s/\_/\//;
       my $checkreferer=1;
       my ($match,$cond)=&is_on_map($uri);
       if ($match) {
           $statecond=$cond;
           if ($env{'user.priv.'.$env{'request.role'}.'./'.$courseprivid}
               =~/\Q$priv\E\&([^\:]*)/) {
               $thisallowed.=$1;
               $checkreferer=0;
           }
       }
       
       if ($checkreferer) {
	  my $refuri=$env{'httpref.'.$orguri};
            unless ($refuri) {
                foreach my $key (keys(%env)) {
		    if ($key=~/^httpref\..*\*/) {
			my $pattern=$key;
                        $pattern=~s/^httpref\.\/res\///;
                        $pattern=~s/\*/\[\^\/\]\+/g;
                        $pattern=~s/\//\\\//g;
                        if ($orguri=~/$pattern/) {
			    $refuri=$env{$key};
                        }
                    }
                }
            }

         if ($refuri) { 
	  $refuri=&declutter($refuri);
          my ($match,$cond)=&is_on_map($refuri);
            if ($match) {
              my $refstatecond=$cond;
              if ($env{'user.priv.'.$env{'request.role'}.'./'.$courseprivid}
                  =~/\Q$priv\E\&([^\:]*)/) {
                  $thisallowed.=$1;
                  $uri=$refuri;
                  $statecond=$refstatecond;
              }
          }
        }
       }
   }

#
# Gathered now: all privileges that could apply, and condition number
# 
#
# Full or no access?
#

    if ($thisallowed=~/F/) {
	return 'F';
    }

    unless ($thisallowed) {
        return '';
    }

# Restrictions exist, deal with them
#
#   C:according to course preferences
#   R:according to resource settings
#   L:unless locked
#   X:according to user session state
#

# Possibly locked functionality, check all courses
# Locks might take effect only after 10 minutes cache expiration for other
# courses, and 2 minutes for current course

    my $envkey;
    if ($thisallowed=~/L/) {
        foreach $envkey (keys(%env)) {
           if ($envkey=~/^user\.role\.(st|ta)\.([^\.]*)/) {
               my $courseid=$2;
               my $roleid=$1.'.'.$2;
               $courseid=~s/^\///;
               my $expiretime=600;
               if ($env{'request.role'} eq $roleid) {
		  $expiretime=120;
               }
	       my ($cdom,$cnum,$csec)=split(/\//,$courseid);
               my $prefix='course.'.$cdom.'_'.$cnum.'.';
               if ((time-$env{$prefix.'last_cache'})>$expiretime) {
		   &coursedescription($courseid,{'freshen_cache' => 1});
               }
               if (($env{$prefix.'res.'.$uri.'.lock.sections'}=~/\,\Q$csec\E\,/)
                || ($env{$prefix.'res.'.$uri.'.lock.sections'} eq 'all')) {
		   if ($env{$prefix.'res.'.$uri.'.lock.expire'}>time) {
                       &log($env{'user.domain'},$env{'user.name'},
                            $env{'user.home'},
                            'Locked by res: '.$priv.' for '.$uri.' due to '.
                            $cdom.'/'.$cnum.'/'.$csec.' expire '.
                            $env{$prefix.'priv.'.$priv.'.lock.expire'});
		       return '';
                   }
               }
               if (($env{$prefix.'priv.'.$priv.'.lock.sections'}=~/\,\Q$csec\E\,/)
                || ($env{$prefix.'priv.'.$priv.'.lock.sections'} eq 'all')) {
		   if ($env{'priv.'.$priv.'.lock.expire'}>time) {
                       &log($env{'user.domain'},$env{'user.name'},
                            $env{'user.home'},
                            'Locked by priv: '.$priv.' for '.$uri.' due to '.
                            $cdom.'/'.$cnum.'/'.$csec.' expire '.
                            $env{$prefix.'priv.'.$priv.'.lock.expire'});
		       return '';
                   }
               }
	   }
       }
    }
   
#
# Rest of the restrictions depend on selected course
#

    unless ($env{'request.course.id'}) {
	if ($thisallowed eq 'A') {
	    return 'A';
        } elsif ($thisallowed eq 'B') {
            return 'B';
	} else {
	    return '1';
	}
    }

#
# Now user is definitely in a course
#


# Course preferences

   if ($thisallowed=~/C/) {
       my $rolecode=(split(/\./,$env{'request.role'}))[0];
       my $unamedom=$env{'user.name'}.':'.$env{'user.domain'};
       if ($env{'course.'.$env{'request.course.id'}.'.'.$priv.'.roles.denied'}
	   =~/\Q$rolecode\E/) {
	   if ($priv ne 'pch') { 
	       &logthis($env{'user.domain'}.':'.$env{'user.name'}.':'.$env{'user.home'}.':'.
			'Denied by role: '.$priv.' for '.$uri.' as '.$rolecode.' in '.
			$env{'request.course.id'});
	   }
           return '';
       }

       if ($env{'course.'.$env{'request.course.id'}.'.'.$priv.'.users.denied'}
	   =~/\Q$unamedom\E/) {
	   if ($priv ne 'pch') { 
	       &logthis($env{'user.domain'}.':'.$env{'user.name'}.':'.$env{'user.home'}.
			'Denied by user: '.$priv.' for '.$uri.' as '.$unamedom.' in '.
			$env{'request.course.id'});
	   }
           return '';
       }
   }

# Resource preferences

   if ($thisallowed=~/R/) {
       my $rolecode=(split(/\./,$env{'request.role'}))[0];
       if (&metadata($uri,'roledeny')=~/\Q$rolecode\E/) {
	   if ($priv ne 'pch') { 
	       &logthis($env{'user.domain'}.':'.$env{'user.name'}.':'.$env{'user.home'}.':'.
			'Denied by role: '.$priv.' for '.$uri.' as '.$rolecode);
	   }
	   return '';
       }
   }

# Restricted by state or randomout?

   if ($thisallowed=~/X/) {
      if ($env{'acc.randomout'}) {
	 if (!$symb) { $symb=&symbread($uri,1); }
         if (($symb) && ($env{'acc.randomout'}=~/\&\Q$symb\E\&/)) { 
            return ''; 
         }
      }
      if (&condval($statecond)) {
	 return '2';
      } else {
         return '';
      }
   }

    if ($thisallowed eq 'A') {
	return 'A';
    } elsif ($thisallowed eq 'B') {
        return 'B';
    }
   return 'F';
}

sub split_uri_for_cond {
    my $uri=&deversion(&declutter(shift));
    my @uriparts=split(/\//,$uri);
    my $filename=pop(@uriparts);
    my $pathname=join('/',@uriparts);
    return ($pathname,$filename);
}
# --------------------------------------------------- Is a resource on the map?

sub is_on_map {
    my ($pathname,$filename) = &split_uri_for_cond(shift);
    #Trying to find the conditional for the file
    my $match=($env{'acc.res.'.$env{'request.course.id'}.'.'.$pathname}=~
	       /\&\Q$filename\E\:([\d\|]+)\&/);
    if ($match) {
	return (1,$1);
    } else {
	return (0,0);
    }
}

# --------------------------------------------------------- Get symb from alias

sub get_symb_from_alias {
    my $symb=shift;
    my ($map,$resid,$url)=&decode_symb($symb);
# Already is a symb
    if ($url) { return $symb; }
# Must be an alias
    my $aliassymb='';
    my %bighash;
    if (tie(%bighash,'GDBM_File',$env{'request.course.fn'}.'.db',
                            &GDBM_READER(),0640)) {
        my $rid=$bighash{'mapalias_'.$symb};
	if ($rid) {
	    my ($mapid,$resid)=split(/\./,$rid);
	    $aliassymb=&encode_symb($bighash{'map_id_'.$mapid},
				    $resid,$bighash{'src_'.$rid});
	}
        untie %bighash;
    }
    return $aliassymb;
}

# ----------------------------------------------------------------- Define Role

sub definerole {
  if (allowed('mcr','/')) {
    my ($rolename,$sysrole,$domrole,$courole)=@_;
    foreach my $role (split(':',$sysrole)) {
	my ($crole,$cqual)=split(/\&/,$role);
        if ($pr{'cr:s'}!~/\Q$crole\E/) { return "refused:s:$crole"; }
        if ($pr{'cr:s'}=~/\Q$crole\E\&/) {
	    if ($pr{'cr:s'}!~/\Q$crole\E\&\w*\Q$cqual\E/) { 
               return "refused:s:$crole&$cqual"; 
            }
        }
    }
    foreach my $role (split(':',$domrole)) {
	my ($crole,$cqual)=split(/\&/,$role);
        if ($pr{'cr:d'}!~/\Q$crole\E/) { return "refused:d:$crole"; }
        if ($pr{'cr:d'}=~/\Q$crole\E\&/) {
	    if ($pr{'cr:d'}!~/\Q$crole\W\&\w*\Q$cqual\E/) { 
               return "refused:d:$crole&$cqual"; 
            }
        }
    }
    foreach my $role (split(':',$courole)) {
	my ($crole,$cqual)=split(/\&/,$role);
        if ($pr{'cr:c'}!~/\Q$crole\E/) { return "refused:c:$crole"; }
        if ($pr{'cr:c'}=~/\Q$crole\E\&/) {
	    if ($pr{'cr:c'}!~/\Q$crole\E\&\w*\Q$cqual\E/) { 
               return "refused:c:$crole&$cqual"; 
            }
        }
    }
    my $command="encrypt:rolesput:$env{'user.domain'}:$env{'user.name'}:".
                "$env{'user.domain'}:$env{'user.name'}:".
	        "rolesdef_$rolename=".
                escape($sysrole.'_'.$domrole.'_'.$courole);
    return reply($command,$env{'user.home'});
  } else {
    return 'refused';
  }
}

# ---------------- Make a metadata query against the network of library servers

sub metadata_query {
    my ($query,$custom,$customshow,$server_array)=@_;
    my %rhash;
    my %libserv = &all_library();
    my @server_list = (defined($server_array) ? @$server_array
                                              : keys(%libserv) );
    for my $server (@server_list) {
	unless ($custom or $customshow) {
	    my $reply=&reply("querysend:".&escape($query),$server);
	    $rhash{$server}=$reply;
	}
	else {
	    my $reply=&reply("querysend:".&escape($query).':'.
			     &escape($custom).':'.&escape($customshow),
			     $server);
	    $rhash{$server}=$reply;
	}
    }
    return \%rhash;
}

# ----------------------------------------- Send log queries and wait for reply

sub log_query {
    my ($uname,$udom,$query,%filters)=@_;
    my $uhome=&homeserver($uname,$udom);
    if ($uhome eq 'no_host') { return 'error: no_host'; }
    my $uhost=&hostname($uhome);
    my $command=&escape(join(':',map{$_.'='.$filters{$_}} keys(%filters)));
    my $queryid=&reply("querysend:".$query.':'.$udom.':'.$uname.':'.$command,
                       $uhome);
    unless ($queryid=~/^\Q$uhost\E\_/) { return 'error: '.$queryid; }
    return get_query_reply($queryid);
}

# -------------------------- Update MySQL table for portfolio file

sub update_portfolio_table {
    my ($uname,$udom,$file_name,$query,$group,$action) = @_;
    if ($group ne '') {
        $file_name =~s /^\Q$group\E//;
    }
    my $homeserver = &homeserver($uname,$udom);
    my $queryid=
        &reply("querysend:".$query.':'.&escape($uname.':'.$udom.':'.$group).
               ':'.&escape($file_name).':'.$action,$homeserver);
    my $reply = &get_query_reply($queryid);
    return $reply;
}

# -------------------------- Update MySQL allusers table

sub update_allusers_table {
    my ($uname,$udom,$names) = @_;
    my $homeserver = &homeserver($uname,$udom);
    my $queryid=
        &reply('querysend:allusers:'.&escape($uname).':'.&escape($udom).':'.
               'lastname='.&escape($names->{'lastname'}).'%%'.
               'firstname='.&escape($names->{'firstname'}).'%%'.
               'middlename='.&escape($names->{'middlename'}).'%%'.
               'generation='.&escape($names->{'generation'}).'%%'.
               'permanentemail='.&escape($names->{'permanentemail'}).'%%'.
               'id='.&escape($names->{'id'}),$homeserver);
    my $reply = &get_query_reply($queryid);
    return $reply;
}

# ------- Request retrieval of institutional classlists for course(s)

sub fetch_enrollment_query {
    my ($context,$affiliatesref,$replyref,$dom,$cnum) = @_;
    my $homeserver;
    my $maxtries = 1;
    if ($context eq 'automated') {
        $homeserver = $perlvar{'lonHostID'};
        $maxtries = 10; # will wait for up to 2000s for retrieval of classlist data before timeout
    } else {
        $homeserver = &homeserver($cnum,$dom);
    }
    my $host=&hostname($homeserver);
    my $cmd = '';
    foreach my $affiliate (keys(%{$affiliatesref})) {
        $cmd .= $affiliate.'='.join(",",@{$$affiliatesref{$affiliate}}).'%%';
    }
    $cmd =~ s/%%$//;
    $cmd = &escape($cmd);
    my $query = 'fetchenrollment';
    my $queryid=&reply("querysend:".$query.':'.$dom.':'.$env{'user.name'}.':'.$cmd,$homeserver);
    unless ($queryid=~/^\Q$host\E\_/) { 
        &logthis('fetch_enrollment_query: invalid queryid: '.$queryid.' for host: '.$host.' and homeserver: '.$homeserver.' context: '.$context.' '.$cnum); 
        return 'error: '.$queryid;
    }
    my $reply = &get_query_reply($queryid);
    my $tries = 1;
    while (($reply=~/^timeout/) && ($tries < $maxtries)) {
        $reply = &get_query_reply($queryid);
        $tries ++;
    }
    if ( ($reply =~/^timeout/) || ($reply =~/^error/) ) {
        &logthis('fetch_enrollment_query error: '.$reply.' for '.$dom.' '.$env{'user.name'}.' for '.$queryid.' context: '.$context.' '.$cnum.' maxtries: '.$maxtries.' tries: '.$tries);
    } else {
        my @responses = split(/:/,$reply);
        if ($homeserver eq $perlvar{'lonHostID'}) {
            foreach my $line (@responses) {
                my ($key,$value) = split(/=/,$line,2);
                $$replyref{$key} = $value;
            }
        } else {
            my $pathname = $perlvar{'lonDaemons'}.'/tmp';
            foreach my $line (@responses) {
                my ($key,$value) = split(/=/,$line);
                $$replyref{$key} = $value;
                if ($value > 0) {
                    foreach my $item (@{$$affiliatesref{$key}}) {
                        my $filename = $dom.'_'.$key.'_'.$item.'_classlist.xml';
                        my $destname = $pathname.'/'.$filename;
                        my $xml_classlist = &reply("autoretrieve:".$filename,$homeserver);
                        if ($xml_classlist =~ /^error/) {
                            &logthis('fetch_enrollment_query - autoretrieve error: '.$xml_classlist.' for '.$filename.' from server: '.$homeserver.' '.$context.' '.$cnum);
                        } else {
                            if ( open(FILE,">$destname") ) {
                                print FILE &unescape($xml_classlist);
                                close(FILE);
                            } else {
                                &logthis('fetch_enrollment_query - error opening classlist file '.$destname.' '.$context.' '.$cnum);
                            }
                        }
                    }
                }
            }
        }
        return 'ok';
    }
    return 'error';
}

sub get_query_reply {
    my $queryid=shift;
    my $replyfile=$perlvar{'lonDaemons'}.'/tmp/'.$queryid;
    my $reply='';
    for (1..100) {
	sleep 2;
        if (-e $replyfile.'.end') {
	    if (open(my $fh,$replyfile)) {
		$reply = join('',<$fh>);
		close($fh);
	   } else { return 'error: reply_file_error'; }
           return &unescape($reply);
	}
    }
    return 'timeout:'.$queryid;
}

sub courselog_query {
#
# possible filters:
# url: url or symb
# username
# domain
# action: view, submit, grade
# start: timestamp
# end: timestamp
#
    my (%filters)=@_;
    unless ($env{'request.course.id'}) { return 'no_course'; }
    if ($filters{'url'}) {
	$filters{'url'}=&symbclean(&declutter($filters{'url'}));
        $filters{'url'}=~s/\.(\w+)$/(\\.\\d+)*\\.$1/;
        $filters{'url'}=~s/\.(\w+)\_\_\_/(\\.\\d+)*\\.$1/;
    }
    my $cname=$env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    return &log_query($cname,$cdom,'courselog',%filters);
}

sub userlog_query {
#
# possible filters:
# action: log check role
# start: timestamp
# end: timestamp
#
    my ($uname,$udom,%filters)=@_;
    return &log_query($uname,$udom,'userlog',%filters);
}

#--------- Call auto-enrollment subs in localenroll.pm for homeserver for course 

sub auto_run {
    my ($cnum,$cdom) = @_;
    my $response = 0;
    my $settings;
    my %domconfig = &get_dom('configuration',['autoenroll'],$cdom);
    if (ref($domconfig{'autoenroll'}) eq 'HASH') {
        $settings = $domconfig{'autoenroll'};
        if ($settings->{'run'} eq '1') {
            $response = 1;
        }
    } else {
        my $homeserver;
        if (&is_course($cdom,$cnum)) {
            $homeserver = &homeserver($cnum,$cdom);
        } else {
            $homeserver = &domain($cdom,'primary');
        }
        if ($homeserver ne 'no_host') {
            $response = &reply('autorun:'.$cdom,$homeserver);
        }
    }
    return $response;
}

sub auto_get_sections {
    my ($cnum,$cdom,$inst_coursecode) = @_;
    my $homeserver;
    if (($cdom =~ /^$match_domain$/) && ($cnum =~ /^$match_courseid$/)) { 
        $homeserver = &homeserver($cnum,$cdom);
    }
    if (!defined($homeserver)) { 
        if ($cdom =~ /^$match_domain$/) {
            $homeserver = &domain($cdom,'primary');
        }
    }
    my @secs;
    if (defined($homeserver)) {
        my $response=&unescape(&reply('autogetsections:'.$inst_coursecode.':'.$cdom,$homeserver));
        unless ($response eq 'refused') {
            @secs = split(/:/,$response);
        }
    }
    return @secs;
}

sub auto_new_course {
    my ($cnum,$cdom,$inst_course_id,$owner) = @_;
    my $homeserver = &homeserver($cnum,$cdom);
    my $response=&unescape(&reply('autonewcourse:'.$inst_course_id.':'.$owner.':'.$cdom,$homeserver));
    return $response;
}

sub auto_validate_courseID {
    my ($cnum,$cdom,$inst_course_id) = @_;
    my $homeserver = &homeserver($cnum,$cdom);
    my $response=&unescape(&reply('autovalidatecourse:'.$inst_course_id.':'.$cdom,$homeserver));
    return $response;
}

sub auto_validate_instcode {
    my ($cnum,$cdom,$instcode,$owner) = @_;
    my ($homeserver,$response);
    if (($cdom =~ /^$match_domain$/) && ($cnum =~ /^$match_courseid$/)) {
        $homeserver = &homeserver($cnum,$cdom);
    }
    if (!defined($homeserver)) {
        if ($cdom =~ /^$match_domain$/) {
            $homeserver = &domain($cdom,'primary');
        }
    }
    $response=&unescape(&reply('autovalidateinstcode:'.$cdom.':'.
                        &escape($instcode).':'.&escape($owner),$homeserver));
    my ($outcome,$description) = map { &unescape($_); } split('&',$response,2);
    return ($outcome,$description);
}

sub auto_create_password {
    my ($cnum,$cdom,$authparam,$udom) = @_;
    my ($homeserver,$response);
    my $create_passwd = 0;
    my $authchk = '';
    if ($udom =~ /^$match_domain$/) {
        $homeserver = &domain($udom,'primary');
    }
    if ($homeserver eq '') {
        if (($cdom =~ /^$match_domain$/) && ($cnum =~ /^$match_courseid$/)) {
            $homeserver = &homeserver($cnum,$cdom);
        }
    }
    if ($homeserver eq '') {
        $authchk = 'nodomain';
    } else {
        $response=&unescape(&reply('autocreatepassword:'.$authparam.':'.$cdom,$homeserver));
        if ($response eq 'refused') {
            $authchk = 'refused';
        } else {
            ($authparam,$create_passwd,$authchk) = split(/:/,$response);
        }
    }
    return ($authparam,$create_passwd,$authchk);
}

sub auto_photo_permission {
    my ($cnum,$cdom,$students) = @_;
    my $homeserver = &homeserver($cnum,$cdom);
    my ($outcome,$perm_reqd,$conditions) = 
	split(/:/,&unescape(&reply('autophotopermission:'.$cdom,$homeserver)),3);
    if ($outcome =~ /^(con_lost|unknown_cmd|no_such_host)$/) {
	return (undef,undef);
    }
    return ($outcome,$perm_reqd,$conditions);
}

sub auto_checkphotos {
    my ($uname,$udom,$pid) = @_;
    my $homeserver = &homeserver($uname,$udom);
    my ($result,$resulttype);
    my $outcome = &unescape(&reply('autophotocheck:'.&escape($udom).':'.
				   &escape($uname).':'.&escape($pid),
				   $homeserver));
    if ($outcome =~ /^(con_lost|unknown_cmd|no_such_host)$/) {
	return (undef,undef);
    }
    if ($outcome) {
        ($result,$resulttype) = split(/:/,$outcome);
    } 
    return ($result,$resulttype);
}

sub auto_photochoice {
    my ($cnum,$cdom) = @_;
    my $homeserver = &homeserver($cnum,$cdom);
    my ($update,$comment) = split(/:/,&unescape(&reply('autophotochoice:'.
						       &escape($cdom),
						       $homeserver)));
    if ($update =~ /^(con_lost|unknown_cmd|no_such_host)$/) {
	return (undef,undef);
    }
    return ($update,$comment);
}

sub auto_photoupdate {
    my ($affiliatesref,$dom,$cnum,$photo) = @_;
    my $homeserver = &homeserver($cnum,$dom);
    my $host=&hostname($homeserver);
    my $cmd = '';
    my $maxtries = 1;
    foreach my $affiliate (keys(%{$affiliatesref})) {
        $cmd .= $affiliate.'='.join(",",@{$$affiliatesref{$affiliate}}).'%%';
    }
    $cmd =~ s/%%$//;
    $cmd = &escape($cmd);
    my $query = 'institutionalphotos';
    my $queryid=&reply("querysend:".$query.':'.$dom.':'.$cnum.':'.$cmd,$homeserver);
    unless ($queryid=~/^\Q$host\E\_/) {
        &logthis('institutionalphotos: invalid queryid: '.$queryid.' for host: '.$host.' and homeserver: '.$homeserver.' and course: '.$cnum);
        return 'error: '.$queryid;
    }
    my $reply = &get_query_reply($queryid);
    my $tries = 1;
    while (($reply=~/^timeout/) && ($tries < $maxtries)) {
        $reply = &get_query_reply($queryid);
        $tries ++;
    }
    if ( ($reply =~/^timeout/) || ($reply =~/^error/) ) {
        &logthis('institutionalphotos error: '.$reply.' for '.$dom.' '.$env{'user.name'}.' for '.$queryid.' course: '.$cnum.' maxtries: '.$maxtries.' tries: '.$tries);
    } else {
        my @responses = split(/:/,$reply);
        my $outcome = shift(@responses); 
        foreach my $item (@responses) {
            my ($key,$value) = split(/=/,$item);
            $$photo{$key} = $value;
        }
        return $outcome;
    }
    return 'error';
}

sub auto_instcode_format {
    my ($caller,$codedom,$instcodes,$codes,$codetitles,$cat_titles,
	$cat_order) = @_;
    my $courses = '';
    my @homeservers;
    if ($caller eq 'global') {
	my %servers = &get_servers($codedom,'library');
	foreach my $tryserver (keys(%servers)) {
	    if (!grep(/^\Q$tryserver\E$/,@homeservers)) {
		push(@homeservers,$tryserver);
	    }
        }
    } elsif ($caller eq 'requests') {
        if ($codedom =~ /^$match_domain$/) {
            my $chome = &domain($codedom,'primary');
            unless ($chome eq 'no_host') {
                push(@homeservers,$chome);
            }
        }
    } else {
        push(@homeservers,&homeserver($caller,$codedom));
    }
    foreach my $code (keys(%{$instcodes})) {
        $courses .= &escape($code).'='.&escape($$instcodes{$code}).'&';
    }
    chop($courses);
    my $ok_response = 0;
    my $response;
    while (@homeservers > 0 && $ok_response == 0) {
        my $server = shift(@homeservers); 
        $response=&reply('autoinstcodeformat:'.$codedom.':'.$courses,$server);
        if ($response !~ /(con_lost|error|no_such_host|refused)/) {
            my ($codes_str,$codetitles_str,$cat_titles_str,$cat_order_str) = 
		split(/:/,$response);
            %{$codes} = (%{$codes},&str2hash($codes_str));
            push(@{$codetitles},&str2array($codetitles_str));
            %{$cat_titles} = (%{$cat_titles},&str2hash($cat_titles_str));
            %{$cat_order} = (%{$cat_order},&str2hash($cat_order_str));
            $ok_response = 1;
        }
    }
    if ($ok_response) {
        return 'ok';
    } else {
        return $response;
    }
}

sub auto_instcode_defaults {
    my ($domain,$returnhash,$code_order) = @_;
    my @homeservers;

    my %servers = &get_servers($domain,'library');
    foreach my $tryserver (keys(%servers)) {
	if (!grep(/^\Q$tryserver\E$/,@homeservers)) {
	    push(@homeservers,$tryserver);
	}
    }

    my $response;
    foreach my $server (@homeservers) {
        $response=&reply('autoinstcodedefaults:'.$domain,$server);
        next if ($response =~ /(con_lost|error|no_such_host|refused)/);
	
	foreach my $pair (split(/\&/,$response)) {
	    my ($name,$value)=split(/\=/,$pair);
	    if ($name eq 'code_order') {
		@{$code_order} = split(/\&/,&unescape($value));
	    } else {
		$returnhash->{&unescape($name)}=&unescape($value);
	    }
	}
	return 'ok';
    }

    return $response;
}

sub auto_possible_instcodes {
    my ($domain,$codetitles,$cat_titles,$cat_orders,$code_order) = @_;
    unless ((ref($codetitles) eq 'ARRAY') && (ref($cat_titles) eq 'HASH') && 
            (ref($cat_orders) eq 'HASH') && (ref($code_order) eq 'ARRAY')) {
        return;
    }
    my (@homeservers,$uhome);
    if (defined(&domain($domain,'primary'))) {
        $uhome=&domain($domain,'primary');
        push(@homeservers,&domain($domain,'primary'));
    } else {
        my %servers = &get_servers($domain,'library');
        foreach my $tryserver (keys(%servers)) {
            if (!grep(/^\Q$tryserver\E$/,@homeservers)) {
                push(@homeservers,$tryserver);
            }
        }
    }
    my $response;
    foreach my $server (@homeservers) {
        $response=&reply('autopossibleinstcodes:'.$domain,$server);
        next if ($response =~ /(con_lost|error|no_such_host|refused)/);
        my ($codetitlestr,$codeorderstr,$cat_title,$cat_order) = 
            split(':',$response);
        @{$codetitles} = map { &unescape($_); } (split('&',$codetitlestr));
        @{$code_order} = map { &unescape($_); } (split('&',$codeorderstr));
        foreach my $item (split('&',$cat_title)) {   
            my ($name,$value)=split('=',$item);
            $cat_titles->{&unescape($name)}=&thaw_unescape($value);
        }
        foreach my $item (split('&',$cat_order)) {
            my ($name,$value)=split('=',$item);
            $cat_orders->{&unescape($name)}=&thaw_unescape($value);
        }
        return 'ok';
    }
    return $response;
}

sub auto_courserequest_checks {
    my ($dom) = @_;
    my ($homeserver,%validations);
    if ($dom =~ /^$match_domain$/) {
        $homeserver = &domain($dom,'primary');
    }
    unless ($homeserver eq 'no_host') {
        my $response=&reply('autocrsreqchecks:'.$dom,$homeserver);
        unless ($response =~ /(con_lost|error|no_such_host|refused)/) {
            my @items = split(/&/,$response);
            foreach my $item (@items) {
                my ($key,$value) = split('=',$item);
                $validations{&unescape($key)} = &thaw_unescape($value);
            }
        }
    }
    return %validations; 
}

sub auto_courserequest_validation {
    my ($dom,$owner,$crstype,$inststatuslist,$instcode,$instseclist) = @_;
    my ($homeserver,$response);
    if ($dom =~ /^$match_domain$/) {
        $homeserver = &domain($dom,'primary');
    }
    unless ($homeserver eq 'no_host') {  
          
        $response=&unescape(&reply('autocrsreqvalidation:'.$dom.':'.&escape($owner).
                                    ':'.&escape($crstype).':'.&escape($inststatuslist).
                                    ':'.&escape($instcode).':'.&escape($instseclist),
                                    $homeserver));
    }
    return $response;
}

sub auto_validate_class_sec {
    my ($cdom,$cnum,$owners,$inst_class) = @_;
    my $homeserver = &homeserver($cnum,$cdom);
    my $ownerlist;
    if (ref($owners) eq 'ARRAY') {
        $ownerlist = join(',',@{$owners});
    } else {
        $ownerlist = $owners;
    }
    my $response=&reply('autovalidateclass_sec:'.$inst_class.':'.
                        &escape($ownerlist).':'.$cdom,$homeserver);
    return $response;
}

# ------------------------------------------------------- Course Group routines

sub get_coursegroups {
    my ($cdom,$cnum,$group,$namespace) = @_;
    return(&dump($namespace,$cdom,$cnum,$group));
}

sub modify_coursegroup {
    my ($cdom,$cnum,$groupsettings) = @_;
    return(&put('coursegroups',$groupsettings,$cdom,$cnum));
}

sub toggle_coursegroup_status {
    my ($cdom,$cnum,$group,$action) = @_;
    my ($from_namespace,$to_namespace);
    if ($action eq 'delete') {
        $from_namespace = 'coursegroups';
        $to_namespace = 'deleted_groups';
    } else {
        $from_namespace = 'deleted_groups';
        $to_namespace = 'coursegroups';
    }
    my %curr_group = &get_coursegroups($cdom,$cnum,$group,$from_namespace);
    if (my $tmp = &error(%curr_group)) {
        &Apache::lonnet::logthis('Error retrieving group: '.$tmp.' in '.$cnum.':'.$cdom);
        return ('read error',$tmp);
    } else {
        my %savedsettings = %curr_group; 
        my $result = &put($to_namespace,\%savedsettings,$cdom,$cnum);
        my $deloutcome;
        if ($result eq 'ok') {
            $deloutcome = &del($from_namespace,[$group],$cdom,$cnum);
        } else {
            return ('write error',$result);
        }
        if ($deloutcome eq 'ok') {
            return 'ok';
        } else {
            return ('delete error',$deloutcome);
        }
    }
}

sub modify_group_roles {
    my ($cdom,$cnum,$group_id,$user,$end,$start,$userprivs,$selfenroll,$context) = @_;
    my $url = '/'.$cdom.'/'.$cnum.'/'.$group_id;
    my $role = 'gr/'.&escape($userprivs);
    my ($uname,$udom) = split(/:/,$user);
    my $result = &assignrole($udom,$uname,$url,$role,$end,$start,'',$selfenroll,$context);
    if ($result eq 'ok') {
        &devalidate_getgroups_cache($udom,$uname,$cdom,$cnum);
    }
    return $result;
}

sub modify_coursegroup_membership {
    my ($cdom,$cnum,$membership) = @_;
    my $result = &put('groupmembership',$membership,$cdom,$cnum);
    return $result;
}

sub get_active_groups {
    my ($udom,$uname,$cdom,$cnum) = @_;
    my $now = time;
    my %groups = ();
    foreach my $key (keys(%env)) {
        if ($key =~ m-user\.role\.gr\./($match_domain)/($match_courseid)/(\w+)$-) {
            my ($start,$end) = split(/\./,$env{$key});
            if (($end!=0) && ($end<$now)) { next; }
            if (($start!=0) && ($start>$now)) { next; }
            if ($1 eq $cdom && $2 eq $cnum) {
                $groups{$3} = $env{$key} ;
            }
        }
    }
    return %groups;
}

sub get_group_membership {
    my ($cdom,$cnum,$group) = @_;
    return(&dump('groupmembership',$cdom,$cnum,$group));
}

sub get_users_groups {
    my ($udom,$uname,$courseid) = @_;
    my @usersgroups;
    my $cachetime=1800;

    my $hashid="$udom:$uname:$courseid";
    my ($grouplist,$cached)=&is_cached_new('getgroups',$hashid);
    if (defined($cached)) {
        @usersgroups = split(/:/,$grouplist);
    } else {  
        $grouplist = '';
        my $courseurl = &courseid_to_courseurl($courseid);
        my %roleshash = &dump('roles',$udom,$uname,$courseurl);
        my $access_end = $env{'course.'.$courseid.
                              '.default_enrollment_end_date'};
        my $now = time;
        foreach my $key (keys(%roleshash)) {
            if ($key =~ /^\Q$courseurl\E\/(\w+)\_gr$/) {
                my $group = $1;
                if ($roleshash{$key} =~ /_(\d+)_(\d+)$/) {
                    my $start = $2;
                    my $end = $1;
                    if ($start == -1) { next; } # deleted from group
                    if (($start!=0) && ($start>$now)) { next; }
                    if (($end!=0) && ($end<$now)) {
                        if ($access_end && $access_end < $now) {
                            if ($access_end - $end < 86400) {
                                push(@usersgroups,$group);
                            }
                        }
                        next;
                    }
                    push(@usersgroups,$group);
                }
            }
        }
        @usersgroups = &sort_course_groups($courseid,@usersgroups);
        $grouplist = join(':',@usersgroups);
        &do_cache_new('getgroups',$hashid,$grouplist,$cachetime);
    }
    return @usersgroups;
}

sub devalidate_getgroups_cache {
    my ($udom,$uname,$cdom,$cnum)=@_;
    my $courseid = $cdom.'_'.$cnum;

    my $hashid="$udom:$uname:$courseid";
    &devalidate_cache_new('getgroups',$hashid);
}

# ------------------------------------------------------------------ Plain Text

sub plaintext {
    my ($short,$type,$cid,$forcedefault) = @_;
    if ($short =~ m{^cr/}) {
	return (split('/',$short))[-1];
    }
    if (!defined($cid)) {
        $cid = $env{'request.course.id'};
    }
    my %rolenames = (
                      Course    => 'std',
                      Community => 'alt1',
                    );
    if ($cid ne '') {
        if ($env{'course.'.$cid.'.'.$short.'.plaintext'} ne '') {
            unless ($forcedefault) {
                my $roletext = $env{'course.'.$cid.'.'.$short.'.plaintext'}; 
                &Apache::lonlocal::mt_escape(\$roletext);
                return &Apache::lonlocal::mt($roletext);
            }
        }
    }
    if ((defined($type)) && (defined($rolenames{$type})) &&
        (defined($rolenames{$type})) && 
        (defined($prp{$short}{$rolenames{$type}}))) {
        return &Apache::lonlocal::mt($prp{$short}{$rolenames{$type}});
    } elsif ($cid ne '') {
        my $crstype = $env{'course.'.$cid.'.type'};
        if (($crstype ne '') && (defined($rolenames{$crstype})) &&
            (defined($prp{$short}{$rolenames{$crstype}}))) {
            return &Apache::lonlocal::mt($prp{$short}{$rolenames{$crstype}});
        }
    }
    return &Apache::lonlocal::mt($prp{$short}{'std'});
}

# ----------------------------------------------------------------- Assign Role

sub assignrole {
    my ($udom,$uname,$url,$role,$end,$start,$deleteflag,$selfenroll,
        $context)=@_;
    my $mrole;
    if ($role =~ /^cr\//) {
        my $cwosec=$url;
        $cwosec=~s/^\/($match_domain)\/($match_courseid)\/.*/$1\/$2/;
	unless (&allowed('ccr',$cwosec)) {
           my $refused = 1;
           if ($context eq 'requestcourses') {
               if (($env{'user.name'} ne '') && ($env{'user.domain'} ne '')) {
                   if ($role =~ m{^cr/($match_domain)/($match_username)/([^/]+)$}) {
                       if (($1 eq $env{'user.domain'}) && ($2 eq $env{'user.name'})) {
                           my ($cdom,$cnum) = ($cwosec =~ m{^/?($match_domain)/($match_courseid)$});
                           my %crsenv = &userenvironment($cdom,$cnum,('internal.courseowner'));
                           if ($crsenv{'internal.courseowner'} eq
                               $env{'user.name'}.':'.$env{'user.domain'}) {
                               $refused = '';
                           }
                       }
                   }
               }
           }
           if ($refused) {
               &logthis('Refused custom assignrole: '.
                        $udom.' '.$uname.' '.$url.' '.$role.' '.$end.' '.$start.
                        ' by '.$env{'user.name'}.' at '.$env{'user.domain'});
               return 'refused';
           }
        }
        $mrole='cr';
    } elsif ($role =~ /^gr\//) {
        my $cwogrp=$url;
        $cwogrp=~s{^/($match_domain)/($match_courseid)/.*}{$1/$2};
        unless (&allowed('mdg',$cwogrp)) {
            &logthis('Refused group assignrole: '.
              $udom.' '.$uname.' '.$url.' '.$role.' '.$end.' '.$start.' by '.
                    $env{'user.name'}.' at '.$env{'user.domain'});
            return 'refused';
        }
        $mrole='gr';
    } else {
        my $cwosec=$url;
        $cwosec=~s/^\/($match_domain)\/($match_courseid)\/.*/$1\/$2/;
        if (!(&allowed('c'.$role,$cwosec)) && !(&allowed('c'.$role,$udom))) {
            my $refused;
            if (($env{'request.course.sec'}  ne '') && ($role eq 'st')) {
                if (!(&allowed('c'.$role,$url))) {
                    $refused = 1;
                }
            } else {
                $refused = 1;
            }
            if ($refused) {
                my ($cdom,$cnum) = ($cwosec =~ m{^/?($match_domain)/($match_courseid)$});
                if (!$selfenroll && $context eq 'course') {
                    my %crsenv;
                    if ($role eq 'cc' || $role eq 'co') {
                        %crsenv = &userenvironment($cdom,$cnum,('internal.courseowner'));
                        if (($role eq 'cc') && ($cnum !~ /^$match_community$/)) {
                            if ($env{'request.role'} eq 'cc./'.$cdom.'/'.$cnum) {
                                if ($crsenv{'internal.courseowner'} eq 
                                    $env{'user.name'}.':'.$env{'user.domain'}) {
                                    $refused = '';
                                }
                            }
                        } elsif (($role eq 'co') && ($cnum =~ /^$match_community$/)) { 
                            if ($env{'request.role'} eq 'co./'.$cdom.'/'.$cnum) {
                                if ($crsenv{'internal.courseowner'} eq 
                                    $env{'user.name'}.':'.$env{'user.domain'}) {
                                    $refused = '';
                                }
                            }
                        }
                    }
                } elsif (($selfenroll == 1) && ($role eq 'st') && ($udom eq $env{'user.domain'}) && ($uname eq $env{'user.name'})) {
                    $refused = '';
                } elsif ($context eq 'requestcourses') {
                    my @possroles = ('st','ta','ep','in','cc','co');
                    if ((grep(/^\Q$role\E$/,@possroles)) && ($env{'user.name'} ne '' && $env{'user.domain'} ne '')) {
                        my $wrongcc;
                        if ($cnum =~ /^$match_community$/) {
                            $wrongcc = 1 if ($role eq 'cc');
                        } else {
                            $wrongcc = 1 if ($role eq 'co');
                        }
                        unless ($wrongcc) {
                            my %crsenv = &userenvironment($cdom,$cnum,('internal.courseowner'));
                            if ($crsenv{'internal.courseowner'} eq 
                                 $env{'user.name'}.':'.$env{'user.domain'}) {
                                $refused = '';
                            }
                        }
                    }
                }
                if ($refused) {
                    &logthis('Refused assignrole: '.$udom.' '.$uname.' '.$url.
                             ' '.$role.' '.$end.' '.$start.' by '.
	  	             $env{'user.name'}.' at '.$env{'user.domain'});
                    return 'refused';
                }
            }
        }
        $mrole=$role;
    }
    my $command="encrypt:rolesput:$env{'user.domain'}:$env{'user.name'}:".
                "$udom:$uname:$url".'_'."$mrole=$role";
    if ($end) { $command.='_'.$end; }
    if ($start) {
	if ($end) { 
           $command.='_'.$start; 
        } else {
           $command.='_0_'.$start;
        }
    }
    my $origstart = $start;
    my $origend = $end;
    my $delflag;
# actually delete
    if ($deleteflag) {
	if ((&allowed('dro',$udom)) || (&allowed('dro',$url))) {
# modify command to delete the role
           $command="encrypt:rolesdel:$env{'user.domain'}:$env{'user.name'}:".
                "$udom:$uname:$url".'_'."$mrole";
	   &logthis("$env{'user.name'} at $env{'user.domain'} deletes $mrole in $url for $uname at $udom"); 
# set start and finish to negative values for userrolelog
           $start=-1;
           $end=-1;
           $delflag = 1;
        }
    }
# send command
    my $answer=&reply($command,&homeserver($uname,$udom));
# log new user role if status is ok
    if ($answer eq 'ok') {
	&userrolelog($role,$uname,$udom,$url,$start,$end);
# for course roles, perform group memberships changes triggered by role change.
        &courserolelog($role,$uname,$udom,$url,$origstart,$origend,$delflag,$selfenroll,$context);
        unless ($role =~ /^gr/) {
            &Apache::longroup::group_changes($udom,$uname,$url,$role,$origend,
                                             $origstart,$selfenroll,$context);
        }
        if ($role eq 'cc') {
            &autoupdate_coowners($url,$end,$start,$uname,$udom);
        }
    }
    return $answer;
}

sub autoupdate_coowners {
    my ($url,$end,$start,$uname,$udom) = @_;
    my ($cdom,$cnum) = ($url =~ m{^/($match_domain)/($match_courseid)});
    if (($cdom ne '') && ($cnum ne '')) {
        my $now = time;
        my %domdesign = &Apache::loncommon::get_domainconf($cdom);
        if ($domdesign{$cdom.'.autoassign.co-owners'}) {
            my %coursehash = &coursedescription($cdom.'_'.$cnum);
            my $instcode = $coursehash{'internal.coursecode'};
            if ($instcode ne '') {
                if (($start && $start <= $now) && ($end == 0) || ($end > $now)) {
                    unless ($coursehash{'internal.courseowner'} eq $uname.':'.$udom) {
                        my ($delcoowners,@newcoowners,$putresult,$delresult,$coowners);
                        my ($result,$desc) = &auto_validate_instcode($cnum,$cdom,$instcode,$uname.':'.$udom);
                        if ($result eq 'valid') {
                            if ($coursehash{'internal.co-owners'}) {
                                foreach my $coowner (split(',',$coursehash{'internal.co-owners'})) {
                                    push(@newcoowners,$coowner);
                                }
                                unless (grep(/^\Q$uname\E:\Q$udom\E$/,@newcoowners)) {
                                    push(@newcoowners,$uname.':'.$udom);
                                }
                                @newcoowners = sort(@newcoowners);
                            } else {
                                push(@newcoowners,$uname.':'.$udom);
                            }
                        } else {
                            if ($coursehash{'internal.co-owners'}) {
                                foreach my $coowner (split(',',$coursehash{'internal.co-owners'})) {
                                    unless ($coowner eq $uname.':'.$udom) {
                                        push(@newcoowners,$coowner);
                                    }
                                }
                                unless (@newcoowners > 0) {
                                    $delcoowners = 1;
                                    $coowners = '';
                                }
                            }
                        }
                        if (@newcoowners || $delcoowners) {
                            &store_coowners($cdom,$cnum,$coursehash{'home'},
                                            $delcoowners,@newcoowners);
                        }
                    }
                }
            }
        }
    }
}

sub store_coowners {
    my ($cdom,$cnum,$chome,$delcoowners,@newcoowners) = @_;
    my $cid = $cdom.'_'.$cnum;
    my ($coowners,$delresult,$putresult);
    if (@newcoowners) {
        $coowners = join(',',@newcoowners);
        my %coownershash = (
                            'internal.co-owners' => $coowners,
                           );
        $putresult = &put('environment',\%coownershash,$cdom,$cnum);
        if ($putresult eq 'ok') {
            if ($env{'course.'.$cid.'.num'} eq $cnum) {
                &appenv({'course.'.$cid.'.internal.co-owners' => $coowners});
            }
        }
    }
    if ($delcoowners) {
        $delresult = &Apache::lonnet::del('environment',['internal.co-owners'],$cdom,$cnum);
        if ($delresult eq 'ok') {
            if ($env{'course.'.$cid.'.internal.co-owners'}) {
                &Apache::lonnet::delenv('course.'.$cid.'.internal.co-owners');
            }
        }
    }
    if (($putresult eq 'ok') || ($delresult eq 'ok')) {
        my %crsinfo =
            &Apache::lonnet::courseiddump($cdom,'.',1,'.','.',$cnum,undef,undef,'.');
        if (ref($crsinfo{$cid}) eq 'HASH') {
            $crsinfo{$cid}{'co-owners'} = \@newcoowners;
            my $cidput = &Apache::lonnet::courseidput($cdom,\%crsinfo,$chome,'notime');
        }
    }
}

# -------------------------------------------------- Modify user authentication
# Overrides without validation

sub modifyuserauth {
    my ($udom,$uname,$umode,$upass)=@_;
    my $uhome=&homeserver($uname,$udom);
    unless (&allowed('mau',$udom)) { return 'refused'; }
    &logthis('Call to modify user authentication '.$udom.', '.$uname.', '.
             $umode.' by '.$env{'user.name'}.' at '.$env{'user.domain'}.
             ' in domain '.$env{'request.role.domain'});  
    my $reply=&reply('encrypt:changeuserauth:'.$udom.':'.$uname.':'.$umode.':'.
		     &escape($upass),$uhome);
    &log($env{'user.domain'},$env{'user.name'},$env{'user.home'},
        'Authentication changed for '.$udom.', '.$uname.', '.$umode.
         '(Remote '.$ENV{'REMOTE_ADDR'}.'): '.$reply);
    &log($udom,,$uname,$uhome,
        'Authentication changed by '.$env{'user.domain'}.', '.
                                     $env{'user.name'}.', '.$umode.
         '(Remote '.$ENV{'REMOTE_ADDR'}.'): '.$reply);
    unless ($reply eq 'ok') {
        &logthis('Authentication mode error: '.$reply);
	return 'error: '.$reply;
    }   
    return 'ok';
}

# --------------------------------------------------------------- Modify a user

sub modifyuser {
    my ($udom,    $uname, $uid,
        $umode,   $upass, $first,
        $middle,  $last,  $gene,
        $forceid, $desiredhome, $email, $inststatus, $candelete)=@_;
    $udom= &LONCAPA::clean_domain($udom);
    $uname=&LONCAPA::clean_username($uname);
    my $showcandelete = 'none';
    if (ref($candelete) eq 'ARRAY') {
        if (@{$candelete} > 0) {
            $showcandelete = join(', ',@{$candelete});
        }
    }
    &logthis('Call to modify user '.$udom.', '.$uname.', '.$uid.', '.
             $umode.', '.$first.', '.$middle.', '.
	     $last.', '.$gene.'(forceid: '.$forceid.'; candelete: '.$showcandelete.')'.
             (defined($desiredhome) ? ' desiredhome = '.$desiredhome :
                                     ' desiredhome not specified'). 
             ' by '.$env{'user.name'}.' at '.$env{'user.domain'}.
             ' in domain '.$env{'request.role.domain'});
    my $uhome=&homeserver($uname,$udom,'true');
# ----------------------------------------------------------------- Create User
    if (($uhome eq 'no_host') && 
	(($umode && $upass) || ($umode eq 'localauth'))) {
        my $unhome='';
        if (defined($desiredhome) && &host_domain($desiredhome) eq $udom) { 
            $unhome = $desiredhome;
	} elsif($env{'course.'.$env{'request.course.id'}.'.domain'} eq $udom) {
	    $unhome=$env{'course.'.$env{'request.course.id'}.'.home'};
        } else { # load balancing routine for determining $unhome
            my $loadm=10000000;
	    my %servers = &get_servers($udom,'library');
	    foreach my $tryserver (keys(%servers)) {
		my $answer=reply('load',$tryserver);
		if (($answer=~/\d+/) && ($answer<$loadm)) {
		    $loadm=$answer;
		    $unhome=$tryserver;
		}
	    }
        }
        if (($unhome eq '') || ($unhome eq 'no_host')) {
	    return 'error: unable to find a home server for '.$uname.
                   ' in domain '.$udom;
        }
        my $reply=&reply('encrypt:makeuser:'.$udom.':'.$uname.':'.$umode.':'.
                         &escape($upass),$unhome);
	unless ($reply eq 'ok') {
            return 'error: '.$reply;
        }   
        $uhome=&homeserver($uname,$udom,'true');
        if (($uhome eq '') || ($uhome eq 'no_host') || ($uhome ne $unhome)) {
	    return 'error: unable verify users home machine.';
        }
    }   # End of creation of new user
# ---------------------------------------------------------------------- Add ID
    if ($uid) {
       $uid=~tr/A-Z/a-z/;
       my %uidhash=&idrget($udom,$uname);
       if (($uidhash{$uname}) && ($uidhash{$uname}!~/error\:/) 
         && (!$forceid)) {
	  unless ($uid eq $uidhash{$uname}) {
	      return 'error: user id "'.$uid.'" does not match '.
                  'current user id "'.$uidhash{$uname}.'".';
          }
       } else {
	  &idput($udom,($uname => $uid));
       }
    }
# -------------------------------------------------------------- Add names, etc
    my @tmp=&get('environment',
		   ['firstname','middlename','lastname','generation','id',
                    'permanentemail','inststatus'],
		   $udom,$uname);
    my %names;
    if ($tmp[0] =~ m/^error:.*/) { 
        %names=(); 
    } else {
        %names = @tmp;
    }
#
# If name, email and/or uid are blank (e.g., because an uploaded file
# of users did not contain them), do not overwrite existing values
# unless field is in $candelete array ref.
#
    my @fields = ('firstname','middlename','lastname','generation',
                  'permanentemail','id');
    my %newvalues;
    if (ref($candelete) eq 'ARRAY') {
        foreach my $field (@fields) {
            if (grep(/^\Q$field\E$/,@{$candelete})) {
                if ($field eq 'firstname') {
                    $names{$field} = $first;
                } elsif ($field eq 'middlename') {
                    $names{$field} = $middle;
                } elsif ($field eq 'lastname') {
                    $names{$field} = $last;
                } elsif ($field eq 'generation') {
                    $names{$field} = $gene;
                } elsif ($field eq 'permanentemail') {
                    $names{$field} = $email;
                } elsif ($field eq 'id') {
                    $names{$field}  = $uid;
                }
            }
        }
    }

    if ($first)  { $names{'firstname'}  = $first; }
    if (defined($middle)) { $names{'middlename'} = $middle; }
    if ($last)   { $names{'lastname'}   = $last; }
    if (defined($gene))   { $names{'generation'} = $gene; }
    if ($email) {
       $email=~s/[^\w\@\.\-\,]//gs;
       if ($email=~/\@/) { $names{'permanentemail'} = $email; }
    }
    if ($uid) { $names{'id'}  = $uid; }
    if (defined($inststatus)) {
        $names{'inststatus'} = '';
        my ($usertypes,$typesorder) = &retrieve_inst_usertypes($udom);
        if (ref($usertypes) eq 'HASH') {
            my @okstatuses; 
            foreach my $item (split(/:/,$inststatus)) {
                if (defined($usertypes->{$item})) {
                    push(@okstatuses,$item);  
                }
            }
            if (@okstatuses) {
                $names{'inststatus'} = join(':', map { &escape($_); } @okstatuses);
            }
        }
    }
    my $reply = &put('environment', \%names, $udom,$uname);
    if ($reply ne 'ok') { return 'error: '.$reply; }
    my $sqlresult = &update_allusers_table($uname,$udom,\%names);
    &devalidate_cache_new('namescache',$uname.':'.$udom);
    my $logmsg = 'Success modifying user '.$udom.', '.$uname.', '.$uid.', '.
                 $umode.', '.$first.', '.$middle.', '.
	         $last.', '.$gene.', '.$email.', '.$inststatus;
    if ($env{'user.name'} ne '' && $env{'user.domain'}) {
        $logmsg .= ' by '.$env{'user.name'}.' at '.$env{'user.domain'};
    } else {
        $logmsg .= ' during self creation';
    }
    &logthis($logmsg);
    return 'ok';
}

# -------------------------------------------------------------- Modify student

sub modifystudent {
    my ($udom,$uname,$uid,$umode,$upass,$first,$middle,$last,$gene,$usec,
        $end,$start,$forceid,$desiredhome,$email,$type,$locktype,$cid,
        $selfenroll,$context,$inststatus)=@_;
    if (!$cid) {
	unless ($cid=$env{'request.course.id'}) {
	    return 'not_in_class';
	}
    }
# --------------------------------------------------------------- Make the user
    my $reply=&modifyuser
	($udom,$uname,$uid,$umode,$upass,$first,$middle,$last,$gene,$forceid,
         $desiredhome,$email,$inststatus);
    unless ($reply eq 'ok') { return $reply; }
    # This will cause &modify_student_enrollment to get the uid from the
    # students environment
    $uid = undef if (!$forceid);
    $reply = &modify_student_enrollment($udom,$uname,$uid,$first,$middle,$last,
					$gene,$usec,$end,$start,$type,$locktype,$cid,$selfenroll,$context);
    return $reply;
}

sub modify_student_enrollment {
    my ($udom,$uname,$uid,$first,$middle,$last,$gene,$usec,$end,$start,$type,$locktype,$cid,$selfenroll,$context) = @_;
    my ($cdom,$cnum,$chome);
    if (!$cid) {
	unless ($cid=$env{'request.course.id'}) {
	    return 'not_in_class';
	}
	$cdom=$env{'course.'.$cid.'.domain'};
	$cnum=$env{'course.'.$cid.'.num'};
    } else {
	($cdom,$cnum)=split(/_/,$cid);
    }
    $chome=$env{'course.'.$cid.'.home'};
    if (!$chome) {
	$chome=&homeserver($cnum,$cdom);
    }
    if (!$chome) { return 'unknown_course'; }
    # Make sure the user exists
    my $uhome=&homeserver($uname,$udom);
    if (($uhome eq '') || ($uhome eq 'no_host')) { 
	return 'error: no such user';
    }
    # Get student data if we were not given enough information
    if (!defined($first)  || $first  eq '' || 
        !defined($last)   || $last   eq '' || 
        !defined($uid)    || $uid    eq '' || 
        !defined($middle) || $middle eq '' || 
        !defined($gene)   || $gene   eq '') {
        # They did not supply us with enough data to enroll the student, so
        # we need to pick up more information.
        my %tmp = &get('environment',
                       ['firstname','middlename','lastname', 'generation','id']
                       ,$udom,$uname);

        #foreach my $key (keys(%tmp)) {
        #    &logthis("key $key = ".$tmp{$key});
        #}
        $first  = $tmp{'firstname'}  if (!defined($first)  || $first  eq '');
        $middle = $tmp{'middlename'} if (!defined($middle) || $middle eq '');
        $last   = $tmp{'lastname'}   if (!defined($last)   || $last eq '');
        $gene   = $tmp{'generation'} if (!defined($gene)   || $gene eq '');
        $uid    = $tmp{'id'}         if (!defined($uid)    || $uid  eq '');
    }
    my $fullname = &format_name($first,$middle,$last,$gene,'lastname');
    my $reply=cput('classlist',
		   {"$uname:$udom" => 
			join(':',$end,$start,$uid,$usec,$fullname,$type,$locktype) },
		   $cdom,$cnum);
    unless (($reply eq 'ok') || ($reply eq 'delayed')) {
	return 'error: '.$reply;
    } else {
	&devalidate_getsection_cache($udom,$uname,$cid);
    }
    # Add student role to user
    my $uurl='/'.$cid;
    $uurl=~s/\_/\//g;
    if ($usec) {
	$uurl.='/'.$usec;
    }
    return &assignrole($udom,$uname,$uurl,'st',$end,$start,undef,$selfenroll,$context);
}

sub format_name {
    my ($firstname,$middlename,$lastname,$generation,$first)=@_;
    my $name;
    if ($first ne 'lastname') {
	$name=$firstname.' '.$middlename.' '.$lastname.' '.$generation;
    } else {
	if ($lastname=~/\S/) {
	    $name.= $lastname.' '.$generation.', '.$firstname.' '.$middlename;
	    $name=~s/\s+,/,/;
	} else {
	    $name.= $firstname.' '.$middlename.' '.$generation;
	}
    }
    $name=~s/^\s+//;
    $name=~s/\s+$//;
    $name=~s/\s+/ /g;
    return $name;
}

# ------------------------------------------------- Write to course preferences

sub writecoursepref {
    my ($courseid,%prefs)=@_;
    $courseid=~s/^\///;
    $courseid=~s/\_/\//g;
    my ($cdomain,$cnum)=split(/\//,$courseid);
    my $chome=homeserver($cnum,$cdomain);
    if (($chome eq '') || ($chome eq 'no_host')) { 
	return 'error: no such course';
    }
    my $cstring='';
    foreach my $pref (keys(%prefs)) {
	$cstring.=&escape($pref).'='.&escape($prefs{$pref}).'&';
    }
    $cstring=~s/\&$//;
    return reply('put:'.$cdomain.':'.$cnum.':environment:'.$cstring,$chome);
}

# ---------------------------------------------------------- Make/modify course

sub createcourse {
    my ($udom,$description,$url,$course_server,$nonstandard,$inst_code,
        $course_owner,$crstype,$cnum,$context,$category)=@_;
    $url=&declutter($url);
    my $cid='';
    if ($context eq 'requestcourses') {
        my $can_create = 0;
        my ($ownername,$ownerdom) = split(':',$course_owner);
        if ($udom eq $ownerdom) {
            if (&usertools_access($ownername,$ownerdom,$category,undef,
                                  $context)) {
                $can_create = 1;
            }
        } else {
            my %userenv = &userenvironment($ownerdom,$ownername,'reqcrsotherdom.'.
                                           $category);
            if ($userenv{'reqcrsotherdom.'.$category} ne '') {
                my @curr = split(',',$userenv{'reqcrsotherdom.'.$category});
                if (@curr > 0) {
                    my @options = qw(approval validate autolimit);
                    my $optregex = join('|',@options);
                    if (grep(/^\Q$udom\E:($optregex)(=?\d*)$/,@curr)) {
                        $can_create = 1;
                    }
                }
            }
        }
        if ($can_create) {
            unless ($ownername eq $env{'user.name'} && $ownerdom eq $env{'user.domain'}) {
                unless (&allowed('ccc',$udom)) {
                    return 'refused'; 
                }
            }
        } else {
            return 'refused';
        }
    } elsif (!&allowed('ccc',$udom)) {
        return 'refused';
    }
# --------------------------------------------------------------- Get Unique ID
    my $uname;
    if ($cnum =~ /^$match_courseid$/) {
        my $chome=&homeserver($cnum,$udom,'true');
        if (($chome eq '') || ($chome eq 'no_host')) {
            $uname = $cnum;
        } else {
            $uname = &generate_coursenum($udom,$crstype);
        }
    } else {
        $uname = &generate_coursenum($udom,$crstype);
    }
    return $uname if ($uname =~ /^error/);
# -------------------------------------------------- Check supplied server name
    if (!defined($course_server)) {
        if (defined(&domain($udom,'primary'))) {
            $course_server = &domain($udom,'primary');
        } else {
            $course_server = $env{'user.home'}; 
        }
    }
    my %host_servers =
        &Apache::lonnet::get_servers($udom,'library');
    unless ($host_servers{$course_server}) {
        return 'error: invalid home server for course: '.$course_server;
    }
# ------------------------------------------------------------- Make the course
    my $reply=&reply('encrypt:makeuser:'.$udom.':'.$uname.':none::',
                      $course_server);
    unless ($reply eq 'ok') { return 'error: '.$reply; }
    my $uhome=&homeserver($uname,$udom,'true');
    if (($uhome eq '') || ($uhome eq 'no_host')) { 
	return 'error: no such course';
    }
# ----------------------------------------------------------------- Course made
# log existence
    my $now = time;
    my $newcourse = {
                    $udom.'_'.$uname => {
                                     description => $description,
                                     inst_code   => $inst_code,
                                     owner       => $course_owner,
                                     type        => $crstype,
                                     creator     => $env{'user.name'}.':'.
                                                    $env{'user.domain'},
                                     created     => $now,
                                     context     => $context,
                                                },
                    };
    &courseidput($udom,$newcourse,$uhome,'notime');
# set toplevel url
    my $topurl=$url;
    unless ($nonstandard) {
# ------------------------------------------ For standard courses, make top url
        my $mapurl=&clutter($url);
        if ($mapurl eq '/res/') { $mapurl=''; }
        $env{'form.initmap'}=(<<ENDINITMAP);
<map>
<resource id="1" type="start"></resource>
<resource id="2" src="$mapurl"></resource>
<resource id="3" type="finish"></resource>
<link index="1" from="1" to="2"></link>
<link index="2" from="2" to="3"></link>
</map>
ENDINITMAP
        $topurl=&declutter(
        &finishuserfileupload($uname,$udom,'initmap','default.sequence')
                          );
    }
# ----------------------------------------------------------- Write preferences
    &writecoursepref($udom.'_'.$uname,
                     ('description'              => $description,
                      'url'                      => $topurl,
                      'internal.creator'         => $env{'user.name'}.':'.
                                                    $env{'user.domain'},
                      'internal.created'         => $now,
                      'internal.creationcontext' => $context)
                    );
    return '/'.$udom.'/'.$uname;
}

# ------------------------------------------------------------------- Create ID
sub generate_coursenum {
    my ($udom,$crstype) = @_;
    my $domdesc = &domain($udom);
    return 'error: invalid domain' if ($domdesc eq '');
    my $first;
    if ($crstype eq 'Community') {
        $first = '0';
    } else {
        $first = int(1+rand(9)); 
    } 
    my $uname=$first.
        ('a'..'z','A'..'Z','0'..'9')[int(rand(62))].
        substr($$.time,0,5).unpack("H8",pack("I32",time)).
        unpack("H2",pack("I32",int(rand(255)))).$perlvar{'lonHostID'};
# ----------------------------------------------- Make sure that does not exist
    my $uhome=&homeserver($uname,$udom,'true');
    unless (($uhome eq '') || ($uhome eq 'no_host')) {
        if ($crstype eq 'Community') {
            $first = '0';
        } else {
            $first = int(1+rand(9));
        }
        $uname=$first.
               ('a'..'z','A'..'Z','0'..'9')[int(rand(62))].
               substr($$.time,0,5).unpack("H8",pack("I32",time)).
               unpack("H2",pack("I32",int(rand(255)))).$perlvar{'lonHostID'};
        $uhome=&homeserver($uname,$udom,'true');
        unless (($uhome eq '') || ($uhome eq 'no_host')) {
            return 'error: unable to generate unique course-ID';
        }
    }
    return $uname;
}

sub is_course {
    my ($cdom,$cnum) = @_;
    my %courses = &courseiddump($cdom,'.',1,'.','.',$cnum,undef,
				undef,'.');
    if (exists($courses{$cdom.'_'.$cnum})) {
        return 1;
    }
    return 0;
}

sub store_userdata {
    my ($storehash,$datakey,$namespace,$udom,$uname) = @_;
    my $result;
    if ($datakey ne '') {
        if (ref($storehash) eq 'HASH') {
            if ($udom eq '' || $uname eq '') {
                $udom = $env{'user.domain'};
                $uname = $env{'user.name'};
            }
            my $uhome=&homeserver($uname,$udom);
            if (($uhome eq '') || ($uhome eq 'no_host')) {
                $result = 'error: no_host';
            } else {
                $storehash->{'ip'} = $ENV{'REMOTE_ADDR'};
                $storehash->{'host'} = $perlvar{'lonHostID'};

                my $namevalue='';
                foreach my $key (keys(%{$storehash})) {
                    $namevalue.=&escape($key).'='.&freeze_escape($$storehash{$key}).'&';
                }
                $namevalue=~s/\&$//;
                $result =  &reply("store:$env{'user.domain'}:$env{'user.name'}:".
                                  "$namespace:$datakey:$namevalue",$uhome);
            }
        } else {
            $result = 'error: data to store was not a hash reference'; 
        }
    } else {
        $result= 'error: invalid requestkey'; 
    }
    return $result;
}

# ---------------------------------------------------------- Assign Custom Role

sub assigncustomrole {
    my ($udom,$uname,$url,$rdom,$rnam,$rolename,$end,$start,$deleteflag,$selfenroll,$context)=@_;
    return &assignrole($udom,$uname,$url,'cr/'.$rdom.'/'.$rnam.'/'.$rolename,
                       $end,$start,$deleteflag,$selfenroll,$context);
}

# ----------------------------------------------------------------- Revoke Role

sub revokerole {
    my ($udom,$uname,$url,$role,$deleteflag,$selfenroll,$context)=@_;
    my $now=time;
    return &assignrole($udom,$uname,$url,$role,$now,undef,$deleteflag,$selfenroll,$context);
}

# ---------------------------------------------------------- Revoke Custom Role

sub revokecustomrole {
    my ($udom,$uname,$url,$rdom,$rnam,$rolename,$deleteflag,$selfenroll,$context)=@_;
    my $now=time;
    return &assigncustomrole($udom,$uname,$url,$rdom,$rnam,$rolename,$now,
           $deleteflag,$selfenroll,$context);
}

# ------------------------------------------------------------ Disk usage
sub diskusage {
    my ($udom,$uname,$directorypath,$getpropath)=@_;
    $directorypath =~ s/\/$//;
    my $listing=&reply('du2:'.&escape($directorypath).':'
                       .&escape($getpropath).':'.&escape($uname).':'
                       .&escape($udom),homeserver($uname,$udom));
    if ($listing eq 'unknown_cmd') {
        if ($getpropath) {
            $directorypath = &propath($udom,$uname).'/'.$directorypath; 
        }
        $listing = &reply('du:'.$directorypath,homeserver($uname,$udom));
    }
    return $listing;
}

sub is_locked {
    my ($file_name, $domain, $user) = @_;
    my @check;
    my $is_locked;
    push @check, $file_name;
    my %locked = &get('file_permissions',\@check,
		      $env{'user.domain'},$env{'user.name'});
    my ($tmp)=keys(%locked);
    if ($tmp=~/^error:/) { undef(%locked); }
    
    if (ref($locked{$file_name}) eq 'ARRAY') {
        $is_locked = 'false';
        foreach my $entry (@{$locked{$file_name}}) {
           if (ref($entry) eq 'ARRAY') { 
               $is_locked = 'true';
               last;
           }
       }
    } else {
        $is_locked = 'false';
    }
}

sub declutter_portfile {
    my ($file) = @_;
    $file =~ s{^(/portfolio/|portfolio/)}{/};
    return $file;
}

# ------------------------------------------------------------- Mark as Read Only

sub mark_as_readonly {
    my ($domain,$user,$files,$what) = @_;
    my %current_permissions = &dump('file_permissions',$domain,$user);
    my ($tmp)=keys(%current_permissions);
    if ($tmp=~/^error:/) { undef(%current_permissions); }
    foreach my $file (@{$files}) {
	$file = &declutter_portfile($file);
        push(@{$current_permissions{$file}},$what);
    }
    &put('file_permissions',\%current_permissions,$domain,$user);
    return;
}

# ------------------------------------------------------------Save Selected Files

sub save_selected_files {
    my ($user, $path, @files) = @_;
    my $filename = $user."savedfiles";
    my @other_files = &files_not_in_path($user, $path);
    open (OUT, '>'.$tmpdir.$filename);
    foreach my $file (@files) {
        print (OUT $env{'form.currentpath'}.$file."\n");
    }
    foreach my $file (@other_files) {
        print (OUT $file."\n");
    }
    close (OUT);
    return 'ok';
}

sub clear_selected_files {
    my ($user) = @_;
    my $filename = $user."savedfiles";
    open (OUT, '>'.$Apache::lonnet::perlvar{'lonDaemons'}.'/tmp/'.$filename);
    print (OUT undef);
    close (OUT);
    return ("ok");    
}

sub files_in_path {
    my ($user, $path) = @_;
    my $filename = $user."savedfiles";
    my %return_files;
    open (IN, '<'.$Apache::lonnet::perlvar{'lonDaemons'}.'/tmp/'.$filename);
    while (my $line_in = <IN>) {
        chomp ($line_in);
        my @paths_and_file = split (m!/!, $line_in);
        my $file_part = pop (@paths_and_file);
        my $path_part = join ('/', @paths_and_file);
        $path_part.='/';
        my $path_and_file = $path_part.$file_part;
        if ($path_part eq $path) {
            $return_files{$file_part}= 'selected';
        }
    }
    close (IN);
    return (\%return_files);
}

# called in portfolio select mode, to show files selected NOT in current directory
sub files_not_in_path {
    my ($user, $path) = @_;
    my $filename = $user."savedfiles";
    my @return_files;
    my $path_part;
    open(IN, '<'.$Apache::lonnet::perlvar{'lonDaemons'}.'/tmp/'.$filename);
    while (my $line = <IN>) {
        #ok, I know it's clunky, but I want it to work
        my @paths_and_file = split(m|/|, $line);
        my $file_part = pop(@paths_and_file);
        chomp($file_part);
        my $path_part = join('/', @paths_and_file);
        $path_part .= '/';
        my $path_and_file = $path_part.$file_part;
        if ($path_part ne $path) {
            push(@return_files, ($path_and_file));
        }
    }
    close(OUT);
    return (@return_files);
}

#----------------------------------------------Get portfolio file permissions

sub get_portfile_permissions {
    my ($domain,$user) = @_;
    my %current_permissions = &dump('file_permissions',$domain,$user);
    my ($tmp)=keys(%current_permissions);
    if ($tmp=~/^error:/) { undef(%current_permissions); }
    return \%current_permissions;
}

#---------------------------------------------Get portfolio file access controls

sub get_access_controls {
    my ($current_permissions,$group,$file) = @_;
    my %access;
    my $real_file = $file;
    $file =~ s/\.meta$//;
    if (defined($file)) {
        if (ref($$current_permissions{$file."\0".'accesscontrol'}) eq 'HASH') {
            foreach my $control (keys(%{$$current_permissions{$file."\0".'accesscontrol'}})) {
                $access{$real_file}{$control} = $$current_permissions{$file."\0".$control};
            }
        }
    } else {
        foreach my $key (keys(%{$current_permissions})) {
            if ($key =~ /\0accesscontrol$/) {
                if (defined($group)) {
                    if ($key !~ m-^\Q$group\E/-) {
                        next;
                    }
                }
                my ($fullpath) = split(/\0/,$key);
                if (ref($$current_permissions{$key}) eq 'HASH') {
                    foreach my $control (keys(%{$$current_permissions{$key}})) {
                        $access{$fullpath}{$control}=$$current_permissions{$fullpath."\0".$control};
                    }
                }
            }
        }
    }
    return %access;
}

sub modify_access_controls {
    my ($file_name,$changes,$domain,$user)=@_;
    my ($outcome,$deloutcome);
    my %store_permissions;
    my %new_values;
    my %new_control;
    my %translation;
    my @deletions = ();
    my $now = time;
    if (exists($$changes{'activate'})) {
        if (ref($$changes{'activate'}) eq 'HASH') {
            my @newitems = sort(keys(%{$$changes{'activate'}}));
            my $numnew = scalar(@newitems);
            for (my $i=0; $i<$numnew; $i++) {
                my $newkey = $newitems[$i];
                my $newid = &Apache::loncommon::get_cgi_id();
                if ($newkey =~ /^\d+:/) { 
                    $newkey =~ s/^(\d+)/$newid/;
                    $translation{$1} = $newid;
                } elsif ($newkey =~ /^\d+_\d+_\d+:/) {
                    $newkey =~ s/^(\d+_\d+_\d+)/$newid/;
                    $translation{$1} = $newid;
                }
                $new_values{$file_name."\0".$newkey} = 
                                          $$changes{'activate'}{$newitems[$i]};
                $new_control{$newkey} = $now;
            }
        }
    }
    my %todelete;
    my %changed_items;
    foreach my $action ('delete','update') {
        if (exists($$changes{$action})) {
            if (ref($$changes{$action}) eq 'HASH') {
                foreach my $key (keys(%{$$changes{$action}})) {
                    my ($itemnum) = ($key =~ /^([^:]+):/);
                    if ($action eq 'delete') { 
                        $todelete{$itemnum} = 1;
                    } else {
                        $changed_items{$itemnum} = $key;
                    }
                }
            }
        }
    }
    # get lock on access controls for file.
    my $lockhash = {
                  $file_name."\0".'locked_access_records' => $env{'user.name'}.
                                                       ':'.$env{'user.domain'},
                   }; 
    my $tries = 0;
    my $gotlock = &newput('file_permissions',$lockhash,$domain,$user);
   
    while (($gotlock ne 'ok') && $tries <3) {
        $tries ++;
        sleep 1;
        $gotlock = &newput('file_permissions',$lockhash,$domain,$user);
    }
    if ($gotlock eq 'ok') {
        my %curr_permissions = &dump('file_permissions',$domain,$user,$file_name);
        my ($tmp)=keys(%curr_permissions);
        if ($tmp=~/^error:/) { undef(%curr_permissions); }
        if (exists($curr_permissions{$file_name."\0".'accesscontrol'})) {
            my $curr_controls = $curr_permissions{$file_name."\0".'accesscontrol'};
            if (ref($curr_controls) eq 'HASH') {
                foreach my $control_item (keys(%{$curr_controls})) {
                    my ($itemnum) = ($control_item =~ /^([^:]+):/);
                    if (defined($todelete{$itemnum})) {
                        push(@deletions,$file_name."\0".$control_item);
                    } else {
                        if (defined($changed_items{$itemnum})) {
                            $new_control{$changed_items{$itemnum}} = $now;
                            push(@deletions,$file_name."\0".$control_item);
                            $new_values{$file_name."\0".$changed_items{$itemnum}} = $$changes{'update'}{$changed_items{$itemnum}};
                        } else {
                            $new_control{$control_item} = $$curr_controls{$control_item};
                        }
                    }
                }
            }
        }
        my ($group);
        if (&is_course($domain,$user)) {
            ($group,my $file) = split(/\//,$file_name,2);
        }
        $deloutcome = &del('file_permissions',\@deletions,$domain,$user);
        $new_values{$file_name."\0".'accesscontrol'} = \%new_control;
        $outcome = &put('file_permissions',\%new_values,$domain,$user);
        #  remove lock
        my @del_lock = ($file_name."\0".'locked_access_records');
        my $dellockoutcome = &del('file_permissions',\@del_lock,$domain,$user);
        my $sqlresult =
            &update_portfolio_table($user,$domain,$file_name,'portfolio_access',
                                    $group);
    } else {
        $outcome = "error: could not obtain lockfile\n";  
    }
    return ($outcome,$deloutcome,\%new_values,\%translation);
}

sub make_public_indefinitely {
    my ($requrl) = @_;
    my $now = time;
    my $action = 'activate';
    my $aclnum = 0;
    if (&is_portfolio_url($requrl)) {
        my (undef,$udom,$unum,$file_name,$group) =
            &parse_portfolio_url($requrl);
        my $current_perms = &get_portfile_permissions($udom,$unum);
        my %access_controls = &get_access_controls($current_perms,
                                                   $group,$file_name);
        foreach my $key (keys(%{$access_controls{$file_name}})) {
            my ($num,$scope,$end,$start) = 
                ($key =~ /^([^:]+):([a-z]+)_(\d*)_?(\d*)$/);
            if ($scope eq 'public') {
                if ($start <= $now && $end == 0) {
                    $action = 'none';
                } else {
                    $action = 'update';
                    $aclnum = $num;
                }
                last;
            }
        }
        if ($action eq 'none') {
             return 'ok';
        } else {
            my %changes;
            my $newend = 0;
            my $newstart = $now;
            my $newkey = $aclnum.':public_'.$newend.'_'.$newstart;
            $changes{$action}{$newkey} = {
                type => 'public',
                time => {
                    start => $newstart,
                    end   => $newend,
                },
            };
            my ($outcome,$deloutcome,$new_values,$translation) =
                &modify_access_controls($file_name,\%changes,$udom,$unum);
            return $outcome;
        }
    } else {
        return 'invalid';
    }
}

#------------------------------------------------------Get Marked as Read Only

sub get_marked_as_readonly {
    my ($domain,$user,$what,$group) = @_;
    my $current_permissions = &get_portfile_permissions($domain,$user);
    my @readonly_files;
    my $cmp1=$what;
    if (ref($what)) { $cmp1=join('',@{$what}) };
    while (my ($file_name,$value) = each(%{$current_permissions})) {
        if (defined($group)) {
            if ($file_name !~ m-^\Q$group\E/-) {
                next;
            }
        }
        if (ref($value) eq "ARRAY"){
            foreach my $stored_what (@{$value}) {
                my $cmp2=$stored_what;
                if (ref($stored_what) eq 'ARRAY') {
                    $cmp2=join('',@{$stored_what});
                }
                if ($cmp1 eq $cmp2) {
                    push(@readonly_files, $file_name);
                    last;
                } elsif (!defined($what)) {
                    push(@readonly_files, $file_name);
                    last;
                }
            }
        }
    }
    return @readonly_files;
}
#-----------------------------------------------------------Get Marked as Read Only Hash

sub get_marked_as_readonly_hash {
    my ($current_permissions,$group,$what) = @_;
    my %readonly_files;
    while (my ($file_name,$value) = each(%{$current_permissions})) {
        if (defined($group)) {
            if ($file_name !~ m-^\Q$group\E/-) {
                next;
            }
        }
        if (ref($value) eq "ARRAY"){
            foreach my $stored_what (@{$value}) {
                if (ref($stored_what) eq 'ARRAY') {
                    foreach my $lock_descriptor(@{$stored_what}) {
                        if ($lock_descriptor eq 'graded') {
                            $readonly_files{$file_name} = 'graded';
                        } elsif ($lock_descriptor eq 'handback') {
                            $readonly_files{$file_name} = 'handback';
                        } else {
                            if (!exists($readonly_files{$file_name})) {
                                $readonly_files{$file_name} = 'locked';
                            }
                        }
                    }
                } 
            }
        } 
    }
    return %readonly_files;
}
# ------------------------------------------------------------ Unmark as Read Only

sub unmark_as_readonly {
    # unmarks $file_name (if $file_name is defined), or all files locked by $what 
    # for portfolio submissions, $what contains [$symb,$crsid] 
    my ($domain,$user,$what,$file_name,$group) = @_;
    $file_name = &declutter_portfile($file_name);
    my $symb_crs = $what;
    if (ref($what)) { $symb_crs=join('',@$what); }
    my %current_permissions = &dump('file_permissions',$domain,$user,$group);
    my ($tmp)=keys(%current_permissions);
    if ($tmp=~/^error:/) { undef(%current_permissions); }
    my @readonly_files = &get_marked_as_readonly($domain,$user,$what,$group);
    foreach my $file (@readonly_files) {
	my $clean_file = &declutter_portfile($file);
	if (defined($file_name) && ($file_name ne $clean_file)) { next; }
	my $current_locks = $current_permissions{$file};
        my @new_locks;
        my @del_keys;
        if (ref($current_locks) eq "ARRAY"){
            foreach my $locker (@{$current_locks}) {
                my $compare=$locker;
                if (ref($locker) eq 'ARRAY') {
                    $compare=join('',@{$locker});
                    if ($compare ne $symb_crs) {
                        push(@new_locks, $locker);
                    }
                }
            }
            if (scalar(@new_locks) > 0) {
                $current_permissions{$file} = \@new_locks;
            } else {
                push(@del_keys, $file);
                &del('file_permissions',\@del_keys, $domain, $user);
                delete($current_permissions{$file});
            }
        }
    }
    &put('file_permissions',\%current_permissions,$domain,$user);
    return;
}

# ------------------------------------------------------------ Directory lister

sub dirlist {
    my ($uri,$userdomain,$username,$getpropath,$getuserdir,$alternateRoot)=@_;
    $uri=~s/^\///;
    $uri=~s/\/$//;
    my ($udom, $uname);
    if ($getuserdir) {
        $udom = $userdomain;
        $uname = $username;
    } else {
        (undef,$udom,$uname)=split(/\//,$uri);
        if(defined($userdomain)) {
            $udom = $userdomain;
        }
        if(defined($username)) {
            $uname = $username;
        }
    }
    my ($dirRoot,$listing,@listing_results);

    $dirRoot = $perlvar{'lonDocRoot'};
    if (defined($getpropath)) {
        $dirRoot = &propath($udom,$uname);
        $dirRoot =~ s/\/$//;
    } elsif (defined($getuserdir)) {
        my $subdir=$uname.'__';
        $subdir =~ s/(.)(.)(.).*/$1\/$2\/$3/;
        $dirRoot = $Apache::lonnet::perlvar{'lonUsersDir'}
                   ."/$udom/$subdir/$uname";
    } elsif (defined($alternateRoot)) {
        $dirRoot = $alternateRoot;
    }

    if($udom) {
        if($uname) {
            $listing = &reply('ls3:'.&escape('/'.$uri).':'.$getpropath.':'
                              .$getuserdir.':'.&escape($dirRoot)
                              .':'.&escape($uname).':'.&escape($udom),
                              &homeserver($uname,$udom));
            if ($listing eq 'unknown_cmd') {
                $listing = &reply('ls2:'.$dirRoot.'/'.$uri,
                                  &homeserver($uname,$udom));
            } else {
                @listing_results = map { &unescape($_); } split(/:/,$listing);
            }
            if ($listing eq 'unknown_cmd') {
                $listing = &reply('ls:'.$dirRoot.'/'.$uri,
				  &homeserver($uname,$udom));
                @listing_results = split(/:/,$listing);
            } else {
                @listing_results = map { &unescape($_); } split(/:/,$listing);
            }
            return @listing_results;
        } elsif(!$alternateRoot) {
            my %allusers;
	    my %servers = &get_servers($udom,'library');
 	    foreach my $tryserver (keys(%servers)) {
                $listing = &reply('ls3:'.&escape("/res/$udom").':::::'.
                                  &escape($udom),$tryserver);
                if ($listing eq 'unknown_cmd') {
		    $listing = &reply('ls2:'.$perlvar{'lonDocRoot'}.'/res/'.
				      $udom, $tryserver);
                } else {
                    @listing_results = map { &unescape($_); } split(/:/,$listing);
                }
		if ($listing eq 'unknown_cmd') {
		    $listing = &reply('ls:'.$perlvar{'lonDocRoot'}.'/res/'.
				      $udom, $tryserver);
		    @listing_results = split(/:/,$listing);
		} else {
		    @listing_results =
			map { &unescape($_); } split(/:/,$listing);
		}
		if ($listing_results[0] ne 'no_such_dir' && 
		    $listing_results[0] ne 'empty'       &&
		    $listing_results[0] ne 'con_lost') {
		    foreach my $line (@listing_results) {
			my ($entry) = split(/&/,$line,2);
			$allusers{$entry} = 1;
		    }
		}
            }
            my $alluserstr='';
            foreach my $user (sort(keys(%allusers))) {
                $alluserstr.=$user.'&user:';
            }
            $alluserstr=~s/:$//;
            return split(/:/,$alluserstr);
        } else {
            return ('missing user name');
        }
    } elsif(!defined($getpropath)) {
        my @all_domains = sort(&all_domains());
        foreach my $domain (@all_domains) {
            $domain = $perlvar{'lonDocRoot'}.'/res/'.$domain.'/&domain';
        }
        return @all_domains;
    } else {
        return ('missing domain');
    }
}

# --------------------------------------------- GetFileTimestamp
# This function utilizes dirlist and returns the date stamp for
# when it was last modified.  It will also return an error of -1
# if an error occurs

sub GetFileTimestamp {
    my ($studentDomain,$studentName,$filename,$getuserdir)=@_;
    $studentDomain = &LONCAPA::clean_domain($studentDomain);
    $studentName   = &LONCAPA::clean_username($studentName);
    my ($fileStat) = 
        &Apache::lonnet::dirlist($filename,$studentDomain,$studentName, 
                                 undef,$getuserdir);
    my @stats = split('&', $fileStat);
    if($stats[0] ne 'empty' && $stats[0] ne 'no_such_dir') {
        # @stats contains first the filename, then the stat output
        return $stats[10]; # so this is 10 instead of 9.
    } else {
        return -1;
    }
}

sub stat_file {
    my ($uri) = @_;
    $uri = &clutter_with_no_wrapper($uri);

    my ($udom,$uname,$file);
    if ($uri =~ m-^/(uploaded|editupload)/-) {
	($udom,$uname,$file) =
	    ($uri =~ m-/(?:uploaded|editupload)/?($match_domain)/?($match_name)/?(.*)-);
	$file = 'userfiles/'.$file;
    }
    if ($uri =~ m-^/res/-) {
	($udom,$uname) = 
	    ($uri =~ m-/(?:res)/?($match_domain)/?($match_username)/-);
	$file = $uri;
    }

    if (!$udom || !$uname || !$file) {
	# unable to handle the uri
	return ();
    }
    my $getpropath;
    if ($file =~ /^userfiles\//) {
        $getpropath = 1;
    }
    my ($result) = &dirlist($file,$udom,$uname,$getpropath);
    my @stats = split('&', $result);
    
    if($stats[0] ne 'empty' && $stats[0] ne 'no_such_dir') {
	shift(@stats); #filename is first
	return @stats;
    }
    return ();
}

# -------------------------------------------------------- Value of a Condition

# gets the value of a specific preevaluated condition
#    stored in the string  $env{user.state.<cid>}
# or looks up a condition reference in the bighash and if if hasn't
# already been evaluated recurses into docondval to get the value of
# the condition, then memoizing it to 
#   $env{user.state.<cid>.<condition>}
sub directcondval {
    my $number=shift;
    if (!defined($env{'user.state.'.$env{'request.course.id'}})) {
	&Apache::lonuserstate::evalstate();
    }
    if (exists($env{'user.state.'.$env{'request.course.id'}.".$number"})) {
	return $env{'user.state.'.$env{'request.course.id'}.".$number"};
    } elsif ($number =~ /^_/) {
	my $sub_condition;
	if (tie(my %bighash,'GDBM_File',$env{'request.course.fn'}.'.db',
		&GDBM_READER(),0640)) {
	    $sub_condition=$bighash{'conditions'.$number};
	    untie(%bighash);
	}
	my $value = &docondval($sub_condition);
	&appenv({'user.state.'.$env{'request.course.id'}.".$number" => $value});
	return $value;
    }
    if ($env{'user.state.'.$env{'request.course.id'}}) {
       return substr($env{'user.state.'.$env{'request.course.id'}},$number,1);
    } else {
       return 2;
    }
}

# get the collection of conditions for this resource
sub condval {
    my $condidx=shift;
    my $allpathcond='';
    foreach my $cond (split(/\|/,$condidx)) {
	if (defined($env{'acc.cond.'.$env{'request.course.id'}.'.'.$cond})) {
	    $allpathcond.=
		'('.$env{'acc.cond.'.$env{'request.course.id'}.'.'.$cond}.')|';
	}
    }
    $allpathcond=~s/\|$//;
    return &docondval($allpathcond);
}

#evaluates an expression of conditions
sub docondval {
    my ($allpathcond) = @_;
    my $result=0;
    if ($env{'request.course.id'}
	&& defined($allpathcond)) {
	my $operand='|';
	my @stack;
	foreach my $chunk ($allpathcond=~/(\d+|_\d+\.\d+|\(|\)|\&|\|)/g) {
	    if ($chunk eq '(') {
		push @stack,($operand,$result);
	    } elsif ($chunk eq ')') {
		my $before=pop @stack;
		if (pop @stack eq '&') {
		    $result=$result>$before?$before:$result;
		} else {
		    $result=$result>$before?$result:$before;
		}
	    } elsif (($chunk eq '&') || ($chunk eq '|')) {
		$operand=$chunk;
	    } else {
		my $new=directcondval($chunk);
		if ($operand eq '&') {
		    $result=$result>$new?$new:$result;
		} else {
		    $result=$result>$new?$result:$new;
		}
	    }
	}
    }
    return $result;
}

# ---------------------------------------------------- Devalidate courseresdata

sub devalidatecourseresdata {
    my ($coursenum,$coursedomain)=@_;
    my $hashid=$coursenum.':'.$coursedomain;
    &devalidate_cache_new('courseres',$hashid);
}


# --------------------------------------------------- Course Resourcedata Query
#
#  Parameters:
#      $coursenum    - Number of the course.
#      $coursedomain - Domain at which the course was created.
#  Returns:
#     A hash of the course parameters along (I think) with timestamps
#     and version info.

sub get_courseresdata {
    my ($coursenum,$coursedomain)=@_;
    my $coursehom=&homeserver($coursenum,$coursedomain);
    my $hashid=$coursenum.':'.$coursedomain;
    my ($result,$cached)=&is_cached_new('courseres',$hashid);
    my %dumpreply;
    unless (defined($cached)) {
	%dumpreply=&dump('resourcedata',$coursedomain,$coursenum);
	$result=\%dumpreply;
	my ($tmp) = keys(%dumpreply);
	if ($tmp !~ /^(con_lost|error|no_such_host)/i) {
	    &do_cache_new('courseres',$hashid,$result,600);
	} elsif ($tmp =~ /^(con_lost|no_such_host)/) {
	    return $tmp;
	} elsif ($tmp =~ /^(error)/) {
	    $result=undef;
	    &do_cache_new('courseres',$hashid,$result,600);
	}
    }
    return $result;
}

sub devalidateuserresdata {
    my ($uname,$udom)=@_;
    my $hashid="$udom:$uname";
    &devalidate_cache_new('userres',$hashid);
}

sub get_userresdata {
    my ($uname,$udom)=@_;
    #most student don\'t have any data set, check if there is some data
    if (&EXT_cache_status($udom,$uname)) { return undef; }

    my $hashid="$udom:$uname";
    my ($result,$cached)=&is_cached_new('userres',$hashid);
    if (!defined($cached)) {
	my %resourcedata=&dump('resourcedata',$udom,$uname);
	$result=\%resourcedata;
	&do_cache_new('userres',$hashid,$result,600);
    }
    my ($tmp)=keys(%$result);
    if (($tmp!~/^error\:/) && ($tmp!~/^con_lost/)) {
	return $result;
    }
    #error 2 occurs when the .db doesn't exist
    if ($tmp!~/error: 2 /) {
	&logthis("<font color=\"blue\">WARNING:".
		 " Trying to get resource data for ".
		 $uname." at ".$udom.": ".
		 $tmp."</font>");
    } elsif ($tmp=~/error: 2 /) {
	#&EXT_cache_set($udom,$uname);
	&do_cache_new('userres',$hashid,undef,600);
	undef($tmp); # not really an error so don't send it back
    }
    return $tmp;
}
#----------------------------------------------- resdata - return resource data
#  Purpose:
#    Return resource data for either users or for a course.
#  Parameters:
#     $name      - Course/user name.
#     $domain    - Name of the domain the user/course is registered on.
#     $type      - Type of thing $name is (must be 'course' or 'user'
#     @which     - Array of names of resources desired.
#  Returns:
#     The value of the first reasource in @which that is found in the
#     resource hash.
#  Exceptional Conditions:
#     If the $type passed in is not valid (not the string 'course' or 
#     'user', an undefined  reference is returned.
#     If none of the resources are found, an undef is returned
sub resdata {
    my ($name,$domain,$type,@which)=@_;
    my $result;
    if ($type eq 'course') {
	$result=&get_courseresdata($name,$domain);
    } elsif ($type eq 'user') {
	$result=&get_userresdata($name,$domain);
    }
    if (!ref($result)) { return $result; }    
    foreach my $item (@which) {
	if (defined($result->{$item->[0]})) {
	    return [$result->{$item->[0]},$item->[1]];
	}
    }
    return undef;
}

#
# EXT resource caching routines
#

sub clear_EXT_cache_status {
    &delenv('cache.EXT.');
}

sub EXT_cache_status {
    my ($target_domain,$target_user) = @_;
    my $cachename = 'cache.EXT.'.$target_user.'.'.$target_domain;
    if (exists($env{$cachename}) && ($env{$cachename}+600) > time) {
        # We know already the user has no data
        return 1;
    } else {
        return 0;
    }
}

sub EXT_cache_set {
    my ($target_domain,$target_user) = @_;
    my $cachename = 'cache.EXT.'.$target_user.'.'.$target_domain;
    #&appenv({$cachename => time});
}

# --------------------------------------------------------- Value of a Variable
sub EXT {

    my ($varname,$symbparm,$udom,$uname,$usection,$recurse)=@_;
    unless ($varname) { return ''; }
    #get real user name/domain, courseid and symb
    my $courseid;
    my $publicuser;
    if ($symbparm) {
	$symbparm=&get_symb_from_alias($symbparm);
    }
    if (!($uname && $udom)) {
      (my $cursymb,$courseid,$udom,$uname,$publicuser)= &whichuser($symbparm);
      if (!$symbparm) {	$symbparm=$cursymb; }
    } else {
	$courseid=$env{'request.course.id'};
    }
    my ($realm,$space,$qualifier,@therest)=split(/\./,$varname);
    my $rest;
    if (defined($therest[0])) {
       $rest=join('.',@therest);
    } else {
       $rest='';
    }

    my $qualifierrest=$qualifier;
    if ($rest) { $qualifierrest.='.'.$rest; }
    my $spacequalifierrest=$space;
    if ($qualifierrest) { $spacequalifierrest.='.'.$qualifierrest; }
    if ($realm eq 'user') {
# --------------------------------------------------------------- user.resource
	if ($space eq 'resource') {
	    if ( (defined($Apache::lonhomework::parsing_a_problem)
		  || defined($Apache::lonhomework::parsing_a_task))
		 &&
		 ($symbparm eq &symbread()) ) {	
		# if we are in the middle of processing the resource the
		# get the value we are planning on committing
                if (defined($Apache::lonhomework::results{$qualifierrest})) {
                    return $Apache::lonhomework::results{$qualifierrest};
                } else {
                    return $Apache::lonhomework::history{$qualifierrest};
                }
	    } else {
		my %restored;
		if ($publicuser || $env{'request.state'} eq 'construct') {
		    %restored=&tmprestore($symbparm,$courseid,$udom,$uname);
		} else {
		    %restored=&restore($symbparm,$courseid,$udom,$uname);
		}
		return $restored{$qualifierrest};
	    }
# ----------------------------------------------------------------- user.access
        } elsif ($space eq 'access') {
	    # FIXME - not supporting calls for a specific user
            return &allowed($qualifier,$rest);
# ------------------------------------------ user.preferences, user.environment
        } elsif (($space eq 'preferences') || ($space eq 'environment')) {
	    if (($uname eq $env{'user.name'}) &&
		($udom eq $env{'user.domain'})) {
		return $env{join('.',('environment',$qualifierrest))};
	    } else {
		my %returnhash;
		if (!$publicuser) {
		    %returnhash=&userenvironment($udom,$uname,
						 $qualifierrest);
		}
		return $returnhash{$qualifierrest};
	    }
# ----------------------------------------------------------------- user.course
        } elsif ($space eq 'course') {
	    # FIXME - not supporting calls for a specific user
            return $env{join('.',('request.course',$qualifier))};
# ------------------------------------------------------------------- user.role
        } elsif ($space eq 'role') {
	    # FIXME - not supporting calls for a specific user
            my ($role,$where)=split(/\./,$env{'request.role'});
            if ($qualifier eq 'value') {
		return $role;
            } elsif ($qualifier eq 'extent') {
                return $where;
            }
# ----------------------------------------------------------------- user.domain
        } elsif ($space eq 'domain') {
            return $udom;
# ------------------------------------------------------------------- user.name
        } elsif ($space eq 'name') {
            return $uname;
# ---------------------------------------------------- Any other user namespace
        } else {
	    my %reply;
	    if (!$publicuser) {
		%reply=&get($space,[$qualifierrest],$udom,$uname);
	    }
	    return $reply{$qualifierrest};
        }
    } elsif ($realm eq 'query') {
# ---------------------------------------------- pull stuff out of query string
        &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
						[$spacequalifierrest]);
	return $env{'form.'.$spacequalifierrest}; 
   } elsif ($realm eq 'request') {
# ------------------------------------------------------------- request.browser
        if ($space eq 'browser') {
	    if ($qualifier eq 'textremote') {
		if (&Apache::lonlocal::mt('textual_remote_display') eq 'on') {
		    return 1;
		} else {
		    return 0;
		}
	    } else {
		return $env{'browser.'.$qualifier};
	    }
# ------------------------------------------------------------ request.filename
        } else {
            return $env{'request.'.$spacequalifierrest};
        }
    } elsif ($realm eq 'course') {
# ---------------------------------------------------------- course.description
        return $env{'course.'.$courseid.'.'.$spacequalifierrest};
    } elsif ($realm eq 'resource') {

	if (defined($courseid) && $courseid eq $env{'request.course.id'}) {
	    if (!$symbparm) { $symbparm=&symbread(); }
	}

	if ($space eq 'title') {
	    if (!$symbparm) { $symbparm = $env{'request.filename'}; }
	    return &gettitle($symbparm);
	}
	
	if ($space eq 'map') {
	    my ($map) = &decode_symb($symbparm);
	    return &symbread($map);
	}
	if ($space eq 'filename') {
	    if ($symbparm) {
		return &clutter((&decode_symb($symbparm))[2]);
	    }
	    return &hreflocation('',$env{'request.filename'});
	}

	my ($section, $group, @groups);
	my ($courselevelm,$courselevel);
	if ($symbparm && defined($courseid) && 
	    $courseid eq $env{'request.course.id'}) {

	    #print '<br>'.$space.' - '.$qualifier.' - '.$spacequalifierrest;

# ----------------------------------------------------- Cascading lookup scheme
	    my $symbp=$symbparm;
	    my $mapp=&deversion((&decode_symb($symbp))[0]);

	    my $symbparm=$symbp.'.'.$spacequalifierrest;
	    my $mapparm=$mapp.'___(all).'.$spacequalifierrest;

	    if (($env{'user.name'} eq $uname) &&
		($env{'user.domain'} eq $udom)) {
		$section=$env{'request.course.sec'};
                @groups = split(/:/,$env{'request.course.groups'});  
                @groups=&sort_course_groups($courseid,@groups); 
	    } else {
		if (! defined($usection)) {
		    $section=&getsection($udom,$uname,$courseid);
		} else {
		    $section = $usection;
		}
                @groups = &get_users_groups($udom,$uname,$courseid);
	    }

	    my $seclevel=$courseid.'.['.$section.'].'.$spacequalifierrest;
	    my $seclevelr=$courseid.'.['.$section.'].'.$symbparm;
	    my $seclevelm=$courseid.'.['.$section.'].'.$mapparm;

	    $courselevel=$courseid.'.'.$spacequalifierrest;
	    my $courselevelr=$courseid.'.'.$symbparm;
	    $courselevelm=$courseid.'.'.$mapparm;

# ----------------------------------------------------------- first, check user

	    my $userreply=&resdata($uname,$udom,'user',
				       ([$courselevelr,'resource'],
					[$courselevelm,'map'     ],
					[$courselevel, 'course'  ]));
	    if (defined($userreply)) { return &get_reply($userreply); }

# ------------------------------------------------ second, check some of course
            my $coursereply;
            if (@groups > 0) {
                $coursereply = &check_group_parms($courseid,\@groups,$symbparm,
                                       $mapparm,$spacequalifierrest);
                if (defined($coursereply)) { return &get_reply($coursereply); }
            }

	    $coursereply=&resdata($env{'course.'.$courseid.'.num'},
				  $env{'course.'.$courseid.'.domain'},
				  'course',
				  ([$seclevelr,   'resource'],
				   [$seclevelm,   'map'     ],
				   [$seclevel,    'course'  ],
				   [$courselevelr,'resource']));
	    if (defined($coursereply)) { return &get_reply($coursereply); }

# ------------------------------------------------------ third, check map parms
	    my %parmhash=();
	    my $thisparm='';
	    if (tie(%parmhash,'GDBM_File',
		    $env{'request.course.fn'}.'_parms.db',
		    &GDBM_READER(),0640)) {
		$thisparm=$parmhash{$symbparm};
		untie(%parmhash);
	    }
	    if ($thisparm) { return &get_reply([$thisparm,'resource']); }
	}
# ------------------------------------------ fourth, look in resource metadata

	$spacequalifierrest=~s/\./\_/;
	my $filename;
	if (!$symbparm) { $symbparm=&symbread(); }
	if ($symbparm) {
	    $filename=(&decode_symb($symbparm))[2];
	} else {
	    $filename=$env{'request.filename'};
	}
	my $metadata=&metadata($filename,$spacequalifierrest);
	if (defined($metadata)) { return &get_reply([$metadata,'resource']); }
	$metadata=&metadata($filename,'parameter_'.$spacequalifierrest);
	if (defined($metadata)) { return &get_reply([$metadata,'resource']); }

# ---------------------------------------------- fourth, look in rest of course
	if ($symbparm && defined($courseid) && 
	    $courseid eq $env{'request.course.id'}) {
	    my $coursereply=&resdata($env{'course.'.$courseid.'.num'},
				     $env{'course.'.$courseid.'.domain'},
				     'course',
				     ([$courselevelm,'map'   ],
				      [$courselevel, 'course']));
	    if (defined($coursereply)) { return &get_reply($coursereply); }
	}
# ------------------------------------------------------------------ Cascade up
	unless ($space eq '0') {
	    my @parts=split(/_/,$space);
	    my $id=pop(@parts);
	    my $part=join('_',@parts);
	    if ($part eq '') { $part='0'; }
	    my @partgeneral=&EXT('resource.'.$part.'.'.$qualifierrest,
				 $symbparm,$udom,$uname,$section,1);
	    if (defined($partgeneral[0])) { return &get_reply(\@partgeneral); }
	}
	if ($recurse) { return undef; }
	my $pack_def=&packages_tab_default($filename,$varname);
	if (defined($pack_def)) { return &get_reply([$pack_def,'resource']); }
# ---------------------------------------------------- Any other user namespace
    } elsif ($realm eq 'environment') {
# ----------------------------------------------------------------- environment
	if (($uname eq $env{'user.name'})&&($udom eq $env{'user.domain'})) {
	    return $env{'environment.'.$spacequalifierrest};
	} else {
	    if ($uname eq 'anonymous' && $udom eq '') {
		return '';
	    }
	    my %returnhash=&userenvironment($udom,$uname,
					    $spacequalifierrest);
	    return $returnhash{$spacequalifierrest};
	}
    } elsif ($realm eq 'system') {
# ----------------------------------------------------------------- system.time
	if ($space eq 'time') {
	    return time;
        }
    } elsif ($realm eq 'server') {
# ----------------------------------------------------------------- system.time
	if ($space eq 'name') {
	    return $ENV{'SERVER_NAME'};
        }
    }
    return '';
}

sub get_reply {
    my ($reply_value) = @_;
    if (ref($reply_value) eq 'ARRAY') {
        if (wantarray) {
	    return @$reply_value;
        }
        return $reply_value->[0];
    } else {
        return $reply_value;
    }
}

sub check_group_parms {
    my ($courseid,$groups,$symbparm,$mapparm,$what) = @_;
    my @groupitems = ();
    my $resultitem;
    my @levels = ([$symbparm,'resource'],[$mapparm,'map'],[$what,'course']);
    foreach my $group (@{$groups}) {
        foreach my $level (@levels) {
             my $item = $courseid.'.['.$group.'].'.$level->[0];
             push(@groupitems,[$item,$level->[1]]);
        }
    }
    my $coursereply = &resdata($env{'course.'.$courseid.'.num'},
                            $env{'course.'.$courseid.'.domain'},
                                     'course',@groupitems);
    return $coursereply;
}

sub sort_course_groups { # Sort groups based on defined rankings. Default is sort().
    my ($courseid,@groups) = @_;
    @groups = sort(@groups);
    return @groups;
}

sub packages_tab_default {
    my ($uri,$varname)=@_;
    my (undef,$part,$name)=split(/\./,$varname);

    my (@extension,@specifics,$do_default);
    foreach my $package (split(/,/,&metadata($uri,'packages'))) {
	my ($pack_type,$pack_part)=split(/_/,$package,2);
	if ($pack_type eq 'default') {
	    $do_default=1;
	} elsif ($pack_type eq 'extension') {
	    push(@extension,[$package,$pack_type,$pack_part]);
	} elsif ($pack_part eq $part || $pack_type eq 'part') {
	    # only look at packages defaults for packages that this id is
	    push(@specifics,[$package,$pack_type,$pack_part]);
	}
    }
    # first look for a package that matches the requested part id
    foreach my $package (@specifics) {
	my (undef,$pack_type,$pack_part)=@{$package};
	next if ($pack_part ne $part);
	if (defined($packagetab{"$pack_type&$name&default"})) {
	    return $packagetab{"$pack_type&$name&default"};
	}
    }
    # look for any possible matching non extension_ package
    foreach my $package (@specifics) {
	my (undef,$pack_type,$pack_part)=@{$package};
	if (defined($packagetab{"$pack_type&$name&default"})) {
	    return $packagetab{"$pack_type&$name&default"};
	}
	if ($pack_type eq 'part') { $pack_part='0'; }
	if (defined($packagetab{$pack_type."_".$pack_part."&$name&default"})) {
	    return $packagetab{$pack_type."_".$pack_part."&$name&default"};
	}
    }
    # look for any posible extension_ match
    foreach my $package (@extension) {
	my ($package,$pack_type)=@{$package};
	if (defined($packagetab{"$pack_type&$name&default"})) {
	    return $packagetab{"$pack_type&$name&default"};
	}
	if (defined($packagetab{$package."&$name&default"})) {
	    return $packagetab{$package."&$name&default"};
	}
    }
    # look for a global default setting
    if ($do_default && defined($packagetab{"default&$name&default"})) {
	return $packagetab{"default&$name&default"};
    }
    return undef;
}

sub add_prefix_and_part {
    my ($prefix,$part)=@_;
    my $keyroot;
    if (defined($prefix) && $prefix !~ /^__/) {
	# prefix that has a part already
	$keyroot=$prefix;
    } elsif (defined($prefix)) {
	# prefix that is missing a part
	if (defined($part)) { $keyroot='_'.$part.substr($prefix,1); }
    } else {
	# no prefix at all
	if (defined($part)) { $keyroot='_'.$part; }
    }
    return $keyroot;
}

# ---------------------------------------------------------------- Get metadata

my %metaentry;
sub metadata {
    my ($uri,$what,$liburi,$prefix,$depthcount)=@_;
    $uri=&declutter($uri);
    # if it is a non metadata possible uri return quickly
    if (($uri eq '') || 
	(($uri =~ m|^/*adm/|) && 
	     ($uri !~ m|^adm/includes|) && ($uri !~ m|/bulletinboard$|)) ||
        ($uri =~ m|/$|) || ($uri =~ m|/.meta$|) ) {
	return undef;
    }
    if (($uri =~ /^~/ || $uri =~ m{home/$match_username/public_html/}) 
	&& &Apache::lonxml::get_state('target') =~ /^(|meta)$/) {
	return undef;
    }
    my $filename=$uri;
    $uri=~s/\.meta$//;
#
# Is the metadata already cached?
# Look at timestamp of caching
# Everything is cached by the main uri, libraries are never directly cached
#
    if (!defined($liburi)) {
	my ($result,$cached)=&is_cached_new('meta',$uri);
	if (defined($cached)) { return $result->{':'.$what}; }
    }
    {
#
# Is this a recursive call for a library?
#
#	if (! exists($metacache{$uri})) {
#	    $metacache{$uri}={};
#	}
	my $cachetime = 60*60;
        if ($liburi) {
	    $liburi=&declutter($liburi);
            $filename=$liburi;
        } else {
	    &devalidate_cache_new('meta',$uri);
	    undef(%metaentry);
	}
        my %metathesekeys=();
        unless ($filename=~/\.meta$/) { $filename.='.meta'; }
	my $metastring;
	if ($uri =~ /^~/ || $uri =~ m{home/$match_username/public_html/}) {
	    my $which = &hreflocation('','/'.($liburi || $uri));
	    $metastring = 
		&Apache::lonnet::ssi_body($which,
					  ('grade_target' => 'meta'));
	    $cachetime = 1; # only want this cached in the child not long term
	} elsif ($uri !~ m -^(editupload)/-) {
	    my $file=&filelocation('',&clutter($filename));
	    #push(@{$metaentry{$uri.'.file'}},$file);
	    $metastring=&getfile($file);
	}
        my $parser=HTML::LCParser->new(\$metastring);
        my $token;
        undef %metathesekeys;
        while ($token=$parser->get_token) {
	    if ($token->[0] eq 'S') {
		if (defined($token->[2]->{'package'})) {
#
# This is a package - get package info
#
		    my $package=$token->[2]->{'package'};
		    my $keyroot=&add_prefix_and_part($prefix,$token->[2]->{'part'});
		    if (defined($token->[2]->{'id'})) { 
			$keyroot.='_'.$token->[2]->{'id'}; 
		    }
		    if ($metaentry{':packages'}) {
			$metaentry{':packages'}.=','.$package.$keyroot;
		    } else {
			$metaentry{':packages'}=$package.$keyroot;
		    }
		    foreach my $pack_entry (keys(%packagetab)) {
			my $part=$keyroot;
			$part=~s/^\_//;
			if ($pack_entry=~/^\Q$package\E\&/ || 
			    $pack_entry=~/^\Q$package\E_0\&/) {
			    my ($pack,$name,$subp)=split(/\&/,$pack_entry);
			    # ignore package.tab specified default values
                            # here &package_tab_default() will fetch those
			    if ($subp eq 'default') { next; }
			    my $value=$packagetab{$pack_entry};
			    my $unikey;
			    if ($pack =~ /_0$/) {
				$unikey='parameter_0_'.$name;
				$part=0;
			    } else {
				$unikey='parameter'.$keyroot.'_'.$name;
			    }
			    if ($subp eq 'display') {
				$value.=' [Part: '.$part.']';
			    }
			    $metaentry{':'.$unikey.'.part'}=$part;
			    $metathesekeys{$unikey}=1;
			    unless (defined($metaentry{':'.$unikey.'.'.$subp})) {
				$metaentry{':'.$unikey.'.'.$subp}=$value;
			    }
			    if (defined($metaentry{':'.$unikey.'.default'})) {
				$metaentry{':'.$unikey}=
				    $metaentry{':'.$unikey.'.default'};
			    }
			}
		    }
		} else {
#
# This is not a package - some other kind of start tag
#
		    my $entry=$token->[1];
		    my $unikey;
		    if ($entry eq 'import') {
			$unikey='';
		    } else {
			$unikey=$entry;
		    }
		    $unikey.=&add_prefix_and_part($prefix,$token->[2]->{'part'});

		    if (defined($token->[2]->{'id'})) { 
			$unikey.='_'.$token->[2]->{'id'}; 
		    }

		    if ($entry eq 'import') {
#
# Importing a library here
#
			if ($depthcount<20) {
			    my $location=$parser->get_text('/import');
			    my $dir=$filename;
			    $dir=~s|[^/]*$||;
			    $location=&filelocation($dir,$location);
			    my $metadata = 
				&metadata($uri,'keys', $location,$unikey,
					  $depthcount+1);
			    foreach my $meta (split(',',$metadata)) {
				$metaentry{':'.$meta}=$metaentry{':'.$meta};
				$metathesekeys{$meta}=1;
			    }
			}
		    } else { 
			
			if (defined($token->[2]->{'name'})) { 
			    $unikey.='_'.$token->[2]->{'name'}; 
			}
			$metathesekeys{$unikey}=1;
			foreach my $param (@{$token->[3]}) {
			    $metaentry{':'.$unikey.'.'.$param} =
				$token->[2]->{$param};
			}
			my $internaltext=&HTML::Entities::decode($parser->get_text('/'.$entry));
			my $default=$metaentry{':'.$unikey.'.default'};
			if ( $internaltext =~ /^\s*$/ && $default !~ /^\s*$/) {
		 # only ws inside the tag, and not in default, so use default
		 # as value
			    $metaentry{':'.$unikey}=$default;
			} elsif ( $internaltext =~ /\S/ ) {
		  # something interesting inside the tag
			    $metaentry{':'.$unikey}=$internaltext;
			} else {
		  # no interesting values, don't set a default
			}
# end of not-a-package not-a-library import
		    }
# end of not-a-package start tag
		}
# the next is the end of "start tag"
	    }
	}
	my ($extension) = ($uri =~ /\.(\w+)$/);
	$extension = lc($extension);
	if ($extension eq 'htm') { $extension='html'; }

	foreach my $key (keys(%packagetab)) {
	    #no specific packages #how's our extension
	    if ($key!~/^extension_\Q$extension\E&/) { next; }
	    &metadata_create_package_def($uri,$key,'extension_'.$extension,
					 \%metathesekeys);
	}

	if (!exists($metaentry{':packages'})
	    || $packagetab{"import_defaults&extension_$extension"}) {
	    foreach my $key (keys(%packagetab)) {
		#no specific packages well let's get default then
		if ($key!~/^default&/) { next; }
		&metadata_create_package_def($uri,$key,'default',
					     \%metathesekeys);
	    }
	}
# are there custom rights to evaluate
	if ($metaentry{':copyright'} eq 'custom') {

    #
    # Importing a rights file here
    #
	    unless ($depthcount) {
		my $location=$metaentry{':customdistributionfile'};
		my $dir=$filename;
		$dir=~s|[^/]*$||;
		$location=&filelocation($dir,$location);
		my $rights_metadata =
		    &metadata($uri,'keys',$location,'_rights',
			      $depthcount+1);
		foreach my $rights (split(',',$rights_metadata)) {
		    #$metaentry{':'.$rights}=$metacache{$uri}->{':'.$rights};
		    $metathesekeys{$rights}=1;
		}
	    }
	}
	# uniqifiy package listing
	my %seen;
	my @uniq_packages =
	    grep { ! $seen{$_} ++ } (split(',',$metaentry{':packages'}));
	$metaentry{':packages'} = join(',',@uniq_packages);

	$metaentry{':keys'} = join(',',keys(%metathesekeys));
	&metadata_generate_part0(\%metathesekeys,\%metaentry,$uri);
	$metaentry{':allpossiblekeys'}=join(',',keys %metathesekeys);
	&do_cache_new('meta',$uri,\%metaentry,$cachetime);
# this is the end of "was not already recently cached
    }
    return $metaentry{':'.$what};
}

sub metadata_create_package_def {
    my ($uri,$key,$package,$metathesekeys)=@_;
    my ($pack,$name,$subp)=split(/\&/,$key);
    if ($subp eq 'default') { next; }
    
    if (defined($metaentry{':packages'})) {
	$metaentry{':packages'}.=','.$package;
    } else {
	$metaentry{':packages'}=$package;
    }
    my $value=$packagetab{$key};
    my $unikey;
    $unikey='parameter_0_'.$name;
    $metaentry{':'.$unikey.'.part'}=0;
    $$metathesekeys{$unikey}=1;
    unless (defined($metaentry{':'.$unikey.'.'.$subp})) {
	$metaentry{':'.$unikey.'.'.$subp}=$value;
    }
    if (defined($metaentry{':'.$unikey.'.default'})) {
	$metaentry{':'.$unikey}=
	    $metaentry{':'.$unikey.'.default'};
    }
}

sub metadata_generate_part0 {
    my ($metadata,$metacache,$uri) = @_;
    my %allnames;
    foreach my $metakey (keys(%$metadata)) {
	if ($metakey=~/^parameter\_(.*)/) {
	  my $part=$$metacache{':'.$metakey.'.part'};
	  my $name=$$metacache{':'.$metakey.'.name'};
	  if (! exists($$metadata{'parameter_0_'.$name.'.name'})) {
	    $allnames{$name}=$part;
	  }
	}
    }
    foreach my $name (keys(%allnames)) {
      $$metadata{"parameter_0_$name"}=1;
      my $key=":parameter_0_$name";
      $$metacache{"$key.part"}='0';
      $$metacache{"$key.name"}=$name;
      $$metacache{"$key.type"}=$$metacache{':parameter_'.
					   $allnames{$name}.'_'.$name.
					   '.type'};
      my $olddis=$$metacache{':parameter_'.$allnames{$name}.'_'.$name.
			     '.display'};
      my $expr='[Part: '.$allnames{$name}.']';
      $olddis=~s/\Q$expr\E/\[Part: 0\]/;
      $$metacache{"$key.display"}=$olddis;
    }
}

# ------------------------------------------------------ Devalidate title cache

sub devalidate_title_cache {
    my ($url)=@_;
    if (!$env{'request.course.id'}) { return; }
    my $symb=&symbread($url);
    if (!$symb) { return; }
    my $key=$env{'request.course.id'}."\0".$symb;
    &devalidate_cache_new('title',$key);
}

# ------------------------------------------------- Get the title of a course

sub current_course_title {
    return $env{ 'course.' . $env{'request.course.id'} . '.description' };
}
# ------------------------------------------------- Get the title of a resource

sub gettitle {
    my $urlsymb=shift;
    my $symb=&symbread($urlsymb);
    if ($symb) {
	my $key=$env{'request.course.id'}."\0".$symb;
	my ($result,$cached)=&is_cached_new('title',$key);
	if (defined($cached)) { 
	    return $result;
	}
	my ($map,$resid,$url)=&decode_symb($symb);
	my $title='';
	if (!$map && $resid == 0 && $url =~/default\.sequence$/) {
	    $title = $env{'course.'.$env{'request.course.id'}.'.description'};
	} else {
	    if (tie(my %bighash,'GDBM_File',$env{'request.course.fn'}.'.db',
		    &GDBM_READER(),0640)) {
		my $mapid=$bighash{'map_pc_'.&clutter($map)};
		$title=$bighash{'title_'.$mapid.'.'.$resid};
		untie(%bighash);
	    }
	}
	$title=~s/\&colon\;/\:/gs;
	if ($title) {
	    return &do_cache_new('title',$key,$title,600);
	}
	$urlsymb=$url;
    }
    my $title=&metadata($urlsymb,'title');
    if (!$title) { $title=(split('/',$urlsymb))[-1]; }    
    return $title;
}

sub get_slot {
    my ($which,$cnum,$cdom)=@_;
    if (!$cnum || !$cdom) {
	(undef,my $courseid)=&whichuser();
	$cdom=$env{'course.'.$courseid.'.domain'};
	$cnum=$env{'course.'.$courseid.'.num'};
    }
    my $key=join("\0",'slots',$cdom,$cnum,$which);
    my %slotinfo;
    if (exists($remembered{$key})) {
	$slotinfo{$which} = $remembered{$key};
    } else {
	%slotinfo=&get('slots',[$which],$cdom,$cnum);
	&Apache::lonhomework::showhash(%slotinfo);
	my ($tmp)=keys(%slotinfo);
	if ($tmp=~/^error:/) { return (); }
	$remembered{$key} = $slotinfo{$which};
    }
    if (ref($slotinfo{$which}) eq 'HASH') {
	return %{$slotinfo{$which}};
    }
    return $slotinfo{$which};
}
# ------------------------------------------------- Update symbolic store links

sub symblist {
    my ($mapname,%newhash)=@_;
    $mapname=&deversion(&declutter($mapname));
    my %hash;
    if (($env{'request.course.fn'}) && (%newhash)) {
        if (tie(%hash,'GDBM_File',$env{'request.course.fn'}.'_symb.db',
                      &GDBM_WRCREAT(),0640)) {
	    foreach my $url (keys(%newhash)) {
		next if ($url eq 'last_known'
			 && $env{'form.no_update_last_known'});
		$hash{declutter($url)}=&encode_symb($mapname,
						    $newhash{$url}->[1],
						    $newhash{$url}->[0]);
            }
            if (untie(%hash)) {
		return 'ok';
            }
        }
    }
    return 'error';
}

# --------------------------------------------------------------- Verify a symb

sub symbverify {
    my ($symb,$thisurl)=@_;
    my $thisfn=$thisurl;
    $thisfn=&declutter($thisfn);
# direct jump to resource in page or to a sequence - will construct own symbs
    if ($thisfn=~/\.(page|sequence)$/) { return 1; }
# check URL part
    my ($map,$resid,$url)=&decode_symb($symb);

    unless ($url eq $thisfn) { return 0; }

    $symb=&symbclean($symb);
    $thisurl=&deversion($thisurl);
    $thisfn=&deversion($thisfn);

    my %bighash;
    my $okay=0;

    if (tie(%bighash,'GDBM_File',$env{'request.course.fn'}.'.db',
                            &GDBM_READER(),0640)) {
        if (($thisurl =~ m{^/adm/wrapper/ext/}) || ($thisurl =~ m{^ext/})) {
            $thisurl =~ s/\?.+$//;
        }
        my $ids=$bighash{'ids_'.&clutter($thisurl)};
        unless ($ids) { 
           $ids=$bighash{'ids_/'.$thisurl};
        }
        if ($ids) {
# ------------------------------------------------------------------- Has ID(s)
	    foreach my $id (split(/\,/,$ids)) {
	       my ($mapid,$resid)=split(/\./,$id);
               if ($thisfn =~ m{^/adm/wrapper/ext/}) {
                   $symb =~ s/\?.+$//;
               }
               if (
  &symbclean(&declutter($bighash{'map_id_'.$mapid}).'___'.$resid.'___'.$thisfn)
   eq $symb) { 
		   if (($env{'request.role.adv'}) ||
		       $bighash{'encrypted_'.$id} eq $env{'request.enc'}) {
		       $okay=1; 
		   }
	       }
	   }
        }
	untie(%bighash);
    }
    return $okay;
}

# --------------------------------------------------------------- Clean-up symb

sub symbclean {
    my $symb=shift;
    if ($symb=~m|^/enc/|) { $symb=&Apache::lonenc::unencrypted($symb); }
# remove version from map
    $symb=~s/\.(\d+)\.(\w+)\_\_\_/\.$2\_\_\_/;

# remove version from URL
    $symb=~s/\.(\d+)\.(\w+)$/\.$2/;

# remove wrapper

    $symb=~s/(\_\_\_\d+\_\_\_)adm\/wrapper\/(res\/)*/$1/;
    $symb=~s/(\_\_\_\d+\_\_\_)adm\/coursedocs\/showdoc\/(res\/)*/$1/;
    return $symb;
}

# ---------------------------------------------- Split symb to find map and url

sub encode_symb {
    my ($map,$resid,$url)=@_;
    return &symbclean(&declutter($map).'___'.$resid.'___'.&declutter($url));
}

sub decode_symb {
    my $symb=shift;
    if ($symb=~m|^/enc/|) { $symb=&Apache::lonenc::unencrypted($symb); }
    my ($map,$resid,$url)=split(/___/,$symb);
    return (&fixversion($map),$resid,&fixversion($url));
}

sub fixversion {
    my $fn=shift;
    if ($fn=~/^(adm|uploaded|editupload|public)/) { return $fn; }
    my %bighash;
    my $uri=&clutter($fn);
    my $key=$env{'request.course.id'}.'_'.$uri;
# is this cached?
    my ($result,$cached)=&is_cached_new('courseresversion',$key);
    if (defined($cached)) { return $result; }
# unfortunately not cached, or expired
    if (tie(%bighash,'GDBM_File',$env{'request.course.fn'}.'.db',
	    &GDBM_READER(),0640)) {
 	if ($bighash{'version_'.$uri}) {
 	    my $version=$bighash{'version_'.$uri};
 	    unless (($version eq 'mostrecent') || 
		    ($version==&getversion($uri))) {
 		$uri=~s/\.(\w+)$/\.$version\.$1/;
 	    }
 	}
 	untie %bighash;
    }
    return &do_cache_new('courseresversion',$key,&declutter($uri),600);
}

sub deversion {
    my $url=shift;
    $url=~s/\.\d+\.(\w+)$/\.$1/;
    return $url;
}

# ------------------------------------------------------ Return symb list entry

sub symbread {
    my ($thisfn,$donotrecurse)=@_;
    my $cache_str='request.symbread.cached.'.$thisfn;
    if (defined($env{$cache_str})) { return $env{$cache_str}; }
# no filename provided? try from environment
    unless ($thisfn) {
        if ($env{'request.symb'}) {
	    return $env{$cache_str}=&symbclean($env{'request.symb'});
	}
	$thisfn=$env{'request.filename'};
    }
    if ($thisfn=~m|^/enc/|) { $thisfn=&Apache::lonenc::unencrypted($thisfn); }
# is that filename actually a symb? Verify, clean, and return
    if ($thisfn=~/\_\_\_\d+\_\_\_(.*)$/) {
	if (&symbverify($thisfn,$1)) {
	    return $env{$cache_str}=&symbclean($thisfn);
	}
    }
    $thisfn=declutter($thisfn);
    my %hash;
    my %bighash;
    my $syval='';
    if (($env{'request.course.fn'}) && ($thisfn)) {
        my $targetfn = $thisfn;
        if ( ($thisfn =~ m/^(uploaded|editupload)\//) && ($thisfn !~ m/\.(page|sequence)$/) ) {
            $targetfn = 'adm/wrapper/'.$thisfn;
        }
	if ($targetfn =~ m|^adm/wrapper/(ext/.*)|) {
	    $targetfn=$1;
	}
        if (tie(%hash,'GDBM_File',$env{'request.course.fn'}.'_symb.db',
                      &GDBM_READER(),0640)) {
	    $syval=$hash{$targetfn};
            untie(%hash);
        }
# ---------------------------------------------------------- There was an entry
        if ($syval) {
	    #unless ($syval=~/\_\d+$/) {
		#unless ($env{'form.request.prefix'}=~/\.(\d+)\_$/) {
		    #&appenv({'request.ambiguous' => $thisfn});
		    #return $env{$cache_str}='';
		#}    
		#$syval.=$1;
	    #}
        } else {
# ------------------------------------------------------- Was not in symb table
           if (tie(%bighash,'GDBM_File',$env{'request.course.fn'}.'.db',
                            &GDBM_READER(),0640)) {
# ---------------------------------------------- Get ID(s) for current resource
              my $ids=$bighash{'ids_'.&clutter($thisfn)};
              unless ($ids) { 
                 $ids=$bighash{'ids_/'.$thisfn};
              }
              unless ($ids) {
# alias?
		  $ids=$bighash{'mapalias_'.$thisfn};
              }
              if ($ids) {
# ------------------------------------------------------------------- Has ID(s)
                 my @possibilities=split(/\,/,$ids);
                 if ($#possibilities==0) {
# ----------------------------------------------- There is only one possibility
		     my ($mapid,$resid)=split(/\./,$ids);
		     $syval=&encode_symb($bighash{'map_id_'.$mapid},
						    $resid,$thisfn);
                 } elsif (!$donotrecurse) {
# ------------------------------------------ There is more than one possibility
                     my $realpossible=0;
                     foreach my $id (@possibilities) {
			 my $file=$bighash{'src_'.$id};
                         if (&allowed('bre',$file)) {
         		    my ($mapid,$resid)=split(/\./,$id);
                            if ($bighash{'map_type_'.$mapid} ne 'page') {
				$realpossible++;
                                $syval=&encode_symb($bighash{'map_id_'.$mapid},
						    $resid,$thisfn);
                            }
			 }
                     }
		     if ($realpossible!=1) { $syval=''; }
                 } else {
                     $syval='';
                 }
	      }
              untie(%bighash)
           }
        }
        if ($syval) {
	    return $env{$cache_str}=$syval;
        }
    }
    &appenv({'request.ambiguous' => $thisfn});
    return $env{$cache_str}='';
}

# ---------------------------------------------------------- Return random seed

sub numval {
    my $txt=shift;
    $txt=~tr/A-J/0-9/;
    $txt=~tr/a-j/0-9/;
    $txt=~tr/K-T/0-9/;
    $txt=~tr/k-t/0-9/;
    $txt=~tr/U-Z/0-5/;
    $txt=~tr/u-z/0-5/;
    $txt=~s/\D//g;
    if ($_64bit) { if ($txt > 2**32) { return -1; } }
    return int($txt);
}

sub numval2 {
    my $txt=shift;
    $txt=~tr/A-J/0-9/;
    $txt=~tr/a-j/0-9/;
    $txt=~tr/K-T/0-9/;
    $txt=~tr/k-t/0-9/;
    $txt=~tr/U-Z/0-5/;
    $txt=~tr/u-z/0-5/;
    $txt=~s/\D//g;
    my @txts=split(/(\d\d\d\d\d\d\d\d\d)/,$txt);
    my $total;
    foreach my $val (@txts) { $total+=$val; }
    if ($_64bit) { if ($total > 2**32) { return -1; } }
    return int($total);
}

sub numval3 {
    use integer;
    my $txt=shift;
    $txt=~tr/A-J/0-9/;
    $txt=~tr/a-j/0-9/;
    $txt=~tr/K-T/0-9/;
    $txt=~tr/k-t/0-9/;
    $txt=~tr/U-Z/0-5/;
    $txt=~tr/u-z/0-5/;
    $txt=~s/\D//g;
    my @txts=split(/(\d\d\d\d\d\d\d\d\d)/,$txt);
    my $total;
    foreach my $val (@txts) { $total+=$val; }
    if ($_64bit) { $total=(($total<<32)>>32); }
    return $total;
}

sub digest {
    my ($data)=@_;
    my $digest=&Digest::MD5::md5($data);
    my ($a,$b,$c,$d)=unpack("iiii",$digest);
    my ($e,$f);
    {
        use integer;
        $e=($a+$b);
        $f=($c+$d);
        if ($_64bit) {
            $e=(($e<<32)>>32);
            $f=(($f<<32)>>32);
        }
    }
    if (wantarray) {
	return ($e,$f);
    } else {
	my $g;
	{
	    use integer;
	    $g=($e+$f);
	    if ($_64bit) {
		$g=(($g<<32)>>32);
	    }
	}
	return $g;
    }
}

sub latest_rnd_algorithm_id {
    return '64bit5';
}

sub get_rand_alg {
    my ($courseid)=@_;
    if (!$courseid) { $courseid=(&whichuser())[1]; }
    if ($courseid) {
	return $env{"course.$courseid.rndseed"};
    }
    return &latest_rnd_algorithm_id();
}

sub validCODE {
    my ($CODE)=@_;
    if (defined($CODE) && $CODE ne '' && $CODE =~ /^\w+$/) { return 1; }
    return 0;
}

sub getCODE {
    if (&validCODE($env{'form.CODE'})) { return $env{'form.CODE'}; }
    if ( (defined($Apache::lonhomework::parsing_a_problem) ||
	  defined($Apache::lonhomework::parsing_a_task) ) &&
	 &validCODE($Apache::lonhomework::history{'resource.CODE'})) {
	return $Apache::lonhomework::history{'resource.CODE'};
    }
    return undef;
}

sub rndseed {
    my ($symb,$courseid,$domain,$username)=@_;
    my ($wsymb,$wcourseid,$wdomain,$wusername)=&whichuser();
    if (!defined($symb)) {
	unless ($symb=$wsymb) { return time; }
    }
    if (!$courseid) { $courseid=$wcourseid; }
    if (!$domain) { $domain=$wdomain; }
    if (!$username) { $username=$wusername }
    my $which=&get_rand_alg();

    if (defined(&getCODE())) {
	if ($which eq '64bit5') {
	    return &rndseed_CODE_64bit5($symb,$courseid,$domain,$username);
	} elsif ($which eq '64bit4') {
	    return &rndseed_CODE_64bit4($symb,$courseid,$domain,$username);
	} else {
	    return &rndseed_CODE_64bit($symb,$courseid,$domain,$username);
	}
    } elsif ($which eq '64bit5') {
	return &rndseed_64bit5($symb,$courseid,$domain,$username);
    } elsif ($which eq '64bit4') {
	return &rndseed_64bit4($symb,$courseid,$domain,$username);
    } elsif ($which eq '64bit3') {
	return &rndseed_64bit3($symb,$courseid,$domain,$username);
    } elsif ($which eq '64bit2') {
	return &rndseed_64bit2($symb,$courseid,$domain,$username);
    } elsif ($which eq '64bit') {
	return &rndseed_64bit($symb,$courseid,$domain,$username);
    }
    return &rndseed_32bit($symb,$courseid,$domain,$username);
}

sub rndseed_32bit {
    my ($symb,$courseid,$domain,$username)=@_;
    {
	use integer;
	my $symbchck=unpack("%32C*",$symb) << 27;
	my $symbseed=numval($symb) << 22;
	my $namechck=unpack("%32C*",$username) << 17;
	my $nameseed=numval($username) << 12;
	my $domainseed=unpack("%32C*",$domain) << 7;
	my $courseseed=unpack("%32C*",$courseid);
	my $num=$symbseed+$nameseed+$domainseed+$courseseed+$namechck+$symbchck;
	#&logthis("$symbseed:$nameseed;$domainseed|$courseseed;$namechck:$symbchck");
	#&logthis("rndseed :$num:$symb");
	if ($_64bit) { $num=(($num<<32)>>32); }
	return $num;
    }
}

sub rndseed_64bit {
    my ($symb,$courseid,$domain,$username)=@_;
    {
	use integer;
	my $symbchck=unpack("%32S*",$symb) << 21;
	my $symbseed=numval($symb) << 10;
	my $namechck=unpack("%32S*",$username);
	
	my $nameseed=numval($username) << 21;
	my $domainseed=unpack("%32S*",$domain) << 10;
	my $courseseed=unpack("%32S*",$courseid);
	
	my $num1=$symbchck+$symbseed+$namechck;
	my $num2=$nameseed+$domainseed+$courseseed;
	#&logthis("$symbseed:$nameseed;$domainseed|$courseseed;$namechck:$symbchck");
	#&logthis("rndseed :$num:$symb");
	if ($_64bit) { $num1=(($num1<<32)>>32); $num2=(($num2<<32)>>32); }
	return "$num1,$num2";
    }
}

sub rndseed_64bit2 {
    my ($symb,$courseid,$domain,$username)=@_;
    {
	use integer;
	# strings need to be an even # of cahracters long, it it is odd the
        # last characters gets thrown away
	my $symbchck=unpack("%32S*",$symb.' ') << 21;
	my $symbseed=numval($symb) << 10;
	my $namechck=unpack("%32S*",$username.' ');
	
	my $nameseed=numval($username) << 21;
	my $domainseed=unpack("%32S*",$domain.' ') << 10;
	my $courseseed=unpack("%32S*",$courseid.' ');
	
	my $num1=$symbchck+$symbseed+$namechck;
	my $num2=$nameseed+$domainseed+$courseseed;
	#&logthis("$symbseed:$nameseed;$domainseed|$courseseed;$namechck:$symbchck");
	#&logthis("rndseed :$num:$symb");
	if ($_64bit) { $num1=(($num1<<32)>>32); $num2=(($num2<<32)>>32); }
	return "$num1,$num2";
    }
}

sub rndseed_64bit3 {
    my ($symb,$courseid,$domain,$username)=@_;
    {
	use integer;
	# strings need to be an even # of cahracters long, it it is odd the
        # last characters gets thrown away
	my $symbchck=unpack("%32S*",$symb.' ') << 21;
	my $symbseed=numval2($symb) << 10;
	my $namechck=unpack("%32S*",$username.' ');
	
	my $nameseed=numval2($username) << 21;
	my $domainseed=unpack("%32S*",$domain.' ') << 10;
	my $courseseed=unpack("%32S*",$courseid.' ');
	
	my $num1=$symbchck+$symbseed+$namechck;
	my $num2=$nameseed+$domainseed+$courseseed;
	#&logthis("$symbseed:$nameseed;$domainseed|$courseseed;$namechck:$symbchck");
	#&logthis("rndseed :$num1:$num2:$_64bit");
	if ($_64bit) { $num1=(($num1<<32)>>32); $num2=(($num2<<32)>>32); }
	
	return "$num1:$num2";
    }
}

sub rndseed_64bit4 {
    my ($symb,$courseid,$domain,$username)=@_;
    {
	use integer;
	# strings need to be an even # of cahracters long, it it is odd the
        # last characters gets thrown away
	my $symbchck=unpack("%32S*",$symb.' ') << 21;
	my $symbseed=numval3($symb) << 10;
	my $namechck=unpack("%32S*",$username.' ');
	
	my $nameseed=numval3($username) << 21;
	my $domainseed=unpack("%32S*",$domain.' ') << 10;
	my $courseseed=unpack("%32S*",$courseid.' ');
	
	my $num1=$symbchck+$symbseed+$namechck;
	my $num2=$nameseed+$domainseed+$courseseed;
	#&logthis("$symbseed:$nameseed;$domainseed|$courseseed;$namechck:$symbchck");
	#&logthis("rndseed :$num1:$num2:$_64bit");
	if ($_64bit) { $num1=(($num1<<32)>>32); $num2=(($num2<<32)>>32); }
	
	return "$num1:$num2";
    }
}

sub rndseed_64bit5 {
    my ($symb,$courseid,$domain,$username)=@_;
    my ($num1,$num2)=&digest("$symb,$courseid,$domain,$username");
    return "$num1:$num2";
}

sub rndseed_CODE_64bit {
    my ($symb,$courseid,$domain,$username)=@_;
    {
	use integer;
	my $symbchck=unpack("%32S*",$symb.' ') << 16;
	my $symbseed=numval2($symb);
	my $CODEchck=unpack("%32S*",&getCODE().' ') << 16;
	my $CODEseed=numval(&getCODE());
	my $courseseed=unpack("%32S*",$courseid.' ');
	my $num1=$symbseed+$CODEchck;
	my $num2=$CODEseed+$courseseed+$symbchck;
	#&logthis("$symbseed:$CODEchck|$CODEseed:$courseseed:$symbchck");
	#&logthis("rndseed :$num1:$num2:$symb");
	if ($_64bit) { $num1=(($num1<<32)>>32); }
	if ($_64bit) { $num2=(($num2<<32)>>32); }
	return "$num1:$num2";
    }
}

sub rndseed_CODE_64bit4 {
    my ($symb,$courseid,$domain,$username)=@_;
    {
	use integer;
	my $symbchck=unpack("%32S*",$symb.' ') << 16;
	my $symbseed=numval3($symb);
	my $CODEchck=unpack("%32S*",&getCODE().' ') << 16;
	my $CODEseed=numval3(&getCODE());
	my $courseseed=unpack("%32S*",$courseid.' ');
	my $num1=$symbseed+$CODEchck;
	my $num2=$CODEseed+$courseseed+$symbchck;
	#&logthis("$symbseed:$CODEchck|$CODEseed:$courseseed:$symbchck");
	#&logthis("rndseed :$num1:$num2:$symb");
	if ($_64bit) { $num1=(($num1<<32)>>32); }
	if ($_64bit) { $num2=(($num2<<32)>>32); }
	return "$num1:$num2";
    }
}

sub rndseed_CODE_64bit5 {
    my ($symb,$courseid,$domain,$username)=@_;
    my $code = &getCODE();
    my ($num1,$num2)=&digest("$symb,$courseid,$code");
    return "$num1:$num2";
}

sub setup_random_from_rndseed {
    my ($rndseed)=@_;
    if ($rndseed =~/([,:])/) {
	my ($num1,$num2)=split(/[,:]/,$rndseed);
	&Math::Random::random_set_seed(abs($num1),abs($num2));
    } else {
	&Math::Random::random_set_seed_from_phrase($rndseed);
    }
}

sub latest_receipt_algorithm_id {
    return 'receipt3';
}

sub recunique {
    my $fucourseid=shift;
    my $unique;
    if ($env{"course.$fucourseid.receiptalg"} eq 'receipt2' ||
	$env{"course.$fucourseid.receiptalg"} eq 'receipt3' ) {
	$unique=$env{"course.$fucourseid.internal.encseed"};
    } else {
	$unique=$perlvar{'lonReceipt'};
    }
    return unpack("%32C*",$unique);
}

sub recprefix {
    my $fucourseid=shift;
    my $prefix;
    if ($env{"course.$fucourseid.receiptalg"} eq 'receipt2'||
	$env{"course.$fucourseid.receiptalg"} eq 'receipt3' ) {
	$prefix=$env{"course.$fucourseid.internal.encpref"};
    } else {
	$prefix=$perlvar{'lonHostID'};
    }
    return unpack("%32C*",$prefix);
}

sub ireceipt {
    my ($funame,$fudom,$fucourseid,$fusymb,$part)=@_;

    my $return =&recprefix($fucourseid).'-';

    if ($env{"course.$fucourseid.receiptalg"} eq 'receipt3' ||
	$env{'request.state'} eq 'construct') {
	$return .= (&digest("$funame,$fudom,$fucourseid,$fusymb,$part")%10000);
	return $return;
    }

    my $cuname=unpack("%32C*",$funame);
    my $cudom=unpack("%32C*",$fudom);
    my $cucourseid=unpack("%32C*",$fucourseid);
    my $cusymb=unpack("%32C*",$fusymb);
    my $cunique=&recunique($fucourseid);
    my $cpart=unpack("%32S*",$part);
    if ($env{"course.$fucourseid.receiptalg"} eq 'receipt2') {

	#&logthis("doing receipt2  using parts $cpart, uname $cuname and udom $cudom gets  ".($cpart%$cuname)." and ".($cpart%$cudom));
			       
	$return.= ($cunique%$cuname+
		   $cunique%$cudom+
		   $cusymb%$cuname+
		   $cusymb%$cudom+
		   $cucourseid%$cuname+
		   $cucourseid%$cudom+
		   $cpart%$cuname+
		   $cpart%$cudom);
    } else {
	$return.= ($cunique%$cuname+
		   $cunique%$cudom+
		   $cusymb%$cuname+
		   $cusymb%$cudom+
		   $cucourseid%$cuname+
		   $cucourseid%$cudom);
    }
    return $return;
}

sub receipt {
    my ($part)=@_;
    my ($symb,$courseid,$domain,$name) = &whichuser();
    return &ireceipt($name,$domain,$courseid,$symb,$part);
}

sub whichuser {
    my ($passedsymb)=@_;
    my ($symb,$courseid,$domain,$name,$publicuser);
    if (defined($env{'form.grade_symb'})) {
	my ($tmp_courseid)=&get_env_multiple('form.grade_courseid');
	my $allowed=&allowed('vgr',$tmp_courseid);
	if (!$allowed &&
	    exists($env{'request.course.sec'}) &&
	    $env{'request.course.sec'} !~ /^\s*$/) {
	    $allowed=&allowed('vgr',$tmp_courseid.
			      '/'.$env{'request.course.sec'});
	}
	if ($allowed) {
	    ($symb)=&get_env_multiple('form.grade_symb');
	    $courseid=$tmp_courseid;
	    ($domain)=&get_env_multiple('form.grade_domain');
	    ($name)=&get_env_multiple('form.grade_username');
	    return ($symb,$courseid,$domain,$name,$publicuser);
	}
    }
    if (!$passedsymb) {
	$symb=&symbread();
    } else {
	$symb=$passedsymb;
    }
    $courseid=$env{'request.course.id'};
    $domain=$env{'user.domain'};
    $name=$env{'user.name'};
    if ($name eq 'public' && $domain eq 'public') {
	if (!defined($env{'form.username'})) {
	    $env{'form.username'}.=time.rand(10000000);
	}
	$name.=$env{'form.username'};
    }
    return ($symb,$courseid,$domain,$name,$publicuser);

}

# ------------------------------------------------------------ Serves up a file
# returns either the contents of the file or 
# -1 if the file doesn't exist
#
# if the target is a file that was uploaded via DOCS, 
# a check will be made to see if a current copy exists on the local server,
# if it does this will be served, otherwise a copy will be retrieved from
# the home server for the course and stored in /home/httpd/html/userfiles on
# the local server.   

sub getfile {
    my ($file) = @_;
    if ($file =~ m -^/*(uploaded|editupload)/-) { $file=&filelocation("",$file); }
    &repcopy($file);
    return &readfile($file);
}

sub repcopy_userfile {
    my ($file)=@_;
    if ($file =~ m -^/*(uploaded|editupload)/-) { $file=&filelocation("",$file); }
    if ($file =~ m|^/home/httpd/html/lonUsers/|) { return 'ok'; }
    my ($cdom,$cnum,$filename) = 
	($file=~m|^\Q$perlvar{'lonDocRoot'}\E/+userfiles/+($match_domain)/+($match_name)/+(.*)|);
    my $uri="/uploaded/$cdom/$cnum/$filename";
    if (-e "$file") {
# we already have a local copy, check it out
	my @fileinfo = stat($file);
	my $rtncode;
	my $info;
	my $lwpresp = &getuploaded('HEAD',$uri,$cdom,$cnum,\$info,\$rtncode);
	if ($lwpresp ne 'ok') {
# there is no such file anymore, even though we had a local copy
	    if ($rtncode eq '404') {
		unlink($file);
	    }
	    return -1;
	}
	if ($info < $fileinfo[9]) {
# nice, the file we have is up-to-date, just say okay
	    return 'ok';
	} else {
# the file is outdated, get rid of it
	    unlink($file);
	}
    }
# one way or the other, at this point, we don't have the file
# construct the correct path for the file
    my @parts = ($cdom,$cnum); 
    if ($filename =~ m|^(.+)/[^/]+$|) {
	push @parts, split(/\//,$1);
    }
    my $path = $perlvar{'lonDocRoot'}.'/userfiles';
    foreach my $part (@parts) {
	$path .= '/'.$part;
	if (!-e $path) {
	    mkdir($path,0770);
	}
    }
# now the path exists for sure
# get a user agent
    my $ua=new LWP::UserAgent;
    my $transferfile=$file.'.in.transfer';
# FIXME: this should flock
    if (-e $transferfile) { return 'ok'; }
    my $request;
    $uri=~s/^\///;
    my $homeserver = &homeserver($cnum,$cdom);
    my $protocol = $protocol{$homeserver};
    $protocol = 'http' if ($protocol ne 'https');
    $request=new HTTP::Request('GET',$protocol.'://'.&hostname($homeserver).'/raw/'.$uri);
    my $response=$ua->request($request,$transferfile);
# did it work?
    if ($response->is_error()) {
	unlink($transferfile);
	&logthis("Userfile repcopy failed for $uri");
	return -1;
    }
# worked, rename the transfer file
    rename($transferfile,$file);
    return 'ok';
}

sub tokenwrapper {
    my $uri=shift;
    $uri=~s|^https?\://([^/]+)||;
    $uri=~s|^/||;
    $env{'user.environment'}=~/\/([^\/]+)\.id/;
    my $token=$1;
    my (undef,$udom,$uname,$file)=split('/',$uri,4);
    if ($udom && $uname && $file) {
	$file=~s|(\?\.*)*$||;
        &appenv({"userfile.$udom/$uname/$file" => $env{'request.course.id'}});
        my $homeserver = &homeserver($uname,$udom);
        my $protocol = $protocol{$homeserver};
        $protocol = 'http' if ($protocol ne 'https');
        return $protocol.'://'.&hostname($homeserver).'/'.$uri.
               (($uri=~/\?/)?'&':'?').'token='.$token.
                               '&tokenissued='.$perlvar{'lonHostID'};
    } else {
        return '/adm/notfound.html';
    }
}

# call with reqtype HEAD: get last modification time
# call with reqtype GET: get the file contents
# Do not call this with reqtype GET for large files! It loads everything into memory
#
sub getuploaded {
    my ($reqtype,$uri,$cdom,$cnum,$info,$rtncode) = @_;
    $uri=~s/^\///;
    my $homeserver = &homeserver($cnum,$cdom);
    my $protocol = $protocol{$homeserver};
    $protocol = 'http' if ($protocol ne 'https');
    $uri = $protocol.'://'.&hostname($homeserver).'/raw/'.$uri;
    my $ua=new LWP::UserAgent;
    my $request=new HTTP::Request($reqtype,$uri);
    my $response=$ua->request($request);
    $$rtncode = $response->code;
    if (! $response->is_success()) {
	return 'failed';
    }      
    if ($reqtype eq 'HEAD') {
	$$info = &HTTP::Date::str2time( $response->header('Last-modified') );
    } elsif ($reqtype eq 'GET') {
	$$info = $response->content;
    }
    return 'ok';
}

sub readfile {
    my $file = shift;
    if ( (! -e $file ) || ($file eq '') ) { return -1; };
    my $fh;
    open($fh,"<$file");
    my $a='';
    while (my $line = <$fh>) { $a .= $line; }
    return $a;
}

sub filelocation {
    my ($dir,$file) = @_;
    my $location;
    $file=~ s/^\s*(\S+)\s*$/$1/; ## strip off leading and trailing spaces

    if ($file =~ m-^/adm/-) {
	$file=~s-^/adm/wrapper/-/-;
	$file=~s-^/adm/coursedocs/showdoc/-/-;
    }

    if ($file=~m:^/~:) { # is a contruction space reference
        $location = $file;
        $location =~ s:/~(.*?)/(.*):/home/$1/public_html/$2:;
    } elsif ($file=~m{^/home/$match_username/public_html/}) {
	# is a correct contruction space reference
        $location = $file;
    } elsif ($file =~ m-^\Q$Apache::lonnet::perlvar{'lonTabDir'}\E/-) {
        $location = $file;
    } elsif ($file=~/^\/*(uploaded|editupload)/) { # is an uploaded file
        my ($udom,$uname,$filename)=
  	    ($file=~m -^/+(?:uploaded|editupload)/+($match_domain)/+($match_name)/+(.*)$-);
        my $home=&homeserver($uname,$udom);
        my $is_me=0;
        my @ids=&current_machine_ids();
        foreach my $id (@ids) { if ($id eq $home) { $is_me=1; } }
        if ($is_me) {
  	    $location=&propath($udom,$uname).'/userfiles/'.$filename;
        } else {
  	  $location=$Apache::lonnet::perlvar{'lonDocRoot'}.'/userfiles/'.
  	      $udom.'/'.$uname.'/'.$filename;
        }
    } elsif ($file =~ m-^/adm/-) {
	$location = $perlvar{'lonDocRoot'}.'/'.$file;
    } else {
        $file=~s/^\Q$perlvar{'lonDocRoot'}\E//;
        $file=~s:^/res/:/:;
        if ( !( $file =~ m:^/:) ) {
            $location = $dir. '/'.$file;
        } else {
            $location = '/home/httpd/html/res'.$file;
        }
    }
    $location=~s://+:/:g; # remove duplicate /
    while ($location=~m{/\.\./}) {
	if ($location =~ m{/[^/]+/\.\./}) {
	    $location=~ s{/[^/]+/\.\./}{/}g;
	} else {
	    $location=~ s{/\.\./}{/}g;
	}
    } #remove dir/..
    while ($location=~m:/\./:) {$location=~ s:/\./:/:g;} #remove /./
    return $location;
}

sub hreflocation {
    my ($dir,$file)=@_;
    unless (($file=~m-^https?\://-i) || ($file=~m-^/-)) {
	$file=filelocation($dir,$file);
    } elsif ($file=~m-^/adm/-) {
	$file=~s-^/adm/wrapper/-/-;
	$file=~s-^/adm/coursedocs/showdoc/-/-;
    }
    if ($file=~m-^\Q$perlvar{'lonDocRoot'}\E-) {
	$file=~s-^\Q$perlvar{'lonDocRoot'}\E--;
    } elsif ($file=~m-/home/($match_username)/public_html/-) {
	$file=~s-^/home/($match_username)/public_html/-/~$1/-;
    } elsif ($file=~m-^\Q$perlvar{'lonUsersDir'}\E-) {
	$file=~s-^/home/httpd/lonUsers/($match_domain)/./././($match_name)/userfiles/
	    -/uploaded/$1/$2/-x;
    }
    if ($file=~ m{^/userfiles/}) {
	$file =~ s{^/userfiles/}{/uploaded/};
    }
    return $file;
}

sub current_machine_domains {
    return &machine_domains(&hostname($perlvar{'lonHostID'}));
}

sub machine_domains {
    my ($hostname) = @_;
    my @domains;
    my %hostname = &all_hostnames();
    while( my($id, $name) = each(%hostname)) {
#	&logthis("-$id-$name-$hostname-");
	if ($hostname eq $name) {
	    push(@domains,&host_domain($id));
	}
    }
    return @domains;
}

sub current_machine_ids {
    return &machine_ids(&hostname($perlvar{'lonHostID'}));
}

sub machine_ids {
    my ($hostname) = @_;
    $hostname ||= &hostname($perlvar{'lonHostID'});
    my @ids;
    my %name_to_host = &all_names();
    if (ref($name_to_host{$hostname}) eq 'ARRAY') {
	return @{ $name_to_host{$hostname} };
    }
    return;
}

sub additional_machine_domains {
    my @domains;
    open(my $fh,"<$perlvar{'lonTabDir'}/expected_domains.tab");
    while( my $line = <$fh>) {
        $line =~ s/\s//g;
        push(@domains,$line);
    }
    return @domains;
}

sub default_login_domain {
    my $domain = $perlvar{'lonDefDomain'};
    my $testdomain=(split(/\./,$ENV{'HTTP_HOST'}))[0];
    foreach my $posdom (&current_machine_domains(),
                        &additional_machine_domains()) {
        if (lc($posdom) eq lc($testdomain)) {
            $domain=$posdom;
            last;
        }
    }
    return $domain;
}

# ------------------------------------------------------------- Declutters URLs

sub declutter {
    my $thisfn=shift;
    if ($thisfn=~m|^/enc/|) { $thisfn=&Apache::lonenc::unencrypted($thisfn); }
    $thisfn=~s/^\Q$perlvar{'lonDocRoot'}\E//;
    $thisfn=~s/^\///;
    $thisfn=~s|^adm/wrapper/||;
    $thisfn=~s|^adm/coursedocs/showdoc/||;
    $thisfn=~s/^res\///;
    unless (($thisfn =~ /^ext/) || ($thisfn =~ /\.(page|sequence)___\d+___ext/)) {
        $thisfn=~s/\?.+$//;
    }
    return $thisfn;
}

# ------------------------------------------------------------- Clutter up URLs

sub clutter {
    my $thisfn='/'.&declutter(shift);
    if ($thisfn !~ m{^/(uploaded|editupload|adm|userfiles|ext|raw|priv|public)/}
	|| $thisfn =~ m{^/adm/(includes|pages)} ) { 
       $thisfn='/res'.$thisfn; 
    }
    if ($thisfn !~m|^/adm|) {
	if ($thisfn =~ m|^/ext/|) {
	    $thisfn='/adm/wrapper'.$thisfn;
	} else {
	    my ($ext) = ($thisfn =~ /\.(\w+)$/);
	    my $embstyle=&Apache::loncommon::fileembstyle($ext);
	    if ($embstyle eq 'ssi'
		|| ($embstyle eq 'hdn')
		|| ($embstyle eq 'rat')
		|| ($embstyle eq 'prv')
		|| ($embstyle eq 'ign')) {
		#do nothing with these
	    } elsif (($embstyle eq 'img') 
		|| ($embstyle eq 'emb')
		|| ($embstyle eq 'wrp')) {
		$thisfn='/adm/wrapper'.$thisfn;
	    } elsif ($embstyle eq 'unk'
		     && $thisfn!~/\.(sequence|page)$/) {
		$thisfn='/adm/coursedocs/showdoc'.$thisfn;
	    } else {
#		&logthis("Got a blank emb style");
	    }
	}
    }
    return $thisfn;
}

sub clutter_with_no_wrapper {
    my $uri = &clutter(shift);
    if ($uri =~ m-^/adm/-) {
	$uri =~ s-^/adm/wrapper/-/-;
	$uri =~ s-^/adm/coursedocs/showdoc/-/-;
    }
    return $uri;
}

sub freeze_escape {
    my ($value)=@_;
    if (ref($value)) {
	$value=&nfreeze($value);
	return '__FROZEN__'.&escape($value);
    }
    return &escape($value);
}


sub thaw_unescape {
    my ($value)=@_;
    if ($value =~ /^__FROZEN__/) {
	substr($value,0,10,undef);
	$value=&unescape($value);
	return &thaw($value);
    }
    return &unescape($value);
}

sub correct_line_ends {
    my ($result)=@_;
    $$result =~s/\r\n/\n/mg;
    $$result =~s/\r/\n/mg;
}
# ================================================================ Main Program

sub goodbye {
   &logthis("Starting Shut down");
#not converted to using infrastruture and probably shouldn't be
   &logthis(sprintf("%-20s is %s",'%badServerCache',length(&nfreeze(\%badServerCache))));
#converted
#   &logthis(sprintf("%-20s is %s",'%metacache',scalar(%metacache)));
   &logthis(sprintf("%-20s is %s",'%homecache',length(&nfreeze(\%homecache))));
#   &logthis(sprintf("%-20s is %s",'%titlecache',length(&nfreeze(\%titlecache))));
#   &logthis(sprintf("%-20s is %s",'%courseresdatacache',length(&nfreeze(\%courseresdatacache))));
#1.1 only
#   &logthis(sprintf("%-20s is %s",'%userresdatacache',length(&nfreeze(\%userresdatacache))));
#   &logthis(sprintf("%-20s is %s",'%getsectioncache',length(&nfreeze(\%getsectioncache))));
#   &logthis(sprintf("%-20s is %s",'%courseresversioncache',length(&nfreeze(\%courseresversioncache))));
#   &logthis(sprintf("%-20s is %s",'%resversioncache',length(&nfreeze(\%resversioncache))));
   &logthis(sprintf("%-20s is %s",'%remembered',length(&nfreeze(\%remembered))));
   &logthis(sprintf("%-20s is %s",'kicks',$kicks));
   &logthis(sprintf("%-20s is %s",'hits',$hits));
   &flushcourselogs();
   &logthis("Shutting down");
}

sub get_dns {
    my ($url,$func,$ignore_cache) = @_;
    if (!$ignore_cache) {
	my ($content,$cached)=
	    &Apache::lonnet::is_cached_new('dns',$url);
	if ($cached) {
	    &$func($content);
	    return;
	}
    }

    my %alldns;
    open(my $config,"<$perlvar{'lonTabDir'}/hosts.tab");
    foreach my $dns (<$config>) {
	next if ($dns !~ /^\^(\S*)/x);
        my $line = $1;
        my ($host,$protocol) = split(/:/,$line);
        if ($protocol ne 'https') {
            $protocol = 'http';
        }
	$alldns{$host} = $protocol;
    }
    while (%alldns) {
	my ($dns) = keys(%alldns);
	my $ua=new LWP::UserAgent;
	my $request=new HTTP::Request('GET',"$alldns{$dns}://$dns$url");
	my $response=$ua->request($request);
        delete($alldns{$dns});
	next if ($response->is_error());
	my @content = split("\n",$response->content);
	&Apache::lonnet::do_cache_new('dns',$url,\@content,30*24*60*60);
	&$func(\@content);
	return;
    }
    close($config);
    my $which = (split('/',$url))[3];
    &logthis("unable to contact DNS defaulting to on disk file dns_$which.tab\n");
    open($config,"<$perlvar{'lonTabDir'}/dns_$which.tab");
    my @content = <$config>;
    &$func(\@content);
    return;
}
# ------------------------------------------------------------ Read domain file
{
    my $loaded;
    my %domain;

    sub parse_domain_tab {
	my ($lines) = @_;
	foreach my $line (@$lines) {
	    next if ($line =~ /^(\#|\s*$ )/x);

	    chomp($line);
	    my ($name,@elements) = split(/:/,$line,9);
	    my %this_domain;
	    foreach my $field ('description', 'auth_def', 'auth_arg_def',
			       'lang_def', 'city', 'longi', 'lati',
			       'primary') {
		$this_domain{$field} = shift(@elements);
	    }
	    $domain{$name} = \%this_domain;
	}
    }

    sub reset_domain_info {
	undef($loaded);
	undef(%domain);
    }

    sub load_domain_tab {
	my ($ignore_cache) = @_;
	&get_dns('/adm/dns/domain',\&parse_domain_tab,$ignore_cache);
	my $fh;
	if (open($fh,"<".$perlvar{'lonTabDir'}.'/domain.tab')) {
	    my @lines = <$fh>;
	    &parse_domain_tab(\@lines);
	}
	close($fh);
	$loaded = 1;
    }

    sub domain {
	&load_domain_tab() if (!$loaded);

	my ($name,$what) = @_;
	return if ( !exists($domain{$name}) );

	if (!$what) {
	    return $domain{$name}{'description'};
	}
	return $domain{$name}{$what};
    }

    sub domain_info {
        &load_domain_tab() if (!$loaded);
        return %domain;
    }

}


# ------------------------------------------------------------- Read hosts file
{
    my %hostname;
    my %hostdom;
    my %libserv;
    my $loaded;
    my %name_to_host;

    sub parse_hosts_tab {
	my ($file) = @_;
	foreach my $configline (@$file) {
	    next if ($configline =~ /^(\#|\s*$ )/x);
	    next if ($configline =~ /^\^/);
	    chomp($configline);
	    my ($id,$domain,$role,$name,$protocol)=split(/:/,$configline);
	    $name=~s/\s//g;
	    if ($id && $domain && $role && $name) {
		$hostname{$id}=$name;
		push(@{$name_to_host{$name}}, $id);
		$hostdom{$id}=$domain;
		if ($role eq 'library') { $libserv{$id}=$name; }
                if (defined($protocol)) {
                    if ($protocol eq 'https') {
                        $protocol{$id} = $protocol;
                    } else {
                        $protocol{$id} = 'http'; 
                    }
                } else {
                    $protocol{$id} = 'http';
                }
	    }
	}
    }
    
    sub reset_hosts_info {
	&purge_remembered();
	&reset_domain_info();
	&reset_hosts_ip_info();
	undef(%name_to_host);
	undef(%hostname);
	undef(%hostdom);
	undef(%libserv);
	undef($loaded);
    }

    sub load_hosts_tab {
	my ($ignore_cache) = @_;
	&get_dns('/adm/dns/hosts',\&parse_hosts_tab,$ignore_cache);
	open(my $config,"<$perlvar{'lonTabDir'}/hosts.tab");
	my @config = <$config>;
	&parse_hosts_tab(\@config);
	close($config);
	$loaded=1;
    }

    sub hostname {
	&load_hosts_tab() if (!$loaded);

	my ($lonid) = @_;
	return $hostname{$lonid};
    }

    sub all_hostnames {
	&load_hosts_tab() if (!$loaded);

	return %hostname;
    }

    sub all_names {
	&load_hosts_tab() if (!$loaded);

	return %name_to_host;
    }

    sub all_host_domain {
        &load_hosts_tab() if (!$loaded);
        return %hostdom;
    }

    sub is_library {
	&load_hosts_tab() if (!$loaded);

	return exists($libserv{$_[0]});
    }

    sub all_library {
	&load_hosts_tab() if (!$loaded);

	return %libserv;
    }

    sub get_servers {
	&load_hosts_tab() if (!$loaded);

	my ($domain,$type) = @_;
	my %possible_hosts = ($type eq 'library') ? %libserv
	                                          : %hostname;
	my %result;
	if (ref($domain) eq 'ARRAY') {
	    while ( my ($host,$hostname) = each(%possible_hosts)) {
		if (grep(/^\Q$hostdom{$host}\E$/,@$domain)) {
		    $result{$host} = $hostname;
		}
	    }
	} else {
	    while ( my ($host,$hostname) = each(%possible_hosts)) {
		if ($hostdom{$host} eq $domain) {
		    $result{$host} = $hostname;
		}
	    }
	}
	return %result;
    }

    sub host_domain {
	&load_hosts_tab() if (!$loaded);

	my ($lonid) = @_;
	return $hostdom{$lonid};
    }

    sub all_domains {
	&load_hosts_tab() if (!$loaded);

	my %seen;
	my @uniq = grep(!$seen{$_}++, values(%hostdom));
	return @uniq;
    }
}

{ 
    my %iphost;
    my %name_to_ip;
    my %lonid_to_ip;

    sub get_hosts_from_ip {
	my ($ip) = @_;
	my %iphosts = &get_iphost();
	if (ref($iphosts{$ip})) {
	    return @{$iphosts{$ip}};
	}
	return;
    }
    
    sub reset_hosts_ip_info {
	undef(%iphost);
	undef(%name_to_ip);
	undef(%lonid_to_ip);
    }

    sub get_host_ip {
	my ($lonid) = @_;
	if (exists($lonid_to_ip{$lonid})) {
	    return $lonid_to_ip{$lonid};
	}
	my $name=&hostname($lonid);
   	my $ip = gethostbyname($name);
	return if (!$ip || length($ip) ne 4);
	$ip=inet_ntoa($ip);
	$name_to_ip{$name}   = $ip;
	$lonid_to_ip{$lonid} = $ip;
	return $ip;
    }
    
    sub get_iphost {
	my ($ignore_cache) = @_;

	if (!$ignore_cache) {
	    if (%iphost) {
		return %iphost;
	    }
	    my ($ip_info,$cached)=
		&Apache::lonnet::is_cached_new('iphost','iphost');
	    if ($cached) {
		%iphost      = %{$ip_info->[0]};
		%name_to_ip  = %{$ip_info->[1]};
		%lonid_to_ip = %{$ip_info->[2]};
		return %iphost;
	    }
	}

	# get yesterday's info for fallback
	my %old_name_to_ip;
	my ($ip_info,$cached)=
	    &Apache::lonnet::is_cached_new('iphost','iphost');
	if ($cached) {
	    %old_name_to_ip = %{$ip_info->[1]};
	}

	my %name_to_host = &all_names();
	foreach my $name (keys(%name_to_host)) {
	    my $ip;
	    if (!exists($name_to_ip{$name})) {
		$ip = gethostbyname($name);
		if (!$ip || length($ip) ne 4) {
		    if (defined($old_name_to_ip{$name})) {
			$ip = $old_name_to_ip{$name};
			&logthis("Can't find $name defaulting to old $ip");
		    } else {
			&logthis("Name $name no IP found");
			next;
		    }
		} else {
		    $ip=inet_ntoa($ip);
		}
		$name_to_ip{$name} = $ip;
	    } else {
		$ip = $name_to_ip{$name};
	    }
	    foreach my $id (@{ $name_to_host{$name} }) {
		$lonid_to_ip{$id} = $ip;
	    }
	    push(@{$iphost{$ip}},@{$name_to_host{$name}});
	}
	&Apache::lonnet::do_cache_new('iphost','iphost',
				      [\%iphost,\%name_to_ip,\%lonid_to_ip],
				      48*60*60);

	return %iphost;
    }

    #
    #  Given a DNS returns the loncapa host name for that DNS 
    # 
    sub host_from_dns {
        my ($dns) = @_;
        my @hosts;
        my $ip;

        if (exists($name_to_ip{$dns})) {
            $ip = $name_to_ip{$dns};
        }
        if (!$ip) {
            $ip = gethostbyname($dns); # Initial translation to IP is in net order.
            if (length($ip) == 4) { 
	        $ip   = &IO::Socket::inet_ntoa($ip);
            }
        }
        if ($ip) {
	    @hosts = get_hosts_from_ip($ip);
	    return $hosts[0];
        }
        return undef;
    }

}

BEGIN {

# ----------------------------------- Read loncapa.conf and loncapa_apache.conf
    unless ($readit) {
{
    my $configvars = LONCAPA::Configuration::read_conf('loncapa.conf');
    %perlvar = (%perlvar,%{$configvars});
}


# ------------------------------------------------------ Read spare server file
{
    open(my $config,"<$perlvar{'lonTabDir'}/spare.tab");

    while (my $configline=<$config>) {
       chomp($configline);
       if ($configline) {
	   my ($host,$type) = split(':',$configline,2);
	   if (!defined($type) || $type eq '') { $type = 'default' };
	   push(@{ $spareid{$type} }, $host);
       }
    }
    close($config);
}
# ------------------------------------------------------------ Read permissions
{
    open(my $config,"<$perlvar{'lonTabDir'}/roles.tab");

    while (my $configline=<$config>) {
	chomp($configline);
	if ($configline) {
	    my ($role,$perm)=split(/ /,$configline);
	    if ($perm ne '') { $pr{$role}=$perm; }
	}
    }
    close($config);
}

# -------------------------------------------- Read plain texts for permissions
{
    open(my $config,"<$perlvar{'lonTabDir'}/rolesplain.tab");

    while (my $configline=<$config>) {
	chomp($configline);
	if ($configline) {
	    my ($short,@plain)=split(/:/,$configline);
            %{$prp{$short}} = ();
	    if (@plain > 0) {
                $prp{$short}{'std'} = $plain[0];
                for (my $i=1; $i<@plain; $i++) {
                    $prp{$short}{'alt'.$i} = $plain[$i];  
                }
            }
	}
    }
    close($config);
}

# ---------------------------------------------------------- Read package table
{
    open(my $config,"<$perlvar{'lonTabDir'}/packages.tab");

    while (my $configline=<$config>) {
	if ($configline !~ /\S/ || $configline=~/^#/) { next; }
	chomp($configline);
	my ($short,$plain)=split(/:/,$configline);
	my ($pack,$name)=split(/\&/,$short);
	if ($plain ne '') {
	    $packagetab{$pack.'&'.$name.'&name'}=$name; 
	    $packagetab{$short}=$plain; 
	}
    }
    close($config);
}

# ------------- set up temporary directory
{
    $tmpdir = $perlvar{'lonDaemons'}.'/tmp/';

}

$memcache=new Cache::Memcached({'servers'           => ['127.0.0.1:11211'],
				'compress_threshold'=> 20_000,
 			        });

$processmarker='_'.time.'_'.$perlvar{'lonHostID'};
$dumpcount=0;
$locknum=0;

&logtouch();
&logthis('<font color="yellow">INFO: Read configuration</font>');
$readit=1;
    {
	use integer;
	my $test=(2**32)+1;
	if ($test != 0) { $_64bit=1; } else { $_64bit=0; }
	&logthis(" Detected 64bit platform ($_64bit)");
    }
}
}

1;
__END__

=pod

=head1 NAME

Apache::lonnet - Subroutines to ask questions about things in the network.

=head1 SYNOPSIS

Invoked by other LON-CAPA modules, when they need to talk to or about objects in the network.

 &Apache::lonnet::SUBROUTINENAME(ARGUMENTS);

Common parameters:

=over 4

=item *

$uname : an internal username (if $cname expecting a course Id specifically)

=item *

$udom : a domain (if $cdom expecting a course's domain specifically)

=item *

$symb : a resource instance identifier

=item *

$namespace : the name of a .db file that contains the data needed or
being set.

=back

=head1 OVERVIEW

lonnet provides subroutines which interact with the
lonc/lond (TCP) network layer of LON-CAPA. They can be used to ask
about classes, users, and resources.

For many of these objects you can also use this to store data about
them or modify them in various ways.

=head2 Symbs

To identify a specific instance of a resource, LON-CAPA uses symbols
or "symbs"X<symb>. These identifiers are built from the URL of the
map, the resource number of the resource in the map, and the URL of
the resource itself. The latter is somewhat redundant, but might help
if maps change.

An example is

 msu/korte/parts/part1.sequence___19___msu/korte/tests/part12.problem

The respective map entry is

 <resource id="19" src="/res/msu/korte/tests/part12.problem"
  title="Problem 2">
 </resource>

Symbs are used by the random number generator, as well as to store and
restore data specific to a certain instance of for example a problem.

=head2 Storing And Retrieving Data

X<store()>X<cstore()>X<restore()>Three of the most important functions
in C<lonnet.pm> are C<&Apache::lonnet::cstore()>,
C<&Apache::lonnet:restore()>, and C<&Apache::lonnet::store()>, which
is is the non-critical message twin of cstore. These functions are for
handlers to store a perl hash to a user's permanent data space in an
easy manner, and to retrieve it again on another call. It is expected
that a handler would use this once at the beginning to retrieve data,
and then again once at the end to send only the new data back.

The data is stored in the user's data directory on the user's
homeserver under the ID of the course.

The hash that is returned by restore will have all of the previous
value for all of the elements of the hash.

Example:

 #creating a hash
 my %hash;
 $hash{'foo'}='bar';

 #storing it
 &Apache::lonnet::cstore(\%hash);

 #changing a value
 $hash{'foo'}='notbar';

 #adding a new value
 $hash{'bar'}='foo';
 &Apache::lonnet::cstore(\%hash);

 #retrieving the hash
 my %history=&Apache::lonnet::restore();

 #print the hash
 foreach my $key (sort(keys(%history))) {
   print("\%history{$key} = $history{$key}");
 }

Will print out:

 %history{1:foo} = bar
 %history{1:keys} = foo:timestamp
 %history{1:timestamp} = 990455579
 %history{2:bar} = foo
 %history{2:foo} = notbar
 %history{2:keys} = foo:bar:timestamp
 %history{2:timestamp} = 990455580
 %history{bar} = foo
 %history{foo} = notbar
 %history{timestamp} = 990455580
 %history{version} = 2

Note that the special hash entries C<keys>, C<version> and
C<timestamp> were added to the hash. C<version> will be equal to the
total number of versions of the data that have been stored. The
C<timestamp> attribute will be the UNIX time the hash was
stored. C<keys> is available in every historical section to list which
keys were added or changed at a specific historical revision of a
hash.

B<Warning>: do not store the hash that restore returns directly. This
will cause a mess since it will restore the historical keys as if the
were new keys. I.E. 1:foo will become 1:1:foo etc.

Calling convention:

 my %record=&Apache::lonnet::restore($symb,$courseid,$domain,$uname,$home);
 &Apache::lonnet::cstore(\%newrecord,$symb,$courseid,$domain,$uname,$home);

For more detailed information, see lonnet specific documentation.

=head1 RETURN MESSAGES

=over 4

=item * B<con_lost>: unable to contact remote host

=item * B<con_delayed>: unable to contact remote host, message will be delivered
when the connection is brought back up

=item * B<con_failed>: unable to contact remote host and unable to save message
for later delivery

=item * B<error:>: an error a occurred, a description of the error follows the :

=item * B<no_such_host>: unable to fund a host associated with the user/domain
that was requested

=back

=head1 PUBLIC SUBROUTINES

=head2 Session Environment Functions

=over 4

=item * 
X<appenv()>
B<appenv($hashref,$rolesarrayref)>: the value of %{$hashref} is written to
the user envirnoment file, and will be restored for each access this
user makes during this session, also modifies the %env for the current
process. Optional rolesarrayref - if defined contains a reference to an array
of roles which are exempt from the restriction on modifying user.role entries 
in the user's environment.db and in %env.    

=item *
X<delenv()>
B<delenv($delthis,$regexp)>: removes all items from the session
environment file that begin with $delthis. If the 
optional second arg - $regexp - is true, $delthis is treated as a 
regular expression, otherwise \Q$delthis\E is used. 
The values are also deleted from the current processes %env.

=item * get_env_multiple($name) 

gets $name from the %env hash, it seemlessly handles the cases where multiple
values may be defined and end up as an array ref.

returns an array of values

=back

=head2 User Information

=over 4

=item *
X<queryauthenticate()>
B<queryauthenticate($uname,$udom)>: try to determine user's current 
authentication scheme

=item *
X<authenticate()>
B<authenticate($uname,$upass,$udom)>: try to
authenticate user from domain's lib servers (first use the current
one). C<$upass> should be the users password.

=item *
X<homeserver()>
B<homeserver($uname,$udom)>: find the server which has
the user's directory and files (there must be only one), this caches
the answer, and also caches if there is a borken connection.

=item *
X<idget()>
B<idget($udom,@ids)>: find the usernames behind a list of IDs
(IDs are a unique resource in a domain, there must be only 1 ID per
username, and only 1 username per ID in a specific domain) (returns
hash: id=>name,id=>name)

=item *
X<idrget()>
B<idrget($udom,@unames)>: find the IDs behind a list of
usernames (returns hash: name=>id,name=>id)

=item *
X<idput()>
B<idput($udom,%ids)>: store away a list of names and associated IDs

=item *
X<rolesinit()>
B<rolesinit($udom,$username,$authhost)>: get user privileges

=item *
X<getsection()>
B<getsection($udom,$uname,$cname)>: finds the section of student in the
course $cname, return section name/number or '' for "not in course"
and '-1' for "no section"

=item *
X<userenvironment()>
B<userenvironment($udom,$uname,@what)>: gets the values of the keys
passed in @what from the requested user's environment, returns a hash

=item * 
X<userlog_query()>
B<userlog_query($uname,$udom,%filters)>: retrieves data from a user's
activity.log file. %filters defines filters applied when parsing the
log file. These can be start or end timestamps, or the type of action
- log to look for Login or Logout events, check for Checkin or
Checkout, role for role selection. The response is in the form
timestamp1:hostid1:event1&timestamp2:hostid2:event2 where events are
escaped strings of the action recorded in the activity.log file.

=back

=head2 User Roles

=over 4

=item *

allowed($priv,$uri,$symb,$role) : check for a user privilege; returns codes for allowed actions
 F: full access
 U,I,K: authentication modes (cxx only)
 '': forbidden
 1: user needs to choose course
 2: browse allowed
 A: passphrase authentication needed

=item *

definerole($rolename,$sysrole,$domrole,$courole) : define role; define a custom
role rolename set privileges in format of lonTabs/roles.tab for system, domain,
and course level

=item *

plaintext($short,$type,$cid,$forcedefault) : return value in %prp hash 
(rolesplain.tab); plain text explanation of a user role term.
$type is Course (default) or Community.
If $forcedefault evaluates to true, text returned will be default 
text for $type. Otherwise, if this is a course, the text returned 
will be a custom name for the role (if defined in the course's 
environment).  If no custom name is defined the default is returned.
   
=item *

get_my_roles($uname,$udom,$context,$types,$roles,$roledoms,$withsec) :
All arguments are optional. Returns a hash of a roles, either for
co-author/assistant author roles for a user's Construction Space
(default), or if $context is 'userroles', roles for the user himself,
In the hash, keys are set to colon-separated $uname,$udom,$role, and
(optionally) if $withsec is true, a fourth colon-separated item - $section.
For each key, value is set to colon-separated start and end times for
the role.  If no username and domain are specified, will default to
current user/domain. Types, roles, and roledoms are references to arrays
of role statuses (active, future or previous), roles 
(e.g., cc,in, st etc.) and domains of the roles which can be used
to restrict the list of roles reported. If no array ref is 
provided for types, will default to return only active roles.

=back

=head2 User Modification

=over 4

=item *

assignrole($udom,$uname,$url,$role,$end,$start,$deleteflag,$selfenroll,$context) : assign role; give a role to a
user for the level given by URL.  Optional start and end dates (leave empty
string or zero for "no date")

=item *

changepass($uname,$udom,$currentpass,$newpass,$server) : attempts to
change a users, password, possible return values are: ok,
pwchange_failure, non_authorized, auth_mode_error, unknown_user,
refused

=item *

modifyuserauth($udom,$uname,$umode,$upass) : modify user authentication

=item *

modifyuser($udom,$uname,$uid,$umode,$upass,$first,$middle,$last, $gene,
           $forceid,$desiredhome,$email,$inststatus,$candelete) :

will update user information (firstname,middlename,lastname,generation,
permanentemail), and if forceid is true, student/employee ID also.
A user's institutional affiliation(s) can also be updated.
User information fields will not be overwritten with empty entries
unless the field is included in the $candelete array reference.
This array is included when a single user is modified via "Manage Users",
or when Autoupdate.pl is run by cron in a domain.

=item *

modifystudent

modify a student's enrollment and identification information.
The course id is resolved based on the current users environment.  
This means the envoking user must be a course coordinator or otherwise
associated with a course.

This call is essentially a wrapper for lonnet::modifyuser and
lonnet::modify_student_enrollment

Inputs: 

=over 4

=item B<$udom> Student's loncapa domain

=item B<$uname> Student's loncapa login name

=item B<$uid> Student/Employee ID

=item B<$umode> Student's authentication mode

=item B<$upass> Student's password

=item B<$first> Student's first name

=item B<$middle> Student's middle name

=item B<$last> Student's last name

=item B<$gene> Student's generation

=item B<$usec> Student's section in course

=item B<$end> Unix time of the roles expiration

=item B<$start> Unix time of the roles start date

=item B<$forceid> If defined, allow $uid to be changed

=item B<$desiredhome> server to use as home server for student

=item B<$email> Student's permanent e-mail address

=item B<$type> Type of enrollment (auto or manual)

=item B<$locktype> boolean - enrollment type locked to prevent Autoenroll.pl changing manual to auto    

=item B<$cid> courseID - needed if a course role is assigned by a user whose current role is DC

=item B<$selfenroll> boolean - 1 if user role change occurred via self-enrollment

=item B<$context> role change context (shown in User Management Logs display in a course)

=item B<$inststatus> institutional status of user - : separated string of escaped status types  

=back

=item *

modify_student_enrollment

Change a students enrollment status in a class.  The environment variable
'role.request.course' must be defined for this function to proceed.

Inputs:

=over 4

=item $udom, students domain

=item $uname, students name

=item $uid, students user id

=item $first, students first name

=item $middle

=item $last

=item $gene

=item $usec

=item $end

=item $start

=item $type

=item $locktype

=item $cid

=item $selfenroll

=item $context

=back


=item *

assigncustomrole($udom,$uname,$url,$rdom,$rnam,$rolename,$end,$start) : assign
custom role; give a custom role to a user for the level given by URL.  Specify
name and domain of role author, and role name

=item *

revokerole($udom,$uname,$url,$role) : revoke a role for url

=item *

revokecustomrole($udom,$uname,$url,$role) : revoke a custom role

=back

=head2 Course Infomation

=over 4

=item *

coursedescription($courseid) : returns a hash of information about the
specified course id, including all environment settings for the
course, the description of the course will be in the hash under the
key 'description'

=item *

resdata($name,$domain,$type,@which) : request for current parameter
setting for a specific $type, where $type is either 'course' or 'user',
@what should be a list of parameters to ask about. This routine caches
answers for 5 minutes.

=item *

get_courseresdata($courseid, $domain) : dump the entire course resource
data base, returning a hash that is keyed by the resource name and has
values that are the resource value.  I believe that the timestamps and
versions are also returned.


=back

=head2 Course Modification

=over 4

=item *

writecoursepref($courseid,%prefs) : write preferences (environment
database) for a course

=item *

createcourse($udom,$description,$url,$course_server,$nonstandard,$inst_code,$course_owner,$crstype,$cnum) : make course

=item *

generate_coursenum($udom,$crstype) : get a unique (unused) course number in domain $udom for course type $crstype (Course or Community).

=back

=head2 Resource Subroutines

=over 4

=item *

subscribe($fname) : subscribe to a resource, returns URL if possible (probably should use repcopy instead)

=item *

repcopy($filename) : subscribes to the requested file, and attempts to
replicate from the owning library server, Might return
'unavailable', 'not_found', 'forbidden', 'ok', or
'bad_request', also attempts to grab the metadata for the
resource. Expects the local filesystem pathname
(/home/httpd/html/res/....)

=back

=head2 Resource Information

=over 4

=item *

EXT($varname,$symb,$udom,$uname) : evaluates and returns the value of
a vairety of different possible values, $varname should be a request
string, and the other parameters can be used to specify who and what
one is asking about.

Possible values for $varname are environment.lastname (or other item
from the envirnment hash), user.name (or someother aspect about the
user), resource.0.maxtries (or some other part and parameter of a
resource)

=item *

directcondval($number) : get current value of a condition; reads from a state
string

=item *

condval($condidx) : value of condition index based on state

=item *

metadata($uri,$what,$liburi,$prefix,$depthcount) : request a
resource's metadata, $what should be either a specific key, or either
'keys' (to get a list of possible keys) or 'packages' to get a list of
packages that this resource currently uses, the last 3 arguments are only used internally for recursive metadata.

this function automatically caches all requests

=item *

metadata_query($query,$custom,$customshow) : make a metadata query against the
network of library servers; returns file handle of where SQL and regex results
will be stored for query

=item *

symbread($filename) : return symbolic list entry (filename argument optional);
returns the data handle

=item *

symbverify($symb,$thisfn) : verifies that $symb actually exists and is
a possible symb for the URL in $thisfn, and if is an encryypted
resource that the user accessed using /enc/ returns a 1 on success, 0
on failure, user must be in a course, as it assumes the existance of
the course initial hash, and uses $env('request.course.id'}


=item *

symbclean($symb) : removes versions numbers from a symb, returns the
cleaned symb

=item *

is_on_map($uri) : checks if the $uri is somewhere on the current
course map, user must be in a course for it to work.

=item *

numval($salt) : return random seed value (addend for rndseed)

=item *

rndseed($symb,$courseid,$udom,$uname) : create a random sum; returns
a random seed, all arguments are optional, if they aren't sent it uses the
environment to derive them. Note: if symb isn't sent and it can't get one
from &symbread it will use the current time as its return value

=item *

ireceipt($funame,$fudom,$fucourseid,$fusymb) : return unique,
unfakeable, receipt

=item *

receipt() : API to ireceipt working off of env values; given out to users

=item *

countacc($url) : count the number of accesses to a given URL

=item *

checkout($symb,$tuname,$tudom,$tcrsid) :  creates a record of a user having looked at an item, most likely printed out or otherwise using a resource

=item *

checkin($token) : updates that a resource has beeen returned (a hard copy version for instance) and returns the data that $token was Checkout with ($symb, $tuname, $tudom, and $tcrsid)

=item *

expirespread($uname,$udom,$stype,$usymb) : set expire date for spreadsheet

=item *

devalidate($symb) : devalidate temporary spreadsheet calculations,
forcing spreadsheet to reevaluate the resource scores next time.

=back

=head2 Storing/Retreiving Data

=over 4

=item *

store($storehash,$symb,$namespace,$udom,$uname) : stores hash permanently
for this url; hashref needs to be given and should be a \%hashname; the
remaining args aren't required and if they aren't passed or are '' they will
be derived from the env

=item *

cstore($storehash,$symb,$namespace,$udom,$uname) : same as store but
uses critical subroutine

=item *

restore($symb,$namespace,$udom,$uname) : returns hash for this symb;
all args are optional

=item *

dumpstore($namespace,$udom,$uname,$regexp,$range) : 
dumps the complete (or key matching regexp) namespace into a hash
($udom, $uname, $regexp, $range are optional) for a namespace that is
normally &store()ed into

$range should be either an integer '100' (give me the first 100
                                           matching records)
              or be  two integers sperated by a - with no spaces
                 '30-50' (give me the 30th through the 50th matching
                          records)


=item *

putstore($namespace,$symb,$version,$storehash,$udomain,$uname) :
replaces a &store() version of data with a replacement set of data
for a particular resource in a namespace passed in the $storehash hash 
reference

=item *

tmpstore($storehash,$symb,$namespace,$udom,$uname) : storage that
works very similar to store/cstore, but all data is stored in a
temporary location and can be reset using tmpreset, $storehash should
be a hash reference, returns nothing on success

=item *

tmprestore($symb,$namespace,$udom,$uname) : storage that works very
similar to restore, but all data is stored in a temporary location and
can be reset using tmpreset. Returns a hash of values on success,
error string otherwise.

=item *

tmpreset($symb,$namespace,$udom,$uname) : temporary storage reset,
deltes all keys for $symb form the temporary storage hash.

=item *

get($namespace,$storearr,$udom,$uname) : returns hash with keys from array
reference filled in from namesp ($udom and $uname are optional)

=item *

del($namespace,$storearr,$udom,$uname) : deletes keys out of array from
namesp ($udom and $uname are optional)

=item *

dump($namespace,$udom,$uname,$regexp,$range) : 
dumps the complete (or key matching regexp) namespace into a hash
($udom, $uname, $regexp, $range are optional)

$range should be either an integer '100' (give me the first 100
                                           matching records)
              or be  two integers sperated by a - with no spaces
                 '30-50' (give me the 30th through the 50th matching
                          records)
=item *

inc($namespace,$store,$udom,$uname) : increments $store in $namespace.
$store can be a scalar, an array reference, or if the amount to be 
incremented is > 1, a hash reference.

($udom and $uname are optional)

=item *

put($namespace,$storehash,$udom,$uname) : stores hash in namesp
($udom and $uname are optional)

=item *

cput($namespace,$storehash,$udom,$uname) : critical put
($udom and $uname are optional)

=item *

newput($namespace,$storehash,$udom,$uname) :

Attempts to store the items in the $storehash, but only if they don't
currently exist, if this succeeds you can be certain that you have 
successfully created a new key value pair in the $namespace db.


Args:
 $namespace: name of database to store values to
 $storehash: hashref to store to the db
 $udom: (optional) domain of user containing the db
 $uname: (optional) name of user caontaining the db

Returns:
 'ok' -> succeeded in storing all keys of $storehash
 'key_exists: <key>' -> failed to anything out of $storehash, as at
                        least <key> already existed in the db (other
                        requested keys may also already exist)
 'error: <msg>' -> unable to tie the DB or other error occurred
 'con_lost' -> unable to contact request server
 'refused' -> action was not allowed by remote machine


=item *

eget($namespace,$storearr,$udom,$uname) : returns hash with keys from array
reference filled in from namesp (encrypts the return communication)
($udom and $uname are optional)

=item *

log($udom,$name,$home,$message) : write to permanent log for user; use
critical subroutine

=item *

get_dom($namespace,$storearr,$udom,$uhome) : returns hash with keys from
array reference filled in from namespace found in domain level on either
specified domain server ($uhome) or primary domain server ($udom and $uhome are optional).

=item *

put_dom($namespace,$storehash,$udom,$uhome) :  stores hash in namespace at 
domain level either on specified domain server ($uhome) or primary domain 
server ($udom and $uhome are optional)

=item * 

get_domain_defaults($target_domain) : returns hash with defaults for
authentication and language in the domain. Keys are: auth_def, auth_arg_def,
lang_def; corresponsing values are authentication type (internal, krb4, krb5,
or localauth), initial password or a kerberos realm, language (e.g., en-us).
Values are retrieved from cache (if current), or from domain's configuration.db
(if available), or lastly from values in lonTabs/dns_domain,tab, 
or lonTabs/domain.tab. 

%domdefaults = &get_auth_defaults($target_domain);

=back

=head2 Network Status Functions

=over 4

=item *

dirlist($uri) : return directory list based on URI

=item *

spareserver() : find server with least workload from spare.tab


=item *

host_from_dns($dns) : Returns the loncapa hostname corresponding to a DNS name or undef
if there is no corresponding loncapa host.

=back


=head2 Apache Request

=over 4

=item *

ssi($url,%hash) : server side include, does a complete request cycle on url to
localhost, posts hash

=back

=head2 Data to String to Data

=over 4

=item *

hash2str(%hash) : convert a hash into a string complete with escaping and '='
and '&' separators, supports elements that are arrayrefs and hashrefs

=item *

hashref2str($hashref) : convert a hashref into a string complete with
escaping and '=' and '&' separators, supports elements that are
arrayrefs and hashrefs

=item *

arrayref2str($arrayref) : convert an arrayref into a string complete
with escaping and '&' separators, supports elements that are arrayrefs
and hashrefs

=item *

str2hash($string) : convert string to hash using unescaping and
splitting on '=' and '&', supports elements that are arrayrefs and
hashrefs

=item *

str2array($string) : convert string to hash using unescaping and
splitting on '&', supports elements that are arrayrefs and hashrefs

=back

=head2 Logging Routines

=over 4

These routines allow one to make log messages in the lonnet.log and
lonnet.perm logfiles.

=item *

logtouch() : make sure the logfile, lonnet.log, exists

=item *

logthis() : append message to the normal lonnet.log file, it gets
preiodically rolled over and deleted.

=item *

logperm() : append a permanent message to lonnet.perm.log, this log
file never gets deleted by any automated portion of the system, only
messages of critical importance should go in here.

=back

=head2 General File Helper Routines

=over 4

=item *

getfile($file,$caller) : two cases - requests for files in /res or in /uploaded.
(a) files in /uploaded
  (i) If a local copy of the file exists - 
      compares modification date of local copy with last-modified date for 
      definitive version stored on home server for course. If local copy is 
      stale, requests a new version from the home server and stores it. 
      If the original has been removed from the home server, then local copy 
      is unlinked.
  (ii) If local copy does not exist -
      requests the file from the home server and stores it. 
  
  If $caller is 'uploadrep':  
    This indicates a call from lonuploadrep.pm (PerlHeaderParserHandler phase)
    for request for files originally uploaded via DOCS. 
     - returns 'ok' if fresh local copy now available, -1 otherwise.
  
  Otherwise:
     This indicates a call from the content generation phase of the request.
     -  returns the entire contents of the file or -1.
     
(b) files in /res
   - returns the entire contents of a file or -1; 
   it properly subscribes to and replicates the file if neccessary.


=item *

stat_file($url) : $url is expected to be a /res/ or /uploaded/ style file
                  reference

returns either a stat() list of data about the file or an empty list
if the file doesn't exist or couldn't find out about it (connection
problems or user unknown)

=item *

filelocation($dir,$file) : returns file system location of a file
based on URI; meant to be "fairly clean" absolute reference, $dir is a
directory that relative $file lookups are to looked in ($dir of /a/dir
and a file of ../bob will become /a/bob)

=item *

hreflocation($dir,$file) : returns file system location or a URL; same as
filelocation except for hrefs

=item *

declutter() : declutters URLs (remove docroot, beginning slashes, 'res' etc)

=back

=head2 Usererfile file routines (/uploaded*)

=over 4

=item *

userfileupload(): main rotine for putting a file in a user or course's
                  filespace, arguments are,

 formname - required - this is the name of the element in $env where the
           filename, and the contents of the file to create/modifed exist
           the filename is in $env{'form.'.$formname.'.filename'} and the
           contents of the file is located in $env{'form.'.$formname}
 coursedoc - if true, store the file in the course of the active role
             of the current user
 subdir - required - subdirectory to put the file in under ../userfiles/
         if undefined, it will be placed in "unknown"

 (This routine calls clean_filename() to remove any dangerous
 characters from the filename, and then calls finuserfileupload() to
 complete the transaction)

 returns either the url of the uploaded file (/uploaded/....) if successful
 and /adm/notfound.html if unsuccessful

=item *

clean_filename(): routine for cleaing a filename up for storage in
                 userfile space, argument is:

 filename - proposed filename

returns: the new clean filename

=item *

finishuserfileupload(): routine that creaes and sends the file to
userspace, probably shouldn't be called directly

  docuname: username or courseid of destination for the file
  docudom: domain of user/course of destination for the file
  formname: same as for userfileupload()
  fname: filename (inculding subdirectories) for the file

 returns either the url of the uploaded file (/uploaded/....) if successful
 and /adm/notfound.html if unsuccessful

=item *

renameuserfile(): renames an existing userfile to a new name

  Args:
   docuname: username or courseid of destination for the file
   docudom: domain of user/course of destination for the file
   old: current file name (including any subdirs under userfiles)
   new: desired file name (including any subdirs under userfiles)

=item *

mkdiruserfile(): creates a directory is a userfiles dir

  Args:
   docuname: username or courseid of destination for the file
   docudom: domain of user/course of destination for the file
   dir: dir to create (including any subdirs under userfiles)

=item *

removeuserfile(): removes a file that exists in userfiles

  Args:
   docuname: username or courseid of destination for the file
   docudom: domain of user/course of destination for the file
   fname: filname to delete (including any subdirs under userfiles)

=item *

removeuploadedurl(): convience function for removeuserfile()

  Args:
   url:  a full /uploaded/... url to delete

=item * 

get_portfile_permissions():
  Args:
    domain: domain of user or course contain the portfolio files
    user: name of user or num of course contain the portfolio files
  Returns:
    hashref of a dump of the proper file_permissions.db
   

=item * 

get_access_controls():

Args:
  current_permissions: the hash ref returned from get_portfile_permissions()
  group: (optional) the group you want the files associated with
  file: (optional) the file you want access info on

Returns:
    a hash (keys are file names) of hashes containing
        keys are: path to file/file_name\0uniqueID:scope_end_start (see below)
        values are XML containing access control settings (see below) 

Internal notes:

 access controls are stored in file_permissions.db as key=value pairs.
    key -> path to file/file_name\0uniqueID:scope_end_start
        where scope -> public,guest,course,group,domains or users.
              end -> UNIX time for end of access (0 -> no end date)
              start -> UNIX time for start of access

    value -> XML description of access control
           <scope type=""> (type =1 of: public,guest,course,group,domains,users">
            <start></start>
            <end></end>

            <password></password>  for scope type = guest

            <domain></domain>     for scope type = course or group
            <number></number>
            <roles id="">
             <role></role>
             <access></access>
             <section></section>
             <group></group>
            </roles>

            <dom></dom>         for scope type = domains

            <users>             for scope type = users
             <user>
              <uname></uname>
              <udom></udom>
             </user>
            </users>
           </scope> 
              
 Access data is also aggregated for each file in an additional key=value pair:
 key -> path to file/file_name\0accesscontrol 
 value -> reference to hash
          hash contains key = value pairs
          where key = uniqueID:scope_end_start
                value = UNIX time record was last updated

          Used to improve speed of look-ups of access controls for each file.  
 
 Locks on files (resulting from submission of portfolio file to a homework problem stored in array of arrays.

modify_access_controls():

Modifies access controls for a portfolio file
Args
1. file name
2. reference to hash of required changes,
3. domain
4. username
  where domain,username are the domain of the portfolio owner 
  (either a user or a course) 

Returns:
1. result of additions or updates ('ok' or 'error', with error message). 
2. result of deletions ('ok' or 'error', with error message).
3. reference to hash of any new or updated access controls.
4. reference to hash used to map incoming IDs to uniqueIDs assigned to control.
   key = integer (inbound ID)
   value = uniqueID  

=back

=head2 HTTP Helper Routines

=over 4

=item *

escape() : unpack non-word characters into CGI-compatible hex codes

=item *

unescape() : pack CGI-compatible hex codes into actual non-word ASCII character

=back

=head1 PRIVATE SUBROUTINES

=head2 Underlying communication routines (Shouldn't call)

=over 4

=item *

subreply() : tries to pass a message to lonc, returns con_lost if incapable

=item *

reply() : uses subreply to send a message to remote machine, logs all failures

=item *

critical() : passes a critical message to another server; if cannot
get through then place message in connection buffer directory and
returns con_delayed, if incapable of saving message, returns
con_failed

=item *

reconlonc() : tries to reconnect lonc client processes.

=back

=head2 Resource Access Logging

=over 4

=item *

flushcourselogs() : flush (save) buffer logs and access logs

=item *

courselog($what) : save message for course in hash

=item *

courseacclog($what) : save message for course using &courselog().  Perform
special processing for specific resource types (problems, exams, quizzes, etc).

=item *

goodbye() : flush course logs and log shutting down; it is called in srm.conf
as a PerlChildExitHandler

=back

=head2 Other

=over 4

=item *

symblist($mapname,%newhash) : update symbolic storage links

=back

=cut

