# The LearningOnline Network
# Create a course
#
# $Id: loncreatecourse.pm,v 1.137.4.1 2010/05/24 06:43:32 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###

=head1 NAME

Apache::loncreatecourse.pm

=head1 SYNOPSIS

Allows domain coordinators to create new
courses and assign course coordinators.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 SUBROUTINES

=over

=item handler()

=item print_course_creation_page()

=item create_course()

=item print_intro_page()

=item upload_batchfile()

=item process_batchfile()

=item courserequestbrowser_javascript()

=item print_creation_logs()

=item creation_display_filter()

=item context_names()

=item instcode_srchstr()

=item syllabuslink_javascript()

=back

=cut

package Apache::loncreatecourse;

use strict;
use Apache::Constants qw(:common :http);
use Apache::lonnet;
use Apache::loncommon;
use Apache::londocs;
use Apache::lonlocal;
use Apache::lonuserutils;
use Apache::lonclonecourse;
use LONCAPA::batchcreatecourse;
use LONCAPA;

# ===================================================== Phase one: fill-in form

sub print_course_creation_page {
    my $r=shift;
    my $crstype = 'Community';
    if ($env{'form.phase'} eq 'courseone') {
        $crstype = 'Course';
    }
    my $defdom=$env{'request.role.domain'};
    my %host_servers = &Apache::lonnet::get_servers($defdom,'library');
    my $course_home = '<select name="course_home" size="1">'."\n";
    foreach my $server (sort(keys(%host_servers))) {
        $course_home .= qq{<option value="$server"};
        if ($server eq $Apache::lonnet::perlvar{'lonHostID'}) {
            $course_home .= " selected ";
        }
        $course_home .= qq{>$server $host_servers{$server}</option>};
    }
    $course_home .= "\n</select>\n";
    my $domform = &Apache::loncommon::select_dom_form($defdom,'ccdomain');
    my $cloneform=&Apache::loncommon::select_dom_form
	($env{'request.role.domain'},'clonedomain').
		     &Apache::loncommon::selectcourse_link
	     ('ccrs','clonecourse','clonedomain',undef,undef,undef,$crstype);
    my $coursebrowserjs=&Apache::loncommon::coursebrowser_javascript($env{'request.role.domain'});
    my ($enroll_table,$krbdef,$krbdefdom,$krbform,$intform,$locform,
        $javascript_validations);
    my $starttime = time;
    my $endtime = time+(6*30*24*60*60); # 6 months from now, approx
    my $access_table = &Apache::lonuserutils::date_setting_table($starttime,
                                             $endtime,'create_defaultdates');
    if ($crstype eq 'Course') {
        $enroll_table = &Apache::lonuserutils::date_setting_table($starttime,
                                              $endtime,'create_enrolldates');
        ($krbdef,$krbdefdom) =
        &Apache::loncommon::get_kerberos_defaults($defdom);
        $javascript_validations=&Apache::lonuserutils::javascript_validations(
                                                    'createcourse',$krbdefdom);
        my %param = ( formname      => 'document.ccrs',
                      kerb_def_dom  => $krbdefdom,
                      kerb_def_auth => $krbdef
                    );
        $krbform = &Apache::loncommon::authform_kerberos(%param);
        $intform = &Apache::loncommon::authform_internal(%param);
        $locform = &Apache::loncommon::authform_local(%param);
    } else {
        my $title_alert = &mt('A Community title is required');
        my $coord_alert = &mt('The username of the Coordinator is required');
        $javascript_validations = qq|
function validate(formname) {
    if (formname.title == '') {
        alert("$title_alert");
        return;
    }
    if (formname.ccuname == '') {
        alert("$coord_alert");
    }
    formname.submit();
}
        |;
    }
    my %lt=&Apache::lonlocal::texthash(
		    'cinf' => "Course Information",
                    'ctit' => "Course Title",
                    'chsr' => "Course Home Server",
                    'cidn' => "Course ID/Number",
                    'opt'  => "optional",
                    'iinf' => "Institutional Information",
                    'stat' => "The following entries will be used to identify the course according to the naming scheme adopted by your institution. Your choices will be used to map an internal LON-CAPA course ID to the corresponding course section ID(s) used by the office responsible for providing official class lists for courses at your institution. This mapping is required if you choose to employ automatic population of class lists.",
                    'ccod' => "Course Code",
                    'toin' => "to interface with institutional data, e.g., fs03glg231 for Fall 2003 Geology 231",
                    'snid' => "Section Numbers and corresponding LON-CAPA section IDs",
                    'csli' => "a comma separated list of institutional section numbers, each separated by a colon from the (optional) corresponding section ID to be used in LON-CAPA e.g., 001:1,002:2",
                    'crcs' => "Crosslisted courses",
                    'cscs' => "a comma separated list of course sections crosslisted with the current course, with each entry including the institutional course section name followed by a colon and then the (optional) sectionID to be used in LON-CAPA, e.g., fs03ent231001:ent1,fs03bot231001:bot1,fs03zol231002:zol2",
                    'crco' => "Course Content",
                    'cncr' => "Completely new course",
                    'cecr' => "Clone an existing course", 
                    'map'  => "Map",
                    'smap' => "Select Map",
                    'sacr' => "Do NOT generate as standard course",
                    'ocik' => "only check if you know what you are doing",
                    'fres' => "First Resource",
                    'stco' => "standard courses only",
                    'blnk' => "Blank",
                    'sllb' => "Syllabus",
                    'navi' => "Course Contents",
                    'cid'  => "Course ID",
                    'dmn'  => "Domain",
                    'dsh'  => "Date Shift",
                    'asov' => "Additional settings, if specified below, will override cloned settings",
                    'ncd'  => "Do not clone date parameters",
                    'prd'  => 'Clone date parameters as-is',
                    'shd'  => 'Shift date parameters by number of days',
                    'assp' => "Assessment Parameters",
                    'oaas' => "Open all assessments",
                    'mssg' => "Messaging",
                    'scpf' => "Set course policy feedback to Course Coordinator",
                    'scfc' => "Set content feedback to Course Coordinator",
                    'cmmn' => "Communication",
                    'dsrd' => "Disable student resource discussion",
                    'dsuc' => "Disable student use of chat rooms",
                    'acco' => "Access Control",
                    'snak' => "Students need access key to enter course",
                    'ow'   => "Course Owner",
                    'cc'   => "(will be assigned Course Coordinator role)",
                    'user' => "Username",
                    'aens' => "Automated enrollment settings",
                    'aesc' => "The following settings control automatic enrollment of students in this class based on information available for this specific course from your institution's official classlists.",
                    'aadd' => "Automated adds",
                    'yes'  => "Yes",
                    'no'   => "No",
                    'audr' => "Automated drops",
                    'dacu' => "Duration of automated classlist updates",
                    'dads' => 'Default Access Dates for Students',
                    'dacc' => "Default start and end dates for student access",
                    'psam' => "Authentication mechanism",
                    'pcda' => "Default authentication method for new users added to this domain by the automated enrollment process",
                    'nech' => "Notification of enrollment changes",
                    'nccl' => "Notification to course coordinator via LON-CAPA message when enrollment changes occur during the automated update?",
                    'ndcl' => "Notification to domain coordinator via LON-CAPA message when enrollment changes occur during the automated update?",
                    'irsp' => "Include retrieval of student photographs?",
		    'rshm' => 'Resource Space Home',
                    'cgrs' => "Course Group Settings",
                    'cgrq' => 'Set a quota for the total disk space available for storage of course group portfolio files',
                    'opco' => 'Open Course',
                    'opcm' => 'Open Community',
	       );
    $lt{'kaut'} = &mt('Key authority ([_1]id:domain[_2]) if other than course','<tt>','</tt>');

    if ($crstype eq 'Community') {
        $lt{'cinf'} = &mt('Community Information');
        $lt{'ctit'} = &mt('Community Title');
        $lt{'chsr'} = &mt('Community Home Server');
        $lt{'cidn'} = &mt('Community ID/Number');
        $lt{'crco'} = &mt('Community Content');
        $lt{'cncr'} = &mt('Completely new community');
        $lt{'cecr'} = &mt('Clone an existing community');
        $lt{'sacr'} = &mt('Do NOT generate as standard community');
        $lt{'stco'} = &mt('standard communities only');
        $lt{'navi'} = &mt('Community Contents');
        $lt{'cid'}  = &mt('Community ID');
        $lt{'scpf'} = &mt('Set community policy feedback to Coordinator');
        $lt{'scfc'} = &mt('Set content feedback to Coordinator');
        $lt{'dsrd'} = &mt('Disable member resource discussion');
        $lt{'dsuc'} = &mt('Disable member use of chat rooms');
        $lt{'dads'} = &mt('Default Access Dates for Members');
        $lt{'dacc'} = &mt('Default start and end dates for member access');
        $lt{'snak'} = &mt('Members need access key to enter course');
        $lt{'kaut'} = &mt('Key authority ([_1]id:domain[_2]) if other than community','<tt>','</tt>');
        $lt{'ow'}  = &mt('Community Owner');
        $lt{'cc'}  = &mt('(will be assigned Coordinator role)');
        $lt{'cgrs'} = &mt('Community Group Settings');
        $lt{'cgrq'} = &mt('Set a quota for the total disk space available for storage of community group portfolio files');
    }
    my $js = <<END;
<script type="text/javascript">
var editbrowser = null;
function openbrowser(formname,elementname) {
    var url = '/res/?';
    if (editbrowser == null) {
        url += 'launch=1&';
    }
    url += 'catalogmode=interactive&';
    url += 'mode=edit&';
    url += 'form=' + formname + '&';
    url += 'element=' + elementname + '&';
    url += 'only=sequence' + '';
    var title = 'Browser';
    var options = 'scrollbars=1,resizable=1,menubar=0';
    options += ',width=700,height=600';
    editbrowser = open(url,title,options,'1');
    editbrowser.focus();
}
$javascript_validations
</script>
$coursebrowserjs
END

    my %titles = &Apache::lonlocal::texthash(
                  courseone => 'Create a New Course',
                  groupone => 'Create a New Community',
    );  
    my $start_page = 
        &Apache::loncommon::start_page($titles{$env{'form.phase'}},$js);
    my $end_page = 
        &Apache::loncommon::end_page();
    my $crumbs = 
	&Apache::lonhtmlcommon::breadcrumbs($crstype.' Information',
					    'Create_Course',undef,
					    'Create_Courses');
    $r->print($start_page.$crumbs);

    $r->print('<form action="/adm/createcourse" method="post" name="ccrs"><div class="LC_createcourse">'
             .&Apache::lonhtmlcommon::start_pick_box()

             .&Apache::lonhtmlcommon::row_headline()
             .'<h3>'.$lt{'ow'}.'&nbsp;'.$lt{'cc'}.'</h3>'
             .&Apache::lonhtmlcommon::row_closure()

             .&Apache::lonhtmlcommon::row_title($lt{'user'})
             .'<input type="text" size="15" name="ccuname" />'
             .&Apache::loncommon::studentbrowser_javascript()
             .&Apache::loncommon::selectstudent_link('ccrs','ccuname','ccdomain')
             .&Apache::lonhtmlcommon::row_closure()

             .&Apache::lonhtmlcommon::row_title($lt{'dmn'},"","LC_odd_row_value")
             .$domform
             .&Apache::lonhtmlcommon::row_closure()

             .&Apache::lonhtmlcommon::row_headline()
             .'<h3>'.$lt{'cinf'}.'</h3>'
             .&Apache::lonhtmlcommon::row_closure()

             .&Apache::lonhtmlcommon::row_title($lt{'ctit'})
             .'<input type="text" size="50" name="title" />'
             .&Apache::lonhtmlcommon::row_closure()

             .&Apache::lonhtmlcommon::row_title($lt{'chsr'})
             .$course_home
             .&Apache::lonhtmlcommon::row_closure()

             .&Apache::lonhtmlcommon::row_title($lt{'cidn'}.' ('.$lt{'opt'}.')')
             .'<input type="text" size="30" name="crsid" />'
             .&Apache::lonhtmlcommon::row_closure()
    );
    if ($crstype eq 'Course') {
        $r->print(&Apache::lonhtmlcommon::row_headline()
                 .'<h3>'.$lt{'iinf'}.'</h3>'
                 .$lt{'stat'}
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'ccod'})
                 .'<input type="text" size="30" name="crscode" />'
                 .'<br />('.$lt{'toin'}.')'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'snid'})
                 .'<input type="text" size="30" name="crssections" />'
                 .'<br />('.$lt{'csli'}.')'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'crcs'})
                 .'<input type="text" size="30" name="crsxlist" />'
                 .'<br />('.$lt{'cscs'}.')'
                 .&Apache::lonhtmlcommon::row_closure()
        );
    }

    # Table: New Course / Clone Course
    $r->print(&Apache::lonhtmlcommon::row_headline()
                 .'<h3>'.$lt{'crco'}.'</h3>'
                 .&Apache::loncommon::start_data_table()
                 .&Apache::loncommon::start_data_table_header_row()
                 .'<th>'.$lt{'cncr'}.'</th>'
                 .'<th>&nbsp;'.&mt('or').'&nbsp;</th>'
                 .'<th>'.$lt{'cecr'}.'</th>'
                 .&Apache::loncommon::end_data_table_header_row()
                 .&Apache::loncommon::start_data_table_row()

                 # New Course:
                 .'<td>'
                 .&Apache::lonhtmlcommon::start_pick_box()
                 .&Apache::lonhtmlcommon::row_title($lt{'map'})
                 .'<input type="text" size="50" name="topmap" />'
                 .'<a href="javascript:openbrowser(\'ccrs\',\'topmap\')">'.$lt{'smap'}.'</a>'
                 .&Apache::lonhtmlcommon::row_closure()
                 .&Apache::lonhtmlcommon::row_title('<label for="nonstd">'.$lt{'sacr'}.'</label>')
                 .'<input id="nonstd" type="checkbox" name="nonstandard" />'
                 .'<br />('.$lt{'ocik'}.')'
                 .&Apache::lonhtmlcommon::row_closure()
                 .&Apache::lonhtmlcommon::row_title($lt{'fres'})
                 .'<label>'
                 .'<input type="radio" name="firstres" value="blank" />'.$lt{'blnk'}
                 .'</label>'
                 .'&nbsp;');
    if ($crstype eq 'Community') {
        $r->print('<label>'
                 .'<input type="radio" name="firstres" value="nav" checked="checked" />'.$lt{'navi'}
                 .'</label>');
    } else {
        $r->print('<label>'
                 .'<input type="radio" name="firstres" value="syl" checked="checked" />'.$lt{'sllb'}
                 .'</label>'
                 .'&nbsp;'
                 .'<label>'
                 .'<input type="radio" name="firstres" value="nav" />'.$lt{'navi'}
                 .'</label>');
    }
    $r->print('<br />('.$lt{'stco'}.')'
                 .&Apache::lonhtmlcommon::row_closure(1)
                 .&Apache::lonhtmlcommon::end_pick_box()

                 .'<td>&nbsp;</td>'

                 # Clone Course:
                 .'<td>'
                 .&Apache::lonhtmlcommon::start_pick_box()
                 .&Apache::lonhtmlcommon::row_title($lt{'cid'})
                 .'<label>'
                 .'<input type="text" size="25" name="clonecourse" value="" />'
                 .'</label>'
                 .&Apache::lonhtmlcommon::row_closure(1)
                 .&Apache::lonhtmlcommon::row_title($lt{'dmn'})
                 .'<label>'
                 .$cloneform
                 .'</label>'
                 .&Apache::lonhtmlcommon::row_closure()
                 .&Apache::lonhtmlcommon::row_title($lt{'dsh'})
                 .'<label>'
                 .'<input type="radio" name="datemode" value="delete" /> '.$lt{'ncd'}
                 .'</label><br />'
                 .'<label>'
                 .'<input type="radio" name="datemode" value="preserve" /> '.$lt{'prd'}
                 .'</label><br />'
                 .'<label>'
                 .'<input type="radio" name="datemode" value="shift" checked="checked" /> '.$lt{'shd'}
                 .'</label>'
                 .' <input type="text" size="5" name="dateshift" value="365" />'
                 .&Apache::lonhtmlcommon::row_closure()
                 .&Apache::lonhtmlcommon::row_headline()
                 .$lt{'asov'}
                 .&Apache::lonhtmlcommon::row_closure(1)
                 .&Apache::lonhtmlcommon::end_pick_box()
                 .'</td>'
                 .&Apache::loncommon::end_data_table_row()
                 .&Apache::loncommon::end_data_table()
                 .&Apache::lonhtmlcommon::row_closure()
    );

    $r->print(&Apache::lonhtmlcommon::row_headline()
                 .'<h3>'.$lt{'dads'}.'</h3>'
                 .&Apache::lonhtmlcommon::row_closure()
                 .&Apache::lonhtmlcommon::row_title($lt{'dacc'})
                 .$access_table
                 .&Apache::lonhtmlcommon::row_closure()
    );

    $r->print(&Apache::lonhtmlcommon::row_headline()
                 .'<h3>'.$lt{'assp'}.'</h3>'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'oaas'})
                 .'<input type="checkbox" name="openall" />'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_headline()
                 .'<h3>'.$lt{'mssg'}.'</h3>'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'scpf'})
                 .'<input type="checkbox" name="setpolicy" checked="checked" />'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'scfc'})
                 .'<input type="checkbox" name="setcontent" checked="checked" />'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_headline()
                 .'<h3>'.$lt{'cmmn'}.'</h3>'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'dsrd'})
                 .'<input type="checkbox" name="disresdis" />'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'dsuc'})
                 .'<input type="checkbox" name="disablechat" />'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_headline()
                 .'<h3>'.$lt{'acco'}.'</h3>'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'snak'})
                 .'<input type="checkbox" name="setkeys" />'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'kaut'})
                 .'<input type="text" size="30" name="keyauth" />'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_headline()
                 .'<h3>'.$lt{'rshm'}.'</h3>'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'rshm'})
                 .'<input type="text" name="reshome" size="30" value="/res/'.$defdom.'/" />'
                 .&Apache::lonhtmlcommon::row_closure()
    );
    if ($crstype eq 'Course') {
        $r->print(&Apache::lonhtmlcommon::row_headline()
                 .'<h3>'.$lt{'aens'}.'</h3>'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'aadd'})
                 .'<label><input type="radio" name="autoadds" value="1" />'.$lt{'yes'}.'</label>'
                 .'<label><input type="radio" name="autoadds" value="0" checked="checked" />'.$lt{'no'}.'</label>'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'audr'})
                 .'<label><input type="radio" name="autodrops" value="1" />'.$lt{'yes'}.'</label>'
                 .'<label><input type="radio" name="autodrops" value="0" checked="checked" />'.$lt{'no'}.'</label>'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'dacu'})
                 .$enroll_table
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'psam'})
                 .$lt{'pcda'}.'<br />'
                 .$krbform.'<br />'
                 .$intform.'<br />'
                 .$locform
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'nech'})
                 .$lt{'nccl'}.'<br/>'
                 .'<label><input type="radio" name="notify_owner" value="1" />'.$lt{'yes'}.'</label>'
                 .'<label><input type="radio" name="notify_owner" value="0" checked="checked" />'.$lt{'no'}.'</label>'
                 .'<br />'
                 .$lt{'ndcl'}.'<br/>'
                 .'<label><input type="radio" name="notify_dc" value="1" />'.$lt{'yes'}.'</label>'
                 .'<label><input type="radio" name="notify_dc" value="0" checked="checked" />'.$lt{'no'}.'</label>'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'irsp'})
                 .'<label>'
                 .'<input type="radio" name="showphotos" value="1" />'.$lt{'yes'}
                 .'</label> '
                 .'<label>'
                 .'<input type="radio" name="showphotos" value="0" checked="checked" />'.$lt{'no'}
                 .'</label> '
                 .&Apache::lonhtmlcommon::row_closure()
        );
    }
 
    $r->print(&Apache::lonhtmlcommon::row_headline()
                 .'<h3>'.$lt{'cgrs'}.'</h3>'
                 .&Apache::lonhtmlcommon::row_closure()

                 .&Apache::lonhtmlcommon::row_title($lt{'cgrq'})
                 .'<input type="text" name="crsquota" value="20" size="6" />Mb'
                 .&Apache::lonhtmlcommon::row_closure(1)

                 .&Apache::lonhtmlcommon::end_pick_box()
                 .'</div>'
    );
    if ($crstype eq 'Course') {
        $r->print('
<p>
<input type="hidden" name="prevphase" value="courseone" />
<input type="hidden" name="phase" value="coursetwo" />
<input type="button" onClick="verify_message(this.form)" value="'.$lt{'opco'}.'" />'
        );
    } elsif ($crstype eq 'Community') {
        $r->print('
<p>
<input type="hidden" name="prevphase" value="groupone" />
<input type="hidden" name="phase" value="grouptwo" />
<input type="submit" value="'.$lt{'opcm'}.'" />
</p>'
        );
    }
    $r->print('</form>'.
              $end_page);
}

# ====================================================== Phase two: make course

sub create_course {
    my $r=shift;
    my $ccuname =&LONCAPA::clean_username($env{'form.ccuname'});
    my $ccdomain=&LONCAPA::clean_domain($env{'form.ccdomain'});
    my $crstype = 'Community';
    my ($enrollstart,$enrollend,$startaccess,$endaccess);

    if ($env{'form.phase'} eq 'coursetwo') {
        $crstype='Course';
        $enrollstart=&Apache::lonhtmlcommon::get_date_from_form('startenroll');
        $enrollend=&Apache::lonhtmlcommon::get_date_from_form('endenroll');
    }
    $startaccess = &Apache::lonhtmlcommon::get_date_from_form('startaccess');
    $endaccess   = &Apache::lonhtmlcommon::get_date_from_form('endaccess');

    my $autharg;
    my $authtype;

    if ($env{'form.login'} eq 'krb') {
        $authtype = 'krb';
        $authtype .=$env{'form.krbver'};
        $autharg = $env{'form.krbarg'};
    } elsif ($env{'form.login'} eq 'int') {
        $authtype ='internal';
        if ((defined($env{'form.intarg'})) && ($env{'form.intarg'})) {
            $autharg = $env{'form.intarg'};
        }
    } elsif ($env{'form.login'} eq 'loc') {
        $authtype = 'localauth';
        if ((defined($env{'form.locarg'})) && ($env{'form.locarg'})) {
            $autharg = $env{'form.locarg'};
        }
    }
    my $logmsg;
    my $start_page=&Apache::loncommon::start_page('Create a New '.$crstype);
    my $crumbs = &Apache::lonhtmlcommon::breadcrumbs('Creation Outcome','Create_Course',undef,'Create_Courses');

    $r->print($start_page.$crumbs);

    my $args = {
               crstype => $crstype,
               ccuname => $ccuname,
               ccdomain => $ccdomain,
               cdescr => $env{'form.title'},
               curl => $env{'form.topmap'},
               course_domain => $env{'request.role.domain'},
               course_home =>  $env{'form.course_home'},
               nonstandard => $env{'form.nonstandard'},
               crscode => $env{'form.crscode'},
               crsquota => $env{'form.crsquota'},
               clonecourse => $env{'form.clonecourse'},
               clonedomain => $env{'form.clonedomain'},
               datemode => $env{'form.datemode'},
               dateshift => $env{'form.dateshift'},
               crsid => $env{'form.crsid'},
               curruser => $env{'user.name'}.':'.$env{'user.domain'},
               crssections => $env{'form.crssections'},
               crsxlist => $env{'form.crsxlist'},
               autoadds => $env{'form.autoadds'},
               autodrops => $env{'form.autodrops'},
               notify_owner => $env{'form.notify_owner'},
               notify_dc => $env{'form.notify_dc'},
               no_end_date => $env{'form.no_end_date'},
               showphotos => $env{'form.showphotos'},
               authtype => $authtype,
               autharg => $autharg,
               enrollstart => $enrollstart,
               enrollend => $enrollend,
               startaccess => $startaccess,
               endaccess => $endaccess,
               setpolicy => $env{'form.setpolicy'},
               setcontent => $env{'form.setcontent'},
               reshome => $env{'form.reshome'},
               setkeys => $env{'form.setkeys'},
               keyauth => $env{'form.keyauth'},
               disresdis => $env{'form.disresdis'},
               disablechat => $env{'form.disablechat'},
               openall => $env{'form.openall'},
               firstres => $env{'form.firstres'}
               };

    #
    # Verify data
    #
    # Check the veracity of the course coordinator
    if (&Apache::lonnet::homeserver($ccuname,$ccdomain) eq 'no_host') {
	$r->print('<form action="/adm/createuser" method="post" name="crtuser">');
        $r->print('<div class="LC_warning">'
                  .&mt('No such user [_1] at domain [_2].','<tt>'.$ccuname.'</tt>','<tt>'.$ccdomain.'</tt>')
                 .'</div>');
	$r->print(&mt('Please click Back on your browser and select another user, or [_1]Create User[_2]'
	             , '<input type="hidden" name="phase" value="get_user_info" />'
                      .'<input type="hidden" name="ccuname" value="'.$ccuname.'" />'
                      .'<input type="hidden" name="ccdomain" value="'.$ccdomain.'" />'
                      .'<input name="userrole" type="submit" value="'
                     , '" />'
                     )
                 .'</form>'.&Apache::loncommon::end_page()
                 );
	return;
    }
    # Check the proposed home server for the course
    my %host_servers = 
	&Apache::lonnet::get_servers($env{'request.role.domain'},'library');
    if (! exists($host_servers{$env{'form.course_home'}})) {
        $r->print(&mt('Invalid home server for course: [_1]'
                     ,$env{'form.course_home'}.&Apache::loncommon::end_page()));
        return;
    }
    my ($courseid,$crsudom,$crsunum);
    my ($success,$output) = 
	&Apache::loncommon::construct_course($args,\$logmsg,\$courseid,
					     \$crsudom,\$crsunum,
					     $env{'user.domain'},
					     $env{'user.name'},'dc_create');
    $r->print($output);
    if ($success) {
        #
	# Make the requested user a course coordinator or group coordinator
        #
	if (($ccdomain) && ($ccuname)) {
            my $ccrole = 'cc';
            if ($crstype eq 'Community') {
                $ccrole = 'co';
            }    
	    $r->print(&mt('Assigning role of '.$crstype.' Coordinator to [_1]:',
			 ,'<i>'.$ccuname.':'.$ccdomain.'</i>')
		      .&Apache::lonnet::assignrole($ccdomain,$ccuname,$courseid,
						  $ccrole,'','','','','createcourse').'<p>');
	}
	if ($env{'form.setkeys'}) {
	    $r->print(
		      '<p><a href="/adm/managekeys?cid='.$crsudom.'_'.$crsunum.'">'.&mt('Manage Access Keys').'</a></p>');
	}
        # Flush the course logs so reverse user roles immediately updated
	&Apache::lonnet::flushcourselogs();
	$r->print('<p>'.&mt('Roles will be active at next login').'.</p>');
    }
    $r->print('<p><a href="/adm/createcourse?phase='.lc($crstype).'one">'.
	      &mt("Create Another $crstype").'</a></p>'.
	      &Apache::loncommon::end_page());
}

sub print_intro_page {
    my ($r,$show_all_choices) = @_;
    my $start_page =
        &Apache::loncommon::start_page('Create a New Course or Community');
    my $crumbs = &Apache::lonhtmlcommon::breadcrumbs('Creation Options','Create_Course',undef,'Create_Courses');
    my $end_page =
        &Apache::loncommon::end_page();
    my $helplink=&Apache::loncommon::help_open_topic('Create_Course_Community',&mt('Help on Creating Courses and Communities'));
    my @choices = ({ internal_name     => 'courseone',
                     name              => &mt('Create a single course'),
                     short_description => &mt('Create a new course by completing an online form.'),
                   });
    if ($show_all_choices) {
        push(@choices,{ internal_name     => 'groupone',
                        name              => &mt('Create a single community'),
                        short_description =>
    &mt('Create a new collaborative community by completing an online form.'),
                      },
                      { internal_name     => 'batchone',
                        name              => &mt('Create courses/communities by uploading an attributes file'),
                        short_description =>
    &mt('Upload an attributes file containing specifications for one or more courses or communities in XML format.'),
                        help => 'Batch_Creation',
                      },
                      { internal_name     => 'requestdisplay',
                        name              => &mt('Approve or reject requests'),
                        short_description =>
    &mt('Display course and community creation requests submitted by authorized users, held pending approval by a Domain Coordinator.'),
                      },
                      { internal_name     => 'pendingdisplay',
                        name              => &mt('Display official course requests'),
                        short_description =>
    &mt('Display official course requests submitted by authorized users, held pending validation of instructor status.'),
                      },
                      { internal_name     => 'creationlog',
                        name              => &mt('View creation log'),
                        short_description =>
    &mt('Display information about when, how and by whom courses and communities were created in this domain.'),
                      });
    } else {
        push(@choices,{ internal_name     => 'batchone',
                        name              => &mt('Create courses by uploading an attributes file'),
                        short_description =>
    &mt('Upload an attributes file containing specifications for one or more courses in XML format.'),
                        help => 'Batch_Creation',
                      });
    }
    my $options;
    foreach my $choice (@choices) {
        $options .='    <h3><a href="/adm/createcourse?phase='.
            $choice->{'internal_name'}.'" >'.
            $choice->{'name'}.'</a>';
        if (exists($choice->{'help'})) {
            $options .= &Apache::loncommon::help_open_topic($choice->{'help'});
        }
        $options .= "</h3>\n".'    '.('&nbsp;'x8).$choice->{'short_description'}.
            "\n";
    }
    $r->print(<<ENDDOCUMENT);
$start_page
$crumbs
$options
$end_page
ENDDOCUMENT
}

sub upload_batchfile {
    my $r = shift;
    my $start_page =
        &Apache::loncommon::start_page('Create a New Course or Community');
    my $crumbs = &Apache::lonhtmlcommon::breadcrumbs('Upload Course/Community Attributes File','Create_Course',undef,'Create_Courses');
    my $end_page =
        &Apache::loncommon::end_page();
    $r->print($start_page.$crumbs);
    $r->print('<h3>'.&mt('Upload a courses or communities attributes file').'</h3>');
    $r->print('<form name="batchcreate" method="post" '.
                'enctype="multipart/form-data" action="/adm/createcourse">'.
              '<input type="file" name="coursecreatorxml" />'.
              '<input type="hidden" name="phase" value="batchtwo" /><br /><br />'.
              '<input type="submit" name="batchsubmit" '.
              'value="Create Courses/Communities" /></form>');
    $r->print($end_page);
    return;
}

sub process_batchfile {
    my $r = shift;
    my $start_page =
        &Apache::loncommon::start_page('Create a New Course or Community');
    my $crumbs = &Apache::lonhtmlcommon::breadcrumbs('Creation Outcome','Create_Course',undef,'Create_Courses');
    my $end_page =
        &Apache::loncommon::end_page();
    my $defdom=$env{'request.role.domain'};
    my $batchfilepath=&Apache::lonnet::userfileupload('coursecreatorxml',undef,
                                                      'batchupload',undef,undef,
                                                       undef,undef,$defdom);
    my ($batchdir,$filename) = ($batchfilepath =~ m-^(.+)/pending/([^/]+)$-);
    my ($result,$logmsg);
    if (-e "$batchfilepath") {
        open(FILE,"<$batchfilepath");
        my @buffer = <FILE>;
        close(FILE);
        if ((defined($filename)) && (defined($batchdir))) {
            my @requests = ($filename);
            my %courseids = ();
            ($result,$logmsg) = &LONCAPA::batchcreatecourse::create_courses(
                                        \@requests,\%courseids,'web',$defdom,
                                        $env{'user.name'},$env{'user.domain'});
            if ($result) {
                if (!-e "$batchdir/processed") {
                    mkdir("$batchdir/processed", 0755);
                    open(FILE,">$batchdir/processed/$filename");
                    print FILE @buffer;
                    close(FILE);
                    if (-e "$batchdir/processed/$filename") {
                        unlink("$batchdir/pending/$filename");
                    }
                }
            }
        }
    }
    $r->print($start_page.$crumbs.$logmsg.$result.'<br /><a href="/adm/createcourse">'.
              &mt('Creation options menu').'</a>'.$end_page);
 
}

sub courserequestbrowser_javascript {
    return <<"ENDREQBRW";
<script type="text/javascript">
// <![CDATA[
var steditbrowser;
function opencoursereqdisplay(cdom,cnum,queue) {
    var url = '/adm/requestcourse?action=display';
    url += '&showdom='+cdom+'&cnum='+cnum+'&queue='+queue;
    var title = 'Course_Request_Browser';
    var options = 'scrollbars=1,resizable=1,menubar=0';
    options += ',width=700,height=600';
    var stdeditbrowser = open(url,title,options,'1');
    stdeditbrowser.focus();
}

// ]]>
</script>
ENDREQBRW
}

sub print_creation_logs {
    my ($r) = @_;
    my $formname = 'creationlog';
    my $dom = $env{'request.role.domain'};
    $r->print('<form action="/adm/createcourse" method="post" name="'.$formname.'">');
    # set defaults
    my $now = time();
    my $defstart = $now - (7*24*3600); #7 days ago
    my %defaults = (
                     page                => '1',
                     show                => '10',
                     type                => 'any',
                     context             => 'any',
                     created_before_date => $now,
                     created_after_date  => $defstart,
                   );
    my ($contexts,$contextnames) = &context_names();
    my $more_records = 0;
    my %curr;
    foreach my $item ('show','page','type','context') {
        $curr{$item} = $env{'form.'.$item};
    }
    $curr{'created_after_date'} = &Apache::lonhtmlcommon::get_date_from_form('created_after_date');
    $curr{'created_before_date'} = &Apache::lonhtmlcommon::get_date_from_form('created_before_date');
    foreach my $key (keys(%defaults)) {
        if ($curr{$key} eq '') {
            $curr{$key} = $defaults{$key};
        }
    }
    my %whodunit;
    $r->print(&creation_display_filter($formname,\%curr));
    my $showntablehdr = 0;
    my $tablehdr = &Apache::loncommon::start_data_table().
                   &Apache::loncommon::start_data_table_header_row().
                   '<th>&nbsp;</th><th>'.&mt('Creation Date').'</th>'.
                   '<th>'.&mt('Creator').'</th><th>'.&mt('Description').'</th>'.
                   '<th>'.&mt('Owner(s)').'</th>';
    if (($curr{'type'} eq 'official') || ($curr{'type'} eq 'any')) {
       $tablehdr .= '<th>'.&mt('Institutional Code').'</th>';
    }
    if ($curr{'type'} eq 'any') { 
        $tablehdr .= '<th>'.&mt('Course Type').'</th>';
    }
    if ($curr{'context'} eq 'any') {
        $tablehdr .= '<th>'.&mt('Creation Context').'</th>';
    }
    $tablehdr .= &Apache::loncommon::end_data_table_header_row();
    my ($minshown,$maxshown);
    $minshown = 1;
    my $count = 0;
    if ($curr{'show'} ne &mt('all')) {
        $maxshown = $curr{'page'} * $curr{'show'};
        if ($curr{'page'} > 1) {
            $minshown = 1 + ($curr{'page'} - 1) * $curr{'show'};
        }
    }
    my $crstype = 'Course';
    if ($curr{'type'} eq 'any') {
        $crstype = '.';
    } elsif ($curr{'type'} eq 'community') {
        $crstype = 'Community';
    }

    my ($instcodefilter,$regexpok);
    my (@codetitles,%cat_titles,%cat_order,%cat_items,$officialjs);
    my ($jscript,$totcodes,$numtitles,$lasttitle) =
        &Apache::courseclassifier::instcode_selectors_data($dom,$formname,
                           \%cat_items,\@codetitles,\%cat_titles,\%cat_order,
                           $officialjs);
    if ($numtitles) {
        if (($curr{'type'} eq 'official') || ($curr{'type'} eq 'unofficial')) {
            $instcodefilter = &instcode_srchstr($dom,$numtitles);
                &Apache::courseclassifier::instcode_search_str($dom,$numtitles);
            if ($curr{'type'} eq 'official') {
                $regexpok = 1;
            } else {
                unless ($instcodefilter eq '') {
                    $regexpok = -1;
                }
            }
        }
    }
    if ($instcodefilter eq '') { $instcodefilter = '.'; }

    my $creationcontext = '.';
    my $context_regexp = join('|',@{$contexts});
    if ($curr{'context'} =~ /^($context_regexp)$/) {
        $creationcontext = $curr{'context'};
    } 
    my %courses = 
        &Apache::lonnet::courseiddump($dom,'.',1,$instcodefilter,'.','.',undef,undef,
                                      $crstype,$regexpok,undef,undef,'1','creationlog',
                                      undef,undef,undef,$curr{'created_before_date'},
                                      $curr{'created_after_date'},$creationcontext);
    foreach my $cid (sort { $courses{$b}{'created'}<=>$courses{$a}{'created'} } (keys(%courses))) {
        if ($curr{'show'} ne &mt('all')) {
            if ($count >= $curr{'page'} * $curr{'show'}) {
                $more_records = 1;
                last;
            }
        }
        $count ++;
        next if ($count < $minshown);
        if (!$showntablehdr) {
            $r->print($tablehdr);
            $showntablehdr = 1;
        }
        if ($courses{$cid}{'creator'} ne '') {
            if ($whodunit{$courses{$cid}{'creator'}} eq '') {
                my ($uname,$udom) = split(':',$courses{$cid}{'creator'});
                $whodunit{$courses{$cid}{'creator'}} =
                    &Apache::loncommon::plainname($uname,$udom);
            }
        }
        my $description = $courses{$cid}{'description'};
        my @owners;
        my $singleowner = $courses{$cid}{'owner'};
        push(@owners,$singleowner);
        if (ref($courses{$cid}{'co-owners'}) eq 'ARRAY') {
            foreach my $item (@{$courses{$cid}{'co-owners'}}) {
                push(@owners,$item);
            }
        }
        my %ownernames;
        foreach my $owner (@owners) {
            my ($ownername,$ownerdom);     
            if ($owner =~ /:/) {
                ($ownername,$ownerdom) = split(/:/,$owner);
            } else {
                $ownername = $owner;
                if ($owner ne '') {
                    $ownerdom = $dom;
                }
            }
            if ($ownername ne '' && $ownerdom ne '') {
                unless (ref($ownernames{$ownername.':'.$ownerdom}) eq 'HASH') { 
                    my %namehash=&Apache::loncommon::getnames($ownername,$ownerdom);
                    $ownernames{$ownername.':'.$ownerdom} = \%namehash;
                }
            }
        }
        my @lastnames;
        foreach my $owner (keys(%ownernames)) {
            if (ref($ownernames{$owner}) eq 'HASH') {
                push(@lastnames,$ownernames{$owner}{'lastname'});
            }
        }
        my $ownerlastnames = join(', ',sort(@lastnames));
        my $showtype;
        if ($courses{$cid}{type} eq 'Community') {
            $showtype = &mt('community');
        } else {
            my $instcode = $courses{$cid}{inst_code};
            if ($instcode ne '')  {
                $showtype = &mt('official');
            } else {
                $showtype = &mt('unofficial');
            }
        }
        my $showcontext;
        
        if ($courses{$cid}{context} =~ /^($context_regexp)$/) {
            $showcontext = $contextnames->{$courses{$cid}{context}};
        }
        my $created = &Apache::lonlocal::locallocaltime($courses{$cid}{'created'});
        my ($cdom,$cnum) = split('_',$cid);
        $r->print(&Apache::loncommon::start_data_table_row()."\n".
                  '<td>'.$count.'</td><td>'.$created.'</td>'."\n".
                  '<td>'.$whodunit{$courses{$cid}{creator}}.'</td>'."\n".
                  '<td>'.$description.'&nbsp;<a href="javascript:ToSyllabus('."'$cdom','$cnum'".')">'.&mt('Syllabus').'</a></td><td>'.$ownerlastnames.'</td>');
        if (($curr{'type'} eq 'official') || ($curr{'type'} eq 'any')) {
            $r->print('<td>'.$courses{$cid}{'inst_code'}.'</td>');
        }
        if ($curr{'type'} eq 'any') {
            $r->print('<td>'.$showtype.'</td>');
        }
        if ($curr{'context'} eq 'any') {
            $r->print('<td>'.$showcontext.'</td>');
        }
        $r->print('<td>'.&Apache::loncommon::end_data_table_row()."\n");
    }
    if ($showntablehdr) {
        $r->print(&Apache::loncommon::end_data_table().'<br />');
        if (($curr{'page'} > 1) || ($more_records)) {
            $r->print('<table><tr>');
            if ($curr{'page'} > 1) {
                $r->print('<td><a href="javascript:chgPage('."'previous'".');">'.&mt('Previous [_1] changes',$curr{'show'}).'</a></td>');
            }
            if ($more_records) {
                $r->print('<td><a href="javascript:chgPage('."'next'".');">'.&mt('Next [_1] changes',$curr{'show'}).'</a></td>');
            }
            $r->print('</tr></table>');
            $r->print(<<"ENDSCRIPT");
<script type="text/javascript">
// <![CDATA[
function chgPage(caller) {
    if (caller == 'previous') {
        document.$formname.page.value --;
    }
    if (caller == 'next') {
        document.$formname.page.value ++;
    }
    document.$formname.submit();
    return;
}
// ]]>
</script>
ENDSCRIPT
        }
    } else {
        $r->print('<p class="LC_info">'.
                  &mt('There are no records to display').
                  '</p>');
    }
    $r->print('<input type="hidden" name="page" value="'.$curr{'page'}.'" />'.
              '<input type="hidden" name="phase" value="creationlog" /></form>');
    return;
}

sub creation_display_filter {
    my ($formname,$curr) = @_;
    my $nolink = 1;
    my ($contexts,$contextnames) = &context_names();
    my $output = '<table><tr><td valign="top">'.
                 '<span class="LC_nobreak"><b>'.&mt('Records/page:').'</b></span><br />'.
                 &Apache::lonmeta::selectbox('show',$curr->{'show'},undef,
                                              (&mt('all'),5,10,20,50,100,1000,10000)).
                 '</td><td>&nbsp;&nbsp;</td>';
    my $startform =
        &Apache::lonhtmlcommon::date_setter($formname,'created_after_date',
                                            $curr->{'created_after_date'},undef,
                                            undef,undef,undef,undef,undef,undef,$nolink);
    my $endform =
        &Apache::lonhtmlcommon::date_setter($formname,'created_before_date',
                                            $curr->{'created_before_date'},undef,
                                            undef,undef,undef,undef,undef,undef,$nolink);
    $output .= '<td valign="top"><b>'.&mt('Window during which course/community was created:').'</b><br />'.
               '<table><tr><td>'.&mt('After:').
               '</td><td>'.$startform.'</td></tr>'.
               '<tr><td>'.&mt('Before:').'</td>'.
               '<td>'.$endform.'</td></tr></table>'.
               '</td>'.
               '<td>&nbsp;&nbsp;</td>';
    my ($types,$typenames) = &Apache::loncommon::course_types();
    if (ref($types) eq 'ARRAY') {
        if (@{$types} > 1) {
            $output .= '<td valign="top"><b>'.
                       &mt('Course Type:').'</b><br /><select name="type">';
            my $selstr = '';
            if ($curr->{'type'} eq 'any') {
                $selstr = ' selected="selected"';
            }
            $output .= '<option value="any"'.$selstr.'>'.&mt('Any').'</option>'."\n";
            foreach my $type (@{$types}) {
                my $selstr = '';
                if ($curr->{'type'} eq $type) {
                    $selstr = ' selected="selected"';
                }
                my $typename = $type;
                if (ref($typenames) eq 'HASH') {
                    if ($typenames->{$type} ne '') {
                        $typename = $typenames->{$type};
                    }
                }
                $output .= '<option value="'.$type.'"'.$selstr.'>'.$typename.'</option>'."\n";
            }
            $output .= '</select></td>';
        }
    }
    $output .= '<td>&nbsp;&nbsp;</td>'.
               '<td valign="top"><b>'.
               &mt('Creation Context:').'</b><br /><select name="context">';
    foreach my $context ('any',@{$contexts}) {
        my $selstr = '';
        if ($curr->{'context'} eq $context) {
            $selstr = ' selected="selected"';
        }
        $output .= '<option value="'.$context.'"'.$selstr.'>';
        if ($context eq 'any') {
            $output .= &mt('Any');
        } else {
            $output .= $contextnames->{$context};
        }
        $output .= '</option>'."\n";
    }
    $output .= '</select></td>'.
               '</tr></table>';

    # Update Display button
    $output .= '<p>'.
               '<input type="submit" value="'.&mt('Update Display').'" />'.
               '</p><hr />';
    return $output;
}

sub context_names {
    my @contexts = qw(auto web dc_create requestcourses);
    my %contextnames = 
        &Apache::lonlocal::texthash (
                        auto           => 'Automated creation from batch file',
                        web            => 'Batch creation from uploaded file',
                        dc_create      => 'Course or community created by Dom. Coord.', 
                        requestcourses => 'Processing of submitted course request',
        );
    return (\@contexts,\%contextnames);
}

sub instcode_srchstr {
    my ($domain,$numtitles) = @_;
    my $instcode;
    if (defined($numtitles) && $numtitles == 0) {
        $instcode = '.+';
    } else {
        my (%codedefaults,@code_order);
        my $defaults_result =
            &Apache::lonnet::auto_instcode_defaults($domain,\%codedefaults,
                                                    \@code_order);
        if ($defaults_result eq 'ok') {
            $instcode ='^';
            foreach my $item (@code_order) {
                $instcode .= $codedefaults{$item};
            }
            $instcode .= '$';
        } else {
            $instcode = '.+';
        }
    }
    return $instcode;
}

sub syllabuslink_javascript {
    return <<"ENDJS";

function ToSyllabus(cdom,cnum) {
    if (cdom == '' || cdom == null) {
        return;
    }
    if (cnum == '' || cnum == null) {
        return;
    }
    var options = 'height=600,width=800,resizable=yes,scrollbars=yes,location=no,menubar=no,toolbar=no';
    var url = "/public/"+cdom+"/"+cnum+"/syllabus";
    syllwin = window.open(url,'',options,1);
    syllwin.focus();
    return;
}

ENDJS

}

# ===================================================================== Handler
sub handler {
    my $r = shift;

    if ($r->header_only) {
       &Apache::loncommon::content_type($r,'text/html');
       $r->send_http_header;
       return OK;
    }

    my $show_all_choices = 0;
    my $primary_rev = &Apache::lonnet::get_server_loncaparev($env{'request.role.domain'});
    if (($primary_rev ne 'refused') && ($primary_rev ne 'error') &&
        ($primary_rev ne 'unknown_cmd') && ($primary_rev ne 'no_such_host')) {
        $show_all_choices = 1;
    }

    if (&Apache::lonnet::allowed('ccc',$env{'request.role.domain'})) {
       &Apache::loncommon::content_type($r,'text/html');
       $r->send_http_header;

       &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
                                            ['phase']);
       &Apache::lonhtmlcommon::clear_breadcrumbs();
       &Apache::lonhtmlcommon::add_breadcrumb
          ({href=>"/adm/createcourse",
            text=>"Creation Options",
            faq=>79,bug=>'Dom Coord Interface',});
       if (($env{'form.phase'} eq 'coursetwo') ||
           (($env{'form.phase'} eq 'grouptwo') && 
            ($show_all_choices))) { 
           &Apache::lonhtmlcommon::add_breadcrumb
                 ({href=>"/adm/createcourse?phase=$env{'form.prevphase'}",
                   text=>&mt('[_1] Creation Settings',),
                   faq=>9,bug=>'Dom Coord Interface',});
          &Apache::lonhtmlcommon::add_breadcrumb
                 ({href=>"/adm/createcourse?phase=$env{'form.phase'}",
                   text=>"Creation Outcome",
                   faq=>9,bug=>'Dom Coord Interface',});
           &create_course($r);
       } elsif (($env{'form.phase'} eq 'courseone') || 
                (($env{'form.phase'} eq 'groupone') && 
                ($show_all_choices))) {
           &Apache::lonhtmlcommon::add_breadcrumb
                 ({href=>"/adm/createcourse?phase=$env{'form.phase'}",
                   text=>&mt('[_1] Creation Settings',),
                   faq=>9,bug=>'Dom Coord Interface',});
	   &print_course_creation_page($r);
       } elsif ($env{'form.phase'} eq 'batchone') {
           &Apache::lonhtmlcommon::add_breadcrumb
                 ({href=>"/adm/createcourse?phase=$env{'form.phase'}",
                   text=>"Upload Description File",
                   faq=>9,bug=>'Dom Coord Interface',});
           &upload_batchfile($r);
       } elsif ($env{'form.phase'} eq 'batchtwo') {
           &Apache::lonhtmlcommon::add_breadcrumb
                 ({href=>"/adm/createcourse?phase=$env{'form.prevphase'}",
                   text=>"Upload Description File",
                   faq=>9,bug=>'Dom Coord Interface',});
           &Apache::lonhtmlcommon::add_breadcrumb
                 ({href=>"/adm/createcourse?phase=$env{'form.phase'}",
                   text=>"Creation Outcome",
                   faq=>9,bug=>'Dom Coord Interface',});
           &process_batchfile($r);
       } elsif (($env{'form.phase'} eq 'requestdisplay') && ($show_all_choices)) {
           &Apache::lonhtmlcommon::add_breadcrumb
                 ({href=>"/adm/createcourse?phase=$env{'form.phase'}",
                   text=>&mt('[_1] Display Request Queue',),
                   faq=>9,bug=>'Dom Coord Interface',});
           my $js = &courserequestbrowser_javascript();
           my $start_page=&Apache::loncommon::start_page('Display Requests',$js);
           my $crumbs = &Apache::lonhtmlcommon::breadcrumbs('Display Requests','Course_Request',undef,'Course_Request');
           $r->print($start_page.$crumbs."\n".'<div>'.
                     &Apache::loncoursequeueadmin::display_queued_requests(
                         'domain',$env{'request.role.domain'}).'</div>'.
                     &Apache::loncommon::end_page());
       } elsif (($env{'form.phase'} eq 'requestchange') && ($show_all_choices)) {
           if ($env{'form.queue'} eq 'pending') {
               &Apache::lonhtmlcommon::add_breadcrumb
                   ({href=>"/adm/createcourse?phase=pendingdisplay",
                     text=>&mt('Display Pending Queue',),
                     faq=>9,bug=>'Dom Coord Interface',});
           } else {
               &Apache::lonhtmlcommon::add_breadcrumb
                     ({href=>"/adm/createcourse?phase=requestdisplay",
                       text=>&mt('[_1] Display Request Queue',),
                       faq=>9,bug=>'Dom Coord Interface',});
           }
           &Apache::lonhtmlcommon::add_breadcrumb
               ({href=>"/adm/createcourse?phase=requestchange",
                 text=>&mt('[_1] Requests Updated',),
                 faq=>9,bug=>'Dom Coord Interface',},);
           my $start_page=&Apache::loncommon::start_page('Update Requests Result');
           my $crumbs = &Apache::lonhtmlcommon::breadcrumbs('Requests Updated','Course_Request',undef,'Course_Request');
           $r->print($start_page.$crumbs."\n".'<div>'.
                     &Apache::loncoursequeueadmin::update_request_queue(
                     'domain',$env{'request.role.domain'}).'</div>'.
                     &Apache::loncommon::end_page());
       } elsif (($env{'form.phase'} eq 'pendingdisplay') && ($show_all_choices)) {
           &Apache::lonhtmlcommon::add_breadcrumb
                 ({href=>"/adm/createcourse?phase=$env{'form.phase'}",
                   text=>&mt('Display Pending Queue',),
                   faq=>9,bug=>'Dom Coord Interface',});
           my $js = &courserequestbrowser_javascript();
           my $start_page=&Apache::loncommon::start_page('Display Pending Queue',$js);
           my $crumbs = &Apache::lonhtmlcommon::breadcrumbs('Display Pending Queue','Course_Request',undef,'Course_Request');
           $r->print($start_page.$crumbs."\n".'<div>'.
                     &Apache::loncoursequeueadmin::display_queued_requests(
                         'pending',$env{'request.role.domain'}).'</div>'.
                     &Apache::loncommon::end_page());
       } elsif (($env{'form.phase'} eq 'requestvalidation') && ($show_all_choices)) {
           my $js = &courserequestbrowser_javascript();
           &Apache::lonhtmlcommon::add_breadcrumb
                 ({href=>"/adm/createcourse?phase=pendingdisplay",
                   text=>&mt('Display Pending Queue'),},
                 {href=>"/adm/createcourse?phase=requestvalidation",
                   text=>&mt('Validation Attempted'),},);
           my $start_page=&Apache::loncommon::start_page('Requests Validation Result',$js);
           my $crumbs = &Apache::lonhtmlcommon::breadcrumbs('Validation Attempted','Course_Request',undef,'Course_Request');
           $r->print($start_page.$crumbs."\n".'<div>'.
                     &Apache::loncoursequeueadmin::process_official_reqs('domain',$env{'request.role.domain'}).'</div>'.
                     &Apache::loncommon::end_page());
       } elsif (($env{'form.phase'} eq 'creationlog') && ($show_all_choices)) {
           &Apache::lonhtmlcommon::add_breadcrumb
                 ({href=>"/adm/createcourse?phase=creationlog",
                   text=>&mt('[_1] Display Creation History',),});
           my $js = '<script type="text/javascript">'."\n".
                    '// <![CDATA['."\n".
                    &syllabuslink_javascript()."\n".
                    '// ]]>'."\n".
                    '</script>';
           my $start_page=&Apache::loncommon::start_page('Course/Community Creation Logs',$js);
           my $crumbs = &Apache::lonhtmlcommon::breadcrumbs('Created Courses/Communities','Creation_Log',undef,'Creation_Log');
           $r->print($start_page.$crumbs."\n".'<div>');
           &print_creation_logs($r);
           $r->print('</div>'.&Apache::loncommon::end_page());
       } else {
           &print_intro_page($r,$show_all_choices);
       }
   } else {
      $env{'user.error.msg'}=
        "/adm/createcourse:ccc:0:0:Cannot create courses or communities";
      return HTTP_NOT_ACCEPTABLE; 
   }
   return OK;
}

1;
__END__
